/** AVGFSIZE.C - Average file size of all files on a volume   Revision 1.04
***************************************************************************
** Written 17 July 1996 by Robert B. Clark <rclark@iquest.net>           **
** Turbo C v3.0 code.  Donated to the public domain.                     **
**                                                                       **
** Tested and developed on NW v3.11/MS-DOS v6.22, P5-100 16Mb RAM/1.2Gb  **
**                                                                       **
** This program totals all of the files and directories on a volume and  **
** reports the average file size.  Useful for determining the most       **
** efficient block size when setting up a new hard drive.                **
**                                                                       **
** See AVGFSIZE.H for history information.                               **
**************************************************************************/
#include "avgfsize.h"

static const char copyleft[]=
   "\n"PROGNAME" v"VERSION" Written 25 July 1996 by Robert B. Clark "
   "<rclark@iquest.net>.\nDonated to the public domain.\n";
/*************************************************************************/
int isredirected(void)
/* Returns 1 if this process' STDOUT and/or STDIN have been redirected;
   returns 0 otherwise. */
{
   asm {
      push ds                 /* Save DS                                */
      mov ax, _psp            /* AX = PSP segment                       */
      mov ds, ax              /* Ptr to Job File Table is at PSP:0034   */
      xor bx, bx              /* (nominally -> PSP:0018                 */
      les bx,[bx+0x34]        /* Now ES:BX = ptr to JFT                 */
      mov al,es:[bx]          /* Get JFT #0                             */
      mov ah,es:[bx+1]        /* Get JFT #1                             */
      pop ds                  /* Restore DS                             */
      cmp al, ah              /* If JFT0==JFT1 stdin/out not redirected */
      mov al,0x01             /* -> Stdin/out redirected; return true   */
      jne done                
      xor al,al               /* -> Not redirected; return false        */
   }
   done:
   return _AL;
} /* isredirected */


int drvexists(char drv)
/* Returns nonzero if drive 'drv' exists.  'drv' should be [a..z, A..Z] */
{
   int cdisk, retval;

   drv=(char)toupper(drv)-'A';
   cdisk=getdisk();        /* Returns current drive number 0=A, 1=B ... */
   setdisk(drv);                          /* Change drive to 'drv' */
   retval=(getdisk()!=drv ? 0: 1);        /* Are we on requested drv? */
   setdisk(cdisk);                        /* Restore original drive */
   return (retval);
} /* drvexists */


int process_dir(const char * pathspec)
/* Recursively span disk and keep running total of number of user files,
   directories, and the sum of their sizes.  Returns nonzero if 'pathspec'
   is a valid directory. */
{
   struct ffblk ff;
   char dirspec[MAXDIR], tspec[MAXDIR];
   register int x, y;
   int found;

   strcpy(dirspec,pathspec);
   if( LAST_CHAR(dirspec)!='\\')
      strcat(dirspec,"\\");

   if (!conio_redirected) {      /* Only if stdout not redirected */
      cprintf("\r%s",dirspec);   /* Display directory spec */
      clreol();
      x=wherex();                /* Everybody remember where we parked */
      y=wherey();
   }

/* Complete filespec mask for search */
   strcat(dirspec,"*.*");

   if((found=findfirst(dirspec, &ff, ANYFILE))==0)
   do {
      if( *ff.ff_name=='.' ) {   /* Skip current & parent dir entries */
         dir_cnt++;              /* but do keep count */
         continue;   
      }

      if( ff.ff_attrib & FA_DIREC ) {     /* Found directory -- recurse */
         dir_cnt++;                       /* at this point and pick up */
                                          /* its files and subdirs */
         strcpy(tspec, pathspec);
         if( LAST_CHAR(tspec)!='\\')
            strcat(tspec,"\\");
         strcat(tspec,ff.ff_name);
         process_dir(tspec);
      }
      else {                              /* Found a normal user file */
         file_cnt++;                      /* Tally count and file size */
         file_size_tot+=ff.ff_fsize;
         sum_sq+=ff.ff_fsize*ff.ff_fsize; /* Sum of squares for std dev */

         if( !conio_redirected) {         /* Don't display filenames if */
            gotoxy(x,y);                  /* stdout is not set to CON */
            cprintf("%s",ff.ff_name);
         }
      }
   } while (findnext(&ff)==0);
/* No files were found, but this is still a valid disk device if ENOENT */
   if(found==-1 && errno==ENOENT) found=0;
   return (found==-1?0:1);
} /* process_dir */


void disp_block_table(unsigned long avgsize, unsigned int au_size)
/* Display table showing how much disk space is wasted for various block
   sizes.  'avgsize' is average file size in bytes, 'au_size' is allocation
   unit size, also in bytes. */
{
   unsigned long wbytes,         /* Wasted bytes per avg file */
                 blk=512UL;      /* Start table at 512 bytes block size */
   char wdiskbytes[EXPNUM_LEN];
   float nblks;

   au_size/=1024;                /* Convert A.U. to kilobytes */
   printf("\n\nBlock Size     Blocks/Avg File    Slack Space/Avg File"
          "     Wasted Space");
   printf("  \n======================================================"
          "=================");
   do {
      nblks=(float)avgsize / blk;
      wbytes=(1-(nblks-(float)floor(nblks))) * blk;
      strcpy(wdiskbytes, Expand(wbytes * file_cnt, NULL));
      printf("\n%4lu%-8s\t%s\t%10s bytes%17s bytes %s",
         blk < 1024UL ? blk : blk/1024UL, blk < 1024UL ? " bytes" : "K",
         iDecimal(nblks, "%8.3lf", NULL), Expand(wbytes, NULL),
         wdiskbytes, au_size==(blk/1024UL) ? "*":"");
      blk+=blk;
   } while (blk<=65536UL);
} /* disp_block_table */


int get_dir_stats(char drv)
/* Gather and display directory/file count data */
{
   char root_path[4];
   unsigned long sstd_dev=0L, bytes_avail=0L, avg_size=0L;
   unsigned au_size=0;
   float pct=0.0;
   struct dfree df;
   int retval;

   sprintf(root_path,"%c:\\",drv);

/* First, check to see if drive exists */
   if(!drvexists(drv))
      fprintf(stderr,"\n\a%c: %s", drv, sys_errlist[EINVDRV]);
   else
   {
      printf("\nGathering file and directory information "
         "for drive %c: ...\n",drv);

      /* If files/directories were successfully read, show stats */
      if((retval=process_dir(root_path))!=0) {
         printf("\r");
         clreol();

         /* Calculate mean, sample std dev (On-1) and percent std dev */
         if(file_cnt) {
            avg_size = file_size_tot / file_cnt;
            if( file_cnt>1 )
               sstd_dev = sqrt(((file_cnt * sum_sq) - (file_size_tot *
                  file_size_tot)) / (file_cnt*(file_cnt-1)));
            if( avg_size )
               pct = (float)sstd_dev / avg_size * 100.0;
         }

         /* Calculate the block size and available disk space for drv */
         getdfree(drv-'A'+1, &df);
         if( df.df_sclus != 0xffff) {
            au_size=df.df_sclus * df.df_bsec;
            bytes_avail=(long)au_size * (long)df.df_avail;
         }

#define BFLD  "%14s"        /* "Bytes" field width */

         printf("\nDisk space used      :"BFLD" byte%s",
            Expand(file_size_tot,NULL), PLURAL(file_size_tot));
         printf("\nDisk space available :"BFLD" byte%s",
            Expand(bytes_avail,NULL), PLURAL(bytes_avail));
         printf("\nAllocation unit size :"BFLD" bytes (%uK)",
            Expand(au_size,NULL), au_size/1024);
         printf("\nDirectory count      :"BFLD" director%s",
            Expand(dir_cnt,NULL), PLURALY(dir_cnt));
         printf("\nUser file count      :"BFLD" file%s",
            Expand(file_cnt,NULL),PLURAL(file_cnt));
         printf("\nTotal file count     :"BFLD" file%s",
            Expand(dir_cnt+file_cnt,NULL), PLURAL(dir_cnt+file_cnt));
         printf("\n----------------------------------------------------"
            "\nAVERAGE FILE SIZE    :"BFLD"", Expand(avg_size, NULL));
         printf(" \xf1 %s byte%s (\xf1 %s%%)",
            Expand(sstd_dev, NULL), PLURAL(sstd_dev),
            iDecimal(pct,"%.1lf",NULL));
         disp_block_table(avg_size, au_size);
      }
      else
         perror("\a");        /* Error reading file directory */
   }
   return (retval);
} /* get_dir_stats */

int main(int argc, char * argv[])
{
   char fname[MAXFILE], drv;

/* Display attribution banner (and help if requested) */
   fprintf(stderr, copyleft);
   if (argc>1 && strchr(argv[1],'?')!=NULL) {
      fnsplit(argv[0],NULL,NULL,fname,NULL);
      printf("\nUsage: %s [drive:]\n\t"
         "Reports the average size of all files on drive [drive:]\n\t"
         "[drive:] defaults to the current drive.\n", fname);
      exit(EXIT_FAILURE);
   }

/* If no arg given, default to current drive */
   if(argc==1)
      drv=getdisk()+'A';
   else 
      drv=toupper(argv[1][0]);

/* Clear global tally variables */
   file_cnt=dir_cnt=file_size_tot=sum_sq=0L;

/* Global conio_redirected true if stdin or stdout have been redirected */
   conio_redirected = isredirected();

/* Send text output directly to video RAM (faster) */
   directvideo=1;             

/* Process volume directories & display data */
   return( get_dir_stats(drv) ? EXIT_SUCCESS : EXIT_FAILURE );
} /* main */
