#include <stdio.h>
//#include <stdlib.h>
#include <string.h>
#include <glib.h>
//#include <unistd.h>

#include "../libendeavour2-base/endeavour2.h"

#include "zip_tool_io.h"
#include "config.h"


static gchar last_error[1024];


static void zt_open_file_to_string(
	const gchar *path, gchar *buf, gint buf_len
);
static gchar *zt_get_ziptool_program(void);

gchar *zt_last_error(void);

gint zt_mount(EDVDevice *dev_ptr);
gint zt_unmount(EDVDevice *dev_ptr);

ZipToolLockState zt_device_is_protected(EDVDevice *dev_ptr);

gint zt_lock(
	EDVDevice *dev_ptr, const gchar *password
);
gint zt_unlock(
	EDVDevice *dev_ptr, const gchar *password
);

gint zt_spin_down(EDVDevice *dev_ptr);
gint zt_eject(EDVDevice *dev_ptr);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Reads the contents of the file into the specified string.
 */
static void zt_open_file_to_string(
	const gchar *path, gchar *buf, gint buf_len
)
{
	FILE *fp;

	if(STRISEMPTY(path))
	{
		if((buf != NULL) && (buf_len > 0))
			*buf = '\0';
		return;
	}

	fp = fopen((const char *)path, "rb");
	if(fp == NULL)
	{
		if((buf != NULL) && (buf_len > 0))
			*buf = '\0';
		return;
	}

	if(buf != NULL)
	{
		const gint	units_to_read = buf_len / sizeof(gchar),
				units_read = (gint)fread(
			buf,
			sizeof(gchar),
			(size_t)units_to_read,
			fp
		);
		if(units_read >= 0)
			buf[units_read] = '\0';
		else
			*buf = '\0';
	}

	fclose(fp);
}

/*
 *	Returns the full path to the ZipTool program or NULL on error.
 *
 *	The last_error will be updated if the ZipTool program is not
 *	found.
 *
 *	The calling function must deleted the returned string.
 */
static gchar *zt_get_ziptool_program(void)
{
	gchar *prog = edv_which(ZIPTOOL_PROG);
	if(prog == NULL)
		g_snprintf(
			last_error, sizeof(last_error),
"Unable to find the program \"%s\".",
			ZIPTOOL_PROG
		);
	return(prog);
}

/*
 *	Returns a statically allocated string describing the last
 *	error or NULL if there was no error.
 */
gchar *zt_last_error(void)
{
	return(STRISEMPTY(last_error) ? NULL : last_error);
}

/*
 *	Mounts the device.
 *
 *      Returns:
 *
 *      0       Success (or device is already mounted).
 *      -1      General error
 *      -2      Bad or ambiguous value
 *      -3      Systems error or unable to execute command
 */
gint zt_mount(EDVDevice *dev_ptr)
{
	gint pid;
	gchar		*cmd,
			*prog,
			*stderr_path;

	*last_error = '\0';		/* Reset last error message */

	if(dev_ptr == NULL)
		return(-2);
	if(STRISEMPTY(dev_ptr->device_path))
		return(-2);
	if(STRISEMPTY(dev_ptr->mount_path))
		return(-2);

	/* Get the ZipTool program */
	prog = zt_get_ziptool_program();
	if(prog == NULL)
		return(-1);

	/* Get the output file paths */
	stderr_path = edv_tmp_name(NULL);

	/* Format and execute mount command */
	cmd = g_strdup_printf(
		"\"%s\" -m \"%s\" \"%s\" > \"%s\" 2> \"%s\"",
		prog,
		dev_ptr->device_path,
		dev_ptr->mount_path,
		PATH_DEV_NULL,
		stderr_path
	);
	pid = edv_system_wait(
		cmd,
		NULL
	);
	g_free(cmd);

	/* Get any error message */
	zt_open_file_to_string(
		stderr_path,
		last_error,
		sizeof(last_error)
	);

	/* Remove the output files */
	(void)edv_unlink(stderr_path);
	g_free(stderr_path);

	g_free(prog);

	return((pid > -1) ? 0 : -1);
}

/*
 *	Unmounts the device.
 *
 *      Returns:
 *
 *      0       Success (or device is already unmounted).
 *      -1      General error
 *      -2      Bad or ambiguous value
 *      -3      Systems error or unable to execute command
 */
gint zt_unmount(EDVDevice *dev_ptr)
{
	gint pid;
	gchar		*cmd,
			*prog,
			*stderr_path;

	*last_error = '\0';			/* Reset last error message */

	if(dev_ptr == NULL)
		return(-2);
	if(STRISEMPTY(dev_ptr->device_path))
		return(-2);

	/* Get the ZipTool program */
	prog = zt_get_ziptool_program();
	if(prog == NULL)
		return(-1);

	/* Get the output file paths */
	stderr_path = edv_tmp_name(NULL);

	/* Format and execute unmount command */
	cmd = g_strdup_printf(
		"\"%s\" -u \"%s\" > \"%s\" 2> \"%s\"",
		prog,
		dev_ptr->device_path,
		PATH_DEV_NULL,
		stderr_path
	);
	pid = edv_system_wait(
		cmd,
		NULL
	);
	g_free(cmd);

	/* Get any error message */
	zt_open_file_to_string(
		stderr_path,
		last_error,
		sizeof(last_error)
	);

	/* Remove the output files */
	(void)edv_unlink(stderr_path);
	g_free(stderr_path);

	g_free(prog);

	return((pid > -1) ? 0 : -1);
}

/*
 *	Checks if the device is unlocked, locked, or locked with
 *	password.
 */
ZipToolLockState zt_device_is_protected(EDVDevice *dev_ptr)
{
	ZipToolLockState status = ZIP_TOOL_LOCK_STATE_UNLOCKED;
	gint pid;
	gchar		*cmd,
			*prog,
			*stdout_path,
			*stderr_path;

	*last_error = '\0';		/* Reset last error message */

	if(dev_ptr == NULL)
		return(status);
	if(STRISEMPTY(dev_ptr->device_path))
		return(status);

	/* Get the ZipTool program */
	prog = zt_get_ziptool_program();
	if(prog == NULL)
		return(status);

	/* Get the output file paths */
	stdout_path = edv_tmp_name(NULL);
	stderr_path = edv_tmp_name(NULL);

	/* Format and execute device status command */
	cmd = g_strdup_printf(
		"\"%s\" -s \"%s\" > \"%s\" 2> \"%s\"",
		prog,
		dev_ptr->device_path,
		stdout_path,
		stderr_path
	);
	pid = edv_system_wait(
		cmd,
		NULL
	);
	g_free(cmd);

	if(pid < 0)
	{
		/* Error */
		(void)edv_unlink(stdout_path);
		g_free(stdout_path);
		(void)edv_unlink(stderr_path);
		g_free(stderr_path);
		g_free(prog);
		return(status);
	}
	else
	{
		/* Success, the read first line of the output file */
		FILE *fp = fopen((const char *)stdout_path, "rb");
		if(fp != NULL)
		{
			gchar buf[1024];
			if(fgets(buf, sizeof(buf), fp) != NULL)
			{
				if(strstr(
					buf,
					"password write-protected"
				) != NULL)
					status = ZIP_TOOL_LOCK_STATE_LOCKED_PASSWORD;
				else if(strstr(
					buf,
					"write-protected"
				) != NULL)
					status = ZIP_TOOL_LOCK_STATE_LOCKED;
				else if(strstr(
					buf,
					"not protected"
				) != NULL)
					status = ZIP_TOOL_LOCK_STATE_UNLOCKED;

				/* All else assume not protected */
			}

			(void)fclose(fp);
		}
	}

	/* Get any error message */
	zt_open_file_to_string(
		stderr_path,
		last_error,
		sizeof(last_error)
	);

	/* Remove the output files */
	(void)edv_unlink(stdout_path);
	g_free(stdout_path);
	(void)edv_unlink(stderr_path);
	g_free(stderr_path);

	g_free(prog);

	return(status);
}


/*
 *	Locks the device (must be currently unlocked). If password
 *	is not NULL then the device will be locked with password.
 *
 *	Returns:
 *
 *	0	Success
 *	-1	General error
 *	-2	Bad or ambiguous value
 *	-3	Systems error or unable to execute command
 *	-4	Unable to lock
 *	-5	Device is already locked (with or without password),
 *		must unlock before locking again
 */
gint zt_lock(
	EDVDevice *dev_ptr, const gchar *password
)
{
	gchar *prog;

	*last_error = '\0';		/* Reset last error message */

	if(dev_ptr == NULL)
		return(-2);
	if(STRISEMPTY(dev_ptr->device_path))
		return(-2);

	/* Get the ZipTool program */
	prog = zt_get_ziptool_program();
	if(prog == NULL)
		return(-1);

	/* Device is already locked? */
	if(zt_device_is_protected(dev_ptr) != ZIP_TOOL_LOCK_STATE_UNLOCKED)
	{
		g_free(prog);
		return(-5);
	}

	/* Lock with password? */
	if(!STRISEMPTY(password))
	{
		/* Format and execute lock device with password command */
		gchar *cmd = g_strdup_printf(
			"%s -rp %s",
			prog,
			dev_ptr->device_path
		);
		FILE *fp = popen(cmd, "w");
		g_free(cmd);
		if(fp == NULL)
		{
			g_free(prog);
			return(-3);
		}

		/* Send password */
		fprintf(fp, "%s\n", password);

		pclose(fp);
	}
	else
	{
		/* Format and execute lock device command */
		gchar *cmd = g_strdup_printf(
			"%s -ro %s",
			prog,
			dev_ptr->device_path
		);
		FILE *fp = popen(cmd, "w");
		g_free(cmd);
		if(fp == NULL)
		{
			g_free(prog);
			return(-3);
		}

		/* Do not send any password */

		pclose(fp);
	}

	g_free(prog);

#if 0
	/* Device was not successfully locked? */
/* This does not work quite well, because the device is ejected after
 * locking and there is no way to see if it was locked correctly
 */
	if(zt_device_is_protected(dev_ptr) == ZIP_TOOL_LOCK_STATE_UNLOCKED)
		return(0);
	else
		return(-4);
#endif
	return(0);
}


/*
 *	Unlocks the device (if it is already locked) and uses the
 *	given password.
 *
 *	Returns:
 *
 *	0	Success (or device is already unlocked)
 *	-1	General error
 *	-2	Bad or ambiguous value
 *	-3	Systems error or unable to execute command
 *	-4	Bad password or unable to unlock
 */
gint zt_unlock(
	EDVDevice *dev_ptr, const gchar *password
)
{
	gchar		*cmd,
			*prog;
	FILE *fp;

	*last_error = '\0';		/* Reset last error message */

	if(dev_ptr == NULL)
		return(-2);
	if(STRISEMPTY(dev_ptr->device_path))
		return(-2);

	/* Get the ZipTool program */
	prog = zt_get_ziptool_program();
	if(prog == NULL)
		return(-1);

	/* Device is not locked? */
	if(zt_device_is_protected(dev_ptr) == ZIP_TOOL_LOCK_STATE_UNLOCKED)
	{
		g_free(prog);
		return(0);
	}

	if(password == NULL)
		password = "";

	/* Format and execute unlock device command */
	cmd = g_strdup_printf(
		"%s -rw %s",
		prog,
		dev_ptr->device_path
	);
	fp = popen(cmd, "w");
	g_free(cmd);
	if(fp == NULL)
	{
		g_free(prog);
		return(-3);
	}

	/* Send password */
	fprintf(fp, "%s\n", password);

	pclose(fp);

	g_free(prog);

	/* Device was not successfully unlocked? */
	return((zt_device_is_protected(dev_ptr) != ZIP_TOOL_LOCK_STATE_UNLOCKED) ?
		-4 : 0
	);
}

/*
 *	Spins down the device
 *
 *	Returns:
 *
 *	0	Success (or device already spinned down)
 *	-1	General error
 *	-2	Bad or ambiguous value
 *	-3	Systems error or unable to execute command
 */
gint zt_spin_down(EDVDevice *dev_ptr)
{
	gint pid;
	gchar		*cmd,
			*prog,
			*stderr_path;

	*last_error = '\0';			/* Reset last error message */

	if(dev_ptr == NULL)
		return(-2);
	if(STRISEMPTY(dev_ptr->device_path))
		return(-2);

	/* Get the ZipTool program */
	prog = zt_get_ziptool_program();
	if(prog == NULL)
		return(-1);

	/* Get the output file paths */
	stderr_path = edv_tmp_name(NULL);

	/* Format and execute spin down command */
	cmd = g_strdup_printf(
		"\"%s\" -p \"%s\" > /dev/null 2> \"%s\"",
		prog,
		dev_ptr->device_path,
		stderr_path
	);
	pid = edv_system_wait(
		cmd,
		NULL
	);
	g_free(cmd);

	/* Get any error message */
	zt_open_file_to_string(
		stderr_path,
		last_error,
		sizeof(last_error)
	);
	  
	/* Remove the output files */   
	(void)edv_unlink(stderr_path);
	g_free(stderr_path);

	g_free(prog);

	return((pid > -1) ? 0 : -1);
}

/*
 *	Ejects the media from the device
 *
 *      Returns:
 *
 *      0       Success (or media is already ejected)
 *      -1      General error
 *      -2      Bad or ambiguous value
 *      -3      Systems error or unable to execute command
 *      -4      Device is currently mounted
 */
gint zt_eject(EDVDevice *dev_ptr)
{
	gint pid;
	gchar		*cmd,
			*prog,
			*stderr_path;

	*last_error = '\0';			/* Reset last error message */

	if(dev_ptr == NULL)
		return(-2);
	if(STRISEMPTY(dev_ptr->device_path))
		return(-2);

	/* Get the ZipTool program */
	prog = zt_get_ziptool_program();
	if(prog == NULL)
		return(-1);

	/* Get the output file paths */
	stderr_path = edv_tmp_name(NULL);

	/* Format and execute eject command */
	cmd = g_strdup_printf(
		"\"%s\" -e \"%s\" > \"%s\" 2> \"%s\"",
		prog,
		dev_ptr->device_path,
		PATH_DEV_NULL,
		stderr_path
	);
	pid = edv_system_wait(
		cmd,
		NULL
	);
	g_free(cmd);

	/* Get any error message */
	zt_open_file_to_string(
		stderr_path,
		last_error,
		sizeof(last_error)
	);

	/* Remove the output files */   
	(void)edv_unlink(stderr_path);
	g_free(stderr_path);

	g_free(prog);

	return((pid > -1) ? 0 : -1);
}
