#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <unistd.h>

#include "../../include/prochandle.h"

#include "../guiutils.h"
#include "../cdialog.h"
#include "../pdialog.h"

#include "../libendeavour2-base/endeavour2.h"

#include "zip_tool_io.h"
#include "zip_tool.h"
#include "zip_tool_cb.h"
#include "config.h"


gint zt_display_button_event_cb(
	GtkWidget *widget, GdkEventButton *button, gpointer data
);

gint zt_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);

gint zt_menu_item_enter_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
gint zt_menu_item_leave_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);

void zt_opid_enter_cb(
	ToolBarItem *item, gint id, gpointer data
);
void zt_opid_leave_cb(
	ToolBarItem *item, gint id, gpointer data
);
void zt_opid_cb(
	ToolBarItem *item, gint id, gpointer data
);

void zt_mount_cb(GtkWidget *widget, gpointer data);
void zt_unmount_cb(GtkWidget *widget, gpointer data);
void zt_spin_down_cb(GtkWidget *widget, gpointer data);
void zt_eject_cb(GtkWidget *widget, gpointer data);
void zt_password_cb(GtkWidget *widget, gpointer data);
void zt_browse_cb(GtkWidget *widget, gpointer data);
void zt_refresh_cb(GtkWidget *widget, gpointer data);
void zt_fsck_cb(GtkWidget *widget, gpointer data);
void zt_close_cb(GtkWidget *widget, gpointer data);
void zt_exit_cb(GtkWidget *widget, gpointer data);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	"button_press_event" signal callback.
 */
gint zt_display_button_event_cb(
	GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	gint status = FALSE;
	ZipTool *zt = ZIP_TOOL(data);
	if((widget == NULL) || (button == NULL) || (zt == NULL))
		return(status);

	if(zt->freeze_count > 0)
		return(status);

	switch((gint)button->type)
	{
	  case GDK_BUTTON_PRESS:
		switch(button->button)
		{
		  case GDK_BUTTON3:			/* Map the right click GtkMenu  */
			gtk_menu_popup(
				GTK_MENU(zt->menu),
				NULL, NULL,
				NULL, NULL,
				button->button, button->time
			);
			status = TRUE;
			break;
		}
		break;
	}

	return(status);
}


/*
 *	Toplevel GtkWindow "delete_event" signal callback.
 */
gint zt_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	ZipTool *zt = ZIP_TOOL(data);
	if((widget == NULL) || (zt == NULL))
		return(FALSE);

	if(zt->freeze_count > 0)
		return(FALSE);

	zt_close_cb(NULL, zt);

	return(TRUE);
}


/*
 *	Menu item enter callback.
 */
gint zt_menu_item_enter_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	ZipTool *zt;
	ZipToolOPID *opid_ptr = ZIP_TOOL_OPID(data);
	if(opid_ptr == NULL)
		return(FALSE);

	zt = ZIP_TOOL(opid_ptr->zt);
	if(zt == NULL)
		return(FALSE);

	if(zt->freeze_count > 0)
		return(FALSE);

	zt_status_message(zt, opid_ptr->tooltip, FALSE);

	return(TRUE);
}

/*
 *	Menu item leave callback.
 */
gint zt_menu_item_leave_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	ZipTool *zt;
	ZipToolOPID *opid_ptr = ZIP_TOOL_OPID(data);
	if(opid_ptr == NULL)
		return(FALSE);

	zt = opid_ptr->zt;
	if(zt == NULL)
		return(FALSE);

	if(zt->freeze_count > 0)
		return(FALSE);

	zt_status_message(zt, NULL, FALSE);

	return(TRUE);
}

/*
 *	Operation ID enter callback.
 */
void zt_opid_enter_cb(
	ToolBarItem *item, gint id, gpointer data
)
{
	ZipToolOPID *opid_ptr = ZIP_TOOL_OPID(data);
	if((item == NULL) || (opid_ptr == NULL))
		return;

	if(opid_ptr->func_enter_cb != NULL)
		opid_ptr->func_enter_cb(
			ToolBarItemGetWidget(item),
			NULL,
			opid_ptr
		);
}

/*
 *      Operation ID leave callback.
 */
void zt_opid_leave_cb(
	ToolBarItem *item, gint id, gpointer data
)
{
	ZipToolOPID *opid_ptr = ZIP_TOOL_OPID(data);
	if((item == NULL) || (opid_ptr == NULL))
		return;

	if(opid_ptr->func_leave_cb != NULL)
		opid_ptr->func_leave_cb(
			ToolBarItemGetWidget(item),
			NULL,
			opid_ptr
		);
}

/*
 *	Operation id callback.
 */
void zt_opid_cb(
	ToolBarItem *item, gint id, gpointer data
)
{
	ZipTool *zt;
	ZipToolOPID *opid_ptr = ZIP_TOOL_OPID(data);
	if((item == NULL) || (opid_ptr == NULL))
		return;

	zt = opid_ptr->zt;
	if(zt == NULL)
		return;

	if(zt->freeze_count > 0)
		return;

	if(opid_ptr->func_cb != NULL)
		opid_ptr->func_cb(
			ToolBarItemGetWidget(item),	/* GtkWidget */
			zt				/* ZipTool */
		);
}


/*
 *	Mount callback.
 */
void zt_mount_cb(GtkWidget *widget, gpointer data)
{
	gboolean got_error = FALSE;
	gint status;
	GtkWidget *toplevel;
	EDVDevice *dev_ptr;
	ZipTool *zt = ZIP_TOOL(data);
	if(zt == NULL)
		return;

	toplevel = zt->toplevel;
	dev_ptr = zt->device;
	if(dev_ptr == NULL)
		return;

	zt_set_busy(zt, TRUE);

	zt_status_message(
		zt, "Mounting device...", TRUE
	);
	status = zt_mount(dev_ptr);
	if(status != 0)
	{
		const gchar *msg = zt_last_error();
		if(!STRISEMPTY(msg))
		{
			edv_play_sound_error(zt->ctx);
			CDialogSetTransientFor(toplevel);
			CDialogGetResponse(
"Error Mounting Device",
				msg,
"Please check to make sure that you have\n\
sufficient permission to run ziptool and access\n\
the device.  Also make sure that the device is\n\
properly defined from Endeavour Mark II.\n\
\n\
You should run Endeavour Mark II and configure the\n\
device references by going to Device->Devices...\n\
and then exit Endeavour Mark II to ensure that\n\
the changes have been saved. Afterwards run this\n\
program again.",
				CDIALOG_ICON_ERROR,
				CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
				CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);
		}
		got_error = TRUE;
	}

	zt_refresh_device(zt, dev_ptr);
	zt_update_display(zt);
	zt_status_message(
		zt,
		got_error ?
			"Error mounting device" : "Device mounted",
		FALSE
	);
	if(!got_error)
		edv_notify_queue_vfs_object_mounted(zt->ctx, dev_ptr->mount_path);
	edv_context_sync(zt->ctx);
	zt_set_busy(zt, FALSE);
}

/*
 *	Unmount callback.
 */
void zt_unmount_cb(GtkWidget *widget, gpointer data)
{
	gboolean got_error = FALSE;
	gint status;
	GtkWidget *toplevel;
	EDVDevice *dev_ptr;
	ZipTool *zt = ZIP_TOOL(data);
	if(zt == NULL)
		return;

	toplevel = zt->toplevel;
	dev_ptr = zt->device;
	if(dev_ptr == NULL)
		return;

	zt_set_busy(zt, TRUE);

	zt_status_message(
		zt, "Unmounting device...", TRUE
	);
	status = zt_unmount(dev_ptr);
	if(status != 0)
	{
		const gchar *buf = zt_last_error();
		if(!STRISEMPTY(buf))
		{
			edv_play_sound_error(zt->ctx);
			CDialogSetTransientFor(toplevel);
			CDialogGetResponse(
"Error Unmounting Device",
buf,
"Please check to make sure that you have\n\
sufficient permission to run ziptool and access\n\
the device.  Also make sure that the device is\n\
properly defined from Endeavour Mark II.\n\
\n\
You should run Endeavour Mark II and configure the\n\
device references by going to Device->Devices...\n\
and then exit Endeavour Mark II to ensure that\n\
the changes have been saved. Afterwards run this\n\
program again.",
				CDIALOG_ICON_ERROR,
				CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
				CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);
		}
		got_error = TRUE;
	}

	zt_refresh_device(zt, dev_ptr);
	zt_update_display(zt);
	zt_status_message(
		zt,
		got_error ?
			"Error unmounting device" : "Device unmounted",
		FALSE
	);
	if(!got_error)
		edv_notify_queue_vfs_object_unmounted(zt->ctx, dev_ptr->mount_path);
	edv_context_sync(zt->ctx);
	zt_set_busy(zt, FALSE);
}

/*
 *	Spin down callback.
 */
void zt_spin_down_cb(GtkWidget *widget, gpointer data)
{
	gboolean got_error = FALSE;
	gint status;
	GtkWidget *toplevel;
	EDVDevice *dev_ptr;
	ZipTool *zt = ZIP_TOOL(data);
	if(zt == NULL)
		return;

	toplevel = zt->toplevel;
	dev_ptr = zt->device;
	if(dev_ptr == NULL)
		return;

	zt_set_busy(zt, TRUE);

	zt_status_message(
		zt, "Spinning down device...", TRUE
	);
	status = zt_spin_down(dev_ptr);
	if(status != 0)
	{
		const gchar *buf = zt_last_error();
		if(!STRISEMPTY(buf))
		{
			edv_play_sound_error(zt->ctx);
			CDialogSetTransientFor(toplevel);
			CDialogGetResponse(
"Error Spinning Down Device",
buf,
"Please check to make sure that you have\n\
sufficient permission to run ziptool and access\n\
the device.  Also make sure that the device is\n\
properly defined from Endeavour Mark II.\n\
\n\
You should run Endeavour Mark II and configure the\n\
device references by going to Device->Devices...\n\
and then exit Endeavour Mark II to ensure that\n\
the changes have been saved. Afterwards run this\n\
program again.",
				CDIALOG_ICON_ERROR,
				CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
				CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);
		}
		got_error = TRUE;
	}

#if 0
/* Do not refresh after spinning down device, because otherwise it
 * would check the device and cause things to spin up again
 */
	zt_refresh_device(zt, dev_ptr);
#endif
	zt_update_display(zt);
	zt_status_message(
		zt,
		got_error ?
			"Error spinning down device" : "Device spinned down",
		FALSE
	);
	zt_set_busy(zt, FALSE);
}

/*
 *	Eject callback.
 */
void zt_eject_cb(GtkWidget *widget, gpointer data)
{
	gboolean got_error = FALSE;
	gint status;
	GtkWidget *toplevel;
	EDVDevice *dev_ptr;
	ZipTool *zt = ZIP_TOOL(data);
	if(zt == NULL)
		return;

	toplevel = zt->toplevel;
	dev_ptr = zt->device;
	if(dev_ptr == NULL)
		return;

	zt_set_busy(zt, TRUE);

	zt_status_message(
		zt, "Ejecting media...", TRUE
	);
	status = zt_eject(dev_ptr);
	if(status != 0)
	{
		const gchar *buf = zt_last_error();
		if(!STRISEMPTY(buf))
		{
			edv_play_sound_error(zt->ctx);
			CDialogSetTransientFor(toplevel);
			CDialogGetResponse(
"Error Ejecting Media",
buf,
"Please check to make sure that you have\n\
sufficient permission to run ziptool and access\n\
the device.  Also make sure that the device is\n\
properly defined from Endeavour Mark II.\n\
\n\
You should run Endeavour Mark II and configure the\n\
device references by going to Device->Devices...\n\
and then exit Endeavour Mark II to ensure that\n\
the changes have been saved. Afterwards run this\n\
program again.",
				CDIALOG_ICON_ERROR,
				CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
				CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);
		}
		got_error = TRUE;
	}

	zt_refresh_device(zt, dev_ptr);
	zt_update_display(zt);
	zt_status_message(
		zt,
		got_error ?
			"Error ejecting media" : "Media ejected",
		FALSE
	);
	if(!got_error)
		edv_notify_queue_vfs_object_unmounted(zt->ctx, dev_ptr->mount_path);
	edv_context_sync(zt->ctx);
	zt_set_busy(zt, FALSE);
}

/*
 *	Lock/unlock with password callback.
 */
void zt_password_cb(GtkWidget *widget, gpointer data)
{
	gint status;
	gchar **strv;
	gint strc;
	GtkWidget *toplevel;
	EDVDevice *dev;
	ZipToolLockState lock_state;
	ZipTool *zt = ZIP_TOOL(data);
	if((zt == NULL) || CDialogIsQuery() || PDialogIsQuery())
		return;

	toplevel = zt->toplevel;
	dev = zt->device;
	if(dev == NULL)
		return;

	if(EDV_DEVICE_IS_MOUNTED(dev))
	{
		edv_play_sound_warning(zt->ctx);
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
"Unable To Lock/Unlock Media",
"The device must be unmounted first, before\n\
locking or unlocking the media.",
			NULL,
			CDIALOG_ICON_WARNING,
			CDIALOG_BTNFLAG_OK,
			CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		return;
	}

	zt_set_busy(zt, TRUE);

	/* Get current lock state */
	lock_state = zt_device_is_protected(zt->device);
	switch(lock_state)
	{
	  case ZIP_TOOL_LOCK_STATE_LOCKED_PASSWORD:
		/* Already locked with password, prompt for unlock
		 * with password
		 */
		zt_status_message(
			zt, "Unlocking media with password...", TRUE
		);
		PDialogDeleteAllPrompts();
		PDialogAddPromptPassword(
			NULL, "Password:", NULL
		);
		PDialogSetSize(300, -1);
		PDialogSetTransientFor(toplevel);
		strv = PDialogGetResponse(
			"Unlock Media",
"A password is required to unlock this media,\n\
please enter the password below.",
			NULL,
			PDIALOG_ICON_SECURITY,
			"Unlock", "Cancel",
			PDIALOG_BTNFLAG_SUBMIT | PDIALOG_BTNFLAG_CANCEL,
			PDIALOG_BTNFLAG_SUBMIT,
			&strc
		);
		PDialogSetTransientFor(NULL);
		if((strv != NULL) && (strc >= 1))
		{
			const gchar *password = strv[0];
			status = zt_unlock(zt->device, password);
			if(status)
			{
				const gchar *buf =
"General error encountered while unlocking media";
				switch(status)
				{
				  case -4:
					buf =
"Bad password";
					break;

				  case -3:
					buf =
"Unable to execute unlock command";
					break;

				  case -2:
					buf =
"Invalid device reference value";
					break;
				}
				edv_play_sound_error(zt->ctx);
				CDialogSetTransientFor(toplevel);
				CDialogGetResponse(
					"Error Unlocking Media",
					buf,
"Please check to make sure that you have\n\
sufficient permission to run ziptool and access\n\
the device.  Also make sure that the device is\n\
properly defined from Endeavour Mark II.\n\
\n\
You should run Endeavour Mark II and configure the\n\
device references by going to Device->Devices...\n\
and then exit Endeavour Mark II to ensure that\n\
the changes have been saved. Afterwards run this\n\
program again.",
					CDIALOG_ICON_ERROR,
					CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
					CDIALOG_BTNFLAG_OK
				);
				CDialogSetTransientFor(NULL);
				zt_status_message(
					zt, "Failed to unlock media", FALSE
				);
			}
			else
			{
				zt_status_message(
					zt, "Unlocked media", FALSE
				);
			}
		}
		else
		{
			zt_status_message(
				zt, "Unlock media aborted", FALSE
			);
		}
		break;

	  case ZIP_TOOL_LOCK_STATE_LOCKED:
		/* Locked (without password), just unlock */
		zt_status_message(
			zt, "Unlocking media...", TRUE
		);
		status = zt_unlock(zt->device, NULL);
		if(status)
		{
			const gchar *buf =
"General error encountered while unlocking media";
			switch(status)
			{
			  case -4:
				buf =
"Bad password";
				break;

			  case -3:
				buf =
"Unable to execute unlock command";
				break;

			  case -2:
				buf =
"Invalid device reference value";
				break;
			}
			edv_play_sound_error(zt->ctx);
			CDialogSetTransientFor(toplevel);
			CDialogGetResponse(
				"Error Unlocking Media",
				buf,
"Please check to make sure that you have\n\
sufficient permission to run ziptool and access\n\
the device.  Also make sure that the device is\n\
properly defined from Endeavour Mark II.\n\
\n\
You should run Endeavour Mark II and configure the\n\
device references by going to Device->Devices...\n\
and then exit Endeavour Mark II to ensure that\n\
the changes have been saved. Afterwards run this\n\
program again.",
				CDIALOG_ICON_ERROR,
				CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
				CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);
			zt_status_message(
				zt, "Failed to unlock media", FALSE
			);
		}
		else
		{
			zt_status_message(
				zt, "Unlocked media", FALSE
			);
		}
		break;

	  case ZIP_TOOL_LOCK_STATE_UNLOCKED:
		/* Unlocked, prompt for lock with optional password */
		zt_status_message(
			zt, "Locking media...", TRUE
		);
		PDialogDeleteAllPrompts();
		PDialogAddPromptPassword(
			NULL, "Password:", NULL
		);
		PDialogAddPromptPassword(
			NULL, "Confirm:", NULL
		);
		PDialogSetSize(300, -1);
		PDialogSetTransientFor(toplevel);
		strv = PDialogGetResponse(
			"Lock Media",
"Enter a new password that will be used to\n\
lock this media, or leave it blank to set this\n\
media as read only",
			NULL,
			PDIALOG_ICON_SECURITY,
			"Lock", "Cancel",
			PDIALOG_BTNFLAG_SUBMIT | PDIALOG_BTNFLAG_CANCEL,
			PDIALOG_BTNFLAG_SUBMIT,
			&strc
		);
		PDialogSetTransientFor(NULL);
		if((strv != NULL) && (strc >= 2))
		{
			const gchar	*password = strv[0],
							*confirm = strv[1];
			if(!STRISEMPTY(password))
			{
				/* Password given, now make sure password matches
				 * with confirm
				 */
				if((confirm != NULL) ?
					!strcmp(password, confirm) : FALSE
				)
					status = zt_lock(zt->device, password);
				else
					status = -100;
			}
			else
			{
				/* No password, just lock media */
				status = zt_lock(zt->device, NULL);
			}
			if(status)
			{
				const gchar *buf =
"General error encountered while locking media";
				switch(status)
				{
				  case -100:
					buf =
"Passwords do not match";
					break;

				  case -4:
					buf =
"Unable to lock media";
					break;

				  case -3:
					buf =
"Unable to execute lock command";
					break;

				  case -2:
					buf =
"Invalid device reference value";
					break;
				}
				edv_play_sound_error(zt->ctx);
				CDialogSetTransientFor(toplevel);
				CDialogGetResponse(
					"Error Locking Media",
					buf,
"Please check to make sure that you have\n\
sufficient permission to run ziptool and access\n\
the device.  Also make sure that the device is\n\
properly defined from Endeavour Mark II.\n\
\n\
You should run Endeavour Mark II and configure the\n\
device references by going to Device->Devices...\n\
and then exit Endeavour Mark II to ensure that\n\
the changes have been saved. Afterwards run this\n\
program again.",
					CDIALOG_ICON_ERROR,
					CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
					CDIALOG_BTNFLAG_OK
				);
				CDialogSetTransientFor(NULL);
				zt_status_message(
					zt, "Failed to lock media", FALSE
				);
			}
			else
			{
				zt_status_message(
					zt, "Locked media", FALSE
				);
			}
		}
		else
		{
			zt_status_message(
				zt, "Lock media aborted", FALSE
			);
		}
		break;
	}

	zt_refresh_device(zt, zt->device);
	zt_update_display(zt);
	zt_set_busy(zt, FALSE);
}

/*
 *	Run file browser to browse media callback.
 */
void zt_browse_cb(GtkWidget *widget, gpointer data)
{
	EDVDevice *dev_ptr;
	ZipTool *zt = ZIP_TOOL(data);
	if(zt == NULL)
		return;

	dev_ptr = zt->device;
	if(dev_ptr == NULL)
		return;

	zt_set_busy(zt, TRUE);
	edv_window_vfs_browser_new(
		zt->ctx,
		dev_ptr->mount_path,		/* Path */
		dev_ptr->mount_path		/* Tree origin */
	);
	zt_set_busy(zt, FALSE);
}

/*
 *	Refresh calback.
 */
void zt_refresh_cb(GtkWidget *widget, gpointer data)
{
	ZipTool *zt = ZIP_TOOL(data);
	if(zt == NULL)
		return;

	zt_set_busy(zt, TRUE);
	zt_status_message(
		zt, "Refreshing device information...", TRUE
	);
	zt_refresh_device(zt, zt->device);
	zt_update_display(zt);
	zt_status_message(
		zt, "Refreshed device information", FALSE
	);
	zt_set_busy(zt, FALSE);
}

/*
 *	FSCK callback.
 */
void zt_fsck_cb(GtkWidget *widget, gpointer data)
{
	const gchar *cmd;
	GtkWidget *toplevel;
	ZipTool *zt = ZIP_TOOL(data);
	if(zt == NULL)
		return;

	toplevel = zt->toplevel;

	zt_set_busy(zt, TRUE);

	cmd = (zt->device != NULL) ? zt->device->command_check : NULL;
	if((cmd != NULL) ? (*cmd != '\0') : FALSE)
	{
		Exec(cmd);
	}
	else
	{
		edv_play_sound_warning(zt->ctx);
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
			"FSCK Program Undefined",
"There is no defined command to run the FSCK program\n\
for this device.",
"To define a command to run the FSCK program for this\n\
device, run Endeavour Mark II and configure the\n\
device references by going to Device->Devices...\n\
and define the Check Command to run the FSCK program.\n\
Then exit Endeavour Mark II to ensure that\n\
the changes have been saved. Afterwards run this\n\
program again.",
			CDIALOG_ICON_WARNING,
			CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
			CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
	}

	zt_set_busy(zt, FALSE);
}

/*
 *	Close callback.
 */
void zt_close_cb(GtkWidget *widget, gpointer data)
{
	ZipTool *zt = ZIP_TOOL(data);
	if(zt == NULL)
		return;

	if(zt->freeze_count > 0)
		return;

	zt_unmap(zt);
}

/*
 *	Exit callback.
 */
void zt_exit_cb(GtkWidget *widget, gpointer data)
{
	ZipTool *zt = ZIP_TOOL(data);
	if(zt == NULL)
		return;
/* Need to work on this */
	zt_unmap(zt);
}

