#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../guiutils.h"
#include "../pie_chart.h"
#include "../tool_bar.h"

#include "../libendeavour2-base/endeavour2.h"

#include "zip_tool_io.h"
#include "zip_tool.h"
#include "zip_tool_cb.h"

#include "config.h"

#include "../images/icon_drive_zip_48x48.xpm"
#include "../images/icon_mount_20x20.xpm"
#include "../images/icon_unmount_20x20.xpm"
#include "../images/icon_mounted_20x20.xpm"
#include "../images/icon_not_mounted_20x20.xpm"
#include "../images/icon_eject_20x20.xpm"
#include "../images/icon_power_20x20.xpm"
#include "../images/icon_security_20x20.xpm"
#include "../images/icon_folder_opened_20x20.xpm"
#include "../images/icon_reload_20x20.xpm"
#include "../images/icon_fsck_20x20.xpm"
#include "../images/icon_exit_20x20.xpm"
#include "../images/icon_close_20x20.xpm"
#include "../images/icon_secure_20x20.xpm"
#include "../images/icon_insecure_20x20.xpm"


/* OPID */
ZipToolOPID *zt_opid_new(void);
void zt_opid_delete(ZipToolOPID *opid);

/* OPIDs List */
ZipToolOPID *zt_opids_list_find(
	GList *opids_list,
	const gint id
);
GList *zt_opids_list_new(ZipTool *zt);
void zt_opids_list_delete(GList *opids_list);

/* Set Device */
void zt_set_device(
	ZipTool *zt,
	EDVDevice *dev
);

/* Refresh Device */
void zt_refresh_device(
	ZipTool *zt,
	EDVDevice *dev
);

/* ZipTool */
ZipTool *zt_new(EDVContext *ctx);
void zt_update_display(ZipTool *zt);
void zt_set_busy(
	ZipTool *zt,
	const gboolean busy
);
void zt_status_message(
	ZipTool *zt,
	const gchar *msg,
	const gboolean allow_gtk_iteration

);
gboolean zt_is_mapped(ZipTool *zt);
void zt_map(ZipTool *zt);
void zt_unmap(ZipTool *zt);
void zt_delete(ZipTool *zt);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


#define ZIP_TOOL_WIDTH			550
#define ZIP_TOOL_HEIGHT			350

#define EDV_STATUS_BAR_HEIGHT		26


/*
 *	Creates a new OPID.
 */
ZipToolOPID *zt_opid_new(void)
{
	return(ZIP_TOOL_OPID(g_malloc0(sizeof(ZipToolOPID))));
}

/*
 *	Deletes the OPID.
 */
void zt_opid_delete(ZipToolOPID *opid)
{
        if(opid == NULL)
            return;

        g_free(opid->name);
        g_free(opid->button_name);
        g_free(opid->tooltip);

        g_free(opid);
}


/*
 *	Gets the OPID by ID.
 *
 *	The opids_list specifies a GList of ZipToolOPID *
 *	OPIDs.
 *
 *	The id specifies the ID.
 *
 *	Returns the OPID who's ID matches the specified ID or NULL on
 *	error.
 */
ZipToolOPID *zt_opids_list_find(
	GList *opids_list,
	const gint id
)
{
	GList *glist;
	ZipToolOPID *opid;

	for(glist = opids_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    opid = ZIP_TOOL_OPID(glist->data);
	    if(opid == NULL)
		continue;

	    if(opid->id == id)
		return(opid);
	}

	return(NULL);
}


/*
 *	Creates a new list of Operation IDs.
 */
GList *zt_opids_list_new(ZipTool *zt)
{
	GList *opid = NULL;
	const gchar *name;
	const gchar *button_name;
	guint8 **icon_data;
	const gchar *tooltip;
	void (*func_cb)(GtkWidget *, gpointer);
	gint (*enter_notify_cb)(GtkWidget *, GdkEventCrossing *, gpointer);
	gint (*leave_notify_cb)(GtkWidget *, GdkEventCrossing *, gpointer);

#define APPEND_OPID(_id_)	{				\
 ZipToolOPID *opid_ptr = ZIP_TOOL_OPID(	\
  g_malloc0(sizeof(ZipToolOPID))		\
 );							\
 opid_ptr->id = (_id_);					\
 opid_ptr->zt = zt;					\
 opid_ptr->name = STRDUP(name);				\
 opid_ptr->button_name = STRDUP(button_name);		\
 opid_ptr->icon_data = icon_data;			\
 opid_ptr->tooltip = STRDUP(tooltip);			\
 opid_ptr->func_cb = func_cb;				\
 opid_ptr->func_enter_cb = enter_notify_cb;		\
 opid_ptr->func_leave_cb = leave_notify_cb;		\
							\
 opid = g_list_append(opid, opid_ptr);			\
}

	enter_notify_cb = zt_menu_item_enter_cb;
	leave_notify_cb = zt_menu_item_leave_cb;

	func_cb = zt_close_cb;
	icon_data = (guint8 **)icon_close_20x20_xpm;
	name = "Close";
	button_name = name;
	tooltip = "Close this window";
	APPEND_OPID(ZIP_TOOL_OPID_CLOSE);

	func_cb = zt_exit_cb;
	icon_data = (guint8 **)icon_exit_20x20_xpm;
	name = "Exit";
	button_name = name;
	tooltip = "Close all windows belonging to this application";
	APPEND_OPID(ZIP_TOOL_OPID_EXIT);

	func_cb = zt_mount_cb;
	icon_data = (guint8 **)icon_mount_20x20_xpm;
	name = "Mount";
	button_name = name;
	tooltip = "Mount the device";
	APPEND_OPID(ZIP_TOOL_OPID_MOUNT);

	func_cb = zt_unmount_cb;
	icon_data = (guint8 **)icon_unmount_20x20_xpm;
	name = "Unmount";
	button_name = name;
	tooltip = "Unmount the device";
	APPEND_OPID(ZIP_TOOL_OPID_UNMOUNT);

	func_cb = zt_spin_down_cb;
	icon_data = (guint8 **)icon_power_20x20_xpm;
	name = "Spin Down";
	button_name = "SpinDn";
	tooltip = "Spin down the device";
	APPEND_OPID(ZIP_TOOL_OPID_SPIN_DOWN);

	func_cb = zt_eject_cb;
	icon_data = (guint8 **)icon_eject_20x20_xpm;
	name = "Eject";
	button_name = name;
	tooltip = "Eject the media from the device";
	APPEND_OPID(ZIP_TOOL_OPID_EJECT);

	func_cb = zt_password_cb;
	icon_data = (guint8 **)icon_security_20x20_xpm;
	name = "Password";
	button_name = name;
	tooltip = "Lock/unlock the media";
	APPEND_OPID(ZIP_TOOL_OPID_PASSWORD);

	func_cb = zt_browse_cb;
	icon_data = (guint8 **)icon_folder_opened_20x20_xpm;
	name = "Browse";
	button_name = name;
	tooltip = "Run the File Browser to browse the media";
	APPEND_OPID(ZIP_TOOL_OPID_BROWSE);

	func_cb = zt_refresh_cb;
	icon_data = (guint8 **)icon_reload_20x20_xpm;
	name = "Refresh";
	button_name = name;
	tooltip = "Refresh device information";
	APPEND_OPID(ZIP_TOOL_OPID_REFRESH);

	func_cb = zt_fsck_cb;
	icon_data = (guint8 **)icon_fsck_20x20_xpm;
	name = "File System Check";
	button_name = "FSCK";
	tooltip = "Run FSCK to check the media";
	APPEND_OPID(ZIP_TOOL_OPID_FSCK);

#undef APPEND_OPID

	return(opid);
}

/*
 *	Deletes the list of Operation IDs.
 */
void zt_opids_list_delete(GList *opids_list)
{
	if(opids_list != NULL)
	{
	    g_list_foreach(
		opids_list,
		(GFunc)zt_opid_delete,
		NULL
	    );
	    g_list_free(opids_list);
	}
}


/*
 *	Sets the new device specified by dev on the zip tools
 *	window. If a device already exists then it will be deallocated
 *	first.
 *
 *	The given device will be taken by this function and should no
 *	longer be referenced by the calling function.
 */
void zt_set_device(
	ZipTool *zt,
	EDVDevice *dev
)
{
	GtkWidget *w;

	if(zt == NULL)
	    return;

	/* Delete existing device reference (if any) and set new
	 * device reference
	 */
	edv_device_delete(zt->device);
	zt->device = dev;

	/* Update title */
	w = zt->toplevel;
	if(w != NULL)
	{
	    if(dev != NULL)
	    {
		gchar *msg;

		if((dev->name != NULL) && (dev->device_path != NULL))
		    msg = g_strdup_printf(
			"%s: %s (%s)",
			PROG_NAME, dev->name, dev->device_path
		    );
		else
		    msg = g_strdup_printf(
			"%s: %s",
			PROG_NAME,
			(dev->name != NULL) ? dev->name :
			    dev->device_path
		    );
		gtk_window_set_title(GTK_WINDOW(w), msg);
		g_free(msg);
	    }
	    else
	    {
		gtk_window_set_title(GTK_WINDOW(w), PROG_NAME);
	    }
	}
}

/*
 *	Rescans the specified device (or the current device on the Zip
 *	Tool Window if dev is NULL) and updates the display widgets.
 */
void zt_refresh_device(
	ZipTool *zt, EDVDevice *dev
)
{
	const gint	border_major = 5,
			border_minor = 2;
	GtkWidget *w, *parent, *parent2, *parent3, *parent4;

	if(zt == NULL)
	    return;

	if(dev == NULL)
	    dev = zt->device;
	if(dev == NULL)
	    return;

	/* Update the Device's mount state and stats */
	edv_device_update_mount_state(dev);
	edv_device_update_stats(dev);

	/* Get lock state */
	zt->lock_state = zt_device_is_protected(dev);


	/* Destroy existing display widgets */
	GTK_WIDGET_DESTROY(zt->capacity_pie_chart);
	zt->capacity_pie_chart = NULL;
	GTK_WIDGET_DESTROY(zt->display_mount_path_btn);
	zt->display_mount_path_btn = NULL;
 	GTK_WIDGET_DESTROY(zt->display_mount_btn);
	zt->display_mount_btn = NULL;
	GTK_WIDGET_DESTROY(zt->display_lock_btn);
	zt->display_lock_btn = NULL;
	GTK_WIDGET_DESTROY(zt->display_child);
	zt->display_child = NULL;


	/* Begin creating the display widgets */

	/* Get the display parent and begin recreating display widgets */
	parent = zt->display_parent;
	if(parent != NULL)
	{
	    struct stat stat_buf;
	    gint prefix_label_width, stat_status, font_size;
	    gulong adj_division = 10;
	    const gchar *font_encoding;
	    gchar	*msg, *msg2,
			*font_name_h1_bold,
			*font_name_h1,
			*font_name_h2_bold,
			*font_name_h2;
	    GdkFont *font;
	    GdkColor c;
	    GdkPixmap *pixmap = NULL;
	    GdkBitmap *mask = NULL;
	    GtkAdjustment *adj;
	    GtkRcStyle *rcstyle;
	    GtkStyle *style = gtk_widget_get_style(parent);

#define SET_FLAT_BUTTON_STYLE(_w_)      {               \
 if((_w_) != NULL) {                                    \
  GtkStateType state, state2 = GTK_STATE_NORMAL;        \
  GtkStyle *style = gtk_widget_get_style(_w_);          \
  GtkRcStyle *rcstyle = gtk_rc_style_new();		\
  state = GTK_STATE_NORMAL;                             \
  rcstyle->color_flags[state] = GTK_RC_FG | GTK_RC_BG | \
   GTK_RC_TEXT;                                         \
  memcpy(&rcstyle->fg[state], &style->fg[state2], sizeof(GdkColor));\
  memcpy(&rcstyle->bg[state], &style->bg[state2], sizeof(GdkColor));\
  GDK_COLOR_SET_COEFF(&rcstyle->text[state], 0.0f, 0.0f, 1.0f)\
  state = GTK_STATE_PRELIGHT;                           \
  rcstyle->color_flags[state] = GTK_RC_FG | GTK_RC_BG | \
   GTK_RC_TEXT;                                         \
  memcpy(&rcstyle->fg[state], &style->fg[state2], sizeof(GdkColor));\
  memcpy(&rcstyle->bg[state], &style->bg[state2], sizeof(GdkColor));\
  GDK_COLOR_SET_COEFF(&rcstyle->text[state], 0.0f, 0.0f, 1.0f)\
  state = GTK_STATE_ACTIVE;                             \
  rcstyle->color_flags[state] = GTK_RC_FG | GTK_RC_BG | \
   GTK_RC_TEXT;                                         \
  memcpy(&rcstyle->fg[state], &style->fg[state2], sizeof(GdkColor));\
  memcpy(&rcstyle->bg[state], &style->bg[state2], sizeof(GdkColor));\
  GDK_COLOR_SET_COEFF(&rcstyle->text[state], 1.0f, 0.0f, 0.0f)\
							\
  gtk_widget_modify_style_recursive((_w_), rcstyle);    \
  GTK_RC_STYLE_UNREF(rcstyle);                          \
							\
  gtk_button_set_relief(GTK_BUTTON(_w_), GTK_RELIEF_NONE);\
} }

#define SET_BUTTON_CB(_w_,_id_)         {               \
 ZipToolOPID *opid_ptr = zt_opids_list_find(  \
  zt->opid, (_id_)                                      \
 );                                                     \
 if(((_w_) != NULL) && (opid_ptr != NULL)) {            \
  gtk_signal_connect(                                   \
   GTK_OBJECT(_w_), "clicked",                          \
   GTK_SIGNAL_FUNC(opid_ptr->func_cb), opid_ptr->zt     \
  );                                                    \
  gtk_signal_connect(                                   \
   GTK_OBJECT(_w_), "enter_notify_event",               \
   GTK_SIGNAL_FUNC(opid_ptr->func_enter_cb), opid_ptr   \
  );                                                    \
  gtk_signal_connect(                                   \
   GTK_OBJECT(_w_), "leave_notify_event",               \
   GTK_SIGNAL_FUNC(opid_ptr->func_leave_cb), opid_ptr   \
  );                                                    \
  GUISetWidgetTip((_w_), opid_ptr->tooltip);            \
} }

	    /* Get the base font size */
	    font = (style != NULL) ? style->font : NULL;
	    font_size = GDK_FONT_GET_FONT_NAME_SIZE(font);
	    if(font_size < 3)
		font_size = 3;

	    /* Format the font names */
#if defined(PROG_LANGUAGE_POLISH)
	    font_encoding = "iso8859-2";
#else
	    font_encoding = "iso8859-1";
#endif

	    font_name_h1_bold = g_strdup_printf(
"-adobe-helvetica-bold-r-normal-*-%i-*-*-*-*-*-%s",
		font_size + 2,
		font_encoding
	    );
	    font_name_h1 = g_strdup_printf(
"-adobe-helvetica-medium-r-normal-*-%i-*-*-*-*-*-%s",
		font_size + 2,
		font_encoding
	    );
	    font_name_h2_bold = g_strdup_printf(
"-adobe-helvetica-bold-r-normal-*-%i-*-*-*-*-*-%s",
		font_size,
		font_encoding
	    );
	    font_name_h2 = g_strdup_printf(
"-adobe-helvetica-medium-r-normal-*-%i-*-*-*-*-*-%s",
		font_size,
		font_encoding
	    );

	    /* Get the mount path's statistics */
	    stat_status = !STRISEMPTY(dev->mount_path) ?
		(gint)stat((const char *)dev->mount_path, &stat_buf) : -1;

	    /* GtkHBox for multiple columns */
	    zt->display_child = w = gtk_hbox_new(TRUE, 0);
	    gtk_container_border_width(GTK_CONTAINER(w), border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	    gtk_widget_show(w);
	    parent = w;

	    /* Left side column GtkVBox */
	    w = gtk_vbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    /* GtkHBox the for device icon and label */
	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;
 
	    /* Get the icon to display the device */
	    if(EDV_DEVICE_IS_MOUNTED(dev) &&
	       (EDV_DEVICE_TOTAL_ICON_STATES > EDV_DEVICE_ICON_STATE_STANDARD)
	    )
	    {
		EDVDeviceIconState i = EDV_DEVICE_ICON_STATE_STANDARD;
		pixmap = GDK_PIXMAP_NEW_FROM_XPM_FILE(
		    &mask, dev->large_icon_file[i]
		);
		if(pixmap == NULL)
		    pixmap = GDK_PIXMAP_NEW_FROM_XPM_FILE(
			&mask, dev->medium_icon_file[i]
		    );
		if(pixmap == NULL)
		    pixmap = GDK_PIXMAP_NEW_FROM_XPM_FILE(
			&mask, dev->small_icon_file[i]
		    );
	    }
	    else if(EDV_DEVICE_TOTAL_ICON_STATES > EDV_DEVICE_ICON_STATE_UNMOUNTED)
	    {
		EDVDeviceIconState i = EDV_DEVICE_ICON_STATE_UNMOUNTED;
		pixmap = GDK_PIXMAP_NEW_FROM_XPM_FILE(
		    &mask, dev->large_icon_file[i]
		);
		if(pixmap == NULL)
		    pixmap = GDK_PIXMAP_NEW_FROM_XPM_FILE(
			&mask, dev->medium_icon_file[i]
		    );
		if(pixmap == NULL)
		    pixmap = GDK_PIXMAP_NEW_FROM_XPM_FILE(
			&mask, dev->small_icon_file[i]
		    );
		i = EDV_DEVICE_ICON_STATE_STANDARD;
		if(pixmap == NULL)
		    pixmap = GDK_PIXMAP_NEW_FROM_XPM_FILE(
			&mask, dev->large_icon_file[i]
		    );
		if(pixmap == NULL)
		    pixmap = GDK_PIXMAP_NEW_FROM_XPM_FILE(
			&mask, dev->medium_icon_file[i]
		    );
		if(pixmap == NULL)
		    pixmap = GDK_PIXMAP_NEW_FROM_XPM_FILE(
			&mask, dev->small_icon_file[i]
		    );
	    }
	    if(pixmap != NULL)
	    {
		w = gtk_pixmap_new(pixmap, mask);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
	    }
	    /* GtkVBox for the name and location labels */
	    w = gtk_vbox_new(TRUE, 0);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;
	    /* Device Name GtkLabel */
	    rcstyle = gtk_rc_style_new();
	    rcstyle->font_name = STRDUP(font_name_h1_bold);
	    if(!STRISEMPTY(dev->name))
		msg = STRDUP(dev->name);
	    else
		msg = STRDUP("Untitled");
	    w = gtk_label_new(msg);
	    g_free(msg);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_modify_style(w, rcstyle);                     
	    gtk_widget_show(w);
	    GTK_RC_STYLE_UNREF(rcstyle);

	    /* Device Path GtkLabel */
	    if(!STRISEMPTY(dev->device_path))
		msg = edv_path_shorten(dev->device_path, 35);
	    else
		msg = STRDUP("*path not set*");
	    w = gtk_label_new(msg);
	    g_free(msg);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);

	    /* Mount Path GtkButton */
	    if(!STRISEMPTY(dev->mount_path))
	    {
		/* GtkHBox */
		w = gtk_hbox_new(TRUE, 0); 
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent3 = w;
		/* GtkButton */
		msg2 = edv_path_shorten(dev->mount_path, 35);
		msg = g_strdup_printf(
"Mount Path: %s",
		    msg2
		);
		zt->display_mount_path_btn = w = GUIButtonPixmapLabelH(
		    (guint8 **)icon_folder_opened_20x20_xpm,
		    msg,
		    NULL
		);
		g_free(msg);
		g_free(msg2);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		SET_FLAT_BUTTON_STYLE(w);
		SET_BUTTON_CB(w, ZIP_TOOL_OPID_BROWSE);
		gtk_widget_show(w);
	    }

	    /* Usage Pie Chart */
	    if(EDV_DEVICE_IS_MOUNTED(dev))
	    {
		gchar *ss_kb, *ss_mb;

		adj = (GtkAdjustment *)gtk_adjustment_new(
		    0.0f, 0.0f,
		    (gfloat)(dev->blocks_total / adj_division),
		    0.0f, 0.0f, 0.0f
		);
		GDK_COLOR_SET_COEFF(&c, 0.0f, 0.0f, 1.0f)
		ss_kb = STRDUP(
		    edv_str_size_delim(dev->blocks_total)
		);
		ss_mb = STRDUP(
		    edv_str_size_delim(dev->blocks_total / 1000)
		);
		msg = g_strdup_printf(
		    "%s: %s kb (%s mb)",
#ifdef PROG_LANGUAGE_ENGLISH
"Total Capacity"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"La Capacidad Total"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Capacit Totale"
#endif
		    , ss_kb, ss_mb
		);
		g_free(ss_kb);
		g_free(ss_mb);

		ss_kb = STRDUP(
		    edv_str_size_delim(dev->blocks_free)
		);
		ss_mb = STRDUP(
		    edv_str_size_delim(dev->blocks_free / 1000)
		);
		msg2 = g_strdup_printf(
		    "%s kb (%s mb)",
		    ss_kb, ss_mb
		);
		g_free(ss_kb);
		g_free(ss_mb);

		zt->capacity_pie_chart = w = PieChartNew(
		    PIE_CHART_DOUBLE_BUFFER |
			PIE_CHART_SHOW_LABELS |
			PIE_CHART_SHOW_SHADOW |
			PIE_CHART_SHOW_OUTLINE,
		    110, 70,
		    adj,
		    &c,
		    NULL, msg,
#ifdef PROG_LANGUAGE_ENGLISH
"Free"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Libre"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Libre"
#endif
		    ":", msg2
		);
		adj = (GtkAdjustment *)GTK_OBJECT_UNREF(GTK_OBJECT(adj));
		g_free(msg);
		g_free(msg2);
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_widget_show(w);

		GDK_COLOR_SET_COEFF(&c, 0.0f, 1.0f, 1.0f);
		adj = (GtkAdjustment *)gtk_adjustment_new(
		    0.0f, 0.0f,
		    (gfloat)(dev->blocks_available / adj_division),
		    0.0f, 0.0f, 0.0f
		);
		ss_kb = STRDUP(
		    edv_str_size_delim(dev->blocks_available)
		);
		ss_mb = STRDUP(
		    edv_str_size_delim(dev->blocks_available / 1000)
		);
		msg = g_strdup_printf(
		    "%s kb (%s mb)",
		    ss_kb, ss_mb
		);
		g_free(ss_kb);
		g_free(ss_mb);
		PieChartValueAdd(
		    w,
		    adj,
		    &c,
#ifdef PROG_LANGUAGE_ENGLISH
"Free & Available"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Libre & Disponible"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Libre & Disponible"
#endif
		    ":", msg
		);
		adj = (GtkAdjustment *)GTK_OBJECT_UNREF(GTK_OBJECT(adj));
		g_free(msg);

		GDK_COLOR_SET_COEFF(&c, 1.0f, 0.0f, 1.0f);
		adj = (GtkAdjustment *)gtk_adjustment_new(
		    0.0, 0.0,
		    (gfloat)(
			(dev->blocks_total - dev->blocks_free) /
			    adj_division
		    ),
		    0.0f, 0.0f, 0.0f
		);
		ss_kb = STRDUP(
		    edv_str_size_delim(
			dev->blocks_total - dev->blocks_free
		    )
		);
		ss_mb = STRDUP(
		    edv_str_size_delim(
			(dev->blocks_total - dev->blocks_free) / 1000
		    )
		);
		msg = g_strdup_printf(
		    "%s kb (%s mb)",
		    ss_kb, ss_mb
		);
		g_free(ss_kb);                      
		g_free(ss_mb); 
		PieChartValueAdd(
		    w,
		    adj,
		    &c,
#ifdef PROG_LANGUAGE_ENGLISH
"Used"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Usado"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Utilis"
#endif
		    ":", msg
		);
		adj = (GtkAdjustment *)GTK_OBJECT_UNREF(GTK_OBJECT(adj));
		g_free(msg);
	    }

	    /* Right side column GtkVBox */
	    w = gtk_vbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

#define ADD_LABEL(_pfx_,_val_)		{		\
 w = gtk_hbox_new(FALSE, border_major);			\
 gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);\
 gtk_widget_show(w);					\
 parent3 = w;						\
 w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);		\
 gtk_widget_set_usize(w, prefix_label_width, -1);	\
 gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);\
 gtk_widget_show(w);					\
 parent4 = w;						\
 w = gtk_label_new(_pfx_);				\
 gtk_container_add(GTK_CONTAINER(parent4), w);		\
 gtk_widget_show(w);					\
 w = gtk_label_new(_val_);				\
 gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);\
 gtk_widget_show(w);					\
}

	    /* Status Banner */
	    font = gdk_font_load(font_name_h1_bold);
	    if(font == NULL)
		font = GDK_FONT_REF(style->font);
	    w = GUIBannerCreate(
		"Status",
		font,
                &style->bg[GTK_STATE_NORMAL],
                &style->text[GTK_STATE_NORMAL],
                &style->fg[GTK_STATE_NORMAL],
		GTK_JUSTIFY_CENTER,
		FALSE,				/* Do not expand */
                border_minor,			/* Inside border */
                border_minor			/* Outside border */
	    );
	    font = GDK_FONT_UNREF(font);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);

	    /* GtkHBox for the mount or unmount button */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;
	    /* Mount/Unmount GtkButton */
	    if(EDV_DEVICE_IS_MOUNTED(dev))
	    {
		msg = g_strdup_printf(
		    "Mounted (%s)",
		    EDV_DEVICE_IS_READ_ONLY(dev) ? "Read Only" : "Read/Write"
		);
		zt->display_mount_btn = w = GUIButtonPixmapLabelH(
		    (guint8 **)icon_mounted_20x20_xpm,
		    msg,
		    NULL
		);
		g_free(msg);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		SET_FLAT_BUTTON_STYLE(w)
		SET_BUTTON_CB(w, ZIP_TOOL_OPID_UNMOUNT)
		gtk_widget_show(w);
	    }
	    else
	    {
		zt->display_mount_btn = w = GUIButtonPixmapLabelH(
		    (guint8 **)icon_not_mounted_20x20_xpm,
		    "Not Mounted",
		    NULL
		);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		SET_FLAT_BUTTON_STYLE(w)                                  
		SET_BUTTON_CB(w, ZIP_TOOL_OPID_MOUNT)
		gtk_widget_show(w);
	    }

	    /* GtkHBox for the locked/unlocked button */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;
	    /* Locked icon and label */
	    if(zt->lock_state != ZIP_TOOL_LOCK_STATE_UNLOCKED)
		w = GUIButtonPixmapLabelH(
		    (guint8 **)icon_secure_20x20_xpm,
		    (zt->lock_state == ZIP_TOOL_LOCK_STATE_LOCKED_PASSWORD) ?
			"Locked With Password" : "Write Protected",
		    NULL
		);
	    else
		w = GUIButtonPixmapLabelH(
		    (guint8 **)icon_insecure_20x20_xpm,
		    "Not Protected",
		    NULL
		);
	    zt->display_lock_btn = w;
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    SET_FLAT_BUTTON_STYLE(w)
	    SET_BUTTON_CB(w, ZIP_TOOL_OPID_PASSWORD)
	    gtk_widget_show(w);

	    /* Ownership widgets (only if mounted) */
	    if(EDV_DEVICE_IS_MOUNTED(dev))
	    {
		prefix_label_width = 70;

		/* Ownership Banner */
		font = gdk_font_load(font_name_h1_bold);
		if(font == NULL)
		    font = GDK_FONT_REF(style->font);
		w = GUIBannerCreate(
		    "Ownership",
		    font,
                    &style->bg[GTK_STATE_NORMAL],
                    &style->text[GTK_STATE_NORMAL],
                    &style->fg[GTK_STATE_NORMAL],
		    GTK_JUSTIFY_CENTER,
		    FALSE,			/* Do not expand */
                    border_minor,		/* Inside border */
                    border_minor		/* Outside border */
		);
		font = GDK_FONT_UNREF(font);
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_widget_show(w);

		/* Owner GtkLabel */
		if(stat_status)
		    msg = STRDUP("");
		else
		    msg = edv_uid_uid_to_name(
			zt->ctx,
			(gint)stat_buf.st_uid
		    );
		ADD_LABEL("Owner:", msg);
		g_free(msg);

		/* Group GtkLabel */
		if(stat_status)
		    msg = STRDUP("");
		else
		    msg = edv_gid_gid_to_name(
			zt->ctx,
			(gint)stat_buf.st_gid
		    );
		ADD_LABEL("Group:", msg);
		g_free(msg);

		/* Permissions Label */
		if(stat_status)
		{
		    msg = STRDUP("");
		}
		else
		{
		    msg = edv_str_permissions(
			edv_stat_mode_to_edv_permissions(stat_buf.st_mode)
		    );
		}
		ADD_LABEL("Permissions:", msg)
		g_free(msg);
	    }

	    /* Details widgets (only if mounted) */
	    if(EDV_DEVICE_IS_MOUNTED(dev))
	    {
		prefix_label_width = 90;

		/* Details Banner */
		font = gdk_font_load(font_name_h1_bold);
		if(font == NULL)
		    font = GDK_FONT_REF(style->font);
		w = GUIBannerCreate(
		    "Details",
		    font,
                    &style->bg[GTK_STATE_NORMAL],
                    &style->text[GTK_STATE_NORMAL],
                    &style->fg[GTK_STATE_NORMAL],
		    GTK_JUSTIFY_CENTER,
		    FALSE,			/* Do not expand */
                    border_minor,		/* Inside border */
                    border_minor		/* Outside border */
		);
		font = GDK_FONT_UNREF(font);
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_widget_show(w);

		/* Device Label */
		if(stat_status)
		    msg = STRDUP("");
		else  
		    msg = g_strdup_printf(
			"%ld",
			(gulong)stat_buf.st_dev
		    );
		ADD_LABEL("Device:", msg)
		g_free(msg);

		/* Filesystem GtkLabel */
		msg = STRDUP(dev->fs_type_name);
		ADD_LABEL("Filesystem:", msg)   
		g_free(msg);

		/* IO Block Size GtkLabel */
		if(stat_status)
		    msg = STRDUP("");
		else
		    msg = g_strdup_printf(
			"%ld %s",
			(gulong)stat_buf.st_blksize,
			(stat_buf.st_blksize == 1l) ? "byte" : "bytes"
		    );
		ADD_LABEL("IO Block Size:", msg)
		g_free(msg);      
	    }

	    g_free(font_name_h1_bold);
	    g_free(font_name_h1);
	    g_free(font_name_h2_bold);
	    g_free(font_name_h2);

#undef SET_BUTTON_CB
#undef SET_FLAT_BUTTON_STYLE
#undef ADD_LABEL
	}

	gtk_widget_queue_resize(zt->toplevel);
}



/*
 *	Creates a new zip tools window.
 */
ZipTool *zt_new(EDVContext *ctx)
{
	gint border_minor = 2;
	GdkWindow *window;
	GList *tb_items_list;
	GtkWidget	*w, *parent, *parent2, *parent3,
			*toplevel, *main_vbox;
	GtkAccelGroup *accelgrp;
	GtkWidget *tb;
	ToolBarItem *tb_item;
	ZipTool *zt = (ZipTool *)g_malloc0(
	    sizeof(ZipTool)
	);
	if(zt == NULL)
	    return(NULL);

	zt->toplevel = toplevel = gtk_window_new(GTK_WINDOW_DIALOG);
	zt->accelgrp = accelgrp = gtk_accel_group_new();
	zt->freeze_count = 0;
	zt->busy_count = 0;
	zt->ctx = ctx;
	zt->opid = zt_opids_list_new(zt);
	zt->device = NULL;
	zt->lock_state = ZIP_TOOL_LOCK_STATE_UNLOCKED;
	zt->busy_cur = gdk_cursor_new(GDK_WATCH);
	zt->text_cur = gdk_cursor_new(GDK_XTERM);

	zt->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_window_set_wmclass(                        
	    GTK_WINDOW(w), "ziptool", PROG_NAME
	);
        gtk_window_set_policy(GTK_WINDOW(w), FALSE, FALSE, FALSE);
	gtk_window_set_title(GTK_WINDOW(w), PROG_NAME);
	gtk_widget_set_usize(w, ZIP_TOOL_WIDTH, ZIP_TOOL_HEIGHT);
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE | GDK_DECOR_MENU |
		    GDK_DECOR_MINIMIZE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
	    );
	    GUISetWMIcon(
		window,
		(guint8 **)icon_drive_zip_48x48_xpm
	    );
	}
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(zt_delete_event_cb), zt
	);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	parent = w;

	/* Main GtkVBox */
	zt->main_vbox = main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Create the Tool Bar Items list */
	tb_items_list = NULL;
#define APPEND_BUTTON(_id_)	{			\
 ZipToolOPID *opid = zt_opids_list_find(	\
  zt->opid, (_id_)					\
 );							\
 if(opid != NULL) {					\
  tb_item = ToolBarItemNew(				\
   TOOL_BAR_ITEM_BUTTON,					\
   opid->button_name,					\
   opid->icon_data,					\
   opid->tooltip,					\
   opid->id,						\
   zt_opid_cb, opid,					\
   zt_opid_enter_cb, opid,				\
   zt_opid_leave_cb, opid				\
  );							\
  tb_items_list = g_list_append(			\
   tb_items_list,					\
   tb_item						\
  );							\
} }
#define APPEND_SEPARATOR	{			\
 tb_item = ToolBarItemNew(				\
  TOOL_BAR_ITEM_SEPARATOR,				\
  NULL,							\
  NULL,							\
  NULL,							\
  0,							\
  NULL, NULL,						\
  NULL, NULL,						\
  NULL, NULL						\
 );							\
 tb_items_list = g_list_append(				\
  tb_items_list,					\
  tb_item						\
 );							\
}
	APPEND_BUTTON(ZIP_TOOL_OPID_MOUNT);
	APPEND_BUTTON(ZIP_TOOL_OPID_UNMOUNT);
	APPEND_SEPARATOR
	APPEND_BUTTON(ZIP_TOOL_OPID_SPIN_DOWN);
	APPEND_BUTTON(ZIP_TOOL_OPID_EJECT);
	APPEND_BUTTON(ZIP_TOOL_OPID_PASSWORD);
	APPEND_SEPARATOR
	APPEND_BUTTON(ZIP_TOOL_OPID_FSCK);
	APPEND_SEPARATOR
	APPEND_BUTTON(ZIP_TOOL_OPID_REFRESH);
	APPEND_SEPARATOR
	APPEND_BUTTON(ZIP_TOOL_OPID_CLOSE);

#undef APPEND_BUTTON
#undef APPEND_SEPARATOR

	/* Create the Tool Bar */
	zt->tool_bar_handle = w = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	zt->tool_bar = tb = ToolBarNew(
	    GTK_ORIENTATION_HORIZONTAL,
	    TOOL_BAR_BUTTON_DECALS_PICTURES_AND_TEXT,
	    GTK_RELIEF_NONE,
	    GTK_POLICY_AUTOMATIC,
	    tb_items_list
	);
	gtk_container_add(GTK_CONTAINER(parent2), tb);
        gtk_container_border_width(GTK_CONTAINER(w), border_minor);
	gtk_widget_show(tb);

	/* Delete the Tool Bar Items list */
	if(tb_items_list != NULL)
	{
	    g_list_foreach(
		tb_items_list,
		(GFunc)ToolBarItemDelete,
		NULL
	    );
	    g_list_free(tb_items_list);
	}


	/* Display GtkEventBox */
	zt->display_event_box = w = gtk_event_box_new();
	gtk_widget_add_events(
	    w,
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(zt_display_button_event_cb), zt
	);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_realize(w);
	gtk_widget_show(w);
	parent2 = w;

	/* Set the style for the GtkEventBox */
	if(TRUE)
	{
#if 0
	    const gchar *rcstr = "\n\
style \"ziptool-base-style\" { \n\
 bg[NORMAL] = \"#ffffc0\"\n\
 fg[NORMAL] = \"#000000\"\n\
}\n\
widget \"*ziptool-base\" style \"ziptool-base-style\"\n\
";
	    gtk_rc_parse_string(rcstr);

#endif
	}

	/* Display parent GtkVBox */
	zt->display_parent = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent2 = w;

	zt->display_child = NULL;
	zt->capacity_pie_chart = NULL;
	zt->display_mount_path_btn = NULL;
	zt->display_mount_btn = NULL;
	zt->display_lock_btn = NULL;



	/* Status bar frame */
	w = gtk_frame_new(NULL);
	gtk_widget_set_usize(w, -1, EDV_STATUS_BAR_HEIGHT);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_OUT);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Table in main frame */
	w = gtk_table_new(1, 1, FALSE);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent2 = w;

	/* Frame for label */
	w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_table_attach(
	    GTK_TABLE(parent2), w,
	    0, 1, 0, 1,
	    GTK_SHRINK | GTK_EXPAND | GTK_FILL,
	    GTK_SHRINK,
	    border_minor, border_minor
	);
	gtk_widget_show(w);
	parent3 = w;
	/* Hbox in frame for label */
	w = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_show(w);
	parent3 = w;
	/* Label */
	zt->status_label = w = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, border_minor);
	gtk_widget_show(w);


	/* Display area right click menu */
	if(TRUE)
	{
#define ADD_MENU_ITEM_LABEL(_id_,_ak_,_am_)	{	\
 ZipToolOPID *opid = zt_opids_list_find(	\
  zt->opid, (_id_)					\
 );							\
 if(opid != NULL) {					\
  w = GUIMenuItemCreate(				\
   menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp,		\
   opid->icon_data, opid->name,				\
   (_ak_), (_am_),					\
   opid->func_cb, zt					\
  );							\
  GUISetMenuItemCrossingCB(				\
   w,							\
   (gpointer)opid->func_enter_cb, opid,			\
   (gpointer)opid->func_leave_cb, opid			\
  );							\
} }
#define ADD_MENU_ITEM_SEPARATOR			{	\
 w = GUIMenuItemCreate(					\
  menu,							\
  GUI_MENU_ITEM_TYPE_SEPARATOR,				\
  NULL,							\
  NULL,							\
  NULL,							\
  0, 0,							\
  NULL, NULL						\
 );							\
}

	    GtkWidget *menu = GUIMenuCreate();
	    GtkAccelGroup *accelgrp = NULL;

	    zt->menu = menu;

	    ADD_MENU_ITEM_LABEL(ZIP_TOOL_OPID_MOUNT, 0, 0)
	    zt->mount_mi = w;
	    ADD_MENU_ITEM_LABEL(ZIP_TOOL_OPID_UNMOUNT, 0, 0)
	    zt->unmount_mi = w;

	    ADD_MENU_ITEM_SEPARATOR

	    ADD_MENU_ITEM_LABEL(ZIP_TOOL_OPID_SPIN_DOWN, 0, 0)
	    zt->spin_down_mi = w;
	    ADD_MENU_ITEM_LABEL(ZIP_TOOL_OPID_EJECT, 0, 0)
	    zt->eject_mi = w;
	    ADD_MENU_ITEM_LABEL(ZIP_TOOL_OPID_PASSWORD, 0, 0)
	    zt->password_mi = w;

	    ADD_MENU_ITEM_SEPARATOR

	    ADD_MENU_ITEM_LABEL(ZIP_TOOL_OPID_FSCK, 0, 0)
	    zt->fsck_mi = w;

	    ADD_MENU_ITEM_SEPARATOR

	    ADD_MENU_ITEM_LABEL(ZIP_TOOL_OPID_REFRESH, 0, 0)
	    zt->refresh_mi = w;

#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_SEPARATOR
	}



	zt_update_display(zt);

	zt->freeze_count--;

	return(zt);
}

/*
 *	Updates the Zip Tool Window's widgets to reflect current
 *	values.
 */
void zt_update_display(ZipTool *zt)
{
	gboolean sensitive;
	GtkWidget *w;
	GtkWidget *tb;
	EDVDevice *dev;

	if(zt == NULL)
	    return;

	zt->freeze_count++;

	tb = zt->tool_bar;
	dev = zt->device;

	/* Mount */
	sensitive = (dev != NULL) ? !EDV_DEVICE_IS_MOUNTED(dev) : TRUE;
	ToolBarSetItemSensitive(
	    tb, ZIP_TOOL_OPID_MOUNT, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(zt->mount_mi, sensitive)

	/* Unmount */
	sensitive = (dev != NULL) ? EDV_DEVICE_IS_MOUNTED(dev) : FALSE;
	ToolBarSetItemSensitive(
	    tb, ZIP_TOOL_OPID_UNMOUNT, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(zt->unmount_mi, sensitive)

	/* Eject */
	sensitive = (dev != NULL) ? !EDV_DEVICE_IS_MOUNTED(dev) : TRUE;
	ToolBarSetItemSensitive(
	    tb, ZIP_TOOL_OPID_EJECT, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(zt->eject_mi, sensitive)

	/* Lock/unlock */
	sensitive = (dev != NULL) ? !EDV_DEVICE_IS_MOUNTED(dev) : TRUE;
	ToolBarUpdateItem(
	    tb,
	    ZIP_TOOL_OPID_PASSWORD,
	    (zt->lock_state != ZIP_TOOL_LOCK_STATE_UNLOCKED) ?
		"Unlock" : "Lock",
	    NULL,
	    NULL
	);
	ToolBarSetItemSensitive(
	    tb, ZIP_TOOL_OPID_PASSWORD, sensitive
	);
	w = zt->password_mi,
	GUIMenuItemSetLabel(
	    w,
	    (zt->lock_state != ZIP_TOOL_LOCK_STATE_UNLOCKED) ?
		"Unlock" : "Lock"
	);
	GTK_WIDGET_SET_SENSITIVE(w, sensitive)

	/* FSCK */
	sensitive = (dev != NULL) ? !EDV_DEVICE_IS_MOUNTED(dev) : TRUE;
	ToolBarSetItemSensitive(
	    tb, ZIP_TOOL_OPID_FSCK, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(zt->fsck_mi, sensitive)

	zt->freeze_count--;
}

/*
 *	Sets the Zip Tool Window as busy or ready.
 */
void zt_set_busy(
	ZipTool *zt,
	const gboolean busy
)
{
	GdkCursor *cur;
	GdkWindow *window;
	GtkWidget *w;

	if(zt == NULL)
	    return;

	w = zt->toplevel;

	if(busy)
	{
	    /* Increase the busy count */
	    zt->busy_count++;

	    /* Already busy? */
	    if(zt->busy_count > 1)
		return;

	    cur = zt->busy_cur;
	}
	else
	{
	    /* Reduce the busy count */
	    zt->busy_count--;
	    if(zt->busy_count < 0)
		zt->busy_count = 0;

	    /* Still busy? */
	    if(zt->busy_count > 0)
		return;

	    cur = NULL;
	}

	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_cursor(window, cur);
	    gdk_flush();
	}
}

/*
 *	Sets the Zip Tool Window status bar's message.
 */
void zt_status_message(
	ZipTool *zt,
	const gchar *msg,
	const gboolean allow_gtk_iteration

)
{
	GtkWidget *w = (zt != NULL) ? zt->status_label : NULL;
	if(w == NULL)
	    return;

	gtk_label_set_text(
	    GTK_LABEL(w),
	    (msg != NULL) ? msg : ""
	);

	if(allow_gtk_iteration)
	    gtk_events_process();
}

/*
 *	Checks if the Zip Tool Window is mapped.
 */
gboolean zt_is_mapped(ZipTool *zt)
{
	if(zt == NULL)
	    return(FALSE);

	return(GTK_WIDGET_MAPPED(zt->toplevel));
}

/*
 *      Maps the Zip Tool Window.
 */
void zt_map(ZipTool *zt)
{
	if(zt == NULL)
	    return;

	gtk_widget_show_raise(zt->toplevel);
}

/*
 *	Unmaps the Zip Tool Window.
 */
void zt_unmap(ZipTool *zt)
{
	if(zt == NULL)
	    return;

	gtk_widget_hide(zt->toplevel);
}

/*
 *	Deletes the Zip Tool Window.
 */
void zt_delete(ZipTool *zt)
{
	if(zt == NULL)
	    return;

	zt_unmap(zt);

	zt->freeze_count++;

	GTK_WIDGET_DESTROY(zt->menu);

	GTK_WIDGET_DESTROY(zt->toplevel);

	GTK_ACCEL_GROUP_UNREF(zt->accelgrp);

	zt_opids_list_delete(zt->opid);

	GDK_CURSOR_DESTROY(zt->busy_cur);
	GDK_CURSOR_DESTROY(zt->text_cur);

	edv_device_delete(zt->device);

	zt->freeze_count--;

	g_free(zt);
}
