#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <signal.h>
#include <gtk/gtk.h>

#include "../guiutils.h"
#include "../con_msg.h"
#include "../cdialog.h"
#include "../pdialog.h"

#include "../libendeavour2-base/endeavour2.h"

#include "zip_tool.h"
#include "zip_tool_cb.h"
#include "config.h"


static void zt_signal_cb(int s);
static gint zt_timeout_cb(gpointer data);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	UNIX signal callback.
 */
static void zt_signal_cb(int s)
{
	switch(s)
	{
	  case SIGINT:
	  case SIGTERM:
	  case SIGSEGV:
		exit(1);
		break;
	}
}

/*
 *	Main timeout callback.
 */
static gint zt_timeout_cb(gpointer data)
{
	ZipTool *zt = ZIP_TOOL(data);
	if(!zt_is_mapped(zt))
	{
		gtk_main_quit();
		return(FALSE);
	}

	return(TRUE);
}


int main(int argc, char *argv[])
{
	gboolean initialized_gtk = FALSE;
	gint i;
	const gchar	*arg,
			*startup_device = "/dev/sda4";
	GList *devices_list;
	ZipTool *zt;
	EDVContext *ctx;

#define CLEANUP_RETURN(_v_)	{	\
					\
 return(_v_);				\
}

	/* Set up signal callbacks */
	signal(SIGINT, zt_signal_cb);
	signal(SIGTERM, zt_signal_cb);
	signal(SIGKILL, zt_signal_cb);
	signal(SIGSEGV, zt_signal_cb);
	signal(SIGSTOP, zt_signal_cb);
	signal(SIGCONT, zt_signal_cb);
	signal(SIGPIPE, zt_signal_cb);


	/* Parse arguments */
	for(i = 1; i < argc; i++)
	{
		arg = argv[i];
		if(arg == NULL)
			continue;

		/* Help */
		if(!g_strcasecmp(arg, "--help") ||
		   !g_strcasecmp(arg, "-help") ||
		   !g_strcasecmp(arg, "--h") ||
		   !g_strcasecmp(arg, "-h") ||
		   !g_strcasecmp(arg, "-?")
		)
		{
			g_print("%s", PROG_HELP_MESG);
			CLEANUP_RETURN(0);
		}
		/* Version */
		else if(!g_strcasecmp(arg, "--version") ||
			!g_strcasecmp(arg, "-version")
		)
		{
			g_print(
				"%s %s\n%s",
				PROG_NAME,
				PROG_VERSION,
				PROG_COPYRIGHT
			);
			CLEANUP_RETURN(0);
		}
		/* Skip these arguments so that gtk_window_apply_args()
		 * handles them
		 */
		else if(gtk_is_window_arg(arg))
		{
			i++;
		}
		/* Single character argument? */
		else if((*arg == '-') ? (arg[1] != '-') : FALSE)
		{
			const gchar *v = arg + 1;
/*			gchar c; */

			while(*v != '\0')
			{
#if 0
				c = *v;
				if(c == 's')
				{
					auto_start = TRUE;
				}
				else
				{
					g_printerr(
"-%c: Unsupported argument.\n",
						c  
					);
					CLEANUP_RETURN(2);
				}
#endif
				v++;
			}
		}
		/* Non-option argument? */
		else if((*arg != '-') && (*arg != '+'))
		{
			/* Startup device */
			startup_device = arg;
		}
		else
		{
			g_printerr(
"%s: Unsupported argument.\n",
				arg
			);
			CLEANUP_RETURN(2);
		}
	}


	/* Initialize GTK as needed */
	if(!initialized_gtk)
	{
		if(!gtk_init_check(&argc, &argv))
		{
			g_printerr("Unable to initialize GTK.\n");
			CLEANUP_RETURN(1);
		}
		initialized_gtk = TRUE;

		/* Initialize the GDK RGB buffers */
		gdk_rgb_init();
	}

	/* Initialize dialogs */
	ConMsgInit(
		PROG_NAME,
		NULL,
		0,
		0,
		TRUE,				/* Show stdout dialog */
		TRUE				/* Show stderr dialog */
	);
	CDialogInit();
	PDialogInit();

	/* Initialize Endeavour context */
	ctx = edv_context_new();
	edv_context_init(ctx, NULL);

	/* Get the Devices List */
	devices_list = edv_devices_list(ctx);

	/* Update the device mount states and stats */
	edv_devices_list_update_mount_states(ctx);
	edv_devices_list_update_stats(ctx);

	/* Check if no devices have been loaded, suggesting that
	 * either no device references exist in the devices.ini or
	 * that the file does not exist
	 */
	if(devices_list == NULL)
	{
		edv_play_sound_warning(ctx);
		CDialogSetTransientFor(NULL);
		CDialogGetResponse(
			"No Devices Found",
"No devices were found.\n\
\n\
If you do have devices then you should run\n\
Endeavour Mark II to configure the device\n\
references by going to Device->Devices...\n\
and then exit Endeavour Mark II to ensure that\n\
the changes have been saved. Afterwards run this\n\
program again.",
			NULL,
			CDIALOG_ICON_WARNING,
			CDIALOG_BTNFLAG_OK,
			CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
	}


	/* Create Zip Tool Window */
	zt = zt_new(ctx);
	if(zt != NULL)
	{
		gtk_window_apply_args(GTK_WINDOW(zt->toplevel), argc, argv);
		zt_map(zt);

		zt_set_busy(zt, TRUE);

		/* Find the startup Device in the Devices List */
		if(startup_device != NULL)
		{
			GList *glist;
			EDVDevice *d;

			for(glist = devices_list;
				glist != NULL;
				glist = g_list_next(glist)
			)
			{
				d = EDV_DEVICE(glist->data);
				if(d == NULL)
					continue;

				/* This device's path matches the startup device? */
				if(d->device_path != NULL)
				{
					if(!strcmp(
						(const char *)d->device_path,
						(const char *)startup_device
					))
					{
						/* Copy this Device to the ZipTools */
						d = edv_device_copy(d);
						zt_set_device(zt, d);
						zt_refresh_device(zt, d);
						zt_update_display(zt);
						break;
					}
				}
			}
			/* No such device? */
			if(glist == NULL)
			{
				gchar *msg = g_strdup_printf(
"Could not find device:\n\
\n\
    %s\n\
\n\
It is possible that no device references were defined\n\
or that the device references file does not exist.",
					startup_device
				);
				edv_play_sound_warning(ctx);
				CDialogSetTransientFor(zt->toplevel);
				CDialogGetResponse(
					"Device Not Found",
					msg,
"You should run Endeavour Mark II and configure the\n\
device references by going to Device->Devices...\n\
and then exit Endeavour Mark II to ensure that\n\
the changes have been saved. Afterwards run this\n\
program again.",
					CDIALOG_ICON_WARNING,
					CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
					CDIALOG_BTNFLAG_OK
				);
				CDialogSetTransientFor(NULL);
				g_free(msg);
			}
		}

		zt_set_busy(zt, FALSE);


		gtk_timeout_add(
			1000l,
			(GtkFunction)zt_timeout_cb, zt
		);
		gtk_main();

		zt_delete(zt);
	}

	edv_context_sync(ctx);
	edv_context_delete(ctx);
	ctx = NULL;

	/* Shutdown the dialogs */
	PDialogShutdown();
	CDialogShutdown();
	ConMsgShutdown();

	CLEANUP_RETURN(0);
#undef CLEANUP_RETURN
}
