#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <gtk/gtk.h>

#include "../include/string.h"

#include "cfg.h"

#include "guiutils.h"
#include "fprompt.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_directory.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_property.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "edv_device.h"
#include "edv_devices_list.h"
#include "edv_obj_info_match.h"
#include "edv_utils_gtk.h"
#include "edv_confirm.h"
#include "edv_status_bar.h"
#include "vfs_browser.h"
#include "vfs_browser_cb.h"
#include "vfs_browser_tree.h"
#include "edv_vfs_obj_op.h"
#include "edv_emit.h"
#include "edv_open.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


typedef struct _EDVBrowserDirTreeFPromptData	EDVBrowserDirTreeFPromptData;
#define EDV_VFS_BROWSER_DIR_TREE_FPROMPT_DATA(p)	((EDVBrowserDirTreeFPromptData *)(p))


/* Utilities */
static EDVVFSObject *EDVBrowserDirTreeHasSubDirs(
	const gchar *path,
	const gboolean hide_object_hidden
);
static GtkCTreeNode *EDVBrowserDirTreeInsertDummyNode(
	EDVVFSBrowser *browser,
	GtkCTreeNode *parent_node,
	GtkCTreeNode *sibling_node
);
static gboolean EDVBrowserDirTreeIsPathNoScan(
	EDVCore *core,
	const gchar *path
);

/* Node Setting */
static void EDVBrowserDirTreeSetNode(
	EDVVFSBrowser *browser,
	GtkCTreeNode *node,
	EDVVFSObject *obj
);

/* Node Utilities */
GtkCTreeNode *edv_vfs_browser_tree_get_toplevel_node(EDVVFSBrowser *browser);
static GtkCTreeNode *EDVBrowserDirTreeInsertNode(
	EDVVFSBrowser *browser,
	GtkCTreeNode *parent_node,
	GtkCTreeNode *sibling_node,
	EDVVFSObject *obj
);

/* Origin Path */
void edv_vfs_browser_tree_set_origin_path(
	EDVVFSBrowser *browser,
	const gchar *path
);

/* Get Listing */
static void EDVBrowserDirTreeGetChildrenListIterate(
	EDVVFSBrowser *browser,
	GtkCTreeNode *parent_node,
	const gboolean show_progress,
	const gboolean recurse,
	const gboolean hide_object_hidden,
	const gboolean hide_object_noaccess,
	const gboolean probe_only
);
static void EDVBrowserDirTreeGetChildrenList(
	EDVVFSBrowser *browser,
	GtkCTreeNode *parent_node,
	const gboolean show_progress,
	const gboolean recurse
);
void edv_vfs_browser_tree_create_toplevel_nodes(EDVVFSBrowser *browser);
void edv_vfs_browser_tree_get_node_listing(
	EDVVFSBrowser *browser,
	GtkCTreeNode *node,
	const gboolean show_progress
);
void edv_vfs_browser_tree_clear(EDVVFSBrowser *browser);

/* Expanding */
void edv_vfs_browser_tree_expand_node(
	EDVVFSBrowser *browser,
	GtkCTreeNode *node,
	const gboolean show_progress
);

/* Realize Listing */
static void edv_vfs_browser_tree_realize_node_listingIterate(
	EDVVFSBrowser *browser,
	GtkCTreeNode *node,
	const gboolean parent_is_expanded
);
void edv_vfs_browser_tree_realize_node_listing(
	EDVVFSBrowser *browser,
	GtkCTreeNode *node
);

/* Selecting */
static void edv_vfs_browser_tree_select_pathIterate(
	EDVVFSBrowser *browser,
	GtkCTreeNode *node,
	const gchar *path,
	GtkCTreeNode **matched_node_rtn,
	GtkCTreeNode **partial_matched_node_rtn
);
void edv_vfs_browser_tree_select_path(
	EDVVFSBrowser *browser,
	const gchar *path
);

/* Finding */
static void edv_vfs_browser_tree_find_by_indexIterate(
	GtkCTree *ctree,
	GtkCTreeNode *node,
	const gulong device_index, const gulong index,
	GtkCTreeNode **matched_node
);
GtkCTreeNode *edv_vfs_browser_tree_find_by_index(
	EDVVFSBrowser *browser,
	const gulong device_index, const gulong index
);
static void edv_vfs_browser_tree_find_by_pathIterate(
	GtkCTree *ctree,
	GtkCTreeNode *node,
	const gchar *path,
	GtkCTreeNode **matched_node
);
GtkCTreeNode *edv_vfs_browser_tree_find_by_path(
	EDVVFSBrowser *browser,
	const gchar *path
);

/* Removing */
void edv_vfs_browser_tree_remove_grand_children_nodes(
	EDVVFSBrowser *browser, GtkCTreeNode *node
);

/* Opening */
void edv_vfs_browser_tree_open_with(
	EDVVFSBrowser *browser,
	GtkCTreeNode *node
);

/* Renaming */
static void EDVBrowserDirTreeFPromptRenameApplyCB(
	gpointer data, const gchar *value
);
static void EDVBrowserDirTreeFPromptRenameCancelCB(gpointer data);
void edv_vfs_browser_tree_rename_query(
	EDVVFSBrowser *browser,
	GtkCTreeNode *node
);

/* Object Callbacks */
void edv_vfs_browser_tree_vfs_object_added_cb(
	EDVVFSBrowser *browser,
	const gchar *path,
	EDVVFSObject *obj
);
void edv_vfs_browser_tree_vfs_object_modified_cb(
	EDVVFSBrowser *browser,
	const gchar *path,
	const gchar *new_path,
	EDVVFSObject *obj
);
void edv_vfs_browser_tree_vfs_object_removed_cb(
	EDVVFSBrowser *browser,
	const gchar *path
);

/* Mount Callbacks */
void edv_vfs_browser_tree_vfs_object_device_mount_cb(
	EDVVFSBrowser *browser,
	EDVDevice *d,
	const gboolean mounted
);


struct _EDVBrowserDirTreeFPromptData {
	EDVVFSBrowser	*browser;
	GtkCTreeNode	*node;
};


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Checks if the directory has any subdirectories.
 *
 *	The path specifies the full path to the directory.
 *
 *	If hide_object_hidden is TRUE then hidden objects will be
 *	excluded.
 *
 *	Returns a new object describing the first subdirectory found
 *	in the directory or NULL if there were no subdirectories.
 */
static EDVVFSObject *EDVBrowserDirTreeHasSubDirs(
	const gchar *path,
	const gboolean hide_object_hidden
)
{
	const gchar *name;
	gchar *child_path;
	EDVDirectory *dp;

	/* Open the directory */
	dp = edv_directory_open(
		path,
		FALSE,				/* Unsorted */
		FALSE				/* Exclude notations */
	);
	if(dp == NULL)
		return(NULL);

	/* Check if there is a subdirectory */
	for(name = edv_directory_next(dp);
		name != NULL;
		name = edv_directory_next(dp)
	)
	{
		/* Skip hidden objects? */
		if(hide_object_hidden)
		{
			if(edv_path_is_hidden(name))
				continue;
		}

		/* Get the full path to this object */
		child_path = g_strconcat(
			path,
			G_DIR_SEPARATOR_S,
			name,
			NULL
		);
		if(child_path == NULL)
			continue;

		/* Does this object lead to a directory? */
		if(edv_path_is_directory(child_path))
		{
			/* Get this object's local statistics */
			EDVVFSObject *obj = edv_vfs_object_lstat(child_path);

			/* Close the directory and return the new object */
			g_free(child_path);
			edv_directory_close(dp); 
			return(obj);
		}

		g_free(child_path);
	}

	/* Close the directory */
	edv_directory_close(dp);

	return(NULL);
}

/*
 *	Inserts a dummy node.
 *
 *	Used to mark an unexpanded directory that it has contents.
 *
 *	Returns the dummy GtkCTreeNode or NULL on error.
 */
static GtkCTreeNode *EDVBrowserDirTreeInsertDummyNode(
	EDVVFSBrowser *browser,
	GtkCTreeNode *parent_node,
	GtkCTreeNode *sibling_node
)
{
	gint i;
	gchar **strv;
	GtkCTreeNode *node;
	GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
	GtkCList *clist = GTK_CLIST(ctree);
	const gint ncolumns = MAX(clist->columns, 1);

	gtk_clist_freeze(clist);

	/* Allocate the new node's cell values */
	strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
		strv[i] = "";

	/* Insert the new node */
	node = gtk_ctree_insert_node(
		ctree,
		parent_node,
		sibling_node,
		strv,				/* Cell values */
		EDV_LIST_PIXMAP_TEXT_SPACING,	/* Pixmap text spacing */
		NULL, NULL,				/* Closed pixmap, mask */
		NULL, NULL,				/* Opened pixmap, mask */
		TRUE,				/* Is leaf */
		FALSE				/* Not expanded */
	);

	/* Delete the node's cell values */
	g_free(strv);

	gtk_clist_thaw(clist);

	return(node);
}

/*
 *	Checks if the path should not be scanned for its contents.
 *
 *	All inputs assumed valid.
 */
static gboolean EDVBrowserDirTreeIsPathNoScan(
	EDVCore *core,
	const gchar *path
)
{
	/* Check if this directory is a mounted device that should
	 * not be scanned for its contents
	 */
#if 1
	EDVDevice *dev = edv_devices_list_match_object(
		core->devices_list,
		NULL,
		path
	);
#else
	EDVDevice *dev = edv_devices_list_match_mount_path(
		core->devices_list,
		NULL,
		path			/* Mount path */
	);
#endif
	if(dev == NULL)
		return(FALSE);

	if(EDV_DEVICE_IS_NO_SCAN(dev))
	{
		if(EDV_DEVICE_IS_MOUNTED(dev))
			return(TRUE);
	}

	return(FALSE);
}

/*
 *	Sets the Directory Tree node's icon, text, and style.
 *
 *	The ctree specifies the Directory Tree.
 *
 *	The node specifies the node.
 *
 *	The obj specifies the object who's values will be used to
 *	set the node's icon, text, and style.
 *
 *	All inputs assumed valid.
 */
static void EDVBrowserDirTreeSetNode(
	EDVVFSBrowser *browser,
	GtkCTreeNode *node,
	EDVVFSObject *obj
)
{
	const gchar *text = obj->name;
	GtkStyle	**cell_styles = browser->cell_style,
					*style = NULL;
	GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
	GtkCList *clist = GTK_CLIST(ctree);
	GtkCTreeRow *row_ptr = GTK_CTREE_ROW(node);
	EDVPixmap	*icon,
					*icon_opened,
					*icon_inaccessable,
					*icon_hidden;
	EDVCore *core = browser->core;

	if(row_ptr == NULL)
		return;

	/* Get the pixmap and mask for this object's icon */
	(void)edv_match_object_icon(
		core->devices_list,
		core->mime_types_list,
		obj->type,
		obj->path,
		EDV_VFS_OBJECT_LINK_TARGET_EXISTS(obj),
		obj->permissions,
		EDV_ICON_SIZE_20,
		&icon,
		&icon_opened,
		&icon_inaccessable,
		&icon_hidden
	);
	/* Hidden? */
	if(edv_is_object_hidden(obj))
	{
		/* Use the hidden icon if it is available */
		if(edv_pixmap_is_loaded(icon_hidden))
		{
			(void)edv_pixmap_unref(icon);
			icon = edv_pixmap_ref(icon_hidden);
		}
	}

	/* Directory not accessable? */
	if(!edv_is_object_accessable(core, obj))
	{
		/* Use the directory not accessable icon if it is available */
		if(edv_pixmap_is_loaded(icon_inaccessable))
		{
			(void)edv_pixmap_unref(icon);
			icon = edv_pixmap_ref(icon_inaccessable);
		}

		/* Use the directory not accessable style */
		if(style == NULL)
			style = cell_styles[EDV_VFS_BROWSER_CELL_STYLE_NO_ACCESS];
	}

	/* Link target is a grand parent? */
	if(EDV_VFS_OBJECT_IS_LINK_TARGET_GRAND_PARENT(obj))
	{
		/* Use the recursive link style */
		if(style == NULL)
			style = cell_styles[EDV_VFS_BROWSER_CELL_STYLE_RECURSIVE_LINK];
	}

	/* If the opened icon was not available then use the closed
	 * icon as the opened icon
	 */
	if(!edv_pixmap_is_loaded(icon_opened))
	{
		(void)edv_pixmap_unref(icon_opened);
		icon_opened = edv_pixmap_ref(icon);
	}

	/* Set this node */
	gtk_clist_freeze(clist);
	if(edv_pixmap_is_loaded(icon) &&
	   edv_pixmap_is_loaded(icon_opened)
	)
		gtk_ctree_set_node_info(
			ctree,
			node,
			(text != NULL) ? text : "(null)",
			EDV_LIST_PIXMAP_TEXT_SPACING,
			icon->pixmap, icon->mask,
			icon_opened->pixmap, icon_opened->mask,
			row_ptr->is_leaf,
			row_ptr->expanded
		);
	else
		gtk_ctree_set_node_info(
			ctree,
			node,
			(text != NULL) ? text : "(null)",
			EDV_LIST_PIXMAP_TEXT_SPACING,
			NULL, NULL,
			NULL, NULL,
			row_ptr->is_leaf,
			row_ptr->expanded
		);
	gtk_ctree_node_set_row_style(
		ctree,
		node,
		style
	);
	gtk_clist_thaw(clist);

	(void)edv_pixmap_unref(icon);
	(void)edv_pixmap_unref(icon_opened);
	(void)edv_pixmap_unref(icon_inaccessable);
	(void)edv_pixmap_unref(icon_hidden);
}


/*
 *	Gets the toplevel node.
 */
GtkCTreeNode *edv_vfs_browser_tree_get_toplevel_node(EDVVFSBrowser *browser)
{
	GtkCTree *ctree;

	if(browser == NULL)
		return(NULL);

	ctree = GTK_CTREE(browser->directory_ctree);

	/* For now just get the first toplevel, later on we may need
	 * to return a node other than the first toplevel if multiple
	 * vfs are to be supported
	 */
	return(edv_ctree_node_get_toplevel(ctree));
}


/*
 *	Inserts a node.
 *
 *	The parent_node specifies the parent GtkCTreeNode that the
 *	new node will be inserted to. If parent_node is NULL then the
 *	new node will be a new toplevel node.
 *
 *	The sibling_node specifies the sibling GtkCTreeNode that the
 *	new node will be inserted after. If sibling_node is NULL then
 *	the new node will be appended to parent_node.
 *
 *	The obj specifies the object that will be set as the new
 *	node's data. The obj should not be referenced again after
 *	this call.
 *
 *	Returns the new node or NULL on error.
 *
 *	All inputs assumed valid.
 */
static GtkCTreeNode *EDVBrowserDirTreeInsertNode(
	EDVVFSBrowser *browser,
	GtkCTreeNode *parent_node,
	GtkCTreeNode *sibling_node,
	EDVVFSObject *obj
)
{
	gint i;
	gchar **strv;
	GtkCTreeNode *node;
	GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
	GtkCList *clist = GTK_CLIST(ctree);
	const gint ncolumns = MAX(clist->columns, 1);

	gtk_clist_freeze(clist);

	/* Allocate the new node's cell values */
	strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
		strv[i] = "";

	/* Insert the new node */
	node = gtk_ctree_insert_node(
		ctree,
		parent_node,
		sibling_node,
		strv,				/* Cell values */
		EDV_LIST_PIXMAP_TEXT_SPACING,	/* Pixmap text spacing */
		NULL, NULL,				/* Closed pixmap, mask */
		NULL, NULL,				/* Opened pixmap, mask */
		FALSE,				/* Not leaf */
		FALSE				/* Not expanded */
	);

	/* Delete the cell values */
	g_free(strv);

	/* Failed to insert a new node? */
	if(node == NULL)
	{
		gtk_clist_thaw(clist);
		edv_vfs_object_delete(obj);
		return(NULL);
	}

	/* Set the new node's text, pixmaps, and style */
	EDVBrowserDirTreeSetNode(
		browser,
		node,
		obj
	);

	/* Set the new node's row data as the specified object */
	gtk_ctree_node_set_row_data_full(
		ctree,
		node,
		obj, (GtkDestroyNotify)edv_vfs_object_delete
	);

	gtk_clist_thaw(clist);

	return(node);
}


/*
 *	Sets the origin path and recreates the toplevel nodes.
 */
void edv_vfs_browser_tree_set_origin_path(
	EDVVFSBrowser *browser,
	const gchar *path
)
{
	const gchar *origin_path;
	GtkCList *clist;
	GtkCTreeNode *node;
	GtkCTree *ctree;

	if((browser == NULL) || STRISEMPTY(path))
		return;

	origin_path = browser->directory_ctree_origin_path;
	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);

	/* No change? */
	if(path == origin_path)
		return;
	if(origin_path != NULL)
	{
		if(!strcmp((const char *)origin_path, (const char *)path))
			return;
	}

	/* Set the new origin path */
	g_free(browser->directory_ctree_origin_path);
	browser->directory_ctree_origin_path = g_strdup(path);

	/* Delete all the existing nodes and recreate the toplevels */
	edv_vfs_browser_tree_create_toplevel_nodes(browser);

	/* Select and expand the toplevel node */
	node = edv_vfs_browser_tree_get_toplevel_node(browser);
	if(node != NULL)
	{
		gtk_clist_freeze(clist);
		gtk_ctree_expand(ctree, node);
		gtk_ctree_select(ctree, node);
		gtk_clist_thaw(clist);
	}
}


/*
 *	Called by EDVBrowserDirTreeGetChildrenList() to get a list
 *	of child nodes.
 *
 *	The parent_node specifies the parent GtkCTreeNode and the
 *	path.
 *
 *	If show_progress is TRUE then the progress and messages will
 *	be displayed on the status bar during this operation.
 *
 *	If recurse is TRUE then this call will recurse no more than
 *	2 levels from the parent node. If recurse is FALSE then no
 *	recursion will take place.
 *
 *	If hide_object_hidden is TRUE then hidden objects will be
 *	excluded.
 *
 *	If hide_object_noaccess is TRUE then no access objects will
 *	be excluded.
 *
 *	If probe_only is TRUE then only one child node will be added
 *	to the parent node if and only if one subdirectory was
 *	encountered. This will improve efficiency and conserve memory
 *	and is typically used when this function calls itself for
 *	the second iteration.
 */
static void EDVBrowserDirTreeGetChildrenListIterate(
	EDVVFSBrowser *browser,
	GtkCTreeNode *parent_node,
	const gboolean show_progress,
	const gboolean recurse,
	const gboolean hide_object_hidden,
	const gboolean hide_object_noaccess,
	const gboolean probe_only
)
{
	gchar *path;
	GList *names_list;
	GtkWidget *sb = browser->status_bar;
	GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
	EDVVFSObject *obj;
	EDVCore *core = browser->core;

	if(browser->stop_count > 0)
		return;

	/* Get the parent node's object */
	obj = EDV_VFS_OBJECT(gtk_ctree_node_get_row_data(
		ctree,
		parent_node
	));
	if(obj == NULL)
		return;

	/* Get the path from the parent node's object */
	path = STRDUP(obj->path);
	if(path == NULL)
		return;

	/* Update the status bar message? */
	if(show_progress)
	{
		const gchar *name = g_basename(path);
		gchar *msg = g_strdup_printf(
			"Scanning directory: %s",
			(name != NULL) ? name : path
		);
		edv_status_bar_message(sb, msg, TRUE);
		g_free(msg);
	}

	if(browser->stop_count > 0)
	{
		g_free(path);
		return;
	}

	/* Probe only? */
	if(probe_only)
	{
		/* Check if this directory has at least one subdirectory */
		EDVVFSObject *obj = EDVBrowserDirTreeHasSubDirs(
			path,
			hide_object_hidden
		);
		if(obj != NULL)
		{
			gchar *s;

			/* Add a child node with a fictious object's name so
			 * that the GtkCTree will denote that this parent
			 * node is expandable (has subdirectories)
			 *
			 * One potential problem with this is that if all the
			 * subdirectories were removed the child node with the
			 * fictious object's name will still exist and
			 * indicate this parent node is expandable, but this
			 * is still the best solution available besides
			 * wasting time and memory to add all the child nodes
			 * representing all the subdirectories
			 */
			s = obj->name;
			obj->name = g_strconcat(
				s,
				"_dummy_node",
				NULL
			);
			g_free(s);

			s = obj->path;
			obj->path = g_strconcat(
				s,
				"_dummy_node",
				NULL
			);
			g_free(s);

			EDVBrowserDirTreeInsertNode(
				browser,
				parent_node,
				NULL,			/* No sibling, append to
											 * parent_node */
				obj
			);
		}
		g_free(path);
		return;
	}

	/* Get this directory's contents listing */
	names_list = edv_directory_list(
		path,
		TRUE,				/* Sort */
		FALSE				/* Exclude notations */
	);
	if(names_list != NULL)
	{
		gboolean no_scan_device;
		const gchar *name;
		gchar *full_path;
		GList *glist;
		GtkCTreeNode *node;

		for(glist = names_list;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			if(browser->stop_count > 0)
			{
				do {
					g_free(glist->data);
					glist = g_list_next(glist);
				} while(glist != NULL);
				break;
			}

			name = (const gchar *)glist->data;
			if(name == NULL)
				continue;

			/* Get this object's full path */
			full_path = edv_paths_join(
				path,
				name
			);
			if(full_path == NULL)
			{
				g_free(glist->data);
				continue;
			}

			/* Does this object not lead to a directory? */
			if(!edv_path_is_directory(full_path))
			{
				g_free(full_path);
				g_free(glist->data);
				continue;
			}

			/* Get this object's local statistics */
			obj = edv_vfs_object_lstat(full_path);
			if(obj == NULL)
			{
#if 0
/* Do not add error objects on the Directory GtkCTree because we
 * do not know if this is a directory or not
 */
				const gint error_code = (gint)errno;
				obj = edv_vfs_object_new_error(
					full_path,
					error_code,
					g_strerror(error_code)
				);
				node = EDVBrowserDirTreeInsertNode(
					browser,
					parent_node,
					NULL,			/* No sibling, append to
											 * parent_node */
					obj
				);
#endif
				g_free(full_path);
				g_free(glist->data);
				continue;
			}

			/* Begin filter checks */
			if((hide_object_hidden ?
				edv_is_object_hidden(obj) : FALSE) ||
			   (hide_object_noaccess ?
				!edv_is_object_accessable(core, obj) : FALSE)
			)
			{
				edv_vfs_object_delete(obj);
				node = NULL;
				no_scan_device = FALSE;
			}
			else
			{
				/* Check if this directory should not be
				 * scanned for its contents
				 */
				no_scan_device = EDVBrowserDirTreeIsPathNoScan(
					core,
					obj->path
				);

				/* Add/transfer the object to the tree */
				node = EDVBrowserDirTreeInsertNode(
					browser,
					parent_node,
					NULL,			/* No sibling, append to
											 * parent_node */
					obj
				);
			}
/*		obj = NULL */

			/* Report progress */
			if(show_progress)
				edv_status_bar_progress(sb, -1.0f, TRUE);

			/* Recurse into subdirectories? */
			if(recurse && (node != NULL))
			{
				/* Do not recurse if this is a device that is
				 * marked "no scan"
				 */
				if(no_scan_device)
					EDVBrowserDirTreeInsertDummyNode(
						browser,
						node,
						NULL
					);
				else
					EDVBrowserDirTreeGetChildrenListIterate(
						browser,
						node,
						show_progress,
						FALSE,		/* Do not recurse on the
											 * second iteration */
						hide_object_hidden,
						hide_object_noaccess,
						TRUE		/* Always probe on the
											 * second iteration */
					);
/* TODO Make probe_only on the second iteration an option based on
 * the conserve memory setting in the configuration?
 */
			}

			g_free(full_path);
			g_free(glist->data);
		}

		/* Delete the directory entries list, each entry should
		 * already be deleted
		 */
		g_list_free(names_list);
	}

	g_free(path);
}

/*
 *	Gets a list of child nodes from the parent node, any existing
 *	child nodes in the parent node will be deleted.
 *
 *	The parent_node specifies the parent GtkCTreeNode.
 *
 *	If show_progress is TRUE then the progress and messages will
 *	be displayed on the status bar during this operation.
 *
 *	If recurse is TRUE then this call will recurse no more than
 *	2 levels from the parent node. If recurse is FALSE then no
 *	recursion will take place.
 *
 *	All inputs assumed valid.
 */
static void EDVBrowserDirTreeGetChildrenList(
	EDVVFSBrowser *browser,
	GtkCTreeNode *parent_node,
	const gboolean show_progress,
	const gboolean recurse
)
{
	gboolean	hide_object_hidden,
					hide_object_noaccess;
	GtkWidget *sb = browser->status_bar;
	GtkCTreeRow *row_ptr;
	GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
	GtkCList *clist = GTK_CLIST(ctree);
	EDVCore *core = browser->core;
	CfgList *cfg_list = core->cfg_list;

	/* Get the display options */
	hide_object_hidden = !EDV_GET_B(
		EDV_CFG_PARM_BROWSER_SHOW_OBJECT_HIDDEN
	);
	hide_object_noaccess = !EDV_GET_B(
		EDV_CFG_PARM_BROWSER_SHOW_OBJECT_NOACCESS
	);

	/* Report the initial progress? */
	if(show_progress)
	{
		edv_status_bar_message(
			sb,
			"Scanning directories...",
			FALSE
		);
		edv_status_bar_progress(sb, -1.0f, TRUE);
	}

	gtk_clist_freeze(clist);

	/* Delete all the child nodes from the parent node */
	row_ptr = GTK_CTREE_ROW(parent_node);
	if(row_ptr != NULL)
	{
		while(row_ptr->children != NULL)
			gtk_ctree_remove_node(ctree, row_ptr->children);
	}

	/* Get the new listing */
	EDVBrowserDirTreeGetChildrenListIterate(
		browser,
		parent_node,
		show_progress,
		recurse,
		hide_object_hidden,
		hide_object_noaccess,
		FALSE				/* Do not probe */
	);

	gtk_clist_thaw(clist);

	/* Report the final progress? */
	if(show_progress)
	{
		edv_status_bar_message(sb, NULL, FALSE);
		edv_status_bar_progress(sb, 0.0f, FALSE);
	}
}

/*
 *	Creates the toplevel nodes.
 *
 *	Any existing nodes will be deleted.
 */
void edv_vfs_browser_tree_create_toplevel_nodes(EDVVFSBrowser *browser)
{
	const gchar *vfs_true_toplevel_path = "/";
	const gchar *path;
	GtkCList *clist;
	GtkCTree *ctree;
	GtkCTreeNode *node;
	CfgList *cfg_list;
	EDVVFSObject *obj;
	EDVCore *core;

	if(browser == NULL)
		return;

	browser->stop_count = 0;

	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);
	core = browser->core;
	cfg_list = core->cfg_list;

	gtk_clist_freeze(clist);

	/* Delete all the existing nodes in the Directory GtkCTree*/
	edv_vfs_browser_tree_clear(browser);

	/* Get the path to the toplevel object as the current
	 * Directory Tree Origin
	 *
	 * Note that since we allow a user-settable Directory Tree
	 * Origin, the toplevel node does not always refer to the
	 * toplevel object on the VFS
	 */
	path = browser->directory_ctree_origin_path;
	if(STRISEMPTY(path))
#if defined(_WIN32)
		path = "C:\\";
#else
		path = "/";
#endif

	/* Set the show stub based on if the origin is the same as the
	 * true VFS toplevel origin or not
	 */
	gtk_ctree_set_show_stub(
		ctree,
		strcmp((const char *)path, vfs_true_toplevel_path) ? TRUE : FALSE
	);

	/* Create the object for the toplevel node */
	obj = edv_vfs_object_lstat(path);
	if(obj == NULL)
	{
		const gint error_code = (gint)errno;
		obj = edv_vfs_object_new_error(
			path,
			error_code,
			g_strerror(error_code)
		);
	}

	/* Add/transfer the object to the tree */
	node = EDVBrowserDirTreeInsertNode(
		browser,
		NULL,				/* No parent, append as
											 * a toplevel node */
		NULL,				/* No sibling, append to
											 * parent_node */
		obj
	);
	if((node != NULL) &&
	   (browser->stop_count == 0)
	)
	{
		/* Get one level of child nodes */
		if(EDVBrowserDirTreeIsPathNoScan(core, path))
			EDVBrowserDirTreeInsertDummyNode(
				browser,
				node,
				NULL
			);
		else
			EDVBrowserDirTreeGetChildrenList(
				browser,
				node,			/* Parent node */
				EDV_GET_B(EDV_CFG_PARM_LISTS_SHOW_PROGRESS),
				FALSE			/* Do not recurse, get
											 * only one level */
			);
	}

	gtk_clist_thaw(clist);
}

/*
 *	Gets a listing of child nodes from the specified node,
 *	recursing no more than 2 levels.
 *
 *	The node specifies the GtkCTreeNode at which to get the
 *	listing of child nodes for. If node is NULL then the toplevel
 *	GtkCTreeNode will be used.
 *
 *	Any existing child nodes in the specified node will be deleted
 *	first before getting the new list of child nodes.
 */
void edv_vfs_browser_tree_get_node_listing(
	EDVVFSBrowser *browser,
	GtkCTreeNode *node,
	const gboolean show_progress
)
{
	GtkCList *clist;
	GtkCTree *ctree;

	if(browser == NULL)
		return;

	browser->stop_count = 0;

	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);

	gtk_clist_freeze(clist);

	/* Use the toplevel node if the specified node is NULL */
	if(node == NULL)
		node = edv_vfs_browser_tree_get_toplevel_node(browser);

	/* Delete any existing child nodes from the specified node
	 * and then get a new list of child nodes
	 */
	if((node != NULL) && (browser->stop_count == 0))
		EDVBrowserDirTreeGetChildrenList(
			browser,
			node,				/* Parent node */
			show_progress,
			TRUE				/* Recurse 2 levels */
		);

	gtk_clist_thaw(clist);
}

/*
 *	Deletes all nodes on the Directory Tree.
 */
void edv_vfs_browser_tree_clear(EDVVFSBrowser *browser)
{
	GtkCList *clist;
	GtkCTreeNode *node;
	GtkCTree *ctree;

	if(browser == NULL)
		return;

	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);
	node = edv_vfs_browser_tree_get_toplevel_node(browser);
	if(node != NULL)
	{
		gtk_clist_freeze(clist);
		gtk_ctree_remove_node(ctree, node);
		gtk_clist_thaw(clist);
	}
}


/*
 *	Expands the node and gets a listing of child nodes that
 *	recurse no more than 2 levels.
 *
 *	The node specifies the GtkCTreeNode to expand and to
 *	get a listing of child nodes for.
 */
void edv_vfs_browser_tree_expand_node(
	EDVVFSBrowser *browser,
	GtkCTreeNode *node,
	const gboolean show_progress
)
{
	GtkCList *clist;
	GtkCTree *ctree;
	EDVCore *core;

	if((browser == NULL) || (node == NULL))
		return;

	browser->stop_count = 0;

	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);
	core = browser->core;

	gtk_clist_freeze(clist);

	/* Remove all the existing child nodes of this expanded node
	 * and get a new list of child nodes, recursing no more than
	 * 2 levels
	 */
	EDVBrowserDirTreeGetChildrenList(
		browser,
		node,				/* Parent node */
		show_progress,
		TRUE				/* Recurse 2 levels */
	);

	gtk_clist_thaw(clist);
}


/*
 *	Called by edv_vfs_browser_tree_realize_node_listing() to update existing
 *	nodes and to remove all non-existant nodes, then recurse into
 *	any child nodes.
 *
 *	All inputs assumed valid.
 */
static void edv_vfs_browser_tree_realize_node_listingIterate(
	EDVVFSBrowser *browser,
	GtkCTreeNode *node,
	const gboolean parent_is_expanded
)
{
	const gchar *path;
	GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
	GtkCTreeRow *row_ptr;
	GtkCTreeNode *next_node;
	EDVVFSObject *obj;
	EDVCore *core = browser->core;

	/* Realize the specified node and its siblings */
	while(node != NULL)
	{
		row_ptr = GTK_CTREE_ROW(node);
		if(row_ptr == NULL)
			break;

		/* Get this node's object */
		obj = EDV_VFS_OBJECT(gtk_ctree_node_get_row_data(ctree, node));
		path = (obj != NULL) ? obj->path : NULL;

		/* Realize this node's child objects only if this node's
		 * device is not marked "no scan"
		 */
		if(!EDVBrowserDirTreeIsPathNoScan(core, path))
		{
			/* Recurse into and realize each child node */
			edv_vfs_browser_tree_realize_node_listingIterate(
				browser,
				row_ptr->children,	/* Child node */
				row_ptr->expanded
			);

			/* Does this node no longer have any children? */
			if(row_ptr->children == NULL)
			{
				/* If the parent was expanded then check if there
				 * are any child nodes now by getting a new
				 * children listing
				 *
				 * This should only be done if the parent is
				 * expanded so that we do not get child listings
				 * for nodes that do not need to be indicated as
				 * expandable
				 *
				 * Recurse one level
				 */
				if(parent_is_expanded && (path != NULL))
					EDVBrowserDirTreeGetChildrenList(
						browser,
						node,		/* Parent node */
						FALSE,		/* Do not update progress */
						FALSE		/* Do not recurse, get only
											 * one level */
					);
			}
		}

		/* Get this node's sibling */
		next_node = row_ptr->sibling;

		/* Does this node have an object and its path is set? */
		if(path != NULL)
		{
			/* Check if this node's object still exists locally */
			EDVVFSObject *obj_lvalues = edv_vfs_object_lstat(path);
			if(obj_lvalues != NULL)
			{
				/* Check if its destination still exists */
				EDVVFSObject *obj_values = edv_vfs_object_stat(path);
				if(obj_values != NULL)
				{
					/* Check if its destination still leads to a directory */
					if(EDV_VFS_OBJECT_IS_DIRECTORY(obj_values))
					{
						/* Update this object and its node with the
						 * local statistics
						 */
						edv_vfs_object_set_object(
							obj,
							obj_lvalues
						);
						EDVBrowserDirTreeSetNode(
							browser,
							node,
							obj
						);
					}
					else
					{
						/* Destination is no longer a directory,
						 * remove this node
						 */
						gtk_ctree_remove_node(ctree, node);
/*
						node = NULL;
						row_ptr = NULL;
						obj = NULL;
						path = NULL;
 */
					}

					edv_vfs_object_delete(obj_values);
				}
				else
				{
					/* Destination no longer exists, remove this node */
					gtk_ctree_remove_node(ctree, node);
/*
					node = NULL;
					row_ptr = NULL;
					obj = NULL;
					path = NULL;
 */
				}

				edv_vfs_object_delete(obj_lvalues);
			}
			else
			{
				/* No longer exists, remove this node */
				gtk_ctree_remove_node(ctree, node);
/*
				node = NULL;
				row_ptr = NULL;
				obj = NULL;
				path = NULL;
 */
			}
		}
		/* Does this node not have an object? */
		else if(obj == NULL)
		{
			/* Remove this node */
			gtk_ctree_remove_node(ctree, node);
/*
			node = NULL;
			row_ptr = NULL;
 */
		}

		/* The node, row_ptr, obj, and path should be considered
		 * invalid after this point
		 */

		/* Get the next sibling */
		node = next_node;
	}
}

/*
 *	Realizes all the nodes from the specified node.
 *
 *	Each node will be checked to see if its object exists and is
 *	a directory, if it is not then the node will be removed.
 *
 *	If a node currently does not have any children then a new
 *	listing of child objects will be obtained for that node only
 *	if the node's parent is expanded. If the specified node does
 *	not have any children then a new listing of child objects will
 *	be obtained for the specified node regardless of if its parent
 *	is expanded or not.
 *
 *	The node specifies the GtkCTreeNode at which to realize the
 *	listing at. If node is NULL then the toplevel GtkCTreeNode
 *	will be used.
 */
void edv_vfs_browser_tree_realize_node_listing(
	EDVVFSBrowser *browser,
	GtkCTreeNode *node
)
{
	GtkCList *clist;
	GtkCTree *ctree;
	EDVCore *core;

	if(browser == NULL)
		return;

	browser->stop_count = 0;

	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);
	core = browser->core;

	/* Use toplevel node if the specified node is NULL */
	if(node == NULL)
		node = edv_vfs_browser_tree_get_toplevel_node(browser);
	if(node == NULL)
		return;

	gtk_clist_freeze(clist);

	/* Begin updating the nodes and removing any non-existant
	 * nodes
	 */
	edv_vfs_browser_tree_realize_node_listingIterate(
		browser,
		node,
		TRUE				/* Always indicate the
											 * specified node's
											 * parent as expanded */
	);

	gtk_clist_thaw(clist);
}


/*
 *	Called by edv_vfs_browser_tree_select_path() to seek through all
 *	child directories for a prefix that matches match the
 *	specified path and recurse into child directories.
 *
 *	The path specifies the full path that is intended to be
 *	selected if it is found.
 */
static void edv_vfs_browser_tree_select_pathIterate(
	EDVVFSBrowser *browser,
	GtkCTreeNode *node,
	const gchar *path,
	GtkCTreeNode **matched_node_rtn,
	GtkCTreeNode **partial_matched_node_rtn	
)
{
	const gchar *cur_path;
	GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
	EDVVFSObject *obj;

	/* If a match has already been made then do not continue */
	if(*matched_node_rtn != NULL)
		return;

	if(node == NULL)
		return;

	/* Get this node's object */
	obj = EDV_VFS_OBJECT(gtk_ctree_node_get_row_data(ctree, node));
	if(obj == NULL)
		return;

	/* Get this object's full path as the current path */
	cur_path = obj->path;
	if(STRISEMPTY(cur_path))
		return;

	/* Is the current path a parent or grand parent of the search
	 * path?
	 */
	if(edv_path_is_parent(
		cur_path,				/* Parent */
		path				/* Child */
	))
	{
		GtkCTreeRow *row_ptr;

		/* Update the partial matched node */
		*partial_matched_node_rtn = node;

		/* Check if the search path and the current path are the
		 * same
		 */
		if(!strcmp((const char *)path, (const char *)cur_path))
		{
			/* Got complete match, set matched node and return */
			*matched_node_rtn = node;
			return;
		}
		else
		{
			/* Not a complete match, but cur_path is a parent of
			 * the search path so we are on the right track
			 *
			 * Get child node and iterate through all its siblings
			 * and children
			 */

			/* Expand current node as needed
			 *
			 * Note that this may call the expand callback and that
			 * it will load more child nodes
			 */
			row_ptr = GTK_CTREE_ROW(node);
			if(row_ptr != NULL)
			{
				if(!row_ptr->expanded)
				{
					GtkCList *clist = GTK_CLIST(ctree);
					gtk_clist_freeze(clist);
					gtk_ctree_expand(ctree, node);
					gtk_clist_thaw(clist);
				}
			}

			/* Get the child nodes */
			node = edv_node_get_child(node);
			while(node != NULL)
			{
				/* Check this node, recurse if it is a match */
				edv_vfs_browser_tree_select_pathIterate(
					browser,
					node,
					path,
					matched_node_rtn,
					partial_matched_node_rtn
				);
				/* If we got a match, then do not continue search */
				if((matched_node_rtn != NULL) ? (*matched_node_rtn != NULL) : TRUE)
					return;

				/* Get the next sibling */
				node = edv_node_get_sibling(node);
			}
		}
	}
}

/*
 *	Selects the node that corresponds to the path or a parent
 *	node of the path if the exact node is not found. Nodes will
 *	be expanded and updated as needed.
 *
 *	The path specifies the full path.
 */
void edv_vfs_browser_tree_select_path(
	EDVVFSBrowser *browser,
	const gchar *path
)
{
	gchar *dpath;
	GtkCList *clist;
	GtkCTree *ctree;
	GtkCTreeNode *node, *matched_node, *partial_matched_node;

	if((browser == NULL) || STRISEMPTY(path))
		return;

	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);

	/* Must have a full path */
	if(!g_path_is_absolute(path))
		return;

	/* Copy the path */
	dpath = g_strdup(path);
	if(dpath == NULL)
		return;

	/* Remove any tailing deliminators in the path */
	edv_path_simplify(dpath);

	/* Get the toplevel node to start the search from */
	node = edv_vfs_browser_tree_get_toplevel_node(browser);
	if(node == NULL)
	{
		g_free(dpath);
		return;
	}

	/* Search for the path, expanding and updating nodes as
	 * needed
	 */
	matched_node = NULL;
	partial_matched_node = NULL;
	edv_vfs_browser_tree_select_pathIterate(
		browser,
		node,
		dpath,
		&matched_node,
		&partial_matched_node
	);

	g_free(dpath);

	/* Found the node? */
	if(matched_node != NULL)
	{
		/* Select the matched node */
		gtk_clist_freeze(clist);
		gtk_ctree_select(ctree, matched_node);
		gtk_clist_thaw(clist);
	}
	/* Found a partially matched parent node? */
	else if(partial_matched_node != NULL)
	{
		/* Select the matched parent node */
		gtk_clist_freeze(clist);
		gtk_ctree_select(ctree, partial_matched_node);
		gtk_clist_thaw(clist);
	}
}


/*
 *	Performs one iteration of finding the Directory Tree node by
 *	location index.
 */
static void edv_vfs_browser_tree_find_by_indexIterate(
	GtkCTree *ctree, 
	GtkCTreeNode *node,
	const gulong device_index, const gulong index,
	GtkCTreeNode **matched_node
)
{
	EDVVFSObject *obj;

	if((*matched_node != NULL) || (node == NULL))
		return;

	obj = EDV_VFS_OBJECT(gtk_ctree_node_get_row_data(ctree, node));
	if(obj == NULL)
		return;

	/* Does this object's index location match the specified index
	 * location?
	 */
	if((obj->device_index == device_index) &&
	   (obj->index == index)
	)
	{
		*matched_node = node;
		return;
	}
	else
	{
		/* No match, now check if any of this node's child nodes
		 * match
		 */
		GtkCTreeRow *row_ptr = GTK_CTREE_ROW(node);
		if(row_ptr == NULL)
			return;

		node = row_ptr->children;
		while(node != NULL)
		{
			edv_vfs_browser_tree_find_by_indexIterate(
				ctree,
				node,
				device_index, index,
				matched_node
			);
			if(*matched_node != NULL)
				return;

			/* Get the sibling */
			row_ptr = GTK_CTREE_ROW(node);
			if(row_ptr == NULL)
				break;

			node = row_ptr->sibling;
		}
	}
}

/*
 *	Finds the node by index.
 *
 *	The device_index specifies the index of the device that the
 *	object resides on.
 *
 *	The index specifies the index on the device that the object
 *	resides at.
 *
 *	Returns the matched node or NULL on error.
 */
GtkCTreeNode *edv_vfs_browser_tree_find_by_index(
	EDVVFSBrowser *browser,                  
	const gulong device_index, const gulong index
)
{
	GtkCTreeNode *node, *matched_node;
	GtkCTree *ctree;

	if(browser == NULL)
		return(NULL);

	ctree = GTK_CTREE(browser->directory_ctree);

	/* Get the toplevel node to start the search at */
	node = edv_vfs_browser_tree_get_toplevel_node(browser);

	/* Begin the search */
	matched_node = NULL;
	edv_vfs_browser_tree_find_by_indexIterate(
		ctree,
		node,
		device_index, index,
		&matched_node
	);

	return(matched_node);
}

/*
 *	Performs one iteration of finding the Directory Tree node by
 *	path.
 */
static void edv_vfs_browser_tree_find_by_pathIterate(
	GtkCTree *ctree,
	GtkCTreeNode *node,
	const gchar *path,
	GtkCTreeNode **matched_node
)
{
	const gchar *cur_path;
	EDVVFSObject *obj;

	if((*matched_node != NULL) || (node == NULL))
		return;

	obj = EDV_VFS_OBJECT(gtk_ctree_node_get_row_data(ctree, node));
	if(obj == NULL)
		return;

	/* Get this object's full path */
	cur_path = obj->path;
	if(STRISEMPTY(cur_path))
		return;

	/* Is the specified node's path a child of the specified path? */
	if(edv_path_is_parent(
		cur_path,				/* Parent */
		path				/* Child */
	))
	{
		/* Yes it is, now check if it is a complete match */
		if(!strcmp((const char *)path, (const char *)cur_path))
		{
			*matched_node = node;
			return;
		}
		else
		{
			/* Not a complete match, check the child nodes */
			node = edv_node_get_child(node);
			while(node != NULL)
			{
				edv_vfs_browser_tree_find_by_pathIterate(
					ctree,
					node,
					path,
					matched_node
				);
				if(*matched_node != NULL)
				    return;

				/* Get the sibling */
				node = edv_node_get_sibling(node);
			}
		}
	}
}

/*
 *	Finds the node by path.
 *
 *	The path specifies the string describing the full path to
 *	match a node that has an object who has the same path.
 *
 *	Returns the matched node or NULL on error.
 */
GtkCTreeNode *edv_vfs_browser_tree_find_by_path(
	EDVVFSBrowser *browser,
	const gchar *path
)
{
	gchar *full_path;
	GtkCTreeNode *node, *matched_node;
	GtkCTree *ctree;

	if((browser == NULL) || STRISEMPTY(path))
		return(NULL);

	ctree = GTK_CTREE(browser->directory_ctree);

	/* Must have a full path */
	if(!g_path_is_absolute(path))
		return(NULL);

	/* Copy the path */
	full_path = g_strdup(path);
	if(full_path == NULL)
		return(NULL);

	/* Remove any tailing deliminators */
	edv_path_simplify(full_path);

	/* Get the toplevel node to start search from */
	node = edv_vfs_browser_tree_get_toplevel_node(browser);

	/* Begin searching for a node who's object's fuil path matches
	 * the specified path
	 *
	 * matched_node_ptr will point to the matched node if a match
	 * was made
	 */
	matched_node = NULL;
	edv_vfs_browser_tree_find_by_pathIterate(
		ctree,
		node,
		full_path,
		&matched_node
	);

	g_free(full_path);

	return(matched_node);
}


/*
 *	Removes all the grandchild nodes.
 *
 *	The node specifies the parent node.
 */
void edv_vfs_browser_tree_remove_grand_children_nodes(
	EDVVFSBrowser *browser, GtkCTreeNode *node
)
{
	GtkCList *clist;
	GtkCTreeRow *row_ptr;
	GtkCTree *ctree;

	if((browser == NULL) || (node == NULL))
		return;

	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);

	row_ptr = GTK_CTREE_ROW(node);
	if(row_ptr == NULL)
		return;

	/* Get the specified node's childrens list */
	node = row_ptr->children;

	/* No children? */
	if(node == NULL)
		return;

	gtk_clist_freeze(clist);

	/* Iterate through the specified node's childrens list */
	while(node != NULL)
	{
		row_ptr = GTK_CTREE_ROW(node);
		if(row_ptr == NULL)
			break;

		/* Remove any child nodes in this node */
		while(row_ptr->children != NULL)
			gtk_ctree_remove_node(ctree, row_ptr->children);

		/* Get the next node */
		node = row_ptr->sibling;
	}

	gtk_clist_thaw(clist);
}


/*
 *	Open with.
 */
void edv_vfs_browser_tree_open_with(
	EDVVFSBrowser *browser,
	GtkCTreeNode *node
)
{
	const gchar *name;
	gchar *full_path;
	GtkWidget *toplevel;
	GtkCTree *ctree;
	EDVVFSObject	*obj,
					*obj_values;
	EDVCore *core;

	if((browser == NULL) || (node == NULL))
		return;

	toplevel = browser->toplevel;
	ctree = GTK_CTREE(browser->directory_ctree);
	core = browser->core;
	obj = EDV_VFS_OBJECT(gtk_ctree_node_get_row_data(ctree, node));
	if(obj == NULL)
		return;

	/* Check the name of the object for special notations */
	name = obj->name;
	if(name != NULL)
	{
		if(!strcmp((const char *)name, "."))
		{
			full_path = NULL;
		}
		else if(!strcmp((const char *)name, ".."))
		{
			full_path = g_dirname(
				edv_vfs_browser_get_location(browser)
			);
		}
		else
		{
			full_path = STRDUP(obj->path);
		}
	}
	else
	{
		/* No name available, use the path */
		full_path = STRDUP(obj->path);
	}
	if(full_path == NULL)
		return;

	/* Get the destination object's statistics, the object
	 * structure may have that information already but we need to
	 * have the most up to date information in case the object's
	 * type has changed
	 */
	obj_values = edv_vfs_object_stat(full_path);
	if(obj_values == NULL)
	{
		g_free(full_path);
		return;
	}

	edv_vfs_object_delete(obj_values);

	/* Confirm open */
	if(edv_confirm_open(
		core,
		toplevel,
		full_path,
		1
	) != CDIALOG_RESPONSE_YES)
	{
		g_free(full_path);
		return;
	}

	/* Open with */
	if(TRUE)
	{
		GList *paths_list = NULL;

		paths_list = g_list_append(
			paths_list,
			g_strdup(full_path)
		);

		(void)edv_open_with(
			core,
			paths_list,
			NULL,			/* Default command */
			toplevel,
			TRUE			/* Verbose */
		);

		g_list_foreach(
			paths_list,
			(GFunc)g_free,
			NULL
		);
		g_list_free(paths_list);
	}

	g_free(full_path);
}


/*
 *	FPrompt rename apply callback.
 */
static void EDVBrowserDirTreeFPromptRenameApplyCB(
	gpointer data, const gchar *value
)
{
	GtkCTreeNode *node;
	EDVVFSBrowser *browser;
	EDVBrowserDirTreeFPromptData *d = EDV_VFS_BROWSER_DIR_TREE_FPROMPT_DATA(data);
	if(d == NULL)
		return;

	browser = d->browser;
	node = d->node;

	/* Inputs valid? */
	if((browser != NULL) && (node != NULL) && (value != NULL))
	{
		GtkWidget	*toplevel = browser->toplevel,
					*sb = browser->status_bar;
		GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
		EDVCore *core = browser->core;

		/* Get the object from the selected node */
		EDVVFSObject *obj = EDV_VFS_OBJECT(
			gtk_ctree_node_get_row_data(ctree, node)
		);
		if((obj != NULL) ? (obj->path != NULL) : FALSE)
		{
			gboolean yes_to_all = FALSE;
			const gchar *error_msg;
			gchar *old_full_path = g_strdup(obj->path);
			GList *modified_paths_list;

			/* Rename */
			edv_vfs_object_op_rename(
				core,
				old_full_path, value,
				&modified_paths_list,
				toplevel,
				FALSE,			/* Do not show progress */
				TRUE,			/* Interactive */
				&yes_to_all
			);

			/* Unmap the progress dialog */
			ProgressDialogBreakQuery(FALSE);
			ProgressDialogSetTransientFor(NULL);

			/* Check for errors */
			error_msg = edv_vfs_object_op_get_error(core);
			if(!STRISEMPTY(error_msg))
			{
				/* Report the error */
				edv_play_sound_error(core);
				edv_message_error(
					"Rename Error",
					error_msg,
					NULL,
					toplevel
				);
			}

			/* Notify about the renamed objects */
			if(modified_paths_list != NULL)
			{
				const gchar *modified_path;
				GList *glist;
				EDVVFSObject *obj;

				for(glist = modified_paths_list;
					glist != NULL;
					glist = g_list_next(glist)
				)
				{
					modified_path = (const gchar *)glist->data;
					if(modified_path == NULL)
						continue;

					obj = edv_vfs_object_lstat(modified_path);
					if(obj != NULL)
					{
						gchar *msg = g_strdup_printf(
							"Object \"%s\" renamed to \"%s\"",
							g_basename(old_full_path),
							g_basename(modified_path)
						);
						edv_status_bar_message(sb, msg, FALSE);
						g_free(msg);

						edv_emit_vfs_object_modified(
							core,
							old_full_path,
							modified_path,
							obj
						);

						edv_vfs_object_delete(obj);
					}
				}

				if(modified_paths_list != NULL)
				{
					g_list_foreach(
						modified_paths_list, (GFunc)g_free, NULL
					);
					g_list_free(modified_paths_list);
				}
			}
			else
			{
				/* Did not get the modified object path so this
				 * implies failure
				 */
				edv_status_bar_message(
					sb,
					"Rename object failed",
					FALSE
				);
			}

			g_free(old_full_path);
		}
	}

	g_free(d);
}

/*
 *	FPrompt rename cancel callback.
 */
static void EDVBrowserDirTreeFPromptRenameCancelCB(gpointer data)
{
	EDVBrowserDirTreeFPromptData *d = EDV_VFS_BROWSER_DIR_TREE_FPROMPT_DATA(data);
	if(d == NULL)
		return;

	g_free(d);
}

/*
 *	Maps the FPrompt over the File Browser directory tree item
 *	to rename.
 */
void edv_vfs_browser_tree_rename_query(
	EDVVFSBrowser *browser,
	GtkCTreeNode *node
)
{
	gint row, column;
	gint cx, cy, px, py, pwidth, pheight;
	GtkWidget *toplevel;
	GtkCList *clist;
	GtkCTree *ctree;
	EDVVFSObject *obj;
	EDVCore *core;

	if((browser == NULL) || (node == NULL) || FPromptIsQuery())
		return;

	toplevel = browser->toplevel;
	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);
	core = browser->core;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
		return;

	edv_vfs_browser_sync_data(browser);

	/* Get the row and column of the specified node */
	if(!edv_ctree_node_get_index(ctree, node, &row, &column))
		return;

	/* Get the cell's geometry */
	if(!gtk_clist_get_cell_geometry(
		clist, column, row,
		&cx, &cy, &pwidth, &pheight
	))
		return;

	/* Get the root window relative coordinates */
	px = 0;
	py = 0;
	gdk_window_get_deskrelative_origin(
		clist->clist_window, &px, &py
	);
/*	px += cx + 0; */
	py += cy - 2;	/* Move up a little */

	/* Set the prompt width to match width of the GtkCTree */
	pwidth = GTK_WIDGET(ctree)->allocation.width;

	/* Get the object */
	obj = EDV_VFS_OBJECT(gtk_ctree_node_get_row_data(ctree, node));
	if(obj == NULL)
		return;

	/* Check if the object's name is a special notation that
	 * may not be renamed
	 */
	if(obj->name != NULL)
	{
		const gchar *name = obj->name;
		if(!strcmp((const char *)name, ".") ||
		   !strcmp((const char *)name, "..") ||
		   !strcmp((const char *)name, "/")
		)
			return;
	}

	if(TRUE)
	{
		gchar *value = STRDUP(obj->name);
		EDVBrowserDirTreeFPromptData *d = EDV_VFS_BROWSER_DIR_TREE_FPROMPT_DATA(
			g_malloc(sizeof(EDVBrowserDirTreeFPromptData))
		);
		if(d != NULL)
		{
			d->browser = browser;
			d->node = node;
		}

		/* Map floating prompt to change values */
		FPromptSetTransientFor(browser->toplevel);
		FPromptSetPosition(px, py);
		FPromptMapQuery(
			NULL,			/* No label */
			value,
			NULL,			/* No tooltip message */
			FPROMPT_MAP_TO_POSITION,	/* Map code */
			pwidth, -1,		/* Width and height */
			0,			/* No buttons */
			d,			/* Callback data */
			NULL,			/* No browse callback */
			EDVBrowserDirTreeFPromptRenameApplyCB,
			EDVBrowserDirTreeFPromptRenameCancelCB
		);

		g_free(value);
	}
}


/*
 *	Object added callback.
 */
void edv_vfs_browser_tree_vfs_object_added_cb(
	EDVVFSBrowser *browser,
	const gchar *path,
	EDVVFSObject *obj
)
{
	gchar *parent_path;
	GtkCList *clist;
	GtkCTreeNode	*node,
					*parent_node;
	GtkCTree *ctree;
	CfgList *cfg_list;
	EDVCore *core;

	if((browser == NULL) || STRISEMPTY(path) || (obj == NULL))
		return;

	browser->stop_count = 0;

	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);
	core = browser->core;
	cfg_list = core->cfg_list;

	/* Skip if the added object's destination is not a directory */
	if(!edv_path_is_directory(path))
		return;

	/* Update the object who's path matches the added path */
	node = edv_vfs_browser_tree_find_by_path(browser, path);
	if(node != NULL)
	{
		/* Update this object */
		EDVVFSObject *tar_obj = EDV_VFS_OBJECT(
			gtk_ctree_node_get_row_data(ctree, node)
		);
		if(tar_obj != NULL)
		{
			GtkCTreeRow *row_ptr = GTK_CTREE_ROW(node);
			const gboolean was_expanded = (row_ptr != NULL) ?
				row_ptr->expanded : FALSE;

			edv_vfs_object_set_object(tar_obj, obj);

			gtk_clist_freeze(clist);

			/* Update the node */
			EDVBrowserDirTreeSetNode(
				browser,
				node,
				tar_obj
			);

			/* Get the new listing of child objects for the
			 * updated node, recurse no more than 2 levels
			 */
			if(EDVBrowserDirTreeIsPathNoScan(core, path))
				EDVBrowserDirTreeInsertDummyNode(
					browser,
					node,
					NULL
				);
			else
				EDVBrowserDirTreeGetChildrenList(
					browser,
					node,			/* Parent node */
					EDV_GET_B(EDV_CFG_PARM_LISTS_SHOW_PROGRESS),
					TRUE			/* Recurse 2 levels */
				);

			if(was_expanded)
				gtk_ctree_expand(ctree, node);

			gtk_clist_thaw(clist);
		}
		return;
	}

	/* Get the added object's parent */
	parent_path = g_dirname(path);
	if(parent_path == NULL)
		return;

	/* Get the parent node of the added object */
	parent_node = edv_vfs_browser_tree_find_by_path(browser, parent_path);
	if(parent_node != NULL)
	{
		gtk_clist_freeze(clist);

		/* Create the new node */
		node = EDVBrowserDirTreeInsertNode(
			browser,
			parent_node,
			NULL,				/* No sibling, append to
											 * parent_node */
			edv_vfs_object_copy(obj)
		);
		if(node != NULL)
		{
			/* Get the new listing of child objects for the new
			 * node, recursing no more than 2 levels
			 */
			if(EDVBrowserDirTreeIsPathNoScan(core, path))
				EDVBrowserDirTreeInsertDummyNode(
					browser,
					node,
					NULL
				);
			else
				EDVBrowserDirTreeGetChildrenList(
					browser,
					node,			/* Parent node */
					EDV_GET_B(EDV_CFG_PARM_LISTS_SHOW_PROGRESS),
					TRUE			/* Recurse 2 levels */
				);
		}

		gtk_clist_thaw(clist);
	}

	g_free(parent_path);
}

/*
 *	Object modified callback.
 */
void edv_vfs_browser_tree_vfs_object_modified_cb(
	EDVVFSBrowser *browser,
	const gchar *path,
	const gchar *new_path,
	EDVVFSObject *obj
)
{
	GtkCList *clist;
	GtkCTreeNode *node;
	GtkCTree *ctree;
	CfgList *cfg_list;
	EDVCore *core;
	if((browser == NULL) || STRISEMPTY(path) || (obj == NULL))
		return;

	browser->stop_count = 0;

	if(new_path == NULL)
		new_path = path;

	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);
	core = browser->core;
	cfg_list = core->cfg_list;

	/* Check if the object's destination is no longer a directory */
	if(!edv_path_is_directory(new_path))
	{
		/* Remove all the nodes who's object's path matches the
		 * modified path
		 */
		GtkCTreeNode *node = edv_vfs_browser_tree_find_by_path(browser, path);
		if(node != NULL)
		{
			gtk_clist_freeze(clist);
			do {
				gtk_ctree_remove_node(ctree, node);
				node = edv_vfs_browser_tree_find_by_path(browser, path);
			} while(node != NULL);
			gtk_clist_thaw(clist);
		}
		return;
	}

	/* Update the node who's object's path matches the modified
	 * path
	 */
	node = edv_vfs_browser_tree_find_by_path(browser, path);
	if(node != NULL)
	{
		/* Update this object */
		EDVVFSObject *tar_obj = EDV_VFS_OBJECT(
			gtk_ctree_node_get_row_data(ctree, node)
		);
		if(tar_obj != NULL)
		{
			GtkCTreeRow *row_ptr = GTK_CTREE_ROW(node);
			const gboolean was_expanded = (row_ptr != NULL) ?
				row_ptr->expanded : FALSE;

			edv_vfs_object_set_object(tar_obj, obj);

			gtk_clist_freeze(clist);

			/* Update this node */
			EDVBrowserDirTreeSetNode(
				browser,
				node,
				tar_obj
			);

			/* Get the new listing of child objects for the updated
			 * node, recurse no more than 2 levels
			 */
			if(EDVBrowserDirTreeIsPathNoScan(core, path))
				EDVBrowserDirTreeInsertDummyNode(
					browser,
					node,
					NULL
				);
			else
				EDVBrowserDirTreeGetChildrenList(
					browser,
					node,			/* Parent node */
					EDV_GET_B(EDV_CFG_PARM_LISTS_SHOW_PROGRESS),
					TRUE			/* Recurse 2 levels */
				);

			if(was_expanded)
				gtk_ctree_expand(ctree, node);

			gtk_clist_thaw(clist);
		}
	}
	else
	{
		/* No node existed for the modified path, create a new
		 * node for the new path
		 */
		gchar *new_parent_path = g_dirname(new_path);
		if(new_parent_path != NULL)
		{
			/* Find the node for the new path's parent */
			GtkCTreeNode *parent_node = edv_vfs_browser_tree_find_by_path(
				browser, new_parent_path
			);
			if(parent_node != NULL)
			{
				/* Create a new node for the new path */
				gtk_clist_freeze(clist);

				/* Add/transfer the object to the tree */
				node = EDVBrowserDirTreeInsertNode(
					browser,
					parent_node,
					NULL,			/* No sibling, append to
											 * parent_node */
					edv_vfs_object_copy(obj)
				);
				if(node != NULL)
				{
					/* Get the new listing of child objects for
					 * the new node, recurse no more than 2 levels
					 */
					if(EDVBrowserDirTreeIsPathNoScan(core, path))
						EDVBrowserDirTreeInsertDummyNode(
							browser,
							node,
							NULL
						);
					else
						EDVBrowserDirTreeGetChildrenList(
							browser,
							node,		/* Parent node */
							EDV_GET_B(EDV_CFG_PARM_LISTS_SHOW_PROGRESS),
							TRUE		/* Recurse 2 levels */
						);
				}

				gtk_clist_thaw(clist);
			}
			g_free(new_parent_path);
		}
	}
}

/*
 *	Object removed callback.
 */
void edv_vfs_browser_tree_vfs_object_removed_cb(
	EDVVFSBrowser *browser,
	const gchar *path
)
{
	GtkCList *clist;
	GtkCTreeNode *node;
	GtkCTree *ctree;
	EDVCore *core;

	if((browser == NULL) || STRISEMPTY(path))
		return;

	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);
	core = browser->core;
 
	/* Remove all the nodes that match the removed path */
	node = edv_vfs_browser_tree_find_by_path(browser, path);
	if(node != NULL)
	{
		gtk_clist_freeze(clist);
		do {
			gtk_ctree_remove_node(ctree, node);
			node = edv_vfs_browser_tree_find_by_path(browser, path);
		} while(node != NULL);
		gtk_clist_thaw(clist);
	}
}


/*
 *	Mount/unmount callback.
 */
void edv_vfs_browser_tree_vfs_object_device_mount_cb(
	EDVVFSBrowser *browser,
	EDVDevice *d,
	const gboolean mounted
)
{
	gchar *mount_path;
	GtkCList *clist;
	GtkCTreeNode *node;
	GtkCTree *ctree;
	CfgList *cfg_list;
	EDVCore *core;

	if((browser == NULL) || (d == NULL))
		return;

	browser->stop_count = 0;

	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);
	core = browser->core;
	cfg_list = core->cfg_list;

	/* Get the mount path */
	mount_path = STRDUP(d->mount_path);
	if(mount_path == NULL)
		return;

	/* Simplify the mount path */
	edv_path_simplify(mount_path);

	/* Find the node that matches this mount path */
	node = edv_vfs_browser_tree_find_by_path(browser, mount_path);
	if(node != NULL)
	{
		/* Update this node and its object */
		EDVVFSObject *obj = EDV_VFS_OBJECT(gtk_ctree_node_get_row_data(
			ctree,
			node
		));
		if(obj != NULL)
		{
			EDVVFSObject *obj_values = edv_vfs_object_lstat(mount_path);
			if(obj_values != NULL)
			{
				edv_vfs_object_set_object(
					obj,
					obj_values
				);

				gtk_clist_freeze(clist);

				EDVBrowserDirTreeSetNode(
					browser,
					node,
					obj
				);

				/* Get the new listing of child objects for the
				 * updated node, recurse no more than 2 levels
				 */
				if(EDVBrowserDirTreeIsPathNoScan(core, mount_path))
					EDVBrowserDirTreeInsertDummyNode(
						browser,
						node,
						NULL
					);
				else
					EDVBrowserDirTreeGetChildrenList(
						browser,
						node,		/* Parent node */
						EDV_GET_B(EDV_CFG_PARM_LISTS_SHOW_PROGRESS),
						TRUE		/* Recurse 2 levels */
					);

				gtk_clist_thaw(clist);

				edv_vfs_object_delete(obj_values);
			}
		}
	}

	/* Unmounted? */
	if(!mounted)
	{
		/* Is the current location within the mount path? */
		gchar *cur_path = STRDUP(edv_vfs_browser_get_location(browser));
		if(cur_path != NULL)
		{
			if(strpfx((const char *)cur_path, (const char *)mount_path))
			{
				/* Select the unmounted device's node as needed */
				GtkCTreeNode *cur_node = edv_vfs_browser_tree_find_by_path(
					browser,
					cur_path
				);
				if((node != NULL) && (node != cur_node))
				{
					gtk_clist_freeze(clist);
					gtk_ctree_select(ctree, node);
					gtk_clist_thaw(clist);
				}
			}
			g_free(cur_path);
		}
	}

	g_free(mount_path);
}
