#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <gtk/gtk.h>

#include "url.h"
#include "cfg.h"

#include "guiutils.h"
#include "tool_bar.h"
#include "cdialog.h"
#include "fb.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_recycled_obj.h"
#include "libendeavour2-base/edv_recycled_obj_stat.h"
#include "edv_recycle_obj.h"
#include "edv_device.h"
#include "edv_obj_info_match.h"
#include "edv_utils_gtk.h"
#include "edv_device_mount.h"
#include "edv_devices_list.h"
#include "edv_mount_bar.h"
#include "edv_find_bar.h"
#include "edv_status_bar.h"
#include "obj_op_dlg.h"
#include "edv_confirm.h"
#include "vfs_browser.h"
#include "vfs_browser_cb.h"
#include "vfs_browser_op.h"
#include "vfs_browser_list.h"
#include "vfs_browser_tree.h"
#include "vfs_browser_list.h"
#include "edv_vfs_obj_create.h"
#include "edv_vfs_obj_op.h"
#include "edv_help.h"
#include "edv_emit.h"
#include "edv_op.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


void edv_vfs_browser_op_cb(
	ToolBarItem *item,
	const gint id,
	gpointer data
);
void edv_vfs_browser_op_enter_cb(
	ToolBarItem *item,
	const gint id,
	gpointer data
);
void edv_vfs_browser_op_leave_cb(
	ToolBarItem *item,
	const gint id,
	gpointer data
);

void edv_vfs_browser_op_close(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_exit(EDVVFSBrowser *browser);

static void edv_vfs_browser_op_sync(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_run(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_master_write_protect(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_delete_method_recycle(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_delete_method_purge(EDVVFSBrowser *browser);

static void edv_vfs_browser_op_new_object_nexus(
	EDVVFSBrowser *browser,
	const EDVObjectType type,
	GtkWidget *toplevel
);
static void edv_vfs_browser_op_new_object_menu(
	EDVVFSBrowser *browser,
	ToolBarItem *item
);
static void edv_vfs_browser_op_new_file(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_new_directory(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_new_link(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_new_fifo(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_new_device_block(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_new_device_character(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_new_socket(EDVVFSBrowser *browser);

static void edv_vfs_browser_op_open(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_open_to(EDVVFSBrowser *browser, ToolBarItem *item);
static void edv_vfs_browser_op_open_with(EDVVFSBrowser *browser);

static void edv_vfs_browser_op_copy_path(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_copy_url(EDVVFSBrowser *browser);
void edv_vfs_browser_op_paste(EDVVFSBrowser *browser);
void edv_vfs_browser_op_paste2(
	EDVVFSBrowser *browser,
	EDVVFSObject *obj
);

static void edv_vfs_browser_op_move(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_copy(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_link(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_rename(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_chmod(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_chown(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_chtime(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_delete(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_properties(EDVVFSBrowser *browser);

static void edv_vfs_browser_op_select_all(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_unselect_all(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_invert_selection(EDVVFSBrowser *browser);

static void edv_vfs_browser_op_download(EDVVFSBrowser *browser);

void edv_vfs_browser_op_refresh(EDVVFSBrowser *browser);
void edv_vfs_browser_op_refresh_all(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_goto_parent(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_goto_home(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_expand(EDVVFSBrowser *browser);

static void edv_vfs_browser_op_list_filter(EDVVFSBrowser *browser);

static void edv_vfs_browser_op_directory_tree_origin(EDVVFSBrowser *browser);

static void edv_vfs_browser_op_mount(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_eject(EDVVFSBrowser *browser);

static void edv_vfs_browser_op_mime_types(EDVVFSBrowser *browser);
static void edv_vfs_browser_op_devices(EDVVFSBrowser *browser);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Operation ID callback nexus.
 *
 *	The data must be a EDVVFSBrowserOp *.
 */
void edv_vfs_browser_op_cb(
	ToolBarItem *item,
	const gint id,
	gpointer data  
)
{
	gint dev_num;
	GtkWidget *toplevel;
	CfgList *cfg_list;
	EDVDevice *dev;
	EDVVFSBrowser *browser;
	EDVCore *core;
	EDVVFSBrowserOp *opid = EDV_VFS_BROWSER_OP(data);
	if(opid == NULL)
		return;

	browser = opid->vfs_browser;
	if((browser == NULL) || (opid->flags & EDV_OPID_NO_OP))
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser) || (browser->freeze_count > 0))
		return;

	toplevel = browser->toplevel;
	core = browser->core;
	cfg_list = core->cfg_list;

	browser->freeze_count++;

	/* Get last selected device (if any) */
	dev_num = browser->selected_dev_num;
	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));

	/* Handle by operation id code */
	switch(opid->id)
	{
	  case EDV_VFS_BROWSER_OP_NONE:
	  case EDV_VFS_BROWSER_OP_SEPARATOR:
		break;

	  case EDV_VFS_BROWSER_OP_CLOSE:
		edv_vfs_browser_op_close(browser);
		break;
	  case EDV_VFS_BROWSER_OP_EXIT:
		edv_vfs_browser_op_exit(browser);
		break;

	  case EDV_VFS_BROWSER_OP_SYNC_DISKS:
		edv_vfs_browser_op_sync(browser);
		break;
	  case EDV_VFS_BROWSER_OP_HISTORY:
		edv_map_history(
			core,
			-1,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL,
			toplevel
		);
		break;
	  case EDV_VFS_BROWSER_OP_RUN:
		edv_vfs_browser_op_run(browser);
		break;
	  case EDV_VFS_BROWSER_OP_RUN_TERMINAL:
		if(core != NULL)
		{
			gchar *working_dir = STRDUP(edv_vfs_browser_get_location(browser));
			edv_run_terminal(
				core,
				NULL,
				working_dir,
				toplevel
			);
			g_free(working_dir);
		}
		break;
	  case EDV_VFS_BROWSER_OP_WRITE_PROTECT:
		edv_vfs_browser_op_master_write_protect(browser);
		break;
	  case EDV_VFS_BROWSER_OP_DELETE_METHOD_RECYCLE:
		edv_vfs_browser_op_delete_method_recycle(browser);
		break;
	  case EDV_VFS_BROWSER_OP_DELETE_METHOD_PURGE:
		edv_vfs_browser_op_delete_method_purge(browser);
		break;

	  case EDV_VFS_BROWSER_OP_NEW:
		edv_vfs_browser_op_new_object_menu(browser, item);
		break;
	  case EDV_VFS_BROWSER_OP_NEW_FILE:
		edv_vfs_browser_op_new_file(browser);
		break;
	  case EDV_VFS_BROWSER_OP_NEW_DIRECTORY:
		edv_vfs_browser_op_new_directory(browser);
		break;
	  case EDV_VFS_BROWSER_OP_NEW_LINK:
		edv_vfs_browser_op_new_link(browser);
		break;
	  case EDV_VFS_BROWSER_OP_NEW_FIFO:
		edv_vfs_browser_op_new_fifo(browser);
		break;
	  case EDV_VFS_BROWSER_OP_NEW_DEVICE_BLOCK:
		edv_vfs_browser_op_new_device_block(browser);
		break;
	  case EDV_VFS_BROWSER_OP_NEW_DEVICE_CHARACTER:
		edv_vfs_browser_op_new_device_character(browser);
		break;
	  case EDV_VFS_BROWSER_OP_NEW_SOCKET:
		edv_vfs_browser_op_new_socket(browser);
		break;

	  case EDV_VFS_BROWSER_OP_OPEN:
		edv_vfs_browser_op_open(browser);
		break;
	  case EDV_VFS_BROWSER_OP_OPEN_TO:
		edv_vfs_browser_op_open_to(browser, item);
		break;
	  case EDV_VFS_BROWSER_OP_OPEN_WITH:
		edv_vfs_browser_op_open_with(browser);
		break;

	  case EDV_VFS_BROWSER_OP_COPY_PATH:
		edv_vfs_browser_op_copy_path(browser);
		break;
	  case EDV_VFS_BROWSER_OP_COPY_URL:
		edv_vfs_browser_op_copy_url(browser);
		break;
	  case EDV_VFS_BROWSER_OP_PASTE:
		edv_vfs_browser_op_paste(browser);
		break;

	  case EDV_VFS_BROWSER_OP_MOVE:
		edv_vfs_browser_op_move(browser);
		break;
	  case EDV_VFS_BROWSER_OP_COPY:
		edv_vfs_browser_op_copy(browser);
		break;
	  case EDV_VFS_BROWSER_OP_LINK:
		edv_vfs_browser_op_link(browser);
		break;
	  case EDV_VFS_BROWSER_OP_RENAME:
		edv_vfs_browser_op_rename(browser);
		break;
	  case EDV_VFS_BROWSER_OP_CHMOD:
		edv_vfs_browser_op_chmod(browser);
		break;
	  case EDV_VFS_BROWSER_OP_CHOWN:
		edv_vfs_browser_op_chown(browser);
		break;
	  case EDV_VFS_BROWSER_OP_CHTIME:
		edv_vfs_browser_op_chtime(browser);
		break;
	  case EDV_VFS_BROWSER_OP_DELETE:
		edv_vfs_browser_op_delete(browser);
		break;
	  case EDV_VFS_BROWSER_OP_PROPERTIES:
		edv_vfs_browser_op_properties(browser);
		break;

	  case EDV_VFS_BROWSER_OP_SELECT_ALL:
		edv_vfs_browser_op_select_all(browser);
		break;
	  case EDV_VFS_BROWSER_OP_UNSELECT_ALL:
		edv_vfs_browser_op_unselect_all(browser);
		break;
	  case EDV_VFS_BROWSER_OP_INVERT_SELECTION:
		edv_vfs_browser_op_invert_selection(browser);
		break;
	  case EDV_VFS_BROWSER_OP_FIND:
		edv_map_find_vfs(core, browser);
		break;

	  case EDV_VFS_BROWSER_OP_DOWNLOAD:
		edv_vfs_browser_op_download(browser);
		break;

	  case EDV_VFS_BROWSER_OP_REFRESH:
		edv_vfs_browser_op_refresh(browser);
		break;
	  case EDV_VFS_BROWSER_OP_REFRESH_ALL:
		edv_vfs_browser_op_refresh_all(browser);
		break;
	  case EDV_VFS_BROWSER_OP_GOTO_PARENT:
		edv_vfs_browser_op_goto_parent(browser);
		break;
	  case EDV_VFS_BROWSER_OP_GOTO_HOME:
		edv_vfs_browser_op_goto_home(browser);
		break;
	  case EDV_VFS_BROWSER_OP_EXPAND:
		edv_vfs_browser_op_expand(browser);
		break;

	  case EDV_VFS_BROWSER_OP_SHOW_TOOL_BAR:
		if(core != NULL)
		{
			const gboolean state = !EDV_GET_B(
				EDV_CFG_PARM_BROWSER_SHOW_TOOL_BAR
			);
			EDV_SET_B(
				EDV_CFG_PARM_BROWSER_SHOW_TOOL_BAR,
				state
			);
			edv_queue_emit_reconfigured(core);
		}
		break;
	  case EDV_VFS_BROWSER_OP_SHOW_LOCATION_BAR:
		if(core != NULL)
		{
			const gboolean state = !EDV_GET_B(
				EDV_CFG_PARM_BROWSER_SHOW_LOCATION_BAR
			);
			EDV_SET_B(
				EDV_CFG_PARM_BROWSER_SHOW_LOCATION_BAR,
				state
			);
			edv_queue_emit_reconfigured(core);
		}
		break;
	  case EDV_VFS_BROWSER_OP_SHOW_MOUNT_BAR:
		if(core != NULL)
		{
			const gboolean state = !EDV_GET_B(
				EDV_CFG_PARM_BROWSER_SHOW_MOUNT_BAR
			);
			EDV_SET_B(
				EDV_CFG_PARM_BROWSER_SHOW_MOUNT_BAR,
				state
			);
			edv_queue_emit_reconfigured(core);
		}
		break;
	  case EDV_VFS_BROWSER_OP_SHOW_FIND_BAR:
		if(core != NULL)
		{
			const gboolean state = !EDV_GET_B(
				EDV_CFG_PARM_BROWSER_SHOW_FIND_BAR
			);
			EDV_SET_B(
				EDV_CFG_PARM_BROWSER_SHOW_FIND_BAR,
				state
			);
			edv_queue_emit_reconfigured(core);
		}
		break;
	  case EDV_VFS_BROWSER_OP_SHOW_STATUS_BAR:
		if(core != NULL)
		{
			const gboolean state = !EDV_GET_B(
				EDV_CFG_PARM_BROWSER_SHOW_STATUS_BAR
			);
			EDV_SET_B(
				EDV_CFG_PARM_BROWSER_SHOW_STATUS_BAR,
				state
			);
			edv_queue_emit_reconfigured(core);
		}
		break;

	  case EDV_VFS_BROWSER_OP_LIST_FILTER:
		edv_vfs_browser_op_list_filter(browser);
		break;

	  case EDV_VFS_BROWSER_OP_CONTENTS_LIST_AUTO_RESIZE_COLUMNS:
		if(core != NULL)
		{
			const gboolean state = !EDV_GET_B(
				EDV_CFG_PARM_BROWSER_CONTENTS_LIST_AUTO_RESIZE_COLUMNS
			);
			EDV_SET_B(
				EDV_CFG_PARM_BROWSER_CONTENTS_LIST_AUTO_RESIZE_COLUMNS,
				state
			);
			edv_queue_emit_reconfigured(core);
		}
		break;

	  case EDV_VFS_BROWSER_OP_SHOW_HIDDEN_OBJECTS:
		if(core != NULL)
		{
			const gboolean state = !EDV_GET_B(
				EDV_CFG_PARM_BROWSER_SHOW_OBJECT_HIDDEN
			);
			EDV_SET_B(
				EDV_CFG_PARM_BROWSER_SHOW_OBJECT_HIDDEN,
				state
			);
			edv_vfs_browser_op_refresh(browser);
		}
		break;
	  case EDV_VFS_BROWSER_OP_SHOW_NOACCESS_OBJECTS:
		if(core != NULL)
		{
			const gboolean state = !EDV_GET_B(
				EDV_CFG_PARM_BROWSER_SHOW_OBJECT_NOACCESS
			);
			EDV_SET_B(
				EDV_CFG_PARM_BROWSER_SHOW_OBJECT_NOACCESS,
				state
			);
			edv_vfs_browser_op_refresh(browser);
		}
		break;
	  case EDV_VFS_BROWSER_OP_DIRECTORY_TREE_ORIGIN:
		edv_vfs_browser_op_directory_tree_origin(browser);
		break;

	  case EDV_VFS_BROWSER_OP_MOUNT:
		edv_vfs_browser_op_mount(browser);
		break;
	  case EDV_VFS_BROWSER_OP_EJECT:
		edv_vfs_browser_op_eject(browser);
		break;
	  case EDV_VFS_BROWSER_OP_DEVICE_CHECK:
		edv_run_device_check(core, dev, toplevel);
		break;
	  case EDV_VFS_BROWSER_OP_DEVICE_TOOLS:
		edv_run_device_tools(core, dev, toplevel);
		break;
	  case EDV_VFS_BROWSER_OP_DEVICE_FORMAT:
		edv_run_device_format(core, dev, toplevel);
		break;

	  case EDV_VFS_BROWSER_OP_MIME_TYPES:
		edv_vfs_browser_op_mime_types(browser);
		break;
	  case EDV_VFS_BROWSER_OP_DEVICES:
		edv_vfs_browser_op_devices(browser);
		break;

	  case EDV_VFS_BROWSER_OP_NEW_BROWSER:
		edv_new_vfs_browser(
			core,
			edv_vfs_browser_get_location(browser),
			browser->directory_ctree_origin_path,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL
		);
		break;

	  case EDV_VFS_BROWSER_OP_NEW_IMBR:
		edv_new_image_browser(
			core,
			edv_vfs_browser_get_location(browser),
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL
		);
		break;

	  case EDV_VFS_BROWSER_OP_NEW_ARCHIVER:
		edv_new_archiver(
			core,
			NULL,
			NULL,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL
		);
		break;

	  case EDV_VFS_BROWSER_OP_RECYCLE_BIN:
		edv_map_recycle_bin(
			core,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL
		);
		break;

	  case EDV_VFS_BROWSER_OP_OPTIONS:
		edv_map_options(
			core,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL,
			toplevel
		);
		break;
	  case EDV_VFS_BROWSER_OP_CUSTOMIZE:
		edv_map_customize(
			core,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL,
			toplevel
		);
		break;

	  case EDV_VFS_BROWSER_OP_HELP_CONTENTS:
		edv_help(core, "Contents", toplevel);
		break;
	  case EDV_VFS_BROWSER_OP_HELP_FILE_BROWSER:
		edv_help(core, "File Browser", toplevel);
		break;
	  case EDV_VFS_BROWSER_OP_HELP_IMAGE_BROWSER:
		edv_help(core, "Image Browser", toplevel);
		break;
	  case EDV_VFS_BROWSER_OP_HELP_ARCHIVER:
		edv_help(core, "Archiver", toplevel);
		break;
	  case EDV_VFS_BROWSER_OP_HELP_RECYCLE_BIN:
		edv_help(core, "Recycle Bin", toplevel);
		break;
	  case EDV_VFS_BROWSER_OP_HELP_KEYS_LIST:
		edv_help(core, "Keys List", toplevel);
		break;
	  case EDV_VFS_BROWSER_OP_HELP_MIME_TYPES:
		edv_help(core, "MIME Types", toplevel);
		break;
	  case EDV_VFS_BROWSER_OP_HELP_DEVICES:
		edv_help(core, "Devices", toplevel);
		break;
	  case EDV_VFS_BROWSER_OP_HELP_COMMON_OPERATIONS:
		edv_help(core, "Common Operations", toplevel);
		break;
	  case EDV_VFS_BROWSER_OP_HELP_ABOUT:
		edv_map_about_dialog(
			core,
			core->geometry_flags, 
			(core->geometry_flags != 0) ? &core->geometry : NULL,
			toplevel
		);
		break;
	}

	browser->freeze_count--;
}

/*
 *	Operation ID enter notify callback nexus.
 *
 *	The data must be a EDVVFSBrowserOp *.
 */
void edv_vfs_browser_op_enter_cb(
	ToolBarItem *item,
	const gint id,
	gpointer data  
)
{
	const gchar *tooltip;
	EDVVFSBrowser *browser;
	EDVVFSBrowserOp *opid = EDV_VFS_BROWSER_OP(data);
	if(opid == NULL)
		return;

	browser = opid->vfs_browser;
	if(browser == NULL)
		return;

	if(browser->freeze_count > 0)
		return;

	tooltip = opid->tooltip;
	if(!STRISEMPTY(tooltip))
		edv_status_bar_message(browser->status_bar, tooltip, FALSE);
}

/*
 *	Operation ID leave notify callback nexus.
 */
void edv_vfs_browser_op_leave_cb(
	ToolBarItem *item,
	const gint id,
	gpointer data
)
{
	EDVVFSBrowser *browser;
	EDVVFSBrowserOp *opid = EDV_VFS_BROWSER_OP(data);
	if(opid == NULL)
		return;

	browser = opid->vfs_browser;
	if(browser == NULL)
		return;

	if(browser->freeze_count > 0)
		return;

	edv_status_bar_message(browser->status_bar, NULL, FALSE);
}


/*
 *	Sync Disks.
 */
void edv_vfs_browser_op_sync(EDVVFSBrowser *browser)
{
	GtkWidget *sb;
	EDVCore *core;

	if(browser == NULL)
		return;

	sb = browser->status_bar;
	core = browser->core;

	edv_vfs_browser_set_busy(browser, TRUE);
	edv_status_bar_message(
		sb,
		"Syncing disks...",
		TRUE
	);

	/* Sync disks */
	edv_sync_edv(core);

	edv_status_bar_message(
		sb,
		"Disk sync done",
		FALSE
	);
	edv_status_bar_progress(sb, 0.0f, FALSE);
	edv_vfs_browser_set_busy(browser, FALSE);
}

/*
 *	Run.
 */
void edv_vfs_browser_op_run(EDVVFSBrowser *browser)
{
	gchar		*s,
			*cmd;
	GList		*glist,
			*objs_list;
	GtkWidget *toplevel;
	EDVVFSObject *obj;
	EDVCore *core;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	toplevel = browser->toplevel;
	core = browser->core;

	cmd = NULL;

	/* Format the run command with the list of selected objects */
	objs_list = edv_vfs_browser_get_selected_objects(
		browser,
		FALSE
	);
	for(glist = objs_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		obj = EDV_VFS_OBJECT(glist->data);
		if(obj == NULL)
			continue;

		if(STRISEMPTY(obj->path))
			continue;

		if(cmd != NULL)
			s = g_strconcat(
				cmd,
				" ",
				obj->path,
				NULL
			);
		else
			s = STRDUP(obj->path);
		g_free(cmd);
		cmd = s;
	}
	g_list_free(objs_list);

	edv_map_run_dialog_command(
		core,
		cmd,
		edv_vfs_browser_get_location(browser),
		toplevel
	);

	g_free(cmd);

	edv_vfs_browser_set_busy(browser, FALSE);
}

/*
 *	Master Write Protect.
 */
void edv_vfs_browser_op_master_write_protect(EDVVFSBrowser *browser)
{
	gboolean state;
	CfgList *cfg_list;
	EDVCore *core;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	core = browser->core;
	cfg_list = core->cfg_list;

	/* Get the current Master Write Protect state */
	state = EDV_GET_B(EDV_CFG_PARM_WRITE_PROTECT);

	/* Toggle the Master Write Protect state */
	state = !state;

	/* Set the new Master Write Protect state */
	EDV_SET_B(EDV_CFG_PARM_WRITE_PROTECT, state);

	/* Notify about the Master Write Protect state change */
	edv_queue_emit_master_write_protect_changed(core);

	edv_vfs_browser_set_busy(browser, FALSE);
}

/*
 *	Delete Method: Recycle.
 */
void edv_vfs_browser_op_delete_method_recycle(EDVVFSBrowser *browser)
{
	CfgList *cfg_list;
	EDVCore *core;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	core = browser->core;
	cfg_list = core->cfg_list;

	EDV_SET_I(
		EDV_CFG_PARM_DELETE_METHOD,
		EDV_DELETE_METHOD_RECYCLE
	);
	edv_queue_emit_delete_method_changed(core);

	edv_vfs_browser_set_busy(browser, FALSE);
}

/*
 *	Delete Method: Purge.
 */
void edv_vfs_browser_op_delete_method_purge(EDVVFSBrowser *browser)
{
	CfgList *cfg_list;
	EDVCore *core;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	core = browser->core;
	cfg_list = core->cfg_list;

	EDV_SET_I(
		EDV_CFG_PARM_DELETE_METHOD,
		EDV_DELETE_METHOD_PURGE
	);
	edv_queue_emit_delete_method_changed(core);

	edv_vfs_browser_set_busy(browser, FALSE);
}


/*
 *	Close.
 */
void edv_vfs_browser_op_close(EDVVFSBrowser *browser)
{
	if(browser == NULL)
		return;

	edv_vfs_browser_sync_configuration(browser);
	edv_vfs_browser_unmap(browser);
}

/*
 *	Exit.
 */
static void edv_vfs_browser_op_exit(EDVVFSBrowser *browser)
{
	EDVCore *core;

	if(browser == NULL)
		return;

	core = browser->core;

	edv_vfs_browser_sync_configuration(browser);
	edv_vfs_browser_unmap(browser);

	/* Schedual a new pending operation on the core to close all
	 * the windows
	 */
	core->pending_flags |= EDV_CORE_PENDING_CLOSE_ALL_WINDOWS;
}


/*
 *	New Object Nexus.
 */
static void edv_vfs_browser_op_new_object_nexus(
	EDVVFSBrowser *browser,
	const EDVObjectType type,
	GtkWidget *toplevel
)
{
	gboolean yes_to_all = FALSE;
	gint status;
	const char *obj_type_name;
	gchar *new_path = NULL;
	const gchar	*cur_path,
			*error_msg;
	EDVCore *core = browser->core;
	if(core == NULL)
		return;

#define CLEANUP_RETURN	{		\
 g_free(new_path);			\
					\
 return;				\
}

	edv_vfs_browser_set_busy(browser, TRUE);

	/* Check and warn if the master write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		CLEANUP_RETURN;
	}

	/* Get current location as the path to create the new object
	 * at
	 */
	cur_path = edv_vfs_browser_get_location(browser);
	if(cur_path == NULL)
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		CLEANUP_RETURN;
	}

	/* Create new object by type */
	status = -1;
	obj_type_name = NULL;
	switch(type)
	{
	  case EDV_OBJECT_TYPE_UNKNOWN:
	  case EDV_OBJECT_TYPE_ERROR:
	  case EDV_OBJECT_TYPE_FILE:
		obj_type_name = "file";
		status = EDVVFSObjectCreateFile(
			core, cur_path, &new_path,
			toplevel, TRUE, TRUE, &yes_to_all
		);
		break;
	  case EDV_OBJECT_TYPE_DIRECTORY:
		obj_type_name = "directory";
		status = EDVVFSObjectCreateDirectory(
			core, cur_path, &new_path,
			toplevel, TRUE, TRUE, &yes_to_all
		);
		break;
	  case EDV_OBJECT_TYPE_LINK:
		obj_type_name = "symbolic link";
		status = EDVVFSObjectCreateLink(
			core, cur_path, &new_path,
			toplevel, TRUE, TRUE, &yes_to_all
		);
		break;
	  case EDV_OBJECT_TYPE_DEVICE_BLOCK:
		obj_type_name = "block device";
		status = EDVVFSObjectCreateDeviceBlock(
			core, cur_path, &new_path,
			toplevel, TRUE, TRUE, &yes_to_all
		);
		break;
	  case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
		obj_type_name = "character device";
		status = EDVVFSObjectCreateDeviceCharacter(
			core, cur_path, &new_path,
			toplevel, TRUE, TRUE, &yes_to_all
		);
		break;
	  case EDV_OBJECT_TYPE_FIFO:
		obj_type_name = "fifo pipe";
		status = EDVVFSObjectCreateFifo(
			core, cur_path, &new_path,
			toplevel, TRUE, TRUE, &yes_to_all
		);
		break;
	  case EDV_OBJECT_TYPE_SOCKET:
		obj_type_name = "socket";
		status = EDVVFSObjectCreateSocket(
			core, cur_path, &new_path,
			toplevel, TRUE, TRUE, &yes_to_all
		);
		break;
	}

	/* Unmap progress dialog since it may have been mapped in
	 * the above operation
	 */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Error creating new object? */
	if(status != 0)
	{
		error_msg = EDVVFSObjectCreateGetError(core);
		if(!STRISEMPTY(error_msg))
		{
			edv_play_sound_error(core);
			edv_message_error(
				"Create Failed",
				error_msg,
				NULL,
				browser->toplevel
			);
		}

		edv_status_bar_message(browser->status_bar, NULL, FALSE);
	}
	else
	{
		/* Successfully created new object */
		gchar *msg;
		EDVVFSObject *obj = edv_vfs_object_lstat(new_path);
		if(obj != NULL)
		{
			gint row;
			GtkCList *clist = GTK_CLIST(browser->contents_clist);

			/* Emit a disk object added signal to all of endeavour's
			 * resources
			 */
			edv_emit_vfs_object_added(
				core,
				new_path,
				obj
			);

			/* Select new row on contents clist that is listing the
			 * new object.
			 */
			row = edv_vfs_browser_list_find_by_path(browser, new_path);
			if((row > -1) && (clist != NULL))
			{
				gtk_clist_freeze(clist);
				gtk_clist_unselect_all(clist);
				gtk_clist_select_row(clist, row, 0);
				gtk_clist_thaw(clist);
			}

			edv_vfs_object_delete(obj);
		}

		msg = g_strdup_printf(
			"Created new %s",
			obj_type_name
		);
		edv_status_bar_message(browser->status_bar, msg, FALSE);
		g_free(msg);
	}

	edv_vfs_browser_set_busy(browser, FALSE);

	CLEANUP_RETURN;
#undef CLEANUP_RETURN
}

/*
 *	New Object.
 */
static void edv_vfs_browser_op_new_object_menu(
	EDVVFSBrowser *browser,
	ToolBarItem *item
)
{
	if(browser == NULL)
		return;

	edv_map_menu_to_button(
		browser->new_object_menu,
		ToolBarItemGetWidget(item)
	);
}

/*
 *	New File.
 */
static void edv_vfs_browser_op_new_file(EDVVFSBrowser *browser)
{
	if(browser == NULL)
		return;

	edv_vfs_browser_op_new_object_nexus(
		browser, EDV_OBJECT_TYPE_FILE, browser->toplevel
	);
}

/*
 *	New Directory.
 */
static void edv_vfs_browser_op_new_directory(EDVVFSBrowser *browser)
{
	if(browser == NULL)
		return;

	edv_vfs_browser_op_new_object_nexus(
		browser, EDV_OBJECT_TYPE_DIRECTORY, browser->toplevel  
	);
}

/*
 *	New Link.
 */
static void edv_vfs_browser_op_new_link(EDVVFSBrowser *browser)
{
	if(browser == NULL)
		return;

	edv_vfs_browser_op_new_object_nexus(
		browser, EDV_OBJECT_TYPE_LINK, browser->toplevel  
	);
}

/*
 *	New FIFO Pipe.
 */
static void edv_vfs_browser_op_new_fifo(EDVVFSBrowser *browser)
{
	if(browser == NULL)
		return;

	edv_vfs_browser_op_new_object_nexus(
		browser, EDV_OBJECT_TYPE_FIFO, browser->toplevel  
	);
}

/*
 *	New Block Device.
 */
static void edv_vfs_browser_op_new_device_block(EDVVFSBrowser *browser)
{
	if(browser == NULL)
		return;

	edv_vfs_browser_op_new_object_nexus(
		browser, EDV_OBJECT_TYPE_DEVICE_BLOCK, browser->toplevel  
	);
}

/*
 *	New Character Device.
 */
static void edv_vfs_browser_op_new_device_character(EDVVFSBrowser *browser)
{
	if(browser == NULL)
		return;

	edv_vfs_browser_op_new_object_nexus(
		browser, EDV_OBJECT_TYPE_DEVICE_CHARACTER, browser->toplevel  
	);
}

/*
 *	New Socket.
 */
static void edv_vfs_browser_op_new_socket(EDVVFSBrowser *browser)
{
	if(browser == NULL)
		return;

	edv_vfs_browser_op_new_object_nexus(
		browser, EDV_OBJECT_TYPE_SOCKET, browser->toplevel  
	);
}


/*
 *	Open.
 */
static void edv_vfs_browser_op_open(EDVVFSBrowser *browser)
{
	if(browser == NULL)
		return;

	edv_vfs_browser_list_open(
		browser,
		-1, -1,				/* Use the selected object(s) */
		0
	);
}

/*
 *	Open To.
 */
static void edv_vfs_browser_op_open_to(EDVVFSBrowser *browser, ToolBarItem *item)
{
	if(browser == NULL)
		return;

	edv_map_menu_to_button(
		browser->open_to_menu,
		ToolBarItemGetWidget(item)
	);
}

/*
 *	Open With.
 */
static void edv_vfs_browser_op_open_with(EDVVFSBrowser *browser)
{
	GtkCList *clist;

	if(browser == NULL)
		return;

	/* Is there an object selected on the Contents List? */
	clist = GTK_CLIST(browser->contents_clist);
	if(clist->selection != NULL)
	{
		edv_vfs_browser_list_open_with(
			browser,
			-1, -1		/* Use selected object(s) */
		);
	}
	else
	{
		/* Since no object was selected on the Contents List then
		 * check if an object is selected on the Directory Tree
		 */
		GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
		GtkCTreeNode *node = edv_ctree_get_selected_last(ctree, NULL);
		if(node != NULL)
			edv_vfs_browser_tree_open_with(
				browser,
				node
			);
	}
}


/*
 *	Copy Path To DDE.
 */
static void edv_vfs_browser_op_copy_path(EDVVFSBrowser *browser)
{
	GList *objs_list;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	edv_vfs_browser_sync_data(browser);

	/* Get the list of selected objects */
	objs_list = edv_vfs_browser_get_selected_objects(
		browser,
		FALSE
	);
	if(objs_list == NULL)
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Copy the paths of the selected objects to the dde */
	edv_copy_paths_to_clipboard(
		browser->core,
		objs_list,
		browser->toplevel
	);

	/* Delete the selected objects list */
	g_list_free(objs_list);

	edv_vfs_browser_set_busy(browser, FALSE);
}

/*
 *	Copy Location To DDE.
 */
static void edv_vfs_browser_op_copy_url(EDVVFSBrowser *browser)
{
	GList *objs_list;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	edv_vfs_browser_sync_data(browser);

	/* Get the list of selected objects */
	objs_list = edv_vfs_browser_get_selected_objects(
		browser,
		FALSE
	);
	if(objs_list == NULL)
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Copy the urls of the selected objects to the dde */
	edv_copy_urls_to_clipboard(
		browser->core,
		objs_list,
		browser->toplevel
	);

	/* Delete the selected objects list */
	g_list_free(objs_list);

	edv_vfs_browser_set_busy(browser, FALSE);
}

/*
 *	Paste to the current location.
 */
void edv_vfs_browser_op_paste(EDVVFSBrowser *browser)
{
	edv_vfs_browser_op_paste2(
		browser,
		NULL
	);
}

/*
 *	Paste to target location.
 *
 *	If obj is not NULL then obj specifies the target location.
 */
void edv_vfs_browser_op_paste2(
	EDVVFSBrowser *browser,
	EDVVFSObject *obj
)
{
	gboolean yes_to_all = FALSE;
	gint status = -1;
	const gchar *protocol;
	gchar *buf;
	GList		*glist,
			*url_list = NULL;
	GtkWidget *toplevel;
	URLStruct *url;
	EDVCore *core;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	toplevel = browser->toplevel;
	core = browser->core;

	/* Check and warn if the master write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	edv_vfs_browser_sync_data(browser);

	/* Get the data from the clipboard as a string */
	buf = GUIDDEGetString(
		toplevel,
		GDK_SELECTION_PRIMARY,
		GDK_CURRENT_TIME
	);
	if(buf == NULL)
	{
		edv_play_sound_warning(core);
		edv_message_warning(
			"Paste Failed",
"The clipboard is empty or does not contain the\n\
correct data type.",
			NULL,
			toplevel
		);
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Use the selected object as the target? */
	if(obj == NULL)
	{
		/* Contents GtkCList was last selected?
		 *
		 * Get the last selected object from the contents list
		 * only if it is a directory, otherwise get last
		 * selected object from the directory tree
		 */
		if(GTK_CLIST(browser->contents_clist)->selection != NULL)
		{
			GtkCList *clist = GTK_CLIST(browser->contents_clist);
			GList *glist = (clist != NULL) ? clist->selection_end : NULL;
			if(glist != NULL)
			{
				const gint row = (gint)glist->data;
				obj = EDV_VFS_OBJECT(gtk_clist_get_row_data(clist, row));
			}
		}
	}
	/* The target object's destination must be a directory */
	if(obj != NULL)
	{
		if(!edv_path_is_directory(obj->path))
			obj = NULL;
	}
	/* No valid object selected on the Contents GtkCList? */
	if(obj == NULL)
	{
		GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
		GtkCList *clist = GTK_CLIST(ctree);
		GList *glist = (clist != NULL) ? clist->selection_end : NULL;
		if(glist != NULL)
		{
			GtkCTreeNode *node = (GtkCTreeNode *)glist->data;
			obj = EDV_VFS_OBJECT(gtk_ctree_node_get_row_data(ctree, node));
		}
	}
	/* The target object's destination must be a directory */
	if(obj != NULL)
	{
		if(!edv_path_is_directory(obj->path))
			obj = NULL;
	}
	/* No valid target object? */
	if(obj == NULL)
	{
		g_free(buf);
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Decode the URL list string into a list of URLs */
	url_list = url_decode_string(buf);
	g_free(buf);
	buf = NULL;

	/* Nothing to paste? */
	if(url_list == NULL)
	{
		edv_play_sound_warning(core);
		edv_message_warning(
			"Paste Failed",
"There were no objects found in the clipboard's data.",
			NULL,
			toplevel
		);
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Confirm paste objects */
	url = URL(url_list->data);
	if(edv_confirm_copy(
		core,
		toplevel,
		(url != NULL) ? url->path : NULL, g_list_length(url_list),
		obj->path
	) != CDIALOG_RESPONSE_YES)
	{
		g_list_foreach(url_list, (GFunc)url_delete, NULL);
		g_list_free(url_list);
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Paste each URL */
	for(glist = url_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		url = URL(glist->data);
		if(url == NULL)
			continue;

		protocol = url->protocol;
		if(STRISEMPTY(protocol))
			protocol = "file";

		/* File? */
		if(!g_strcasecmp(protocol, "file"))
		{
			const gchar *error_msg;
			GList *new_paths_list = NULL;

			/* Copy this object */
			status = edv_vfs_object_op_copy(
				core,
				url->path,
				obj->path,
				&new_paths_list,
				TRUE,		/* Archive */
				toplevel,
				TRUE,		/* Show progress */
				TRUE,		/* Interactive */
				&yes_to_all
			);

			/* Check for error */
			error_msg = edv_vfs_object_op_get_error(core);
			if(!STRISEMPTY(error_msg))
			{
				/* Report the error */
				edv_play_sound_error(core);
				edv_message_object_op_error(
					"Copy Error",
					error_msg,
					NULL,
					toplevel
				);
			}

			/* Report the new objects? */
			if(new_paths_list != NULL)
			{
				const gchar *path;
				GList *glist;
				EDVVFSObject *obj;

				for(glist = new_paths_list;
					glist != NULL;
					glist = g_list_next(glist)
				)
				{
					path = (const gchar *)glist->data;
					if(STRISEMPTY(path))
						continue;

					obj = edv_vfs_object_lstat(path);
					if(obj != NULL)
					{
						edv_emit_vfs_object_added(
							core,
							path,
							obj
						);
						edv_vfs_object_delete(obj);
					}
				}

				g_list_foreach(new_paths_list, (GFunc)g_free, NULL);
				g_list_free(new_paths_list);
			}

			/* Skip handling of the rest of the objects on error
			 * (status != 0) and that the error was not a user
			 * response of no (status != -5)
			 */
			if((status != 0) && (status != -5))
				break;
		}
		/* Download? */
		else if(!g_strcasecmp(protocol, "http") ||
			!g_strcasecmp(protocol, "ftp") ||
			!g_strcasecmp(protocol, "https")
		)
		{
			edv_internet_download_object(
				core,
				url,
				obj->path,
				toplevel
			);
		}
	}

	/* Delete the URLs list */
	g_list_foreach(url_list, (GFunc)url_delete, NULL);
	g_list_free(url_list);

	/* Unmap progress dialog, it may have been mapped if any
	 * operations occured in the above loop
	 */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Play the "completed" sound on success */
	if(status == 0)
		edv_play_sound_completed(core);

	edv_vfs_browser_set_busy(browser, FALSE);
}


/*
 *	Move.
 */
static void edv_vfs_browser_op_move(EDVVFSBrowser *browser)
{
	GList *sel_objs_list;
	GtkWidget *toplevel;
	edv_obj_op_dlg_struct *d;
	EDVCore *core;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	toplevel = browser->toplevel;
	core = browser->core;

	/* Check and warn if the master write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Get the object operations dialog and create it as needed */
	d = edv_get_object_operations_dialog(core);
	if(d == NULL)
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Sync data to ensure current values to operate on */
	edv_vfs_browser_sync_data(browser);

	/* Get the selected objects list */
	sel_objs_list = edv_vfs_browser_get_selected_objects(
		browser,
		FALSE
	);
	if(sel_objs_list == NULL)
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Map the object operations dialog to move */
	EDVObjOpDlgMapValues(
		d,
		EDV_OBJ_OP_DLG_OP_MOVE,
		EDV_LOCATION_TYPE_VFS,
		sel_objs_list,
		edv_vfs_browser_get_location(browser),
		toplevel
	);

	/* Delete the selected objects list */
	g_list_free(sel_objs_list);

	edv_vfs_browser_set_busy(browser, FALSE);
}

/*
 *	Copy.
 */
static void edv_vfs_browser_op_copy(EDVVFSBrowser *browser)
{
	GList *sel_objs_list;
	GtkWidget *toplevel;
	edv_obj_op_dlg_struct *d;
	EDVCore *core;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	toplevel = browser->toplevel;
	core = browser->core;

	/* Check and warn if the master write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Get the object operations dialog and create it as needed */
	d = edv_get_object_operations_dialog(core);
	if(d == NULL)
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Sync data to ensure current values to operate on */
	edv_vfs_browser_sync_data(browser);

	/* Get the selected objects list */
	sel_objs_list = edv_vfs_browser_get_selected_objects(
		browser,
		FALSE
	);
	if(sel_objs_list == NULL)
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Map the object operations dialog to copy */
	EDVObjOpDlgMapValues(
		d,
		EDV_OBJ_OP_DLG_OP_COPY,
		EDV_LOCATION_TYPE_VFS,
		sel_objs_list,
		edv_vfs_browser_get_location(browser),
		toplevel
	);

	/* Delete the selected objects list */
	g_list_free(sel_objs_list);

	edv_vfs_browser_set_busy(browser, FALSE);
}

/*
 *	Link.
 */
static void edv_vfs_browser_op_link(EDVVFSBrowser *browser)
{
	GList		*glist,
			*sel_objs_list;
	GtkWidget *toplevel;
	EDVCore *core;
	edv_obj_op_dlg_struct *d;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	toplevel = browser->toplevel;
	core = browser->core;

	/* Check and warn if the master write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Get the object operations dialog and create it as needed */
	d = edv_get_object_operations_dialog(core);
	if(d == NULL)
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Sync data to ensure current values to operate on */
	edv_vfs_browser_sync_data(browser);

	/* Get the selected objects list */
	sel_objs_list = edv_vfs_browser_get_selected_objects(
		browser,
		FALSE
	);
	if(sel_objs_list == NULL)
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Linking only allows one object in the list, so create
	 * a list with only one object
	 */
	glist = NULL;
	glist = g_list_append(glist, sel_objs_list->data);
	g_list_free(sel_objs_list);
	sel_objs_list = glist;

	/* Map the object operations dialog to link */
	EDVObjOpDlgMapValues(
		d,
		EDV_OBJ_OP_DLG_OP_LINK,
		EDV_LOCATION_TYPE_VFS,
		sel_objs_list,
		edv_vfs_browser_get_location(browser),
		toplevel
	);

	/* Delete the selected objects list */
	g_list_free(sel_objs_list);

	edv_vfs_browser_set_busy(browser, FALSE);
}

/*
 *	Rename.
 */
static void edv_vfs_browser_op_rename(EDVVFSBrowser *browser)
{
	GtkWidget *toplevel;
	GtkCList *clist;
	EDVCore *core;

	if(browser == NULL)
		return;

	toplevel = browser->toplevel;
	core = browser->core;

	/* Check and warn if the master write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
		return;

	/* Contents GtkCList has a selected object? */
	clist = GTK_CLIST(browser->contents_clist);
	if(clist->selection_end != NULL)
	{
		GList *glist = clist->selection_end;
		const gint row = (gint)glist->data;
		if(row > -1)
		{
			edv_vfs_browser_list_rename_query(
				browser,
				row,
				-1
			);
			return;
		}
	}

	/* Directory GtkCTree has a selected object? */
	clist = GTK_CLIST(browser->directory_ctree);
	if(clist->selection_end != NULL)
	{
		GList *glist = clist->selection_end;
		GtkCTreeNode *node = (GtkCTreeNode *)glist->data;
		if(node != NULL)
		{
			edv_vfs_browser_tree_rename_query(
				browser,
				node
			);
			return;
		}
	}
}

/*
 *	Change Permissions.
 */
static void edv_vfs_browser_op_chmod(EDVVFSBrowser *browser)
{
	GList *sel_objs_list;
	GtkWidget *toplevel;
	edv_obj_op_dlg_struct *d;
	EDVCore *core;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	toplevel = browser->toplevel;
	core = browser->core;

	/* Check and warn if the master write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Get the object operations dialog and create it as needed */
	d = edv_get_object_operations_dialog(core);
	if(d == NULL)
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Sync data to ensure current values to operate on */
	edv_vfs_browser_sync_data(browser);

	/* Get the selected objects list */
	sel_objs_list = edv_vfs_browser_get_selected_objects(
		browser,
		FALSE
	);
	if(sel_objs_list == NULL)
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Map the object operations dialog to change permissions */
	EDVObjOpDlgMapValues(
		d,
		EDV_OBJ_OP_DLG_OP_CHMOD,
		EDV_LOCATION_TYPE_VFS,
		sel_objs_list,
		edv_vfs_browser_get_location(browser),
		toplevel
	);

	/* Delete the selected objects list */
	g_list_free(sel_objs_list);

	edv_vfs_browser_set_busy(browser, FALSE);
}

/*
 *	Change Ownership.
 */
static void edv_vfs_browser_op_chown(EDVVFSBrowser *browser)
{
	GList *sel_objs_list;
	GtkWidget *toplevel;
	edv_obj_op_dlg_struct *d;
	EDVCore *core;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	toplevel = browser->toplevel;
	core = browser->core;

	/* Check and warn if the master write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Get the object operations dialog and create it as needed */
	d = edv_get_object_operations_dialog(core);
	if(d == NULL)
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Sync data to ensure current values to operate on */
	edv_vfs_browser_sync_data(browser);

	/* Get the selected objects list */
	sel_objs_list = edv_vfs_browser_get_selected_objects(
		browser,
		FALSE
	);
	if(sel_objs_list == NULL)
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Map the object operations dialog to change ownership */
	EDVObjOpDlgMapValues(
		d,
		EDV_OBJ_OP_DLG_OP_CHOWN,
		EDV_LOCATION_TYPE_VFS,
		sel_objs_list,
		edv_vfs_browser_get_location(browser),
		toplevel
	);

	/* Delete the selected objects list */
	g_list_free(sel_objs_list);

	edv_vfs_browser_set_busy(browser, FALSE);
}

/*
 *	Set Time Stamps.
 */
static void edv_vfs_browser_op_chtime(EDVVFSBrowser *browser)
{
	GList *sel_objs_list;
	GtkWidget *toplevel;
	EDVCore *core;
	edv_obj_op_dlg_struct *d;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	toplevel = browser->toplevel;
	core = browser->core;

	/* Check and warn if the master write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Get the object operations dialog and create it as needed */
	d = edv_get_object_operations_dialog(core);
	if(d == NULL)
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Sync data to ensure current values to operate on */
	edv_vfs_browser_sync_data(browser);

	/* Get the selected objects list */
	sel_objs_list = edv_vfs_browser_get_selected_objects(
		browser,
		FALSE
	);
	if(sel_objs_list == NULL)
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Map the object operations dialog to set time stamps */
	EDVObjOpDlgMapValues(
		d,
		EDV_OBJ_OP_DLG_OP_CHTIME,
		EDV_LOCATION_TYPE_VFS,
		sel_objs_list,
		edv_vfs_browser_get_location(browser),
		toplevel
	);

	/* Delete the selected objects list */
	g_list_free(sel_objs_list);

	edv_vfs_browser_set_busy(browser, FALSE);
}

/*
 *	Delete.
 */
static void edv_vfs_browser_op_delete(EDVVFSBrowser *browser)
{
	gboolean yes_to_all = FALSE;
	gint status, response, npaths, objects_deleted = 0;
	const gchar *path, *error_msg;
	GList *glist, *paths_list, *index_list;
	GtkWidget *toplevel;
	EDVCore *core;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

#define CLEANUP_RETURN	{			\
 if(paths_list != NULL)	{			\
  g_list_foreach(				\
   paths_list, (GFunc)g_free, NULL		\
  );						\
  g_list_free(paths_list);			\
 }						\
						\
 return;					\
}

	toplevel = browser->toplevel;
	core = browser->core;

	/* Check and warn if the master write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Get the list of selected objects */
	paths_list = edv_vfs_browser_get_selected_paths(browser);

	/* No objects selected? */
	if(paths_list == NULL)
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		CLEANUP_RETURN;
	}

	npaths = g_list_length(paths_list);

	/* Confirm delete */
	response = edv_confirm_delete(
		core, toplevel,
		(npaths == 1) ? (const gchar *)paths_list->data : NULL,
		npaths
	);
	switch(response)
	{
	  case CDIALOG_RESPONSE_YES_TO_ALL:
		yes_to_all = TRUE;
	  case CDIALOG_RESPONSE_YES:
	  case CDIALOG_RESPONSE_OK:
		break;

	  default:
		edv_vfs_browser_set_busy(browser, FALSE);
		CLEANUP_RETURN;
		break;
	}

	/* Delete each object */
	status = 0;
	for(glist = paths_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		path = (const gchar *)glist->data;
		if(path == NULL)
			continue;

		/* Delete this object */
		index_list = NULL;
		status = edv_recycle_object(
			core, path,
			&index_list,
			toplevel,
			TRUE,			/* Show progress */
			TRUE,			/* Interactive */
			&yes_to_all
		);

		/* Check for errors */
		error_msg = edv_recycle_object_get_error(core);
		if(!STRISEMPTY(error_msg))
		{
			/* Report the error */
			edv_play_sound_error(core);
			edv_message_error(
				"Delete Object Error",
				error_msg,
				NULL,
				toplevel
			);
		}

		/* Check if this object no longer exists */
		if(!edv_path_lexists(path))
			edv_emit_vfs_object_removed(
				core,
				path
			);

		/* Notify about the recycled objects added */
		if(index_list != NULL)
		{
			guint index;
			GList *glist;

			for(glist = index_list;
				glist != NULL;
				glist = g_list_next(glist)
			)
			{
				index = (guint)glist->data;
				if(index != 0)
					edv_emit_recycled_object_added(core, index);

				objects_deleted++;
			}

			/* Delete the recycle objects index list */
			g_list_free(index_list);
		}

		/* User aborted? */
		if(status == -4)
			break;
	}

	/* Update the status bar */
	if(TRUE)
	{
		gchar *s;
		if(status == -4)
			s = STRDUP(
				"Delete operation canceled"
			);
		else if(objects_deleted > 0)
			s = g_strdup_printf(
				"Deleted %i %s",
				objects_deleted,
				(objects_deleted == 1) ? "object" : "objects"
			);
		else
			s = g_strdup_printf(
				"Unable to delete %s",
				(npaths == 1) ? "object" : "objects"
			);
		edv_status_bar_message(browser->status_bar, s, FALSE);
		g_free(s);
	}

	/* Unmap progress dialog, it may have been mapped if any
	 * operations occured in the above loop
	 */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Play the "completed" sound on success */
	if(status == 0)
		edv_play_sound_completed(core);

	edv_vfs_browser_set_busy(browser, FALSE);

	CLEANUP_RETURN;
#undef CLEANUP_RETURN
}

/*
 *	Properties.
 */
static void edv_vfs_browser_op_properties(EDVVFSBrowser *browser)
{
	GtkWidget *toplevel;
	EDVVFSObject *obj = NULL;
	EDVCore *core;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	toplevel = browser->toplevel;
	core = browser->core;

	edv_vfs_browser_sync_data(browser);

	/* Get the selected object */
	if(browser->contents_clist_selected_row > -1)
	{
		GtkCList *clist = GTK_CLIST(browser->contents_clist);
		GList *glist = clist->selection_end;
		if(glist != NULL)
			obj = EDV_VFS_OBJECT(gtk_clist_get_row_data(
				clist,
				(gint)glist->data
			));
	}
	else if(browser->directory_ctree_selected_node != NULL)
	{
		GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
		GtkCList *clist = GTK_CLIST(ctree);
		GList *glist = clist->selection_end;
		if(glist != NULL)
			obj = EDV_VFS_OBJECT(gtk_ctree_node_get_row_data(
				ctree,
				(GtkCTreeNode *)glist->data
			));
	}

	/* No selected object */
	if(obj == NULL)
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Create a new Properties Dialog to display the object */
	edv_new_properties_dialog_vfs(
		core,
		obj->path,
		NULL,				/* Default page */
		obj->meta_data_list,
		core->geometry_flags,
		(core->geometry_flags != 0) ? &core->geometry : NULL,
		toplevel
	);

	edv_vfs_browser_set_busy(browser, FALSE);
}


/*
 *	Select All.
 */
static void edv_vfs_browser_op_select_all(EDVVFSBrowser *browser)
{
	EDVCore *core;
	GtkCList *clist;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	clist = GTK_CLIST(browser->contents_clist);
	core = browser->core;

	/* Select all rows on clist */
	gtk_clist_freeze(clist);
	gtk_clist_select_all(clist);
	gtk_clist_thaw(clist);

	/* Mark node as unselected on directory ctree */
	browser->directory_ctree_selected_node = NULL;

	/* Assume highest row index as the last selected row */
	browser->contents_clist_selected_row = clist->rows - 1;

	edv_status_bar_message(
		browser->status_bar, "All objects selected", FALSE
	);

	edv_vfs_browser_set_busy(browser, FALSE);
}

/*
 *	Unselect All.
 */
static void edv_vfs_browser_op_unselect_all(EDVVFSBrowser *browser)
{
	EDVCore *core;
	GtkCList *clist;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	clist = GTK_CLIST(browser->contents_clist);
	core = browser->core;

	/* Unselect all rows on clist */
	gtk_clist_freeze(clist);
	gtk_clist_unselect_all(clist);
	gtk_clist_thaw(clist);

	/* Mark node as unselected on directory ctree */
	browser->directory_ctree_selected_node = NULL;

	/* Mark contents clist's row as unselected */
	browser->contents_clist_selected_row = -1;

	edv_status_bar_message(
		browser->status_bar, "All objects unselected", FALSE
	);

	edv_vfs_browser_set_busy(browser, FALSE);
}

/*
 *	Invert Selection.
 */
static void edv_vfs_browser_op_invert_selection(EDVVFSBrowser *browser)
{
	gint		row,
			total_rows;
	GList		*glist,
			*selection;
	GtkCList *clist;
	EDVCore *core;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	clist = GTK_CLIST(browser->contents_clist);
	core = browser->core;

	/* Get a copy of the selected rows list */
	selection = (clist->selection != NULL) ?
		g_list_copy(clist->selection) : NULL;

	/* Invert the selection */
	gtk_clist_freeze(clist);
	for(row = 0, total_rows = clist->rows;
	    row < total_rows;
	    row++
	)
	{
		/* Unselect all rows in the selections list */
		for(glist = selection;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			if(row == (gint)glist->data)
			{
				gtk_clist_unselect_row(
					clist,
					row,
					0
				);
				break;
			}
		}

		/* If this row is not selected then select it? */
		if(glist == NULL)
			gtk_clist_select_row(
				clist,
				row,
				0
			);
	}
	gtk_clist_thaw(clist);

	g_list_free(selection);

	edv_status_bar_message(
		browser->status_bar,
		"Selection inverted",
		FALSE
	);
	edv_vfs_browser_set_busy(browser, FALSE);
}


/*
 *	Download.
 */
static void edv_vfs_browser_op_download(EDVVFSBrowser *browser)
{
	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);
	edv_internet_download_object(
		browser->core,
		NULL,
		edv_vfs_browser_get_location(browser),
		browser->toplevel
	);
	edv_vfs_browser_set_busy(browser, FALSE);
}


/*
 *	Refresh.
 */
void edv_vfs_browser_op_refresh(EDVVFSBrowser *browser)
{
	gchar *cur_path;
	GtkWidget	*toplevel,
					*sb;
	GtkCList *clist;
	GtkCTree *ctree;
	GtkCTreeNode *node;
	GtkCTreeRow *row_ptr;
	CfgList *cfg_list;
	EDVCore *core;

	if(browser == NULL)
		return;

	toplevel = browser->toplevel;
	cur_path = STRDUP(edv_vfs_browser_get_location(browser));
	sb = browser->status_bar;
	core = browser->core;
	cfg_list = core->cfg_list;

	edv_vfs_browser_set_busy(browser, TRUE);
	GUIBlockInput(toplevel, TRUE);

	/* Update the Directory GtkCTree */
	ctree = GTK_CTREE(browser->directory_ctree);
	clist = GTK_CLIST(ctree);

	/* Get the selected node */
	node = edv_ctree_get_selected_last(ctree, &row_ptr);
	if((node != NULL) && (row_ptr != NULL))
	{
		const gboolean was_expanded = row_ptr->expanded;

		/* Record last scroll position */
		const gfloat	last_x = GTK_ADJUSTMENT_GET_VALUE(clist->hadjustment),
							last_y = GTK_ADJUSTMENT_GET_VALUE(clist->vadjustment);

		gtk_clist_freeze(clist);

		/* Get a new listing of child nodes for the selected node */
		edv_vfs_browser_tree_get_node_listing(
			browser,
			node,
			EDV_GET_B(EDV_CFG_PARM_LISTS_SHOW_PROGRESS)
		);

		/* Realize all the nodes */
		edv_vfs_browser_tree_realize_node_listing(browser, NULL);

		/* Reget the selected node */
		node = edv_ctree_get_selected_last(ctree, &row_ptr);
		if(node != NULL)
		{
			/* Expand it as needed */
			if(was_expanded)
				gtk_ctree_expand(ctree, node);
		}

		gtk_clist_thaw(clist);

		/* Scroll back to the original position */
		edv_clist_scroll_to_position(clist, last_x, last_y);

	}

	/* Update the Contents GtkCList */
	clist = GTK_CLIST(browser->contents_clist);
	if(cur_path != NULL)
	{
		/* Record the last scroll positions */
		const gfloat	last_x = GTK_ADJUSTMENT_GET_VALUE(clist->hadjustment),
							last_y = GTK_ADJUSTMENT_GET_VALUE(clist->vadjustment);

		gtk_clist_freeze(clist);

		/* Reget the listing */
		edv_vfs_browser_list_get_listing(
			browser,
			cur_path,
			EDV_GET_B(EDV_CFG_PARM_LISTS_SHOW_PROGRESS)
		);

		gtk_clist_thaw(clist);

		/* Scroll back to the original position */
		edv_clist_scroll_to_position(clist, last_x, last_y);
	}

	edv_vfs_browser_update_display(browser);
	edv_status_bar_message(sb, "Refreshed contents listing", FALSE);

	g_free(cur_path);

	GUIBlockInput(toplevel, FALSE);
	edv_vfs_browser_set_busy(browser, FALSE);
}

/*
 *	Refresh All.
 */
void edv_vfs_browser_op_refresh_all(EDVVFSBrowser *browser)
{
	EDVCore *core;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	core = browser->core;

	/* Refresh device mount states and stats */
	edv_devices_list_update_mount_states(core->devices_list);
	edv_devices_list_update_statistics(core->devices_list);

	/* Refresh File Browser */
	edv_vfs_browser_op_refresh(browser);

	edv_vfs_browser_set_busy(browser, FALSE);
}

/*
 *	Go To Parent.
 */
static void edv_vfs_browser_op_goto_parent(EDVVFSBrowser *browser)
{
	const gchar *path;
	GtkWidget *toplevel;
	GtkCTreeNode *node;
	GtkCTreeRow *row_ptr;
	GtkCTree *ctree;
	EDVVFSObject *obj;
	EDVCore *core;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	toplevel = browser->toplevel;
	ctree = GTK_CTREE(browser->directory_ctree);
	core = browser->core;

	/* Get the selected node */
	node = edv_ctree_get_selected_last(ctree, &row_ptr);
	if((node == NULL) || (row_ptr == NULL))
	{
		/* No node selected */
		edv_play_sound_beep(core);
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Get the selected node's Object */
	obj = EDV_VFS_OBJECT(gtk_ctree_node_get_row_data(ctree, node));
	if(obj == NULL)
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	path = obj->path;
	if(STRISEMPTY(path))
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Already at toplevel? */
	if(!strcmp((const char *)path, "/"))
	{
		edv_play_sound_beep(core);
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Get the parent node
	 *
	 * If the node does not have a parent then it implies that
	 * we need to update the Directory GtkCTree's Origin path
	 */
	GUIBlockInput(toplevel, TRUE);
	node = row_ptr->parent;
	if(node != NULL)
	{
		/* Select the parent node */
		gtk_ctree_select(ctree, node);
	}
	else
	{
		gchar *parent_path = g_dirname(path);
		if(parent_path != NULL)
		{
			/* Update the Directory Tree Origin to be the parent
			 * and select the parent
			 */
			edv_vfs_browser_tree_set_origin_path(browser, parent_path);
			g_free(parent_path);
		}
	}
	GUIBlockInput(toplevel, FALSE);

	edv_vfs_browser_set_busy(browser, FALSE);
}

/*
 *	Go To Home.
 */
static void edv_vfs_browser_op_goto_home(EDVVFSBrowser *browser)
{
	gchar *path;
	GtkWidget *toplevel;
	GtkCTree *ctree;
	EDVCore *core;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	toplevel = browser->toplevel;
	ctree = GTK_CTREE(browser->directory_ctree);
	core = browser->core;

	/* Get home directory */
	path = STRDUP(core->home_path);
	if(STRISEMPTY(path))
	{
		edv_play_sound_warning(core);
		edv_message_warning(
#if defined(PROG_LANGUAGE_SPANISH)
"Vaya A En Casa Gua Fallada",
"Incapaz de encontrar la gua de hogar, cercirese que el\n\
ambiente HOME variable se pone.\n",
			NULL,
#elif defined(PROG_LANGUAGE_FRENCH)
"Aller A L'Annuaire De Maison Echou",
"Incapable de trouver l'annuaire de maison, s'assure que\n\
l'environnement HOME variable est rgl.\n",
			NULL,
#elif defined(PROG_LANGUAGE_GERMAN)
"Gehen Sie Zu Heim Verzeichnis Hat Versagt",
"Unfhig, das heim verzeichnis zu finden, vergewissert\n\
sich, da die umwelt vernderliche HOME gesetzt ist.\n",
			NULL,
#elif defined(PROG_LANGUAGE_ITALIAN)
"Andare All'Elenco Di Casa Fallito",
"Incapace per trovare l'elenco di casa, si assicura che\n\
l'ambiente HOME variabile  regolato.\n",
			NULL,
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga Te Huis Gids Verzuimde",
"Onbekwame de huis gids te vinden, vergewist zich ervan dat de\n\
omgeving, die veranderlijke HOME gezet is.\n",
			NULL,
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"V A Guia De Lar Fracassado",
"Incapaz de achar o guia de lar, assegura-se que o ambiente\n\
HOME varivel  posto.\n",
			NULL,
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Dra Til Hjem Sviktet Katalog",
"Maktesls finne hjemkatalogen, sjekker at miljet variabel\n\
HOME setter.\n",
			NULL,
#else
"Go To Home Directory Failed",
"Unable to find the home directory, make sure that the\n\
environment variable HOME is set.\n",
			NULL,
#endif
			toplevel
		);
		g_free(path);
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Change the Directory Tree origin as needed */
	GUIBlockInput(toplevel, TRUE);
	if(edv_vfs_browser_is_path_from_tree_origin(browser, path))
	{
		/* Select the home directory */
		edv_vfs_browser_tree_select_path(browser, path);
	}
	else
	{
		/* Set the Directory Tree origin to the home directory
		 * and select the home directory
		 */
		edv_vfs_browser_tree_set_origin_path(browser, path);
	}
	GUIBlockInput(toplevel, FALSE);

	g_free(path);

	edv_vfs_browser_set_busy(browser, FALSE);
}

/*
 *	Expands/Colapse.
 */
static void edv_vfs_browser_op_expand(EDVVFSBrowser *browser)
{
	GtkCTree *ctree;
	GtkCTreeNode *node;
	GtkCTreeRow *row_ptr;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	ctree = GTK_CTREE(browser->directory_ctree);

	/* Get selected node */
	node = edv_ctree_get_selected_last(ctree, &row_ptr);
	if((node == NULL) || (row_ptr == NULL))
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Collapse or expand node? Note that appropriate callbacks
	 * will be made when expanded or collapsed.
	 */
	if(row_ptr->expanded)
		gtk_ctree_collapse(ctree, node);
	else
		gtk_ctree_expand(ctree, node);

	edv_vfs_browser_update_display(browser);

	edv_vfs_browser_set_busy(browser, FALSE);
}


/*
 *	Sets the contents list filter.
 */
static void edv_vfs_browser_op_list_filter(EDVVFSBrowser *browser)
{
	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	if(edv_query_name_filter(
		browser->core,
		&browser->contents_list_filter,
		browser->toplevel
	))
	{
		edv_vfs_browser_set_title(
			browser,
			edv_vfs_browser_get_location(browser)
		);
		edv_vfs_browser_op_refresh(browser);
	}

	edv_vfs_browser_set_busy(browser, FALSE);
}


/*
 *	Directory Tree Origin.
 */
static void edv_vfs_browser_op_directory_tree_origin(EDVVFSBrowser *browser)
{
	gboolean response;
	gint		nftypes = 0,
			npaths = 0;
	gchar		*path,
			**paths_list = NULL,
			*new_origin_path;
	GtkWidget *toplevel;
	fb_type_struct	**ftypes_list = NULL,
			*ftype_rtn = NULL;
	EDVVFSObject *obj;
	EDVCore *core;
	if((browser == NULL) || FileBrowserIsQuery())
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	toplevel = browser->toplevel;
	path = STRDUP(browser->directory_ctree_origin_path);
	core = browser->core;

	/* Create a new file types list */
	FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		"*.*", "All Files"
	);
	  
	/* Query the user for the origin directory */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
		"Set Directory Tree Origin",
		"Set", "Cancel",
		path,
		ftypes_list, nftypes,
		&paths_list, &npaths,
		&ftype_rtn
	);
	g_free(path);
	FileBrowserSetTransientFor(NULL);

	/* Got new directory tree origin path and it exists and it leads
	 * to a directory?
	 */
	if(response)
	{
		new_origin_path = (npaths > 0) ?
			STRDUP(paths_list[npaths - 1]) : NULL;
	}
	else
	{
		new_origin_path = NULL;
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	if(STRISEMPTY(new_origin_path))
	{
		g_free(new_origin_path);
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}
	if(!g_path_is_absolute(new_origin_path))
	{
		gchar *msg = g_strdup_printf(
"Not an absolute path:\n\
\n\
    %s\n",
			new_origin_path
		);
		edv_play_sound_warning(core);
		edv_message_warning(
			"Set Directory Tree Origin Failed",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
		g_free(new_origin_path);
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}
	obj = edv_vfs_object_stat(new_origin_path);
	if(obj == NULL)
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s\n",
			g_strerror(error_code),
			new_origin_path
		);
		edv_play_sound_warning(core);
		edv_message_warning(
			"Set Directory Tree Origin Failed",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
		g_free(new_origin_path);
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}
	if(!EDV_VFS_OBJECT_IS_DIRECTORY(obj))
	{
		gchar *msg = g_strdup_printf(
"Object is not a directory:\n\
\n\
    %s\n",
			new_origin_path
		);
		edv_play_sound_warning(core);
		edv_message_warning(
			"Set Directory Tree Origin Failed",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
		edv_vfs_object_delete(obj);
		g_free(new_origin_path);
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	edv_vfs_object_delete(obj);

	/* Set the new Directory Tree Origin path */
	GUIBlockInput(toplevel, TRUE);
	edv_vfs_browser_tree_set_origin_path(
		browser,
		new_origin_path
	);
	GUIBlockInput(toplevel, FALSE);

	g_free(new_origin_path);

	edv_vfs_browser_set_busy(browser, FALSE);
}


/*
 *	Mount/Unmount.
 */
static void edv_vfs_browser_op_mount(EDVVFSBrowser *browser)
{
	gboolean original_mount_state;
	gint		status,
			dev_num;
	GtkWidget *toplevel;
	EDVDevice *dev;
	EDVCore *core;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	toplevel = browser->toplevel;
	core = browser->core;

	/* Get the last selected Device */
	dev_num = browser->selected_dev_num;
	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));
	if(dev == NULL)
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Record original mount state */
	original_mount_state = EDV_DEVICE_IS_MOUNTED(dev);

	/* Unmount or mount? */
	if(EDV_DEVICE_IS_MOUNTED(dev))
		status = edv_device_unmount(
			core,
			dev,
			TRUE,				/* Show progress */
			TRUE,				/* Verbose */
			toplevel
		);
	else
		status = edv_device_mount(
			core,
			dev,
			TRUE,				/* Show progress */
			TRUE,				/* Verbose */
			toplevel
		);

	/* Update all the devices' statistics */
	edv_devices_list_update_statistics(core->devices_list);

	/* Mount error? */
	if(status != 0)
	{
		const gchar *last_error = edv_device_mount_get_error(core);
		if(!STRISEMPTY(last_error))
		{
			edv_play_sound_error(core);
			edv_message_error(
				original_mount_state ?
					"Unmount Failed" : "Mount Failed",
				last_error,
				NULL,
				browser->toplevel
			);
		}
	}
	else
	{
		/* Report un mount signal to all of endeavour's resources */
		edv_emit_device_mount(core, dev_num, dev, EDV_DEVICE_IS_MOUNTED(dev));
	}

	edv_vfs_browser_set_busy(browser, FALSE);
}

/*
 *	Eject.
 */
static void edv_vfs_browser_op_eject(EDVVFSBrowser *browser)
{
	gboolean original_mount_state;
	gint		status,
			dev_num;
	GtkWidget *toplevel;
	EDVDevice *dev;
	EDVCore *core;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	toplevel = browser->toplevel;
	core = browser->core;

	/* Get the last selected Device */
	dev_num = browser->selected_dev_num;
	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));
	if(dev == NULL)
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Record original mount state */
	original_mount_state = EDV_DEVICE_IS_MOUNTED(dev);

	/* Need to unmount first? */
	if(EDV_DEVICE_IS_MOUNTED(dev))
		status = edv_device_unmount(
			core, dev,
			TRUE, TRUE,
			toplevel
		);

	/* Eject */
	status = edv_device_eject(
		core, dev,
		TRUE, TRUE,
		toplevel
	);

	/* Update all device mount states and stats */
	edv_devices_list_update_mount_states(
		core->devices_list
	);
	edv_devices_list_update_statistics(
		core->devices_list
	);

	/* Eject error? */
	if(status != 0)
	{
		const gchar *last_error = edv_device_mount_get_error(core);
		if(!STRISEMPTY(last_error))
		{
			edv_play_sound_error(core);
			edv_message_error(
				"Eject Failed",
				last_error,
				NULL,
				browser->toplevel
			);
		}
	}
	else
	{
		/* Report eject (unmount) signal to all of endeavour's
		 * resources
		 */
		edv_emit_device_mount(core, dev_num, dev, EDV_DEVICE_IS_MOUNTED(dev));
	}

	edv_vfs_browser_set_busy(browser, FALSE);
}

/*
 *      MIME Types.
 */
static void edv_vfs_browser_op_mime_types(EDVVFSBrowser *browser)
{
	gchar *type_string = NULL;
	GtkWidget *toplevel;
	EDVCore *core;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	toplevel = browser->toplevel;
	core = browser->core;

	if(browser->contents_clist_selected_row > -1)
	{
		GtkCList *clist = GTK_CLIST(browser->contents_clist);
		const gint i = edv_clist_get_selected_last(clist, NULL);
		if(i > -1)
		{
			EDVVFSObject *obj = EDV_VFS_OBJECT(
				gtk_clist_get_row_data(clist, i)
			);
			if(obj != NULL)
			{
				(void)edv_match_object_type_string(
					core->mime_types_list,
					obj->type,
					obj->path,
					obj->permissions,
					&type_string
				);
			}
		}
	}
	else if(browser->directory_ctree_selected_node != NULL)
	{
		GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
		GtkCTreeNode *node = edv_ctree_get_selected_last(ctree, NULL);
		if(node != NULL)
		{
			EDVVFSObject *obj = EDV_VFS_OBJECT(
				gtk_ctree_node_get_row_data(ctree, node)
			);
			if(obj != NULL)
			{
				(void)edv_match_object_type_string(
					core->mime_types_list,
					obj->type,
					obj->path,
					obj->permissions,
					&type_string
				);
			}
		}
	}

	edv_map_mime_types(
		core,
		type_string,
		core->geometry_flags,
		(core->geometry_flags != 0) ? &core->geometry : NULL,
		toplevel
	);

	g_free(type_string);

	edv_vfs_browser_set_busy(browser, FALSE);
}

/*
 *	Devices.
 */
static void edv_vfs_browser_op_devices(EDVVFSBrowser *browser)
{
	gchar *path = NULL;
	GtkWidget *toplevel;
	EDVCore *core;

	if(browser == NULL)
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	toplevel = browser->toplevel;
	core = browser->core;

	if(browser->contents_clist_selected_row > -1)
	{
		GtkCList *clist = GTK_CLIST(browser->contents_clist);
		const gint i = edv_clist_get_selected_last(clist, NULL);
		if(i > -1)
		{
			EDVVFSObject *obj = EDV_VFS_OBJECT(
				gtk_clist_get_row_data(clist, i)
			);
			if(obj != NULL)
				path = STRDUP(obj->path);
		}
	}
	else if(browser->directory_ctree_selected_node != NULL)
	{
		GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
		GtkCTreeNode *node = edv_ctree_get_selected_last(ctree, NULL);
		if(node != NULL)
		{
			EDVVFSObject *obj = EDV_VFS_OBJECT(
				gtk_ctree_node_get_row_data(ctree, node)
			);
			if(obj != NULL)
				path = STRDUP(obj->path);
		}
	}

	edv_map_devices(
		core,
		path,
		core->geometry_flags,
		(core->geometry_flags != 0) ? &core->geometry : NULL,
		toplevel
	);

	g_free(path);

	edv_vfs_browser_set_busy(browser, FALSE);
}
