#include <stdlib.h>
#include <string.h>
#include <errno.h>
#if defined(HAVE_REGEX)
# include <regex.h>
#else
# include <fnmatch.h>
#endif
#include <gtk/gtk.h>

#include "cfg.h"

#include "guiutils.h"
#include "fprompt.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_directory.h"
#include "libendeavour2-base/edv_property.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_id.h"
#include "edv_ids_list.h"
#include "edv_date_format.h"
#include "edv_mime_type.h"
#include "edv_confirm.h"
#include "edv_obj_info_match.h"
#include "edv_utils_gtk.h"
#include "edv_status_bar.h"
#include "vfs_browser.h"
#include "vfs_browser_cb.h"
#include "vfs_browser_tree.h"
#include "vfs_browser_list.h"
#include "edv_vfs_obj_op.h"
#include "edv_open.h"
#include "edv_emit.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


typedef struct _EDVVFSBrowserListPromptData	\
					EDVVFSBrowserListPromptData;
#define EDV_VFS_BROWSER_LIST_PROMPT_DATA(p)	\
					((EDVVFSBrowserListPromptData *)(p))


/* Columns */
gint edv_vfs_browser_list_get_column_index_by_type(
	EDVVFSBrowser *browser,
	const EDVVFSBrowserColumnType column_type
);
gint edv_vfs_browser_list_get_column_width_by_type(
	EDVVFSBrowser *browser,
	const EDVVFSBrowserColumnType column_type
);
EDVVFSBrowserColumnType edv_vfs_browser_list_get_column_type_by_index(
	EDVVFSBrowser *browser,
	const gint column_num
);
void edv_vfs_browser_list_set_column_width_by_type(
	EDVVFSBrowser *browser,
	const EDVVFSBrowserColumnType column_type,
	const gint width
);
static void edv_vfs_browser_list_reset_columns(EDVVFSBrowser *browser);
void edv_vfs_browser_list_resize_column_optimul(
	EDVVFSBrowser *browser,
	const gint column_num
);
void edv_vfs_browser_list_resize_columns_optimul(EDVVFSBrowser *browser);

/* Cell Setting */
static void edv_vfs_browser_list_set_cell_name(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column
);
static void edv_vfs_browser_list_set_cell_size(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column,
	const gboolean hide_dir_size, const gboolean hide_link_size,
	const EDVSizeFormat size_format,
	const gulong block_size
);
static void edv_vfs_browser_list_set_cell_type(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column
);
static void edv_vfs_browser_list_set_cell_permissions(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column, gboolean hide_link_permissions
);
static void edv_vfs_browser_list_set_cell_owner(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column
);
static void edv_vfs_browser_list_set_cell_group(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column
);
static void edv_vfs_browser_list_set_cell_date_access(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column,
	EDVDateRelativity date_relativity, const gchar *date_format
);
static void edv_vfs_browser_list_set_cell_date_modified(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column,
	EDVDateRelativity date_relativity, const gchar *date_format
);
static void edv_vfs_browser_list_set_cell_date_changed(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column,
	EDVDateRelativity date_relativity, const gchar *date_format
);
static void edv_vfs_browser_list_set_cell_hard_links(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column
);
static void edv_vfs_browser_list_set_cell_linked_to(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column
);
static void edv_vfs_browser_list_set_cell_device(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column
);
static void edv_vfs_browser_list_set_cell_index(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column
);
static void edv_vfs_browser_list_set_cell_device_type(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column
);
static void edv_vfs_browser_list_set_cell_block_size(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column
);
static void edv_vfs_browser_list_set_cell_blocks(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column
);
static void edv_vfs_browser_list_set_row(
	EDVVFSBrowser *browser,
	const gint row,
	EDVVFSObject *obj
);

static gint edv_vfs_browser_list_append_object(
	EDVVFSBrowser *browser,
	EDVVFSObject *obj
);
static void edv_vfs_browser_list_append_listing(
	EDVVFSBrowser *browser,
	const gchar *path,
	const gboolean show_progress
);

/* Selection */
static GList *edv_vfs_browser_list_get_selected_paths(EDVVFSBrowser *browser);

/* Finding */
gint edv_vfs_browser_list_find_by_index(
	EDVVFSBrowser *browser,
	const gulong device_index, const gulong index
);
gint edv_vfs_browser_list_find_by_path(
	EDVVFSBrowser *browser,
	const gchar *path
);

/* Realize Listing */
void edv_vfs_browser_list_realize_listing(EDVVFSBrowser *browser);

/* Get Listing */
void edv_vfs_browser_list_get_listing(
	EDVVFSBrowser *browser,
	const gchar *path,
	const gboolean show_progress
);
void edv_vfs_browser_list_clear(EDVVFSBrowser *browser);

/* Opening */
void edv_vfs_browser_list_open(
	EDVVFSBrowser *browser,
	const gint row, const gint column,
	const guint state			/* Modifier keys */
);
void edv_vfs_browser_list_open_with(
	EDVVFSBrowser *browser,
	const gint row, const gint column
);

/* Renaming */
static void edv_vfs_browser_list_prompt_rename_apply_cb(
	gpointer data, const gchar *value
);
static void edv_vfs_browser_list_prompt_rename_cancel_cb(gpointer data);
void edv_vfs_browser_list_rename_query(
	EDVVFSBrowser *browser,
	const gint row, const gint column
);

/* Object Callbacks */
void edv_vfs_browser_list_vfs_object_added_cb(
	EDVVFSBrowser *browser,
	const gchar *path,
	EDVVFSObject *obj
);
void edv_vfs_browser_list_vfs_object_modified_cb(
	EDVVFSBrowser *browser,
	const gchar *path,
	const gchar *new_path,
	EDVVFSObject *obj
);
void edv_vfs_browser_list_vfs_object_removed_cb(
	EDVVFSBrowser *browser,
	const gchar *path
);

/* Mount Callbacks */
void edv_vfs_browser_list_device_mount_cb(
	EDVVFSBrowser *browser,
	EDVDevice *d,
	const gboolean mounted
);


struct _EDVVFSBrowserListPromptData {
	EDVVFSBrowser	*vfs_browser;
	gint		row,
			column;
};


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the column index displaying the column type.
 *
 *	The column_type specifies the EDVVFSBrowserColumnType.
 *
 *	Returns the column index that is set to column_type or
 *	negative on error.
 */
gint edv_vfs_browser_list_get_column_index_by_type(
	EDVVFSBrowser *browser,
	const EDVVFSBrowserColumnType column_type
)
{
	gint column_num;
	GtkCList *clist;
	CfgList *cfg_list;
	CfgIntList *column_types_intlist;
	EDVCore *core;

	if(browser == NULL)
		return(-2);

	clist = GTK_CLIST(browser->contents_clist);
	core = browser->core;
	cfg_list = core->cfg_list;

	column_types_intlist = EDV_GET_INT_LIST(
		EDV_CFG_PARM_BROWSER_CONTENTS_COLUMN
	);
	if(column_types_intlist == NULL)
		return(-2);

	column_num = (gint)g_list_index(
		column_types_intlist->list,
		(gpointer)column_type
	);
	if(column_num < 0)
		return(-2);

	if(column_num >= clist->columns)
		return(clist->columns - 1);

	return(column_num);
}

/*
 *	Gets the width of the column displaying the column type.
 *
 *	The column_type specifies the column type.
 *
 *	Returns the column width.
 */
gint edv_vfs_browser_list_get_column_width_by_type(
	EDVVFSBrowser *browser,
	const EDVVFSBrowserColumnType column_type
)
{
	GtkCList *clist;
	const gint column_num = edv_vfs_browser_list_get_column_index_by_type(
		browser,
		column_type
	);
	if(column_num < 0)
		return(0);

	clist = GTK_CLIST(browser->contents_clist);

	return(clist->column[column_num].width);
}

/*
 *	Gets the column type that the column index is displaying.
 *
 *	The column_num specifies the column index.
 *
 *	Returns the EDVVFSBrowserColumnType that column_num is
 *	displaying.
 */
EDVVFSBrowserColumnType edv_vfs_browser_list_get_column_type_by_index(
	EDVVFSBrowser *browser,
	const gint column_num
)
{
	CfgList *cfg_list;
	CfgIntList *column_types_intlist;
	EDVCore *core;

	if(browser == NULL)
		return(EDV_VFS_BROWSER_COLUMN_TYPE_NAME);

	core = browser->core;
	cfg_list = core->cfg_list;

	column_types_intlist = EDV_GET_INT_LIST(
		EDV_CFG_PARM_BROWSER_CONTENTS_COLUMN
	);
	if(column_types_intlist == NULL)
		return(EDV_VFS_BROWSER_COLUMN_TYPE_NAME);

	return((EDVVFSBrowserColumnType)g_list_nth_data(
		column_types_intlist->list,
		column_num
	));
}

/*
 *	Sets the width of the column that is displaying the column
 *	type.
 *
 *	The column_type specifies the column type.
 *
 *	The width specifies the width of the column.
 */
void edv_vfs_browser_list_set_column_width_by_type(
	EDVVFSBrowser *browser,
	const EDVVFSBrowserColumnType column_type,
	const gint width
)
{
	GtkCList *clist;
	const gint column_num = edv_vfs_browser_list_get_column_index_by_type(
		browser,
		column_type
	);
	if(column_num < 0)
		return;

	clist = GTK_CLIST(browser->contents_clist);

	gtk_clist_set_column_width(
		clist,
		column_num,
		width
	);
}


/*
 *	Resets the column headings to the names and ordering specified
 *	by the configuration.
 *
 *	All inputs assumed valid.
 */
static void edv_vfs_browser_list_reset_columns(EDVVFSBrowser *browser)
{
	gint		i,
			width;
	const gchar *title = NULL;
	GList *glist;
	GtkJustification justify = GTK_JUSTIFY_LEFT;
	GtkCList *clist = GTK_CLIST(browser->contents_clist);
	const gint ncolumns = clist->columns;
	EDVCore *core = browser->core;
	GtkRcStyle *lists_rcstyle = core->lists_rcstyle;
	CfgIntList	*column_types_intlist,
			*column_width_intlist;
	CfgList *cfg_list = core->cfg_list;
	EDVVFSBrowserColumnType column_type;

	/* Get the column types mapping */
	column_types_intlist = EDV_GET_INT_LIST(
		EDV_CFG_PARM_BROWSER_CONTENTS_COLUMN
	);
	if(column_types_intlist == NULL)
		return;

	/* Get the column widths */
	column_width_intlist = EDV_GET_INT_LIST(
		EDV_CFG_PARM_BROWSER_CONTENTS_COLUMN_WIDTH
	);
	if(column_width_intlist == NULL)
		return;

	gtk_clist_freeze(clist);

	/* Update the clist's column settings */
	gtk_clist_column_titles_active(clist);
	gtk_clist_column_titles_show(clist);
	gtk_clist_set_auto_sort(
		clist,
		FALSE
	);
	gtk_clist_set_sort_type(
		clist,
		GTK_SORT_DESCENDING
	);

#if 0
/* Already set */
	/* Change clist selection mode to GTK_SELECTION_EXTENDED
	 *
	 * The selection mode can change whenever the contents list is
	 * updated
	 */
	gtk_clist_set_selection_mode(
		clist,
		GTK_SELECTION_EXTENDED
	);
#endif

	/* Iterate through each column */
	for(glist = column_types_intlist->list, i = 0;
	    glist != NULL;
	    glist = g_list_next(glist), i++
	)
	{
		column_type = (EDVVFSBrowserColumnType)glist->data;

		/* Get the width for this column type */
		width = (gint)g_list_nth_data(
			column_width_intlist->list,
			(guint)column_type
		);

		/* Get column title and justification  based on the
		 * column type
		 */
		switch(column_type)
		{
		  case EDV_VFS_BROWSER_COLUMN_TYPE_NAME:
			title = "Name";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_SIZE:
			title = "Size";
			justify = GTK_JUSTIFY_RIGHT;
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_TYPE:
			title = "Type";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_PERMISSIONS:
			title = "Permissions";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_OWNER:
			title = "Owner";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_GROUP:
			title = "Group";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_DATE_ACCESS:
			title = "Date Access";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_DATE_MODIFIED:
			title = "Date Modified";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_DATE_CHANGED:
			title = "Date Changed";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_HARD_LINKS:
			title = "Hard Links";
			justify = GTK_JUSTIFY_RIGHT;
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_LINKED_TO:
			title = "Linked To";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_DEVICE_INDEX:
			title = "Device Index";
			justify = GTK_JUSTIFY_RIGHT;
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_INDEX:
			title = "Index";
			justify = GTK_JUSTIFY_RIGHT;
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_DEVICE_TYPE:
			title = "Device Type";
			justify = GTK_JUSTIFY_RIGHT;
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_BLOCK_SIZE:
			title = "Block Size";
			justify = GTK_JUSTIFY_RIGHT;
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_BLOCKS:
			title = "Blocks";
			justify = GTK_JUSTIFY_RIGHT;
			break;
		}

		gtk_clist_set_column_visibility(
			clist,
			i,
			TRUE
		);
		gtk_clist_set_column_auto_resize(
			clist,
			i,
			FALSE
		);
		gtk_clist_set_column_title(
			clist,
			i,
			title
		);
		gtk_clist_set_column_justification(
			clist,
			i,
			justify
		);
		gtk_clist_set_column_width(
			clist,
			i,
			width
		);
	}
	/* Reset the rest of the columns */
	while(i < ncolumns)
	{
		gtk_clist_set_column_visibility(
			clist,
			i,
			FALSE
		);
		i++;
	}

	/* Set RC style after column headings have been mapped */
	if(lists_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			GTK_WIDGET(clist),
			lists_rcstyle
		);

	gtk_clist_thaw(clist);
}

/*
 *	Resizes the column to optimul size.
 */
void edv_vfs_browser_list_resize_column_optimul(
	EDVVFSBrowser *browser,
	const gint column_num
)
{
	gint ncolumns, width;
	GtkCList *clist;

	if(browser == NULL)
		return;

	clist = GTK_CLIST(browser->contents_clist);
	ncolumns = clist->columns;
	if((column_num < 0) || (column_num >= ncolumns))
		return;

	gtk_clist_freeze(clist);

	width = gtk_clist_optimal_column_width(
		clist,
		column_num
	);
	if(width > 0)
		gtk_clist_set_column_width(
			clist,
			column_num,
			width
		);

	gtk_clist_thaw(clist);
}

/*
 *	Resizes all the columns to optimul widths.
 */
void edv_vfs_browser_list_resize_columns_optimul(EDVVFSBrowser *browser)
{
	gint		i,
			ncolumns;
	GtkCList *clist;

	if(browser == NULL)
		return;

	clist = GTK_CLIST(browser->contents_clist);
	ncolumns = clist->columns;

	gtk_clist_freeze(clist);

	for(i = 0; i < ncolumns; i++)
		edv_vfs_browser_list_resize_column_optimul(
			browser,
			i
		);

	gtk_clist_thaw(clist);
}


/*
 *	Sets the cell's name value and attributes.
 */
static void edv_vfs_browser_list_set_cell_name(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column
)
{
	const gchar *name = (obj->name != NULL) ? obj->name : "(null)";
	GtkStyle	**cell_styles = browser->cell_style,
			*style = NULL;
	EDVPixmap	*icon,
			*icon_opened,
			*icon_inaccessable,
			*icon_hidden;

	/* Get the object's icon */
	(void)edv_match_object_icon(
		core->devices_list,
		core->mime_types_list,
		obj->type,
		obj->path,
		EDV_VFS_OBJECT_LINK_TARGET_EXISTS(obj),
		obj->permissions,
		EDV_ICON_SIZE_20,
		&icon,
		&icon_opened,
		&icon_inaccessable,
		&icon_hidden
	);
	/* Get the style and alternate icon based on any
	 * type-specific conditions
	 */
	switch(obj->type)
	{
	  case EDV_OBJECT_TYPE_UNKNOWN:
		break;
	  case EDV_OBJECT_TYPE_FILE:
		/* Hidden? */
		if(edv_is_object_hidden(obj))
		{
			if(edv_pixmap_is_loaded(icon_hidden))
			{
				(void)edv_pixmap_unref(icon);
				icon = edv_pixmap_ref(icon_hidden);
			}
		}
		break;
	  case EDV_OBJECT_TYPE_DIRECTORY:
		/* Hidden? */
		if(edv_is_object_hidden(obj))
		{
			if(edv_pixmap_is_loaded(icon_hidden))
			{
				(void)edv_pixmap_unref(icon);
				icon = edv_pixmap_ref(icon_hidden);
			}
		}
		/* Directory not accessable? */
		if(!edv_is_object_accessable(core, obj))
		{
			if(edv_pixmap_is_loaded(icon_inaccessable))
			{
				(void)edv_pixmap_unref(icon);
				icon = edv_pixmap_ref(icon_inaccessable);
			}
			if(style == NULL)
				style = cell_styles[
					EDV_VFS_BROWSER_CELL_STYLE_NO_ACCESS
				];
		}
		break;
	  case EDV_OBJECT_TYPE_LINK:
		/* Link target exists? */
		if(EDV_VFS_OBJECT_LINK_TARGET_EXISTS(obj))
		{
			/* Hidden? */
			if(edv_is_object_hidden(obj))
			{
				if(edv_pixmap_is_loaded(icon_hidden))
				{
					(void)edv_pixmap_unref(icon);
					icon = edv_pixmap_ref(icon_hidden);
				}
			}
		}
		else
		{
			/* Dangling link */
			if(edv_pixmap_is_loaded(icon_inaccessable))
			{
				(void)edv_pixmap_unref(icon);
				icon = edv_pixmap_ref(icon_inaccessable);
			}
			if(style == NULL)
				style = cell_styles[
				    EDV_VFS_BROWSER_CELL_STYLE_DANGLING_LINK
				];
		}
		/* Infinately recursive? */
		if(EDV_VFS_OBJECT_IS_LINK_TARGET_GRAND_PARENT(obj))
		{
			if(style == NULL)
				style = cell_styles[
					EDV_VFS_BROWSER_CELL_STYLE_RECURSIVE_LINK
				];
		}
		break;
	  case EDV_OBJECT_TYPE_DEVICE_BLOCK:
		/* Hidden? */
		if(edv_is_object_hidden(obj))
		{
			if(edv_pixmap_is_loaded(icon_hidden))
			{
				(void)edv_pixmap_unref(icon);
				icon = edv_pixmap_ref(icon_hidden);
			}
		}
		break;
	  case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
		/* Hidden? */
		if(edv_is_object_hidden(obj))
		{
			if(edv_pixmap_is_loaded(icon_hidden))
			{
				(void)edv_pixmap_unref(icon);
				icon = edv_pixmap_ref(icon_hidden);
			}
		}
		break;
	  case EDV_OBJECT_TYPE_FIFO:
		/* Hidden? */
		if(edv_is_object_hidden(obj))
		{
			if(edv_pixmap_is_loaded(icon_hidden)) 
			{
				(void)edv_pixmap_unref(icon);
				icon = edv_pixmap_ref(icon_hidden);
			}
		}
		break;
	  case EDV_OBJECT_TYPE_SOCKET:
		/* Hidden? */
		if(edv_is_object_hidden(obj))
		{
			if(edv_pixmap_is_loaded(icon_hidden))
			{
				(void)edv_pixmap_unref(icon);
				icon = edv_pixmap_ref(icon_hidden);
			}
		}
		break;
	  case EDV_OBJECT_TYPE_ERROR:
		break;
	}

	/* Special directory notations? */
	if(!strcmp((const char *)name, ".."))
	{
		if(edv_pixmap_is_loaded(browser->directory_parent_icon))
		{
			(void)edv_pixmap_unref(icon);
			icon = edv_pixmap_ref(browser->directory_parent_icon);
		}
	}

	/* Set the cell */
	gtk_clist_freeze(clist);
	if(edv_pixmap_is_loaded(icon))
		 gtk_clist_set_pixtext(
			clist,
			row, column,
			name,
			EDV_LIST_PIXMAP_TEXT_SPACING,
			icon->pixmap, icon->mask
		);
	else
		gtk_clist_set_text(
			clist,
			row, column,
			name
		);
	gtk_clist_set_cell_style(
		clist,
		row, column,
		style
	);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);

	(void)edv_pixmap_unref(icon);
	(void)edv_pixmap_unref(icon_opened);
	(void)edv_pixmap_unref(icon_inaccessable);
	(void)edv_pixmap_unref(icon_hidden);
}

/*
 *	Sets the cell's size value and attributes.
 */
static void edv_vfs_browser_list_set_cell_size(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column,
	const gboolean hide_dir_size, const gboolean hide_link_size,
	const EDVSizeFormat size_format,
	const gulong block_size
)
{
	const gint border_minor = 2;
	const gchar *s;
	const EDVObjectType type = obj->type;

	if(hide_dir_size && (type == EDV_OBJECT_TYPE_DIRECTORY))
		s = "";
	else if(hide_link_size && (type == EDV_OBJECT_TYPE_LINK))
		s = "";
	else if((type == EDV_OBJECT_TYPE_DEVICE_BLOCK) ||
			(type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
	)
		s = "";
	else
		s = edv_str_size_format(
			obj->size,
			size_format,
			block_size,
			',',
			TRUE
		);

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(s != NULL) ? s : ""
	);
	gtk_clist_set_cell_style(
		clist,
		row, column,
		NULL
	);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, -border_minor
	);
	gtk_clist_thaw(clist);
}

/*
 *	Sets the cell's MIME Type value and attributes.
 */
static void edv_vfs_browser_list_set_cell_type(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column
)
{
	gchar *type_string;

	/* Get MIME Type type string for the given object */
	(void)edv_match_object_type_string(
		core->mime_types_list,
		obj->type,
		obj->path,
		obj->permissions,
		&type_string
	);

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(type_string != NULL) ? type_string : ""
	);
	gtk_clist_set_cell_style(
		clist,
		row, column,
		NULL
	);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);

	g_free(type_string);
}

/*
 *	Sets the cell's permissions value and attributes.
 */
static void edv_vfs_browser_list_set_cell_permissions(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column, gboolean hide_link_permissions
)
{
	EDVPermissionFlags permissions = obj->permissions;
	gchar *s;

	if(hide_link_permissions && (obj->type == EDV_OBJECT_TYPE_LINK))
		s = NULL;
	else
		s = edv_str_permissions(permissions);

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist, row, column,
		(s != NULL) ? s : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist, row, column, 0, 0
	);
	gtk_clist_thaw(clist);

	g_free(s);
}

/*
 *	Sets the cell's owner value and attributes.
 */
static void edv_vfs_browser_list_set_cell_owner(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column
)
{
	/* Get owner name from object's user id */
	gchar *owner_name = edv_uid_uid_to_name(
		core->uids_list,
		obj->owner_id, NULL
	);

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist, row, column,
		(owner_name != NULL) ? owner_name : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist, row, column, 0, 0
	);
	gtk_clist_thaw(clist);

	g_free(owner_name);
}

/*
 *	Sets the cell's group value and attributes.
 */
static void edv_vfs_browser_list_set_cell_group(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column
)
{
	/* Get group name from object's group id */
	gchar *group_name = edv_gid_gid_to_name(
		core->gids_list,
		obj->group_id, NULL
	);

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist, row, column,
		(group_name != NULL) ? group_name : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist, row, column, 0, 0
	);
	gtk_clist_thaw(clist);

	g_free(group_name);
}

/*
 *	Sets the cell's access date value and attributes.
 */
static void edv_vfs_browser_list_set_cell_date_access(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column,
	EDVDateRelativity date_relativity, const gchar *date_format
)
{
	/* Get/format the date string */
	const gulong t = obj->access_time;
	gchar *s = (t > 0l) ?
		edv_date_string_format(t, date_format, date_relativity) :
		g_strdup("");

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(s != NULL) ? s : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);

	g_free(s);
}

/*
 *	Sets the cell's modified date value and attributes.
 */
static void edv_vfs_browser_list_set_cell_date_modified(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column,
	EDVDateRelativity date_relativity, const gchar *date_format
)
{
	/* Get/format the date string */
	const gulong t = obj->modify_time;
	gchar *s = (t > 0l) ?
		edv_date_string_format(t, date_format, date_relativity) :
		g_strdup("");

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(s != NULL) ? s : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);

	g_free(s);
}

/*
 *	Sets the cell's changed date value and attributes.
 */
static void edv_vfs_browser_list_set_cell_date_changed(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column,
	EDVDateRelativity date_relativity, const gchar *date_format
)
{
	/* Get/format the date string */
	const gulong t = obj->change_time;
	gchar *s = (t > 0l) ?
		edv_date_string_format(t, date_format, date_relativity) :
		g_strdup("");

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(s != NULL) ? s : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);

	g_free(s);
}

/*
 *	Sets the cell's hard links value and attributes.
 */
static void edv_vfs_browser_list_set_cell_hard_links(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column
)
{
	const gint border_minor = 2;
	gchar num_str[80];

	/* Get string for number of hard links */
	g_snprintf(
		num_str, sizeof(num_str),
		"%i",
		obj->hard_links
	);

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist, row, column, num_str
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist, row, column, 0, -border_minor
	);
	gtk_clist_thaw(clist);
}

/*
 *	Sets the cell's linked to value and attributes.
 */
static void edv_vfs_browser_list_set_cell_linked_to(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column
)
{
	const gchar	*path = (obj->path != NULL) ?
						obj->path : "",
					*link_value = (obj->link_target != NULL) ?
						obj->link_target : "";
	GtkStyle	*style = NULL,
					**cell_styles = browser->cell_style;
	EDVPixmap       *icon = NULL,
					*icon_inaccessable = NULL,
					*icon_hidden = NULL;

	/* Begin checking for which GtkStyle to be used */

	/* Dangling link?
	 *
	 * Note that we use the no access style for the link
	 * destination of a dangling link. The actual dangling link
	 * style is only used for the name of a dangling link
	 */
	if(!EDV_VFS_OBJECT_LINK_TARGET_EXISTS(obj))
	{
		if(style == NULL)
			style = cell_styles[EDV_VFS_BROWSER_CELL_STYLE_NO_ACCESS];
	}
	if(EDV_VFS_OBJECT_IS_LINK_TARGET_GRAND_PARENT(obj))
	{
		if(style == NULL)
			style = cell_styles[EDV_VFS_BROWSER_CELL_STYLE_RECURSIVE_LINK];
	}

	/* Get link value */
	link_value = (obj->link_target != NULL) ? obj->link_target : "";
	if(EDV_VFS_OBJECT_LINK_TARGET_EXISTS(obj))
	{
		gchar	*parent = g_dirname(path),
					*dest_path = edv_path_evaluate(parent, link_value);
		EDVVFSObject *dest_obj = edv_vfs_object_lstat(dest_path);
		if(dest_obj != NULL)
		{
			/* Get the pixmap and mask for the link destination icon */
			(void)edv_match_object_icon(
				core->devices_list,
				core->mime_types_list,
				dest_obj->type,
				dest_obj->path,
				EDV_VFS_OBJECT_LINK_TARGET_EXISTS(dest_obj),
				dest_obj->permissions,
				EDV_ICON_SIZE_20,
				&icon,
				NULL,
				&icon_inaccessable,
				&icon_hidden
			);

			/* Checking if an alternate GtkStyle and icon
			 * should be used based on certain object types
			 */
			switch(dest_obj->type)
			{
			  case EDV_OBJECT_TYPE_UNKNOWN:
				break;
			  case EDV_OBJECT_TYPE_FILE:
				/* Hidden? */
				if(edv_is_object_hidden(dest_obj))
				{
					if(edv_pixmap_is_loaded(icon_hidden))
					{
						(void)edv_pixmap_unref(icon);
						icon = edv_pixmap_ref(icon_hidden);
					}
				}
				break;
			  case EDV_OBJECT_TYPE_DIRECTORY:
				/* Hidden? */
				if(edv_is_object_hidden(dest_obj))
				{
					if(edv_pixmap_is_loaded(icon_hidden))
					{
						(void)edv_pixmap_unref(icon);
						icon = edv_pixmap_ref(icon_hidden);
					}
				}
				/* Directory not accessable? */
				if(!edv_is_object_accessable(core, dest_obj))
				{
					if(edv_pixmap_is_loaded(icon_inaccessable))
					{
						(void)edv_pixmap_unref(icon);
						icon = edv_pixmap_ref(icon_inaccessable);
					}
					if(style == NULL)
						style = cell_styles[
							EDV_VFS_BROWSER_CELL_STYLE_NO_ACCESS
						];
				}
				break;
			  case EDV_OBJECT_TYPE_LINK:
				/* Hidden? */
				if(edv_is_object_hidden(dest_obj))
				{
					if(edv_pixmap_is_loaded(icon_hidden))
					{
						(void)edv_pixmap_unref(icon);
						icon = edv_pixmap_ref(icon_hidden);
					}
				}
				/* Infinately recursive? */
				if(EDV_VFS_OBJECT_IS_LINK_TARGET_GRAND_PARENT(dest_obj))
				{
					if(style == NULL)
						style = cell_styles[
							EDV_VFS_BROWSER_CELL_STYLE_RECURSIVE_LINK
						];
				}
				break;
			  case EDV_OBJECT_TYPE_DEVICE_BLOCK:
				/* Hidden? */
				if(edv_is_object_hidden(dest_obj))
				{
					if(edv_pixmap_is_loaded(icon_hidden))
					{
						(void)edv_pixmap_unref(icon);
						icon = edv_pixmap_ref(icon_hidden);
					}
				}
				break;
			  case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
				/* Hidden? */
				if(edv_is_object_hidden(dest_obj))
				{
					if(edv_pixmap_is_loaded(icon_hidden))
					{
						(void)edv_pixmap_unref(icon);
						icon = edv_pixmap_ref(icon_hidden);
					}
				}
				break;
			  case EDV_OBJECT_TYPE_FIFO:
				/* Hidden? */
				if(edv_is_object_hidden(dest_obj))
				{
					if(edv_pixmap_is_loaded(icon_hidden))
					{
						(void)edv_pixmap_unref(icon);
						icon = edv_pixmap_ref(icon_hidden);
					}
				}
				break;
			  case EDV_OBJECT_TYPE_SOCKET:
				/* Hidden? */
				if(edv_is_object_hidden(dest_obj))
				{
					if(edv_pixmap_is_loaded(icon_hidden))
					{
						(void)edv_pixmap_unref(icon);
						icon = edv_pixmap_ref(icon_hidden);
					}
				}
				break;
			  case EDV_OBJECT_TYPE_ERROR:
				break;
			}

			/* Parent directory? */
			if(dest_obj->name != NULL)
			{
				if(!strcmp((const char *)dest_obj->name, ".."))
				{
					if(edv_pixmap_is_loaded(browser->directory_parent_icon))
					{
						(void)edv_pixmap_unref(icon);
						icon = edv_pixmap_ref(browser->directory_parent_icon);
					}
				}
			}

			edv_vfs_object_delete(dest_obj);
		}

		g_free(dest_path);
		g_free(parent);
	}

	/* Set the cell */
	gtk_clist_freeze(clist);
	if(edv_pixmap_is_loaded(icon))
		gtk_clist_set_pixtext(
			clist,
			row, column,
			link_value,
			EDV_LIST_PIXMAP_TEXT_SPACING,
			icon->pixmap, icon->mask
		);
	else
		gtk_clist_set_text(
			clist,
			row, column,
			link_value
		);
	gtk_clist_set_cell_style(
		clist,
		row, column,
		style
	);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);

	(void)edv_pixmap_unref(icon);
	(void)edv_pixmap_unref(icon_inaccessable);
	(void)edv_pixmap_unref(icon_hidden);
}

/*
 *	Sets the cell's device index value and attributes.
 */
static void edv_vfs_browser_list_set_cell_device(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column
)
{
	const gint border_minor = 2;
	gchar num_str[80];

	/* Get string for device */
	g_snprintf(
		num_str, sizeof(num_str),
		"%ld",
		obj->device_index
	);

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist, row, column, num_str
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist, row, column, 0, -border_minor
	);
	gtk_clist_thaw(clist);
}

/*
 *	Sets the cell's index value and attributes.
 */
static void edv_vfs_browser_list_set_cell_index(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column
)
{
	const gint border_minor = 2;
	gchar num_str[80];

	g_snprintf(
		num_str, sizeof(num_str),
		"%ld",
		obj->index
	);

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist, row, column, num_str
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist, row, column, 0, -border_minor
	);
	gtk_clist_thaw(clist);
}

/*
 *	Sets the cell's device type value and attributes.
 */
static void edv_vfs_browser_list_set_cell_device_type(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column
)
{
	const gint border_minor = 2;
	const EDVObjectType type = obj->type;
	gint major, minor;
	gchar num_str[2 * 80];

	/* Is object a block or character device? */
	if((type == EDV_OBJECT_TYPE_DEVICE_BLOCK) ||
	   (type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
	)
	{
		edv_device_numbers_parse(
			(dev_t)obj->device_type, &major, &minor
		);
		g_snprintf(
			num_str, sizeof(num_str),
			"%i, %i",
			major, minor
		);
	}
	else
	{
		*num_str = '\0';
	}

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist, row, column, num_str
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist, row, column, 0, -border_minor
	);
	gtk_clist_thaw(clist);
}

/*
 *	Sets the cell's block size value and attributes.
 */
static void edv_vfs_browser_list_set_cell_block_size(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column
)
{
	const gint border_minor = 2;
	const gchar *s = edv_str_size_delim(obj->block_size);

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist, row, column,
		(s != NULL) ? s : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist, row, column, 0, -border_minor
	);
	gtk_clist_thaw(clist);
}

/*
 *	Sets the cell's blocks value and attributes.
 */
static void edv_vfs_browser_list_set_cell_blocks(
	EDVCore *core, EDVVFSBrowser *browser,
	GtkCList *clist, EDVVFSObject *obj,
	const gint row, const gint column
)
{
	const gint border_minor = 2;
	gchar num_str[80];

	/* Get string for number of blocks */
	g_snprintf(
		num_str, sizeof(num_str),
		"%ld",
		obj->blocks
	);

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist, row, column, num_str
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist, row, column, 0, -border_minor
	);
	gtk_clist_thaw(clist);
}

/*
 *	Sets the Contents List's row to the values of the object.
 *
 *	The clist specifies the Contents List.
 *
 *	The obj specifies the object.
 *
 *	The row specifies the row.
 *
 *	All inputs assumed valid.
 */
static void edv_vfs_browser_list_set_row(
	EDVVFSBrowser *browser,
	const gint row,
	EDVVFSObject *obj
)
{
	gboolean	hide_dir_size,
					hide_link_size,
					hide_link_permissions;
	gint i;
	gulong block_size;
	const gchar *date_format;
	GList *glist;
	CfgIntList *column_types_intlist;
	EDVSizeFormat size_format;
	EDVDateRelativity date_relativity;
	EDVVFSBrowserColumnType column_type;
	GtkCList *clist = GTK_CLIST(browser->contents_clist);
	EDVCore *core = browser->core;
	CfgList *cfg_list = core->cfg_list;

	/* Get the column types mapping */
	column_types_intlist = EDV_GET_INT_LIST(
		EDV_CFG_PARM_BROWSER_CONTENTS_COLUMN
	);
	if(column_types_intlist == NULL)
		return;

	/* Get the display options */
	size_format = (EDVSizeFormat)EDV_GET_I(
		EDV_CFG_PARM_SIZE_FORMAT
	);
	block_size = EDV_GET_UL(EDV_CFG_PARM_BLOCK_SIZE);
	date_relativity = (EDVDateRelativity)EDV_GET_I(
		EDV_CFG_PARM_DATE_RELATIVITY
	);
	date_format = EDV_GET_S(EDV_CFG_PARM_DATE_FORMAT);
	hide_dir_size = EDV_GET_B(EDV_CFG_PARM_BROWSER_CONTENTS_HIDE_DIR_SIZE);
	hide_link_size = EDV_GET_B(EDV_CFG_PARM_BROWSER_CONTENTS_HIDE_LINK_SIZE);
	hide_link_permissions = EDV_GET_B(EDV_CFG_PARM_BROWSER_CONTENTS_HIDE_LINK_PERMISSIONS);

	gtk_clist_freeze(clist);

	/* Iterate through each column */
	for(i = 0, glist = column_types_intlist->list;
		glist != NULL;
		i++, glist = g_list_next(glist)
	)
	{
		column_type = (EDVVFSBrowserColumnType)glist->data;
		switch(column_type)
		{
		  case EDV_VFS_BROWSER_COLUMN_TYPE_NAME:
			edv_vfs_browser_list_set_cell_name(
				core, browser, clist, obj,
				row, i
			);
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_SIZE:
			edv_vfs_browser_list_set_cell_size(
				core, browser, clist, obj,
				row, i, hide_dir_size, hide_link_size,
				size_format, block_size
			);
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_TYPE:
			edv_vfs_browser_list_set_cell_type(
				core, browser, clist, obj,
				row, i
			);
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_PERMISSIONS:
			edv_vfs_browser_list_set_cell_permissions(
				core, browser, clist, obj,
				row, i, hide_link_permissions
			);
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_OWNER:
			edv_vfs_browser_list_set_cell_owner(
				core, browser, clist, obj,
				row, i
			);
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_GROUP:
			edv_vfs_browser_list_set_cell_group(
				core, browser, clist, obj,
				row, i
			);
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_DATE_ACCESS:
			edv_vfs_browser_list_set_cell_date_access(
				core, browser, clist, obj,
				row, i, date_relativity, date_format
			);
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_DATE_MODIFIED:
			edv_vfs_browser_list_set_cell_date_modified(
				core, browser, clist, obj,
				row, i, date_relativity, date_format
			);
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_DATE_CHANGED:
			edv_vfs_browser_list_set_cell_date_changed(
				core, browser, clist, obj,
				row, i, date_relativity, date_format
			);
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_HARD_LINKS:
			edv_vfs_browser_list_set_cell_hard_links(
				core, browser, clist, obj,
				row, i
			);
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_LINKED_TO:
			edv_vfs_browser_list_set_cell_linked_to(
				core, browser, clist, obj,
				row, i
			);
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_DEVICE_INDEX:
			edv_vfs_browser_list_set_cell_device(
				core, browser, clist, obj,
				row, i
			);
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_INDEX:
			edv_vfs_browser_list_set_cell_index(
				core, browser, clist, obj,
				row, i
			);
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_DEVICE_TYPE:
			edv_vfs_browser_list_set_cell_device_type(
				core, browser, clist, obj,
				row, i
			);
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_BLOCK_SIZE:
			edv_vfs_browser_list_set_cell_block_size(
				core, browser, clist, obj,
				row, i
			);
			break;
		  case EDV_VFS_BROWSER_COLUMN_TYPE_BLOCKS:
			edv_vfs_browser_list_set_cell_blocks(
				core, browser, clist, obj,
				row, i
			);
			break;
		}
	}

	gtk_clist_thaw(clist);
}

/*
 *	Appends the object to the Contents List.
 *
 *	The obj specifies the object who's values will be used to
 *	append a new row on the Contents List and will be transfered
 *	to the Content List's row data. The obj should not be
 *	referenced again after this call.
 *
 *	Returns the new row index or -1 on error.
 *
 *	All inputs assumed valid.
 */
static gint edv_vfs_browser_list_append_object(
	EDVVFSBrowser *browser,
	EDVVFSObject *obj
)
{
	gint		i,
					new_row;
	gchar **strv;
	GtkCList *clist = GTK_CLIST(browser->contents_clist);
	const gint ncolumns = MAX(clist->columns, 1);

	gtk_clist_freeze(clist);

	/* Allocate the row cell values */
	strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
		strv[i] = "";

	/* Append the new row */
	new_row = gtk_clist_append(clist, strv);

	/* Delete the row cell values */
	g_free(strv);

	/* Failed to append row? */
	if(new_row < 0)
	{
		gtk_clist_thaw(clist);
		edv_vfs_object_delete(obj);
		return(-1);
	}

	/* Set the new row's values */
	edv_vfs_browser_list_set_row(
		browser,
		new_row,
		obj
	);

	/* Set this object as the new row's data */
	gtk_clist_set_row_data_full(
		clist,
		new_row,
		obj, (GtkDestroyNotify)edv_vfs_object_delete
	);

	gtk_clist_thaw(clist);

	return(new_row);
}

/*
 *	Appends the objects from a specified location to the Contents
 *	List.
 *
 *	The path specifies the full path to the location.
 *
 *	If show_progress is TRUE then the status bar will be
 *	updated during this operation.
 *
 *	All inputs assumed valid.
 */
static void edv_vfs_browser_list_append_listing(
	EDVVFSBrowser *browser,
	const gchar *path,
	const gboolean show_progress
)
{
#ifdef HAVE_REGEX
	regex_t *regex_filter;
#endif
	gboolean	hide_object_hidden,
					hide_object_noaccess;
	const gchar *filter = browser->contents_list_filter;
	GList *names_list;
	GtkWidget *sb = browser->status_bar;
	EDVCore *core = browser->core;
	CfgList *cfg_list = core->cfg_list;

	if(browser->stop_count > 0)
		return;

	hide_object_hidden = !EDV_GET_B(
		EDV_CFG_PARM_BROWSER_SHOW_OBJECT_HIDDEN
	);
	hide_object_noaccess = !EDV_GET_B(
		EDV_CFG_PARM_BROWSER_SHOW_OBJECT_NOACCESS
	);

#if defined(HAVE_REGEX)
	/* Compile the regex search criteria */
	if(STRISEMPTY(filter) ?
		FALSE : strcmp((const char *)filter, "*")
	)
	{
		regex_filter = (regex_t *)g_malloc(sizeof(regex_t));
		if(regcomp(
			regex_filter,
			(const char *)filter,
#ifdef REG_EXTENDED
			REG_EXTENDED |		/* Use POSIX extended regex */
#endif
			REG_NOSUB		/* Do not report subpattern matches */
		))
		{
			g_free(regex_filter);
			regex_filter = NULL;
		}
	}
	else
	{
		regex_filter = NULL;
	}
#else
	if(STRISEMPTY(filter) ?
		TRUE : !strcmp((const char *)filter, "*")
	)
		filter = NULL;
#endif

	/* Add the parent directory */
	if(TRUE)
	{
		gchar *full_path = g_dirname(path);
		EDVVFSObject *obj = edv_vfs_object_lstat(full_path);
		if(obj != NULL)
		{
			/* Set the name of the object as the previous directory
			 * notation
			 */
			edv_vfs_object_set_path2(
				obj,
				"..",
				full_path
			);

			/* Append/transfer this object to the listing */
			(void)edv_vfs_browser_list_append_object(
				browser,
				obj
			);
		}

		g_free(full_path);
	}

	/* Get the list of object names from the specified location */
	names_list = edv_directory_list(
		path,
		TRUE,				/* Sort */
		FALSE				/* Exclude notations */
	);
	if(names_list != NULL)
	{
		const gint nobjs = g_list_length(names_list);
		gint	nobjs_loaded = 0,
					last_progress_percent = -1;
		const gchar *name;
		gchar *full_path;
		GList *glist;
		EDVVFSObject *obj;

		/* Report the number of objects being loaded? */
		if(show_progress && (nobjs > 0))
		{
/* Do not print how many objects because some objects may be
 * filtered out?
 */
			gchar *msg = g_strdup_printf(
"Loading directory contents (%i %s)...",
				nobjs,
				(nobjs == 1) ? "object" : "objects"
			);
			edv_status_bar_message(sb, msg, TRUE);
			g_free(msg);
		}

#define UPDATE_PROGRESS	{				\
 if(show_progress && (nobjs > 0)) {			\
  const gint progress_percent = nobjs_loaded *		\
   100 / nobjs;						\
  if(progress_percent > last_progress_percent) {	\
   edv_status_bar_progress(				\
    sb,							\
    (gfloat)progress_percent / 100.0f,			\
    TRUE						\
   );							\
   last_progress_percent = progress_percent;		\
  }							\
 }							\
}

		/* Load the directories */
		for(glist = names_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			if(browser->stop_count > 0)
			{
				do {
					g_free(glist->data);
					glist->data = NULL;
					glist = g_list_next(glist);
				} while(glist != NULL);
				break;
			}

			name = (const gchar *)glist->data;
			if(name == NULL)
				continue;

			/* Get this object's full path */
			full_path = edv_paths_join(
				path,
				name
			);
			if(full_path == NULL)
			{
				/* Count this object as being loaded */
				nobjs_loaded++;
				UPDATE_PROGRESS
				g_free(glist->data);
				glist->data = NULL;
				continue;
			}

			/* Get this object's destination statistics */
			obj = edv_vfs_object_stat(full_path);
			if(obj == NULL)
			{
				/* Failed to get the statistics of this object
				 * but do not add it as an error object on
				 * this directory iteration, it will be added
				 * later in the second iteration
				 */
				g_free(full_path);
				continue;
			}

			/* Skip if the destination is not a directory */
			if(!EDV_VFS_OBJECT_IS_DIRECTORY(obj))
			{
				edv_vfs_object_delete(obj);
				g_free(full_path);
				continue;
			}

			edv_vfs_object_delete(obj);

			/* Get this directory's local statistics */
			obj = edv_vfs_object_lstat(full_path);
			if(obj == NULL)
			{
				/* Unable to get the directory's local statistics
				 * so add it as an error object
				 */
				const gint error_code = (gint)errno;
				obj = edv_vfs_object_new_error(
					full_path,
					error_code,
					g_strerror(error_code)
				);
				if(obj != NULL)
					edv_vfs_browser_list_append_object(
						browser,
						obj
					);

				/* Count this object as being loaded */
				nobjs_loaded++;
				UPDATE_PROGRESS
				g_free(full_path);
				g_free(glist->data);
				glist->data = NULL;
				continue;
			}

			/* Skip hidden or no access objects? */
			if((hide_object_hidden ?
				edv_is_object_hidden(obj) : FALSE) ||
			   (hide_object_noaccess ?
				!edv_is_object_accessable(core, obj) : FALSE)
			)
			{
				edv_vfs_object_delete(obj);
			}
			else
			{
				/* Append/transfer this object to the listing */
				edv_vfs_browser_list_append_object(
					browser,
					obj
				);
			}

			/* Count this object as being loaded */
			nobjs_loaded++;
			UPDATE_PROGRESS
			g_free(full_path);
			g_free(glist->data);
			glist->data = NULL;
		}

		/* Load the rest of the objects
		 *
		 * This pass will delete all the entries in names_list
		 *
		 * Do not reset last_progress_percent, leave it as the
		 * last value from the previous loop
		 */
		for(glist = names_list;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			if(browser->stop_count > 0)
			{
				do {
					g_free(glist->data);
					glist = g_list_next(glist);
				} while(glist != NULL);
				break;
			}

			name = (const gchar *)glist->data;
			if(name == NULL)
				continue;

			/* Filter check */
#if defined(HAVE_REGEX)
			if(regex_filter != NULL)
			{
				if(regexec(
					regex_filter,
					name,
					0, NULL,
					0
				) == REG_NOMATCH)
				{
					/* Count this object as being loaded */
					nobjs_loaded++;
					UPDATE_PROGRESS
					g_free(glist->data);
					continue;
				}
			}
#else
			if(filter != NULL)
			{
				if(fnmatch(filter, name, 0) == FNM_NOMATCH)
				{
					/* Count this object as being loaded */
					nobjs_loaded++;
					UPDATE_PROGRESS
					g_free(glist->data);
					continue;
				}
			}
#endif

			/* Get this object's full path */
			full_path = edv_paths_join(
				path,
				name
			);
			if(full_path == NULL)
			{
				/* Count this object as being loaded */
				nobjs_loaded++;
				UPDATE_PROGRESS
				g_free(glist->data);
				continue;
			}

			/* Get this object's local statistics */
			obj = edv_vfs_object_lstat(full_path);
			if(obj == NULL)
			{
				/* Unable to get the object's local statistics
				 * so add it as an error object
				 */
				const gint error_code = (gint)errno;
				obj = edv_vfs_object_new_error(
					full_path,
					error_code,
					g_strerror(error_code)
				);
				if(obj != NULL)
					edv_vfs_browser_list_append_object(
						browser,
						obj
					);

				/* Count this object as being loaded */
				nobjs_loaded++;
				UPDATE_PROGRESS
				g_free(full_path);
				g_free(glist->data);
				continue;
			}

			/* Skip hidden objects */
			if((hide_object_hidden ?
				edv_is_object_hidden(obj) : FALSE)
/*                  ||
			   (hide_object_noaccess ?
				!edv_is_object_accessable(core, obj) : FALSE)
 */
			)
			{
				edv_vfs_object_delete(obj);
			}
			else
			{
				/* Append/transfer this object to the listing */
				edv_vfs_browser_list_append_object(
					browser,
					obj
				);
			}

			/* Count this object as being loaded */
			nobjs_loaded++;
			UPDATE_PROGRESS
			g_free(full_path);
			g_free(glist->data);
		}

#undef UPDATE_PROGRESS

		/* Delete the directory entries list, each entry should
		 * already be deleted
		 */
		g_list_free(names_list);
	}

#ifdef HAVE_REGEX
	if(regex_filter != NULL)
	{
		regfree(regex_filter);
		g_free(regex_filter);
	}
#endif
}


/* 
 *	Gets a list of selected paths.
 *
 *	Returns a GList of gchar * strings describing the full paths
 *	of the selected objects or NULL on error.
 */
static GList *edv_vfs_browser_list_get_selected_paths(EDVVFSBrowser *browser)
{
	GList		*glist,
			*paths_list = NULL;
	GtkCList *clist = GTK_CLIST(browser->contents_clist);
	EDVVFSObject *obj;

	/* Create the selected paths list */
	for(glist = clist->selection;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		obj = EDV_VFS_OBJECT(gtk_clist_get_row_data(
			clist,
			(gint)glist->data
		));
		if(obj == NULL)
			continue;

		if(STRISEMPTY(obj->path))
			continue;

		paths_list = g_list_append(
			paths_list,
			g_strdup(obj->path)
		);
	}

	return(paths_list);
}


/*
 *	Finds the row who's object matches the location index.
 *
 *	The device_index specifies the index of the device that the
 *	object resides on.
 *
 *	The index specifies the index on the device that the object
 *	resides at.
 *
 *	Returns the matched row or negative on error.
 */
gint edv_vfs_browser_list_find_by_index(
	EDVVFSBrowser *browser, 
	const gulong device_index, const gulong index
)
{
	gint i, n;
	GtkCList *clist;
	EDVVFSObject *obj;

	if(browser == NULL)
		return(-2);

	clist = GTK_CLIST(browser->contents_clist);

	n = clist->rows;
	for(i = 0; i < n; i++)
	{
		obj = EDV_VFS_OBJECT(gtk_clist_get_row_data(clist, i));
		if(obj == NULL)
			continue;

		if((obj->device_index == device_index) &&
		   (obj->index == index)
		)
			return(i);
	}

	return(-1);
}

/*
 *	Finds the row who's object matches the path.
 *
 *	The path specifies the full path of the object.
 *
 *	Returns the matched row or negative on error.
 */
gint edv_vfs_browser_list_find_by_path(
	EDVVFSBrowser *browser,
	const gchar *path
)
{
	gint i, n;
	GtkCList *clist;
	EDVVFSObject *obj;

	if((browser == NULL) || STRISEMPTY(path))
		return(-2);

	clist = GTK_CLIST(browser->contents_clist);

	n = clist->rows;
	for(i = 0; i < n; i++)
	{
		obj = EDV_VFS_OBJECT(gtk_clist_get_row_data(clist, i));
		if(obj == NULL)
			continue;

		if(obj->path == NULL)
			continue;

		/* Full paths match? */
		if(!strcmp((const char *)obj->path, (const char *)path))
			return(i);
	}

	return(-1);
}

/*
 *	Updates all the existing rows on the Contents GtkCList by
 *	getting each row's object and updating the row's cell values
 *	with it.
 */
void edv_vfs_browser_list_realize_listing(EDVVFSBrowser *browser)
{
	gint i, n;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVVFSObject *obj;
	EDVCore *core;

	if(browser == NULL)
		return;

	browser->stop_count = 0;

	clist = GTK_CLIST(browser->contents_clist);
	core = browser->core;
	cfg_list = core->cfg_list;

	gtk_clist_freeze(clist);

	/* Reset the columns */
	edv_vfs_browser_list_reset_columns(browser);

	/* Automatically resize the columns to optimul widths based on
	 * their headings?
	 */
	if(EDV_GET_B(EDV_CFG_PARM_BROWSER_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
		edv_vfs_browser_list_resize_columns_optimul(browser);

	/* Update the rows */
	n = clist->rows;
	for(i = 0; i < n; i++)
	{
		if(browser->stop_count > 0)
			break;

		obj = EDV_VFS_OBJECT(gtk_clist_get_row_data(clist, i));
		if(obj == NULL)
			continue;

		edv_vfs_browser_list_set_row(
			browser,
			i,				/* Row */
			obj
		);
	}

	/* Automatically resize the columns to optimul widths based on
	 * their row values?
	 */
	if(EDV_GET_B(EDV_CFG_PARM_BROWSER_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
		edv_vfs_browser_list_resize_columns_optimul(browser);

	gtk_clist_thaw(clist);
}


/*
 *	Clears the Contents List, resets the columns, and gets a new
 *	listing of objects.
 *
 *	The path specifies the full path to the location.
 *
 *	If show_progress is TRUE then the progress and messages will
 *	be displayed on the status bar during this operation.
 */
void edv_vfs_browser_list_get_listing(
	EDVVFSBrowser *browser,
	const gchar *path,
	const gboolean show_progress
)
{
	GtkWidget	*w,
					*sb;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVCore *core;

	if(browser == NULL)
		return;

	browser->stop_count = 0;

	w = browser->contents_clist;
	clist = GTK_CLIST(w);
	sb = browser->status_bar;
	core = browser->core;
	cfg_list = core->cfg_list;

	/* Report the initial progress? */
	if(show_progress)
	{
		edv_status_bar_message(
			sb,
"Loading directory contents...",
			FALSE
		);
		edv_status_bar_progress(sb, 0.0f, TRUE);
	}

	gtk_clist_freeze(clist);

	/* Clear the listing */
	edv_vfs_browser_list_clear(browser);

	/* Reset the columns */
	edv_vfs_browser_list_reset_columns(browser);

	/* Automatically resize the columns to optimul widths based on
	 * their headings?
	 */
	if(EDV_GET_B(EDV_CFG_PARM_BROWSER_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
		edv_vfs_browser_list_resize_columns_optimul(browser);

	/* Append the list of objects to the listing */
	if(!STRISEMPTY(path) &&
	   (browser->stop_count == 0)
	)
		edv_vfs_browser_list_append_listing(
			browser,
			path,				/* Location */
			show_progress
		);

	/* Automatically resize the columns to optimul widths based on
	 * their row values?
	 */
	if(EDV_GET_B(EDV_CFG_PARM_BROWSER_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
		edv_vfs_browser_list_resize_columns_optimul(browser);

	gtk_clist_thaw(clist);

	/* Report the final progress? */
	if(show_progress)
	{
		edv_status_bar_message(
			sb,
"Directory contents loaded",
			FALSE
		);
		edv_status_bar_progress(sb, 0.0f, TRUE);
	}
}

/*
 *	Clears the Contents List.
 */
void edv_vfs_browser_list_clear(EDVVFSBrowser *browser)
{
	GtkCList *clist;

	if(browser == NULL)
		return;

	clist = GTK_CLIST(browser->contents_clist);
	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	gtk_clist_thaw(clist);
}


/*
 *	Opens the object.
 *
 *	The row and column specify the object displayed on the
 *	Contents List. If row and column are -1 then the selected
 *	object(s) will be used instead.
 *
 *	The state specifies the current modifier key states.
 */
void edv_vfs_browser_list_open(
	EDVVFSBrowser *browser,
	const gint row, const gint column,
	const guint state
)
{
	gint npaths;
	const gchar *path;
	GList		*glist,
			*paths_list;
	GtkWidget *toplevel;
	EDVCore *core;

	if(browser == NULL)
		return;

	toplevel = browser->toplevel;
	core = browser->core;

	/* Use the selected object(s)? */
	if(row < 0)
	{
		paths_list = edv_vfs_browser_list_get_selected_paths(browser);
	}
	else
	{
		GtkCList *clist = GTK_CLIST(browser->contents_clist);
		EDVVFSObject *obj = EDV_VFS_OBJECT(
			gtk_clist_get_row_data(clist, row)
		);
		if(obj == NULL)
			return;

		if(STRISEMPTY(obj->path))
			return;

		paths_list = NULL;
		paths_list = g_list_append( 
			paths_list,
			g_strdup(obj->path)
		);
	}

	/* Count the number of paths in the list and get the path to
	 * the last path in the list
	 */
	npaths = 0;
	path = NULL;
	glist = paths_list;
	while(glist != NULL)
	{
		npaths++;
		path = (const gchar *)glist->data;
		glist = g_list_next(glist);
	}

	/* Check if there is only one object selected and that it
	 * refers to a directory
	 */
	if(npaths == 1)
	{
		gchar *path = edv_path_evaluate(
			NULL, (gchar *)paths_list->data
		);
		if(edv_path_is_directory(path))
		{
			/* Change directory and return */
			edv_vfs_browser_goto_directory_cb(browser, path);
			g_list_foreach(paths_list, (GFunc)g_free, NULL);
			g_list_free(paths_list);
			g_free(path);
			return;
		}
		g_free(path);
	}

	/* Confirm open */
	if(edv_confirm_open(
		core,
		toplevel,
		path,
		npaths
	) != CDIALOG_RESPONSE_YES)
	{
		g_list_foreach(paths_list, (GFunc)g_free, NULL);
		g_list_free(paths_list);
		return;
	}

	/* Open */
	if(paths_list != NULL)
	{
		const gchar *command_name = NULL;

		edv_vfs_browser_set_busy(browser, TRUE);

		if(state & GDK_CONTROL_MASK)
			command_name = "edit";
		else if(state & GDK_SHIFT_MASK)
			command_name = "edit";

		(void)edv_open(
			core,
			paths_list,
			command_name,
			toplevel,
			TRUE			/* Verbose */
		);

		g_list_foreach(
			paths_list,
			(GFunc)g_free,
			NULL
		);
		g_list_free(paths_list);

		edv_vfs_browser_set_busy(browser, FALSE);
	}
}

/*
 *	Maps the Open With list to open the object.
 *
 *	The row and column specify the object displayed on the
 *	Contents List. If row and column are -1 then the selected
 *	object(s) will be used instead.
 */
void edv_vfs_browser_list_open_with(
	EDVVFSBrowser *browser,
	const gint row, const gint column
)
{
	gint npaths;
	const gchar *path;
	GList		*glist,
					*paths_list;
	GtkWidget *toplevel;
	EDVCore *core;

	if(browser == NULL)
		return;

	toplevel = browser->toplevel;
	core = browser->core;

	/* Use selected object paths? */
	if(row < 0)
	{
		/* Get list of selected object paths */
		paths_list = edv_vfs_browser_list_get_selected_paths(browser);
	}
	else
	{   
		GtkCList *clist = GTK_CLIST(browser->contents_clist);
		EDVVFSObject *obj = EDV_VFS_OBJECT(
			gtk_clist_get_row_data(clist, row)
		);
		if(obj == NULL)
			return;

		if(STRISEMPTY(obj->path))
			return;

		paths_list = NULL;
		paths_list = g_list_append(
			paths_list,
			g_strdup(obj->path)
		);
	}

	/* Count the number of paths in the list and get the path to
	 * the last path in the list
	 */
	npaths = 0;
	path = NULL;
	glist = paths_list;
	while(glist != NULL)
	{
		npaths++;
		path = (const gchar *)glist->data;
		glist = g_list_next(glist);
	}

	/* Confirm open */
	if(edv_confirm_open(
		core,
		toplevel,
		path,
		npaths
	) != CDIALOG_RESPONSE_YES)
	{
		g_list_foreach(
			paths_list,
			(GFunc)g_free,
			NULL
		);
		g_list_free(paths_list);
		return;
	}

	/* Open With */
	if(paths_list != NULL)
	{
		(void)edv_open_with(
			core,
			paths_list,
			NULL,			/* Default command */
			toplevel,
			TRUE			/* Verbose */
		);

		g_list_foreach(
			paths_list,
			(GFunc)g_free,
			NULL
		);
		g_list_free(paths_list);
	}
}


/*
 *	FPrompt rename apply callback.
 */
static void edv_vfs_browser_list_prompt_rename_apply_cb(
	gpointer data, const gchar *value
)
{
	gint row;
	EDVVFSBrowser *browser;
	EDVVFSBrowserListPromptData *d = EDV_VFS_BROWSER_LIST_PROMPT_DATA(data);
	if(d == NULL)
		return;

	browser = d->vfs_browser;
	row = d->row;

	/* Inputs valid? */
	if((browser != NULL) && (row > -1) && (value != NULL))
	{
		GtkWidget	*toplevel = browser->toplevel,
					*sb = browser->status_bar;
		GtkCList *clist = GTK_CLIST(browser->contents_clist);
		EDVCore *core = browser->core;

		/* Get the object from the selected row */
		EDVVFSObject *obj = EDV_VFS_OBJECT(
			gtk_clist_get_row_data(clist, row)
		);

		/* Check if the selected object's disk object structure
		 * is valid
		 */
		if((obj != NULL) ? (obj->path != NULL) : FALSE)
		{
			gboolean yes_to_all = FALSE;
			const gchar *error_msg;
			gchar *old_full_path = g_strdup(obj->path);
			GList *modified_paths_list;

			/* Rename */
			(void)edv_vfs_object_op_rename(
				core,
				old_full_path, value,
				&modified_paths_list,
				toplevel,
				FALSE,			/* Do not show progress */
				TRUE,			/* Interactive */
				&yes_to_all
			);

			/* Unmap the progress dialog */
			ProgressDialogBreakQuery(FALSE);
			ProgressDialogSetTransientFor(NULL);

			/* Check for errors */
			error_msg = edv_vfs_object_op_get_error(core);
			if(!STRISEMPTY(error_msg))
			{
				/* Report the error */
				edv_play_sound_error(core);
				edv_message_error(
					"Rename Error",
					error_msg,
					NULL,
					toplevel
				);
			}

			/* Report the modified objects */
			if(modified_paths_list != NULL)
			{
				const gchar *modified_path;
				GList *glist;
				EDVVFSObject *obj;

				for(glist = modified_paths_list;
					glist != NULL;
					glist = g_list_next(glist)
				)
				{
					modified_path = (const gchar *)glist->data;
					if(modified_path == NULL)
						continue;

					obj = edv_vfs_object_lstat(modified_path);
					if(obj != NULL)
					{
						gchar *msg = g_strdup_printf(
							"Object \"%s\" renamed to \"%s\"",
							g_basename(old_full_path),
							g_basename(modified_path)
						);
						edv_status_bar_message(sb, msg, FALSE);
						g_free(msg);

						edv_emit_vfs_object_modified(
							core,
							old_full_path,
							modified_path,
							obj
						);

						edv_vfs_object_delete(obj);
					}
				}

				/* Delete the modified paths list */
				g_list_foreach(
					modified_paths_list, (GFunc)g_free, NULL
				);
				g_list_free(modified_paths_list);
			}
			else
			{   
				/* Did not get the modified object path so this
				 * implies failure
				 */
				edv_status_bar_message(
					sb,
					"Rename object failed",
					FALSE
				);
			}

			g_free(old_full_path);
		}
	}

	g_free(d);
}

/*
 *	FPrompt rename cancel callback.
 */
static void edv_vfs_browser_list_prompt_rename_cancel_cb(gpointer data)
{
	EDVVFSBrowserListPromptData *d = EDV_VFS_BROWSER_LIST_PROMPT_DATA(data);
	if(d == NULL)
		return;

	g_free(d);
}

/*
 *	Prompts to rename the object.
 *
 *	The row and column specifies the object on the list to map
 *	the rename prompt at. If column is -1 then the column that
 *	is set to EDV_VFS_BROWSER_COLUMN_TYPE_NAME will be used.
 */
void edv_vfs_browser_list_rename_query(
	EDVVFSBrowser *browser,
	const gint row, const gint column
)
{
	gint cx, cy, px, py, pwidth, pheight, name_column;
	GtkWidget *toplevel;
	GtkCList *clist;
	const CfgList *cfg_list;
	CfgIntList *column_type_intlist;
	EDVVFSObject *obj;
	EDVCore *core;

	if((browser == NULL) || FPromptIsQuery())
		return;

	toplevel = browser->toplevel;
	clist = GTK_CLIST(browser->contents_clist);
	core = browser->core;
	cfg_list = core->cfg_list;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
		return;

	edv_vfs_browser_sync_data(browser);

	/* Row does not exist? */
	if((row < 0) || (row >= clist->rows))
		return;

	/* Find which column is displaying the name */
	name_column = -1;
	column_type_intlist = EDV_GET_INT_LIST(
		EDV_CFG_PARM_BROWSER_CONTENTS_COLUMN
	);
	if(column_type_intlist != NULL)
	{
		gint i = 0;
		GList *glist;

		for(glist = column_type_intlist->list;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			if((EDVVFSBrowserColumnType)glist->data ==
				EDV_VFS_BROWSER_COLUMN_TYPE_NAME
			)
			{
				name_column = i;
				break;
			}
			i++;
		}
	}
	/* No column displaying the name? */
	if(name_column < 0)
		return;

	/* Get the cell's geometry */
	if(!gtk_clist_get_cell_geometry(
		clist, name_column, row,
		&cx, &cy, &pwidth, &pheight
	))
		return;

	/* Get the root window relative coordinates */
	px = 0;
	py = 0;
	gdk_window_get_deskrelative_origin(
		clist->clist_window, &px, &py
	);
	px += cx + 0;
	py += cy - 2;		/* Move up a little */

	/* Get the object */
	obj = EDV_VFS_OBJECT(gtk_clist_get_row_data(clist, row));
	if(obj == NULL)
		return;

	/* Check if the object's name is a special notation that
	 * may not be renamed
	 */
	if(!STRISEMPTY(obj->name))
	{
		const gchar *name = obj->name;
		if(!strcmp(name, ".") || !strcmp(name, "..") ||
		   !strcmp(name, "/")
		)
			return;
	}

	/* Is the specified column the name column or -1? */
	if((column == name_column) || (column < 0))
	{
		gchar *value = STRDUP(obj->name);
		EDVVFSBrowserListPromptData *d = EDV_VFS_BROWSER_LIST_PROMPT_DATA(
			g_malloc(sizeof(EDVVFSBrowserListPromptData))
		);
		if(d != NULL)
		{
			d->vfs_browser = browser;
			d->row = row;
			d->column = column;
		}

		/* Map floating prompt to change values */
		FPromptSetTransientFor(toplevel);
		FPromptSetPosition(px, py);
		FPromptMapQuery(
			NULL,			/* No label */
			value,
			NULL,			/* No tooltip message */
			FPROMPT_MAP_TO_POSITION,	/* Map code */
			pwidth, -1,		/* Width and height */
			0,			/* No buttons */
			d,			/* Callback data */
			NULL,			/* No browse callback */
			edv_vfs_browser_list_prompt_rename_apply_cb,
			edv_vfs_browser_list_prompt_rename_cancel_cb
		);

		g_free(value);
	}
}


/*
 *	Object added callback.
 */
void edv_vfs_browser_list_vfs_object_added_cb(
	EDVVFSBrowser *browser,
	const gchar *path,
	EDVVFSObject *obj
)
{
	gint row;
	gchar		*cur_path,
					*parent_path;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVCore *core;

	if((browser == NULL) || STRISEMPTY(path) || (obj == NULL))
		return;

	cur_path = STRDUP(edv_vfs_browser_get_location(browser));
	if(cur_path == NULL)
		return;

	clist = GTK_CLIST(browser->contents_clist);
	core = browser->core;
	cfg_list = core->cfg_list;

	/* Is the current location the same as the added object? */
	if(!strcmp((const char *)cur_path, (const char *)path))
	{
		/* Reget the listing */
		edv_vfs_browser_list_get_listing(
			browser,
			path,
			EDV_GET_B(EDV_CFG_PARM_LISTS_SHOW_PROGRESS)
		);
		g_free(cur_path);
		return;
	}

	/* Update the object who's path matches the added path */
	row = edv_vfs_browser_list_find_by_path(browser, path);
	if(row > -1)
	{
		/* Update this object */
		EDVVFSObject *tar_obj = EDV_VFS_OBJECT(
			gtk_clist_get_row_data(clist, row)
		);
		if(obj != NULL)
		{
			edv_vfs_object_set_object(tar_obj, obj);

			/* Update the Contents List row with the new values
			 * from this object
			 */
			edv_vfs_browser_list_set_row(
				browser,
				row,
				tar_obj
			);

			/* Resize the columns */
			if(EDV_GET_B(EDV_CFG_PARM_BROWSER_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
				edv_vfs_browser_list_resize_columns_optimul(browser);
		}
		g_free(cur_path);
		return;
	}

	/* Get the parent directory of the added object */
	parent_path = g_dirname(path);
	if(parent_path != NULL)
	{
		/* Is the parent directory of the added object the same as
		 * the current location?
		 */
		if(!strcmp((const char *)parent_path, (const char *)cur_path))
		{
			/* Add this object to the list */
			edv_vfs_browser_list_append_object(
				browser,
				edv_vfs_object_copy(obj)
			);

			/* Resize the columns */
			if(EDV_GET_B(EDV_CFG_PARM_BROWSER_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
				edv_vfs_browser_list_resize_columns_optimul(browser);
		}
		g_free(parent_path);
	}

	g_free(cur_path);
}

/*
 *	Object modified callback.
 */
void edv_vfs_browser_list_vfs_object_modified_cb(
	EDVVFSBrowser *browser,
	const gchar *path,
	const gchar *new_path,
	EDVVFSObject *obj
)
{
	gint row;
	gchar *cur_path;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVCore *core;

	if((browser == NULL) || STRISEMPTY(path) || (obj == NULL))
		return;

	if(new_path == NULL)
		new_path = path;

	cur_path = STRDUP(edv_vfs_browser_get_location(browser));
	if(cur_path == NULL)
		return;

	clist = GTK_CLIST(browser->contents_clist);
	core = browser->core;
	cfg_list = core->cfg_list;

	/* Is the new modified path the same as the current location? */
	if(!strcmp((const char *)new_path, (const char *)cur_path))
	{
		/* Reget the listing */
		edv_vfs_browser_list_get_listing(
			browser,
			new_path,
			EDV_GET_B(EDV_CFG_PARM_LISTS_SHOW_PROGRESS)
		);
		g_free(cur_path);
		return;
	}

	/* Update the object who's path matches the modified path */
	row = edv_vfs_browser_list_find_by_path(browser, path);
	if(row > -1)
	{
		/* Update this object */
		EDVVFSObject *tar_obj = EDV_VFS_OBJECT(
			gtk_clist_get_row_data(clist, row)
		);
		if(tar_obj != NULL)
		{
			edv_vfs_object_set_object(tar_obj, obj);

			/* Update the Contents List row with the new values
			 * from this object
			 */
			edv_vfs_browser_list_set_row(
				browser,
				row,
				tar_obj
			);

			/* Resize the columns */
			if(EDV_GET_B(EDV_CFG_PARM_BROWSER_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
				edv_vfs_browser_list_resize_columns_optimul(browser);
		}
	}

	g_free(cur_path);
}

/*
 *	Object removed callback/
 */
void edv_vfs_browser_list_vfs_object_removed_cb(
	EDVVFSBrowser *browser,
	const gchar *path
)
{
	gchar *cur_path;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVCore *core;

	if((browser == NULL) || STRISEMPTY(path))
		return;

	cur_path = STRDUP(edv_vfs_browser_get_location(browser));
	if(cur_path == NULL)
		return;

	clist = GTK_CLIST(browser->contents_clist);
	core = browser->core;
	cfg_list = core->cfg_list;

	/* Is the removed path the same as the current location? */
	if(!strcmp((const char *)path, (const char *)cur_path))
	{
		/* Clear the list */
		edv_vfs_browser_list_clear(browser);
	}
	else
	{
		/* Remove all the row who's object's path matches the
		 * removed path
		 */
		gint row = edv_vfs_browser_list_find_by_path(browser, path);
		if(row > -1)
		{
			gtk_clist_freeze(clist);
			do {
				gtk_clist_remove(clist, row);
				row = edv_vfs_browser_list_find_by_path(browser, path);
			} while(row > -1);
			gtk_clist_thaw(clist);

			/* Resize the columns */
			if(EDV_GET_B(EDV_CFG_PARM_BROWSER_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
				edv_vfs_browser_list_resize_columns_optimul(browser);
		}
	}

	g_free(cur_path);
}


/*
 *	Device mount/unmount callback.
 */
void edv_vfs_browser_list_device_mount_cb(
	EDVVFSBrowser *browser,
	EDVDevice *d,
	const gboolean mounted
)
{
	gchar		*cur_path,
					*mount_path;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVCore *core;

	if((browser == NULL) || (d == NULL))
		return;

	clist = GTK_CLIST(browser->contents_clist);
	core = browser->core;
	cfg_list = core->cfg_list;

	/* Get the current location and mount path */
	cur_path = STRDUP(edv_vfs_browser_get_location(browser));
	mount_path = STRDUP(d->mount_path);
	if((cur_path == NULL) || (mount_path == NULL))
	{
		g_free(cur_path);
		g_free(mount_path);
		return;
	}

	edv_path_simplify(cur_path);
	edv_path_simplify(mount_path);

	/* Check if mount path was the current location, if it was
	 * then the Contents List needs to be updated
	 */
	if(!strcmp((const char *)cur_path, (const char *)mount_path))
	{
		/* Reget the listing */
		edv_vfs_browser_list_get_listing(
			browser,
			cur_path,
			EDV_GET_B(EDV_CFG_PARM_LISTS_SHOW_PROGRESS)
		);
	}

	g_free(cur_path);
	g_free(mount_path);
}
