#include <gtk/gtk.h>

#include "url.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "edv_utils_gtk.h"
#include "edv_status_bar.h"
#include "vfs_browser.h"
#include "vfs_browser_dnd.h"
#include "edv_dnd.h"

#include "config.h"


static gint EDVBrowserDragDataReceivedNexus(
	EDVVFSBrowser *browser,
	GdkDragContext *dc,
	const guint info,
	GtkSelectionData *selection_data,
	EDVVFSObject *obj
);

gint edv_vfs_browser_location_bar_icon_crossing_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);

void edv_vfs_browser_tree_drag_data_get_cb(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
void edv_vfs_browser_tree_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
void edv_vfs_browser_tree_drag_data_delete_cb(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
);

void edv_vfs_browser_list_drag_data_get_cb(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
void edv_vfs_browser_list_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
void edv_vfs_browser_list_drag_data_delete_cb(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Drag data received nextus.
 *
 *	The dc specifies the GdkDragContext.
 *
 *	The info specifies the source object's location type,
 *
 *	The selection_data specifies the DND data describing the
 *	source objects in the form of a null-separated list of
 *	URL strings which will be parsed into a list of URL strings.
 *
 *	The tar_obj specifies the target location. If tar_obj is NULL
 *	then the current location will be used as the target location.
 *
 *	Returns 0 on success or non-zero on error.
 */
static gint EDVBrowserDragDataReceivedNexus(
	EDVVFSBrowser *browser,
	GdkDragContext *dc,
	const guint info,
	GtkSelectionData *selection_data,
	EDVVFSObject *tar_obj
)
{
	gint status;
	EDVCore *core = browser->core;
	gchar *tar_path = (tar_obj != NULL) ? STRDUP(tar_obj->path) : NULL;
	if(tar_path == NULL)
	    tar_path = STRDUP(edv_vfs_browser_get_location(browser));

	edv_vfs_browser_set_busy(browser, TRUE);

	/* Check if the object is an archive, and if so, then add to
         * the archive isntead
         */
        if(edv_check_archive_format_supported(core, g_basename(tar_path)))
        {
            gchar *password = NULL;

            /* Process the DND operation to the VFS */
            status = edv_dnd_vfs_to_archive(
                core,
                dc,
                info,				/* Source location type */
                selection_data,			/* Source objects */
                tar_path,			/* Target archive path */
                &password,			/* May be modified */
                TRUE,				/* Verbose */
                browser->toplevel,
                browser->status_bar
            );

	    g_free(password);
        }
        else
        {
            /* Process a DND operation to the VFS */
	    status = edv_dnd_any_to_vfs(
                core,
                dc,
                info,				/* Source location type */
                selection_data,			/* Source objects */
                tar_path,			/* Target location */
                TRUE,				/* Verbose */
                browser->toplevel,
                browser->status_bar
            );
        }

	g_free(tar_path);

	edv_vfs_browser_set_busy(browser, FALSE);

	return(status);
}

/*
 *	Location Bar icon "enter_notify_event" or "leave_notify_event"
 *	signal callback.
 */
gint edv_vfs_browser_location_bar_icon_crossing_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	gint status = FALSE;
	GtkWidget *sb;
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((widget == NULL) || (crossing == NULL) || (browser == NULL))
	    return(status);

	sb = browser->status_bar;

	switch((gint)crossing->type)
	{
	  case GDK_ENTER_NOTIFY:
	    if(browser->location_icon_pm != NULL)
	    {
		GdkBitmap *mask;
		GdkPixmap *pixmap;
		GtkWidget *w = browser->location_icon_pm;
		gtk_pixmap_get(GTK_PIXMAP(w), &pixmap, &mask);
		if(pixmap != NULL)
		    GUIDNDSetDragIcon(
			widget,
			pixmap, mask,
			GUI_DND_DRAG_ICON_DEF_HOT_X, GUI_DND_DRAG_ICON_DEF_HOT_Y
		    );
	    }
	    edv_status_bar_message(
		sb,
#if defined(PROG_LANGUAGE_SPANISH)
"Arrastre esto crear una conexin a esta ubicacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Traner ceci pour crer un lien  cet emplacement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Schleppen sie dies, eine verbindung zu diesem ort zu schaffen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Trascinare questo per creare una maglia a questa posizione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Sleep dit om een schakel aan deze plaats te creren"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Arraste isto criar um elo a esta localidade"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Slep dette skape et ledd til denne plasseringen"
#else
"Drag this to create a link to this location"
#endif
		, FALSE
	    );
	    status = TRUE;
	    break;

	  case GDK_LEAVE_NOTIFY:
	    edv_status_bar_message(sb, NULL, FALSE);
	    status = TRUE;
	    break;
	}

	return(status);
}


/*
 *	Directory GtkCTree "drag_data_get" signal callback.
 */
void edv_vfs_browser_tree_drag_data_get_cb(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gboolean data_sent = FALSE;
	GList		*glist,
			*url_list;
	GtkWidget *w;
	GtkCList *clist;
	GtkCTree *ctree;
	URLStruct *url;
	EDVVFSObject *obj;
	EDVCore *core;
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((dc == NULL) || (browser == NULL))
	    return;

	w = browser->directory_ctree;
	ctree = GTK_CTREE(w);
	clist = GTK_CLIST(w);
	core = browser->core;

	edv_vfs_browser_sync_data(browser);

	/* Generate a list of URLs from the selected nodes */
	url_list = NULL;
	for(glist = clist->selection;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    obj = EDV_VFS_OBJECT(gtk_ctree_node_get_row_data(
		ctree,
		(GtkCTreeNode *)glist->data
	    ));
	    if(obj == NULL)
		continue;

	    url = url_new();
	    if(url == NULL)
		break;

	    url->path = STRDUP(obj->path);
	    url_list = g_list_append(
		url_list,
		url
	    );
	}

	/* Encode the DDE buffer from the URL list */
	if(url_list != NULL)
	{
	    gint buf_len;
	    guint8 *buf = url_encode(
		url_list,
		&buf_len
	    );
	    if(buf != NULL)
	    {
		/* Send out the buffer */
		gtk_selection_data_set(
		    selection_data,
		    GDK_SELECTION_TYPE_STRING,
		    8,				/* Bits Per Character */
		    buf,			/* Data */
		    buf_len			/* Length */
		);
		data_sent = TRUE;
		g_free(buf);
	    }
	}

	/* Delete the URLs list */
	if(url_list != NULL)
	{
	    g_list_foreach(
		url_list,
		(GFunc)url_delete,
		NULL
	    );
	    g_list_free(url_list);
	}

	/* If failed to send out data then respond with error */
	if(!data_sent)
	{
	    const gchar *s = "Error";
	    gtk_selection_data_set(
		selection_data,
		GDK_SELECTION_TYPE_STRING,
		8,				/* Bits Per Character */
		s,				/* Data */
		strlen(s)			/* Length */
	    );
	    data_sent = TRUE;
	}
}

/*
 *	Directory GtkCTree "drag_data_received" signal callback.
 */
void edv_vfs_browser_tree_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	GtkWidget *w;
	GtkCList *clist;
	GtkCTree *ctree;
	GtkCTreeNode *node;
	EDVVFSObject *obj;
	EDVCore *core;
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((dc == NULL) || (browser == NULL))
	    return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser) || (browser->freeze_count > 0))
	    return;

	/* No source data? */
	if(selection_data == NULL)
	    return;
	if(selection_data->length <= 0)
	    return;

	w = browser->directory_ctree;
	ctree = GTK_CTREE(w);
	clist = GTK_CLIST(w);
	core = browser->core;

	edv_vfs_browser_sync_data(browser);

	/* Find the node and object that this drop occured on */
	node = edv_ctree_node_get_by_coordinates(
	    ctree,
	    x,
	    y - ((clist->flags & GTK_CLIST_SHOW_TITLES) ?
		clist->column_title_area.height +
		clist->column_title_area.y : 0)
	);
	if(node == NULL)
	    return;

	obj = EDV_VFS_OBJECT(gtk_ctree_node_get_row_data(
	    ctree,
	    node
	));
	if(obj == NULL)
	    return;

	if(obj->path == NULL)
	    return;

	/* Perform the drag operation */
	(void)EDVBrowserDragDataReceivedNexus(
	    browser,
	    dc,
	    info,
	    selection_data,
	    obj
	);
}

/*
 *	Directory GtkCTree "drag_data_delete" signal callback.
 */
void edv_vfs_browser_tree_drag_data_delete_cb(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{

}


/*
 *	Contents GtkCList "drag_data_get" signal callback.
 */
void edv_vfs_browser_list_drag_data_get_cb(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gboolean data_sent = FALSE;
	GList		*glist,
			*url_list = NULL;
	GtkWidget *w;
	GtkCList *clist;
	URLStruct *url;
	EDVVFSObject *obj;
	EDVCore *core;
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((dc == NULL) || (browser == NULL))
	    return;

	w = browser->contents_clist;
	clist = GTK_CLIST(w);
	core = browser->core;

	edv_vfs_browser_sync_data(browser);

	/* Generate a list of URLs from the selected rows */
	for(glist = clist->selection;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    obj = EDV_VFS_OBJECT(gtk_clist_get_row_data(
		clist,
		(gint)glist->data
	    ));
	    if(obj == NULL)
		continue;

	    url = url_new();
	    if(url == NULL)
		break;

	    url->path = STRDUP(obj->path);
	    url_list = g_list_append(
		url_list,
		url
	    );
	}

	/* Encode DDE buffer from the URL list */
	if(url_list != NULL)
	{
	    gint buf_len;
	    guint8 *buf = url_encode(url_list, &buf_len);
	    if(buf != NULL)
	    {
		/* Send out DND data buffer */
		gtk_selection_data_set(
		    selection_data,
		    GDK_SELECTION_TYPE_STRING,
		    8,				/* Bits Per Character */
		    buf,			/* Data */
		    buf_len			/* Length */
		);
		data_sent = TRUE;
		g_free(buf);
	    }
	}

	/* Delete the URL list */
	if(url_list != NULL)
	{
	    g_list_foreach(
		url_list,
		(GFunc)url_delete,
		NULL
	    );
	    g_list_free(url_list);
	}

	/* If failed to send out data then respond with error */ 
	if(!data_sent)
	{
	    const gchar *s = "Error";
	    gtk_selection_data_set(
		selection_data,
		GDK_SELECTION_TYPE_STRING,
		8,				/* Bits Per Character */
		s,				/* Data */
		strlen(s)			/* Length */
	    );
	}
}

/*
 *	Contents GtkCList "drag_data_received" signal callback.
 */
void edv_vfs_browser_list_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gint row, column;
	GtkWidget *w;
	GtkCList *clist;
	EDVVFSObject *obj;
	EDVCore *core;
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((dc == NULL) || (browser == NULL))
	    return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser) || (browser->freeze_count > 0))
	    return;

	/* No source data? */
	if(selection_data == NULL)
	    return;
	if(selection_data->length <= 0)
	    return;

	w = browser->contents_clist;
	clist = GTK_CLIST(w);
	core = browser->core;

	edv_vfs_browser_sync_data(browser);

	/* Find row that this drop occured on, then get the object
	 * referenced by the row
	 */
	/* Find row and column based on given coordinates */
	if(!gtk_clist_get_selection_info(
	    clist,
	    x,
	    y - ((clist->flags & GTK_CLIST_SHOW_TITLES) ?
		clist->column_title_area.height +
		clist->column_title_area.y : 0),
	    &row, &column
	))
	{
	    row = -1;
	    column = 0;
	}

	if((row >= 0) && (row < clist->rows))
	    obj = EDV_VFS_OBJECT(gtk_clist_get_row_data(
		clist,
		row
	    ));
	else
	    obj = NULL;

	/* Perform the drag operation */
	(void)EDVBrowserDragDataReceivedNexus(
	    browser,
	    dc,
	    info,
	    selection_data,
	    obj
	);
}

/*
 *	File Browser Contents List "drag_data_delete" signal callback.
 */
void edv_vfs_browser_list_drag_data_delete_cb(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{

}
