#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"

#include "cfg.h"

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "edv_device.h"
#include "edv_utils_gtk.h"
#include "edv_device_mount.h"
#include "edv_devices_list.h"
#include "edv_list_seek.h"
#include "edv_status_bar.h"
#include "vfs_browser.h"
#include "vfs_browser_cb.h"
#include "vfs_browser_op.h"
#include "vfs_browser_tree.h"
#include "vfs_browser_list.h"
#include "vfs_browser_dnd.h"
#include "edv_vfs_obj_create.h"
#include "edv_open.h"
#include "edv_emit.h"
#include "edv_cb.h"
#include "edv_op.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


#include "images/icon_endeavour_file_browser_48x48.xpm"


typedef struct _EDVVFSBrowserTreeCBData	EDVVFSBrowserTreeCBData;
#define EDV_VFS_BROWSER_DIR_TREE_CB_DATA(p)	\
					((EDVVFSBrowserTreeCBData *)(p))


/* GTK+ Signal Callbacks */
void edv_vfs_browser_realize_cb(GtkWidget *widget, gpointer data);
gint edv_vfs_browser_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
gint edv_vfs_browser_key_event_cb(
	 GtkWidget *widget, GdkEventKey *key, gpointer data
);
gint edv_vfs_browser_button_event_cb(
	GtkWidget *widget, GdkEventButton *button, gpointer data
);

void edv_vfs_browser_handle_child_attach_cb(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
);
void edv_vfs_browser_handle_child_detach_cb(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
);

void edv_vfs_browser_location_bar_icon_realize_cb(GtkWidget *widget, gpointer data);

static gint EDVBrowserTreeSelectRowIdleCB(gpointer data);
static gint EDVBrowserTreeExpandScrollOptimizeIdleCB(gpointer data);
void edv_vfs_browser_tree_select_row_cb(
	GtkCTree *ctree, GtkCTreeNode *node, gint column,
	gpointer data
);
void edv_vfs_browser_tree_unselect_row_cb(
	GtkCTree *ctree, GtkCTreeNode *node, gint column,
	gpointer data
);
void edv_vfs_browser_tree_expand_cb(
	GtkCTree *ctree, GtkCTreeNode *node, gpointer data
);
void edv_vfs_browser_tree_collapse_cb(
	GtkCTree *ctree, GtkCTreeNode *node, gpointer data
);

void edv_vfs_browser_list_realize_cb(GtkWidget *widget, gpointer data);
static gint EDVBrowserCListColumnSortSizeCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint EDVBrowserCListColumnSortDateNexus(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2,
	gint sort_code
);
static gint EDVBrowserCListColumnSortDateAccessCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint EDVBrowserCListColumnSortDateModifyCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint EDVBrowserCListColumnSortDateChangeCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint EDVBrowserCListColumnSortHardLinksCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint EDVBrowserCListColumnSortIndexCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint EDVBrowserCListColumnSortDeviceIndexCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint EDVBrowserCListColumnSortBlockSizeCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint EDVBrowserCListColumnSortBlocksCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
void edv_vfs_browser_resize_column_cb(
	GtkCList *clist, gint column, gint width, gpointer data
);
void edv_vfs_browser_click_column_cb(
	GtkCList *clist, gint column, gpointer data
);
void edv_vfs_browser_list_select_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
void edv_vfs_browser_list_unselect_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);

void edv_vfs_browser_location_combo_activate_cb(GtkWidget *widget, gpointer data);


/* Menu Item Callbacks */
void edv_vfs_browser_menu_item_cb(GtkWidget *widget, gpointer data);
gint edv_vfs_browser_menu_item_enter_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
gint edv_vfs_browser_menu_item_leave_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
void edv_vfs_browser_menu_item_cmd_object_op_cb(GtkWidget *widget, gpointer data);
void edv_vfs_browser_menu_item_cmd_new_object_cb(GtkWidget *widget, gpointer data);
gint edv_vfs_browser_menu_item_cmd_enter_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
gint edv_vfs_browser_menu_item_cmd_leave_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);


/* Paths List Get Callback */
GList *edv_vfs_browser_get_selected_paths_cb(gpointer data);


/* Goto Directory Callback */
void edv_vfs_browser_goto_directory_cb(gpointer data, const gchar *path);

/* Mount Bar Callbacks */
void edv_vfs_browser_mount_bar_mount_cb(
	GtkWidget *widget, const gint dev_num, EDVDevice *dev,
	gpointer data
);
void edv_vfs_browser_mount_bar_eject_cb(
	GtkWidget *widget, const gint dev_num, EDVDevice *dev,
	gpointer data
);
void edv_vfs_browser_mount_bar_goto_cb(
	GtkWidget *widget, const gint dev_num, EDVDevice *dev,
	gpointer data
);

/* Find Bar Callbacks */
const gchar *edv_vfs_browser_find_bar_location_cb(
	GtkWidget *bar,
	gpointer data
);
void edv_vfs_browser_find_bar_start_cb(
	GtkWidget *bar,
	gpointer data
);
void edv_vfs_browser_find_bar_end_cb(
	GtkWidget *bar,
	const gint nmatches,
	gpointer data
);
void edv_vfs_browser_find_bar_match_cb(
	GtkWidget *bar,
	const gchar *path,
	GList *properties_list,
	const gint line_index,
	const gchar *excerpt,
	gpointer data
);

/* Status Bar Callbacks */
void edv_vfs_browser_status_message_cb(
	GtkWidget *widget,
	const gchar *message,
	gpointer data
);
void edv_vfs_browser_status_progress_cb(
	GtkWidget *widget,
	const gfloat progress,
	gpointer data
);

/* Window Created/Deleted Callbacks */
void edv_vfs_browser_window_created_cb(
	EDVVFSBrowser *browser,
	const EDVWindowType win_type,
	gpointer win
);
void edv_vfs_browser_window_deleted_cb(
	EDVVFSBrowser *browser,
	const EDVWindowType win_type,
	gpointer win
);

/* Write Protect Changed Callback */
void edv_vfs_browser_master_write_protect_changed_cb(
	EDVVFSBrowser *browser,
	const gboolean state
);

/* Delete Method Changed Callback */
void edv_vfs_browser_delete_method_changed_cb(
	EDVVFSBrowser *browser,
	const EDVDeleteMethod delete_method
);

/* Object Callbacks */
void edv_vfs_browser_vfs_object_added_cb(
	EDVVFSBrowser *browser,
	const gchar *path,
	EDVVFSObject *obj
);
void edv_vfs_browser_vfs_object_modified_cb(
	EDVVFSBrowser *browser,
	const gchar *path,
	const gchar *new_path,
	EDVVFSObject *obj
);
void edv_vfs_browser_vfs_object_removed_cb(
	EDVVFSBrowser *browser,
	const gchar *path
);

/* Mount/Unmount Callback */
void edv_vfs_browser_device_mount_cb(
	EDVVFSBrowser *browser,
	const gint dev_num, EDVDevice *d,
	const gboolean mounted
);

/* Recycled Object Callbacks */
void edv_vfs_browser_recycled_object_added_cb(
	EDVVFSBrowser *browser,
	const guint index
);
void edv_vfs_browser_recycled_object_modified_cb(
	EDVVFSBrowser *browser,
	const guint index
);
void edv_vfs_browser_recycled_object_removed_cb(
	EDVVFSBrowser *browser,
	const guint index
);

/* Reconfigured Callback */
void edv_vfs_browser_reconfigured_cb(EDVVFSBrowser *browser);

/* MIME Type Callbacks */
void edv_vfs_browser_mime_type_added_cb(
	EDVVFSBrowser *browser,
	const gint mt_num, EDVMIMEType *m
);
void edv_vfs_browser_mime_type_modified_cb(
	EDVVFSBrowser *browser,
	const gint mt_num, EDVMIMEType *m
);
void edv_vfs_browser_mime_type_removed_cb(
	EDVVFSBrowser *browser,
	const gint mt_num
);

/* Device Callbacks */
void edv_vfs_browser_device_added_cb(
	EDVVFSBrowser *browser,
	const gint dev_num, EDVDevice *d
);
void edv_vfs_browser_device_modified_cb(
	EDVVFSBrowser *browser,
	const gint dev_num, EDVDevice *d
);
void edv_vfs_browser_device_removed_cb(
	EDVVFSBrowser *browser,
	const gint dev_num
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


struct _EDVVFSBrowserTreeCBData {
	EDVVFSBrowser	*vfs_browser;
	GtkCTreeNode	*node;
};


/*
 *	Toplevel GtkWidget "realize" signal callback.
 */
void edv_vfs_browser_realize_cb(GtkWidget *widget, gpointer data)
{
	GdkWindow *window;
	GtkStyle *style;
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((widget == NULL) || (browser == NULL))
		return;

	window = widget->window;
	style = gtk_widget_get_style(widget);

	if(window != NULL)
	{
		GdkGeometry geo;
		geo.min_width = 320;
		geo.min_height = 240;
		geo.base_width = 0;
		geo.base_height = 0;
		geo.width_inc = 1;
		geo.height_inc = 1;
		gdk_window_set_geometry_hints(
			window,
			&geo,
			GDK_HINT_MIN_SIZE | GDK_HINT_BASE_SIZE |
				GDK_HINT_RESIZE_INC
		);
		gdk_window_set_hints(
			window,
			0, 0,				/* Position */
			geo.min_width, geo.min_height,	/* Min size */
			geo.max_width, geo.max_height,	/* Max size */
			GDK_HINT_MIN_SIZE
		);
		GUISetWMIcon(
			window,
			(guint8 **)icon_endeavour_file_browser_48x48_xpm
		);
	}

	browser->flags |= EDV_VFS_BROWSER_REALIZED;
}

/*
 *	Toplevel GtkWidget "delete_event" signal callback.
 */
gint edv_vfs_browser_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if(browser == NULL)
		return(FALSE);

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser) || (browser->freeze_count > 0))
	{
		browser->stop_count++;
		return(TRUE);
	}

	edv_vfs_browser_op_close(browser);

	return(TRUE);
}

/*
 *	Any GtkWidget "key_press_event" or "key_release_event" signal
 *	callback.
 */
gint edv_vfs_browser_key_event_cb(
	 GtkWidget *widget, GdkEventKey *key, gpointer data
)
{
	gint status = FALSE;
	gboolean is_press;
	gint etype;
	guint		keyval,
					state;
	GtkWidget	*w,
					*focus_widget;
	CfgList *cfg_list;
	EDVCore *core;
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((widget == NULL) || (key == NULL) || (browser == NULL))
		return(status);

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser) || (browser->freeze_count > 0))
		return(status);

	w = browser->toplevel;
	focus_widget = GTK_WINDOW(w)->focus_widget;
	core = browser->core;
	cfg_list = core->cfg_list;

	etype = key->type;
	is_press = (etype == GDK_KEY_PRESS) ? TRUE : FALSE;
	keyval = key->keyval;
	state = key->state;

#define STOP_SIGNAL_EMISSION(_w_) {		\
 gtk_signal_emit_stop_by_name(			\
  GTK_OBJECT(_w_),				\
  is_press ?					\
   "key_press_event" : "key_release_event"	\
 );						\
}

	/* If the focus_widget is not a GtkEditable then check if the
	 * keyval is an accelerator key before all subsequence checks
	 */
	if((focus_widget != NULL) ?
		!GTK_IS_EDITABLE(focus_widget) : TRUE
	)
	{
		EDVVFSBrowserOpID op = (EDVVFSBrowserOpID)edv_match_accel_key_op_id(
			cfg_list, EDV_CFG_PARM_BROWSER_ACCELERATOR_KEYS,
			keyval, state
		);
		if(op > 0)
		{
			if(is_press)
			{
				EDVVFSBrowserOp *opid = edv_vfs_browser_op_match_by_id(
					browser, op
				);
				if(opid != NULL)
					edv_vfs_browser_op_cb(NULL, -1, opid);
			}
			STOP_SIGNAL_EMISSION(widget);
			status = TRUE;
			return(status);
		}
	}

	/* Check which widget this signal is for
	 *
	 * Directory GtkCTree
	 */
	if(widget == browser->directory_ctree)
	{
		GtkCList *clist = GTK_CLIST(widget);
		GtkCTree *ctree = GTK_CTREE(widget);
		GtkCTreeNode *node = edv_ctree_get_selected_last(ctree, NULL);

		switch(keyval)
		{
		  case GDK_space:
		  case GDK_KP_Space:
			node = gtk_ctree_node_nth(ctree, clist->focus_row);
			if((node != NULL) && is_press)
			{
				gboolean is_selected;
				GList *glist;

				/* Check if this node is already selected */
				is_selected = FALSE;
				for(glist = clist->selection;
					glist != NULL;
					glist = g_list_next(glist)
				)
				{
					if(node == (GtkCTreeNode *)glist->data)
					{
						is_selected = TRUE;
						break;
					}
				}

				gtk_clist_freeze(clist);
				if(is_selected)
					gtk_ctree_unselect(ctree, node);
				else
					gtk_ctree_select(ctree, node);
				gtk_clist_thaw(clist);
			}
			STOP_SIGNAL_EMISSION(widget);
			status = TRUE;
			break;
		}
	}
	/* Contents GtkCList */
	else if(widget == browser->contents_clist)
	{
		GtkCList *clist = GTK_CLIST(widget);
		gint row = edv_clist_get_selected_last(clist, NULL);

		switch(keyval)
		{
		  case GDK_space:
		  case GDK_KP_Space:
			row = clist->focus_row;
			if((row >= 0) && (row < clist->rows) && is_press)
			{
				gboolean is_selected;
				GList *glist;

				/* Check if this row is already selected */
				is_selected = FALSE;
				for(glist = clist->selection;
					glist != NULL;
					glist = g_list_next(glist)
				)
				{
					if(row == (gint)glist->data)
					{
						is_selected = TRUE;
						break;
					}
				}

				gtk_clist_freeze(clist);
				if(is_selected)
					gtk_clist_unselect_row(clist, row, 0);
				else
					gtk_clist_select_row(clist, row, 0);
				gtk_clist_thaw(clist);
			}
			STOP_SIGNAL_EMISSION(widget);
			status = TRUE;
			break;

		  default:
			/* For all other alphanumeric character keys and while
			 * no modifier keys are held, attempt to seek to the
			 * item who's name starts with the letter of the key
			 * that was pressed
			 */
			if(EDV_GET_B(EDV_CFG_PARM_LISTS_KEYBOARD_SCROLL_TO_KEY_NAME) &&
			   isalnum((int)keyval) && is_press)
			{
				const gboolean backwards = (state & GDK_MOD1_MASK) ? TRUE : FALSE;
				const gint start_row = (backwards) ?
					(clist->rows - 1) : 0;

				/* Find the column that is set to display the name */
				gint column_num_name = edv_vfs_browser_list_get_column_index_by_type(
					browser,
					EDV_VFS_BROWSER_COLUMN_TYPE_NAME
				);
				if(column_num_name < 0)
					column_num_name = 0;

				/* Seek to the row who's text matches this
				 * key event's character
				 */
				gtk_clist_freeze(clist);
				edv_clist_seek_character(
					clist,
					column_num_name,	/* Column */
					start_row,
					backwards,
					0,			/* Text index */
					keyval			/* Character */
				);
				gtk_clist_thaw(clist);

				STOP_SIGNAL_EMISSION(widget);
				status = TRUE;
			}
			break;
		}
	}

#undef STOP_SIGNAL_EMISSION

	return(status);
}

/*
 *	Any GtkWidget "button_press_event" signal callback.
 */
gint edv_vfs_browser_button_event_cb(
	GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	gint status = FALSE;
	gint etype;
	guint state;
	const CfgList *cfg_list;
	EDVCore *core;
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((widget == NULL) || (button == NULL) || (browser == NULL))
		return(status);

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser) || (browser->freeze_count > 0))
		return(status);

	core = browser->core;
	cfg_list = core->cfg_list;
	etype = (gint)button->type;
	state = button->state;

	/* Check which widget this signal is for
	 *
	 * Directory GtkCTree
	 */
	if(widget == browser->directory_ctree)
	{
		gint	row, column,
					nodes_selected = 0;
		GList *glist;
		GtkCTreeNode *selected_node = NULL;
		GtkCList *clist = GTK_CLIST(widget);
		GtkCTree *ctree = GTK_CTREE(widget);
		GtkCTreeNode *node = edv_ctree_node_get_by_coordinates(
			ctree,
			button->x,
			button->y - ((clist->flags & GTK_CLIST_SHOW_TITLES) ?
			clist->column_title_area.height +
			clist->column_title_area.y : 0)
		);

		/* Find row and column based on given coordinates */
		if(!gtk_clist_get_selection_info(
			clist,
			(gint)button->x, (gint)button->y,
			&row, &column
		))
		{
			row = -1;
			column = 0;
		}

		/* Get number of selected rows and the last selected row */
		glist = clist->selection;
		while(glist != NULL)
		{
			selected_node = (GtkCTreeNode *)glist->data;
			if(selected_node != NULL)
				nodes_selected++;
			glist = g_list_next(glist);
		}

		/* Handle by the button number */
		switch(button->button)
		{
		  case GDK_BUTTON2:
			switch((EDVListsPointerOpButton2)EDV_GET_I(EDV_CFG_PARM_LISTS_POINTER_OP_BUTTON2))
			{
			  case EDV_LISTS_POINTER_OP_BUTTON2_NONE:
			  case EDV_LISTS_POINTER_OP_BUTTON2_SCROLL_XY:
				/* Generic list callback handles this */
				break;
			  case EDV_LISTS_POINTER_OP_BUTTON2_RENAME:
				if(etype == GDK_BUTTON_PRESS)
				{
					/* Rename */
					edv_vfs_browser_tree_rename_query(
						browser,
						node
					);
				}
				status = TRUE;
				break;
			  case EDV_LISTS_POINTER_OP_BUTTON2_PASTE:
				if(etype == GDK_BUTTON_PRESS)
				{
					/* Paste objects */
					edv_vfs_browser_op_paste2(
						browser,
						EDV_VFS_OBJECT(gtk_ctree_node_get_row_data(ctree, node))
					);
				}
				status = TRUE;
				break;
			}
			break;

		  case GDK_BUTTON3:			/* Right Click GtkMenu */
			if(etype == GDK_BUTTON_PRESS)
			{
				/* Select the item before mapping the menu? */
				if(EDV_GET_B(EDV_CFG_PARM_RIGHT_CLICK_MENU_SELECTS) &&
				   (node != NULL)
				)
				{
					/* Select the node that the button was pressed over */
					gtk_clist_freeze(clist);
					if((row >= 0) && (row < clist->rows))
						clist->focus_row = row;
					gtk_ctree_select(ctree, node);
					gtk_clist_thaw(clist);
				}

				/* Update all the menus and then map the
				 * right-click menu
				 */
				edv_vfs_browser_update_display(browser);
				gtk_menu_popup(
					GTK_MENU(browser->directory_ctree_menu),
					NULL, NULL,
					NULL, NULL,
					button->button, button->time
				);
			}
			status = TRUE;
			break;
		}
	}
	/* Contents GtkCList */
	else if(widget == browser->contents_clist)
	{
		gint	row, column,
					rows_selected = 0,
					selected_row = -1;
		GList *glist;
		GtkCList *clist = GTK_CLIST(widget);

		/* Find the row and column */
		if(!gtk_clist_get_selection_info(
			clist,
			(gint)button->x, (gint)button->y,
			&row, &column
		))
		{
			row = -1;
			column = 0;
		}

		/* Get the number of selected rows and the last selected row */
		glist = clist->selection;
		while(glist != NULL)
		{
			rows_selected++;
			selected_row = (gint)glist->data;
			glist = g_list_next(glist);
		}

		/* Handle by button number */
		switch(button->button)
		{
		  case GDK_BUTTON2:
			switch((EDVListsPointerOpButton2)EDV_GET_I(EDV_CFG_PARM_LISTS_POINTER_OP_BUTTON2))
			{
			  case EDV_LISTS_POINTER_OP_BUTTON2_NONE:
			  case EDV_LISTS_POINTER_OP_BUTTON2_SCROLL_XY:
				/* Generic list callback handles this */
				break;
			  case EDV_LISTS_POINTER_OP_BUTTON2_RENAME:
				if(etype == GDK_BUTTON_PRESS)
				{
					/* Rename */
					edv_vfs_browser_list_rename_query(
						browser,
						row,
						column
					);
				}
				status = TRUE;
				break;
			  case EDV_LISTS_POINTER_OP_BUTTON2_PASTE:
				if(etype == GDK_BUTTON_PRESS)
				{
					/* Paste objects */
					edv_vfs_browser_op_paste2(
						browser,
						EDV_VFS_OBJECT(gtk_clist_get_row_data(clist, row))
					);
				}
				status = TRUE;
				break;
			}
			break;

		  case GDK_BUTTON3:			/* Right Click GtkMenu */
			if(etype == GDK_BUTTON_PRESS)
			{
				/* Select the item before mapping the menu? */
				if(EDV_GET_B(EDV_CFG_PARM_RIGHT_CLICK_MENU_SELECTS) &&
				   (row >= 0) && (row < clist->rows)
				)
				{
					/* Select the row that the button was pressed over.
					 * if no key modifiers are held then this will also
					 * unselect all previously selected rows
					 */
					gtk_clist_freeze(clist);
					if(!(state & GDK_CONTROL_MASK) &&
					   !(state & GDK_SHIFT_MASK)
					)
						gtk_clist_unselect_all(clist);
					clist->focus_row = row;
					gtk_clist_select_row(clist, row, 0);
					gtk_clist_thaw(clist);
				}

				/* Update all the menus and then map the
				 * right-click menu
				 */
				edv_vfs_browser_update_display(browser);
				gtk_menu_popup(
					GTK_MENU(browser->contents_clist_menu),
					NULL, NULL,
					NULL, NULL,
					button->button, button->time
				);
			}
			status = TRUE;
			break;
		}
	}

	return(status);
}

/*
 *	GtkHandleBox "child_attached" signal callback.
 */
void edv_vfs_browser_handle_child_attach_cb(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
)
{
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((handle_box == NULL) || (browser == NULL))
		return;

	gtk_widget_queue_resize(
		gtk_widget_get_toplevel(GTK_WIDGET(handle_box))
	);
}

/*
 *	GtkHandleBox "child_detached" signal callback.
 */
void edv_vfs_browser_handle_child_detach_cb(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
)
{
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((handle_box == NULL) || (browser == NULL))
		return;

	gtk_widget_queue_resize(
		gtk_widget_get_toplevel(GTK_WIDGET(handle_box))
	);
}


/*
 *	Location Bar Icon GtkFixed "realize" signal callback.
 */
void edv_vfs_browser_location_bar_icon_realize_cb(GtkWidget *widget, gpointer data)
{
	GdkWindow *window;
	EDVCore *core;
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((widget == NULL) || (browser == NULL))
		return;

	window = widget->window;
	core = browser->core;

	if(window != NULL)
	{
		GdkCursor *cursor = edv_get_cursor(core, EDV_CURSOR_CODE_HAND);
		gdk_window_set_cursor(window, cursor);
	}
}


/*
 *	Directory GtkCTree selct row idle callback.
 */
static gint EDVBrowserTreeSelectRowIdleCB(gpointer data)
{
	GtkWidget *toplevel;
	GtkCTreeNode *node;
	GtkCList *clist;
	GtkCTree *ctree;
	CfgList *cfg_list;
	EDVVFSObject *obj;
	EDVVFSBrowser *browser;
	EDVCore *core;
	EDVVFSBrowserTreeCBData *d = EDV_VFS_BROWSER_DIR_TREE_CB_DATA(data);
	if(d == NULL)
		return(FALSE);

	browser = d->vfs_browser;
	toplevel = browser->toplevel;
	ctree = GTK_CTREE(browser->directory_ctree);
	core = browser->core;
	cfg_list = core->cfg_list;
	node = d->node;
	obj = EDV_VFS_OBJECT(gtk_ctree_node_get_row_data(ctree, node));

	edv_vfs_browser_set_busy(browser, TRUE);
	GUIBlockInput(toplevel, TRUE);
	browser->flags |= EDV_VFS_BROWSER_PROCESSING;

	/* If the selected node is not fully visible then scroll to
	 * position the node in the middle of the list
	 */
	if(gtk_ctree_node_is_visible(ctree, node)
		!= GTK_VISIBILITY_FULL
	)
		gtk_ctree_node_moveto(
			ctree,
			node, -1,
			0.5f, 0.0f		/* Row, column */
		);

	/* Update the current selected node and reset the last
	 * selected object on the contents list
	 */
	browser->directory_ctree_selected_node = node;
	browser->contents_clist_selected_row = -1;

	/* Record the device who's mount path matches the selected
	 * object's path
	 */
	if(obj != NULL)
		edv_devices_list_match_mount_path(
			core->devices_list,
			&browser->selected_dev_num,
			obj->path
		);
	else
		browser->selected_dev_num = -1;

	/* Update the DND drag icon */
	GUIDNDSetDragIconFromCTreeSelection(ctree);

	/* Update the contents list with the contents of the
	 * selected directory
	 */
	clist = GTK_CLIST(browser->contents_clist);
	if(obj != NULL)
	{
		gint objects_in_directory;
		const gchar *path = obj->path;
		EDVMIMEType *m = edv_mime_types_list_match_path(
			core->mime_types_list,
			path
		);

		/* Update the title, location, and location icon */
		edv_vfs_browser_set_title(
			browser,
			path
		);
		edv_vfs_browser_set_location(
			browser,
			path,
			FALSE
		);
		edv_vfs_browser_update_location_icon(
			browser,
			path
		);

		/* Update the open to menus */
		edv_vfs_browser_open_to_menu_regenerate(browser, m);

		/* Get listing of contents in the directory */
		edv_vfs_browser_list_get_listing(
			browser,
			path,
			EDV_GET_B(EDV_CFG_PARM_LISTS_SHOW_PROGRESS)
		);

		/* Get the number of objects in the directory by
		 * calculating the number of rows in the contents list
		 * (excluding the parent directory notation row)
		 *
		 * Note that this calculation may be inaccurate if the
		 * contents list filter is set
		 */
		objects_in_directory = MAX(clist->rows - 1, 0);

		/* Update the status bar message */
		if(!STRISEMPTY(obj->name))
		{
			gchar *s;
			if(objects_in_directory > 0)
				s = g_strdup_printf(
"Directory \"%s\" selected (containing %i %s)",
					obj->name,
					objects_in_directory,
					(objects_in_directory == 1) ? "object" : "objects"
				);
			else
				s = g_strdup_printf(
"Directory \"%s\" selected (empty)",
					obj->name
				);
			edv_status_bar_message(
				browser->status_bar,
				s,
				FALSE
			);
			g_free(s);
		}
		else
		{
			edv_status_bar_message(
				browser->status_bar,
				"Directory with no name selected",
				FALSE
			);
		}
	}

	browser->flags &= ~EDV_VFS_BROWSER_PROCESSING;
	GUIBlockInput(toplevel, FALSE);
	edv_vfs_browser_set_busy(browser, FALSE);

	edv_vfs_browser_update_display(browser);

	/* Reset the idle ID */
	browser->directory_ctree_expand_optimize_idleid = 0;

	/* Delete the callback data */
	g_free(d);

	return(FALSE);
}

/*
 *	Directory GtkCTree expand optimize idle callback.
 */
static gint EDVBrowserTreeExpandScrollOptimizeIdleCB(gpointer data)
{
	GtkCTreeNode *node;
	GtkCTree *ctree;
	EDVVFSBrowser *browser;
	EDVVFSBrowserTreeCBData *d = EDV_VFS_BROWSER_DIR_TREE_CB_DATA(data);
	if(d == NULL)
		return(FALSE);

	browser = d->vfs_browser;
	ctree = GTK_CTREE(browser->directory_ctree);
	node = d->node;

	/* Scroll to the optimul viewing position */
	GUICTreeOptimizeExpandPosition(ctree, node);

	/* Reset the idle ID */
	browser->directory_ctree_expand_optimize_idleid = 0;

	/* Delete the callback data */
	g_free(d);

	return(FALSE);
}

/*
 *	Directory GtkCTree "tree_select_row" signal callback.
 */
void edv_vfs_browser_tree_select_row_cb(
	GtkCTree *ctree, GtkCTreeNode *node, gint column,
	gpointer data
)
{
	const CfgList *cfg_list;
	EDVCore *core;
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((ctree == NULL) || (browser == NULL))
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	core = browser->core;
	cfg_list = core->cfg_list;

	/* Check which widget this signal is for */
	/* Directory Tree */
	if(GTK_WIDGET(ctree) == browser->directory_ctree)
	{
		/* Change in selection? */
		if((browser->directory_ctree_selected_node != node) &&
		   (node != NULL)
		)
		{
			/* Set the idle callback to handle the selected row */
			EDVVFSBrowserTreeCBData *d = EDV_VFS_BROWSER_DIR_TREE_CB_DATA(g_malloc(
				sizeof(EDVVFSBrowserTreeCBData)
			));
			if(d != NULL)
			{
				d->vfs_browser = browser;
				d->node = node;
				browser->directory_ctree_expand_optimize_idleid = gtk_idle_add_priority(
					G_PRIORITY_LOW,
					EDVBrowserTreeSelectRowIdleCB, d
				);
			}
		}


	}
}

/*
 *	Directory GtkCTree "tree_unselect_row" signal callback.
 */
void edv_vfs_browser_tree_unselect_row_cb(
	GtkCTree *ctree, GtkCTreeNode *node, gint column,
	gpointer data
)
{
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((ctree == NULL) || (browser == NULL))
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	/* Check which widget this signal is for */
	/* Directory Tree */
	if(GTK_WIDGET(ctree) == browser->directory_ctree)
	{
		gchar *s;
		GtkCList *clist = GTK_CLIST(ctree);

		/* Get the total number of objects selected */
		const gint nselected = g_list_length(clist->selection);

		/* Was an item selected? */
		if(browser->directory_ctree_selected_node != NULL)
		{
			/* Mark that no node or device is selected */
			browser->directory_ctree_selected_node = NULL;
			browser->selected_dev_num = -1;
		}

		/* Update the title */
		edv_vfs_browser_set_title(browser, NULL);

		/* Update the open to menus */
		edv_vfs_browser_open_to_menu_regenerate(browser, NULL);

		/* Update the status bar message */
		if(nselected > 0)
			s = g_strdup_printf(
				"%s %s selected",
				edv_str_size_delim(nselected),
				(nselected == 1) ? "object" : "objects"
			);
		else
			s = STRDUP("No objects selected");
		edv_status_bar_message(
			browser->status_bar, s, FALSE
		);
		g_free(s);

		edv_vfs_browser_update_display(browser);
	}
}

/*
 *	Directory GtkCTree "tree_expand" signal callback.
 */
void edv_vfs_browser_tree_expand_cb(
	GtkCTree *ctree, GtkCTreeNode *node, gpointer data
)
{
	GtkWidget *toplevel;
	CfgList *cfg_list;
	EDVCore *core;
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((ctree == NULL) || (browser == NULL))
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	toplevel = browser->toplevel;
	core = browser->core;
	cfg_list = core->cfg_list;

	/* Check which widget this signal is for */
	/* Directory Tree */
	if(GTK_WIDGET(ctree) == browser->directory_ctree)
	{
		edv_vfs_browser_set_busy(browser, TRUE);
		GUIBlockInput(toplevel, TRUE);
		browser->flags |= EDV_VFS_BROWSER_PROCESSING;

		/* Expand this node and get a listing of child nodes that
		 * recurse no more than 2 levels
		 */
		edv_vfs_browser_tree_expand_node(
			browser,
			node,
			EDV_GET_B(EDV_CFG_PARM_LISTS_SHOW_PROGRESS)
		);

		/* Scroll to optimize viewing of expand nodes? */
		if(EDV_GET_B(EDV_CFG_PARM_TREE_EXPAND_OPTIMIZE_POS))
		{
			/* Set the idle callback to scroll to optimize view of
			 * the expanded nodes
			 */
			EDVVFSBrowserTreeCBData *d = EDV_VFS_BROWSER_DIR_TREE_CB_DATA(g_malloc(
				sizeof(EDVVFSBrowserTreeCBData)
			));
			if(d != NULL)
			{
				d->vfs_browser = browser;
				d->node = node;
				browser->directory_ctree_expand_optimize_idleid = gtk_idle_add_priority(
					G_PRIORITY_LOW,
					EDVBrowserTreeExpandScrollOptimizeIdleCB, d
				);
			}
		}

		browser->flags &= ~EDV_VFS_BROWSER_PROCESSING;
		GUIBlockInput(toplevel, FALSE);
		edv_vfs_browser_set_busy(browser, FALSE);

		edv_vfs_browser_update_display(browser);
	}
}

/*
 *	Directory GtkCTree "tree_collapse" signal callback.
 */
void edv_vfs_browser_tree_collapse_cb(
	GtkCTree *ctree, GtkCTreeNode *node, gpointer data
)
{
	GtkWidget *toplevel;
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((ctree == NULL) || (browser == NULL))
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	toplevel = browser->toplevel;

	/* Check which widget this signal is for */
	/* Directory Tree */
	if(GTK_WIDGET(ctree) == browser->directory_ctree)
	{
		GtkCList *clist = GTK_CLIST(ctree);
		EDVVFSObject *obj;

		edv_vfs_browser_set_busy(browser, TRUE);
		GUIBlockInput(toplevel, TRUE);
		browser->flags |= EDV_VFS_BROWSER_PROCESSING;

		/* Remove all the grandchild nodes of this node in order
		 * to save memory
		 */
		edv_vfs_browser_tree_remove_grand_children_nodes(browser, node);

		browser->flags &= ~EDV_VFS_BROWSER_PROCESSING;
		GUIBlockInput(toplevel, FALSE);
		edv_vfs_browser_set_busy(browser, FALSE);

		edv_vfs_browser_update_display(browser);

		/* If the current location is a child of this collapsed
		 * node then change the current location to this node
		 */
		obj = EDV_VFS_OBJECT(gtk_ctree_node_get_row_data(ctree, node));
		if(obj != NULL)
		{
			if(edv_path_is_parent(
				obj->path,
				edv_vfs_browser_get_location(browser)
			))
			{
				GList *glist;

				/* If this node is not selected then select it */
				for(glist = clist->selection;
					glist != NULL;
					glist = g_list_next(glist)
				)
				{
					if((GtkCTreeNode *)glist->data == node)
						break;
				}
				if(glist == NULL)
				{
					gtk_clist_freeze(clist);
					gtk_ctree_select(ctree, node);
					gtk_clist_thaw(clist);
				}
			}
		}
	}
}


/*
 *	Contents GtkCList "realize" signal callback.
 */
void edv_vfs_browser_list_realize_cb(GtkWidget *widget, gpointer data)
{
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((widget == NULL) || (browser == NULL))
		return;

	/* Create the list GtkStyles */
	edv_vfs_browser_cell_styles_regenerate(browser);
}

/*
 *	Contents GtkCList column sort size callback.
 */
static gint EDVBrowserCListColumnSortSizeCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	EDVVFSObject	*obj1 = EDV_VFS_OBJECT(
			((const GtkCListRow *)ptr1)->data
	),
			*obj2 = EDV_VFS_OBJECT(
			((const GtkCListRow *)ptr2)->data
	);
	if((obj1 == NULL) || (obj2 == NULL))
		return(0);

	if(obj1->size <= obj2->size)
		return(obj1->size < obj2->size);
	else
		return(-1);
}

/*
 *	Contents GtkCList column sort date nexus.
 *
 *	The ptr1 and ptr2 specifies the two GtkCListRow * rows
 *	that contain the object date data to compare.
 *
 *	The sort_code specifies which date value to compare.
 */
static gint EDVBrowserCListColumnSortDateNexus(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2,
	gint sort_code
)
{
	EDVVFSObject	*obj1 = EDV_VFS_OBJECT(
			((const GtkCListRow *)ptr1)->data
	),
			*obj2 = EDV_VFS_OBJECT(
			((const GtkCListRow *)ptr2)->data
	);
	if((clist == NULL) || (obj1 == NULL) || (obj2 == NULL))
		return(0);

	/* Handle by the sort code */
	switch(sort_code)
	{
	  case 0:	/* Access time */
		if(obj1->access_time <= obj2->access_time)
			return(obj1->access_time < obj2->access_time);
		else
			return(-1);
		break;

	  case 1:	/* Modify time */
		if(obj1->modify_time <= obj2->modify_time)
			return(obj1->modify_time < obj2->modify_time);
		else
			return(-1);
		break;

	  case 2:	/* Change time */
		if(obj1->change_time <= obj2->change_time)
			return(obj1->change_time < obj2->change_time);
		else
			return(-1);
		break;
	}

	return(0);
}

static gint EDVBrowserCListColumnSortDateAccessCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	return(EDVBrowserCListColumnSortDateNexus(
		clist, ptr1, ptr2, 0
	));
}

static gint EDVBrowserCListColumnSortDateModifyCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	return(EDVBrowserCListColumnSortDateNexus(
		clist, ptr1, ptr2, 1
	));
}

static gint EDVBrowserCListColumnSortDateChangeCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	return(EDVBrowserCListColumnSortDateNexus(
		clist, ptr1, ptr2, 2
	));
}

/*
 *	Contents GtkCList column sort hard links callback.
 */
static gint EDVBrowserCListColumnSortHardLinksCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	EDVVFSObject	*obj1 = EDV_VFS_OBJECT(
			((const GtkCListRow *)ptr1)->data
	),
			*obj2 = EDV_VFS_OBJECT(
			((const GtkCListRow *)ptr2)->data
	);
	if((obj1 == NULL) || (obj2 == NULL))
		return(0);

	if(obj1->hard_links <= obj2->hard_links)
		return(obj1->hard_links < obj2->hard_links);
	else
		return(-1);
}

/*
 *	Contents GtkCList column sort indices callback.
 */
static gint EDVBrowserCListColumnSortIndexCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	EDVVFSObject	*obj1 = EDV_VFS_OBJECT(
			((const GtkCListRow *)ptr1)->data
	),
			*obj2 = EDV_VFS_OBJECT(
			((const GtkCListRow *)ptr2)->data
	);
	if((obj1 == NULL) || (obj2 == NULL))
		return(0);

	if(obj1->index <= obj2->index)
		return(obj1->index < obj2->index);
	else
		return(-1);
}

/*
 *	Contents GtkCList column sort device indices callback.
 */
static gint EDVBrowserCListColumnSortDeviceIndexCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	EDVVFSObject	*obj1 = EDV_VFS_OBJECT(
			((const GtkCListRow *)ptr1)->data
	),
			*obj2 = EDV_VFS_OBJECT(
			((const GtkCListRow *)ptr2)->data
	);
	if((obj1 == NULL) || (obj2 == NULL))
		return(0);

	if(obj1->device_index <= obj2->device_index)
		return(obj1->device_index < obj2->device_index);
	else
		return(-1);
}

/*
 *	Contents GtkCList column sort block size callback.
 */
static gint EDVBrowserCListColumnSortBlockSizeCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	EDVVFSObject	*obj1 = EDV_VFS_OBJECT(
			((const GtkCListRow *)ptr1)->data
	),
			*obj2 = EDV_VFS_OBJECT(
			((const GtkCListRow *)ptr2)->data
	);
	if((obj1 == NULL) || (obj2 == NULL))
		return(0);

	if(obj1->block_size <= obj2->block_size)
		return(obj1->block_size < obj2->block_size);
	else
		return(-1);
}

/*
 *	Contents GtkCList column sort blocks callback.
 */
static gint EDVBrowserCListColumnSortBlocksCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	EDVVFSObject	*obj1 = EDV_VFS_OBJECT(
			((const GtkCListRow *)ptr1)->data
	),
			*obj2 = EDV_VFS_OBJECT(
			((const GtkCListRow *)ptr2)->data
	);
	if((obj1 == NULL) || (obj2 == NULL))
		return(0);

	if(obj1->blocks <= obj2->blocks)
		return(obj1->blocks < obj2->blocks);
	else
		return(-1);
}

/*
 *	Contents GtkCList "resize_column" signal callback.
 */
void edv_vfs_browser_resize_column_cb(
	GtkCList *clist, gint column, gint width, gpointer data
)
{
	CfgList *cfg_list;
	EDVCore *core;
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((clist == NULL) || (browser == NULL))
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	core = browser->core;
	cfg_list = core->cfg_list;

	/* Check which widget this signal is for */
	/* Contents List */
	if(GTK_WIDGET(clist) == browser->contents_clist)
	{
		/* Get the column type that the column is displaying */
		const EDVVFSBrowserColumnType column_type = edv_vfs_browser_list_get_column_type_by_index(
			browser,
			column
		);

		/* Get the column widths intlist */
		CfgIntList *column_width_intlist = EDV_GET_INT_LIST(
			EDV_CFG_PARM_BROWSER_CONTENTS_COLUMN_WIDTH
		);
		if((column_type >= 0) && (column_width_intlist != NULL))
		{
			GList *glist = g_list_nth(
				column_width_intlist->list,
				(guint)column_type
			);
			if(glist != NULL)
			{
				glist->data = (gpointer)width;
			}
			else
			{
				/* The column type is not in the column widths      
				 * list so create a new column widths list from
				 * the current column widths list and include   
				 * the missing column type                   
				 */
				const gint m = EDV_VFS_BROWSER_NCOLUMN_TYPES;
				gint i;
				GList *glist_new = NULL;

				glist = column_width_intlist->list;

				for(i = 0; i < m; i++)
				{
					if(glist != NULL)
					{
						glist_new = g_list_append(
							glist_new,
							glist->data
						);
						glist = g_list_next(glist);
					}
					else
						glist_new = g_list_append(glist_new, NULL);
				}
				 
				g_list_free(column_width_intlist->list);
				column_width_intlist->list = glist_new;

				glist = g_list_nth(
					glist_new,
					(guint)column_type
				);
				if(glist != NULL)
					glist->data = (gpointer)width;
			}
		}
	}
}

/*
 *	Contents GtkCList "click_column" signal callback.
 */
void edv_vfs_browser_click_column_cb(
	GtkCList *clist, gint column, gpointer data
)
{
	GtkWidget *widget, *toplevel;
	const CfgList *cfg_list;
	EDVCore *core;
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((clist == NULL) || (browser == NULL))
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser) || (browser->freeze_count > 0))
		return;

	widget = GTK_WIDGET(clist);
	toplevel = browser->toplevel;
	core = browser->core;
	cfg_list = core->cfg_list;

	/* Check which widget this signal is for
	 *
	 * Contents GtkCList
	 */
	if(widget == browser->contents_clist)
	{
		GtkCListCompareFunc cmp_func = NULL;
		GtkCListCompareFunc cmp_func_str =
			 (GtkCListCompareFunc)EDVCListColumnSortStringCB;
#if 0
		GtkCListCompareFunc cmp_func_num =
			(GtkCListCompareFunc)edv_clist_column_sort_number_cb;
#endif
		CfgIntList *column_types_intlist;

		edv_vfs_browser_set_busy(browser, TRUE);
		GUIBlockInput(toplevel, TRUE);
		browser->flags |= EDV_VFS_BROWSER_PROCESSING;	    

		/* Get the column types mapping list */
		column_types_intlist = EDV_GET_INT_LIST(
			EDV_CFG_PARM_BROWSER_CONTENTS_COLUMN
		);
		if(column_types_intlist != NULL)
		{
			const EDVVFSBrowserColumnType column_type = (EDVVFSBrowserColumnType)g_list_nth_data(
				column_types_intlist->list,
				column
			);
			switch(column_type)
			{
			  case EDV_VFS_BROWSER_COLUMN_TYPE_NAME:
				cmp_func = cmp_func_str;
				break;
			  case EDV_VFS_BROWSER_COLUMN_TYPE_SIZE:
				cmp_func = EDVBrowserCListColumnSortSizeCB;
				break;
			  case EDV_VFS_BROWSER_COLUMN_TYPE_TYPE:
				cmp_func = cmp_func_str;
				break;
			  case EDV_VFS_BROWSER_COLUMN_TYPE_PERMISSIONS:
				cmp_func = cmp_func_str;
				break;
			  case EDV_VFS_BROWSER_COLUMN_TYPE_OWNER:
				cmp_func = cmp_func_str;
				break;
			  case EDV_VFS_BROWSER_COLUMN_TYPE_GROUP:
				cmp_func = cmp_func_str;
				break;
			  case EDV_VFS_BROWSER_COLUMN_TYPE_DATE_ACCESS:
				cmp_func = EDVBrowserCListColumnSortDateAccessCB;
				break;
			  case EDV_VFS_BROWSER_COLUMN_TYPE_DATE_MODIFIED:
				cmp_func = EDVBrowserCListColumnSortDateModifyCB;
				break;
			  case EDV_VFS_BROWSER_COLUMN_TYPE_DATE_CHANGED:
				cmp_func = EDVBrowserCListColumnSortDateChangeCB;
				break;
			  case EDV_VFS_BROWSER_COLUMN_TYPE_HARD_LINKS:
				cmp_func = EDVBrowserCListColumnSortHardLinksCB;
				break;
			  case EDV_VFS_BROWSER_COLUMN_TYPE_LINKED_TO:
				cmp_func = cmp_func_str;
				break;
			  case EDV_VFS_BROWSER_COLUMN_TYPE_DEVICE_INDEX:
				cmp_func = EDVBrowserCListColumnSortDeviceIndexCB;
				break;
			  case EDV_VFS_BROWSER_COLUMN_TYPE_INDEX:
				cmp_func = EDVBrowserCListColumnSortIndexCB;
				break;
			  case EDV_VFS_BROWSER_COLUMN_TYPE_DEVICE_TYPE:
				cmp_func = cmp_func_str;
				break;
			  case EDV_VFS_BROWSER_COLUMN_TYPE_BLOCK_SIZE:
				cmp_func = EDVBrowserCListColumnSortBlockSizeCB;
				break;
			  case EDV_VFS_BROWSER_COLUMN_TYPE_BLOCKS:
				cmp_func = EDVBrowserCListColumnSortBlocksCB;
				break;
			}
		}


		gtk_clist_freeze(clist);

		/* Set or invert the sort column on the GtkCList based
		 * on if the clicked on column is already the current
		 * sort column
		 */
		if(column != clist->sort_column)
			gtk_clist_set_sort_column(clist, column);
		else
			gtk_clist_set_sort_type(
				clist,
				(clist->sort_type == GTK_SORT_ASCENDING) ?
					GTK_SORT_DESCENDING : GTK_SORT_ASCENDING
			);
		if(cmp_func != NULL)
			gtk_clist_set_compare_func(clist, cmp_func);

		/* Sort the rows, this will call the GtkCList column
		 * sort callbacks
		 */
		gtk_clist_sort(clist);

		gtk_clist_thaw(clist);

		browser->flags &= ~EDV_VFS_BROWSER_PROCESSING;
		GUIBlockInput(toplevel, FALSE);
		edv_vfs_browser_set_busy(browser, FALSE);

		edv_vfs_browser_update_display(browser);
	}
}

/*
 *	Browser GtkCList "select_row" signal callback.
 */
void edv_vfs_browser_list_select_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((clist == NULL) || (browser == NULL))
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	/* Check which widget this signal is for */
	/* Contents List */
	if(GTK_WIDGET(clist) == browser->contents_clist)
	{
		EDVVFSObject *obj;
		EDVCore *core = browser->core;

		/* Get total number of objects selected */
		const gint nselected = g_list_length(clist->selection);

		/* Update selected row */
		browser->contents_clist_selected_row = row;

		/* Mark last selected directory and device is no longer
		 * selected due to selection of row on the contents clist
		 */
		browser->directory_ctree_selected_node = NULL;
		browser->selected_dev_num = -1;

		/* Set the DND drag icon */
		GUIDNDSetDragIconFromCListSelection(clist);

		/* Get the object */
		obj = EDV_VFS_OBJECT(gtk_clist_get_row_data(clist, row));
		if(obj != NULL)
		{
			EDVMIMEType *m = edv_mime_types_list_match_path(
				core->mime_types_list,
				obj->path
			);

			/* Update the open to menus */
			edv_vfs_browser_open_to_menu_regenerate(browser, m);

			/* Update the status bar message */
			if(!STRISEMPTY(obj->name))
			{
				gchar *s, *size_str = NULL;
				const gchar *type_str = NULL;

				/* Get object type and size strings */
				switch(obj->type)
				{
				  case EDV_OBJECT_TYPE_UNKNOWN:
					type_str = "Object";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_FILE:
					type_str = "File";
					size_str = g_strdup_printf(
						" (%s %s)",
						edv_str_size_delim(obj->size),
						(obj->size == 1l) ? "byte" : "bytes"
					);
					break;
				  case EDV_OBJECT_TYPE_DIRECTORY:
					type_str = "Directory";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_LINK:
					type_str = "Link";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_DEVICE_BLOCK:
					type_str = "Block device";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
					type_str = "Character device";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_FIFO:
					type_str = "FIFO pipe";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_SOCKET:
					type_str = "Socket";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_ERROR:
					type_str = "Error";
					size_str = NULL;
					break;
				}

				/* Set the status bar message */
				if(nselected > 1)
				{
					gulong total_size = 0l;
					EDVVFSObject *obj;
					GList *glist = clist->selection;
					while(glist != NULL)
					{
						obj = EDV_VFS_OBJECT(gtk_clist_get_row_data(
							clist, (gint)glist->data
						));
						if(obj != NULL)
							total_size += obj->size;
						glist = g_list_next(glist);
					}
					g_free(size_str);
					size_str = g_strdup_printf(
						"%s %s",
						edv_str_size_delim(total_size),
						(total_size == 1l) ? "byte" : "bytes"
					);
					s = g_strdup_printf(
						"%s objects selected (totaling %s)",
						edv_str_size_delim(nselected),
						size_str
					);
				}
				else if(!strcmp((const char *)obj->name, ".."))
					s = g_strdup_printf(
						"Parent directory selected"
					);
				else
					s = g_strdup_printf(
						"%s \"%s\" selected%s",
						type_str, obj->name,
						(size_str != NULL) ? size_str : ""
					);
				edv_status_bar_message(
					browser->status_bar, s, FALSE
				);
				g_free(s);
				g_free(size_str);
			}
			else
			{
				edv_status_bar_message(
					browser->status_bar,
					"Object with no name selected",
					FALSE
				);
			}
		}

		/* If the selected row is not fully visible then scroll
		 * to position the selected row in the center of the list
		 */
		if(gtk_clist_row_is_visible(clist, row) !=
			GTK_VISIBILITY_FULL
		)
			gtk_clist_moveto(
				clist,
				row, -1,			/* Row, column */
				0.5f, 0.0f			/* Row, column */
			);

		edv_vfs_browser_update_display(browser);

		/* Double click? */
		if(event != NULL)
		{
			if(event->type == GDK_2BUTTON_PRESS)
			{
				/* Open */
				GdkEventButton *button = (GdkEventButton *)event;
				edv_vfs_browser_list_open(
					browser,
					row, column,
					button->state		/* Modifier keys */
				);
			}
		}
	}
}

/*
 *	Browser GtkCList "unselect_row" signal callback.
 */
void edv_vfs_browser_list_unselect_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((clist == NULL) || (browser == NULL))
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	/* Check which widget this signal is for */
	/* Contents List */
	if(GTK_WIDGET(clist) == browser->contents_clist)
	{
		gint nselected;
		gchar *s;
		GList *glist;
		EDVVFSObject *obj;
		EDVMIMEType *m;
		EDVCore *core = browser->core;

		/* Get the total number of objects selected */
		nselected = g_list_length(clist->selection);

		/* Get the last selected object (if any) */
		glist = clist->selection_end;
		if(glist != NULL)
			obj = EDV_VFS_OBJECT(gtk_clist_get_row_data(clist, (gint)glist->data));
		else
			obj = NULL;

		if(obj != NULL)
			m = edv_mime_types_list_match_path(
				core->mime_types_list,
				obj->path
			);
		else
			m = NULL;

		/* Update the open to menus */
		edv_vfs_browser_open_to_menu_regenerate(browser, m);

		/* Update the status bar message */
		if(nselected > 0)
		{
			gchar *size_str;
			gulong total_size = 0l;
			EDVVFSObject *obj;
			GList *glist = clist->selection;
			while(glist != NULL)
			{
				obj = EDV_VFS_OBJECT(gtk_clist_get_row_data(
					clist, (gint)glist->data
				));
				if(obj != NULL)
					total_size += obj->size;
				glist = g_list_next(glist);
			}
			size_str = g_strdup_printf(
				"%s %s",
				edv_str_size_delim(total_size),
				(total_size == 1l) ? "byte" : "bytes"
			);
			s = g_strdup_printf(
				"%s %s selected (totaling %s)",
				edv_str_size_delim(nselected),
				(nselected == 1) ? "object" : "objects",
				size_str
			);
			g_free(size_str);
		}
		else
			s = STRDUP("No objects selected");
		edv_status_bar_message(
			browser->status_bar, s, FALSE
		);
		g_free(s);

		edv_vfs_browser_update_display(browser);
	}
}


/*
 *	Location combo "activate" callback.
 */
void edv_vfs_browser_location_combo_activate_cb(GtkWidget *widget, gpointer data)
{
	gboolean set_origin_path;
	gchar *path;
	GtkWidget *toplevel;
	GtkCombo *combo;
	EDVVFSObject *obj;
	EDVCore *core;
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((widget == NULL) || (browser == NULL))
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	combo = GTK_COMBO(widget);
	toplevel = browser->toplevel;
	core = browser->core;

	/* Evaluate/copy the location combo's current path */
	path = edv_path_evaluate(
		NULL,				/* No parent reference */
		gtk_entry_get_text(GTK_ENTRY(combo->entry))
	);
	if(path == NULL)
	{
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Make sure that the path exists and leads to a directory */
	obj = edv_vfs_object_stat(path);
	if(obj == NULL)
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
			g_strerror(error_code),
			path
		);
		edv_play_sound_warning(core);
		edv_message_warning(
			"Set Location Failed",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
		g_free(path);
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Not a directory? */
	if(!EDV_VFS_OBJECT_IS_DIRECTORY(obj))
	{
		gchar *msg = g_strdup_printf(
"Not a directory:\n\
\n\
    %s",
			path
		);
		edv_play_sound_warning(core);
		edv_message_warning(
			"Set Location Failed",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
		edv_vfs_object_delete(obj);
		g_free(path);
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	edv_vfs_object_delete(obj);

	GUIBlockInput(toplevel, TRUE);

	/* Do not mark as processing or else
	 * edv_vfs_browser_tree_select_path() will not be able to expand
	 * the nodes because the expand signal callback will detect
	 * that processing is set and not expand the node
	 */

	/* If the specified path's prefix is the same as the current
	 * Directory Tree Origin then do not change the Directory
	 * Tree Origin, otherwise the Directory Tree Origin needs to
	 * be changed to the specified path
	 */
	if(edv_vfs_browser_is_path_from_tree_origin(browser, path))
	{
		 set_origin_path = FALSE;
	}
	else
	{
		edv_vfs_browser_tree_set_origin_path(browser, path);
		set_origin_path = TRUE;
	}

	/* Update the location combo to record the new location in the
	 * locations history
	 *
	 * The combo will be updated again (without recording history)
	 * if the new path is valid and selected in the callbacks
	 * triggered by edv_vfs_browser_tree_select_path() farther below
	 */
	edv_vfs_browser_set_location(browser, path, TRUE);

	/* Attempt to select the node who's object's path matches path
	 * (recurse and expand as needed)
	 *
	 * The Directory GtkCTree will be frozen and thawed by the
	 * callbacks each time it is expanded so we do not need to do
	 * that here
	 *
	 * Also, the location GtkCombo will be updated again (but its
	 * history will not be re-recorded) when the the node is
	 * selected
	 */
	if(!set_origin_path)
		edv_vfs_browser_tree_select_path(browser, path);

	g_free(path);

	GUIBlockInput(toplevel, FALSE);
	edv_vfs_browser_set_busy(browser, FALSE);
}

/*
 *	Menu item "activate" callback.
 *
 *	The data must be a EDVVFSBrowserOp *.
 */
void edv_vfs_browser_menu_item_cb(GtkWidget *widget, gpointer data)
{
	edv_vfs_browser_op_cb(NULL, -1, data);
}

/*
 *	Menu item "enter_notify_event" signal callback.
 *
 *	The data must be a EDVVFSBrowserOp *.
 */
gint edv_vfs_browser_menu_item_enter_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	edv_vfs_browser_op_enter_cb(NULL, -1, data);
	return(TRUE);
}

/*
 *	Menu item "leave_notify_event" signal callback.
 *
 *	The data must be a EDVVFSBrowserOp *.
 */
gint edv_vfs_browser_menu_item_leave_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	edv_vfs_browser_op_leave_cb(NULL, -1, data);
	return(TRUE);
}

/*
 *	Menu item command object op "activate" signal callback.
 */
void edv_vfs_browser_menu_item_cmd_object_op_cb(GtkWidget *widget, gpointer data)
{
	const gchar *cmd_src;
	gchar *cmd;
	GtkWidget *toplevel;
	GList *paths_list;
	CfgList *cfg_list;
	EDVVFSBrowser *browser;
	EDVCore *core;
	EDVVFSBrowserCommand *d = EDV_VFS_BROWSER_COMMAND(data);
	if(d == NULL)
		return;

	browser = d->vfs_browser;
	toplevel = browser->toplevel;
	core = browser->core;
	cfg_list = core->cfg_list;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	/* Get the command for this menu item */
	cmd_src = d->command;
	if(STRISEMPTY(cmd_src))
	{
		gchar *msg = g_strdup_printf(
"The custom menu item:\n\
\n\
    %s\n\
\n\
Does not have a defined command.",
			d->label
		);
		edv_play_sound_warning(core);
		edv_message_warning(
			"Open Failed",
			msg,
"To set the command for this custom menu item,\n\
go to Settings->Customize...->Global->Object Ops Menu.",
			toplevel
		);
		g_free(msg);
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Format the command */
	cmd = STRDUP(cmd_src);

	/* Prefix the run in terminal command? */
	if(d->flags & CFG_MENU_ITEM_RUN_IN_TERMINAL)
	{
		const gchar *cmd_terminal_run = EDV_GET_S(
			EDV_CFG_PARM_PROG_TERMINAL_RUN
		);
		if(!STRISEMPTY(cmd_terminal_run))
		{
			/* Format the command for run in terminal */
			gchar *s = g_strconcat(
				cmd_terminal_run,
				" ",
				cmd,
				NULL
			);
			g_free(cmd);
			cmd = s;
		}
		else
		{
			edv_play_sound_warning(core);
			edv_message_warning(
				"Open Failed",
"The \"Run In Terminal\" command is not set.\n\
\n\
To set the \"Run In Terminal\" command go to\n\
Settings->Options...->Programs.",
				NULL,
				toplevel
			);
			g_free(cmd);
			edv_vfs_browser_set_busy(browser, FALSE);
			return;
		}
	}

	/* Create the selected paths list */
	paths_list = edv_vfs_browser_get_selected_paths(browser);
	if(paths_list != NULL)
	{
		/* Open the selected objects */
		(void)edv_open_command(
			core,
			paths_list,
			cmd,
			toplevel,
			TRUE			/* Verbose */
		);

		g_list_foreach(
			paths_list,
			(GFunc)g_free,
			NULL
		);
		g_list_free(paths_list);
	}

	/* Delete the formatted command */
	g_free(cmd);

	edv_vfs_browser_set_busy(browser, FALSE);
}

/*
 *	Menu item command new object "activate" signal callback.
 */
void edv_vfs_browser_menu_item_cmd_new_object_cb(GtkWidget *widget, gpointer data)
{
	gboolean yes_to_all = FALSE;
	gint status;
	const gchar	*cmd_src,
					*error_msg;
	gchar		*cmd,
					*new_name,
					*new_path;
	GList *new_paths_list;
	GtkWidget *toplevel;
	CfgList *cfg_list;
	EDVVFSBrowser *browser;
	EDVCore *core;
	EDVVFSBrowserCommand *d = EDV_VFS_BROWSER_COMMAND(data);
	if(d == NULL)
		return;

	browser = d->vfs_browser;
	toplevel = browser->toplevel;
	core = browser->core;
	cfg_list = core->cfg_list;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	edv_vfs_browser_set_busy(browser, TRUE);

	/* Get the command for this menu item */
	cmd_src = d->command;
	if(STRISEMPTY(cmd_src))
	{
		gchar *msg = g_strdup_printf(
"The custom menu item:\n\
\n\
    %s\n\
\n\
Does not have a defined command.",
			d->label
		);
		edv_play_sound_warning(core);
		edv_message_warning(
			"Create New Object Failed",
			msg,
"To set the command for this custom menu item,\n\
go to Settings->Customize...->Global->New Objects Menu.",
			toplevel
		);
		g_free(msg);
		edv_vfs_browser_set_busy(browser, FALSE);
		return;
	}

	/* Format the suggested name for the new object */
	if(!STRISEMPTY(d->ext_data))
	{
		const gchar *ext = d->ext_data;
		gchar *s, *new_ext;

		if(*ext != '.')
			new_ext = g_strconcat(".", ext, NULL);
		else
			new_ext = STRDUP(ext);
		s = (gchar *)strpbrk((char *)new_ext, " \t,;");
		if(s != NULL)
			*s = '\0';

		new_name = g_strconcat(
			"new",
			new_ext,
			NULL
		);

		g_free(new_ext);
	}
	else
	{
		new_name = STRDUP("new");
	}

	/* Prompt the user for the name of the new file and create the
	 * new file
	 */
	status = EDVVFSObjectCreateFileQueryName(
		core,
		edv_vfs_browser_get_location(browser),	/* Parent path */
		new_name,				/* Suggested name */
		&new_path,
		toplevel,
		TRUE,				/* Show progress */
		&yes_to_all
	);

	g_free(new_name);

	/* Check for errors */
	error_msg = EDVVFSObjectCreateGetError(core);
	if(!STRISEMPTY(error_msg))
	{
		/* Report the error */
		edv_play_sound_error(core);
		edv_message_error(
			"Create New Object Failed",
			error_msg,   
			NULL,
			toplevel
		);
	}

	/* Report the new object being added and create the new paths
	 * list
	 */
	new_paths_list = NULL;
	if(new_path != NULL)
	{
		EDVVFSObject *obj = edv_vfs_object_lstat(new_path);
		if(obj != NULL)
		{
			edv_emit_vfs_object_added(
				core,
				new_path,
				obj
			);
			edv_vfs_object_delete(obj);
		}

		new_paths_list = g_list_append(
			new_paths_list,
			STRDUP(new_path)
		);
	}

	/* Unmap the progress dialog */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Format the command */
	cmd = STRDUP(cmd_src);

	/* Prefix the run in terminal command? */
	if(d->flags & CFG_MENU_ITEM_RUN_IN_TERMINAL)
	{
		const gchar *cmd_terminal_run = EDV_GET_S(
			EDV_CFG_PARM_PROG_TERMINAL_RUN
		);
		if(!STRISEMPTY(cmd_terminal_run))
		{
			/* Format the command for run in terminal */
			gchar *s = g_strconcat(
				cmd_terminal_run,
				" ",
				cmd,
				NULL
			);
			g_free(cmd);
			cmd = s;
		}
		else
		{
			edv_play_sound_warning(core);
			edv_message_warning(
				"Open Failed",
"The \"Run In Terminal\" command is not set.\n\
\n\
To set the \"Run In Terminal\" command go to\n\
Settings->Options...->Programs.",
				NULL,
				toplevel
			);
			if(new_paths_list != NULL)
			{
				g_list_foreach(new_paths_list, (GFunc)g_free, NULL);
				g_list_free(new_paths_list);
			}
			g_free(cmd);
			g_free(new_path);
			edv_vfs_browser_set_busy(browser, FALSE);
			return;
		}
	}

	/* Open the new object */
	if(new_paths_list != NULL)
	{
		(void)edv_open_command(
			core,
			new_paths_list,
			cmd,
			toplevel,
			TRUE			/* Verbose */
		);

		g_list_foreach(
			new_paths_list,
			(GFunc)g_free,
			NULL
		);
		g_list_free(new_paths_list);
	}

	/* Delete the formatted command */
	g_free(cmd);

	/* Delete the new path */
	g_free(new_path);

	/* Play the "completed" sound on success */
	if(status == 0)
		edv_play_sound_completed(core);

	edv_vfs_browser_set_busy(browser, FALSE);
}

/*
 *	Menu item command "leave_enter_event" signal callback.
 */
gint edv_vfs_browser_menu_item_cmd_enter_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	EDVVFSBrowser *browser;
	EDVVFSBrowserCommand *d = EDV_VFS_BROWSER_COMMAND(data);
	if(d == NULL)
		return(FALSE);

	browser = d->vfs_browser;

	edv_status_bar_message(
		browser->status_bar,
		STRISEMPTY(d->description) ?
			d->command : d->description,
		FALSE
	);

	return(TRUE);
}

/*
 *	Menu item command "leave_notify_event" signal callback.
 */
gint edv_vfs_browser_menu_item_cmd_leave_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	EDVVFSBrowser *browser;
	EDVVFSBrowserCommand *d = EDV_VFS_BROWSER_COMMAND(data);
	if(d == NULL)
		return(FALSE);

	browser = d->vfs_browser;

	edv_status_bar_message(
		browser->status_bar,
		NULL,
		FALSE
	);

	return(TRUE);
}


/*
 *	Paths list get callback.
 */
GList *edv_vfs_browser_get_selected_paths_cb(gpointer data)
{
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if(browser == NULL)
		return(NULL);

	return(edv_vfs_browser_get_selected_paths(browser));
}

/*
 *	Go to directory callback.
 */
void edv_vfs_browser_goto_directory_cb(gpointer data, const gchar *path)
{
	GtkWidget *toplevel;
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((browser == NULL) || (path == NULL))
		return;

	toplevel = browser->toplevel;

	edv_vfs_browser_set_busy(browser, TRUE);
	GUIBlockInput(toplevel, TRUE);  

	/* Is the path within the Directory Tree? */
	if(edv_vfs_browser_is_path_from_tree_origin(browser, path))
	{
		/* Select the path */
		edv_vfs_browser_tree_select_path(browser, path);
	}
	else
	{
		/* Set the Directory Tree origin and select the path */
		edv_vfs_browser_tree_set_origin_path(browser, path);
	}

	GUIBlockInput(toplevel, FALSE);
	edv_vfs_browser_set_busy(browser, FALSE);
}


/*
 *	EDVMountBar mount callback.
 */
void edv_vfs_browser_mount_bar_mount_cb(
	GtkWidget *widget, const gint dev_num, EDVDevice *dev,
	gpointer data
)
{
	gboolean original_mount_state;
	gint status;
	GtkWidget *toplevel;
	EDVCore *core;
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((browser == NULL) || (dev == NULL))
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	toplevel = browser->toplevel;
	core = browser->core;

	edv_vfs_browser_set_busy(browser, TRUE);

	/* Record original mount state */
	original_mount_state = EDV_DEVICE_IS_MOUNTED(dev);

	/* Unmount or mount? */
	if(EDV_DEVICE_IS_MOUNTED(dev))
		status = edv_device_unmount(
			core,
			dev,
			TRUE,				/* Show progress */
			TRUE,				/* Verbose */
			toplevel
		);
	else
		status = edv_device_mount(
			core,
			dev,
			TRUE,				/* Show progress */
			TRUE,				/* Verbose */
			toplevel
		);

	/* Update all the devices' statistics */
	edv_devices_list_update_statistics(core->devices_list);

	edv_vfs_browser_set_busy(browser, FALSE);

	/* Mount error? */
	if(status != 0)
	{
		const gchar *last_error = edv_device_mount_get_error(core);
		if(!STRISEMPTY(last_error))
		{
			edv_play_sound_error(core);
			edv_message_error(
				original_mount_state ?
					"Unmount Failed" : "Mount Failed",
				last_error,
				NULL,
				browser->toplevel
			);
		}
	}
	else
	{
		/* Report mount signal to all of endeavour's resources */
		edv_emit_device_mount(core, dev_num, dev, EDV_DEVICE_IS_MOUNTED(dev));
	}
}

/*
 *	EDVMountBar eject callback.
 */
void edv_vfs_browser_mount_bar_eject_cb(
	GtkWidget *widget, const gint dev_num, EDVDevice *dev,
	gpointer data
)
{
	gboolean original_mount_state;
	gint status;
	GtkWidget *toplevel;
	EDVCore *core;
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((browser == NULL) || (dev == NULL))
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	toplevel = browser->toplevel;
	core = browser->core;

	edv_vfs_browser_set_busy(browser, TRUE);

	/* Record original mount state */
	original_mount_state = EDV_DEVICE_IS_MOUNTED(dev);

	/* Need to unmount first? */
	if(EDV_DEVICE_IS_MOUNTED(dev))
		status = edv_device_unmount(
			core, dev,
			TRUE, TRUE,
			toplevel
		);
	/* Now eject */
	status = edv_device_eject(
		core, dev,
		TRUE, TRUE,
		toplevel
	);

	/* Update all device mount states and stats */
	edv_devices_list_update_mount_states(
		core->devices_list
	);
	edv_devices_list_update_statistics(
		core->devices_list
	);

	edv_vfs_browser_set_busy(browser, FALSE);

	/* Mount error? */
	if(status != 0)
	{
		const gchar *last_error = edv_device_mount_get_error(core);
		if(!STRISEMPTY(last_error))
		{
			edv_play_sound_error(core);
			edv_message_error(
				"Eject Failed",
				last_error,
				NULL,
				browser->toplevel
			);
		}
	}
	else
	{
		/* Report eject (unmount) signal to all of endeavour's
		 * resources
		 */
		edv_emit_device_mount(core, dev_num, dev, EDV_DEVICE_IS_MOUNTED(dev));
	}
}

/*
 *	EDVMountBar go to mount path callback.
 */
extern void edv_vfs_browser_mount_bar_goto_cb(
	GtkWidget *widget, const gint dev_num, EDVDevice *dev,
	gpointer data
)
{
	gchar *path;
	GtkWidget *toplevel;
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((browser == NULL) || (dev == NULL))
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	path = STRDUP(dev->mount_path);
	toplevel = browser->toplevel;

	edv_vfs_browser_set_busy(browser, TRUE);
	GUIBlockInput(toplevel, TRUE);

	/* Is the path within the Directory Tree? */
	if(edv_vfs_browser_is_path_from_tree_origin(browser, path))
	{
		/* Select the path */
		edv_vfs_browser_tree_select_path(browser, path);
	}
	else
	{
		/* Set the Directory Tree origin and select the path */
		edv_vfs_browser_tree_set_origin_path(browser, path);
	}

	g_free(path);

	GUIBlockInput(toplevel, FALSE);
	edv_vfs_browser_set_busy(browser, FALSE);
}


/*
 *	Find Bar get current location callback.
 */
const gchar *edv_vfs_browser_find_bar_location_cb(
	GtkWidget *bar,
	gpointer data
)
{
	return(edv_vfs_browser_get_location(EDV_VFS_BROWSER(data)));
}

/*
 *	Find Bar start find callback.
 */
void edv_vfs_browser_find_bar_start_cb(
	GtkWidget *bar,
	gpointer data
)
{
	GtkCList *clist;
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if(browser == NULL)
		return;

	clist = GTK_CLIST(browser->contents_clist);

	edv_vfs_browser_set_busy(browser, TRUE);

	gtk_clist_freeze(clist);
	gtk_clist_unselect_all(clist);
	gtk_clist_thaw(clist);

	edv_vfs_browser_update_display(browser);

	browser->freeze_count++;
}

/*
 *	Find Bar end find callback.
 */
void edv_vfs_browser_find_bar_end_cb(
	GtkWidget *bar,
	const gint nmatches,
	gpointer data
)
{
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if(browser == NULL)
		return;

	browser->freeze_count--;
	edv_vfs_browser_set_busy(browser, FALSE);
}

/*
 *	Find Bar match callback.
 */
void edv_vfs_browser_find_bar_match_cb(
	GtkWidget *bar,
	const gchar *path,
	GList *properties_list,
	const gint line_index,
	const gchar *excerpt,
	gpointer data
)
{
	gint row;
	GtkCList *clist;
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if((path == NULL) || (browser == NULL))
		return;

	clist = GTK_CLIST(browser->contents_clist);

	row = edv_vfs_browser_list_find_by_path(browser, path);
	if(row > -1)
	{
		gtk_clist_freeze(clist);
		gtk_clist_select_row(clist, row, 0);
		gtk_clist_thaw(clist);
	}
}


/*
 *	Status message callback.
 */
void edv_vfs_browser_status_message_cb(
	GtkWidget *widget,
	const gchar *message,
	gpointer data
)
{
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if(browser == NULL)
		return;

	edv_status_bar_message(browser->status_bar, message, FALSE);
}

/*
 *	Status progress callback.
 */
void edv_vfs_browser_status_progress_cb(
	GtkWidget *widget,
	const gfloat progress,
	gpointer data
)
{
	EDVVFSBrowser *browser = EDV_VFS_BROWSER(data);
	if(browser == NULL)
		return;

	edv_status_bar_progress(browser->status_bar, progress, FALSE);
}


/*
 *	Window created callback.
 */
void edv_vfs_browser_window_created_cb(
	EDVVFSBrowser *browser,
	const EDVWindowType win_type,
	gpointer win
) 
{
	if(browser == NULL)
		return;

	if(browser == EDV_VFS_BROWSER(win))
		return;

	edv_vfs_browser_update_display(browser);
}

/*
 *	Window deleted callback.
 */
void edv_vfs_browser_window_deleted_cb(
	EDVVFSBrowser *browser,
	const EDVWindowType win_type,
	gpointer win
) 
{
	if(browser == NULL)
		return;

	if(browser == EDV_VFS_BROWSER(win))
		return;

	edv_vfs_browser_update_display(browser);
}


/*
 *	Master Write Protect changed callback.
 */
void edv_vfs_browser_master_write_protect_changed_cb(
	EDVVFSBrowser *browser,
	const gboolean state
)
{
	if(browser == NULL)
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	edv_vfs_browser_update_display(browser);
}

/*
 *	Delete Method changed callback.
 */
void edv_vfs_browser_delete_method_changed_cb(
	EDVVFSBrowser *browser,
	const EDVDeleteMethod delete_method
)
{
	if(browser == NULL)
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	edv_vfs_browser_update_display(browser);
}

/*
 *	Object added callback.
 */
void edv_vfs_browser_vfs_object_added_cb(
	EDVVFSBrowser *browser,
	const gchar *path,
	EDVVFSObject *obj
)
{
	if(browser == NULL)
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	/* Contents Tree */
	edv_vfs_browser_list_vfs_object_added_cb(
		browser,
		path,
		obj
	);

	/* Directory List */
	edv_vfs_browser_tree_vfs_object_added_cb(
		browser,
		path,
		obj
	);

	/* Mount Bar */
	edv_mount_bar_update_display(browser->mount_bar);

/*	edv_vfs_browser_update_display(browser); */
}

/*
 *	Object modified callback.
 */
void edv_vfs_browser_vfs_object_modified_cb(
	EDVVFSBrowser *browser,
	const gchar *path,
	const gchar *new_path,
	EDVVFSObject *obj
)
{
	gchar *current_location_path;

	if((browser == NULL) || STRISEMPTY(path))
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	if(new_path == NULL)
		new_path = path;

	current_location_path = STRDUP(edv_vfs_browser_get_location(browser));
	if(current_location_path == NULL)
		return;

	/* Is the modified path the current path? */
	if(!strcmp((const char *)current_location_path, (const char *)path))
	{
		/* Path changed? */
		if(path != new_path)
		{
			if(strcmp((const char *)path, (const char *)new_path))
			{
				/* Update the current location */
				edv_vfs_browser_set_title(browser, new_path);
				edv_vfs_browser_set_location(browser, new_path, FALSE);
				edv_vfs_browser_update_location_icon(browser, new_path);
			}
		}
	}

	/* Contents List */
	edv_vfs_browser_list_vfs_object_modified_cb(
		browser,
		path,
		new_path,
		obj
	);

	/* Directory Tree */
	edv_vfs_browser_tree_vfs_object_modified_cb(
		browser,
		path,
		new_path,
		obj
	);

	/* Mount Bar */
	edv_mount_bar_update_display(browser->mount_bar);

/*	edv_vfs_browser_update_display(browser); */

	g_free(current_location_path);
}

/*
 *	Object removed callback.
 */
void edv_vfs_browser_vfs_object_removed_cb(
	EDVVFSBrowser *browser,
	const gchar *path
)
{
	gchar *current_location_path;

	if((browser == NULL) || STRISEMPTY(path))
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	current_location_path = STRDUP(edv_vfs_browser_get_location(browser));
	if(current_location_path == NULL)
		return;

	/* Contents List */
	edv_vfs_browser_list_vfs_object_removed_cb(browser, path);

	/* Directory Tree */
	edv_vfs_browser_tree_vfs_object_removed_cb(browser, path);

	/* Mount Bar */
	edv_mount_bar_update_display(browser->mount_bar);

	/* Is the removed path the current location or the parent
	 * of the current location?
	 */
	if(strpfx((const char *)current_location_path, (const char *)path))
	{
		/* Change the current location to the parent of the
		 * removed path
		 */
		gchar *parent_path = g_dirname(path);
		if(parent_path != NULL)
		{
			edv_vfs_browser_set_title(browser, parent_path);
			edv_vfs_browser_set_location(browser, parent_path, FALSE);
			edv_vfs_browser_update_location_icon(browser, parent_path);
			g_free(parent_path);
		}
	}

/*	edv_vfs_browser_update_display(browser); */

	g_free(current_location_path);
}


/*
 *	Device mount/unmount callback.
 */
void edv_vfs_browser_device_mount_cb(
	EDVVFSBrowser *browser,
	const gint dev_num, EDVDevice *d,
	const gboolean mounted
)
{
	if(browser == NULL)
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	/* Update the location icon */
	edv_vfs_browser_update_location_icon(
		browser,
		edv_vfs_browser_get_location(browser)
	);

	/* Contents List */
	edv_vfs_browser_list_device_mount_cb(
		browser,
		d,
		mounted
	);

	/* Directory Tree */
	edv_vfs_browser_tree_vfs_object_device_mount_cb(
		browser,
		d,
		mounted
	);

	edv_vfs_browser_update_display(browser);
}


/*
 *	Recycled object added callback.
 */
void edv_vfs_browser_recycled_object_added_cb(
	EDVVFSBrowser *browser,
	const guint index
)
{
	EDVCore *core;

	if(browser == NULL)
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	core = browser->core;

	/* There is not much interest when a recycled object has been
	 * added or removed. Only the menus need to be updated if there
	 * is a change in the number of recycled objects
	 */
	if(core->last_nrecycle_bin_items != browser->last_nrecycle_bin_items)
		edv_vfs_browser_update_display(browser);
}

/*
 *	Recycled object modified callback.
 */
void edv_vfs_browser_recycled_object_modified_cb(
	EDVVFSBrowser *browser,
	const guint index
)
{
	EDVCore *core;

	if(browser == NULL)
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	core = browser->core;

	/* There is not much interest when a recycled object has been
	 * added or removed. Only the menus need to be updated if there
	 * is a change in the number of recycled objects
	 */
	if(core->last_nrecycle_bin_items != browser->last_nrecycle_bin_items)
		edv_vfs_browser_update_display(browser);
}

/*
 *	Recycled object removed callback.
 */
void edv_vfs_browser_recycled_object_removed_cb(
	EDVVFSBrowser *browser,
	const guint index
)
{
	EDVCore *core;

	if(browser == NULL)
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	core = browser->core;

	/* There is not much interest when a recycled object has been
	 * added or removed. Only the menus need to be updated if there
	 * is a change in the number of recycled objects
	 */
	if(core->last_nrecycle_bin_items != browser->last_nrecycle_bin_items)
		edv_vfs_browser_update_display(browser);
}


/*
 *	Reconfigured callback.
 */
void edv_vfs_browser_reconfigured_cb(EDVVFSBrowser *browser)
{
	gchar *current_location_path;
	GtkRcStyle *standard_rcstyle, *lists_rcstyle;
	GtkWidget *w;
	CfgList *cfg_list;
	EDVCore *core;

	if(browser == NULL)
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	core = browser->core;
	cfg_list = core->cfg_list;
	standard_rcstyle = core->standard_rcstyle;
	lists_rcstyle = core->lists_rcstyle;


	/* Reset last state markers so that resources get explicitly
	 * checked due to reconfiguring
	 */
	browser->last_nrecycle_bin_items = -1;
	browser->last_write_protect_state = -1;


	/* Get the current location */
	current_location_path = STRDUP(edv_vfs_browser_get_location(browser));


	/* Update the title */
	edv_vfs_browser_set_title(browser, current_location_path);

	/* Recreate the Tool Bar */
	edv_vfs_browser_tool_bar_regenerate(browser);

	/* Recreate the File menu items */
	edv_vfs_browser_file_menu_items_regenerate(browser);

	/* Show/hide the Tool Bar */
	w = browser->tool_bar_handle;
	if(w != NULL)
	{
		if(EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_TOOL_BAR))
		{
			browser->flags |= EDV_VFS_BROWSER_TOOL_BAR_MAPPED;
			gtk_widget_show(w);
		}
		else
		{
			browser->flags &= ~EDV_VFS_BROWSER_TOOL_BAR_MAPPED;
			gtk_widget_hide(w);
		}
	}

	/* Show/hide the Location Bar */
	w = browser->location_bar_handle;
	if(w != NULL)
	{
		if(EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_LOCATION_BAR))
		{
			browser->flags |= EDV_VFS_BROWSER_LOCATION_BAR_MAPPED;
			gtk_widget_show(w);
		}
		else
		{
			browser->flags &= ~EDV_VFS_BROWSER_LOCATION_BAR_MAPPED;
			gtk_widget_hide(w);
		}
	}

	/* Show/hide the Mount Bar */
	w = browser->mount_bar_handle;
	if(w != NULL)
	{
		if(EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_MOUNT_BAR))
		{
			browser->flags |= EDV_VFS_BROWSER_MOUNT_BAR_MAPPED;
			gtk_widget_show(w);
		}
		else
		{
			browser->flags &= ~EDV_VFS_BROWSER_MOUNT_BAR_MAPPED;
			gtk_widget_hide(w);
		}
	}

	/* Show/hide the Find Bar */
	w = browser->find_bar_handle;
	if(w != NULL)
	{
		if(EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_FIND_BAR))
		{
			browser->flags |= EDV_VFS_BROWSER_FIND_BAR_MAPPED;
			gtk_widget_show(w);
		}
		else
		{
			browser->flags &= ~EDV_VFS_BROWSER_FIND_BAR_MAPPED;
			gtk_widget_hide(w);
		}
	}

	/* Show/hide the Status Bar */
	w = browser->status_bar;
	if(w != NULL)
	{
		if(EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_STATUS_BAR))
		{
			browser->flags |= EDV_VFS_BROWSER_STATUS_BAR_MAPPED;
			gtk_widget_show(w);
		}
		else
		{
			browser->flags &= ~EDV_VFS_BROWSER_STATUS_BAR_MAPPED;
			gtk_widget_hide(w);
		}
	}

	/* Recreate the New Object submenu items */
	edv_vfs_browser_new_object_menu_items_regenerate(browser);

	/* Recreate the Open To menus */
	if(browser->contents_clist_selected_row > -1)
	{
		const gint row = browser->contents_clist_selected_row;
		EDVVFSObject *obj = EDV_VFS_OBJECT(gtk_clist_get_row_data(
			GTK_CLIST(browser->contents_clist), row
		));
		const gchar *path = (obj != NULL) ? obj->path : NULL;
		EDVMIMEType *m = edv_mime_types_list_match_path(
			core->mime_types_list,
			path
		);
		edv_vfs_browser_open_to_menu_regenerate(browser, m);
	}
	else if(browser->directory_ctree_selected_node != NULL)
	{
		GtkCTreeNode *node = browser->directory_ctree_selected_node;
		EDVVFSObject *obj = EDV_VFS_OBJECT(gtk_ctree_node_get_row_data(
			GTK_CTREE(browser->directory_ctree), node
		));
		const gchar *path = (obj != NULL) ? obj->path : NULL;
		EDVMIMEType *m = edv_mime_types_list_match_path(
			core->mime_types_list,
			path
		);
		edv_vfs_browser_open_to_menu_regenerate(browser, m);
	}

	/* Recreate the Accelkey labels */
	edv_vfs_browser_accelkeys_regenerate(browser);

	/* Update the RC styles */
	w = browser->toplevel;
	if((w != NULL) && (standard_rcstyle != NULL))
		gtk_widget_modify_style_recursive(w, standard_rcstyle);
	w = browser->directory_ctree;
	if((w != NULL) && (lists_rcstyle != NULL))
		gtk_widget_modify_style_recursive(w, lists_rcstyle);
	w = browser->contents_clist;
	if((w != NULL) && (lists_rcstyle != NULL))
		gtk_widget_modify_style_recursive(w, lists_rcstyle);
	w = browser->directory_ctree_menu;
	if((w != NULL) && (standard_rcstyle != NULL))
		gtk_widget_modify_style_recursive(w, standard_rcstyle);
	w = browser->contents_clist_menu;
	if((w != NULL) && (standard_rcstyle != NULL))
		gtk_widget_modify_style_recursive(w, standard_rcstyle);

	edv_vfs_browser_cell_styles_regenerate(browser);

	/* Update the Tree */
	w = browser->directory_ctree;
	if(w != NULL)
	{
		gint i;
		GtkCTree *ctree = GTK_CTREE(w);

		i = EDV_GET_I(EDV_CFG_PARM_TREE_LINES_STYLE);
		if(ctree->line_style != i)
			gtk_ctree_set_line_style(
				ctree,
				(GtkCTreeLineStyle)i
			);

		i = EDV_GET_I(EDV_CFG_PARM_TREE_EXPANDER_STYLE);
		if(ctree->expander_style != i)
			gtk_ctree_set_expander_style(
				ctree,
				(GtkCTreeExpanderStyle)i
			);
	}

	/* Realize the listings */
	edv_vfs_browser_tree_realize_node_listing(browser, NULL);
	edv_vfs_browser_list_realize_listing(browser);

	edv_vfs_browser_update_display(browser);

	/* Notify GTK about possible size changes (such as the tool
	 * bar resizing)
	 */
	gtk_widget_queue_resize(browser->toplevel);

	g_free(current_location_path);
}


/*
 *	MIME Type added callback.
 */
void edv_vfs_browser_mime_type_added_cb(
	EDVVFSBrowser *browser,
	const gint mt_num, EDVMIMEType *m
)
{
	/* Treat a MIME Type added the same as it would be for a
	 * MIME Type modified, forward signal to the MIME Type modified
	 * callback
	 */
	edv_vfs_browser_mime_type_modified_cb(browser, mt_num, m);
}

/*
 *	MIME Type modified callback.
 */
void edv_vfs_browser_mime_type_modified_cb(
	EDVVFSBrowser *browser,
	const gint mt_num, EDVMIMEType *m
)
{
	gchar *current_location_path;
	EDVCore *core;

	if(browser == NULL)
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	core = browser->core;

	/* Get the current location */
	current_location_path = STRDUP(edv_vfs_browser_get_location(browser));

	/* Update the location icon */
	edv_vfs_browser_update_location_icon(
		browser,
		current_location_path
	);

	/* Realize listings */
	edv_vfs_browser_tree_realize_node_listing(browser, NULL);
	edv_vfs_browser_list_realize_listing(browser);

/*	edv_vfs_browser_update_display(browser); */

	g_free(current_location_path);
}

/*
 *	MIME Type removed callback.
 */
void edv_vfs_browser_mime_type_removed_cb(
	EDVVFSBrowser *browser,
	const gint mt_num
)
{
	gchar *current_location_path;
	EDVCore *core;

	if(browser == NULL)
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	core = browser->core;

	/* Get the current location */
	current_location_path = STRDUP(edv_vfs_browser_get_location(browser));

	/* Update the location icon */
	edv_vfs_browser_update_location_icon(
		browser,
		current_location_path
	);

	/* Realize listings */
	edv_vfs_browser_tree_realize_node_listing(browser, NULL);
	edv_vfs_browser_list_realize_listing(browser);

/*	edv_vfs_browser_update_display(browser); */

	g_free(current_location_path);
}


/*
 *	Device added callback.
 */
void edv_vfs_browser_device_added_cb(
	EDVVFSBrowser *browser,
	const gint dev_num, EDVDevice *d
)
{
	/* Treat a device added the same as it would be for a device
	 * modified, forward signal to the device modified callback.
	 */
	edv_vfs_browser_device_modified_cb(browser, dev_num, d);
}

/*
 *	Device modified callback.
 */
void edv_vfs_browser_device_modified_cb(
	EDVVFSBrowser *browser,
	const gint dev_num, EDVDevice *d
)
{
	gchar *current_location_path;
	EDVCore *core;

	if(browser == NULL)
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	core = browser->core;

	/* Get the current location */
	current_location_path = STRDUP(edv_vfs_browser_get_location(browser));

	/* Update the location icon */
	edv_vfs_browser_update_location_icon(
		browser,
		current_location_path
	);

	/* Realize listings */
	edv_vfs_browser_tree_realize_node_listing(browser, NULL);
	edv_vfs_browser_list_realize_listing(browser);

	edv_vfs_browser_update_display(browser);

	g_free(current_location_path);
}

/*
 *      Device removed callback.
 */
void edv_vfs_browser_device_removed_cb(
	EDVVFSBrowser *browser,
	const gint dev_num
)
{
	gchar *current_location_path;
	EDVCore *core;

	if(browser == NULL)
		return;

	if(EDV_VFS_BROWSER_IS_PROCESSING(browser))
		return;

	core = browser->core;

	/* Get the current location */
	current_location_path = STRDUP(edv_vfs_browser_get_location(browser));

	/* Update the location icon */
	edv_vfs_browser_update_location_icon(
		browser,
		current_location_path
	);

	/* Check if mount bar is currently referencing this device, if it
	 * is then its selected device needs to be set to -1
	 */
	if(edv_mount_bar_current_device(browser->mount_bar) == dev_num)
		edv_mount_bar_select(
			browser->mount_bar,
			-1
		);

	/* Realize listings */
	edv_vfs_browser_tree_realize_node_listing(browser, NULL);
	edv_vfs_browser_list_realize_listing(browser);

	edv_vfs_browser_update_display(browser);

	g_free(current_location_path);
}
