#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"

#include "cfg.h"

#include "guiutils.h"
#include "cfg_gtk.h"
#include "tool_bar.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "edv_device.h"
#include "edv_mime_type.h"
#include "edv_obj_info_match.h"
#include "edv_utils_gtk.h"
#include "edv_list_cb.h"
#include "edv_mount_bar.h"
#include "edv_find_bar.h"
#include "edv_status_bar.h"
#include "vfs_browser.h"
#include "vfs_browser_cb.h"
#include "vfs_browser_op.h"
#include "vfs_browser_tree.h"
#include "vfs_browser_list.h"
#include "vfs_browser_dnd.h"
#include "edv_cb.h"
#include "edv_open_to_menu.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_hsep_20x20.xpm"
#include "images/icon_terminal2_20x20.xpm"
#include "images/icon_copy_20x20.xpm"
#include "images/icon_paste_20x20.xpm"
#include "images/icon_new_20x20.xpm"
#include "images/icon_save_20x20.xpm"
#include "images/icon_trace_20x20.xpm"
#include "images/icon_close_20x20.xpm"
#include "images/icon_exit_20x20.xpm"
#include "images/icon_mount_20x20.xpm"
#include "images/icon_unmount_20x20.xpm"
#include "images/icon_eject_20x20.xpm"
#include "images/icon_move_file_20x20.xpm"
#include "images/icon_copy_file_20x20.xpm"
#include "images/icon_rename_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_chmod_20x20.xpm"
#include "images/icon_owned_20x20.xpm"
#include "images/icon_time_stamp_20x20.xpm"
#include "images/icon_properties_20x20.xpm"
#include "images/icon_run_20x20.xpm"
#include "images/icon_select_all_20x20.xpm"
#include "images/icon_unselect_all_20x20.xpm"
#include "images/icon_invert_selection_20x20.xpm"
#include "images/icon_reload_20x20.xpm"
#include "images/icon_download_file_20x20.xpm"
#include "images/icon_search_20x20.xpm"
#include "images/icon_mimetypes_20x20.xpm"
#include "images/icon_devices_list_20x20.xpm"
#include "images/icon_tool_bar_20x20.xpm"
#include "images/icon_status_bar_20x20.xpm"
#include "images/icon_wildcards_20x20.xpm"
#include "images/icon_resize_column_20x20.xpm"
#include "images/icon_options_20x20.xpm"
#include "images/icon_customize_20x20.xpm"
#include "images/icon_tree_origin_20x20.xpm"
#include "images/icon_tree_branch_expand_20x20.xpm"
#include "images/icon_tree_branch_colapse_20x20.xpm"
#include "images/icon_write_protect_off_20x20.xpm"
#include "images/icon_write_protect_on_20x20.xpm"
#include "images/icon_fsck_20x20.xpm"
#include "images/icon_tools_20x20.xpm"
#include "images/icon_floppy_20x20.xpm"

#include "images/icon_open_20x20.xpm"
#include "images/icon_file_20x20.xpm"
#include "images/icon_file_hidden_20x20.xpm"
#include "images/icon_file_noaccess_20x20.xpm"
#include "images/icon_folder_closed_20x20.xpm"
#include "images/icon_folder_parent_20x20.xpm"
#include "images/icon_folder_home_20x20.xpm"
#include "images/icon_link_20x20.xpm"
#include "images/icon_pipe_20x20.xpm"
#include "images/icon_device_block_20x20.xpm"
#include "images/icon_device_character_20x20.xpm"
#include "images/icon_socket_20x20.xpm"

#include "images/icon_purge_20x20.xpm"
#include "images/icon_help_20x20.xpm"
#include "images/icon_about_20x20.xpm"

#include "images/icon_endeavour_file_browser_20x20.xpm"
#include "images/icon_endeavour_image_browser_20x20.xpm"
#include "images/icon_endeavour_archiver_20x20.xpm"


/* Utilities */
EDVVFSBrowserOp *edv_vfs_browser_op_match_by_id(
	EDVVFSBrowser *browser,
	const EDVVFSBrowserOpID id
);
gboolean edv_vfs_browser_is_path_from_tree_origin(
	EDVVFSBrowser *browser,
	const gchar *path
);

/* EDVVFSBrowserCommand */
static EDVVFSBrowserCommand *edv_vfs_browser_command_new(void);
static void edv_vfs_browser_command_delete(EDVVFSBrowserCommand *cmd);

/* Title */
void edv_vfs_browser_set_title(
	EDVVFSBrowser *browser,
	const gchar *path
);

/* Location */
const gchar *edv_vfs_browser_get_location(EDVVFSBrowser *browser);
void edv_vfs_browser_set_location(
	EDVVFSBrowser *browser,
	const gchar *path,
	const gboolean record_history
);
void edv_vfs_browser_update_location_icon(
	EDVVFSBrowser *browser,
	const gchar *path
);

/* Selected Objects & Paths */
GList *edv_vfs_browser_get_selected_objects(
	EDVVFSBrowser *browser,
	const gboolean copy
);
GList *edv_vfs_browser_get_selected_paths(EDVVFSBrowser *browser);

/* Regenerate Menus, Styles, and Bars */
void edv_vfs_browser_accelkeys_regenerate(EDVVFSBrowser *browser);
void edv_vfs_browser_cell_styles_regenerate(EDVVFSBrowser *browser);
void edv_vfs_browser_tool_bar_regenerate(EDVVFSBrowser *browser);
void edv_vfs_browser_file_menu_items_regenerate(EDVVFSBrowser *browser);
void edv_vfs_browser_new_object_menu_items_regenerate(EDVVFSBrowser *browser);
void edv_vfs_browser_open_to_menu_regenerate(
	EDVVFSBrowser *browser,
	EDVMIMEType *m
);
void edv_vfs_browser_tree_menu_regenerate(EDVVFSBrowser *browser);
void edv_vfs_browser_list_menu_regenerate(EDVVFSBrowser *browser);

/* Sync Configuration */
void edv_vfs_browser_sync_configuration(EDVVFSBrowser *browser);

/* EDVVFSBrowserOp */
static EDVVFSBrowserOp *edv_vfs_browser_op_new(void);
static void edv_vfs_browser_op_delete(EDVVFSBrowserOp *op);
GList *edv_vfs_browser_ops_list_new(EDVVFSBrowser *browser);
void edv_vfs_browser_ops_list_delete(GList *op_ids_list);

/* Accel Keys */
CfgAccelkeyList *edv_vfs_browser_accelkeys_new(void);

/* File Browser */
static void EDVBrowserCreateMenuBar(
	EDVVFSBrowser *browser,
	GtkWidget *parent
);
EDVVFSBrowser *edv_vfs_browser_new(
	EDVCore *core,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry
);
void edv_vfs_browser_sync_data(EDVVFSBrowser *browser);
void edv_vfs_browser_update_display(EDVVFSBrowser *browser);
void edv_vfs_browser_set_busy(EDVVFSBrowser *browser, const gboolean busy);
gboolean edv_vfs_browser_is_mapped(EDVVFSBrowser *browser);
void edv_vfs_browser_map(EDVVFSBrowser *browser);
void edv_vfs_browser_unmap(EDVVFSBrowser *browser);
void edv_vfs_browser_delete(EDVVFSBrowser *browser);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Returns the File Browser Operation ID that matches the given
 *	operation.
 */
EDVVFSBrowserOp *edv_vfs_browser_op_match_by_id(
	EDVVFSBrowser *browser,
	const EDVVFSBrowserOpID id
)
{
	GList *glist;
	EDVVFSBrowserOp *op;

	if(browser == NULL)
		return(NULL);

	for(glist = browser->op_ids_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		op = EDV_VFS_BROWSER_OP(glist->data);
		if(op == NULL)
			continue;

		if(op->id == id)
			return(op);
	}

	return(NULL);
}

/*
 *	Checks if the path is within the current directory origin.
 *
 *	The path specifies the string describing the full path to
 *	test against the current directory tree origin.
 *
 *	Returns TRUE if the path falls within the current directory
 *	tree origin or FALSE if it falls outside of the current
 *	directory tree origin or error has occured.
 */
gboolean edv_vfs_browser_is_path_from_tree_origin(
	EDVVFSBrowser *browser,
	const gchar *path
)
{
	gchar		*origin_path,
			*test_path,
			*origin_path_ptr,
			*test_path_ptr;

	if((browser == NULL) || (path == NULL))
		return(FALSE);

	/* Get/copy the current origin path */
	origin_path = STRDUP(browser->directory_ctree_origin_path);
	if(origin_path == NULL)
		return(FALSE);

	edv_path_simplify(origin_path);

	/* If the origin path is toplevel then all paths fall within
	 * it, this needs to be checked since if the origin path is
	 * toplevel it would be too short to compare
	 */
	if(!strcmp((const char *)origin_path, "/"))
	{
		g_free(origin_path);
		return(TRUE);
	}

	/* Get/copy the specified path and simplify it */
	test_path = STRDUP(path);
	if(test_path == NULL)
	{
		g_free(origin_path);
		return(FALSE);
	}

	edv_path_simplify(test_path);

	/* Seek through the origin path and the specified path to see
	 * if any characters differ
	 */
	origin_path_ptr = origin_path;
	test_path_ptr = test_path;
	while((*origin_path_ptr != '\0') && (*test_path_ptr != '\0'))
	{
		/* If the characters differ then the specified path falls
		 * outside of the origin path
		 */
		if(*origin_path_ptr != *test_path_ptr)
		{
			g_free(origin_path);
			g_free(test_path);
			return(FALSE);
		}

		origin_path_ptr++;
		test_path_ptr++;
	}

	/* Reached the end of the specified path in the above
	 * iteration?
	 */
	if(*test_path_ptr == '\0')
	{
		/* If the origin path and the specified path are the
		 * same then the specified path falls within it otherwise
		 * it falls outside of it since the specified path is
		 * shorter than the origin path
		 */
		if(*origin_path_ptr == '\0')
		{
			/* Paths are the same */
			g_free(origin_path);
			g_free(test_path);
			return(TRUE);
		}
		else
		{
			/* Paths are different and the origin path is longer
			 * than the specified path so therefore the specified
			 * path falls outside of the origin path
			 */
			g_free(origin_path);
			g_free(test_path);
			return(FALSE);
		}
	}

	/* At this point the specified path is longer than the origin
	 * path
	 *
	 * Check if the specified path is a child/grandchild of the
	 * origin path by checking if the specified path's character
	 * at this length is a deliminator character
	 */
	if(*test_path_ptr == G_DIR_SEPARATOR)
	{
		g_free(origin_path);
		g_free(test_path);
		return(TRUE);
	}

	/* Delete the coppies of the origin and specified paths */
	g_free(origin_path);
	g_free(test_path);

	/* The specified path is not a child of the origin path
	 * and is not the same as the origin path so therefore the
	 * specified path falls outside of it
	 */
	return(FALSE);
}


/*
 *	Creates a new EDVVFSBrowserCommand.
 */
static EDVVFSBrowserCommand *edv_vfs_browser_command_new(void)
{
	return(EDV_VFS_BROWSER_COMMAND(g_malloc0(sizeof(EDVVFSBrowserCommand))));
}

/*
 *	Deletes the EDVVFSBrowserCommand.
 */
static void edv_vfs_browser_command_delete(EDVVFSBrowserCommand *cmd)
{
	if(cmd == NULL)
		return;

	g_free(cmd->label);
	g_free(cmd->command);
	g_free(cmd->description);
	g_free(cmd->ext_data);
	g_free(cmd);
}


/*
 *	Sets the title.
 *
 *	The path specifies the full path to the current location
 *	that is to be displayed on the title.
 */
void edv_vfs_browser_set_title(
	EDVVFSBrowser *browser,
	const gchar *path
)
{
	gboolean	show_program_name,
			show_user_name,
			show_location;
	const gchar	*prog_name = PROG_NAME_FULL,
			*user_name,
			*filter;
	gchar *title;
	GtkWidget *w;
	CfgList *cfg_list;
	EDVCore *core;

	if(browser == NULL)
		return;

	w = browser->toplevel;
	core = browser->core;
	cfg_list = core->cfg_list;

	/* Get the values to show in the title */
	if(!STRISEMPTY(core->title))
		prog_name = core->title;

	user_name = core->effective_user_id_host_str;
	filter = browser->contents_list_filter;

	show_program_name = EDV_GET_B(EDV_CFG_PARM_BROWSER_TITLE_WITH_PROGRAM);
	if(show_program_name && STRISEMPTY(prog_name))
		show_program_name = FALSE;

	show_user_name = EDV_GET_B(EDV_CFG_PARM_BROWSER_TITLE_WITH_USER);
	if(show_user_name && STRISEMPTY(user_name))
		show_user_name = FALSE;

	show_location = EDV_GET_B(EDV_CFG_PARM_BROWSER_TITLE_WITH_LOCATION);
	if(show_location && STRISEMPTY(path))
		show_location = FALSE;

	/* Format the new title string */
	title = g_strdup("");
	if(EDV_GET_B(EDV_CFG_PARM_BROWSER_TITLE_REVERSE))
	{
		if(show_location)
		{
			gchar *s;
			if(STRISEMPTY(filter))
				s = g_strconcat(
					title,
					path,
					NULL
				);
			else
				s = g_strconcat(
					title,
					path,
					G_DIR_SEPARATOR_S,
					filter,
					NULL
				);
			g_free(title);
			title = s;
		}
		if(show_user_name)
		{
			gchar *s = g_strconcat(
				title,
				(show_location) ? " - " : "",
				user_name,
				NULL
			);
			g_free(title);
			title = s;
		}
		if(show_program_name)
		{
			gchar *s = g_strconcat(
				title,
				(show_user_name || show_location) ? " - " : "",
				prog_name,
				NULL
			);
			g_free(title);
			title = s;
		}
	}
	else
	{
		if(show_program_name)
		{
			gchar *s = g_strconcat(
				title,
				prog_name,
				(show_user_name || show_location) ? ": " : "",
				NULL
			);
			g_free(title);
			title = s;
		}
		if(show_user_name)
		{
			gchar *s = g_strconcat(
				title,
				user_name,
				(show_location) ? "" : "",
				NULL
			);
			g_free(title);
			title = s;
		}
		if(show_location)
		{
			gchar *s;
			if(STRISEMPTY(filter))
				s = g_strconcat(
					title,
					path,
					NULL
				);
			else
				s = g_strconcat(
					title,
					path,
					G_DIR_SEPARATOR_S,
					filter,
					NULL
				);
			g_free(title);
			title = s;
		}
	}

	/* Set the new title */
	gtk_window_set_title(GTK_WINDOW(w), title);

	g_free(title);
}

/*
 *	Returns the File Browser's current location.
 */
const gchar *edv_vfs_browser_get_location(EDVVFSBrowser *browser)
{
	GtkCombo *combo;

	if(browser == NULL)
		return(NULL);

	combo = GTK_COMBO(browser->location_combo);
	return(gtk_entry_get_text(GTK_ENTRY(combo->entry)));
}

/*
 *	Sets the File Browser's location to the specified path.
 *
 *	If record_history is TRUE then the new location will be
 *	recorded on the locations history.
 */
void edv_vfs_browser_set_location(
	EDVVFSBrowser *browser,
	const gchar *path,
	const gboolean record_history
)
{
	const gchar *s;
	gchar *new_path;
	GtkEntry *entry;
	GtkCombo *combo;

	if((browser == NULL) || STRISEMPTY(path))
		return;

	combo = GTK_COMBO(browser->location_combo);
	entry = GTK_ENTRY(combo->entry);

	/* Make a copy of the specified path */
	new_path = STRDUP(path);
	if(new_path == NULL)
		return;

	/* Simplify the new path */
	edv_path_simplify(new_path);

	/* No change? */
	s = gtk_entry_get_text(entry);
	if(s != NULL)
	{
		if(!strcmp((const char *)s, (const char *)new_path))
		{
			g_free(new_path);
			return;
		}
	}

	/* Record new location on the location history? */
	if(record_history)
		GUIComboAddItem(GTK_WIDGET(combo), new_path);

	/* Set the new location */
	gtk_entry_set_text(entry, new_path);
	gtk_entry_set_position(entry, -1);

	g_free(new_path);
}

/*
 *	Updates the File Browser's location bar icon with the object
 *	specified by path.
 *
 *	If the given path is NULL then the icon will be set as unknown.
 */
void edv_vfs_browser_update_location_icon(
	EDVVFSBrowser *browser,
	const gchar *path
)
{
	GtkWidget	*w,
			*parent;
	EDVPixmap	*icon,
			*icon_inaccessable,
			*icon_hidden;
	EDVVFSObject *obj;
	EDVCore *core;

	if(browser == NULL)
		return;

	parent = browser->location_icon_fixed;
	core = browser->core;

	obj = edv_vfs_object_lstat(path);
	if(obj == NULL)
	{
		obj = edv_vfs_object_new();
		if(obj != NULL)
		{
			obj->type = EDV_OBJECT_TYPE_DIRECTORY;
			edv_vfs_object_set_path(obj, path);
		}
	}
	if(obj == NULL)
		return;

	/* Get the location icon */
	(void)edv_match_object_icon(
		core->devices_list,
		core->mime_types_list,
		obj->type,
		obj->path,
		EDV_VFS_OBJECT_LINK_TARGET_EXISTS(obj),
		obj->permissions,
		EDV_ICON_SIZE_20,
		&icon,
		NULL,
		&icon_inaccessable,
		&icon_hidden
	);
	/* Hidden? */
	if(edv_is_object_hidden(obj))
	{
		if(edv_pixmap_is_loaded(icon_hidden))
		{
			(void)edv_pixmap_unref(icon);
			icon = edv_pixmap_ref(icon_hidden);
		}
	}
	/* No access? */
	if(!edv_is_object_accessable(core, obj))
	{
		if(edv_pixmap_is_loaded(icon_inaccessable))
		{
			(void)edv_pixmap_unref(icon);
			icon = edv_pixmap_ref(icon_inaccessable);
		}
	}

	edv_vfs_object_delete(obj);

	/* Create or set the location icon */ 
	w = browser->location_icon_pm;
	if(w != NULL)
	{
		edv_pixmap_set_gtk_pixmap(icon, w);
	}
	else
	{
		browser->location_icon_pm = w = edv_pixmap_new_gtk_pixmap(icon);
		if(w != NULL)
		{
			gtk_fixed_put(GTK_FIXED(parent), w, 0, 0);
			gtk_widget_show(w);
		}
	}

	(void)edv_pixmap_unref(icon);
	(void)edv_pixmap_unref(icon_inaccessable);
	(void)edv_pixmap_unref(icon_hidden);
}

/*
 *	Gets a list of the selected objects.
 *
 *	If copy is TRUE then each object in the returned list will
 *	be a copy of the original object.
 *
 *	Returns a GList of EDVVFSObject * objects. If copy
 *	was TRUE then the calling function must delete each object
 *	in the list and the list itself, otherwise the calling
 *	function need only delete the returned list.
 */
GList *edv_vfs_browser_get_selected_objects(
	EDVVFSBrowser *browser,
	const gboolean copy
)
{
	GList *objs_list;
	GtkCList *contents_clist;
	GtkCTree *directory_ctree;

	if(browser == NULL)
		return(NULL);

	contents_clist = GTK_CLIST(browser->contents_clist);
	directory_ctree = GTK_CTREE(browser->directory_ctree);

	objs_list = NULL;

	/* Contents GtkCLiist has selected objects? */
	if(contents_clist->selection != NULL)
	{
		gint row;
		GList *glist;
		GtkCList *clist = contents_clist;
		EDVVFSObject *obj;

		for(glist = clist->selection;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			row = (gint)glist->data;
			obj = EDV_VFS_OBJECT(gtk_clist_get_row_data(clist, row));
			if(obj == NULL)
				continue;

			objs_list = g_list_append(
				objs_list,
				(copy) ? edv_vfs_object_copy(obj) : obj
			);
		}
	}
	/* Directory GtkCTree has selected objects? */
	else if(GTK_CLIST(directory_ctree)->selection != NULL)
	{
		GList *glist;
		GtkCTreeNode *node;
		GtkCTree *ctree = directory_ctree;
		GtkCList *clist = GTK_CLIST(ctree);
		EDVVFSObject *obj;

		for(glist = clist->selection;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			node = (GtkCTreeNode *)glist->data;
			obj = EDV_VFS_OBJECT(gtk_ctree_node_get_row_data(ctree, node));
			if(obj == NULL)
				continue;

			objs_list = g_list_append(
				objs_list,
				(copy) ? edv_vfs_object_copy(obj) : obj
			);
		}
	}

	return(objs_list);
}

/*
 *	Gets the list of selected paths.
 *
 *	Returns a dynamically allocated GList of gchar * paths.
 */
GList *edv_vfs_browser_get_selected_paths(EDVVFSBrowser *browser)
{
	GList *paths_list;

	if(browser == NULL)
		return(NULL);

	paths_list = NULL;

	/* Contents GtkCLiist was last selected? */
	if(browser->contents_clist_selected_row > -1)
	{
		gint row;
		GList *glist;
		GtkCList *clist = GTK_CLIST(browser->contents_clist);
		EDVVFSObject *obj;

		for(glist = clist->selection;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			row = (gint)glist->data;
			obj = EDV_VFS_OBJECT(gtk_clist_get_row_data(clist, row));
			if(obj == NULL)
				continue;

			if(obj->path == NULL)
				continue;

			paths_list = g_list_append(
				paths_list,
				STRDUP(obj->path)
			);
		}    
	}
	/* Directory GtkCTree was last selected? */
	else if(browser->directory_ctree_selected_node != NULL)
	{
		GList *glist;
		GtkCTreeNode *node;
		GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
		GtkCList *clist = GTK_CLIST(ctree);
		EDVVFSObject *obj;

		for(glist = clist->selection;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			node = (GtkCTreeNode *)glist->data;
			obj = EDV_VFS_OBJECT(gtk_ctree_node_get_row_data(ctree, node));
			if(obj == NULL)
				continue;

			if(obj->path == NULL)
				continue;

			paths_list = g_list_append(
				paths_list,
				STRDUP(obj->path)
			);
		}
	}

	return(paths_list);
}


/*
 *	Resets the File Browser's accelkeys displayed on the widgets.
 */
void edv_vfs_browser_accelkeys_regenerate(EDVVFSBrowser *browser)
{
	GtkAccelGroup *accelgrp = NULL;
	CfgAccelkeyList *ak_list;
	CfgList *cfg_list;
	EDVCore *core;

	if(browser == NULL)
		return;

	core = browser->core;
	cfg_list = core->cfg_list;

	ak_list = EDV_GET_ACCELKEY_LIST(EDV_CFG_PARM_BROWSER_ACCELERATOR_KEYS);
	if(ak_list == NULL)
		return;

#define UPDATE_ACCELKEY(_w_,_id_)	{	\
 if(((_w_) != NULL) && ((_id_) > 0)) {		\
  GList *glist;					\
  CfgAccelkey *ak;				\
						\
  for(glist = ak_list->list;			\
      glist != NULL;				\
      glist = g_list_next(glist)		\
  ) {						\
   ak = CFG_ACCELKEY(glist->data);		\
   if(ak == NULL)				\
    continue;					\
						\
   if(ak->op_id == (gint)(_id_)) {		\
    GUIMenuItemSetAccelKey(			\
     (_w_), accelgrp, ak->key, ak->modifiers	\
    );						\
    break;					\
   }						\
  }						\
 }						\
}

	UPDATE_ACCELKEY(
		browser->open_mi,
		EDV_VFS_BROWSER_OP_OPEN
	);
	UPDATE_ACCELKEY(
		browser->open_to_mi,
		EDV_VFS_BROWSER_OP_OPEN_TO
	);
	UPDATE_ACCELKEY(
		browser->open_with_mi,
		EDV_VFS_BROWSER_OP_OPEN_WITH
	);
	UPDATE_ACCELKEY(
		browser->new_file_mi,
		EDV_VFS_BROWSER_OP_NEW_FILE
	);
	UPDATE_ACCELKEY(
		browser->new_directory_mi,
		EDV_VFS_BROWSER_OP_NEW_DIRECTORY
	);
	UPDATE_ACCELKEY(
		browser->new_link_mi,
		EDV_VFS_BROWSER_OP_NEW_LINK
	);
	UPDATE_ACCELKEY(
		browser->new_fifo_mi,
		EDV_VFS_BROWSER_OP_NEW_FIFO
	);
	UPDATE_ACCELKEY(
		browser->new_device_block_mi,
		EDV_VFS_BROWSER_OP_NEW_DEVICE_BLOCK
	);
	UPDATE_ACCELKEY(
		browser->new_device_character_mi,
		EDV_VFS_BROWSER_OP_NEW_DEVICE_CHARACTER
	);
	UPDATE_ACCELKEY(
		browser->new_socket_mi,
		EDV_VFS_BROWSER_OP_NEW_SOCKET
	);
	UPDATE_ACCELKEY(
		browser->sync_disks_mi,
		EDV_VFS_BROWSER_OP_SYNC_DISKS
	);
	UPDATE_ACCELKEY(
		browser->history_mi,
		EDV_VFS_BROWSER_OP_HISTORY
	);
	UPDATE_ACCELKEY(
		browser->run_mi,
		EDV_VFS_BROWSER_OP_RUN
	);
	UPDATE_ACCELKEY(
		browser->close_mi,
		EDV_VFS_BROWSER_OP_CLOSE
	);

	UPDATE_ACCELKEY(
		browser->move_mi,
		EDV_VFS_BROWSER_OP_MOVE
	);
	UPDATE_ACCELKEY(
		browser->copy_mi,
		EDV_VFS_BROWSER_OP_COPY
	);
	UPDATE_ACCELKEY(
		browser->link_mi,
		EDV_VFS_BROWSER_OP_LINK
	);
	UPDATE_ACCELKEY(
		browser->rename_mi,
		EDV_VFS_BROWSER_OP_RENAME
	);
	UPDATE_ACCELKEY(
		browser->chmod_mi,
		EDV_VFS_BROWSER_OP_CHMOD
	);
	UPDATE_ACCELKEY(
		browser->chown_mi,
		EDV_VFS_BROWSER_OP_CHOWN
	);
	UPDATE_ACCELKEY(
		browser->chtime_mi,
		EDV_VFS_BROWSER_OP_CHTIME
	);
	UPDATE_ACCELKEY(
		browser->delete_mi,
		EDV_VFS_BROWSER_OP_DELETE
	);
	UPDATE_ACCELKEY(
		browser->copy_path_mi,
		EDV_VFS_BROWSER_OP_COPY_PATH
	);
	UPDATE_ACCELKEY(
		browser->copy_url_mi,
		EDV_VFS_BROWSER_OP_COPY_URL
	);
	UPDATE_ACCELKEY(
		browser->paste_mi,
		EDV_VFS_BROWSER_OP_PASTE
	);
	UPDATE_ACCELKEY(
		browser->select_all_mi,
		EDV_VFS_BROWSER_OP_SELECT_ALL
	);
	UPDATE_ACCELKEY(
		browser->unselect_all_mi,
		EDV_VFS_BROWSER_OP_UNSELECT_ALL
	);
	UPDATE_ACCELKEY(
		browser->invert_selection_mi,
		EDV_VFS_BROWSER_OP_INVERT_SELECTION
	);
	UPDATE_ACCELKEY(
		browser->find_mi,
		EDV_VFS_BROWSER_OP_FIND
	);
	UPDATE_ACCELKEY(
		browser->properties_mi,
		EDV_VFS_BROWSER_OP_PROPERTIES
	);

	UPDATE_ACCELKEY(
		browser->mount_mi,
		EDV_VFS_BROWSER_OP_MOUNT
	);
	UPDATE_ACCELKEY(
		browser->eject_mi,
		EDV_VFS_BROWSER_OP_EJECT
	);
	UPDATE_ACCELKEY(
		browser->device_check_mi,
		EDV_VFS_BROWSER_OP_DEVICE_CHECK
	);
	UPDATE_ACCELKEY(
		browser->device_tools_mi,
		EDV_VFS_BROWSER_OP_DEVICE_TOOLS
	);
	UPDATE_ACCELKEY(
		browser->device_format_mi,
		EDV_VFS_BROWSER_OP_DEVICE_FORMAT
	);
	UPDATE_ACCELKEY(
		browser->devices_mi,
		EDV_VFS_BROWSER_OP_DEVICES
	);

	UPDATE_ACCELKEY(
		browser->refresh_mi,
		EDV_VFS_BROWSER_OP_REFRESH
	);
	UPDATE_ACCELKEY(
		browser->refresh_all_mi,
		EDV_VFS_BROWSER_OP_REFRESH_ALL
	);
	UPDATE_ACCELKEY(
		browser->show_tool_bar_micheck,
		EDV_VFS_BROWSER_OP_SHOW_TOOL_BAR
	);
	UPDATE_ACCELKEY(
		browser->show_location_bar_micheck,
		EDV_VFS_BROWSER_OP_SHOW_LOCATION_BAR
	);
	UPDATE_ACCELKEY(
		browser->show_mount_bar_micheck,
		EDV_VFS_BROWSER_OP_SHOW_MOUNT_BAR
	);
	UPDATE_ACCELKEY(
		browser->show_find_bar_micheck,
		EDV_VFS_BROWSER_OP_SHOW_FIND_BAR
	);
	UPDATE_ACCELKEY(
		browser->show_status_bar_micheck,
		EDV_VFS_BROWSER_OP_SHOW_STATUS_BAR
	);
	UPDATE_ACCELKEY(
		browser->directory_tree_origin_mi,
		EDV_VFS_BROWSER_OP_DIRECTORY_TREE_ORIGIN
	);
	UPDATE_ACCELKEY(
		browser->show_hidden_objects_micheck,
		EDV_VFS_BROWSER_OP_SHOW_HIDDEN_OBJECTS
	);
	UPDATE_ACCELKEY(
		browser->show_noaccess_objects_micheck,
		EDV_VFS_BROWSER_OP_SHOW_NOACCESS_OBJECTS
	);
	UPDATE_ACCELKEY(
		browser->contents_list_filter_mi,
		EDV_VFS_BROWSER_OP_LIST_FILTER
	);
	UPDATE_ACCELKEY(
		browser->contents_list_auto_resize_columns_micheck,
		EDV_VFS_BROWSER_OP_CONTENTS_LIST_AUTO_RESIZE_COLUMNS
	);
	UPDATE_ACCELKEY(
		browser->mime_types_mi,
		EDV_VFS_BROWSER_OP_MIME_TYPES
	);

	UPDATE_ACCELKEY(
		browser->write_protect_mi,
		EDV_VFS_BROWSER_OP_WRITE_PROTECT
	);
	UPDATE_ACCELKEY(
		browser->delete_method_recycle_micheck,
		EDV_VFS_BROWSER_OP_DELETE_METHOD_RECYCLE
	);
	UPDATE_ACCELKEY(
		browser->delete_method_purge_micheck,
		EDV_VFS_BROWSER_OP_DELETE_METHOD_PURGE
	);
	UPDATE_ACCELKEY(
		browser->options_mi,
		EDV_VFS_BROWSER_OP_OPTIONS
	);
	UPDATE_ACCELKEY(
		browser->customize_mi,
		EDV_VFS_BROWSER_OP_CUSTOMIZE
	);

	UPDATE_ACCELKEY(
		browser->window_new_browser_mi,
		EDV_VFS_BROWSER_OP_NEW_BROWSER
	);
	UPDATE_ACCELKEY(
		browser->window_new_imbr_mi,
		EDV_VFS_BROWSER_OP_NEW_IMBR
	);
	UPDATE_ACCELKEY(
		browser->window_new_archiver_mi,
		EDV_VFS_BROWSER_OP_NEW_ARCHIVER
	);
	UPDATE_ACCELKEY(
		browser->window_recycle_bin_mi,
		EDV_VFS_BROWSER_OP_RECYCLE_BIN
	);

	UPDATE_ACCELKEY(
		browser->help_contents_mi,
		EDV_VFS_BROWSER_OP_HELP_CONTENTS
	);
	UPDATE_ACCELKEY(
		browser->help_file_browser_mi,
		EDV_VFS_BROWSER_OP_HELP_FILE_BROWSER
	);
	UPDATE_ACCELKEY(
		browser->help_keys_list_mi,
		EDV_VFS_BROWSER_OP_HELP_KEYS_LIST
	);
	UPDATE_ACCELKEY(
		browser->help_mime_types_mi,
		EDV_VFS_BROWSER_OP_HELP_MIME_TYPES
	);
	UPDATE_ACCELKEY(
		browser->help_devices_mi,
		EDV_VFS_BROWSER_OP_HELP_DEVICES
	);
	UPDATE_ACCELKEY(
		browser->help_common_operations_mi,
		EDV_VFS_BROWSER_OP_HELP_COMMON_OPERATIONS
	);
	UPDATE_ACCELKEY(
		browser->help_about_mi,
		EDV_VFS_BROWSER_OP_HELP_ABOUT
	);


	UPDATE_ACCELKEY(
		browser->directory_ctree_expand_mi,
		EDV_VFS_BROWSER_OP_EXPAND
	);
	UPDATE_ACCELKEY(
		browser->directory_ctree_mount_mi,
		EDV_VFS_BROWSER_OP_MOUNT
	);
	UPDATE_ACCELKEY(
		browser->directory_ctree_eject_mi,
		EDV_VFS_BROWSER_OP_EJECT
	);
	UPDATE_ACCELKEY(
		browser->directory_ctree_move_mi,
		EDV_VFS_BROWSER_OP_MOVE
	);
	UPDATE_ACCELKEY(
		browser->directory_ctree_copy_mi,
		EDV_VFS_BROWSER_OP_COPY
	);
	UPDATE_ACCELKEY(
		browser->directory_ctree_link_mi,
		EDV_VFS_BROWSER_OP_LINK
	);
	UPDATE_ACCELKEY(
		browser->directory_ctree_rename_mi,
		EDV_VFS_BROWSER_OP_RENAME
	);
	UPDATE_ACCELKEY(
		browser->directory_ctree_chmod_mi,
		EDV_VFS_BROWSER_OP_CHMOD
	);
	UPDATE_ACCELKEY(
		browser->directory_ctree_chown_mi,
		EDV_VFS_BROWSER_OP_CHOWN
	);
	UPDATE_ACCELKEY(
		browser->directory_ctree_chtime_mi,
		EDV_VFS_BROWSER_OP_CHTIME
	);
	UPDATE_ACCELKEY(
		browser->directory_ctree_delete_mi,
		EDV_VFS_BROWSER_OP_DELETE
	);
	UPDATE_ACCELKEY(
		browser->directory_ctree_copy_path_mi,
		EDV_VFS_BROWSER_OP_COPY_PATH
	);
	UPDATE_ACCELKEY(
		browser->directory_ctree_copy_url_mi,
		EDV_VFS_BROWSER_OP_COPY_URL
	);
	UPDATE_ACCELKEY(
		browser->directory_ctree_paste_mi,
		EDV_VFS_BROWSER_OP_PASTE
	);
	UPDATE_ACCELKEY(
		browser->directory_ctree_properties_mi,
		EDV_VFS_BROWSER_OP_PROPERTIES
	);


	UPDATE_ACCELKEY(
		browser->contents_clist_open_mi,
		EDV_VFS_BROWSER_OP_OPEN
	);
	UPDATE_ACCELKEY(
		browser->contents_clist_open_to_mi,
		EDV_VFS_BROWSER_OP_OPEN_TO
	);
	UPDATE_ACCELKEY(
		browser->contents_clist_open_with_mi,
		EDV_VFS_BROWSER_OP_OPEN_WITH
	);
	UPDATE_ACCELKEY(
		browser->contents_clist_move_mi,
		EDV_VFS_BROWSER_OP_MOVE
	);
	UPDATE_ACCELKEY(
		browser->contents_clist_copy_mi,
		EDV_VFS_BROWSER_OP_COPY
	);
	UPDATE_ACCELKEY(
		browser->contents_clist_link_mi,
		EDV_VFS_BROWSER_OP_LINK
	);
	UPDATE_ACCELKEY(
		browser->contents_clist_rename_mi,
		EDV_VFS_BROWSER_OP_RENAME
	);
	UPDATE_ACCELKEY(
		browser->contents_clist_chmod_mi,
		EDV_VFS_BROWSER_OP_CHMOD
	);
	UPDATE_ACCELKEY(
		browser->contents_clist_chown_mi,
		EDV_VFS_BROWSER_OP_CHOWN
	);
	UPDATE_ACCELKEY(
		browser->contents_clist_chtime_mi,
		EDV_VFS_BROWSER_OP_CHTIME
	);
	UPDATE_ACCELKEY(
		browser->contents_clist_delete_mi,
		EDV_VFS_BROWSER_OP_DELETE
	);
	UPDATE_ACCELKEY(
		browser->contents_clist_copy_path_mi,
		EDV_VFS_BROWSER_OP_COPY_PATH
	);
	UPDATE_ACCELKEY(
		browser->contents_clist_copy_url_mi,
		EDV_VFS_BROWSER_OP_COPY_URL
	);
	UPDATE_ACCELKEY(
		browser->contents_clist_paste_mi,
		EDV_VFS_BROWSER_OP_PASTE
	);
	UPDATE_ACCELKEY(
		browser->contents_clist_properties_mi,
		EDV_VFS_BROWSER_OP_PROPERTIES
	);


#undef UPDATE_ACCELKEY
}

/*
 *	Recreates the list GtkStyles.
 */
void edv_vfs_browser_cell_styles_regenerate(EDVVFSBrowser *browser)
{
	const EDVVFSBrowserCellStyleType ncell_styles = EDV_VFS_BROWSER_NCELL_STYLE_TYPES;
	EDVVFSBrowserCellStyleType i;
	GdkColormap *colormap;
	GtkStyle	*style,
					*src_style;
	GtkWidget *w;

	if(browser == NULL)
		return;

	w = browser->contents_clist;

	/* Unref the existing GtkStyles */
	for(i = 0; i < ncell_styles; i++)
	{
		GTK_STYLE_UNREF(browser->cell_style[i]);
		browser->cell_style[i] = NULL;
	}

	/* Get the source GdkColormap and GtkStyle */
	colormap = gtk_widget_get_colormap(w);
	src_style = gtk_widget_get_style(w);
	if((colormap == NULL) || (src_style == NULL))
		return;


	/* Begin creating the GtkStyles */

#define REALLOC_COLOR(_ct_,_cs_)		{		\
 GDK_COLORMAP_FREE_COLOR(colormap, (_ct_));			\
 memcpy((_ct_), (_cs_), sizeof(GdkColor));			\
 GDK_COLORMAP_ALLOC_COLOR(colormap, (_ct_));			\
}

	/* No Access */
	i = EDV_VFS_BROWSER_CELL_STYLE_NO_ACCESS;
	browser->cell_style[i] = style = gtk_style_copy(src_style);
	if(style != NULL)
	{
		const GdkColor	cs = EDV_GDKCOLOR_NO_ACCESS_FG,
							cs_sel = EDV_GDKCOLOR_NO_ACCESS_FG_SELECTED,
							cs_insen = EDV_GDKCOLOR_NO_ACCESS_FG_INSENSITIVE;

		GtkStateType state = GTK_STATE_NORMAL;
		REALLOC_COLOR(&style->fg[state], &cs);

		state = GTK_STATE_ACTIVE;
		REALLOC_COLOR(&style->fg[state], &cs);

		state = GTK_STATE_PRELIGHT;
		REALLOC_COLOR(&style->fg[state], &cs);

		state = GTK_STATE_SELECTED;
		REALLOC_COLOR(&style->fg[state], &cs_sel);

		state = GTK_STATE_INSENSITIVE;
		REALLOC_COLOR(&style->fg[state], &cs_insen);
	}

	/* Dangling Link */
	i = EDV_VFS_BROWSER_CELL_STYLE_DANGLING_LINK;
	browser->cell_style[i] = style = gtk_style_copy(src_style);
	if(style != NULL)
	{
		const GdkColor	cs = EDV_GDKCOLOR_DANGLING_LINK_FG,
							cs_sel = EDV_GDKCOLOR_DANGLING_LINK_FG_SELECTED,
							cs_insen = EDV_GDKCOLOR_DANGLING_LINK_FG_INSENSITIVE;

		GtkStateType state = GTK_STATE_NORMAL;
		REALLOC_COLOR(&style->fg[state], &cs);

		state = GTK_STATE_ACTIVE;
		REALLOC_COLOR(&style->fg[state], &cs);

		state = GTK_STATE_PRELIGHT;
		REALLOC_COLOR(&style->fg[state], &cs);

		state = GTK_STATE_SELECTED;
		REALLOC_COLOR(&style->fg[state], &cs_sel);

		state = GTK_STATE_INSENSITIVE;
		REALLOC_COLOR(&style->fg[state], &cs_insen);
	}

	/* Infinately Recursive Link */
	i = EDV_VFS_BROWSER_CELL_STYLE_RECURSIVE_LINK;
	browser->cell_style[i] = style = gtk_style_copy(src_style);
	if(style != NULL)
	{
		const GdkColor      cs = EDV_GDKCOLOR_RECURSIVE_LINK_FG,
							cs_sel = EDV_GDKCOLOR_RECURSIVE_LINK_FG_SELECTED,
							cs_insen = EDV_GDKCOLOR_RECURSIVE_LINK_FG_INSENSITIVE;

		GtkStateType state = GTK_STATE_NORMAL;
		REALLOC_COLOR(&style->fg[state], &cs);

		state = GTK_STATE_ACTIVE;
		REALLOC_COLOR(&style->fg[state], &cs);

		state = GTK_STATE_PRELIGHT; 
		REALLOC_COLOR(&style->fg[state], &cs);

		state = GTK_STATE_SELECTED;
		REALLOC_COLOR(&style->fg[state], &cs_sel);

		state = GTK_STATE_INSENSITIVE;
		REALLOC_COLOR(&style->fg[state], &cs_insen);
	}

	/* Hidden Object */
	i = EDV_VFS_BROWSER_CELL_STYLE_HIDDEN;
	browser->cell_style[i] = style = gtk_style_copy(src_style);
	if(style != NULL)
	{
		const GdkColor	cs = EDV_GDKCOLOR_HIDDEN_OBJECT_FG,
							cs_sel = EDV_GDKCOLOR_HIDDEN_OBJECT_FG_SELECTED,
							cs_insen = EDV_GDKCOLOR_HIDDEN_OBJECT_FG_INSENSITIVE;

		GtkStateType state = GTK_STATE_NORMAL;
		REALLOC_COLOR(&style->fg[state], &cs);

		state = GTK_STATE_ACTIVE;
		REALLOC_COLOR(&style->fg[state], &cs);

		state = GTK_STATE_PRELIGHT;
		REALLOC_COLOR(&style->fg[state], &cs);

		state = GTK_STATE_SELECTED;
		REALLOC_COLOR(&style->fg[state], &cs_sel);

		state = GTK_STATE_INSENSITIVE;
		REALLOC_COLOR(&style->fg[state], &cs_insen);
	}

	/* Executable */
	i = EDV_VFS_BROWSER_CELL_STYLE_EXECUTABLE;
	browser->cell_style[i] = style = gtk_style_copy(src_style);
	if(style != NULL)
	{
		const GdkColor	cs = EDV_GDKCOLOR_EXECUTABLE_FG,
							cs_sel = EDV_GDKCOLOR_EXECUTABLE_FG_SELECTED,
							cs_insen = EDV_GDKCOLOR_EXECUTABLE_FG_INSENSITIVE;

		GtkStateType state = GTK_STATE_NORMAL;
		REALLOC_COLOR(&style->fg[state], &cs);

		state = GTK_STATE_ACTIVE;
		REALLOC_COLOR(&style->fg[state], &cs);

		state = GTK_STATE_PRELIGHT;
		REALLOC_COLOR(&style->fg[state], &cs);

		state = GTK_STATE_SELECTED;
		REALLOC_COLOR(&style->fg[state], &cs_sel);

		state = GTK_STATE_INSENSITIVE;
		REALLOC_COLOR(&style->fg[state], &cs_insen);
	}

#undef REALLOC_COLOR
}

/*
 *	Recreates the File Browser's Tool Bar.
 */
void edv_vfs_browser_tool_bar_regenerate(EDVVFSBrowser *browser)
{
	const gint border_minor = 2;
	GList		*glist,
			*items_list;
	GtkReliefStyle tb_relief;
	GtkWidget	*w,
			*parent;
	CfgIntList *intlist;
	CfgList *cfg_list;
	ToolBarButtonDecals tb_button_decals;
	ToolBarItem *item;
	EDVVFSBrowserOp *op;
	EDVCore *core;

	if(browser == NULL)
		return;

	parent = browser->tool_bar_handle;
	core = browser->core;
	cfg_list = core->cfg_list;

	/* Get the Tool Bar configuration */
	tb_button_decals = (ToolBarButtonDecals)EDV_GET_I(
		EDV_CFG_PARM_BROWSER_TOOL_BAR_BUTTON_DECALS
	);
	tb_relief = (GtkReliefStyle)EDV_GET_I(
		EDV_CFG_PARM_BROWSER_TOOL_BAR_RELIEF
	);
	intlist = EDV_GET_INT_LIST(EDV_CFG_PARM_BROWSER_TOOL_BAR);
	if(intlist == NULL)
		return;

	/* Delete the existing Tool Bar */
	GTK_WIDGET_DESTROY(browser->tool_bar);
	browser->tool_bar = NULL;

	/* Create the ToolBarItems list */
	items_list = NULL;
	for(glist = intlist->list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		/* Get the OPID */ 
		op = edv_vfs_browser_op_match_by_id(
			browser,
			(EDVVFSBrowserOpID)glist->data
		);
		if(op == NULL)
			continue;

		/* Exclude this OPID from Tool Bars? */
		if(op->flags & EDV_OPID_EX_TOOL_BAR)
			continue;

		/* Create a new ToolBarItem from this OPID */
		item = ToolBarItemNew(
			op->tb_item_type,
			op->button_name,
			op->button_icon_data,
			op->tooltip,
			op->id,
			edv_vfs_browser_op_cb, op,
			edv_vfs_browser_op_enter_cb, op,
			edv_vfs_browser_op_leave_cb, op
		);
		if(item == NULL)
			continue;

		/* Append the new ToolBarItem to the list */
		items_list = g_list_append(
			items_list,
			item
		);
	}

	/* Create the Tool Bar */
	browser->tool_bar = w = ToolBarNew(
		GTK_ORIENTATION_HORIZONTAL,
		tb_button_decals,
		tb_relief,
		GTK_POLICY_AUTOMATIC,
		items_list
	);
	gtk_widget_set_name(w, EDV_WIDGET_NAME_TOOL_BAR);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_minor);
	gtk_widget_show(w);

	/* Delete the Tool Bar items list */
	if(items_list != NULL)
	{
		g_list_foreach(
			items_list,
			(GFunc)ToolBarItemDelete,
			NULL
		);
		g_list_free(items_list);
	}
}

/*
 *	Recreates the File Browser's File menu items.
 */
void edv_vfs_browser_file_menu_items_regenerate(EDVVFSBrowser *browser)
{
	edv_pixmap_data *icon_data = NULL;
	GList *glist;
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget	*w,
			*menu,
			*submenu;
	GtkMenuShell *menushell;
	CfgList *cfg_list;
	CfgMenu *cfg_menu;
	EDVVFSBrowserOpID id;
	EDVCore *core;

	if(browser == NULL)
		return;

	core = browser->core;
	cfg_list = core->cfg_list;

	/* Get the File menu */
	menu = browser->file_menu;
	if(menu == NULL)
		return;

	menushell = GTK_MENU_SHELL(menu);

	/* Delete the existing menu items */
	glist = g_list_copy(menushell->children);
	if(glist != NULL)
	{
		g_list_foreach(glist, (GFunc)gtk_widget_destroy, NULL);
		g_list_free(glist);
	}

#define ADD_MENU_ITEM_LABEL	{		\
 EDVVFSBrowserOp *op =				\
  edv_vfs_browser_op_match_by_id(browser, id);		\
 if(op != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu,					\
   GUI_MENU_ITEM_TYPE_LABEL,			\
   accelgrp,					\
   (icon_data != NULL) ?			\
    icon_data : op->menu_icon_data,		\
   op->menu_name,				\
   op->accel_key, op->accel_mods,		\
   edv_vfs_browser_menu_item_cb, op			\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)edv_vfs_browser_menu_item_enter_cb, op,	\
   (gpointer)edv_vfs_browser_menu_item_leave_cb, op	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_ITEM_CMD_OBJ_OP(_item_) {	\
 if((_item_) != NULL) {				\
  EDVVFSBrowserCommand *cmd;			\
  const gchar *icon_file = (_item_)->icon_file;	\
  GdkBitmap *mask;				\
  GdkPixmap *pixmap;				\
  if(icon_file != NULL) {  			\
   pixmap = GDK_PIXMAP_NEW_FROM_XPM_FILE(	\
    &mask, icon_file				\
   );						\
  } else {					\
   pixmap = NULL;				\
   mask = NULL;					\
  }						\
						\
  cmd = edv_vfs_browser_command_new();			\
  if(cmd != NULL) {				\
   cmd->vfs_browser = browser;			\
   cmd->label = STRDUP((_item_)->label);	\
   cmd->command = STRDUP((_item_)->command);	\
   cmd->description = STRDUP((_item_)->description);\
   cmd->ext_data = STRDUP((_item_)->ext_data);	\
   cmd->flags = (_item_)->flags;		\
						\
   if((_item_)->flags & CFG_MENU_ITEM_IS_SEPARATOR) { \
    w = GUIMenuItemCreate(			\
     menu,					\
     GUI_MENU_ITEM_TYPE_SEPARATOR,		\
     NULL,					\
     NULL,					\
     NULL,					\
     0, 0,					\
     NULL, NULL					\
    );						\
   } else {					\
    w = GUIMenuItemCreate(			\
     menu,					\
     GUI_MENU_ITEM_TYPE_LABEL,			\
     accelgrp,					\
     NULL,					\
     (_item_)->label,				\
     0, 0,					\
     edv_vfs_browser_menu_item_cmd_object_op_cb, cmd	\
    );						\
    GUISetMenuItemCrossingCB(			\
     w,						\
     (gpointer)edv_vfs_browser_menu_item_cmd_enter_cb, cmd,\
     (gpointer)edv_vfs_browser_menu_item_cmd_leave_cb, cmd \
    );						\
    if(pixmap != NULL)				\
     GUIMenuItemSetPixmap2(			\
      w,					\
      pixmap, mask				\
     );						\
   }						\
						\
   gtk_object_set_data_full(			\
    GTK_OBJECT(w), EDV_VFS_BROWSER_COMMAND_DATA_KEY,	\
    cmd, (GtkDestroyNotify)edv_vfs_browser_command_delete	\
   );						\
  }						\
						\
  (void)GDK_PIXMAP_UNREF(pixmap);		\
  (void)GDK_BITMAP_UNREF(mask);			\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_ITEM_SUBMENU_LABEL	{	\
 EDVVFSBrowserOp *op =				\
  edv_vfs_browser_op_match_by_id(browser, id);		\
 if(op != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu,					\
   GUI_MENU_ITEM_TYPE_LABEL,			\
   NULL,					\
   (icon_data != NULL) ?			\
    icon_data : op->menu_icon_data,		\
   op->menu_name,				\
   0, 0,					\
   NULL, NULL					\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)edv_vfs_browser_menu_item_enter_cb, op,	\
   (gpointer)edv_vfs_browser_menu_item_leave_cb, op	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_ITEM_SEPARATOR	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL,						\
  NULL,						\
  0, 0,						\
  NULL, NULL					\
 );						\
}
	id = EDV_VFS_BROWSER_OP_OPEN;
	ADD_MENU_ITEM_LABEL
	browser->open_mi = w;

	id = EDV_VFS_BROWSER_OP_OPEN_TO;
	ADD_MENU_ITEM_SUBMENU_LABEL
	browser->open_to_mi = w;
	browser->open_to_menu = NULL;

	id = EDV_VFS_BROWSER_OP_OPEN_WITH;
	ADD_MENU_ITEM_LABEL
	browser->open_with_mi = w;

	ADD_MENU_ITEM_SEPARATOR

	/* Customized Object Ops menu items */
	cfg_menu = EDV_GET_MENU(EDV_CFG_PARM_OBJECT_OPS_MENU);
	if((cfg_menu != NULL) ? (cfg_menu->list != NULL) : FALSE)
	{
		GList *glist;
		CfgMenuItem *item;

		for(glist = cfg_menu->list;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			item = CFG_MENU_ITEM(glist->data);
			ADD_MENU_ITEM_CMD_OBJ_OP(item);
		}

		ADD_MENU_ITEM_SEPARATOR
	}

	id = EDV_VFS_BROWSER_OP_NEW;
	ADD_MENU_ITEM_SUBMENU_LABEL
	browser->new_object_mi = w;
	browser->new_object_menu = submenu = GUIMenuCreate();
	GUIMenuItemSetSubMenu(w, submenu);

	ADD_MENU_ITEM_SEPARATOR

	id = EDV_VFS_BROWSER_OP_SYNC_DISKS;
	ADD_MENU_ITEM_LABEL
	browser->sync_disks_mi = w;

	ADD_MENU_ITEM_SEPARATOR

	id = EDV_VFS_BROWSER_OP_HISTORY;
	ADD_MENU_ITEM_LABEL
	browser->history_mi = w;

	ADD_MENU_ITEM_SEPARATOR

	id = EDV_VFS_BROWSER_OP_RUN;
	ADD_MENU_ITEM_LABEL
	browser->run_mi = w;

	ADD_MENU_ITEM_SEPARATOR

	id = EDV_VFS_BROWSER_OP_CLOSE;
	ADD_MENU_ITEM_LABEL
	browser->close_mi = w;

#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_CMD_OBJ_OP
#undef ADD_MENU_ITEM_SUBMENU_LABEL
#undef ADD_MENU_ITEM_SEPARATOR
}

/*
 *	Recreates the File Browser's New Object Submenu.
 */
void edv_vfs_browser_new_object_menu_items_regenerate(EDVVFSBrowser *browser)
{
	edv_pixmap_data *icon_data = NULL;
	GList *glist;
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget *w, *menu;
	GtkMenuShell *menushell;
	CfgList *cfg_list;
	CfgMenu *cfg_menu;
	EDVVFSBrowserOpID id;
	EDVCore *core;

	if(browser == NULL)
		return;

	core = browser->core;
	cfg_list = core->cfg_list;

	/* Get the New Object submenu */
	menu = browser->new_object_menu;
	if(menu == NULL)
		return;

	menushell = GTK_MENU_SHELL(menu);

	/* Delete the existing menu items */
	glist = g_list_copy(menushell->children);
	if(glist != NULL)
	{
		g_list_foreach(glist, (GFunc)gtk_widget_destroy, NULL);
		g_list_free(glist);
	}

#define ADD_MENU_ITEM_LABEL	{		\
 EDVVFSBrowserOp *op =				\
  edv_vfs_browser_op_match_by_id(browser, id);		\
 if(op != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu,					\
   GUI_MENU_ITEM_TYPE_LABEL,			\
   accelgrp,					\
   (icon_data != NULL) ?			\
    icon_data : op->menu_icon_data,		\
   op->menu_name,				\
   op->accel_key, op->accel_mods,		\
   edv_vfs_browser_menu_item_cb, op			\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)edv_vfs_browser_menu_item_enter_cb, op,	\
   (gpointer)edv_vfs_browser_menu_item_leave_cb, op	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_ITEM_CMD_NEW_OBJ(_item_) {	\
 if((_item_) != NULL) {				\
  EDVVFSBrowserCommand *cmd;			\
  const gchar *icon_file = (_item_)->icon_file;	\
  GdkBitmap *mask;				\
  GdkPixmap *pixmap;				\
  if(icon_file != NULL) {  			\
   pixmap = GDK_PIXMAP_NEW_FROM_XPM_FILE(	\
    &mask, icon_file				\
   );						\
  } else {					\
   pixmap = NULL;				\
   mask = NULL;					\
  }						\
						\
  cmd = edv_vfs_browser_command_new();			\
  if(cmd != NULL) {				\
   cmd->vfs_browser = browser;			\
   cmd->label = STRDUP((_item_)->label);	\
   cmd->command = STRDUP((_item_)->command);	\
   cmd->description = STRDUP((_item_)->description);\
   cmd->ext_data = STRDUP((_item_)->ext_data);	\
   cmd->flags = (_item_)->flags;		\
						\
   if((_item_)->flags & CFG_MENU_ITEM_IS_SEPARATOR) { \
    w = GUIMenuItemCreate(			\
     menu,					\
     GUI_MENU_ITEM_TYPE_SEPARATOR,		\
     NULL,					\
     NULL,					\
     NULL,					\
     0, 0,					\
     NULL, NULL					\
    );						\
   } else {					\
    w = GUIMenuItemCreate(			\
     menu,					\
     GUI_MENU_ITEM_TYPE_LABEL,			\
     accelgrp,					\
     NULL,					\
     (_item_)->label,				\
     0, 0,					\
     edv_vfs_browser_menu_item_cmd_new_object_cb, cmd	\
    );						\
    GUISetMenuItemCrossingCB(			\
     w,						\
     (gpointer)edv_vfs_browser_menu_item_cmd_enter_cb, cmd,\
     (gpointer)edv_vfs_browser_menu_item_cmd_leave_cb, cmd \
    );						\
    if(pixmap != NULL)				\
     GUIMenuItemSetPixmap2(			\
      w,					\
      pixmap, mask				\
     );						\
   }						\
						\
   gtk_object_set_data_full(			\
    GTK_OBJECT(w), EDV_VFS_BROWSER_COMMAND_DATA_KEY,	\
    cmd, (GtkDestroyNotify)edv_vfs_browser_command_delete	\
   );						\
  }						\
						\
  (void)GDK_PIXMAP_UNREF(pixmap);		\
  (void)GDK_BITMAP_UNREF(mask);			\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_ITEM_SEPARATOR	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL,						\
  NULL,						\
  0, 0,						\
  NULL, NULL					\
 );						\
}

	id = EDV_VFS_BROWSER_OP_NEW_FILE;
	ADD_MENU_ITEM_LABEL
	browser->new_file_mi = w;

	id = EDV_VFS_BROWSER_OP_NEW_DIRECTORY;
	ADD_MENU_ITEM_LABEL
	browser->new_directory_mi = w;

	id = EDV_VFS_BROWSER_OP_NEW_LINK;
	ADD_MENU_ITEM_LABEL
	browser->new_link_mi = w;

	/* Customized New Object menu items */
	cfg_menu = EDV_GET_MENU(EDV_CFG_PARM_NEW_OBJECTS_MENU);
	if((cfg_menu != NULL) ? (cfg_menu->list != NULL) : FALSE)
	{
		GList *glist;
		CfgMenuItem *item;

		ADD_MENU_ITEM_SEPARATOR

		for(glist = cfg_menu->list;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			item = CFG_MENU_ITEM(glist->data);
			ADD_MENU_ITEM_CMD_NEW_OBJ(item);
		}

		ADD_MENU_ITEM_SEPARATOR
	}

	id = EDV_VFS_BROWSER_OP_NEW_FIFO;
	ADD_MENU_ITEM_LABEL
	browser->new_fifo_mi = w;

	id = EDV_VFS_BROWSER_OP_NEW_DEVICE_BLOCK;
	ADD_MENU_ITEM_LABEL
	browser->new_device_block_mi = w;

	id = EDV_VFS_BROWSER_OP_NEW_DEVICE_CHARACTER;
	ADD_MENU_ITEM_LABEL
	browser->new_device_character_mi = w;

	id = EDV_VFS_BROWSER_OP_NEW_SOCKET;
	ADD_MENU_ITEM_LABEL
	browser->new_socket_mi = w;

#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_CMD_NEW_OBJ
#undef ADD_MENU_ITEM_SEPARATOR         
}

/* 
 *	Regenerates the Open To menu.
 */
void edv_vfs_browser_open_to_menu_regenerate(
	EDVVFSBrowser *browser,
	EDVMIMEType *m
)
{
	GtkWidget	*submenu,
			*toplevel;
	EDVCore *core;

	if(browser == NULL)
		return;

	toplevel = browser->toplevel;
	core = browser->core;

	submenu = edv_open_to_menu_new_from_mime_type(
		core, m,
		edv_vfs_browser_get_selected_paths_cb, browser,
		edv_vfs_browser_goto_directory_cb, browser,
		toplevel,
		TRUE				/* Verbose */
	);
	GUIMenuItemSetSubMenu(browser->open_to_mi, submenu);
	browser->open_to_menu = submenu;

	submenu = edv_open_to_menu_new_from_mime_type(
		core, m,
		edv_vfs_browser_get_selected_paths_cb, browser,
		edv_vfs_browser_goto_directory_cb, browser,
		toplevel,
		TRUE				/* Verbose */
	);
	GUIMenuItemSetSubMenu(browser->contents_clist_open_to_mi, submenu);
	browser->contents_clist_open_to_menu = submenu;
}

/*
 *	Recreates the File Browser Directory Tree's right-click menu.
 */
void edv_vfs_browser_tree_menu_regenerate(EDVVFSBrowser *browser)
{
	edv_pixmap_data *icon_data = NULL;
	EDVVFSBrowserOpID id;
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget	*w,
			*menu;

	if(browser == NULL)
		return;

	/* Create/recreate the menu */
	GTK_WIDGET_DESTROY(browser->directory_ctree_menu);
	browser->directory_ctree_menu = menu = GUIMenuCreate();

#define ADD_MENU_ITEM_LABEL		{	\
 EDVVFSBrowserOp *op =				\
  edv_vfs_browser_op_match_by_id(browser, id);		\
 if(op != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu,					\
   GUI_MENU_ITEM_TYPE_LABEL,			\
   accelgrp,					\
   (icon_data != NULL) ?			\
    icon_data : op->menu_icon_data,		\
   op->menu_name,				\
   op->accel_key, op->accel_mods,		\
   edv_vfs_browser_menu_item_cb, op			\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)edv_vfs_browser_menu_item_enter_cb, op,	\
   (gpointer)edv_vfs_browser_menu_item_leave_cb, op	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_ITEM_SEPARATOR	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL,						\
  NULL,						\
  0, 0,						\
  NULL, NULL					\
 );						\
}
	id = EDV_VFS_BROWSER_OP_EXPAND;
	ADD_MENU_ITEM_LABEL
	browser->directory_ctree_expand_mi = w;

	ADD_MENU_ITEM_SEPARATOR

	id = EDV_VFS_BROWSER_OP_MOUNT;
	ADD_MENU_ITEM_LABEL
	browser->directory_ctree_mount_mi = w;

	id = EDV_VFS_BROWSER_OP_EJECT;
	ADD_MENU_ITEM_LABEL
	browser->directory_ctree_eject_mi = w;

	ADD_MENU_ITEM_SEPARATOR
	browser->directory_ctree_mount_misep = w;

	id = EDV_VFS_BROWSER_OP_MOVE;
	ADD_MENU_ITEM_LABEL
	browser->directory_ctree_move_mi = w;

	id = EDV_VFS_BROWSER_OP_COPY;
	ADD_MENU_ITEM_LABEL
	browser->directory_ctree_copy_mi = w;

	id = EDV_VFS_BROWSER_OP_LINK;
	ADD_MENU_ITEM_LABEL
	browser->directory_ctree_link_mi = w;

	id = EDV_VFS_BROWSER_OP_RENAME;
	ADD_MENU_ITEM_LABEL
	browser->directory_ctree_rename_mi = w;

	id = EDV_VFS_BROWSER_OP_CHMOD;
	ADD_MENU_ITEM_LABEL
	browser->directory_ctree_chmod_mi = w;

	id = EDV_VFS_BROWSER_OP_CHOWN;
	ADD_MENU_ITEM_LABEL
	browser->directory_ctree_chown_mi = w;

	id = EDV_VFS_BROWSER_OP_CHTIME;
	ADD_MENU_ITEM_LABEL
	browser->directory_ctree_chtime_mi = w;

	id = EDV_VFS_BROWSER_OP_DELETE;
	ADD_MENU_ITEM_LABEL
	browser->directory_ctree_delete_mi = w;

	ADD_MENU_ITEM_SEPARATOR

	id = EDV_VFS_BROWSER_OP_COPY_PATH;
	ADD_MENU_ITEM_LABEL
	browser->directory_ctree_copy_path_mi = w;

	id = EDV_VFS_BROWSER_OP_COPY_URL;
	ADD_MENU_ITEM_LABEL
	browser->directory_ctree_copy_url_mi = w;

	id = EDV_VFS_BROWSER_OP_PASTE;
	ADD_MENU_ITEM_LABEL
	browser->directory_ctree_paste_mi = w;

	ADD_MENU_ITEM_SEPARATOR

	id = EDV_VFS_BROWSER_OP_PROPERTIES;
	ADD_MENU_ITEM_LABEL
	browser->directory_ctree_properties_mi = w;

#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_SEPARATOR
}

/*
 *	Recreates the File Browser Contents List's right-click menu.
 */
void edv_vfs_browser_list_menu_regenerate(EDVVFSBrowser *browser)
{
	edv_pixmap_data *icon_data = NULL;
	EDVVFSBrowserOpID id;
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget	*w,
			*menu;

	if(browser == NULL)
		return;

	/* Create/recreate the menu */
	GTK_WIDGET_DESTROY(browser->contents_clist_menu);
	browser->contents_clist_menu = menu = GUIMenuCreate();

#define ADD_MENU_ITEM_LABEL	{		\
 EDVVFSBrowserOp *op =				\
  edv_vfs_browser_op_match_by_id(browser, id);		\
 if(op != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu,					\
   GUI_MENU_ITEM_TYPE_LABEL,			\
   accelgrp,					\
   (icon_data != NULL) ?			\
    icon_data : op->menu_icon_data,		\
   op->menu_name,				\
   op->accel_key, op->accel_mods,		\
   edv_vfs_browser_menu_item_cb, op			\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)edv_vfs_browser_menu_item_enter_cb, op,	\
   (gpointer)edv_vfs_browser_menu_item_leave_cb, op	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_ITEM_SUBMENU_LABEL	{	\
 EDVVFSBrowserOp *op =				\
  edv_vfs_browser_op_match_by_id(browser, id);		\
 if(op != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu,					\
   GUI_MENU_ITEM_TYPE_LABEL,			\
   NULL,					\
   (icon_data != NULL) ?			\
    icon_data : op->menu_icon_data,		\
   op->menu_name,				\
   0, 0,					\
   NULL, NULL					\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)edv_vfs_browser_menu_item_enter_cb, op,	\
   (gpointer)edv_vfs_browser_menu_item_leave_cb, op	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_ITEM_SEPARATOR	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL,						\
  NULL,						\
  0, 0,						\
  NULL, NULL					\
 );						\
}

	id = EDV_VFS_BROWSER_OP_OPEN;
	ADD_MENU_ITEM_LABEL
	GUISetMenuItemSetDefault(w, TRUE);
	browser->contents_clist_open_mi = w;

	id = EDV_VFS_BROWSER_OP_OPEN_TO;
	ADD_MENU_ITEM_SUBMENU_LABEL
	browser->contents_clist_open_to_mi = w;

	id = EDV_VFS_BROWSER_OP_OPEN_WITH;
	ADD_MENU_ITEM_LABEL
	browser->contents_clist_open_with_mi = w;

	ADD_MENU_ITEM_SEPARATOR

	id = EDV_VFS_BROWSER_OP_MOVE;
	ADD_MENU_ITEM_LABEL
	browser->contents_clist_move_mi = w;

	id = EDV_VFS_BROWSER_OP_COPY;
	ADD_MENU_ITEM_LABEL
	browser->contents_clist_copy_mi = w;

	id = EDV_VFS_BROWSER_OP_LINK;
	ADD_MENU_ITEM_LABEL
	browser->contents_clist_link_mi = w;

	id = EDV_VFS_BROWSER_OP_RENAME;
	ADD_MENU_ITEM_LABEL
	browser->contents_clist_rename_mi = w;

	id = EDV_VFS_BROWSER_OP_CHMOD;
	ADD_MENU_ITEM_LABEL
	browser->contents_clist_chmod_mi = w;

	id = EDV_VFS_BROWSER_OP_CHOWN;
	ADD_MENU_ITEM_LABEL
	browser->contents_clist_chown_mi = w;

	id = EDV_VFS_BROWSER_OP_CHTIME;
	ADD_MENU_ITEM_LABEL
	browser->contents_clist_chtime_mi = w;

	id = EDV_VFS_BROWSER_OP_DELETE;
	ADD_MENU_ITEM_LABEL
	browser->contents_clist_delete_mi = w;

	ADD_MENU_ITEM_SEPARATOR

	id = EDV_VFS_BROWSER_OP_COPY_PATH;
	ADD_MENU_ITEM_LABEL
	browser->contents_clist_copy_path_mi = w;

	id = EDV_VFS_BROWSER_OP_COPY_URL;
	ADD_MENU_ITEM_LABEL
	browser->contents_clist_copy_url_mi = w;

	id = EDV_VFS_BROWSER_OP_PASTE;
	ADD_MENU_ITEM_LABEL
	browser->contents_clist_paste_mi = w;

	ADD_MENU_ITEM_SEPARATOR

	id = EDV_VFS_BROWSER_OP_PROPERTIES;
	ADD_MENU_ITEM_LABEL
	browser->contents_clist_properties_mi = w;

#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_SUBMENU_LABEL
#undef ADD_MENU_ITEM_SEPARATOR
}


/*
 *	Sets the File Browser's values to the configuration.
 */
void edv_vfs_browser_sync_configuration(EDVVFSBrowser *browser)
{
	GtkWidget *w;
	CfgList *cfg_list;
	EDVCore *core;

	if(browser == NULL)
		return;

	core = browser->core;
	cfg_list = core->cfg_list;

	/* Location GtkCombo history */
	w = browser->location_combo;
	if(w != NULL)
		edv_save_text_file_glist(
			EDV_GET_S(EDV_CFG_PARM_FILE_BROWSER_LOCATION_HISTORY),
			GUIComboGetList(w)
		);

	/* Toplevel GtkWindow position and size */
	CfgGtkGetGtkWindow(
		cfg_list,
		EDV_CFG_PARM_BROWSER_X,
		EDV_CFG_PARM_BROWSER_Y,
		EDV_CFG_PARM_BROWSER_WIDTH,
		EDV_CFG_PARM_BROWSER_HEIGHT,
		browser->toplevel
	);

	/* Main GtkPaned position */
	CfgGtkGetGtkPaned(
		cfg_list,
		EDV_CFG_PARM_BROWSER_PANED_POSITION,
		browser->main_paned
	);

	/* EDV_CFG_PARM_BROWSER_CONTENTS_COLUMN_WIDTH parameter is updated
	 * as they change in edv_vfs_browser_resize_column_cb() or when the
	 * configuration list is changed directly
	 */

	/* EDV_CFG_PARM_BROWSER_CONTENTS_COLUMN parameter is updated
	 * when the configuration list is changed directly
	 */

	/* Set the new startup directory as the last directory of
	 * this window's location
	 */
	if(EDV_GET_B(EDV_CFG_PARM_RECORD_START_UP_DIR))
		EDV_SET_S(
			EDV_CFG_PARM_DIR_START_UP,
			edv_vfs_browser_get_location(browser)
		);

	/* Directory tree origin path */
	if(!STRISEMPTY(browser->directory_ctree_origin_path))
		EDV_SET_S(
			EDV_CFG_PARM_BROWSER_DIR_TREE_ORIGIN,
			browser->directory_ctree_origin_path
		);

	/* Contents list filter */
	if(STRISEMPTY(browser->contents_list_filter))
		EDV_SET_S(
			EDV_CFG_PARM_BROWSER_CONTENTS_LIST_FILTER,
			"*"
		);
	else
		EDV_SET_S(
			EDV_CFG_PARM_BROWSER_CONTENTS_LIST_FILTER,
			browser->contents_list_filter
		);
}


/*
 *	Creates a new EDVVFSBrowserOp.
 */
static EDVVFSBrowserOp *edv_vfs_browser_op_new(void)
{
	return(EDV_VFS_BROWSER_OP(g_malloc0(sizeof(EDVVFSBrowserOp))));
}

/*
 *	Deletes the EDVVFSBrowserOp.
 */
static void edv_vfs_browser_op_delete(EDVVFSBrowserOp *op)
{
	if(op == NULL)
		return;

	g_free(op->button_name);
	g_free(op->menu_name);
	g_free(op->tooltip);
	g_free(op);
}

/*
 *	Creates a new File Browser Operaton IDs list.
 *
 *	The specified File Browser may be NULL.
 */
GList *edv_vfs_browser_ops_list_new(EDVVFSBrowser *browser)
{
	guint		accel_key,
			accel_mods;
	const gchar	*button_name,
			*menu_name,
			*tooltip;
	guint8		**button_icon_data,
			**menu_icon_data;
	GList *op_ids_list = NULL;
	ToolBarItemType tb_item_type;
	EDVOpIDFlags flags;
	EDVVFSBrowserOpID id;
	void (*func_cb)(ToolBarItem *, gint, gpointer);
	void (*enter_func_cb)(ToolBarItem *, gint, gpointer);

#define APPEND_OPID	{			\
 EDVVFSBrowserOp *op = edv_vfs_browser_op_new();	\
 if(op != NULL) {				\
  op->id = id;					\
  op->flags = flags;				\
  op->tb_item_type = tb_item_type;		\
  op->func_cb = func_cb;			\
  op->enter_func_cb = enter_func_cb;		\
  op->button_name = STRDUP(button_name);	\
  op->menu_name = STRDUP(menu_name);		\
  op->tooltip = STRDUP(tooltip);		\
  op->accel_key = accel_key;			\
  op->accel_mods = accel_mods;			\
  op->button_icon_data = button_icon_data;	\
  op->menu_icon_data = menu_icon_data;		\
  op->vfs_browser = browser;			\
  op_ids_list = g_list_append(			\
   op_ids_list,					\
   op						\
  );						\
 }						\
}

	func_cb = edv_vfs_browser_op_cb;
	enter_func_cb = edv_vfs_browser_op_enter_cb;


	id = EDV_VFS_BROWSER_OP_SEPARATOR;
	flags = EDV_OPID_ALLOW_MULTIPLE | EDV_OPID_NO_OP;
	tb_item_type = TOOL_BAR_ITEM_SEPARATOR;
	button_icon_data = (edv_pixmap_data *)icon_hsep_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Separador";
	menu_name = "Separador";
	tooltip = "Separador";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Sparateur";
	menu_name = "Sparateur";
	tooltip = "Sparateur";
#elif defined(PROG_LANGUAGE_GERMAN)
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#elif defined(PROG_LANGUAGE_ITALIAN)
	button_name = "Separatore";
	menu_name = "Separatore";
	tooltip = "Separatore";
#elif defined(PROG_LANGUAGE_DUTCH)
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	button_name = "Separador";
	menu_name = "Separador";
	tooltip = "Separador";
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#else
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID


	id = EDV_VFS_BROWSER_OP_CLOSE;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_close_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Cierre";
	menu_name = "Cierre";
	tooltip = "Cierre esta ventana";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Proche";
	menu_name = "Proche";
	tooltip = "Fermer cette fentre";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Zamknij";
	menu_name = "Zamknij";
	tooltip = "Zamknij to okno";
#else
	button_name = "Close";
	menu_name = "Close";
	tooltip = "Close this window";
#endif
	accel_key = GDK_w;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_EXIT;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_exit_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "La Salida";
	menu_name = "La Salida";
	tooltip = "Cierra todas ventanas en la aplicacin y la";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Sortie";
	menu_name = "Sortie";
	tooltip = "Ferme toutes fentres dans l'application et sortie";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Wyjcie";
	menu_name = "Wyjcie";
	tooltip = "Zamknij wszystkie okna aplikacji i wyjd";
#else
	button_name = "Exit";
	menu_name = "Exit";
	tooltip = "Close all windows in the application and exit";
#endif
	accel_key = GDK_q;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID


	id = EDV_VFS_BROWSER_OP_SYNC_DISKS;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_save_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Sincronizar Discos";
	menu_name = "Sincronizar Discos";
	tooltip = "Limpan cualquiera transferencias pendientes de datos al disco";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Disq Syn";
	menu_name = "Les Disques De Syn";
	tooltip = "Rincer des transferts de donnes en attente au disque";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Synchr.dyski";
	menu_name = "Synchronizuj dyski";
	tooltip = "Wykonaj wszystkie oczekujce transfery na dysk";
#else
	button_name = "Sync Disks";
	menu_name = "Sync Disks";
	tooltip = "Flush any pending data transfers to disk";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_HISTORY;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_trace_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Historia";
	menu_name = "La Historia...";
	tooltip = "La Historia de rdenes procesadas por esta aplicacin";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "L'Histoire";
	menu_name = "L'Histoire...";
	tooltip = "L'Histoire d'ordres traits par cette application";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Historia";
	menu_name = "Historia...";
	tooltip = "Historia komend wykonanych przez t aplikacj";
#else
	button_name = "History";
	menu_name = "History...";
	tooltip = "History of commands processed by this application";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_RUN;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_run_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Corra";
	menu_name = "Corra...";
	tooltip = "Corra un programa";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Courir";
	menu_name = "Courir...";
	tooltip = "Excute un programme";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Uruchom";
	menu_name = "Uruchom...";
	tooltip = "Uruchom program";
#else
	button_name = "Run";
	menu_name = "Run...";
	tooltip = "Run a program";
#endif
	accel_key = GDK_r;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_RUN_TERMINAL;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_terminal2_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "La Terminal";
	menu_name = "La Terminal";
	tooltip = "Corre una terminal";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Le Terminal";
	menu_name = "Le Terminal";
	tooltip = "Courir un terminal";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Terminal";
	menu_name = "Terminal";
	tooltip = "Uruchom terminal";
#else
	button_name = "Terminal";
	menu_name = "Terminal";
	tooltip = "Run a terminal";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_WRITE_PROTECT;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_write_protect_on_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Proteja";
	menu_name = "Proteja contra escritura";
	tooltip = "Habilite/inhabilita la proteccion contra escritura";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Protge";
	menu_name = "Protge En criture";
	tooltip = "Rend capable/rend infirme protge en criture";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Zabezpieczenie";
	menu_name = "Zabezpieczenie przed zapisem";
	tooltip = "Wcz/wycz zabezpieczenie przed zapisem";
#else
	button_name = "Protect";
	menu_name = "Master Write Protect";
	tooltip = "Turn the master write protect on/off";
#endif
	accel_key = GDK_y;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_DELETE_METHOD_RECYCLE;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = edv_get_recycle_bin_icon_data(
		EDV_ICON_SIZE_20,
		1,
		NULL
	);
	menu_icon_data = button_icon_data;
	button_name = "Recycle";
	menu_name = "Delete Method: Recycle";
	tooltip = "Set the delete method to recycle";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_DELETE_METHOD_PURGE;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_purge_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Purge";
	menu_name = "Delete Method: Purge";
	tooltip = "Set the delete method to purge";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID


	id = EDV_VFS_BROWSER_OP_NEW;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_new_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Nuevo";
	menu_name = "Nuevo";
	tooltip = "Crea un objeto nuevo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Nouveau";
	menu_name = "Nouveau";
	tooltip = "Crer un nouvel objet";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Nowy";
	menu_name = "Nowy";
	tooltip = "Stwrz nowy obiekt";
#else
	button_name = "New";
	menu_name = "New";
	tooltip = "Create a new object";
#endif
	accel_key = GDK_n;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_NEW_FILE;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_file_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Archivo";
	menu_name = "El Archivo";
	tooltip = "Crea un objeto nuevo del archivo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Fichier";
	menu_name = "Fichier";
	tooltip = "Crer un nouvel objet de fichier";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Nowy plik";
	menu_name = "Plik";
	tooltip = "Stwrz nowy plik";
#else
	button_name = "New File";
	menu_name = "File";
	tooltip = "Create a new file";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_NEW_DIRECTORY;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_folder_closed_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Carpeta";
	menu_name = "Carpeta";
	tooltip = "Crea una nueva carpeta";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Annuaire";
	menu_name = "Annuaire";
	tooltip = "Crer un nouvel objet d'annuaire";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Nowy katalog";
	menu_name = "Katalog";
	tooltip = "Stwrz nowy katalog";
#else
	button_name = "New Dir";
	menu_name = "Directory";
	tooltip = "Create a new directory";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_NEW_LINK;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_link_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Enlace";
	menu_name = "Enlace";
	tooltip = "Crea un nuevo enlace simblico";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Lien";
	menu_name = "Lien";
	tooltip = "Cre un nouveau symbolique lien";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Nowy odnonik";
	menu_name = "Odnonik";
	tooltip = "Stwrz nowy odnonik symboliczny";
#else
	button_name = "New Lnk";
	menu_name = "Link";
	tooltip = "Create a new symbolic link";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_NEW_FIFO;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_pipe_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "FIFO";
	menu_name = "FIFO Tubo";
	tooltip = "Crea un objeto nuevo de tubo de FIFO";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "FIFO";
	menu_name = "FIFO Tuyau";
	tooltip = "Crer un nouvel objet de tuyau de FIFO";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Nowe FIFO";
	menu_name = "Rura FIFO";
	tooltip = "Stwrz now rur FIFO";
#else
	button_name = "New FIFO";
	menu_name = "FIFO Pipe";
	tooltip = "Create a new FIFO pipe";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_NEW_DEVICE_BLOCK;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_device_block_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "BDispositivo";
	menu_name = "El Dispositivo Del Bloque";
	tooltip = "Crea un objeto nuevo de nodo de Dispositivo de bloque (buffered)";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "BL'Appareil";
	menu_name = "Le Priphrique  Blocs";
	tooltip = "Crer un nouveau bloc (buffered) l'objet de noeu";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Nowe urz.blok.";
	menu_name = "Urzdzenie blokowe";
	tooltip = "Stwrz nowy wze urzdzenia blokowego (buforowanego)";
#else
	button_name = "New BDev";
	menu_name = "Block Device";
	tooltip = "Create a new block (buffered) device node";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_NEW_DEVICE_CHARACTER;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_device_character_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "CDispositivo";
	menu_name = "El Dispositivo Del Carcter";
	tooltip = "Crea un objeto nuevo de nodo de Dispositivo de carcter (unbuffered)";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "CL'Appareil";
	menu_name = "Le Priphrique  Caractres";
	tooltip = "Crer un nouveau caractre (unbuffered) l'objet de noeud d'appareil";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Nowe urz.znak.";
	menu_name = "Urzdzenie znakowe";
	tooltip = "Stwrz nowy wze urzdzenia znakowego (niebuforowanego)";
#else
	button_name = "New CDev";
	menu_name = "Character Device";
	tooltip = "Create a new character (unbuffered) device node";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_NEW_SOCKET;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_socket_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Enchufe";
	menu_name = "El Enchufe";
	tooltip = "Crea un objeto nuevo del enchufe";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Douille";
	menu_name = "La Douille";
	tooltip = "Crer un nouvel objet de douille";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Nowe gniazdo";
	menu_name = "Gniazdo";
	tooltip = "Stwrz nowe gniazdo";
#else
	button_name = "New Sock";
	menu_name = "Socket";
	tooltip = "Create a new socket";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID


	id = EDV_VFS_BROWSER_OP_OPEN;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_open_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Abrir";
	menu_name = "Abrir";
	tooltip = "Abre archivo escogido";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Ouvert";
	menu_name = "Ouvert";
	tooltip = "Ouvrir l'objet choisi";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Otwrz";
	menu_name = "Otwrz";
	tooltip = "Otwrz wybrany obiekt";
#else
	button_name = "Open";
	menu_name = "Open";
	tooltip = "Open the selected object(s)";
#endif
	accel_key = GDK_Return;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_OPEN_TO;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_open_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "Open To";
	menu_name = "Open To...";
	tooltip = "Open the selected object(s) using a specific method";
	accel_key = GDK_Return;
	accel_mods = GDK_CONTROL_MASK | GDK_SHIFT_MASK;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_OPEN_WITH;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_open_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Abrir Con";
	menu_name = "Abrir Con...";
	tooltip = "Abrir archivo escogido usando un mtodo especfico";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Ouvrir Avec";
	menu_name = "Ouvrir Avec...";
	tooltip = "Ouvrir l'objet choisi pour utiliser une mthode spcifique";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Otwrz z...";
	menu_name = "Otwrz z...";
	tooltip = "Otwrz wybrany obiekt uywajc sbosobu wybranego przez uytkownika";
#else
	button_name = "Open With";
	menu_name = "Open With...";
	tooltip = "Open the selected object(s) using a specific application";
#endif
	accel_key = GDK_Return;
	accel_mods = GDK_SHIFT_MASK;
	APPEND_OPID


	id = EDV_VFS_BROWSER_OP_COPY_PATH;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_copy_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "CopiSende";
	menu_name = "Copie Sendero";
	tooltip = "Copia la ubicacin del objeto(s) escogido como sendero(s)";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "CopiSenti";
	menu_name = "Copier Le Sentier";
	tooltip = "Copier l'emplacement de l'objet choisi(s) comme le sentier(s)";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Kopiuj ciek";
	menu_name = "Kopiuj ciek";
	tooltip = "Kopiuj lokacj zaznaczonego (zaznaczonych) obiektu (obiektw) jako ciek (cieki)";
#else
	button_name = "CopyPath";
	menu_name = "Copy Path";
	tooltip = "Copy the location of the selected object(s) as path(s)";
#endif
	accel_key = GDK_c;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_COPY_URL;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_copy_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "CopiURL";
	menu_name = "Copie URL";
	tooltip = "Copie la ubicacin del objeto(s) escogido como url(s)";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "CopiURL";
	menu_name = "Copier URL";
	tooltip = "Copier l'emplacement de l'objet choisi(s) comme url(s)";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Kopiuj URL";
	menu_name = "Kopiuj URL";
	tooltip = "Kopiuj lokacj zaznaczonego (zaznaczonych) obiektu (obiektw) jako URL";
#else
	button_name = "CopyURL";
	menu_name = "Copy URL";
	tooltip = "Copy the location of the selected object(s) as url(s)";
#endif
	accel_key = GDK_c;
	accel_mods = GDK_CONTROL_MASK | GDK_SHIFT_MASK;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_PASTE;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_paste_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_FRENCH)
	button_name = "Coller";
	menu_name = "Coller";
	tooltip = "Coller object(s) depuis l'objet source(s)";
#else
	button_name = "Paste";
	menu_name = "Paste";
	tooltip = "Paste object(s) from coppied location source(s)";
#endif
	accel_key = GDK_v;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID


	id = EDV_VFS_BROWSER_OP_MOVE;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_move_file_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Mueva";
	menu_name = "Mueva...";
	tooltip = "Mueva los objeto(s) escogido(s)";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Mouvement";
	menu_name = "Le mouvement...";
	tooltip = "Le Mouvement a choisi l'objet(s)";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Przenie";
	menu_name = "Przenie...";
	tooltip = "Przenie zaznaczony(e) obiekt(y)";
#else
	button_name = "Move";
	menu_name = "Move...";
	tooltip = "Move the selected object(s)";
#endif
	accel_key = GDK_F7;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_COPY;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_copy_file_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Copia";
	menu_name = "Copia...";
	tooltip = "Copia los objeto(s) escogido(s)";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Copia";
	menu_name = "La Copie...";
	tooltip = "La Copie a choisi l'objet(s)";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Kopiuj";
	menu_name = "Kopiuj...";
	tooltip = "Kopiuj zaznaczony(e) obiekt(y)";
#else
	button_name = "Copy";
	menu_name = "Copy...";
	tooltip = "Copy the selected object(s)";
#endif
	accel_key = GDK_F8;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_LINK;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_link_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Enlace";
	menu_name = "Enlace...";
	tooltip = "Enlace los objeto(s) escogido(s)";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Le Lien";
	menu_name = "Le Lien...";
	tooltip = "Le Lien a choisi l'objet(s)";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Odnonik";
	menu_name = "Odnonik...";
	tooltip = "Utwrz odnonik do zaznaczonego (zaznaczonych) obiektu (obiektw)";
#else
	button_name = "Link";
	menu_name = "Link...";
	tooltip = "Link the selected object(s)";
#endif
	accel_key = GDK_F6;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_RENAME;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_rename_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Renombrar";
	menu_name = "Renombrar";
	tooltip = "Renombrar archivo escogido";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Renommer";
	menu_name = "Renommer";
	tooltip = "Renomme l'objet choisi";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Zmie nazw";
	menu_name = "Zmie nazw";
	tooltip = "Zmie nazw zaznaczonego (zaznaczonych) obiektu (obiektw)";
#else
	button_name = "Rename";
	menu_name = "Rename";
	tooltip = "Rename the selected object";
#endif
	accel_key = GDK_F2;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_CHMOD;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_chmod_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "ChMod";
	menu_name = "Cambie permisos...";
	tooltip = "El Cambio escogi los permisos de objetos";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "ChMod";
	menu_name = "Changer Le Mode...";
	tooltip = "Le Changement a choisi des permissions des objets";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "ChMod";
	menu_name = "Zmiana uprawnie...";
	tooltip = "Zmie uprawnienia dostpu do zaznaczonego (zaznaczonych) obiektu (obiektw)";
#else
	button_name = "ChMod";
	menu_name = "Change Permissions...";
	tooltip = "Change the permissions of the selected object(s)";
#endif
	accel_key = GDK_F9;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_CHOWN;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_owned_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "ChOwn";
	menu_name = "Change Ownership...";
	tooltip = "Chown escogido(s)";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "ChOwn";
	menu_name = "Change Ownership...";
	tooltip = "Chown a choisi l'objet(s)";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "ChOwn";
	menu_name = "Zmiana waciciela...";
	tooltip = "Zmie wacicela zaznaczonego (zaznaczonych) obiektu (obiektw)";
#else
	button_name = "ChOwn";
	menu_name = "Change Ownership...";
	tooltip = "Change the ownership of the selected object(s)";
#endif
	accel_key = GDK_F10;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_CHTIME;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_time_stamp_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "ChTime";
	menu_name = "Change Time Stamps...";
	tooltip = "Change the time stamps of the selected object(s)";
	accel_key = GDK_F11;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_DELETE;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_cancel_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Borre";
	menu_name = "Borre";
	tooltip = "Borre los objeto(s)escogido(s)";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Effacer";
	menu_name = "Effacer";
	tooltip = "Efface l'objet choisi(s)";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Usu";
	menu_name = "Usu";
	tooltip = "Usu zaznaczony(e) obiekt(y)";
#else
	button_name = "Delete";
	menu_name = "Delete";
	tooltip = "Delete the selected object(s)";
#endif
	accel_key = GDK_Delete;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_PROPERTIES;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_properties_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Props";
	menu_name = "Las Propiedades";
	tooltip = "Modifican escogi las propiedades de objeto";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Props";
	menu_name = "Les Proprits";
	tooltip = "Modifier a choisi les proprits de l'objet";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Waciwoci";
	menu_name = "Waciwoci...";
	tooltip = "Modyfikuj waciwoci zaznaczonego obiektu";
#else
	button_name = "Properties";
	menu_name = "Properties...";
	tooltip = "Modify the selected object's properties";
#endif
	accel_key = GDK_Return;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID


	id = EDV_VFS_BROWSER_OP_SELECT_ALL;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_select_all_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "SelAll";
	menu_name = "Escoja Todo";
	tooltip = "Escoge todos los objetos";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "SelAll";
	menu_name = "Choisir Tout";
	tooltip = "Choisir tous objets";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "ZaznWszyst";
	menu_name = "Zaznacz wszystko";
	tooltip = "Zaznacz wszystkie obiekty";
#else
	button_name = "SelAll";
	menu_name = "Select All";
	tooltip = "Select all objects";
#endif
	accel_key = GDK_a;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_UNSELECT_ALL;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_unselect_all_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "UnSelAll";
	menu_name = "Unescoja Todo";
	tooltip = "Unescoge todo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "UnSelAll";
	menu_name = "Unchoisir Tout";
	tooltip = "Unchoisir tous objets";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "OdznWszyst";
	menu_name = "Odznacz wszystko";
	tooltip = "Usu zaznaczenie ze wszystkich obiektw";
#else
	button_name = "UnSelAll";
	menu_name = "Unselect All";
	tooltip = "Unselect all objects";
#endif
	accel_key = GDK_u;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_INVERT_SELECTION;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_invert_selection_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "InvSel";
	menu_name = "Invierta Seleccin";
	tooltip =
 "Selectos todos objetos y unselect no seleccionados todos objetos escogidos";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "InvSel";
	menu_name = "Inverser Slection";
	tooltip =
 "Choisir tous objets dslections et deslectionner tous objets choisis";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "OdwrZazn";
	menu_name = "Odwr zaznaczenie";
	tooltip =
 "Zaznacz wszystkie niezaznaczone obiekty i usu zaznaczenie ze wszystkich zaznaczonych obiektw";
#else
	button_name = "InvSel";
	menu_name = "Invert Selection";
	tooltip =
 "Select all unselected objects and unselect all selected objects";
#endif
	accel_key = GDK_i;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_FIND;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_search_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Buscar";
	menu_name = "Buscar";
	tooltip = "Buscar objeto(s) por atributos especificados";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Dcouverte";
	menu_name = "La Dcouverte";
	tooltip = "L'objet de Dcouverte (s) par les attributs spcifis ";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Znajd";
	menu_name = "Znajd...";
	tooltip = "Znajd obiekt(y) o podanych atrybutach";
#else
	button_name = "Find";
	menu_name = "Find...";
	tooltip = "Find object(s) by specific attributes";
#endif
	accel_key = GDK_f;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID


	id = EDV_VFS_BROWSER_OP_DOWNLOAD;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_download_file_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Download";
	menu_name = "Download";
	tooltip = "Download del objeto";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Download";
	menu_name = "Download";
	tooltip = "Download de l'objet";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Pobierz";
	menu_name = "Pobierz";
	tooltip = "Pobierz obiekt";
#else
	button_name = "Download";
	menu_name = "Download";
	tooltip = "Download an object";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID


	id = EDV_VFS_BROWSER_OP_REFRESH;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_reload_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Refresque";
	menu_name = "Refresque";
	tooltip = "Refresque carpeta escogida para listar";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Rafrachir";
	menu_name = "Rafrachir";
	tooltip = "Rafrachit a choisi l'numrer de l'annuaire";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Odwie";
	menu_name = "Odwie";
	tooltip = "Odwie list obiektw biecego katalogu";
#else
	button_name = "Refresh";
	menu_name = "Refresh";
	tooltip = "Refresh the selected directory's listing";
#endif
	accel_key = GDK_F5;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_REFRESH_ALL;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_reload_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Refresque Todo";
	menu_name = "Refresque Todo";
	tooltip = "Refresca todas listas";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Rafrachir Tout";
	menu_name = "Rafrachir Tout";
	tooltip = "Rafrachir toutes listes";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Odwie wszystko";
	menu_name = "Odwie wszystko";
	tooltip = "Odwie wszystkie listy";
#else
	button_name = "Refresh All";
	menu_name = "Refresh All";
	tooltip = "Refresh all listings";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_GOTO_PARENT;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_folder_parent_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Carpeta padre";
	menu_name = "Ir a carpeta padre";
	tooltip = "Para ir a la carpeta padre";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Parent";
	menu_name = "Aller Au Parent";
	tooltip = "Aller  l'annuaire de parent";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Nadrzdny";
	menu_name = "Id do nadrzdnego";
	tooltip = "Id do katalogu nadrzdnego";
#else
	button_name = "Parent";
	menu_name = "Go To Parent";
	tooltip = "Go to the parent directory";
#endif
	accel_key = GDK_BackSpace;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_GOTO_HOME;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_folder_home_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_FRENCH)
	button_name = "Accueil";
	menu_name = "Aller  l'accueil";
	tooltip = "Aller au rpertoire d'accueil (HOME)";
#else
	button_name = "Home";
	menu_name = "Go To Home";
	tooltip = "Go to the home directory";
#endif
	accel_key = GDK_grave;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_EXPAND;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_tree_branch_expand_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Ensanchar";
	menu_name = "Ensanchar";
	tooltip = "Ensanchar/el desplome escogi rama de rbol";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Augmenter";
	menu_name = "Augmenter";
	tooltip = "L'Augmente/Effondrement branche d'arbre choisie";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Rozwi";
	menu_name = "Rozwi";
	tooltip = "Rozwi/zwi wybran ga drzewa katalogw";
#else
	button_name = "Expand";
	menu_name = "Expand";
	tooltip = "Expand/collapse the selected tree branch";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID


	id = EDV_VFS_BROWSER_OP_SHOW_TOOL_BAR;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	flags = EDV_OPID_EX_TOOL_BAR;
	button_icon_data = (edv_pixmap_data *)icon_tool_bar_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "ToolBar";
	menu_name = "Tool Bar";
	tooltip = "La Exposicin/la barra de herramienta de cuero";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "ToolBar";
	menu_name = "Tool Bar";
	tooltip = "La barre d'outil de montre/peau";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PasNarz";
	menu_name = "Pasek narzdzi";
	tooltip = "Poka/ukryj pasek narzdzi";
#else
	button_name = "ToolBar";
	menu_name = "Tool Bar";
	tooltip = "Show/hide tool bar";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_SHOW_LOCATION_BAR;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_tool_bar_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "LocBar";
	menu_name = "Location Bar";
	tooltip = "La exposicin/la barra de la ubicacin del cuero";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "LocBar";
	menu_name = "Location Bar";
	tooltip = "La barre d'emplacement de montre/peau";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Pascieki";
	menu_name = "Pasek cieki";
	tooltip = "Poka/ukryj pasek cieki";
#else
	button_name = "LocBar";
	menu_name = "Location Bar";
	tooltip = "Show/hide location bar";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_SHOW_MOUNT_BAR;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_tool_bar_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "MountBar";
	menu_name = "Mount Bar";
	tooltip = "La Exposicin/la barra del monte del cuero";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "MountBar";
	menu_name = "Mount Bar";
	tooltip = "La barre de mont de Show/Peau";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PasMontowania";
	menu_name = "Pasek montowania";
	tooltip = "Poka/ukryj pasek montowania";
#else
	button_name = "MountBar";
	menu_name = "Mount Bar";
	tooltip = "Show/hide mount bar";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_SHOW_FIND_BAR;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_tool_bar_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "FindBar";
	menu_name = "Find Bar";
	tooltip = "La exposicin/la barra del hallazgo del cuero";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "FindBar";
	menu_name = "Find Bar";
	tooltip = "La barre de dcouverte de montre/peau";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PasSzuk";
	menu_name = "Pasek wyszukiwania";
	tooltip = "Poka/ukryj pasek wyszukiwania";
#else
	button_name = "FindBar";
	menu_name = "Find Bar";
	tooltip = "Show/hide find bar";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_SHOW_STATUS_BAR;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_status_bar_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "StatusBar";
	menu_name = "Status Bar";
	tooltip = "La exposicin/la barra de la posicin del cuero";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "StatusBar";
	menu_name = "Status Bar";
	tooltip = "La barre de statut de montre/peau";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PasStatusu";
	menu_name = "Pasek statusu";
	tooltip = "Poka/ukryj pasek statusu";
#else
	button_name = "StatusBar";
	menu_name = "Status Bar";
	tooltip = "Show/hide status bar";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID


	id = EDV_VFS_BROWSER_OP_LIST_FILTER;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_wildcards_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Filter";
	menu_name = "Set Name Filter...";
	tooltip = "Set the filter to list only the objects that have\
 a specific name pattern";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_CONTENTS_LIST_AUTO_RESIZE_COLUMNS;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_resize_column_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "ReSzCol";
	menu_name = "Auto Resize Columns";
	tooltip = "Automatically resize the columns each time the list is updated";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_SHOW_HIDDEN_OBJECTS;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_file_hidden_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_FRENCH)
	button_name = "HidObj";
	menu_name = "Objets Cachs";
	tooltip = "Monter/cacher objets cachs(.)";
#else
	button_name = "HidObj";
	menu_name = "Hidden Objects";
	tooltip = "Show/hide hidden objects";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_SHOW_NOACCESS_OBJECTS;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_file_noaccess_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_FRENCH)
	button_name = "NoAccObj";
	menu_name = "Objet  accs rserv";
	tooltip = "Monter/cacher les objets  accs rserv";
#else
	button_name = "NoAccObj";
	menu_name = "No Access Objects";
	tooltip = "Show/hide no access objects";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_DIRECTORY_TREE_ORIGIN;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_tree_origin_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_FRENCH)
	button_name = "TreeOrig";
	menu_name = "Rpertoire Initial...";
	tooltip = "Positionner au rpertoire initial";
#else
	button_name = "TreeOrig";
	menu_name = "Directory Tree Origin...";
	tooltip = "Set the directory tree origin";
#endif
	accel_key = GDK_j;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID


	id = EDV_VFS_BROWSER_OP_MOUNT;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_mount_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Montar";
	menu_name = "Montar";
	tooltip = "Montar/desmontar carpeta de punto de montaje seleccionada";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Le Mont";
	menu_name = "Le Mont";
	tooltip = "Le Mont/unmount a choisi l'annuaire de point de mont";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Montowanie";
	menu_name = "Montowanie";
	tooltip = "Zamontuj odmontuj wybrany punkt montowania";
#else
	button_name = "Mount";
	menu_name = "Mount";
	tooltip = "Mount/unmount selected mount point directory";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_EJECT;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_eject_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Expulse";
	menu_name = "Expulse";
	tooltip = "Expulse medios del Dispositivo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Ejecter";
	menu_name = "Ejecter";
	tooltip = "Ejecter le presse de l'appareil";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Wysu";
	menu_name = "Wysu";
	tooltip = "Wysu nonik z napdu";
#else
	button_name = "Eject";
	menu_name = "Eject";
	tooltip = "Eject media from device";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_DEVICE_CHECK;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_fsck_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "FSCK";
	menu_name = "Archive Cheque De Sistema...";
	tooltip = "Verifique el sistema del archivo de Dispositivo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "FSCK";
	menu_name = "Classer Le Chque De Systme...";
	tooltip = "Vrifier le systme de fichier de l'appareil";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "FSCK";
	menu_name = "Sprawd system plikw...";
	tooltip = "Sprawd system plikw urzdzenia pod ktem bdw";
#else
	button_name = "FSCK";
	menu_name = "File System Check...";
	tooltip = "Check the device's file system";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_DEVICE_TOOLS;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_tools_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "DevTool";
	menu_name = "Herramienta Equipar Dispositivo  ...";
	tooltip = "Corra el programa de herramientas de Dispositivo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "DevTool";
	menu_name = "Outils D'Appareil...";
	tooltip = "Courir le programme d'outils de l'appareil";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "NarzUrzdzenia";
	menu_name = "Narzdzia urzdzenia...";
	tooltip = "Uruchom program narzdzi urzdzenia";
#else
	button_name = "DevTool";
	menu_name = "Device Tools...";
	tooltip = "Run the device's tools program";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_DEVICE_FORMAT;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_floppy_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Format";
	menu_name = "Formatear Medios...";
	tooltip = "Formatear los medios en el Dispositivo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Format";
	menu_name = "Met Format Presse...";
	tooltip = "Met en format le presse dans l'appareil";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Formatuj";
	menu_name = "Formatuj nonik...";
	tooltip = "Formatuj nonik w napdzie";
#else
	button_name = "Format";
	menu_name = "Format Media...";
	tooltip = "Format the media in the device";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID


	id = EDV_VFS_BROWSER_OP_MIME_TYPES;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_mimetypes_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "MIMETypes";
	menu_name = "MIME Types...";
	tooltip = "El panorama/modifica MIME Types";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "MIMETypes";
	menu_name = "MIME Types...";
	tooltip = "Vue/modifie MIME Types";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Typy MIME";
	menu_name = "Typy MIME...";
	tooltip = "Obejrzyj/zmodyfikuj typy MIME";
#else
	button_name = "MIMETypes";
	menu_name = "MIME Types...";
	tooltip = "View/modify MIME Types";
#endif
	accel_key = GDK_m;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_DEVICES;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_devices_list_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Dispositivos";
	menu_name = "Los Dispositivos...";
	tooltip = "El Panorama/modifica las referencias de Dispositivo de sistema de archivo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Appareils";
	menu_name = "Les Appareils...";
	tooltip = "Les rfrences de priphrique-systme de fichier de Vue/Modifie ";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Urzdzenia";
	menu_name = "Urzdzenia...";
	tooltip = "Obejrzyj/zmodyfikuj systemowe ustawienia urzdze";
#else
	button_name = "Devices";
	menu_name = "Devices...";
	tooltip = "View/modify file system device references";
#endif
	accel_key = GDK_d;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID


	id = EDV_VFS_BROWSER_OP_NEW_BROWSER;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_endeavour_file_browser_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "FBrowser";
	menu_name = "Nuevo Examinador De Archivos";
	tooltip = "Muestra un nuevo examinador de archivos";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "FBrowser";
	menu_name = "Nouveau Browser de Fichier";
	tooltip = "Crer un nouveau browser de fichier";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PrzeglPlik";
	menu_name = "Nowa przegldarka plikw";
	tooltip = "Utwrz now przegldark plikw";
#else
	button_name = "FBrowser";
	menu_name = "New File Browser";
	tooltip = "Create a new file browser";
#endif
	accel_key = GDK_1;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_NEW_IMBR;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_endeavour_image_browser_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "IBrowser";
	menu_name = "Nuevo Examinador De Imagenes";
	tooltip = "Muestra un nuevo examinador de imagenes";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "IBrowser";
	menu_name = "Nouveau Browser D'Image";
	tooltip = "Crer un nouveau browser d'image";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PrzeglObr";
	menu_name = "Nowa przegldarka obrazw";
	tooltip = "Utwrz now przegldark obrazw";
#else
	button_name = "IBrowser";
	menu_name = "New Image Browser";
	tooltip = "Create a new image browser";
#endif
	accel_key = GDK_2;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_NEW_ARCHIVER;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_endeavour_archiver_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Archiver";
	menu_name = "Nuevo Compresor De Archivos";
	tooltip = "Muestra un archivo comprimido nuevo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Archiver";
	menu_name = "Nouveau Archiver";
	tooltip = "Crer un nouveau archiver";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Archiwizator";
	menu_name = "Nowy archiwizator";
	tooltip = "Utwrz nowego archwizatora";
#else
	button_name = "Archiver";
	menu_name = "New Archiver";
	tooltip = "Create a new archiver";
#endif
	accel_key = GDK_3;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_RECYCLE_BIN;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = edv_get_recycle_bin_icon_data(
		EDV_ICON_SIZE_20,
		1,
		NULL
	);
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "RecBin";
	menu_name = "El Cajn De Reciclaje";
	tooltip = "Abrir Cajn de Reciclaje (basura)";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "RecBin";
	menu_name = "Recycler L'Huche";
	tooltip = "Recycler l'huche (la poubelle)";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Kosz";
	menu_name = "Kosz";
	tooltip = "Otwrz kosz";
#else
	button_name = "RecBin";
	menu_name = "Recycle Bin";
	tooltip = "Open Recycle Bin";
#endif
	accel_key = GDK_4;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID


	id = EDV_VFS_BROWSER_OP_OPTIONS;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_options_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Las Opciones";
	menu_name = "Las Opciones...";
	tooltip = "Modifican estas opciones de la aplicacin";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Les Options";
	menu_name = "Les Options";
	tooltip = "Modifier ces options de l'application";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Opcje";
	menu_name = "Opcje...";
	tooltip = "Modyfikuj opcje tej aplikacji";
#else
	button_name = "Options";
	menu_name = "Options...";
	tooltip = "Modify this application's options";
#endif
	accel_key = GDK_o;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_CUSTOMIZE;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_customize_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Personal";
	menu_name = "Personaliza...";
	tooltip = "Personaliza la mirada y se siente de esta aplicacin";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Personnal";
	menu_name = "Personnalise...";
	tooltip = "Personnalise le regard et sens de cette application";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Dostosuj";
	menu_name = "Dostosuj...";
	tooltip = "Dostosuj wygld i zachowanie tej aplikacji";
#else
	button_name = "Custom";
	menu_name = "Customize...";
	tooltip = "Customize the look and feel of this application";
#endif
	accel_key = GDK_t;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID


	id = EDV_VFS_BROWSER_OP_HELP_CONTENTS;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "La Ayuda";
	menu_name = "El Contenido";
	tooltip = "El Indice de toda ayuda pagina";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "L'Aide";
	menu_name = "Les Contenus";
	tooltip = "L'Index de toutes pages d'aide";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Pomoc";
	menu_name = "Tematy pomocy";
	tooltip = "Spis treci wszystkich stron pomocy";
#else
	button_name = "Help";
	menu_name = "Contents";
	tooltip = "Index of all help pages";
#endif
	accel_key = GDK_F1;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_HELP_FILE_BROWSER;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "Help";
	menu_name = "File Browser";
	tooltip = "Help on using the file browser";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_HELP_IMAGE_BROWSER;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "Help";
	menu_name = "Image Browser";
	tooltip = "Help on using the image browser";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_HELP_ARCHIVER;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "Help";
	menu_name = "Archiver";
	tooltip = "Help on using the archiver";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_HELP_RECYCLE_BIN;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "Help";
	menu_name = "Recycle Bin";
	tooltip = "Help on using the recycle bin";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_HELP_KEYS_LIST;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "HelpKeys";
	menu_name = "La Lista De Llaves";
	tooltip = "La Lista de todo teclado las operaciones claves";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "HelpKeys";
	menu_name = "La Liste De Clefs";
	tooltip = "La Liste de tout clavier toutes oprations principales";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PomocKlaw";
	menu_name = "Lista klawiszy";
	tooltip = "Lista skrtw klawiaturowych";
#else
	button_name = "HelpKeys";
	menu_name = "Keys List";
	tooltip = "List of all keyboard key operations";

#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_HELP_MIME_TYPES;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "HelpMIME";
	menu_name = "Mime Types";
	tooltip = "Ayude a establecer MIME Types";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "HelpMIME";
	menu_name = "Mime Types";
	tooltip = "Aider sur montant MIME Types";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Pomoc MIME";
	menu_name = "Typy MIME";
	tooltip = "Pomoc w ustawianiu typw MIME";
#else
	button_name = "HelpMIME";
	menu_name = "Mime Types";
	tooltip = "Help on setting up MIME Types";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_HELP_DEVICES;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "HelpDev";
	menu_name = "Los Dispositivos";
	tooltip = "La Ayuda a establecer Dispositivos";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "HelpDev";
	menu_name = "Les Appareils";
	tooltip = "L'Aide sur montant des appareils";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PomUrzdzenia";
	menu_name = "Urzdzenia";
	tooltip = "Pomoc w ustawianiu urzdze";
#else
	button_name = "HelpDev";
	menu_name = "Devices";
	tooltip = "Help on setting up devices";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_HELP_COMMON_OPERATIONS;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "HelpOp";
	menu_name = "Las Operaciones Comunes";
	tooltip = "La Ayuda en las operaciones ms comunes";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "HelpOp";
	menu_name = "Les Oprations Communes";
	tooltip = "L'Aide sur montant des appareils";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PomOperacje";
	menu_name = "Najczstsze operacje";
	tooltip = "Pomoc dotyczca najczstszych operacji";
#else
	button_name = "HelpOp";
	menu_name = "Common Operations";
	tooltip = "Help on the most common operations";

#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_VFS_BROWSER_OP_HELP_ABOUT;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_about_20x20_xpm;
	menu_icon_data = (edv_pixmap_data *)icon_about_20x20_xpm;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Acerca";
	menu_name = "Acerca De";
	tooltip = "Acerca de esta aplicacin";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "De";
	menu_name = "De";
	tooltip = "De cette application";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "O programie";
	menu_name = "O programie";
	tooltip = "Informacje o programie";
#else
	button_name = "About";
	menu_name = "About";
	tooltip = "About this application";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID


	return(op_ids_list);

#undef APPEND_OPID
}

/*
 *	Deletes the File Browser Operation IDs list.
 */
void edv_vfs_browser_ops_list_delete(GList *op_ids_list)
{
	if(op_ids_list == NULL)
		return;

	g_list_foreach(
		op_ids_list,
		(GFunc)edv_vfs_browser_op_delete,
		NULL
	);
	g_list_free(op_ids_list);
}


/*
 *	Creates a new Accel Key List with default values.
 */
CfgAccelkeyList *edv_vfs_browser_accelkeys_new(void)
{
	GList		*glist,
			*op_ids_list,
			*ak_glist = NULL;
	CfgAccelkeyList *ak_list;
	EDVVFSBrowserOp *op;

	op_ids_list = edv_vfs_browser_ops_list_new(NULL);
	for(glist = op_ids_list; glist != NULL; glist = g_list_next(glist))
	{
		op = EDV_VFS_BROWSER_OP(glist->data);
		if(op == NULL)
			continue;

		if(op->flags & EDV_OPID_NO_OP)
			continue;

		ak_glist = g_list_append(
			ak_glist,
			CFGAccelkeyNew(
				(gint)op->id,
				op->accel_key,
				op->accel_mods
			)
		);
	}
	edv_vfs_browser_ops_list_delete(op_ids_list);

	ak_list = CFGAccelkeyListNew(ak_glist);

	g_list_foreach(ak_glist, (GFunc)CFGAccelkeyDelete, NULL);
	g_list_free(ak_glist);

	return(ak_list);
}


/*
 *	Creates the menu bar.
 */
static void EDVBrowserCreateMenuBar(
	EDVVFSBrowser *browser,
	GtkWidget *parent
)
{
	edv_pixmap_data *icon_data = NULL;
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget	*w,
			*menu,
			*menu_bar;
	EDVVFSBrowserOpID id;

	/* Create the GtkMenuBar */
	browser->menu_bar = menu_bar = w = GUIMenuBarCreate(NULL);
	gtk_widget_set_name(w, EDV_WIDGET_NAME_MENU_BAR);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);

#define ADD_MENU_ITEM_LABEL	{		\
 EDVVFSBrowserOp *op =				\
  edv_vfs_browser_op_match_by_id(browser, id);		\
 if(op != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu,					\
   GUI_MENU_ITEM_TYPE_LABEL,			\
   accelgrp,					\
   (icon_data != NULL) ?			\
    icon_data : op->menu_icon_data,		\
   op->menu_name,				\
   op->accel_key, op->accel_mods,		\
   edv_vfs_browser_menu_item_cb, op			\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)edv_vfs_browser_menu_item_enter_cb, op,	\
   (gpointer)edv_vfs_browser_menu_item_leave_cb, op	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_ITEM_CHECK	{		\
 EDVVFSBrowserOp *op =				\
  edv_vfs_browser_op_match_by_id(browser, id);		\
 if(op != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu,					\
   GUI_MENU_ITEM_TYPE_CHECK,			\
   accelgrp,					\
   NULL,					\
   op->menu_name,				\
   op->accel_key, op->accel_mods,		\
   edv_vfs_browser_menu_item_cb, op			\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)edv_vfs_browser_menu_item_enter_cb, op,	\
   (gpointer)edv_vfs_browser_menu_item_leave_cb, op	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_ITEM_SUBMENU_LABEL	{	\
 EDVVFSBrowserOp *op =				\
  edv_vfs_browser_op_match_by_id(browser, id);		\
 if(op != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_LABEL, NULL,	\
   (icon_data != NULL) ?			\
    icon_data : op->menu_icon_data,		\
   op->menu_name,				\
   0, 0,					\
   NULL, NULL					\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)edv_vfs_browser_menu_item_enter_cb, op,	\
   (gpointer)edv_vfs_browser_menu_item_leave_cb, op	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_ITEM_SEPARATOR	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL,						\
  NULL,						\
  0, 0,						\
  NULL, NULL					\
 );						\
}
	/* Create menus */

	/* File Menu */
	browser->file_menu = menu = GUIMenuCreateTearOff();
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Archivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fichier"
#elif defined(PROG_LANGUAGE_GERMAN)
"Akte"
#elif defined(PROG_LANGUAGE_ITALIAN)
"File"
#elif defined(PROG_LANGUAGE_DUTCH)
"Dossier"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Arquivo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Arkiv"
#elif defined(PROG_LANGUAGE_POLISH)
"Plik"
#else
"File"
#endif
		, GTK_JUSTIFY_LEFT
	);
	edv_vfs_browser_file_menu_items_regenerate(browser);

	/* Edit Menu */
	browser->edit_menu = menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		id = EDV_VFS_BROWSER_OP_MOVE;
		ADD_MENU_ITEM_LABEL
		browser->move_mi = w;

		id = EDV_VFS_BROWSER_OP_COPY;
		ADD_MENU_ITEM_LABEL
		browser->copy_mi = w;

		id = EDV_VFS_BROWSER_OP_LINK;
		ADD_MENU_ITEM_LABEL
		browser->link_mi = w;

		id = EDV_VFS_BROWSER_OP_RENAME;
		ADD_MENU_ITEM_LABEL
		browser->rename_mi = w;

		id = EDV_VFS_BROWSER_OP_CHMOD;
		ADD_MENU_ITEM_LABEL
		browser->chmod_mi = w;

		id = EDV_VFS_BROWSER_OP_CHOWN;
		ADD_MENU_ITEM_LABEL
		browser->chown_mi = w;

		id = EDV_VFS_BROWSER_OP_CHTIME;
		ADD_MENU_ITEM_LABEL
		browser->chtime_mi = w;

		id = EDV_VFS_BROWSER_OP_DELETE;
		ADD_MENU_ITEM_LABEL
		browser->delete_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_VFS_BROWSER_OP_COPY_PATH;
		ADD_MENU_ITEM_LABEL
		browser->copy_path_mi = w;

		id = EDV_VFS_BROWSER_OP_COPY_URL;
		ADD_MENU_ITEM_LABEL
		browser->copy_url_mi = w;

		id = EDV_VFS_BROWSER_OP_PASTE;
		ADD_MENU_ITEM_LABEL
		browser->paste_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_VFS_BROWSER_OP_SELECT_ALL;
		ADD_MENU_ITEM_LABEL
		browser->select_all_mi = w;

		id = EDV_VFS_BROWSER_OP_UNSELECT_ALL;
		ADD_MENU_ITEM_LABEL
		browser->unselect_all_mi = w;

		id = EDV_VFS_BROWSER_OP_INVERT_SELECTION;
		ADD_MENU_ITEM_LABEL
		browser->invert_selection_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_VFS_BROWSER_OP_FIND;
		ADD_MENU_ITEM_LABEL
		browser->find_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_VFS_BROWSER_OP_PROPERTIES;
		ADD_MENU_ITEM_LABEL
		browser->properties_mi = w;
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Editar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Editer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Redigieren"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Redigere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bewerking"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Edite"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Rediger"
#elif defined(PROG_LANGUAGE_POLISH)
"Edycja"
#else
"Edit"
#endif
		, GTK_JUSTIFY_LEFT
	);


	/* View Menu */
	browser->view_menu = menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		id = EDV_VFS_BROWSER_OP_REFRESH;
		ADD_MENU_ITEM_LABEL
		browser->refresh_mi = w;

		id = EDV_VFS_BROWSER_OP_REFRESH_ALL;
		ADD_MENU_ITEM_LABEL
		browser->refresh_all_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_VFS_BROWSER_OP_SHOW_TOOL_BAR;
		ADD_MENU_ITEM_CHECK
		browser->show_tool_bar_micheck = w;

		id = EDV_VFS_BROWSER_OP_SHOW_LOCATION_BAR;
		ADD_MENU_ITEM_CHECK
		browser->show_location_bar_micheck = w;

		id = EDV_VFS_BROWSER_OP_SHOW_MOUNT_BAR;
		ADD_MENU_ITEM_CHECK
		browser->show_mount_bar_micheck = w;

		id = EDV_VFS_BROWSER_OP_SHOW_FIND_BAR;
		ADD_MENU_ITEM_CHECK
		browser->show_find_bar_micheck = w;

		id = EDV_VFS_BROWSER_OP_SHOW_STATUS_BAR;
		ADD_MENU_ITEM_CHECK
		browser->show_status_bar_micheck = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_VFS_BROWSER_OP_DIRECTORY_TREE_ORIGIN;
		ADD_MENU_ITEM_LABEL
		browser->directory_tree_origin_mi = w;

		ADD_MENU_ITEM_SEPARATOR       

		id = EDV_VFS_BROWSER_OP_CONTENTS_LIST_AUTO_RESIZE_COLUMNS;
		ADD_MENU_ITEM_CHECK
		browser->contents_list_auto_resize_columns_micheck = w;

		id = EDV_VFS_BROWSER_OP_SHOW_HIDDEN_OBJECTS;
		ADD_MENU_ITEM_CHECK
		browser->show_hidden_objects_micheck = w;

		id = EDV_VFS_BROWSER_OP_SHOW_NOACCESS_OBJECTS;
		ADD_MENU_ITEM_CHECK
		browser->show_noaccess_objects_micheck = w;

		id = EDV_VFS_BROWSER_OP_LIST_FILTER;
		ADD_MENU_ITEM_LABEL
		browser->contents_list_filter_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_VFS_BROWSER_OP_MIME_TYPES;
		ADD_MENU_ITEM_LABEL
		browser->mime_types_mi = w;
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Vista"
#elif defined(PROG_LANGUAGE_FRENCH)
"Vue"
#elif defined(PROG_LANGUAGE_GERMAN)
"Blick"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Veduta"
#elif defined(PROG_LANGUAGE_DUTCH)
"Overzicht"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Vista"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Sikt"
#elif defined(PROG_LANGUAGE_POLISH)
"Widok"
#else
"View"
#endif
		, GTK_JUSTIFY_LEFT
	);

	/* Device Menu */
	browser->device_menu = menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		id = EDV_VFS_BROWSER_OP_MOUNT;
		ADD_MENU_ITEM_LABEL
		browser->mount_mi = w;

		id = EDV_VFS_BROWSER_OP_EJECT;
		ADD_MENU_ITEM_LABEL
		browser->eject_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_VFS_BROWSER_OP_DEVICE_CHECK;
		ADD_MENU_ITEM_LABEL
		browser->device_check_mi = w;

		id = EDV_VFS_BROWSER_OP_DEVICE_TOOLS;
		ADD_MENU_ITEM_LABEL
		browser->device_tools_mi = w;

		id = EDV_VFS_BROWSER_OP_DEVICE_FORMAT;
		ADD_MENU_ITEM_LABEL
		browser->device_format_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_VFS_BROWSER_OP_DEVICES;
		ADD_MENU_ITEM_LABEL
		browser->devices_mi = w;
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Dispositivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Appareil"
#elif defined(PROG_LANGUAGE_GERMAN)
"Vorrichtung"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Apparaat"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Innretning"
#elif defined(PROG_LANGUAGE_POLISH)
"Urzdzenie"
#else
"Device"
#endif
		, GTK_JUSTIFY_LEFT
	);


	/* Settings Menu */
	browser->settings_menu = menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		id = EDV_VFS_BROWSER_OP_WRITE_PROTECT;
		ADD_MENU_ITEM_LABEL
		browser->write_protect_mi = w;

		id = EDV_VFS_BROWSER_OP_DELETE_METHOD_RECYCLE;
		ADD_MENU_ITEM_CHECK
		browser->delete_method_recycle_micheck = w;

		id = EDV_VFS_BROWSER_OP_DELETE_METHOD_PURGE;
		ADD_MENU_ITEM_CHECK
		browser->delete_method_purge_micheck = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_VFS_BROWSER_OP_OPTIONS;
		ADD_MENU_ITEM_LABEL
		browser->options_mi = w;

		id = EDV_VFS_BROWSER_OP_CUSTOMIZE;
		ADD_MENU_ITEM_LABEL
		browser->customize_mi = w;
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Configuracion"
#elif defined(PROG_LANGUAGE_FRENCH)
"Montages"
#elif defined(PROG_LANGUAGE_GERMAN)
"Setzen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Montaggi"
#elif defined(PROG_LANGUAGE_DUTCH)
"Zetten"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Por"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Innstillinger"
#elif defined(PROG_LANGUAGE_POLISH)
"Ustawienia"
#else
"Settings"
#endif
		, GTK_JUSTIFY_LEFT
	);


	/* Windows Menu */
	browser->windows_menu = menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		id = EDV_VFS_BROWSER_OP_NEW_BROWSER;
		ADD_MENU_ITEM_LABEL
		browser->window_new_browser_mi = w;

		id = EDV_VFS_BROWSER_OP_NEW_IMBR;
		ADD_MENU_ITEM_LABEL
		browser->window_new_imbr_mi = w;

		id = EDV_VFS_BROWSER_OP_NEW_ARCHIVER;
		ADD_MENU_ITEM_LABEL
		browser->window_new_archiver_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_VFS_BROWSER_OP_RECYCLE_BIN;
		ADD_MENU_ITEM_LABEL
		browser->window_recycle_bin_mi = w;
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Ventanas"
#elif defined(PROG_LANGUAGE_FRENCH)
"Windows"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fenster"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Finestre"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ramen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Janelas"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Vinduer"
#elif defined(PROG_LANGUAGE_POLISH)
"Okna"
#else
"Windows"
#endif
		, GTK_JUSTIFY_LEFT
	);


	/* Help Menu */
	browser->help_menu = menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		id = EDV_VFS_BROWSER_OP_HELP_CONTENTS;
		ADD_MENU_ITEM_LABEL
		browser->help_contents_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_VFS_BROWSER_OP_HELP_FILE_BROWSER;
		ADD_MENU_ITEM_LABEL
		browser->help_file_browser_mi = w;

		id = EDV_VFS_BROWSER_OP_HELP_KEYS_LIST;
		ADD_MENU_ITEM_LABEL
		browser->help_keys_list_mi = w;

		id = EDV_VFS_BROWSER_OP_HELP_MIME_TYPES;
		ADD_MENU_ITEM_LABEL
		browser->help_mime_types_mi = w;

		id = EDV_VFS_BROWSER_OP_HELP_DEVICES;
		ADD_MENU_ITEM_LABEL
		browser->help_devices_mi = w;

		id = EDV_VFS_BROWSER_OP_HELP_COMMON_OPERATIONS;
		ADD_MENU_ITEM_LABEL
		browser->help_common_operations_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_VFS_BROWSER_OP_HELP_ABOUT;
		ADD_MENU_ITEM_LABEL
		browser->help_about_mi = w;
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Ayuda"
#elif defined(PROG_LANGUAGE_FRENCH)
"Aide"
#elif defined(PROG_LANGUAGE_GERMAN)
"Hilfe"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Aiuto"
#elif defined(PROG_LANGUAGE_DUTCH)
"Hulp"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Ajuda"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Hjelp"
#elif defined(PROG_LANGUAGE_POLISH)
"Pomoc"
#else
"Help"
#endif
		, GTK_JUSTIFY_RIGHT
	);


#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_CHECK
#undef ADD_MENU_ITEM_SUBMENU_LABEL
#undef ADD_MENU_ITEM_SEPARATOR
}


/*
 *	Creates a new File Browser.
 */
EDVVFSBrowser *edv_vfs_browser_new(
	EDVCore *core,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry
)
{
	const gchar	*wm_name,
					*wm_class,
					*directory_ctree_origin_path;
	gpointer combo_rtn;
	GList *glist;
	GtkRcStyle	*standard_rcstyle,
					*lists_rcstyle;
	GtkWidget	*w,
					*parent, *parent2, *parent3, *parent4,
					*toplevel;
	GtkCombo *combo;
	GtkCList *clist;
	GtkCTree *ctree;
	const GtkTargetEntry dnd_tar_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,	0,	EDV_DND_INFO_TEXT_PLAIN},
{GUI_TARGET_NAME_TEXT_URI_LIST,	0,	EDV_DND_INFO_TEXT_URI_LIST},
{GUI_TARGET_NAME_STRING,	0,	EDV_DND_INFO_STRING},
{EDV_DND_TARGET_RECYCLED_OBJECT, 0,	EDV_DND_INFO_RECYCLED_OBJECT},
{EDV_DND_TARGET_ARCHIVE_OBJECT,	0,	EDV_DND_INFO_ARCHIVE_OBJECT}
	};
	const GtkTargetEntry dnd_src_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,	0,	EDV_DND_INFO_TEXT_PLAIN},
{GUI_TARGET_NAME_TEXT_URI_LIST,	0,	EDV_DND_INFO_TEXT_URI_LIST},
{GUI_TARGET_NAME_STRING,	0,	EDV_DND_INFO_STRING}
	};
	CfgList *cfg_list;
	EDVVFSBrowser *browser;

	if(core == NULL)
		return(NULL);

	cfg_list = core->cfg_list;

	standard_rcstyle = core->standard_rcstyle;
	lists_rcstyle = core->lists_rcstyle;

	wm_name = core->wm_name;
	wm_class = core->wm_class;

	directory_ctree_origin_path = EDV_GET_S(
		EDV_CFG_PARM_BROWSER_DIR_TREE_ORIGIN
	);


	/* Create the File Browser */
	browser = EDV_VFS_BROWSER(g_malloc0(sizeof(EDVVFSBrowser)));
	if(browser == NULL)
		return(NULL);

	browser->toplevel = toplevel = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	browser->core = core;
#if 0
	browser->flags = 0;
	browser->freeze_count = 0;
	browser->busy_count = 0;
	browser->stop_count = 0;
#endif

	browser->op_ids_list = edv_vfs_browser_ops_list_new(browser);

	if(EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_TOOL_BAR))
		browser->flags |= EDV_VFS_BROWSER_TOOL_BAR_MAPPED;
	if(EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_LOCATION_BAR))
		browser->flags |= EDV_VFS_BROWSER_LOCATION_BAR_MAPPED;
	if(EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_MOUNT_BAR))
		browser->flags |= EDV_VFS_BROWSER_MOUNT_BAR_MAPPED;
	if(EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_FIND_BAR))
		browser->flags |= EDV_VFS_BROWSER_FIND_BAR_MAPPED;
	if(EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_STATUS_BAR))
		browser->flags |= EDV_VFS_BROWSER_STATUS_BAR_MAPPED;

	browser->contents_list_filter = STRDUP(EDV_GET_S(
		EDV_CFG_PARM_BROWSER_CONTENTS_LIST_FILTER
	));

/*	browser->directory_ctree_expand_optimize_idleid = 0; */

/*	browser->directory_ctree_selected_node = NULL; */
	browser->contents_clist_selected_row = -1;
	browser->selected_dev_num = -1;

	browser->directory_ctree_origin_path = STRDUP(directory_ctree_origin_path);

	browser->last_nrecycle_bin_items = -1;
	browser->last_write_protect_state = -1;

	browser->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	if(!STRISEMPTY(wm_name) && !STRISEMPTY(wm_class))
		gtk_window_set_wmclass(GTK_WINDOW(w), wm_name, wm_class);
	else
		gtk_window_set_wmclass(
			GTK_WINDOW(w), EDV_FILE_BROWSER_WM_CLASS_WINDOW_NAME, PROG_NAME
		);
	gtk_widget_set_name(w, EDV_WIDGET_NAME_FILE_BROWSER_TOPLEVEL);
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, FALSE);
	if((geometry_flags != 0) && (geometry != NULL))
	{
		if((geometry_flags & GDK_GEOMETRY_X) || (geometry_flags & GDK_GEOMETRY_Y))
			gtk_widget_set_uposition(w, geometry->x, geometry->y);

		if((geometry_flags & GDK_GEOMETRY_WIDTH) || (geometry_flags & GDK_GEOMETRY_HEIGHT))
			gtk_widget_set_usize(w, geometry->width, geometry->height);
		else
			CfgGtkSetGtkWindow(
				cfg_list,
				NULL, NULL,
				EDV_CFG_PARM_BROWSER_WIDTH,
				EDV_CFG_PARM_BROWSER_HEIGHT,
				w
			);
	}
	else
	{
		CfgGtkSetGtkWindow(
			cfg_list,
			EDV_CFG_PARM_BROWSER_X,
			EDV_CFG_PARM_BROWSER_Y,
			EDV_CFG_PARM_BROWSER_WIDTH,
			EDV_CFG_PARM_BROWSER_HEIGHT,
			w
		);
	}
	gtk_widget_add_events(
		w,
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "realize",
		GTK_SIGNAL_FUNC(edv_vfs_browser_realize_cb), browser
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "delete_event",
		GTK_SIGNAL_FUNC(edv_vfs_browser_delete_event_cb), browser
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(edv_vfs_browser_key_event_cb), browser
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(edv_vfs_browser_key_event_cb), browser
	);
	parent = w;


	/* Main GtkVBox */
	browser->main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Bars GtkVBox */
	w = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* Menu Bar GtkHandleBox */
	browser->menu_bar_handle = w = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_attached",
		GTK_SIGNAL_FUNC(edv_vfs_browser_handle_child_attach_cb), browser
	);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_detached",
		GTK_SIGNAL_FUNC(edv_vfs_browser_handle_child_detach_cb), browser
	);
	gtk_widget_show(w);
	/* Menu Bar */
	EDVBrowserCreateMenuBar(browser, w);


	/* Tool Bar GtkHandleBox */
	browser->tool_bar_handle = w = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_attached",
		GTK_SIGNAL_FUNC(edv_vfs_browser_handle_child_attach_cb), browser
	);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_detached",
		GTK_SIGNAL_FUNC(edv_vfs_browser_handle_child_detach_cb), browser
	);
	if(browser->flags & EDV_VFS_BROWSER_TOOL_BAR_MAPPED)
		gtk_widget_show(w);
	/* Tool Bar will be built later */
/*	browser->tool_bar = NULL; */


	/* Location Bar GtkHandleBox */
	browser->location_bar_handle = w = gtk_handle_box_new();
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_attached",
		GTK_SIGNAL_FUNC(edv_vfs_browser_handle_child_attach_cb), browser
	);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_detached",
		GTK_SIGNAL_FUNC(edv_vfs_browser_handle_child_detach_cb), browser
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	if(browser->flags & EDV_VFS_BROWSER_LOCATION_BAR_MAPPED)
		gtk_widget_show(w);
	parent2 = w;

	/* Location Bar GtkHBox */
	browser->location_bar_hbox = w = gtk_hbox_new(FALSE, 0);
	gtk_widget_set_name(w, EDV_WIDGET_NAME_LOCATION_BAR);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent2 = w;


	/* Location Icon and Combo GtkHBox */
	w = gtk_hbox_new(FALSE, 0);	/* Don't need spacing between icon
									 * and combo because combo has it
									 * already.
									 */
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* GtkVBox to orient the location icon fixed parent */
	w = gtk_vbox_new(TRUE, 0);
	gtk_widget_set_usize(w, 20, 20);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Location Icon GtkFixed */
	browser->location_icon_fixed = w = gtk_fixed_new();
	gtk_widget_add_events(
		w,
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK |
		GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "realize",
		GTK_SIGNAL_FUNC(edv_vfs_browser_location_bar_icon_realize_cb), browser
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_vfs_browser_location_bar_icon_crossing_cb), browser
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_vfs_browser_location_bar_icon_crossing_cb), browser
	);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Arrastre esto para crear un enlace en esta ubicacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Traner ceci pour crer un lien  cet emplacement"
#elif defined(PROG_LANGUAGE_POLISH)
"Przecignij to by utworzy odnonik do tej lokacji"
#else
"Drag this to create a link to this location"
#endif
	);
	gtk_widget_show(w);

	/* Reset the location icon pixmap */
	browser->location_icon_pm = NULL;

	/* Location GtkCombo */
	glist = edv_open_text_file_glist(
		EDV_GET_S(EDV_CFG_PARM_FILE_BROWSER_LOCATION_HISTORY),
		-1,					/* No lines limit */
		TRUE				/* Strip CRs */
	);
	w = GUIComboCreate(
#if defined(PROG_LANGUAGE_SPANISH)
		"La Ubicacin:",
#elif defined(PROG_LANGUAGE_FRENCH)
		"Emplacement:",
#elif defined(PROG_LANGUAGE_POLISH)
		"cieka:",
#else
		"Location:",
#endif
		"/",
		glist,
		20,					/* Maximum items */
		&combo_rtn,
		browser,
		edv_vfs_browser_location_combo_activate_cb,
		NULL
	);
	if(glist != NULL)
	{
		g_list_foreach(glist, (GFunc)g_free, NULL);
		g_list_free(glist);
		glist = NULL;
	}
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);

	browser->location_combo = w = (GtkWidget *)combo_rtn;
	combo = GTK_COMBO(w);
	gtk_combo_set_case_sensitive(combo, TRUE);
	GUIDNDSetSrc(
		browser->location_icon_fixed,
		dnd_src_types,
		sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY | GDK_ACTION_MOVE |
			GDK_ACTION_LINK,		/* Allowed actions */
		GDK_BUTTON1_MASK,			/* Buttons */
		NULL,
		edv_location_bar_icon_drag_data_get_cb,
		NULL,
		NULL,
		combo
	);
	w = combo->entry;
	edv_entry_set_dnd(core, w);
	edv_entry_set_complete_path(core, w);
	GUIEditableEndowPopupMenu(w, 0);

	/* GtkHandleBox for the Mount Bar */
	browser->mount_bar_handle = w = gtk_handle_box_new();
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_attached",
		GTK_SIGNAL_FUNC(edv_vfs_browser_handle_child_attach_cb), browser
	);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_detached",
		GTK_SIGNAL_FUNC(edv_vfs_browser_handle_child_detach_cb), browser
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	if(browser->flags & EDV_VFS_BROWSER_MOUNT_BAR_MAPPED)
		gtk_widget_show(w);
	parent2 = w;

	/* Create the Mount Bar */
	browser->mount_bar = w = edv_mount_bar_new(
		core,
		edv_vfs_browser_mount_bar_mount_cb,
		browser,
		edv_vfs_browser_mount_bar_eject_cb,
		browser,
		edv_vfs_browser_mount_bar_goto_cb,
		browser,
		edv_vfs_browser_status_message_cb,
		browser
	);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);


	/* GtkHandleBox for the Find Bar */
	browser->find_bar_handle = w = gtk_handle_box_new();
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_attached",
		GTK_SIGNAL_FUNC(edv_vfs_browser_handle_child_attach_cb), browser
	);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_detached",
		GTK_SIGNAL_FUNC(edv_vfs_browser_handle_child_detach_cb), browser
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	if(browser->flags & EDV_VFS_BROWSER_FIND_BAR_MAPPED)
		gtk_widget_show(w);
	parent2 = w;

	/* Create the Find Bar */
	browser->find_bar = w = edv_find_bar_new(
		core,
		EDV_LOCATION_TYPE_VFS,
		edv_vfs_browser_find_bar_location_cb,
		browser,
		edv_vfs_browser_find_bar_start_cb,
		browser,
		edv_vfs_browser_find_bar_end_cb,
		browser,
		edv_vfs_browser_find_bar_match_cb,
		browser,
		edv_vfs_browser_status_message_cb,
		browser,
		edv_vfs_browser_status_progress_cb,
		browser
	);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);


	/* Main GtkHPaned */
	browser->main_paned = w = gtk_hpaned_new();
	CfgGtkSetGtkPaned(
		cfg_list,
		EDV_CFG_PARM_BROWSER_PANED_POSITION,
		w
	);
	gtk_paned_set_handle_size(
		GTK_PANED(w), EDV_PANED_HANDLE_SIZE
	);
	gtk_paned_set_gutter_size(
		GTK_PANED(w), EDV_PANED_GUTTER_SIZE
	);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* GtkScrolledWindow for the Directory GtkCTree */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
		GTK_SCROLLED_WINDOW(w),
		GTK_POLICY_AUTOMATIC,
		GTK_POLICY_AUTOMATIC
	);
	gtk_paned_add1(GTK_PANED(parent2), w);
	gtk_widget_show(w);
	parent3 = w;

	/* Directory GtkCTree */
	browser->directory_ctree = w = gtk_ctree_new(
		1,					/* Number of columns */
		0					/* Tree column */
	);
	clist = GTK_CLIST(w);
	ctree = GTK_CTREE(w);
	gtk_widget_set_name(w, EDV_WIDGET_NAME_FILE_BROWSER_DIRECTORY_TREE);
	gtk_widget_add_events(
		w,
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "motion_notify_event",
		GTK_SIGNAL_FUNC(edv_clist_motion_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(edv_vfs_browser_key_event_cb), browser
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(edv_vfs_browser_key_event_cb), browser
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_vfs_browser_button_event_cb), browser
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "tree_select_row",
		GTK_SIGNAL_FUNC(edv_vfs_browser_tree_select_row_cb), browser
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "tree_unselect_row",
		GTK_SIGNAL_FUNC(edv_vfs_browser_tree_unselect_row_cb), browser
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "tree_expand",
		GTK_SIGNAL_FUNC(edv_vfs_browser_tree_expand_cb), browser
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "tree_collapse",
		GTK_SIGNAL_FUNC(edv_vfs_browser_tree_collapse_cb), browser
	);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_clist_set_shadow_type(
		clist,
		GTK_SHADOW_IN
	);
	gtk_clist_set_selection_mode(
		clist,
		GTK_SELECTION_SINGLE
	);
	gtk_clist_set_column_auto_resize(
		clist,
		0,
		TRUE
	);
	gtk_clist_set_column_justification(
		clist,
		0,
		GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_row_height(
		clist,
		EDV_LIST_ROW_SPACING
	);
	gtk_ctree_set_line_style(
		ctree,
		(GtkCTreeLineStyle)EDV_GET_I(EDV_CFG_PARM_TREE_LINES_STYLE)
	);
	gtk_ctree_set_expander_style(
		ctree,
		(GtkCTreeExpanderStyle)EDV_GET_I(EDV_CFG_PARM_TREE_EXPANDER_STYLE)
	);
	GUIDNDSetSrc(
		w,
		dnd_src_types,
		sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY | GDK_ACTION_MOVE |
			GDK_ACTION_LINK,	/* Allowed actions */
		GDK_BUTTON1_MASK,		/* Buttons */
		NULL,
		edv_vfs_browser_tree_drag_data_get_cb,
		edv_vfs_browser_tree_drag_data_delete_cb,
		NULL,
		browser
	);
	GUIDNDSetTar(
		w,
		dnd_tar_types,
		sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY | GDK_ACTION_MOVE |
			GDK_ACTION_LINK,	/* Actions */
		GDK_ACTION_MOVE,		/* Default action if same */
		GDK_ACTION_MOVE,		/* Default action */
		edv_vfs_browser_tree_drag_data_received_cb,
		browser,
		TRUE				/* Highlight */
	);
	gtk_widget_show(w);


	/* GtkScrolledWindow for the Contents GtkCList */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
		GTK_SCROLLED_WINDOW(w),
		GTK_POLICY_AUTOMATIC,
		GTK_POLICY_AUTOMATIC
	);
	gtk_paned_add2(GTK_PANED(parent2), w);
	gtk_widget_show(w);
	parent3 = w;

	/* Contents GtkCList */
	browser->contents_clist = w = gtk_clist_new(EDV_VFS_BROWSER_NCOLUMN_TYPES);
	clist = GTK_CLIST(w);
	gtk_widget_set_name(w, EDV_WIDGET_NAME_FILE_BROWSER_CONTENTS_LIST);
	gtk_widget_add_events(
		w,
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "realize",
		GTK_SIGNAL_FUNC(edv_vfs_browser_list_realize_cb), browser
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "motion_notify_event",
		GTK_SIGNAL_FUNC(edv_clist_motion_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(edv_vfs_browser_key_event_cb), browser
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(edv_vfs_browser_key_event_cb), browser
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_vfs_browser_button_event_cb), browser
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "resize_column",
		GTK_SIGNAL_FUNC(edv_vfs_browser_resize_column_cb), browser
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "click_column",
		GTK_SIGNAL_FUNC(edv_vfs_browser_click_column_cb), browser
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "select_row",
		GTK_SIGNAL_FUNC(edv_vfs_browser_list_select_row_cb), browser
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "unselect_row",
		GTK_SIGNAL_FUNC(edv_vfs_browser_list_unselect_row_cb), browser
	);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED);
	if(clist->columns > 0)
	{
		CfgIntList *column_widths_intlist = EDV_GET_INT_LIST(
			EDV_CFG_PARM_BROWSER_CONTENTS_COLUMN_WIDTH
		);
		if(column_widths_intlist != NULL)
		{
			EDVVFSBrowserColumnType column_type = 0;
			GList *glist = column_widths_intlist->list;
			while(glist != NULL)
			{
				edv_vfs_browser_list_set_column_width_by_type(
					browser,
					column_type,
					MAX((gint)glist->data, 10)	/* Width */
				);
				column_type++;
				glist = g_list_next(glist);
			}
		}
	}
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
	GUIDNDSetSrc(
		w,
		dnd_src_types,
		sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY | GDK_ACTION_MOVE |
			GDK_ACTION_LINK,	/* Allowed actions */
		GDK_BUTTON1_MASK,		/* Buttons */
		NULL,
		edv_vfs_browser_list_drag_data_get_cb,
		edv_vfs_browser_list_drag_data_delete_cb,
		NULL,
		browser
	);
	GUIDNDSetTar(
		w,
		dnd_tar_types,
		sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY | GDK_ACTION_MOVE |
			GDK_ACTION_LINK,	/* Actions */
		GDK_ACTION_MOVE,		/* Default action if same */
		GDK_ACTION_MOVE,		/* Default action */
		edv_vfs_browser_list_drag_data_received_cb,
		browser,
		TRUE				/* Highlight */
	);
	gtk_widget_show(w);


	/* Create the Tool Bar */
	edv_vfs_browser_tool_bar_regenerate(browser);

	/* Create the Directory GtkCTree's right-click menu */
	edv_vfs_browser_tree_menu_regenerate(browser);

	/* Create the Contents GtkCList's right-click menu */
	edv_vfs_browser_list_menu_regenerate(browser);


	/* Status Bar */
	browser->status_bar = w = edv_status_bar_new(core);
	gtk_box_pack_end(GTK_BOX(parent), w, FALSE, FALSE, 0);
	if(browser->flags & EDV_VFS_BROWSER_STATUS_BAR_MAPPED)
		gtk_widget_show(w);


	/* Create the New Objects submenu */
	edv_vfs_browser_new_object_menu_items_regenerate(browser);

	/* Create the Open To submenu */
	edv_vfs_browser_open_to_menu_regenerate(browser, NULL);


	/* Set the initial RC styles */
	if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			browser->toplevel,
			standard_rcstyle
		);
	if(lists_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			browser->directory_ctree,
			lists_rcstyle
		);
	if(lists_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			browser->contents_clist,
			lists_rcstyle
		);
	if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			browser->directory_ctree_menu,
			standard_rcstyle
		);
	if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			browser->contents_clist_menu,
			standard_rcstyle
		);

	/* Load special icons */
	if(TRUE)
	{
		gchar *icons_path = g_strconcat(
			EDV_GET_S(EDV_CFG_PARM_DIR_GLOBAL),
			G_DIR_SEPARATOR_S,
			EDV_NAME_ICONS_SUBDIR,
			NULL
		);

#define LOAD_ICON(_icon_,_path_)	{		\
 gchar *path = g_strconcat(				\
  icons_path,						\
  G_DIR_SEPARATOR_S,					\
  (_path_),						\
  NULL							\
 );							\
 if(path != NULL) {					\
  (_icon_) = edv_pixmap_new_from_file(path);		\
  g_free(path);						\
 }							\
}
		LOAD_ICON(
			browser->directory_parent_icon,
			"icon_folder_parent_20x20.xpm"
		);
#undef LOAD_ICON

		g_free(icons_path);
	}

	/* Set the initial Accel Keys */
	edv_vfs_browser_accelkeys_regenerate(browser);

	browser->freeze_count--;

	return(browser);
}

/*
 *	Syncronizes the File Browser's widget values with other values.
 */
void edv_vfs_browser_sync_data(EDVVFSBrowser *browser)
{
	if(browser == NULL)
		return;

/* This function may be removed in the future, since there really is
 * no floating data from widgets to be synced to disk on the browser
 */

}


/*
 *	Updates the File Browser's widgets to reflect current values.
 */
void edv_vfs_browser_update_display(EDVVFSBrowser *browser)
{
	gboolean	write_protect,
			write_protect_changed = FALSE,
			sensitive,
			state,
			sel_dir_node_expanded = FALSE,
			sel_dir_node_has_child = FALSE;
	gint		i,
			sel_dev_num,
			sel_obj_row;
	guint8		**icon_data,
			**recbin_icon_data;
	GtkWidget	*w,
			*tb;
	GtkCTreeNode *sel_dir_node;
	CfgList *cfg_list;
	EDVDeleteMethod delete_method;
	EDVDevice *sel_dev_ptr;
	EDVCore *core;

	if(browser == NULL)
		return;

	browser->freeze_count++;

	tb = browser->tool_bar;
	core = browser->core;
	cfg_list = core->cfg_list;

#define MENU_ITEM_SET_TOGGLE(_w_,_state_)		\
 GUIMenuItemSetCheck(					\
  (_w_),						\
  (_state_),						\
  FALSE							\
 )
#define TOOL_BAR_ITEM_SET_SENSITIVE(_op_,_sensitive_)	\
 ToolBarSetItemSensitive(				\
  tb,							\
  (_op_),						\
  (_sensitive_)						\
 )
#define TOOL_BAR_ITEM_SET_TOGGLE(_op_,_state_)		\
 ToolBarSetItemToggle(					\
  tb,							\
  (_op_),						\
  (_state_)						\
 )
#define TOOL_BAR_ITEM_UPDATE(_op_,_text_,_icon_data_,_tip_) \
 ToolBarUpdateItem(					\
  tb,							\
  (_op_),						\
  (_text_),						\
  (_icon_data_),					\
  (_tip_)						\
 )

	/* Get the master write protect state */
	write_protect = EDV_GET_B(EDV_CFG_PARM_WRITE_PROTECT);
	if(browser->last_write_protect_state < 0)
	{
		write_protect_changed = TRUE;
	}
	else
	{
		if((write_protect && !browser->last_write_protect_state) ||
		   (!write_protect && browser->last_write_protect_state)
		)
			write_protect_changed = TRUE;
	}
	browser->last_write_protect_state = write_protect ? 1 : 0;

	/* Delete method */
	delete_method = (EDVDeleteMethod)EDV_GET_I(
		EDV_CFG_PARM_DELETE_METHOD
	);

	/* Get last selected directory node and contents row, note
	 * that even not NULL or positive does not mean they are
	 * valid. Check if they actually exist on the list before
	 * referencing them
	 */
	sel_dir_node = browser->directory_ctree_selected_node;
	sel_obj_row = browser->contents_clist_selected_row;

	/* Get last selected device (if any) */
	sel_dev_num = browser->selected_dev_num;
	sel_dev_ptr = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)sel_dev_num
	));

	/* Get the Recycle Bin icon data */
	icon_data = edv_get_recycle_bin_icon_data(
		EDV_ICON_SIZE_20,
		core->last_nrecycle_bin_items,
		NULL
	);
	browser->last_nrecycle_bin_items = core->last_nrecycle_bin_items;
	recbin_icon_data = icon_data;

	/* Get the selected directory node expand state (if any) */
	if(browser->directory_ctree != NULL)
	{
		GtkCTreeRow *row_ptr;
		GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
		(void)edv_ctree_get_selected_last(ctree, &row_ptr);
		if(row_ptr != NULL)
		{
			sel_dir_node_expanded = row_ptr->expanded;
			sel_dir_node_has_child = (row_ptr->children != NULL) ?
				TRUE : FALSE;
		}
	}


	/* Update the mount bar */
	edv_mount_bar_update_display(browser->mount_bar);

	/* Update the find bar */
	edv_find_bar_update_display(browser->find_bar);


	/* Update the GtkMenus, GtkMenuItems, and ToolBar */

	/* File Menu */
	/* Open & Open With */
	sensitive = ((sel_dir_node != NULL) ||
		(sel_obj_row > -1)) ? TRUE : FALSE;
	/* Open */
	GTK_WIDGET_SET_SENSITIVE(browser->open_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_OPEN, sensitive);
	/* Open To */
	GTK_WIDGET_SET_SENSITIVE(browser->open_to_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_OPEN_TO, sensitive);
	/* Open with */
	GTK_WIDGET_SET_SENSITIVE(browser->open_with_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_OPEN_WITH, sensitive);

	/* New Object Submenu Item */
	sensitive = ((sel_dir_node != NULL) ||
		(sel_obj_row > -1)) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(browser->new_object_mi, sensitive);
	/* New File */
	GTK_WIDGET_SET_SENSITIVE(browser->new_file_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_NEW_FILE, sensitive);
	/* New Directory */
	GTK_WIDGET_SET_SENSITIVE(browser->new_directory_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_NEW_DIRECTORY, sensitive);
	/* New Link */
	GTK_WIDGET_SET_SENSITIVE(browser->new_link_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_NEW_LINK, sensitive);
	/* New FIFO */
	GTK_WIDGET_SET_SENSITIVE(browser->new_fifo_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_NEW_FIFO, sensitive);
	/* New Block Device */
	GTK_WIDGET_SET_SENSITIVE(browser->new_device_block_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_NEW_DEVICE_BLOCK, sensitive);
	/* New Character Device */
	GTK_WIDGET_SET_SENSITIVE(browser->new_device_character_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_NEW_DEVICE_CHARACTER, sensitive);
	/* New Socket */
	GTK_WIDGET_SET_SENSITIVE(browser->new_socket_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_NEW_SOCKET, sensitive);

	/* Go To Parent */
	sensitive = TRUE;
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_GOTO_PARENT, sensitive);

	/* Go To Home */
	sensitive = TRUE;
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_GOTO_HOME, sensitive);

	/* Expand/Colapse */
	if(sel_dir_node != NULL)
		sensitive = sel_dir_node_has_child;
	else
		sensitive = FALSE;
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_EXPAND, sensitive);
	if(TRUE)
	{
		const gchar *text;
		edv_pixmap_data *icon_data;
		if(sensitive)
		{
			if(sel_dir_node_expanded)
			{
				text = "Collapse";
				icon_data = (edv_pixmap_data *)icon_tree_branch_colapse_20x20_xpm;
			}
			else
			{
				text = "Expand";
				icon_data = (edv_pixmap_data *)icon_tree_branch_expand_20x20_xpm;
			}
		}
		else
		{
			text = "Expand/Collapse";
			icon_data = (edv_pixmap_data *)icon_tree_branch_expand_20x20_xpm;
		}
		TOOL_BAR_ITEM_UPDATE(
			EDV_VFS_BROWSER_OP_EXPAND,
			text,
			icon_data,
			NULL
		);
	}


	/* Edit Menu */
	sensitive = ((sel_dir_node != NULL) ||
		(sel_obj_row > -1)) ? TRUE : FALSE;
	/* Move */
	GTK_WIDGET_SET_SENSITIVE(browser->move_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_MOVE, sensitive);
	/* Copy */
	GTK_WIDGET_SET_SENSITIVE(browser->copy_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_COPY, sensitive);
	/* Link */
	GTK_WIDGET_SET_SENSITIVE(browser->link_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_LINK, sensitive);
	/* Rename */
	GTK_WIDGET_SET_SENSITIVE(browser->rename_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_RENAME, sensitive);
	/* Change Permissions */
	GTK_WIDGET_SET_SENSITIVE(browser->chmod_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_CHMOD, sensitive);
	/* Change Ownership */
	GTK_WIDGET_SET_SENSITIVE(browser->chown_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_CHOWN, sensitive);
	/* Set Time Stamps */
	GTK_WIDGET_SET_SENSITIVE(browser->chtime_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_CHTIME, sensitive);
	/* Delete */
	GTK_WIDGET_SET_SENSITIVE(browser->delete_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_DELETE, sensitive);
	/* Copy Path */
	GTK_WIDGET_SET_SENSITIVE(browser->copy_path_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_COPY_PATH, sensitive);
	/* Copy URL */
	GTK_WIDGET_SET_SENSITIVE(browser->copy_url_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_COPY_URL, sensitive);
	/* Paste */
	GTK_WIDGET_SET_SENSITIVE(browser->paste_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_PASTE, sensitive);
	/* Properties */
	GTK_WIDGET_SET_SENSITIVE(browser->properties_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_PROPERTIES, sensitive);

	/* View Menu */
	/* Show Tool Bar */
	state = EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_TOOL_BAR);
	MENU_ITEM_SET_TOGGLE(browser->show_tool_bar_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_VFS_BROWSER_OP_SHOW_TOOL_BAR, state);
	/* Show Location Bar */
	state = EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_LOCATION_BAR);
	MENU_ITEM_SET_TOGGLE(browser->show_location_bar_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_VFS_BROWSER_OP_SHOW_LOCATION_BAR, state);
	/* Show Mount Bar */
	state = EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_MOUNT_BAR);
	MENU_ITEM_SET_TOGGLE(browser->show_mount_bar_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_VFS_BROWSER_OP_SHOW_MOUNT_BAR, state);
	/* Show Find Bar */
	state = EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_FIND_BAR);
	MENU_ITEM_SET_TOGGLE(browser->show_find_bar_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_VFS_BROWSER_OP_SHOW_FIND_BAR, state);
	/* Show Status Bar */
	state = EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_STATUS_BAR);
	MENU_ITEM_SET_TOGGLE(browser->show_status_bar_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_VFS_BROWSER_OP_SHOW_STATUS_BAR, state);
	/* Show Hidden Objects */
	state = EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_OBJECT_HIDDEN);
	MENU_ITEM_SET_TOGGLE(browser->show_hidden_objects_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_VFS_BROWSER_OP_SHOW_HIDDEN_OBJECTS, state);
	/* Show No Access Objects */
	state = EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_OBJECT_NOACCESS);
	MENU_ITEM_SET_TOGGLE(browser->show_noaccess_objects_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_VFS_BROWSER_OP_SHOW_NOACCESS_OBJECTS, state);
	/* Auto Resize Columns */
	state = EDV_GET_B(EDV_CFG_PARM_BROWSER_CONTENTS_LIST_AUTO_RESIZE_COLUMNS);
	MENU_ITEM_SET_TOGGLE(browser->contents_list_auto_resize_columns_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_VFS_BROWSER_OP_CONTENTS_LIST_AUTO_RESIZE_COLUMNS, state);


	/* Device Menu */
	sensitive = (sel_dev_ptr != NULL) ?
		!EDV_DEVICE_IS_NO_UNMOUNT(sel_dev_ptr) : FALSE;
	w = browser->mount_mi;
	GTK_WIDGET_SET_SENSITIVE(w, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_MOUNT, sensitive);
	if(w != NULL)
	{
		const gchar *text;
		edv_pixmap_data *icon_data;
		if(sel_dev_ptr != NULL)
		{
			if(EDV_DEVICE_IS_MOUNTED(sel_dev_ptr))
			{
				text = "Unmount";
				icon_data = (edv_pixmap_data *)icon_unmount_20x20_xpm;
			}
			else
			{
				text = "Mount";
				icon_data = (edv_pixmap_data *)icon_mount_20x20_xpm;
			}
		}
		else
		{
			text = "Mount/Unmount";
			icon_data = (edv_pixmap_data *)icon_unmount_20x20_xpm;
		}
		GUIMenuItemSetPixmap(w, icon_data);
		GUIMenuItemSetLabel(w, text);
		TOOL_BAR_ITEM_UPDATE(
			EDV_VFS_BROWSER_OP_MOUNT,
			text,
			icon_data,
			NULL
		);
	}

	/* Eject */
	if(sel_dev_ptr != NULL)
		sensitive = !STRISEMPTY(sel_dev_ptr->command_eject);
	else
		sensitive = FALSE;
	GTK_WIDGET_SET_SENSITIVE(browser->eject_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_EJECT, sensitive);

	/* Device check, tools, and format commands */
	if(sel_dev_ptr != NULL)
		sensitive = !STRISEMPTY(sel_dev_ptr->command_check);
	else
		sensitive = FALSE;
	GTK_WIDGET_SET_SENSITIVE(browser->device_check_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_DEVICE_CHECK, sensitive);

	if(sel_dev_ptr != NULL)
		sensitive = !STRISEMPTY(sel_dev_ptr->command_tools);
	else
		sensitive = FALSE;
	GTK_WIDGET_SET_SENSITIVE(browser->device_tools_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_DEVICE_TOOLS, sensitive);

	if(sel_dev_ptr != NULL)
		sensitive = !STRISEMPTY(sel_dev_ptr->command_format);
	else
		sensitive = FALSE;
	GTK_WIDGET_SET_SENSITIVE(browser->device_format_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_DEVICE_FORMAT, sensitive);


	/* Settings Menu */
	/* Master Write Protect */
	if(write_protect_changed)
	{
		w = browser->write_protect_mi;
		GUIMenuItemSetPixmap(
			w,
			(guint8 **)(write_protect ?
				icon_write_protect_on_20x20_xpm : icon_write_protect_off_20x20_xpm)
		);
		GUIMenuItemSetLabel(
			w,
			write_protect ?
				"Master Write Protect: On" :
				"Master Write Protect: Off"
		);
		TOOL_BAR_ITEM_UPDATE(
			EDV_VFS_BROWSER_OP_WRITE_PROTECT,
			NULL,
			(guint8 **)(write_protect ?
				icon_write_protect_on_20x20_xpm : icon_write_protect_off_20x20_xpm),
			NULL
		);
	}

	/* Delete Method: Recycle */
	state = (delete_method == EDV_DELETE_METHOD_RECYCLE) ? TRUE : FALSE;
	MENU_ITEM_SET_TOGGLE(browser->delete_method_recycle_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_VFS_BROWSER_OP_DELETE_METHOD_RECYCLE, state);

	/* Delete Method: Purge */
	state = (delete_method == EDV_DELETE_METHOD_PURGE) ? TRUE : FALSE;
	MENU_ITEM_SET_TOGGLE(browser->delete_method_purge_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_VFS_BROWSER_OP_DELETE_METHOD_PURGE, state);


	/* Windows
	 *
	 * New Browser
	 */
	w = browser->window_new_browser_mi;
	i = EDV_GET_I(EDV_CFG_PARM_MAX_BROWSERS);
	sensitive = ((core->total_browsers < i) || (i < 0)) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(w, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_NEW_BROWSER, sensitive);

	/* New Image Browser */
	w = browser->window_new_imbr_mi;
	i = EDV_GET_I(EDV_CFG_PARM_MAX_IMBRS);
	sensitive = ((core->total_imbrs < i) || (i < 0)) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(w, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_NEW_IMBR, sensitive);

	/* New Archiver */
	w = browser->window_new_archiver_mi;
	i = EDV_GET_I(EDV_CFG_PARM_MAX_ARCHIVERS);
	sensitive = ((core->total_archivers < i) || (i < 0)) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(w, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_VFS_BROWSER_OP_NEW_ARCHIVER, sensitive);

	/* Recycle Bin */
	if(recbin_icon_data != NULL)
	{
		GUIMenuItemSetPixmap(
			browser->window_recycle_bin_mi,
			recbin_icon_data
		);
		TOOL_BAR_ITEM_UPDATE(
			EDV_VFS_BROWSER_OP_RECYCLE_BIN,
			NULL,
			recbin_icon_data,
			NULL
		);
	}


	/* Directory Tree right-click menu */
	sensitive = (sel_dir_node != NULL) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(
		browser->directory_ctree_move_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		browser->directory_ctree_copy_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		browser->directory_ctree_link_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		browser->directory_ctree_rename_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		browser->directory_ctree_chmod_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		browser->directory_ctree_chown_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		browser->directory_ctree_delete_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		browser->directory_ctree_copy_path_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		browser->directory_ctree_copy_url_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		browser->directory_ctree_paste_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		browser->directory_ctree_properties_mi, sensitive
	);

	w = browser->directory_ctree_expand_mi;
	if(sel_dir_node != NULL)
		sensitive = sel_dir_node_has_child;
	else
		sensitive = FALSE;
	GTK_WIDGET_SET_SENSITIVE(w, sensitive);
	if(w != NULL)
	{
		const gchar *text;
		edv_pixmap_data *icon_data;
		if(sensitive)
		{
			if(sel_dir_node_expanded)
			{
				text = "Collapse";
				icon_data = (edv_pixmap_data *)icon_tree_branch_colapse_20x20_xpm;
			}
			else
			{
				text = "Expand";
				icon_data = (edv_pixmap_data *)icon_tree_branch_expand_20x20_xpm;
			}
		}
		else
		{
			text = "Expand/Collapse";
			icon_data = (edv_pixmap_data *)icon_tree_branch_expand_20x20_xpm;
		}
		GUIMenuItemSetLabel(
			w,
			text
		);
		GUIMenuItemSetPixmap(
			w,
			icon_data
		);
	}

	w = browser->directory_ctree_mount_mi;
	if(sel_dev_ptr != NULL)
	{
		GTK_WIDGET_SHOW(w);
	}
	else
	{
		GTK_WIDGET_HIDE(w);
	}
	sensitive = (sel_dev_ptr != NULL) ?
		!EDV_DEVICE_IS_NO_UNMOUNT(sel_dev_ptr) : FALSE;
	GTK_WIDGET_SET_SENSITIVE(w, sensitive);
	if(sel_dev_ptr != NULL)
	{
		const gchar *text;
		edv_pixmap_data *icon_data;
		if(EDV_DEVICE_IS_MOUNTED(sel_dev_ptr))
		{
			text = "Unmount";
			icon_data = (edv_pixmap_data *)icon_unmount_20x20_xpm;
		}
		else
		{   
			text = "Mount";
			icon_data = (edv_pixmap_data *)icon_mount_20x20_xpm;
		}
		GUIMenuItemSetPixmap(w, icon_data);
		GUIMenuItemSetLabel(w, text);
	}

	w = browser->directory_ctree_mount_misep;
	if(sel_dev_ptr != NULL)
	{
		GTK_WIDGET_SHOW(w);
	}
	else
	{
		GTK_WIDGET_HIDE(w);
	}

	/* Eject */
	w = browser->directory_ctree_eject_mi;
	if(sel_dev_ptr != NULL)
	{
		GTK_WIDGET_SHOW(w);
	}
	else
	{
		GTK_WIDGET_HIDE(w);
	}
	if(sel_dev_ptr != NULL)
		sensitive = (sel_dev_ptr->command_eject != NULL) ?
			(*sel_dev_ptr->command_eject != '\0') : FALSE;
	else
		sensitive = FALSE;
	GTK_WIDGET_SET_SENSITIVE(w, sensitive);


	/* Contents List right-click menu */
	sensitive = (sel_obj_row > -1) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(
		browser->contents_clist_open_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		browser->contents_clist_open_to_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		browser->contents_clist_open_with_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		browser->contents_clist_move_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		browser->contents_clist_copy_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		browser->contents_clist_link_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		browser->contents_clist_rename_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		browser->contents_clist_chmod_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		browser->contents_clist_chown_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		browser->contents_clist_delete_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		browser->contents_clist_copy_path_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		browser->contents_clist_copy_url_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		browser->contents_clist_paste_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		browser->contents_clist_properties_mi, sensitive
	);

	/* Status Bar */
	edv_status_bar_update_display(browser->status_bar);

#undef MENU_ITEM_SET_TOGGLE
#undef TOOL_BAR_ITEM_SET_SENSITIVE
#undef TOOL_BAR_ITEM_SET_TOGGLE
#undef TOOL_BAR_ITEM_UPDATE

	browser->freeze_count--;
}

/*
 *	Sets the File Browser as busy or ready.
 */
void edv_vfs_browser_set_busy(EDVVFSBrowser *browser, const gboolean busy)
{
	GdkCursor *cur;
	GtkWidget *w;
	EDVCore *core;

	if(browser == NULL)
		return;

	w = browser->toplevel;
	core = browser->core;

	if(busy)
	{
		browser->busy_count++;
		if(browser->busy_count > 1)
			return;
		cur = edv_get_cursor(core, EDV_CURSOR_CODE_BUSY);
	}
	else
	{
		browser->busy_count--;
		if(browser->busy_count < 0)
			browser->busy_count = 0;
		if(browser->busy_count > 0)
			return;
		cur = NULL;			/* Use default cursor */
	}

	if(w->window != NULL)
	{
		gdk_window_set_cursor(w->window, cur);
		gdk_flush();
	}
}

/*
 *	Checks if the File Browser is mapped.
 */
gboolean edv_vfs_browser_is_mapped(EDVVFSBrowser *browser)
{
	if(browser == NULL)
		return(FALSE);

	return(GTK_WIDGET_MAPPED(browser->toplevel));
}

/*
 *	Maps the File Browser.
 */
void edv_vfs_browser_map(EDVVFSBrowser *browser)
{
	if(browser == NULL)
		return;

	gtk_widget_show_raise(browser->toplevel);
	browser->flags |= EDV_VFS_BROWSER_MAPPED;
}

/*
 *	Unmaps the File Browser.
 */
void edv_vfs_browser_unmap(EDVVFSBrowser *browser)
{
	if(browser == NULL)
		return;

	gtk_widget_hide(browser->toplevel);
	browser->flags &= ~EDV_VFS_BROWSER_MAPPED;
}

/*
 *	Deletes the File Browser.
 */
void edv_vfs_browser_delete(EDVVFSBrowser *browser)
{
	gint i;

	if(browser == NULL)
		return;

	/* Clear the Contents List */
	edv_vfs_browser_list_clear(browser);

	/* Clear the Directory Tree */
	edv_vfs_browser_tree_clear(browser);

	/* Remove the idle & timeout callbacks */
	browser->directory_ctree_expand_optimize_idleid =
		GTK_IDLE_REMOVE(browser->directory_ctree_expand_optimize_idleid);

	/* Reset the last selected item references */
	browser->directory_ctree_selected_node = NULL;
	browser->contents_clist_selected_row = -1;


	edv_vfs_browser_unmap(browser);

	browser->freeze_count++;

	GTK_WIDGET_DESTROY(browser->directory_ctree_menu);
	GTK_WIDGET_DESTROY(browser->contents_clist_open_to_menu);
	GTK_WIDGET_DESTROY(browser->contents_clist_menu);

	GTK_WIDGET_DESTROY(browser->new_object_menu);
	GTK_WIDGET_DESTROY(browser->open_to_menu);

	GTK_WIDGET_DESTROY(browser->toplevel);

	/* Unref all the cell styles */
	for(i = 0; i < EDV_VFS_BROWSER_NCELL_STYLE_TYPES; i++)
		GTK_STYLE_UNREF(browser->cell_style[i]);

	/* Unref the special icons */
	(void)edv_pixmap_unref(browser->directory_parent_icon);

	/* Operation IDs */
	edv_vfs_browser_ops_list_delete(browser->op_ids_list);

	g_free(browser->directory_ctree_origin_path);
	g_free(browser->contents_list_filter);

	browser->freeze_count--;

	g_free(browser);
}
