#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include "../splash.h"


gint runlevel;

static void print_help(const gchar *prog_name);
static void signal_cb(int s);
static void splash_button_cb(
	GtkWidget *splash, GdkEventButton *button, gpointer data
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


static void print_help(const gchar *prog_name)
{
	g_print(
"Usage: %s [splash_xpm_path] [options]\n",
	    prog_name
	);
	g_print(
"\n\
    The [splash_xpm_path] specifies the path to the splash XPM\n\
    image file.\n\
\n\
    The [options] can be any of the following:\n\
\n\
        --effects <effects>     Specifies the effects, <effects> can be:\n\
                                    none\n\
                                    fade_white\n\
                                    fade_black\n\
                                    fade_background\n\
        -e                      Same as --effects.\n\
        --map-int <ms>          Specifies the map and unmap intervals\n\
                                in milliseconds.\n\
        -i                      Same as --map-int.\n\
        --message <message>     Specifies the splash message.\n\
        -m                      Same as --message.\n\
        --delay <seconds>       Specifies the delay in seconds to keep\n\
                                the splash mapped before it is unmapped.\n\
                                If this is not specified then the splash\n\
                                will stay mapped until SIGINT is\n\
                                received or the splash is clicked on.\n\
        -d                      Same as --delay.\n\
\n\
        --help                  Prints this help screen and exits.\n\
\n"
	);
}


static void signal_cb(int s)
{
	switch(s)
	{
	  case SIGINT:
	    runlevel = 1;
	    break;
	}
}


static void splash_button_cb(
	GtkWidget *splash, GdkEventButton *button, gpointer data
)
{
	runlevel = 1;
}


int main(int argc, char *argv[])
{
	const gchar *splash_xpm_path =
"/usr/share/endeavour2/images/endeavour_splash.xpm";
        gulong	delay = 0l,
		map_int_ms = SPLASH_DEF_MAP_DURATION_MS;
	gint i;
	const gchar *arg, *message = NULL;
	GdkFont *font;
	GtkWidget *splash;
	SplashEffects effects = SPLASH_EFFECTS_FADE_BACKGROUND;

	/* Set the runlevel to starting */
	runlevel = 1;

	/* Set the SIGINT signal callback */
	signal(SIGINT, signal_cb);

	/* Handle the arguments */
	for(i = 1; i < argc; i++)
	{
	    arg = argv[i];
	    if(arg == NULL)
		continue;

	    if(!g_strcasecmp(arg, "--help") ||
	       !g_strcasecmp(arg, "-help") ||
	       !g_strcasecmp(arg, "--h") ||
	       !g_strcasecmp(arg, "-h") ||
	       !g_strcasecmp(arg, "-?")
	    )
	    {
		print_help(argv[0]);
		return(0);
	    }
	    else if(!g_strcasecmp(arg, "--version") ||
	            !g_strcasecmp(arg, "-version")
	    )
	    {
		return(0);
	    }
	    /* Effects */
	    else if(!g_strcasecmp(arg, "--effects") ||
	            !g_strcasecmp(arg, "-e")
	    )
	    {
		i++;
		if(i < argc)
		{
		    arg = argv[i];
		    if(arg == NULL)
			break;

		    if(!g_strcasecmp(arg, "none"))
			effects = SPLASH_EFFECTS_NONE;
		    else if(!g_strcasecmp(arg, "fade_white"))
			effects = SPLASH_EFFECTS_FADE_WHITE;
		    else if(!g_strcasecmp(arg, "fade_black"))
			effects = SPLASH_EFFECTS_FADE_BLACK;
		    else if(!g_strcasecmp(arg, "fade_background"))
			effects = SPLASH_EFFECTS_FADE_BACKGROUND;
		}
		else
		{
		    g_printerr(
"%s: Requires argument.\n",
			arg
		    );
		    return(2);
		}
	    }
	    /* Map Interval */
	    else if(!g_strcasecmp(arg, "--map-int") ||
	            !g_strcasecmp(arg, "-i")
	    )
	    {
		i++;
		if(i < argc)
		{
		    map_int_ms = ATOL(argv[i]);
		}
		else
		{
		    g_printerr(
"%s: Requires argument.\n",
			arg
		    );
		    return(2);
		}
	    }
	    /* Message */
	    else if(!g_strcasecmp(arg, "--message") ||
	            !g_strcasecmp(arg, "-m")
	    )
	    {
		i++;
		if(i < argc)
		{
		    message = argv[i];
		}
		else
		{
		    g_printerr(
"%s: Requires argument.\n",
			arg
		    );
		    return(2);
		}
	    }
	    /* Delay */
	    else if(!g_strcasecmp(arg, "--delay") ||
	            !g_strcasecmp(arg, "-d")
	    )
	    {
		i++;
		if(i < argc)
		{
		    delay = ATOL(argv[i]);
		}
		else
		{
		    g_printerr(
"%s: Requires argument.\n",
			arg
		    );
		    return(2);
		}
	    }
	    /* Splash XPM Path */
	    else if((*arg != '+') && (*arg != '-'))
	    {
		splash_xpm_path = arg;
	    }
	    else
	    {
		g_printerr(
"%s: Unsupported argument.\n",
		    arg
		);
		return(2);
	    }
	}


	/* Initialize GTK */
	gtk_init_check(&argc, &argv);

	/* Initialize the GDK RGB buffers needed by the splash */
	gdk_rgb_init();

	/* Create a new splash */
	splash = splash_new();
	splash_set_button_cb(
	    splash,
	    splash_button_cb, NULL
	);
	SplashSetXPMFile(
	    splash,
	    splash_xpm_path
	);
	font = gdk_font_load(
"-adobe-helvetica-bold-r-normal-*-24-*-*-*-p-*-iso8859-1"
	);
	if(font != NULL)
	{
	    splash_set_message_font(splash, font);
	    gdk_font_unref(font);
	}
	splash_set_message_justification(splash, GTK_JUSTIFY_CENTER);
	splash_set_message_position(splash, GTK_POS_BOTTOM);

	/* Map the splash */
	splash_map(
	    splash,
	    effects,
	    map_int_ms,
	    GTK_WIN_POS_CENTER
	);

	if(message != NULL)
	    splash_update_message(
		splash,
		-1.0f,
		message
	    );

	/* Set the runlevel to running */
	runlevel = 2;

	/* Redraw the splash and manage any GTK events */
	splash_update(splash, TRUE);

	/* Wait for the delay */
	if(delay > 0l)
	{
	    sleep(delay);
	}
	else
	{
	    /* Wait for SIGINT */
	    while(runlevel >= 2)
	    {
		while(gtk_events_pending() > 0)
		    gtk_main_iteration();
	    }
	}

	/* Redraw the splash and manage any GTK events */
	splash_update(splash, TRUE);

	/* Unmap the splash */
	splash_unmap(
	    splash,
	    effects,
	    map_int_ms
	);

	/* Delete the splash */
	splash = splash_unref(splash);

	/* Manage any pending GTK events */
	while(gtk_events_pending() > 0)
	    gtk_main_iteration();

	return(0);
}
