/*
	Recycle Bin List

	Lists the contents of the recycle bin, works similarly to ls.

	Use "recover" to recover a recycled object or "purge" to
	permanently purge a recycled object.
 */

#include <stdlib.h>
#include <fnmatch.h>				/* For fnmatch() */
#include <glib.h>
#include <endeavour2.h>
#include "../config.h"


/*
 *	RLS Environment Variable Names:
 */
#define RLS_ENV_NAME_COLUMNS		"COLUMNS"
#define RLS_ENV_NAME_RLS_COLORS		"RLS_COLORS"
#define RLS_ENV_NAME_LS_COLORS		"LS_COLORS"


/*
 *	Color Codes:
 */
typedef enum {
	RLS_COLOR_CODE_UNKNOWN,
	RLS_COLOR_CODE_REGULAR,
	RLS_COLOR_CODE_DIRECTORY,
	RLS_COLOR_CODE_LINK,
	RLS_COLOR_CODE_FIFO,
	RLS_COLOR_CODE_DEVICE_BLOCK,
	RLS_COLOR_CODE_DEVICE_CHARACTER,
	RLS_COLOR_CODE_SOCKET,
	RLS_COLOR_CODE_EXECUTABLE
} RLSColorCodes;

/*
 *	Sort Methods:
 */
typedef enum {
	RLS_SORT_NONE,				/* Order of deletion (same
						 * as unsorted) */
	RLS_SORT_NAME,
	RLS_SORT_INDEX,
	RLS_SORT_SIZE,
	RLS_SORT_LAST_ACCESSED_TIME,
	RLS_SORT_LAST_MODIFIED_TIME,
	RLS_SORT_LAST_CHANGED_TIME
} RLSSort;

/*
 *	List Formats:
 */
typedef enum {
	RLS_LIST_FORMAT_TABLE_COLUMN,		/* Table ordered by column */
	RLS_LIST_FORMAT_TABLE_ROW,		/* Table ordered by row */
	RLS_LIST_FORMAT_SINGLE_COLUMN,		/* One object per line */
	RLS_LIST_FORMAT_SINGLE_LINE_COMMA,	/* Single continuous line with
						 * ',' separated object names */
	RLS_LIST_FORMAT_LONG			/* ls's long list format */
} RLSListFormat;

/*
 *	Print Options:
 */
typedef enum {
	RLS_PRINT_COLOR			= (1 << 0),
	RLS_PRINT_NAME_WITH_INDICIES	= (1 << 1),
	RLS_PRINT_ORIGINAL_LOCATION	= (1 << 2),
	RLS_PRINT_FULL_TIME		= (1 << 3),
	RLS_PRINT_UID_GID_NAMES		= (1 << 4),
	RLS_PRINT_FRIENDLY_SIZES	= (1 << 5),
	RLS_PRINT_FRIENDLY_SIZES_1000	= (1 << 6),	/* Use 1000 sized
							 * blocks instead of
							 * 1024 */
	RLS_PRINT_GROUP			= (1 << 7)
} RLSPrintOptions;

/*
 *	Color Code:
 */
typedef struct {
	gint		code,
			attributes;
} RLSColorCode;
#define RLS_COLOR_CODE(p)		((RLSColorCode *)(p))

/*
 *	Object:
 */
typedef struct {
	gchar		*name,
			*name_index;		/* "name (#12345)" */
	gulong		index;
	EDVRecycledObject	*recycled_object;
	gchar		*original_path;		/* Original location without the
						 * object's name */
} RLSObject;
#define RLS_OBJ(p)			((RLSObject *)(p))


/* Print Help */
static void print_help(const gchar *prog_name);

/* Sort Callbacks */
static int rls_obj_sort_name_cb(const void *a, const void *b);
static int rls_obj_sort_name_rev_cb(const void *a, const void *b);
static int rls_obj_sort_index_cb(const void *a, const void *b);
static int rls_obj_sort_index_rev_cb(const void *a, const void *b);
static int rls_obj_sort_size_cb(const void *a, const void *b);
static int rls_obj_sort_size_rev_cb(const void *a, const void *b);
static int rls_obj_sort_accessed_time_cb(const void *a, const void *b);
static int rls_obj_sort_accessed_time_rev_cb(const void *a, const void *b);
static int rls_obj_sort_modified_time_cb(const void *a, const void *b);
static int rls_obj_sort_modified_time_rev_cb(const void *a, const void *b);
static int rls_obj_sort_changed_time_cb(const void *a, const void *b);
static int rls_obj_sort_changed_time_rev_cb(const void *a, const void *b);

/* RLS Color Codes */
static RLSColorCode *RLSColorCodes_new(void);
static void RLSColorCodes_delete(RLSColorCode *c);

/* RLS Object */
static RLSObject *rls_obj_new(void);
static void rls_obj_delete(RLSObject *obj);
static GList *rls_get_listing(
	EDVContext *ctx,
	gint *nobjs_rtn,
	gulong *total_size_rtn,
	const RLSSort sort,
	const gboolean sort_reversed
);

/* Printing */
static gint rls_print_obj_name(
	RLSObject *obj,
	GList *color_codes_list,
	const RLSPrintOptions print_options
);
static void rls_print_obj_size(
	const gulong x, const RLSPrintOptions print_options
);
static void rls_print_obj_list_table(
	EDVContext *ctx,
	GList *rls_objects_list, const gint nobjs,
	GList *color_codes_list,
	const gint column_width,
	const gboolean order_by_row,
	const RLSPrintOptions print_options
);
static void rls_print_obj_list_single_column(
	EDVContext *ctx,
	GList *rls_objects_list, const gint nobjs,
	GList *color_codes_list,
	const RLSPrintOptions print_options
);
static void rls_print_obj_list_comma(
	EDVContext *ctx,
	GList *rls_objects_list, const gint nobjs,
	GList *color_codes_list,
	const RLSPrintOptions print_options
);
static void rls_print_obj_list_long(
	EDVContext *ctx,
	GList *rls_objects_list, const gint nobjs, const gulong total_size,
	GList *color_codes_list,
	const gint column_width,
	const RLSSort sort,
	const RLSPrintOptions print_options
);
static void rls_print_obj_list(
	EDVContext *ctx,
	GList *rls_objects_list, const gint nobjs, const gulong total_size,
	GList *color_codes_list,
	const gint column_width,
	const RLSSort sort,
	const RLSListFormat list_format,
	const RLSPrintOptions print_options
);

static gint rls(const gint argc, const gchar **argv);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Prints the help message.
 */
static void print_help(const gchar *prog_name)
{
	g_print(   
"Usage: %s [object(s)...] [options]\n",
		prog_name
	);

	g_print(
		"%s",
"\n\
    The [object(s)...] specifies one or more recycled objects to\n\
    list.\n\
\n\
    Note that if you specify wildcards (* or ?) then you need to put\n\
    them between double quotes (\") such as \"*\" or else the shell\n\
    will expand them as regular object names instead of recycled\n\
    object names.\n\
\n\
    The [options] can be any of the following:\n\
\n\
	-1                      Print single column list format.\n\
	-C                      Print table format ordered by column\n\
				(default).\n\
	-x                      Print table format ordered by row.\n\
	-m                      Print a single line comma-separated\n\
				object names.\n\
	-l                      Print long list format (similar to\n\
				ls's long list format).\n\
\n\
	-U                      Sort by order of deletion (same as\n\
				unsorted).\n\
	-i                      Sort by index.\n\
	-S                      Sort by size.\n\
	-u                      Sort by last accessed time.\n\
	-t                      Sort by last modified time.\n\
	-c                      Sort by last changed time.\n\
	--reverse               Sort reversed.\n\
	-r                      Same as --reverse.\n\
\n\
	--color <when>          Use color to distinguish object types,\n\
				valid values are; never, always, auto.\n\
				Color codes are defined in the\n\
				environment variables RLS_COLORS\n\
				(checked first) or LS_COLORS.\n\
	--clean-names           Print names without indices.\n\
	-N                      Same as --clean-names.\n\
	--original-location     Print the full path to the original\n\
				object.\n\
	-O                      Same as --original-location.\n\
	--full-time             Print full date and time when using\n\
				the -l long list format.\n\
	--numeric-uid-gid       Print numeric UID and GID values\n\
				instead of names.\n\
	-n                      Same as --numeric-uid-gid.\n\
	--human-readable        Print sizes in human readable format\n\
				using 1024 size blocks (i.e. 1K 234M\n\
				2G).\n\
	-h                      Same as --human-readable.\n\
	--si                    Print sizes in human readable format\n\
				using 1000 size blocks (i.e. 1K 234M\n\
				2G).\n\
	-H                      Same as --si.\n\
\n\
	--no-group              Do not print group information.\n\
	-G                      Same as --no-group.\n\
\n\
	--width <i>             Specifies the width of the terminal.\n\
	-w                      Same as --width.\n\
\n\
	--force                 List even if the recycle bin is locked,\n\
				in which the information may not be\n\
				accurate.\n\
\n\
	--help                  Prints this help screen and exits.\n\
	--version               Prints version information and exits.\n\
\n\
    Return values:\n\
\n\
	0       Success.\n\
	1       General error.\n\
	2       Invalid value.\n\
	3       Systems error or memory allocation error.\n\
	4       User aborted.\n\
	5       User responded with \"no\" to a query.\n\
	6       Busy/retry.\n\
	7       Function not supported.\n\
\n\
    To purge recycled objects, use \"purge\".\n\
    To recover recycled objects, use \"recover\".\n\
    To recycle objects, use \"recycle\".\n\
\n"
	);
}


/*
 *	Sort by name callback.
 */
static int rls_obj_sort_name_cb(const void *a, const void *b)
{
	const RLSObject	*oa = *((const RLSObject **)a),
			*ob = *((const RLSObject **)b);
	return(strcmp(oa->name, ob->name));
} 
static int rls_obj_sort_name_rev_cb(const void *a, const void *b)
{
	const RLSObject	*oa = *((const RLSObject **)a),
			*ob = *((const RLSObject **)b);
	return(strcmp(ob->name, oa->name));
} 

/*
 *	Sort by index callback.
 */
static int rls_obj_sort_index_cb(const void *a, const void *b)
{
	const RLSObject	*oa = *((const RLSObject **)a),
			*ob = *((const RLSObject **)b);
	return(oa->index >= ob->index);
}
static int rls_obj_sort_index_rev_cb(const void *a, const void *b)
{
	const RLSObject	*oa = *((const RLSObject **)a),
			*ob = *((const RLSObject **)b);
	return(ob->index >= oa->index);
}

/*
 *	Sort by size callback.
 */
static int rls_obj_sort_size_cb(const void *a, const void *b)
{
	const RLSObject	*oa = *((const RLSObject **)a),
			*ob = *((const RLSObject **)b);
	return(oa->recycled_object->size <= ob->recycled_object->size);
}
static int rls_obj_sort_size_rev_cb(const void *a, const void *b)
{
	const RLSObject	*oa = *((const RLSObject **)a),
			*ob = *((const RLSObject **)b);
	return(ob->recycled_object->size <= oa->recycled_object->size);
}

/*
 *	Sort by last accessed time callback.
 */
static int rls_obj_sort_accessed_time_cb(const void *a, const void *b)
{
	const RLSObject	*oa = *((const RLSObject **)a),
			*ob = *((const RLSObject **)b);
	return(oa->recycled_object->access_time <= ob->recycled_object->access_time);
}
static int rls_obj_sort_accessed_time_rev_cb(const void *a, const void *b)
{
	const RLSObject	*oa = *((const RLSObject **)a),
			*ob = *((const RLSObject **)b);
	return(ob->recycled_object->access_time <= oa->recycled_object->access_time);
}

/*
 *	Sort by last modified time callback.
 */
static int rls_obj_sort_modified_time_cb(const void *a, const void *b)
{
	const RLSObject	*oa = *((const RLSObject **)a),
			*ob = *((const RLSObject **)b);
	return(oa->recycled_object->modify_time <= ob->recycled_object->modify_time);
}
static int rls_obj_sort_modified_time_rev_cb(const void *a, const void *b)
{
	const RLSObject	*oa = *((const RLSObject **)a),
			*ob = *((const RLSObject **)b);
	return(ob->recycled_object->modify_time <= oa->recycled_object->modify_time);
}

/*
 *	Sort by last changed time callback.
 */
static int rls_obj_sort_changed_time_cb(const void *a, const void *b)
{
	const RLSObject	*oa = *((const RLSObject **)a),
			*ob = *((const RLSObject **)b);
	return(oa->recycled_object->change_time <= ob->recycled_object->change_time);
}
static int rls_obj_sort_changed_time_rev_cb(const void *a, const void *b)
{
	const RLSObject	*oa = *((const RLSObject **)a),
			*ob = *((const RLSObject **)b);
	return(ob->recycled_object->change_time <= oa->recycled_object->change_time);
}


/*
 *	Creates a new color code.
 */
static RLSColorCode *RLSColorCodes_new(void)
{
	return(RLS_COLOR_CODE(g_malloc0(sizeof(RLSColorCode))));
}

/*
 *	Deletes the color code.
 */
static void RLSColorCodes_delete(RLSColorCode *c)
{
	if(c == NULL)
		return;

	g_free(c);
}


/*
 *	Creates a new object.
 */
static RLSObject *rls_obj_new(void)
{
	return(RLS_OBJ(g_malloc0(sizeof(RLSObject))));
}

/*
 *	Deletes the object.
 */
static void rls_obj_delete(RLSObject *obj)
{
	if(obj == NULL)
		return;

	g_free(obj->name);
	g_free(obj->name_index);
	g_free(obj->original_path);
	edv_recycled_object_delete(obj->recycled_object);
	g_free(obj);
}


/*
 *	Gets a listing of all recycled objects.
 *
 *	The sort specifies the sort order of the returned list.
 */
static GList *rls_get_listing(
	EDVContext *ctx,
	gint *nobjs_rtn,
	gulong *total_size_rtn,
	const RLSSort sort,
	const gboolean sort_reversed
)
{
	const gchar *index_path = edv_get_s(ctx, EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);
	gint nobjs;
	GList *rls_objects_list;
	EDVRecycledObject *obj;
	EDVRecycleBinIndex *rp;
	RLSObject *rls_obj;
	int (*sort_func)(const void *, const void *) = NULL;

	if(nobjs_rtn != NULL)
		*nobjs_rtn = 0;
	if(total_size_rtn != NULL)
		*total_size_rtn = 0l;

	/* Get the stats of all the recycled objects */
	rp = edv_recycle_bin_index_open(index_path);
	nobjs = 0;
	rls_objects_list = NULL;
	for(obj = edv_recycle_bin_index_next(rp);
		obj != NULL;
		obj = edv_recycle_bin_index_next(rp)
	)
	{
		rls_obj = rls_obj_new();
		if(rls_obj == NULL)
			break;

		rls_obj->name = STRDUP(obj->name);
		rls_obj->index = obj->index;
		rls_obj->name_index = g_strdup_printf(
			"%s (#%ld)",
			obj->name,
			obj->index
		);
		rls_obj->recycled_object = edv_recycled_object_copy(obj);
		rls_obj->original_path = STRDUP(obj->original_path);

		if(total_size_rtn != NULL)
			*total_size_rtn = *total_size_rtn + obj->size;

		rls_objects_list = g_list_append(
			rls_objects_list,
			rls_obj
		);
		nobjs++;
	}

	edv_recycle_bin_index_close(rp);

	if(nobjs_rtn != NULL)
		*nobjs_rtn = nobjs;

	/* Sort the pointer array list */
	switch(sort)
	{
	  case RLS_SORT_NONE:
		sort_func = NULL;
		break;
	  case RLS_SORT_NAME:
		sort_func = sort_reversed ?
			rls_obj_sort_name_rev_cb : rls_obj_sort_name_cb;
		break;
	  case RLS_SORT_INDEX:
		sort_func = sort_reversed ?
			rls_obj_sort_index_rev_cb : rls_obj_sort_index_cb;
		break;
	  case RLS_SORT_SIZE:
		sort_func = sort_reversed ?
			rls_obj_sort_size_rev_cb : rls_obj_sort_size_cb;
		break;
	  case RLS_SORT_LAST_ACCESSED_TIME:
		sort_func = sort_reversed ?
			rls_obj_sort_accessed_time_rev_cb : rls_obj_sort_accessed_time_cb;
		break;
	  case RLS_SORT_LAST_MODIFIED_TIME:
		sort_func = sort_reversed ?
			rls_obj_sort_modified_time_rev_cb : rls_obj_sort_modified_time_cb;
		break;
	  case RLS_SORT_LAST_CHANGED_TIME:
		sort_func = sort_reversed ?
			rls_obj_sort_changed_time_rev_cb : rls_obj_sort_changed_time_cb;
		break;
	}
	if(sort_func != NULL)
	{
		gint i;
		GList *glist = rls_objects_list;
		RLSObject **list = (RLSObject **)g_malloc(nobjs * sizeof(RLSObject *));
		for(i = 0; (i < nobjs) && (glist != NULL); i++)
		{
			list[i] = (RLSObject *)glist->data;
			glist = g_list_next(glist);
		}
		while(i < nobjs)
		{
			list[i] = NULL;
			i++;
		}

		(void)qsort(
			list, nobjs, sizeof(RLSObject *),
			sort_func
		);

		glist = rls_objects_list;
		for(i = 0; (i < nobjs) && (glist != NULL); i++)
		{
			glist->data = list[i];
			glist = g_list_next(glist);
		}

		g_free(list);
	}

	return(rls_objects_list);
}


/*
 *	Returns a string describing the object's name.
 *
 *	If print_options specifies the RLS_PRINT_COLOR option then
 *	the object's name will be printed in the appropriate
 *	color based on the object's type.
 *
 *	If print_options specifies the RLS_PRINT_ORIGINAL_LOCATION
 *	option then the object's original location will be printed
 *	in front with a directory deliminator added.
 *
 *	If print_options specifies the RLS_PRINT_NAME_WITH_INDICIES
 *	option then the object's index will be printed after the
 *	name in the format "name (#12345)"
 *
 *	Returns the number of characters printed.
 */
static gint rls_print_obj_name(
	RLSObject *obj,
	GList *color_codes_list,
	const RLSPrintOptions print_options
)
{
	const EDVObjectType type = obj->recycled_object->type;
	const EDVPermissionFlags permissions = obj->recycled_object->permissions;
	gchar		*color_on_str = g_strdup(""),
			*color_off_str = g_strdup("");

	if(print_options & RLS_PRINT_COLOR)
	{
#define MK_COLOR_STR(_ci_)	{	\
 RLSColorCode *c = RLS_COLOR_CODE(g_list_nth_data( \
  color_codes_list, (_ci_)		\
 ));					\
 if(c != NULL) {			\
  g_free(color_on_str);			\
  color_on_str = g_strdup_printf(	\
   "\033[%i;%im",			\
   c->attributes, c->code		\
  );					\
 }					\
}

		if(FALSE)
		{

		}
		else if(type == EDV_OBJECT_TYPE_DIRECTORY)
		{
			MK_COLOR_STR(RLS_COLOR_CODE_DIRECTORY);
		}
		else if(type == EDV_OBJECT_TYPE_LINK)
		{
			MK_COLOR_STR(RLS_COLOR_CODE_LINK);
		}
		else if(type == EDV_OBJECT_TYPE_FIFO)
		{
			MK_COLOR_STR(RLS_COLOR_CODE_FIFO);
		}
		else if(type == EDV_OBJECT_TYPE_DEVICE_BLOCK)
		{
			MK_COLOR_STR(RLS_COLOR_CODE_DEVICE_BLOCK);
		}
		else if(type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
		{
			MK_COLOR_STR(RLS_COLOR_CODE_DEVICE_CHARACTER);
		}
		else if(type == EDV_OBJECT_TYPE_SOCKET)
		{
			MK_COLOR_STR(RLS_COLOR_CODE_SOCKET);
		}
		else if((permissions & EDV_PERMISSION_UX) ||
				(permissions & EDV_PERMISSION_GX) ||
				(permissions & EDV_PERMISSION_OX)
		)
		{
			MK_COLOR_STR(RLS_COLOR_CODE_EXECUTABLE);
		}

		g_free(color_off_str);
		color_off_str = STRDUP("\033[0;0m");

#undef MK_COLOR_STR
	}

	if(print_options & RLS_PRINT_ORIGINAL_LOCATION)
	{
		if(print_options & RLS_PRINT_NAME_WITH_INDICIES)
		{
			g_print(
"%s%s%s%s%s (#%ld)",
				color_on_str,
				obj->original_path,
				G_DIR_SEPARATOR_S,
				obj->name,
				color_off_str,
				obj->index
			);
			g_free(color_on_str);
			g_free(color_off_str);
			return(
				STRLEN(obj->original_path) +
				STRLEN(G_DIR_SEPARATOR_S) +
				STRLEN(obj->name_index)
			);
		}
		else
		{
			g_print(
"%s%s%s%s%s",
				color_on_str,
				obj->original_path,
				G_DIR_SEPARATOR_S,
				obj->name,
				color_off_str
			);
			g_free(color_on_str);
			g_free(color_off_str);
			return(
				STRLEN(obj->original_path) +
				STRLEN(G_DIR_SEPARATOR_S) +
				STRLEN(obj->name)
			);
		}
	}
	else
	{
		if(print_options & RLS_PRINT_NAME_WITH_INDICIES)
		{
			g_print(
"%s%s%s (#%ld)",
				color_on_str,
				obj->name,
				color_off_str,
				obj->index
			);
			g_free(color_on_str);
			g_free(color_off_str);
			return(STRLEN(obj->name_index));
		}
		else
		{
			g_print(
"%s%s%s",
				color_on_str,
				obj->name,
				color_off_str
			);
			g_free(color_on_str);
			g_free(color_off_str);
			return(STRLEN(obj->name));
		}
	}
}


/*
 *	Prints the size.
 */
static void rls_print_obj_size(
	const gulong x, const RLSPrintOptions print_options
)
{
	if(print_options & RLS_PRINT_FRIENDLY_SIZES)
	{
		gchar *s = g_strconcat(
			"          ",
			edv_str_size_format(
				x,
				EDV_SIZE_FORMAT_HUMAN_READABLE,
				1000l,			/* Block size */
				',',			/* Deliminator character */
				TRUE			/* Allow unit conversions */
			),
			NULL
		);
		if(s != NULL)
		{
			g_print("%s", s + (strlen(s) - 11));
			g_free(s);
		}
	}
	else if(print_options & RLS_PRINT_FRIENDLY_SIZES_1000)
	{
		gchar *s = g_strconcat(
			"          ",
			edv_str_size_format(
				x,
				EDV_SIZE_FORMAT_HUMAN_READABLE,
				1024l,			/* Block size */
				',',			/* Deliminator character */
				TRUE			/* Allow unit conversions */
			),
			NULL
		);
		if(s != NULL)
		{
			g_print("%s", s + (strlen(s) - 11));
			g_free(s);
		}
	}
	else
	{
		gchar *s = g_strdup_printf(
			"          %ld",
			x
		);
		if(s != NULL)
		{
			g_print("%s", s + (strlen(s) - 11));
			g_free(s);
		}
	}
}

/*
 *	Prints the list of objects using the short list format.
 *
 *	If order_by_row is TRUE then the objects will be printed in
 *	the order of each row. Otherwise the objects will be printed
 *	in the order of each column.
 */
static void rls_print_obj_list_table(
	EDVContext *ctx,
	GList *rls_objects_list, const gint nobjs,
	GList *color_codes_list,
	const gint column_width,
	const gboolean order_by_row,
	const RLSPrintOptions print_options
)
{
	gint		len,
			longest_name_len,
			per_column_width,
			ncolumns,
			nobjs_per_column;
	GList *glist;
	RLSObject *obj;

	/* Find the object with the longest name */
	longest_name_len = 0;
	for(glist = rls_objects_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		obj = RLS_OBJ(glist->data);
		if(obj == NULL)
			continue;

		len = ((print_options & RLS_PRINT_ORIGINAL_LOCATION) ?
			(STRLEN(obj->original_path) + STRLEN(G_DIR_SEPARATOR_S)) : 0) +
			((print_options & RLS_PRINT_NAME_WITH_INDICIES) ?
			STRLEN(obj->name_index) : STRLEN(obj->name)) +
			1;
		if(len > longest_name_len)
			longest_name_len = len;
	}

	/* Calculate the number of columns that we can fit based on
	 * the longest name length and the column width
	 */
	ncolumns = MAX((column_width / longest_name_len), 1);

	/* Calculate the number of objects to be displayed per
	 * column
	 */
	nobjs_per_column = (nobjs / ncolumns) +
		(((nobjs % ncolumns) != 0) ? 1 : 0);
	if(nobjs_per_column <= 1)
		nobjs_per_column = 1;

	/* Calculate the width of each column */
	per_column_width = MAX((column_width / ncolumns), 1) +
		(((column_width % ncolumns) != 0) ? 1 : 0);

	/* Print the list */
	if(order_by_row)
	{
		/* Print ordered by row */
		gint	i, j, k,
			len;
		for(i = 0; i < nobjs; i += ncolumns)
		{
			for(j = 0; j < ncolumns; j++)
			{
				k = i + j;
				if(k >= nobjs)
					break;

				obj = RLS_OBJ(g_list_nth_data(rls_objects_list, k));
				if(obj == NULL)
					continue;

				len = rls_print_obj_name(
					obj,
					color_codes_list,
					print_options
				);
				if((j + 1) < ncolumns)
				{
					gint n;
					for(n = len; n < per_column_width; n++)
						g_print(" ");
				}
			}

			g_print("\n");
		}
	}
	else
	{
		/* Print ordered by column */
		gint		i, j, k,
				len,
				column_num;

		for(i = 0; i < nobjs_per_column; i++)
		{
			for(j = 0, column_num = 0;
			    j < nobjs;
			    j += nobjs_per_column, column_num++
			)
			{
				k = i + j;
				if(k >= nobjs)
					break;

				obj = RLS_OBJ(g_list_nth_data(rls_objects_list, k));
				if(obj == NULL)
					continue;

				len = rls_print_obj_name(
					obj,
					color_codes_list,
					print_options
				);
				if((column_num + 1) < ncolumns)
				{
					gint n;
					for(n = len; n < per_column_width; n++)
						g_print(" ");
				}
			}

			g_print("\n");
		}
	}
}

/*
 *	Prints the list of objects using the single list format.
 */
static void rls_print_obj_list_single_column(
	EDVContext *ctx,
	GList *rls_objects_list, const gint nobjs,
	GList *color_codes_list,
	const RLSPrintOptions print_options
)
{
	GList *glist;
	RLSObject *obj;

	/* Print each object */
	for(glist = rls_objects_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		obj = RLS_OBJ(glist->data);
		if(obj == NULL)
			continue;

		rls_print_obj_name(
			obj,
			color_codes_list,
			print_options
		);
		g_print("\n");
	}
}

/*
 *	Prints the recycled object on a single line separated by commas.
 */
static void rls_print_obj_list_comma(
	EDVContext *ctx,
	GList *rls_objects_list, const gint nobjs,
	GList *color_codes_list,
	const RLSPrintOptions print_options
)
{
	gint nobjs_printed = 0;
	GList *glist;
	RLSObject *obj;

	/* Print each object */
	for(glist = rls_objects_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		obj = RLS_OBJ(glist->data);
		if(obj == NULL)
			continue;

		if(nobjs_printed > 0)
			g_print(",");

		rls_print_obj_name(
			obj,
			color_codes_list,
			print_options
		);

		nobjs_printed++;
	}

	g_print("\n");
}

/*
 *	Prints the recycled object using the long list format.
 */
static void rls_print_obj_list_long(
	EDVContext *ctx,
	GList *rls_objects_list, const gint nobjs, const gulong total_size,
	GList *color_codes_list,
	const gint column_width,
	const RLSSort sort,
	const RLSPrintOptions print_options
)
{
	gint		len,
			longest_name_len;
	gulong t;
	const gulong ct = edv_time();
	gchar *s;
	GList *glist;
	EDVObjectType type;
	EDVPermissionFlags permissions;
	EDVRecycledObject *recycled_object;
	RLSObject *obj;

	/* Find the object with the longest name */
	longest_name_len = 0;
	for(glist = rls_objects_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		obj = RLS_OBJ(glist->data);
		if(obj == NULL)
			continue;

		len = STRLEN(obj->name);
		if(len > longest_name_len)
			longest_name_len = len;
	}

	/* Print heading only if the total_size is positive, which
	 * suggests that all the objects were requested to be printed
	 */
	if(total_size > 0l)
	{
		g_print(
			"Total %i  Size ",
			nobjs
		);
		rls_print_obj_size(total_size, print_options);
		g_print("\n");
	}

	/* Print each object */
	for(glist = rls_objects_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		obj = RLS_OBJ(glist->data);
		if(obj == NULL)
			continue;

		recycled_object = obj->recycled_object;
		type = recycled_object->type;
		permissions = recycled_object->permissions;

		/* Type */
		if(type == EDV_OBJECT_TYPE_FILE)
			g_print("-");
		else if(type == EDV_OBJECT_TYPE_DIRECTORY)
			g_print("d");
		else if(type == EDV_OBJECT_TYPE_LINK)
			g_print("l");
		else if(type == EDV_OBJECT_TYPE_FIFO)
			g_print("p");
		else if(type == EDV_OBJECT_TYPE_DEVICE_BLOCK)
			g_print("b");
		else if(type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
			g_print("c");
		else if(type == EDV_OBJECT_TYPE_SOCKET)
			g_print("s");
		else
			g_print("-");

		/* Permissions */
		g_print(
"%c%c%c%c%c%c%c%c%c   ",
			(permissions & EDV_PERMISSION_UR) ? 'r' : '-',
			(permissions & EDV_PERMISSION_UW) ? 'w' : '-',
			(permissions & EDV_PERMISSION_UX) ? 'x' :
				((permissions & EDV_PERMISSION_SETUID) ? 'S' : '-'),
			(permissions & EDV_PERMISSION_GR) ? 'r' : '-',
			(permissions & EDV_PERMISSION_GW) ? 'w' : '-',
			(permissions & EDV_PERMISSION_GX) ? 'x' :
				((permissions & EDV_PERMISSION_SETGID) ? 'G' : '-'),
			(permissions & EDV_PERMISSION_OR) ? 'r' : '-',
			(permissions & EDV_PERMISSION_OW) ? 'w' : '-',
			(permissions & EDV_PERMISSION_OX) ? 'x' :
				((permissions & EDV_PERMISSION_STICKY) ? 'S' : '-')
		);

		/* User */
		if(ctx != NULL)
		{
			gchar	*user_name = edv_uid_uid_to_name(
					ctx,
					recycled_object->owner_id
				),
				*s = g_strdup_printf(
					"%s      ",
					user_name
				);
			g_free(user_name);
			s[8] = '\0';
			g_print("%s ", s);
			g_free(s);
		}

		/* Group */
		if(print_options & RLS_PRINT_GROUP)
		{
			gchar	*group_name = edv_gid_gid_to_name(
					ctx,
					recycled_object->group_id
				),
				*s = g_strdup_printf(
					"%s      ",
					group_name
				);
			g_free(group_name);
			s[8] = '\0';
			g_print("%s ", s);
			g_free(s);
		}

		/* Size */
		rls_print_obj_size(
			recycled_object->size,
			print_options
		);
		g_print(" ");

		/* Time */
		switch(sort)
		{
		  case RLS_SORT_LAST_ACCESSED_TIME:
			t = (time_t)recycled_object->access_time;
			break;
		  case RLS_SORT_LAST_CHANGED_TIME:
			t = (time_t)recycled_object->change_time;
			break;
		  default:
			t = (time_t)recycled_object->modify_time;
			break;
		}
		if(print_options & RLS_PRINT_FULL_TIME)
		{
			s = g_strdup_printf(
				"%s                         ",
				edv_date_format_absolute_string(
					ctx,
					t,
					"%a %b %e %H:%M:%S %Y"
				)
			);
			if(s != NULL)
				s[24] = '\0';
		}
		else
		{
			/* Time is in the future? */
			if(t > ct)
				s = g_strdup_printf(
					"%s             ",
					edv_date_format_absolute_string(
						ctx,
						t,
						"%b %e %H:%M"
					)
				);
			/* Less than a year old? */
			else if((ct - t) < (365l * 24l * 60l * 60l))
				s = g_strdup_printf(
					"%s             ",
					edv_date_format_absolute_string(
						ctx,
						t,
						"%b %e %H:%M"
					)
				);
			else
				s = g_strdup_printf(
					"%s             ",
					edv_date_format_absolute_string(
						ctx,
						t,
						"%b %e  %Y"
					)
				);
			s[12] = '\0';
		}
		g_print("%s ", s);
		g_free(s);

		/* Name */
		rls_print_obj_name(
			obj,
			color_codes_list,
			print_options
		);
		g_print("\n");
	}
}

/*
 *	Prints the list of objects.
 */
static void rls_print_obj_list(
	EDVContext *ctx,
	GList *rls_objects_list, const gint nobjs, const gulong total_size,
	GList *color_codes_list,
	const gint column_width,
	const RLSSort sort,
	const RLSListFormat list_format,
	const RLSPrintOptions print_options
)
{
	if((rls_objects_list == NULL) || (nobjs <= 0) || (column_width <= 0))
		return;

	switch(list_format)
	{
	    case RLS_LIST_FORMAT_TABLE_COLUMN:
		rls_print_obj_list_table(
			ctx,
			rls_objects_list, nobjs,
			color_codes_list,
			column_width,
			FALSE,			/* Order by column */
			print_options
		);
		break;
	    case RLS_LIST_FORMAT_TABLE_ROW:
		rls_print_obj_list_table(
			ctx,
			rls_objects_list, nobjs,
			color_codes_list,
			column_width,
			TRUE,			/* Order by row */
			print_options
		);
		break;
	    case RLS_LIST_FORMAT_SINGLE_COLUMN:
		rls_print_obj_list_single_column(
			ctx,
			rls_objects_list, nobjs,
			color_codes_list,
			print_options
		);
		break;
	    case RLS_LIST_FORMAT_SINGLE_LINE_COMMA:
		rls_print_obj_list_comma(
			ctx,
			rls_objects_list, nobjs,
			color_codes_list,
			print_options
		);
		break;
	    case RLS_LIST_FORMAT_LONG:
		rls_print_obj_list_long(
			ctx,
			rls_objects_list, nobjs, total_size,
			color_codes_list,
			column_width,
			sort,
			print_options
		);
		break;
	}
}


/*
 *	Recycle Bin List.
 */
static gint rls(const gint argc, const gchar **argv)
{
	gboolean	force = FALSE,
			sort_reversed = FALSE;
	gint		i,
			column_width = ATOI(g_getenv(RLS_ENV_NAME_COLUMNS));
	const gchar *arg;
	GList		*names_list = NULL,
			*color_codes_list = NULL;
	RLSSort sort = RLS_SORT_NAME;
	RLSListFormat list_format = RLS_LIST_FORMAT_TABLE_COLUMN;
	RLSPrintOptions print_options = RLS_PRINT_COLOR |
				RLS_PRINT_NAME_WITH_INDICIES |
				RLS_PRINT_UID_GID_NAMES |
				RLS_PRINT_GROUP;

	/* Initialize the Endeavour2 Context */
	EDVContext *ctx = edv_context_new();
	edv_context_init(ctx, NULL);

#define CLEANUP_RETURN(_v_)	{		\
 if(names_list != NULL) {			\
  g_list_foreach(				\
   names_list, (GFunc)g_free, NULL		\
  );						\
  g_list_free(names_list);			\
 }						\
						\
 if(color_codes_list != NULL) {			\
  g_list_foreach(				\
   color_codes_list,				\
   (GFunc)RLSColorCodes_delete,			\
   NULL						\
  );						\
  g_list_free(color_codes_list);		\
 }						\
						\
 /* Shutdown the Endeavour2 context */		\
 edv_context_delete(ctx);			\
						\
 return(_v_);					\
}

	if(column_width <= 0)
		column_width = 80;

	/* Handle command line arguments */
	for(i = 1; i < argc; i++)
	{
		arg = argv[i];
		if(arg == NULL)
			continue;

		/* Single Column List Format */
		if(!strcmp(arg, "--1") ||
		   !strcmp(arg, "-1")
		)
		{
			list_format = RLS_LIST_FORMAT_SINGLE_COLUMN;
		}
		/* Table Format Ordered By Column */
		else if(!strcmp(arg, "--C") ||
			!strcmp(arg, "-C")
		)
		{
			list_format = RLS_LIST_FORMAT_TABLE_COLUMN;
		}
		/* Table Format Ordered By Row */
		else if(!strcmp(arg, "--x") ||
			!strcmp(arg, "-x")
		)
		{
			list_format = RLS_LIST_FORMAT_TABLE_ROW;
		}
		/* Single Line Comma-Separated */
		else if(!strcmp(arg, "--m") ||
			!strcmp(arg, "-m")
		)
		{
			list_format = RLS_LIST_FORMAT_SINGLE_LINE_COMMA;
		}
		/* Long List Format */
		else if(!strcmp(arg, "--l") ||
			!strcmp(arg, "-l")
		)
		{
			list_format = RLS_LIST_FORMAT_LONG;
		}
		/* Sort By Delete order */
		else if(!strcmp(arg, "--U") ||
			!strcmp(arg, "-U")
		)
		{
			sort = RLS_SORT_NONE;
		}
		/* Sort By Index */
		else if(!strcmp(arg, "--i") ||
			!strcmp(arg, "-i")
		)
		{
			sort = RLS_SORT_INDEX;
		}
		/* Sort By Size */
		else if(!strcmp(arg, "--S") ||
			!strcmp(arg, "-S")
		)
		{
			sort = RLS_SORT_SIZE;
		}
		/* Sort By Last Accessed Time */
		else if(!strcmp(arg, "--u") ||
			!strcmp(arg, "-u")
		)
		{
			sort = RLS_SORT_LAST_ACCESSED_TIME;
		}
		/* Sort By Last Modified Time */
		else if(!strcmp(arg, "--t") ||
			!strcmp(arg, "-t")
		)
		{
			sort = RLS_SORT_LAST_MODIFIED_TIME;
		}
		/* Sort By Last Changed Time */
		else if(!strcmp(arg, "--c") ||
			!strcmp(arg, "-c")
		)
		{
			sort = RLS_SORT_LAST_CHANGED_TIME;
		}
		/* Sort Reversed */
		else if(!g_strcasecmp(arg, "--reverse") ||
			!g_strcasecmp(arg, "-reverse") ||
			!strcmp(arg, "--r") ||
			!strcmp(arg, "-r")
		)
		{
			sort_reversed = TRUE;
		}
		/* Color */
		else if(!g_strcasecmp(arg, "--color") ||
			!g_strcasecmp(arg, "-color")
		)
		{
			i++;
			arg = (i < argc) ? argv[i] : NULL;
			if(arg != NULL)
			{
				if(!g_strcasecmp(arg, "never"))
					print_options &= ~RLS_PRINT_COLOR;
				else if(!g_strcasecmp(arg, "always"))
						print_options |= RLS_PRINT_COLOR;
				else if(!g_strcasecmp(arg, "auto"))
					print_options |= RLS_PRINT_COLOR;
				else
				{
					g_printerr(
"%s: Invalid value, valid values are; never always auto\n",
						argv[i - 1]
					);
					CLEANUP_RETURN(-2);
				}
			}
			else
			{
			    g_printerr(
"%s: Requires argument.\n",
					argv[i - 1]
			);
			CLEANUP_RETURN(-2);
			}
		}
		/* Print Names Without Indicies */
		else if(!g_strcasecmp(arg, "--clean-names") ||
			!g_strcasecmp(arg, "-clean-names") ||
			!strcmp(arg, "--N") ||
			!strcmp(arg, "-N")
		)
		{
			print_options &= ~RLS_PRINT_NAME_WITH_INDICIES;
		}
		/* Print Original Location */
		else if(!g_strcasecmp(arg, "--original-location") ||
			!g_strcasecmp(arg, "-original-location") ||
			!strcmp(arg, "--O") ||
			!strcmp(arg, "-O")
		)
		{
			print_options |= RLS_PRINT_ORIGINAL_LOCATION;
		}
		/* Full Time */
		else if(!g_strcasecmp(arg, "--full-time") ||
			!g_strcasecmp(arg, "-full-time")
		)
		{
			print_options |= RLS_PRINT_FULL_TIME;
		}
		/* Print Numeric UID & GID Values */
		else if(!g_strcasecmp(arg, "--numeric-uid-gid") ||
			!g_strcasecmp(arg, "-numeric-uid-gid") ||
			!strcmp(arg, "--n") ||
			!strcmp(arg, "-n")
		)
		{
			print_options &= ~RLS_PRINT_UID_GID_NAMES;
		}
		/* Print Friendly Sizes */
		else if(!g_strcasecmp(arg, "--human-readable") ||
			!g_strcasecmp(arg, "-human-readable") ||
			!strcmp(arg, "--h") ||
			!strcmp(arg, "-h")
		)
		{
			print_options |= RLS_PRINT_FRIENDLY_SIZES;
		}
		/* Print Friendly Sizes in 1000 blocks*/
		else if(!g_strcasecmp(arg, "--si") ||
			!g_strcasecmp(arg, "-si") ||
			!strcmp(arg, "--H") ||
			!strcmp(arg, "-H")
		)
		{
			print_options |= RLS_PRINT_FRIENDLY_SIZES_1000;
		}
		/* Do Not Print Group */
		else if(!g_strcasecmp(arg, "--no-group") ||
			!g_strcasecmp(arg, "-no-group") ||
			!strcmp(arg, "--G") ||
			!strcmp(arg, "-G")
		)
		{
			print_options &= ~RLS_PRINT_GROUP;
		}
		/* Width */
		else if(!g_strcasecmp(arg, "--width") ||
			!g_strcasecmp(arg, "-width") ||
			!strcmp(arg, "--w") ||
			!strcmp(arg, "-w")
		)
		{
			i++;
			arg = (i < argc) ? argv[i] : NULL;
			if(arg != NULL)
			{
				column_width = MAX(ATOI(arg), 1);
			}
			else
			{
				g_printerr(
"%s: Requires argument.\n",
					argv[i - 1]
				);
				CLEANUP_RETURN(-2);
			}
		}
		/* Force */
		else if(!g_strcasecmp(arg, "--force") ||
			!g_strcasecmp(arg, "-force")
		)
		{
			force = TRUE;
		}
		/* Single character argument? */
		else if((*arg == '-') ? (arg[1] != '-') : FALSE)
		{
			const gchar *v;
			for(v = arg + 1; *v != '\0'; v++)
			{
				switch(*v)
				{
				    case '1':
					list_format = RLS_LIST_FORMAT_SINGLE_COLUMN;
					break;
				    case 'C':
					list_format = RLS_LIST_FORMAT_TABLE_COLUMN;
					break;
				    case 'x':
					list_format = RLS_LIST_FORMAT_TABLE_ROW;
					break;
				    case 'm':
					list_format = RLS_LIST_FORMAT_SINGLE_LINE_COMMA;
					break;
				    case 'l':
					list_format = RLS_LIST_FORMAT_LONG;
					break;
				    case 'U':
					sort = RLS_SORT_NONE;
					break;
				    case 'i':
					sort = RLS_SORT_INDEX;
					break;
				    case 'S':
					sort = RLS_SORT_SIZE;
					break;
				    case 'u':
					sort = RLS_SORT_LAST_ACCESSED_TIME;
					break;
				    case 't':
					sort = RLS_SORT_LAST_MODIFIED_TIME;
					break;
				    case 'c':
					sort = RLS_SORT_LAST_CHANGED_TIME;
					break;
				    case 'r':
					sort_reversed = TRUE; 
					break;
				    case 'N':
					print_options &= ~RLS_PRINT_NAME_WITH_INDICIES;
					break;
				    case 'O':
					print_options |= RLS_PRINT_FULL_TIME;
					break;
				    case 'n':
					print_options &= ~RLS_PRINT_UID_GID_NAMES;
					break;
				    case 'h':
					print_options |= RLS_PRINT_FRIENDLY_SIZES;
					break;
				    case 'H':
					print_options |= RLS_PRINT_FRIENDLY_SIZES_1000;
					break;
				    case 'G':
					print_options &= ~RLS_PRINT_GROUP;
					break;
				    default:
				        g_printerr(
"-%c: Unsupported argument.\n",
					    *v
					);
					CLEANUP_RETURN(-2);
					break;
				}
			}
		}
		/* Object? */
		else if((*arg != '-') && (*arg != '+'))
		{
			names_list = g_list_append(
				names_list,
				g_strdup(arg)
			);
		}
		else
		{
			g_printerr(
"%s: Unsupported argument.\n",
				arg
			);
			CLEANUP_RETURN(-2);
		}
	}

	/* If the Recycle Bin is locked then do not continue unless
	 * forcing
	 */
	if(edv_recycle_bin_index_get_lock(
		edv_get_s(ctx, EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX)
	) != 0)
	{
		if(force)
		{
			g_printerr(
"%s: The recycle bin is currently locked, the information may not be accurate.\n",
				argv[0]
			);
		}
		else
		{
			g_printerr(
"%s: The recycle bin is currently locked, please try again later or use --force.\n",
				argv[0]
			);
			CLEANUP_RETURN(-6);
		}
	}

	/* Build the colors list */
	if(print_options & RLS_PRINT_COLOR)
	{
		RLSColorCode *c;

		/* Create the colors list based on the order of the
		 * RLS_COLOR_CODE_* enumerations
		 */

		/* RLS_COLOR_CODE_UNKNOWN */
		c = RLSColorCodes_new();
		c->code = 0;
		c->attributes = 0;
		color_codes_list = g_list_append(color_codes_list, c);

		/* RLS_COLOR_CODE_REGULAR */
		c = RLSColorCodes_new();
		c->code = 0;
		c->attributes = 0;
		color_codes_list = g_list_append(color_codes_list, c);

		/* RLS_COLOR_CODE_DIRECTORY */
		c = RLSColorCodes_new();
		c->code = 34;
		c->attributes = 1;
		color_codes_list = g_list_append(color_codes_list, c);

		/* RLS_COLOR_CODE_LINK */
		c = RLSColorCodes_new();
		c->code = 36;
		c->attributes = 1;
		color_codes_list = g_list_append(color_codes_list, c);

		/* RLS_COLOR_CODE_FIFO */
		c = RLSColorCodes_new();
		c->code = 31;
		c->attributes = 1;
		color_codes_list = g_list_append(color_codes_list, c);

		/* RLS_COLOR_CODE_DEVICE_BLOCK */
		c = RLSColorCodes_new();
		c->code = 33;
		c->attributes = 1;
		color_codes_list = g_list_append(color_codes_list, c);

		/* RLS_COLOR_CODE_DEVICE_CHARACTER */
		c = RLSColorCodes_new();
		c->code = 33;
		c->attributes = 1;
		color_codes_list = g_list_append(color_codes_list, c);

		/* RLS_COLOR_CODE_SOCKET */
		c = RLSColorCodes_new();
		c->code = 35;
		c->attributes = 1;
		color_codes_list = g_list_append(color_codes_list, c);

		/* RLS_COLOR_CODE_EXECUTABLE */
		c = RLSColorCodes_new();
		c->code = 32;
		c->attributes = 1;
		color_codes_list = g_list_append(color_codes_list, c);

		/* Check if there are user defined colors set in the
		 * environment
		 */
		arg = g_getenv(RLS_ENV_NAME_RLS_COLORS);
		if(arg == NULL)
			arg = g_getenv(RLS_ENV_NAME_LS_COLORS);
		if(arg != NULL)
		{
			/* Get each color entry */
			gchar **strv = g_strsplit(
				arg,
				":",
				-1
			);
			if(strv != NULL)
			{
				const gchar	*type_str,
						*val_str;
				gchar **entry_strv;

				/* Iterate through each color entry */
				for(i = 0; strv[i] != NULL; i++)
				{
					/* Parse this color entry */
					entry_strv = g_strsplit(
						strv[i],
						"=",
						-1
					);
					if(entry_strv == NULL)
						continue;

					/* Type not available? */
					if(entry_strv[0] == NULL)
					{
						g_strfreev(entry_strv);
						continue;
					}
					/* Value not available? */
					if(entry_strv[1] == NULL)
					{
						g_strfreev(entry_strv);
						continue;
					}

					/* Get the type and value */
					type_str = entry_strv[0];
					val_str = entry_strv[1];

#define SET_COLOR(_c_)	{				\
 if(c != NULL) {					\
  gchar **val_strv = g_strsplit(val_str, ";", -1);	\
  if(val_strv != NULL) {				\
   /* Code */						\
   if(val_strv[0] != NULL) {				\
    c->code = (gint)atoi((const char *)val_strv[0]);	\
    /* Attributes */					\
    if(val_strv[1] != NULL)				\
     c->attributes = (gint)atoi((const char *)val_strv[1]); \
   }							\
   g_strfreev(val_strv);				\
  }							\
 }							\
}

					/* Directory */
					if(!g_strcasecmp(type_str, "di"))
					{
						c = RLS_COLOR_CODE(g_list_nth_data(
							color_codes_list,
							RLS_COLOR_CODE_DIRECTORY
						));
						SET_COLOR(c);
					}
					/* Link */
					else if(!g_strcasecmp(type_str, "ln"))
					{
						c = RLS_COLOR_CODE(g_list_nth_data(
							color_codes_list,
							RLS_COLOR_CODE_LINK
						));
						SET_COLOR(c);
					}
					/* FIFO */
					else if(!g_strcasecmp(type_str, "pi"))
					{
						c = RLS_COLOR_CODE(g_list_nth_data(
							color_codes_list,
							RLS_COLOR_CODE_FIFO
						));
						SET_COLOR(c);
					}
					/* Device Block */
					else if(!g_strcasecmp(type_str, "bd"))
					{
						c = RLS_COLOR_CODE(g_list_nth_data(
							color_codes_list,
							RLS_COLOR_CODE_DEVICE_BLOCK
						));
						SET_COLOR(c);
					}
					/* Device Character */
					else if(!g_strcasecmp(type_str, "cd"))
					{
						c = RLS_COLOR_CODE(g_list_nth_data(
							color_codes_list,
							RLS_COLOR_CODE_DEVICE_CHARACTER
						));
						SET_COLOR(c);
					}
					/* Socket */
					else if(!g_strcasecmp(type_str, "sk"))
					{
						c = RLS_COLOR_CODE(g_list_nth_data(
							color_codes_list,
							RLS_COLOR_CODE_SOCKET
						));
						SET_COLOR(c);
					}
					/* Executable */
					else if(!g_strcasecmp(type_str, "ex"))
					{
						c = RLS_COLOR_CODE(g_list_nth_data(
							color_codes_list,
							RLS_COLOR_CODE_EXECUTABLE
						));
						SET_COLOR(c);
					}
#undef SET_COLOR

					g_strfreev(entry_strv);
				}

				g_strfreev(strv);
			}
		}
	}

	/* List specified objects? */
	if(names_list != NULL)
	{
		gint		nobjs,
				nmatches;
		const gchar	*arg,
				*name;
		GList		*glist,
				*glist2,
				*rls_objects_list = rls_get_listing(
					ctx,
					&nobjs,
					NULL,	/* No total size return */
					sort,
					sort_reversed
				),
				*rec_obj_print_list = NULL;
		RLSObject *obj;

		/* Iterate through each specified object and create
		 * a list of objects to print
		 */
		for(glist = names_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			arg = (const gchar *)glist->data;
			if(arg == NULL)
				continue;

			/* Check if this argument matches one of the
			 * names of the recycled objects
			 */
			nmatches = 0;
			for(glist2 = rls_objects_list;
			    glist2 != NULL;
			    glist2 = g_list_next(glist2)
			)
			{
				obj = RLS_OBJ(glist2->data);
				if(obj == NULL)
					continue;

				name = obj->name;
				if(name == NULL)
					continue;

				if(fnmatch(
					(const char *)arg,
					(const char *)name,
					0	/* No options */
				) == 0)
				{
					/* Add this object to the list
					 * of objects to print
					 */
					rec_obj_print_list = g_list_append(
						rec_obj_print_list,
						obj
					);
					nmatches++;
				}
			}
			if(nmatches == 0)
				g_printerr(
"%s: No such object.\n",
					arg
				);
		}

		/* Print the list of objects to print */
		rls_print_obj_list(
			ctx,
			rec_obj_print_list,
			g_list_length(rec_obj_print_list),
			0l,			/* No size */
			color_codes_list,
			column_width,
			sort,
			list_format,
			print_options
		);

		/* Delete the object lists */
		g_list_free(rec_obj_print_list);

		if(rls_objects_list != NULL)
		{
			g_list_foreach(
				rls_objects_list,
				(GFunc)rls_obj_delete,
				NULL
			);
			g_list_free(rls_objects_list);
		}
	}
	else
	{
		/* Print the list of all recycled objects */
		gint nobjs;
		gulong total_size;
		GList *rls_objects_list = rls_get_listing(
			ctx,
			&nobjs,
			&total_size,
			sort,
			sort_reversed
		);
		rls_print_obj_list(
			ctx,
			rls_objects_list,
			nobjs,
			total_size,
			color_codes_list,
			column_width,
			sort,
			list_format,
			print_options
		);

		/* Delete the object list */
		g_list_foreach(rls_objects_list, (GFunc)rls_obj_delete, NULL);
		g_list_free(rls_objects_list);
	}

	/* Flush any pending Endeavour2 operations */
	edv_context_sync(ctx);

	CLEANUP_RETURN(0);
#undef CLEANUP_RETURN
}


int main(int argc, char *argv[])
{
	gint		i,
			status;
	const gchar *arg;

	/* Handle basic command line arguments */
	for(i = 1; i < argc; i++)
	{
		arg = argv[i];
		if(arg == NULL)
			continue;

		/* Help */
		if(!g_strcasecmp(arg, "--help") ||
		   !g_strcasecmp(arg, "-help") ||
		   (!strcmp(arg, "--h") && (argc == 2)) ||
		   (!strcmp(arg, "-h") && (argc == 2))
		)
		{
			print_help(g_basename(argv[0]));
			return(0);
		}
		/* Version */
		else if(!g_strcasecmp(arg, "--version") ||
			!g_strcasecmp(arg, "-version")
		)
		{
			g_print(
"Endeavour Mark II Recycle Bin List Version " PROG_VERSION "\n"
PROG_COPYRIGHT
			);
			return(0);
		}
	}

	/* Recycle Bin List */
	status = rls((const gint)argc, (const gchar **)argv);

	/* Return value must be positive */
	if(status < 0)
		status = -status;

	return(status);
}
