/*
	Recycle

	Recycles VFS objects by placing them into the recycle bin.

	Use "rls" to get a list of the recycle bin and "recover" to
	recover a recycled object.
 */

#include <errno.h>
#include <glib.h>
#include <endeavour2.h>
#include "../config.h"


static void print_help(const gchar *prog_name);

static gboolean recycle_query_user_recycle_object(const gchar *name);
static gboolean recycle_query_user_recycle_read_only(
	const gchar *name,
	const EDVObjectType type
);

static gint recycle_progress_cb(
	gpointer progress_data,
	const gulong i, const gulong n
);

static gulong recycle_object_iterate(
	EDVContext *ctx,
	const gchar *path,
	const gboolean interactive,
	const gboolean verbose,
	const gboolean force,
	gint *status_rtn
);

static gint recycle(const gint argc, const gchar **argv);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Prints the help message.
 */
static void print_help(const gchar *prog_name)
{
	g_print(
"Usage: %s <object(s)...> [options]\n",
		prog_name
	);

	g_print(
		"%s",
"\n\
    The <object(s)...> specifies one or more object(s) to recycle.\n\
\n\
    The [options] can be any of the following:\n\
\n\
	--interactive           Prompt before recycling an object.\n\
	-i                      Same as --interactive.\n\
	--quiet                 Do not print any messages to stdout.\n\
	-q                      Same as --quiet.\n\
	--help                  Prints this help screen and exits.\n\
	--version               Prints version information and exits.\n\
\n\
    Return values:\n\
\n\
	0       Success.\n\
	1       General error.\n\
	2       Invalid value.\n\
	3       Systems error or memory allocation error.\n\
	4       User aborted.\n\
	5       User responded with \"no\" to a query.\n\
	6       Busy/retry.\n\
	7       Function not supported.\n\
\n\
    To list recycled objects, use \"rls\".\n\
    To recover recycled objects, use \"recover\".\n\
    To purge recycled objects, use \"purge\".\n\
\n"
	);
}


/*
 *	Queries the user to recycle the object.
 *
 *	Returns TRUE if the user responded with yes.
 */
static gboolean recycle_query_user_recycle_object(const gchar *name)
{
	gint c;

	g_print(
"Recycle `%s'? ",
		name
	);
	c = (gint)fgetc(stdin);
	while(fgetc(stdin) != '\n');
	if((c == 'y') || (c == 'Y'))
		return(TRUE);
	else
		return(FALSE);
}

/*
 *	Queries the user to recycle a read-only object.
 *
 *	Returns TRUE if the user responded with yes.
 */
static gboolean recycle_query_user_recycle_read_only(
	const gchar *name,
	const EDVObjectType type
)
{
	gint c;

	g_print(
"Recycle write-protected %s `%s'? ",
		edv_object_type_to_object_name_lower(type),
		name
	);
	c = (gint)fgetc(stdin);
	while(fgetc(stdin) != '\n');
	if((c == 'y') || (c == 'Y'))
		return(TRUE);
	else
		return(FALSE);
}

/*
 *	Recycle progress callback.
 *
 *	Prints a line stating the object being recycled and the progress
 *	bar to stdout.
 *
 *	No new line character is printed, so this prints on to the
 *	current line only.
 */
static gint recycle_progress_cb(
	gpointer progress_data,
	const gulong i, const gulong n
)
{
	gint		ci,
			cb,
			cn;
	gchar *s;
	const gchar *path = (const gchar *)progress_data;
	gfloat coeff = (n > 0l) ? ((gfloat)i / (gfloat)n) : 0.0f;

	/* Print name */
	s = g_strdup_printf("\rRecycling %s", path);
	ci = strlen(s);
	printf(s);
	g_free(s);

	/* Print spacing between name and progress bar */
	cn = 50;				/* Column to stop at */
	for(; ci < cn; ci++)
		fputc(' ', stdout);

	/* Print progress bar */
	fputc('[', stdout);			/* Left start bracket */
	cn = 23;				/* Width of bar - 2 */
	cb = (int)(cn * coeff) - 1;
	for(ci = 0; ci < cb; ci++)
		fputc('=', stdout);
	fputc((coeff >= 1.0f) ? '=' : '>', stdout);
	ci++;
	for(; ci < cn; ci++)
		fputc('-', stdout);
	fputc(']', stdout);			/* Right end bracket */

	fflush(stdout);				/* Needed since no newline */

	return(0);
}

/*
 *	Recycles the object specified by path.
 *
 *	If the path is a directory then its child objects will be
 *	recycled too.
 *
 *	Returns the index or 0 on error.
 */
static gulong recycle_object_iterate(
	EDVContext *ctx,
	const gchar *path,
	const gboolean interactive,
	const gboolean verbose,
	const gboolean force,
	gint *status_rtn
)
{
	gint		error_code,
			recycle_status;
	gulong		index,
			time_start;
	const gchar *short_name;
	EDVVFSObject *obj = edv_vfs_object_lstat(path);
	if(obj == NULL)
	{
		const gint error_code = (gint)errno;
		g_printerr(
"%s: %s\n",
			g_basename(path),
			g_strerror(error_code)
		);
		*status_rtn = -1;
		return(0l);
	}

	short_name = g_basename(path);

	/* Is this object a directory? */
	if(EDV_VFS_OBJECT_IS_DIRECTORY(obj))
	{
		gint status2;
		const char *name;
		gchar *full_path;
		EDVDirectory *dp = edv_directory_open(
			path,
			FALSE,			/* Not sorted */
			FALSE			/* Exclude notations */
		);
		if(dp != NULL)
		{
			if(verbose)
				g_print(
"Recycling all entries of directory `%s'\n",
					short_name
				);
		}
		else
		{
			const gint error_code = errno;
			g_printerr(
"%s: Unable to open directory: %s\n",
				short_name,
				g_strerror(error_code)
			);
			*status_rtn = -1;
			return(0l);
		}

		for(name = edv_directory_next(dp);
		    name != NULL;
		    name = edv_directory_next(dp)
		)
		{
			/* Generate full path to this child object */
			full_path = edv_paths_join(
				path,
				name
			);

			/* Recycle this child object (recurse if it is
			 * a directory)
			 */
			status2 = 0;
			index = recycle_object_iterate(
				ctx,
				full_path,
				interactive,
				verbose,
				force,
				&status2
			);
			if(status2 != 0)
				*status_rtn = status2;

			g_free(full_path);
		}

		edv_directory_close(dp);

		/* Do not continue to recycle the directory itself
		 * if an error occured or the user responded with
		 * "no" while recycling its child objects
		 */
		if(*status_rtn != 0)
			return(0l);

		if(verbose)
			g_print(
"Recycling directory itself `%s'\n",
				short_name
			);
	}

	/* Confirm recycle read-only object */
	if(!(obj->permissions & EDV_PERMISSION_UW))
	{
		if(force ? FALSE : !recycle_query_user_recycle_read_only(
			short_name,
			obj->type
		))
		{
			edv_vfs_object_delete(obj);
			*status_rtn = -5;
			return(0l);
		}
	}

	edv_vfs_object_delete(obj);

	/* Do confirmation */
	if(interactive ? !recycle_query_user_recycle_object(short_name) : FALSE)
	{
		*status_rtn = -5;
		return(0l);
	}


	time_start = edv_time();

	/* Recycle this object */
	index = edv_recycle(
		ctx,				/* Context */
		path,				/* Object */
		TRUE,				/* Notify Endeavour */
		verbose ? recycle_progress_cb : NULL,	/* Progress callback */
		(gpointer)short_name		/* Progress data */
	);
	error_code = (gint)errno;
	recycle_status = (index != 0l) ? 0 : -1;

	/* Record history */
	edv_history_append(
		ctx,
		EDV_HISTORY_VFS_OBJECT_DELETE,
		time_start,
		edv_time(),
		recycle_status,
		path,
		NULL,				/* No target */
		edv_recycle_get_error(ctx)
	);

	/* Recycle Failed? */
	if(recycle_status != 0)
	{
		const gchar *error_msg = edv_recycle_get_error(ctx);
		g_printerr(
			"%s%s: %s.\n",
			verbose ? "\n" : "",
			path,
			(error_msg != NULL) ? error_msg : g_strerror(error_code)
		);
		edv_context_flush(ctx);
		*status_rtn = -1;
		return(0l);
	}
	else
	{
		if(verbose)
			printf("\n");
		edv_context_flush(ctx);
		return(index);
	}
}


/*
 *	Recycle.
 */
static gint recycle(const gint argc, const gchar **argv)
{
	gboolean	interactive = FALSE,
			verbose = TRUE,
			force = FALSE;
	gint		i,
			status;
	gchar *cwd = edv_getcwd();
	const gchar *arg;
	GList *paths_list = NULL;

	/* Initialize the Endeavour 2 Context */
	EDVContext *ctx = edv_context_new();
	edv_context_init(ctx, NULL);

#define CLEANUP_RETURN(_v_)	{		\
						\
 if(paths_list != NULL) {			\
  g_list_foreach(				\
   paths_list,					\
   (GFunc)g_free,				\
   NULL						\
  );						\
  g_list_free(paths_list);			\
 }						\
						\
 g_free(cwd);					\
						\
 edv_context_delete(ctx);			\
						\
 return(_v_);					\
}

	/* Handle command line arguments */
	for(i = 1; i < argc; i++)
	{
		arg = argv[i];
		if(arg == NULL)
			continue;

		/* Interactive */
		if(!g_strcasecmp(arg, "--interactive") ||
		   !g_strcasecmp(arg, "-i")
		)
		{
			interactive = TRUE;
		}
		/* Quiet */
		else if(!g_strcasecmp(arg, "--quiet") ||
			!g_strcasecmp(arg, "-q")
		)
		{
			verbose = FALSE;
		}
		/* Verbose */
		else if(!g_strcasecmp(arg, "--verbose") ||
			!g_strcasecmp(arg, "-v")
		)
		{
			verbose = TRUE;
		}
		/* Recursive */
		else if(!g_strcasecmp(arg, "--recursive") ||
			!g_strcasecmp(arg, "-R") ||
			!g_strcasecmp(arg, "-r")
		)
		{
			/* Recursive is the default so change nothing */
		}
		/* Force */
		else if(!g_strcasecmp(arg, "--force") ||
			!g_strcasecmp(arg, "-f")
		)
		{
			force = TRUE;
		}
		/* Unsupported argument */
		else if((*arg == '-') || (*arg == '+'))
		{
			g_printerr(
"%s: Unsupported argument.\n",
				arg
			);
			CLEANUP_RETURN(-2);
		}
		/* Path */
		else
		{
			paths_list = g_list_append(
				paths_list,
				edv_path_evaluate(
					cwd,	/* Parent */
					arg	/* Path */
				)
			);
		}
	}

	status = 0;

	if(paths_list != NULL)
	{
		gulong index;
		GList *glist;

		for(glist = paths_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			index = recycle_object_iterate(
				ctx,
				(const gchar *)glist->data,
				interactive,
				verbose,
				force,
				&status
			);
		}
	}
	else
	{
		print_help(g_basename(argv[0]));
		CLEANUP_RETURN(-2);
	}

	/* Flush any pending Endeavour2 operations */
	edv_context_sync(ctx);

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}


int main(int argc, char *argv[])
{
	gint		i,
			status;
	const gchar *arg;

	/* Handle basic command line arguments */
	for(i = 1; i < argc; i++)
	{
		arg = argv[i];
		if(arg == NULL)
			continue;

		/* Help */
		if(!g_strcasecmp(arg, "--help") ||
		   !g_strcasecmp(arg, "-help") ||
		   !g_strcasecmp(arg, "--h") ||
		   !g_strcasecmp(arg, "-h")
		) 
		{ 
			print_help(g_basename(argv[0]));
			return(0);
		}
		/* Version */
		else if(!g_strcasecmp(arg, "--version") ||
			!g_strcasecmp(arg, "-version")
		)
		{ 
			g_print(
"Endeavour Mark II Recycle Version " PROG_VERSION "\n"
PROG_COPYRIGHT
			);
			return(0); 
		}
	}

	/* Recycle */
	status = recycle((const gint)argc, (const gchar **)argv);

	/* Return value must be positive */
	if(status < 0)
		status = -status;

	return(status);
}
