/*
	Recover

	Recovers recycled objects from the recycle bin back to the VFS.
 */

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <glib.h>
#include <endeavour2.h>

#include "../config.h"


static void print_help(const gchar *prog_name);

static gint recover_rmkdir_query(
	EDVContext *ctx,
	const gchar *path
);

static gboolean recover_query_user_recover_object(
	const gchar *name,
	const gulong index
);

static gint recover_progress_cb(
	gpointer progress_data,
	const gulong i, const gulong n
);

static gint recover(const gint argc, const gchar **argv);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *      Prints the help message.
 */
static void print_help(const gchar *prog_name)
{
	g_print(
"Usage: %s <index(ies)...> [alternate_recovery_path] [options]\n",
		prog_name
	);
	g_print(
		"%s",
"\n\
    The <index(ies)...> specifies the index(ies) of one or more\n\
    recycled object(s) to recover. You may need to enclose each\n\
    index in double quotes or prefix it with a backslash in order\n\
    for the shell to process it correctly (e.g. \"#12345\" or \\#12345).\n\
\n\
    The [alternate_recovery_path] specifies the path to the\n\
    directory to be used as the alternate recovery location. If this\n\
    is not specified then each recycled object's original location\n\
    will be used as the recovery location.\n\
\n\
    The [options] can be any of the following:\n\
\n\
	--interactive           Prompt before recovering an object.\n\
	-i                      Same as --interactive.\n\
	--quiet                 Do not print any messages to stdout.\n\
	-q                      Same as --quiet.\n\
	--help                  Prints this help screen and exits.\n\
	--version               Prints version information and exits.\n\
\n\
    Return values:\n\
\n\
	0       Success.\n\
	1       General error.\n\
	2       Invalid value.\n\
	3       Systems error or memory allocation error.\n\
	4       User aborted.\n\
	5       User responded with \"no\" to a query.\n\
	6       Busy/retry.\n\
	7       Function not supported.\n\
\n\
    To list recycled objects, use \"rls\".\n\
    To recycle objects, use \"recycle\".\n\
    To purge recycled objects, use \"purge\".\n\
\n"
	);
}


/*
 *	Creates the directory as needed and queries the user.
 */
static gint recover_rmkdir_query(
	EDVContext *ctx,
	const gchar *path
)
{
	gint		c,
			status;
	GList *new_paths_list;

	if(path == NULL)
	{
		errno = EINVAL;
		return(-2);
	}

	if(edv_path_exists(path))
		return(0);

	/* Query the user to create the directory */
	g_print(
		"Create directory `%s'? ",
		path
	);
	c = (gint)fgetc(stdin);
	while(fgetc(stdin) != '\n');
	if((c != 'y') && (c != 'Y'))
	{
		errno = EINTR;
		return(-4);
	}

	/* Create the directory */
	status = edv_directory_create(
		path,
		TRUE,				/* Create parents */
		&new_paths_list			/* No paths list return */
	);
	if(status != 0)
	{
		const gint error_code = (gint)errno;
		g_printerr(
"Unable to create directory `%s': %s.\n",
			path,
			g_strerror(error_code)
		);
	}

	/* Notify about the created directories */
	if(new_paths_list != NULL)
	{
		GList *glist;
		for(glist = new_paths_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
			edv_notify_queue_vfs_object_added(
				ctx,
				(const gchar *)glist->data
			);
		edv_context_flush(ctx);

		g_list_foreach(new_paths_list, (GFunc)g_free, NULL);
		g_list_free(new_paths_list);
	}

	return(status);
}


/*
 *	Queries the user to recover the object specified by name.
 *
 *	Returns TRUE if the user responded with yes.
 */
static gboolean recover_query_user_recover_object(
	const gchar *name,
	const gulong index
)
{
	gint c;

	g_print(
		"Recover `%s (#%ld)'? ",
		name,
		index
	);
	c = (gint)fgetc(stdin);
	while(fgetc(stdin) != '\n');
	if((c == 'y') || (c == 'Y'))
		return(TRUE);
	else
		return(FALSE);
}


/*
 *	Recover progress callback.
 *
 *	Prints a line stating the object being recycled and the progress
 *	bar to stdout.
 *
 *	No new line character is printed, so this prints on to the
 *	current line only.
 */
static gint recover_progress_cb(
	gpointer progress_data,
	const gulong i, const gulong n
)
{
	gint		ci,
			cb,
			cn;
	gchar *s;
	const gchar *path = (const gchar *)progress_data;
	gfloat coeff = (n > 0l) ? ((gfloat)i / (gfloat)n) : 0.0f; 

	/* Print name */
	s = g_strdup_printf("\rRecovering %s", path);
	ci = strlen(s);                             
	printf(s);
	g_free(s);

	/* Print spacing between name and progress bar */
	cb = 50;				/* Column to stop at */
	for(; ci < cb; ci++)   
		fputc(' ', stdout);

	/* Print progress bar */
	fputc('[', stdout);			/* Left start bracket */
	cn = 23;				/* Width of bar - 2 */  
	cb = (int)(cn * coeff) - 1;
	for(ci = 0; ci < cb; ci++) 
		fputc('=', stdout);   
	fputc((coeff >= 1.0f) ? '=' : '>', stdout);
	ci++;
	for(; ci < cn; ci++)
		fputc('-', stdout);
	fputc(']', stdout);			/* Right end bracket */   

	fflush(stdout);				/* Needed since no newline */

	return(0);
}


/*
 *	Recover.
 */
static gint recover(const gint argc, const gchar **argv)
{
	gboolean	interactive = FALSE,
			verbose = TRUE;
	gint		i,
			status,
			error_code,
			recover_status;
	const gchar	*arg,
			*index_path;
	gchar		*recovery_path,
			*parent_path,
			*alternate_recovery_path = NULL;
	GList		*glist,
			*indicies_list = NULL;
	gulong index;
	gulong time_start;
	EDVRecycledObject *obj;

	/* Initialize the Endeavour2 Context */
	EDVContext *ctx = edv_context_new();
	edv_context_init(ctx, NULL);

#define CLEANUP_RETURN(_v_)	{		\
 if(indicies_list != NULL) {			\
  g_list_free(indicies_list);			\
 }						\
						\
 g_free(alternate_recovery_path);		\
						\
 /* Shutdown the Endeavour2 context */		\
 edv_context_delete(ctx);			\
						\
 return(_v_);					\
}

	/* Handle command line arguments */
	for(i = 1; i < argc; i++)
	{
		arg = argv[i];
		if(arg == NULL)
			continue;

		/* Interactive */
		if(!g_strcasecmp(arg, "--interactive") ||
		   !g_strcasecmp(arg, "-i")
		)
		{
			interactive = TRUE;
		}
		/* Quiet */
		else if(!g_strcasecmp(arg, "--quiet") ||
			!g_strcasecmp(arg, "-q")
		)
		{
			verbose = FALSE;
		}
		/* Verbose */
		else if(!g_strcasecmp(arg, "--verbose") ||
			!g_strcasecmp(arg, "-v")
		)
		{
			verbose = TRUE;
		}
		/* Unsupported Argument */
		else if((*arg == '-') || (*arg == '+'))
		{
			g_printerr(
"%s: Unsupported argument.\n",
				arg 
			);     
			CLEANUP_RETURN(-2);
		}
		/* Index */
		else if(*arg == '#')
		{
			const gulong index = (gulong)atol(arg + 1);
			indicies_list = g_list_append(
				indicies_list,
				(gpointer)index
			);
		}
		/* All else assume it is the alternate recovery path */
		else
		{
			const gchar *path = arg;
			g_free(alternate_recovery_path);
			alternate_recovery_path = (g_path_is_absolute(path)) ?
				g_strdup(path) :
				g_strconcat(
					g_get_current_dir(),
					G_DIR_SEPARATOR_S,
					path,
					NULL
				);
		}
	}

	/* No recycled object indicies specified? */
	if(indicies_list == NULL)
	{
		print_help(g_basename(argv[0]));
		CLEANUP_RETURN(-2);
	}

	/* Get the path to the recycle bin index file */
	index_path = edv_get_s(ctx, EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);

	status = 0;

	/* Iterate through each recycled object index */
	for(glist = indicies_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		index = (gulong)glist->data;

		obj = edv_recycled_object_stat(
			index_path,
			index
		);
		if(obj == NULL)
		{
			g_printerr(
"Unable to recover `#%ld': No such recycled object.\n",
				index
			);
			status = -1;
			continue;
		}

		if(alternate_recovery_path != NULL)
			recovery_path = edv_paths_join(
				alternate_recovery_path,
				obj->name
			);
		else
			recovery_path = edv_paths_join(
				obj->original_path,
				obj->name
			);

		/* Confirm recover */
		if(interactive ?
			!recover_query_user_recover_object(recovery_path, index) :
			FALSE
		)
		{
			edv_recycled_object_delete(obj);
			g_free(recovery_path);
			continue;
		}

		/* Recreate the parent directory as needed */
		parent_path = g_dirname(recovery_path);
		status = recover_rmkdir_query(
			ctx,
			parent_path
		);
		g_free(parent_path);
		if(status != 0)
		{
			edv_recycled_object_delete(obj);
			g_free(recovery_path);
			continue;
		}

		time_start = edv_time();

		/* Recover this object */
		recover_status = edv_recover(
			ctx,
			index,
			alternate_recovery_path,
			TRUE,			/* Notify Endeavour */
			verbose ? recover_progress_cb : NULL,
			recovery_path		/* Data */
		);

		error_code = (gint)errno;

		/* Record history */
		edv_history_append(
			ctx,
			EDV_HISTORY_RECYCLED_OBJECT_RECOVER,
			time_start,
			edv_time(),
			recover_status,
			recovery_path,
			alternate_recovery_path,
			edv_recycle_get_error(ctx)
		);

		/* Recover failed? */
		if(recover_status != 0)
		{
			const gchar *error_msg = edv_recycle_get_error(ctx);
			g_printerr(
				"%s%s (#%ld): %s.\n",
				verbose ? "\n" : "",
				recovery_path,
				index,
				(error_msg != NULL) ? error_msg : g_strerror(error_code)
			);
			status = -1;
		}
		else
		{
			if(verbose)
				printf("\n");
		}

		edv_context_flush(ctx);

		edv_recycled_object_delete(obj);
		g_free(recovery_path);
	}

	/* Flush any pending Endeavour2 operations */
	edv_context_sync(ctx);

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}


int main(int argc, char *argv[])
{
	gint		i,
			status;
	const gchar *arg;

	/* Handle basic command line arguments */
	for(i = 1; i < argc; i++)
	{
		arg = argv[i];
		if(arg == NULL)
			continue;

		/* Help */
		if(!g_strcasecmp(arg, "--help") ||
		   !g_strcasecmp(arg, "-help") ||
		   !g_strcasecmp(arg, "--h") ||
		   !g_strcasecmp(arg, "-h")
		)
		{
			print_help(g_basename(argv[0]));
			return(0);
		}
		/* Version */
		else if(!g_strcasecmp(arg, "--version") ||
				!g_strcasecmp(arg, "-version")
		)
		{
			g_print(
"Endeavour Mark II Recover Version " PROG_VERSION "\n"
PROG_COPYRIGHT
			);
			return(0);
		}
	}

	/* Recover */
	status = recover((const gint)argc, (const gchar **)argv);

	/* Return value must be positive */
	if(status < 0)
		status = -status;

	return(status);
}
