/*
	Purge

	Purges recycled objects from the recycle bin.
 */

#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <glib.h>
#include <endeavour2.h>

#include "../config.h"


static void print_help(const gchar *prog_name);

static gboolean purge_query_user_purge_object(
	const gchar *name,
	const gulong index
);

static gint purge_progress_cb(
	gpointer progress_data,
	const gulong i, const gulong n
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *      Prints the help message.
 */
static void print_help(const gchar *prog_name)
{
	g_print(
"Usage: %s <index(ies)...> [options]\n\
       %s all [options]\n",
		prog_name, prog_name
	);
	g_print(
		"%s",
"\n\
    The <index(ies)...> specifies the index(ies) of one or more\n\
    recycled object(s) to recover. You may need to enclose each\n\
    index in double quotes or prefix it with a backslash in order\n\
    for the shell to process it correctly (e.g. \"#12345\" or \\#12345).\n\
\n\
    You may also specify \"all\" to purge all the recycled objects.\n\
\n\
    The [options] can be any of the following:\n\
\n\
	--interactive           Prompt before recycling an object.\n\
	-i                      Same as --interactive.\n\
	--quiet                 Do not print any messages to stdout.\n\
	-q                      Same as --quiet.\n\
	--help                  Prints this help screen and exits.\n\
	--version               Prints version information and exits.\n\
\n\
    Return values:\n\
\n\
	0       Success.\n\
	1       General error.\n\
	2       Invalid value.\n\
	3       Systems error or memory allocation error.\n\
	4       User aborted.\n\
	5       User responded with \"no\" to a query.\n\
	6       Busy/retry.\n\
	7       Function not supported.\n\
\n\
    To list recycled objects, use \"rls\".\n\
    To recover recycled objects, use \"recover\".\n\
    To recycle objects, use \"recycle\".\n\
\n"
	);
}


/*
 *	Queries the user to purge the recycled object specified by name.
 *
 *	Returns TRUE if the user responded with yes.
 */
static gboolean purge_query_user_purge_object(
	const gchar *name,
	const gulong index
)
{
	gint c;

	g_print(
		"Purge `%s (#%ld)'? ",
		name, index
	);
	c = (gint)fgetc(stdin);
	while(fgetc(stdin) != '\n');
	if((c == 'y') || (c == 'Y'))
		return(TRUE);
	else
		return(FALSE);
}


/*
 *	Purge progress callback.
 *
 *	Prints a line stating the object being recycled and the progress
 *	bar to stdout.
 *
 *	No new line character is printed, so this prints on to the
 *	current line only.
 */
static gint purge_progress_cb(
	gpointer progress_data,
	const gulong i, const gulong n
)
{
	gint		ci,
			cb,
			cn;
	gchar *s;
	const gchar *path = (const gchar *)progress_data;
	gfloat coeff = (n > 0l) ? ((gfloat)i / (gfloat)n) : 0.0f;

	/* Print name */
	s = g_strdup_printf("\rPurging %s", path);
	ci = strlen(s);
	printf(s);
	g_free(s);

	/* Print spacing between name and progress bar */     
	cb = 50;				/* Column to stop at */
	for(; ci < cb; ci++)
		fputc(' ', stdout);

	/* Print progress bar */
	fputc('[', stdout);		/* Left start bracket */
	cn = 23;			/* Width of bar - 2 */
	cb = (int)(cn * coeff) - 1;
	for(ci = 0; ci < cb; ci++) 
		fputc('=', stdout);
	fputc((coeff >= 1.0f) ? '=' : '>', stdout);
	ci++;
	for(; ci < cn; ci++)
		fputc('-', stdout);
	fputc(']', stdout);		/* Right end bracket */   

	fflush(stdout);			/* Needed since no newline */

	return(0);
}


static gint purge(const gint argc, const gchar **argv)
{
	gboolean	interactive = FALSE,
			verbose = TRUE,
			purge_all = FALSE;
	gint		i,
			status,
			purge_status;
	gulong time_start;
	const gchar *arg;
	GList *indicies_list = NULL;

	/* Initialize the Endeavour2 Context */
	EDVContext *ctx = edv_context_new();
	edv_context_init(ctx, NULL);

#define CLEANUP_RETURN(_v_) {		\
 if(indicies_list != NULL) {		\
  g_list_free(indicies_list);		\
 }					\
					\
 /* Shutdown the Endeavour2 context */	\
 edv_context_delete(ctx);		\
					\
 return(_v_);				\
}

	/* Handle command line arguments */
	for(i = 1; i < argc; i++)
	{
		arg = argv[i];
		if(arg == NULL)
			continue;

		/* Interactive */
		if(!g_strcasecmp(arg, "--interactive") ||
		   !g_strcasecmp(arg, "-i")
		)
		{
			interactive = TRUE;
		}
		/* Quiet */
		else if(!g_strcasecmp(arg, "--quiet") ||
			!g_strcasecmp(arg, "-q")
		)
		{
			verbose = FALSE;
		}
		/* Verbose */
		else if(!g_strcasecmp(arg, "--verbose") ||
			!g_strcasecmp(arg, "-v")
		)
		{
			verbose = TRUE;
		}
		/* Unsupported Argument */
		else if((*arg == '-') || (*arg == '+'))
		{
			g_printerr(
"%s: Unsupported argument.\n",
				arg
			);
			CLEANUP_RETURN(-2);
		}
		/* Index */
		else if(*arg == '#')
		{
			const gulong index = (gulong)atol(arg + 1);
			indicies_list = g_list_append(
				indicies_list,
				(gpointer)index
			);
		}
		/* All? */
		else if(!g_strcasecmp(arg, "all"))
		{
			purge_all = TRUE;
		}
	}

	/* No recycled object indicies specified? */
	if((indicies_list == NULL) && !purge_all)
	{
		print_help(g_basename(argv[0]));
		CLEANUP_RETURN(-2);
	}

	status = 0;

	if(purge_all)
	{
		gint error_code;
		const gchar *index_path = edv_get_s(ctx, EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);
		GList	*glist,
			*objects_list = NULL;
		EDVRecycledObject *obj;
		EDVRecycleBinIndex *rp = edv_recycle_bin_index_open(index_path);
		for(obj = edv_recycle_bin_index_next(rp);
		    obj != NULL;
		    obj = edv_recycle_bin_index_next(rp)
		)
			objects_list = g_list_append(
				objects_list,
				edv_recycled_object_copy(obj)
			);
		edv_recycle_bin_index_close(rp);

		for(glist = objects_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			obj = EDV_RECYCLED_OBJECT(glist->data);
			if(obj == NULL)
				continue;

			/* Confirm purge */
			if(interactive ?
				!purge_query_user_purge_object(obj->name, obj->index) :
				FALSE
			)
				continue;

			time_start = edv_time();

			/* Purge this object */
			purge_status = edv_purge(
				ctx,
				obj->index,
				TRUE,		/* Notify Endeavour */
				verbose ? purge_progress_cb : NULL,
				obj->name
			);

			error_code = (gint)errno;

			/* Record history */
			edv_history_append(
			        ctx,
			        EDV_HISTORY_RECYCLED_OBJECT_PURGE,
			        time_start,
				edv_time(),
			        purge_status,
			        obj->name,
			        NULL,		/* No target */
			        edv_recycle_get_error(ctx)
			);

			/* Purge failed? */
			if(purge_status != 0)
			{
				const gchar *error_msg = edv_recycle_get_error(ctx);
				g_printerr(
					"%s%s (#%ld): %s.\n",
					verbose ? "\n" : "",
					obj->name,
					obj->index,
					(error_msg != NULL) ? error_msg : g_strerror(error_code)
				);
				status = 1;
			}
			else
			{
				if(verbose)
					printf("\n");
			}

			edv_context_flush(ctx);
		}

		g_list_foreach(objects_list, (GFunc)edv_recycled_object_delete, NULL);
		g_list_free(objects_list);
	}
	else
	{
		const gchar *index_path = edv_get_s(ctx, EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);
		gint error_code;
		gulong index;
		GList *glist;
		EDVRecycledObject *obj;

		/* Iterate through each recycled object index */
		for(glist = indicies_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			index = (gulong)glist->data;

			obj = edv_recycled_object_stat(
				index_path,
				index
			);
			if(obj == NULL)
			{
				g_printerr(
"Unable to purge `#%ld': No such recycled object.\n",
					index
				);
				status = 1;
				continue;
			}

			/* Do confirmation */
			if(interactive ?
				!purge_query_user_purge_object(obj->name, obj->index) :
				FALSE
			)
			{
				edv_recycled_object_delete(obj);
				continue;
			}

			time_start = edv_time();

			/* Purge this object */
			purge_status = edv_purge(
				ctx,
				obj->index,
				TRUE,		/* Notify Endeavour */
				verbose ? purge_progress_cb : NULL,
				obj->name
			);

			error_code = (gint)errno;

			/* Record history */
			edv_history_append(
			        ctx,
			        EDV_HISTORY_RECYCLED_OBJECT_PURGE,
			        time_start,
				edv_time(),
			        purge_status,
			        obj->name,
			        NULL,		/* No target */
			        edv_recycle_get_error(ctx)
			);

			/* Purged failed? */
			if(purge_status != 0)
			{
				const gchar *error_msg = edv_recycle_get_error(ctx);
				g_printerr(
					"%s%s (#%ld): %s.\n",
					verbose ? "\n" : "",
					obj->name,
					obj->index,
					(error_msg != NULL) ? error_msg : g_strerror(error_code)
				);
				status = 1;
			}
			else
			{
				if(verbose)
					printf("\n");
			}

			edv_recycled_object_delete(obj);

			edv_context_flush(ctx);
		}
	}

	/* Flush any pending Endeavour2 operations */
	edv_context_sync(ctx); 

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}


int main(int argc, char *argv[])
{
	gint		i,
			status;
	const gchar *arg;

	/* Handle basic command line arguments */
	for(i = 1; i < argc; i++)
	{
		arg = argv[i];
		if(arg == NULL)
			continue;

		/* Help */
		if(!g_strcasecmp(arg, "--help") ||
		   !g_strcasecmp(arg, "-help") ||
		   !g_strcasecmp(arg, "--h") ||
		   !g_strcasecmp(arg, "-h")
		)
		{
			print_help(g_basename(argv[0]));
			return(0);
		}
		/* Version */
		else if(!g_strcasecmp(arg, "--version") ||
			!g_strcasecmp(arg, "-version")
		)
		{
			g_print(
"Endeavour Mark II Purge Version " PROG_VERSION "\n"
PROG_COPYRIGHT
			);
			return(0);
		}
	}

	/* Purge */
	status = purge((const gint)argc, (const gchar **)argv);

	/* Return value must be positive */
	if(status < 0)
		status = -status;

	return(status);
}


