#include <sys/types.h>
#include <stdio.h>
#include <dirent.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <gtk/gtk.h>

#include "../guiutils.h"

#include "../images/icon_memory_20x20.xpm"
#include "../images/icon_drive_fixed_20x20.xpm"
#include "../images/icon_close_20x20.xpm"

#include "../images/icon_memory_48x48.xpm"


typedef struct {
	GtkWidget	*toplevel,
			*mem_usage_toplevel,
			*mem_usage_da,
			*mem_usage_label,
			*swap_usage_toplevel,
			*swap_usage_da,
			*swap_usage_label,
			*menu;
	GtkAccelGroup	*accelgrp;
	guint		timeout_id;
	GdkColormap	*colormap;
	gint		ncolors;
	GdkColor	*fg_colors_list,
			*bg_colors_list,
			base_color;
	GdkPixmap	*pixmap;
	GdkGC		*gc;

	gulong		mem_total_kb,
			mem_current_kb,
			swap_total_kb,
			swap_current_kb;
} Win;
#define WIN(p)				((Win *)(p))


static void get_capacity_totals(
	gulong *mem_total_kb_rtn,
	gulong *swap_total_kb_rtn
);
static void get_pid_mem_usage(
	const gint p,
	gulong *data_kb_rtn,
	gulong *stk_kb_rtn,
	gulong *exe_kb_rtn
);
static void get_all_process_mem(gulong *total_kb_rtn);

static gint win_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static gint win_timeout_cb(gpointer data);
static gint win_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void win_close_cb(GtkWidget *widget, gpointer data);

static void win_draw_gauge(
	Win *win,
	GtkWidget *w,
	gfloat coeff
);
static void win_queue_draw_gauges(Win *win);

static void win_update(Win *win);

static Win *win_new(const gint argc, gchar **argv);
static void win_delete(Win *win);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define ISBLANK(c)	(((c) == ' ') || ((c) == '\t'))


/*
 *	Gets the total capacities of memory and swap in kilobytes.
 *
 *	If swap is not available then *swap_total_kb_rtn will be 0.
 */
static void get_capacity_totals(
	gulong *mem_total_kb_rtn,
	gulong *swap_total_kb_rtn
)
{
	FILE *fp;
	const gchar *meminfo_path = "/proc/meminfo";

	*mem_total_kb_rtn = 0l;
	*swap_total_kb_rtn = 0l;

	fp = fopen((const char *)meminfo_path, "rb");
	if(fp != NULL)
	{
		gchar buf[10000];
		const gint nunits = sizeof(buf) / sizeof(gchar);
		const gint units_read = (gint)fread(
			buf,
			sizeof(gchar),
			nunits,
			fp
		);
		if(units_read > 0)
		{
			const gchar *param = "MemTotal:";
			const gchar *s;

			if(units_read < nunits)
				buf[units_read * sizeof(gchar)] = '\0';
			else
				buf[sizeof(buf) - 1] = '\0';

			s = (const gchar *)strstr(
				(const char *)buf,
				(const char *)param
			);
			if(s != NULL)
			{
				s += strlen(param);
				while(ISBLANK(*s))
					s++;

				*mem_total_kb_rtn = atol(s);
			}

			param = "SwapTotal:";
			s = (const gchar *)strstr(
				(const char *)buf,
				(const char *)param
			);
			if(s != NULL)
			{
				s += strlen(param);
				while(ISBLANK(*s))
					s++;

				*swap_total_kb_rtn = atol(s);
			}
		}

		fclose(fp);
	}
}

/*
 *	Gets the total amount of non-shared memory (memory and swap)
 *	the process is using in kilobytes.
 *
 *	The p specifies the process.
 *
 *	The data_kb_rtn, stk_kb_rtn, and exe_kb_rtn specifies the
 *	return values (when added up) for the total amount of
 *	non-shared memory (memory and swap) the process is using in
 *	kilobytes.
 */
static void get_pid_mem_usage(
	const gint p,
	gulong *data_kb_rtn,
	gulong *stk_kb_rtn,
	gulong *exe_kb_rtn
)
{
	FILE *fp;
	const gchar *proc_path = "/proc";
	gchar *path;

	*data_kb_rtn = 0l;
	*stk_kb_rtn = 0l;
	*exe_kb_rtn = 0l;

	/* Format the path to the process's memory statistics */
	path = g_strdup_printf(
		"%s%c%i%c%s",
		proc_path,
		G_DIR_SEPARATOR,
		p,
		G_DIR_SEPARATOR,
		"status"
	);

	/* Open the process's memory statistics as a file for reading */
	fp = fopen((const char *)path, "rb");
	if(fp != NULL)
	{
		gchar buf[10000];
		const gint nunits = sizeof(buf) / sizeof(gchar);
		const gint units_read = (gint)fread(
			buf,
			sizeof(gchar),
			nunits,
			fp
		);
		if(units_read > 0)
		{
			const gchar *param;
			const gchar *s;

			if(units_read < nunits)
				buf[units_read * sizeof(gchar)] = '\0';
			else
				buf[sizeof(buf) - 1] = '\0';

			/* Get the memory used by data */
			param = "VmData:";
			s = (const gchar *)strstr(
				(const char *)buf,
				(const char *)param
			);
			if(s != NULL)
			{
				s += strlen(param);
				while(ISBLANK(*s))
					s++;

				*data_kb_rtn = atol(s);
			}

			/* Get the memory used by stk */
			param = "VmStk:";
			s = (const gchar *)strstr(
				(const char *)buf,
				(const char *)param
			);
			if(s != NULL)
			{
				s += strlen(param);
				while(ISBLANK(*s))
					s++;

				*stk_kb_rtn = atol(s);
			}

			/* Get the memory used by the exe */
			param = "VmExe:";
			s = (const gchar *)strstr(
				(const char *)buf,
				(const char *)param
			);
			if(s != NULL)
			{
				s += strlen(param);
				while(ISBLANK(*s))
					s++;

				*exe_kb_rtn = atol(s);
			}
		}

		/* Close the process's memory statistics */
		fclose(fp);
	}

	/* Delete the path to the process's memory statistics */
	g_free(path);
}

/*
 *	Gets the total amount of memory (memory and swap) all the
 *	process are using in kilobytes.
 */
static void get_all_process_mem(gulong *total_kb_rtn)
{
	DIR *dir;
	struct dirent *de;
	const gchar *s;
	const gchar *proc_path = "/proc";
	gulong		data_mem_kb,
			stk_mem_kb,
			exe_mem_kb;

	*total_kb_rtn = 0l;

	/* Open the processes directory */
	dir = opendir(proc_path);
	if(dir == NULL)
		return;

	/* Get the memory usage of all the processes */
	while(1)
	{
		de = readdir(dir);
		if(de == NULL)
			break;

		s = de->d_name;

		if(!g_strcasecmp(s, ".") || !g_strcasecmp(s, ".."))
			continue;

		if(!isdigit(*s))
			continue;

		/* Get the memory usage of this process */
		get_pid_mem_usage(
			(gint)atoi(s),
			&data_mem_kb,
			&stk_mem_kb,
			&exe_mem_kb
		);

		/* Add the memory usage of this process to the total */
		*total_kb_rtn = (*total_kb_rtn) + data_mem_kb +
			stk_mem_kb + exe_mem_kb;
	}

	/* Close the processes directory */
	closedir(dir);
}


static gint win_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gtk_main_quit();
	return(TRUE);
}

static gint win_timeout_cb(gpointer data)
{
	win_update(WIN(data));
	return(TRUE);
}

static gint win_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	GdkEventButton *button;
	Win *win = WIN(data);

	switch((gint)event->type)
	{
	  case GDK_CONFIGURE:
		if(widget == win->mem_usage_da)
		{
			/* Unref the pixmap, it will be (re)created when on the
			 * next draw
			 */
			if(win->pixmap != NULL)
			{
				gdk_pixmap_unref(win->pixmap);
				win->pixmap = NULL;
			}
			status = TRUE;
		}
		else if(widget == win->swap_usage_da)
		{
			/* Unref the pixmap, it will be (re)created when on the
			 * next draw
			 */
			if(win->pixmap != NULL)
			{
				gdk_pixmap_unref(win->pixmap);
				win->pixmap = NULL;
			}
			status = TRUE;
		}
		break;

	  case GDK_EXPOSE:
		if(widget == win->mem_usage_da)
		{
			if(win->mem_total_kb > 0l)
				win_draw_gauge(
					win,
					win->mem_usage_da,
					(gfloat)win->mem_current_kb /
						(gfloat)win->mem_total_kb
				);
			else
				win_draw_gauge(
					win,
					win->mem_usage_da,
					0.0f
				);
			status = TRUE;
		}
		else if(widget == win->swap_usage_da)
		{
			if(win->swap_total_kb > 0l)
				win_draw_gauge(
					win,
					win->swap_usage_da,
					(gfloat)win->swap_current_kb /
						(gfloat)win->swap_total_kb
				);
			else
				win_draw_gauge(
					win,
					win->swap_usage_da,
					0.0f
				);
			status = TRUE;
		}
		break;

	  case GDK_BUTTON_PRESS:
		button = (GdkEventButton *)event;
		switch(button->button)
		{
		  case GDK_BUTTON3:
			gtk_menu_popup(
				GTK_MENU(win->menu),
				NULL, NULL,
				NULL, NULL,
				button->button, button->time
			);
 		status = TRUE;
			break;
		}
		break;
	}

	return(status);
}


static void win_close_cb(GtkWidget *widget, gpointer data)
{
	gtk_main_quit();
	return;
}


static void win_draw_gauge(
	Win *win,
	GtkWidget *w,
	gfloat coeff
)
{
	gint		i, n, m,
					x, x_step,
					width, height;
	GdkWindow *window = w->window;
	GdkDrawable *drawable;
	GdkGC *gc = win->gc;

	if((window == NULL) || (gc == NULL))
		return;

	gdk_window_get_size(window, &width, &height);
	if((width <= 0) || (height <= 0))
		return;

	/* Get/create the buffer GdkPixmap as needed */
	drawable = (GdkDrawable *)win->pixmap;
	if(drawable == NULL)
		win->pixmap = drawable = gdk_pixmap_new(
			window,
			width, height,
			-1
		);

	/* Draw the background */
	gdk_gc_set_foreground(
		gc,
		&win->base_color
	);
	gdk_draw_rectangle(
		drawable,
		gc,
		TRUE,				/* Fill */
		0, 0,
		width, height
	);

	/* Draw the foreground */
	m = win->ncolors;
	n = MIN((m * coeff), m);
	x = 0;
	x_step = (m > 0) ? (width / m) : 0;
	for(i = 0; i < n; i++)
	{
		gdk_gc_set_foreground(
			gc,
			&win->fg_colors_list[i]
		);
		gdk_draw_rectangle(
			drawable,
			gc,
			TRUE,				/* Fill */
			x, 0,
			x_step - 1, height
		);
		x += x_step;
	}
	for(; i < m; i++)
	{
		gdk_gc_set_foreground(
			gc,
			&win->bg_colors_list[i]
		);
		gdk_draw_rectangle(
			drawable,
			gc,
			TRUE,				/* Fill */
			x, 0,
			x_step - 1, height
		);
		x += x_step;
	}

	/* Send the drawable to the window if the two are not the same */
	if(drawable != (GdkDrawable *)window)
		gdk_draw_pixmap(
			window,
			gc,
			drawable,
			0, 0,
			0, 0,
			width, height
		);
}

static void win_queue_draw_gauges(Win *win)
{
	gtk_widget_queue_draw(win->mem_usage_da);
	gtk_widget_queue_draw(win->swap_usage_da);
}


static void win_update(Win *win)
{
	gulong mem_used_all_kb;

	get_all_process_mem(&mem_used_all_kb);

	if(win->mem_total_kb > 0l)
	{
		gchar *s;

		/* Swap being used? */
		if(mem_used_all_kb > win->mem_total_kb)
		{
			win->mem_current_kb = win->mem_total_kb;
			win->swap_current_kb = mem_used_all_kb - win->mem_total_kb;
		}
		else
		{
			win->mem_current_kb = mem_used_all_kb;
			win->swap_current_kb = 0l;
		}

		win_queue_draw_gauges(win);

		s = g_strdup_printf(
			"%.1f / %.1f mb",
			(gfloat)win->mem_current_kb / 1000.0f,
			(gfloat)win->mem_total_kb / 1000.0f
		);
		gtk_label_set_text(GTK_LABEL(win->mem_usage_label), s);
		g_free(s);

		s = g_strdup_printf(
			"%.1f / %.1f mb",
			(gfloat)win->swap_current_kb / 1000.0f,
			(gfloat)win->swap_total_kb / 1000.0f
		);
		gtk_label_set_text(GTK_LABEL(win->swap_usage_label), s);
		g_free(s);
	}
}


static Win *win_new(const gint argc, gchar **argv)
{
	const gint ncolors = 30;
	GdkColor *c;
	GdkColormap *colormap;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GdkWindow *window;
	GtkAccelGroup *accelgrp;
	GtkStyle *style;
	GtkWidget	*w,
			*parent, *parent2, *parent3, *parent4;
	Win *win = WIN(g_malloc0(sizeof(Win)));

	win->accelgrp = accelgrp = gtk_accel_group_new();

	win->toplevel = w = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_wmclass(
		GTK_WINDOW(w),
		"console",
		"MemoryUsageEstimator"
	);
	gtk_widget_set_name(
		w,
		"memory-usage-estimator-console"
	);
	gtk_window_set_policy(
		GTK_WINDOW(w),
		FALSE, FALSE, FALSE
	);
	gtk_window_set_title(
		GTK_WINDOW(w),
		"Memory Usage Estimator"
	);
/*	gtk_widget_set_usize(w, 250, -1); */
	gtk_widget_add_events(
		w,
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "delete_event",
		GTK_SIGNAL_FUNC(win_delete_event_cb), win
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(win_event_cb), win
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(win_event_cb), win
	);
	gtk_widget_realize(w);
	win->colormap = colormap = gtk_widget_get_colormap(w);
	gdk_colormap_ref(colormap);
	style = gtk_widget_get_style(w);
	window = w->window;
	if(window != NULL)
	{
		GdkGeometry geo;
		geo.min_width = 10;
		geo.min_height = 10;
		geo.base_width = 0;
		geo.base_height = 0;
		geo.width_inc = 1;
		geo.height_inc = 1;
		gdk_window_set_geometry_hints(
			window,
			&geo,
			GDK_HINT_MIN_SIZE |
			GDK_HINT_BASE_SIZE |
			GDK_HINT_RESIZE_INC
		);
	}
	gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE
	);
	gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_CLOSE
	);
	GUISetWMIcon(
		window,
		(guint8 **)icon_memory_48x48_xpm
	);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	parent = w;

	w = gtk_vbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_container_set_border_width(GTK_CONTAINER(w), 5);
	gtk_widget_show(w);
	parent = w;

	/* Mem usage toplevel GtkHBox */
	win->mem_usage_toplevel = w = gtk_hbox_new(FALSE, 2);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	parent2 = w;

	w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);
	gtk_widget_set_usize(w, 70, -1);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_hbox_new(FALSE, 2);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_show(w);
	parent4 = w;

	pixmap = gdk_pixmap_create_from_xpm_d(
		window,
		&mask,
		&style->bg[GTK_STATE_NORMAL],
		icon_memory_20x20_xpm
	);
	w = gtk_pixmap_new(pixmap, mask);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	gdk_pixmap_unref(pixmap);
	gdk_bitmap_unref(mask);

	w = gtk_label_new("Memory:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = gtk_vbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Mem usage GtkDrawingArea */
	win->mem_usage_da = w = gtk_drawing_area_new();
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_set_usize(w, 150, 10);
	gtk_widget_add_events(
		w,
		GDK_STRUCTURE_MASK | GDK_EXPOSURE_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "configure_event",
		GTK_SIGNAL_FUNC(win_event_cb), win
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "expose_event",
		GTK_SIGNAL_FUNC(win_event_cb), win
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(win_event_cb), win
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(win_event_cb), win
	);
	gtk_widget_show(w);

	w = gtk_alignment_new(0.0f, 0.5f, 0.0f, 0.0f);
	gtk_widget_set_usize(w, 100, -1);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Memory usage GtkLabel */
	win->mem_usage_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_show(w);


	/* Swap usage toplevel GtkHBox */
	win->swap_usage_toplevel = w = gtk_hbox_new(FALSE, 2);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	parent2 = w;

	w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);
	gtk_widget_set_usize(w, 70, -1);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_hbox_new(FALSE, 2);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_show(w);
	parent4 = w;

	pixmap = gdk_pixmap_create_from_xpm_d(
		window,
		&mask,
		&style->bg[GTK_STATE_NORMAL],
		icon_drive_fixed_20x20_xpm
	);
	w = gtk_pixmap_new(pixmap, mask);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	gdk_pixmap_unref(pixmap);
	gdk_bitmap_unref(mask);

	w = gtk_label_new("Swap:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = gtk_vbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Swap usage GtkDrawingArea */
	win->swap_usage_da = w = gtk_drawing_area_new();
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_set_usize(w, 150, 10);
	gtk_widget_add_events(
		w,
		GDK_STRUCTURE_MASK | GDK_EXPOSURE_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "configure_event",
		GTK_SIGNAL_FUNC(win_event_cb), win
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "expose_event",
		GTK_SIGNAL_FUNC(win_event_cb), win
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(win_event_cb), win
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(win_event_cb), win
	);
	gtk_widget_show(w);

	w = gtk_alignment_new(0.0f, 0.5f, 0.0f, 0.0f);
	gtk_widget_set_usize(w, 100, -1);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Swap usage GtkLabel */
	win->swap_usage_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_show(w);


	/* Allocate the colors */
	win->ncolors = ncolors;
	win->fg_colors_list = (GdkColor *)g_malloc(
		ncolors * sizeof(GdkColor)
	);
	if(win->fg_colors_list != NULL)
	{
		gint i;
		const gfloat mid_coeff = 0.68f;
		gfloat coeff;

		for(i = 0; i < ncolors; i++)
		{
			c = &win->fg_colors_list[i];
			coeff = (gfloat)(i + 1) / (gfloat)ncolors;
			if(coeff < mid_coeff)
			{
				c->red	= 0xFFFF * (coeff / mid_coeff);
				c->green	= 0xFFFF;
				c->blue	= 0x0000;
				c->pixel	= 0l;
			}
			else
			{
				c->red	= 0xFFFF;
				c->green	= 0xFFFF * (1.0f -
					((coeff - mid_coeff) / (1.0f - mid_coeff))
				);
				c->blue	= 0x0000;
				c->pixel	= 0l;
			}

			gdk_colormap_alloc_color(
				colormap,
				c,
				TRUE,			/* Writeable */
				TRUE			/* Best match */
			);
		}
	}
	win->bg_colors_list = (GdkColor *)g_malloc(
		ncolors * sizeof(GdkColor)
	);
	if(win->bg_colors_list != NULL)
	{
		gint i;
		const gfloat mid_coeff = 0.68f;
		gfloat coeff;

		for(i = 0; i < ncolors; i++)
		{
			c = &win->bg_colors_list[i];
			coeff = (gfloat)(i + 1) / (gfloat)ncolors;
			if(coeff < mid_coeff)
			{
				c->red	= 0x5050 * (coeff / mid_coeff);
				c->green	= 0x5050;
				c->blue	= 0x0000;
				c->pixel	= 0l;
			}
			else
			{
				c->red	= 0x5050;
				c->green	= 0x5050 * (1.0f -
					((coeff - mid_coeff) / (1.0f - mid_coeff))
				);
				c->blue	= 0x0000;
				c->pixel	= 0l;
			}

			gdk_colormap_alloc_color(
				colormap,
				c,
				TRUE,			/* Writeable */
				TRUE			/* Best match */
			);
		}
	}

	c = &win->base_color;
	c->red	= 0x0000;
	c->green	= 0x0000;
	c->blue	= 0x0000;
	c->pixel	= 0l;
	(void)gdk_colormap_alloc_color(
		colormap,
		c,
		TRUE,				/* Writeable */
		TRUE				/* Best match */
	);

	win->pixmap = NULL;

	/* Create the GC */
	win->gc = gdk_gc_new(window);

	/* Get the capacity totals */
	get_capacity_totals(
		&win->mem_total_kb,
		&win->swap_total_kb
	);
	if(win->mem_total_kb > 0l)
		gtk_widget_show(win->mem_usage_toplevel);
	if(win->swap_total_kb > 0l)
		gtk_widget_show(win->swap_usage_toplevel);

	/* Right-click popup menu */
	win->menu = w = GUIMenuCreate();
	if(w != NULL)
	{
		guint accel_key, accel_mods;
		guint8 **icon;
		const gchar *label;
		GtkAccelGroup *accelgrp = NULL;
		GtkWidget *menu = w;
		void (*func_cb)(GtkWidget *w, gpointer);
		gpointer data = win;

#define ADD_MENU_ITEM_LABEL     {               \
 w = GUIMenuItemCreate(                         \
  menu,                                         \
  GUI_MENU_ITEM_TYPE_LABEL,                     \
  accelgrp,                                     \
  icon, label,                                  \
  accel_key, accel_mods,                        \
  func_cb, data                                 \
 );                                             \
}
#define ADD_MENU_ITEM_CHECK     {               \
 w = GUIMenuItemCreate(                         \
  menu,                                         \
  GUI_MENU_ITEM_TYPE_CHECK,                     \
  accelgrp,                                     \
  icon, label,                                  \
  accel_key, accel_mods,                        \
  func_cb, data                                 \
 );                                             \
}
#define ADD_MENU_SEPARATOR      {               \
 w = GUIMenuItemCreate(                         \
  menu,                                         \
  GUI_MENU_ITEM_TYPE_SEPARATOR,                 \
  NULL,                                         \
  NULL, NULL,                                   \
  0, 0,                                         \
  NULL, NULL                                    \
 );                                             \
}

		icon = (guint8 **)icon_close_20x20_xpm;
		label = "Close";
		accel_key = 0;
		accel_mods = 0;
		func_cb = win_close_cb;
		ADD_MENU_ITEM_LABEL

#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_CHECK
#undef ADD_MENU_SEPARATOR
	}

	/* Apply command line arguments */
	gtk_window_apply_args(GTK_WINDOW(win->toplevel), argc, argv);

	/* Map the Win */
	gtk_widget_show(win->toplevel);

	/* Set the timeout callback */
	win->timeout_id = gtk_timeout_add(
		1000l,
		win_timeout_cb, win
	);

	return(win);
}

static void win_delete(Win *win)
{
	gint i;

	if(win == NULL)
		return;

	gtk_timeout_remove(win->timeout_id);

	gtk_widget_destroy(win->menu);

	gtk_widget_destroy(win->toplevel);

	gdk_gc_unref(win->gc);

	for(i = 0; i < win->ncolors; i++)
	{
		gdk_colormap_free_colors(
			win->colormap,
			&win->fg_colors_list[i],
			1
		);
		gdk_colormap_free_colors(
			win->colormap,
			&win->bg_colors_list[i],
			1
		);
	}
	g_free(win->fg_colors_list);
	g_free(win->bg_colors_list);

	gdk_colormap_free_colors(win->colormap, &win->base_color, 1);

	gdk_colormap_unref(win->colormap);

	if(win->pixmap != NULL)
		gdk_pixmap_unref(win->pixmap);

	g_free(win);
}


int main(int argc, char **argv)
{
	Win *win;

	if(!gtk_init_check(&argc, &argv))
		return(1);

	win = win_new(argc, argv);

	gtk_main();

	win_delete(win);

	return(0);
}

