#include <gtk/gtk.h>
#include "../hview.h"


typedef struct {
	GtkWidget	*toplevel,
			*targets_clist,
			*target_entry,
			*target_da,		/* Drop here GtkDrawingArea */
			*move_tb,
			*copy_tb,
			*link_tb,
			*details_clist;
	hview_struct	*hview;
	gchar		*msg;
	gint		drop_x, drop_y;
} Win;
#define WIN(p)			((Win *)(p))


static gint delete_event_cb(
	GtkWidget *widget, GdkEvent *event,
	gpointer data
);
static gint target_da_expose_event_cb(
	GtkWidget *widget, GdkEventExpose *expose,
	gpointer data
);
static gboolean drag_motion_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y, guint t,
	gpointer data
);
static void drag_leave_cb(
	GtkWidget *widget, GdkDragContext *dc,
	guint t,
	gpointer data
);
static gboolean drag_drop_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y, guint t,
	gpointer data
);
static void drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
static void targets_clist_select_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
static void add_target_cb(GtkWidget *widget, gpointer data);
static void remove_target_cb(GtkWidget *widget, gpointer data);
static void toggled_cb(GtkWidget *widget, gpointer data);

static void update_target_da(Win *win);

static void target_da_draw(Win *win);

static Win *win_new(void);
static void win_delete(Win *win);


static gint delete_event_cb(
	GtkWidget *widget, GdkEvent *event,
	gpointer data
)
{
	gtk_main_quit();
	return(TRUE);
}


static gint target_da_expose_event_cb(
	GtkWidget *widget, GdkEventExpose *expose,
	gpointer data
)
{
	target_da_draw(WIN(data));
	return(TRUE);
}

static gboolean drag_motion_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y, guint t,
	gpointer data
)
{
	Win *win = WIN(data);
	win->drop_x = x;
	win->drop_y = y;
	gtk_widget_queue_draw(win->target_da);
	return(FALSE);
}

static void drag_leave_cb(
	GtkWidget *widget, GdkDragContext *dc,
	guint t,
	gpointer data
)
{
	Win *win = WIN(data);
	win->drop_x = -1;
	win->drop_y = -1;
	gtk_widget_queue_draw(win->target_da);
	return;
}

static gboolean drag_drop_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y, guint t,
	gpointer data
)
{
	Win *win = WIN(data);
	win->drop_x = x;
	win->drop_y = y;
	gtk_widget_queue_draw(win->target_da);
	return(FALSE);
}

static void drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gint i, ncolumns;
	const gchar	*protocol_name = "",
			*action_name = "";
	gchar *s, **strv;
	GtkCList *clist;
	Win *win = WIN(data);

	if(selection_data == NULL)
	    return;

	/* Get the protocol name */
	switch(dc->protocol)
	{
	  case GDK_DRAG_PROTO_MOTIF:
	    protocol_name = "Motif";
	    break;
	  case GDK_DRAG_PROTO_XDND:
	    protocol_name = "XDND";
	    break;
	  case GDK_DRAG_PROTO_ROOTWIN:
	    protocol_name = "Root Window";
	    break;
	  case GDK_DRAG_PROTO_NONE:
	    protocol_name = "None";
	    break;
	}

	/* Get the action name */
	switch(dc->action)
	{
	  case GDK_ACTION_DEFAULT:
	    action_name = "Default";
	    break;
	  case GDK_ACTION_COPY:
	    action_name = "Copy";
	    break;
	  case GDK_ACTION_MOVE:
	    action_name = "Move";
	    break;
	  case GDK_ACTION_LINK:
	    action_name = "Link";
	    break;
	  case GDK_ACTION_PRIVATE:
	    action_name = "Private";
	    break;
	  case GDK_ACTION_ASK:
	    action_name = "Ask";
	    break;
	}

	/* Get the target name */
	s = gdk_atom_name(selection_data->target);

	/* Update the drop message for the target GtkDrawingArea */
	g_free(win->msg);
	win->msg = g_strdup_printf(
	    "GOT \"%s\" %i bytes",
	    s,
	    selection_data->length
	);

	g_free(s);

	win->drop_x = x;
	win->drop_y = y;

	gtk_widget_queue_draw(win->target_da);


	/* Set the Details GtkCList */
	clist = GTK_CLIST(win->details_clist);
	ncolumns = clist->columns;
	strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
	    strv[i] = "";

	gtk_clist_freeze(clist);

	gtk_clist_clear(clist);
	gtk_clist_set_column_justification(clist, 0, GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_justification(clist, 1, GTK_JUSTIFY_LEFT);

#define APPEND(_n_,_v_)				{	\
 if(ncolumns >= 2) {					\
  gint row;						\
  strv[0] = (gchar *)(_n_);				\
  strv[1] = (gchar *)(_v_);				\
  row = gtk_clist_append(clist, strv);			\
  if(row > -1) {					\
   gtk_clist_set_selectable(clist, row, FALSE);		\
  }							\
 }							\
}
	s = gdk_atom_name(selection_data->selection);
	APPEND("Selection", s);
	g_free(s);
	s = gdk_atom_name(selection_data->target);
	APPEND("Target", s);
	g_free(s);
	s = gdk_atom_name(selection_data->type);
	APPEND("Type", s);
	g_free(s);
	s = g_strdup_printf("%i bits", selection_data->format);
	APPEND("Format", s);
	g_free(s);
	s = g_strdup_printf("%ld bytes", (gulong)selection_data->length);
	APPEND("Length", s);
	g_free(s);

	APPEND("Protocol", protocol_name);
	APPEND("Is Source", (dc->is_source) ? "Yes" : "No");
	s = g_strdup_printf("%p", dc->source_window);
	APPEND("Source GdkWindow", s);
	g_free(s);
	s = g_strdup_printf("%p", dc->dest_window);
	APPEND("Target GdkWindow", s);
	g_free(s);

	APPEND("Action", action_name);

	s = g_strdup_printf("%ld", (gulong)dc->start_time);
	APPEND("Start Time", s);
	g_free(s);


#undef APPEND
	gtk_clist_thaw(clist);

	/* Add the data to the HexView */
	if(selection_data->format == 8)
	{
	    if(selection_data->length > 0)
		HViewOpenData(
		    win->hview,
		    (const guint8 *)selection_data->data,
		    selection_data->length
		);
	    else
		HViewClear(win->hview);
	}
}

static void targets_clist_select_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	gchar *text = NULL;
	Win *win = WIN(data);
	gtk_clist_get_text(
	    clist,
	    row, 0,
	    &text
	);
	if(text != NULL)
	    gtk_entry_set_text(GTK_ENTRY(win->target_entry), text);
}

static void add_target_cb(GtkWidget *widget, gpointer data)
{
	gchar *strv[1];
	GtkCList *clist;
	Win *win = WIN(data);

	strv[0] = gtk_entry_get_text(GTK_ENTRY(win->target_entry));

	clist = GTK_CLIST(win->targets_clist);
	if(clist->selection_end != NULL)
	    gtk_clist_insert(
		clist,
		(gint)clist->selection_end->data,
		strv
	    );
	else
	    gtk_clist_append(clist, strv);

	update_target_da(win);
}

static void remove_target_cb(GtkWidget *widget, gpointer data)
{
	GtkCList *clist;
	Win *win = WIN(data);

	clist = GTK_CLIST(win->targets_clist);
	while(clist->selection != NULL)
	    gtk_clist_remove(
		clist,
		(gint)clist->selection->data
	    );

	update_target_da(win);
}


static void toggled_cb(GtkWidget *widget, gpointer data)
{
	update_target_da(WIN(data));
}


static void update_target_da(Win *win)
{
	GtkTargetEntry *target_entries_list;
	GtkCList *clist = GTK_CLIST(win->targets_clist);
	const gint ntarget_entries = clist->rows;

	/* Generate the target entries list from the targets GtkCList */
	if(ntarget_entries > 0)
	{
	    gint i;
	    gchar *text = NULL;
	    GtkTargetEntry *target_entry;
	    target_entries_list = (GtkTargetEntry *)g_malloc0(
		ntarget_entries * sizeof(GtkTargetEntry)
	    );
	    for(i = 0; i < ntarget_entries; i++)
	    {
		target_entry = &target_entries_list[i];
		gtk_clist_get_text(
		    clist,
		    i, 0,
		    &text
		);
		target_entry->target = g_strdup(text);
		target_entry->flags = 0;
		target_entry->info = (guint)i;
	    }
	    g_free(win->msg);
	    win->msg = g_strdup("DROP AN OBJECT HERE");
	}
	else
	{
	    target_entries_list = NULL;
	    g_free(win->msg);
	    win->msg = g_strdup("<- ADD A TARGET");
	}
	win->drop_x = -1;
	win->drop_y = -1;

	/* Update the target entries list on the target GtkDrawingArea */
	gtk_drag_dest_unset(win->target_da);
	if(target_entries_list != NULL)
	    gtk_drag_dest_set(
		win->target_da,
		GTK_DEST_DEFAULT_MOTION |
		    GTK_DEST_DEFAULT_HIGHLIGHT |
		    GTK_DEST_DEFAULT_DROP,
		target_entries_list, ntarget_entries,
		(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(win->move_tb)) ?
		    GDK_ACTION_MOVE : 0) |
		(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(win->copy_tb)) ?
		    GDK_ACTION_COPY : 0) |
		(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(win->link_tb)) ?
		    GDK_ACTION_LINK : 0)
	    );

	gtk_widget_queue_draw(win->target_da);

	/* Delete the target entries list */
	if(target_entries_list != NULL)
	{
	    gint i;
	    GtkTargetEntry *target_entry;

	    for(i = 0; i < ntarget_entries; i++)
	    {
		target_entry = &target_entries_list[i];
		g_free(target_entry->target);
	    }
	    g_free(target_entries_list);
	}
}


static void target_da_draw(Win *win)
{
	const gchar *msg = win->msg;
	gint		width, height,
			lbearing, rbearing, swidth, ascent, descent;
	GtkWidget *w = win->target_da;
	GdkWindow *window = w->window;
	GdkDrawable *drawable = (GdkDrawable *)window;
	const GtkStateType state = GTK_WIDGET_STATE(w);
	GtkStyle *style = gtk_widget_get_style(w);
	GdkFont *font = style->font;

	gdk_window_get_size(drawable, &width, &height);

	gdk_draw_rectangle(
	    drawable,
	    style->base_gc[state],
	    TRUE,				/* Fill */
	    0, 0,
	    width, height
	);

	gdk_draw_line(
	    drawable,
	    style->bg_gc[GTK_STATE_SELECTED],
	    win->drop_x, 0,
	    win->drop_x, height
	);
	gdk_draw_line(
	    drawable,
	    style->bg_gc[GTK_STATE_SELECTED],
	    0, win->drop_y,
	    width, win->drop_y
	);

	gdk_string_extents(
	    font, msg,
	    &lbearing, &rbearing, &swidth, &ascent, &descent
	);
	gdk_draw_string(
	    drawable,
	    font,
	    style->text_gc[state],
	    ((width - swidth) / 2) - lbearing,
	    ((height - (font->ascent + font->descent)) / 2) +
	    font->ascent,
	    msg
	);
}


static Win *win_new(void)
{
	GtkWidget *w, *parent, *parent2, *parent3, *parent4;
	GtkCList *clist;
	hview_struct *hview;
	Win *win = WIN(g_malloc0(sizeof(Win)));

	win->msg = NULL;

	win->toplevel = w = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(w), "DND Drop Reader");
	gtk_window_set_policy(GTK_WINDOW(w), FALSE, FALSE, FALSE);
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(delete_event_cb), win
	);
	gtk_widget_realize(w);
	parent = w;

	w = gtk_vbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_container_set_border_width(GTK_CONTAINER(w), 5);
	gtk_widget_show(w);
	parent = w;

	/* Targets GtkHBox */
	w = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Targets GtkVBox */
	w = gtk_vbox_new(FALSE, 2);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
	    GTK_SCROLLED_WINDOW(w),
	    GTK_POLICY_AUTOMATIC,
	    GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Targets GtkCList */
	win->targets_clist = w = gtk_clist_new(1);
	clist = GTK_CLIST(w);
	gtk_widget_set_usize(w, 320, 150);
	gtk_signal_connect(
	    GTK_OBJECT(w), "select_row",
	    GTK_SIGNAL_FUNC(targets_clist_select_row_cb), win
	);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED);
	gtk_clist_set_column_title(clist, 0, "Targets");
	gtk_clist_set_column_auto_resize(clist, 0, TRUE);
	gtk_clist_column_titles_show(clist);
	gtk_clist_column_titles_passive(clist);
	gtk_clist_set_row_height(clist, 20);
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_widget_show(w);

	w = gtk_hbox_new(FALSE, 2);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	win->target_entry = w = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, TRUE, 0);
	gtk_widget_show(w);

	w = gtk_button_new_with_label("+");
	gtk_widget_set_usize(w, 20, -1);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(add_target_cb), win
	);
	gtk_widget_show(w);

	w = gtk_button_new_with_label("-");
	gtk_widget_set_usize(w, 20, -1);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(remove_target_cb), win
	);
	gtk_widget_show(w);


	/* Target GtkVBox */
	w = gtk_vbox_new(FALSE, 2);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Target */
	w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Target GtkDrawingArea */
	win->target_da = w = gtk_drawing_area_new();
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_set_usize(w, 250, -1);
	gtk_widget_add_events(
	    w,
	    GDK_STRUCTURE_MASK | GDK_EXPOSURE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "expose_event",
	    GTK_SIGNAL_FUNC(target_da_expose_event_cb), win
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "drag_motion",
	    GTK_SIGNAL_FUNC(drag_motion_cb), win
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "drag_leave",
	    GTK_SIGNAL_FUNC(drag_leave_cb), win
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "drag_drop",
	    GTK_SIGNAL_FUNC(drag_drop_cb), win
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "drag_data_received",
	    GTK_SIGNAL_FUNC(drag_data_received_cb), win
	);
	gtk_widget_show(w);

	w = gtk_hbox_new(FALSE, 2);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	win->move_tb = w = gtk_toggle_button_new_with_label("Move");
	gtk_widget_set_usize(w, 60, -1);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	GTK_TOGGLE_BUTTON(w)->active = TRUE;
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(toggled_cb), win
	);
	gtk_widget_show(w);

	win->copy_tb = w = gtk_toggle_button_new_with_label("Copy");
	gtk_widget_set_usize(w, 60, -1);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	GTK_TOGGLE_BUTTON(w)->active = TRUE;
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(toggled_cb), win
	);
	gtk_widget_show(w);

	win->link_tb = w = gtk_toggle_button_new_with_label("Link");
	gtk_widget_set_usize(w, 60, -1);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	GTK_TOGGLE_BUTTON(w)->active = TRUE;
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(toggled_cb), win
	);
	gtk_widget_show(w);


	/* Details */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
	    GTK_SCROLLED_WINDOW(w),
	    GTK_POLICY_AUTOMATIC,
	    GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Details GtkCList */
	win->details_clist = w = gtk_clist_new(2);
	clist = GTK_CLIST(w);
	gtk_widget_set_usize(w, -1, 120);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_SINGLE);
	gtk_clist_set_column_title(clist, 0, "Details");
	gtk_clist_set_column_title(clist, 1, "");
	gtk_clist_set_column_auto_resize(clist, 0, TRUE);
	gtk_clist_set_column_auto_resize(clist, 1, TRUE);
	gtk_clist_column_titles_show(clist);
	gtk_clist_column_titles_passive(clist);
	gtk_clist_set_row_height(clist, 20);
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_widget_show(w);


	/* HexView */
	win->hview = hview = HViewNew(parent);
	gtk_widget_set_usize(hview->toplevel, -1, 200);
	HViewShowHeading(hview, TRUE);
	HViewShowStatusBar(hview, FALSE);
	HViewSetReadOnly(hview, TRUE);
	HViewMap(hview);


	/* Add default targets to the targets list */
	clist = GTK_CLIST(win->targets_clist);
	if(clist != NULL)
	{
	    gchar *strv[1];
#define APPEND(_s_)	{		\
 strv[0] = (_s_);			\
 gtk_clist_append(clist, strv);		\
}
	    APPEND("text/plain");
	    APPEND("text/uri-list");
	    APPEND("STRING");
	    APPEND("Endeavour2/MimeType");
	    APPEND("Endeavour2/MimeType/Command");
	    APPEND("Endeavour2/Device");
	    APPEND("Endeavour2/RecycledObject");
	    APPEND("Endeavour2/ArchiveObject");
#undef APPEND
	    update_target_da(win);
	}

	gtk_widget_show(win->toplevel);


	return(win);
}

static void win_delete(Win *win)
{
	if(win == NULL)
	    return;

	HViewDelete(win->hview);
	gtk_widget_destroy(win->toplevel);
	g_free(win->msg);
	g_free(win);
}


int main(int argc, char **argv)
{
	Win *win;

	if(!gtk_init_check(&argc, &argv))
	    return(1);

	win = win_new();

	gtk_main();

	win_delete(win);

	return(0);
}

