#include <gtk/gtk.h>
#include "../hview.h"
#include "../images/icon_x_48x48.xpm"


typedef struct {
	GtkWidget	*toplevel,
			*targets_clist,
			*target_entry,
			*source_da,		/* Drop here GtkDrawingArea */
			*move_tb,
			*copy_tb,
			*link_tb;
	hview_struct	*hview;

	gchar		*msg;

	GdkPixmap	*drag_icon_pixmap;
	GdkBitmap	*drag_icon_mask;

	GtkWidget	*drag_icon_toplevel;
} Win;
#define WIN(p)			((Win *)(p))


static gint delete_event_cb(
	GtkWidget *widget, GdkEvent *event,
	gpointer data
);
static gint source_da_expose_event_cb(
	GtkWidget *widget, GdkEventExpose *expose,
	gpointer data
);
static void dtag_begin_cb(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
);
static void drag_data_get_cb(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info,
	guint t,
	gpointer data
);
static void drag_end_cb(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
);
static void targets_clist_select_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
static void targets_clist_unselect_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);

static void add_target_cb(GtkWidget *widget, gpointer data);
static void remove_target_cb(GtkWidget *widget, gpointer data);
static void toggled_cb(GtkWidget *widget, gpointer data);

static void update_source_da(Win *win);

static void source_da_draw(Win *win);

static void recreate_drag_icon(Win *win);
static Win *win_new(void);
static void win_delete(Win *win);


static gint delete_event_cb(
	GtkWidget *widget, GdkEvent *event,
	gpointer data
)
{
	gtk_main_quit();
	return(TRUE);
}


static gint source_da_expose_event_cb(
	GtkWidget *widget, GdkEventExpose *expose,
	gpointer data
)
{
	source_da_draw(WIN(data));
	return(TRUE);
}

static void dtag_begin_cb(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{
	Win *win = WIN(data);

        /* Set the DND drag icon */
        gtk_drag_set_icon_widget(
            dc,
            win->drag_icon_toplevel,
	    -5, -5				/* Hot spot */
	);

#if 0
        /* Stop the signal so that other callbacks do not override
         * what we have set here
         */
        gtk_signal_emit_stop_by_name(
            GTK_OBJECT(widget),
            "drag_begin"
        );
#endif
}

static void drag_data_get_cb(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info,
	guint t,
	gpointer data
)
{
	Win *win = WIN(data);
	hview_struct *hv = win->hview;
	if(hv->buf != NULL)
	    gtk_selection_data_set(
		selection_data,
		GDK_SELECTION_TYPE_STRING,
		8,                      /* Bits Per Character */
		hv->buf,		/* Data */
		hv->buf_len		/* Length */
	    );
}

static void drag_end_cb(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{

}

static void targets_clist_select_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	gchar *text = NULL;
	Win *win = WIN(data);
	gtk_clist_get_text(
	    clist,
	    row, 0,
	    &text
	);
	if(text != NULL)
	    gtk_entry_set_text(GTK_ENTRY(win->target_entry), text);
	update_source_da(win);
	recreate_drag_icon(win);
}

static void targets_clist_unselect_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	Win *win = WIN(data);
	update_source_da(win);
	recreate_drag_icon(win);
}


static void add_target_cb(GtkWidget *widget, gpointer data)
{
	gchar *strv[1];
	GtkCList *clist;
	Win *win = WIN(data);

	strv[0] = gtk_entry_get_text(GTK_ENTRY(win->target_entry));

	clist = GTK_CLIST(win->targets_clist);
	if(clist->selection_end != NULL)
	    gtk_clist_insert(
		clist,
		(gint)clist->selection_end->data,
		strv
	    );
	else
	    gtk_clist_append(clist, strv);

	update_source_da(win);
	recreate_drag_icon(win);
}

static void remove_target_cb(GtkWidget *widget, gpointer data)
{
	GtkCList *clist;
	Win *win = WIN(data);

	clist = GTK_CLIST(win->targets_clist);
	while(clist->selection != NULL)
	    gtk_clist_remove(
		clist,
		(gint)clist->selection->data
	    );

	update_source_da(win);
	recreate_drag_icon(win);
}


static void toggled_cb(GtkWidget *widget, gpointer data)
{
	Win *win = WIN(data);
	update_source_da(win);
	recreate_drag_icon(win);
}


static void update_source_da(Win *win)
{
	GtkTargetEntry *target_entries_list;
	GtkCList *clist = GTK_CLIST(win->targets_clist);
	const gint ntarget_entries = g_list_length(clist->selection);

	/* Generate the target entries list from the targets GtkCList */
	if(ntarget_entries > 0)
	{
	    gint i, row;
	    GList *glist = clist->selection;
	    gchar *text = NULL;
	    GtkTargetEntry *target_entry;
	    target_entries_list = (GtkTargetEntry *)g_malloc0(
		ntarget_entries * sizeof(GtkTargetEntry)
	    );
	    for(i = 0; (glist != NULL) && (i < ntarget_entries); i++)
	    {
		row = (gint)glist->data;
		gtk_clist_get_text(
		    clist,
		    row, 0,
		    &text
		);

		target_entry = &target_entries_list[i];
		target_entry->target = g_strdup(text);
		target_entry->flags = 0;
		target_entry->info = (guint)row;

		glist = g_list_next(glist);
	    }
	    g_free(win->msg);
	    win->msg = g_strdup("DRAG ME SOMEWHERE");
	}
	else
	{
	    target_entries_list = NULL;
	    g_free(win->msg);
	    win->msg = g_strdup("<- SELECT A TARGET");
	}

	/* Update the target entries list on the source GtkDrawingArea */
	gtk_drag_source_unset(win->source_da);
	if(target_entries_list != NULL)
	    gtk_drag_source_set(
		win->source_da,
		GDK_BUTTON1_MASK,
		target_entries_list, ntarget_entries,
		(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(win->move_tb)) ?
		    GDK_ACTION_MOVE : 0) |
		(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(win->copy_tb)) ?
		    GDK_ACTION_COPY : 0) |
		(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(win->link_tb)) ?
		    GDK_ACTION_LINK : 0)
	    );

	gtk_widget_queue_draw(win->source_da);

	/* Delete the target entries list */
	if(target_entries_list != NULL)
	{
	    gint i;
	    GtkTargetEntry *target_entry;

	    for(i = 0; i < ntarget_entries; i++)
	    {
		target_entry = &target_entries_list[i];
		g_free(target_entry->target);
	    }
	    g_free(target_entries_list);
	}
}


static void source_da_draw(Win *win)
{
	const gchar *msg = win->msg;
	gint		width, height,
			lbearing, rbearing, swidth, ascent, descent;
	GtkWidget *w = win->source_da;
	GtkCList *clist;
	GdkWindow *window = w->window;
	GdkDrawable *drawable = (GdkDrawable *)window;
	const GtkStateType state = GTK_WIDGET_STATE(w);
	GtkStyle *style = gtk_widget_get_style(w);
	GdkFont *font = style->font;

	gdk_window_get_size(drawable, &width, &height);

	gdk_draw_rectangle(
	    drawable,
	    style->base_gc[state],
	    TRUE,				/* Fill */
	    0, 0,
	    width, height
	);

	/* Drag Icon */
	clist = GTK_CLIST(win->targets_clist);
	if(clist->selection != NULL)
	{
	    gint x, y, icon_width, icon_height;
	    GdkPixmap *pixmap = win->drag_icon_pixmap;
	    GdkBitmap *mask = win->drag_icon_mask;
	    GdkGC *gc = style->white_gc;

	    gdk_window_get_size(pixmap, &icon_width, &icon_height);
	    x = (width - icon_width) / 2;
	    y = (height - icon_height) / 2;

	    gdk_gc_set_clip_origin(gc, x, y);
	    gdk_gc_set_clip_mask(gc, mask);
	    gdk_draw_pixmap(
		drawable,
		gc,
		pixmap,
		0, 0,				/* Source x, y */
		x, y,				/* Destination x, y */
		icon_width, icon_height		/* Source size */
	    );
	    gdk_gc_set_clip_mask(gc, NULL);

	    gdk_string_extents(
		font, msg,
		&lbearing, &rbearing, &swidth, &ascent, &descent
	    );
	    gdk_draw_string(
		drawable,
		font,
		style->text_gc[state],
		((width - swidth) / 2) - lbearing,
		y + icon_height + 2 + font->ascent,
		msg
	    );
	}
	else
	{
	    gdk_string_extents(
		font, msg,
		&lbearing, &rbearing, &swidth, &ascent, &descent
	    );
	    gdk_draw_string(
		drawable,
		font,
		style->text_gc[state],
		((width - swidth) / 2) - lbearing,
		((height - (font->ascent + font->descent)) / 2) +
		font->ascent,
		msg
	    );
	}
}


static void recreate_drag_icon(Win *win)
{
	gint		x, y,
			width, height,
			icon_width, icon_height,
			lbearing = 0, rbearing = 0, swidth = 0, ascent, descent,
			font_height;
	gchar *bitmap_data;
	gchar *text = NULL;
	GdkBitmap *icon_mask = win->drag_icon_mask;
	GdkPixmap *icon_pixmap = win->drag_icon_pixmap;
	GdkBitmap *mask;
	GdkPixmap *pixmap;
	GdkFont *font;
	GdkWindow *window;
	GdkGC *gc;
	GtkStyle *style;
	GtkWidget *w, *parent;
	GtkCList *clist;

	/* Destroy the existing DND drag icon GtkWidget */
	if(win->drag_icon_toplevel != NULL)
	    gtk_widget_destroy(win->drag_icon_toplevel);

	/* Create the DND drag icon GtkWidget */
        win->drag_icon_toplevel = w = gtk_window_new(GTK_WINDOW_POPUP);
        gtk_window_set_policy(GTK_WINDOW(w), FALSE, FALSE, FALSE);
        gtk_widget_realize(w);
	window = w->window;
	style = gtk_widget_get_style(w);
	font = style->font;
	font_height = font->ascent + font->descent;
        parent = w;

	/* Get the selected target's text */
	clist = GTK_CLIST(win->targets_clist);
	if(clist->selection_end != NULL)
	    gtk_clist_get_text(
		clist,
		(gint)clist->selection_end->data,
		0,
		&text
	    );

	gdk_window_get_size(icon_pixmap, &icon_width, &icon_height);
	if(text != NULL)
	{
	    gdk_string_extents(
		font, text,
		&lbearing, &rbearing, &swidth, &ascent, &descent
	    );

	    /* Calculate the size needed to include the icon and the text */
	    width = MAX(icon_width, (swidth + 4));
	    height = icon_height + 2 + (font_height + 4);
	}
	else
	{
	    width = icon_width;
	    height = icon_height;
	}

        /* Create an empty GdkBitmap */
        bitmap_data = (gchar *)g_malloc0(
            (((width / 8) + 1) * height) * sizeof(gchar)
        );
        if(bitmap_data != NULL)
	{
            mask = gdk_bitmap_create_from_data(
                window,
                bitmap_data,
                width, height
            );
	    g_free(bitmap_data);
	}
        else
            mask = NULL;

	/* Create the GdkPixmap */
	pixmap = gdk_pixmap_new(
	    window,
	    width, height,
	    -1
	);

	/* Draw the icon on to the pixmap and mask */
	x = (width - icon_width) / 2;
	y = 0;

	gc = gdk_gc_new((GdkWindow *)mask);
	gdk_gc_set_function(gc, GDK_SET);
	gdk_gc_set_clip_origin(gc, x, y);
	gdk_gc_set_clip_mask(gc, icon_mask);
        gdk_draw_rectangle(
            mask,
            gc,
            TRUE,                               /* Fill */
            x, y,
            icon_width, icon_height
        );
	gdk_gc_unref(gc);

        gdk_draw_pixmap(
            pixmap,
	    style->white_gc,
	    icon_pixmap,
	    0, 0,				/* Source x, y */
	    x, y,				/* Destination x, y */
	    icon_width, icon_height		/* Source size */
	);

	/* Draw the text on to the pixmap and mask */
	if(text != NULL)
	{
	    x = (width - (swidth + 4)) / 2;
	    y = icon_height + 2 + 2 + font->ascent;

#define DRAW_STRING(_xo, _yo)	{	\
 gdk_draw_string(			\
  mask,					\
  font,					\
  gc,					\
  x + (_xo), y + (_yo),			\
  text					\
 );					\
}
	    gc = gdk_gc_new((GdkWindow *)mask);
	    gdk_gc_set_function(gc, GDK_SET);
	    DRAW_STRING(0, 0);
	    DRAW_STRING(-1, 0);
	    DRAW_STRING(0, -1);
	    DRAW_STRING(1, 0);
	    DRAW_STRING(0, 1);
	    gdk_gc_unref(gc);

#undef DRAW_STRING

#define DRAW_STRING_BASE(_xo, _yo) {	\
 gdk_draw_string(			\
  pixmap,				\
  font,					\
  style->base_gc[GTK_STATE_NORMAL],	\
  x + (_xo), y + (_yo),			\
  text					\
 );					\
}
	    DRAW_STRING_BASE(-1, 0);
	    DRAW_STRING_BASE(0, -1);
	    DRAW_STRING_BASE(1, 0);
	    DRAW_STRING_BASE(0, 1);
#undef DRAW_STRING_BASE
	    gdk_draw_string(
		pixmap,
		font,
		style->text_gc[GTK_STATE_NORMAL],
		x, y,
		text
	    );
	}

	/* Set the new pixmap and mask on to DND drag icon GtkWidget */
        gtk_widget_set_usize(w, width, height);
        gdk_window_set_back_pixmap(window, pixmap, 0);
        gtk_widget_shape_combine_mask(w, mask, 0, 0);

        w = gtk_pixmap_new(pixmap, mask);
        gtk_container_add(GTK_CONTAINER(parent), w);
        gtk_widget_show(w);

	gdk_pixmap_unref(pixmap);
	gdk_bitmap_unref(mask);
}


static Win *win_new(void)
{
	GdkWindow *window;
	GtkWidget *w, *parent, *parent2, *parent3, *parent4;
	GtkCList *clist;
	hview_struct *hview;
	Win *win = WIN(g_malloc0(sizeof(Win)));

	win->msg = NULL;

	win->toplevel = w = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_policy(GTK_WINDOW(w), FALSE, FALSE, FALSE);
	gtk_window_set_title(GTK_WINDOW(w), "DND Drag Maker");
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(delete_event_cb), win
	);
	gtk_widget_realize(w);
	window = w->window;
	parent = w;

        win->drag_icon_pixmap = gdk_pixmap_create_from_xpm_d(
            window,
            &win->drag_icon_mask,
            NULL,                               /* No background color */
            (gchar **)icon_x_48x48_xpm
        );

	w = gtk_vbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_container_set_border_width(GTK_CONTAINER(w), 5);
	gtk_widget_show(w);
	parent = w;

	/* Targets GtkHBox */
	w = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Targets GtkVBox */
	w = gtk_vbox_new(FALSE, 2);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
	    GTK_SCROLLED_WINDOW(w),
	    GTK_POLICY_AUTOMATIC,
	    GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Targets GtkCList */
	win->targets_clist = w = gtk_clist_new(1);
	clist = GTK_CLIST(w);
	gtk_widget_set_usize(w, 320, 150);
	gtk_signal_connect(
	    GTK_OBJECT(w), "select_row",
	    GTK_SIGNAL_FUNC(targets_clist_select_row_cb), win
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "unselect_row",
	    GTK_SIGNAL_FUNC(targets_clist_unselect_row_cb), win
	);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED);
	gtk_clist_set_column_title(clist, 0, "Targets");
	gtk_clist_set_column_auto_resize(clist, 0, TRUE);
	gtk_clist_column_titles_show(clist);
	gtk_clist_column_titles_passive(clist);
	gtk_clist_set_row_height(clist, 20);
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_widget_show(w);

	w = gtk_hbox_new(FALSE, 2);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	win->target_entry = w = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, TRUE, 0);
	gtk_widget_show(w);

	w = gtk_button_new_with_label("+");
	gtk_widget_set_usize(w, 20, -1);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(add_target_cb), win
	);
	gtk_widget_show(w);

	w = gtk_button_new_with_label("-");
	gtk_widget_set_usize(w, 20, -1);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(remove_target_cb), win
	);
	gtk_widget_show(w);


	/* Source GtkVBox */
	w = gtk_vbox_new(FALSE, 2);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Source GtkFrame */
	w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Source GtkDrawingArea */
	win->source_da = w = gtk_drawing_area_new();
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_set_usize(w, 250, -1);
	gtk_widget_add_events(
	    w,
	    GDK_STRUCTURE_MASK | GDK_EXPOSURE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "expose_event",
	    GTK_SIGNAL_FUNC(source_da_expose_event_cb), win
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "drag_begin",
	    GTK_SIGNAL_FUNC(dtag_begin_cb), win
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "drag_data_get",
	    GTK_SIGNAL_FUNC(drag_data_get_cb), win
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "drag_end",
	    GTK_SIGNAL_FUNC(drag_end_cb), win
	);
	gtk_widget_show(w);

	w = gtk_hbox_new(FALSE, 2);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	win->move_tb = w = gtk_toggle_button_new_with_label("Move");
	gtk_widget_set_usize(w, 60, -1);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	GTK_TOGGLE_BUTTON(w)->active = TRUE;
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(toggled_cb), win
	);
	gtk_widget_show(w);

	win->copy_tb = w = gtk_toggle_button_new_with_label("Copy");
	gtk_widget_set_usize(w, 60, -1);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	GTK_TOGGLE_BUTTON(w)->active = TRUE;
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(toggled_cb), win
	);
	gtk_widget_show(w);

	win->link_tb = w = gtk_toggle_button_new_with_label("Link");
	gtk_widget_set_usize(w, 60, -1);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	GTK_TOGGLE_BUTTON(w)->active = TRUE;
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(toggled_cb), win
	);
	gtk_widget_show(w);


	/* HexView */
	win->hview = hview = HViewNew(parent);
	gtk_widget_set_usize(hview->toplevel, -1, 200);
	HViewShowHeading(hview, TRUE);
	HViewShowStatusBar(hview, FALSE);
	HViewSetReadOnly(hview, FALSE);
	HViewMap(hview);


	/* Add the default targets to the targets list */
	clist = GTK_CLIST(win->targets_clist);
	if(clist != NULL)
	{
	    gchar *strv[1];
#define APPEND(_s_)	{		\
 strv[0] = (_s_);			\
 gtk_clist_append(clist, strv);		\
}
	    APPEND("text/plain");
	    APPEND("text/uri-list");
	    APPEND("STRING");
	    APPEND("Endeavour2/MimeType");
	    APPEND("Endeavour2/MimeType/Command");
	    APPEND("Endeavour2/Device");
	    APPEND("Endeavour2/RecycledObject");
	    APPEND("Endeavour2/ArchiveObject");
#undef APPEND
	    update_source_da(win);
	}

	/* Add default data */
	if(TRUE)
	{
	    const gchar *data = "file:///directory/file.ext";
	    HViewOpenData(
		win->hview,
		data,
		strlen(data) + 1
	    );
	}

	gtk_widget_show(win->toplevel);


	return(win);
}

static void win_delete(Win *win)
{
	if(win == NULL)
	    return;

        if(win->drag_icon_toplevel != NULL)
            gtk_widget_destroy(win->drag_icon_toplevel);

	HViewDelete(win->hview);
	gtk_widget_destroy(win->toplevel);
	gdk_pixmap_unref(win->drag_icon_pixmap);
	gdk_bitmap_unref(win->drag_icon_mask);
	g_free(win->msg);
	g_free(win);
}


int main(int argc, char **argv)
{
	Win *win;

	if(!gtk_init_check(&argc, &argv))
	    return(1);

	win = win_new();

	gtk_main();

	win_delete(win);

	return(0);
}
