/*
                 Endeavour Mark II - Directory Statistics

	Used by Endeavour's Properties Dialog to get statistics on a
	directory.

	Directory statistic results are sent to stdout where the
	Endeavour process will read them from.

 */

#include <stdlib.h>
#include <signal.h>
#include <glib.h>
#include <endeavour2.h>
#include "../config.h"


/*
 *	Directory Statistics:
 */
typedef struct {
	gulong		total_size_bytes,
			total_children,
			total_subdirectories,
			recursion_level,
			ngenerations;
} DSStatistics;
#define DS_STATISTICS(p)		((DSStatistics *)(p))


static void ds_print_usage(const gchar *prog_name);

static void ds_signal_cb(int s);
static void ds_print_statistics(
	DSStatistics *stats,
	const gulong block_size,
	const gboolean human_readable
);
static gint ds_print_directory_iterate(
	const gchar *path,
	const gulong starting_device_index,
	const gulong block_size,
	const gboolean include_directory_sizes,
	const gboolean recurse,
	const gboolean recurse_different_devices,
	const gboolean dereference_links,
	const gboolean print_progress,
	const gboolean human_readable,
	DSStatistics *stats
);
static gint ds(const gint argc, const gchar **argv);


static gboolean user_aborted;


/*
 *	Print usage.
 */
static void ds_print_usage(const gchar *prog_name)
{
	g_print(
"Usage: %s <path> [options]\n",
		prog_name
	);

	g_print(
"\n\
    This program is designed to be only called by Endeavour Mark II.\n\
\n\
    The <path> specifies the directory.\n\
\n\
    The [options] can be any of the following:\n\
\n\
        --block-size [n]        Specifies the block size in bytes, where\n\
                                [n] is the number of bytes per block\n\
                                (default is 1024).\n\
\n\
        --dereference-links     Specifies to dereference links.\n\
        --dereference           Same as --dereference-links.\n\
        -L                      Same as --dereference-links.\n\
\n\
        --exclude-directory-sizes   Specifies to not count the sizes of\n\
                                directories themselves.\n\
        -e                      Same as --exclude-directory-sizes.\b\
\n\
        --one-file-system       Specifies to not recurse into directories\n\
                                that are not on the same device as the\n\
                                specified directory.\n\
        -x                      Same as --one-file-system.\n\
\n\
        --no-progress           Specifies to not print subtotals during\n\
                                the process of getting the statistics.\n\
        -c                      Same as --no-progress.\n\
\n\
        --human-readable        Specifies the print the output in human\n\
                                readable format.\n\
        -v                      Same as --human-readable.\n\
\n\
        --help                  Prints this help screen and exits.\n\
        --version               Prints the version information and exits.\n\
\n"
	);
}

/*
 *	Signal callback.
 */
static void ds_signal_cb(int s)
{
	switch(s)
	{
		case SIGINT:
		case SIGTERM:
		user_aborted = TRUE;
		break;
	}
}


/*
 *	Print statistics.
 */
static void ds_print_statistics(
	DSStatistics *stats,
	const gulong block_size,
	const gboolean human_readable
)
{
	if(human_readable)
		g_print(
"Total Size: %ld kb (%ld mb)\n\
Children: %ld\n\
Subdirectories: %ld\n\
Generations: %ld\n",
			stats->total_size_bytes / block_size,
			stats->total_size_bytes / block_size / block_size,
			stats->total_children,
			stats->total_subdirectories,
			stats->ngenerations
		);
	else
		g_print(
			"%ld %ld %ld %ld\n",
			stats->total_size_bytes / block_size,
			stats->total_children,
			stats->total_subdirectories,
			stats->ngenerations
		);
}

/*
 *	Gather and print the directory's statistics.
 */
static gint ds_print_directory_iterate(
	const gchar *path,
	const gulong starting_device_index,
	const gulong block_size,
	const gboolean include_directory_sizes,
	const gboolean recurse,
	const gboolean recurse_different_devices,
	const gboolean dereference_links,
	const gboolean print_progress,
	const gboolean human_readable,
	DSStatistics *stats
)
{
	EDVDirectory *dp = edv_directory_open(
		path,
		FALSE,			/* Unsorted */
		FALSE			/* Exclude notations */
	);
	if(dp != NULL)
	{
		const gchar *name;
		gchar *child_path;
		EDVVFSObject *obj;

		for(	name = edv_directory_next(dp);
			(name != NULL) && !user_aborted;
			name = edv_directory_next(dp)
		)
		{
			stats->total_children++;

			child_path = g_strconcat(
				path,
				G_DIR_SEPARATOR_S,
				name,
				NULL
			);
			if(child_path == NULL)
				break;

			obj = (dereference_links) ?
				edv_vfs_object_stat(child_path) :
				edv_vfs_object_lstat(child_path);
			if(obj == NULL)
			{
				g_free(child_path);
				continue;
			}

			if(EDV_VFS_OBJECT_IS_DIRECTORY(obj))
			{
				if(include_directory_sizes)
					stats->total_size_bytes += obj->size;
				stats->total_subdirectories++;
				if((obj->device_index != starting_device_index) ?
					(recurse && recurse_different_devices) :
					recurse
				)
				{
					stats->recursion_level++;
					if(stats->recursion_level > stats->ngenerations)
					    stats->ngenerations = stats->recursion_level;
					ds_print_directory_iterate(
						child_path,
						starting_device_index,
						block_size,
						include_directory_sizes,
						recurse,
						recurse_different_devices,
						dereference_links,
						print_progress,
						human_readable,
						stats
					);
					stats->recursion_level--;
					if(print_progress)
						ds_print_statistics(
							stats,
							block_size,
							human_readable
						);
				}
			}
			else
			{
				stats->total_size_bytes += obj->size;
			}

			edv_vfs_object_delete(obj);
			g_free(child_path);
		}

		edv_directory_close(dp);
	}

	return(user_aborted ? 4 : 0);
}


/*
 *	Directory statistics.
 */
static gint ds(const gint argc, const gchar **argv)
{
	gboolean	include_directory_sizes = TRUE,
			recurse = TRUE,
			dereference_links = FALSE,
			recurse_different_devices = TRUE,
			print_progress = TRUE,
			human_readable = FALSE;
	gint		i,
			status;
	const gchar	*arg,
			*path = NULL;
	gulong		block_size = 1024l,
			starting_device_index;
	EDVVFSObject *obj;
	DSStatistics *stats;

	/* Parse arguments */
	for(i = 1; i < argc; i++)
	{
		arg = argv[i];
		if(arg == NULL)
			continue;

		if(!g_strcasecmp(arg, "--help") ||
		   !g_strcasecmp(arg, "-help") ||
		   !g_strcasecmp(arg, "--h") ||
		   !g_strcasecmp(arg, "-h")
		)
		{
			ds_print_usage(g_basename(argv[0]));
			return(0);
		}
		else if(!g_strcasecmp(arg, "--version") ||
		        !g_strcasecmp(arg, "-version")
		)
		{
			g_print("%s %s\n\%s\n",
				g_basename(argv[0]),
				PROG_VERSION,
				PROG_COPYRIGHT
			);
			return(0);
		}
		else if(!g_strcasecmp(arg, "--block-size") ||
			!g_strcasecmp(arg, "-block-size")
		)
		{
			i++;
			arg = (i < argc) ? argv[i] : NULL;
			if(arg == NULL)
			{
				g_printerr(
"%s: Requires argument.\n",
					argv[i - 1]
				);
				return(2);
			}
			block_size = MAX((gulong)atol(arg), 1l);
		}
		else if(!g_strcasecmp(arg, "--dereference-links") ||
			!g_strcasecmp(arg, "-dereference-links") ||
			!g_strcasecmp(arg, "--dereference") ||
			!g_strcasecmp(arg, "-dereference") ||
			!g_strcasecmp(arg, "-L")
		)
		{
			dereference_links = TRUE;
		}
		else if(!g_strcasecmp(arg, "--exclude-directory-sizes") ||
			!g_strcasecmp(arg, "-exclude-directory-sizes") ||
			!g_strcasecmp(arg, "-e")
		)
		{
			include_directory_sizes = FALSE;
		}
		else if(!g_strcasecmp(arg, "--one-file-system") ||
			!g_strcasecmp(arg, "-one-file-system") ||
			!g_strcasecmp(arg, "-x")
		)
		{
			recurse_different_devices = FALSE;
		}
		else if(!g_strcasecmp(arg, "--no-progress") ||
			!g_strcasecmp(arg, "--no-progress") ||
			!g_strcasecmp(arg, "-c")
		)
		{
			print_progress = FALSE;
		}
		else if(!g_strcasecmp(arg, "--human-readable") ||
			!g_strcasecmp(arg, "-human-readable") ||
			!g_strcasecmp(arg, "-v")
		)
		{
			human_readable = TRUE;
		}
		else if((*arg != '-') && (*arg != '+'))
		{
			path = arg;
		}
	}
	if((path == NULL) || (block_size == 0l))
	{
		ds_print_usage(g_basename(argv[0]));
		return(2);
	}

	stats = DS_STATISTICS(g_malloc0(sizeof(DSStatistics)));
	if(stats == NULL)
		return(3);

	obj = (dereference_links) ?
		edv_vfs_object_stat(path) :
		edv_vfs_object_lstat(path);
	if(obj == NULL)
	{
		g_free(stats);
		return(1);
	}

	if(EDV_VFS_OBJECT_IS_DIRECTORY(obj))
	{
		starting_device_index = obj->device_index;
		if(include_directory_sizes)
			stats->total_size_bytes += obj->size;
		status = ds_print_directory_iterate(
			path,
			starting_device_index,
			block_size,
			include_directory_sizes,
			recurse,
			recurse_different_devices,
			dereference_links,
			print_progress,
			human_readable,
			stats
		);
		ds_print_statistics(
			stats,
			block_size,
			human_readable
		);
	}
	else
	{
		status = 2;
	}

	edv_vfs_object_delete(obj);
	g_free(stats);

	return(status);
}

int main(int argc, char *argv[])
{
	user_aborted = FALSE;
	signal(SIGINT, ds_signal_cb);
	signal(SIGTERM, ds_signal_cb);
	return(ds((const gint)argc, (const gchar **)argv));
}
