#include <stdio.h>
#include <sys/stat.h>
#include <gtk/gtk.h>

#include "guiutils.h"
#include "tview.h"
#include "tviewcb.h"

#include "images/icon_cut_20x20.xpm"
#include "images/icon_copy_20x20.xpm"
#include "images/icon_paste_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_edit_48x48.xpm"


void TViewScrollToLine(
	tview_struct *tv, const gint n,
	const gfloat ycoeff, const gboolean move_cursor 
);
GtkVisibility TViewIsLineVisible(tview_struct *tv, const gint n);

gint TViewOpenFile(tview_struct *tv, const gchar *path);
gint TViewSaveFile(tview_struct *tv, const gchar *path);
gint TViewOpenData(tview_struct *tv, const gchar *data, gint data_len);
void TViewClear(tview_struct *tv);

void TViewCut(tview_struct *tv);
void TViewCopy(tview_struct *tv);
void TViewPaste(tview_struct *tv);
void TViewDeleteText(tview_struct *tv);
void TViewSelectAll(tview_struct *tv);
void TViewUnselectAll(tview_struct *tv);

gboolean TViewToplevelIsWindow(tview_struct *tv);
GtkWidget *TViewGetToplevelWidget(tview_struct *tv);
GtkText *TViewGetTextWidget(tview_struct *tv);
GtkMenu *TViewGetMenuWidget(tview_struct *tv);

tview_struct *TViewNew(GtkWidget *parent);
tview_struct *TViewNewWithToplevel(
	const gint width, const gint height
);
void TViewSetReadOnly(tview_struct *tv, const gboolean read_only);
void TViewSetChangedCB(   
	tview_struct *tv,
	void (*changed_cb)(tview_struct *, gpointer),
	gpointer data
);
void TViewUpdate(tview_struct *tv);
void TViewDraw(tview_struct *tv);
void TViewSetBusy(tview_struct *tv, const gboolean busy);
void TViewMap(tview_struct *tv);
void TViewUnmap(tview_struct *tv);
void TViewDelete(tview_struct *tv);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


#define TVIEW_DEF_TITLE			"Text View"
#define TVIEW_DEF_WIDTH			640
#define TVIEW_DEF_HEIGHT		480


/*
 *	Scrolls the Text View to the specified line.
 */
void TViewScrollToLine(
	tview_struct *tv, const gint n,
	const gfloat ycoeff, const gboolean move_cursor
)
{
	gfloat v;
	gint line_height;
	GdkFont *font;
	GtkAdjustment *adj;
	GtkWidget *w = (tv != NULL) ? tv->text : NULL;
	if(w == NULL)
	    return;

	font = tv->text_font;
	adj = tv->vadj;
	if((font == NULL) || (adj == NULL))
	    return;

	/* Get height of each line */
	line_height = font->ascent + font->descent;

	/* Calculate scroll position from line number and apply
	 * ycoeff
	 */
	v = (line_height * n) -
	    (ycoeff * MAX(adj->page_size - line_height, 0));

	/* Clip */
	if(v > (adj->upper - adj->page_size))
	    v = adj->upper - adj->page_size;
	if(v < adj->lower)
	    v = adj->lower;

	/* Scroll as needed */
	gtk_adjustment_set_value(adj, v);

	/* Move cursor */
	if(move_cursor)
	{
	    gint i, m = (gint)gtk_text_get_length(GTK_TEXT(w));
	    gint line_count = 0;

	    for(i = 0; i < m; i++)
	    {
		if(line_count == n)
		{
		    gtk_text_set_point(GTK_TEXT(w), i);
		    break;
		}

		if(GTK_TEXT_INDEX(GTK_TEXT(w), (guint)i) == '\n')
		    line_count++;
	    }
	}
}

/*
 *	Checks if the specified line on the Text View is visible.
 */
GtkVisibility TViewIsLineVisible(tview_struct *tv, const gint n)
{
	gfloat v;
	gint line_height;
	GdkFont *font;
	GtkAdjustment *adj;
	GtkWidget *w = (tv != NULL) ? tv->text : NULL;
	if(w == NULL)
	    return(GTK_VISIBILITY_NONE);

	font = tv->text_font;
	adj = tv->vadj;
	if((font == NULL) || (adj == NULL))
	    return(GTK_VISIBILITY_NONE);

	/* Get height of each line */
	line_height = font->ascent + font->descent;
	if(line_height <= 0)
	    return(GTK_VISIBILITY_NONE);

	/* Calculate scroll position from line number */
	v = (n * line_height) - adj->value;

	/* Line completely not visible? */
	if(((v + line_height) <= 0) || (v >= adj->page_size))
	    return(GTK_VISIBILITY_NONE);

	/* Line partially visible? */
	if((v < 0) || ((v + line_height) > adj->page_size))
	    return(GTK_VISIBILITY_PARTIAL);
	else
	    return(GTK_VISIBILITY_FULL);
}

/*
 *	Opens the specified file to the Text View.
 *
 *	TViewClear() will be called first.
 */
gint TViewOpenFile(tview_struct *tv, const gchar *path)
{
	FILE *fp;
	gint status;
	GtkWidget *w;
	GtkEditable *editable;
	GtkText *text;

	if(tv == NULL)             
	    return(-2);

	/* Delete all text */                        
	TViewClear(tv);

	/* Reset the has changes marker */
	if(TVIEW_HAS_CHANGES(tv))
	    tv->flags &= ~TVIEW_FLAG_HAS_CHANGES;

	if(STRISEMPTY(path))
	    return(-2);

	w = tv->text;
	editable = GTK_EDITABLE(w);
	text = GTK_TEXT(w);

	status = 0;

	/* Open the file for reading */
	fp = fopen((const char *)path, "rb");
	if(fp != NULL)
	{
	    struct stat stat_buf;

	    /* Get the file's statistics */
	    if(fstat(fileno(fp), &stat_buf))
	    {
		status = -1;
	    }
	    else
	    {
		size_t units_read;
		guint8 *io_buf;
#if defined(_WIN32)
		const gulong io_buf_len = 1024l;
#else
		const gulong io_buf_len = (gulong)stat_buf.st_blksize;
#endif

		gtk_text_freeze(text);

		/* Set the GtkText's insert position to the beginning */
		gtk_text_set_point(GTK_TEXT(w), 0);

		/* Allocate the I/O buffer */
		if(io_buf_len > 0l)
		    io_buf = (guint8 *)g_malloc(io_buf_len * sizeof(guint8));
		else
		    io_buf = NULL;

		if(io_buf != NULL)
		{
		    /* Read from the file */
		    while(!feof(fp))
		    {
			/* Read this block */
			units_read = fread(
			    io_buf,
			    sizeof(guint8),
			    io_buf_len,
			    fp
			);
			if(units_read > 0l)
			{
			    /* Append this block to the GtkText */
			    gtk_text_insert(
				text,
				tv->text_font,
				NULL, NULL,
				(const gchar *)io_buf, units_read
			    );
			}

			/* Read error? */
			if(ferror(fp))
			{
			    status = -1;
			    break;
			}
		    }

		    /* Delete the I/O buffer */
		    g_free(io_buf);
		}
		else
		{
		    /* No read buffer, read one character at a time */
		    gint c;
		    gchar buf[1];

		    /* Read from the file */
		    while(!feof(fp))
		    {
			c = (gint)fgetc(fp);
			if(c == EOF)
			    break;

			buf[0] = (gchar)c;

			/* Append this character to the GtkText */
			gtk_text_insert(
			    text,
			    tv->text_font,
			    NULL, NULL,
			    buf, 1
			);
		    }
		}

		gtk_text_thaw(text);
	    }

	    /* Close the file */
	    (void)fclose(fp);
	}
	else
	{
	    status = -1;
	}

	TViewUpdate(tv);

	return(status);
}

/*
 *	Saves the data on the Text View to the specified file.
 */
gint TViewSaveFile(
	tview_struct *tv,
	const gchar *path
)
{
	FILE *fp;
	gint		status,
			len;
	gchar		*buf,
			*buf_end;
	GtkWidget *w;

	if((tv == NULL) || STRISEMPTY(path))
	    return(-2);

	w = tv->text;

	/* Get copy of data from text */
	len = (gint)gtk_text_get_length(GTK_TEXT(w));
	buf = gtk_editable_get_chars(GTK_EDITABLE(w), 0, len);
	if(buf == NULL)
	    return(-1);

	buf_end = buf + len;

	status = 0;

	/* Open the file for writing */
	fp = fopen(path, "wb");
	if(fp != NULL)
	{
	    struct stat stat_buf;

	    /* Get the file's statistics */
	    if(fstat(fileno(fp), &stat_buf))
	    {
		status = -1;
	    }
	    else
	    {
		size_t	units_to_write,
			units_written;
#if defined(_WIN32)
		const gulong io_buf_len = 1024l;
#else
		const gulong io_buf_len = MAX(
		    (gulong)stat_buf.st_blksize,
		    1l
		);
#endif
		const gchar *buf_ptr = buf;

		/* Write to the file */
		while(buf_ptr < buf_end)
		{
		    units_to_write = MIN(
			(gint)(buf_end - buf_ptr),
			(gint)io_buf_len
		    );

		    /* Write this block */
		    units_written = fwrite(
			buf_ptr,
			sizeof(gchar),
			units_to_write,
			fp
		    );
		    if(units_written != units_to_write)
		    {
			status = -1;
			break;
		    }

		    buf_ptr += units_written;
		}
	    }

	    /* Close the file */
	    if(fclose(fp))
	    {
		status = -1;
	    }
	}
	else
	{
	    status = -1;
	}

	/* Delete the copy of the text */
	g_free(buf);

	if(status == 0)
	{
	    /* Reset the has changes marker */
	    if(TVIEW_HAS_CHANGES(tv))
		tv->flags &= ~TVIEW_FLAG_HAS_CHANGES;
	}

	TViewUpdate(tv);

	return(status);
}

/*
 *	Coppies the given data to the Text View.
 *
 *	TViewClear() will be called first.
 */
gint TViewOpenData(tview_struct *tv, const gchar *data, gint data_len)
{
	GtkWidget *w;
	GtkText *text;

	if(tv == NULL)
	    return(-2);

	/* Delete all the text */
	TViewClear(tv);

	/* Reset the has changes marker */
	if(TVIEW_HAS_CHANGES(tv))
	    tv->flags &= ~TVIEW_FLAG_HAS_CHANGES;

	if((data == NULL) || (data_len <= 0))
	    return(0);

	w = tv->text;
	text = GTK_TEXT(w);

	/* Insert text from data */
	gtk_text_freeze(text);
	gtk_text_set_point(text, 0);
	gtk_text_insert(
	    text,
	    tv->text_font,
	    NULL, NULL,
	    data, data_len
	);
	gtk_text_thaw(text);

	TViewUpdate(tv);

	return(0);
}

/*
 *	Clears the Text View.
 */
void TViewClear(tview_struct *tv)
{
	GtkEditable *editable;
	GtkText *text;

	if(tv == NULL)
	    return;

	editable = GTK_EDITABLE(tv->text);
	text = GTK_TEXT(editable);

	tv->freeze_count++;

	gtk_events_process();

	gtk_text_freeze(text);
	gtk_text_set_point(text, 0);
	gtk_editable_delete_text(editable, 0, -1);
	gtk_text_thaw(text);

	TViewUpdate(tv);

	tv->freeze_count--;
}


/*
 *	Cuts the selected data on the Text View to the clipboard.
 */ 
void TViewCut(tview_struct *tv)
{
	GtkEditable *editable;
	GtkText *text;

	if(tv == NULL)
	    return;

	editable = GTK_EDITABLE(tv->text);
	text = GTK_TEXT(editable);

	tv->freeze_count++;

	gtk_events_process();

	gtk_text_freeze(text);
	gtk_editable_cut_clipboard(editable);
	gtk_text_thaw(text);

	TViewUpdate(tv);

	tv->freeze_count--;
}

/*
 *	Coppies the selected data on the Text View to the clipboard.
 */
void TViewCopy(tview_struct *tv)
{
	GtkEditable *editable;
	GtkText *text;

	if(tv == NULL)
	    return;

	editable = GTK_EDITABLE(tv->text);
	text = GTK_TEXT(editable);

	tv->freeze_count++;

	gtk_events_process();

	gtk_text_freeze(text);
	gtk_editable_copy_clipboard(editable);
	gtk_text_thaw(text);

	tv->freeze_count--;
}

/*
 *	Pastes any data from the clipboard to the Text View.
 */
void TViewPaste(tview_struct *tv)
{
	GtkEditable *editable;
	GtkText *text;

	if(tv == NULL)
	    return;

	editable = GTK_EDITABLE(tv->text);
	text = GTK_TEXT(editable);

	tv->freeze_count++;

	gtk_events_process();

	gtk_text_freeze(text);
	gtk_editable_paste_clipboard(editable);
	gtk_text_thaw(text);

	TViewUpdate(tv);

	tv->freeze_count--;
}

/*
 *	Delete selected text in the Text View.
 */
void TViewDeleteText(tview_struct *tv)
{
	gint len, sel_pos, sel_len;
	GtkEditable *editable;
	GtkText *text;

	if(tv == NULL)
	    return;

	editable = GTK_EDITABLE(tv->text);
	text = GTK_TEXT(editable);

	gtk_text_freeze(text);

	/* Get length of data in text */
	len = (gint)gtk_text_get_length(text);

	/* Get selection start position (if any) */
	sel_pos = (gint)editable->selection_start_pos;
	if((sel_pos < 0) || (sel_pos >= len))
	{
	    gtk_text_thaw(text);
	    return;
	}
	 
	/* Calculate selection length (if any) */
	sel_len = (gint)editable->selection_end_pos - sel_pos;
	if(sel_len <= 0)
	{
	    gtk_text_thaw(text);
	    return;
	}

	/* Make sure selection region does not exceed length of data */
	if((sel_pos + sel_len) > len)
	    sel_len = len - sel_pos;
	if(sel_len <= 0)
	{
	    gtk_text_thaw(text);
	    return;
	}

	/* Delete selected text */
	gtk_text_set_point(text, sel_pos);
	gtk_text_forward_delete(text, sel_len);

	gtk_text_thaw(text);

	TViewUpdate(tv);
}

/*
 *	Selects all the text in the Text View.
 */
void TViewSelectAll(tview_struct *tv)
{
	GtkEditable *editable;
	GtkText *text;

	if(tv == NULL)
	    return;

	editable = GTK_EDITABLE(tv->text);
	text = GTK_TEXT(editable);

	tv->freeze_count++;

	gtk_text_freeze(text);
	gtk_editable_select_region(editable, -1, -1);
	gtk_text_thaw(text);

	gtk_events_process();

	gtk_text_freeze(text);
	gtk_editable_select_region(editable, 0, -1);
	gtk_text_thaw(text);

	gtk_events_process();

	TViewUpdate(tv);

	tv->freeze_count--;
}

/*
 *	Unselects all the text in the Text View.
 */
void TViewUnselectAll(tview_struct *tv)
{
	GtkEditable *editable;
	GtkText *text;

	if(tv == NULL)
	    return;

	editable = GTK_EDITABLE(tv->text);
	text = GTK_TEXT(editable);

	tv->freeze_count++;

	gtk_text_freeze(text);
	gtk_editable_select_region(editable, 0, 0);
	gtk_text_thaw(text);

	gtk_events_process();

	TViewUpdate(tv);

	tv->freeze_count--;
}


/*
 *	Checks if the Text View's toplevel GtkWidget is a GtkWindow.
 */
gboolean TViewToplevelIsWindow(tview_struct *tv)
{
	if(tv == NULL)
	    return(FALSE);

	return(GTK_IS_WINDOW(tv->toplevel));
}

/*
 *	Returns the Text View's toplevel GtkWidget.
 */
GtkWidget *TViewGetToplevelWidget(tview_struct *tv)
{
	return((tv != NULL) ? tv->toplevel : NULL);
}

/*
 *	Returns the Text View's GtkText.
 */
GtkText *TViewGetTextWidget(tview_struct *tv)
{
	return((tv != NULL) ? (GtkText *)tv->text : NULL);
}

/*
 *	Returns the Text View's GtkMenu.
 */
GtkMenu *TViewGetMenuWidget(tview_struct *tv)
{
	return((tv != NULL) ? (GtkMenu *)tv->menu : NULL);
}


/*
 *	Creates a new Text View.
 */ 
tview_struct *TViewNew(GtkWidget *parent)
{
	gint border_minor = 2;
	GtkWidget *w, *parent2;
	GtkEditable *editable;
	GtkText *text;
	tview_struct *tv = TVIEW(
	    g_malloc0(sizeof(tview_struct))
	);
	if(tv == NULL)
	    return(NULL);

	tv->map_state = FALSE;
	tv->freeze_count = 0;
	tv->busy_count = 0;
	tv->flags = 0;
	tv->busy_cur = gdk_cursor_new(GDK_WATCH);
	tv->text_cur = gdk_cursor_new(GDK_XTERM);
	tv->translate_cur = gdk_cursor_new(GDK_FLEUR);
	tv->text_font = gdk_font_load(
"-adobe-courier-medium-r-normal-*-12-*-*-*-m-*-iso8859-1"
	);
	tv->changed_cb = NULL;
	tv->changed_data = NULL;

	tv->freeze_count++;

	/* Toplevel GtkVBox */
	tv->toplevel = w = gtk_vbox_new(FALSE, 0);
	if(GTK_IS_BOX(parent))
	    gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	else
	    gtk_container_add(GTK_CONTAINER(parent), w);
	parent = w;

	/* Table for text and scroll bar widgets */
	w = gtk_table_new(2, 2, FALSE);
	gtk_table_set_row_spacing(GTK_TABLE(w), 0, border_minor);
	gtk_table_set_col_spacing(GTK_TABLE(w), 0, border_minor);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;
	/* Text */
	tv->text = w = gtk_text_new(NULL, NULL);
	editable = GTK_EDITABLE(w);
	text = GTK_TEXT(w);
	gtk_widget_set_name(w, TVIEW_WIDGET_NAME);
	gtk_widget_add_events(
	    w,
	    GDK_STRUCTURE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "configure_event",
	    GTK_SIGNAL_FUNC(TViewTextEventCB), tv
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(TViewTextEventCB), tv 
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(TViewTextEventCB), tv 
	);
	text->default_tab_width = 8;
	gtk_text_set_editable(text, !TVIEW_READ_ONLY(tv));
	gtk_text_set_word_wrap(text, TRUE);
	gtk_table_attach(
	    GTK_TABLE(parent2), w,
	    0, 1, 0, 1,
	    GTK_EXPAND | GTK_SHRINK | GTK_FILL,
	    GTK_EXPAND | GTK_SHRINK | GTK_FILL,
	    0, 0
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(TViewChangedCB), tv
	);
	gtk_widget_show(w);

	tv->hadj = GTK_TEXT(w)->hadj;
	tv->vadj = GTK_TEXT(w)->vadj;

#if 0
	/* Horizontal scroll bar */
	if(tv->hadj != NULL)
	{
	    GtkAdjustment *adj = tv->hadj;
	    tv->hscrollbar = w = gtk_hscrollbar_new(adj);
	    gtk_table_attach(
		GTK_TABLE(parent2), w,
		0, 1, 1, 2,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		GTK_FILL,
		0, 0
	    );
	    gtk_widget_show(w);
	}
#endif

	/* Vertical GtkScrollbar */
	if(tv->vadj != NULL)
	{
	    GtkAdjustment *adj = tv->vadj;
	    tv->vscrollbar = w = gtk_vscrollbar_new(adj);
	    gtk_table_attach(
		GTK_TABLE(parent2), w,
		1, 2, 0, 1,
		GTK_FILL,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		0, 0
	    );
	    gtk_widget_show(w);
	}


	/* Right-click popup menu */
	if(TRUE)
	{
#define ADD_MENU_ITEM_LABEL	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_LABEL,			\
  accelgrp,					\
  icon, label,					\
  accel_key, accel_mods,			\
  func_cb, menu_item_data			\
 );						\
}
#define ADD_MENU_ITEM_CHECK	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_CHECK,			\
  accelgrp,					\
  NULL,						\
  label,					\
  accel_key, accel_mods,			\
  func_cb, menu_item_data			\
 );						\
}
#define ADD_MENU_ITEM_SEPARATOR	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL, NULL,					\
  0, 0,						\
  NULL, NULL					\
 );						\
}
	    GtkWidget *menu = GUIMenuCreate();
	    guint8 **icon;
	    const gchar *label;
	    guint accel_key, accel_mods;
	    GtkAccelGroup *accelgrp = NULL;
	    gpointer menu_item_data = tv;
	    void (*func_cb)(GtkWidget *w, gpointer);

	    tv->menu = menu;

	    icon = (guint8 **)icon_cut_20x20_xpm;
	    label =
#if defined(PROG_LANGUAGE_SPANISH)
"Corte"
#elif defined(PROG_LANGUAGE_FRENCH)
"Couper"
#elif defined(PROG_LANGUAGE_GERMAN)
"Schnitt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Taglio"
#elif defined(PROG_LANGUAGE_DUTCH)
"Snee"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Corte"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Snitt"
#else
"Cut"
#endif
	    ;
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = TViewCutCB;
	    ADD_MENU_ITEM_LABEL
	    tv->cut_mi = w;

	    icon = (guint8 **)icon_copy_20x20_xpm;
	    label =
#if defined(PROG_LANGUAGE_SPANISH)
"Copia"
#elif defined(PROG_LANGUAGE_FRENCH)
"Copier"
#elif defined(PROG_LANGUAGE_GERMAN)
"Kopie"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Copia"
#elif defined(PROG_LANGUAGE_DUTCH)
"Kopie"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Cpia"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kopi"
#else  
"Copy"
#endif 
	    ;
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = TViewCopyCB;
	    ADD_MENU_ITEM_LABEL
	    tv->copy_mi = w;

	    icon = (guint8 **)icon_paste_20x20_xpm;
	    label =
#if defined(PROG_LANGUAGE_SPANISH) 
"Pasta"
#elif defined(PROG_LANGUAGE_FRENCH)
"Coller"
#elif defined(PROG_LANGUAGE_GERMAN)
"Paste"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Pasta"
#elif defined(PROG_LANGUAGE_DUTCH)
"Plakmiddel"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Pasta"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Masse"
#else
"Paste"
#endif
	    ;
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = TViewPasteCB;
	    ADD_MENU_ITEM_LABEL
	    tv->paste_mi = w;

	    icon = (guint8 **)icon_cancel_20x20_xpm;
	    label =
#if defined(PROG_LANGUAGE_SPANISH)
"Borre"
#elif defined(PROG_LANGUAGE_FRENCH)
"Supprimer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Lschen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cancellare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrap"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anule"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stryk"
#else
"Delete"
#endif
	    ;
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = TViewDeleteTextCB;
	    ADD_MENU_ITEM_LABEL
	    tv->delete_mi = w;

	    ADD_MENU_ITEM_SEPARATOR

	    icon = NULL;
	    label = "Select All";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = TViewSelectAllCB;
	    ADD_MENU_ITEM_LABEL
	    tv->select_all_mi = w;

	    icon = NULL;
	    label = "Unselect All";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = TViewUnselectAllCB;
	    ADD_MENU_ITEM_LABEL  
	    tv->unselect_all_mi = w;

#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_CHECK
#undef ADD_MENU_ITEM_SEPARATOR
	}

	tv->freeze_count--;

	return(tv);
}

/*
 *	Creates a new Text View with a toplevel GtkWindow.
 */
tview_struct *TViewNewWithToplevel(
	const gint width, const gint height
)
{
	tview_struct *tv;
	GdkWindow *window;

	/* Create a toplevel GtkWindow for the Text View */
	GtkWidget *w = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, TRUE);
	gtk_widget_set_usize(
	    w,
	    (width > 0) ? width : TVIEW_DEF_WIDTH,
	    (height > 0) ? height : TVIEW_DEF_HEIGHT
	);
	gtk_window_set_title(GTK_WINDOW(w), TVIEW_DEF_TITLE);
#ifdef PROG_NAME
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "textview", PROG_NAME
	);
#endif
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    GUISetWMIcon(
		window,
		(guint8 **)icon_edit_48x48_xpm
	    );
	}

	/* Create the Text View and parent it to the toplevel
	 * GtkWindow
	 */
	tv = TViewNew(w);
	if(tv == NULL)
	{
	    gtk_widget_destroy(w);
	    return(NULL);
	}

	tv->freeze_count++;

	gtk_widget_show(tv->toplevel);

	/* Change the Text View's toplevel to the newly created
	 * toplevel GtkWindow
	 */
	tv->toplevel = w;

	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(TViewDeleteEventCB), tv
	);

	tv->freeze_count--;

	return(tv);
}

/*
 *	Sets the Text View as read only or read/write.
 */
void TViewSetReadOnly(tview_struct *tv, const gboolean read_only)
{
	if(tv == NULL)
	    return;

	if(TVIEW_READ_ONLY(tv) != read_only)
	{
	    GtkWidget *w = tv->text;
	    if(w != NULL)
	    {
		gtk_text_set_editable(GTK_TEXT(w), !read_only);
	    }

	    if(read_only)
		tv->flags |= TVIEW_FLAG_READ_ONLY;
	    else
		tv->flags &= ~TVIEW_FLAG_READ_ONLY;

	    TViewDraw(tv);
	    TViewUpdate(tv);
	}
}

/*
 *	Sets the Text View's changed callback.
 */
void TViewSetChangedCB(   
	tview_struct *tv,
	void (*changed_cb)(tview_struct *, gpointer),
	gpointer data
)
{
	if(tv == NULL)
	    return;

	tv->changed_cb = changed_cb;
	tv->changed_data = data;
}

/*
 *	Updates the Text View's widgets to reflect current values.
 */
void TViewUpdate(tview_struct *tv)
{
	gboolean sensitive, read_only, has_selection, has_changes;
	gint sel_len;
	GtkWidget *w;

	if(tv == NULL)
	    return;

	tv->freeze_count++;

	w = tv->text;
	read_only = TVIEW_READ_ONLY(tv);
	sel_len = (w != NULL) ?
	    ((gint)GTK_EDITABLE(w)->selection_end_pos -
	     (gint)GTK_EDITABLE(w)->selection_start_pos) : 0;
	has_selection = (sel_len != 0) ? TRUE : FALSE;
	has_changes = TVIEW_HAS_CHANGES(tv);


	/* Right-click popup menu */

	/* Cut, Copy and Paste */
	sensitive = !read_only ? has_selection : FALSE;
	GTK_WIDGET_SET_SENSITIVE(tv->cut_mi, sensitive);
	sensitive = has_selection;
	GTK_WIDGET_SET_SENSITIVE(tv->copy_mi, sensitive);
	sensitive = !read_only;
	GTK_WIDGET_SET_SENSITIVE(tv->paste_mi, sensitive);
	sensitive = !read_only ? has_selection : FALSE;
	GTK_WIDGET_SET_SENSITIVE(tv->delete_mi, sensitive);
	if(read_only)
	{
	    gtk_widget_hide(tv->cut_mi);
	    gtk_widget_show(tv->copy_mi);
	    gtk_widget_hide(tv->paste_mi);
	    gtk_widget_hide(tv->delete_mi);
	}
	else
	{
	    gtk_widget_show(tv->cut_mi);
	    gtk_widget_show(tv->copy_mi);
	    gtk_widget_show(tv->paste_mi);
	    gtk_widget_show(tv->delete_mi);
	}

	sensitive = TRUE;
	GTK_WIDGET_SET_SENSITIVE(tv->select_all_mi, sensitive);
	sensitive = has_selection;
	GTK_WIDGET_SET_SENSITIVE(tv->unselect_all_mi, sensitive);
	if(read_only)
	{
	    gtk_widget_show(tv->select_all_mi);
	    gtk_widget_show(tv->unselect_all_mi);
	}
	else
	{
	    gtk_widget_show(tv->select_all_mi);
	    gtk_widget_show(tv->unselect_all_mi);
	}

	tv->freeze_count--;
}

/*
 *	Redraws the Text View.
 */
void TViewDraw(tview_struct *tv)
{
	if(tv == NULL)
	    return;

	gtk_widget_draw(tv->text, NULL);
}

/*
 *	Sets the Text View as busy or ready.
 */
void TViewSetBusy(tview_struct *tv, const gboolean busy)
{
#define SET_CURSOR(_w_,_c_)		\
{ if((_w_) != NULL) {			\
 GdkWindow *window = (_w_)->window;	\
 if(window != NULL)			\
  gdk_window_set_cursor(window, (_c_));	\
} }

	if(tv == NULL)
	    return;

	if(busy)
	{
	    /* Increase busy count */
	    tv->busy_count++;

	    /* If already busy then don't change anything */
	    if(tv->busy_count > 1)
		return;

	    SET_CURSOR(tv->toplevel, tv->busy_cur)
	    SET_CURSOR(tv->text, tv->busy_cur)
	}                     
	else                  
	{   
	    /* Reduce busy count */
	    tv->busy_count--;
	    if(tv->busy_count < 0)
		tv->busy_count = 0;

	    /* If still busy do not change anything */
	    if(tv->busy_count > 0)                                     
		return;

	    SET_CURSOR(tv->toplevel, NULL)
	    SET_CURSOR(tv->text, tv->text_cur)
	}

	gdk_flush();

#undef SET_CURSOR
}

/*
 *	Maps the Text View.
 */
void TViewMap(tview_struct *tv)
{
	GtkWidget *w;

	if(tv == NULL)
	    return;

	w = tv->toplevel;
	if(GTK_IS_WINDOW(w))
	    gtk_widget_show_raise(w);
	else
	    gtk_widget_show(w);

	tv->map_state = TRUE;
}

/*
 *	Unmaps the Text View.
 */
void TViewUnmap(tview_struct *tv)
{
	if(tv == NULL)
	    return;

	gtk_widget_hide(tv->toplevel);
	tv->map_state = FALSE;
}

/*
 *	Deletes the Text View.
 */
void TViewDelete(tview_struct *tv)
{
	if(tv == NULL)
	    return;

	TViewUnmap(tv);

	tv->freeze_count++;

	GTK_WIDGET_DESTROY(tv->menu);
	GTK_WIDGET_DESTROY(tv->toplevel);

	(void)GDK_FONT_UNREF(tv->text_font);

	GDK_CURSOR_DESTROY(tv->busy_cur);
	GDK_CURSOR_DESTROY(tv->text_cur);
	GDK_CURSOR_DESTROY(tv->translate_cur);

	tv->freeze_count--;

	g_free(tv);
}
