#include <stdlib.h>
#include <string.h>

#include <gtk/gtk.h>

#include "guiutils.h"
#include "menu_button.h"
#include "tool_bar.h"


#define TOOL_BAR_ITEM_KEY		"/ToolBar/Item"

typedef struct _ToolBar			ToolBar;
#define TOOL_BAR(p)			((ToolBar *)(p))
#define TOOL_BAR_KEY			"/ToolBar"


/*
 *      Tool Bar Flags:
 */
typedef enum {
	TOOL_BAR_MAPPED                 = (1 << 0),
	TOOL_BAR_REALIZED               = (1 << 1),
	TOOL_BAR_SCROLL_BUTTONS_MAPPED  = (1 << 2)
} ToolBarFlags;


/* Callbacks */
static void ToolBarDestroyCB(gpointer data);
static void ToolBarRealizeCB(GtkWidget *widget, gpointer data);

static void ToolBarOutsideSizeAllocateCB(
	GtkWidget *widget, GtkAllocation *allocation, gpointer data
);
static void ToolBarInsideSizeAllocateCB(
	GtkWidget *widget, GtkAllocation *allocation, gpointer data
);

static void ToolBarShowCB(GtkWidget *widget, gpointer data);
static void ToolBarHideCB(GtkWidget *widget, gpointer data);

static gint ToolBarResizeIdleCB(gpointer data);

static void ToolBarScrollAdjustmentValueChangedCB(
	GtkAdjustment *adj, gpointer data
);

static void ToolBarScrollLeftPressedCB(GtkWidget *widget, gpointer data);
static void ToolBarScrollLeftReleasedCB(GtkWidget *widget, gpointer data);
static void ToolBarScrollRightPressedCB(GtkWidget *widget, gpointer data);
static void ToolBarScrollRightReleasedCB(GtkWidget *widget, gpointer data);
static gint ToolBarScrollLeftTOCB(gpointer data);
static gint ToolBarScrollRightTOCB(gpointer data);

static gint ToolBarEnterNotifyCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
static gint ToolBarLeaveNotifyCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
static void ToolBarClickedCB(GtkWidget *widget, gpointer data);

static void ToolBarItemDestroyCB(gpointer data);
static gint ToolBarItemFocusEventCB(
	GtkWidget *widget, GdkEventFocus *focus, gpointer data
);
static gint ToolBarItemVisibilityEventCB(
	GtkWidget *widget, GdkEventVisibility *visibility, gpointer data
);

/* ToolBarItem */
ToolBarItem *ToolBarItemNew(
	const ToolBarItemType type,
	const gchar *text,
	guint8 **icon_data,
	const gchar *tooltip,
	const gint id,
	void (*activate_cb)(
		ToolBarItem *,
		const gint,
		gpointer
	),
	gpointer activate_data,
	void (*enter_cb)(
		ToolBarItem *,
		const gint,
		gpointer
	),
	gpointer enter_data,
	void (*leave_cb)(
		ToolBarItem *,
		const gint,
		gpointer
	),
	gpointer leave_data
);
ToolBarItem *ToolBarItemNewSeparator(void);

void ToolBarItemDelete(ToolBarItem *item);

ToolBarItemType ToolBarItemGetType(ToolBarItem *item);
GtkWidget *ToolBarItemGetWidget(ToolBarItem *item);
GtkWidget *ToolBarItemGetToolBar(ToolBarItem *item);
gint ToolBarItemGetID(ToolBarItem *item);
void ToolBarItemSetData(
	ToolBarItem *item,
	gpointer data
);
void ToolBarItemSetDataFull(
	ToolBarItem *item,
	gpointer data,
	GtkDestroyNotify destroy_cb
);
gpointer ToolBarItemGetData(ToolBarItem *item);
GdkVisibilityState ToolBarItemGetVisibility(ToolBarItem *item);
void ToolBarItemSetToggle(
	ToolBarItem *item,
	const gboolean toggled
);
gboolean ToolBarItemGetToggle(ToolBarItem *item);
void ToolBarItemSetMenu(
	ToolBarItem *item,
	GtkWidget *menu
);
GtkWidget *ToolBarItemGetMenu(ToolBarItem *item);

/* ToolBarItems List */
ToolBarItem *ToolBarItemListMatchByID(
	GList *items_list,
	const gint id
);

/* Scrolling */
static void ToolBarClipScrollPosition(ToolBar *tb);
static void ToolBarScrollLeftIncrement(ToolBar *tb);
static void ToolBarScrollRightIncrement(ToolBar *tb);

/* Remap Scroll Buttons */
static void ToolBarRemapScrollButtons(ToolBar *tb);

/* ToolBar */
static ToolBar *ToolBarGetWidgetData(
	GtkWidget *w,
	const gchar *func_name
);
GtkWidget *ToolBarNew(
	const GtkOrientation orientation,
	const ToolBarButtonDecals button_decals,
	const GtkReliefStyle relief,
	const GtkPolicyType scroll_policy,
	GList *items_list
);
void ToolBarSetButtonDecals(
	GtkWidget *w,
	const ToolBarButtonDecals button_decals
);
void ToolBarSetRelief(
	GtkWidget *w,
	const GtkReliefStyle relief
);
void ToolBarSetButtonSize(
	GtkWidget *w,
	const gint width, const gint height
);
void ToolBarGetButtonSize(
	GtkWidget *w,
	gint *width_rtn, gint *height_rtn
);
void ToolBarSetScrollPolicy(
	GtkWidget *w,
	const GtkPolicyType scroll_policy
);
void ToolBarScrollToItem(
	GtkWidget *w,
	const gint id,
	const gfloat coeff
);
void ToolBarUpdateItem(
	GtkWidget *w,
	const gint id,
	const gchar *text,
	guint8 **icon_data,
	const gchar *tooltip
);
ToolBarItem *ToolBarGetItem(
	GtkWidget *w,
	const gint id
);
GtkWidget *ToolBarGetItemWidget(
	GtkWidget *w,
	const gint id
);
void ToolBarSetItemSensitive(
	GtkWidget *w,
	const gint id,
	const gboolean sensitive
);
void ToolBarSetItemToggle(
	GtkWidget *w,
	const gint id,
	const gboolean toggled
);
gboolean ToolBarGetItemToggle(
	GtkWidget *w,
	const gint id
);
void ToolBarSetItemMenu(
	GtkWidget *w,
	const gint id,
	GtkWidget *menu
);
GtkWidget *ToolBarGetItemMenu(
	GtkWidget *w,
	const gint id
);
gboolean ToolBarIsItemMapped(
	GtkWidget *w,
	const gint id
);
void ToolBarMapItem(
	GtkWidget *w,
	const gint id
);
void ToolBarUnmapItem(
	GtkWidget *w,
	const gint id
);
void ToolBarQueueResize(GtkWidget *w);


#define TOOL_BAR_BUTTON_PICTURE_AND_TEXT_WIDTH   GUI_BUTTON_VLABEL_WIDTH
#define TOOL_BAR_BUTTON_PICTURE_AND_TEXT_HEIGHT  GUI_BUTTON_VLABEL_HEIGHT

#define TOOL_BAR_BUTTON_PICTURE_WIDTH            30
#define TOOL_BAR_BUTTON_PICTURE_HEIGHT           30

#define TOOL_BAR_BUTTON_TEXT_WIDTH               -1
#define TOOL_BAR_BUTTON_TEXT_HEIGHT              30


#define TOOL_BAR_SCROLL_INT			100
#define TOOL_BAR_SCROLL_INITIAL_INT		300


#define TOOL_BAR_CHANGE_BUTTON_LAYOUT(_w_,_button_decals_,_bw_,_bh_) { \
 if((_w_) != NULL) {					\
  switch(_button_decals_) {				\
   case TOOL_BAR_BUTTON_DECALS_PICTURES_AND_TEXT:	\
    GUIButtonChangeLayout((_w_), TRUE, TRUE);		\
    break;						\
   case TOOL_BAR_BUTTON_DECALS_PICTURES:		\
    GUIButtonChangeLayout((_w_), TRUE, FALSE);		\
    break;						\
   case TOOL_BAR_BUTTON_DECALS_TEXT:			\
    GUIButtonChangeLayout((_w_), FALSE, TRUE);		\
    break;						\
  }							\
  gtk_widget_set_usize((_w_), (_bw_), (_bh_));		\
 }							\
}


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	ToolBarItem:
 */
struct _ToolBarItem {

	ToolBarItemType	type;
	gint		freeze_count;
	GdkVisibilityState	visibility;
	ToolBar		*tb;			/* ToolBar that this is on */
	GtkWidget	*w;			/* Item's toplevel GtkWidget */
	gchar		*text;			/* Label */
	guint8		**icon_data;		/* Item icon data (shared) */
	gchar		*tooltip;

	gint		id;			/* Unique ID to identify this
						 * item, set by user */

	gpointer	data;			/* User data */
	GtkDestroyNotify	destroy_cb;

	/* "clicked" or "toggled" callback */
	void	(*activate_cb)(
		ToolBarItem *,			/* ToolBarItem */
		const gint,			/* ID */
		gpointer			/* Function Data */
	);
	gpointer	activate_data;

	/* "enter_notify_event" and "leave_notify_event" callbacks */
	void	(*enter_cb)(
		ToolBarItem *,			/* ToolBarItem */
		const gint,			/* ID */
		gpointer			/* Enter Data */
	);
	gpointer	enter_data;
	void	(*leave_cb)(
		ToolBarItem *,			/* ToolBarItem */
		const gint,			/* ID */
		gpointer			/* Leave Data */
	);
	gpointer	leave_data;

};


/*
 *      Tool Bar:
 */
struct _ToolBar {

	GtkWidget	*toplevel;		/* GtkBox */
	gint		freeze_count,
			callback_level;
	ToolBarFlags	flags;

	GtkOrientation	orientation;
	ToolBarButtonDecals	button_decals;
	GtkReliefStyle	relief;
	gint		button_width,		/* User set button size if
						 * TOOL_BAR_BUTTON_SIZE_SET is
						 * set */
			button_height;
	GtkPolicyType	scroll_buttons_policy;
	GtkAdjustment	*adj;
	guint		scroll_toid,
			resize_idle_id;

	GtkWidget	*fixed,			/* GtkFixed for scrolling */
			*buttons_event_box,
			*buttons_toplevel,	/* Buttons parent GtkBox */
			*scroll_buttons_toplevel,
			*scroll_left_btn,
			*scroll_right_btn;

	GList		*items_list;
};


/*
 *	Tool Bar toplevel GtkBox "destroy" signal callback.
 */
static void ToolBarDestroyCB(gpointer data)
{
	ToolBar *tb = TOOL_BAR(data);
	if(tb == NULL)
	    return;

	if(tb->callback_level > 0)
	{
	    g_printerr(
"ToolBarDestroyCB():\
 Warning: tb=%p ToolBar toplevel GtkWidget %p was destroyed\
 with %i callback levels remaining.\n\
Was this ToolBar destroyed by one of its callback functions?\n",
		tb,
		tb->toplevel,
		tb->callback_level
	    );
	}

	tb->freeze_count++;

	tb->scroll_toid = GTK_TIMEOUT_REMOVE(tb->scroll_toid);

	tb->resize_idle_id = GTK_IDLE_REMOVE(tb->resize_idle_id);

	if(tb->items_list != NULL)
	{
	    g_printerr(
"ToolBarDestroyCB():\
 Warning: tb=%p ToolBarItems list\
 tb->items_list=%p (length=%i)\
 was not cleared properly.\n",
 tb,
 tb->items_list,
 g_list_length(tb->items_list)
	    );
	    g_list_free(tb->items_list);
	}

	(void)GTK_OBJECT_UNREF(GTK_OBJECT(tb->adj));

	tb->freeze_count--;

	g_free(tb);
}

/*
 *	Tool Bar buttons toplevel GtkEventBox "realized" signal
 *	callback.
 */
static void ToolBarRealizeCB(GtkWidget *widget, gpointer data)
{
	ToolBar *tb = TOOL_BAR(data);
	if((widget == NULL) || (tb == NULL))
	    return;

	tb->flags |= TOOL_BAR_REALIZED;
}

/*
 *	Tool Bar GtkFixed "size_allocate" signal callback.
 */
static void ToolBarOutsideSizeAllocateCB(
	GtkWidget *widget, GtkAllocation *allocation, gpointer data
)
{
	gfloat prev_page_size;
	GtkAdjustment *adj;
	ToolBar *tb = TOOL_BAR(data);
	if((allocation == NULL) || (tb == NULL))
	    return;

	/* Update the viewport size based on the new size of the
	 * GtkFixed
	 */
	adj = tb->adj;
	prev_page_size = adj->page_size;
	adj->page_size = (gfloat)((tb->orientation == GTK_ORIENTATION_VERTICAL) ?
	    allocation->height : allocation->width
	);

	/* No change in the viewport's size? */
	if(adj->page_size == prev_page_size)
	    return;

	/* Calculate the scroll increments */
	switch(tb->button_decals)
	{
	  case TOOL_BAR_BUTTON_DECALS_PICTURES_AND_TEXT:
	    adj->step_increment = (gfloat)((tb->orientation == GTK_ORIENTATION_VERTICAL) ?
		TOOL_BAR_BUTTON_PICTURE_AND_TEXT_HEIGHT :
		TOOL_BAR_BUTTON_PICTURE_AND_TEXT_WIDTH
	    );
	    break;
	  case TOOL_BAR_BUTTON_DECALS_PICTURES:
	    adj->step_increment = (gfloat)((tb->orientation == GTK_ORIENTATION_VERTICAL) ?
		TOOL_BAR_BUTTON_PICTURE_HEIGHT:
		TOOL_BAR_BUTTON_PICTURE_WIDTH
	    );
	    break;
	  case TOOL_BAR_BUTTON_DECALS_TEXT:
	    adj->step_increment = (gfloat)((tb->orientation == GTK_ORIENTATION_VERTICAL) ?
		TOOL_BAR_BUTTON_TEXT_HEIGHT :
		TOOL_BAR_BUTTON_TEXT_WIDTH
	    );
	    break;
	}
	if(adj->step_increment <= 0.0f)
	    adj->step_increment = (gfloat)adj->page_size * 0.25f;
	adj->page_increment = adj->page_size;

	if(tb->freeze_count > 0)
	    return;

	/* Remap the scroll buttons due to the viewport size changing
	 * and clip the scroll position
	 */
	ToolBarRemapScrollButtons(tb);
	ToolBarClipScrollPosition(tb);
}

/*
 *	Tool Bar buttons toplevel GtkEventBox "size_allocate" signal
 *	callback.
 */
static void ToolBarInsideSizeAllocateCB(
	GtkWidget *widget, GtkAllocation *allocation, gpointer data
)
{
	gfloat prev_size;
	GtkAdjustment *adj;
	ToolBar *tb = TOOL_BAR(data);
	if((allocation == NULL) || (tb == NULL))
	    return;

	/* Update the buttons toplevel size */
	adj = tb->adj;
	prev_size = adj->upper;
	adj->upper = (gfloat)(
	    (tb->orientation == GTK_ORIENTATION_VERTICAL) ?
		allocation->height : allocation->width
	);
	adj->lower = 0.0f;

	/* No change in the button toplevel's size? */
	if(adj->upper == prev_size)
	    return;

	if(tb->freeze_count > 0)
	    return;

	/* Remap the scroll buttons due to the buttons toplevel GtkBox
	 * size changing and clip the scroll positions
	 */
	ToolBarRemapScrollButtons(tb);
	ToolBarClipScrollPosition(tb);
}


/*
 *	Tool Bar toplevel GtkBox "show" signal callback.
 */
static void ToolBarShowCB(GtkWidget *widget, gpointer data)
{
	ToolBar *tb = TOOL_BAR(data);
	if(tb == NULL)
	    return;

	tb->flags |= TOOL_BAR_MAPPED;
}

/*
 *	Tool Bar toplevel GtkBox "hide" signal callback.
 */
static void ToolBarHideCB(GtkWidget *widget, gpointer data)
{
	ToolBar *tb = TOOL_BAR(data);
	if(tb == NULL)
	    return;

	tb->flags &= ~TOOL_BAR_MAPPED;
}


/*
 *	Resize idle callback.
 */
static gint ToolBarResizeIdleCB(gpointer data)
{
	GtkWidget	*toplevel,
			*parent_toplevel;
	ToolBar *tb = TOOL_BAR(data);
	if(tb == NULL)
	    return(FALSE);

	tb->resize_idle_id = 0;

	toplevel = tb->toplevel;

	/* Get the Tool Bar's GtkFixed and scroll buttons toplevel
	 * GtkBox to properly resize when the scroll buttons toplevel
	 * GtkBox gets mapped/unmapped
	 */
	gtk_container_check_resize(GTK_CONTAINER(toplevel));

	/* Queue the ToolBar's toplevel GtkWindow so that the
	 * ToolBar's sibling GtkWidgets are resized alongside it
	 * when its size changes
	 */
	parent_toplevel = gtk_widget_get_toplevel(toplevel);
	if(parent_toplevel != NULL)
	    gtk_widget_queue_resize(parent_toplevel);

	return(FALSE);
}


/*
 *	Tool Bar scroll GtkAdjustment "value_changed" signal callback.
 */
static void ToolBarScrollAdjustmentValueChangedCB(
	GtkAdjustment *adj, gpointer data
)
{
	ToolBar *tb = TOOL_BAR(data);
	if((adj == NULL) || (tb == NULL))
	    return;

	if(tb->freeze_count > 0)
	    return;

	if(tb->orientation == GTK_ORIENTATION_VERTICAL)
	    gtk_fixed_move(
		GTK_FIXED(tb->fixed),
		tb->buttons_event_box,
		0,
		(gint)-adj->value
	    );
	else
	    gtk_fixed_move(
		GTK_FIXED(tb->fixed),
		tb->buttons_event_box,
		(gint)-adj->value,
		0
	    );
}

/*
 *	Tool Bar scroll left GtkButton "pressed" signal callback.
 */
static void ToolBarScrollLeftPressedCB(GtkWidget *widget, gpointer data)
{
	ToolBar *tb = TOOL_BAR(data);
	if(tb == NULL)
	    return;

	if(tb->freeze_count > 0)
	    return;

	if(tb->scroll_toid != 0)
	    return;

	ToolBarScrollLeftIncrement(tb);

	tb->scroll_toid = gtk_timeout_add(
	    TOOL_BAR_SCROLL_INITIAL_INT,
	    ToolBarScrollLeftTOCB, tb
	);
}

/*
 *	Tool Bar scroll left GtkButton "released" signal callback.
 */
static void ToolBarScrollLeftReleasedCB(GtkWidget *widget, gpointer data)
{
	ToolBar *tb = TOOL_BAR(data);
	if(tb == NULL)
	    return;

	if(tb->freeze_count > 0)
	    return;

	tb->scroll_toid = GTK_TIMEOUT_REMOVE(tb->scroll_toid);
}

/*
 *	Tool Bar scroll right GtkButton "pressed" signal callback.
 */
static void ToolBarScrollRightPressedCB(GtkWidget *widget, gpointer data)
{
	ToolBar *tb = TOOL_BAR(data);
	if(tb == NULL)
	    return;

	if(tb->freeze_count > 0)
	    return;

	if(tb->scroll_toid != 0)
	    return;

	ToolBarScrollRightIncrement(tb);

	tb->scroll_toid = gtk_timeout_add(
	    TOOL_BAR_SCROLL_INITIAL_INT,
	    ToolBarScrollRightTOCB, tb
	);
}

/*
 *	Tool Bar scroll right GtkButton "released" signal callback.
 */
static void ToolBarScrollRightReleasedCB(GtkWidget *widget, gpointer data)
{
	ToolBar *tb = TOOL_BAR(data);
	if(tb == NULL)
	    return;

	if(tb->freeze_count > 0)
	    return;

	tb->scroll_toid = GTK_TIMEOUT_REMOVE(tb->scroll_toid);
}

/*
 *	Scroll left timeout callback.
 */
static gint ToolBarScrollLeftTOCB(gpointer data)
{
	ToolBar *tb = TOOL_BAR(data);
	if(tb == NULL)
	    return(FALSE);

	tb->scroll_toid = 0;

	if(tb->freeze_count > 0)
	    return(FALSE);

	ToolBarScrollLeftIncrement(tb);

	tb->scroll_toid = gtk_timeout_add(
	    TOOL_BAR_SCROLL_INT,
	    ToolBarScrollLeftTOCB, tb
	);

	return(FALSE);
}

/*
 *	Scroll right timeout callback.
 */
static gint ToolBarScrollRightTOCB(gpointer data)
{
	ToolBar *tb = TOOL_BAR(data);
	if(tb == NULL)
	    return(FALSE);

	tb->scroll_toid = 0;

	if(tb->freeze_count > 0)
	    return(FALSE);

	ToolBarScrollRightIncrement(tb);

	tb->scroll_toid = gtk_timeout_add(
	    TOOL_BAR_SCROLL_INT,
	    ToolBarScrollRightTOCB, tb
	);

	return(FALSE);
}


/*
 *	ToolBarItem "enter_notify_event" signal callback.
 */
static gint ToolBarEnterNotifyCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	ToolBar *tb;
	ToolBarItem *item = TOOL_BAR_ITEM(data);
	if(item == NULL)
	    return(TRUE);

	if(item->freeze_count > 0)
	    return(TRUE);

	tb = item->tb;
	if(tb == NULL)
	    return(TRUE);

	tb->callback_level++;

	if(item->enter_cb != NULL)
	    item->enter_cb(
		item,			/* ToolBarItem */
		item->id,		/* ID */
		item->enter_data	/* Data */
	    );

	tb->callback_level--;

	return(TRUE);
}

/*
 *	ToolBarItem "leave_notify_event" signal callback.
 */
static gint ToolBarLeaveNotifyCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	ToolBar *tb;
	ToolBarItem *item = TOOL_BAR_ITEM(data);
	if(item == NULL)
	    return(TRUE);

	if(item->freeze_count > 0)
	    return(TRUE);

	tb = item->tb;
	if(tb == NULL)
	    return(TRUE);

	tb->callback_level++;

	if(item->leave_cb != NULL)
	    item->leave_cb(
		item,			/* ToolBarItem */
		item->id,		/* ID */
		item->leave_data	/* Data */
	    );

	tb->callback_level--;

	return(TRUE);
}

/*
 *	ToolBarItem "clicked" or "toggled" signal callback.
 */
static void ToolBarClickedCB(GtkWidget *widget, gpointer data)
{
	ToolBar *tb;
	ToolBarItem *item = TOOL_BAR_ITEM(data);
	if(item == NULL)
	    return;

	if(item->freeze_count > 0)
	    return;

	tb = item->tb;
	if(tb == NULL)
	    return;

	tb->callback_level++;

	if(item->activate_cb != NULL)
	    item->activate_cb(
		item,				/* ToolBarItem */
		item->id,			/* ID */
		item->activate_data		/* Data */
	    );

	tb->callback_level--;
}


/*
 *	ToolBarItem GtkWidget "destroy" signal callback.
 */
static void ToolBarItemDestroyCB(gpointer data)
{
	ToolBarItem *item = TOOL_BAR_ITEM(data);
	if(item == NULL)
	    return;

#if 0
g_print(
 "ToolBarItemDestroyCB() item=%p item->text=\"%s\"\n",
 item,
 item->text
);
#endif

	item->freeze_count++;

	/* Remove any references to this ToolBarItem on its ToolBar */
	if(item->tb != NULL)
	{
	    ToolBar *tb = item->tb;
	    tb->items_list = g_list_remove(
		tb->items_list,
		item
	    );
	}

	/* Mark the GtkWidget as being destroyed */
	item->w = NULL;

	item->freeze_count--;

	/* Notify about this ToolBarItem being deleted and delete the
	 * rest of this ToolBarItem except for its GtkWidget
	 */
	ToolBarItemDelete(item);
}

/*
 *	ToolBarItem toplevel GtkWidget "focus_in_event" or
 *	"focus_out_event" signal callback.
 */
static gint ToolBarItemFocusEventCB(
	GtkWidget *widget, GdkEventFocus *focus, gpointer data
)
{
	gint status = FALSE;
	ToolBar *tb;
	ToolBarItem *item = TOOL_BAR_ITEM(data);
	if((widget == NULL) || (focus == NULL) || (item == NULL))
	    return(status);

	tb = item->tb;
	if(tb == NULL)
	    return(status);

	if(focus->in)
	{
	    if(ToolBarItemGetVisibility(item) != GDK_VISIBILITY_UNOBSCURED)
		ToolBarScrollToItem(
		    tb->toplevel,
		    ToolBarItemGetID(item),
		    0.5f
		);

	    status = TRUE;
	}

	return(status);
}

/*
 *	ToolBarItem toplevel GtkWidget "visibility_notify_event"
 *	signal callback.
 */
static gint ToolBarItemVisibilityEventCB(
	GtkWidget *widget, GdkEventVisibility *visibility, gpointer data
)
{
	ToolBarItem *item = TOOL_BAR_ITEM(data);
	if((widget == NULL) || (visibility == NULL) || (item == NULL))
	    return(FALSE);

	item->visibility = visibility->state;

	return(TRUE);
}


/*
 *	Creates a new ToolBarItem.
 *
 *	The item's GtkWidget will not be created.
 */
ToolBarItem *ToolBarItemNew(
	const ToolBarItemType type,
	const gchar *text,
	guint8 **icon_data,
	const gchar *tooltip,
	const gint id,
	void (*activate_cb)(
		ToolBarItem *,
		const gint,
		gpointer
	),
	gpointer activate_data,
	void (*enter_cb)(
		ToolBarItem *,
		const gint,
		gpointer
	),
	gpointer enter_data,
	void (*leave_cb)(
		ToolBarItem *,
		const gint,
		gpointer
	),
	gpointer leave_data
)
{
	ToolBarItem *item = TOOL_BAR_ITEM(g_malloc0(
	    sizeof(ToolBarItem)
	));
	if(item == NULL)
	    return(NULL);

	item->type = type;
/*
	item->freeze_count = 0;
	item->w = NULL;
 */
 	item->text = STRDUP(text);
	item->icon_data = icon_data;
	item->tooltip = STRDUP(tooltip);
	item->id = id;
	item->activate_cb = activate_cb;
	item->activate_data = activate_data;
	item->enter_cb = enter_cb;
	item->enter_data = enter_data;
	item->leave_cb = leave_cb;
	item->leave_data = leave_data;

	return(item);
}

/*
 *	Creates a new ToolBarItem of type TOOL_BAR_ITEM_SEPARATOR.
 */
ToolBarItem *ToolBarItemNewSeparator(void)
{
	return(ToolBarItemNew(
	    TOOL_BAR_ITEM_SEPARATOR,
	    NULL,
	    NULL,
	    NULL,
	    0,
	    NULL, NULL,
	    NULL, NULL,
	    NULL, NULL
	));
}

/*
 *	Deletes the ToolBarItem.
 *
 *	The ToolBarItem's GtkWidget will not be destroyed, because
 *	this function is intended to be called by user functions and
 *	ToolBarItemDestroyCB() when its GtkWidget is destroyed.
 */
void ToolBarItemDelete(ToolBarItem *item)
{
	if(item == NULL)
	    return;

	if(item->destroy_cb != NULL)
	    item->destroy_cb(item->data);

	item->freeze_count++;

	g_free(item->text);
	g_free(item->tooltip);

	item->freeze_count--;

	g_free(item);
}

/*
 *	Gets the ToolBarItem's GtkWidget.
 */
ToolBarItemType ToolBarItemGetType(ToolBarItem *item)
{
	if(item == NULL)
	    return(0);

	return(item->type);
}

/*
 *	Gets the ToolBarItem's GtkWidget.
 */
GtkWidget *ToolBarItemGetWidget(ToolBarItem *item)
{
	if(item == NULL)
	    return(NULL);

	return(item->w);
}

/*
 *	Gets the ToolBarItem's ToolBar.
 */
GtkWidget *ToolBarItemGetToolBar(ToolBarItem *item)
{
	ToolBar *tb;

	if(item == NULL)
	    return(NULL);

	tb = item->tb;
	if(tb == NULL)
	    return(NULL);

	return(tb->toplevel);
}

/*
 *	Gets the ToolBarItem's ID.
 */
gint ToolBarItemGetID(ToolBarItem *item)
{
	if(item == NULL)
	    return(0);

	return(item->id);
}

/*
 *	Sets the ToolBarItem's user data.
 */
void ToolBarItemSetData(
	ToolBarItem *item,
	gpointer data
)
{
	ToolBarItemSetDataFull(
	    item,
	    data,
	    NULL
	);
}

/*
 *	Sets the ToolBarItem's user data with a destroy_cb.
 */
void ToolBarItemSetDataFull(
	ToolBarItem *item,
	gpointer data,
	GtkDestroyNotify destroy_cb
)
{
	if(item == NULL)
	    return;

	item->data = data;
	item->destroy_cb = destroy_cb;
}

/*
 *	Gets the ToolBarItem's user data.
 */
gpointer ToolBarItemGetData(ToolBarItem *item)
{
	if(item == NULL)
	    return(NULL);

	return(item->data);
}

/*
 *	Gets the ToolBarItem's visibility.
 */
GdkVisibilityState ToolBarItemGetVisibility(ToolBarItem *item)
{
	if(item == NULL)
	    return(GDK_VISIBILITY_FULLY_OBSCURED);

	return(item->visibility);
}

/*
 *	Sets the ToolBarItem's toggled value.
 *
 *	The item specifies the ToolBarItem which must be of type
 *	TOOL_BAR_ITEM_TOGGLE_BUTTON.
 *
 *	The toggled specifies the toggled value.
 *
 *	The ToolBarItem's activate signal callback will be called.
 */
void ToolBarItemSetToggle(
	ToolBarItem *item,
	const gboolean toggled
)
{
	GtkToggleButton *toggle_button;
	GtkWidget *w = ToolBarItemGetWidget(item);
	if(w == NULL)
	    return;

	if(item->type != TOOL_BAR_ITEM_TOGGLE_BUTTON)
	    return;

	toggle_button = GTK_TOGGLE_BUTTON(w);

	if(toggle_button->active == toggled)
	    return;

	/* Set the GtkToggleButton's toggled value and call the
	 * ToolBarItem's "toggled" signal callback
	 */
	gtk_toggle_button_set_active(toggle_button, toggled);
}

/*
 *	Gets the ToolBarItem's toggled value.
 *
 *	The item specifies the ToolBarItem which must be of type
 *	TOOL_BAR_ITEM_TOGGLE_BUTTON.
 *
 *	Returns the ToolBarItem's toggled value.
 */
gboolean ToolBarItemGetToggle(ToolBarItem *item)
{
	GtkWidget *w = ToolBarItemGetWidget(item);
	if(w == NULL)
	    return(FALSE);

	if(item->type != TOOL_BAR_ITEM_TOGGLE_BUTTON)
	    return(FALSE);

	return(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)));
}

/*
 *	Sets the ToolBarItem's GtkMenu.
 *
 *	The item specifies the ToolBarItem which must be of type
 *	TOOL_BAR_ITEM_MENU_BUTTON.
 *
 *	The menu specifies the GtkMenu to set. If there is an existing
 *	GtkMenu set on the ToolBarItem then that GtkMenu will be
 *	destroyed.
 */
void ToolBarItemSetMenu(
	ToolBarItem *item,
	GtkWidget *menu
)
{
	GtkWidget *w = ToolBarItemGetWidget(item);
	if(w == NULL)
	    return;

	if(item->type != TOOL_BAR_ITEM_MENU_BUTTON)
	    return;

	menu_button_set_menu(w, GTK_MENU(menu));
}

/*
 *	Gets the ToolBarItem's GtkMenu.
 *
 *	The item specifies the ToolBarItem which must be of type
 *	TOOL_BAR_ITEM_MENU_BUTTON.
 *
 *	Returns the GtkMenu.
 */
GtkWidget *ToolBarItemGetMenu(ToolBarItem *item)
{
	GtkWidget *w = ToolBarItemGetWidget(item);
	if(w == NULL)
	    return(NULL);

	if(item->type != TOOL_BAR_ITEM_MENU_BUTTON)
	    return(NULL);

	return(menu_button_get_menu(w));
}


/*
 *	Gets the ToolBarItem in the list by ID.
 *
 *	The items_list specifies a GList of ToolBarItem * items.
 *
 *	The id specifies the ID.
 *
 *	Returns the ToolBarItem in the items_list who's ID matches
 *	the specified ID or NULL on error.
 */
ToolBarItem *ToolBarItemListMatchByID(
	GList *items_list,
	const gint id
)
{
	GList *glist;
	ToolBarItem *item;

	for(glist = items_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    item = TOOL_BAR_ITEM(glist->data);
	    if(item == NULL)
		continue;

	    if(item->id == id)
		return(item);
	}

	return(NULL);
}


/*
 *	Clips the scroll bar positions.
 *
 *	If the scroll position has been clipped then a "value_changed"
 *	signal will be emitted to the scroll GtkAdjustment.
 */
static void ToolBarClipScrollPosition(ToolBar *tb)
{
	GtkAdjustment *adj = tb->adj;
	const gfloat prev_value = adj->value;

	/* If the scroll buttons are mapped then we need to clip
	 * the scroll position to the current size of the buttons
	 * toplevel GtkBox, otherwise the scroll position should
	 * always be at the lower bounds
	 */
	if(tb->flags & TOOL_BAR_SCROLL_BUTTONS_MAPPED)
	{
	    if(adj->value > (adj->upper - adj->page_size))
		adj->value = adj->upper - adj->page_size;
	    if(adj->value < adj->lower)
		adj->value = adj->lower;
	}
	else
	{
	    adj->value = adj->lower;
	}
	if(adj->value != prev_value)
	    gtk_adjustment_value_changed(adj);
}

/*
 *	Scrolls one increment to the left.
 */
static void ToolBarScrollLeftIncrement(ToolBar *tb)
{
	GtkAdjustment *adj = tb->adj;
	GTK_ADJUSTMENT_SET_VALUE(
	    adj,
	    adj->value - adj->step_increment
	);
}

/*
 *	Scrolls one increment to the right.
 */
static void ToolBarScrollRightIncrement(ToolBar *tb)
{
	GtkAdjustment *adj = tb->adj;
	GTK_ADJUSTMENT_SET_VALUE(
	    adj,
	    adj->value + adj->step_increment
	);
}

/*
 *	Maps/unmaps the Tool Bar's scroll buttons toplevel GtkBox
 *	based on the scroll_buttons_policy and the Tool Bar's
 *	current scroll GtkAdjustment values.
 *
 *	If the scroll buttons are mapped or unmapped then
 *	ToolBarQueueResize() will be called to resize. The scroll
 *	position is not clipped so you need to call
 *	ToolBarClipScrollPosition() afterwards.
 */
static void ToolBarRemapScrollButtons(ToolBar *tb)
{
	GtkAdjustment *adj = tb->adj;

	/* Are the scroll buttons currently mapped? */
	if(tb->flags & TOOL_BAR_SCROLL_BUTTONS_MAPPED)
	{
	    /* Check if the scroll buttons need to be unmapped */
	    switch(tb->scroll_buttons_policy)
	    {
	      case GTK_POLICY_ALWAYS:
		break;
	      case GTK_POLICY_AUTOMATIC:
		if(tb->toplevel != NULL)
		{
		    GtkWidget *w = tb->toplevel;
		    GtkAllocation *allocation = &w->allocation;
	            const gint  viewport_size = (gint)((tb->orientation == GTK_ORIENTATION_VERTICAL) ?
			allocation->height : allocation->width
		    ),
	                        content_size = (gint)(adj->upper - adj->lower);
	            if((viewport_size > 0) && (viewport_size >= content_size))
	            {
	                tb->flags &= ~TOOL_BAR_SCROLL_BUTTONS_MAPPED;
	                gtk_widget_hide(tb->scroll_buttons_toplevel);
		        ToolBarQueueResize(tb->toplevel);
	            }
		}
		break;
	      case GTK_POLICY_NEVER:
		tb->flags &= ~TOOL_BAR_SCROLL_BUTTONS_MAPPED;
		gtk_widget_hide(tb->scroll_buttons_toplevel);
		ToolBarQueueResize(tb->toplevel);
		break;
	    }
	}
	else
	{
	    /* Check if the scroll buttons need to be mapped */
	    switch(tb->scroll_buttons_policy)
	    {
	      case GTK_POLICY_ALWAYS:
		tb->flags |= TOOL_BAR_SCROLL_BUTTONS_MAPPED;
	        gtk_widget_show(tb->scroll_buttons_toplevel);
	        ToolBarQueueResize(tb->toplevel);
		break;
	      case GTK_POLICY_AUTOMATIC:
		if(tb->toplevel != NULL)
		{
	            GtkWidget *w = tb->toplevel;
		    GtkAllocation *allocation = &w->allocation;
	            const gint  viewport_size = (gint)((tb->orientation == GTK_ORIENTATION_VERTICAL) ?
			allocation->height : allocation->width
		    ),
	                        content_size = (gint)(adj->upper - adj->lower);
	            if((viewport_size > 0) && (viewport_size < content_size))
		    {
	                tb->flags |= TOOL_BAR_SCROLL_BUTTONS_MAPPED;
	                gtk_widget_show(tb->scroll_buttons_toplevel);
	                ToolBarQueueResize(tb->toplevel);
	            }
		}
		break;
	      case GTK_POLICY_NEVER:
		break;
	    }
	}
}


/*
 *	Gets the ToolBar data from the GtkWidget.
 */
static ToolBar *ToolBarGetWidgetData(
	GtkWidget *w,
	const gchar *func_name
)
{
	const gchar *key = TOOL_BAR_KEY;
	ToolBar *tb;

	if(w == NULL)
	    return(NULL);

	tb = TOOL_BAR(gtk_object_get_data(
	    GTK_OBJECT(w),
	    key
	));
	if(tb == NULL)
	{
	    g_printerr(
"%s(): Warning: GtkWidget %p:\
 Unable to find the data that matches the key \"%s\".\n",
		func_name,
		w,
		key
	    );
	    return(NULL);
	}

	return(tb);
} 

/*
 *	Creates a new Tool Bar.
 *
 *	The items_list specifies a GList of ToolBarItem * items
 *	to create. This list and each item is not modified or deleted by
 *	this function.
 *
 *	Returns the new Tool Bar or NULL on error.
 */
GtkWidget *ToolBarNew(
	const GtkOrientation orientation,
	const ToolBarButtonDecals button_decals,
	const GtkReliefStyle relief,
	const GtkPolicyType scroll_policy,
	GList *items_list
)
{
	const gint border_major = 5;
	gint		button_width,
			button_height;
	GList *glist;
	GtkWidget	*w,
			*parent, *parent2, *parent3,
			*label_rtn,
			*menu_rtn;
	ToolBarItem *item;
	ToolBar *tb = TOOL_BAR(g_malloc0(
	    sizeof(ToolBar)
	));
	if(tb == NULL)
	    return(NULL);

/*
	tb->freeze_count = 0;
	tb->callback_level = 0;
	tb->flags = 0;
 */
	tb->orientation = orientation;
	tb->button_decals = button_decals;
	tb->relief = relief;
	tb->button_width = -1;
	tb->button_height = -1;
	tb->scroll_buttons_policy = scroll_policy;
/*
	tb->scroll_tocb = 0;
	tb->resize_idle_id = 0;
 */
	tb->adj = (GtkAdjustment *)gtk_adjustment_new(
	    0.0f, 0.0f, 0.0f,
	    0.0f, 0.0f, 0.0f
	);
	gtk_signal_connect(
	    GTK_OBJECT(tb->adj), "value_changed",
	    GTK_SIGNAL_FUNC(ToolBarScrollAdjustmentValueChangedCB), tb
	);
/*	tb->items_list = NULL; */

	tb->freeze_count++;

	/* Toplevel GtkBox */
	if(orientation == GTK_ORIENTATION_VERTICAL)
	    w = gtk_vbox_new(FALSE, 0);
	else
	    w = gtk_hbox_new(FALSE, 0);
	tb->toplevel = w;
	gtk_object_set_data_full(
	    GTK_OBJECT(w), TOOL_BAR_KEY,
	    tb, ToolBarDestroyCB
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "show",
	    GTK_SIGNAL_FUNC(ToolBarShowCB), tb
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "hide",
	    GTK_SIGNAL_FUNC(ToolBarHideCB), tb
	);
	parent = w;

	/* Create a GtkFixed to use as the viewport and control the
	 * scrolling
	 */
	tb->fixed = w = gtk_fixed_new();
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "size_allocate",
	    GTK_SIGNAL_FUNC(ToolBarOutsideSizeAllocateCB), tb
	);
	gtk_widget_show(w);
	parent2 = w;

	/* GtkEventBox for the ToolBarItem GtkWidgets so that they
	 * have a GdkWindow within the viewport
	 */
	tb->buttons_event_box = w = gtk_event_box_new();
	gtk_fixed_put(GTK_FIXED(parent2), w, 0, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "realize",
	    GTK_SIGNAL_FUNC(ToolBarRealizeCB), tb
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "size_allocate",
	    GTK_SIGNAL_FUNC(ToolBarInsideSizeAllocateCB), tb
	);
	gtk_widget_show(w);
	parent3 = w;

	/* Buttons Toplevel GtkBox to parent all ToolBarItem
	 * GtkWidgets to
	 */
	if(orientation == GTK_ORIENTATION_VERTICAL)
	    w = gtk_vbox_new(FALSE, 0);
	else
	    w = gtk_hbox_new(FALSE, 0);
	tb->buttons_toplevel = w;
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_show(w);
	parent3 = w;


	/* Scroll Buttons Toplevel GtkBox */
	if(orientation == GTK_ORIENTATION_VERTICAL)
	    w = gtk_vbox_new(FALSE, 0);
	else
	    w = gtk_hbox_new(FALSE, 0);
	tb->scroll_buttons_toplevel = w;
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	parent2 = w;

	/* Scroll Left GtkButton */
	if(orientation == GTK_ORIENTATION_VERTICAL)
	    w = GUIButtonArrow(
		GTK_ARROW_UP,
		-1, 10
	    );
	else
	    w = GUIButtonArrow(
		GTK_ARROW_LEFT,
		10, -1
	    );
	tb->scroll_left_btn = w;
	gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "pressed",
	    GTK_SIGNAL_FUNC(ToolBarScrollLeftPressedCB), tb
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "released",
	    GTK_SIGNAL_FUNC(ToolBarScrollLeftReleasedCB), tb
	);
	gtk_widget_show(w);

	/* Scroll Right GtkButton */
	if(orientation == GTK_ORIENTATION_VERTICAL)
	    w = GUIButtonArrow(
		GTK_ARROW_DOWN,
		-1, 10
	    );
	else
	    w = GUIButtonArrow(
		GTK_ARROW_RIGHT,
		10, -1
	    );
	tb->scroll_right_btn = w;
	gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "pressed",
	    GTK_SIGNAL_FUNC(ToolBarScrollRightPressedCB), tb
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "released",
	    GTK_SIGNAL_FUNC(ToolBarScrollRightReleasedCB), tb
	);
	gtk_widget_show(w);

	/* Get the button size */
	ToolBarGetButtonSize(
	    tb->toplevel,
	    &button_width, &button_height
	);

	/* Create the ToolBarItems */
	parent = tb->buttons_toplevel;
	for(glist = items_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    item = TOOL_BAR_ITEM(glist->data);
	    if(item == NULL)
		continue;

	    /* Copy this item and add it to the Toolbar's Items list */
	    item = ToolBarItemNew(
		item->type,
		item->text,
		item->icon_data,
		item->tooltip,
		item->id,
		item->activate_cb,
		item->activate_data,
		item->enter_cb,
		item->enter_data,
		item->leave_cb,
		item->leave_data
	    );
	    if(item == NULL)
		continue;

	    item->tb = tb;

	    tb->items_list = g_list_append(
		tb->items_list,
		item
	    );

	    /* Create this item's GtkWidgets based on the item's type */
	    switch(item->type)
	    {
	      case TOOL_BAR_ITEM_SEPARATOR:
		if(orientation == GTK_ORIENTATION_VERTICAL)
		    w = gtk_hbox_new(TRUE, 0);
		else
		    w = gtk_vbox_new(TRUE, 0);
		item->w = w;
		if(orientation == GTK_ORIENTATION_VERTICAL)
		    gtk_widget_set_usize(
			w,
			-1, 5
		    );
		else
		    gtk_widget_set_usize(
			w,
			5, -1
		    );
		gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent2 = w;

		if(orientation == GTK_ORIENTATION_VERTICAL)
		    w = gtk_hseparator_new();
		else
		    w = gtk_vseparator_new();
		gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, border_major);
		gtk_widget_show(w);
		break;

	      case TOOL_BAR_ITEM_BUTTON:
		item->w = w = GUIButtonPixmapLabelV(
		    (guint8 **)item->icon_data,
		    item->text,
		    &label_rtn
		);
		gtk_button_set_relief(GTK_BUTTON(w), relief);
		gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
		gtk_widget_add_events(
		    w,
		    GDK_FOCUS_CHANGE_MASK | GDK_VISIBILITY_NOTIFY_MASK
		);
		gtk_signal_connect(
		    GTK_OBJECT(w), "focus_in_event",
		    GTK_SIGNAL_FUNC(ToolBarItemFocusEventCB), item
		);
		gtk_signal_connect(
		    GTK_OBJECT(w), "focus_out_event",
		    GTK_SIGNAL_FUNC(ToolBarItemFocusEventCB), item
		);
		gtk_signal_connect(
		    GTK_OBJECT(w), "visibility_notify_event",
		    GTK_SIGNAL_FUNC(ToolBarItemVisibilityEventCB), item
		);
		if(item->activate_cb != NULL)
		    gtk_signal_connect(
			GTK_OBJECT(w), "clicked",
			GTK_SIGNAL_FUNC(ToolBarClickedCB), item
		    );
		if(item->enter_cb != NULL)
		    gtk_signal_connect(
			GTK_OBJECT(w), "enter_notify_event",
			GTK_SIGNAL_FUNC(ToolBarEnterNotifyCB), item
		    );
		if(item->leave_cb != NULL)
		    gtk_signal_connect(
			GTK_OBJECT(w), "leave_notify_event",
			GTK_SIGNAL_FUNC(ToolBarLeaveNotifyCB), item
		    );
		if(item->tooltip != NULL)
		    GUISetWidgetTip(w, item->tooltip);
		TOOL_BAR_CHANGE_BUTTON_LAYOUT(
		    w,
		    button_decals,
		    button_width, button_height
		);
		gtk_widget_show(w);
		break;

	      case TOOL_BAR_ITEM_TOGGLE_BUTTON:
		item->w = w = GUIToggleButtonPixmapLabelV(
		    (guint8 **)item->icon_data,
		    item->text,
		    &label_rtn
		);
		gtk_button_set_relief(GTK_BUTTON(w), relief);
		gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
		gtk_widget_add_events(
		    w,
		    GDK_FOCUS_CHANGE_MASK | GDK_VISIBILITY_NOTIFY_MASK
		);
		gtk_signal_connect(
		    GTK_OBJECT(w), "focus_in_event",
		    GTK_SIGNAL_FUNC(ToolBarItemFocusEventCB), item
		);
		gtk_signal_connect(
		    GTK_OBJECT(w), "focus_out_event",
		    GTK_SIGNAL_FUNC(ToolBarItemFocusEventCB), item
		);
		gtk_signal_connect(
		    GTK_OBJECT(w), "visibility_notify_event",
		    GTK_SIGNAL_FUNC(ToolBarItemVisibilityEventCB), item
		);
		if(item->activate_cb != NULL)
		    gtk_signal_connect(
			GTK_OBJECT(w), "toggled",
			GTK_SIGNAL_FUNC(ToolBarClickedCB), item
		    );
		if(item->enter_cb != NULL)
		    gtk_signal_connect(
			GTK_OBJECT(w), "enter_notify_event",
			GTK_SIGNAL_FUNC(ToolBarEnterNotifyCB), item
		    );
		if(item->leave_cb != NULL)
		    gtk_signal_connect(
			GTK_OBJECT(w), "leave_notify_event",
			GTK_SIGNAL_FUNC(ToolBarLeaveNotifyCB), item
		    );
		if(item->tooltip != NULL)
		    GUISetWidgetTip(w, item->tooltip);
		TOOL_BAR_CHANGE_BUTTON_LAYOUT(
		    w,
		    button_decals,
		    button_width, button_height
		);
		gtk_widget_show(w);
		break;

	      case TOOL_BAR_ITEM_MENU_BUTTON:
		item->w = w = menu_vbutton_new(
		    item->text,
		    (guint8 **)item->icon_data,
		    &menu_rtn
		);
		gtk_button_set_relief(GTK_BUTTON(w), relief);
		gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
		gtk_widget_add_events(
		    w,
		    GDK_FOCUS_CHANGE_MASK | GDK_VISIBILITY_NOTIFY_MASK
		);
		gtk_signal_connect(
		    GTK_OBJECT(w), "focus_in_event",
		    GTK_SIGNAL_FUNC(ToolBarItemFocusEventCB), item
		);
		gtk_signal_connect(
		    GTK_OBJECT(w), "focus_out_event",
		    GTK_SIGNAL_FUNC(ToolBarItemFocusEventCB), item
		);
		gtk_signal_connect(
		    GTK_OBJECT(w), "visibility_notify_event",
		    GTK_SIGNAL_FUNC(ToolBarItemVisibilityEventCB), item
		);
		if(item->activate_cb != NULL)
		    gtk_signal_connect(
			GTK_OBJECT(w), "clicked",
			GTK_SIGNAL_FUNC(ToolBarClickedCB), item
		    );
		if(item->enter_cb != NULL)
		    gtk_signal_connect(
			GTK_OBJECT(w), "enter_notify_event",
			GTK_SIGNAL_FUNC(ToolBarEnterNotifyCB), item
		    );
		if(item->leave_cb != NULL)
		    gtk_signal_connect(
			GTK_OBJECT(w), "leave_notify_event",
			GTK_SIGNAL_FUNC(ToolBarLeaveNotifyCB), item
		    );
		if(item->tooltip != NULL)
		    GUISetWidgetTip(w, item->tooltip);
		TOOL_BAR_CHANGE_BUTTON_LAYOUT(
		    w,
		    button_decals,
		    button_width, button_height
		);
		gtk_widget_show(w);
		break;
	    }
	    if(item->w != NULL)
		gtk_object_set_data_full(
		    GTK_OBJECT(item->w), TOOL_BAR_ITEM_KEY,
		    item, ToolBarItemDestroyCB
		);
	}

	tb->freeze_count--;

	return(tb->toplevel);
}

/*
 *	Sets the button decals.
 */
void ToolBarSetButtonDecals(
	GtkWidget *w,
	const ToolBarButtonDecals button_decals
)
{
	gint		nitems_updated,
			button_width,
			button_height;
	GList *glist;
	ToolBarItem *item;
	ToolBar *tb = ToolBarGetWidgetData(
	    w,
	    "ToolBarSetButtonDecals"
	);
	if(tb == NULL)
	    return;

	if(tb->button_decals == button_decals)
	    return;

	tb->button_decals = button_decals;

	/* Get the button size */
	ToolBarGetButtonSize(
	    tb->toplevel,
	    &button_width, &button_height
	);

	/* Update each ToolBarItem's button decals and size */
	nitems_updated = 0;
	for(glist = tb->items_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    item = TOOL_BAR_ITEM(glist->data);
	    w = ToolBarItemGetWidget(item);
	    if(w == NULL)
		continue;

	    switch(item->type)
	    {
	      case TOOL_BAR_ITEM_SEPARATOR:
		break;
	      case TOOL_BAR_ITEM_BUTTON:
	      case TOOL_BAR_ITEM_TOGGLE_BUTTON:
	      case TOOL_BAR_ITEM_MENU_BUTTON:
		TOOL_BAR_CHANGE_BUTTON_LAYOUT(
		    w,
		    button_decals,
		    button_width, button_height
		);
		nitems_updated++;
		break;
	    }
	}

	/* Queue the ToolBar to resize since changing the button decals
	 * will change the size of the Tool Bar, the scroll position
	 * will be clipped later when the size change is detected
	 */
	if(nitems_updated > 0)
	    ToolBarQueueResize(tb->toplevel);
}

/*
 *	Sets the relief.
 */
void ToolBarSetRelief(
	GtkWidget *w,
	const GtkReliefStyle relief
)
{
	GList *glist;
	ToolBarItem *item;
	ToolBar *tb = ToolBarGetWidgetData(
	    w,
	    "ToolBarSetRelief"
	);
	if(tb == NULL)
	    return;

	if(tb->relief == relief)
	    return;

	tb->relief = relief;

	/* Set the relief on each item */
	for(glist = tb->items_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    item = TOOL_BAR_ITEM(glist->data);
	    w = ToolBarItemGetWidget(item);
	    if(w == NULL)
		continue;

	    switch(item->type)
	    {
	      case TOOL_BAR_ITEM_SEPARATOR:
		break;
	      case TOOL_BAR_ITEM_BUTTON:
	      case TOOL_BAR_ITEM_TOGGLE_BUTTON:
	      case TOOL_BAR_ITEM_MENU_BUTTON:
		if(GTK_IS_BUTTON(w))
		    gtk_button_set_relief(GTK_BUTTON(w), relief);
		break;
	    }
	}
}

/*
 *	Sets the size of each ToolBarItem.
 *
 *	The width and height specifies the size of each ToolBarItem.
 *	If either width or height is -1 then the default size will be
 *	used for that dimension.
 */
void ToolBarSetButtonSize(
	GtkWidget *w,
	const gint width, const gint height
)
{
	gint		nitems_updated,
			button_width,
			button_height;
	GList *glist;
	ToolBarItem *item;
	ToolBar *tb = ToolBarGetWidgetData(
	    w,
	    "ToolBarSetButtonSize"
	);
	if(tb == NULL)
	    return;

	if((width == tb->button_width) &&
	   (height == tb->button_height)
	)
	    return;

	/* Set the new button size */
	tb->button_width = width;
	tb->button_height = height;

	/* Reget the button size and apply it to each button, the we
	 * need to reget the buttons because some sizes have default
	 * sizes which are set if width or height were -1
	 */
	ToolBarGetButtonSize(
	    tb->toplevel,
	    &button_width, &button_height
	);
	nitems_updated = 0;
	for(glist = tb->items_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    item = TOOL_BAR_ITEM(glist->data);
	    w = ToolBarItemGetWidget(item);
	    if(w == NULL)
		continue;

	    switch(item->type)
	    {
	      case TOOL_BAR_ITEM_SEPARATOR:
		break;
	      case TOOL_BAR_ITEM_BUTTON:
	      case TOOL_BAR_ITEM_TOGGLE_BUTTON:
	      case TOOL_BAR_ITEM_MENU_BUTTON:
		gtk_widget_set_usize(w, button_width, button_height);
		nitems_updated++;
		break;
	    }
	}

	if(nitems_updated > 0)
	    ToolBarQueueResize(tb->toplevel);
}

/*
 *	Gets the current size for all the ToolBarItems.
 */
void ToolBarGetButtonSize(
	GtkWidget *w,
	gint *width_rtn, gint *height_rtn
)
{
	ToolBar *tb = ToolBarGetWidgetData(
	    w,
	    "ToolBarGetButtonSize"
	);

	if(width_rtn != NULL)
	    *width_rtn = 0;
	if(height_rtn != NULL)
	    *height_rtn = 0;

	if(tb == NULL)
	    return;

	switch(tb->button_decals)
	{
	  case TOOL_BAR_BUTTON_DECALS_PICTURES_AND_TEXT:
	    if(width_rtn != NULL)
		*width_rtn = (tb->button_width > 0) ?
		    tb->button_width :
		    TOOL_BAR_BUTTON_PICTURE_AND_TEXT_WIDTH;
	    if(height_rtn != NULL)
		*height_rtn = (tb->button_height > 0) ?
		    tb->button_height :
		    TOOL_BAR_BUTTON_PICTURE_AND_TEXT_HEIGHT;
	    break;
	  case TOOL_BAR_BUTTON_DECALS_PICTURES:
	    if(width_rtn != NULL)
		*width_rtn = (tb->button_width > 0) ?
		    tb->button_width :
		    TOOL_BAR_BUTTON_PICTURE_WIDTH;
	    if(height_rtn != NULL)
		*height_rtn = (tb->button_height > 0) ?
		    tb->button_height :
		    TOOL_BAR_BUTTON_PICTURE_HEIGHT;
	    break;
	  case TOOL_BAR_BUTTON_DECALS_TEXT:
	    if(width_rtn != NULL)
		*width_rtn = (tb->button_width > 0) ?
		    tb->button_width :
		    TOOL_BAR_BUTTON_TEXT_WIDTH;
	    if(height_rtn != NULL)
		*height_rtn = (tb->button_height > 0) ?
		    tb->button_height :
		    TOOL_BAR_BUTTON_TEXT_HEIGHT;
	    break;
	}
}

/*
 *	Sets the scroll policy.
 */
void ToolBarSetScrollPolicy(
	GtkWidget *w,
	const GtkPolicyType scroll_policy
)
{
	ToolBar *tb = ToolBarGetWidgetData(
	    w,
	    "ToolBarSetScrollPolicy"
	);
	if(tb == NULL)
	    return;

	if(tb->scroll_buttons_policy == scroll_policy)
	    return;

	tb->scroll_buttons_policy = scroll_policy;

	/* Map/unmap the scroll buttons due to the scroll policy
	 * changing
	 */
	ToolBarRemapScrollButtons(tb);
}

/*
 *	Scrolls to the ToolBarItem.
 *
 *	The w specifies the ToolBar.
 *
 *	The id specifies the TooBarItem's ID.
 *
 *	The coeff specifies the coefficient to offset the position by
 *	relative to the current size of the ToolBar's viewport.
 */
void ToolBarScrollToItem(
	GtkWidget *w,
	const gint id,
	const gfloat coeff
)
{
	gint		position,
			viewport_size;
	GtkAllocation *allocation;
	GtkAdjustment *adj;
	GtkWidget *parent;
	ToolBarItem *item;
	ToolBar *tb = ToolBarGetWidgetData(
	    w,
	    "ToolBarScrollToItem"
	);
	if(tb == NULL)
	    return;

	adj = tb->adj;
	viewport_size = (gint)adj->page_size;

	item = ToolBarItemListMatchByID(tb->items_list, id);
	if(item == NULL)
	    return;

	w = ToolBarItemGetWidget(item);
	if(w == NULL)
	    return;

	allocation = &w->allocation;
	parent = w->parent;
	if(parent == NULL)
	    return;

	if(tb->orientation == GTK_ORIENTATION_VERTICAL)
	    position = (gint)allocation->y -
		(gint)((viewport_size - (gint)allocation->height) * coeff);
	else
	    position = (gint)allocation->x -
		(gint)((viewport_size - (gint)allocation->width) * coeff);

	GTK_ADJUSTMENT_SET_VALUE(adj, (gfloat)position);
}

/*
 *	Updates the ToolBarItem's text, icon, or tooltip.
 *
 *	The w specifies the ToolBar.
 *
 *	The id specifies the TooBarItem's ID.
 *
 *	The text specifies the TooBarItem's new text to set. If text
 *	is NULL then the TooBarItem's text will not be changed.
 *
 *	The icon_data specifies the TooBarItem's new picture to set.
 *	If icon_data is NULL then the TooBarItem's picture will not
 *	be changed.
 *
 *	The tooltip specifies the TooBarItem's new tooltip to set. If
 *	text is NULL then the TooBarItem's tooltip will not be changed.
 */
void ToolBarUpdateItem(
	GtkWidget *w,
	const gint id,
	const gchar *text,
	guint8 **icon_data,
	const gchar *tooltip
)
{
	gboolean	size_fixed,
			size_changed;
	gint		button_width,
			button_height,
			nitems_updated;
	gchar		*dtext,
			*dtooltip;
	GList *glist;
	ToolBarItem *item;
	ToolBar *tb = ToolBarGetWidgetData(
	    w,
	    "ToolBarUpdateItem"
	);
	if(tb == NULL)
	    return;

	dtext = (text != NULL) ? g_strdup(text) : NULL;
	dtooltip = (tooltip != NULL) ? g_strdup(tooltip) : NULL;

	/* Get the size of each ToolBarItem so that we know if it has
	 * a fixed size or not
	 */
	ToolBarGetButtonSize(
	    tb->toplevel,
	    &button_width, &button_height
	);
	size_fixed = ((button_width > 0) && (button_height > 0)) ? TRUE : FALSE;

	/* Update each item in the items list who's ID matches the
	 * specified ID
	 */
	size_changed = FALSE;
	nitems_updated = 0;
	for(glist = tb->items_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    item = TOOL_BAR_ITEM(glist->data);
	    if(item == NULL)
		continue;

	    if(item->id != id)
		continue;

	    w = ToolBarItemGetWidget(item);
	    if(w != NULL)
	    {
		gboolean changed = FALSE;

		/* Change the text? */
		if(dtext != NULL)
		{
		    if((item->text != NULL) ?
			strcmp((const char *)item->text, (const char *)dtext) : TRUE
		    )
		    {
			g_free(item->text);
			item->text = g_strdup(dtext);
			changed = TRUE;
			switch(tb->button_decals)
			{
			  case TOOL_BAR_BUTTON_DECALS_PICTURES_AND_TEXT:
			  case TOOL_BAR_BUTTON_DECALS_TEXT:
			    if(!size_fixed)
				size_changed = TRUE;
			    break;
			  case TOOL_BAR_BUTTON_DECALS_PICTURES:
			    break;
			}
		    }
		}

		/* Change the picture? */
		if(icon_data != NULL)
		{
		    /* Change in XPM data? */
	    	    if(item->icon_data != icon_data)
		    {
			item->icon_data = icon_data;
			changed = TRUE;
			switch(tb->button_decals)
			{
			  case TOOL_BAR_BUTTON_DECALS_PICTURES_AND_TEXT:
			  case TOOL_BAR_BUTTON_DECALS_PICTURES:
			    if(!size_fixed)
				size_changed = TRUE;
			    break;
			  case TOOL_BAR_BUTTON_DECALS_TEXT:
			    break;
			}
		    }
		}

		/* Change the tooltip? */
		if(tooltip != NULL)
		{
		    if((item->tooltip != NULL) ?
			strcmp((const char *)item->tooltip, (const char *)dtooltip) : TRUE
		    )
		    {
			g_free(item->tooltip);
			item->tooltip = g_strdup(dtooltip);
			changed = TRUE;
		    }
		}

		/* Nothing to change? */
		if(!changed)
		    continue;

		/* Update widget, handle by item type */
		switch(item->type)
		{
		  case TOOL_BAR_ITEM_SEPARATOR:
		    break;
		  case TOOL_BAR_ITEM_BUTTON:
		    GUIButtonPixmapUpdate(
			w,
			icon_data,
			text
		    );
		    break;
		  case TOOL_BAR_ITEM_TOGGLE_BUTTON:
		    GUIButtonPixmapUpdate(
			w,
			icon_data,
			text
		    );
		    break;
		  case TOOL_BAR_ITEM_MENU_BUTTON:
		    GUIButtonPixmapUpdate(
			w,
			icon_data,
			text
		    );
		    break;
		}

		nitems_updated++;

		/* Do not break after matching this item, there may
		 * be other items with the same ID that need to be
		 * updated
		 */
	    }
	}

	g_free(dtext);
	g_free(dtooltip);

	/* If the size has changed then queue the ToolBar to resize,
	 * the scroll position will be clipped later when the size
	 * change is detected
	 */
	if(size_changed)
	    ToolBarQueueResize(tb->toplevel);
}

/*
 *	Gets the ToolBarItem on the ToolBar.
 *
 *	The w specifies the ToolBar.
 *
 *	The id specifies the ToolBarItem.
 *
 *	Returns the ToolBarItem or NULL on error.
 */
ToolBarItem *ToolBarGetItem(
	GtkWidget *w,
	const gint id
)
{
	ToolBar *tb = ToolBarGetWidgetData(
	    w,
	    "ToolBarGetItem"
	);
	if(tb == NULL)
	    return(NULL);

	return(ToolBarItemListMatchByID(tb->items_list, id));
}

/*
 *	Gets the ToolBarItem's GtkWidget by id.
 */
GtkWidget *ToolBarGetItemWidget(
	GtkWidget *w,
	const gint id
)
{
	ToolBarItem *item;
	ToolBar *tb = ToolBarGetWidgetData(
	    w,
	    "ToolBarGetItemWidget"
	);
	if(tb == NULL)
	    return(NULL);

	item = ToolBarItemListMatchByID(tb->items_list, id);
	if(item == NULL)
	    return(NULL);

	return(item->w);
}


/*
 *	Sets the ToolBarItem specified by id as sensitive or
 *	insensitive.
 */
void ToolBarSetItemSensitive(
	GtkWidget *w,
	const gint id,
	const gboolean sensitive
)
{
	w = ToolBarGetItemWidget(w, id);
	if(w == NULL)
	    return;

	gtk_widget_set_sensitive(w, sensitive);
}


/*
 *	Sets the ToolBarItem specified by id as toggled or untoggled.
 *
 *	The ToolBarItem must be of type TOOL_BAR_ITEM_TOGGLE_BUTTON.
 *
 *	The ToolBarItem's activate signal callback will be called.
 */
void ToolBarSetItemToggle(
	GtkWidget *w,
	const gint id,
	const gboolean toggled
)
{
	ToolBarItem *item;
	ToolBar *tb = ToolBarGetWidgetData(
	    w,
	    "ToolBarSetItemToggle"
	);
	if(tb == NULL)
	    return;

	item = ToolBarItemListMatchByID(tb->items_list, id);
	if(item == NULL)
	    return;

	ToolBarItemSetToggle(item, toggled);
}

/*
 *	Gets the ToolBarItem's toggle state.
 */
gboolean ToolBarGetItemToggle(
	GtkWidget *w,
	const gint id
)
{
	ToolBarItem *item;
	ToolBar *tb = ToolBarGetWidgetData(
	    w,
	    "ToolBarGetItemToggle"
	);
	if(tb == NULL)
	    return(FALSE);

	item = ToolBarItemListMatchByID(tb->items_list, id);
	if(item == NULL)
	    return(FALSE);

	return(ToolBarItemGetToggle(item));
}

/*
 *	Sets the ToolBarItem's menu.
 */
void ToolBarSetItemMenu(
	GtkWidget *w,
	const gint id,
	GtkWidget *menu
)
{
	ToolBarItem *item;
	ToolBar *tb = ToolBarGetWidgetData(
	    w,
	    "ToolBarSetItemMenu"
	);
	if(tb == NULL)
	    return;

	item = ToolBarItemListMatchByID(tb->items_list, id);
	if(item == NULL)
	    return;

	ToolBarItemSetMenu(item, menu);
}

/*
 *	Gets the ToolBarItem's menu.
 */
GtkWidget *ToolBarGetItemMenu(
	GtkWidget *w,
	const gint id
)
{
	ToolBarItem *item;
	ToolBar *tb = ToolBarGetWidgetData(
	    w,
	    "ToolBarGetItemMenu"
	);
	if(tb == NULL)
	    return(NULL);

	item = ToolBarItemListMatchByID(tb->items_list, id);
	if(item == NULL)
	    return(NULL);

	return(ToolBarItemGetMenu(item));
}


/*
 *	Checks if the ToolBarItem specified by id is mapped.
 */
gboolean ToolBarIsItemMapped(
	GtkWidget *w,
	const gint id
)
{
	w = ToolBarGetItemWidget(w, id);
	if(w == NULL)
	    return(FALSE);

	return(GTK_WIDGET_MAPPED(w));
}


/*
 *	Maps the ToolBarItem specified by id.
 */
void ToolBarMapItem(
	GtkWidget *w,
	const gint id
)
{
	w = ToolBarGetItemWidget(w, id);
	if(w == NULL)
	    return;

	gtk_widget_show(w);
}

/*
 *	Unmaps the ToolBarItem specified by id.
 */
void ToolBarUnmapItem(
	GtkWidget *w,
	const gint id
)
{
	w = ToolBarGetItemWidget(w, id);
	if(w == NULL)
	    return;

	gtk_widget_hide(w);
}

/*
 *	Queues the Tool Bar to resize.
 */
void ToolBarQueueResize(GtkWidget *w)
{
	ToolBar *tb = ToolBarGetWidgetData(
	    w,
	    "ToolBarQueueResize"
	);
	if(tb == NULL)
	    return;

	if(tb->resize_idle_id != 0)
	    return;

	tb->resize_idle_id = gtk_idle_add_priority(
	    GTK_PRIORITY_RESIZE,
	    ToolBarResizeIdleCB, tb
	);
}
