/*
			    Thumbs List
 */

#ifndef TLIST_H
#define TLIST_H

#include <gtk/gtk.h>


typedef struct _tlist_struct		tlist_struct;
#define TLIST(p)			((tlist_struct *)(p))

typedef struct _tlist_thumb_struct	tlist_thumb_struct;
#define TLIST_THUMB(p)			((tlist_thumb_struct *)(p))


/*
 *	Widget Name:
 */
#define TLIST_WIDGET_NAME		"thumbs-list"


/*
 *	Flags:
 */
typedef enum {
	TLIST_MAPPED			= (1 << 0),
	TLIST_REALIZED			= (1 << 1),
	TLIST_DOUBLE_BUFFER		= (1 << 2),	/* Use double buffer */
	TLIST_SHOW_THUMB_FRAMES		= (1 << 3),	/* Show Thumb Frames */
 	TLIST_SHOW_THUMB_LABELS		= (1 << 4),	/* Show Thumb Labels */
	TLIST_SHOW_TEXTTIPS		= (1 << 5),	/* Show Texttips */
	TLIST_ENABLE_LIST_DRAG_SCROLL	= (1 << 6),	/* Enable middle-click
							 * list scroll */

	TLIST_LIST_DRAG_SCROLL_ACTIVE	= (1 << 8),	/* Middle-click list
							 * scroll is active */
	TLIST_DRAG_ACTIVE		= (1 << 9),	/* A drag is in progress
							 * that this list
							 * started */
	TLIST_DRAG_ACTIVE_MOVE		= (1 << 10),	/* A drag move is in
							 * progress */
	TLIST_DRAG_OVER			= (1 << 11)	/* A drag is currently
							 * over the list */
} tlist_flags;


/*
 *	Thumb Flags:
 */
typedef enum {
	TLIST_THUMB_SENSITIVE		= (1 << 0),
	TLIST_THUMB_SELECTABLE		= (1 << 1)
} tlist_thumb_flags;


/*
 *	Thumb Load States:
 */
typedef enum {
	TLIST_LOAD_STATE_NOT_LOADED,
	TLIST_LOAD_STATE_LOADED,
	TLIST_LOAD_STATE_FAILED
} tlist_load_state;


/*
 *	Thumb:
 */
struct _tlist_thumb_struct {

	tlist_thumb_flags	flags;
	tlist_load_state	load_state;

	GdkPixmap	*pixmap;
	GdkBitmap	*mask;

	gchar		*text;
	GtkVisibility	text_visibility;
	GdkColor	*text_color_fg,
			*text_color_bg;

	GtkCornerType	attribute_icon_placement;
	GList		*attribute_icon_pixmaps_list,	/* Glist of GdkPixmap
							 * * pixmaps */
			*attribute_icon_masks_list;	/* GList of GdkBitmap
							 * * masks */

	gulong		loaded_time;		/* In seconds since EPOCH */

	gpointer	data;

	GtkDestroyNotify	destroy_cb;

};

#define TLIST_THUMB_IS_SENSITIVE(_thumb_)			\
 (((_thumb_) != NULL) ?						\
  ((_thumb_)->flags & TLIST_THUMB_SENSITIVE) : FALSE		\
 )
#define TLIST_THUMB_IS_SELECTABLE(_thumb_)			\
 (((_thumb_) != NULL) ?						\
  ((_thumb_)->flags & TLIST_THUMB_SELECTABLE) : FALSE		\
 )


/*
 *	Thumbs List:
 */
struct _tlist_struct {

	GtkWidget	*toplevel;
	gint		freeze_count;
	GdkColormap	*colormap;
	GdkGC		*gc;

	tlist_flags	flags;
	GtkOrientation	orientation;

	GdkBitmap	*transparent_stipple_bm;

	GdkCursor	*translate_cur;

	GtkWidget	*list_da,		/* List GtkDrawingArea */
			*vsb,			/* Vertical GtkScrollBar */
			*hsb;			/* Horizontal GtkScrollBar */

	GdkPixmap	*list_pm;		/* Back buffer for list_da */

	/* Scroll GtkAdjustments */
	GtkAdjustment	*hadjustment,
			*vadjustment;

	/* Last drag position */
	gint		drag_last_x,
			drag_last_y;

	/* Thumbs */
	tlist_thumb_struct	**thumb;
	gint			total_thumbs;

	GtkSelectionMode	selection_mode;

	/* Selection list, a GList of gint thumb indicies */
	GList		*selection,
			*selection_end;

	gint		focus_thumb,		/* Thumb in focus or -1 for none */
			pointer_over_thumb;	/* Pointer is over this
						 * thumb or -1 for none */

	/* Thumb geometry */
	gint		thumb_width,		/* Size of each thumb */
			thumb_height,
			thumb_border,		/* Padding between a thumb's
						 * contents and its frame */
			thumbs_per_line;	/* Thumbs visible on one "line" */

	/* Move to idle id */
	gint		moveto_thumb;
	gfloat		moveto_coeff;
	guint		moveto_idleid;

	/* Callbacks */
	void	(*select_cb)(
		tlist_struct *,			/* Thumbs List */
		GdkEventButton *,		/* Triggering Event (may be NULL) */
		gint,				/* Thumb */
		gpointer			/* Data */
	);
	gpointer	select_data;

	void	(*unselect_cb)(
		tlist_struct *,			/* Thumbs List */
		GdkEventButton *,		/* Triggering Event (may be NULL) */
		gint,				/* Thumb */
		gpointer			/* Data */
	);
	gpointer	unselect_data;

};


/* Calculate Suitable Thumb Pixmap Size */
extern void TListQueryThumbPixmapSize(
	tlist_struct *tlist,
	const gint img_width, const gint img_height,
	gint *width_rtn, gint *height_rtn
);


/* Drawing */
extern void TListDraw(tlist_struct *tlist);
extern void TListQueueDraw(tlist_struct *tlist);


/* Resize */
extern void TListResize(
	tlist_struct *tlist,
	const gint width, const gint height
);


/* Freeze/Thaw */
extern void TListFreeze(tlist_struct *tlist);
extern void TListThaw(tlist_struct *tlist);


/* Thumbs Add, Set, and Delete */
extern gint TListInsert(
	tlist_struct *tlist,
	const gint thumb_num,
	const gchar *text
);
extern gint TListAppend(
	tlist_struct *tlist,
	const gchar *text
);
extern void TListSetLoadState(
	tlist_struct *tlist,
	const gint thumb_num,
	const tlist_load_state load_state
);
extern void TListSetText(
	tlist_struct *tlist,
	const gint thumb_num,
	const gchar *text
);
extern void TListSetTextColor(
	tlist_struct *tlist,
	const gint thumb_num,
	GdkColor *fg,
	GdkColor *bg
);
extern void TListSetPixmap(
	tlist_struct *tlist,
	const gint thumb_num,
	GdkPixmap *pixmap, GdkBitmap *mask
);
extern void TListSetRGBA(
	tlist_struct *tlist,
	const gint thumb_num,
	const gint width, const gint height,
	const gint bpl,
	const GdkRgbDither dith,
	const guint8 *rgba,
	const gboolean no_enlarge
);
extern void TListSetSensitive(
	tlist_struct *tlist,
	const gint thumb_num,
	const gboolean sensitive
);
extern void TListSetSelectable(
	tlist_struct *tlist,
	const gint thumb_num,
	const gboolean selectable
);
extern void TListSetAttributeIconPlacement(
        tlist_struct *tlist,
        const int thumb_num,
        const GtkCornerType placement
);
extern void TListAppendAttributeIcon(
        tlist_struct *tlist,
        const int thumb_num,
        GdkPixmap *pixmap,
        GdkBitmap *mask
);
extern void TListCleardAttributeIcons(
        tlist_struct *tlist,
        const int thumb_num
);
extern void TListSetThumbData(
	tlist_struct *tlist,
	const gint thumb_num,
	gpointer data
);
extern void TListSetThumbDataFull(
	tlist_struct *tlist,
	const gint thumb_num,
	gpointer data,
	GtkDestroyNotify destroy_cb
);

extern void TListRemove(
	tlist_struct *tlist,
	const gint thumb_num
);
extern void TListClear(tlist_struct *tlist);

/* Thumbs Get */
extern tlist_thumb_struct *TListGetThumb(
	tlist_struct *tlist,
	const gint thumb_num
);
extern tlist_load_state TListGetLoadState(
	tlist_struct *tlist,
	const gint thumb_num
);
extern gboolean TListGetText(
	tlist_struct *tlist,
	const gint thumb_num,
	const gchar **text
);
extern gboolean TListGetPixmap(
	tlist_struct *tlist,
	const gint thumb_num,
	GdkPixmap **pixmap, GdkBitmap **mask
);
extern gpointer TListGetThumbData(
	tlist_struct *tlist,
	const gint thumb_num
);

/* Thumbs Find */
extern gint TListFindThumbFromData(
	tlist_struct *tlist,
	gpointer data
);

/* Selecting */
extern gboolean TListIsThumbSelected(
	tlist_struct *tlist,
	const gint thumb_num
);
extern void TListSelectThumb(
	tlist_struct *tlist,
	const gint thumb_num
);
extern void TListUnselectThumb(
	tlist_struct *tlist,
	const gint thumb_num
);
extern void TListSelectAll(tlist_struct *tlist);
extern void TListUnselectAll(tlist_struct *tlist);

extern gboolean TListGetSelection(
	tlist_struct *tlist,
	const gint x, const gint y,
	gint *thumb_num_rtn,
	gint *thumb_ix_rtn, gint *thumb_iy_rtn
);
extern gboolean TListGetThumbPosition(
	tlist_struct *tlist,
	const gint thumb_num,
	gint *x_rtn, gint *y_rtn
);
extern gboolean TListGetThumbPixmapGeometry(
	tlist_struct *tlist,
	const gint thumb_num,
	gint *x_rtn, gint *y_rtn,
	gint *width_rtn, gint *height_rtn
);
extern gboolean TListGetThumbLabelGeometry(
	tlist_struct *tlist,
	const gint thumb_num,
	gint *x_rtn, gint *y_rtn,
	gint *width_rtn, gint *height_rtn
);


/* Visibility */
extern GtkVisibility TListIsThumbVisible(
	tlist_struct *tlist,
	const gint thumb_num
);


/* Scrolling */
extern void TListMoveTo(
	tlist_struct *tlist,
	const gint thumb_num,
	const gfloat coeff
);
extern void TListQueueMoveTo(
	tlist_struct *tlist,
	const gint thumb_num,
	const gfloat coeff
);


/* Thumbs List */
extern tlist_struct *TListNew(
	const GtkOrientation orientation,
	const gint thumb_width, const gint thumb_height,
	const gint thumb_border,
	void (*select_cb)(tlist_struct *, GdkEventButton *, gint, gpointer),
	gpointer select_data,
	void (*unselect_cb)(tlist_struct *, GdkEventButton *, gint, gpointer),
	gpointer unselect_data
);
extern GtkWidget *TListGetToplevelWidget(tlist_struct *tlist);
extern GtkWidget *TListGetListWidget(tlist_struct *tlist);
extern void TListRealize(tlist_struct *tlist);
extern void TListThumbGeometry(
	tlist_struct *tlist,
	const gint thumb_width, const gint thumb_height,
	const gint thumb_border
);
extern void TListSelectionMode(
	tlist_struct *tlist,
	const GtkSelectionMode selection_mode
);
extern void TListDoubleBuffer(
	tlist_struct *tlist,
	const gboolean double_buffer
);
extern void TListOrientation(
	tlist_struct *tlist,
	const GtkOrientation orientation
);
extern void TListShowThumbFrames(
	tlist_struct *tlist,
	const gboolean show
);
extern void TListShowThumbLabels(
	tlist_struct *tlist,
	const gboolean show
);
extern void TListShowTextTips(
	tlist_struct *tlist,
	const gboolean show
);
extern void TListEnableListDragScroll(
	tlist_struct *tlist,
	const gboolean enable
);
extern void TListMap(tlist_struct *tlist);
extern void TListUnmap(tlist_struct *tlist);
extern void TListGrabFocus(tlist_struct *tlist);
extern void TListDelete(tlist_struct *tlist);


#endif	/* TLIST_H */
