#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkrgb.h>
#include <gdk/gdkkeysyms.h>

#include "guiutils.h"
#include "guirgbimg.h"
#include "statictip.h"

#include "tlist.h"


/* Callbacks */
static gint TListConfigureEventCB(
	GtkWidget *widget, GdkEventConfigure *configure, gpointer data
);
static gint TListExposeEventCB(
	GtkWidget *widget, GdkEventExpose *expose, gpointer data
);
static gint TListKeyEventCB(
	GtkWidget *widget, GdkEventKey *key, gpointer data
);
static gint TListButtonEventCB(
	GtkWidget *widget, GdkEventButton *button, gpointer data
);
static gint TListMotionEventCB(
	GtkWidget *widget, GdkEventMotion *motion, gpointer data
);
static gint TListCrossingEventCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
static gint TListFocusEventCB(
	GtkWidget *widget, GdkEventFocus *focus, gpointer data
);
static void TListRealizeCB(GtkWidget *widget, gpointer data);

static void TListDragBeginCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
);
static void TListDragEndCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
);
static void TListDragLeaveCB(
	GtkWidget *widget, GdkDragContext *dc,
	guint t,
	gpointer data
);
static gboolean TListDragMotionCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y, guint t,
	gpointer data
);
static gboolean TListDragDropCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	guint t,
	gpointer data
);
static void TListAdjustmentValueChangedCB(GtkAdjustment *adjustment, gpointer data);
static gint TListMoveToIdleCB(gpointer data);


/* Thumbs */
static tlist_thumb_struct *TListThumbNew(tlist_struct *tlist);
static void TListThumbDelete(
	tlist_struct *tlist,
	tlist_thumb_struct *thumb
);


/* Internal Selecting */
static void TListDoSelectThumb(
	tlist_struct *tlist,
	const gint thumb_num,
	GdkEventButton *button,
	const gboolean append
);
static void TListDoSelectThumbRange(
	tlist_struct *tlist,
	const gint thumb_num,
	GdkEventButton *button
);
static void TListDoUnselectThumb(
	tlist_struct *tlist,
	const gint thumb_num,
	GdkEventButton *button
);
static void TListDoSelectAllThumbs(tlist_struct *tlist);
static void TListDoUnselectAllThumbs(tlist_struct *tlist);


/* Calculate Suitable Thumb Pixmap Size */
void TListQueryThumbPixmapSize(
	tlist_struct *tlist,
	const gint img_width, const gint img_height,
	gint *width, gint *height
);


/* Drawing */
static void TListDrawThumbIterate(
	tlist_struct *tlist,
	tlist_thumb_struct *thumb, const gint thumb_num,
	const tlist_flags flags,
	const gboolean has_focus,
	const gboolean drag_active,
	const gboolean drag_over,
	const gint thumb_border,
	const gint thumb_pixmap_height_max,
	GdkRectangle *thumb_rect,
	GdkRectangle *rel_label_rect,
	GdkDrawable *drawable,
	GdkFont *font,
	const gint font_height,
	GdkGC *gc,
	const GtkStateType state,
	GtkStyle *style,
	GtkWidget *w
);
void TListDraw(tlist_struct *tlist);
void TListQueueDraw(tlist_struct *tlist);


/* Resize */
void TListResize(
	tlist_struct *tlist,
	const gint width, const gint height
);


/* Freeze/Thaw */
void TListFreeze(tlist_struct *tlist);
void TListThaw(tlist_struct *tlist);


/* Thumbs Add, Set, and Delete */
static GtkVisibility TListThumbTextVisiblity(
	tlist_struct *tlist,
	const gchar *text
);
gint TListInsert(
	tlist_struct *tlist,
	const gint thumb_num,
	const gchar *text
);
gint TListAppend(
	tlist_struct *tlist,
	const gchar *text
);
void TListSetLoadState(
	tlist_struct *tlist,
	const gint thumb_num,
	const tlist_load_state load_state
);
void TListSetText(
	tlist_struct *tlist,
	const gint thumb_num,
	const gchar *text
);
void TListSetTextColor(
	tlist_struct *tlist,
	const gint thumb_num,
	GdkColor *fg,
	GdkColor *bg
);
void TListSetPixmap(
	tlist_struct *tlist,
	const gint thumb_num,
	GdkPixmap *pixmap, GdkBitmap *mask
);
void TListSetRGBA(
	tlist_struct *tlist,
	const gint thumb_num,
	const gint width, const gint height,
	const gint bpl,
	const GdkRgbDither dith,
	const guint8 *rgba,
	const gboolean no_enlarge
);
void TListSetSensitive(
	tlist_struct *tlist,
	const gint thumb_num,
	const gboolean sensitive
);
void TListSetSelectable(
	tlist_struct *tlist,
	const gint thumb_num,
	const gboolean selectable
);
void TListSetAttributeIconPlacement(
	tlist_struct *tlist,
	const int thumb_num,
	const GtkCornerType placement
);
void TListAppendAttributeIcon(
	tlist_struct *tlist,
	const int thumb_num,
	GdkPixmap *pixmap,
	GdkBitmap *mask
);
void TListSetThumbData(
	tlist_struct *tlist,
	const gint thumb_num,
	gpointer data
);
void TListSetThumbDataFull(
	tlist_struct *tlist,
	const gint thumb_num,
	gpointer data,
	GtkDestroyNotify destroy_cb
);
void TListRemove(
	tlist_struct *tlist,
	const gint thumb_num
);
void TListClear(tlist_struct *tlist);

/* Thumbs Get */
tlist_thumb_struct *TListGetThumb(
	tlist_struct *tlist,
	const gint thumb_num
);
tlist_load_state TListGetLoadState(
	tlist_struct *tlist,
	const gint thumb_num
);
gboolean TListGetText(
	tlist_struct *tlist,
	const gint thumb_num,
	const gchar **text
);
gboolean TListGetPixmap(
	tlist_struct *tlist,
	const gint thumb_num,
	GdkPixmap **pixmap, GdkBitmap **mask
);
gpointer TListGetThumbData(
	tlist_struct *tlist,
	const gint thumb_num
);

/* Thumbs Find */
gint TListFindThumbFromData(
	tlist_struct *tlist,
	gpointer data
);

/* Selecting */
gboolean TListIsThumbSelected(
	tlist_struct *tlist,
	const gint thumb_num
);
void TListSelectThumb(
	tlist_struct *tlist,
	const gint thumb_num
);
void TListUnselectThumb(
	tlist_struct *tlist,
	const gint thumb_num
);
void TListSelectAll(tlist_struct *tlist);
void TListUnselectAll(tlist_struct *tlist);

gboolean TListGetSelection(
	tlist_struct *tlist,
	const gint x, const gint y,
	gint *thumb_num_rtn,
	gint *thumb_ix_rtn, gint *thumb_iy_rtn
);
gboolean TListGetThumbPosition(
	tlist_struct *tlist,
	const gint thumb_num,
	gint *x_rtn, gint *y_rtn
);
gboolean TListGetThumbPixmapGeometry(
	tlist_struct *tlist,
	const gint thumb_num, 
	gint *x_rtn, gint *y_rtn,
	gint *width_rtn, gint *height_rtn
);
gboolean TListGetThumbLabelGeometry(
	tlist_struct *tlist,
	const gint thumb_num,
	gint *x_rtn, gint *y_rtn,
	gint *width_rtn, gint *height_rtn
);


/* Visibility */
GtkVisibility TListIsThumbVisible(
	tlist_struct *tlist,
	const gint thumb_num
);


/* Scrolling */
void TListMoveTo(
	tlist_struct *tlist,
	const gint thumb_num,
	const gfloat coeff
);
void TListQueueMoveTo(
	tlist_struct *tlist,
	const gint thumb_num,
	const gfloat coeff
);


/* Thumbs List */
tlist_struct *TListNew(
	const GtkOrientation orientation,
	const gint thumb_width, const gint thumb_height,
	const gint thumb_border,
	void (*select_cb)(tlist_struct *, GdkEventButton *, gint, gpointer),
	gpointer select_data,
	void (*unselect_cb)(tlist_struct *, GdkEventButton *, gint, gpointer),
	gpointer unselect_data
);
GtkWidget *TListGetToplevelWidget(tlist_struct *tlist);
GtkWidget *TListGetListWidget(tlist_struct *tlist);
void TListRealize(tlist_struct *tlist);
void TListThumbGeometry(
	tlist_struct *tlist, 
	const gint thumb_width, const gint thumb_height,
	const gint thumb_border
);
void TListSelectionMode(
	tlist_struct *tlist,
	const GtkSelectionMode selection_mode
);
void TListDoubleBuffer(
	tlist_struct *tlist,
	const gboolean double_buffer
);
void TListOrientation(
	tlist_struct *tlist,
	const GtkOrientation orientation
);
void TListShowThumbFrames(
	tlist_struct *tlist,
	const gboolean show
);
void TListShowThumbLabels(
	tlist_struct *tlist,
	const gboolean show
);
void TListShowTextTips(
	tlist_struct *tlist,
	const gboolean show
);
void TListEnableListDragScroll(
	tlist_struct *tlist,
	const gboolean enable
);
void TListMap(tlist_struct *tlist);
void TListUnmap(tlist_struct *tlist);
void TListGrabFocus(tlist_struct *tlist);
void TListDelete(tlist_struct *tlist);


/* Size of GtkFrame borders */
#define TLIST_THUMB_DEF_FRAME_WIDTH		2
#define TLIST_THUMB_DEF_FRAME_HEIGHT		2


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Thumbs List GtkDrawingArea "configure_event" signal callback.
 */
static gint TListConfigureEventCB(
	GtkWidget *widget, GdkEventConfigure *configure, gpointer data
)
{
	tlist_struct *tlist = TLIST(data);
	if((widget == NULL) || (configure == NULL) || (tlist == NULL))
		return(FALSE);

	/* Update the size of the List GtkDrawingArea, recreate the
	 * back buffer GdkPixmap, and update the GtkScrollBar values
	 */
	TListResize(
		tlist,
		configure->width, configure->height
	);

	return(TRUE);
}

/*
 *	Thumbs List GtkDrawingArea "expose_event" signal callback.
 */
static gint TListExposeEventCB(
	GtkWidget *widget, GdkEventExpose *expose, gpointer data
)
{
	tlist_struct *tlist = TLIST(data);
	if((widget == NULL) || (expose == NULL) || (tlist == NULL))
		return(FALSE);

	TListDraw(tlist);

	return(TRUE);
}

/*
 *	Thumbs List GtkDrawingArea "key_press_event" or
 *	"key_release_event" signal callback.
 */
static gint TListKeyEventCB(
	GtkWidget *widget, GdkEventKey *key, gpointer data
)
{
	gint status = FALSE;
	gint etype;
	gboolean press;
	guint keyval, state;
	tlist_struct *tlist = TLIST(data);
	if((key == NULL) || (tlist == NULL))
		return(status);

	etype = key->type;
	press = (etype == GDK_KEY_PRESS) ? TRUE : FALSE;
	keyval = key->keyval;
	state = key->state;

#define STOP_KEY_SIGNAL_EMIT	{		\
 gtk_signal_emit_stop_by_name(			\
  GTK_OBJECT(widget),				\
  press ?					\
   "key_press_event" : "key_release_event"	\
 );						\
}
#define ADJ_CLAMP_EMIT(_adj_)	{		\
 if((_adj_)->value > ((_adj_)->upper - (_adj_)->page_size))	\
  (_adj_)->value = (_adj_)->upper - (_adj_)->page_size;	\
 if((_adj_)->value < (_adj_)->lower)		\
  (_adj_)->value = (_adj_)->lower;		\
 gtk_adjustment_set_value(			\
  (_adj_), (_adj_)->value			\
 );						\
}
/* Clamp the focus thumb and queue redraw */
#define TLIST_FOCUS_THUMB_CLAMP_DRAW(_tlist_) {	\
 if((_tlist_)->focus_thumb >= (_tlist_)->total_thumbs)	\
  (_tlist_)->focus_thumb = (_tlist_)->total_thumbs - 1;	\
 if((_tlist_)->focus_thumb < 0)			\
  (_tlist_)->focus_thumb = 0;			\
 TListQueueDraw(_tlist_);			\
}

	/* Handle by key value */
	switch(keyval)
	{
	  case GDK_space:
		if(press)
		{
			const gint thumb_num = tlist->focus_thumb;

			TListFreeze(tlist);

			switch(tlist->selection_mode)
			{
			  case GTK_SELECTION_EXTENDED:
			  case GTK_SELECTION_MULTIPLE:
				if(TListIsThumbSelected(tlist, thumb_num))
					TListDoUnselectThumb(
						tlist, thumb_num, NULL
					);
				else
					TListDoSelectThumb(
						tlist, thumb_num, NULL, TRUE
					);
				break;
			  case GTK_SELECTION_BROWSE:
				TListDoSelectThumb(
					tlist, thumb_num, NULL, FALSE
				);
				break;
			  case GTK_SELECTION_SINGLE:
				if(TListIsThumbSelected(tlist, thumb_num))
				{
					TListDoUnselectAllThumbs(tlist);
				}
				else
				{
					TListDoUnselectAllThumbs(tlist);
					TListDoSelectThumb(
						tlist, thumb_num, NULL, TRUE
					);
				}
				break;
			}

			TListThaw(tlist);
		}
		status = TRUE;
		break;

	  case GDK_Up:
	  case GDK_KP_Up:
		if((state & GDK_CONTROL_MASK) && press)
		{
			/* Get adjustment and scroll up */
			GtkAdjustment *adj = tlist->vadjustment;
			if(adj != NULL)
			{
				adj->value -= adj->step_increment;
				ADJ_CLAMP_EMIT(adj);
			}
		}
		else if((state & GDK_SHIFT_MASK) && press)
		{
			TListFreeze(tlist);

			/* Move focus thumb up */
			tlist->focus_thumb -= (tlist->orientation == GTK_ORIENTATION_HORIZONTAL) ?
				1 : tlist->thumbs_per_line;
			TLIST_FOCUS_THUMB_CLAMP_DRAW(tlist);

			switch(tlist->selection_mode)
			{
			  case GTK_SELECTION_EXTENDED:
			  case GTK_SELECTION_MULTIPLE:
				TListDoSelectThumbRange(
					tlist, tlist->focus_thumb, NULL
				);
				break;
			  case GTK_SELECTION_BROWSE:
			  case GTK_SELECTION_SINGLE:
				TListDoUnselectAllThumbs(tlist);
				TListDoSelectThumb(
					tlist, tlist->focus_thumb, NULL, TRUE
				);
				break;
			}
			if(TListIsThumbVisible(tlist, tlist->focus_thumb) !=
				GTK_VISIBILITY_FULL
			)
				TListMoveTo(tlist, tlist->focus_thumb, 0.0f);

			TListThaw(tlist);
		}
		else if(press)
		{
			TListFreeze(tlist);

			/* Move focus thumb up */
			tlist->focus_thumb -= (tlist->orientation == GTK_ORIENTATION_HORIZONTAL) ?
				1 : tlist->thumbs_per_line;
			TLIST_FOCUS_THUMB_CLAMP_DRAW(tlist);

			if(TListIsThumbVisible(tlist, tlist->focus_thumb) !=
				GTK_VISIBILITY_FULL
			)
				TListMoveTo(tlist, tlist->focus_thumb, 0.0f);

			TListThaw(tlist);
		}
		STOP_KEY_SIGNAL_EMIT
		status = TRUE;
		break;

	  case GDK_Down:
	  case GDK_KP_Down:
		if((state & GDK_CONTROL_MASK) && press)
		{
			/* Get adjustment and scroll down */
			GtkAdjustment *adj = tlist->vadjustment;
			if(adj != NULL)
			{
				adj->value += adj->step_increment;
				ADJ_CLAMP_EMIT(adj);
			}
		}
		else if((state & GDK_SHIFT_MASK) && press)
		{
			TListFreeze(tlist);

			/* Move focus thumb down */
			tlist->focus_thumb += (tlist->orientation == GTK_ORIENTATION_HORIZONTAL) ?
				1 : tlist->thumbs_per_line;
			TLIST_FOCUS_THUMB_CLAMP_DRAW(tlist);

			switch(tlist->selection_mode)
			{
			  case GTK_SELECTION_EXTENDED:
			  case GTK_SELECTION_MULTIPLE:
				TListDoSelectThumbRange(
					tlist, tlist->focus_thumb, NULL
				);
				break;
			  case GTK_SELECTION_BROWSE:
			  case GTK_SELECTION_SINGLE:
				TListDoUnselectAllThumbs(tlist);
				TListDoSelectThumb(
					tlist, tlist->focus_thumb, NULL, TRUE
				);
				break;
			}

			if(TListIsThumbVisible(tlist, tlist->focus_thumb) !=
				GTK_VISIBILITY_FULL
			)
				TListMoveTo(tlist, tlist->focus_thumb, 1.0f);

			TListThaw(tlist);
		}
		else if(press)
		{
			TListFreeze(tlist);

			/* Move focus thumb down */
			tlist->focus_thumb += (tlist->orientation == GTK_ORIENTATION_HORIZONTAL) ?
				1 : tlist->thumbs_per_line;
			TLIST_FOCUS_THUMB_CLAMP_DRAW(tlist);

			if(TListIsThumbVisible(tlist, tlist->focus_thumb) !=
				GTK_VISIBILITY_FULL
			)
				TListMoveTo(tlist, tlist->focus_thumb, 1.0f);

			TListThaw(tlist);
		}
		STOP_KEY_SIGNAL_EMIT
		status = TRUE;
		break;

	  case GDK_Left:
	  case GDK_KP_Left:
		if((state & GDK_CONTROL_MASK) && press)
		{
			/* Get adjustment and scroll left */
			GtkAdjustment *adj = tlist->hadjustment;
			if(adj != NULL)
			{
				adj->value -= adj->step_increment;
				ADJ_CLAMP_EMIT(adj);
			}
		}
		else if((state & GDK_SHIFT_MASK) && press)
		{
			TListFreeze(tlist);

			/* Move focus thumb left */
			tlist->focus_thumb -= (tlist->orientation == GTK_ORIENTATION_HORIZONTAL) ?
				tlist->thumbs_per_line : 1;
			TLIST_FOCUS_THUMB_CLAMP_DRAW(tlist);

			switch(tlist->selection_mode)
			{
			  case GTK_SELECTION_EXTENDED:
			  case GTK_SELECTION_MULTIPLE:
				TListDoSelectThumbRange(
					tlist, tlist->focus_thumb, NULL
				);
				break;
			  case GTK_SELECTION_BROWSE:
			  case GTK_SELECTION_SINGLE:
				TListDoUnselectAllThumbs(tlist);
				TListDoSelectThumb(
					tlist, tlist->focus_thumb, NULL, TRUE
				);
				break;
			}

			if(TListIsThumbVisible(tlist, tlist->focus_thumb) !=
				GTK_VISIBILITY_FULL
			)
				TListMoveTo(tlist, tlist->focus_thumb, 0.0f);

			TListThaw(tlist);
		}
		else if(press)
		{
			TListFreeze(tlist);

			/* Move focus thumb left */
			tlist->focus_thumb -= (tlist->orientation == GTK_ORIENTATION_HORIZONTAL) ?
				tlist->thumbs_per_line : 1;
			TLIST_FOCUS_THUMB_CLAMP_DRAW(tlist);

			if(TListIsThumbVisible(tlist, tlist->focus_thumb) !=
				GTK_VISIBILITY_FULL
			)
				TListMoveTo(tlist, tlist->focus_thumb, 0.0f);

			TListThaw(tlist);
		}
		STOP_KEY_SIGNAL_EMIT
		status = TRUE;
		break;

	  case GDK_Right:
	  case GDK_KP_Right:
		if((state & GDK_CONTROL_MASK) && press)
		{
			/* Get adjustment and scroll right */
			GtkAdjustment *adj = tlist->hadjustment;
			if(adj != NULL)
			{
				adj->value += adj->step_increment;
				ADJ_CLAMP_EMIT(adj);
			}
		}
		else if((state & GDK_SHIFT_MASK) && press)
		{
			TListFreeze(tlist);

			/* Move focus thumb right */
			tlist->focus_thumb += (tlist->orientation == GTK_ORIENTATION_HORIZONTAL) ?
				tlist->thumbs_per_line : 1;
			TLIST_FOCUS_THUMB_CLAMP_DRAW(tlist);

			switch(tlist->selection_mode)
			{
			  case GTK_SELECTION_EXTENDED:
			  case GTK_SELECTION_MULTIPLE:
				TListDoSelectThumbRange(
					tlist, tlist->focus_thumb, NULL
				);
				break;
			  case GTK_SELECTION_BROWSE:
			  case GTK_SELECTION_SINGLE:
				TListDoUnselectAllThumbs(tlist);
				TListDoSelectThumb(
					tlist, tlist->focus_thumb, NULL, TRUE
				);
				break;
			}

			if(TListIsThumbVisible(tlist, tlist->focus_thumb) !=
				GTK_VISIBILITY_FULL
			)
				TListMoveTo(tlist, tlist->focus_thumb, 1.0f);

			TListThaw(tlist);
		}
		else if(press)
		{
			TListFreeze(tlist);

			/* Move focus thumb right */
			tlist->focus_thumb += (tlist->orientation == GTK_ORIENTATION_HORIZONTAL) ?
				tlist->thumbs_per_line : 1;
			TLIST_FOCUS_THUMB_CLAMP_DRAW(tlist);

			if(TListIsThumbVisible(tlist, tlist->focus_thumb) !=
				GTK_VISIBILITY_FULL
			)
				TListMoveTo(tlist, tlist->focus_thumb, 1.0f);

			TListThaw(tlist);
		}
		STOP_KEY_SIGNAL_EMIT
		status = TRUE;
		break;

	  case GDK_Page_Up:
	  case GDK_KP_Page_Up:
		if((state & GDK_CONTROL_MASK) && press)
		{
			GtkAdjustment *adj = (tlist->orientation == GTK_ORIENTATION_HORIZONTAL) ?
				tlist->hadjustment : tlist->vadjustment;
			if(adj != NULL)
			{
				adj->value -= adj->page_increment;
				ADJ_CLAMP_EMIT(adj);
			}
		}
		else if(press)
		{
			const gint thumb_len = (tlist->orientation == GTK_ORIENTATION_HORIZONTAL) ?
				tlist->thumb_width : tlist->thumb_height;
			GtkAdjustment *adj = (tlist->orientation == GTK_ORIENTATION_HORIZONTAL) ?
				tlist->hadjustment : tlist->vadjustment;
			if((adj != NULL) && (thumb_len > 0))
			{
				gint n = (gint)(adj->page_increment / thumb_len) *
					tlist->thumbs_per_line;

				TListFreeze(tlist);

				/* Move focus thumb */
				tlist->focus_thumb -= n;
				TLIST_FOCUS_THUMB_CLAMP_DRAW(tlist);

				/* Select thumbs in between */
				if(state & GDK_SHIFT_MASK)
				{
					switch(tlist->selection_mode)
					{
					  case GTK_SELECTION_EXTENDED:
					  case GTK_SELECTION_MULTIPLE:
						TListDoSelectThumbRange(
							tlist, tlist->focus_thumb, NULL 
						);
						break;
					  case GTK_SELECTION_BROWSE:
					  case GTK_SELECTION_SINGLE:
						TListDoUnselectAllThumbs(tlist);
						TListDoSelectThumb(
							tlist, tlist->focus_thumb, NULL, TRUE
						);
						break;
					}
				}

				if(TListIsThumbVisible(tlist, tlist->focus_thumb) !=
					GTK_VISIBILITY_FULL
				)
					TListMoveTo(tlist, tlist->focus_thumb, 0.0f);

				TListThaw(tlist);
			}
		}
		STOP_KEY_SIGNAL_EMIT
		status = TRUE;
		break;

	  case GDK_Page_Down:
	  case GDK_KP_Page_Down:
		if((state & GDK_CONTROL_MASK) && press)
		{
			GtkAdjustment *adj = (tlist->orientation == GTK_ORIENTATION_HORIZONTAL) ?
				tlist->hadjustment : tlist->vadjustment;
			if(adj != NULL)
			{
				adj->value += adj->page_increment;
				ADJ_CLAMP_EMIT(adj);
			}
		}
		else if(press)
		{
			const gint thumb_len = (tlist->orientation == GTK_ORIENTATION_HORIZONTAL) ?
				tlist->thumb_width : tlist->thumb_height;
			GtkAdjustment *adj = (tlist->orientation == GTK_ORIENTATION_HORIZONTAL) ?
				tlist->hadjustment : tlist->vadjustment;
			if((adj != NULL) && (thumb_len > 0))
			{
				gint n = (gint)(adj->page_increment / thumb_len) *
					tlist->thumbs_per_line;

				TListFreeze(tlist);

				/* Move focus thumb */
				tlist->focus_thumb += n;
				TLIST_FOCUS_THUMB_CLAMP_DRAW(tlist);

				/* Select thumbs in between */
				if(state & GDK_SHIFT_MASK)
				{
					switch(tlist->selection_mode)
					{
					  case GTK_SELECTION_EXTENDED:
					  case GTK_SELECTION_MULTIPLE:
						TListDoSelectThumbRange(
							tlist, tlist->focus_thumb, NULL 
						);
						break;
					  case GTK_SELECTION_BROWSE:
					  case GTK_SELECTION_SINGLE:
						TListDoUnselectAllThumbs(tlist);
						TListDoSelectThumb(
							tlist, tlist->focus_thumb, NULL, TRUE
						);
						break;
					}
				}

				if(TListIsThumbVisible(tlist, tlist->focus_thumb) !=
					GTK_VISIBILITY_FULL
				)
					TListMoveTo(tlist, tlist->focus_thumb, 1.0f);

				TListThaw(tlist);
			}
		}
		STOP_KEY_SIGNAL_EMIT
		status = TRUE;
		break;

	  case GDK_Home:
	  case GDK_KP_Home:
		if((state & GDK_CONTROL_MASK) && press)
		{
			/* Get adjustment and scroll all the way up */
			GtkAdjustment *adj = (tlist->orientation == GTK_ORIENTATION_HORIZONTAL) ?
				tlist->hadjustment : tlist->vadjustment;
			if(adj != NULL)
			{
				adj->value = adj->lower;
				ADJ_CLAMP_EMIT(adj);
		    }
		}
		else if(press)
		{
			TListFreeze(tlist);

			/* Move focus thumb */
			tlist->focus_thumb = 0;
			TLIST_FOCUS_THUMB_CLAMP_DRAW(tlist);

			/* Select thumbs in between */
			if(state & GDK_SHIFT_MASK)
			{
				switch(tlist->selection_mode)
				{
				  case GTK_SELECTION_EXTENDED:
				  case GTK_SELECTION_MULTIPLE:
					TListDoSelectThumbRange(
						tlist, tlist->focus_thumb, NULL
					);
					break; 
				  case GTK_SELECTION_BROWSE:
				  case GTK_SELECTION_SINGLE:
					TListDoUnselectAllThumbs(tlist);
					TListDoSelectThumb(
						tlist, tlist->focus_thumb, NULL, TRUE
					);
					break;
				}
			}

			if(TListIsThumbVisible(tlist, tlist->focus_thumb) !=
				GTK_VISIBILITY_FULL
			)
				TListMoveTo(tlist, tlist->focus_thumb, 0.0f);

			TListThaw(tlist);
		}
		STOP_KEY_SIGNAL_EMIT
		status = TRUE;
		break;

	  case GDK_End:
	  case GDK_KP_End:
		if((state & GDK_CONTROL_MASK) && press)
		{
			/* Get adjustment and scroll all the way down */
			GtkAdjustment *adj = (tlist->orientation == GTK_ORIENTATION_HORIZONTAL) ?
				tlist->hadjustment : tlist->vadjustment;
			if(adj != NULL)
			{
				adj->value = adj->upper - adj->page_size;
				ADJ_CLAMP_EMIT(adj);
			}
		}
		else if(press)
		{
			TListFreeze(tlist);

			/* Move focus thumb */
			tlist->focus_thumb = tlist->total_thumbs - 1;
			TLIST_FOCUS_THUMB_CLAMP_DRAW(tlist);

			/* Select thumbs in between */
			if(state & GDK_SHIFT_MASK)
			{
				switch(tlist->selection_mode)
				{
				  case GTK_SELECTION_EXTENDED:
				  case GTK_SELECTION_MULTIPLE:
					TListDoSelectThumbRange(
						tlist, tlist->focus_thumb, NULL
					);
				    break; 
				  case GTK_SELECTION_BROWSE:
				  case GTK_SELECTION_SINGLE:
					TListDoUnselectAllThumbs(tlist);
					TListDoSelectThumb(
						tlist, tlist->focus_thumb, NULL, TRUE
					);
					break;
				}
			}

			if(TListIsThumbVisible(tlist, tlist->focus_thumb) !=
				GTK_VISIBILITY_FULL
			)
				TListMoveTo(tlist, tlist->focus_thumb, 1.0f);

			TListThaw(tlist);
		}
		STOP_KEY_SIGNAL_EMIT
		status = TRUE;
		break;
	}

	/* If a key was handled then hide the texttip */
	if(status)
	{
		/* Reset the pointer over thumb and hide the texttip */
		if(tlist->pointer_over_thumb > -1)
		{
			tlist->pointer_over_thumb = -1;

			/* Hide the texttip */
			StaticTipSet(widget, NULL, 0, 0, 0, 0);
		}
	}

#undef TLIST_FOCUS_THUMB_CLAMP_DRAW
#undef ADJ_CLAMP_EMIT
#undef STOP_KEY_SIGNAL_EMIT

	return(status);
}

/*
 *	Thumbs List GtkDrawingArea "button_press_event" or
 *	"button_release_event" signal callback.
 */
static gint TListButtonEventCB(
	GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	gint status = FALSE;
	gint thumb_num;
	tlist_struct *tlist = TLIST(data);
	if((widget == NULL) || (button == NULL) || (tlist == NULL))
		return(status);

	switch((gint)button->type)
	{
	  case GDK_BUTTON_PRESS:
		/* Grab the focus as needed */
		if(!GTK_WIDGET_HAS_FOCUS(widget))
			gtk_widget_grab_focus(widget);

		/* Handle by the button number */
		switch(button->button)
		{
		  case GDK_BUTTON1:
			/* Select/unselect thumb
			 *
			 * Match the thumb the button was pressed over
			 */
			if(TListGetSelection(
				tlist, (gint)button->x, (gint)button->y,
				&thumb_num, NULL, NULL
			))
			{
				TListFreeze(tlist);

				/* Set the new focus thumb as this thumb */
				tlist->focus_thumb = thumb_num;

				/* Select/unselect thumb(s) by the modifier
				 * key
				 *
				 * Control Key (Invert Individual Selection)
				 */
				if(button->state & GDK_CONTROL_MASK)
				{
					switch(tlist->selection_mode)
					{
					  case GTK_SELECTION_EXTENDED:
					  case GTK_SELECTION_MULTIPLE:
						if(TListIsThumbSelected(tlist, thumb_num))
							TListDoUnselectThumb(
								tlist, thumb_num, button
							);
						else
							TListDoSelectThumb(
								tlist, thumb_num, button, TRUE
							);
						break;
					  case GTK_SELECTION_BROWSE:
						TListDoSelectThumb(
							tlist, thumb_num, button, FALSE
						);
						break;
					  case GTK_SELECTION_SINGLE:
						if(TListIsThumbSelected(tlist, thumb_num))
						{
							TListDoUnselectAllThumbs(tlist);
						}
						else
						{
							TListDoUnselectAllThumbs(tlist);
							TListDoSelectThumb(
								tlist, thumb_num, button, TRUE
							);
						}
						break;
					}
				}
				/* Shift Key (Group Selection) */
				else if(button->state & GDK_SHIFT_MASK)
				{
					switch(tlist->selection_mode)
					{
					  case GTK_SELECTION_EXTENDED:
					  case GTK_SELECTION_MULTIPLE:
						TListDoSelectThumbRange(
							tlist, thumb_num, button
						);
						break;
					  case GTK_SELECTION_BROWSE:
					  case GTK_SELECTION_SINGLE:
						if(TListIsThumbSelected(tlist, thumb_num))
						{
							TListDoUnselectAllThumbs(tlist);
						}
						else
						{
							TListDoUnselectAllThumbs(tlist);
							TListDoSelectThumb(
								tlist, thumb_num, button, TRUE
							);
						}
						break;
					}
				}
				/* No important modifier keys */
				else
				{
					/* If there was exactly one thumb selected and
					 * it matches thumb_num then do nothing
					 */
					gint i;
					GList	*glist,
							*selection = g_list_copy(tlist->selection);

					/* Unselect all thumbs except for the thumb
					 * indicated by thumb_num
					 */
					for(glist = selection;
						glist != NULL;
						glist = g_list_next(glist)
					)
					{
						i = (gint)glist->data;
						if(i != thumb_num)
							TListDoUnselectThumb(tlist, i, NULL);
					}
					TListDoSelectThumb(
						tlist, thumb_num, button, TRUE
					);

					g_list_free(selection);
				}

				TListThaw(tlist);
			}
			status = TRUE;
			break;

		  case GDK_BUTTON2:
			/* Middle click list drag scroll? */
			if(tlist->flags & TLIST_ENABLE_LIST_DRAG_SCROLL)
			{
				/* Start scrolling */
				tlist->flags |= TLIST_LIST_DRAG_SCROLL_ACTIVE;
				tlist->drag_last_x = (gint)button->x;
				tlist->drag_last_y = (gint)button->y;
				gdk_pointer_grab(
					button->window,
					FALSE,
					GDK_BUTTON_PRESS_MASK |
						GDK_BUTTON_RELEASE_MASK |
						GDK_POINTER_MOTION_MASK |
						GDK_POINTER_MOTION_HINT_MASK,
					NULL,
					tlist->translate_cur,
					button->time
				);
				status = TRUE;
			}
			break;

		  case GDK_BUTTON4:
			/* Scroll left? */
			if(tlist->orientation == GTK_ORIENTATION_HORIZONTAL)
			{
				GtkAdjustment *adj = tlist->hadjustment;
				if(adj != NULL)
				{
					const gfloat inc = MAX(
						(0.25f * adj->page_size),
						adj->step_increment
					);
					gfloat v = adj->value - inc;
					if(v > (adj->upper - adj->page_size))
						v = adj->upper - adj->page_size;
					if(v < adj->lower)
						v = adj->lower;
					gtk_adjustment_set_value(adj, v);
				}
			}
			/* Scroll up */
			else
			{
				GtkAdjustment *adj = tlist->vadjustment;
				if(adj != NULL)
				{
					const gfloat inc = MAX(
						(0.25f * adj->page_size),
						adj->step_increment
					);
					gfloat v = adj->value - inc;
					if(v > (adj->upper - adj->page_size))
						v = adj->upper - adj->page_size;
					if(v < adj->lower)
						v = adj->lower;
					gtk_adjustment_set_value(adj, v);
				}
			}
			/* Reset the pointer over thumb and hide the texttip
			 * due to the scroll
			 */
			if(tlist->pointer_over_thumb > -1)
			{
				tlist->pointer_over_thumb = -1;

				/* Hide the texttip */
				StaticTipSet(widget, NULL, 0, 0, 0, 0);
			}
			status = TRUE;
			break;

		  case GDK_BUTTON5:
			/* Scroll right? */
			if(tlist->orientation == GTK_ORIENTATION_HORIZONTAL)
			{
				GtkAdjustment *adj = tlist->hadjustment;
				if(adj != NULL)
				{
					const gfloat inc = MAX(
						(0.25f * adj->page_size),
						adj->step_increment
					);
					gfloat v = adj->value + inc;
					if(v > (adj->upper - adj->page_size))
						v = adj->upper - adj->page_size;
					if(v < adj->lower)
						v = adj->lower;
					gtk_adjustment_set_value(adj, v);
				}
			}
			/* Scroll down */
			else
			{
				GtkAdjustment *adj = tlist->vadjustment;
				if(adj != NULL)
				{
					const gfloat inc = MAX(
						(0.25f * adj->page_size),
						adj->step_increment
					);
					gfloat v = adj->value + inc;
					if(v > (adj->upper - adj->page_size))
						v = adj->upper - adj->page_size;
					if(v < adj->lower)
						v = adj->lower;
					gtk_adjustment_set_value(adj, v);
				}
			}
			/* Reset the pointer over thumb and hide the texttip
			 * due to the scroll
			 */
			if(tlist->pointer_over_thumb > -1)
			{
				tlist->pointer_over_thumb = -1;

				/* Hide the texttip */
				StaticTipSet(widget, NULL, 0, 0, 0, 0);
			}
			status = TRUE;
			break;
		}
		break;

	  case GDK_BUTTON_RELEASE:
		/* Handle by the button number */
		switch(button->button)
		{
		  case GDK_BUTTON2:
			/* Middle click list drag scroll? */
			if(tlist->flags & TLIST_ENABLE_LIST_DRAG_SCROLL)
			{
				/* Stop scrolling */
				tlist->flags &= ~TLIST_LIST_DRAG_SCROLL_ACTIVE;
				if(gdk_pointer_is_grabbed())
					gdk_pointer_ungrab(button->time);
				status = TRUE;
			}
			break;
		}
		break;

	  case GDK_2BUTTON_PRESS:
		/* Handle by the button number */
		switch(button->button)
		{
		  case GDK_BUTTON1:
			/* Get the clicked-on thumb */
			if(TListGetSelection(
				tlist, (gint)button->x, (gint)button->y,
				&thumb_num, NULL, NULL
			))
			{
				/* Notify about this thumb being double clicked on */
				if(tlist->select_cb != NULL)
					tlist->select_cb(
						tlist,		/* Thumbs List */
						button,		/* GdkEventButton */
						thumb_num,		/* Thumb Index */
						tlist->select_data	/* Data */
					);
			}
			status = TRUE;
			break;
		}
		break;
	}

	return(status);
}

/*
 *	Thumbs List GtkDrawingArea "motion_notify_event" signal
 *	callback.
 */
static gint TListMotionEventCB(
	GtkWidget *widget, GdkEventMotion *motion, gpointer data
)
{
	gint status = FALSE;
	gint x, y;
	tlist_flags flags;
	tlist_struct *tlist = TLIST(data);
	if((widget == NULL) || (motion == NULL) || (tlist == NULL))
		return(status);

	if(motion->is_hint)
	{
		GdkModifierType mask;
		gdk_window_get_pointer(widget->window, &x, &y, &mask);
	}
	else
	{
		x = (gint)motion->x;
		y = (gint)motion->y;
	}

	flags = tlist->flags;

	/* Is a middle click list drag scroll active? */
	if((flags & TLIST_ENABLE_LIST_DRAG_SCROLL) &&
	   (flags & TLIST_LIST_DRAG_SCROLL_ACTIVE)
	)
	{
		GtkAdjustment *adj;

		/* Calculate the drag deltas */
		const gint	dx = x - tlist->drag_last_x,
					dy = y - tlist->drag_last_y;

		/* Update the drag position */
		tlist->drag_last_x = x;
		tlist->drag_last_y = y;

		/* Scroll horizontal */
		adj = tlist->hadjustment;
		if((adj != NULL) && (dx != 0))
		{
			if((adj->upper - adj->page_size) > adj->lower)
				gtk_adjustment_set_value(
					adj,
					CLIP(
						(adj->value - dx),
						adj->lower, (adj->upper - adj->page_size)
					)
				);
		}

		/* Scroll vertical */
		adj = tlist->vadjustment;
		if((adj != NULL) && (dy != 0))
		{
			if((adj->upper - adj->page_size) > adj->lower)
				gtk_adjustment_set_value(
					adj,
					CLIP(
						(adj->value - dy),
						adj->lower, (adj->upper - adj->page_size)
					)
				);
		}

		/* Reset the pointer over thumb and hide the texttip due
		 * to the scroll
		 */
		if(tlist->pointer_over_thumb > -1)
		{
			tlist->pointer_over_thumb = -1;

			/* Hide the texttip */
			StaticTipSet(widget, NULL, 0, 0, 0, 0);

			/* Queue redraw */
			TListQueueDraw(tlist);
		}
	}
	else
	{
		/* All else update the texttips */
		gint thumb_num;
		tlist_thumb_struct *thumb;

		/* Get the thumb that this motion occured over */
		if(!TListGetSelection(
			tlist,
			x, y,
			&thumb_num,
			NULL, NULL
		))
			thumb_num = -1;

		thumb = TListGetThumb(tlist, thumb_num);
		if(thumb != NULL)
		{
			/* Is the thumb that this motion occured over different
			 * from the thumb that the pointed was previously over?
			 */
			if(tlist->pointer_over_thumb != thumb_num)
			{
				const gchar *text = thumb->text;

				/* Update the thumb that the pointer is currently
				 * over
				 */
				tlist->pointer_over_thumb = thumb_num;

				/* Thumb text not fully visible? */
				if((flags & TLIST_SHOW_TEXTTIPS) &&
				   (thumb->text_visibility != GTK_VISIBILITY_FULL) &&
				   (text != NULL)
				)
				{
					gint x, y;
					GtkStyle *style = gtk_widget_get_style(widget);
					GdkFont *font = (style != NULL) ? style->font : NULL;
					GdkTextBounds b;

					gdk_string_bounds(font, text, &b);

					if(TListGetThumbPosition(tlist, thumb_num, &x, &y))
					{
						x -= ((b.width + (2 * 4)) - tlist->thumb_width) / 2;
						y += tlist->thumb_height;

						/* Set and display thumb's texttip */
						StaticTipSet(
							widget,
							text,
							STATIC_TIP_ALIGN_WIDGET_VALUE,
							STATIC_TIP_ALIGN_WIDGET_VALUE,
							x, y
						);
					}
				}
				else
				{
					/* Hide the texttip */
					StaticTipSet(widget, NULL, 0, 0, 0, 0);
				}

				/* Queue redraw */
				TListQueueDraw(tlist);
			}
		}
		else
		{
			/* Motion did not occure over a thumb */
			if(tlist->pointer_over_thumb > -1)
			{
				tlist->pointer_over_thumb = -1;

				/* Hide the texttip */
				StaticTipSet(widget, NULL, 0, 0, 0, 0);

				/* Queue redraw */
				TListQueueDraw(tlist);
			}
		}
		status = TRUE;
	}

	return(status);
}

/*
 *	Thumbs List GtkDrawingArea "enter_notify_event" or
 *	"leave_notify_event" signal callback.
 */
static gint TListCrossingEventCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	tlist_struct *tlist = TLIST(data);
	if((widget == NULL) || (crossing == NULL) || (data == NULL))
		return(FALSE);

	switch((gint)crossing->type)
	{
	  case GDK_ENTER_NOTIFY:
		break;

	  case GDK_LEAVE_NOTIFY:
		/* Hide thumb tip */
		StaticTipSet(widget, NULL, 0, 0, 0, 0);

		/* Mark that the pointer is not over any thumb */
		tlist->pointer_over_thumb = -1;

		TListQueueDraw(tlist);
		break;
	}

	return(TRUE);
}


/*
 *	Thumbs List GtkDrawingArea "focus_in_event" or "focus_out_event"
 *	signal callback.
 */
static gint TListFocusEventCB(
	GtkWidget *widget, GdkEventFocus *focus, gpointer data
)
{
	tlist_struct *tlist = TLIST(data);
	if((widget == NULL) || (focus == NULL) || (tlist == NULL))
		return(FALSE);

	if(focus->in && !GTK_WIDGET_HAS_FOCUS(widget))
	{
		GTK_WIDGET_SET_FLAGS(widget, GTK_HAS_FOCUS);
		TListQueueDraw(TLIST(data));
	}
	else if(!focus->in && GTK_WIDGET_HAS_FOCUS(widget))
	{
		GTK_WIDGET_UNSET_FLAGS(widget, GTK_HAS_FOCUS);
		TListQueueDraw(TLIST(data));
	}

	return(TRUE);
}

/*
 *	Thumbs List GtkDrawingArea "realize" signal callback.
 */
static void TListRealizeCB(GtkWidget *widget, gpointer data)
{
	GdkWindow *window;
	GtkStyle *style;
	tlist_struct *tlist = TLIST(data);
	if((widget == NULL) || (tlist == NULL))
		return;

	window = widget->window;
	if((window == NULL) || (tlist->flags & TLIST_REALIZED))
		return;

	/* Set this GdkWindow's background color to this GtkWidget
	 * GtkStyle's base color
	 */
	style = gtk_widget_get_style(widget);
	gdk_window_set_background(
		window,
		&style->base[GTK_STATE_NORMAL]
	);

	/* Create the transparent stipple GdkBitmap */
	if(tlist->transparent_stipple_bm == NULL)
	{
		const gchar data[] = { 0x55, 0xaa, 0x55, 0xaa,
				   0x55, 0xaa, 0x55, 0xaa };
		tlist->transparent_stipple_bm = gdk_bitmap_create_from_data(
			window,
			data,
			8, 8
		);
	}

	/* Get the colormap */
	if(tlist->colormap == NULL)
		tlist->colormap = GDK_COLORMAP_REF(gdk_window_get_colormap(window));

	/* Create the GC */
	if(tlist->gc == NULL)
		tlist->gc = gdk_gc_new(window);

	/* Mark the Thumbs List as realized */
	tlist->flags |= TLIST_REALIZED;
}


/*
 *	Thumbs List GtkDrawingArea "drag_begin" signal callback.
 */
static void TListDragBeginCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{
	tlist_struct *tlist = TLIST(data);
	if((widget == NULL) || (dc == NULL) || (tlist == NULL))
		return;

	tlist->flags |= TLIST_DRAG_ACTIVE;

	/* Hide the texttip since it would otherwise interfere
	 * with the drag operation
	 */
	StaticTipSet(widget, NULL, 0, 0, 0, 0);

	TListQueueDraw(tlist);
}

/*
 *	Thumbs List GtkDrawingArea "drag_end" signal callback.
 */
static void TListDragEndCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{
	tlist_struct *tlist = TLIST(data);
	if((widget == NULL) || (dc == NULL) || (tlist == NULL))
		return;

	tlist->flags &= ~TLIST_DRAG_ACTIVE;
	tlist->flags &= ~TLIST_DRAG_ACTIVE_MOVE;
	TListQueueDraw(tlist);
}

/*
 *	Thumbs List GtkDrawingArea "drag_leave" signal callback.
 */
static void TListDragLeaveCB(
	GtkWidget *widget, GdkDragContext *dc,
	guint t,
	gpointer data
)
{
	gboolean need_draw = FALSE;
	tlist_struct *tlist = TLIST(data);
	if((widget == NULL) || (dc == NULL) || (tlist == NULL))
		return;

	if(!GTK_WIDGET_SENSITIVE(widget))
		return;

	/* Clear the drag over flag */
	if(tlist->flags & TLIST_DRAG_OVER)
	{
		tlist->flags &= ~TLIST_DRAG_OVER;
		need_draw = TRUE;
	}

	/* Clear the focus thumb */
	if(tlist->focus_thumb > -1)
	{
		tlist->focus_thumb = -1;
		need_draw = TRUE;
	}

	if(need_draw)
		TListQueueDraw(tlist);
}

/*
 *	Thumbs List GtkDrawingArea "drag_motion" signal callback.
 */
static gboolean TListDragMotionCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y, guint t,
	gpointer data
)
{
	gboolean same, need_draw = FALSE;
	gint thumb_num;
	gchar *target_name;
	GList *glist;
	GdkDragAction	allowed_actions,
					default_action_same,
					default_action,
					action;
	GtkTargetFlags target_flags;
	GtkWidget *src_w;
	tlist_struct *tlist = TLIST(data);
	if((widget == NULL) || (dc == NULL) || (tlist == NULL))
		return(FALSE);

	if(!GTK_WIDGET_SENSITIVE(widget))
		return(FALSE);

	/* Get the GtkWidget that the drag originated from */
	src_w = gtk_drag_get_source_widget(dc);

	/* Are they the same GtkWidgets? */
	same = (src_w == widget) ? TRUE : FALSE;

	/* Set the target flags to denote if this is the same
	 * GtkWidget or not
	 */
	target_flags = ((same) ? GTK_TARGET_SAME_WIDGET : 0);

	/* Iterate through the list of targets on the drag context
	 * and check if we will accept one of them
	 */
	for(glist = dc->targets;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		/* Get the next target name on the drag context */
		target_name = gdk_atom_name((GdkAtom)glist->data);
		if(target_name != NULL)
		{
			/* Check if this target name is in the list of
			 * accepted targets on the GtkWidget
			 */
			if(GUIDNDIsTargetAccepted(
				widget,
				target_name,
				target_flags
			))
			{
				g_free(target_name);
				break;
			}

			g_free(target_name);
		}
	}
	/* Drag not accepted? */
	if(glist == NULL)
		return(FALSE);

	/* Determine the appropriate drag action */
	GUIDNDGetTarActions(
		widget,
		&allowed_actions,
		&default_action_same,
		&default_action
	);
	action = GUIDNDDetermineDragAction(
		dc->actions,			/* Requested actions */
		allowed_actions,
		(same) ? default_action_same : default_action
	);

	/* Did the drag originate from this Thumbs List? */
	if(same)
	{
		if(action == GDK_ACTION_MOVE)
		{
			if(!(tlist->flags & TLIST_DRAG_ACTIVE_MOVE))
			{
				tlist->flags |= TLIST_DRAG_ACTIVE_MOVE;
				need_draw = TRUE;
			}
		}
		else
		{
			if(tlist->flags & TLIST_DRAG_ACTIVE_MOVE)
			{
				tlist->flags &= ~TLIST_DRAG_ACTIVE_MOVE;
				need_draw = TRUE;
			}
		}
	}

	/* Set the drag over flag */
	if((action != 0) &&
	   !(tlist->flags & TLIST_DRAG_OVER)
	)
	{
		tlist->flags |= TLIST_DRAG_OVER;
		need_draw = TRUE;
	}
	else if((action == 0) &&
		    (tlist->flags & TLIST_DRAG_OVER) 
	)
	{
		tlist->flags &= ~TLIST_DRAG_OVER;
		need_draw = TRUE;
	}

	/* Set the focus thumb */
	if(action != 0)
	{
		if(!TListGetSelection(
			tlist,
			x, y,
			&thumb_num,
			NULL, NULL
		))
			thumb_num = -1;
	}
	else
		thumb_num = -1;
	
	if(tlist->focus_thumb != thumb_num)
	{
		tlist->focus_thumb = thumb_num;
		need_draw = TRUE;
	}

	if(need_draw)
		TListQueueDraw(tlist);

	return(FALSE);
}

/*
 *	Thumbs List GtkDrawingArea "drag_drop" signal callback.
 */
static gboolean TListDragDropCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	guint t,
	gpointer data
)
{
	gboolean need_draw = FALSE;
	tlist_struct *tlist = TLIST(data);
	if((widget == NULL) || (dc == NULL) || (tlist == NULL))
		return(FALSE);

	if(!GTK_WIDGET_SENSITIVE(widget))
		return(FALSE);

	/* Clear the drag over flag */
	if(tlist->flags & TLIST_DRAG_OVER)
	{
		tlist->flags &= ~TLIST_DRAG_OVER;
		need_draw = TRUE;
	}

	/* Clear the focus thumb */
	if(tlist->focus_thumb > -1)
	{
		tlist->focus_thumb = -1;
		need_draw = TRUE;
	}

	if(need_draw)
		TListQueueDraw(tlist);

	return(FALSE);
}

/*
 *	Horizontal or vertical scroll bar GtkAdjustment "value_changed"
 *	signal callback.
 */
static void TListAdjustmentValueChangedCB(GtkAdjustment *adjustment, gpointer data)
{
	TListQueueDraw(TLIST(data));
}

/*
 *	Moveto idle callback.
 */
static gint TListMoveToIdleCB(gpointer data)
{
	tlist_struct *tlist = TLIST(data);
	if(tlist == NULL)
		return(FALSE);

	TListMoveTo(
		tlist,
		tlist->moveto_thumb,
		tlist->moveto_coeff
	);

	tlist->moveto_idleid = 0;

	return(FALSE);
}


/*
 *	Creates a new Thumb.
 */
static tlist_thumb_struct *TListThumbNew(tlist_struct *tlist)
{
	tlist_thumb_struct *thumb = TLIST_THUMB(g_malloc0(
		sizeof(tlist_thumb_struct)
	));
	if(thumb == NULL)
		return(NULL);

	thumb->flags = TLIST_THUMB_SENSITIVE |
					TLIST_THUMB_SELECTABLE;
	thumb->load_state = TLIST_LOAD_STATE_NOT_LOADED;

#if 0
	thumb->pixmap = NULL;
	thumb->mask = NULL;

	thumb->text = NULL;
#endif
	thumb->text_visibility = GTK_VISIBILITY_FULL;
#if 0
	thumb->text_color_fg = NULL;
	thumb->text_color_bg = NULL;

	thumb->loaded_time = 0l;

	thumb->data = NULL;

	thumb->destroy_cb = NULL;
#endif

	return(thumb);
}

/*
 *	Deletes the Thumb.
 *
 *	If the destroy_cb is not NULL then it will be called first
 *	before this Thumb is deleted.
 */
static void TListThumbDelete(
	tlist_struct *tlist,
	tlist_thumb_struct *thumb
)
{
	GdkColormap *colormap = tlist->colormap;

	if(thumb == NULL)
		return;

	/* Notify about this Thumb being deleted */
	if(thumb->destroy_cb != NULL)
		thumb->destroy_cb(thumb->data);

	/* Unref the pixmap and mask */
	(void)GDK_PIXMAP_UNREF(thumb->pixmap);
	(void)GDK_BITMAP_UNREF(thumb->mask);

	/* Delete the text */
	g_free(thumb->text);

	/* Delete the colors */
#define DELETE_COLOR(_c_)	{		\
 if((_c_) != NULL) {				\
  GDK_COLORMAP_FREE_COLOR(colormap, (_c_));	\
  g_free(_c_);					\
 }						\
}
	DELETE_COLOR(thumb->text_color_fg);
	DELETE_COLOR(thumb->text_color_bg);
#undef DELETE_COLOR

	/* Delete the attribute icons */
	if(thumb->attribute_icon_pixmaps_list != NULL)
	{
		g_list_foreach(
			thumb->attribute_icon_pixmaps_list,
			(GFunc)GDK_PIXMAP_UNREF,
			NULL
		);
		g_list_free(thumb->attribute_icon_pixmaps_list);
	}
	if(thumb->attribute_icon_masks_list != NULL)
	{
		g_list_foreach(
			thumb->attribute_icon_masks_list,
			(GFunc)GDK_BITMAP_UNREF,
			NULL
		);
		g_list_free(thumb->attribute_icon_masks_list);
	}

	g_free(thumb);
}


/*
 *	Adds the thumb to the list of selected thumbs.
 *
 *	If append is FALSE then all the previously selected thumbs
 *	will be unselected.
 */
static void TListDoSelectThumb(
	tlist_struct *tlist,
	const gint thumb_num,
	GdkEventButton *button,
	const gboolean append
)
{
	tlist_thumb_struct *thumb;

	if(tlist == NULL)
		return;

	thumb = TListGetThumb(tlist, thumb_num);

	/* Is this thumb selectable? */
	if(!TLIST_THUMB_IS_SELECTABLE(thumb))
		return;

	/* Is this thumb already selected? */
	if(g_list_find(tlist->selection, (gpointer)thumb_num) != NULL)
		return;

	/* If not appending a selection then all selected thumbs must be
	 * unselected first
	 */
	if(!append)
	{
		GList	*selection = tlist->selection,
					*selection_end = tlist->selection_end;

		tlist->selection = NULL;
		tlist->selection_end = NULL;

		/* Notify about the unselecting of all the thumbs */
		if(tlist->unselect_cb != NULL)
		{
			GList *glist;

			for(glist = selection_end;
				glist != NULL;
				glist = g_list_previous(glist)
			)
				tlist->unselect_cb(
					tlist,			/* Thumbs List */
					NULL,			/* No GdkEventButton
											 * since we're unselecting
											 * these for an upcomming
											 * select below */
					(gint)glist->data,	/* Thumb */
					tlist->unselect_data	/* Data */
				);
		}

		/* Delete entire selection */
		if(selection != NULL)
			g_list_free(selection);
	}


	/* Append new selected thumb */
	tlist->selection = g_list_append(
		tlist->selection, (gpointer)thumb_num
	);

	/* Update end selection pointer since main selection changed */
	tlist->selection_end = (tlist->selection != NULL) ?
		g_list_last(tlist->selection) : NULL;

	/* Notify about this thumb being selected */
	if(tlist->select_cb != NULL)
		tlist->select_cb(
			tlist,				/* Thumbs List */
			button,				/* GdkEventButton */
			thumb_num,			/* Thumb Index */
			tlist->select_data		/* Data */
		);

	TListQueueDraw(tlist);
}

/*
 *	Similar to TListDoSelectThumb() except that it selects all
 *	thumbs in sequential indices from the last selected thumb to
 *	the specified thumb.
 *
 *	If there was no previously selected thumb then only the
 *	specified thumb will be selected.
 */
static void TListDoSelectThumbRange(
	tlist_struct *tlist,
	const gint thumb_num,
	GdkEventButton *button
)
{
	gint i, last_thumb_num;
	GList *glist;

	if(tlist == NULL)
		return;

	if((thumb_num < 0) || (thumb_num >= tlist->total_thumbs))
		return;

	/* Get the last selected thumb if any */
	glist = tlist->selection_end;
	last_thumb_num = (glist != NULL) ? (gint)glist->data : -1;
	if((last_thumb_num < 0) || (last_thumb_num >= tlist->total_thumbs))
	{
		/* No previously selected thumb so just select this thumb */
		TListDoSelectThumb(tlist, thumb_num, button, FALSE);
		return;
	}

	/* Now select all thumbs from last_thumb_num to thumb_num */
	if(last_thumb_num < thumb_num)
	{
		/* Select low to high */
		for(i = last_thumb_num + 1; i <= thumb_num; i++)
			TListDoSelectThumb(tlist, i, button, TRUE);
	}
	else if(last_thumb_num > thumb_num)
	{
		/* Select high to low */
		for(i = last_thumb_num - 1; i >= thumb_num; i--)
			TListDoSelectThumb(tlist, i, button, TRUE);
	}
}

/*
 *	Procedure to unselect the given thumb from the list of selected
 *	thumbs on the thumbs list.
 */
static void TListDoUnselectThumb(
	tlist_struct *tlist,
	gint thumb_num,
	GdkEventButton *button
)
{
	if(tlist == NULL)
		return;

	/* Is the specified Thumb not selected? */
	if(g_list_find(tlist->selection, (gpointer)thumb_num) == NULL)
		return;

	/* Remove the Thumb from the selection list */
	if(tlist->selection != NULL)
		tlist->selection = g_list_remove(
			tlist->selection, (gpointer)thumb_num
		);

	/* Update end selection pointer since main selection changed */
	tlist->selection_end = (tlist->selection != NULL) ?
		g_list_last(tlist->selection) : NULL;

	/* Report unselect as needed */
	if(tlist->unselect_cb != NULL)
		tlist->unselect_cb(
			tlist,				/* Thumbs List */
			button,				/* GdkEventButton */
			thumb_num,			/* Thumb Index */
			tlist->unselect_data		/* Data */
		);

	TListQueueDraw(tlist);
}


/*
 *	Selects all the thumbs by first unselected the selection
 *	list and the nselecting all thumbs.
 */
static void TListDoSelectAllThumbs(tlist_struct *tlist)
{
	gint i;
	tlist_thumb_struct *thumb;

	if(tlist == NULL)
		return;

	/* Delete the entire selection list first */
	if(tlist->selection != NULL)
		g_list_free(tlist->selection);
	tlist->selection = NULL;
	tlist->selection_end = NULL;

	/* Select all the thumbs */
	for(i = 0; i < tlist->total_thumbs; i++)
	{
		thumb = tlist->thumb[i];
		if(!TLIST_THUMB_IS_SELECTABLE(thumb))
			continue;

		tlist->selection = g_list_append(
			tlist->selection, (gpointer)i
		);
	}

	/* Update the end selection pointer since main selection changed */
	tlist->selection_end = (tlist->selection != NULL) ?
		g_list_last(tlist->selection) : NULL;


	/* Notify about the selecting of all the thumbs */
	if(tlist->select_cb != NULL)
	{
		GList *glist;

		for(glist = tlist->selection;
			glist != NULL;
			glist = g_list_next(glist)
		)
			tlist->select_cb(
				tlist,			/* Thumbs List */
				NULL,			/* No GdkEventButton */
				(gint)glist->data,		/* Thumb Index */
				tlist->select_data		/* Data */
			);
	}

	TListQueueDraw(tlist);
}

/*
 *	Unselects all the thumbs.
 */
static void TListDoUnselectAllThumbs(tlist_struct *tlist)
{
	GList		*selection,
					*selection_end;

	if(tlist == NULL)
		return;

	/* Transfer the selections list locally */
	selection = tlist->selection;
	selection_end = tlist->selection_end;
	tlist->selection = NULL;
	tlist->selection_end = NULL;

	/* Notify about the unselecting of all the thumbs */
	if(tlist->unselect_cb != NULL)
	{
		GList *glist;

		for(glist = selection_end;
			glist != NULL;
			glist = g_list_previous(glist)
		)
			tlist->unselect_cb(
				tlist,			/* Thumbs List */
				NULL,			/* No GdkEventButton */
				(gint)glist->data,		/* Thumb Index */
				tlist->unselect_data	/* Data */
			);
	}

	/* Delete the entire selection */
	if(selection != NULL)
		g_list_free(selection);

	TListQueueDraw(tlist);
}


/*
 *	Gets the prefered thumb size based on an actual image size.
 *
 *	The img_width and img_height specifies the actual image size.
 *
 *	The width_rtn and height_rtn specifies the return values for
 *	the prefered thumb size.
 */
void TListQueryThumbPixmapSize(
	tlist_struct *tlist,
	const gint img_width, const gint img_height,
	gint *width_rtn, gint *height_rtn 
)
{
	gfloat aspect;
	gint	font_height, thumb_border,
			twidth, theight,
			twidth_clip, theight_clip;
	tlist_flags flags;
	GdkFont *font;
	GtkStyle *style;
	GtkWidget *w;

	if(width_rtn != NULL)
		*width_rtn = 0;
	if(height_rtn != NULL)
		*height_rtn = 0;

	if((tlist == NULL) || (img_width <= 0) || (img_height <= 0))
		return;

	flags = tlist->flags;
	thumb_border = tlist->thumb_border;
	w = tlist->list_da;
	style = gtk_widget_get_style(w);
	font = (style != NULL) ? style->font : NULL;
	font_height = (font != NULL) ? (font->ascent + font->descent) : 0;

	/* Calculate the exact size of the thumb's pixmap
	 *
	 * Note that it will be smaller than the overall thumb's size
	 * because we need to accomidate for the borders, labels, and
	 * frames
	 */
	if(flags & TLIST_SHOW_THUMB_LABELS)
	{
		const gint label_height = font_height + (2 * TLIST_THUMB_DEF_FRAME_HEIGHT);
		twidth_clip = tlist->thumb_width - (2 * thumb_border);
		theight_clip = tlist->thumb_height - (3 * thumb_border) -
			label_height;
	}
	else
	{
		twidth_clip = tlist->thumb_width - (2 * thumb_border);
		theight_clip = tlist->thumb_height - (2 * thumb_border);
	}
	if(flags & TLIST_SHOW_THUMB_FRAMES)
	{
		twidth_clip -= (2 * TLIST_THUMB_DEF_FRAME_WIDTH);
		theight_clip -= (2 * TLIST_THUMB_DEF_FRAME_HEIGHT);
	}
	else
	{
		twidth_clip -= 2;
		theight_clip -= 2;
	}                                          
	if((twidth_clip <= 0) || (theight_clip <= 0))
		return;

	/* Begin calculating thumb width and height, mind the aspect
	 * ratio of the specified image size
	 */
	aspect = (gfloat)img_width / (gfloat)img_height;

	theight = theight_clip;
	twidth = (gint)(theight * aspect);

	if(twidth > twidth_clip)
	{
		aspect = (gfloat)img_height / (gfloat)img_width;

		twidth = twidth_clip;
		theight = (gint)(twidth * aspect);
	}

	if(width_rtn != NULL)
		*width_rtn = twidth;
	if(height_rtn != NULL)
		*height_rtn = theight;
}


/*
 *	Draws a thumb.
 *
 *	The thumb_rect is relative to the origin of the Thumbs List's
 *	GtkDrawingArea.
 *
 *	The rel_label_rect is relative to the thumb and includes the
 *	label's frame.
 *
 *	The drawable is the Thumbs List's GtkDrawingArea.
 */
static void TListDrawThumbIterate(
	tlist_struct *tlist,
	tlist_thumb_struct *thumb, const gint thumb_num,
	const tlist_flags flags,
	const gboolean has_focus,
	const gboolean drag_active,
	const gboolean drag_over,
	const gint thumb_border,
	const gint thumb_pixmap_height_max,
	GdkRectangle *thumb_rect,
	GdkRectangle *rel_label_rect,
	GdkDrawable *drawable,
	GdkFont *font,
	const gint font_height,
	GdkGC *gc,
	const GtkStateType state,
	GtkStyle *style,
	GtkWidget *w
)
{
	const gboolean	is_selected = TListIsThumbSelected(tlist, thumb_num),
					thumb_has_focus = (tlist->focus_thumb == thumb_num) ? TRUE : FALSE,
					thumb_dragged = (is_selected && drag_active &&
						(flags & TLIST_DRAG_ACTIVE_MOVE)) ? TRUE : FALSE;
	gboolean	thumb_sensitive;
	GdkRectangle	pm_rect_buf,
					*pm_rect = &pm_rect_buf,
					label_rect_buf,
					*label_rect = &label_rect_buf,
					label_base_rect_buf,
					*label_base_rect = &label_base_rect_buf;
	tlist_thumb_flags thumb_flags;

	if(thumb == NULL)
		return;

	thumb_flags = thumb->flags;
	thumb_sensitive = (thumb_flags & TLIST_THUMB_SENSITIVE) ? TRUE : FALSE;

	/* Calculate the pixmap's geometry */
	if(thumb->pixmap != NULL)
	{
		gint width, height;
		gdk_window_get_size(
			(GdkWindow *)thumb->pixmap,
			&width, &height
		);
		pm_rect->width = width;
		pm_rect->height = MIN(height, thumb_pixmap_height_max);
	}
	else
	{
		pm_rect->width = 0;
		pm_rect->height = 0;
	}
	/* Include the size of the label? */
	if(flags & TLIST_SHOW_THUMB_LABELS)
	{
		if(flags & TLIST_SHOW_THUMB_FRAMES)
		{
			pm_rect->x = thumb_rect->x + MAX(
				((thumb_rect->width - pm_rect->width) / 2),
				(TLIST_THUMB_DEF_FRAME_WIDTH + thumb_border)
			);
			pm_rect->y = thumb_rect->y + MAX(
				(TLIST_THUMB_DEF_FRAME_HEIGHT + thumb_border +
				((thumb_pixmap_height_max - pm_rect->height) / 2)),
				(TLIST_THUMB_DEF_FRAME_HEIGHT + thumb_border)
			);
		}
		else
		{
			const gint pixmap_and_label_height = pm_rect->height +
				((pm_rect->height > 0) ? thumb_border : 0) +
				rel_label_rect->height;
			pm_rect->x = thumb_rect->x + MAX(
				((thumb_rect->width - pm_rect->width) / 2),
				(1 + thumb_border)
			);
			pm_rect->y = thumb_rect->y + MAX(
				((thumb_rect->height - pixmap_and_label_height) / 2),
				(1 + thumb_border)
			);
		}
	}    
	else
	{
		if(flags & TLIST_SHOW_THUMB_FRAMES)
		{
			pm_rect->x = thumb_rect->x + MAX(
				((thumb_rect->width - pm_rect->width) / 2),
				(TLIST_THUMB_DEF_FRAME_WIDTH + thumb_border) 
			);
			pm_rect->y = thumb_rect->y + MAX(
				((thumb_rect->height - pm_rect->height) / 2),
				(TLIST_THUMB_DEF_FRAME_HEIGHT + thumb_border)
			);
		}
		else
		{
			pm_rect->x = thumb_rect->x + MAX(
				((thumb_rect->width - pm_rect->width) / 2),
				(1 + thumb_border)
			);
			pm_rect->y = thumb_rect->y + MAX(
				((thumb_rect->height - pm_rect->height) / 2),
				(1 + thumb_border)
			);
		}
	}


	/* If this thumb is being dragged then use a the transparent
	 * stippled clip mask to make it look transparent
	 */
	if(thumb_dragged)
	{
		gdk_gc_set_stipple(
			gc,
			(GdkPixmap *)tlist->transparent_stipple_bm
		);
		gdk_gc_set_ts_origin(gc, 0, 0);
		gdk_gc_set_fill(gc, GDK_STIPPLED);
	}


	/* Draw the thumb's frame and the selection? */
	if(flags & TLIST_SHOW_THUMB_FRAMES)
	{
		if(is_selected)
			gdk_gc_set_foreground(
				gc,
				&style->bg[GTK_STATE_SELECTED]
			);
		else if(!thumb_sensitive)
			gdk_gc_set_foreground(
				gc,
				&style->bg[GTK_STATE_INSENSITIVE]
			);
		else if(tlist->pointer_over_thumb == thumb_num)
			gdk_gc_set_foreground(
				gc,
				&style->bg[GTK_STATE_PRELIGHT]
			);
		else
			gdk_gc_set_foreground(
				gc,
				&style->bg[state]
			);
		gdk_draw_rectangle(
			drawable,
			gc,
			TRUE,				/* Fill */
			thumb_rect->x, thumb_rect->y,
			thumb_rect->width, thumb_rect->height
		);
	}
	/* Draw the selection without the thumb's frame? */
	else if(is_selected)
	{
		gdk_gc_set_foreground(
			gc,
			&style->bg[GTK_STATE_SELECTED]
		);
		gdk_draw_rectangle(
			drawable,
			gc,
			TRUE,				/* Fill */
			thumb_rect->x + 1, thumb_rect->y + 1,
			thumb_rect->width - 2, thumb_rect->height - 2
		);
	}

	/* Draw the pixmap */
	if(thumb->pixmap != NULL)
	{
		GdkBitmap *mask = thumb->mask;
		GdkPixmap *pixmap = thumb->pixmap;

		gdk_gc_set_clip_rectangle(gc, thumb_rect);
		gdk_gc_set_clip_mask(gc, mask);
		gdk_gc_set_clip_origin(gc, pm_rect->x, pm_rect->y);
		gdk_draw_pixmap(
			drawable,
			gc,
			(GdkDrawable *)pixmap,
			0, 0,
			pm_rect->x, pm_rect->y,
			pm_rect->width, pm_rect->height
		);
		gdk_gc_set_clip_mask(gc, NULL);
		if(thumb_dragged)
		{
			if(flags & TLIST_SHOW_THUMB_FRAMES)
			{
				if(is_selected)
					gdk_gc_set_foreground(
						gc,
						&style->bg[GTK_STATE_SELECTED]
					);
				else if(!thumb_sensitive)
					gdk_gc_set_foreground(
						gc,
						&style->bg[GTK_STATE_INSENSITIVE]
					);
				else if(tlist->pointer_over_thumb == thumb_num)
					gdk_gc_set_foreground(
						gc,
						&style->bg[GTK_STATE_PRELIGHT]
					);
				else
					gdk_gc_set_foreground(
						gc,
						&style->bg[state]
					);
			}
			else
			{
				if(is_selected)
					gdk_gc_set_foreground(
						gc,
						&style->bg[GTK_STATE_SELECTED]
					);
				else
					gdk_gc_set_foreground(
						gc,
						&style->base[state]
					);
			}
			gdk_draw_rectangle(
				drawable,
				gc,
				TRUE,			/* Fill */
				pm_rect->x, pm_rect->y,
				pm_rect->width, pm_rect->height
			);
		}
		gdk_gc_set_clip_rectangle(gc, NULL);
	}


	/* Calculate the label rectangle, draw the text, text base,
	 * and text frame
	 */
	if((flags & TLIST_SHOW_THUMB_LABELS) &&
	   (font != NULL) && (thumb->text != NULL)
	)
	{
		const gchar *s = thumb->text;
		GdkTextBounds b;

		/* Calculate the text geometry relative to the Thumb
		 * List's GtkDrawingArea
		 */
		if(flags & TLIST_SHOW_THUMB_FRAMES)
		{
			/* When frames are shown the label should be at the
			 * bottom of the thumb, the label rectangle's
			 * relative position is already at the bottom of the
			 * thumb so just use it as is
			 */
			label_rect->x = thumb_rect->x + rel_label_rect->x;
			label_rect->y = thumb_rect->y + rel_label_rect->y;
			label_rect->width = rel_label_rect->width;
			label_rect->height = rel_label_rect->height;
		}
		else
		{
			/* When frames are not shown the label should be at
			 * the bottom of the pixmap with thumb_border
			 * separating the pixmap and label
			 */
			label_rect->x = thumb_rect->x + rel_label_rect->x;
			label_rect->y = pm_rect->y + pm_rect->height +
				((pm_rect->height > 0) ? thumb_border : 0);
			label_rect->width = rel_label_rect->width;
			label_rect->height = rel_label_rect->height;
		}

		/* Get the geometry of label's text */
		gdk_string_bounds(font, s, &b);

		/* Calculate the label's base rectangle which should be
		 * only as wide enough to accomidate the text plus the
		 * frames
		 */
		label_base_rect->width = MIN(
			(4 * TLIST_THUMB_DEF_FRAME_WIDTH) + b.width,
			label_rect->width
		);
		label_base_rect->height = label_rect->height;
		label_base_rect->x = thumb_rect->x + MAX(
			((thumb_rect->width - label_base_rect->width) / 2),
			thumb_border + TLIST_THUMB_DEF_FRAME_WIDTH
		);
		label_base_rect->y = label_rect->y;

		/* Draw the label's background */
		if(flags & TLIST_SHOW_THUMB_FRAMES)
		{
			if(thumb->text_color_bg != NULL)
				gdk_gc_set_foreground(
					gc,
					thumb->text_color_bg
				);
			else if(is_selected)
				gdk_gc_set_foreground(
					gc,
					&style->bg[GTK_STATE_SELECTED]
				);
			else if(!thumb_sensitive)
				gdk_gc_set_foreground(
					gc,
					&style->base[GTK_STATE_INSENSITIVE]
				);
			else
				gdk_gc_set_foreground(
					gc,
					&style->base[state]
				);
			gdk_draw_rectangle(
				drawable,
				gc,
				TRUE,			/* Fill */
				label_base_rect->x, label_base_rect->y,
				label_base_rect->width, label_base_rect->height
			);
		}

		/* Draw the label's text */
		gdk_gc_set_clip_rectangle(gc, label_rect);
		if(thumb->text_color_fg != NULL)
			gdk_gc_set_foreground(
				gc,
				thumb->text_color_fg
			);
		else if(is_selected)
			gdk_gc_set_foreground(
				gc,
				&style->text[GTK_STATE_SELECTED]
			);
		else if(!thumb_sensitive)
			gdk_gc_set_foreground(
				gc,
				&style->text[GTK_STATE_INSENSITIVE]
			);
		else
			gdk_gc_set_foreground(
				gc,
				&style->text[state]
			);
		gdk_gc_set_font(gc, font);
		gdk_draw_string(
			drawable,
			font,
			gc,
			thumb_rect->x + ((thumb_rect->width - b.width) / 2) -
				b.lbearing,
			label_rect->y + ((label_rect->height - font_height) / 2) +
				font->ascent,
			s
		);
		gdk_gc_set_clip_rectangle(gc, NULL);

		/* Draw the label's frame */
		if(flags & TLIST_SHOW_THUMB_FRAMES)
			gtk_paint_shadow(
				style,
				(GdkWindow *)drawable,
				thumb_sensitive ? state : GTK_STATE_INSENSITIVE,
				GTK_SHADOW_IN,
				NULL,			/* Entire area */
				w,
				NULL,			/* No detail */
				label_base_rect->x, label_base_rect->y,
				label_base_rect->width, label_base_rect->height
			);
	}


	/* Draw the thumb's attribute icons */
	if(thumb->attribute_icon_pixmaps_list != NULL)
	{
		gint	x, y,
					width, height;
		GList	*pixmaps_list,
					*masks_list;
		GdkBitmap *mask;
		GdkPixmap *pixmap;

		/* Calculate the initial X coordinate position */
		x = 0;
		switch(thumb->attribute_icon_placement)
		{
		  case GTK_CORNER_TOP_LEFT:
			if(flags & TLIST_SHOW_THUMB_FRAMES)
				x = thumb_rect->x;
			else
				x = pm_rect->x;
			break;
		  case GTK_CORNER_BOTTOM_LEFT:
			if(flags & TLIST_SHOW_THUMB_LABELS)
				x = label_base_rect->x - 2;
			else if(flags & TLIST_SHOW_THUMB_FRAMES)
				x = thumb_rect->x;
			else
				x = pm_rect->x;
			break;
		  case GTK_CORNER_TOP_RIGHT:
			if(flags & TLIST_SHOW_THUMB_FRAMES)
				x = thumb_rect->x + thumb_rect->width;
			else
				x = pm_rect->x + pm_rect->width;
			break;
		  case GTK_CORNER_BOTTOM_RIGHT:
			if(flags & TLIST_SHOW_THUMB_LABELS)
				x = label_base_rect->x + label_base_rect->width + 2;
			else if(flags & TLIST_SHOW_THUMB_FRAMES)
				x = thumb_rect->x + thumb_rect->width;
			else
				x = pm_rect->x + pm_rect->width;
			break;
		}

		for(pixmaps_list = thumb->attribute_icon_pixmaps_list,
			masks_list = thumb->attribute_icon_masks_list;
			(pixmaps_list != NULL) && (masks_list != NULL);
			pixmaps_list = g_list_next(pixmaps_list),
			masks_list = g_list_next(masks_list)
		)
		{
			pixmap = (GdkPixmap *)pixmaps_list->data;
			mask = (GdkBitmap *)masks_list->data;
			if(pixmap == NULL)
				continue;

			gdk_window_get_size((GdkWindow *)pixmap, &width, &height);
			y = 0;
			switch(thumb->attribute_icon_placement)
			{
			  case GTK_CORNER_TOP_LEFT:
				if(flags & TLIST_SHOW_THUMB_FRAMES)
					y = thumb_rect->y;
				else
					y = pm_rect->y - height;
				break;
			  case GTK_CORNER_BOTTOM_LEFT:
				if(flags & TLIST_SHOW_THUMB_LABELS)
				{
					x -= width;
					y = label_base_rect->y +
						((label_base_rect->height - height) / 2);
				}
				else if(flags & TLIST_SHOW_THUMB_FRAMES)
				{
					y = thumb_rect->y + thumb_rect->height;
				}
				else
				{
					y = pm_rect->y + pm_rect->height;
				}
				break;
			  case GTK_CORNER_TOP_RIGHT:
				if(flags & TLIST_SHOW_THUMB_FRAMES)
				{
					x -= width;
					y = thumb_rect->y;
				}
				else
				{
					x -= width;
					y = pm_rect->y - height;
				}
				break;
			  case GTK_CORNER_BOTTOM_RIGHT:
				if(flags & TLIST_SHOW_THUMB_LABELS)
				{
					y = label_base_rect->y +
						((label_base_rect->height - height) / 2);
				}
				else if(flags & TLIST_SHOW_THUMB_FRAMES)
				{
					x -= width;
					y = thumb_rect->y + thumb_rect->height;
				}
				else
				{
					x -= width;
					y = pm_rect->y + pm_rect->height;
				}
				break;
			}

			/* Keep the icon within the thumb's rectangle */
			if(flags & TLIST_SHOW_THUMB_FRAMES)
			{
				if((x + width) > (thumb_rect->x + thumb_rect->width - 2))
					x = thumb_rect->x + thumb_rect->width - 2 - width;
				if(x < (thumb_rect->x + 2))
					x = thumb_rect->x + 2;
				if((y + height) > (thumb_rect->y + thumb_rect->height - 2))
					y = thumb_rect->y + thumb_rect->height - 2 - height;
				if(y < (thumb_rect->y + 2))
					y = thumb_rect->y + 2;
			}
			else
			{
				if((x + width) > (thumb_rect->x + thumb_rect->width))
					x = thumb_rect->x + thumb_rect->width - width;
				if(x < (thumb_rect->x))
					x = thumb_rect->x;
				if((y + height) > (thumb_rect->y + thumb_rect->height))
					y = thumb_rect->y + thumb_rect->height - height;
				if(y < (thumb_rect->y))
					y = thumb_rect->y;
			}
			gdk_gc_set_clip_rectangle(gc, thumb_rect);
			gdk_gc_set_clip_mask(gc, mask);
			gdk_gc_set_clip_origin(gc, x, y);
			gdk_draw_pixmap(
				drawable,
				gc,
				(GdkDrawable *)pixmap,
				0, 0,
				x, y,
				width, height
			);
			gdk_gc_set_clip_mask(gc, NULL);
			gdk_gc_set_clip_rectangle(gc, NULL);

			/* Increment the X position */
			switch(thumb->attribute_icon_placement)
			{
			  case GTK_CORNER_TOP_LEFT:
			  case GTK_CORNER_BOTTOM_LEFT:
				x += width;
				break;
			  case GTK_CORNER_TOP_RIGHT:
			  case GTK_CORNER_BOTTOM_RIGHT:
				x -= width;
				break;
			}
		}
	}

	/* Draw the thumb's frame */
	if(flags & TLIST_SHOW_THUMB_FRAMES)
		gtk_paint_shadow(
			style,
			(GdkWindow *)drawable,
			thumb_sensitive ? state : GTK_STATE_INSENSITIVE,
			GTK_SHADOW_OUT,
			NULL,				/* Entire area */
			w,
			NULL,				/* No detail */
			thumb_rect->x, thumb_rect->y,
			thumb_rect->width, thumb_rect->height
		);


	if(thumb_dragged)
	{
		gdk_gc_set_fill(gc, GDK_SOLID);
	}


	/* Draw the focus rectangle if the list is in focus or a drag
	 * is over it and this is the focus_thumb
	 */
	if((has_focus || drag_over) && thumb_has_focus)
	{
		gdk_gc_set_foreground(gc, &style->fg[state]);
		if(is_selected)
			gdk_gc_set_function(gc, GDK_INVERT);
		if(flags & TLIST_SHOW_THUMB_FRAMES)
			gdk_draw_rectangle(
				drawable,
				gc,
				FALSE,			/* Outline */
				thumb_rect->x + 2, thumb_rect->y + 2,
				thumb_rect->width - 5, thumb_rect->height - 5
			);
		else
			gdk_draw_rectangle(
				drawable,
				gc,
				FALSE,			/* Outline */
				thumb_rect->x + 1, thumb_rect->y + 1,
				thumb_rect->width - 3, thumb_rect->height - 3
			);
		gdk_gc_set_function(gc, GDK_COPY);
	}
}

/*
 *	Redraws the Thumbs List.
 *
 *	If the Thumbs List. is currently frozen or unmapped then nothing
 *	will be done.
 */
void TListDraw(tlist_struct *tlist)
{
	gboolean	has_focus,
					drag_active,
					drag_over;
	gint		width, height,
					thumb_border, thumb_pixmap_height_max,
					tpl,
					font_height,
					label_height;
	GdkRectangle rel_label_rect, thumb_rect;
	GdkFont *font;
	GdkPixmap *bg_pixmap;
	GdkWindow *window;
	GdkDrawable *drawable;
	GdkGC *gc;
	GtkStateType state;
	GtkAdjustment *hadj, *vadj;
	GtkStyle *style;
	GtkWidget *w;
	tlist_flags flags;

	if(tlist == NULL)
		return;

	if(tlist->freeze_count > 0)
		return;

	w = tlist->list_da;

	if(!GTK_WIDGET_VISIBLE(w))
		return;

	/* Realize the Thumbs List as needed to get the GC */
	if(!(tlist->flags & TLIST_REALIZED))
		gtk_widget_realize(tlist->list_da);

	flags = tlist->flags;
	has_focus = GTK_WIDGET_HAS_FOCUS(w);
	drag_active = (flags & TLIST_DRAG_ACTIVE) ? TRUE : FALSE;
	drag_over = (flags & TLIST_DRAG_OVER) ? TRUE : FALSE;
	window = w->window;
	gc = tlist->gc;
	state = GTK_WIDGET_STATE(w);
	style = gtk_widget_get_style(w);
	if((window == NULL) || (gc == NULL) || (style == NULL))
		return;

	gdk_window_get_size(window, &width, &height);
	if((width <= 0) || (height <= 0))
		return;

	drawable = (GdkDrawable *)tlist->list_pm;
	if(drawable == NULL)
		drawable = (GdkDrawable *)window;

	font = style->font;
	font_height = (font != NULL) ? (font->ascent + font->descent) : 0;
	label_height = font_height + (2 * TLIST_THUMB_DEF_FRAME_HEIGHT);

	gdk_gc_set_function(gc, GDK_COPY);

	/* Draw the background */
	bg_pixmap = style->bg_pixmap[state];
	if(bg_pixmap != NULL)
	{
		gdk_gc_set_tile(gc, bg_pixmap);
		gdk_gc_set_ts_origin(gc, 0, 0);
		gdk_gc_set_fill(gc, GDK_TILED);
	}
	else
	{
		gdk_gc_set_fill(gc, GDK_SOLID);
		gdk_gc_set_foreground(gc, &style->base[state]);
	}
	gdk_draw_rectangle(
		drawable,
		gc,
		TRUE,				/* Fill */
		0, 0,
		width, height
	);
	gdk_gc_set_fill(gc, GDK_SOLID);


	/* Begin drawing thumbs */

	hadj = tlist->hadjustment;
	vadj = tlist->vadjustment;

	/* Calculate thumbs per line */
	tpl = MAX(tlist->thumbs_per_line, 1);

	/* Calculate the thumb geometry */
	thumb_rect.width = tlist->thumb_width;
	thumb_rect.height = tlist->thumb_height;
	thumb_border = tlist->thumb_border;
	thumb_pixmap_height_max = thumb_rect.height -
		((flags & TLIST_SHOW_THUMB_FRAMES) ?
			(2 * TLIST_THUMB_DEF_FRAME_HEIGHT) : 2) -
		(2 * thumb_border) -
		((flags & TLIST_SHOW_THUMB_LABELS) ?
			(label_height + thumb_border) : 0);

	/* Calculate the label geometry */
	rel_label_rect.width = MAX(
		(thumb_rect.width -
		((flags & TLIST_SHOW_THUMB_FRAMES) ?
			(2 * TLIST_THUMB_DEF_FRAME_WIDTH) : 2) -
		(2 * thumb_border)),
		(2 * TLIST_THUMB_DEF_FRAME_WIDTH)
	);
	rel_label_rect.height = label_height;
	rel_label_rect.x = ((flags & TLIST_SHOW_THUMB_FRAMES) ?
		TLIST_THUMB_DEF_FRAME_WIDTH : 1) + thumb_border;
	rel_label_rect.y = thumb_rect.height -
		((flags & TLIST_SHOW_THUMB_FRAMES) ? TLIST_THUMB_DEF_FRAME_HEIGHT : 1) -
		thumb_border -
		rel_label_rect.height;


	/* Enough information to draw the thumbs? */
	if((thumb_rect.width > 0) && (thumb_rect.height > 0) &&
	   (hadj != NULL) && (vadj != NULL)
	)
	{
		gint	i, i_start,
					x_start, y_start;

		if(tlist->orientation == GTK_ORIENTATION_HORIZONTAL)
		{
			/* Calculate starting positions */
			i_start = (gint)(hadj->value / thumb_rect.width) * tpl;
			x_start = -((gint)hadj->value % thumb_rect.width);
			y_start = -(gint)vadj->value;

			/* Begin iterating and drawing each thumb */
			thumb_rect.x = x_start;
			thumb_rect.y = y_start;
			for(i = MAX(i_start, 0); i < tlist->total_thumbs; i++)
			{
				/* Drawn past end of page? */
				if(thumb_rect.x > hadj->page_size)
					break;

				/* Draw this thumb */ 
				TListDrawThumbIterate(
					tlist,
					tlist->thumb[i], i,
					flags,
					has_focus,
					drag_active,
					drag_over,
					thumb_border,
					thumb_pixmap_height_max,
					&thumb_rect,
					&rel_label_rect,
					drawable,
					font, font_height,
					gc,
					state,
					style,
					w
				);

				/* Increment coordinates */
				thumb_rect.y += thumb_rect.height;

				/* Drawn past the edge of the "row"? */
				if((thumb_rect.y + thumb_rect.height) >
					(gint)vadj->page_size
				)
				{
					thumb_rect.x += thumb_rect.width;
					thumb_rect.y = y_start;
				}
			}
		}
		else
		{
			/* Calculate starting positions */
			i_start = (gint)(vadj->value / thumb_rect.height) * tpl;
			x_start = -(gint)hadj->value;
			y_start = -((gint)vadj->value % thumb_rect.height);

			/* Begin iterating and drawing each thumb */
			thumb_rect.x = x_start;
			thumb_rect.y = y_start;
			for(i = MAX(i_start, 0); i < tlist->total_thumbs; i++)
			{
				/* Drawn past end of page? */
				if(thumb_rect.y > vadj->page_size)
					break;

				/* Draw this thumb */
				TListDrawThumbIterate(
					tlist,
					tlist->thumb[i], i,
					flags,
					has_focus,
					drag_active,
					drag_over,
					thumb_border,
					thumb_pixmap_height_max,
					&thumb_rect,
					&rel_label_rect,
					drawable,
					font,
					font_height,
					gc,
					state,
					style,
					w
				);

				/* Increment coordinates */
				thumb_rect.x += thumb_rect.width;

				/* Drawn past the edge of the "row"? */
				if((thumb_rect.x + thumb_rect.width) >
					(gint)hadj->page_size
				)
				{
					thumb_rect.y += thumb_rect.height;
					thumb_rect.x = x_start;
				}
			}
		}
	}

	/* Is a drag occuring over this list? */
	if(drag_over && (state != GTK_STATE_INSENSITIVE))
		gtk_paint_focus(
			style,
			(GdkWindow *)drawable,
			NULL,				/* Entire area */
			w,
			NULL,				/* No detail */
			0, 0,
			width - 1, height - 1
		);

	/* Send drawable to window if drawable is not the window (using
	 * double buffers)
	 */
	if(drawable != (GdkDrawable *)window)
		gdk_draw_pixmap(
			(GdkDrawable *)window,
			gc,
			drawable,
			0, 0,
			0, 0,
			width, height
		);
}

/*
 *	Queues a redraw for the Thumbs List.
 */
void TListQueueDraw(tlist_struct *tlist)
{
	if(tlist == NULL)
		return;

	gtk_widget_queue_draw(tlist->list_da);
}


/*
 *	Resizes the Thumbs List.
 *
 *	The width and height specifies the size. If either width or
 *	height is -1 then the current size will be used.
 *
 *	If the TLIST_DOUBLE_BUFFER is set then the back buffer
 *	GdkPixmap will be recreated.
 *
 *	The values for the GtkAdjustments will be recalculated and a
 *	"changed" signal will be emitted, resulting in the Thumbs
 *	List being redrawn.
 */
void TListResize(
	tlist_struct *tlist,
	const gint width, const gint height
)
{
	gint		tpl,
					_width = width,
					_height = height;
	GdkWindow *window;
	GtkAdjustment *hadj, *vadj;
	GtkWidget *w;
	tlist_flags flags;

	if(tlist == NULL)
		return;

	flags = tlist->flags;
	w = tlist->list_da;
	window = w->window;
	hadj = tlist->hadjustment;
	vadj = tlist->vadjustment;
	if((window == NULL) || (hadj == NULL) || (vadj == NULL))
		return;

	/* Use the current size if none was specified */
	if((_width <= 0) || (_height <= 0))
		gdk_window_get_size(
			window,
			&_width, &_height
		);

	/* Empty size? */
	if((tlist->thumb_width <= 0) || (tlist->thumb_height <= 0) ||
	   (_width <= 0) || (_height <= 0)
	)
		return;

	/* Recreate the back buffer GdkPixmap if the
	 * TLIST_DOUBLE_BUFFER flag is set
	 */
	(void)GDK_PIXMAP_UNREF(tlist->list_pm);
	if(flags & TLIST_DOUBLE_BUFFER)
	{
		tlist->list_pm = gdk_pixmap_new(
			window,
			_width, _height,
			-1
		);
		if(tlist->list_pm == NULL)
			return;
	}
	else
	{
		tlist->list_pm = NULL;
	}

	/* Update the GtkAdjustments, GtkScrollBars, and
	 * thumbs_per_line values
	 */
	if(tlist->orientation == GTK_ORIENTATION_HORIZONTAL)
	{
		/* Calculate new number of thumbs per line */
		tlist->thumbs_per_line = tpl = MAX(
			_height / tlist->thumb_height, 1
		);

		/* Update adjustment ranges and page sizes */
		hadj->lower = 0.0f;
		hadj->upper = (gfloat)MAX(
			((gint)(tlist->total_thumbs / tpl) * tlist->thumb_width) +
			(((tlist->total_thumbs % tpl) == 0) ? 0 : tlist->thumb_width),
			_width
		);
		hadj->page_size = (gfloat)_width;
		hadj->page_increment = hadj->page_size / 2;
		hadj->step_increment = tlist->thumb_width;

		vadj->lower = 0.0f;
		vadj->upper = (gfloat)MAX(_height, tlist->thumb_height);
		vadj->page_size = (gfloat)_height;
		vadj->page_increment = vadj->page_size / 2;
		vadj->step_increment = tlist->thumb_height / 2;
	}
	else
	{
		/* Calculate new number of thumbs per line */
		tlist->thumbs_per_line = tpl = MAX(
			_width / tlist->thumb_width, 1
		);

		/* Update adjustment ranges and page sizes */
		hadj->lower = 0.0f;
		hadj->upper = (gfloat)MAX(_width, tlist->thumb_width);
		hadj->page_size = (gfloat)_width;
		hadj->page_increment = hadj->page_size / 2;
		hadj->step_increment = tlist->thumb_width / 2;

		vadj->lower = 0.0f;
		vadj->upper = (gfloat)MAX(
			((gint)(tlist->total_thumbs / tpl) * tlist->thumb_height) +
			(((tlist->total_thumbs % tpl) == 0) ? 0 : tlist->thumb_height),
			_height
		);
		vadj->page_size = (gfloat)_height;
		vadj->page_increment = vadj->page_size / 2;
		vadj->step_increment = tlist->thumb_height;
	}

	/* Clip scroll adjustment values */
	if(hadj->value > (hadj->upper - hadj->page_size))
		hadj->value = hadj->upper - hadj->page_size;
	if(hadj->value < hadj->lower)
		hadj->value = hadj->lower;

	if(vadj->value > (vadj->upper - vadj->page_size))
		vadj->value = vadj->upper - vadj->page_size;
	if(vadj->value < vadj->lower)
		vadj->value = vadj->lower;

	/* Notify the GtkAdjustments about the changes to their bounds */
	gtk_signal_emit_by_name(GTK_OBJECT(hadj), "changed");
	gtk_signal_emit_by_name(GTK_OBJECT(vadj), "changed");

	/* Notify the GtkAdjustments about the changes to their values */
	gtk_signal_emit_by_name(GTK_OBJECT(hadj), "value_changed");
	gtk_signal_emit_by_name(GTK_OBJECT(vadj), "value_changed");

	/* Show or hide the GtkScrollBars based on the GtkAdjustment's
	 * page_size
	 */
	w = tlist->hsb;
	if(w != NULL)
	{
		if((hadj->upper - hadj->lower) > hadj->page_size)
			gtk_widget_show(w);
		else
			gtk_widget_hide(w);
	}
	w = tlist->vsb;
	if(w != NULL)
	{
		if((vadj->upper - vadj->lower) > vadj->page_size)
			gtk_widget_show(w);
		else
			gtk_widget_hide(w);
	}
}


/*
 *	Freezes the Thumbs List.
 */
void TListFreeze(tlist_struct *tlist)
{
	if(tlist == NULL)
		return;

	tlist->freeze_count++;

	/* Previous freeze count was 0? */
	if(tlist->freeze_count == 1)
	{
		/* Freeze list */


	}
}

/*
 *	Thaws the Thumbs List.
 *
 *	If the previous freeze count was positive then the Thumbs List
 *	will be redrawn.
 */
void TListThaw(tlist_struct *tlist)
{
	if(tlist == NULL)
		return;

	tlist->freeze_count--;

	/* Queue a redraw when freeze counts reach 0 */
	if(tlist->freeze_count == 0)
		TListQueueDraw(tlist);

	if(tlist->freeze_count < 0)
		tlist->freeze_count = 0;
}


/*
 *	Returns the Thumb's text visibility based on the specified
 *	thumb's size and settings of the Thumbs List.
 */
static GtkVisibility TListThumbTextVisiblity(
	tlist_struct *tlist, const gchar *text
)
{
	GdkTextBounds b;
	GdkFont *font;
	GtkStyle *style;
	GtkWidget *w;
	tlist_flags flags;

	if((tlist == NULL) || (text == NULL))
		return(GTK_VISIBILITY_NONE);

	flags = tlist->flags;

	/* Thumb labels never shown? */
	if(!(flags & TLIST_SHOW_THUMB_LABELS))
		return(GTK_VISIBILITY_NONE);

	w = tlist->list_da;
	style = gtk_widget_get_style(w);
	font = (style != NULL) ? style->font : NULL;
	if(font == NULL)
		return(GTK_VISIBILITY_NONE);

	/* Get width of text */
	gdk_string_bounds(font, text, &b);

	/* Is the width of the text greater than the visible space
	 * available to draw the thumb's text?
	 */
	if(flags & TLIST_SHOW_THUMB_FRAMES)
		return(
			(b.width > (tlist->thumb_width - (2 * tlist->thumb_border)
				- (4 * TLIST_THUMB_DEF_FRAME_WIDTH))
			) ? GTK_VISIBILITY_PARTIAL : GTK_VISIBILITY_FULL
		);
	else
		return(
			(b.width > (tlist->thumb_width - (2 * tlist->thumb_border)  
				- (2 * TLIST_THUMB_DEF_FRAME_WIDTH))
			) ? GTK_VISIBILITY_PARTIAL : GTK_VISIBILITY_FULL
		);
}

/*
 *	Inserts a new Thumb to the Thumbs List at the position specified
 *	by thumb_num.
 *
 *	Returns the new Thumb's index or negative on error.
 */
gint TListInsert(
	tlist_struct *tlist,
	const gint thumb_num,
	const gchar *text
)
{
	gint _thumb_num = thumb_num;
	tlist_thumb_struct *thumb;

	if(tlist == NULL)
		return(-2);

	/* Create a new Thumb */
	thumb = TListThumbNew(tlist);
	if(thumb == NULL)
		return(-3);

	/* Set text and text visibility */
	thumb->text = STRDUP((text != NULL) ? text : "");
	thumb->text_visibility = TListThumbTextVisiblity(tlist, text);

	/* Append the Thumb if the specified position is out of range */
	if((_thumb_num < 0) || (_thumb_num >= tlist->total_thumbs))
	{
		/* Append Thumb */
		_thumb_num = MAX(tlist->total_thumbs, 0);
		tlist->total_thumbs = _thumb_num + 1;

		/* Allocate more pointers */
		tlist->thumb = (tlist_thumb_struct **)g_realloc(
			tlist->thumb,
			tlist->total_thumbs * sizeof(tlist_thumb_struct *)
		);
		if(tlist->thumb == NULL)
		{
			TListThumbDelete(tlist, thumb);
			tlist->total_thumbs = 0;
			return(-3);
		}              

		tlist->thumb[_thumb_num] = thumb;
	}
	else
	{
		/* Insert Thumb */
		gint i;
		GList *glist;

		if(tlist->total_thumbs < 0)
			tlist->total_thumbs = 0;
		tlist->total_thumbs++;

		/* Allocate more pointers */
		tlist->thumb = (tlist_thumb_struct **)g_realloc(
			tlist->thumb,
			tlist->total_thumbs * sizeof(tlist_thumb_struct *)
		);
		if(tlist->thumb == NULL)
		{
			TListThumbDelete(tlist, thumb);
			tlist->total_thumbs = 0;
			return(-3);
		}

		/* Shift the pointers */
		for(i = tlist->total_thumbs - 1; i > _thumb_num; i--)
			tlist->thumb[i] = tlist->thumb[i - 1];

		tlist->thumb[_thumb_num] = thumb;

		/* Shift the selection */
		for(glist = tlist->selection;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			i = (gint)glist->data;
			if(i >= _thumb_num)
				glist->data = (gpointer)(i + 1);
		}
	}

	/* Update the GtkScrollBar values */
	TListResize(
		tlist,
		-1, -1
	);

	return(_thumb_num);
}

/*
 *	Appends a new Thumb to the Thumbs List.
 *
 *	Returns the new Thumb's index or negative on error.
 */
gint TListAppend(
	tlist_struct *tlist,
	const gchar *text
)
{
	return(TListInsert(tlist, -1, text));
}

/*
 *	Sets the load state of the thumb to load_state.
 */
void TListSetLoadState(
	tlist_struct *tlist,
	const gint thumb_num,
	const tlist_load_state load_state
)
{
	tlist_thumb_struct *thumb = TListGetThumb(tlist, thumb_num);
	if(thumb == NULL)
		return;

	if(thumb->load_state != load_state)
	{
		thumb->load_state = load_state;
		TListQueueDraw(tlist);
	}
}

/*
 *	Sets the Thumb's text.
 */
void TListSetText(
	tlist_struct *tlist,
	const gint thumb_num,
	const gchar *text
)
{
	tlist_thumb_struct *thumb = TListGetThumb(tlist, thumb_num);
	if(thumb == NULL)
		return;

	/* Set text and text visibility */
	g_free(thumb->text);
	thumb->text = STRDUP((text != NULL) ? text : "");
	thumb->text_visibility = TListThumbTextVisiblity(tlist, text);

	TListQueueDraw(tlist);
}

/*
 *	Sets the Thumbs foreground and background colors.
 */
void TListSetTextColor(
	tlist_struct *tlist,
	const gint thumb_num,
	GdkColor *fg,
	GdkColor *bg
)
{
	GdkColor *c;
	GdkColormap *colormap;
	tlist_thumb_struct *thumb = TListGetThumb(tlist, thumb_num);
	if(thumb == NULL)
		return;

	/* Realize the Thumbs List as needed to get the GdkColormap */
	if(!(tlist->flags & TLIST_REALIZED))
		gtk_widget_realize(tlist->list_da);

	colormap = tlist->colormap;
	if(colormap == NULL)
		return;

	/* Foreground Color */
	c = thumb->text_color_fg;
	if(c != NULL)
	{
		GDK_COLORMAP_FREE_COLOR(colormap, c);
		g_free(c);
	}
	if(fg != NULL)
	{
		thumb->text_color_fg = c = g_memdup(
			fg,
			sizeof(GdkColor)
		);
		GDK_COLORMAP_ALLOC_COLOR(colormap, c);
	}
	else
	{
		thumb->text_color_fg = NULL;
	}

	/* Background Color */
	c = thumb->text_color_bg;
	if(c != NULL)
	{
		GDK_COLORMAP_FREE_COLOR(colormap, c);
		g_free(c);
	}
	if(bg != NULL)
	{
		thumb->text_color_bg = c = (GdkColor *)g_malloc(sizeof(GdkColor));
		if(c != NULL)
		{
			memcpy(c, bg, sizeof(GdkColor));
			GDK_COLORMAP_ALLOC_COLOR(colormap, c);
		}
	}
	else
	{
		thumb->text_color_bg = NULL;
	}

	TListQueueDraw(tlist);
}

/*
 *	Sets the Thumb's pixmap and mask.
 */
void TListSetPixmap(
	tlist_struct *tlist,
	const gint thumb_num,
	GdkPixmap *pixmap, GdkBitmap *mask
)
{
	tlist_thumb_struct *thumb = TListGetThumb(tlist, thumb_num);
	if(thumb == NULL)
		return;

	if(pixmap != NULL)
	{
		/* Set the new pixmap and mask */
		(void)GDK_PIXMAP_UNREF(thumb->pixmap);
		(void)GDK_BITMAP_UNREF(thumb->mask);
		thumb->pixmap = GDK_PIXMAP_REF(pixmap);
		thumb->mask = GDK_BITMAP_REF(mask);
	}
	else
	{
		/* Clear the existing pixmap and mask */
		thumb->pixmap = GDK_PIXMAP_UNREF(thumb->pixmap);
		thumb->mask = GDK_BITMAP_UNREF(thumb->mask);
	}

	TListQueueDraw(tlist);
}

/*
 *	Sets the thumb's pixmap image from RGBA image data.
 *
 *	The thumb_num specifies the thumb.
 *
 *	The width and height specifies the size of the RGBA image data.
 *
 *	The bpl specifies the bytes per line. If bpl is -1 then it will
 *	be automatically calculated.
 *
 *	The dith specifies the GdkRgbDither dither amount.
 *
 *	The rgba specifies the RGBA image data.
 *
 *	If no_enlarge is TRUE then no upscaling of the image will
 *	be done if it is smaller than the thumb.
 */
void TListSetRGBA(
	tlist_struct *tlist,
	const gint thumb_num,
	const gint width, const gint height,
	const gint bpl,
	const GdkRgbDither dith,
	const guint8 *rgba,
	const gboolean no_enlarge
)
{
	gint		twidth, theight,
					_bpl = bpl;
	GdkBitmap *mask;
	GdkPixmap *pixmap;
	GtkStyle *style;
	GtkWidget *w;
	tlist_thumb_struct *thumb = TListGetThumb(tlist, thumb_num);
	if(thumb == NULL)
		return;

	/* Unref existing pixmap and mask pair (if any) */
	thumb->pixmap = GDK_PIXMAP_UNREF(thumb->pixmap);
	thumb->mask = GDK_BITMAP_UNREF(thumb->mask);

	/* If no image data is given then stop at this point so that
	 * the only operation done was unloading of the thumb's
	 * previous pixmap
	 */
	if((rgba == NULL) || (width <= 0) || (height <= 0))
		return;

	/* Calculate bytes per line as needed */
	if(_bpl <= 0)
		_bpl = width * 4;

	/* Get values from list's drawing area widget */
	w = tlist->list_da;
	style = gtk_widget_get_style(w);

	/* Calculate the exact size of the pixmap to be displayed
	 * on the thumb based on the specified image size
	 */
	TListQueryThumbPixmapSize(
		tlist,
		width, height,		/* Actual size of image */
		&twidth, &theight		/* Returned thumb size */
	);
	if((twidth <= 0) || (theight <= 0))
		return;

	/* If the image is not to be enlarged when displayed on the
	 * thumb then update twidth and theight to match the specified
	 * image size
	 */
	if(no_enlarge)
	{
		if(twidth > width)
			twidth = width;
		if(theight > height)
			theight = height;
	}

	/* Make sure that the desired thumb size is non-zero */
	if((twidth <= 0) || (theight <= 0))
		return;

	/* At this point, twidth and theight are the desired size of
	 * the thumb image data
	 */

	/* Do we need to resize the specified image data to
	 * accomidate for the size of the thumb?
	 */
	if((twidth != width) || (theight != height))
	{
		/* Calculate bytes per line for the thumb image data and
		 * allocate tempory buffer for thumb image
		 */
		const gint tbpl = twidth * 4 * (gint)sizeof(guint8);
		guint8 *thumb_rgba = (guint8 *)g_malloc(tbpl * theight);
		if(thumb_rgba == NULL)
			return;

		/* Copy/resize the specified RGBA data to the thumb's
		 * TGBA data
		 */
		GUIImageBufferResize(
			4,
			rgba, width, height, _bpl,
			thumb_rgba, twidth, theight, tbpl,
			NULL, NULL
		);

		/* Create new thumb pixmap and mask */
		pixmap = gdk_pixmap_new(
			w->window,
			twidth, theight,
			-1
		);
		if(pixmap != NULL)
			gdk_draw_rgb_32_image(
				pixmap, style->black_gc,
				0, 0, twidth, theight,
				dith,
				(guchar *)thumb_rgba,
				tbpl
			);
		mask = GUICreateBitmapFromDataRGBA(
			twidth, theight, tbpl,
			thumb_rgba,
			0x80,			/* Alpha byte threshold */
			NULL
		);

		g_free(thumb_rgba);
	}
	else
	{
		/* Create new thumb pixmap and mask */
		pixmap = gdk_pixmap_new(
			w->window,
			twidth, theight,
			-1
		);
		if(pixmap != NULL) 
			gdk_draw_rgb_32_image(
				pixmap, style->black_gc,
				0, 0, twidth, theight,
				dith,
				(guchar *)rgba,
				twidth * 4		/* Bytes per line */
			);
		mask = GUICreateBitmapFromDataRGBA(
			twidth, theight,
			twidth * 4,		/* Bytes per line */
			rgba,
			0x80,                   /* Alpha byte threshold */
			NULL
		);
	}

	/* Set the thumb's new pixmap and mask */
	thumb->pixmap = pixmap;
	thumb->mask = mask;

	TListQueueDraw(tlist);
}

/*
 *	Sets the thumb's sensitivity.
 */
void TListSetSensitive(
	tlist_struct *tlist,
	const gint thumb_num,
	const gboolean sensitive
)
{
	tlist_thumb_struct *thumb = TListGetThumb(tlist, thumb_num);
	if(thumb == NULL)
		return;

	if(sensitive && !(thumb->flags & TLIST_THUMB_SENSITIVE))
	{
		thumb->flags |= TLIST_THUMB_SENSITIVE;
		TListQueueDraw(tlist);
	}
	else if(!sensitive && (thumb->flags & TLIST_THUMB_SENSITIVE))
	{
		thumb->flags &= ~TLIST_THUMB_SENSITIVE;
		TListQueueDraw(tlist);
	}
}

/*
 *	Sets the thumb selectable.
 */
void TListSetSelectable(
	tlist_struct *tlist,
	const gint thumb_num,
	const gboolean selectable
)
{
	tlist_thumb_struct *thumb = TListGetThumb(tlist, thumb_num);
	if(thumb == NULL)
		return;

	if(selectable && !(thumb->flags & TLIST_THUMB_SELECTABLE))
	{
		thumb->flags |= TLIST_THUMB_SELECTABLE;
		TListQueueDraw(tlist);
	}
	else if(!selectable && (thumb->flags & TLIST_THUMB_SELECTABLE))
	{
		thumb->flags &= ~TLIST_THUMB_SELECTABLE;
		TListQueueDraw(tlist);
	}
}

/*
 *	Sets the thumb's attribute icon placement.
 */
void TListSetAttributeIconPlacement(
	tlist_struct *tlist,
	const int thumb_num,
	const GtkCornerType placement
)
{
	tlist_thumb_struct *thumb = TListGetThumb(tlist, thumb_num);
	if(thumb == NULL)
		return;

	if(thumb->attribute_icon_placement != placement)
	{
		thumb->attribute_icon_placement = placement;
		TListQueueDraw(tlist);
	}
}

/*
 *	Appends an attribute icon to the thumb.
 */
void TListAppendAttributeIcon(
	tlist_struct *tlist,
	const int thumb_num,
	GdkPixmap *pixmap,
	GdkBitmap *mask
)
{
	tlist_thumb_struct *thumb = TListGetThumb(tlist, thumb_num);
	if((thumb == NULL) || (pixmap == NULL))
		return;

	thumb->attribute_icon_pixmaps_list = g_list_append(
		thumb->attribute_icon_pixmaps_list,
		GDK_PIXMAP_REF(pixmap)
	);
	thumb->attribute_icon_masks_list = g_list_append(
		thumb->attribute_icon_masks_list,
		GDK_BITMAP_REF(mask)
	);
	TListQueueDraw(tlist);
}

/*
 *	Clears all the attribute icons on the thumb.
 */
void TListCleardAttributeIcons(
	tlist_struct *tlist,
	const int thumb_num
)
{
	tlist_thumb_struct *thumb = TListGetThumb(tlist, thumb_num);
	if(thumb == NULL)
		return;

	if(thumb->attribute_icon_pixmaps_list != NULL)
	{
		g_list_foreach(
			thumb->attribute_icon_pixmaps_list,
			(GFunc)GDK_PIXMAP_UNREF,
			NULL
		);
		g_list_free(thumb->attribute_icon_pixmaps_list);
		thumb->attribute_icon_pixmaps_list = NULL;

		g_list_foreach(
			thumb->attribute_icon_masks_list,
			(GFunc)GDK_BITMAP_UNREF,
			NULL
		);
		g_list_free(thumb->attribute_icon_masks_list);
		thumb->attribute_icon_masks_list = NULL;

		TListQueueDraw(tlist);
	}
}

/*
 *	Sets the Thumb's data.
 */
void TListSetThumbData(
	tlist_struct *tlist,
	const gint thumb_num,
	gpointer data   
)
{
	tlist_thumb_struct *thumb = TListGetThumb(tlist, thumb_num);
	if(thumb == NULL)
		return;

	thumb->data = data;
}

/*
 *	Sets the Thumb's data and destroy callback.
 *
 *	The Thumb's previous destroy callback will not be called.
 */
void TListSetThumbDataFull(
	tlist_struct *tlist,
	const gint thumb_num,
	gpointer data,
	GtkDestroyNotify destroy_cb
)
{
	tlist_thumb_struct *thumb = TListGetThumb(tlist, thumb_num);
	if(thumb == NULL)
		return;

	thumb->data = data;
	thumb->destroy_cb = destroy_cb;
}

/*
 *	Deletes the specified Thumb on the Thumbs List.
 */
void TListRemove(tlist_struct *tlist, const gint thumb_num)
{
	gint i;
	GList *glist;
	tlist_thumb_struct *thumb = TListGetThumb(tlist, thumb_num);
	if(thumb == NULL)
		return;

	/* Unselect this Thumb before removing it */
	TListDoUnselectThumb(
		tlist,
		thumb_num,
		NULL
	);

	/* Shift the selection indicies referencing any thumb index
	 * who's value is greater than this thumb since this thumb is
	 * about to be removed
	 */
	for(glist = tlist->selection;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		i = (gint)glist->data;
		if(i > thumb_num)
			glist->data = (gpointer)(i - 1);
	}

	/* Notify about this Thumb being removed and then remove it */
	TListThumbDelete(tlist, thumb);
	tlist->thumb[thumb_num] = thumb = NULL;

	/* Reduce the pointers on thumbs list and shift */
	tlist->total_thumbs--;
	if(tlist->total_thumbs > 0)
	{
		/* Shift the pointers */
		for(i = thumb_num; i < tlist->total_thumbs; i++)
			tlist->thumb[i] = tlist->thumb[i + 1];

		/* Reduce the pointer array allocation */
		tlist->thumb = (tlist_thumb_struct **)g_realloc(
			tlist->thumb,
			tlist->total_thumbs * sizeof(tlist_thumb_struct *)
		);
		if(tlist->thumb == NULL)
		{
			tlist->total_thumbs = 0;
		}
	}
	else
	{
		g_free(tlist->thumb);
		tlist->thumb = NULL;
		tlist->total_thumbs = 0;
	}

	/* Clip the focus thumb */
	if(tlist->focus_thumb > -1)
	{
		if(tlist->focus_thumb >= tlist->total_thumbs)
			tlist->focus_thumb = tlist->total_thumbs - 1;
		if(tlist->focus_thumb < 0)
			tlist->focus_thumb = 0;
	}

	/* Update the GtkScrollBar values due to this Thumb being
	 * removed
	 */
	TListResize(
		tlist,
		-1, -1				/* No size change */
	);
}

/*
 *	Deletes all the Thumbs on the Thumbs List.
 */
void TListClear(tlist_struct *tlist)
{
	gint i;
	tlist_thumb_struct *thumb;

	if(tlist == NULL)
		return;

	/* Unselect all the Thumbs? */
	if(tlist->selection != NULL)
	{
		/* Transfer selections list locally */
		GList	*selection = tlist->selection,
					*selection_end = tlist->selection_end;

		tlist->selection = NULL;
		tlist->selection_end = NULL;

		/* Notify about all the Thumbs being unselected */
		if(tlist->unselect_cb != NULL)
		{
			GList *glist;
			for(glist = selection_end;
				glist != NULL;
				glist = g_list_previous(glist)
			)
				tlist->unselect_cb(
					tlist,			/* Thumbs List */
					NULL,			/* No GtkEventButton */
					(gint)glist->data,	/* Thumb Index */
					tlist->unselect_data	/* Data */
				);
		}

		/* Delete the selection */
		g_list_free(selection);
	}

	/* Reset the focus and pointer over thumbs */
	tlist->focus_thumb = -1;
	tlist->pointer_over_thumb = -1;

	/* Delete all the Thumbs */
	for(i = tlist->total_thumbs - 1; i >= 0; i--)
	{
		thumb = tlist->thumb[i];
		if(thumb == NULL)
			continue;

		/* Notify about this Thumb being deleted and then delete it */
		TListThumbDelete(tlist, thumb);
		tlist->thumb[i] = thumb = NULL;
	}

	/* Delete the thumbs list */
	g_free(tlist->thumb);
	tlist->thumb = NULL;
	tlist->total_thumbs = 0;

	/* Update the GtkScrollBar values due to all the Thumbs being
	 * deleted
	 */
	TListResize(
		tlist,
		-1, -1				/* No size change */
	);
}


/*
 *	Gets the Thumb's pointer.
 *
 *	The thumb_num specifies the Thumb.
 *
 *	Returns the pointer to the Thumb or NULL on error.
 */
tlist_thumb_struct *TListGetThumb(
	tlist_struct *tlist,
	const gint thumb_num
)
{
	if(tlist == NULL)
		return(NULL);
	else if((thumb_num < 0) || (thumb_num >= tlist->total_thumbs))
		return(NULL);
	else
		return(tlist->thumb[thumb_num]);
}

/*
 *	Gets the Thumb's load state.
 *
 *	The thumb_num specifies the Thumb.
 *
 *	Returns the Thumb's load state.
 */
tlist_load_state TListGetLoadState(
	tlist_struct *tlist,
	const gint thumb_num
)
{
	tlist_thumb_struct *thumb = TListGetThumb(tlist, thumb_num);
	return((thumb != NULL) ? thumb->load_state : TLIST_LOAD_STATE_NOT_LOADED);
}

/*
 *	Gets the Thumb's text.
 *
 *	The thumb_num specifies the Thumb.
 *
 *	The text specifies the return value for the Thumb's text.
 *
 *	Returns TRUE if the Thumb is valid and a value was returned.
 */
gboolean TListGetText(
	tlist_struct *tlist,
	const gint thumb_num,
	const gchar **text
)
{
	tlist_thumb_struct *thumb = TListGetThumb(tlist, thumb_num);

	if(text != NULL)
		*text = NULL;

	if(thumb == NULL)
		return(FALSE);

	if(text != NULL)
		*text = thumb->text;

	return(TRUE);
}

/*
 *	Gets the Thumb's pixmap.
 *
 *	The thumb_num specifies the Thumb.
 *
 *	The pixmap and mask specifies the return values for the pixmap
 *	and mask.
 *
 *	Returns TRUE if the Thumb is valid and a value was returned.
 */
gboolean TListGetPixmap(
	tlist_struct *tlist,
	const gint thumb_num,
	GdkPixmap **pixmap, GdkBitmap **mask
)
{
	tlist_thumb_struct *thumb = TListGetThumb(tlist, thumb_num);

	if(pixmap != NULL)
		*pixmap = NULL;
	if(mask != NULL)
		*mask = NULL;

	if(thumb == NULL)
		return(FALSE);

	if(pixmap != NULL)
		*pixmap = thumb->pixmap;
	if(mask != NULL)
		*mask = thumb->mask;

	return(TRUE);
}

/*
 *	Gets the Thumb's data.
 *
 *	The thumb_num specifies the Thumb.
 *
 *	Returns the Thumb's data.
 */
gpointer TListGetThumbData(
	tlist_struct *tlist,
	const gint thumb_num
)
{
	tlist_thumb_struct *thumb = TListGetThumb(tlist, thumb_num);
	return((thumb != NULL) ? thumb->data : NULL);
}


/*
 *	Checks if the Thumb is selected.
 *
 *	The thumb_num specifies the Thumb to check.
 *
 *	Returns TRUE if the Thumb is selected.
 */
gboolean TListIsThumbSelected(
	tlist_struct *tlist,
	const gint thumb_num
)
{
	GList *glist;

	if((tlist == NULL) || (thumb_num < 0))
		return(FALSE);

	/* Check if the thumb's index is in the selections list */
	for(glist = tlist->selection;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		if((gint)glist->data == thumb_num)
			return(TRUE);
	}

	return(FALSE);
}

/*
 *	Selects the Thumb.
 *
 *	The thumb_num specifies the Thumb to select.
 *
 *	If the thumb is already selected or does not exist then nothing
 *	will be done.
 */
void TListSelectThumb(
	tlist_struct *tlist,
	const gint thumb_num
)
{
	TListDoSelectThumb(tlist, thumb_num, NULL, TRUE);
}

/*
 *      Unselects the thumb.
 *
 *	The thumb_num specifies the Thumb to unselect.
 *
 *	If the thumb is not selected or does not exist then nothing will
 *	be done.
 */
void TListUnselectThumb(
	tlist_struct *tlist,
	const gint thumb_num
)
{
	TListDoUnselectThumb(tlist, thumb_num, NULL);
}

/*
 *	Selects all the Thumbs.
 */
void TListSelectAll(tlist_struct *tlist)
{
	TListDoSelectAllThumbs(tlist);
}

/*
 *	Unselects all the Thumbs.
 */
void TListUnselectAll(tlist_struct *tlist)
{
	TListDoUnselectAllThumbs(tlist);
}


/*
 *	Gets the Thumb and its index position from coordinate
 *	position.
 *
 *	The x and y specifies the coordinate position relative to
 *	the Thumb List's GtkDrawingArea.
 *
 *	The thumb_num_rtn specifies the return value for the Thumb's
 *	index.
 *
 *	The thumb_ix_rtn and thumb_iy_rtn specifies the return value
 *	for the Thumb's coordinate position in units of thumb indicies.
 *
 *	Returns TRUE if a selection is made or FALSE on no match.
 */
gboolean TListGetSelection(
	tlist_struct *tlist,
	const gint x, const gint y,
	gint *thumb_num_rtn,
	gint *thumb_ix_rtn, gint *thumb_iy_rtn
)
{
	gint i, ix, iy, tpl, thumb_width, thumb_height;
	GtkAdjustment *hadj, *vadj;

	if(thumb_num_rtn != NULL)
		*thumb_num_rtn = -1;
	if(thumb_ix_rtn != NULL)
		*thumb_ix_rtn = 0;
	if(thumb_iy_rtn != NULL)
		*thumb_iy_rtn = 0;

	if(tlist == NULL)
		return(FALSE);

	tpl = MAX(tlist->thumbs_per_line, 1);
	thumb_width = tlist->thumb_width;
	thumb_height = tlist->thumb_height;
	hadj = tlist->hadjustment;
	vadj = tlist->vadjustment;
	if((hadj == NULL) || (vadj == NULL) ||
	   (thumb_width <= 0) || (thumb_height <= 0)
	)
		return(FALSE);

	/* Calculate thumb index coordinates */
	ix = (gint)(x + hadj->value) / thumb_width;
	iy = (gint)(y + vadj->value) / thumb_height;
	if(thumb_ix_rtn != NULL)
		*thumb_ix_rtn = ix;
	if(thumb_iy_rtn != NULL)
		*thumb_iy_rtn = iy;

	/* Calculate the thumb's index */
	if(tlist->orientation == GTK_ORIENTATION_HORIZONTAL)
	{
		if((y >= 0) && (y < (tpl * thumb_height)))
			i = (ix * tpl) + CLIP(iy, 0, tpl - 1);
		else
			i = -1;
	}
	else
	{
		if((x >= 0) && (x < (tpl * thumb_width)))
			i = (iy * tpl) + CLIP(ix, 0, tpl - 1);
		else
			i = -1;
	}

	/* Thumb index out of range? */
	if((i < 0) || (i >= tlist->total_thumbs))
		return(FALSE);

	if(thumb_num_rtn != NULL)
		*thumb_num_rtn = i;

	return(TRUE);
}

/*
 *	Gets the thumb's coordinate position relative to the Thumb
 *	List's GtkDrawingArea with scroll adjustments applied.
 *
 *	The thumb_num specifies the thumb.
 *
 *	The x_rtn and y_rtn specifies the return values for the
 *	coordinate position (which may or may not be currently visible).
 *
 *	Returns TRUE if thumb_num is valid and a coordinate position
 *	was obtained.
 */
gboolean TListGetThumbPosition(
	tlist_struct *tlist,
	const gint thumb_num,
	gint *x_rtn, gint *y_rtn
)
{
	gint lx, ly, tpl;
	GtkAdjustment *hadj, *vadj;

	if(x_rtn != NULL)
		*x_rtn = 0;
	if(y_rtn != NULL)
		*y_rtn = 0;

	if(tlist == NULL)
		return(FALSE);

	if((thumb_num < 0) || (thumb_num >= tlist->total_thumbs))
		return(FALSE);

	/* Get thumbs per line and scroll adjustments */
	tpl = MAX(tlist->thumbs_per_line, 1);
	hadj = tlist->hadjustment;
	vadj = tlist->vadjustment;
	if((hadj == NULL) || (vadj == NULL))
		return(FALSE);

	if((tlist->thumb_width <= 0) || (tlist->thumb_height <= 0))
		return(FALSE);

	if(tlist->orientation == GTK_ORIENTATION_HORIZONTAL)
	{
		lx = (gint)(thumb_num / tpl) * tlist->thumb_width;
		ly = (thumb_num % tpl) * tlist->thumb_height;
	}
	else
	{
		lx = (thumb_num % tpl) * tlist->thumb_width;
		ly = (gint)(thumb_num / tpl) * tlist->thumb_height;
	}
	if(x_rtn != NULL)
		*x_rtn = (gint)(lx - hadj->value);
	if(y_rtn != NULL)
		*y_rtn = (gint)(ly - vadj->value);

	return(TRUE);
}

/*
 *	Gets the thumb's pixmap coordinate position and size relative
 *	to the Thumb List's GtkDrawingArea with scroll adjustments
 *	applied.
 *
 *	The thumb_num specifies the thumb.
 *
 *	The x_rtn and y_rtn specifies the return values for the
 *	coordinate position (which may or may not be currently visible).
 *
 *	The width_rtn and height_rtn specifies the reutrn values for
 *	the size.
 *
 *	Returns TRUE if thumb_num is valid and a coordinate position
 *	and size was obtained.
 */
gboolean TListGetThumbPixmapGeometry(
	tlist_struct *tlist,
	const gint thumb_num,
	gint *x_rtn, gint *y_rtn,
	gint *width_rtn, gint *height_rtn
)
{
	gint pm_width, pm_height, font_height;
	GdkFont *font;
	GdkPixmap *pixmap;
	GtkStyle *style;
	GtkWidget *w;
	tlist_flags flags;
	tlist_thumb_struct *thumb = TListGetThumb(tlist, thumb_num);

	if(x_rtn != NULL)
		*x_rtn = 0;
	if(y_rtn != NULL)
		*y_rtn = 0;
	if(width_rtn != NULL)
		*width_rtn = 0;
	if(height_rtn != NULL)
		*height_rtn = 0;

	if(thumb == NULL)
		return(FALSE);

	if(!TListGetThumbPosition(tlist, thumb_num, x_rtn, y_rtn))
		return(FALSE);

	flags = tlist->flags;
	w = tlist->list_da;
	if(w == NULL)
		return(TRUE);

	style = gtk_widget_get_style(w);
	if(style == NULL)
		return(TRUE);
	font = style->font;
	font_height = (font != NULL) ? (font->ascent + font->descent) : 0;

	pixmap = thumb->pixmap;
	if(pixmap != NULL)
	{
		gdk_window_get_size(pixmap, &pm_width, &pm_height);
	}
	else
	{
		pm_width = 0;
		pm_height = 0;
	}
	if(width_rtn != NULL)
		*width_rtn = pm_width;
	if(height_rtn != NULL)
		*height_rtn = pm_height;

	if(flags & TLIST_SHOW_THUMB_LABELS)
	{
		const gint	thumb_width = tlist->thumb_width,
					thumb_height = tlist->thumb_height,
					thumb_border = tlist->thumb_border,
					label_height = font_height +
							(2 * TLIST_THUMB_DEF_FRAME_HEIGHT),
					thumb_pixmap_height_max = thumb_height -
							((flags & TLIST_SHOW_THUMB_FRAMES) ?
								(2 * TLIST_THUMB_DEF_FRAME_HEIGHT) : 2) -
							(3 * thumb_border) - label_height;
		if(flags & TLIST_SHOW_THUMB_FRAMES)
		{
			if(x_rtn != NULL)
				*x_rtn = (*x_rtn) + ((thumb_width - pm_width) / 2);
			if(y_rtn != NULL)
				*y_rtn = (*y_rtn) + TLIST_THUMB_DEF_FRAME_HEIGHT + thumb_border +
					((thumb_pixmap_height_max - pm_height) / 2);
		}
		else
		{
			const gint pixmap_and_label_height = pm_height +
				((pm_height > 0) ? thumb_border : 0) + label_height;
			if(x_rtn != NULL)
				*x_rtn = (*x_rtn) + ((thumb_width - pm_width) / 2);
			if(y_rtn != NULL)
				*y_rtn = (*y_rtn) +
					((thumb_height - pixmap_and_label_height) / 2);
		}
	}
	else
	{
		const gint  thumb_width = tlist->thumb_width,
					thumb_height = tlist->thumb_height;
		if(flags & TLIST_SHOW_THUMB_FRAMES)
		{
			if(x_rtn != NULL)
				*x_rtn = (*x_rtn) + ((thumb_width - pm_width) / 2);
			if(y_rtn != NULL)
				*y_rtn = (*y_rtn) + ((thumb_height - pm_height) / 2);
		}
		else
		{
			if(x_rtn != NULL)
				*x_rtn = (*x_rtn) + ((thumb_width - pm_width) / 2);
			if(y_rtn != NULL)
				*y_rtn = (*y_rtn) + ((thumb_height - pm_height) / 2);
		}
	}

	return(TRUE);
}

/*
 *	Gets the thumb's label coordinate position and size relative
 *	to the Thumb List's GtkDrawingArea with scroll adjustments
 *	applied.
 *
 *	The thumb_num specifies the thumb.
 *
 *	The x_rtn and y_rtn specifies the return values for the
 *	coordinate position (which may or may not be currently visible).
 *
 *	The width_rtn and height_rtn specifies the reutrn values for
 *	the size.
 *
 *	Returns TRUE if thumb_num is valid and a coordinate position
 *	and size was obtained.
 */
gboolean TListGetThumbLabelGeometry(
	tlist_struct *tlist,
	const gint thumb_num, 
	gint *x_rtn, gint *y_rtn,
	gint *width_rtn, gint *height_rtn
)
{
	gint pm_width, pm_height, font_height;
	GdkFont *font;
	GdkPixmap *pixmap;
	GtkStyle *style;
	GtkWidget *w;
	tlist_flags flags;
	tlist_thumb_struct *thumb = TListGetThumb(tlist, thumb_num);

	if(x_rtn != NULL)
		*x_rtn = 0;
	if(y_rtn != NULL)
		*y_rtn = 0;
	if(width_rtn != NULL)
		*width_rtn = 0;
	if(height_rtn != NULL)
		*height_rtn = 0;

	if(thumb == NULL)
		return(FALSE);

	if(!TListGetThumbPosition(tlist, thumb_num, x_rtn, y_rtn))
		return(FALSE);

	flags = tlist->flags;
	w = tlist->list_da;
	if(w == NULL)
		return(TRUE);

	style = gtk_widget_get_style(w);
	if(style == NULL)
		return(TRUE);
	font = style->font;
	font_height = (font != NULL) ? (font->ascent + font->descent) : 0;

	pixmap = thumb->pixmap;
	if(pixmap != NULL)
	{
		gdk_window_get_size(pixmap, &pm_width, &pm_height);
	}
	else
	{
		pm_width = 0;
		pm_height = 0;
	}

	if(flags & TLIST_SHOW_THUMB_LABELS)
	{
		const gint  thumb_width = tlist->thumb_width,
					thumb_height = tlist->thumb_height,
					thumb_border = tlist->thumb_border,
					label_height = font_height +
							(2 * TLIST_THUMB_DEF_FRAME_HEIGHT);
		if(flags & TLIST_SHOW_THUMB_FRAMES)
		{
			if(x_rtn != NULL)
				*x_rtn = (*x_rtn) + TLIST_THUMB_DEF_FRAME_WIDTH + thumb_border;
			if(y_rtn != NULL)
				*y_rtn = (*y_rtn) + thumb_height - TLIST_THUMB_DEF_FRAME_HEIGHT -
					thumb_border - label_height;
			if(width_rtn != NULL)  
				*width_rtn = thumb_width - (2 * TLIST_THUMB_DEF_FRAME_WIDTH) -
					(2 * thumb_border);
			if(height_rtn != NULL)
				*height_rtn = label_height;
		}
		else
		{   
			const gint pixmap_and_label_height = pm_height +
				((pm_height > 0) ? thumb_border : 0) + label_height;
			if(x_rtn != NULL)
				*x_rtn = (*x_rtn) + 1 + thumb_border;
			if(y_rtn != NULL)
				*y_rtn = (*y_rtn) +
					((thumb_height - pixmap_and_label_height) / 2) +
					pm_height + ((pm_height > 0) ? thumb_border : 0);
			if(width_rtn != NULL)  
				*width_rtn = thumb_width - 2 - (2 * thumb_border);
			if(height_rtn != NULL)
				*height_rtn = label_height;
		}
	}
	else
	{   
		const gint  thumb_width = tlist->thumb_width,
					thumb_height = tlist->thumb_height,
					thumb_border = tlist->thumb_border,
					label_height = font_height +
							(2 * TLIST_THUMB_DEF_FRAME_HEIGHT);
		if(flags & TLIST_SHOW_THUMB_FRAMES)
		{
			if(x_rtn != NULL)
				*x_rtn = (*x_rtn) + TLIST_THUMB_DEF_FRAME_WIDTH + thumb_border;
			if(y_rtn != NULL)
				*y_rtn = (*y_rtn) + ((thumb_height - label_height) / 2);
			if(width_rtn != NULL)
				*width_rtn = thumb_width - (2 * TLIST_THUMB_DEF_FRAME_WIDTH) -
					(2 * thumb_border);
			if(height_rtn != NULL)
				*height_rtn = label_height;
		}
		else
		{
			if(x_rtn != NULL)
				*x_rtn = (*x_rtn) + 1 + thumb_border;
			if(y_rtn != NULL)
				*y_rtn = (*y_rtn) + ((thumb_height - label_height) / 2);
			if(width_rtn != NULL)
				*width_rtn = thumb_width - 2 - (2 * thumb_border);
			if(height_rtn != NULL)
				*height_rtn = label_height;
		}
	}    

	return(TRUE);
}


/*
 *	Gets the thumb's visibility.
 */
GtkVisibility TListIsThumbVisible(
	tlist_struct *tlist,
	const gint thumb_num
)
{
	gint x, y, tpl;
	GtkAdjustment *hadj, *vadj;

	if(tlist == NULL)
		return(GTK_VISIBILITY_NONE);

	/* Non-existant thumb index? */
	if((thumb_num < 0) || (thumb_num >= tlist->total_thumbs))
		return(GTK_VISIBILITY_NONE);

	/* Get thumbs per line and scroll adjustments */
	tpl = MAX(tlist->thumbs_per_line, 1);
	hadj = tlist->hadjustment;
	vadj = tlist->vadjustment;
	if((hadj == NULL) || (vadj == NULL))
		return(GTK_VISIBILITY_NONE);

	if((tlist->thumb_width <= 0) || (tlist->thumb_height <= 0))
		return(GTK_VISIBILITY_NONE);

	/* Calculate coordinates of thumb (without scroll adjustments
	 * applied)
	 */
	if(tlist->orientation == GTK_ORIENTATION_HORIZONTAL)
	{
		x = (gint)(thumb_num / tpl) * tlist->thumb_width;
		y = (thumb_num % tpl) * tlist->thumb_height;
	}
	else
	{
		x = (thumb_num % tpl) * tlist->thumb_width;
		y = (gint)(thumb_num / tpl) * tlist->thumb_height;
	}

	/* Check if the thumb coordinate position visiblity by masking
	 * the absolute coordinate values of the adjustments to them
	 */

	/* Totally obscured? */
	if(((gint)(x + tlist->thumb_width) < (gint)hadj->value) ||
	   ((gint)x >= (gint)(hadj->value + hadj->page_size)) ||
	   ((gint)(y + tlist->thumb_height) < (gint)vadj->value) ||
	   ((gint)y >= (gint)(vadj->value + vadj->page_size))
	)
		return(GTK_VISIBILITY_NONE);
	/* Fully visible? */
	else if(((gint)x >= (gint)hadj->value) &&
			((gint)(x + tlist->thumb_width) <=
				(gint)(hadj->value + hadj->page_size)) &&
			((gint)y >= (gint)vadj->value) &&
			((gint)(y + tlist->thumb_height) <=
				(gint)(vadj->value + vadj->page_size))
	)
		return(GTK_VISIBILITY_FULL);
	else
		return(GTK_VISIBILITY_PARTIAL);
}

/* 
 *	Scrolls the Thumb List to the thumb's position with coefficient
 *	offset applied.
 *
 *	The thumb_num specifies the thumb.
 *
 *	The coeff specifies the coefficient offset from 0.0 to 1.0.
 */
void TListMoveTo(
	tlist_struct *tlist,
	const gint thumb_num,
	const gfloat coeff
)
{
	gint tpl;
	gfloat _coeff = coeff;
	GtkAdjustment *hadj, *vadj;

	if(tlist == NULL)
		return;

	/* Non-existant thumb index? */
	if((thumb_num < 0) || (thumb_num >= tlist->total_thumbs))
		return;

	/* Sanitize coefficient */
	if(_coeff < 0.0f)
		_coeff = 0.0f;
	if(_coeff > 1.0f)
		_coeff = 1.0f;

	/* Get thumbs per line and scroll adjustments */
	tpl = MAX(tlist->thumbs_per_line, 1);
	hadj = tlist->hadjustment;
	vadj = tlist->vadjustment;
	if((hadj == NULL) || (vadj == NULL))
		return;

	/* Calculate the position of the thumb, apply the coefficient
	 * offset, and scroll to the new position
	 */
	if(tlist->orientation == GTK_ORIENTATION_HORIZONTAL)
	{
		/* Calculate the position of the thumb */
		gint x = (gint)(thumb_num / tpl) * tlist->thumb_width;

		/* Apply the coefficient offset */
		x = x - (gint)(_coeff *
			(hadj->page_size - (gfloat)tlist->thumb_width)
		);

		/* Clip the x position */
		if((gfloat)x > (hadj->upper - hadj->page_size))
			x = (gint)(hadj->upper - hadj->page_size);
		if((gfloat)x < hadj->lower)
			x = (gint)hadj->lower;

		gtk_adjustment_set_value(hadj, (gfloat)x);
	}
	else
	{
		/* Calculate the position of the thumb */
		gint y = (gint)(thumb_num / tpl) * tlist->thumb_height;

		/* Apply the coefficient offset */
		y = y - (gint)(_coeff *
			(vadj->page_size - (gfloat)tlist->thumb_height)
		);

		/* Clip the y position */
		if((gfloat)y > (vadj->upper - vadj->page_size))
			y = (gint)(vadj->upper - vadj->page_size);
		if((gfloat)y < vadj->lower)
			y = (gint)vadj->lower;

		gtk_adjustment_set_value(vadj, (gfloat)y);
	}
}

/* 
 *	Queues the Thumb List to scroll to the thumb's position with
 *	the coefficient offset applied.
 *
 *	The thumb_num specifies the thumb.
 *
 *	The coeff specifies the coefficient offset from 0.0 to 1.0.
 */
void TListQueueMoveTo(
	tlist_struct *tlist,
	const gint thumb_num,
	const gfloat coeff
)
{
	if(tlist == NULL)
		return;

	tlist->moveto_thumb = thumb_num;
	tlist->moveto_coeff = coeff;

	if(tlist->moveto_idleid == 0)
		tlist->moveto_idleid = gtk_idle_add_priority(
			GTK_PRIORITY_RESIZE,
			TListMoveToIdleCB, tlist
		);
}


/*
 *	Creates a new Thumbs List.
 */
tlist_struct *TListNew(
	const GtkOrientation orientation,
	const gint thumb_width, const gint thumb_height,
	const gint thumb_border,
	void (*select_cb)(tlist_struct *, GdkEventButton *, gint, gpointer),
	gpointer select_data,
	void (*unselect_cb)(tlist_struct *, GdkEventButton *, gint, gpointer),
	gpointer unselect_data
)
{
	const gint border_minor = 3;
	GtkAdjustment *adj;
	GtkWidget *w, *parent, *parent2;

	/* Create a new Thumbs List */
	tlist_struct *tlist = TLIST(g_malloc0(sizeof(tlist_struct)));
	if(tlist == NULL)
		return(tlist);

#if 0
	tlist->freeze_count = 0;
	tlist->colormap = NULL;
	tlist->gc = NULL;
#endif

	tlist->flags = TLIST_SHOW_THUMB_FRAMES |
		TLIST_SHOW_THUMB_LABELS;
	tlist->orientation = orientation;

#if 0
	tlist->transparent_stipple_bm = NULL;
#endif
	tlist->translate_cur = gdk_cursor_new(GDK_FLEUR);

#if 0
	tlist->list_pm = NULL;

	tlist->vadjustment = NULL;
	tlist->hadjustment = NULL;

	tlist->drag_last_x = 0;
	tlist->drag_last_y = 0;

	tlist->thumb = NULL;
	tlist->total_thumbs = 0;
#endif

	tlist->selection_mode = GTK_SELECTION_SINGLE;

#if 0
	tlist->selection = NULL;
	tlist->selection_end = NULL;
#endif

	tlist->focus_thumb = -1;
	tlist->pointer_over_thumb = -1;

	tlist->thumb_width = thumb_width;
	tlist->thumb_height = thumb_height;
	tlist->thumb_border = thumb_border;
	tlist->thumbs_per_line = 1;

#if 0
	tlist->moveto_thumb = 0;
	tlist->moveto_coeff = 0.0f;
	tlist->moveto_idleid = 0;
#endif

	tlist->select_cb = select_cb;
	tlist->select_data = select_data;
	tlist->unselect_cb = unselect_cb;
	tlist->unselect_data = unselect_data;

	tlist->freeze_count++;

	/* Toplevel GtkTable */
	tlist->toplevel = w = gtk_table_new(2, 2, FALSE);
	gtk_table_set_row_spacings(GTK_TABLE(w), (guint)border_minor);
	gtk_table_set_col_spacings(GTK_TABLE(w), (guint)border_minor);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	parent = w;


	/* List GtkFrame */
	w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_table_attach(
		GTK_TABLE(parent), w,
		0, 1,
		0, 1,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND,
		0, 0
	);
	gtk_widget_show(w);
	parent2 = w;

	/* List GtkDrawingArea */
	tlist->list_da = w = gtk_drawing_area_new();
	gtk_widget_set_name(w, TLIST_WIDGET_NAME);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT | GTK_CAN_FOCUS);
	gtk_widget_add_events(
		w,
		GDK_VISIBILITY_NOTIFY_MASK | GDK_FOCUS_CHANGE_MASK |
		GDK_EXPOSURE_MASK | GDK_STRUCTURE_MASK |
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "realize",
		GTK_SIGNAL_FUNC(TListRealizeCB), tlist
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "configure_event",
		GTK_SIGNAL_FUNC(TListConfigureEventCB), tlist
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "expose_event",
		GTK_SIGNAL_FUNC(TListExposeEventCB), tlist
	);                                            
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(TListKeyEventCB), tlist
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(TListKeyEventCB), tlist
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(TListButtonEventCB), tlist
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(TListButtonEventCB), tlist
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "motion_notify_event",
		GTK_SIGNAL_FUNC(TListMotionEventCB), tlist
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(TListCrossingEventCB), tlist
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(TListCrossingEventCB), tlist
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "focus_in_event",
		GTK_SIGNAL_FUNC(TListFocusEventCB), tlist
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "focus_out_event",
		GTK_SIGNAL_FUNC(TListFocusEventCB), tlist
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "drag_begin",
		GTK_SIGNAL_FUNC(TListDragBeginCB), tlist
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "drag_end",
		GTK_SIGNAL_FUNC(TListDragEndCB), tlist
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "drag_leave",
		GTK_SIGNAL_FUNC(TListDragLeaveCB), tlist
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "drag_motion",
		GTK_SIGNAL_FUNC(TListDragMotionCB), tlist
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "drag_drop",
		GTK_SIGNAL_FUNC(TListDragDropCB), tlist
	);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);


	/* Vertical GtkAdjustment and GtkScrollBar */
	adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0f,
		0.0f, 10.0f,
		10.0f, 10.0f, 10.0f
	);
	tlist->vadjustment = adj = (GtkAdjustment *)GTK_OBJECT_REF(GTK_OBJECT(adj));
	tlist->vsb = w = gtk_vscrollbar_new(adj);
	gtk_table_attach(
		GTK_TABLE(parent), w,
		1, 2,
		0, 1,
		GTK_FILL,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		0, 0
	);
	gtk_signal_connect(
		GTK_OBJECT(adj), "value_changed",
		GTK_SIGNAL_FUNC(TListAdjustmentValueChangedCB), tlist
	);
	/* Do not show this now, it will be mapped/unmapped when resizing
	 * of list drawing area occures
	 */

	/* Horizontal GtkAdjustment and GtkScrollBar */
	adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0f,
		0.0f, 10.0f,
		10.0f, 10.0f, 10.0f
	);
	tlist->hadjustment = adj = (GtkAdjustment *)GTK_OBJECT_REF(GTK_OBJECT(adj));
	tlist->hsb = w = gtk_hscrollbar_new(adj);
	gtk_table_attach(
		GTK_TABLE(parent), w,
		0, 1,
		1, 2,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		GTK_FILL,
		0, 0
	);
	gtk_signal_connect(
		GTK_OBJECT(adj), "value_changed",
		GTK_SIGNAL_FUNC(TListAdjustmentValueChangedCB), tlist
	);
	/* Do not show the GtkScrollBars now, they will be mapped or
	 * unmapped when the GDK_CONFIGURE event is received by the
	 * list GtkDrawingArea
	 */

	tlist->freeze_count--;

	return(tlist);
}

/*
 *	Realizes the Thumbs List.
 */
void TListRealize(tlist_struct *tlist)
{
	if(tlist == NULL)
		return;

	/* Already realized? */
	if(tlist->flags & TLIST_REALIZED)
		return;

	/* Realize the Thumbs List */
	gtk_widget_realize(tlist->list_da);
}

/*
 *	Gets the Thumb List's toplevel GtkWidget.
 */
GtkWidget *TListGetToplevelWidget(tlist_struct *tlist)
{
	if(tlist == NULL)
		return(NULL);

	return(tlist->toplevel);
}

/*
 *	Gets the Thumb List's list GtkDrawingArea.
 */
GtkWidget *TListGetListWidget(tlist_struct *tlist)
{
	if(tlist == NULL)
		return(NULL);

	return(tlist->list_da);
}

/*
 *	Sets the Thumb List's thumb geometry.
 */
void TListThumbGeometry(
	tlist_struct *tlist, 
	const gint thumb_width, const gint thumb_height,
	const gint thumb_border
)
{
	if(tlist == NULL)  
		return;

	if((tlist->thumb_width != thumb_width) ||
	   (tlist->thumb_height != thumb_height) ||
	   (tlist->thumb_border != thumb_border)
	)
	{
		tlist->thumb_width = thumb_width;
		tlist->thumb_height = thumb_height;
		tlist->thumb_border = thumb_border;

		TListQueueDraw(tlist);
	}
}

/*
 *	Sets the Thumbs List selection mode.
 */
void TListSelectionMode(
	tlist_struct *tlist,
	const GtkSelectionMode selection_mode
)
{
	if(tlist == NULL)
		return;

	tlist->selection_mode = selection_mode;
}

/*
 *	Sets the Thumbs List in double buffer or single buffer mode.
 */
void TListDoubleBuffer(
	tlist_struct *tlist,
	const gboolean double_buffer
)
{
	if(tlist == NULL)
		return;

	if(double_buffer && !(tlist->flags & TLIST_DOUBLE_BUFFER))
	{
		tlist->flags |= TLIST_DOUBLE_BUFFER;

		/* Recreate the back buffer GdkPixmap */
		TListResize(
			tlist,
			-1, -1				/* No size change */
		);
	}
	else if(!double_buffer && (tlist->flags & TLIST_DOUBLE_BUFFER))
	{
		tlist->flags &= ~TLIST_DOUBLE_BUFFER;

		/* Unref the back buffer GdkPixmap */
		tlist->list_pm = GDK_PIXMAP_UNREF(tlist->list_pm);
	}
}

/*
 *	Sets the Thumbs List orientation.
 */
void TListOrientation(
	tlist_struct *tlist,
	const GtkOrientation orientation
)
{
	if(tlist == NULL)
		return;

	if(tlist->orientation != orientation)
	{
		GtkAdjustment	*hadj = tlist->hadjustment,
							*vadj = tlist->vadjustment;

		tlist->orientation = orientation;

		/* Swap the scroll adjustment values */
		if((hadj != NULL) && (vadj != NULL))
		{
			gfloat v = hadj->value;
			hadj->value = vadj->value;
			vadj->value = v;
		}

		/* Update the GtkScrollBar values and map states due to the
		 * change in orientation
		 */
		TListResize(
			tlist,
			-1, -1				/* No size change */
		);
	}
}

/*
 *	Sets the Thumbs List to show or hide thumb frames.
 */
void TListShowThumbFrames(
	tlist_struct *tlist,
	const gboolean show
)
{
	gboolean was_show;

	if(tlist == NULL)
		return;

	was_show = (tlist->flags & TLIST_SHOW_THUMB_FRAMES) ? TRUE : FALSE;

	if(was_show != show)
	{
		if(show)
			tlist->flags |= TLIST_SHOW_THUMB_FRAMES;
		else
			tlist->flags &= ~TLIST_SHOW_THUMB_FRAMES;

		TListQueueDraw(tlist);
	}
}

/*
 *	Sets the Thumbs List to show or hide thumb labels.
 */
void TListShowThumbLabels(
	tlist_struct *tlist,
	const gboolean show
)
{
	gboolean was_show;

	if(tlist == NULL)
		return;
			   
	was_show = (tlist->flags & TLIST_SHOW_THUMB_LABELS) ? TRUE : FALSE;

	if(was_show != show)
	{
		if(show)
			tlist->flags |= TLIST_SHOW_THUMB_LABELS;
		else
			tlist->flags &= ~TLIST_SHOW_THUMB_LABELS;

		TListQueueDraw(tlist);
	}
}

/*
 *	Sets the Thumbs List to show or not show texttips for obscured
 *	thumb labels.
 */
void TListShowTextTips(
	tlist_struct *tlist,
	const gboolean show
)
{
	gboolean was_show;

	if(tlist == NULL)
		return;
			   
	was_show = (tlist->flags & TLIST_SHOW_TEXTTIPS) ? TRUE : FALSE;

	if(was_show != show)
	{
		if(show)
			tlist->flags |= TLIST_SHOW_TEXTTIPS;
		else                                      
			tlist->flags &= ~TLIST_SHOW_TEXTTIPS;

		TListQueueDraw(tlist);
	}
}

/*
 *	Enable or disable middle click list drag scroll.
 */
void TListEnableListDragScroll(
	tlist_struct *tlist,
	const gboolean enable
)
{
	if(tlist == NULL)
		return;

	if(enable)
	{
		tlist->flags |= TLIST_ENABLE_LIST_DRAG_SCROLL;
	}
	else
	{
		tlist->flags &= ~TLIST_ENABLE_LIST_DRAG_SCROLL;
	}
}

/*
 *	Maps the Thumbs List.
 */
void TListMap(tlist_struct *tlist)
{
	if(tlist == NULL)
		return;

	gtk_widget_show(tlist->toplevel);
	tlist->flags |= TLIST_MAPPED;
}

/*
 *	Unmaps the Thumbs List.
 */
void TListUnmap(tlist_struct *tlist)
{
	if(tlist == NULL)
		return;

	gtk_widget_hide(tlist->toplevel);
	tlist->flags &= ~TLIST_MAPPED;
}

/*
 *	Grabs focus.
 */
void TListGrabFocus(tlist_struct *tlist)
{
	if(tlist == NULL)
		return;

	gtk_widget_grab_focus(tlist->list_da);
}

/*
 *	Delete the Thumbs List.
 */
void TListDelete(tlist_struct *tlist)
{
	if(tlist == NULL)
		return;

	/* Remove all the timeout and idle callbacks */
	tlist->moveto_idleid = GTK_IDLE_REMOVE(tlist->moveto_idleid);

	/* Unselect and delete all the thumbs */
	TListClear(tlist);

	TListUnmap(tlist);

	tlist->freeze_count++;

	/* Destroy all the GtkWidgets */
	gtk_widget_destroy(tlist->toplevel);

	/* Unref the scroll GtkAdjustments */
	(void)GTK_OBJECT_UNREF(GTK_OBJECT(tlist->vadjustment));
	(void)GTK_OBJECT_UNREF(GTK_OBJECT(tlist->hadjustment));

	/* Unref the drawing resources */
	(void)GDK_GC_UNREF(tlist->gc);
	(void)GDK_COLORMAP_UNREF(tlist->colormap);

	/* Unref the back buffer GdkPixmap */
	(void)GDK_PIXMAP_UNREF(tlist->list_pm);

	(void)GDK_BITMAP_UNREF(tlist->transparent_stipple_bm);
	GDK_CURSOR_DESTROY(tlist->translate_cur);

	tlist->freeze_count--;

	g_free(tlist);
}
