#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <gtk/gtk.h>

#include "guiutils.h"
#include "pulist.h"
#include "time_prompt.h"

#include "images/icon_time_stamp_20x20.xpm"

typedef struct _TimePrompt		TimePrompt;
#define TIME_PROMPT(p)			((TimePrompt *)(p))
#define TIME_PROMPT_KEY			"/TimePrompt"


/*
 *	Time Prompt Flags:
 */
typedef enum {
	TIME_PROMPT_ALLOW_SET_CURRENT_TIME	= (1 << 0)
} TimePromptFlags;


/* Callbacks */
static void time_prompt_destroy_cb(gpointer data);
static void time_prompt_changed_cb(GtkWidget *widget, gpointer data);
static void time_prompt_popup_list_box_changed_cb(
	GtkWidget *widget, const gint i, gpointer data
);
static void time_prompt_current_time_cb(GtkWidget *widget, gpointer data);

/* Time Prompt */
static TimePrompt *time_prompt_get_widget_data(
	GtkWidget *w,
	const gchar *func_name
);
GtkWidget *time_prompt_new(
	const GtkOrientation orientation,
	const gchar *label
);
gulong time_prompt_get(GtkWidget *w);
void time_prompt_set(
	GtkWidget *w,
	const gulong t
);
void time_prompt_set_changed_cb(
	GtkWidget *w,
	void (*func)(GtkWidget *, gpointer),
	gpointer data
);
void time_prompt_allow_set_current_time(
	GtkWidget *w,
	const gboolean allow
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Time Prompt:
 */
struct _TimePrompt {

	GtkWidget	*toplevel;		/* GtkBox */
	gint		freeze_count;
	TimePromptFlags	flags;

	GtkWidget	*year_spin,		/* GtkSpinButton */
			*month_pulistbox,	/* PopupListBox */
			*day_spin,		/* GtkSpinButton */
			*hours_spin,		/* GtkSpinButton */
			*minutes_spin,		/* GtkSpinButton */
			*seconds_spin,		/* GtkSpinButton */
			*current_time_button;	/* GtkButton */

	void		(*changed_cb)(
			GtkWidget *,
			gpointer
	);
	gpointer	changed_data;
};


/*
 *	Toplevel "destroy" signal callback.
 */
static void time_prompt_destroy_cb(gpointer data) 
{
	TimePrompt *p = TIME_PROMPT(data);
	if(p == NULL)
		return;

	p->freeze_count++;

	p->freeze_count--;

	g_free(p);
}

/*
 *	Any GtkWidget "changed" signal callback.
 */
static void time_prompt_changed_cb(GtkWidget *widget, gpointer data)
{
	TimePrompt *p = TIME_PROMPT(data);
	if(p == NULL)
		return;

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	/* Notify about the change */
	if(p->changed_cb != NULL)
		p->changed_cb(
			p->toplevel,
			p->changed_data
		);

	p->freeze_count--;
}

/*
 *	PopupListBox "changed" signal callback.
 */
static void time_prompt_popup_list_box_changed_cb(
	GtkWidget *widget, const gint i, gpointer data
)
{
	time_prompt_changed_cb(widget, data);
}

/*
 *	Current time callback.
 */
static void time_prompt_current_time_cb(GtkWidget *widget, gpointer data)
{
	TimePrompt *p = TIME_PROMPT(data);
	if(p == NULL)
		return;

	if(p->freeze_count > 0)
		return;

	if(!(p->flags & TIME_PROMPT_ALLOW_SET_CURRENT_TIME))
		return;

	p->freeze_count++;

	time_prompt_set(
		p->toplevel,
		(gulong)time(NULL)
	);

	p->freeze_count--;
}

/*
 *	Gets the TimePrompt data from the GtkWidget.
 */
static TimePrompt *time_prompt_get_widget_data(
	GtkWidget *w,
	const gchar *func_name
)
{
	const gchar *key = TIME_PROMPT_KEY;
	TimePrompt *p;

	if(w == NULL)
		return(NULL);

	p = TIME_PROMPT(gtk_object_get_data(
		GTK_OBJECT(w),
		key
	));
	if(p == NULL)
	{
		g_printerr(
"%s(): Warning: GtkWidget %p:\
 Unable to find the data that matches the key \"%s\".\n",
			func_name,
			w,
			key
		);
		return(NULL);
	}

	return(p);
}

/*
 *	Creates a new TimePrompt.
 */
GtkWidget *time_prompt_new(
	const GtkOrientation orientation,
	const gchar *label
)
{
	const gint      border_major = 5,
					border_minor = 2;
	gchar *s;
	GtkAdjustment *adj;
	GtkWidget       *w,
					*parent, *parent2, *parent3, *parent4,
					*pulist;

	TimePrompt *p = (TimePrompt *)g_malloc0(
		sizeof(TimePrompt)
	);
	if(p == NULL)
		return(NULL);

	p->freeze_count++;

	/* Toplevel GtkBox */
	if(orientation == GTK_ORIENTATION_VERTICAL)
		w = gtk_vbox_new(FALSE, border_major);
	else
		w = gtk_hbox_new(FALSE, border_major);
	p->toplevel = w;
	gtk_object_set_data_full(
		GTK_OBJECT(w), TIME_PROMPT_KEY,
		p, time_prompt_destroy_cb
	);
	parent = w;

	/* Year/Month/Day GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Year GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Prefix Label GtkAlignment */
	w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);
	gtk_widget_set_usize(w, 80, -1);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Prefix GtkLabel */
	if(label != NULL)
		s = g_strconcat(
			label,
			" Date:",
			NULL
		);
	else
		s = g_strdup("Date:");
	w = gtk_label_new(s);
	g_free(s);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	/* Year GtkSpinButton */
	adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0f, 0.0f, 1000000.0f,
		1.0f, 5.0f, 5.0f
	);
	p->year_spin = w = gtk_spin_button_new(adj, 1.0, 0);
	gtk_widget_set_usize(w, 60, -1);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(time_prompt_changed_cb), p
	);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(w, "Year");
	gtk_widget_show(w);

	/* Month GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Month GtkLabel */
	w = gtk_label_new("/");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Month Popup List Box */
	p->month_pulistbox = w = PUListBoxNew(
		70, -1
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	pulist = PUListBoxGetPUList(w);
	if(pulist != NULL)
	{
		struct tm v;
		gint i, n = -1;
		gchar buf[40];

		/* Generate the month names using strftime() and add each
		 * one to the months PopupListBox
		 */
		v.tm_sec = 0;
		v.tm_min = 0;
		v.tm_hour = 0;
		v.tm_mday = 1;
		v.tm_mon = 0;
		v.tm_year = 1900;
		v.tm_wday = 0;
		v.tm_yday = 0;
		v.tm_isdst = -1;

		for(i = 0; i < 12; i++)
		{
			v.tm_mon = (int)i;
			if(strftime((char *)buf, sizeof(buf), "%b", &v) > 0)
				n = PUListAddItem(pulist, buf);
		}

		if(n > -1)
			PUListBoxSetLinesVisible(w, n + 1);
	}
	PUListBoxSetChangedCB(
		w,
		time_prompt_popup_list_box_changed_cb, p
	);
	PUListBoxSetTip(w, "Month");
	gtk_widget_show(w);

	/* Day GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Day GtkLabel */
	w = gtk_label_new("/");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Day GtkSpinButton */
	adj = (GtkAdjustment *)gtk_adjustment_new(
		1.0f, 1.0f, 32.0f,
		1.0f, 5.0f, 5.0f
	);
	p->day_spin = w = gtk_spin_button_new(adj, 1.0, 0);
	gtk_widget_set_usize(w, 45, -1);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(time_prompt_changed_cb), p
	);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(w, "Day");
	gtk_widget_show(w);


	/* Hours/Minutes/Seconds GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Hours GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Hours GtkLabel */
	w = gtk_label_new("Time:");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Hours GtkSpinButton */
	adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0f, 0.0f, 23.0f,
		1.0f, 5.0f, 5.0f
	);
	p->hours_spin = w = gtk_spin_button_new(adj, 1.0, 0);
	gtk_widget_set_usize(w, 45, -1);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(time_prompt_changed_cb), p
	);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(w, "Hours");
	gtk_widget_show(w);

	/* Minutes GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Minutes GtkLabel */
	w = gtk_label_new(":");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Minutes GtkSpinButton */
	adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0f, 0.0f, 59.0f,
		1.0f, 5.0f, 5.0f
	);
	p->minutes_spin = w = gtk_spin_button_new(adj, 1.0, 0);
	gtk_widget_set_usize(w, 45, -1);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(time_prompt_changed_cb), p
	);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(w, "Minutes");
	gtk_widget_show(w);

	/* Seconds GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Seconds GtkLabel */
	w = gtk_label_new(":");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Seconds GtkSpinButton */
	adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0f, 0.0f, 59.0f,
		1.0f, 5.0f, 5.0f
	);
	p->seconds_spin = w = gtk_spin_button_new(adj, 1.0, 0);
	gtk_widget_set_usize(w, 45, -1);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(time_prompt_changed_cb), p
	);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(w, "Seconds");
	gtk_widget_show(w);

	/* Current Time Button */
	p->current_time_button = w = GUIButtonPixmap(
		(guint8 **)icon_time_stamp_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(time_prompt_current_time_cb), p
	);
	GUISetWidgetTip(w, "Use current time");

	p->freeze_count--;

	return(p->toplevel);
}

/*
 *	Gets the current value in seconds since EPOCH.
 *
 *	Returns the current value in seconds since EPOCH.
 */
gulong time_prompt_get(GtkWidget *w)
{
	gulong t;
	struct tm *tm_buf;
	TimePrompt *p = time_prompt_get_widget_data(
		w,
		"TimePromptGet"
	);
	if(p == NULL)
		return(0l);

	/* Allocate the time compoent values buffer */
	tm_buf = (struct tm *)g_malloc0(sizeof(struct tm));
	if(tm_buf == NULL)
		return(0l);

	/* Gather the time compoent values from the TimePrompt to the
	 * struct tm
	 */
	tm_buf->tm_year = (int)gtk_spin_button_get_value_as_int(
		GTK_SPIN_BUTTON(p->year_spin)
	) - 1900;

	tm_buf->tm_mon = (int)PUListBoxGetSelected(
		p->month_pulistbox
	);

	tm_buf->tm_mday = (int)gtk_spin_button_get_value_as_int(
		GTK_SPIN_BUTTON(p->day_spin)
	);

	tm_buf->tm_hour = (int)gtk_spin_button_get_value_as_int(
		GTK_SPIN_BUTTON(p->hours_spin)
	);

	tm_buf->tm_min = (gint)gtk_spin_button_get_value_as_int(
		GTK_SPIN_BUTTON(p->minutes_spin)
	);

	tm_buf->tm_sec = (gint)gtk_spin_button_get_value_as_int(
		GTK_SPIN_BUTTON(p->seconds_spin)
	);

	tm_buf->tm_isdst = -1;			/* Not specified */

	/* Calculate the seconds since EPOCH from the time compoent
	 * values
	 */
	t = (gulong)mktime(tm_buf);

	/* Delete the time compoent values buffer */
	g_free(tm_buf);

	return(t);
}

/*
 *	Sets the current value in seconds since EPOCH.
 *
 *	The t specifies the value in seconds since EPOCH.
 */
void time_prompt_set(
	GtkWidget *w,
	const gulong t
)
{
	time_t _t = (time_t)t;
	const struct tm *tm_ptr;
	TimePrompt *p = time_prompt_get_widget_data(
		w,
		"TimePromptSet"
	);
	if(p == NULL)
		return;

	/* Get the time compoent values from the specified time */
	tm_ptr = localtime(&_t);
	if(tm_ptr == NULL)
		return;

	p->freeze_count++;

	/* Set the time compoent values to the TimePrompt */
	gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(p->year_spin),
		(gfloat)(1900 + tm_ptr->tm_year)
	);

	PUListBoxSelect(
		p->month_pulistbox,
		tm_ptr->tm_mon
	);

	gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(p->day_spin),
		(gfloat)tm_ptr->tm_mday
	);

	gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(p->hours_spin),
		(gfloat)tm_ptr->tm_hour
	);

	gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(p->minutes_spin),
		(gfloat)tm_ptr->tm_min
	);

	gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(p->seconds_spin),
		(gfloat)tm_ptr->tm_sec
	);

	/* Notify about the change */
	if(p->changed_cb != NULL)
		p->changed_cb(
			p->toplevel,
			p->changed_data
		);

	p->freeze_count--;
}

/*
 *	Sets the changed callback.
 */
void time_prompt_set_changed_cb(
	GtkWidget *w,
	void (*func)(GtkWidget *, gpointer),
	gpointer data
)
{
	TimePrompt *p = time_prompt_get_widget_data(
		w,
		"TimePromptSetChangedCB"
	);
	if(p == NULL)
		return;

	p->changed_cb = func;
	p->changed_data = data;
}

/*
 *	Allows/disallows the user to set the current time.
 *
 *	This will show/hide the current time button.
 */
void time_prompt_allow_set_current_time(
	GtkWidget *w,
	const gboolean allow
)
{
	TimePrompt *p = time_prompt_get_widget_data(
		w,
		"TimePromptAllowSetCurrentTime"
	);
	if(p == NULL)
		return;

	if(allow)
	{
		p->flags |= TIME_PROMPT_ALLOW_SET_CURRENT_TIME;
		gtk_widget_show(p->current_time_button);
	}
	else
	{
		p->flags &= ~TIME_PROMPT_ALLOW_SET_CURRENT_TIME;
		gtk_widget_hide(p->current_time_button);
	}
}
