#include <stdio.h>
#include <errno.h>
#include <glib.h>
#include <unistd.h>

#include "../../include/string.h"


static void print_usage(const gchar *prog_name);
static gchar *fixindent_line(gchar *line);
static gint fixindent_stream(
	FILE *in_fp,
	FILE *out_fp
);
static gint fixindent_file(const gchar *path);
static gint fixindent(const gint argc, const gchar **argv);


static void print_usage(const gchar *prog_name)
{
	g_print(
"Usage: %s <file(s)...>\n",
		prog_name
	);
}


static gchar *fixindent_line(gchar *line)
{
	gint		i,
			orig_ntabs;
	const gchar *s;

	if(line == NULL)
		return(line);

	/* If the line does not start with a tab then we can be sure
	 * it is not a line we need to fix
	 */
	if(*line != '\t')
		return(line);


	/* Start by counting the number of tabs */
	for(s = line + 1, orig_ntabs = 1;
	    *s != '\0';
	    s++
	)
	{
		if(*s != '\t')
			break;

		orig_ntabs++;
	}

	/* Prefix orig_ntabs - 1 more tabs to this line */
	for(i = orig_ntabs - 1;
	    i > 0;
	    i--
	)
	{
		line = strinschr(
			line,
			0,
			'\t'
		);
		if(line == NULL)
			return(line);
	}

	/* Check if there are 4 spaces after the tabs */
	for(s = line; *s != '\0'; s++)
	{
		if(*s != '\t')
			break;
	}
	if(!g_strncasecmp(s, "    ", 4))
	{
		/* Remove the 4 spaces and replace it with a single tab */
		line = strdelchrs(
			line,
			s - line,
			4
		);
		if(line == NULL)
			return(line);

		line = strinschr(
                        line,
                        0,
                        '\t'
                );
                if(line == NULL)
                        return(line);
	}

	return(line);
}

static gint fixindent_stream(
	FILE *in_fp,
	FILE *out_fp
)
{
	gchar		*line,
			buf[10000];

	while(fgets(buf, sizeof(buf), in_fp) != NULL)
	{
		line = g_strdup(buf);
		line = fixindent_line(line);
		if(line == NULL)
			continue;

		if(fputs(line, out_fp) == EOF)
		{
			g_free(line);
			return(-1);
		}

		g_free(line);
	}

	return(0);
}


static gint fixindent_file(const gchar *path)
{
	gchar *out_path = g_strconcat(path, "-fixindent-out", NULL);
	FILE		*in_fp = fopen(path, "rb"),
			*out_fp;
	if(in_fp == NULL)
	{
		const gint error_code = (gint)errno;
		g_printerr("%s: %s\n", g_strerror(error_code), path);
		return(-1);
	}

	g_print("%s\n", path);

	out_fp = fopen(out_path, "wb");
	if(out_fp == NULL)
	{
		fclose(in_fp);
		g_free(out_path);
		return(-1);
	}

	if(fixindent_stream(
		in_fp,
		out_fp
	))
	{
		(void)fclose(in_fp);
		(void)fclose(out_fp);
		g_free(out_path);
		return(-1);
	}

	(void)fclose(in_fp);
	if(fclose(out_fp))
	{
		g_free(out_path);
		return(-1);
	}

	if(unlink(path))
	{
		g_free(out_path);
		return(-1);
	}

	if(rename(
		out_path,			/* Old */
		path				/* New */
	))
	{
		g_free(out_path);
		return(-1);
	}

	return(0);
}


static gint fixindent(const gint argc, const gchar **argv)
{
	gint i;
	const gchar *arg;
	GList		*glist,
			*paths_list = NULL;


	for(i = 1; i < argc; i++)
	{
		arg = argv[i];
		if(arg == NULL)
			continue;

		if(!g_strcasecmp(arg, "--help") ||
		   !g_strcasecmp(arg, "-help") ||
		   !g_strcasecmp(arg, "--h") ||
		   !g_strcasecmp(arg, "-h") ||
		   !g_strcasecmp(arg, "-?")
		)
		{
			print_usage(argv[0]);
			return(0);
		}
		else if((*arg != '-') && (*arg != '+'))
		{
			paths_list = g_list_append(
				paths_list,
				g_strdup(arg)
			);
		}



	}

	if(paths_list == NULL)
	{
		print_usage(argv[0]);
		return(2);
	}

	for(glist = paths_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		fixindent_file((const gchar *)glist->data);
	}

	return(0);
}


int main(int argc, char *argv[])
{
	return((int)fixindent(
		(gint)argc,
		(const gchar **)argv
	));
}
