#include <gtk/gtk.h>
#include "../guiutils.h"
#include "../statictip.h"
#include "si_cpu.h"
#include "si_win.h"
#include "config.h"

#include "../images/icon_cpu_20x20.xpm"
#include "../images/icon_drive_root2_20x20.xpm"
#include "../images/icon_memory_20x20.xpm"
#include "../images/icon_power_20x20.xpm"
#include "../images/icon_close_20x20.xpm"

#include "../images/icon_cpu_48x48.xpm"



typedef enum {
	SI_WIN_DANGER_NONE,
	SI_WIN_DANGER_LOW,
	SI_WIN_DANGER_HIGH
} SIWinDanger;


/* Callbacks */
static void si_win_destroy_cb(gpointer data);
static void si_win_realize_cb(GtkWidget *widget, gpointer data);
static void si_win_bar_realize_cb(GtkWidget *widget, gpointer data);
static gint si_win_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void si_win_show_memory_toggle_cb(GtkWidget *widget, gpointer data);
static void si_win_show_apm_power_toggle_cb(GtkWidget *widget, gpointer data);
static void si_win_show_cpu_details_toggle_cb(GtkWidget *widget, gpointer data);
static void si_win_show_style_continuous_cb(GtkWidget *widget, gpointer data);
static void si_win_show_style_discrete_cb(GtkWidget *widget, gpointer data);
static void si_win_show_style_led_cb(GtkWidget *widget, gpointer data);
static void si_win_close_cb(GtkWidget *widget, gpointer data);

/* Drawing */
static void si_win_draw_bar(
	SIWin *si,
	GtkWidget *w,
	GdkPixmap *pixmap,
	const gfloat v,
	const SIWinDanger danger
);
static void si_win_draw_cpu_load(
	SIWin *si,
	const gfloat v
);
static void si_win_draw_cpu_load_average(
	SIWin *si,
	const gfloat v
);
static void si_win_draw_memory(
	SIWin *si,
	const gfloat v
);
static void si_win_draw_apm_power(
	SIWin *si,
	const gfloat v
);
static void si_win_update_details(SIWin *si);

/* SIWin */
SIWin *si_win_new(
	const gint cpu_num,
	const GtkOrientation orientation,
	const gboolean show_border,
	const gboolean show_title,
	const SIWinDisplay display,
	const SIWinBarStyle bar_style,
	const gint bar_length
);
void si_win_set_display(
	SIWin *si,
	const SIWinDisplay display
);
void si_win_set_bar_style(
	SIWin *si,
	const SIWinBarStyle bar_style
);
void si_win_update(SIWin *si);
gboolean si_win_is_mapped(SIWin *si);
void si_win_map(SIWin *si);
void si_win_unmap(SIWin *si);
void si_win_delete(SIWin *si);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Toplevel GtkWindow "destroy" signal callback.
 */
static void si_win_destroy_cb(gpointer data)
{
	GdkColormap *colormap;
	SIWin *si = SI_WIN(data);
	if(si == NULL)
		return;

	si->freeze_count++;

	GTK_WIDGET_DESTROY(si->menu);
	GTK_ACCEL_GROUP_UNREF(si->accelgrp);

	si->gc = GDK_GC_UNREF(si->gc);

	si->bar_pixmap = GDK_PIXMAP_UNREF(si->bar_pixmap);

	gdk_cursor_destroy(si->translate_cur);

	colormap = si->colormap;
	if(colormap != NULL)
	{
		if(si->bar_led_fg_color != NULL)
		{
			const gint n = si->nled_bars;
			gint i;
			for(i = 0; i < n; i++)
			{
				GDK_COLORMAP_FREE_COLOR(
					colormap,
					&si->bar_led_fg_color[i]
				);
			}
			g_free(si->bar_led_fg_color);
		}
		if(si->bar_led_bg_color != NULL)
		{
			const gint n = si->nled_bars;
			gint i;
			for(i = 0; i < n; i++)
			{
				GDK_COLORMAP_FREE_COLOR(
					colormap,
					&si->bar_led_bg_color[i]
				);
			}
			g_free(si->bar_led_bg_color);
		}
		si->colormap = colormap = GDK_COLORMAP_UNREF(colormap);
	}

	si->freeze_count--;

	g_free(si);
}

/*
 *	Toplevel GtkWindow "realize" signal callback.
 */
static void si_win_realize_cb(GtkWidget *widget, gpointer data)
{
	GdkGeometry geo;
	GdkWindow *window;
	SIWin *si = SI_WIN(data);
	if((widget == NULL) || (si == NULL))
		return;

	window = widget->window;
	if(window == NULL)
		return;

	si->freeze_count++;

	geo.min_width = 10;
	geo.min_height = 10;
	geo.base_width = 0;
	geo.base_height = 0;
	geo.width_inc = 1;
	geo.height_inc = 1;
	gdk_window_set_geometry_hints(
		window,
		&geo,
		GDK_HINT_MIN_SIZE |
		GDK_HINT_BASE_SIZE |
		GDK_HINT_RESIZE_INC
	);
	gdk_window_set_decorations(
		window,
		((si->flags & SI_WIN_SHOW_BORDER) ? GDK_DECOR_BORDER : 0) |
		((si->flags & SI_WIN_SHOW_TITLE) ?  GDK_DECOR_TITLE : 0)
	);
	gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_CLOSE
	);
	GUISetWMIcon(
		window,
		(guint8 **)icon_cpu_48x48_xpm
	);

	si->translate_cur = gdk_cursor_new(GDK_FLEUR);

	if(si->colormap == NULL)
	{
		const gint ncolors = si->nled_bars;
		const gfloat mid_coeff = 0.68f;
		GdkColormap *colormap = GDK_COLORMAP_REF(gtk_widget_get_colormap(widget));

		if((si->bar_led_fg_color == NULL) && (ncolors > 0))
		{
			si->bar_led_fg_color = (GdkColor *)g_malloc(
				ncolors * sizeof(GdkColor)
			);
			if(si->bar_led_fg_color != NULL)
			{
				gint i;
				gfloat coeff;
				GdkColor *c;

				for(i = 0; i < ncolors; i++)
				{
					c = &si->bar_led_fg_color[i];
					coeff = (gfloat)(i + 1) / (gfloat)ncolors;
					if(coeff < mid_coeff)
					{
						c->red		= 0xFFFF * (coeff / mid_coeff);
						c->green	= 0xFFFF;
						c->blue		= 0x0000;
						c->pixel	= 0l;
					}
					else
					{
						c->red		= 0xFFFF;
						c->green	= 0xFFFF * (1.0f -
				((coeff - mid_coeff) / (1.0f - mid_coeff))
							      );
						c->blue		= 0x0000;
						c->pixel	= 0l;
					}

					GDK_COLORMAP_ALLOC_COLOR(colormap, c);
				}
			}
		}

		if((si->bar_led_bg_color == NULL) && (ncolors > 0))
		{
			si->bar_led_bg_color = (GdkColor *)g_malloc(
				ncolors * sizeof(GdkColor)
			);
			if(si->bar_led_bg_color != NULL)
			{
				gint i;
				gfloat coeff;
				GdkColor *c;

				for(i = 0; i < ncolors; i++)
				{
		            c = &si->bar_led_bg_color[i];
		            coeff = (gfloat)(i + 1) / (gfloat)ncolors;
		            if(coeff < mid_coeff)
		            {
		                c->red      = 0x5050 * (coeff / mid_coeff);
		                c->green    = 0x5050;
				        c->blue     = 0x0000;
		                c->pixel    = 0l;
		            }
		            else
				    {
		                c->red      = 0x5050;
		                c->green    = 0x5050 * (1.0f -
		                    ((coeff - mid_coeff) / (1.0f - mid_coeff))
		                );
		                c->blue     = 0x0000;
		                c->pixel    = 0l;
		            }

		            GDK_COLORMAP_ALLOC_COLOR(colormap, c);
				}
			}
		}

		si->colormap = colormap;
	}

	if(si->gc == NULL)
		si->gc = gdk_gc_new(window);

	si->flags |= SI_WIN_REALIZED;

	si->freeze_count--;
}

/*
 *	Bar GtkDrawingArea "realize" signal callback.
 */
static void si_win_bar_realize_cb(GtkWidget *widget, gpointer data)
{
	GdkWindow *window;
        GtkStyle *style;
	SIWin *si = SI_WIN(data);
	if((widget == NULL) || (si == NULL))
		return;

	window = widget->window;
	if(window == NULL)
		return;

	si->freeze_count++;

        style = gtk_widget_get_style(widget);
        if(style != NULL)
                gdk_window_set_background(
                        window,
                        &style->base[GTK_STATE_NORMAL]
                );

	si->freeze_count--;
}

/*
 *	Any GtkWidget event signal callback.
 */
static gint si_win_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	GdkEventConfigure *configure;
	GdkEventExpose *expose;
	GdkEventFocus *focus;
	GdkEventButton *button;
	GdkEventMotion *motion;
	GdkEventCrossing *crossing;
	SIWin *si = SI_WIN(data);
	if((widget == NULL) || (event == NULL) || (si == NULL))
		return(status);

	switch((gint)event->type)
	{
	  case GDK_DELETE:
		if(widget == si->toplevel)
		{
			si_win_unmap(si);
			status = TRUE;
		}
		break;

	  case GDK_CONFIGURE:
		configure = (GdkEventConfigure *)event;
		if(widget == si->toplevel)
		{
			status = TRUE;
		}
		else if(widget == si->cpu_load_da)
		{
			GDK_PIXMAP_UNREF(si->bar_pixmap);
			si->bar_pixmap = gdk_pixmap_new(
				widget->window, configure->width, configure->height,
				-1
			);
			status = TRUE;
		}
		else if(widget == si->cpu_loadavg_da)
		{
			status = TRUE;
		}
		else if(widget == si->memory_da)
		{
			status = TRUE;
		}
		break;

	  case GDK_EXPOSE:
		expose = (GdkEventExpose *)event;
		if(widget == si->toplevel)
		{
			status = TRUE;
		}
		else if(widget == si->cpu_load_da)
		{
			si_win_draw_cpu_load(si, -1.0f);
			status = TRUE;
		}
		else if(widget == si->cpu_loadavg_da)
		{
			si_win_draw_cpu_load_average(si, -1.0f);
			status = TRUE;
		}
		else if(widget == si->memory_da)
		{
			si_win_draw_memory(si, -1.0f);
			status = TRUE;
		}
		else if(widget == si->apm_power_da)
		{
			si_win_draw_apm_power(si, -1.0f);
			status = TRUE;
		}
		break;

	  case GDK_FOCUS_CHANGE:
		focus = (GdkEventFocus *)event;
		if(GTK_WIDGET_CAN_FOCUS(widget))
		{
			if(!GTK_WIDGET_HAS_FOCUS(widget) && focus->in)
				GTK_WIDGET_SET_FLAGS(widget, GTK_HAS_FOCUS);
			else if(GTK_WIDGET_HAS_FOCUS(widget) && !focus->in)
				GTK_WIDGET_UNSET_FLAGS(widget, GTK_HAS_FOCUS);
			status = TRUE;
		}
		break;

	  case GDK_BUTTON_PRESS:
		button = (GdkEventButton *)event;
		/* Grab focus on the event widget if possible */
		if(!GTK_WIDGET_HAS_FOCUS(widget) &&
		   GTK_WIDGET_CAN_FOCUS(widget)
		)
			gtk_widget_grab_focus(widget);

		switch(button->button)
		{
                    case GDK_BUTTON1:
                        if(!(si->flags & SI_WIN_SHOW_TITLE))
                        {
                                GdkModifierType mask;
                                gdk_window_get_pointer(
                                        widget->window,
                                        &si->last_button_x, &si->last_button_y,
                                        &mask
                                );
                                si->flags |= SI_WIN_DRAG_MOVE;
                                gdk_pointer_grab(
                                        button->window,
                                        FALSE,
                                        GDK_BUTTON_RELEASE_MASK |
                                        GDK_POINTER_MOTION_MASK,
                                        NULL,
                                        si->translate_cur,
                                        button->time
                                );
                                status = TRUE;
                        }
                        break;

		   case GDK_BUTTON3:
			gtk_menu_popup(
				GTK_MENU(si->menu),
				NULL, NULL,
				NULL, NULL,
				button->button, button->time
			);
			status = TRUE;
			break;
		}
		break;

          case GDK_BUTTON_RELEASE:
                button = (GdkEventButton *)event;
                switch(button->button)
                {
                    case GDK_BUTTON1:
                        if(si->flags & SI_WIN_DRAG_MOVE)
                        {
                                gdk_pointer_ungrab(button->time);
                                si->flags &= ~SI_WIN_DRAG_MOVE;
                                status = TRUE;
                        }
                        break;
                    case GDK_BUTTON3:
                        status = TRUE;
                        break;
                }
                break;

          case GDK_MOTION_NOTIFY:
                motion = (GdkEventMotion *)event;
                if(si->flags & SI_WIN_DRAG_MOVE)
                {
                        if(motion->is_hint)
                        {
                                gint x, y;
                                GdkModifierType mask;
                                gdk_window_get_pointer(
                                        motion->window,
                                        &x, &y,
                                        &mask
                                );
                        }
                        gtk_widget_set_uposition(
                                widget,
                                (gint)motion->x_root - si->last_button_x,
                                (gint)motion->y_root - si->last_button_y
                        );
                        status = TRUE;
                }
                break;

	  case GDK_ENTER_NOTIFY:
		crossing = (GdkEventCrossing *)event;
		if((widget == si->cpu_load_icon_eb) ||
		   (widget == si->cpu_load_da)
		)
		{
			/* Update the pointer_in_widget and redraw to set/update
			 * the static tip                                 
			 */
			si->pointer_in_widget = si->cpu_load_da;
			si_win_draw_cpu_load(si, -1.0f);
			status = TRUE;
		}
		else if((widget == si->cpu_loadavg_icon_eb) ||
		        (widget == si->cpu_loadavg_da)
		)
		{
			/* Update the pointer_in_widget and redraw to set/update
			 * the static tip
			 */
			si->pointer_in_widget = si->cpu_loadavg_da;
			si_win_draw_cpu_load_average(si, -1.0f);
			status = TRUE;
		}
		else if((widget == si->memory_icon_eb) ||
				(widget == si->memory_da)
		)
		{
			/* Update the pointer_in_widget and redraw to set/update
			 * the static tip
			 */
			si->pointer_in_widget = si->memory_da;
			si_win_draw_memory(si, -1.0f);
			status = TRUE;
		}
		else if((widget == si->apm_power_icon_eb) ||
				(widget == si->apm_power_da)
		)
		{
			/* Update the pointer_in_widget and redraw to set/update
			 * the static tip
			 */
			si->pointer_in_widget = si->apm_power_da;
			si_win_draw_apm_power(si, -1.0f);
			status = TRUE;
		}
		break;

	  case GDK_LEAVE_NOTIFY:
		if((widget == si->cpu_load_icon_eb) ||
		   (widget == si->cpu_load_da) ||
		   (widget == si->cpu_loadavg_icon_eb) ||
		   (widget == si->cpu_loadavg_da) ||
		   (widget == si->memory_icon_eb) ||
		   (widget == si->memory_da) ||
		   (widget == si->apm_power_icon_eb) ||
		   (widget == si->apm_power_da)
		)
		{
			StaticTipSet(si->pointer_in_widget, NULL, 0, 0, 0, 0);
			si->pointer_in_widget = NULL;
			status = TRUE;
		}
		break;
	}
	return(status);
}

/*
 *	SysInfo Window show Memory toggle callback.
 */
static void si_win_show_memory_toggle_cb(GtkWidget *widget, gpointer data)
{
	SIWinDisplay display;
	SIWin *si = SI_WIN(data);

	if(si->freeze_count > 0)
		return;

	display = si->display;
	if(display & SI_WIN_DISPLAY_MEMORY)
		display &= ~SI_WIN_DISPLAY_MEMORY;
	else
		display |= SI_WIN_DISPLAY_MEMORY;

	si_win_set_display(si, display);
}

/*
 *	SysInfo Window show APM Power toggle callback.
 */
static void si_win_show_apm_power_toggle_cb(GtkWidget *widget, gpointer data)
{
	SIWinDisplay display;
	SIWin *si = SI_WIN(data);

	if(si->freeze_count > 0)
		return;

	display = si->display;
	if(display & SI_WIN_DISPLAY_APM_POWER)
		display &= ~SI_WIN_DISPLAY_APM_POWER;
	else
		display |= SI_WIN_DISPLAY_APM_POWER;

	si_win_set_display(si, display);
}

/*
 *	SysInfo Window show CPU Details toggle callback.
 */
static void si_win_show_cpu_details_toggle_cb(GtkWidget *widget, gpointer data)
{
	SIWinDisplay display;
	SIWin *si = SI_WIN(data);

	if(si->freeze_count > 0)
		return;

	display = si->display;
	if(display & SI_WIN_DISPLAY_CPU_DETAILS)
		display &= ~SI_WIN_DISPLAY_CPU_DETAILS;
	else
		display |= SI_WIN_DISPLAY_CPU_DETAILS;

	si_win_set_display(si, display);
}

/*
 *	Bar Style Continuous callback.
 */
static void si_win_show_style_continuous_cb(GtkWidget *widget, gpointer data)
{
	SIWin *si = SI_WIN(data);
	if(si == NULL)
		return;

	if(si->freeze_count > 0)
		return;

	si_win_set_bar_style(si, SI_WIN_BAR_STYLE_CONTINUOUS);
}

/*
 *	Bar Style Discrete callback.
 */
static void si_win_show_style_discrete_cb(GtkWidget *widget, gpointer data)
{
	SIWin *si = SI_WIN(data);
	if(si == NULL)
		return;

	if(si->freeze_count > 0)
		return;

	si_win_set_bar_style(si, SI_WIN_BAR_STYLE_DISCRETE);
}

/*
 *	Bar Style Led callback.
 */
static void si_win_show_style_led_cb(GtkWidget *widget, gpointer data)
{
	SIWin *si = SI_WIN(data);
	if(si == NULL)
		return;

	if(si->freeze_count > 0)
		return;

	si_win_set_bar_style(si, SI_WIN_BAR_STYLE_LED);
}

/*
 *	Close callback.
 */
static void si_win_close_cb(GtkWidget *widget, gpointer data)
{
	SIWin *si = SI_WIN(data);
	if(si == NULL)
		return;

	if(si->freeze_count > 0)
		return;

	si_win_unmap(si);
}


/*
 *	Redraws the SysInfo Window's Bar.
 */
static void si_win_draw_bar(
	SIWin *si,
	GtkWidget *w,
	GdkPixmap *pixmap,
	gfloat v,
	const SIWinDanger danger
)
{
	gint		x, y,
			width, height;
	GdkWindow *window = w->window;
	GdkDrawable *drawable = (pixmap != NULL) ? pixmap : window;
	GdkGC *gc;
	GtkStateType state = GTK_WIDGET_STATE(w);
	GtkStyle *style = gtk_widget_get_style(w);

	if(window == NULL)
		return;

	gdk_window_get_size(window, &width, &height);
	if((width <= 0) || (height <= 0))
		return;

	/* Draw by bar style */
	switch(si->bar_style)
	{
	  case SI_WIN_BAR_STYLE_CONTINUOUS:
		/* Background */
		gc = style->base_gc[state];
		gdk_draw_rectangle(
			drawable,
			gc,
			TRUE,
			0, 0,
			width, height
		);
		/* Value */
		gc = style->bg_gc[GTK_STATE_SELECTED];
		switch(si->orientation)
		{
		  case GTK_ORIENTATION_HORIZONTAL:
			x = 0;
			gdk_draw_rectangle(
				drawable,
				gc,
				TRUE,
				x, 0,
				width * v, height
			);
			break;
		  case GTK_ORIENTATION_VERTICAL:
			y = height - (height * v);
			gdk_draw_rectangle(
				drawable,
				gc,
				TRUE,
				0, y,
				width, height - y
			);
			break;
		}
		break;
	  case SI_WIN_BAR_STYLE_DISCRETE:
		/* Background */
		gc = style->base_gc[state];
		gdk_draw_rectangle(
			drawable,
			gc,
			TRUE,
			0, 0,
			width, height
		);
		/* Value */
		if(si->ndiscrete_bars > 0)
		{
			const gint	nbars = si->ndiscrete_bars,
							n = MIN((nbars * v), nbars);
			gint	i,
					x, y,
					x_step, y_step,
					bar_width, bar_height;

			gc = style->bg_gc[GTK_STATE_SELECTED];

			switch(si->orientation)
			{
			  case GTK_ORIENTATION_HORIZONTAL:
				for(i = 0,
					x_step = width / nbars,
					x = 0,
					y = 0,
					bar_width = x_step - 1,
					bar_height = height;
					i < n;
					i++,
					x += x_step
				)
					gdk_draw_rectangle(
						drawable,
						gc,
						TRUE,		/* Fill */
						x, y,
						bar_width, bar_height
					);
				break;
			  case GTK_ORIENTATION_VERTICAL:
				for(i = 0,
					x = 0,
					y_step = height / nbars,
					y = height - y_step + 1,
					bar_width = width,
					bar_height = y_step - 1;
					i < n;
					i++,
					y -= y_step
				)
					gdk_draw_rectangle(
						drawable,
						gc,
						TRUE,
						x, y,
						bar_width, bar_height
					);
				break;
			}
		}
		break;
	  case SI_WIN_BAR_STYLE_LED:
		/* Background */
		gc = si->gc;
		gdk_gc_set_foreground(gc, &style->black);
		gdk_draw_rectangle(
			drawable,
			gc,
			TRUE,
			0, 0,
			width, height
		);
		/* Value */
		if(si->nled_bars > 0)
		{
			const gint	nbar_leds = si->nled_bars,
							n = MIN((nbar_leds * v), nbar_leds);
			gint	i,
					x, y,
					x_step, y_step,
					bar_width, bar_height;

			switch(si->orientation)
			{
			  case GTK_ORIENTATION_HORIZONTAL:
				x_step = width / nbar_leds;
				x = 0;
				y = 0;
				bar_width = x_step - 1;
				bar_height = height;
				switch(danger)
				{
				  case SI_WIN_DANGER_HIGH:
					for(i = 0; i < n; i++)
					{
						gdk_gc_set_foreground(
							gc,
							&si->bar_led_fg_color[i]
						);
						gdk_draw_rectangle(
							drawable,
							gc,
							TRUE,		/* Fill */
							x, y,
							bar_width, bar_height
						);
						x += x_step;
					}
					for(; i < nbar_leds; i++)
					{
						gdk_gc_set_foreground(
							gc,
							&si->bar_led_bg_color[i]
						);
						gdk_draw_rectangle(
							drawable,
							gc,
							TRUE,		/* Fill */
							x, y,
							bar_width, bar_height
						);
						x += x_step;
					}
					break;
				  case SI_WIN_DANGER_LOW:
					for(i = 0; i < n; i++)
					{
						gdk_gc_set_foreground(
							gc,
							&si->bar_led_fg_color[nbar_leds - 1 - i]
						);
						gdk_draw_rectangle(
							drawable,
							gc,
							TRUE,		/* Fill */
							x, y,
							bar_width, bar_height
						);
						x += x_step;
					}
					for(; i < nbar_leds; i++)
					{
						gdk_gc_set_foreground(
							gc,
							&si->bar_led_bg_color[nbar_leds - 1 - i]
						);
						gdk_draw_rectangle(
							drawable,
							gc,
							TRUE,		/* Fill */
							x, y,
							bar_width, bar_height
						);
						x += x_step;
					}
					break;
				  case SI_WIN_DANGER_NONE:
					gdk_gc_set_foreground(
						gc,
						&si->bar_led_fg_color[0]
					);
					for(i = 0; i < n; i++)
					{
						gdk_draw_rectangle(
							drawable,
							gc,
							TRUE,		/* Fill */
							x, y,
							bar_width, bar_height
						);
						x += x_step;
					}
					gdk_gc_set_foreground(
						gc,
						&si->bar_led_bg_color[0]
					);
					for(; i < nbar_leds; i++)
					{
						gdk_draw_rectangle(
							drawable,
							gc,
							TRUE,		/* Fill */
							x, y,
							bar_width, bar_height
						);
						x += x_step;
					}
					break;
				}
				break;
			  case GTK_ORIENTATION_VERTICAL:
				x = 0;
				y_step = height / nbar_leds;
				y = height - y_step + 1;
				bar_width = width;
				bar_height = y_step - 1;
				switch(danger)
				{
				  case SI_WIN_DANGER_HIGH:
					for(i = 0; i < n; i++)
					{
						gdk_gc_set_foreground(
							gc,
							&si->bar_led_fg_color[i]
						);
						gdk_draw_rectangle(
							drawable,
							gc,
							TRUE,		/* Fill */
							x, y,
							bar_width, bar_height
						);
						y -= y_step;
					}
					for(; i < nbar_leds; i++)
					{
						gdk_gc_set_foreground(
							gc,
							&si->bar_led_bg_color[i]
						);
						gdk_draw_rectangle(
							drawable,
							gc,
							TRUE,		/* Fill */
							x, y,
							bar_width, bar_height
						);
						y -= y_step;
					}
					break;
				  case SI_WIN_DANGER_LOW:
					for(i = 0; i < n; i++)
					{
						gdk_gc_set_foreground(
							gc,
							&si->bar_led_fg_color[nbar_leds - 1 - i]
						);
						gdk_draw_rectangle(
							drawable,
							gc,
							TRUE,		/* Fill */
							x, y,
							bar_width, bar_height
						);
						y -= y_step;
					}
					for(; i < nbar_leds; i++)
					{
						gdk_gc_set_foreground(
							gc,
							&si->bar_led_bg_color[nbar_leds - 1 - i]
						);
						gdk_draw_rectangle(
							drawable,
							gc,
							TRUE,		/* Fill */
							x, y,
							bar_width, bar_height
						);
						y -= y_step;
					}
					break;
				  case SI_WIN_DANGER_NONE:
					gdk_gc_set_foreground(
						gc,
						&si->bar_led_fg_color[0]
					);
					for(i = 0; i < n; i++)
					{
						gdk_draw_rectangle(
							drawable,
							gc,
							TRUE,		/* Fill */
							x, y,
							bar_width, bar_height
						);
						y -= y_step;
					}
					gdk_gc_set_foreground(
						gc,
						&si->bar_led_bg_color[0]
					);
					for(; i < nbar_leds; i++)
					{
						gdk_draw_rectangle(
							drawable,
							gc,
							TRUE,		/* Fill */
							x, y,
							bar_width, bar_height
						);
						y -= y_step;
					}
					break;
				}
				break;
			}
		}
		break;
	}

	/* Send the drawable to the window if it is not the window */
	if(drawable != window)
		gdk_draw_pixmap(
			window,
			style->white_gc,
			drawable,
			0, 0,
			0, 0,
			width, height
		);
}

/*
 *	Redraws the SysInfo Window's CPU Load Bar.
 */
static void si_win_draw_cpu_load(
	SIWin *si,
	const gfloat v
)
{
	gfloat _v = v;
	GtkWidget *w = si->cpu_load_da;

	if(_v < 0.0f)
		_v = si->cpu_load_last_value;
	else if(_v == si->cpu_load_last_value)
		return;

	/* Draw the bar */
	si_win_draw_bar(
		si,
		w,
		si->bar_pixmap,
		_v,
		SI_WIN_DANGER_HIGH
	);

	/* Update the static tip if the pointer is in this widget */
	if(w == si->pointer_in_widget)
	{
		gchar *s = g_strdup_printf(
			"CPU Load: %i%%",
			CLIP(
				(gint)(_v * 100.0f),
				0, 100
			)
		);
		StaticTipSet(
			w, s,
			STATIC_TIP_ALIGN_WIDGET_CENTER,
			(si->orientation == GTK_ORIENTATION_VERTICAL) ?
				STATIC_TIP_ALIGN_WIDGET_MIN :
				STATIC_TIP_ALIGN_WIDGET_MAX,
			0, 5
		);
		g_free(s);
	}
}

/*
 *	Redraws the SysInfo Window's CPU Load Average Bar.
 */
static void si_win_draw_cpu_load_average(
	SIWin *si,
	const gfloat v
)
{
	gfloat _v = v;
	GtkWidget *w = si->cpu_loadavg_da;

	if(_v < 0.0f)
		_v = si->cpu_loadavg_last_value;
	else if(_v == si->cpu_loadavg_last_value)
		return;

	/* Draw the bar */
	si_win_draw_bar(
		si,
		w,
		si->bar_pixmap,
		_v,
		SI_WIN_DANGER_HIGH
	);

	/* Update the static tip if the pointer is in this widget */
	if(w == si->pointer_in_widget)
	{
		gchar *s = g_strdup_printf(
			"System Load Average: %i%%",
			CLIP(
				(gint)(_v * 100.0f),
				0, 100
			)
		);
		StaticTipSet(
			w, s,
			STATIC_TIP_ALIGN_WIDGET_CENTER,
			(si->orientation == GTK_ORIENTATION_VERTICAL) ?
				STATIC_TIP_ALIGN_WIDGET_MIN :
				STATIC_TIP_ALIGN_WIDGET_MAX,
			0, 5
		);
		g_free(s);
	}
}

/*
 *	Redraws the SysInfo Window's Memory Bar.
 */
static void si_win_draw_memory(
	SIWin *si,
	const gfloat v
)
{
	gfloat _v = v;
	GtkWidget *w = si->memory_da;

	if(_v < 0.0f)
		_v = si->memory_last_value;
	else if(_v == si->memory_last_value)
		return;

	/* Draw the bar */        
	si_win_draw_bar(
		si,
		w,
		si->bar_pixmap,
		_v,
		SI_WIN_DANGER_NONE
	);

	/* Update the static tip if the pointer is in this widget */
	if(w == si->pointer_in_widget)                              
	{
		gchar *s = g_strdup_printf(
			"Memory: %i%%  Total: %ld mb  Free: %ld mb",
			CLIP(
				(gint)(_v * 100.0f),
				0, 100
			),
			si->memory_total,
			si->memory_free
		);
		StaticTipSet(   
			w, s,
			STATIC_TIP_ALIGN_WIDGET_CENTER,
			(si->orientation == GTK_ORIENTATION_VERTICAL) ?
				STATIC_TIP_ALIGN_WIDGET_MIN :
				STATIC_TIP_ALIGN_WIDGET_MAX,
			0, 5
		);
		g_free(s);
	}
}

/*
 *      Redraws the SysInfo Window's APM Power Bar.
 */
static void si_win_draw_apm_power(
	SIWin *si,
	const gfloat v
)
{
	gfloat _v = v;
	GtkWidget *w = si->apm_power_da;

	if(_v < 0.0f)
		_v = si->apm_power_last_value;
	else if(_v == si->apm_power_last_value)
		return;

	/* Draw the bar */
	si_win_draw_bar(
		si,
		w,
		si->bar_pixmap,
		_v,
		SI_WIN_DANGER_LOW
	);

	/* Update the static tip if the pointer is in this widget */
	if(w == si->pointer_in_widget)                              
	{
		gchar *s = g_strdup_printf(
			"APM Power: %i%%",
			CLIP(
				(gint)(_v * 100.0f),
				0, 100
			)
		);
		StaticTipSet(
			w, s,
			STATIC_TIP_ALIGN_WIDGET_CENTER,
			(si->orientation == GTK_ORIENTATION_VERTICAL) ?
				STATIC_TIP_ALIGN_WIDGET_MIN :
				STATIC_TIP_ALIGN_WIDGET_MAX,
			0, 5
		);
		g_free(s);
	}
}

/*
 *	Updates the SysInfo Window's Details.
 */
static void si_win_update_details(SIWin *si)
{
	gchar *msg;
	GtkWidget *w;
	GtkEditable *editable;
	GtkText *text;

	if(si == NULL)
		return;

	w = si->details_text;
	editable = GTK_EDITABLE(w);
	text = GTK_TEXT(w);

	gtk_text_freeze(text);

	/* Delete the existing message */
	gtk_editable_delete_text(editable, 0, -1);

	/* Format the new message */
	msg = si_cpu_get_details_message(si->cpu_num);

	/* Insert the new message */
	gtk_text_insert(
		text,
		NULL, NULL,
		NULL,
		msg, -1
	);

	g_free(msg);

	gtk_text_thaw(text);
}


/*
 *	Creates a new SysInfo Window.
 */
SIWin *si_win_new(
	const gint cpu_num,
	const GtkOrientation orientation,
	const gboolean show_border,
	const gboolean show_title,
	const SIWinDisplay display,
	const SIWinBarStyle bar_style,
	const gint bar_length
)
{
	const gint	border_major = 5,
			border_minor = 2;
	const gboolean	vertical = (orientation == GTK_ORIENTATION_VERTICAL) ?
				TRUE : FALSE;
	const gint	bar_width = vertical ? 10 : bar_length,
			bar_height = vertical ? bar_length : 10;
	gchar *s;
	GtkAccelGroup *accelgrp;
	GtkWidget	*w,
			*toplevel,
			*parent, *parent2, *parent3, *parent4,
			*parent5;
	SIWin *si = SI_WIN(g_malloc0(sizeof(SIWin)));
	if(si == NULL)
		return(NULL);

	si->accelgrp = accelgrp = gtk_accel_group_new();
	si->toplevel = toplevel = gtk_window_new(GTK_WINDOW_TOPLEVEL);
/*
	si->busy_count = 0;
	si->freeze_count = 0;
 */
	si->flags =	(show_border ? SI_WIN_SHOW_BORDER : 0) |
			(show_title ? SI_WIN_SHOW_TITLE : 0);

	si->orientation = orientation;
	si->display = display;
	si->bar_style = bar_style;
	si->ndiscrete_bars = bar_length / (6 + 1);
	si->nled_bars = bar_length / (4 + 1);

	si->ncpus = si_cpu_get_total();
	si->cpu_num = cpu_num;

	si->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_object_set_data_full(
		GTK_OBJECT(w), SI_WIN_KEY,
		si, si_win_destroy_cb
	);
	gtk_window_set_wmclass(
		GTK_WINDOW(w),
		SYSINFO_WM_WINDOW_NAME,
		SYSINFO_WM_CLASS_NAME
	);
	gtk_widget_set_name(w, SYSINFO_TOPLEVEL_WIDGET_NAME);
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, TRUE);
	if(cpu_num > -1)
		s = si_cpu_get_model(cpu_num);
	else
		s = g_strdup_printf(
			"%i %s",
			si->ncpus,
			(si->ncpus == 1) ? "CPU" : "CPUs"
		);
	gtk_window_set_title(GTK_WINDOW(w), s);
	g_free(s);
	gtk_widget_add_events(
		w,
		GDK_STRUCTURE_MASK | GDK_EXPOSURE_MASK |
		GDK_FOCUS_CHANGE_MASK |
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "realize",
		GTK_SIGNAL_FUNC(si_win_realize_cb), si
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "delete_event",
		GTK_SIGNAL_FUNC(si_win_event_cb), si
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "configure_event",
		GTK_SIGNAL_FUNC(si_win_event_cb), si
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "expose_event",
		GTK_SIGNAL_FUNC(si_win_event_cb), si
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "focus_in_event",
		GTK_SIGNAL_FUNC(si_win_event_cb), si
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "focus_out_event",
		GTK_SIGNAL_FUNC(si_win_event_cb), si
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(si_win_event_cb), si
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(si_win_event_cb), si
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(si_win_event_cb), si
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(si_win_event_cb), si
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "motion_notify_event",
		GTK_SIGNAL_FUNC(si_win_event_cb), si
	);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	parent = w;

	/* If no borders are to be displayed then add the Main GtkFrame */
	if(!(si->flags & SI_WIN_SHOW_BORDER))
	{
		w = gtk_frame_new(NULL);
		gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_OUT);
		gtk_container_add(GTK_CONTAINER(parent), w);
		gtk_widget_show(w);
		parent = w;
	}

	/* Main GtkBox */
	w = vertical ?
		gtk_hbox_new(FALSE, border_major) :
		gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


#if 0
	/* GtkBox for CPU icon and load bars */
	w = vertical ?
		gtk_vbox_new(FALSE, border_major) :
		gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* GtkBox for CPU Icon */
	w = vertical ? gtk_hbox_new(TRUE, 0) :
		gtk_vbox_new(TRUE, 0);
	if(vertical)
		gtk_box_pack_end(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	else
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* CPU Icon */
	w = gtk_pixmap_new_from_xpm_d(
		NULL,
		NULL,
		(guint8 **)icon_cpu_32x32_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
#else
	parent2 = parent;
#endif

	/* GtkBox for the bars */
	w = vertical ? gtk_hbox_new(TRUE, border_minor) :
		gtk_vbox_new(TRUE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

#define CREATE_BAR(_toplevel_,_widget_name_,_icon_eb_,_bar_da_,_icon_data_)	{ \
 /* Toplevel GtkBox */					\
 (_toplevel_) = w = vertical ?				\
  gtk_vbox_new(FALSE, border_minor) :			\
  gtk_hbox_new(FALSE, border_minor);			\
 gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0); \
 parent4 = w;						\
							\
 /* Icon GtkEventBox */					\
 (_icon_eb_) = w = gtk_event_box_new();			\
 if(vertical)						\
  gtk_box_pack_end(GTK_BOX(parent4), w, FALSE, FALSE, 0);\
 else							\
  gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);\
 gtk_widget_set_events(					\
  w,							\
  GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK		\
 );							\
 gtk_signal_connect(					\
  GTK_OBJECT(w), "enter_notify_event",			\
  GTK_SIGNAL_FUNC(si_win_event_cb), si			\
 );							\
 gtk_signal_connect(					\
  GTK_OBJECT(w), "leave_notify_event",			\
  GTK_SIGNAL_FUNC(si_win_event_cb), si			\
 );							\
 gtk_widget_show(w);					\
 parent5 = w;						\
							\
 /* Icon GtkPixmap */					\
 w = gtk_pixmap_new_from_xpm_d(				\
  NULL,							\
  NULL,							\
  (guint8 **)(_icon_data_)				\
 );							\
 gtk_container_add(GTK_CONTAINER(parent5), w);		\
 gtk_widget_show(w);					\
							\
 w = vertical ?						\
  gtk_hbox_new(TRUE, 0) :				\
  gtk_vbox_new(TRUE, 0);				\
 gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);\
 gtk_widget_show(w);					\
 parent5 = w;						\
							\
 /* Bar GtkFrame */					\
 w = gtk_frame_new(NULL);				\
 gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);\
 gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);\
 gtk_widget_show(w);					\
 parent5 = w;						\
							\
 /* GtkDrawingArea */					\
 (_bar_da_) = w = gtk_drawing_area_new();		\
 gtk_widget_set_name(w, (_widget_name_));		\
 gtk_widget_set_events(					\
  w,							\
  GDK_STRUCTURE_MASK | GDK_EXPOSURE_MASK |		\
  GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |	\
  GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK		\
 );							\
 gtk_signal_connect(					\
  GTK_OBJECT(w), "realize",				\
  GTK_SIGNAL_FUNC(si_win_bar_realize_cb), si		\
 );							\
 gtk_signal_connect(					\
  GTK_OBJECT(w), "configure_event",			\
  GTK_SIGNAL_FUNC(si_win_event_cb), si			\
 );							\
 gtk_signal_connect(					\
  GTK_OBJECT(w), "expose_event",			\
  GTK_SIGNAL_FUNC(si_win_event_cb), si			\
 );							\
 gtk_signal_connect(					\
  GTK_OBJECT(w), "button_press_event",			\
  GTK_SIGNAL_FUNC(si_win_event_cb), si			\
 );							\
 gtk_signal_connect(					\
  GTK_OBJECT(w), "button_release_event",		\
  GTK_SIGNAL_FUNC(si_win_event_cb), si			\
 );							\
 gtk_signal_connect(					\
  GTK_OBJECT(w), "enter_notify_event",			\
  GTK_SIGNAL_FUNC(si_win_event_cb), si			\
 );							\
 gtk_signal_connect(					\
  GTK_OBJECT(w), "leave_notify_event",			\
  GTK_SIGNAL_FUNC(si_win_event_cb), si			\
 );							\
 gtk_widget_set_usize(w, bar_width, bar_height);	\
 gtk_container_add(GTK_CONTAINER(parent5), w);		\
 gtk_widget_show(w);					\
							\
}

	/* CPU Load */
	CREATE_BAR(
		si->cpu_load_box,
		SYSINFO_CPU_LOAD_BAR_WIDGET_NAME,
		si->cpu_load_icon_eb,
		si->cpu_load_da,
		icon_cpu_20x20_xpm
	);
	if(display & SI_WIN_DISPLAY_CPU_LOAD)
		gtk_widget_show(si->cpu_load_box);

	/* System Load Average Bar */
	CREATE_BAR(
		si->cpu_loadavg_box,
		SYSINFO_LOAD_AVERAGE_BAR_WIDGET_NAME,
		si->cpu_loadavg_icon_eb,
		si->cpu_loadavg_da,
		icon_drive_root2_20x20_xpm
	);
	if(display & SI_WIN_DISPLAY_CPU_LOADAVG)
		gtk_widget_show(si->cpu_loadavg_box);

	/* Memory Bar */
	CREATE_BAR(
		si->memory_box,
		SYSINFO_MEMORY_BAR_WIDGET_NAME,
		si->memory_icon_eb,
		si->memory_da,
		icon_memory_20x20_xpm
	);
	if(display & SI_WIN_DISPLAY_MEMORY)
		gtk_widget_show(si->memory_box);

	/* APM Power Bar */
	CREATE_BAR(
		si->apm_power_box,
		SYSINFO_APM_POWER_BAR_WIDGET_NAME,
		si->apm_power_icon_eb,
		si->apm_power_da,
		icon_power_20x20_xpm
	);
	if(display & SI_WIN_DISPLAY_APM_POWER)
		gtk_widget_show(si->apm_power_box);

#undef CREATE_BAR

	/* Box for Details */
	si->details_box = w = vertical ?
		gtk_vbox_new(FALSE, border_minor) :
		gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	if(display & SI_WIN_DISPLAY_CPU_DETAILS)
		gtk_widget_show(w);
	parent2 = w;

	w = gtk_frame_new("Details");
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Table for Details Text */
	w = gtk_table_new(2, 2, FALSE);
	gtk_table_set_row_spacing(GTK_TABLE(w), 0, border_minor);
	gtk_table_set_col_spacing(GTK_TABLE(w), 0, border_minor);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent2 = w;

	if(parent2 != NULL)
	{
		GtkRcStyle *rcstyle = gtk_rc_style_new();
		GtkWidget *sb;
		GtkTable *table = GTK_TABLE(parent2);
		GtkEditable *editable;
		GtkText *text;

		rcstyle->font_name = STRDUP(
"-adobe-courier-medium-r-normal-*-12-*-*-*-*-*-iso8859-1"
		);

		/* Details GtkText */
		si->details_text = w = gtk_text_new(NULL, NULL);
		editable = GTK_EDITABLE(w);
		text = GTK_TEXT(w);
		gtk_widget_set_usize(w, -1, 80);
		gtk_text_set_editable(text, FALSE);
		gtk_text_set_word_wrap(text, TRUE);
		gtk_table_attach(
			table, w,
			0, 1, 0, 1,
			GTK_EXPAND | GTK_SHRINK | GTK_FILL,
			GTK_EXPAND | GTK_SHRINK | GTK_FILL,
			0, 0
		);
		gtk_widget_modify_style(w, rcstyle);
		gtk_widget_show(w);

		/* Vertical scroll bar */
		sb = gtk_vscrollbar_new(GTK_TEXT(w)->vadj);
		gtk_table_attach(
			table, sb,
			1, 2, 0, 1,
			GTK_FILL,
			GTK_EXPAND | GTK_SHRINK | GTK_FILL,
			0, 0
		);
		gtk_widget_show(sb);

		gtk_rc_style_unref(rcstyle);
	}



	/* Right-click popup menu */
	si->menu = w = GUIMenuCreate();
	if(w != NULL)
	{
		guint accel_key, accel_mods;
		guint8 **icon;
		const gchar *label;
		GtkWidget	*menu = w,
					*submenu;
		void (*func_cb)(GtkWidget *w, gpointer);
		gpointer data = si;

#define ADD_MENU_ITEM_LABEL	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_LABEL,			\
  accelgrp,					\
  icon, label,					\
  accel_key, accel_mods,			\
  func_cb, data					\
 );						\
}
#define ADD_MENU_ITEM_CHECK	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_CHECK,			\
  accelgrp,					\
  icon, label,					\
  accel_key, accel_mods,			\
  func_cb, data					\
 );						\
}
#define ADD_MENU_ITEM_SUBMENU_LABEL	{	\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SUBMENU,			\
  accelgrp,					\
  icon, label,					\
  accel_key, accel_mods,			\
  func_cb, data					\
 );						\
 GUIMenuItemSetSubMenu(w, submenu);		\
}
#define ADD_MENU_SEPARATOR	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL, NULL,					\
  0, 0,						\
  NULL, NULL					\
 );						\
}

		icon = NULL;
		label = "Show Memory";
		accel_key = 0;
		accel_mods = 0;
		func_cb = si_win_show_memory_toggle_cb;
		ADD_MENU_ITEM_CHECK
		si->display_memory_micheck = w;

		icon = NULL;
		label = "Show APM Power";
		accel_key = 0;
		accel_mods = 0;
		func_cb = si_win_show_apm_power_toggle_cb;
		ADD_MENU_ITEM_CHECK
		si->display_apm_power_micheck = w;

		icon = NULL;
		label = "Show CPU Details";
		accel_key = 0;
		accel_mods = 0;
		func_cb = si_win_show_cpu_details_toggle_cb;
		ADD_MENU_ITEM_CHECK
		si->display_cpu_details_micheck = w;

		ADD_MENU_SEPARATOR

		submenu = GUIMenuCreate();
		if(submenu != NULL)
		{
			GtkWidget *menu = submenu;

			icon = NULL;
			label = "Continuous";
			accel_key = 0;
			accel_mods = 0;
			func_cb = si_win_show_style_continuous_cb;
			ADD_MENU_ITEM_CHECK
			si->bar_style_continuous_micheck = w;

			icon = NULL;
			label = "Discrete";
			accel_key = 0;
			accel_mods = 0;
			func_cb = si_win_show_style_discrete_cb;
			ADD_MENU_ITEM_CHECK
			si->bar_style_discrete_micheck = w;

			icon = NULL;
			label = "Led";
			accel_key = 0;
			accel_mods = 0;
			func_cb = si_win_show_style_led_cb;
			ADD_MENU_ITEM_CHECK
			si->bar_style_led_micheck = w;
		}
		icon = NULL;
		label = "Bar Style";
		accel_key = 0;
		accel_mods = 0;
		func_cb = NULL;
		ADD_MENU_ITEM_SUBMENU_LABEL

		ADD_MENU_SEPARATOR

		icon = (guint8 **)icon_close_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Cierre"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fermer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nah"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Vicino"
#elif defined(PROG_LANGUAGE_DUTCH)
"Einde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Prximo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Nr"
#else
"Close"
#endif
		;
		accel_key = 0;
		accel_mods = 0;
		func_cb = si_win_close_cb;
		ADD_MENU_ITEM_LABEL

#undef ADD_MENU_SEPARATOR
#undef ADD_MENU_ITEM_CHECK
#undef ADD_MENU_ITEM_SUBMENU_LABEL
#undef ADD_MENU_ITEM_LABEL
	}


	/* Set the CPU information */
	si_win_update_details(si);

	si_win_update(si);

	si->freeze_count--;

	return(si);
}

/*
 *	Sets the SysInfo Window's display flags.
 */
void si_win_set_display(
	SIWin *si, SIWinDisplay display
)
{
	GtkWidget *w;

	if(si == NULL)
		return;

	if(si->display == display)
		return;

	si->display = display;

	/* Display CPU Load */
	w = si->cpu_load_box;
	if(w != NULL)
	{
		if(display & SI_WIN_DISPLAY_CPU_LOAD)
			gtk_widget_show(w);
		else
			gtk_widget_hide(w);
	}

	/* Display CPU Load Average */
	w = si->cpu_loadavg_box;
	if(w != NULL)
	{
		if(display & SI_WIN_DISPLAY_CPU_LOADAVG)
			gtk_widget_show(w);
		else
			gtk_widget_hide(w);
	}

	/* Display Memory */
	w = si->memory_box;
	if(w != NULL)
	{
		if(display & SI_WIN_DISPLAY_MEMORY)
			gtk_widget_show(w);
		else
			gtk_widget_hide(w);
	}

	/* Display APM Power */
	w = si->apm_power_box;
	if(w != NULL)
	{
		if(display & SI_WIN_DISPLAY_APM_POWER)
			gtk_widget_show(w);
		else
			gtk_widget_hide(w);
	}

	/* Display Details */
	w = si->details_box;
	if(w != NULL)
	{
		if(display & SI_WIN_DISPLAY_CPU_DETAILS)
			gtk_widget_show(w);
		else
			gtk_widget_hide(w);
	}

	/* Update widgets and redraw */
	si_win_update(si);
}

/*
 *	Sets the SysInfo Window's bar style.
 */
void si_win_set_bar_style(
	SIWin *si, SIWinBarStyle bar_style
)
{
	if(si == NULL)
		return;

	if(si->bar_style == bar_style)
		return;

	si->bar_style = bar_style;

	si_win_draw_cpu_load(si, -1.0f);
	si_win_draw_cpu_load_average(si, -1.0f);
	si_win_draw_memory(si, -1.0f);
	si_win_draw_apm_power(si, -1.0f);
			
	si_win_update(si);
}

/*
 *	Regets the system information and updates the SIWin's
 *	GtkWidgets to reflect the current values.
 */
void si_win_update(SIWin *si)
{
	gfloat v, dv, max_change = 0.2f;
	gulong free, total;

	if(si == NULL)
		return;

	si->freeze_count++;

#define SET_CHECK_MENU_ITEM(_w_,_b_)		\
{ if((_w_) != NULL) {				\
 if(GTK_CHECK_MENU_ITEM(_w_)->active != (_b_))	\
  gtk_check_menu_item_set_active(		\
   GTK_CHECK_MENU_ITEM(_w_), (_b_)		\
  );						\
} }

	/* Display */
	SET_CHECK_MENU_ITEM(
		si->display_memory_micheck,
		(si->display & SI_WIN_DISPLAY_MEMORY) ? TRUE : FALSE
	);
	SET_CHECK_MENU_ITEM(
		si->display_apm_power_micheck,
		(si->display & SI_WIN_DISPLAY_APM_POWER) ? TRUE : FALSE
	);
	SET_CHECK_MENU_ITEM(
		si->display_cpu_details_micheck,
		(si->display & SI_WIN_DISPLAY_CPU_DETAILS) ? TRUE : FALSE
	);

	/* Bar Style */
	SET_CHECK_MENU_ITEM(
		si->bar_style_continuous_micheck,
		(si->bar_style == SI_WIN_BAR_STYLE_CONTINUOUS) ? TRUE : FALSE
	);
	SET_CHECK_MENU_ITEM(
		si->bar_style_discrete_micheck,
		(si->bar_style == SI_WIN_BAR_STYLE_DISCRETE) ? TRUE : FALSE
	);
	SET_CHECK_MENU_ITEM(
		si->bar_style_led_micheck,
		(si->bar_style == SI_WIN_BAR_STYLE_LED) ? TRUE : FALSE
	);

	/* CPU Load */
	v = si_cpu_get_load(si->cpu_num);
	dv = v - si->cpu_load_last_value;
	if(dv > max_change)
		v = si->cpu_load_last_value + max_change;
	else if(dv < -max_change)
		v = si->cpu_load_last_value - max_change;
/*	v = CLIP(v, 0.0f, 1.0f);*/
	si_win_draw_cpu_load(si, v);
	si->cpu_load_last_value = v;

	/* System Load Average */
	v = si_cpu_get_load_average();
	si_win_draw_cpu_load_average(si, v);
	si->cpu_loadavg_last_value = v;

	/* Memory */
	si_cpu_get_memory(&free, &total);
	si->memory_free = free;
	si->memory_total = total;
	v = (total > 0l) ? (gfloat)free / (gfloat)total : 0.0f;
	si_win_draw_memory(si, v);
	si->memory_last_value = v;

	/* APM Power */
	v = si_cpu_get_apm_power();
	/* On AC? */
	if(v < 0.0f)
		v = 1.0f;
	si_win_draw_apm_power(si, v);
	si->apm_power_last_value = v;

#undef SET_CHECK_MENU_ITEM

	si->freeze_count--;
}

/*
 *	Checks if the SIWin is mapped.
 */
gboolean si_win_is_mapped(SIWin *si)
{
	if(si == NULL)
		return(FALSE);

	return((si->flags & SI_WIN_MAPPED) ? TRUE : FALSE);
}

/*
 *	Maps the SIWin.
 */
void si_win_map(SIWin *si)
{
	if(si == NULL)
		return;

	gtk_widget_show(si->toplevel);
	si->flags |= SI_WIN_MAPPED;
}

/*
 *	Unmaps the SIWin.
 */
void si_win_unmap(SIWin *si)
{
	if(si == NULL)
		return;

	gtk_widget_hide(si->toplevel);
	si->flags &= ~SI_WIN_MAPPED;
}

/*
 *	Deletes the SIWin.
 */
void si_win_delete(SIWin *si)
{
	if(si == NULL)
		return;

	si_win_unmap(si);

	/* Destroy the toplevel GtkWindow and call si_win_destroy_cb()
	 * to delete the rest of the SIWin
	 */
	GTK_WIDGET_DESTROY(si->toplevel);
}
