#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <glib.h>
#include "si_cpu.h"


gint si_cpu_get_total(void);

gfloat si_cpu_get_load(const gint cpu_num);
gfloat si_cpu_get_load_average(void);
void si_cpu_get_memory(gulong *free_rtn, gulong *total_rtn);
gfloat si_cpu_get_apm_power(void);

gchar *si_cpu_get_vendor(const gint cpu_num);
gchar *si_cpu_get_model(const gint cpu_num);
gchar *si_cpu_get_details_message(const gint cpu_num);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define ABSOLUTE(x)	(((x) < 0) ? (-(x)) : (x))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)
#define STRPFX(s,p)     ((((s) != NULL) && ((p) != NULL)) ? \
			(!strncmp((s), (p), strlen(p))) : FALSE)
#define ISBLANK(c)		(((c) == ' ') || ((c) == '\t'))


/*
 *	Gets the total number of CPUs on the system.
 */
gint si_cpu_get_total(void)
{
	gint ncpus;
	gchar buf[1024];

	/* Open the cpu information file for reading */
	FILE *fp = fopen("/proc/cpuinfo", "rb");
	if(fp == NULL)
		return(0);

	/* Read each line until we get to the line that contains the
	 * information we want
	 */
	ncpus = 0;
	while(fgets(buf, sizeof(buf), fp) != NULL)
	{
		if(STRPFX(buf, "processor"))
			ncpus++;

	}

	(void)fclose(fp);

	return(ncpus);
}


/*
 *	Gets the CPU load as a coefficient value from 0.0 to 1.0.
 *
 *	The cpu_num specifies the CPU index from 0. If cpu_num is -1
 *	then the total cpu is used.
 */
gfloat si_cpu_get_load(const gint cpu_num)
{
	static gint n = 0;
	static gint ct[2][4] = { { 0, 0, 0, 0 }, { 0, 0, 0, 0 } };
	gint		i,
			t;
	gfloat v;
	const gchar *pfx = "cpu";
	gchar		*s,
			*s_total,
			*s_val,
			buf[256];
	gint d[4];
	FILE *fp;

	/* Open the cpu statistics file for reading */
	fp = fopen("/proc/stat", "rb");
	if(fp == NULL)
		return(0.0f);

	/* Read each line until we get to the one that contains the
	 * cpu information that we want
	 */
	s_total = NULL;
	s_val = NULL;
	while(fgets(buf, sizeof(buf), fp) != NULL)
	{
		/* Does this line start with the CPU prefix? */
		if(STRPFX(buf, pfx))
		{
			/* Seek past the prefix to the start of the CPU
			 * index
			 */
			const char *s_cpu_num = buf + strlen(pfx);

			/* Record the line containing the total CPU load */
			if((s_total == NULL) && !isdigit(*s_cpu_num))
			{
				const char *s = s_cpu_num;
				while((*s != '\0') && !ISBLANK(*s))
					s++;
				while(ISBLANK(*s))
					s++;
				g_free(s_total);
				s_total = g_strdup(s);
			}

			/* Get the total CPU? */
			if(cpu_num < 0)
			{
				g_free(s_val);
				s_val = STRDUP(s_total);
				break;
			}
			/* A CPU index was specified, does this line specify
			 * a specific CPU index?
			 */
			else if(isdigit(*s_cpu_num))
			{
				if(atoi(s_cpu_num) == cpu_num)
				{
					const char *s = s_cpu_num;
					while((*s != '\0') && !ISBLANK(*s))
						s++;
					while(ISBLANK(*s))
						s++;
					g_free(s_val);
					s_val = g_strdup(s);
					break;
				}
			}
		}
	}

	(void)fclose(fp);

	/* If no specific CPU was specified then use the total CPU */
	if(s_val == NULL)
	{
		s_val = STRDUP(s_total);
		if(s_val == NULL)
		{
			g_free(s_val);
			g_free(s_total);
			return(0.0f);
		}
	}

	/* Get the CPU load */
	s = s_val;

	(void)sscanf(
		s,
		"%u %u %u %u",
		&ct[n][0], &ct[n][1], &ct[n][2], &ct[n][3]
	);

	t = 0;
	for(i = 0; i < 4; i++)
		t += (d[i] = ABSOLUTE(ct[n][i] - ct[1 - n][i]));
	if(t <= 0)
	{
		g_free(s_val);
		g_free(s_total);
		return(0.0f);
	}

	v = (gfloat)(t - d[3]) / (gfloat)t;

	n = 1 - n;

	g_free(s_val);
	g_free(s_total);

	return(v);
}

/*
 *	Gets the load average of all the CPUs as a coefficient value
 *	from 0.0 to 1.0.
 */
gfloat si_cpu_get_load_average(void)
{
	gchar		*s,
			buf[1024];

	/* Open the load average file for reading */
	FILE *fp = fopen("/proc/loadavg", "rb");
	if(fp == NULL)
		return(0.0f);

	if(fgets(buf, sizeof(buf), fp) != NULL)
		buf[sizeof(buf) - 1] = '\0';
	else
		*buf = '\0';

	(void)fclose(fp);

	s = buf;
	while(ISBLANK(*s))
		s++;

	return((gfloat)atof(s));
}


/*
 *	Gets the memory free and total in bytes.
 */
void si_cpu_get_memory(gulong *free_rtn, gulong *total_rtn)
{
	gchar		*s,
			buf[1024];
	const gchar *pfx = "Mem:";

	/* Open the memory information file for reading */
	FILE *fp = fopen("/proc/meminfo", "rb");
	if(free_rtn != NULL)
		*free_rtn = 0l;
	if(total_rtn != NULL)
		*total_rtn = 0l;
	if(fp == NULL)
		return;

	/* Read each line until we get to the line that contains the
	 * information we want
	 */
	s = NULL;
	while(fgets(buf, sizeof(buf), fp) != NULL)
	{
		if(STRPFX(buf, pfx))
		{
			s = buf;
			while((*s != '\0') && (*s != ':'))
				s++;
			if(*s == ':')
				s++;
			while(ISBLANK(*s))
				s++;
			break;
		}
	}

	(void)fclose(fp);

	if(s == NULL)
		return;

	/* Total */
	if(total_rtn != NULL)
		*total_rtn = (gulong)atol(s);
	while((*s != '\0') && !ISBLANK(*s))
		s++;
	while(ISBLANK(*s))
		s++;
	/* Used (skip) */
	while((*s != '\0') && !ISBLANK(*s))
		s++;
	while(ISBLANK(*s))
		s++;

	/* Free */
	if(free_rtn != NULL)
		*free_rtn = (gulong)atol(s);
}

/*
 *	Gets the Advanced Power Management power level as a
 *	coefficient value from 0.0 to 1.0 (or -1.0 if on line/AC power).
 */
gfloat si_cpu_get_apm_power(void)
{
	gchar		*s,
			buf[1024];

	/* Open the APM file for reading */
	FILE *fp = fopen("/proc/apm", "rb");
	if(fp == NULL)
		return(0.0f);

	if(fgets(buf, sizeof(buf), fp) != NULL)
		buf[sizeof(buf) - 1] = '\0';
	else
		*buf = '\0';

	(void)fclose(fp);

	s = buf;
	while(ISBLANK(*s))
		s++;

	while((*s != '\0') && !ISBLANK(*s))
		s++;
	while(ISBLANK(*s))
		s++;

	while((*s != '\0') && !ISBLANK(*s))
		s++;
	while(ISBLANK(*s))
		s++;

	while((*s != '\0') && !ISBLANK(*s))
		s++;
	while(ISBLANK(*s))
		s++;

	while((*s != '\0') && !ISBLANK(*s))
		s++;
	while(ISBLANK(*s))
		s++;

	while((*s != '\0') && !ISBLANK(*s))
		s++;
	while(ISBLANK(*s))
		s++;

	while((*s != '\0') && !ISBLANK(*s))
		s++;
	while(ISBLANK(*s))
		s++;

	return((gfloat)atof(s));
}


/*
 *	Gets the CPU vendor's name.
 */
gchar *si_cpu_get_vendor(const gint cpu_num)
{
	gint cur_cpu_num;
	gchar		buf[1024],
			*rtn_s;

	/* Open the cpu information file for reading */
	FILE *fp = fopen("/proc/cpuinfo", "rb");
	if(fp == NULL)
		return(NULL);

	/* Read each line until we get to the line that contains the
	 * information we want
	 */
	cur_cpu_num = -1;
	rtn_s = NULL;
	while(fgets(buf, sizeof(buf), fp) != NULL)
	{
		if(STRPFX(buf, "processor"))
		{
			gchar *s = buf + strlen("processor");
			while((*s != '\0') && !isdigit(*s))
				s++;
			cur_cpu_num = atoi(s);
		}
		else if(STRPFX(buf, "vendor_id"))
		{
			if((cur_cpu_num == cpu_num) && (cur_cpu_num > -1))
			{
				gchar   *s = buf,
							*s_end;
				while((*s != '\0') && (*s != ':'))
					s++;
				if(*s == ':')
					s++;
				while(ISBLANK(*s))
					s++;

				s_end = strchr(s, '\n');
				if(s_end != NULL)
					*s_end = '\0';

				g_free(rtn_s);
				rtn_s = g_strdup(s);

				break;
			}
		}
	}

	(void)fclose(fp);

	return(rtn_s);
}

/*
 *	Gets the CPU's model name.
 */
gchar *si_cpu_get_model(const gint cpu_num)
{
	gint cur_cpu_num;
	gchar		buf[1024],
			*rtn_s;

	/* Open the cpu information file for reading */
	FILE *fp = fopen("/proc/cpuinfo", "rb");
	if(fp == NULL)
		return(NULL);

	/* Read each line until we get to the line that contains the
	 * information we want
	 */
	cur_cpu_num = -1;
	rtn_s = NULL;
	while(fgets(buf, sizeof(buf), fp) != NULL)
	{
		if(STRPFX(buf, "processor"))
		{
			gchar *s = buf + strlen("processor");
			while((*s != '\0') && !isdigit(*s))
				s++;
			cur_cpu_num = atoi(s);
		}
		else if(STRPFX(buf, "model name"))
		{
			if((cur_cpu_num == cpu_num) && (cur_cpu_num > -1))
			{
				gchar   *s = buf,
					*s_end;
				while((*s != '\0') && (*s != ':'))
					s++;
				if(*s == ':')
					s++;
				while(ISBLANK(*s))
					s++;

				s_end = (gchar *)strchr((char *)s, '\n');
				if(s_end != NULL)
					*s_end = '\0';

				g_free(rtn_s);
				rtn_s = g_strdup(s);

				break;
			}
		}
	}

	(void)fclose(fp);

	return(rtn_s);
}

/*
 *	Gets the CPU details message.
 */
gchar *si_cpu_get_details_message(const gint cpu_num)
{
	gint cur_cpu_num;
	gchar		buf[1024],
			*msg;

	/* Open the cpu information file for reading */
	FILE *fp = fopen("/proc/cpuinfo", "rb");
	if(fp == NULL)
	{
		const gint error_code = (gint)errno;
		return(g_strdup_printf(
"%s:\n\n\
    %s",
			g_strerror(error_code),
			"/proc/cpuinfo"
		));
	}

	/* Read each line until we get to the line that contains the
	 * information we want
	 */
	cur_cpu_num = -1;
	msg = g_strdup("");
	while(fgets(buf, sizeof(buf), fp) != NULL)
	{
		if(STRPFX(buf, "processor"))
		{
			gchar *s = buf + strlen("processor");
			while((*s != '\0') && !isdigit(*s))
				s++;
			cur_cpu_num = atoi(s);
		}

		if((cpu_num < 0) ||
		   ((cur_cpu_num == cpu_num) && (cur_cpu_num > -1))
		)
		{
			gchar *s = g_strconcat(
				msg,
				buf,
				NULL
			);
			if(s != NULL)
			{
				g_free(msg);
				msg = s;
			}
		}
	}

	(void)fclose(fp);

	return(msg);
}
