#include <stdlib.h>
#include <signal.h>
#include <gtk/gtk.h>

#include "../guiutils.h"
#include "si_cb.h"
#include "si_win.h"
#include "config.h"


static gint si(const gint argc, const gchar **argv);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	SysInfo.
 */
static gint si(const gint argc, const gchar **argv)
{
	gboolean	initialized_gtk = FALSE,
			show_border = TRUE,
			show_title = TRUE;
	gint		i,
			cpu_num = 0,
			bar_length = SYSINFO_DEF_BAR_LENGTH;
	const gchar *arg;
	guint id;
	gulong poll_interval = SYSINFO_DEF_POLL_INT;
	GtkOrientation orientation = SYSINFO_DEF_ORIENTATION;
	SIWinDisplay display =	SI_WIN_DISPLAY_CPU_LOAD |
				SI_WIN_DISPLAY_CPU_LOADAVG |
				SI_WIN_DISPLAY_MEMORY |
				SI_WIN_DISPLAY_APM_POWER;
	SIWinBarStyle bar_style = SI_WIN_BAR_STYLE_CONTINUOUS;
	SIWin *si;

#define CLEANUP_RETURN(_v_)	{	\
					\
 return(_v_);				\
}

	/* Set up the UNIX signal callbacks */
#ifdef SIGINT
	signal(SIGINT, si_signal_cb);
#endif
#ifdef SIGTERM
	signal(SIGTERM, si_signal_cb);
#endif

	/* Handle the command line arguments */
	for(i = 1; i < argc; i++)
	{
		arg = argv[i];
		if(arg == NULL)
			continue;

		/* No Border */
		if(!g_strcasecmp(arg, "--no-border") ||
		   !g_strcasecmp(arg, "-no-border") || 
		   !g_strcasecmp(arg, "--no-borders") || 
		   !g_strcasecmp(arg, "-no-borders") || 
		   !g_strcasecmp(arg, "--w") || 
		   !g_strcasecmp(arg, "-w")
		)
		{
			show_border = FALSE;
		}
		/* No Title Bar */
		else if(!g_strcasecmp(arg, "--no-title-bar") ||
		        !g_strcasecmp(arg, "--no-title") ||
		        !g_strcasecmp(arg, "-no-title-bar") ||
		        !g_strcasecmp(arg, "-no-title") ||
		        !g_strcasecmp(arg, "--t") ||
		        !g_strcasecmp(arg, "-t")
		)
		{
			show_title = FALSE;
		}
		/* Display */
		else if(!g_strcasecmp(arg, "-d"))
		{
			display = 0;		/* Reset display flags */
			i++;
			arg = (i < argc) ? argv[i] : NULL;
			while((arg != NULL) ? (*arg != '-') : FALSE)
			{
				if(!g_strcasecmp(arg, "load"))
					display |= SI_WIN_DISPLAY_CPU_LOAD;
				else if(!g_strcasecmp(arg, "loadavg") ||
					!g_strcasecmp(arg, "avg") ||
					!g_strcasecmp(arg, "average")
				)
					display |= SI_WIN_DISPLAY_CPU_LOADAVG;
				else if(!g_strcasecmp(arg, "memory") ||
					!g_strcasecmp(arg, "mem")
				)
					display |= SI_WIN_DISPLAY_MEMORY;
				else if(!g_strcasecmp(arg, "apmpower") ||
					!g_strcasecmp(arg, "apm")
				)
					display |= SI_WIN_DISPLAY_APM_POWER;
				else if(!g_strcasecmp(arg, "details"))
					display |= SI_WIN_DISPLAY_CPU_DETAILS;
				else
					g_printerr(
"-d: Warning: Unsupported display flag \"%s\".\n",
						arg
					);

				i++;
				arg = (i < argc) ? argv[i] : NULL;
			}
			i--;
		}
		/* Bar Style */
		else if(!g_strcasecmp(arg, "--style") ||
			!g_strcasecmp(arg, "-style") ||
			!g_strcasecmp(arg, "-s")
		)
		{
			i++;
			arg = (i < argc) ? argv[i] : NULL;
			if(!STRISEMPTY(arg))
			{
				if(*arg == 'c')
					bar_style = SI_WIN_BAR_STYLE_CONTINUOUS;
				else if(*arg == 'd')
					bar_style = SI_WIN_BAR_STYLE_DISCRETE;
				else if(*arg == 'l')
					bar_style = SI_WIN_BAR_STYLE_LED;
				else
					g_printerr(
"%s: Warning: Unsupported argument \"%s\".\n",
						argv[i - 1], arg
					);
			}
		}
		/* Orientation */
		else if(!g_strcasecmp(arg, "--orientation") ||
			!g_strcasecmp(arg, "-orientation") ||
			!g_strcasecmp(arg, "-o")
		)
		{
			i++;
			arg = (i < argc) ? argv[i] : NULL;
			if(!STRISEMPTY(arg))
			{
				if(*arg == 'h')
					orientation = GTK_ORIENTATION_HORIZONTAL;
				else if(*arg == 'v')
					orientation = GTK_ORIENTATION_VERTICAL;
				else
					g_printerr(
"%s: Warning: Unsupported argument \"%s\".\n",
						argv[i - 1], arg
					);
			}
		}
		/* Bar Length */
		else if(!g_strcasecmp(arg, "--length") ||
			!g_strcasecmp(arg, "-length") ||
			!g_strcasecmp(arg, "-l")
		)
		{
			i++;
			arg = (i < argc) ? argv[i] : NULL;
			if(!STRISEMPTY(arg))
			{
				bar_length = ATOI(arg);
				if(bar_length < 1)
					bar_length = 1;
			}
		}
		/* Poll Interval */
		else if(!g_strcasecmp(arg, "--interval") ||
			!g_strcasecmp(arg, "-interval") ||
			!g_strcasecmp(arg, "-i")
		)
		{
			i++;
			arg = (i < argc) ? argv[i] : NULL;
			if(!STRISEMPTY(arg))
			{
				poll_interval = ATOL(arg);
			}
		}
		/* All CPUs */
		else if(!g_strcasecmp(arg, "-1"))
		{
			cpu_num = -1;
		}
		/* Skip these arguments so that gtk_window_apply_args()
		 * handles them
		 */
		else if(gtk_is_window_arg(arg))
		{
			i++;
		}
		/* Single character argument? */
		else if((*arg == '-') ? (arg[1] != '-') : FALSE)
		{
			const gchar *v = arg + 1;

			while(*v != '\0')
			{
#if 0
				switch(*v)
				{
				    case 's':
					auto_start = TRUE;
					break;
				    default:
					g_printerr(
"-%c: Unsupported argument.\n",
						c
					);
					CLEANUP_RETURN(-2);
					break;
				}
#endif
				v++;
			}
		}
		/* Unsupported argument */
		else if(*arg == '-')
		{
			g_printerr(
"%s: Unsupported argument.\n",
				arg
			);
			CLEANUP_RETURN(2);
		}
		/* CPU */
		else
		{
			cpu_num = ATOI(arg);
		}
	}


	/* Initialize GTK as needed */
	if(!initialized_gtk)
	{
		if(!gtk_init_check((gint *)&argc, (gchar ***)&argv))
		{
			g_printerr("Unable to initialize GTK.\n");
			CLEANUP_RETURN(1);
		}
		initialized_gtk = TRUE;

		/* Initialize GDK RGB buffers system */
		gdk_rgb_init();
	}


	/* Create the SysInfo Window */
	si = si_win_new(
		cpu_num,
		orientation,
		show_border,
		show_title,
		display,
		bar_style,
		bar_length
	);
	gtk_window_apply_args(
		GTK_WINDOW(si->toplevel),
		(gint)argc, (gchar **)argv
	);
	si_win_map(si);

	id = gtk_timeout_add(
		MAX(poll_interval, 10l),
		si_timeout_cb, si
	);

	gtk_main();


	/* Begin deleting */

	si_win_delete(si);

	CLEANUP_RETURN(0);
#undef CLEANUP_RETURN
}


int main(int argc, char *argv[])
{
	gint		i,
			status;
	const gchar *arg;

	/* Handle the basic command line arguments */
	for(i = 1; i < argc; i++)
	{
		arg = (const gchar *)argv[i];
		if(arg == NULL)
			continue;

		/* Help */
		if(!g_strcasecmp(arg, "--help") ||
		   !g_strcasecmp(arg, "-help") ||
		   !g_strcasecmp(arg, "--h") ||
		   !g_strcasecmp(arg, "-h") ||
		   !g_strcasecmp(arg, "-?")
		)
		{
			g_print("%s", PROG_HELP_MESG);
			return(0);
		}
		/* Version */
		else if(!g_strcasecmp(arg, "--version") ||
			!g_strcasecmp(arg, "-version")
		)
		{
			g_print(
				"%s",
				PROG_NAME " Version " PROG_VERSION "\n"
				PROG_COPYRIGHT
			);
			return(0);
		}
	}

	/* SysInfo */
	status = si((gint)argc, (const gchar **)argv);

	/* Return value must be positive */
	if(status < 0)
		status = -status;

	return(status);
}
