#include <string.h>
#include <gtk/gtk.h>
#include "guiutils.h"
#include "statictip.h"


typedef struct _StaticTip		StaticTip;
#define STATIC_TIP(p)			((StaticTip *)(p))
#define STATIC_TIP_KEY			"/StaticTip"


static void StaticTipDestroyCB(gpointer data);
static gint StaticTipExposeEventCB(
	GtkWidget *widget, GdkEventExpose *expose, gpointer data
);
static void StaticTipDrawCB(
	GtkWidget *widget, GdkRectangle *rect, gpointer data
);
static void StaticTipDraw(StaticTip *t);

static StaticTip *StaticTipNew(void);

void StaticTipSet(
	GtkWidget *w,
	const gchar *text,
	const StaticTipAlign x_align, const StaticTipAlign y_align, 
	const gint x, const gint y
);


#define STATIC_TIP_BORDER_MINOR		4


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Static Tip:
 */
struct _StaticTip {
	GtkWidget	*toplevel;		/* GtkWindow */
	gchar		*text;
};


/*
 *	Static Tip reference GtkWidget "destroy" signal callback.
 */
static void StaticTipDestroyCB(gpointer data)
{
	StaticTip *t = STATIC_TIP(data);
	if(t == NULL)
		return;

	GTK_WIDGET_DESTROY(t->toplevel);
	g_free(t->text);
	g_free(t);
}

/*
 *	Static Tip toplevel GtkWindow "expose_event" signal callback.
 */
static gint StaticTipExposeEventCB(
	GtkWidget *widget, GdkEventExpose *expose, gpointer data
)
{
	StaticTip *t = STATIC_TIP(data);
	if((widget == NULL) || (t == NULL))
		return(FALSE);

	StaticTipDraw(t);

	return(TRUE);
}

/*
 *	Static Tip toplevel GtkWindow "draw" signal callback.
 */
static void StaticTipDrawCB(
	GtkWidget *widget, GdkRectangle *rect, gpointer data
)
{
	StaticTip *t = STATIC_TIP(data);
	if((widget == NULL) || (t == NULL))
		return;

	StaticTipDraw(t);
}

/*
 *	Redraws the Static Tip.
 */
static void StaticTipDraw(StaticTip *t)
{
	const gint border_minor = STATIC_TIP_BORDER_MINOR;
	GdkWindow *window;
	GtkStateType state;
	GtkStyle *style;
	GtkWidget *w;

	if(t == NULL)
		return;

	w = t->toplevel;
	window = w->window;
	state = GTK_WIDGET_STATE(w);
	style = gtk_widget_get_style(w);
	if((window == NULL) || (style == NULL))
		return;

	/* Draw the background */
	gtk_paint_flat_box(
		style,
		window,
		state,
		GTK_SHADOW_OUT,
		NULL,
		w,
		"tooltip",
		0, 0,
		-1, -1
	);

	/* Draw the text */
	if((style->font != NULL) && (t->text != NULL))
	{
		gint	x = border_minor,
			y = border_minor,
			line_len;
		const gchar	*s = t->text,
				*s_line_end;
		GdkFont *font = style->font;
		const gint font_height = font->ascent + font->descent;

		do
		{
			/* Calculate the length of this line */
			s_line_end = (const gchar *)strchr((const char *)s, '\n');
			if(s_line_end != NULL)
				line_len = s_line_end - s;
			else
				line_len = (gint)strlen((const char *)s);
			if(line_len > 0)
			{
				gchar *sd = (gchar *)g_malloc((line_len + 1) * sizeof(gchar));
				if(sd != NULL)
				{
					GdkTextBounds b;
					(void)memcpy(
						sd,
						s,
						(size_t)(line_len * sizeof(gchar))
					);
					sd[line_len] = '\0';

					/* Calculate the size of this line */
					gdk_string_bounds(
						font,
						sd,
						&b
					);

					/* Draw this line */
					gtk_paint_string(
						style,
						window,
						state,
						NULL,
						w,
						"tooltip",
						x,
						y + font->ascent,
						sd
					);

					g_free(sd);
				}
			}

			/* Increment the y coordinate position */
			y += font_height;

			/* Seek to the next line or end of string */
			s += line_len;
			if(*s == '\n')
				s++;

		} while(*s != '\0');
	}
}


/*
 *	Creates a new Static Tip.
 */
static StaticTip *StaticTipNew(void)
{
	GtkWidget *w;
	StaticTip *t = STATIC_TIP(g_malloc0(
		sizeof(StaticTip)
	));
	if(t == NULL)
		return(NULL);

	t->toplevel = w = gtk_window_new(GTK_WINDOW_POPUP);
	gtk_window_set_policy(GTK_WINDOW(w), FALSE, FALSE, FALSE);
	gtk_widget_set_name(w, STATIC_TIP_WIDGET_NAME);
	gtk_widget_set_app_paintable(w, TRUE);
	gtk_widget_add_events(
		w,
		GDK_STRUCTURE_MASK | GDK_EXPOSURE_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "expose_event",
		GTK_SIGNAL_FUNC(StaticTipExposeEventCB), t
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "draw",
		GTK_SIGNAL_FUNC(StaticTipDrawCB), t
	);
	gtk_widget_realize(w);

	return(t);
}


/*
 *	Sets the Static Tip on the widget.
 *
 *	The w specifies the GtkWidget to set the Static Tip on. If
 *	the GtkWidget already has a Static Tip set on it then its
 *	Static Tip will only be updated.
 *
 *	The text specifies the string describing the Static Tip text,
 *	which can contain multiple lines. If text is NULL then the
 *	GtkWidget's Static Tip will be removed.
 *
 *	The x_align and y_align specifies the positional alignment
 *	options.
 *
 *	The x and y specifies the position of the Static Tip relative
 *	to the desktop if x_align or y_align is set to
 *	STATIC_TIP_ALIGN_ROOT_VALUE, otherwise x and y specifies the
 *	offset.
 */
void StaticTipSet(
	GtkWidget *w,
	const gchar *text,
	const StaticTipAlign x_align, const StaticTipAlign y_align, 
	const gint x, const gint y
)
{
	const gint border_minor = STATIC_TIP_BORDER_MINOR;
	gint		x1, y1,
			x2, y2,
			width, height,
			root_width, root_height;
	GdkFont *font;
	GdkWindow	*window,
			*root;
	GtkAllocation *allocation;
	GtkStyle *style;
	GtkWidget *toplevel;
	StaticTip *t;

	if(w == NULL)
		return;

	/* Remove the static tip from this GtkWidget? */
	if(text == NULL)
	{
		gtk_object_remove_data(GTK_OBJECT(w), STATIC_TIP_KEY);
		return;
	}

	/* Get/create the Static Tip for this GtkWidget */
	t = STATIC_TIP(GTK_OBJECT_GET_DATA(
		w,
		STATIC_TIP_KEY
	));
	if(t == NULL)
	{
		t = StaticTipNew();
		if(t == NULL)
			return;

		gtk_object_set_data_full(
			GTK_OBJECT(w), STATIC_TIP_KEY,
			t, StaticTipDestroyCB
		);
	}

	toplevel = t->toplevel;
	if(toplevel == NULL)
		return;

	style = gtk_widget_get_style(toplevel);
	if(style == NULL)
		return;

	font = style->font;

	/* Set the text */
	g_free(t->text);
	t->text = STRDUP(text);

	/* Get the specified GtkWidget's window and geometry */
	window = w->window;
	allocation = &w->allocation;


	/* Calculate the size of the Static Tip */
	width = 0;
	height = 0;
	/* Add the text size to the size */
	if(t->text != NULL)
	{
		gint line_len;
		const gchar	*s = t->text,
				*s_line_end;
		GdkFont *font = style->font;
		const gint font_height = font->ascent + font->descent;
		GdkTextBounds b;

		do
		{
			/* Calculate the length of this line */
			s_line_end = (const gchar *)strchr((const char *)s, '\n');
			if(s_line_end != NULL)
				line_len = s_line_end - s;
			else
				line_len = (gint)strlen((const char *)s);

			/* Calculate the size of this line */
			gdk_text_bounds(
				font,
				s,
				line_len,
				&b
			);

			/* Add the size of this line */
			if(b.width > width)
				width = b.width;
			height += font_height;

			/* Seek to the next line or end of string */
			s += line_len;
			if(*s == '\n')
				s++;

		} while(*s != '\0');
	}
	/* Add the borders to the size */
	width += 2 * border_minor;
	height += 2 * border_minor;

	/* Get the size of the root window */
	root = gdk_window_get_root();
	if(root != NULL)
	{
		gdk_window_get_size(
			root,
			&root_width, &root_height
		);
	}
	else
	{
		root_width = 0;
		root_height = 0;
	}

	/* Calculate the x position */
	x1 = x;
	switch(x_align)
	{
	  case STATIC_TIP_ALIGN_POINTER_MIN:
		gdk_window_get_pointer(NULL, &x1, NULL, NULL);
		x1 -= width + border_minor;
		x1 += x;
		break;
	  case STATIC_TIP_ALIGN_POINTER_CENTER:
		gdk_window_get_pointer(NULL, &x1, NULL, NULL);
		x1 -= width / 2;
		x1 += x;
		break;
	  case STATIC_TIP_ALIGN_POINTER_MAX:
		gdk_window_get_pointer(NULL, &x1, NULL, NULL);
		x1 += width + border_minor;
		x1 += x;
		break;

	  case STATIC_TIP_ALIGN_WIDGET_MIN:
		gdk_window_get_origin(window, &x1, NULL);
		x1 -= width + border_minor;
		if(GTK_WIDGET_NO_WINDOW(w))
			x1 += allocation->x;
		x1 += x;
		/* If the position places the the window off the root
		 * window then recalculate the position as
		 * STATIC_TIP_ALIGN_WIDGET_MAX
		 */
		if(x1 < 0)
		{
	                gdk_window_get_origin(window, &x1, NULL);
		        x1 += allocation->width + border_minor;
	                if(GTK_WIDGET_NO_WINDOW(w))
		                x1 += allocation->x;
			x1 += x;
		}
		break;
	  case STATIC_TIP_ALIGN_WIDGET_CENTER:
		gdk_window_get_origin(window, &x1, NULL);
		x1 += (allocation->width / 2) - (width / 2);
		if(GTK_WIDGET_NO_WINDOW(w))
			x1 += allocation->x;
		x1 += x;
		break;
	  case STATIC_TIP_ALIGN_WIDGET_MAX:
		gdk_window_get_origin(window, &x1, NULL);
		x1 += allocation->width + border_minor;
		if(GTK_WIDGET_NO_WINDOW(w))
			x1 += allocation->x;
		x1 += x;
		/* If the position places the the window off the root
		 * window then recalculate the position as
		 * STATIC_TIP_ALIGN_WIDGET_MIN
		 */
		if((root_width > 0) && ((x1 + width) > root_width))
		{
			gdk_window_get_origin(window, &x1, NULL);
			x1 -= width + border_minor;
			if(GTK_WIDGET_NO_WINDOW(w))
				x1 += allocation->x;
			x1 += x;
		}
		break;

	  case STATIC_TIP_ALIGN_WIDGET_VALUE:
		gdk_window_get_origin(window, &x2, NULL);
		x1 += x2;
		if(GTK_WIDGET_NO_WINDOW(w))
			x1 += allocation->x;
		break;
	  case STATIC_TIP_ALIGN_ROOT_VALUE:
		break;
	}

	/* Calculate the y position */
	y1 = y;
	switch(y_align)
	{
	  case STATIC_TIP_ALIGN_POINTER_MIN:
		gdk_window_get_pointer(NULL, NULL, &y1, NULL);
		y1 -= height + border_minor;
		y1 += y;
		break;
	  case STATIC_TIP_ALIGN_POINTER_CENTER:
		gdk_window_get_pointer(NULL, NULL, &y1, NULL);
		y1 -= height / 2;
		y1 += y;
		break;
	  case STATIC_TIP_ALIGN_POINTER_MAX:
		gdk_window_get_pointer(NULL, NULL, &y1, NULL);
		y1 += height + border_minor;
		y1 += y;
		break;

	  case STATIC_TIP_ALIGN_WIDGET_MIN:
		gdk_window_get_origin(window, NULL, &y1);
		y1 -= height + border_minor;
		if(GTK_WIDGET_NO_WINDOW(w))
			y1 += allocation->y;
		y1 += y;
		/* If the position places the the window off the root
		 * window then recalculate the position as
		 * STATIC_TIP_ALIGN_WIDGET_MAX
		 */
		if(y1 < 0)
		{
			gdk_window_get_origin(window, NULL, &y1);
			y1 += allocation->height + border_minor;
			if(GTK_WIDGET_NO_WINDOW(w))
				y1 += allocation->y;
			y1 += y;
		}
		break;
	  case STATIC_TIP_ALIGN_WIDGET_CENTER:
		gdk_window_get_origin(window, NULL, &y1);
		y1 += (allocation->height / 2) - (height / 2);
		if(GTK_WIDGET_NO_WINDOW(w))
			y1 += allocation->y;
		y1 += y;
		break;
	  case STATIC_TIP_ALIGN_WIDGET_MAX:
		gdk_window_get_origin(window, NULL, &y1);
		y1 += allocation->height + border_minor;
		if(GTK_WIDGET_NO_WINDOW(w))
			y1 += allocation->y;
		y1 += y;
		/* If the position places the the window off the root
		 * window then recalculate the position as
		 * STATIC_TIP_ALIGN_WIDGET_MIN
		 */
		if((root_height > 0) && ((y1 + height) > root_height))
		{
			gdk_window_get_origin(window, NULL, &y1);
			y1 -= height + border_minor;
			if(GTK_WIDGET_NO_WINDOW(w))
				y1 += allocation->y;
			y1 += y;
		}
		break;

	  case STATIC_TIP_ALIGN_WIDGET_VALUE:
		gdk_window_get_origin(window, NULL, &y2);
		y1 += y2;
		if(GTK_WIDGET_NO_WINDOW(w))
			y1 += allocation->y;
		break;
	  case STATIC_TIP_ALIGN_ROOT_VALUE:
		break;
	}


#if 1
	/* Set the new geometry and map toplevel as needed */
	gtk_widget_set_usize(toplevel, width, height);
	if(GTK_WIDGET_MAPPED(toplevel))
		gtk_widget_set_uposition(toplevel, x1, y1);
	else
		gtk_widget_popup(toplevel, x1, y1);
#else
	/* Set the new geometry and map toplevel as needed */
	if(GTK_WIDGET_MAPPED(toplevel))
	{
		gtk_widget_set_usize(toplevel, width, height);
		gtk_widget_set_uposition(toplevel, x1, y1);
		gtk_widget_queue_resize(toplevel);
	}
	else
	{
		gtk_widget_set_usize(toplevel, width, height);
		gtk_widget_popup(toplevel, x1, y1);
	}
#endif

	gtk_widget_queue_draw(toplevel);
}
