#include <stdlib.h>
#include <string.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include "guiutils.h"

#include "stacklist.h"

#include "images/icon_add_20x20.xpm"
#include "images/icon_remove_20x20.xpm"


#ifndef DEBUG_LEVEL
# define DEBUG_LEVEL	0
#endif


/* Callbacks */
static gint StackListCListEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void StackListDragDataGetCB(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
static void StackListDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
static void StackListDragDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
);
static void StackListSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
static void StackListUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
static void StackListAddCB(GtkWidget *widget, gpointer data);
static void StackListRemoveCB(GtkWidget *widget, gpointer data);
static void StackListShiftUpCB(GtkWidget *widget, gpointer data);
static void StackListShiftDownCB(GtkWidget *widget, gpointer data);

/* Utilities */
static stack_list_item_struct *StackListMatchItemByID(
	stack_list_struct *slist, const gint id
);
static void StackListSetDescriptionFromItem(
	stack_list_struct *slist, stack_list_item_struct *item
);

static gint StackListInsertTar(
	stack_list_struct *slist,
	const gint row,
	stack_list_item_struct *item
);
static gint StackListAppendTar(
	stack_list_struct *slist,
	stack_list_item_struct *item
);

/* Operations */
void StackListAdd(stack_list_struct *slist);
void StackListRemove(stack_list_struct *slist);
void StackListShiftUp(stack_list_struct *slist);
void StackListShiftDown(stack_list_struct *slist);

/* Stack List Item */
gint StackListAppend(
	stack_list_struct *slist,
	const gchar *name,
	const gchar *description,
	guint8 **icon_data,
	gpointer data,
	const gint id,
	const gboolean allow_multiple,
	const gboolean stay_on_target
);
void StackListClear(stack_list_struct *slist);

gint StackListItemAppendSrc(stack_list_struct *slist, const gint id);
gint StackListItemAppendTar(stack_list_struct *slist, const gint id);
void StackListItemSetAllFromCacheSrc(stack_list_struct *slist);

void StackListItemRemoveByIDSrc(
	stack_list_struct *slist, const gint id,
	const gboolean exclude_allowed_multiples
);
void StackListItemRemoveByIDTar(
	stack_list_struct *slist, const gint id,
	const gboolean exclude_allowed_multiples
);

void StackListItemClearSrc(stack_list_struct *slist);
void StackListItemClearTar(stack_list_struct *slist);

gint *StackListItemGetSrc(stack_list_struct *slist, gint *total);
gint *StackListItemGetTar(stack_list_struct *slist, gint *total);


/* Stack List */
stack_list_struct *StackListNew(
	GtkWidget *parent,
	const gchar *src_title,
	const gchar *tar_title
);
void StackListShowDescription(
	stack_list_struct *slist,
	gboolean show_src_desc,
	gboolean show_tar_desc
);
void StackListSetMaintainSourceOrder(
	stack_list_struct *slist, gboolean maintain_source_order
);
void StackListSetChangedCB(
	stack_list_struct *slist, 
	void (*cb)(
		stack_list_struct *,		/* Stack List */
		gpointer			/* Data */
	),
	gpointer data   
);
void StackListSetAddedCB(
	stack_list_struct *slist,
	void (*cb)(
		stack_list_struct *,		/* Stack List */
		stack_list_item_struct *,	/* Stack List Item */
		gpointer			/* Data */
	),
	gpointer data
);
void StackListSetRemovedCB(
	stack_list_struct *slist, 
	void (*cb)(
		stack_list_struct *,		/* Stack List */
		stack_list_item_struct *,	/* Stack List Item */
		gpointer			/* Data */
	),
	gpointer data
);
void StackListUpdateMenus(stack_list_struct *slist);
gboolean StackListMapped(stack_list_struct *slist);
void StackListMap(stack_list_struct *slist);
void StackListUnmap(stack_list_struct *slist);
void StackListDelete(stack_list_struct *slist);


#define STACK_LIST_ROW_SPACING			20


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	GtkCList event signal callback.
 */
static gint StackListCListEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gboolean status = FALSE;
	gint etype, x, y, row, column;
	gboolean press;
	guint keyval, state;
	GdkEventKey *key;
	GdkEventCrossing *crossing;
	GdkEventButton *button;
	GdkEventMotion *motion;
	GtkCList *clist;
	stack_list_flags flags;
	stack_list_struct *slist = STACK_LIST(data);
	if((widget == NULL) || (event == NULL) || (slist == NULL))
	    return(status);

	if(slist->freeze_count > 0)
	    return(status);

	clist = GTK_CLIST(widget);
	if(clist->clist_window != ((GdkEventAny *)event)->window)
	    return(status);

	flags = slist->flags;

	etype = event->type;
	switch(etype)
	{
	  case GDK_KEY_PRESS:
	  case GDK_KEY_RELEASE:
#define STOP_KEY_SIGNAL_EMIT(_w_)	{	\
 gtk_signal_emit_stop_by_name(			\
  GTK_OBJECT(_w_),				\
  press ?					\
   "key_press_event" : "key_release_event"	\
 );						\
}
	    key = (GdkEventKey *)event;
	    press = (etype == GDK_KEY_PRESS) ? TRUE : FALSE;
	    keyval = key->keyval;
	    state = key->state;
#if (DEBUG_LEVEL >= 1)
g_print(
 "StackListCListEventCB(): slist=0x%.8x %s keyval=0x%.8x(%c) state=0x%.8x\n",
 (guint32)slist,
 press ? "key_press_event" : "key_release_event",
 (guint32)keyval, (gchar)keyval,
 (guint32)state
);
#endif
	    switch(keyval)
	    {
	      case GDK_Return:
	      case GDK_KP_Enter:
	      case GDK_ISO_Enter:
	      case GDK_3270_Enter:
		if(press)
		{
		    /* Check which GtkCList this event is for
		     *
		     * Source GtkCList
		     */
		    if(widget == slist->src_clist)
		    {
			/* Add */
			StackListAdd(slist);
		    }
		    /* Target GtkCList */
		    else if(widget == slist->tar_clist)
		    {
			/* Remove */
			StackListRemove(slist);
		    }
		}
		STOP_KEY_SIGNAL_EMIT(widget);
		status = TRUE;
		break;

	      case GDK_plus:
	      case GDK_equal:
	      case GDK_KP_Add:
	      case GDK_Insert:
	      case GDK_greater:
		if(press)
		    StackListAdd(slist);
		STOP_KEY_SIGNAL_EMIT(widget);
		status = TRUE;
		break;

	      case GDK_minus:
	      case GDK_underscore:
	      case GDK_KP_Subtract:
	      case GDK_Delete:
	      case GDK_less:
		if(press)
		    StackListRemove(slist);
		STOP_KEY_SIGNAL_EMIT(widget);
		status = TRUE;
		break;


	    }
	    break;
#undef STOP_KEY_SIGNAL_EMIT

	  case GDK_ENTER_NOTIFY:
	    crossing = (GdkEventCrossing *)event;

	    /* Clear the description */
	    StackListSetDescriptionFromItem(slist, NULL);

	    status = TRUE;
	    break;

	  case GDK_LEAVE_NOTIFY:
	    crossing = (GdkEventCrossing *)event;

	    /* Clear the description */ 
	    StackListSetDescriptionFromItem(slist, NULL);

	    status = TRUE;
	    break;

	  case GDK_BUTTON_PRESS:
	    button = (GdkEventButton *)event;
	    x = (gint)button->x;
	    y = (gint)button->y;
	    slist->button = button->button;

	    /* Handle by button number */
	    switch(button->button)
	    {
	      case GDK_BUTTON2:
		gdk_pointer_grab(
		    button->window,
		    FALSE,
		    GDK_BUTTON_PRESS_MASK |
		    GDK_BUTTON_RELEASE_MASK |
		    GDK_POINTER_MOTION_MASK |
		    GDK_POINTER_MOTION_HINT_MASK,
		    NULL,
		    slist->translate_cur,
		    button->time
		);
		slist->drag_last_x = (gint)button->x;
		slist->drag_last_y = (gint)button->y;
		status = TRUE;
		break;

	      case GDK_BUTTON3:
		/* Get the row and column that this button press
		 * occured over
		 */
		if(!gtk_clist_get_selection_info(
		    clist,
		    x, y,
		    &row, &column
		))
		{
		    row = -1;
		    column = 0;
		}

		/* Select the row */
		gtk_clist_freeze(clist);
		if(!(button->state & GDK_CONTROL_MASK) &&
		   !(button->state & GDK_SHIFT_MASK)
		)
		    gtk_clist_unselect_all(clist);
		if(row > -1)
		    gtk_clist_select_row(clist, row, column);
		gtk_clist_thaw(clist);

		/* Map the right-click popup menu based on which
		 * GtkCList this event is for
		 *
		 * Source GtkCList
		 */
		if(widget == slist->src_clist)
		{
		    gtk_menu_popup(
			GTK_MENU(slist->src_menu),
			NULL, NULL,
			NULL, NULL,
			button->button, button->time
		    );
		}
		/* Target GtkCList */
		else if(widget == slist->tar_clist)
		{
		    gtk_menu_popup(
			GTK_MENU(slist->tar_menu),
			NULL, NULL,
			NULL, NULL,
			button->button, button->time
		    );
		}
		status = TRUE;
		break;
	    }
	    break;

	  case GDK_BUTTON_RELEASE:
	    button = (GdkEventButton *)event;
	    switch(button->button)
	    {
	      case GDK_BUTTON2:
		if(slist->button > 0)
		    status = TRUE;
		if(gdk_pointer_is_grabbed())
		    gdk_pointer_ungrab(button->time);
		break;
	    }
	    slist->button = 0;
	    slist->drag_last_x = 0;
	    slist->drag_last_y = 0;
	    break;

	  case GDK_MOTION_NOTIFY:
	    motion = (GdkEventMotion *)event;
	    x = (gint)motion->x;
	    y = (gint)motion->y;

#define SCROLL(_adj_,_v_)	{				\
 if((_v_) != 0.0f) {						\
  if(((_adj_)->upper - (_adj_)->page_size) > (_adj_)->lower) {	\
   gtk_adjustment_set_value(					\
    (_adj_),							\
    CLIP(							\
     (_adj_)->value - (_v_),					\
     (_adj_)->lower, ((_adj_)->upper - (_adj_)->page_size)	\
    )								\
   );								\
  }								\
 }								\
}

	    /* Check which GtkCList this event is for
	     *
	     * Source GtkCList
	     */
	    if((widget == slist->src_clist) &&
	       (flags & STACK_LIST_SHOW_DESCRIPTION_SOURCE)
	    )
	    {
		if(slist->button == GDK_BUTTON2)
		{
		    gint dx, dy;
		    GtkCList *clist = GTK_CLIST(slist->src_clist);
		    GtkAdjustment *hadj = clist->hadjustment,
				  *vadj = clist->vadjustment;
		    if(motion->is_hint)
		    {
			gint x, y;
			GdkModifierType mask;
			gdk_window_get_pointer(
			    motion->window, &x, &y, &mask
			);
			dx = x - slist->drag_last_x;
			dy = y - slist->drag_last_y;
			slist->drag_last_x = x;
			slist->drag_last_y = y;
		    }
		    else
		    {
			dx = (gint)(motion->x - slist->drag_last_x);
			slist->drag_last_x = (gint)motion->x;
			dy = (gint)(motion->y - slist->drag_last_y);
			slist->drag_last_y = (gint)motion->y;
		    }
		    SCROLL(hadj, dx);
		    SCROLL(vadj, dy);
		}
		else
		{
		    stack_list_item_struct *item;

		    /* Get the row and column that this motion occured
		     * over
		     */
		    if(!gtk_clist_get_selection_info(
			clist,
			x, y,
			&row, &column
		    ))
		    {
			row = -1;
			column = 0;
		    }

		    /* Get the item */
		    item = STACK_LIST_ITEM(gtk_clist_get_row_data(
			clist,
			row
		    ));

		    /* Set the description */
		    StackListSetDescriptionFromItem(slist, item);
		}
		status = TRUE;
	    }
	    /* Target GtkCList */
	    else if((widget == slist->tar_clist) &&  
		    (flags & STACK_LIST_SHOW_DESCRIPTION_TARGET)
	    )
	    {
		if(slist->button == GDK_BUTTON2)
		{
		    gint dx, dy;
		    GtkCList *clist = GTK_CLIST(slist->tar_clist);
		    GtkAdjustment *hadj = clist->hadjustment,
				  *vadj = clist->vadjustment;
		    if(motion->is_hint)
		    {
			gint x, y;
			GdkModifierType mask;
			gdk_window_get_pointer(
			    motion->window, &x, &y, &mask
			);
			dx = x - slist->drag_last_x;
			dy = y - slist->drag_last_y;
			slist->drag_last_x = x;
			slist->drag_last_y = y;
		    }
		    else
		    {
			dx = (gint)(motion->x - slist->drag_last_x);
			slist->drag_last_x = (gint)motion->x;
			dy = (gint)(motion->y - slist->drag_last_y);
			slist->drag_last_y = (gint)motion->y;
		    }
		    SCROLL(hadj, dx);
		    SCROLL(vadj, dy);
		}
		else
		{
		    stack_list_item_struct *item;

		    /* Get the row and column that this motion occured
		     * over
		     */
		    if(!gtk_clist_get_selection_info(
			clist,
			x, y,
			&row, &column
		    ))
		    {
			row = -1;
			column = 0;
		    }

		    /* Get the item */
		    item = STACK_LIST_ITEM(gtk_clist_get_row_data(
			clist,
			row
		    ));

		    /* Set the description */
		    StackListSetDescriptionFromItem(slist, item);
		}
		status = TRUE;
	    }

#undef SCROLL
	    break;
	}

	return(status);
}

/*
 *	DND "drag_data_get" signal callback.
 */
static void StackListDragDataGetCB(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gint buf_len;
	guint8 *buf;
	stack_list_struct *slist = STACK_LIST(data);
	if((widget == NULL) || (dc == NULL) || (slist == NULL))
	    return;

	if(slist->freeze_count > 0)
	    return;

	/* Respond with the source widget */
	buf_len = sizeof(GtkWidget *);
	buf = (guint8 *)g_malloc(buf_len + 1);
	if(buf == NULL)
	    return;

	*((GtkWidget **)buf) = widget;
	buf[buf_len] = '\0';

	gtk_selection_data_set(
	    selection_data,
	    GDK_SELECTION_TYPE_STRING,
	    8,				/* Bits Per Character */
	    buf,			/* Data */
	    buf_len			/* Length */
	);

	g_free(buf);
}

/*
 *	DND "drag_data_received" signal callback.
 */
static void StackListDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	stack_list_struct *slist = STACK_LIST(data);
	if((widget == NULL) || (dc == NULL) || (slist == NULL))
	    return;

	if(slist->freeze_count > 0)
	    return;

	if(selection_data == NULL)
	    return;

	if((selection_data->data == NULL) ||
	   (selection_data->length <= 0)
	)
	    return;

	/* Check which GtkCList this event is for
	 *
	 * Source GtkCList
	 */
	if(widget == slist->src_clist)
	{
	    if(info == STACK_LIST_ITEM_TAR_INFO)
	    {
		/* Get the data as a GtkWidget and handle it only if it
		 * refered to our target GtkCList
		 */
		GtkWidget *src_w = (selection_data->length >= sizeof(GtkWidget *)) ?
		    *((GtkWidget **)selection_data->data) : NULL;
		if(src_w == slist->tar_clist)
		{
		    /* Remove */
		    StackListRemove(slist);
		}
	    }
	}
	/* Target GtkCList */
	else if(widget == slist->tar_clist)
	{
	    if(info == STACK_LIST_ITEM_SRC_INFO)
	    {
		/* Get the data as a GtkWidget and handle it only if it
		 * refered to our source GtkCList
		 */
		GtkWidget *src_w = (selection_data->length >= sizeof(GtkWidget *)) ?
		    *((GtkWidget **)selection_data->data) : NULL;
		if(src_w == slist->src_clist)
		{
		    gint row = -1, column = 0;
		    GtkCList *clist = GTK_CLIST(widget);
		    GdkRectangle *column_title_area = &clist->column_title_area;

		    /* Find the dropped on row and column and select
		     * it for the upcomming add
		     */
		    gtk_clist_get_selection_info(
			clist,
			x,
			y - (GTK_CLIST_SHOW_TITLES(clist) ?
			    (column_title_area->y + column_title_area->height) : 0),
			&row, &column
		    );
		    gtk_clist_freeze(clist);
		    gtk_clist_unselect_all(clist);
		    if(row > -1)
			gtk_clist_select_row(clist, row, 0);
		    gtk_clist_thaw(clist);

		    /* Add */
		    StackListAdd(slist);
		}
	    }
	    else if(info == STACK_LIST_ITEM_TAR_INFO)
	    {
		/* Get the data as a GtkWidget and handle it only if it
		 * refered to our target GtkCList
		 */
		GtkWidget *src_w = (selection_data->length >= sizeof(GtkWidget *)) ?
		    *((GtkWidget **)selection_data->data) : NULL;
		if(src_w == slist->tar_clist)
		{
		    /* Reorder */
		    gint row = -1, column = 0, insert_row, new_row;
		    GList *glist, *sel_list;
		    GtkCList *clist = GTK_CLIST(widget);
		    GdkRectangle *column_title_area = &clist->column_title_area;
		    stack_list_item_struct *item;

		    /* Find the dropped on row */
		    gtk_clist_get_selection_info(
			clist,
			x,
			y - (GTK_CLIST_SHOW_TITLES(clist) ?
			    (column_title_area->y + column_title_area->height) : 0),
			&row, &column
		    );
		    insert_row = row;

		    /* Create a list of selected items */
		    sel_list = NULL;
		    for(glist = clist->selection;
			glist != NULL;
			glist = g_list_next(glist)
		    )
			sel_list = g_list_append(
			    sel_list,
			    gtk_clist_get_row_data(clist, (gint)glist->data)
			);

		    gtk_clist_freeze(clist);

		    /* Insert the selected items */
		    for(glist = sel_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			item = STACK_LIST_ITEM(glist->data);
			if(item == NULL)
			    continue;

			if(insert_row > -1)
			{
			    new_row = StackListInsertTar(
				slist,
				insert_row,
				item
			    );
			    insert_row++;
			}
			else
			{
			    new_row = StackListAppendTar(
				slist,
				item
			    );
			}
		    }
		    g_list_free(sel_list);

		    /* Remove all the selected rows */
		    while(clist->selection != NULL)
			gtk_clist_remove(
			    clist,
			    (gint)clist->selection->data
			);

		    gtk_clist_columns_autosize(clist);

		    gtk_clist_thaw(clist);

		    StackListUpdateMenus(slist);

		    /* Call the changed callback */
		    if(slist->changed_cb != NULL)
			slist->changed_cb(
			    slist,		/* Stack List */
			    slist->changed_data	/* Data */
			);
		}
	    }
	}
}

/*
 *	DND "drag_data_delete" signal callback.
 */
static void StackListDragDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{
	stack_list_struct *slist = STACK_LIST(data);
	if((widget == NULL) || (dc == NULL) || (slist == NULL))
	    return;

	if(slist->freeze_count > 0)
	    return;
}

/*
 *      GtkCList "select_row" signal callback.
 */
static void StackListSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	GtkWidget *widget;
	stack_list_flags flags;
	stack_list_item_struct *item;
	stack_list_struct *slist = STACK_LIST(data);
	if(slist == NULL)
	    return;

	if(slist->freeze_count > 0)
	    return;

	widget = GTK_WIDGET(clist);
	flags = slist->flags;

	/* Check which GtkCList this event is for
	 *
	 * Source GtkCList
	 */
	if(widget == slist->src_clist)
	{
	    slist->src_last_selected = row;

	    /* If the row is not fully visible then scroll to it */
	    if(gtk_clist_row_is_visible(clist, row) !=
		GTK_VISIBILITY_FULL
	    )
		gtk_clist_moveto(
		    clist,
		    row, -1,			/* Row, column */
		    0.5f, 0.0f			/* Row, column */
		);

	    /* Set the DND drag icon */
	    GUIDNDSetDragIconFromCListSelection(clist);

	    StackListUpdateMenus(slist);

	    /* Double click? */
	    if(event != NULL)
	    {
		if(event->type == GDK_2BUTTON_PRESS)
		{
		    GdkEventButton *button = (GdkEventButton *)event;
		    switch(button->button)
		    {
		      case GDK_BUTTON1:
			/* Add */
			StackListAdd(slist);

			/* Update the description */
			if(flags & STACK_LIST_SHOW_DESCRIPTION_SOURCE)
			{
			    item = STACK_LIST_ITEM(
				gtk_clist_get_row_data(clist, row)
			    );
			    StackListSetDescriptionFromItem(slist, item);
			}
			break;
		    }
		}
	    }
	}
	/* Target GtkCList */
	else if(widget == slist->tar_clist)
	{
	    slist->tar_last_selected = row;

	    /* If the row is not fully visible then scroll to it */
	    if(gtk_clist_row_is_visible(clist, row) !=
		GTK_VISIBILITY_FULL
	    )
		gtk_clist_moveto(
		    clist,
		    row, -1,			/* Row, column */
		    0.5f, 0.0f			/* Row, column */
		);

	    /* Set the DND drag icon */
	    GUIDNDSetDragIconFromCListSelection(clist);

	    StackListUpdateMenus(slist);

	    /* Double click? */
	    if(event != NULL)
	    {
		if(event->type == GDK_2BUTTON_PRESS)
		{
		    GdkEventButton *button = (GdkEventButton *)event;
		    switch(button->button)
		    {
		      case GDK_BUTTON1:
			/* Remove */
			StackListRemove(slist);

			/* Update the description */
			if(flags & STACK_LIST_SHOW_DESCRIPTION_TARGET)
			{
			    item = STACK_LIST_ITEM(
				gtk_clist_get_row_data(clist, row)
			    );
			    StackListSetDescriptionFromItem(slist, item);
			}
			break;
		    }
		}
	    }
	}
}

/*
 *      GtkCList "unselect_row" signal callback.
 */
static void StackListUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	GtkWidget *widget;
	stack_list_struct *slist = STACK_LIST(data);
	if(slist == NULL)
	    return;

	if(slist->freeze_count > 0)
	    return;

	widget = GTK_WIDGET(clist);

	/* Check which GtkCList this event is for
	 *
	 * Source GtkCList
	 */
	if(widget == slist->src_clist)
	{
	    if(row == slist->src_last_selected)
		slist->src_last_selected = -1;

	    StackListUpdateMenus(slist);
	}
	/* Target GtkCList */
	else if(widget == slist->tar_clist)
	{
	   if(row == slist->tar_last_selected)
		slist->tar_last_selected = -1;

	    StackListUpdateMenus(slist);
	}
}

/*
 *	Add callback.
 *
 *	Inserts all selected items from the source list to the target
 *	list at the last selected item on the target list, or appends
 *	them to the target list if nothing is selected on the target
 *	list.
 */
static void StackListAddCB(GtkWidget *widget, gpointer data)
{
	stack_list_struct *slist = STACK_LIST(data);
	if(slist == NULL)
	    return;

	if(slist->freeze_count > 0)
	    return;

	StackListAdd(slist);
}

/*
 *	Remove callback.
 */
static void StackListRemoveCB(GtkWidget *widget, gpointer data)
{
	stack_list_struct *slist = STACK_LIST(data);
	if(slist == NULL)
	    return;

	if(slist->freeze_count > 0)
	    return;

	StackListRemove(slist);
}

/*
 *	Shift up callback.
 */
static void StackListShiftUpCB(GtkWidget *widget, gpointer data)
{
	stack_list_struct *slist = STACK_LIST(data);
	if(slist == NULL)
	    return;

	if(slist->freeze_count > 0)
	    return;

	StackListShiftUp(slist);
}

/*
 *	Shift down callback.
 */
static void StackListShiftDownCB(GtkWidget *widget, gpointer data)
{
	stack_list_struct *slist = STACK_LIST(data);
	if(slist == NULL)
	    return;

	if(slist->freeze_count > 0)
	    return;

	StackListShiftDown(slist);
}


/*
 *	Returns the Stack List Item that matches the specified ID.
 */
static stack_list_item_struct *StackListMatchItemByID(
	stack_list_struct *slist, const gint id
)
{
	GList *glist;
	stack_list_item_struct *item;

	if(slist == NULL)
	    return(NULL);

	for(glist = slist->items_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    item = STACK_LIST_ITEM(glist->data);
	    if(item == NULL)
		continue;

	    if(item->id == id)
		return(item);
	}

	return(NULL);
}


/*
 *	Updates the Stack List's description with the specified item.
 */
static void StackListSetDescriptionFromItem(
	stack_list_struct *slist, stack_list_item_struct *item
)
{
	GtkWidget *w = (slist != NULL) ?
	    slist->description_label : NULL;
	if(w == NULL)
	    return;

#if (DEBUG_LEVEL >= 1)
g_print(
 "StackListSetDescriptionFromItem(): slist=0x%.8x item=0x%.8x\n",
 (guint32)slist, (guint32)item
);
#endif

	/* Change in items? */
	if(item != slist->last_description_item)
	{
	    const gchar *s = (item != NULL) ? item->description : NULL;
#if (DEBUG_LEVEL >= 2)
g_print(
 "StackListSetDescriptionFromItem(): Item changed from 0x%.8x to 0x%.8x\n",
 (guint32)slist->last_description_item, (guint32)item
);
#endif
	    gtk_label_set_text(GTK_LABEL(w), (s != NULL) ? s : "");
	    slist->last_description_item = item;
	}
}


/*
 *	Inserts an item into the target list.
 *
 *	The row specifies the row to insert at. If row is -1 then
 *	the item is appended.
 *
 *	The item specifies the item. The item will be referenced,
 *	which means that it will never be modified or deleted by
 *	this function.
 *
 *	Returns the new row index or negative on error.
 */
static gint StackListInsertTar(
	stack_list_struct *slist,
	const gint row,
	stack_list_item_struct *item
)
{
	gint i, new_row, column, ncolumns;
	gchar **strv;
	GtkCList *clist;

	if((slist == NULL) || (item == NULL))
	    return(-2);

	clist = GTK_CLIST(slist->tar_clist);

	/* Allocate the row cell values */
	ncolumns = clist->columns;
	strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
	    strv[i] = "";

	gtk_clist_freeze(clist);

	/* Insert/append the new item */
	if(row > -1)
	    new_row = gtk_clist_insert(clist, row, strv);
	else
	    new_row = gtk_clist_append(clist, strv);

	/* Delete the row cell values */
	g_free(strv);

	/* Failed to create the new row? */
	if(new_row < 0)
	{
	    gtk_clist_thaw(clist);
	    return(-1);
	}

	/* Set the new item's cell values */
	column = 0;
	if(column < ncolumns)
	{
	    if((item->icon_pixmap != NULL) &&
	       (item->name != NULL)
	    )
		gtk_clist_set_pixtext(
		    clist, new_row, column,
		    item->name,
		    2,
		    item->icon_pixmap, item->icon_mask
		);
	    else if(item->icon_pixmap != NULL)
		gtk_clist_set_pixmap(
		    clist, new_row, column,
		    item->icon_pixmap, item->icon_mask
		);
	    else if(item->name != NULL)
		gtk_clist_set_text(
		    clist, new_row, column,
		    item->name
		);
	}

	/* Set the item as the row data, the item is referenced
	 * and should not be deleted when the row is deleted
	 */
	gtk_clist_set_row_data(clist, new_row, item);

	gtk_clist_columns_autosize(clist);

	gtk_clist_thaw(clist);

	return(new_row);
}

/*
 *	Appends an item into the target list.
 *
 *	The item specifies the item. The item will be referenced,
 *	which means that it will never be modified or deleted by
 *	this function.
 *
 *	Returns the new row index or negative on error.
 */
static gint StackListAppendTar(
	stack_list_struct *slist,
	stack_list_item_struct *item
)
{
	return(StackListInsertTar(slist, -1, item));
}


/*
 *	Add.
 */
void StackListAdd(stack_list_struct *slist)
{
	gint	src_row, tar_insert_row, new_row,
		last_new_row = -1;
	GList *glist, *glist2, *sel_list;
	GtkCList *src_clist, *tar_clist;
	stack_list_item_struct *item;

	if(slist == NULL)
	    return;

	src_clist = GTK_CLIST(slist->src_clist);
	tar_clist = GTK_CLIST(slist->tar_clist);

	/* Get the row index on the target list to insert at */
	glist = tar_clist->selection_end;
	tar_insert_row = (glist != NULL) ? (gint)glist->data : -1;

	/* Begin adding */
	gtk_clist_freeze(src_clist);
	gtk_clist_freeze(tar_clist);

#if 0
/* Do not unselect any target rows, this makes placement more convient
 * for subsequent adds
 */
	/* Unselect all rows on the target list */
	gtk_clist_unselect_all(tar_clist);
#endif

	/* Create a list of selected rows */
	sel_list = NULL;
	for(glist = src_clist->selection;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    glist2 = g_list_nth(
		src_clist->row_list,
		(gint)glist->data
	    );
	    if(glist2 == NULL)
		continue;

	    sel_list = g_list_append(
		sel_list,
		(GtkCListRow *)glist2->data
	    );
	}

	/* Add each selected row to the target list */
	for(glist = sel_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    src_row = g_list_index(         
		src_clist->row_list,
		(GtkCListRow *)glist->data
	    );
	    item = STACK_LIST_ITEM(gtk_clist_get_row_data(
		src_clist, src_row
	    ));
	    if(item == NULL)
		continue;

	    /* Create a new row on the target list for this item */
	    if(tar_insert_row > -1)
	    {
		new_row = StackListInsertTar(
		    slist, tar_insert_row, item
		);
		tar_insert_row++;
	    }
	    else
	    {
		new_row = StackListAppendTar(
		    slist, item
		);
	    }
	    if(new_row < 0)
		break;

	    /* Record the last new row */
	    last_new_row = new_row;

#if 0
/* Do not select the new target row, this makes placement more convient
 * for subsequent adds
 */
	    gtk_clist_select_row(tar_clist, new_row, 0);
#endif

	    /* If multiple occurances of the item is not allowed
	     * then remove this item from the source list
	     */
	    if(!item->allow_multiple)
	    {
		src_row = gtk_clist_find_row_from_data(src_clist, item);
		gtk_clist_remove(src_clist, src_row);
	    }
	}

	gtk_clist_columns_autosize(src_clist);
	gtk_clist_columns_autosize(tar_clist);

	gtk_clist_thaw(src_clist);
	gtk_clist_thaw(tar_clist);

	g_list_free(sel_list);

	/* If there is a last new row on the target list then scroll
	 * to it as needed
	 */
	if(last_new_row > -1)
	{
	    const gint row = last_new_row;
	    if(gtk_clist_row_is_visible(tar_clist, row) !=
		GTK_VISIBILITY_FULL
	    )
		gtk_clist_moveto(
		    tar_clist,
		    row, -1,		/* Row, column */
		    0.5f, 0.0f		/* Row, column */
		);
	}

	StackListUpdateMenus(slist);

	/* Call the changed callback */
	if(slist->changed_cb != NULL)
	    slist->changed_cb(
		slist,			/* Stack List */
		slist->changed_data	/* Data */
	    );
}

/*
 *	Remove.
 *
 *	Removes the selected items from the target list and (if
 *	item->allow_multiple is FALSE) puts them back in order on the
 *	source list.
 */
void StackListRemove(stack_list_struct *slist)
{
	gint i, tar_row, src_ncolumns, last_new_row = -1;
	gchar **strv;
	GList *glist, *glist2, *sel_list;
	GtkCList *src_clist, *tar_clist;
	stack_list_flags flags;
	stack_list_item_struct *item;

	if(slist == NULL)
	    return;

	flags = slist->flags;

	src_clist = GTK_CLIST(slist->src_clist);
	tar_clist = GTK_CLIST(slist->tar_clist);

	/* Allocate the row cell values */
	src_ncolumns = src_clist->columns;
	strv = (gchar **)g_malloc(src_ncolumns * sizeof(gchar *));
	for(i = 0; i < src_ncolumns; i++)
	    strv[i] = "";

	/* Begin removing */
	gtk_clist_freeze(src_clist);
	gtk_clist_freeze(tar_clist);

	/* Create a list of selected rows */
	sel_list = NULL;
	for(glist = tar_clist->selection;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    glist2 = g_list_nth(
		tar_clist->row_list,
		(gint)glist->data
	    );
	    if(glist2 == NULL)
		continue;

	    sel_list = g_list_append(
		sel_list,
		(GtkCListRow *)glist2->data
	    );
	}

#if 0 
/* Do not unselect any source rows, this makes placement more convient
 * for subsequent removes
 */
	/* Unselect all rows on the source list */
	gtk_clist_unselect_all(src_clist);
#endif

	/* Remove each selected row from the target list */
	for(glist = sel_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    tar_row = g_list_index(
		tar_clist->row_list,
		(GtkCListRow *)glist->data
	    );
	    item = STACK_LIST_ITEM(gtk_clist_get_row_data(
		tar_clist, tar_row
	    ));
	    if(item == NULL)
		continue;

	    /* Does this item need to be put back on the source list? */
	    if(!item->allow_multiple && !item->stay_on_target)
	    {
		gint new_row, src_row;

		/* Need to maintain order of items on the source list? */
 		if(flags & STACK_LIST_MAINTAIN_SOURCE_ORDER)
		{
		    /* Put the item back on the source list, find the
		     * row index on the source list where we should put
		     * it by checking the Stack List Item cache
		     */
		    const gint id = item->id;
		    GList *glist;
		    stack_list_item_struct *src_item;

		    src_row = 0;

		    for(glist = slist->items_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			src_item = STACK_LIST_ITEM(glist->data);
		        if(src_item == NULL)
			    continue;

			/* If this cache item is on the source list then
			 * increment src_row
			 */
			if(gtk_clist_find_row_from_data(src_clist, src_item) > -1)
			    src_row++;

			/* Stop iterating when we encounter this item
			 * item in the Stack List Item cache
			 */
			if(src_item->id == id)
			    break;
		    }
		}
		else
		{
		    glist2 = src_clist->selection_end;
		    src_row = (glist2 != NULL) ? (gint)glist2->data : -1;
		}

		/* Insert or append item back into the source list */
		if((src_row >= 0) && (src_row < src_clist->rows))
		    new_row = gtk_clist_insert(src_clist, src_row, strv);
		else
		    new_row = gtk_clist_append(src_clist, strv);
		if(new_row > -1)
		{
		    /* Set new source row cell values */
		    gint column = 0;
		    if(column < src_ncolumns)
		    {
		        if((item->icon_pixmap != NULL) &&
		           (item->name != NULL)
		        )
			    gtk_clist_set_pixtext(
			        src_clist, new_row, column,
			        item->name,
			        2,
			        item->icon_pixmap, item->icon_mask
			    );
		        else if(item->icon_pixmap != NULL)
			    gtk_clist_set_pixmap(
			        src_clist, new_row, column,
			        item->icon_pixmap, item->icon_mask
			    );
		        else if(item->name != NULL)
			    gtk_clist_set_text(
			        src_clist, new_row, column,
			        item->name
			    );
		    }

		    /* Record the last new row */
		    last_new_row = new_row;

		    /* Set row data, but no destroy function since the
		     * row data is shared and not suppose to be
		     * deleted when the clist row is deleted
		     */
		    gtk_clist_set_row_data(src_clist, new_row, item);

#if 0
/* Do not select the new source row, this makes placement more convient
 * for subsequent removes
 */
		    gtk_clist_select_row(src_clist, new_row, 0);
#endif
		}
	    }

	    /* Remove or unselect this row from target list? */
	    if(!item->stay_on_target)
		gtk_clist_remove(tar_clist, tar_row);
	}

	gtk_clist_columns_autosize(src_clist);
	gtk_clist_columns_autosize(tar_clist);

	gtk_clist_thaw(src_clist);
	gtk_clist_thaw(tar_clist);

	g_list_free(sel_list);
	g_free(strv);

	/* If there is a last new row on the source list then scroll
	 * to it as needed
	 */
	if(last_new_row > -1)
	{
	    const gint row = last_new_row;
	    if(gtk_clist_row_is_visible(src_clist, row) !=
		GTK_VISIBILITY_FULL
	    )
		gtk_clist_moveto(
		    src_clist,
		    row, -1,		/* Row, column */
		    0.5f, 0.0f		/* Row, column */
		);
	}

	StackListUpdateMenus(slist);

	/* Call the changed callback */
	if(slist->changed_cb != NULL)
	    slist->changed_cb(
		slist,			/* Stack List */
		slist->changed_data	/* Data */
	    );
}

/*
 *	Shift up.
 */
void StackListShiftUp(stack_list_struct *slist)
{
	gint row, nrows, lead_row;
	GtkCList *clist;

	if(slist == NULL)
	    return;

	clist = GTK_CLIST(slist->tar_clist);
	nrows = clist->rows;
	if((clist->selection == NULL) || (nrows < 2))
	    return;

	/* Already at top? */
	row = 0;
	if(g_list_find(clist->selection, (gpointer)row) != NULL)
	    return;

	/* Shift each selected row up */
	gtk_clist_freeze(clist);
	lead_row = -1;
	for(row = 1; row < nrows; row++)
	{
	    if(g_list_find(clist->selection, (gpointer)row) != NULL)
	    {
		if(lead_row < 0)
		    lead_row = row - 1;

		gtk_clist_swap_rows(clist, row, row - 1);
	    }
	}
	gtk_clist_thaw(clist);

	/* Scroll to the lead row if it is not visible */
	if(lead_row > -1)
	{
	    if(gtk_clist_row_is_visible(clist, lead_row) !=
		GTK_VISIBILITY_FULL
	    )
		gtk_clist_moveto(
		    clist,
		    lead_row, -1,	/* Row, column */
		    0.5f, 0.0f		/* Row, column */
		);
	}

	StackListUpdateMenus(slist);

	/* Call the changed callback */
	if(slist->changed_cb != NULL)
	    slist->changed_cb(
		slist,			/* Stack List */
		slist->changed_data	/* Data */
	    );
}

/*
 *	Shift down.
 */
void StackListShiftDown(stack_list_struct *slist)
{
	gint row, nrows, lead_row;
	GtkCList *clist;

	if(slist == NULL)
	    return;

	clist = GTK_CLIST(slist->tar_clist);
	nrows = clist->rows;
	if((clist->selection == NULL) || (nrows < 2))
	    return;

	/* Already at bottom? */
	row = nrows - 1;
	if(g_list_find(clist->selection, (gpointer)row) != NULL)
	    return;

	/* Shift each selected row down */
	gtk_clist_freeze(clist);
	lead_row = -1;
	for(row = nrows - 2; row >= 0; row--)
	{
	    if(g_list_find(clist->selection, (gpointer)row) != NULL)
	    {
		if(lead_row < 0)
		    lead_row = row + 1;

		gtk_clist_swap_rows(clist, row, row + 1);
	    }
	}
	gtk_clist_thaw(clist);

	/* Scroll to the lead row if it is not visible */
	if(lead_row > -1)
	{
	    if(gtk_clist_row_is_visible(clist, lead_row) !=
		GTK_VISIBILITY_FULL
	    )
		gtk_clist_moveto(
		    clist,
		    lead_row, -1,	/* Row, column */
		    0.5f, 0.0f		/* Row, column */
		);
	}

	StackListUpdateMenus(slist);

	/* Call the changed callback */
	if(slist->changed_cb != NULL)
	    slist->changed_cb(
		slist,			/* Stack List */
		slist->changed_data	/* Data */
	    );
}


/*
 *	Appends the item to the Stack List's items list.
 */
gint StackListAppend(
	stack_list_struct *slist,
	const gchar *name,
	const gchar *description,
	guint8 **icon_data,
	gpointer data,
	const gint id,
	const gboolean allow_multiple,
	const gboolean stay_on_target
)
{
	gint i;
	stack_list_item_struct *item;

	if(slist == NULL)
	    return(-2);

	item = STACK_LIST_ITEM(g_malloc0(
	    sizeof(stack_list_item_struct)
	));
	if(item == NULL)
	    return(-3);

	i = g_list_length(slist->items_list);
	slist->items_list = g_list_append(slist->items_list, item);

	item->name = STRDUP(name);
	item->description = STRDUP(description);
	item->data = data;
	item->id = id;
	item->allow_multiple = allow_multiple;
	item->stay_on_target = stay_on_target;
	item->icon_pixmap = GDK_PIXMAP_NEW_FROM_XPM_DATA(
	    &item->icon_mask, (guint8 **)icon_data
	);

	return(i);
}

/*
 *	Deletes all items in the Stack List's Cache and clears all the
 *	lists.
 */
void StackListClear(stack_list_struct *slist)
{
	if(slist == NULL)
	    return;

	/* Clear the source and target lists */
	StackListItemClearSrc(slist);
	StackListItemClearTar(slist);

	/* Clear the items list */
	if(slist->items_list != NULL)
	{
	    GList *glist;
	    stack_list_item_struct *item;

	    for(glist = slist->items_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		item = STACK_LIST_ITEM(glist->data);
		if(item == NULL)
		    continue;

		g_free(item->name);
		g_free(item->description);
		GDK_PIXMAP_UNREF(item->icon_pixmap);
		GDK_BITMAP_UNREF(item->icon_mask);
		g_free(item);
	    }
	    g_list_free(slist->items_list);
	    slist->items_list = NULL;
	}

	slist->last_description_item = NULL;
}


/*
 *	Appends an item specified by id to the Stack List's source
 *	list.
 */
gint StackListItemAppendSrc(stack_list_struct *slist, const gint id)
{
	gint i, row, ncolumns;
	gchar **strv;
	GtkCList *clist;
	stack_list_item_struct *item;

	if(slist == NULL)
	    return(-2);

	clist = GTK_CLIST(slist->src_clist);
	ncolumns = clist->columns;

	/* Get the Stack List Item that matches the specified id */
	item = StackListMatchItemByID(slist, id);
	if(item == NULL)
	    return(-1);

	/* Allocate the row cell values */
	strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
	    strv[i] = "";

	gtk_clist_freeze(clist);

	/* Append a new item to the clist */
	row = gtk_clist_append(clist, strv);

	/* Delete the row cell values */
	g_free(strv);

	/* Set the new row */
	if(row > -1)
	{
	    const gint column = 0;
	    if((item->icon_pixmap != NULL) && (item->name != NULL))
		gtk_clist_set_pixtext(
		    clist, row, column,
		    item->name,
		    2,
		    item->icon_pixmap, item->icon_mask
		);
	    else if(item->icon_pixmap != NULL)
		gtk_clist_set_pixmap(
		    clist, row, column,
		    item->icon_pixmap, item->icon_mask
		);
	    else if(item->name != NULL)
		gtk_clist_set_text(
		    clist, row, column,
		    item->name
		);

	    /* Set row data, but no destroy function since the row
	     * data is shared and not suppose to be deleted when the
	     * clist row is deleted
	     */
	    gtk_clist_set_row_data(clist, row, item);
	}

	gtk_clist_columns_autosize(clist);

	gtk_clist_thaw(clist);

	return(row);
}

/*
 *	Appends an item specified by id to the Stack List's target
 *	list.
 */
gint StackListItemAppendTar(stack_list_struct *slist, const gint id)
{
	gint i, row, ncolumns;
	gchar **strv;
	GtkCList *clist;
	stack_list_item_struct *item;

	if(slist == NULL)
	    return(-2);

	clist = GTK_CLIST(slist->tar_clist);
	ncolumns = clist->columns;

	/* Get the Stack List Item that matches the specified id */
	item = StackListMatchItemByID(slist, id);
	if(item == NULL)
	    return(-1);

	/* Allocate the row cell values */
	strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
	    strv[i] = "";

	/* Append a new item to the clist */
	row = gtk_clist_append(clist, strv);

	/* Delete the row cell values */
	g_free(strv);

	/* Set the new row */
	if(row > -1)
	{
	    const gint column = 0;
	    if((item->icon_pixmap != NULL) && (item->name != NULL))
		gtk_clist_set_pixtext(
		    clist, row, column,
		    item->name,
		    2,
		    item->icon_pixmap, item->icon_mask
		);
	    else if(item->icon_pixmap != NULL)
		gtk_clist_set_pixmap(
		    clist, row, column,
		    item->icon_pixmap, item->icon_mask
		);
	    else if(item->name != NULL)
		gtk_clist_set_text(
		    clist, row, column,
		    item->name
		);

	    /* Set row data, but no destroy function since the row
	     * data is shared and not suppose to be deleted when the
	     * clist row is deleted
	     */
	    gtk_clist_set_row_data(clist, row, item);
	}

	return(row);
}

/*
 *	Appends the entire items list to the source list.
 */
void StackListItemSetAllFromCacheSrc(stack_list_struct *slist)
{
	GList *glist;
	GtkCList *clist;
	stack_list_item_struct *item;

	if(slist == NULL)
	    return;

	clist = GTK_CLIST(slist->src_clist);

	gtk_clist_freeze(clist);
	for(glist = slist->items_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    item = STACK_LIST_ITEM(glist->data);
	    if(item == NULL)
		continue;

	    StackListItemAppendSrc(slist, item->id);
	}
	gtk_clist_thaw(clist);
}


/*
 *	Removes all items who's id matches the given id from the
 *	source clist.
 *
 *      If exclude_allowed_multiples is TRUE than if the item allows
 *      multiple coppies of itself then it will not be removed.
 */
void StackListItemRemoveByIDSrc(
	stack_list_struct *slist, const gint id,
	const gboolean exclude_allowed_multiples
)
{
	gint i;
	GtkCList *clist;
	stack_list_item_struct *item;

	if(slist == NULL)
	    return;

	clist = GTK_CLIST(slist->src_clist);

	gtk_clist_freeze(clist);

	/* Iterate through all rows, checking each item structure's
	 * id if it matches the given id.
	 */
	for(i = 0; i < clist->rows; i++)
	{
	    item = STACK_LIST_ITEM(gtk_clist_get_row_data(
		clist, i
	    ));
	    if(item == NULL)
		continue;

	    /* Skip if we are to exclude items that allow multiples and
	     * this item allows multiple coppies of itself
	     */
	    if(exclude_allowed_multiples &&
	       item->allow_multiple
	    )
		continue;

	    /* IDs match? */
	    if(item->id == id)
	    {
		/* Remove this row */
		gtk_clist_remove(clist, i);
	    }
	}

	gtk_clist_columns_autosize(clist);

	gtk_clist_thaw(clist);
}

/*
 *      Removes all items who's id matches the given id from the
 *      source clist.
 *
 *	If exclude_allowed_multiples is TRUE than if the item allows
 *	multiple coppies of itself then it will not be removed.
 */
void StackListItemRemoveByIDTar(
	stack_list_struct *slist, const gint id,
	const gboolean exclude_allowed_multiples
)
{
	gint i;
	GtkCList *clist;
	const stack_list_item_struct *item;

	if(slist == NULL)
	    return;

	clist = GTK_CLIST(slist->tar_clist);

	gtk_clist_freeze(clist);

	/* Iterate through all rows, checking each Stack List Item's
	 * ID to see if it matches the specified ID
	 */
	for(i = 0; i < clist->rows; i++)
	{
	    item = STACK_LIST_ITEM(gtk_clist_get_row_data(
		clist, i
	    ));
	    if(item == NULL)
		continue;

	    /* Skip if we are to exclude items that allow multiples and
	     * this item allows multiple coppies of itself
	     */
	    if(exclude_allowed_multiples && item->allow_multiple)
		continue;

	    /* IDs match? */
	    if(item->id == id)
	    {
		/* Remove this row */
		gtk_clist_remove(clist, i);
		continue;
	    }
	}

	gtk_clist_columns_autosize(clist);

	gtk_clist_thaw(clist);
}

/*
 *	Deletes all items in the Stack List's source list.
 */
void StackListItemClearSrc(stack_list_struct *slist)
{
	GtkCList *clist = (slist != NULL) ?
	    GTK_CLIST(slist->src_clist) : NULL;
	if(clist == NULL)
	    return;

	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	gtk_clist_columns_autosize(clist);
	gtk_clist_thaw(clist);
}

/*
 *      Deletes all items in the Stack List's target list.
 */
void StackListItemClearTar(stack_list_struct *slist)
{
	GtkCList *clist = (slist != NULL) ?
	    GTK_CLIST(slist->tar_clist) : NULL;
	if(clist == NULL)
	    return;

	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	gtk_clist_columns_autosize(clist);
	gtk_clist_thaw(clist);
}


/*
 *	Returns a list of gints, each has the value of the id of the
 *	item currently in the source list.
 *
 *	The calling function must delete the returned pointer.
 */
gint *StackListItemGetSrc(stack_list_struct *slist, gint *total)
{
	gint i;
	gint *list;
	const stack_list_item_struct *item;
	GtkCList *clist = (slist != NULL) ?
	    GTK_CLIST(slist->src_clist) : NULL;
	if((clist == NULL) || (total == NULL))
	    return(NULL);

	*total = clist->rows;
	if(*total <= 0)
	    return(NULL);

	/* Allocate gint list */
	list = (gint *)g_malloc0((*total) * sizeof(gint));
	if(list == NULL)
	{
	    *total = 0;
	    return(NULL);
	}

	/* Iterate through each row and set each ID to the gint list */
	for(i = 0; i < clist->rows; i++)
	{
	    item = STACK_LIST_ITEM(
		gtk_clist_get_row_data(clist, i)
	    );
	    list[i] = (item != NULL) ? item->id : -1;
	}

	return(list);
}

/*
 *      Returns a dynamically allocated array of gint's, each
 *      representing the id of the item currently in the target list.
 *      The calling function needs to deallocate the returned pointer.
 *
 *      Can return NULL on error or empty list.
 */
gint *StackListItemGetTar(stack_list_struct *slist, gint *total)
{
	gint i;
	gint *list;
	stack_list_item_struct *item;
	GtkCList *clist = (slist != NULL) ?
	    GTK_CLIST(slist->tar_clist) : NULL;
	if((clist == NULL) || (total == NULL))
	    return(NULL);

	*total = clist->rows;
	if(*total <= 0)
	    return(NULL);

	/* Allocate gint array */
	list = (gint *)g_malloc0((*total) * sizeof(gint));
	if(list == NULL)
	{
	    *total = 0;
	    return(NULL);
	}

	/* Iterate through each row and set each ID to the gint list */
	for(i = 0; i < clist->rows; i++)
	{
	    item = STACK_LIST_ITEM(
		gtk_clist_get_row_data(clist, i)
	    );
	    list[i] = (item != NULL) ? item->id : -1;
	}

	return(list);
}


/*
 *	Creates a new Stack List.
 */
stack_list_struct *StackListNew(
	GtkWidget *parent,
	const gchar *src_title,
	const gchar *tar_title
)
{
	const gint border_minor = 2;
	GtkWidget	*w,
			*parent2, *parent3,
			*toplevel;
	GtkCList *clist;
 	stack_list_struct *slist = STACK_LIST(g_malloc0(
	    sizeof(stack_list_struct)
	));
	if(slist == NULL)
	    return(slist);

	slist->toplevel = toplevel = gtk_vbox_new(FALSE, border_minor);
/*
	slist->freeze_count = 0;
	slist->busy_count = 0;
	slist->flags = 0;
 */
	slist->translate_cur = gdk_cursor_new(GDK_FLEUR);
	slist->src_last_selected = -1;
	slist->tar_last_selected = -1;
/*
	slist->items_list = NULL;
	slist->last_description_item = NULL;

	slist->button = 0;
	slist->drag_last_x = 0;
	slist->drag_last_y = 0;

	slist->changed_cb = NULL;
	slist->changed_data = NULL;
	slist->added_cb = NULL;
	slist->added_data = NULL;
	slist->removed_cb = NULL;
	slist->removed_data = NULL;
 */

	slist->freeze_count++;

	/* Create the toplevel GtkVBox */
	w = toplevel;
	if(parent != NULL)
	{
	    if(GTK_IS_BOX(parent))
		gtk_box_pack_start(
		    GTK_BOX(parent), w, TRUE, TRUE, 0
		);
	    else if(GTK_IS_CONTAINER(parent))
		gtk_container_add(GTK_CONTAINER(parent), w);
	}
	parent = w;

	/* GtkHBox for the source & target lists */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent = w;

	/* GtkScrolledWindow for the source GtkCList */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
	    GTK_SCROLLED_WINDOW(w),
	    GTK_POLICY_AUTOMATIC,
	    GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Create the source GtkCList */
	if(src_title != NULL)
	{
	    gchar *heading[1];
	    heading[0] = STRDUP(src_title);
	    slist->src_clist = w = gtk_clist_new_with_titles(
		sizeof(heading) / sizeof(gchar *),
		heading
	    );
	    g_free(heading[0]);
	}
	else
	{
	    slist->src_clist = w = gtk_clist_new(1);
	}
	clist = GTK_CLIST(w);
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(StackListCListEventCB), slist
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(StackListCListEventCB), slist
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "enter_notify_event",
	    GTK_SIGNAL_FUNC(StackListCListEventCB), slist
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "leave_notify_event",
	    GTK_SIGNAL_FUNC(StackListCListEventCB), slist
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(StackListCListEventCB), slist
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(StackListCListEventCB), slist
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(StackListCListEventCB), slist
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "select_row",
	    GTK_SIGNAL_FUNC(StackListSelectRowCB), slist
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "unselect_row",
	    GTK_SIGNAL_FUNC(StackListUnselectRowCB), slist
	);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_realize(w);
	gtk_clist_column_titles_passive(clist);
	gtk_clist_set_column_auto_resize(clist, 0, TRUE);
	gtk_clist_set_column_resizeable(clist, 0, FALSE);
	gtk_clist_set_column_justification(
	    clist, 0, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED);
	gtk_clist_set_row_height(clist, STACK_LIST_ROW_SPACING);
	if(w != NULL)
	{
	    const GtkTargetEntry dnd_src_types[] = {
{STACK_LIST_ITEM_SRC_TARGET_NAME,	0,	STACK_LIST_ITEM_SRC_INFO}
	    };
	    const GtkTargetEntry dnd_tar_types[] = {
{STACK_LIST_ITEM_TAR_TARGET_NAME,	0,	STACK_LIST_ITEM_TAR_INFO}
	    };
	    GUIDNDSetSrc(
		w,
		dnd_src_types,
		sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_MOVE,		/* Allowed actions */
		GDK_BUTTON1_MASK,		/* Buttons */
		NULL,
		StackListDragDataGetCB,
		StackListDragDataDeleteCB,
		NULL,
		slist
	    );
	    GUIDNDSetTar(
		w,
		dnd_tar_types,
		sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_MOVE,		/* Allowed actions */
		GDK_ACTION_MOVE,		/* Default action if same */
		GDK_ACTION_MOVE,		/* Default action */
		StackListDragDataReceivedCB,
		slist,
		TRUE				/* Highlight */
	    );
	}
	gtk_widget_show(w);


	/* Island GtkVBox between lists for the arrow buttons */
	w = gtk_vbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* GtkVBox for the left and right arrow GtkButtons */
	w = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Right arrow GtkButton */
	slist->right_arrow_btn = w = GUIButtonArrow(
	    GTK_ARROW_RIGHT,
	    -1, -1
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(StackListAddCB), slist
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
	gtk_widget_show(w);

	/* Left arrow GtkButton */
	slist->left_arrow_btn = w = GUIButtonArrow(
	    GTK_ARROW_LEFT,
	    -1, -1
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(StackListRemoveCB), slist
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
	gtk_widget_show(w);


	/* GtkVBox for the up and down arrow GtkButtons */
	w = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Up arrow GtkButton */
	slist->up_arrow_btn = w = GUIButtonArrow(
	    GTK_ARROW_UP,
	    -1, -1
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(StackListShiftUpCB), slist
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
	gtk_widget_show(w);

	/* Down arrow GtkButton */
	slist->down_arrow_btn = w = GUIButtonArrow(
	    GTK_ARROW_DOWN,
	    -1, -1
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(StackListShiftDownCB), slist
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
	gtk_widget_show(w);


	/* GtkScrolledWindow for the target GtkCList */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
	    GTK_SCROLLED_WINDOW(w),
	    GTK_POLICY_AUTOMATIC,
	    GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Create the target GtkCList */
	if(tar_title != NULL)
	{
	    gchar *heading[1];
	    heading[0] = STRDUP(tar_title);
	    slist->tar_clist = w = gtk_clist_new_with_titles(
		sizeof(heading) / sizeof(gchar *),
		heading
	    );
	    g_free(heading[0]);
	}
	else
	{
	    slist->tar_clist = w = gtk_clist_new(1);
	}
	clist = GTK_CLIST(w);
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(StackListCListEventCB), slist
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(StackListCListEventCB), slist
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "enter_notify_event",
	    GTK_SIGNAL_FUNC(StackListCListEventCB), slist
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "leave_notify_event",
	    GTK_SIGNAL_FUNC(StackListCListEventCB), slist
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(StackListCListEventCB), slist
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(StackListCListEventCB), slist
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(StackListCListEventCB), slist
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "select_row",
	    GTK_SIGNAL_FUNC(StackListSelectRowCB), slist
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "unselect_row",
	    GTK_SIGNAL_FUNC(StackListUnselectRowCB), slist
	);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_realize(w);
	gtk_clist_column_titles_passive(clist);
	gtk_clist_set_column_auto_resize(clist, 0, TRUE);
	gtk_clist_set_column_resizeable(clist, 0, FALSE);
	gtk_clist_set_column_justification(
	    clist, 0, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED);
	gtk_clist_set_row_height(clist, STACK_LIST_ROW_SPACING);
	if(w != NULL)
	{
	    const GtkTargetEntry dnd_src_types[] = {
{STACK_LIST_ITEM_TAR_TARGET_NAME,	0,	STACK_LIST_ITEM_TAR_INFO}
	    };
	    const GtkTargetEntry dnd_tar_types[] = {
{STACK_LIST_ITEM_SRC_TARGET_NAME,	0,	STACK_LIST_ITEM_SRC_INFO},
{STACK_LIST_ITEM_TAR_TARGET_NAME,	0,	STACK_LIST_ITEM_TAR_INFO}
	    };
	    GUIDNDSetSrc(
		w,
		dnd_src_types,
		sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_MOVE,		/* Allowed actions */
		GDK_BUTTON1_MASK,		/* Buttons */
		NULL,
		StackListDragDataGetCB,
		StackListDragDataDeleteCB,
		NULL,
		slist
	    );
	    GUIDNDSetTar(
		w,
		dnd_tar_types,
		sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_MOVE,		/* Allowed actions */
		GDK_ACTION_MOVE,		/* Default action if same */
		GDK_ACTION_MOVE,		/* Default action */
		StackListDragDataReceivedCB,
		slist,
		TRUE				/* Highlight */
	    );
	}
	gtk_widget_show(w);


	/* Frame for description label */
	slist->description_toplevel = w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(slist->toplevel), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent = w;
#if 0
	/* Style for background */
	rcstyle = gtk_rc_style_new();
	state = GTK_STATE_NORMAL;
	rcstyle->color_flags[state] = GTK_RC_FG | GTK_RC_BG |
	    GTK_RC_TEXT | GTK_RC_BASE;
	GDK_COLOR_SET_COEFF(
	    &rcstyle->fg[state],
	    0.0f, 0.0f, 0.0f
	)
	GDK_COLOR_SET_COEFF(
	    &rcstyle->bg[state],
	    1.0f, 1.0f, 0.8f
	)
	GDK_COLOR_SET_COEFF(
	    &rcstyle->text[state],
	    0.0f, 0.0f, 0.0f
	)
	GDK_COLOR_SET_COEFF(
	    &rcstyle->base[state],
	    1.0f, 1.0f, 0.8f   
	)
	state = GTK_STATE_INSENSITIVE;
	rcstyle->color_flags[state] = GTK_RC_FG | GTK_RC_BG |
	    GTK_RC_TEXT | GTK_RC_BASE;
	GDK_COLOR_SET_COEFF(
	    &rcstyle->fg[state],
	    0.25f, 0.25f, 0.25f
	)
	GDK_COLOR_SET_COEFF(
	    &rcstyle->bg[state],
	    1.0f, 1.0f, 0.8f
	)
	GDK_COLOR_SET_COEFF(
	    &rcstyle->text[state],
	    0.25f, 0.25f, 0.25f
	)
	GDK_COLOR_SET_COEFF(
	    &rcstyle->base[state],
	    1.0f, 1.0f, 0.8f
	)
#endif
	/* Create a GtkEventBox for the label's background */
	w = gtk_event_box_new();
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_hbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), border_minor);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent2 = w;

	/* Description label */
	slist->description_label = w = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


#define ADD_MENU_ITEM_LABEL	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_LABEL,			\
  accelgrp,					\
  icon,						\
  label,					\
  accel_key, accel_mods,			\
  func_cb, menu_data				\
 );						\
}
#define ADD_MENU_ITEM_SEPARATOR	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL,						\
  NULL,						\
  0, 0,						\
  NULL, NULL					\
 );						\
}
	/* Source clist right-click menu */
	slist->src_menu = w = GUIMenuCreate();
	if(w != NULL)
	{
	    GtkAccelGroup *accelgrp = NULL;
	    GtkWidget *menu = w;
	    guint8 **icon;
	    const gchar *label;
	    guint accel_key, accel_mods;
	    void (*func_cb)(GtkWidget *w, gpointer);
	    gpointer menu_data = slist;

	    icon = (guint8 **)icon_add_20x20_xpm;
	    label = "Add";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = StackListAddCB;
	    ADD_MENU_ITEM_LABEL
	    slist->src_add_mi = w;
	}

	/* Target clist right-click menu */
	slist->tar_menu = w = GUIMenuCreate();
	if(w != NULL)
	{
	    GtkAccelGroup *accelgrp = NULL;
	    GtkWidget *menu = w;
	    guint8 **icon;
	    const gchar *label;
	    guint accel_key, accel_mods;
	    void (*func_cb)(GtkWidget *w, gpointer);
	    gpointer menu_data = slist;

	    icon = (guint8 **)icon_remove_20x20_xpm;
	    label = "Remove";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = StackListRemoveCB;
	    ADD_MENU_ITEM_LABEL
	    slist->tar_remove_mi = w;

	    ADD_MENU_ITEM_SEPARATOR

	    icon = NULL;
	    label = "Shift Up";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = StackListShiftUpCB;
	    ADD_MENU_ITEM_LABEL
	    slist->tar_up_mi = w;

	    icon = NULL;
	    label = "Shift Down";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = StackListShiftDownCB;
	    ADD_MENU_ITEM_LABEL
	    slist->tar_down_mi = w;
	}
#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_SEPARATOR

	slist->freeze_count--;

	return(slist);
}

/*
 *	Sets the Stack List to show the description.
 *
 *	If both show_src_desc and show_tar_desc are FALSE then the
 *	description label is hidden.
 */
void StackListShowDescription(
	stack_list_struct *slist, 
	gboolean show_src_desc,
	gboolean show_tar_desc
)
{
	GtkWidget *w;

	if(slist == NULL)
	    return;

	if(show_src_desc)
	    slist->flags |= STACK_LIST_SHOW_DESCRIPTION_SOURCE;
	else
	    slist->flags &= ~STACK_LIST_SHOW_DESCRIPTION_SOURCE;

	if(show_tar_desc)
	    slist->flags |= STACK_LIST_SHOW_DESCRIPTION_TARGET;
	else
	    slist->flags &= ~STACK_LIST_SHOW_DESCRIPTION_TARGET;

	w = slist->description_toplevel;
	if(w != NULL)
	{
	    if(show_src_desc || show_tar_desc)
		gtk_widget_show(w);
	    else
		gtk_widget_hide(w);
	}
}

/*
 *	Sets the Stack List to maintain the order of items on the
 *	source list.
 */
void StackListSetMaintainSourceOrder(
	stack_list_struct *slist, gboolean maintain_source_order
)
{
	if(slist == NULL)
	    return;

	if(maintain_source_order)
	    slist->flags |= STACK_LIST_MAINTAIN_SOURCE_ORDER;
	else
	    slist->flags &= ~STACK_LIST_MAINTAIN_SOURCE_ORDER;
}

/*
 *	Sets the Stack List's changed callback.
 */
void StackListSetChangedCB(
	stack_list_struct *slist,
	void (*cb)(
		stack_list_struct *,		/* Stack List */
		gpointer			/* Data */
	),
	gpointer data
)
{
	if(slist == NULL)
	    return;

	slist->changed_cb = cb;
	slist->changed_data = data;
}

/*
 *      Sets the Stack List's added callback.
 */
void StackListSetAddedCB(
	stack_list_struct *slist,
	void (*cb)(
		stack_list_struct *,		/* Stack List */
		stack_list_item_struct *,	/* Stack List Item */
		gpointer			/* Data */
	),
	gpointer data
)
{
	if(slist == NULL)
	    return;

	slist->added_cb = cb;
	slist->added_data = data;
}

/*
 *	Sets the Stack List's removed callback.
 */
void StackListSetRemovedCB(
	stack_list_struct *slist,
	void (*cb)(
		stack_list_struct *,		/* Stack List */
		stack_list_item_struct *,	/* Stack List Item */
		gpointer			/* Data */
	),
	gpointer data
)
{
	if(slist == NULL)
	    return;

	slist->removed_cb = cb;
	slist->removed_data = data;
}
 
/*
 *	Updates widgets on the given slist to reflect the current data
 *	represented.
 */
void StackListUpdateMenus(stack_list_struct *slist)
{
	gboolean sensitive, nrows, sel_row, nselected;
	GList *glist;
	GtkCList *clist;
	stack_list_item_struct *item;

	if(slist == NULL)
	    return;

	slist->freeze_count++;

	/* Target GtkCList */
	clist = GTK_CLIST(slist->tar_clist);
	nrows = clist->rows;
	nselected = g_list_length(clist->selection);
	glist = clist->selection_end;
	sel_row = (glist != NULL) ? (gint)glist->data : -1;
	item = STACK_LIST_ITEM(gtk_clist_get_row_data(
	    clist,
	    sel_row
	));

	sensitive = ((g_list_find(clist->selection, (gpointer)0) == NULL) &&
	    (nselected > 0)) ? TRUE : FALSE;    
	GTK_WIDGET_SET_SENSITIVE(slist->up_arrow_btn, sensitive);
	GTK_WIDGET_SET_SENSITIVE(slist->tar_up_mi, sensitive);

	sensitive = ((g_list_find(clist->selection, (gpointer)(nrows - 1)) == NULL) &&
	    (nselected > 0)) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(slist->down_arrow_btn, sensitive);
	GTK_WIDGET_SET_SENSITIVE(slist->tar_down_mi, sensitive);

	sensitive = (sel_row > -1) ? TRUE : FALSE;
	if(sensitive)
	    sensitive = (item != NULL) ?
		!item->stay_on_target : FALSE;
	GTK_WIDGET_SET_SENSITIVE(slist->left_arrow_btn, sensitive);
	GTK_WIDGET_SET_SENSITIVE(slist->tar_remove_mi, sensitive);

	/* Source GtkCList */
	clist = GTK_CLIST(slist->src_clist);
	nrows = clist->rows;
	nselected = g_list_length(clist->selection);
	glist = clist->selection_end;
	sel_row = (glist != NULL) ? (gint)glist->data : -1;
	item = STACK_LIST_ITEM(gtk_clist_get_row_data(
	    clist, sel_row
	));

	sensitive = (sel_row > -1) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(slist->right_arrow_btn, sensitive);
	GTK_WIDGET_SET_SENSITIVE(slist->src_add_mi, sensitive);

	slist->freeze_count--;
}

/*
 *	Checks if the Stack List is mapped.
 */
gboolean StackListMapped(stack_list_struct *slist)
{
	if(slist == NULL)
	    return(FALSE);

	return((slist->flags & STACK_LIST_MAPPED) ? TRUE : FALSE);
}

/*
 *	Maps the Stack List.
 */
void StackListMap(stack_list_struct *slist)
{
	if(slist == NULL)
	    return;

	gtk_widget_show(slist->toplevel);
	slist->flags |= STACK_LIST_MAPPED;
}

/*
 *	Unmaps the Stack List.
 */
void StackListUnmap(stack_list_struct *slist)
{
	if(slist == NULL)
	    return;

	gtk_widget_hide(slist->toplevel);
	slist->flags &= ~STACK_LIST_MAPPED;
}

/*
 *	Deletes the Stack List.
 */
void StackListDelete(stack_list_struct *slist)
{
	if(slist == NULL)
	    return;

	/* Delete all cached items and clear lists */
	StackListClear(slist);

	slist->freeze_count++;

	/* Destroy widgets */
	gtk_widget_destroy(slist->src_menu);
	gtk_widget_destroy(slist->tar_menu);
	gtk_widget_destroy(slist->toplevel);

	gdk_cursor_destroy(slist->translate_cur);

	slist->freeze_count--;

	g_free(slist);
}
