#include <string.h>
#include <sys/time.h>
#include <gtk/gtk.h>

#include "guiutils.h"
#include "guirgbimg.h"

#include "splash.h"


typedef struct _Splash			Splash;
#define SPLASH(p)			((Splash *)(p))
#define SPLASH_KEY			"/Splash"


/*
 *      Flags:
 */
typedef enum {
	SPLASH_MAPPED                   = (1 << 0),
	SPLASH_REALIZED                 = (1 << 1)
} SplashFlags;


/* Callbacks */
static void splash_destroy_cb(gpointer data);
static gint splash_button_cb(GtkWidget *widget, GdkEventButton *button, gpointer data);
static gint splash_expose_cb(GtkWidget *widget, GdkEventExpose *expose, gpointer data);
static gint splash_configure_cb(GtkWidget *widget, GdkEventConfigure *configure, gpointer data);
static gint splash_focus_cb(GtkWidget *widget, GdkEventFocus *focus, gpointer data);
static void splash_realize_cb(GtkWidget *widget, gpointer data);
static void splash_draw_cb(
	GtkWidget *widget, GdkRectangle *area, gpointer data
);
static gint splash_draw_idle_cb(gpointer data);

/* Draw */
static void splash_draw(Splash *splash);
static void splash_queue_draw(Splash *splash);

/* Clear Image Buffers */
static void splash_clear_buffers(Splash *splash);

/* Create GdkPixmap */
static GdkPixmap *splash_create_no_such_file_pixmap(
	const gchar *path,
	GdkWindow *window,
	GtkStyle *style,
	GdkBitmap **mask_rtn
);

/* Splash */
static Splash *splash_get_widget_data(
	GtkWidget *w,
	const gchar *func_name
);
GtkWidget *splash_new(void);
void SplashSetXPMFile(
	GtkWidget *w,
	const gchar *xpm_file
);
void splash_set_xpm_data(
	GtkWidget *w,
	const guint8 **xpm_data
);
void splash_set_message_font(
	GtkWidget *w,
	GdkFont *font
);
void splash_set_message_color(
	GtkWidget *w,
	GdkColor *fg_color,
	GdkColor *bg_color
);
void splash_set_message_justification(
	GtkWidget *w,
	const GtkJustification justify
);
void splash_set_message_position(
	GtkWidget *w,
	const GtkPositionType position
);
void splash_set_button_cb(
	GtkWidget *w,
	void (*cb)(           
		GtkWidget *,
		GdkEventButton *,       
		gpointer       
	),           
	gpointer data           
);
void splash_update_message(
	GtkWidget *w,
	const gfloat v,
	const gchar *msg
);
void splash_update(
	GtkWidget *w,
	const gboolean manage_events
);
void splash_map(
	GtkWidget *w,
	const SplashEffects effects,
	const gulong duration_ms,
	const GtkWindowPosition position
);
void splash_unmap(
	GtkWidget *w,
	const SplashEffects effects,
	const gulong duration_ms
);
GtkWidget *splash_ref(GtkWidget *w);
GtkWidget *splash_unref(GtkWidget *w);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Splash:
 */
struct _Splash {

	GtkWidget	*toplevel;
	gint		freeze_count,
					ref_count;
	GdkColormap	*colormap;
	GdkGC		*gc;
	SplashFlags	flags;

	guint		draw_idleid;

	gchar		*xpm_file;
	const guint8	**xpm_data;

	guint8		*rgba;			/* Splash image */
	gint		rgba_width,
					rgba_height,
					rgba_bpl;

	guint8		*bg_rgba;		/* Background image */
	gint		bg_rgba_width,
					bg_rgba_height,
					bg_rgba_bpl;

	gfloat		value;			/* 0.0 to 1.0 or -1.0 */
	gchar		*message;
	GdkFont		*message_font;
	GdkColor	*message_fg_color,
					*message_bg_color;
	GdkTextBounds	message_bounds;
	GtkJustification	message_justify;
	GtkPositionType	message_position;

	void		(*button_cb)(
		GtkWidget *,			/* Splash */
		GdkEventButton *,		/* GdkEventButton */
		gpointer			/* Data */
	);
	gpointer	button_data;

};


/*
 *	Splash "destroy" signal callback.
 */
static void splash_destroy_cb(gpointer data)
{
	GdkColormap *colormap;
	Splash *splash = SPLASH(data);
	if(splash == NULL)
		return;

	if(splash->ref_count > 0)
	{
		g_printerr(
"splash_destroy_cb():\
 Warning: Destroying Splash %p with %i reference counts remaining.\n\
Was splash_unref() not used properly to unref this Splash?\n",
			splash->toplevel,
			splash->ref_count
		);
	}

	splash->freeze_count++;

	/* Remove any queued draws */
	splash->draw_idleid = GTK_IDLE_REMOVE(splash->draw_idleid);

	/* Delete the colors */
	colormap = splash->colormap;
	if(colormap != NULL)
	{
#define DELETE_COLOR(_cp_)	{				\
 if((_cp_) != NULL) {						\
  gdk_colormap_free_colors(colormap, (_cp_), 1);		\
  g_free(_cp_);							\
  (_cp_) = NULL;						\
 }								\
}
		DELETE_COLOR(splash->message_fg_color);
		DELETE_COLOR(splash->message_bg_color);
#undef DELETE_COLOR
	}

	(void)GDK_GC_UNREF(splash->gc);
	(void)GDK_COLORMAP_UNREF(splash->colormap);
	(void)GDK_FONT_UNREF(splash->message_font);
	g_free(splash->xpm_file);
	g_free(splash->rgba);
	g_free(splash->bg_rgba);
	g_free(splash->message);

	splash->freeze_count--;

	g_free(splash);
}

/*
 *	Splash "button_press_event" or "button_release_event" signal callback.
 */
static gint splash_button_cb(GtkWidget *widget, GdkEventButton *button, gpointer data)
{
	Splash *splash = SPLASH(data);
	if((widget == NULL) || (button == NULL) || (splash == NULL))
		return(FALSE);

	if(splash->freeze_count > 0)
		return(FALSE);

	if(splash->button_cb != NULL)
		splash->button_cb(
			splash->toplevel,
			button,
			splash->button_data
		);

	return(TRUE);
}

/*
 *	Splash "expose_event" signal callback.
 */
static gint splash_expose_cb(GtkWidget *widget, GdkEventExpose *expose, gpointer data)
{
	Splash *splash = SPLASH(data);
	if((widget == NULL) || (expose == NULL) || (splash == NULL))
		return(FALSE);

	splash_draw(splash);

	return(TRUE);
}

/*
 *	Splash "configure_event" signal callback.
 */
static gint splash_configure_cb(GtkWidget *widget, GdkEventConfigure *configure, gpointer data)
{
	Splash *splash = SPLASH(data);
	if((widget == NULL) || (configure == NULL) || (splash == NULL))
		return(FALSE);

	return(FALSE);
}

/*
 *	Splash "focus_in_event" or "focus_out_event" signal callback.
 */
static gint splash_focus_cb(GtkWidget *widget, GdkEventFocus *focus, gpointer data)
{
	Splash *splash = SPLASH(data);
	if((widget == NULL) || (focus == NULL) || (splash == NULL))
		return(FALSE);

	if(focus->in && !GTK_WIDGET_HAS_FOCUS(widget))
		GTK_WIDGET_SET_FLAGS(widget, GTK_HAS_FOCUS);
	else if(!focus->in && GTK_WIDGET_HAS_FOCUS(widget))
		GTK_WIDGET_UNSET_FLAGS(widget, GTK_HAS_FOCUS);

	return(TRUE);
}

/*
 *	Toplevel GtkWindow "realize" signal callback.
 */
static void splash_realize_cb(GtkWidget *widget, gpointer data)
{
	GdkColormap *colormap;
	GdkWindow *window;
	GdkGC *gc;
	GtkStyle *style;
	Splash *splash = SPLASH(data);
	if((widget == NULL) || (splash == NULL))
		return;

	/* Get the splash's realized GdkWindow */
	window = widget->window;
	colormap = gtk_widget_get_colormap(widget);
	style = gtk_widget_get_style(widget);
	if((window == NULL) || (colormap == NULL) || (style == NULL))
		return;

	/* Get the GdkColormap as needed */
	if(splash->colormap == NULL)
		splash->colormap = GDK_COLORMAP_REF(colormap);

	/* Create the GC as needed */
	if(splash->gc == NULL)
		splash->gc = gdk_gc_new(window);
	gc = splash->gc;

	/* If the message font was not set then use the style's font */
	if(splash->message_font == NULL)
		splash->message_font = GDK_FONT_REF(style->font);

	/* Allocate the colors as needed */
	if(colormap != NULL)
	{
		if(splash->message_fg_color == NULL)
		{
			GdkColor color = {0l, 0xffff, 0xffff, 0xffff};
			splash->message_fg_color = g_memdup(&color, sizeof(GdkColor));
		}
		GDK_COLORMAP_ALLOC_COLOR(colormap, splash->message_fg_color);

		if(splash->message_bg_color == NULL)
		{
			GdkColor color = {0l, 0x0000, 0x0000, 0x0000};
			splash->message_bg_color = g_memdup(&color, sizeof(GdkColor));
		}
		GDK_COLORMAP_ALLOC_COLOR(colormap, splash->message_bg_color);
	}

	/* Set the toplevel GdkWindow's background GdkPixmap */
	if(splash->bg_rgba != NULL)
	{
		const gint	width = splash->bg_rgba_width,
				height = splash->bg_rgba_height;
		GdkPixmap *pixmap = gdk_pixmap_new(
			window,
			width, height,
			-1
		);
		if(pixmap != NULL)
		{
			gdk_draw_rgb_32_image(
				(GdkDrawable *)pixmap,
				gc,
				0, 0,
				width, height,
				GDK_RGB_DITHER_NONE,
				splash->bg_rgba,
				splash->bg_rgba_bpl
			);
			gdk_window_set_back_pixmap(
				window,
				pixmap,
				FALSE			/* Not parent relative */
			);
			gdk_pixmap_unref(pixmap);
		}
	}
	else if(splash->rgba != NULL)
	{
		const gint	width = splash->rgba_width,
				height = splash->rgba_height;
		GdkPixmap *pixmap = gdk_pixmap_new(
			window,
			width, height,
			-1
		);
		if(pixmap != NULL)
		{
			gdk_draw_rgb_32_image(
				(GdkDrawable *)pixmap,
				gc,
				0, 0,
				width, height,
				GDK_RGB_DITHER_NONE,
				splash->rgba,
				splash->rgba_bpl
			);
			gdk_window_set_back_pixmap(
				window,
				pixmap,
				FALSE			/* Not parent relative */
			);
			gdk_pixmap_unref(pixmap);
		}
	}

	/* Mark the splash as realized */
	splash->flags |= SPLASH_REALIZED;
}

/*
 *	Toplevel GtkWindow "draw" signal callback.
 */
static void splash_draw_cb(
	GtkWidget *widget, GdkRectangle *area, gpointer data
)
{
	Splash *splash = SPLASH(data);
	if((widget == NULL) || (splash == NULL))
		return;

	splash_draw(splash);
}

/*
 *	Draw idle callback.
 */
static gint splash_draw_idle_cb(gpointer data)
{
	Splash *splash = SPLASH(data);
	if(splash == NULL)
		return(FALSE);

	splash->draw_idleid = 0;

	splash_draw(splash);

	return(FALSE);
}


/*
 *	Draws the background image on to the splash toplevel GtkWindow.
 */
static void splash_draw(Splash *splash)
{
	const gint border_major = 5;
	gint width, height;
	GdkDrawable *drawable;
	GdkGC *gc = splash->gc;
	GtkStateType state;
	GtkWidget *w = splash->toplevel;
	GtkStyle *style;

	drawable = (GdkDrawable *)w->window;
	state = GTK_WIDGET_STATE(w);
	style = gtk_widget_get_style(w);
	if((drawable == NULL) || (gc == NULL) || (style == NULL))
		return;

	gdk_window_get_size(
		(GdkWindow *)drawable,
		&width, &height
	);
	if((width <= 0) || (height <= 0))
		return;

	/* Mapped? (not unmapped or in the mapping/unmapping stage */
	if(splash->flags & SPLASH_MAPPED)
	{
		/* Draw the splash image */
		if(splash->rgba != NULL)
			gdk_draw_rgb_32_image(
				drawable,
				gc,
				0, 0,
				splash->rgba_width, splash->rgba_height,
				GDK_RGB_DITHER_NONE,
				splash->rgba,
				splash->rgba_bpl
			);
	}
	else
	{
		/* Draw the background image */
		if(splash->bg_rgba != NULL)
			gdk_draw_rgb_32_image(
				drawable,
				gc,
				0, 0,
				splash->bg_rgba_width, splash->bg_rgba_height,
				GDK_RGB_DITHER_NONE,
				splash->bg_rgba,
				splash->bg_rgba_bpl
			);
		else if(splash->rgba != NULL)
			gdk_draw_rgb_32_image(
				drawable,
				gc,
				0, 0,
				splash->rgba_width, splash->rgba_height,
				GDK_RGB_DITHER_NONE,
				splash->rgba,
				splash->rgba_bpl
			);
	}

	/* Draw the message */
	if((splash->message != NULL) && (splash->message_font != NULL))
	{
		gint x = 0, y = 0;
		const gchar *msg = splash->message;
		GdkFont *font = splash->message_font;
		const gint font_height = font->ascent + font->descent;
		GdkTextBounds *b = &splash->message_bounds;

		gdk_gc_set_font(gc, font);

		switch(splash->message_position)
		{
		  case GTK_POS_TOP:
			switch(splash->message_justify)
			{
			  case GTK_JUSTIFY_LEFT:
				x = (2 * border_major) - b->lbearing;
				y = (2 * border_major) + font->ascent;
				break;
			  case GTK_JUSTIFY_RIGHT:
				x = width - b->width - (2 * border_major) - b->lbearing;
				y = (2 * border_major) + font->ascent;
				break;
			  case GTK_JUSTIFY_CENTER:
			  case GTK_JUSTIFY_FILL:
				x = (width - b->width) / 2;
				y = (2 * border_major) + font->ascent;
				break;
			}
			break;

		  case GTK_POS_BOTTOM:
			switch(splash->message_justify)
			{
			  case GTK_JUSTIFY_LEFT:
				x = (2 * border_major) - b->lbearing;
				y = height - font_height - (2 * border_major) + font->ascent;
				break;
			  case GTK_JUSTIFY_RIGHT:
				x = width - b->width - (2 * border_major) - b->lbearing;
				y = height - font_height - (2 * border_major) + font->ascent;
				break;
			  case GTK_JUSTIFY_CENTER:
			  case GTK_JUSTIFY_FILL:
				x = (width - b->width) / 2;
				y = height - font_height - (2 * border_major) + font->ascent;
				break;
			}
			break;

		  case GTK_POS_LEFT:
		  case GTK_POS_RIGHT:
			switch(splash->message_justify)
			{
			  case GTK_JUSTIFY_LEFT:
				x = (2 * border_major) - b->lbearing;
				y = ((height - font_height) / 2);
				break;
			  case GTK_JUSTIFY_RIGHT:
				x = width - b->width - (2 * border_major) - b->lbearing;
				y = ((height - font_height) / 2);
				break;
			  case GTK_JUSTIFY_CENTER:
			  case GTK_JUSTIFY_FILL:
				x = (width - b->width) / 2;
				y = ((height - font_height) / 2);
				break;
			}
			break;
		}

#define DRAW_STRING_ITERATE(_x_,_y_)	{	\
 gdk_draw_string(				\
  drawable, font, gc,				\
  (_x_), (_y_),					\
  msg						\
 );						\
}
		/* Draw the message background */
		if(splash->message_bg_color != NULL)
		{
			gdk_gc_set_foreground(
				gc,
				splash->message_bg_color
			);
			DRAW_STRING_ITERATE(x - 1, y);
			DRAW_STRING_ITERATE(x, y - 1);
			DRAW_STRING_ITERATE(x + 1, y);
			DRAW_STRING_ITERATE(x, y + 1);
		}

		/* Draw the message foreground */
		if(splash->message_fg_color != NULL)
			gdk_gc_set_foreground(
				gc,
				splash->message_fg_color
			);
		DRAW_STRING_ITERATE(x, y);

#undef DRAW_STRING_ITERATE
	}
}

/*
 *	Queues the splash to draw.
 */
void splash_queue_draw(Splash *splash)
{
	/* Already queued to draw? */
	if(splash->draw_idleid != 0)
		return;

	splash->draw_idleid = gtk_idle_add_priority(
		GTK_PRIORITY_REDRAW,
		splash_draw_idle_cb, splash
	);
}


/*
 *	Deletes all the image buffers on the splash.
 */
static void splash_clear_buffers(Splash *splash)
{
	g_free(splash->rgba);
	splash->rgba = NULL;
	splash->rgba_width = 0;
	splash->rgba_height = 0;
	splash->rgba_bpl = 0;

	g_free(splash->bg_rgba);
	splash->bg_rgba = NULL;
	splash->bg_rgba_width = 0;
	splash->bg_rgba_height = 0;
	splash->bg_rgba_bpl = 0;
}


/*
 *	Creates a "no such file" GdkPixmap.
 */
static GdkPixmap *splash_create_no_such_file_pixmap(
	const gchar *path,
	GdkWindow *window,
	GtkStyle *style,
	GdkBitmap **mask_rtn
)
{
	const gint border_major = 5;
	gint x, y, width, height, font_height;
	gchar *s;
	GdkTextBounds b;
	GdkFont *font;
	GdkPixmap *pixmap;
	GdkDrawable *drawable;
	GtkStateType state;

	if(mask_rtn != NULL)
		*mask_rtn = NULL;

	if((path == NULL) || (window == NULL) || (style == NULL))
		return(NULL);

	font = style->font;
	if(font == NULL)
		return(NULL);

	state = GTK_STATE_NORMAL;
	font_height = font->ascent + font->descent;
	gdk_string_bounds(font, path, &b);
	width = 25 + b.width + (border_major * 2);
	height = (font_height * 3) + (border_major * 2);

	pixmap = gdk_pixmap_new(window, width, height, -1);
	if(pixmap == NULL)
		return(NULL);

	drawable = (GdkDrawable *)pixmap;

	gtk_paint_box(
		style,
		(GdkWindow *)drawable,
		state,
		GTK_SHADOW_OUT,
		NULL,				/* Entire area */
		NULL,				/* No GtkWidget */
		NULL,				/* No detail */
		0, 0,
		width, height
	);

	x = border_major;
	y = border_major;
	s = STRDUP("No such file:");
	gdk_string_bounds(font, s, &b);
	gdk_draw_string(
		drawable, font, style->text_gc[state],
		x - b.lbearing,
		y + font->ascent,
		s
	);
	g_free(s);

	y += font_height;
	x = border_major;
	/* Empty line */

	y += font_height;
	x = border_major + 25;
	gdk_string_bounds(font, path, &b);
	gdk_draw_string(
		drawable, font, style->text_gc[state],
		x - b.lbearing,
		y + font->ascent,
		path
	);

	return(pixmap);
}


/*
 *	Gets the Splash data from the GtkWidget.
 */
static Splash *splash_get_widget_data(
	GtkWidget *w,
	const gchar *func_name
)
{
	const gchar *key = SPLASH_KEY;
	Splash *splash;

	if(w == NULL)
		return(NULL);

	splash = SPLASH(gtk_object_get_data(
		GTK_OBJECT(w),
		key
	));
	if(splash == NULL)
	{
		g_printerr(
"%s(): Warning: GtkWidget %p:\
 Unable to find the data that matches the key \"%s\".\n",
			func_name,
			w,
			key
		);
		return(NULL);
	}

	return(splash);
}

/*
 *	Create a new splash.
 */
GtkWidget *splash_new(void)
{
	GtkWidget *w;
	Splash *splash = SPLASH(g_malloc0(sizeof(Splash)));
	if(splash == NULL)
		return(NULL);

	gtk_widget_push_visual(gdk_rgb_get_visual());
	gtk_widget_push_colormap(gdk_rgb_get_cmap());
	splash->toplevel = w = gtk_window_new(GTK_WINDOW_POPUP);
	gtk_widget_pop_visual();
	gtk_widget_pop_colormap();
	if(w == NULL)
		return(NULL);

	gtk_object_set_data_full(
		GTK_OBJECT(w), SPLASH_KEY,
		splash, splash_destroy_cb
	);
/*
	splash->freeze_count = 0;
 */
	splash->ref_count = 1;
/*
	splash->colormap = NULL;
	splash->gc = NULL;
	splash->flags = 0;

	splash->draw_idleid = 0;
 */

	splash->value = -1.0f;

/*
	splash->message = NULL;
	splash->message_font = NULL;
	splash->message_fg_color = NULL;
	splash->message_bg_color = NULL;
 */
	splash->message_justify = GTK_JUSTIFY_CENTER;
	splash->message_position = GTK_POS_BOTTOM;

	return(splash->toplevel);
}

/*
 *	Set the XPM file.
 */
void SplashSetXPMFile(
	GtkWidget *w,
	const gchar *xpm_file
)
{
	Splash *splash = splash_get_widget_data(
		w,
		"SplashSetXPMFile"
	);
	if(splash == NULL)
		return;

	g_free(splash->xpm_file);
	splash->xpm_file = STRDUP(xpm_file);

	splash->xpm_data = NULL;	/* Unset the XPM data */
}

/*
 *	Set the XPM data.
 */
void splash_set_xpm_data(
	GtkWidget *w,
	const guint8 **xpm_data
)
{
	Splash *splash = splash_get_widget_data(
		w,
		"SplashSetXPMData"
	);
	if(splash == NULL)
		return;

	g_free(splash->xpm_file);	/* Unset the XPM file */
	splash->xpm_file = NULL;

	splash->xpm_data = xpm_data;
}

/*
 *	Sets the message font.
 */
void splash_set_message_font(
	GtkWidget *w,
	GdkFont *font
)
{
	Splash *splash = splash_get_widget_data(
		w,
		"SplashSetMessageFont"
	);
	if((splash == NULL) || (font == NULL))
		return;

	(void)GDK_FONT_UNREF(splash->message_font);
	splash->message_font = font = GDK_FONT_REF(font);

	/* Update the message's string bounds as needed */
	if(splash->message != NULL)
		gdk_string_bounds(
			font,
			splash->message,
			&splash->message_bounds
		);
}

/*
 *	Sets the message's color.
 *
 *	The fg_color specifies the message's foreground GdkColor.
 *	The GdkColor does not need to be allocated.
 *
 *	The bg_color specifies the message's background GdkColor,
 *	The GdkColor does not need to be allocated.
 */
void splash_set_message_color(
	GtkWidget *w,
	GdkColor *fg_color,
	GdkColor *bg_color
)
{
	GdkColormap *colormap;
	Splash *splash = splash_get_widget_data(
		w,
		"SplashSetMessageColor"
	);
	if(splash == NULL)
		return;

	colormap = splash->colormap;
	if(colormap != NULL)
	{
#define SET_COLOR(_cp_,_cv_)	{				\
 if((_cv_) != NULL) {						\
  if((_cp_) != NULL) {						\
   GDK_COLORMAP_FREE_COLOR(colormap, (_cp_));			\
  } else {							\
   (_cp_) = (GdkColor *)g_malloc(sizeof(GdkColor));		\
  }								\
  if((_cp_) != NULL) {						\
   memcpy((_cp_), (_cv_), sizeof(GdkColor));			\
   GDK_COLORMAP_ALLOC_COLOR(colormap, (_cp_));			\
  }								\
 }								\
}
		SET_COLOR(splash->message_fg_color, fg_color);
		SET_COLOR(splash->message_bg_color, bg_color);
#undef SET_COLOR
	}
	else
	{
#define SET_COLOR(_cp_,_cv_)	{				\
 g_free(_cp_);							\
 (_cp_) = ((_cv_) != NULL) ?					\
  g_memdup((_cv_), sizeof(GdkColor)) : NULL;			\
}
		SET_COLOR(splash->message_fg_color, fg_color);
		SET_COLOR(splash->message_bg_color, bg_color);
#undef SET_COLOR
	}
}

/*
 *	Sets the message justification.
 */
void splash_set_message_justification(
	GtkWidget *w,
	const GtkJustification justify
)
{
	Splash *splash = splash_get_widget_data(
		w,
		"SplashSetMessageJustification"
	);
	if(splash == NULL)
		return;

	splash->message_justify = justify;
}

/*
 *	Sets the message position.
 *
 *	The position specifies the message position, if position is
 *	either GTK_POS_LEFT or GTK_POS_RIGHT then the position is
 *	center vertically. splash_set_message_justification() controls
 *	whether the position will actually be left or right.
 */
void splash_set_message_position(
	GtkWidget *w,
	const GtkPositionType position
)
{
	Splash *splash = splash_get_widget_data(
		w,
		"SplashSetMessagePosition"
	);
	if(splash == NULL)
		return;

	splash->message_position = position;
}

/*
 *	Sets the button event signal callback.
 */
void splash_set_button_cb(
	GtkWidget *w,
	void (*cb)(           
		GtkWidget *,			/* Splash */
		GdkEventButton *,			/* GdkEventButton */
		gpointer				/* Data */
	),           
	gpointer data           
)
{
	Splash *splash = splash_get_widget_data(
		w,
		"SplashSetButtonCB"
	);
	if(splash == NULL)
		return;

	splash->button_cb = cb;
	splash->button_data = data;
}

/*
 *	Updates the splash by setting a new value and message.
 *
 *	The splash will be redrawn as needed.
 */
void splash_update_message(
	GtkWidget *w,
	const gfloat v,
	const gchar *msg
)
{
	gboolean value_changed;
	Splash *splash = splash_get_widget_data(
		w,
		"SplashUpdateMessage"
	);
	if(splash == NULL)
		return;

	value_changed = FALSE;

	if(v != splash->value)
	{
		splash->value = v;
		value_changed = TRUE;
	}

	if(splash->message != msg)
	{
		g_free(splash->message);
		splash->message = STRDUP(msg);

		/* Update the message's string bounds if a new message
		 * is set
		 */
		if((msg != NULL) && (splash->message_font != NULL))
			gdk_string_bounds(
				splash->message_font,
				msg,
				&splash->message_bounds
			);

		value_changed = TRUE;
	}

	if((splash->flags & SPLASH_MAPPED) && value_changed)
	{
		splash_draw(splash);
		gdk_flush();
	}
}

/*
 *	Updates and queues the splash to draw.
 *
 *	If manage_events is TRUE then all pending GTK events will
 *	be handled. Note that this will slow down performance
 *	considerably, GTK events do not need to be handled in
 *	order for the splash to be redrawn by this call.
 */
void splash_update(
	GtkWidget *w,
	const gboolean manage_events
)
{
	Splash *splash = splash_get_widget_data(
		w,
		"SplashUpdate"
	);
	if(splash == NULL)
		return;

	if(manage_events)
		gtk_events_process();

	if(splash->flags & SPLASH_MAPPED)
		splash_queue_draw(splash);
}

/*
 *	Maps the splash and performs any mapping effects.
 */
void splash_map(
	GtkWidget *w,
	const SplashEffects effects,
	const gulong duration_ms,
	const GtkWindowPosition position
)
{
	const gint bpp = 4 * sizeof(guint8);
	gint	x, y, width, height,
			root_width, root_height,
			bpl;
	guint8 *frame_rgba;
	GdkModifierType mask;
	GdkWindow *root = gdk_window_get_root();
	GdkDrawable *drawable;
	GdkGC *gc;
	GtkStyle *style;
	Splash *splash = splash_get_widget_data(
		w,
		"SplashMap"
	);
	if(splash == NULL)
		return;

	/* Get the size of the root window */
	gdk_window_get_size(root, &root_width, &root_height);

	/* Delete all the image buffers */
	splash_clear_buffers(splash);

	/* Get the splash image data
	 *
	 * Open from XPM file?
	 */
	if(splash->xpm_file != NULL)
	{
		GdkBitmap *mask;
		GdkPixmap *pixmap = gdk_pixmap_create_from_xpm(
			root,
			&mask,
			NULL,
			splash->xpm_file
		);
		if(pixmap == NULL)
		{
			/* Create an realize a GtkWindow so we have a GtkStyle
			 * to use
			 */
			GtkWidget *w = gtk_window_new(GTK_WINDOW_TOPLEVEL);
			if(w != NULL)
			{
				gtk_widget_realize(w);

				/* Create the "no such file" GdkPixmap */
				pixmap = splash_create_no_such_file_pixmap(
					splash->xpm_file,
					root,
					gtk_widget_get_style(w),
					&mask
				);

				gtk_widget_destroy(w);
			}
		}
		if(pixmap != NULL)
		{
			splash->rgba = gdk_get_rgba_image(
				pixmap,
				NULL,			/* No mask */
				NULL,			/* Entire area */
				&splash->rgba_width,
				&splash->rgba_height,
				&splash->rgba_bpl
			);
			GDK_PIXMAP_UNREF(pixmap);
			GDK_BITMAP_UNREF(mask);
		}
	}
	/* Open from XPM data? */
	else if(splash->xpm_data != NULL)
	{
		GdkBitmap *mask;
		GdkPixmap *pixmap = gdk_pixmap_create_from_xpm_d(
			root,
			&mask,
			NULL,
			(gchar **)splash->xpm_data
		);
		if(pixmap != NULL)
		{
			splash->rgba = gdk_get_rgba_image(
				pixmap,
				NULL,			/* No mask */
				NULL,			/* Entire area */
				&splash->rgba_width,
				&splash->rgba_height,
				&splash->rgba_bpl
			);
			GDK_PIXMAP_UNREF(pixmap);
			GDK_BITMAP_UNREF(mask);
		}
	}

	width = splash->rgba_width;
	height = splash->rgba_height;
	bpl = splash->rgba_bpl;

	if((splash->rgba == NULL) || (width <= 0) || (height <= 0))
		return;

	/* Calculate the position */
	switch(position)
	{
	  case GTK_WIN_POS_NONE:
		x = 0;
		y = 0;
		break;
	  case GTK_WIN_POS_CENTER:
		x = (root_width - width) / 2;
		y = (root_height - height) / 2;
		break;
	  case GTK_WIN_POS_MOUSE:
		(void)gdk_window_get_pointer(root, &x, &y, &mask);
		break;
	  case GTK_WIN_POS_CENTER_ALWAYS:
		x = (root_width - width) / 2;
		y = (root_height - height) / 2;
		break;
	}

	/* Is the splash toplevel GtkWindow realized? */
	if(splash->flags & SPLASH_REALIZED)
	{
		/* Already realized and therefore its values have been
		 * initialized, get its values
		 */
		x = w->allocation.x;
		y = w->allocation.y;
	}
	else
	{
		/* Not realized yet, which means it has been created but
		 * not initialized, so initialize its values here
		 */
		gtk_widget_set_uposition(w, x, y);
		gtk_widget_set_usize(w, width, height);
		gtk_window_set_policy(GTK_WINDOW(w), FALSE, FALSE, FALSE);
		gtk_widget_set_app_paintable(w, TRUE);
		GTK_WIDGET_SET_FLAGS(w, GTK_CAN_FOCUS);
		GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
		gtk_widget_add_events(
			w,
			GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
			GDK_EXPOSURE_MASK | GDK_STRUCTURE_MASK |
			GDK_FOCUS_CHANGE_MASK
		);
		gtk_signal_connect(
			GTK_OBJECT(w), "realize",
			GTK_SIGNAL_FUNC(splash_realize_cb), splash
		);
		gtk_signal_connect(
			GTK_OBJECT(w), "draw",
			GTK_SIGNAL_FUNC(splash_draw_cb), splash
		);
		gtk_signal_connect(
			GTK_OBJECT(w), "expose_event",
			GTK_SIGNAL_FUNC(splash_expose_cb), splash
		);
		gtk_signal_connect(
			GTK_OBJECT(w), "configure_event",
			GTK_SIGNAL_FUNC(splash_configure_cb), splash
		);
		gtk_signal_connect(
			GTK_OBJECT(w), "button_press_event",
			GTK_SIGNAL_FUNC(splash_button_cb), splash
		);
		/* GTK_WINDOW_POPUPs can not receive key events */
		gtk_signal_connect(
			GTK_OBJECT(w), "focus_in_event",
			GTK_SIGNAL_FUNC(splash_focus_cb), splash
		);
		gtk_signal_connect(
			GTK_OBJECT(w), "focus_out_event",
			GTK_SIGNAL_FUNC(splash_focus_cb), splash
		);
	}

	/* Get the background image data as needed */
	switch(effects)
	{
	  case SPLASH_EFFECTS_NONE:
		break;
	  case SPLASH_EFFECTS_FADE_WHITE:
		if((width > 0) && (height > 0))
		{
			/* Create a white RGBA image buffer */
			splash->bg_rgba_width = width;
			splash->bg_rgba_height = height;
			splash->bg_rgba_bpl = bpl;
			splash->bg_rgba = (guint8 *)g_malloc(bpl * height);
			if(splash->bg_rgba != NULL)
			{
				const guint8 pixel[4] = {0xff, 0xff, 0xff, 0xff};
				guint8 *line_ptr;
				gint x, y;

				for(y = 0; y < height; y++)
				{
					line_ptr = splash->bg_rgba + (bpl * y);
					for(x = 0; x < width; x++)
					{
						memcpy(line_ptr, pixel, bpp);
						line_ptr += bpp;
					}
				}
			}
		}
		break;
	  case SPLASH_EFFECTS_FADE_BLACK:
		if((width > 0) && (height > 0))
		{
			/* Create a black RGBA image buffer */
			splash->bg_rgba_width = width;
			splash->bg_rgba_height = height;
			splash->bg_rgba_bpl = bpl;
			splash->bg_rgba = (guint8 *)g_malloc(bpl * height);
			if(splash->bg_rgba != NULL)
			{
				const guint8 pixel[4] = {0x00, 0x00, 0x00, 0xff};
				guint8 *line_ptr;
				gint x, y;

				for(y = 0; y < height; y++)
				{
					line_ptr = splash->bg_rgba + (bpl * y);
					for(x = 0; x < width; x++)
					{
						memcpy(line_ptr, pixel, bpp);
						line_ptr += bpp;
					}
				}
			}
		}
		break;
	  case SPLASH_EFFECTS_FADE_BACKGROUND:
		if((width > 0) && (height > 0))
		{
			/* Copy the RGBA image buffer from the current root
			 * GdkWindow's contents where the splash is going to
			 * be displayed
			 */
			GdkRectangle rect;
			rect.x = x;
			rect.y = y;
			rect.width = width;
			rect.height = height;
			splash->bg_rgba = gdk_get_rgba_image(
				(GdkDrawable *)root,
				NULL,			/* No mask */
				&rect,
				&splash->bg_rgba_width,
				&splash->bg_rgba_height,
				&splash->bg_rgba_bpl
			);
		}
		break;
	}

	/* Allocate the rendering image buffer */
	frame_rgba = (guint8 *)g_malloc(bpl * height);
	if(frame_rgba == NULL)
		return;

	/* Map and realize the splash */
	gtk_widget_show_raise(w);
	splash->flags |= SPLASH_MAPPED;

	/* Get the resources created from the realized toplevel GtkWindow */
	drawable = (GdkDrawable *)w->window;
	gc = splash->gc;
	style = gtk_widget_get_style(w);
	if((drawable == NULL) || (gc == NULL) || (style == NULL))
	{
		g_free(frame_rgba);
		return;
	}

	/* This is required so that the mapping effect starts off
	 * without any skipped "frames" due to previous unprocessed
	 * GDK operations
	 */
	gdk_flush();

	/* Perform the mapping effect */
	switch(effects)
	{
	  case SPLASH_EFFECTS_NONE:
		break;

	  case SPLASH_EFFECTS_FADE_WHITE:
	  case SPLASH_EFFECTS_FADE_BLACK:
	  case SPLASH_EFFECTS_FADE_BACKGROUND:
		if(splash->bg_rgba != NULL)
		{
			/* Fade the splash image in with the background image */
			const gint bg_bpl = splash->bg_rgba_bpl;
			gulong elapsed_ms = 0l;
			gfloat	coeff,
					inv_coeff;
			const guint8	*bg_line,
							*bg_ptr,
							*src_line,
							*src_line_end,
							*src_end,
							*src_ptr;
			guint8	*tar_line,
					*tar_ptr;
			GTimer *timer = g_timer_new();
			g_timer_start(timer);
			while(elapsed_ms < duration_ms)
			{
				coeff = (gfloat)elapsed_ms / (gfloat)duration_ms;
				inv_coeff = 1.0f - coeff;

				/* Blend the splash image on to the background image */
				for(bg_line = splash->bg_rgba,
					src_line = splash->rgba,
					src_end = src_line + (height * bpl),
					tar_line = frame_rgba;
					src_line < src_end;
					bg_line += bg_bpl,
					src_line += bpl,
					tar_line += bpl
				)
				{
					for(bg_ptr = bg_line,
						src_ptr = src_line,
						src_line_end = src_ptr + (width * bpp),
						tar_ptr = tar_line;
						src_ptr < src_line_end;
						bg_ptr += bpp,
						src_ptr += bpp,
						tar_ptr += bpp
					)
					{
						tar_ptr[0] = (guint8)(
							(bg_ptr[0] * inv_coeff) +
							(src_ptr[0] * coeff)
						);
						tar_ptr[1] = (guint8)(
							(bg_ptr[1] * inv_coeff) +
							(src_ptr[1] * coeff)
						);
						tar_ptr[2] = (guint8)(
							(bg_ptr[2] * inv_coeff) +
							(src_ptr[2] * coeff)
						);
						tar_ptr[3] = 0xff;
					}
				}

				/* Put the rendering image buffer on to the GtkDrawable */
				gdk_draw_rgb_32_image(
					drawable,
					gc,
					0, 0, width, height,
					GDK_RGB_DITHER_NONE,
					frame_rgba,
					bpl
				);

				/* Update the elapsed time */
				elapsed_ms = G_TIMER_ELAPSED_MS(timer);
			}
			G_TIMER_DESTROY(timer);

			/* Do not draw the last frame, it will be drawn
			 * outside of this case
			 */
		}
		break;
	}

	/* Delete the rendering image buffer */
	g_free(frame_rgba);

	/* Draw the final foreground image */
	splash_draw(splash);

	/* This is required to ensure that the final draw is completed
	 * after the mapping effect so that we do not see any partial
	 * mapping effect "frames" left over after this call
	 */
	gdk_flush();
}

/*
 *	Unmaps the splash and performs any mapping effects.
 */
void splash_unmap(
	GtkWidget *w,
	const SplashEffects effects,
	const gulong duration_ms
)
{
	const gint bpp = 4;			/* RGBA */
	gint		width, height,
					bpl;
	guint8 *frame_rgba;
	GdkGC *gc;
	Splash *splash = splash_get_widget_data(
		w,
		"SplashUnmap"
	);
	if(splash == NULL)
		return;

	/* If the splash was not mapped in the frist place then just
	 * unmap it simply
	 */
	if(!(splash->flags & SPLASH_MAPPED))
	{
		gtk_widget_hide(w);
		return;
	}

	width = splash->rgba_width;
	height = splash->rgba_height;
	bpl = splash->rgba_bpl;
	gc = splash->gc;
	if((width <= 0) || (height <= 0) || (gc == NULL))
	{
		gtk_widget_hide(w);
		splash->flags &= ~SPLASH_MAPPED;
		return;
	}

	/* Allocate the rendering image buffer */
	frame_rgba = (guint8 *)g_malloc(bpl * height);
	if(frame_rgba == NULL)
	{
		gtk_widget_hide(w);
		splash->flags &= ~SPLASH_MAPPED;
		return;
	}

	/* This is required so that the unmapping effect starts off
	 * without any skipped "frames" due to previous unprocessed
	 * GDK operations
	 */
	gdk_flush();

	/* Perform the unmapping effect */
	switch(effects)
	{
	  case SPLASH_EFFECTS_NONE:
		break;

	  case SPLASH_EFFECTS_FADE_WHITE:
	  case SPLASH_EFFECTS_FADE_BLACK:
	  case SPLASH_EFFECTS_FADE_BACKGROUND:
		if(splash->bg_rgba != NULL)
		{
			/* Fade the splash image in with the background image */
			const gint bg_bpl = splash->bg_rgba_bpl;
			gulong elapsed_ms = 0l;
			gfloat	coeff,
					inv_coeff;
			const guint8	*bg_line,
							*bg_ptr,
							*src_line,
							*src_line_end,
							*src_end,
							*src_ptr;
			guint8	*tar_line,
					*tar_ptr;
			GdkDrawable *drawable = (GdkDrawable *)w->window;
			GTimer *timer = g_timer_new();
			g_timer_start(timer);
			while(elapsed_ms < duration_ms)
			{
				inv_coeff = (gfloat)elapsed_ms / (gfloat)duration_ms;
				coeff = 1.0f - inv_coeff;

				/* Blend the splash image on to the background image */
				for(bg_line = splash->bg_rgba,
					src_line = splash->rgba,
					src_end = src_line + (height * bpl),
					tar_line = frame_rgba;
					src_line < src_end;
					bg_line += bg_bpl,
					src_line += bpl,
					tar_line += bpl
				)
				{
					for(bg_ptr = bg_line,
						src_ptr = src_line,
						src_line_end = src_ptr + (width * bpp),
						tar_ptr = tar_line;
						src_ptr < src_line_end;
						bg_ptr += bpp,
						src_ptr += bpp,
						tar_ptr += bpp
					)
					{
						tar_ptr[0] = (guint8)(
							(bg_ptr[0] * inv_coeff) +
							(src_ptr[0] * coeff)
						);
						tar_ptr[1] = (guint8)(
							(bg_ptr[1] * inv_coeff) +
							(src_ptr[1] * coeff)
						);
						tar_ptr[2] = (guint8)(
							(bg_ptr[2] * inv_coeff) +
							(src_ptr[2] * coeff)
						);
						tar_ptr[3] = 0xff;
					}
				}

				/* Put the rendering image buffer on to the GtkDrawable */
				gdk_draw_rgb_32_image(
					drawable,
					gc,
					0, 0, width, height,
					GDK_RGB_DITHER_NONE,
					frame_rgba,
					bpl
				);

				/* Update the elapsed time */
				elapsed_ms = G_TIMER_ELAPSED_MS(timer);
			}
			G_TIMER_DESTROY(timer);

			/* Draw the last frame */
			gdk_draw_rgb_32_image(
				drawable,
				gc,
				0, 0,
				splash->bg_rgba_width, splash->bg_rgba_height,
				GDK_RGB_DITHER_NONE,
				splash->bg_rgba,
				splash->bg_rgba_bpl
			);
		}
		break;
	}

	/* Delete the rendering image buffer */
	g_free(frame_rgba);

	/* This is required so that the unmapping effect finishes
	 * with the last frame before the splash is unmapped
	 */
	gdk_flush();

	/* Unmap the splash */
	gtk_widget_hide(w);
	splash->flags &= ~SPLASH_MAPPED;

	/* Delete all the image buffers */
	splash_clear_buffers(splash);

	/* This is required to ensure that the splash has been actually
	 * unmapped before the end of this call
	 */
	gdk_flush();
}

/*
 *	Adds a reference count to the Splash.
 *
 *	Returns the Splash or NULL on error.
 */
GtkWidget *splash_ref(GtkWidget *w)
{
	Splash *splash = splash_get_widget_data(
		w,
		"SplashRef"
	);
	if(splash == NULL)
		return(NULL);

	splash->ref_count++;

	return(w);
}

/*
 *	Removes a reference count from the Splash.
 *
 *	If the reference counts reach 0 then the Splash will be
 *	destroyed.
 *
 *	Always returns NULL.
 */
GtkWidget *splash_unref(GtkWidget *w)
{
	Splash *splash = splash_get_widget_data(
		w,
		"SplashUnref"
	);
	if(splash == NULL)
		return(NULL);

	splash->ref_count--;

	if(splash->ref_count <= 0)
		gtk_widget_destroy(w);

	return(NULL);
}
