#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"

#include "cfg.h"

#include "guiutils.h"
#include "cfg_gtk.h"
#include "cdialog.h"
#include "fb.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_process.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "run_dlg.h"
#include "endeavour2.h"
#include "edv_op.h"
#include "edv_utils_gtk.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_run_20x20.xpm"
#include "images/icon_executable_48x48.xpm"
#include "images/icon_close_20x20.xpm"
#include "images/icon_folder_opened_20x20.xpm"
#include "images/icon_clear_20x20.xpm"
#include "images/icon_run_dir_from_cmd_20x20.xpm"
#include "images/icon_options_20x20.xpm"


static gint edv_run_dlg_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void edv_run_dlg_run_cb(GtkWidget *widget, gpointer data);
static gint edv_run_dlg_command_focus_event_cb(
	 GtkWidget *widget, GdkEventFocus *focus, gpointer data
);
static void edv_run_dlg_browser_command_cb(GtkWidget *widget, gpointer data);
static void edv_run_dlg_browse_working_directory_cb(GtkWidget *widget, gpointer data);
static void edv_run_dlg_clear_cb(GtkWidget *widget, gpointer data);
static void edv_run_dlg_get_working_irectory_from_command_cb(GtkWidget *widget, gpointer data);
static void edv_run_dlg_options_cb(GtkWidget *widget, gpointer data);
static void edv_run_dlg_close_cb(GtkWidget *widget, gpointer data);

/* Utilities */
static void edv_run_dlg_complete_command(EDVRunDlg *d);

/* Set & Get Values */
void edv_run_dlg_set_command(
	EDVRunDlg *d,
	const char *command
);
void edv_run_dlg_set_working_directory(
	EDVRunDlg *d,
	const char *path
);
void edv_run_dlg_get_values(EDVRunDlg *d);

/* Operations */
static void edv_run_dlg_run(
	EDVRunDlg *d,
	const gboolean record_command
);
static void edv_run_dlg_close(
	EDVRunDlg *d,
	const gboolean save_config
);

/* Run Dialog */
EDVRunDlg *edv_run_dlg_new(EDVCore *core);
void edv_run_dlg_set_busy(EDVRunDlg *d, const gboolean busy);
gboolean edv_run_dlg_is_mapped(EDVRunDlg *d);
void edv_run_dlg_map(EDVRunDlg *d);
void edv_run_dlg_unmap(EDVRunDlg *d);
void edv_run_dlg_delete(EDVRunDlg *d);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


#if defined(PROG_LANGUAGE_SPANISH)
# define RUNDLG_TITLE		"Corra"
#elif defined(PROG_LANGUAGE_FRENCH)
# define RUNDLG_TITLE		"Lancer"
#elif defined(PROG_LANGUAGE_GERMAN)
# define RUNDLG_TITLE		"Lauf"
#elif defined(PROG_LANGUAGE_ITALIAN)
# define RUNDLG_TITLE		"Ha Correto"
#elif defined(PROG_LANGUAGE_DUTCH)
# define RUNDLG_TITLE		"Tocht"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
# define RUNDLG_TITLE		"A Corrida"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
# define RUNDLG_TITLE		"Lp"
#else
# define RUNDLG_TITLE		"Run"
#endif

#define RUNDLG_WIDTH		500
#define RUNDLG_HEIGHT		-1


/*
 *	Toplevel GtkWindow "delete_event" signal callback.
 */
static gint edv_run_dlg_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	EDVRunDlg *d = EDV_RUN_DLG(data);
	if(d == NULL)
		return(FALSE);

	if(d->freeze_count > 0)
		return(TRUE);

	edv_run_dlg_close(d, TRUE);

	return(TRUE);
}

/*
 *	Run callback.
 */
static void edv_run_dlg_run_cb(GtkWidget *widget, gpointer data)
{
	EDVRunDlg *d = EDV_RUN_DLG(data);
	if(d == NULL)
		return;

	if(d->freeze_count > 0)
		return;

	edv_run_dlg_run(d, TRUE);
}


/*
 *	Run Dialog Command GtkEntry "focus_in_event" or
 *	"focus_out_event" signal callback.
 */
static gint edv_run_dlg_command_focus_event_cb(
	 GtkWidget *widget, GdkEventFocus *focus, gpointer data
)
{
	gint status = FALSE;
	EDVRunDlg *d = EDV_RUN_DLG(data);
	if((widget == NULL) || (focus == NULL) || (d == NULL))
		return(status);

	if(focus->in)
	{
		status = TRUE;
	}
	else
	{
		/* Complete the command as needed when focusing out */
		edv_run_dlg_complete_command(d);
		status = TRUE;
	}

	return(status);
}


/*
 *	Run Dialog browse command callback.
 */
static void edv_run_dlg_browser_command_cb(GtkWidget *widget, gpointer data)
{
	gboolean response;
	gint		npaths = 0,
					nftypes = 0;
	const gchar *cmd;
	gchar		*prog,
					*args,
					**paths_list = NULL;
	GtkWidget *toplevel;
	GtkEntry *entry;
	GtkCombo *combo;
	fb_type_struct	**ftypes_list = NULL,
					*ftype_rtn = NULL;
	EDVRunDlg *d = EDV_RUN_DLG(data);
	if((d == NULL) || FileBrowserIsQuery())
		return;

	if(d->freeze_count > 0)
		return;

	toplevel = d->toplevel;
	combo = GTK_COMBO(d->run_combo);
	entry = GTK_ENTRY(combo->entry);

	edv_run_dlg_set_busy(d, TRUE);

	/* Get the current command, program, and argumets */
	cmd = gtk_entry_get_text(entry);
	args = STRDUP(edv_strarg(
		cmd,
		&prog,
		FALSE,				/* Parse escapes */
		TRUE				/* Parse quotes */
	));

	/* If the program is not an absolute path then do not use it */
	if(prog != NULL)
	{
		if(!g_path_is_absolute(prog))
		{
			g_free(prog);
			prog = NULL;
		}
	}

	/* Create the file types list */
	FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		"*.*", "All Files"
	);

	/* Query the user for a program */
	d->freeze_count++;
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
		"Select Program",
		"Select", "Cancel",
		prog,
		ftypes_list, nftypes,
		&paths_list, &npaths,
		&ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);
	d->freeze_count--;

	/* Got response? */
	if(response && (npaths > 0))
	{
		gchar *path = STRDUP(paths_list[npaths - 1]);
		if(path != NULL)
		{
			gchar *cmd_new;
			if(args != NULL)
			{
				if(strpbrk((const char *)path, " \t") != NULL)
					cmd_new = g_strconcat(
						"\"",
						path,
						"\" ",
						args,
						NULL
					);
				else
					cmd_new = g_strconcat(
						path,
						" ",
						args,
						NULL
					);
			}
			else
			{
				if(strpbrk((const char *)path, " \t") != NULL)
					cmd_new = g_strconcat(
						"\"",
						path,
						"\"",
						NULL
					);
				else
					cmd_new = g_strdup(path);
			}
			if(cmd_new != NULL)
			{
				cmd_new = g_strstrip(cmd_new);
				gtk_entry_set_text(entry, cmd_new);
				gtk_entry_set_position(entry, -1);
				g_free(cmd_new);
			}

			g_free(path);
		}
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	g_free(prog);
	g_free(args);

	edv_run_dlg_set_busy(d, FALSE);
}

/*
 *	Run Dialog browse working directory callback.
 */
static void edv_run_dlg_browse_working_directory_cb(GtkWidget *widget, gpointer data)
{
	gboolean response;
	gint		npaths = 0,
					nftypes = 0;
	gchar		*cur_dir,
					**paths_list = NULL;
	GtkWidget *toplevel;
	GtkEntry *entry;
	fb_type_struct	**ftypes_list = NULL,
					*ftype_rtn = NULL;
	EDVRunDlg *d = EDV_RUN_DLG(data);
	if((d == NULL) || FileBrowserIsQuery())
		return;

	if(d->freeze_count > 0)
		return;

	toplevel = d->toplevel;
	entry = GTK_ENTRY(d->working_dir_entry);
	cur_dir = STRDUP(gtk_entry_get_text(entry));

	edv_run_dlg_set_busy(d, TRUE);

	/* Create the file types list */
	FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		"*.*", "All Files"
	);

	/* Query the user for a directory */
	d->freeze_count++;
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
		"Select Directory",
		"Select", "Cancel",
		cur_dir,
		ftypes_list, nftypes,
		&paths_list, &npaths,
		&ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);
	d->freeze_count--;

	/* Got response? */
	if(response && (npaths > 0))
	{
		gchar *path = STRDUP(paths_list[npaths - 1]);
		if(path != NULL)
		{
			gtk_entry_set_text(entry, path);
			gtk_entry_set_position(entry, -1);
			g_free(path);
		}
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	g_free(cur_dir);

	edv_run_dlg_set_busy(d, FALSE);
}

/*
 *	Clear callback.
 *
 *	The data must be a GtkEntry.
 */
static void edv_run_dlg_clear_cb(GtkWidget *widget, gpointer data)
{
	GtkEntry *entry;
	GtkWidget *w = (GtkWidget *)data;
	if(w == NULL)
		return;

	if(!GTK_IS_ENTRY(w))
		return;

	entry = GTK_ENTRY(w);

/*	edv_run_dlg_set_busy(d, TRUE); */

	gtk_entry_set_text(entry, "");
	gtk_entry_set_position(entry, 0);

/*	edv_run_dlg_set_busy(d, FALSE); */
}

/*
 *	Run dialog get working directory from current run command value
 *	callback.
 */
static void edv_run_dlg_get_working_irectory_from_command_cb(GtkWidget *widget, gpointer data)
{
	const gchar *cmd;
	GtkEntry *entry;
	GtkCombo *combo;
	EDVRunDlg *d = EDV_RUN_DLG(data);
	if(d == NULL)
		return;

	if(d->freeze_count > 0)
		return;

	edv_run_dlg_set_busy(d, TRUE);

	combo = GTK_COMBO(d->run_combo);
	entry = GTK_ENTRY(d->working_dir_entry);

	/* Get the current command */
	cmd = gtk_entry_get_text(GTK_ENTRY(combo->entry));
	if(cmd != NULL)
	{
		while(ISBLANK(*cmd))
			cmd++;

		/* Run command starts with an absolute path? */
		if(*cmd == G_DIR_SEPARATOR)
		{
			gchar	*s, *parent_path,
					*new_working_dir = STRDUP(cmd);

			/* Seek strptr to first space (if any) in
			 * new_working_dir
			 */
			s = new_working_dir;
			while(!ISBLANK(*s) && (*s != '\0'))
				s++;
			/* Cap new_working_dir at first blank character */
			*s = '\0';

			parent_path = g_dirname(new_working_dir);

			gtk_entry_set_text(entry, parent_path);
			gtk_entry_set_position(entry, -1);

			g_free(new_working_dir);
			g_free(parent_path);
		}
	}

	edv_run_dlg_set_busy(d, FALSE);
}

/*
 *	Options button callback.
 */
static void edv_run_dlg_options_cb(GtkWidget *widget, gpointer data)
{
	GtkWidget *toplevel;
	EDVCore *core;
	EDVRunDlg *d = EDV_RUN_DLG(data);
	if(d == NULL)
		return;

	if(d->freeze_count > 0)
		return;

	edv_run_dlg_set_busy(d, TRUE);

	toplevel = d->toplevel;
	core = d->core;

	edv_map_options_page(
		core,
		"Programs",				/* Page Name */
		core->geometry_flags,
		(core->geometry_flags != 0) ? &core->geometry : NULL,
		d->toplevel
	);

	edv_run_dlg_set_busy(d, FALSE);
}

/*
 *	Run dialog close button callback.
 */
static void edv_run_dlg_close_cb(GtkWidget *widget, gpointer data)
{
	EDVRunDlg *d = EDV_RUN_DLG(data);
	if(d == NULL)
		return;

	if(d->freeze_count > 0)
		return;

	edv_run_dlg_close(d, TRUE);
}


/*
 *	Completes the Run Dialog's command as needed and updates the
 *	command.
 */
static void edv_run_dlg_complete_command(EDVRunDlg *d)
{
	/* Get the current command */
	GtkCombo *combo = GTK_COMBO(d->run_combo);
	const gchar *cmd_src = gtk_entry_get_text(
		GTK_ENTRY(combo->entry)
	);
	if(!STRISEMPTY(cmd_src))
	{
		/* Get the current program and arguments */
		gchar	*prog,
			*args = STRDUP(edv_strarg(
				cmd_src,
				&prog,
				FALSE,		/* Do not parse escapes */
				TRUE		/* Parse quotes */
		));

		/* If the program location is not an absolute path then
		 * we need to complete it
		 */
		if((prog != NULL) ? !g_path_is_absolute(prog) : FALSE)
		{
			gchar *prog_full_path = edv_which(prog);
			if(prog_full_path != NULL)
			{
				gchar *cmd_new;
				GtkCombo *combo = GTK_COMBO(d->run_combo);

				/* Set the new program with the completed path */
				g_free(prog);
				prog = prog_full_path;

				/* Format and set the new command */
				if(args != NULL)
				{
					if(strpbrk((const char *)prog, " \t") != NULL)
						cmd_new = g_strconcat(
							"\"",
							prog,
							"\" ",
							args,
							NULL
						);
					else
						cmd_new = g_strconcat(
							prog,
							" ",
							args,
							NULL
						);
				}
				else
				{
					if(strpbrk((const char *)prog, " \t") != NULL)
						cmd_new = g_strconcat(
							"\"",
							prog,
							"\"",
							NULL
						);
					else
						cmd_new = STRDUP(prog);
				}
				if(cmd_new != NULL)
				{
					cmd_new = g_strstrip(cmd_new);
					gtk_entry_set_text(GTK_ENTRY(combo->entry), cmd_new);
					g_free(cmd_new);
				}
			}
		}

		g_free(prog);
		g_free(args);
	}
}


/*
 *	Sets the command value.
 */
void edv_run_dlg_set_command(
	EDVRunDlg *d,
	const char *command
)
{
	GtkEntry *entry;
	GtkCombo *combo;

	if(d == NULL)
		return;

	combo = GTK_COMBO(d->run_combo);
	entry = GTK_ENTRY(combo->entry);
	gtk_entry_set_text(
		entry,
		(command != NULL) ? command : ""
	);
	gtk_entry_set_position(entry, -1);
}

/*
 *	Sets the working directory value.
 */
void edv_run_dlg_set_working_directory(
	EDVRunDlg *d,
	const char *path
)
{
	GtkEntry *entry;

	if(d == NULL)
		return;

	entry = GTK_ENTRY(d->working_dir_entry);
	gtk_entry_set_text(
		entry,
		(path != NULL) ? path : ""
	);
	gtk_entry_set_position(entry, -1);
}

/*
 *	Regets list of past runned objects and updates the widgets to
 *	match the current configuration.
 */
void edv_run_dlg_get_values(EDVRunDlg *d)
{
	GtkCombo *combo;
	const CfgItem *cfg_list;
	EDVCore *core;

	if(d == NULL)
		return;

	core = d->core;
	cfg_list = core->cfg_list;

	/* Begin fetching values from global configuration and
	 * updating the Run Dialog's widgets
	 */

	/* Run History */
	combo = GTK_COMBO(d->run_combo);
	if(combo != NULL)
	{
		const gchar *path = EDV_GET_S(EDV_CFG_PARM_FILE_RUNDLG_HISTORY);
		GList *glist = edv_open_text_file_glist(
			path,
			-1,				/* No lines limit */
			TRUE				/* Strip CRs */
		);
		if(glist != NULL)
		{
			/* Set the new glist to the run combo's history, this
			 * glist will be managed by the call and should not
			 * be referenced afterwards
			 */
			GUIComboSetList(GTK_WIDGET(combo), glist);
			glist = NULL;
		}
	}

	/* Run In Terminal */
	CfgGtkSetGtkToggleButton(
		cfg_list,
		EDV_CFG_PARM_RUN_DLG_RUN_IN_TERMINAL,
		d->run_in_terminal_check
	);

	/* Keep Dialog */
	CfgGtkSetGtkToggleButton(
		cfg_list,
		EDV_CFG_PARM_RUN_DLG_KEEP_DIALOG,
		d->keep_dialog_check
	);
}


/*
 *	Run.
 *
 *	If record_command is TRUE then the command will be added to the
 *	run GtkCombo's list.
 */
static void edv_run_dlg_run(
	EDVRunDlg *d,
	const gboolean record_command
)
{
	gint euid;
	gchar		*cmd,
			*cmd_src,
			*wd;
	CfgList *cfg_list;
	GtkWidget *toplevel;
	EDVCore *core;

	if(d == NULL)
		return;

	toplevel = d->toplevel;
	core = d->core;
	cfg_list = core->cfg_list;
	euid = core->effective_user_id;

	edv_run_dlg_set_busy(d, TRUE);

	/* Complete the command as needed */
	edv_run_dlg_complete_command(d);

	/* Get the working directory */
	wd = STRDUP(gtk_entry_get_text(GTK_ENTRY(d->working_dir_entry)));

	/* Get the command */
	cmd_src = STRDUP(gtk_entry_get_text(
		GTK_ENTRY(GTK_COMBO(d->run_combo)->entry)
	));
	if(STRISEMPTY(cmd_src))
	{
		g_free(wd);
		g_free(cmd_src);
		edv_run_dlg_set_busy(d, FALSE);
		return;
	}

	/* Record this command in the command GtkCombo's list */
	if(record_command)
	{
		GUIComboAddItem(d->run_combo, cmd_src);
	}

	/* Check if the program is executable */
	if(cmd_src != NULL)
	{
		/* Get the program path from the command */
		gchar *prog;
		(void)edv_strarg(
			cmd_src,
			&prog,
			TRUE,				/* Parse escapes */
			TRUE				/* Parse quotes */
		);
		if(prog != NULL)
		{
			EDVPermissionFlags permissions;
			EDVVFSObject *obj;

			/* Not an absolute path? */
			if(!g_path_is_absolute(prog))
			{
				edv_play_sound_warning(core);
				edv_message_warning(
					"Run Failed",
"The path to the program must be an absolute path.",
					NULL,
					toplevel
				);
				g_free(prog);
				g_free(wd);
				g_free(cmd_src);
				edv_run_dlg_set_busy(d, FALSE);
				return;
			}
			/* Does not exist? */
			obj = edv_vfs_object_stat(prog);
			if(obj == NULL)
			{
				const gint error_code = (gint)errno;
				gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
					g_strerror(error_code),
					prog
				);
				edv_play_sound_warning(core);
				edv_message_warning(
					"Run Failed",
					msg,
					NULL,
					toplevel
				);
				g_free(msg);
				g_free(prog);
				g_free(wd);
				g_free(cmd_src);
				edv_run_dlg_set_busy(d, FALSE);
				return;
			}
			/* Not executable? */
			permissions = obj->permissions;
			if(!(permissions & EDV_PERMISSION_UX) &&
			   !(permissions & EDV_PERMISSION_GX) &&
			   !(permissions & EDV_PERMISSION_OX)
			)
			{
				gchar *msg = g_strdup_printf(
"Not executable:\n\
\n\
    %s",
					prog
				);
				edv_play_sound_warning(core);
				edv_message_warning(
					"Run Failed",
					msg,
					NULL,
					toplevel
				);
				g_free(msg);
				edv_vfs_object_delete(obj);
				g_free(prog);
				g_free(wd);
				g_free(cmd_src);
				edv_run_dlg_set_busy(d, FALSE);
				return;
			}
			edv_vfs_object_delete(obj);
			g_free(prog);
		}
	}

	/* Check if the working directory is valid */
	if(wd != NULL)
	{
		EDVPermissionFlags permissions;
		EDVVFSObject *obj;

		/* Not an absolute path? */
		if(!g_path_is_absolute(wd))
		{
			edv_play_sound_warning(core);
			edv_message_warning(
				"Run Failed",
"The working directory path must be an absolute path.",
				NULL,
				toplevel
			);
			g_free(wd);
			g_free(cmd_src);
			edv_run_dlg_set_busy(d, FALSE);
			return;
		}
		/* Does not exist? */
		obj = edv_vfs_object_stat(wd);
		if(obj == NULL)
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
				g_strerror(error_code),
				wd
			);
			edv_play_sound_warning(core);
			edv_message_warning(
				"Run Failed",
				msg,
				NULL,
				toplevel
			);
			g_free(msg);
			g_free(wd);
			g_free(cmd_src);
			edv_run_dlg_set_busy(d, FALSE);
			return;
		}
		/* Not executable? */
		permissions = obj->permissions;
		if(!(permissions & EDV_PERMISSION_UX) &&
		   !(permissions & EDV_PERMISSION_GX) &&
		   !(permissions & EDV_PERMISSION_OX)
		)
		{
			gchar *msg = g_strdup_printf(
"Not executable:\n\
\n\
    %s",
				wd
			);
			edv_play_sound_warning(core);
			edv_message_warning(
				"Run Failed",
				msg,
				NULL,
				toplevel
			);
			g_free(msg);
			edv_vfs_object_delete(obj);
			g_free(wd);
			g_free(cmd_src);
			edv_run_dlg_set_busy(d, FALSE);
			return;
		}
		edv_vfs_object_delete(obj);
	}

	/* Format the target command */
	if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->run_in_terminal_check))
	{
		/* Format the command for running in a terminal */
		const gchar *cmd_terminal_run = EDV_GET_S(
			EDV_CFG_PARM_PROG_TERMINAL_RUN
		);
		if(!STRISEMPTY(cmd_terminal_run))
		{
			/* Format the command for run in terminal */
			cmd = g_strconcat(
				cmd_terminal_run,
				" ",
				cmd_src,
				NULL
			);
		}
		else
		{
			edv_play_sound_warning(core);
			edv_message_warning(   
				"Run Failed",
"The \"Run In Terminal\" command is not set.\n\
\n\
To set the \"Run In Terminal\" command go to\n\
Settings->Options...->Programs.",
				NULL,
				toplevel
			);
			g_free(wd);
			g_free(cmd_src);
			edv_run_dlg_set_busy(d, FALSE);
			return;
		}
	}
	else
	{
		/* Format a plain command */
		cmd = g_strconcat(
			cmd_src,
			NULL
		);
	}

	/* Generated the target command successfully? */
	if(cmd != NULL)
	{
		gint p;
		gchar *shell_prog;
		const gchar	*shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
					*shell_args = edv_strarg(
			shell_cmd,
			&shell_prog,
			TRUE,				/* Parse escapes */
			TRUE				/* Parse quotes */
		);

		/* Get the current working directory */
		gchar *cwd = edv_getcwd();

		/* Change to the new working directory */
		edv_setcwd(wd);

		/* Execute the command */
		p = edv_system_shell(
			cmd,
			shell_prog,
			shell_args
		);
		if(p < 0)
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"Unable to execute the command:\n\
\n\
    %s\n\
\n\
%s.",
				cmd,
				g_strerror(error_code)
			);
			edv_play_sound_error(core);
			edv_message_error(
"Run Failed",
				msg,
"Please check to make sure that the path to the program\n\
is correct and that the program is set executable.",
				toplevel
			);
			g_free(msg);
		}

		g_free(cmd);
		g_free(shell_prog);

		/* Change back to the previous working directory */
		if(cwd != NULL)
		{
			edv_setcwd(cwd);
			g_free(cwd);
		}
	}

	/* Do not keep dialog after running? */
	if(!GTK_TOGGLE_BUTTON_GET_ACTIVE(d->keep_dialog_check))
		edv_run_dlg_close(d, TRUE);

	g_free(wd);
	g_free(cmd_src);

	edv_run_dlg_set_busy(d, FALSE);
}

/*
 *	Closes the Run Dialog.
 *
 *	If save_config is TRUE then the Run Dialog's values will
 *	be saved to the configuration.
 */
static void edv_run_dlg_close(
	EDVRunDlg *d,
	const gboolean save_config
)
{
	CfgItem *cfg_list;
	EDVCore *core;

	if(d == NULL)
		return;

	core = d->core;
	cfg_list = core->cfg_list;

	if(save_config)
	{
		/* Run history */
		edv_save_text_file_glist(
			EDV_GET_S(EDV_CFG_PARM_FILE_RUNDLG_HISTORY),
			GUIComboGetList(d->run_combo)
		);

		/* Toplevel position and size */
		CfgGtkGetGtkWindow(
			cfg_list,
			EDV_CFG_PARM_RUN_DLG_X,
			EDV_CFG_PARM_RUN_DLG_Y,
			NULL,
			NULL,
			d->toplevel
		);

		/* Last working directory */
		CfgGtkGetGtkEditable(
			cfg_list,
			EDV_CFG_PARM_RUN_DLG_WORKING_DIR,
			d->working_dir_entry
		);

		/* Run in terminal */
		CfgGtkGetGtkToggleButton(
			cfg_list,
			EDV_CFG_PARM_RUN_DLG_RUN_IN_TERMINAL,
			d->run_in_terminal_check
		);

		/* Keep dialog */
		CfgGtkGetGtkToggleButton(
			cfg_list,
			EDV_CFG_PARM_RUN_DLG_KEEP_DIALOG,
			d->keep_dialog_check
		);
	}

	/* Close the dialog */
	edv_run_dlg_unmap(d);
}


/*
 *	Creates a new Run Dialog.
 */
EDVRunDlg *edv_run_dlg_new(EDVCore *core)
{
	const gint	border_major = 5,
					border_minor = 2;
	gpointer combo_rtn;
	GdkWindow *window;
	GtkStyle *style;
	GtkAccelGroup *accelgrp;
	GtkWidget	*w,
					*parent, *parent2,
					*toplevel;
	GtkCombo *combo;
	CfgItem *cfg_list;
	EDVRunDlg *d;

	if(core == NULL)
		return(NULL);

	cfg_list = core->cfg_list;

	/* Create the Run Dialog */
	d = EDV_RUN_DLG(g_malloc0(sizeof(EDVRunDlg)));
	if(d == NULL)
		return(NULL);

	d->toplevel = toplevel = gtk_window_new(GTK_WINDOW_DIALOG);
	d->accelgrp = accelgrp = gtk_accel_group_new();
	d->processing = FALSE;
	d->busy_count = 0;
	d->freeze_count = 0;
	d->core = core;

	d->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_window_set_wmclass(
		GTK_WINDOW(w),
		"dialog",
		PROG_NAME
	);
	gtk_window_set_policy(GTK_WINDOW(w), FALSE, FALSE, FALSE);
	gtk_window_set_title(GTK_WINDOW(w), RUNDLG_TITLE);
	CfgGtkSetGtkWindow(
		cfg_list,
		EDV_CFG_PARM_RUN_DLG_X,
		EDV_CFG_PARM_RUN_DLG_Y,
		NULL,
		NULL,
		w
	);
	gtk_widget_set_usize(w, RUNDLG_WIDTH, RUNDLG_HEIGHT);
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
		gdk_window_set_decorations(
			window,
			GDK_DECOR_BORDER | GDK_DECOR_TITLE
		);
		gdk_window_set_functions(
			window,
			GDK_FUNC_MOVE | GDK_FUNC_CLOSE
		);
		GUISetWMIcon(window, (guint8 **)icon_executable_48x48_xpm);
	}
	gtk_widget_add_events(
		w, 
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "delete_event",
		GTK_SIGNAL_FUNC(edv_run_dlg_delete_event_cb), d
	);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	style = gtk_widget_get_style(w);
	parent = w;


	/* Main GtkVBox */
	d->main_vbox = w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* GtkHBox for the run combo, browse button, and clear button */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	w = GUIComboCreate(
#if defined(PROG_LANGUAGE_SPANISH)
"La Orden:"
#elif defined(PROG_LANGUAGE_FRENCH)
"Commande:"
#elif defined(PROG_LANGUAGE_GERMAN)
"Befehl:"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Comando:"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bevel:"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Comando:"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kommando:"
#else
"Command:"
#endif
		, "",
		NULL,		/* Set list later */
		20,			/* Maximum items */
		&combo_rtn,
		d,
		edv_run_dlg_run_cb,
		NULL
	);
	d->run_combo = (GtkWidget *)combo_rtn;
	combo = (GtkCombo *)combo_rtn;
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);

	w = combo->entry;
	gtk_signal_connect(
		GTK_OBJECT(w), "focus_in_event",
		GTK_SIGNAL_FUNC(edv_run_dlg_command_focus_event_cb), d
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "focus_out_event",
		GTK_SIGNAL_FUNC(edv_run_dlg_command_focus_event_cb), d
	);
	edv_entry_set_dnd(core, w);
	edv_entry_set_complete_path(core, w);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Entre el nombre del programa ms cualquier argumento"
#elif defined(PROG_LANGUAGE_FRENCH)
"Entrer le nom du programme avec les arguments"
#elif defined(PROG_LANGUAGE_GERMAN)
"Tragen Sie den Namen des Programms und irgendeine Argumente ein"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Entrare il nome del programma pi qualunque discussioni"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga de naam van het programma bovendien enig argumenten binnen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Entre o nome do programa com mais qualquer argumentos"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"G inn i navnet av programet pluss noen argumenter"
#else
"Enter the name of the program plus any arguments"
#endif
	);

	/* Browse button */
	d->browse_btn = w = GUIButtonPixmap(
		(guint8 **)icon_folder_opened_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_run_dlg_browser_command_cb), d
	);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Hojee"
#elif defined(PROG_LANGUAGE_FRENCH)
"Parcourir"
#elif defined(PROG_LANGUAGE_GERMAN)
"Brausen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Curiosare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Grasduin"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Olhe"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Browse"
#else
"Browse"
#endif
	);
	gtk_widget_show(w);

	/* Clear button */
	d->clear_btn = w = GUIButtonPixmap(
		(guint8 **)icon_clear_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_run_dlg_clear_cb), combo->entry
	);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Claro"
#elif defined(PROG_LANGUAGE_FRENCH)
"Effacer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Klar"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Chiaro"
#elif defined(PROG_LANGUAGE_DUTCH)
"Helder"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Claro"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Klar"
#else
"Clear"
#endif
	);
	gtk_widget_show(w);


	/* Hbox for working directory entry and get from command
	 * button
	 */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Working Directory entry */
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Gua Trabajo:"
#elif defined(PROG_LANGUAGE_FRENCH)
"Rpertoire de travail:"
#elif defined(PROG_LANGUAGE_GERMAN)
"Arbeitenden Verzeichnis:"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Che Lavorando L'Elenco:"
#elif defined(PROG_LANGUAGE_DUTCH)
"Werkende Gids:"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Que trabalhando Guia:"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Arbeidende Directory:"
#else
"Working Directory:"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	d->working_dir_entry = w = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	CfgGtkSetGtkEditable(
		cfg_list,
		EDV_CFG_PARM_RUN_DLG_WORKING_DIR,
		w
	);
	edv_entry_set_dnd(core, w);
	edv_entry_set_complete_path(core, w);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Entre la gua de trabajo que se usar cundo correr el programa"
#elif defined(PROG_LANGUAGE_FRENCH)
"Entrer le rpertoire de travail utilis par le programme programme"
#elif defined(PROG_LANGUAGE_GERMAN)
"Tragen Sie das arbeitende Verzeichnis, das benutzt werden wird,\
 beim Laufen des Programms ein"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Entrare il che lavorando elenco che sar usato quando correre il\
 programma"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga de werking gids die gebruikt worden zal wanneer lopen het\
 programma binnen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Entre o que trabalhando guia que ser usado quando correr o programa"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"G inn i den arbeidende katalogen som brukt ved kjre av av programet"
#else
"Enter the working directory that will be used when running the\
 program"
#endif
	);
	gtk_widget_show(w);

	/* Browse button */
	d->working_dir_browse_btn = w = GUIButtonPixmap(
		(guint8 **)icon_folder_opened_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	combo = (GtkCombo *)d->run_combo;
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_run_dlg_browse_working_directory_cb), d
	);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Hojee"
#elif defined(PROG_LANGUAGE_FRENCH)
"Parcourir"
#elif defined(PROG_LANGUAGE_GERMAN)
"Brausen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Curiosare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Grasduin"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Olhe"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Browse"
#else
"Browse"
#endif
	);
	gtk_widget_show(w);

	/* Clear button */
	d->working_dir_clear_btn = w = GUIButtonPixmap(
		(guint8 **)icon_clear_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_run_dlg_clear_cb), d->working_dir_entry
	);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Claro"
#elif defined(PROG_LANGUAGE_FRENCH)
"Effacer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Klar"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Chiaro"
#elif defined(PROG_LANGUAGE_DUTCH)
"Helder"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Claro"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Klar"
#else
"Clear"
#endif
	);
	gtk_widget_show(w);

	/* Get from command button */
	d->working_dir_get_from_cmd_btn = w = GUIButtonPixmap(
		(guint8 **)icon_run_dir_from_cmd_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_run_dlg_get_working_irectory_from_command_cb), d
	);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Obtenga trabajando gua del valor actual de la orden"
#elif defined(PROG_LANGUAGE_FRENCH)
"Recevoir le rpertoire de travail depuis la commande"
#elif defined(PROG_LANGUAGE_GERMAN)
"Erhalten Sie arbeitenden Verzeichnis vom jetzigen Befehl Wert"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Prendere che lavorando l'elenco dal valore di comando attuale"
#elif defined(PROG_LANGUAGE_DUTCH)
"Krijg werkende gids van de huidig bevel waarde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Fique trabalhando guia do valor atual de comando"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Bli arbeidende katalog fra den nvrendee kommandoverdi"
#else
"Get working directory from the current command value"
#endif
	);
	gtk_widget_show(w);


	/* Hbox for run in terminal and keep dialog checks */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	d->run_in_terminal_check = w = gtk_check_button_new_with_label(
#if defined(PROG_LANGUAGE_SPANISH)
"Corra En La Terminal"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ouvrir Dans un Terminal"
#elif defined(PROG_LANGUAGE_GERMAN)
"Laufen Sie In Terminal"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Ha Correto Nell'Estremit"
#elif defined(PROG_LANGUAGE_DUTCH)
"Loop In Eindpunt"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Corra Em Terminal"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kjr In Terminal"
#else
"Run In Terminal"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Corra el programa especificado en una terminal, esto a\
 menudo se necesita para programas basados de texto"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ouvrir le programme spcifi dans un terminal, ceci\
 est souvent nexessaire pour les programmes bas sur du texte"
#elif defined(PROG_LANGUAGE_GERMAN)
"Laufen Sie das angegebene Programm in ein Terminal, dies ist often,\
 der fr Text basiert Programme gebraucht wird"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Ha correto il programma specificato in un'estremit, questo  often\
 ha avuto bisogno di per i programmi testo basati"
#elif defined(PROG_LANGUAGE_DUTCH)
"Loop het gespecificeerd programma in een eindpunt, dit is often nood\
 aan tekst gebaseerd programma's"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Corra o programa especificado num terminal, isto  often necessitou\
 para programas texto baseados"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kjr det spesifiserte programet i en terminal, dette er often som\
 trengt for tekst basert programer"
#else
"Check this to run the specified program in a terminal, this is needed\
 when running text-based programs"
#endif
	);
	gtk_widget_show(w);

	d->keep_dialog_check = w = gtk_check_button_new_with_label(
#if defined(PROG_LANGUAGE_SPANISH)
"Mantenga El Dilogo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Garder Le Dialogue"
#elif defined(PROG_LANGUAGE_GERMAN)
"Behalten Sie Dialog"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Tenere Il Dialogo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Houd Dialoog Bij"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mantenha Dialog"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Behold Dialog"
#else
"Keep Dialog"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Mantenga este dilogo trazado despus de correr un programa"
#elif defined(PROG_LANGUAGE_FRENCH)
"Garder cette boite de dialogue aprs avoir excuter le programme"
#elif defined(PROG_LANGUAGE_GERMAN)
"Behalten Sie diesen Dialog, der nach Laufen jedes Programms\
 aufgezeichnet wird"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Tenere questo mapped di dialogo dopo correre ogni programma"
#elif defined(PROG_LANGUAGE_DUTCH)
"Houd deze dialoog bracht in kaart na lopen van elk programma bij"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mantenha este mapped de dialog depois de correr cada programa"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Behold denne dialogen som kartlagt etter kjring av hver program"
#else
"Check this to keep this dialog mapped after running each program"
#endif
	);
	gtk_widget_show(w);

	/* Options button */
	d->options_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_options_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Las Opciones"
#elif defined(PROG_LANGUAGE_FRENCH)
"Options"
#elif defined(PROG_LANGUAGE_GERMAN)
"Optionen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Opzioni"
#elif defined(PROG_LANGUAGE_DUTCH)
"Opties"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Opes"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Alternativer"
#else
"Options"
#endif
		, NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_end(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_run_dlg_options_cb), d
	);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"El clic para configurar los programas terminales"
#elif defined(PROG_LANGUAGE_FRENCH)
"Configuer le terminal"
#elif defined(PROG_LANGUAGE_GERMAN)
"Klicken, letzte Programme zu gestalten"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Lo scatto di configurare i programmi terminali"
#elif defined(PROG_LANGUAGE_DUTCH)
"Klik Eindprogramma's te vormen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O estalido configurar programas terminais"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Klikk forme avsluttende programer"
#else
"Click to configure terminal programs"
#endif
	);
	gtk_accel_group_add(
		accelgrp, GDK_o, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_o);
	gtk_widget_show(w);


	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Hbox for buttons */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Run button */
	d->run_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_run_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Corra"
#elif defined(PROG_LANGUAGE_FRENCH)
"Excuter"
#elif defined(PROG_LANGUAGE_GERMAN)
"Lauf"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Correto"
#elif defined(PROG_LANGUAGE_DUTCH)
"Tocht"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Corrida"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Lp"
#else
"Run"
#endif
		, NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_run_dlg_run_cb), d
	);
	gtk_accel_group_add(
		accelgrp, GDK_Return, 0, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
		accelgrp, GDK_r, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_r);
	gtk_widget_show(w);

	/* Close button */
	d->close_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_close_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cierre"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fermer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nah"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Vicino"
#elif defined(PROG_LANGUAGE_DUTCH)
"Einde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Prximo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Nr"
#else
"Close"
#endif
		, NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_run_dlg_close_cb), d
	);
	gtk_accel_group_add(
		accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
		accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);
	gtk_widget_show(w);


	d->freeze_count--;

	return(d);
}

/*
 *	Marks the Run Dialog as busy or ready.
 */
void edv_run_dlg_set_busy(EDVRunDlg *d, const gboolean busy)
{
	GdkCursor *cursor;
	GtkWidget *w;
	EDVCore *core;

	if(d == NULL)
		return;

	w = d->toplevel;
	core = d->core;

	if(w != NULL)
	{
		if(busy)
		{
			/* Increase busy count */
			d->busy_count++;

			/* If already busy then don't change anything */
			if(d->busy_count > 1)
				return;

			cursor = edv_get_cursor(core, EDV_CURSOR_CODE_BUSY);
		}
		else
		{
			/* Reduce busy count */
			d->busy_count--;
			if(d->busy_count < 0)
				d->busy_count = 0;

			/* If still busy do not change anything */
			if(d->busy_count > 0)
				return;

			cursor = NULL;  /* Use default cursor */
		}

		/* Update toplevel window's cursor */
		if(w->window != NULL)
		{
			gdk_window_set_cursor(w->window, cursor);
			gdk_flush();
		}
	}
}

/*
 *	Checks if the Run Dialog is mapped.
 */
gboolean edv_run_dlg_is_mapped(EDVRunDlg *d)
{
	if(d == NULL)
		return(FALSE);

	return(GTK_WIDGET_MAPPED(d->toplevel));
}

/*
 *	Maps the Run Dialog.
 */
void edv_run_dlg_map(EDVRunDlg *d)
{
	GtkCombo *combo;

	if(d == NULL)
		return;

	gtk_widget_show_raise(d->toplevel);

	combo = GTK_COMBO(d->run_combo);
	gtk_widget_grab_focus(combo->entry);
/*	gtk_entry_set_position(GTK_ENTRY(combo->entry), 0); */
}

/*
 *	Unmaps the Run Dialog.
 */
void edv_run_dlg_unmap(EDVRunDlg *d)
{
	if(d == NULL)
		return;

	gtk_widget_hide(d->toplevel);
}

/*
 *	Deletes the Run Dialog.
 */
void edv_run_dlg_delete(EDVRunDlg *d)
{
	if(d == NULL)
		return;

	edv_run_dlg_unmap(d);

	d->freeze_count++;

	gtk_widget_destroy(d->toplevel);
	gtk_accel_group_unref(d->accelgrp);

	d->freeze_count--;

	g_free(d);
}
