#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <gtk/gtk.h>

#include "cfg.h"

#include "guiutils.h"
#include "tool_bar.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_directory.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_recycled_obj.h"
#include "libendeavour2-base/edv_recycle_bin_index.h"
#include "edv_purge_obj.h"
#include "edv_recover_obj.h"
#include "edv_recycle_bin_sync.h"
#include "edv_obj_info_match.h"
#include "edv_utils_gtk.h"
#include "edv_find_bar.h"
#include "edv_status_bar.h"
#include "edv_confirm.h"
#include "recycle_bin.h"
#include "recycle_bin_cb.h"
#include "recycle_bin_op.h"
#include "recycle_bin_list.h"
#include "recycle_bin_desktop_icon.h"
#include "edv_help.h"
#include "edv_emit.h"
#include "edv_op.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


static gint edv_recycle_bin_op_create_directories(
	EDVCore *core,
	const gchar *path
);

void edv_recycle_bin_op_cb(
	ToolBarItem *item, gint id, gpointer data
);
void edv_recycle_bin_op_enter_cb(
	ToolBarItem *item, gint id, gpointer data
);
void edv_recycle_bin_op_leave_cb(
	ToolBarItem *item, gint id, gpointer data
);

void edv_recycle_bin_op_close(edv_recbin_struct *recbin);
static void edv_recycle_bin_op_exit(edv_recbin_struct *recbin);

static void edv_recycle_bin_op_sync_disks(edv_recbin_struct *recbin);
static void edv_recycle_bin_op_sync_recycle_bin(edv_recbin_struct *recbin);
static void edv_recycle_bin_op_master_write_project(edv_recbin_struct *recbin);
static void edv_recycle_bin_op_delete_method_recycle(edv_recbin_struct *recbin);
static void edv_recycle_bin_op_delete_method_purge(edv_recbin_struct *recbin);

void edv_recycle_bin_op_recover(edv_recbin_struct *recbin);
static void edv_recycle_bin_op_purge(edv_recbin_struct *recbin);
static void edv_recycle_bin_op_purge_all(edv_recbin_struct *recbin);
static void edv_recycle_bin_op_rename(edv_recbin_struct *recbin);
static void edv_recycle_bin_op_chmod(edv_recbin_struct *recbin);
static void edv_recycle_bin_op_chown(edv_recbin_struct *recbin);
static void edv_recycle_bin_op_chtime(edv_recbin_struct *recbin);
static void edv_recycle_bin_op_properties(edv_recbin_struct *recbin);

static void edv_recycle_bin_op_select_all(edv_recbin_struct *recbin);
static void edv_recycle_bin_op_unselect_all(edv_recbin_struct *recbin);
static void edv_recycle_bin_op_invert_selection(edv_recbin_struct *recbin);

void edv_recycle_bin_op_refresh(edv_recbin_struct *recbin);
static void edv_recycle_bin_op_refresh_all(edv_recbin_struct *recbin);

static void edv_recycle_bin_op_contents_filter(edv_recbin_struct *recbin);

static void edv_recycle_bin_op_mime_types(edv_recbin_struct *recbin);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Creates the directory and all parent directories as needed and
 *	then reports them.
 *
 *	The path specifies the directory, if it does not exist then
 *	it and all its parent directories will be created as needed.
 */
static gint edv_recycle_bin_op_create_directories(
	EDVCore *core,
	const gchar *path
)
{
	GList *new_paths_list;
	const gint	status = edv_directory_create(
		path,
		TRUE,				/* Create parents */
		&new_paths_list
	),
			error_code = (int)errno;
	if(new_paths_list != NULL)
	{
		const gchar *path;
		GList *glist;
		EDVVFSObject *obj;

		/* Notify about the new directories being created */
		for(glist = new_paths_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			path = (const gchar *)glist->data;
			if(path == NULL)
				continue;

			obj = edv_vfs_object_lstat(path);
			if(obj != NULL)
			{
				edv_emit_vfs_object_added(
					core,
					path,
					obj
				);
				edv_vfs_object_delete(obj);
			}
		}

		g_list_foreach(
			new_paths_list,
			(GFunc)g_free,
			NULL
		);
		g_list_free(new_paths_list);
	}

	errno = (int)error_code;

	return(status);
}


/*
 *	Operation id callback nexus.
 *
 *	The given client data must be a EDVRecycleBinOp *.
 */
void edv_recycle_bin_op_cb(
	ToolBarItem *item, gint id, gpointer data  
)
{
	GtkWidget *toplevel;
	CfgList *cfg_list;
	edv_recbin_struct *recbin;
	EDVCore *core;
	EDVRecycleBinOp *op = EDV_RECYCLE_BIN_OP(data);
	if(op == NULL)
		return;

	recbin = op->recbin;
	if((recbin == NULL) || (op->flags & EDV_OPID_NO_OP))
		return;

	if(EDV_RECYCLE_BIN_IS_PROCESSING(recbin) || (recbin->freeze_count > 0))
		return;

	toplevel = recbin->toplevel;
	core = recbin->core;
	cfg_list = core->cfg_list;

	recbin->freeze_count++;

	/* Handle by operation id code */
	switch(op->id)
	{
	  case EDV_RECYCLE_BIN_OP_NONE:
	  case EDV_RECYCLE_BIN_OP_SEPARATOR:
		break;

	  case EDV_RECYCLE_BIN_OP_CLOSE:
		edv_recycle_bin_op_close(recbin);
		break;

	  case EDV_RECYCLE_BIN_OP_EXIT:
		edv_recycle_bin_op_exit(recbin);
		break;


	  case EDV_RECYCLE_BIN_OP_SYNC_DISKS:
		edv_recycle_bin_op_sync_disks(recbin);
		break;

	  case EDV_RECYCLE_BIN_OP_HISTORY:
		edv_map_history(
			core,
			-1,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL,
			toplevel
		);
		break;

	  case EDV_RECYCLE_BIN_OP_RUN:
		edv_map_run_dialog_command(
			core,
			NULL,
			NULL,
			toplevel
		);
		break;

	  case EDV_RECYCLE_BIN_OP_RUN_TERMINAL:
		edv_run_terminal(
			core,
			NULL,
			NULL,
			toplevel
		);
		break;

	  case EDV_RECYCLE_BIN_OP_WRITE_PROTECT:
		edv_recycle_bin_op_master_write_project(recbin);
		break;

	  case EDV_RECYCLE_BIN_OP_DELETE_METHOD_RECYCLE:
		edv_recycle_bin_op_delete_method_recycle(recbin);
		break;

	  case EDV_RECYCLE_BIN_OP_DELETE_METHOD_PURGE:
		edv_recycle_bin_op_delete_method_purge(recbin);
		break;


	  case EDV_RECYCLE_BIN_OP_SYNC_RECYCLE_BIN:
		edv_recycle_bin_op_sync_recycle_bin(recbin);
		break;


	  case EDV_RECYCLE_BIN_OP_RECOVER:
		edv_recycle_bin_op_recover(recbin);
		break;

	  case EDV_RECYCLE_BIN_OP_PURGE:
		edv_recycle_bin_op_purge(recbin);
		break;

	  case EDV_RECYCLE_BIN_OP_PURGE_ALL:
		edv_recycle_bin_op_purge_all(recbin);
		break;

	  case EDV_RECYCLE_BIN_OP_RENAME:
		edv_recycle_bin_op_rename(recbin);
		break;

	  case EDV_RECYCLE_BIN_OP_CHMOD:
		edv_recycle_bin_op_chmod(recbin);
		break;

	  case EDV_RECYCLE_BIN_OP_CHOWN:
		edv_recycle_bin_op_chown(recbin);
		break;

	  case EDV_RECYCLE_BIN_OP_CHTIME:
		edv_recycle_bin_op_chtime(recbin);
		break;

	  case EDV_RECYCLE_BIN_OP_PROPERTIES:
		edv_recycle_bin_op_properties(recbin);
		break;


	  case EDV_RECYCLE_BIN_OP_SELECT_ALL:
		edv_recycle_bin_op_select_all(recbin);
		break;

	  case EDV_RECYCLE_BIN_OP_UNSELECT_ALL:
		edv_recycle_bin_op_unselect_all(recbin);
		break;

	  case EDV_RECYCLE_BIN_OP_INVERT_SELECTION:
		edv_recycle_bin_op_invert_selection(recbin);
		break;


	  case EDV_RECYCLE_BIN_OP_FIND:
		edv_map_find_recycle_bin(core, recbin);
		break;


	  case EDV_RECYCLE_BIN_OP_REFRESH:
		edv_recycle_bin_op_refresh(recbin);
		break;

	  case EDV_RECYCLE_BIN_OP_REFRESH_ALL:
		edv_recycle_bin_op_refresh_all(recbin);
		break;

	  case EDV_RECYCLE_BIN_OP_SHOW_TOOL_BAR:
		if(core != NULL)
		{
			const gboolean state = !EDV_GET_B(
				EDV_CFG_PARM_RECBIN_SHOW_TOOL_BAR
			);
			EDV_SET_B(
				EDV_CFG_PARM_RECBIN_SHOW_TOOL_BAR,
				state
			);
			edv_queue_emit_reconfigured(core);
		}
		break;

	  case EDV_RECYCLE_BIN_OP_SHOW_FIND_BAR:
		if(core != NULL)
		{
			const gboolean state = !EDV_GET_B(
				EDV_CFG_PARM_RECBIN_SHOW_FIND_BAR
			);
			EDV_SET_B(
				EDV_CFG_PARM_RECBIN_SHOW_FIND_BAR,
				state
			);
			edv_queue_emit_reconfigured(core);
		}
		break;

	  case EDV_RECYCLE_BIN_OP_SHOW_STATUS_BAR:
		if(core != NULL)
		{
			const gboolean state = !EDV_GET_B(
				EDV_CFG_PARM_RECBIN_SHOW_STATUS_BAR
			);
			EDV_SET_B(
				EDV_CFG_PARM_RECBIN_SHOW_STATUS_BAR,
				state
			);
			edv_queue_emit_reconfigured(core);
		}
		break;


	  case EDV_RECYCLE_BIN_OP_CONTENTS_LIST_FILTER:
		edv_recycle_bin_op_contents_filter(recbin);
		break;

	  case EDV_RECYCLE_BIN_OP_CONTENTS_LIST_AUTO_RESIZE_COLUMNS:
		if(core != NULL)
		{
			const gboolean state = !EDV_GET_B(
				EDV_CFG_PARM_RECBIN_CONTENTS_LIST_AUTO_RESIZE_COLUMNS
			);
			EDV_SET_B(
				EDV_CFG_PARM_RECBIN_CONTENTS_LIST_AUTO_RESIZE_COLUMNS,
				state
			);
			edv_queue_emit_reconfigured(core);
		}
		break;


	  case EDV_RECYCLE_BIN_OP_MIME_TYPES:
		edv_recycle_bin_op_mime_types(recbin);
		break;


	  case EDV_RECYCLE_BIN_OP_NEW_BROWSER:
		edv_new_vfs_browser(
			core,
			NULL,
			NULL,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL
		);
		break;

	  case EDV_RECYCLE_BIN_OP_NEW_IMBR:
		edv_new_image_browser(
			core,
			NULL,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL
		);
		break;

	  case EDV_RECYCLE_BIN_OP_NEW_ARCHIVER:
		edv_new_archiver(
			core,
			NULL,
			NULL,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL
		);
		break;


	  case EDV_RECYCLE_BIN_OP_OPTIONS:
		edv_map_options(
			core,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL,
			toplevel
		);
		break;

	  case EDV_RECYCLE_BIN_OP_CUSTOMIZE:
		edv_map_customize(
			core,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL,
			toplevel
		);
		break;


	  case EDV_RECYCLE_BIN_OP_HELP_CONTENTS:
		edv_help(core, "Contents", toplevel);
		break;
	  case EDV_RECYCLE_BIN_OP_HELP_FILE_BROWSER:
		edv_help(core, "File Browser", toplevel);
		break;
	  case EDV_RECYCLE_BIN_OP_HELP_IMAGE_BROWSER:
		edv_help(core, "Image Browser", toplevel);
		break;
	  case EDV_RECYCLE_BIN_OP_HELP_ARCHIVER:
		edv_help(core, "Archiver", toplevel);
		break;
	  case EDV_RECYCLE_BIN_OP_HELP_RECYCLE_BIN:
		edv_help(core, "Recycle Bin", toplevel);
		break;
	  case EDV_RECYCLE_BIN_OP_HELP_KEYS_LIST:
		edv_help(core, "Keys List", toplevel);
		break;
	  case EDV_RECYCLE_BIN_OP_HELP_COMMON_OPERATIONS:
		edv_help(core, "Common Operations", toplevel);
		break;
	  case EDV_RECYCLE_BIN_OP_HELP_ABOUT:
		edv_map_about_dialog(
			core,
			core->geometry_flags, 
			(core->geometry_flags != 0) ? &core->geometry : NULL,
			toplevel
		);
		break;
	}

	recbin->freeze_count--;
}

/*
 *	Operation id enter notify callback nexus.
 *
 *	The given client data must be a EDVRecycleBinOp *.
 */
void edv_recycle_bin_op_enter_cb(
	ToolBarItem *item, gint id, gpointer data  
)
{
	const gchar *tooltip;
	edv_recbin_struct *recbin;
	EDVRecycleBinOp *op = EDV_RECYCLE_BIN_OP(data);
	if(op == NULL)
		return;

	recbin = op->recbin;
	if(recbin == NULL)
		return;

	if(recbin->freeze_count > 0)
		return;

	tooltip = op->tooltip;
	if(tooltip != NULL)
		edv_status_bar_message(recbin->status_bar, tooltip, FALSE);
}

/*
 *	Operation id leave notify callback nexus.
 */
void edv_recycle_bin_op_leave_cb(
	ToolBarItem *item, gint id, gpointer data  
)
{
	edv_recbin_struct *recbin;
	EDVRecycleBinOp *op = EDV_RECYCLE_BIN_OP(data);
	if(op == NULL)
		return;

	recbin = op->recbin;
	if(recbin == NULL)
		return;

	if(recbin->freeze_count > 0)
		return;

	edv_status_bar_message(recbin->status_bar, NULL, FALSE);
}


/*
 *	Close.
 */
void edv_recycle_bin_op_close(edv_recbin_struct *recbin)
{
	if(recbin == NULL)
		return;

	edv_recycle_bin_sync_configuration(recbin);
	edv_recycle_bin_unmap(recbin);
}

/*
 *	Close all windows.
 */
static void edv_recycle_bin_op_exit(edv_recbin_struct *recbin)
{
	EDVCore *core;

	if(recbin == NULL)
		return;

	core = recbin->core;

	/* Record the Recycle Bin's current configuration */
	edv_recycle_bin_sync_configuration(recbin);

	/* Unmap the Recycle Bin */
	edv_recycle_bin_unmap(recbin);

	/* Schedual a new pending operation on the core to close all
	 * the windows
	 */
	core->pending_flags |= EDV_CORE_PENDING_CLOSE_ALL_WINDOWS;
}


/*
 *	Sync Disks.
 */
static void edv_recycle_bin_op_sync_disks(edv_recbin_struct *recbin)
{
	GtkWidget *sb;
	EDVCore *core;

	if(recbin == NULL)
		return;

	sb = recbin->status_bar;
	core = recbin->core;

	edv_recycle_bin_set_busy(recbin, TRUE);
	edv_status_bar_message(
		sb,
		"Syncing disks...",
		TRUE
	);

	/* Sync disks */
	edv_sync_edv(core);

	edv_status_bar_message(
		sb,
		"Disk sync done",
		FALSE
	);
	edv_status_bar_progress(sb, 0.0f, FALSE);

	edv_recycle_bin_set_busy(recbin, FALSE);
}

/*
 *	Sync Recycle Bin.
 */
static void edv_recycle_bin_op_sync_recycle_bin(edv_recbin_struct *recbin)
{
	gboolean yes_to_all = FALSE;
	gint status;
	GtkWidget	*toplevel,
			*sb;
	EDVCore *core;

	if(recbin == NULL)
		return;

	if(EDV_RECYCLE_BIN_IS_PROCESSING(recbin))
		return;

	toplevel = recbin->toplevel;
	core = recbin->core;
	sb = recbin->status_bar;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
		return;

	edv_status_bar_message(
		sb,
		"Syncing recycle bin...",
		TRUE
	);

	edv_recycle_bin_set_busy(recbin, TRUE);

	/* Sync the recycle bin */
	status = edv_recycle_bin_sync(
		core,
		toplevel,
		TRUE,
		TRUE,
		&yes_to_all,
		edv_recycle_bin_find_bar_status_message_cb, recbin,
		edv_recycle_bin_find_bar_status_progress_cb, recbin
	);

	/* Refresh listing of recycled objects */
	edv_recycle_bin_op_refresh(recbin);

	edv_recycle_bin_set_busy(recbin, FALSE);

	edv_status_bar_message(
		sb,
		"Recycle bin sync done",
		FALSE
	);
	edv_status_bar_progress(sb, 0.0f, FALSE);
}

/*
 *	Write Protect Toggle.
 */
static void edv_recycle_bin_op_master_write_project(edv_recbin_struct *recbin)
{
	gboolean state;
	CfgList *cfg_list;
	EDVCore *core;

	if(recbin == NULL)
		return;

	edv_recycle_bin_set_busy(recbin, TRUE);

	core = recbin->core;
	cfg_list = core->cfg_list;

	/* Get the current Master Write Protect state */
	state = EDV_GET_B(EDV_CFG_PARM_WRITE_PROTECT);

	/* Toggle the Master Write Protect */
	state = !state;

	/* Set the new Master Write Protect state */
	EDV_SET_B(EDV_CFG_PARM_WRITE_PROTECT, state);

	/* Notify about the change in the Master Write Protect state */
	edv_queue_emit_master_write_protect_changed(core);

	edv_recycle_bin_set_busy(recbin, FALSE);
}

/*
 *	Delete Method: Recycle.
 */
static void edv_recycle_bin_op_delete_method_recycle(edv_recbin_struct *recbin)
{
	CfgList *cfg_list;
	EDVCore *core;

	if(recbin == NULL)
		return;

	edv_recycle_bin_set_busy(recbin, TRUE);

	core = recbin->core;
	cfg_list = core->cfg_list;

	EDV_SET_I(
		EDV_CFG_PARM_DELETE_METHOD,
		EDV_DELETE_METHOD_RECYCLE
	);
	edv_queue_emit_delete_method_changed(core);

	edv_recycle_bin_set_busy(recbin, FALSE);
}

/*
 *	Delete Method: Purge.
 */
static void edv_recycle_bin_op_delete_method_purge(edv_recbin_struct *recbin)
{
	CfgList *cfg_list;
	EDVCore *core;

	if(recbin == NULL)
		return;

	edv_recycle_bin_set_busy(recbin, TRUE);

	core = recbin->core;
	cfg_list = core->cfg_list;

	EDV_SET_I(
		EDV_CFG_PARM_DELETE_METHOD,
		EDV_DELETE_METHOD_PURGE
	);
	edv_queue_emit_delete_method_changed(core);

	edv_recycle_bin_set_busy(recbin, FALSE);
}


/*
 *	Recover callback.
 */
void edv_recycle_bin_op_recover(edv_recbin_struct *recbin)
{
	gboolean yes_to_all = FALSE;
	gint		status,
			nobjs,
			nobjs_recovered = 0;
	gulong index;
	gchar		*msg,
			*new_path;
	const gchar *error_msg;
	GList		*glist,
			*obj_list = NULL;
	GtkWidget *toplevel;
	GtkCList *clist;
	EDVRecycledObject *obj = NULL;
	EDVCore *core;

	if(recbin == NULL)
		return;

	toplevel = recbin->toplevel;
	clist = GTK_CLIST(recbin->contents_clist);
	core = recbin->core;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
		return;

#define CLEANUP_RETURN	{				\
 if(obj_list != NULL) {					\
  g_list_foreach(					\
   obj_list,						\
   (GFunc)edv_recycled_object_delete,			\
   NULL							\
  );							\
  g_list_free(obj_list);				\
 }							\
							\
 return;						\
}

	/* Get the list of selected recycled objects */
	for(glist = clist->selection;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		obj = EDV_RECYCLED_OBJECT(gtk_clist_get_row_data(
			clist,
			(gint)glist->data
		));
		if(obj == NULL)
			continue;

		obj_list = g_list_append(
			obj_list,
			edv_recycled_object_copy(obj)
		);
	}

	/* No recycled objects to recover? */
	if(obj_list == NULL)
	{
		CLEANUP_RETURN;
	}

	nobjs = g_list_length(obj_list);

	edv_recycle_bin_set_busy(recbin, TRUE);

	/* Confirm recover */
	status = edv_confirm_recover(
		core,
		toplevel,
		(obj != NULL) ? obj->name : NULL,
		nobjs,
		((obj != NULL) && (nobjs == 1)) ? obj->original_path : NULL
	);
	switch(status)
	{
	  case CDIALOG_RESPONSE_YES:
	  case CDIALOG_RESPONSE_YES_TO_ALL:
		break;

	  default:
		edv_recycle_bin_set_busy(recbin, FALSE);
		CLEANUP_RETURN;
		break;
	}

	/* Recover each recycled object to the VFS */
	status = 0;
	for(glist = obj_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		obj = EDV_RECYCLED_OBJECT(glist->data);
		if(obj == NULL)
			continue;

		index = obj->index;

		/* Check if the original location no longer exists */
		if((obj->original_path != NULL) ?
			!edv_path_exists(obj->original_path) : FALSE
		)
		{
			/* Query the user to recreate the original
			 * location
			 */
			gint	response,
				status2;
			const gchar *parent_path = obj->original_path;
			if(yes_to_all)
			{
				response = CDIALOG_RESPONSE_YES;
			}
			else
			{
				gchar *msg = g_strdup_printf(
"The original location:\n\
\n\
    %s\n\
\n\
Of:\n\
\n\
    %s\n\
\n\
No longer exists, create it?",
					parent_path,
					obj->name
				);
				edv_play_sound_question(core);
				CDialogSetTransientFor(toplevel);
				response = CDialogGetResponse(
					"Create Directory",
					msg,
					NULL,
					CDIALOG_ICON_QUESTION,
					CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_YES_TO_ALL |
						CDIALOG_BTNFLAG_CANCEL,
					CDIALOG_BTNFLAG_YES
				);
				g_free(msg);
				CDialogSetTransientFor(NULL);
			}
			switch(response)
			{
			    case CDIALOG_RESPONSE_YES_TO_ALL:
				yes_to_all = TRUE;
			    case CDIALOG_RESPONSE_YES:
				if(edv_recycle_bin_op_create_directories(
					core,
					parent_path
				))
				{
					const gint error_code = (gint)errno;
					gchar *msg = g_strdup_printf(
"Unable to create directory:\n\
\n\
    %s\n\
\n\
%s.",
						parent_path,
						g_strerror(error_code)
					);
					edv_play_sound_error(core);
					edv_message_error(
						"Create Directory Failed",
						msg,
						NULL,
						toplevel
					);
					g_free(msg);
					status2 = -1;
				}
				else
				{
					status2 = 0;
				}
				break;

			    default:
				status2 = -4;
				break;
			}

			/* Stop recovering if an error occured or the
			 * user aborted
			 */
			if(status2 != 0)
			{
				status = status2;
				break;
			}
		}

		/* Recover this recycled object to the VFS */
		status = edv_recover_object(
			core,
			index,			/* Recycled object to recover */
			NULL,			/* Use the original location
						 * as the recovery path */
			&new_path,
			toplevel,
			TRUE,			/* Show progress */
			TRUE,			/* Interactive */
			&yes_to_all
		);

		/* Check for errors */
		error_msg = edv_recover_object_get_error(core);
		if(!STRISEMPTY(error_msg))
		{
			/* Report the error */
			edv_play_sound_error(core);
			edv_message_error(
				"Recover Recycled Object Error",
				error_msg,
				NULL,
				toplevel
			);
		}

                /* Count the recovered object, notify about the
                 * recycled object being removed, and notify about
		 * the new recovered VFS object
		 */
		if(new_path != NULL)
		{
			EDVVFSObject *obj;
			nobjs_recovered++;
			edv_emit_recycled_object_removed(
				core,
				index
			);
			obj = edv_vfs_object_lstat(new_path);
			if(obj != NULL)
			{
				edv_emit_vfs_object_added(
					core,
					new_path,
					obj
				);
				edv_vfs_object_delete(obj);
			}
			g_free(new_path);
		}

                /* Stop recovering if the user aborted */
		if(status == -4)
			break;
	}

	/* Unmap the progress dialog which may have been mapped in
	 * the above operation
	 */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Update the status bar message */
	if(status == -4)
		msg = g_strdup(
"Recover operation canceled"
		);
	else if(nobjs_recovered > 0)
		msg = g_strdup_printf(
			"Recovered %i %s",
			nobjs_recovered,
			(nobjs_recovered == 1) ? "recycled object" : "recycled objects" 		);
	else
		msg = g_strdup_printf(
			"Unable to recover %s",
			(nobjs == 1) ? "recycled object" : "recycled objects"
		);
	edv_status_bar_message(recbin->status_bar, msg, FALSE);
	g_free(msg);

	/* Play the "completed" sound on success */
	if(status == 0)
		edv_play_sound_completed(core);

	edv_recycle_bin_set_busy(recbin, FALSE);

	CLEANUP_RETURN;
#undef CLEANUP_RETURN
}

/*
 *	Purge callback.
 */
static void edv_recycle_bin_op_purge(edv_recbin_struct *recbin)
{
	gboolean yes_to_all = FALSE;
	gint		status,
			nobjs,
			nobjs_purged = 0;
	gulong index;
	gchar *msg;
	const gchar *error_msg;
	GList		*glist,
			*obj_list = NULL;
	GtkWidget *toplevel;
	GtkCList *clist;
	EDVRecycledObject *obj = NULL;
	EDVCore *core;

	if(recbin == NULL)
		return;

	toplevel = recbin->toplevel;
	clist = GTK_CLIST(recbin->contents_clist);
	core = recbin->core;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
		return;

#define CLEANUP_RETURN	{				\
 if(obj_list != NULL) {					\
  g_list_foreach(					\
   obj_list,						\
   (GFunc)edv_recycled_object_delete,			\
   NULL							\
  );							\
  g_list_free(obj_list);				\
 }							\
							\
 return;						\
}

	/* Get a list of selected recycled objects indices */
	for(glist = clist->selection;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		obj = EDV_RECYCLED_OBJECT(gtk_clist_get_row_data(
			clist,
			(gint)glist->data
		));
		if(obj == NULL)
			continue;

		obj_list = g_list_append(
			obj_list,
			edv_recycled_object_copy(obj)
		);
	}

	/* No recycled objects to purge? */
	if(obj_list == NULL)
	{
		CLEANUP_RETURN;
	}

	nobjs = g_list_length(obj_list);

	edv_recycle_bin_set_busy(recbin, TRUE);

	/* Confirm purge */
	status = edv_confirm_purge(
		core,
		toplevel,
		(obj != NULL) ? obj->name : NULL,
		nobjs
	);
	switch(status)
	{
	    case CDIALOG_RESPONSE_YES:
	    case CDIALOG_RESPONSE_YES_TO_ALL:
		break;
	    default:
		edv_recycle_bin_set_busy(recbin, FALSE);
		CLEANUP_RETURN;
		break;
	}

	/* Purge the recycled objects from the Recycle Bin */
	status = 0;
	for(glist = obj_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		obj = EDV_RECYCLED_OBJECT(glist->data);
		if(obj == NULL)
			continue;

		index = obj->index;

		/* Purge this recycled object */
		status = edv_purge_object(
			core,
			index,
			toplevel,
			(nobjs > 0) ?
				((gfloat)nobjs_purged / (gfloat)nobjs) : 0.0f,
			TRUE,			/* Show progress */
			TRUE,			/* Interactive */
			&yes_to_all
		);

		/* Check for errors */
		error_msg = edv_purge_object_get_error(core);
		if(!STRISEMPTY(error_msg))
		{
			/* Report the error */
			edv_play_sound_error(core);
			edv_message_error(
				"Purge Recycled Object Error",
				error_msg,
				NULL,
				toplevel
			);
		}

		/* Count this recycled object being purged and notify
		 * about this recycled object being purged
		 */
		if(status == 0)
		{
			nobjs_purged++;
			edv_emit_recycled_object_removed(
				core,
				index
			);
		}

		/* Stop purging if the user aborted */
		if(status == -4)
			break;
	}

	/* Unmap the progress dialog which may have been mapped in
	 * the above operation
	 */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Update the status bar message */
	if(status == -4)
		msg = g_strdup(
"Purge operation canceled"
		);
	else if(nobjs_purged > 0)
		msg = g_strdup_printf(
"Purged %i %s",
			nobjs_purged,
			(nobjs_purged == 1) ? "recycled object" : "recycled objects"
		);
	else
		msg = g_strdup_printf(
"Unable to purge %s",
			(nobjs == 1) ? "recycled object" : "recycled objects"
		);
	edv_status_bar_message(recbin->status_bar, msg, FALSE);
	g_free(msg);

	/* Play the "completed" sound on success */
	if(status == 0) 
		edv_play_sound_completed(core);


	edv_recycle_bin_set_busy(recbin, FALSE);

	CLEANUP_RETURN;
#undef CLEANUP_RETURN
}

/*
 *	Purge all.
 */
static void edv_recycle_bin_op_purge_all(edv_recbin_struct *recbin)
{
	gboolean yes_to_all = FALSE;
	gint		status,
			nobjs,
			nobjs_purged = 0;
	gulong index;
	gchar *msg;
	const gchar	*error_msg,
			*index_path;
	GList		*glist,
			*indicies_list = NULL;
	GtkWidget *toplevel;
	CfgList *cfg_list;
	EDVRecycledObject *obj;
	EDVRecycleBinIndex *rp;
	EDVCore *core;

	if(recbin == NULL)
		return;

	toplevel = recbin->toplevel;
	core = recbin->core;
	cfg_list = core->cfg_list;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
		return;

	/* Get the path to the recycled objects index file */
	index_path = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);
	if(index_path == NULL)
		return;

#define CLEANUP_RETURN	{		\
 g_list_free(indicies_list);		\
					\
 return;				\
}

	/* Get the list of recycled objects */
	rp = edv_recycle_bin_index_open(index_path);
	for(obj = edv_recycle_bin_index_next(rp);
	    obj != NULL;
	    obj = edv_recycle_bin_index_next(rp)
	)
		indicies_list = g_list_append(
			indicies_list,
			(gpointer)obj->index
		);
	edv_recycle_bin_index_close(rp);

	/* No recycled objects to purge? */
	if(indicies_list == NULL)
	{
		CLEANUP_RETURN;
	}

	nobjs = g_list_length(indicies_list);

	edv_recycle_bin_set_busy(recbin, TRUE);

	/* Confirm purge */
	status = edv_confirm_purge(
		core,
		toplevel,
		NULL,
		nobjs
	);
	switch(status)
	{
	    case CDIALOG_RESPONSE_YES:
	    case CDIALOG_RESPONSE_YES_TO_ALL:
		break;
	    default:
		edv_recycle_bin_set_busy(recbin, FALSE);
		CLEANUP_RETURN;
		break;
	}

	/* Purge all the objects from the Recycle Bin
	 *
	 * Note that edv_purge_all_objects() is not used here because
	 * it does not allow the reporting of each recycled object
	 * purged during the purge
	 */
	status = 0;
	for(glist = indicies_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		index = (gulong)glist->data;

		/* Purge this recycled object */
		status = edv_purge_object(
			core,
			index,			/* Recycled object to purge */
			toplevel,
			(nobjs > 0) ?
				((gfloat)nobjs_purged / (gfloat)nobjs) : -1.0f,
			TRUE,			/* Show progress */
			TRUE,			/* Interactive */
			&yes_to_all
		);

		/* Check for errors */
		error_msg = edv_purge_object_get_error(core);
		if(!STRISEMPTY(error_msg))
		{
			/* Report the error */
			edv_play_sound_error(core);
			edv_message_error(
				"Purge Recycled Object Error",
				error_msg,
				NULL,
				toplevel
			);
		}

		/* Count this recycled object being purged and notify
		 * about this recycled object being purged
		 */
		if(status == 0)
		{
			nobjs_purged++;
			edv_emit_recycled_object_removed(
				core,
				index
			);
		}

		/* Stop purging if the user aborted */
		if(status == -4)
			break;
	}

	/* If no errors occured or the user did not abort then call
	 * edv_purge_all_objects() to remove any stray recycled
	 * object files and the recycled objects index file from the
	 * Recycle Bin
	 */
	if(status == 0)
	{
		/* Remove any stray recycled object files and the recycled
		 * objects index file
		 */
		status = edv_purge_all_objects(
			core,
			toplevel,
			FALSE,			/* Do not show progress */
			TRUE,			/* Interactive */
			&yes_to_all
		);

		/* Check for errors */
		error_msg = edv_purge_object_get_error(core);
		if(!STRISEMPTY(error_msg))
		{
			/* Report the error */
			edv_play_sound_error(core);
			edv_message_error(
				"Purge Recycled Object Error",
				error_msg,
				NULL,
				toplevel
			);
		}
	}

	/* Unmap the progress dialog which may have been mapped in
	 * the above operation
	 */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Update the status bar message */
	if(status == -4)
		msg = g_strdup(
"Purge operation canceled"
		);
	else if(nobjs_purged > 0)
		msg = g_strdup_printf(
"Purged %i %s",
			nobjs_purged,
			(nobjs_purged == 1) ? "recycled object" : "recycled objects"
		);
	else
		msg = g_strdup_printf(
"Unable to purge %s",
			(nobjs == 1) ? "recycled object" : "recycled objects"
		);
	edv_status_bar_message(recbin->status_bar, msg, FALSE);
	g_free(msg);

	/* Play the "completed" sound on success */
	if(status == 0)
		edv_play_sound_completed(core);

	edv_recycle_bin_set_busy(recbin, FALSE);

	CLEANUP_RETURN;
#undef CLEANUP_RETURN
}

/*
 *	Rename.
 */
static void edv_recycle_bin_op_rename(edv_recbin_struct *recbin)
{
	gint row;
	GtkWidget *toplevel;
	GtkCList *clist;
	EDVCore *core;

	if(recbin == NULL)
		return;

	toplevel = recbin->toplevel;
	clist = GTK_CLIST(recbin->contents_clist);
	core = recbin->core;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, recbin->toplevel))
		return;

	/* Get the last selected row */
	row = edv_clist_get_selected_last(clist, NULL);
	if(row > -1)
	{
		/* Map the rename prompt */
		edv_recycle_bin_list_rename_query(
			recbin,
			row, -1
		);
	}
}

/*
 *	Change Permissions.
 */
static void edv_recycle_bin_op_chmod(edv_recbin_struct *recbin)
{
	GList *sel_objs_list;
	GtkWidget *toplevel;
	edv_obj_op_dlg_struct *d;
	EDVCore *core;

	if(recbin == NULL)
		return;

	edv_recycle_bin_set_busy(recbin, TRUE);

	toplevel = recbin->toplevel;
	core = recbin->core;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_recycle_bin_set_busy(recbin, FALSE);
		return;
	}

	/* Get the object operations dialog and create it as needed */
	d = edv_get_object_operations_dialog(core);
	if(d == NULL)
	{
		edv_recycle_bin_set_busy(recbin, FALSE);
		return;
	}

	/* Sync data to ensure that current values to operate on */
	edv_recycle_bin_sync_data(recbin);

	/* Get the selected objects list */
	sel_objs_list = edv_recycle_bin_get_selected_objects(recbin, FALSE);
	if(sel_objs_list == NULL)
	{
		edv_recycle_bin_set_busy(recbin, FALSE);
		return;
	}

	/* Map the object operations dialog to change permissions */
	EDVObjOpDlgMapValues(
		d,
		EDV_OBJ_OP_DLG_OP_CHMOD,
		EDV_LOCATION_TYPE_RECYCLE_BIN,
		sel_objs_list,
		NULL,				/* No srouce directory */
		toplevel
	);

	/* Delete the selected objects list */
	g_list_free(sel_objs_list);

	edv_recycle_bin_set_busy(recbin, FALSE);
}

/*
 *	Change Ownership.
 */
static void edv_recycle_bin_op_chown(edv_recbin_struct *recbin)
{
	GList *sel_objs_list;
	GtkWidget *toplevel;
	edv_obj_op_dlg_struct *d;
	EDVCore *core;

	if(recbin == NULL)
		return;

	edv_recycle_bin_set_busy(recbin, TRUE);

	toplevel = recbin->toplevel;
	core = recbin->core;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_recycle_bin_set_busy(recbin, FALSE);
		return;
	}

	/* Get the object operations dialog and create it as needed */
	d = edv_get_object_operations_dialog(core);
	if(d == NULL)
	{
		edv_recycle_bin_set_busy(recbin, FALSE);
		return;
	}

	/* Sync data to ensure that current values to operate on */
	edv_recycle_bin_sync_data(recbin);

	/* Get the selected objects list */
	sel_objs_list = edv_recycle_bin_get_selected_objects(recbin, FALSE);
	if(sel_objs_list == NULL)
	{
		edv_recycle_bin_set_busy(recbin, FALSE);
		return;
	}

	/* Map the object operations dialog to change ownership */
	EDVObjOpDlgMapValues(
		d,
		EDV_OBJ_OP_DLG_OP_CHOWN,
		EDV_LOCATION_TYPE_RECYCLE_BIN,
		sel_objs_list,
		NULL,				/* No srouce directory */
		toplevel
	);

	/* Delete the selected objects list */
	g_list_free(sel_objs_list);

	edv_recycle_bin_set_busy(recbin, FALSE);
}

/*
 *	Change Time Stamps.
 */
static void edv_recycle_bin_op_chtime(edv_recbin_struct *recbin)
{
	GList *sel_objs_list;
	GtkWidget *toplevel;
	edv_obj_op_dlg_struct *d;
	EDVCore *core;

	if(recbin == NULL)
		return;

	edv_recycle_bin_set_busy(recbin, TRUE);

	toplevel = recbin->toplevel;
	core = recbin->core;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_recycle_bin_set_busy(recbin, FALSE);
		return;
	}

	/* Get the object operations dialog and create it as needed */
	d = edv_get_object_operations_dialog(core);
	if(d == NULL)
	{
		edv_recycle_bin_set_busy(recbin, FALSE);
		return;
	}

	/* Sync data to ensure that current values to operate on */
	edv_recycle_bin_sync_data(recbin);

	/* Get the selected objects list */
	sel_objs_list = edv_recycle_bin_get_selected_objects(recbin, FALSE);
	if(sel_objs_list == NULL)
	{
		edv_recycle_bin_set_busy(recbin, FALSE);
		return;
	}

	/* Map the object operations dialog to change time stamps */
	EDVObjOpDlgMapValues(
		d,
		EDV_OBJ_OP_DLG_OP_CHTIME,
		EDV_LOCATION_TYPE_RECYCLE_BIN,
		sel_objs_list,
		NULL,				/* No srouce directory */
		toplevel
	);

	/* Delete the selected objects list */
	g_list_free(sel_objs_list);

	edv_recycle_bin_set_busy(recbin, FALSE);
}

/*
 *	Properties.
 */
static void edv_recycle_bin_op_properties(edv_recbin_struct *recbin)
{
	GtkWidget *toplevel;
	EDVRecycledObject *obj = NULL;
	EDVCore *core;

	if(recbin == NULL)
		return;

	edv_recycle_bin_set_busy(recbin, TRUE);

	toplevel = recbin->toplevel;
	core = recbin->core;

	edv_recycle_bin_sync_data(recbin);

	/* Get the selected object */
	if(recbin->contents_clist_selected_row > -1)
	{
		GtkCList *clist = GTK_CLIST(recbin->contents_clist);
		GList *glist = clist->selection_end;
		if(glist != NULL)
			obj = EDV_RECYCLED_OBJECT(gtk_clist_get_row_data(
				clist,
				(gint)glist->data
			));
	}

	/* No selected object? */
	if(obj == NULL)
	{
		edv_recycle_bin_set_busy(recbin, FALSE);
		return;
	}

	/* Create a new Properties Dialog to display the object */
	edv_new_properties_dialog_recycle_bin(
		core,
		obj->index,
		NULL,				/* Default page */
		NULL,				/* No extended properties list */
		core->geometry_flags,
		(core->geometry_flags != 0) ? &core->geometry : NULL,
		toplevel
	);

	edv_recycle_bin_set_busy(recbin, FALSE);
}


/*
 *	Select all.
 */
static void edv_recycle_bin_op_select_all(edv_recbin_struct *recbin)
{
	GtkCList *clist;
	EDVCore *core;

	if(recbin == NULL)
		return;

	clist = GTK_CLIST(recbin->contents_clist);
	core = recbin->core;

	edv_recycle_bin_set_busy(recbin, TRUE);

	/* Select all rows on clist */
	gtk_clist_freeze(clist);
	gtk_clist_select_all(clist);
	gtk_clist_thaw(clist);

	/* Assume highest row index as the last selected row */
	recbin->contents_clist_selected_row = clist->rows - 1;

	edv_status_bar_message(
		recbin->status_bar, "All objects selected", FALSE
	);

	edv_recycle_bin_set_busy(recbin, FALSE);
}

/*
 *	Unselect all.
 */
static void edv_recycle_bin_op_unselect_all(edv_recbin_struct *recbin)
{
	GtkCList *clist;
	EDVCore *core;

	if(recbin == NULL)
		return;

	clist = GTK_CLIST(recbin->contents_clist);
	core = recbin->core;

	edv_recycle_bin_set_busy(recbin, TRUE);

	/* Unselect all rows on clist */
	gtk_clist_freeze(clist);
	gtk_clist_unselect_all(clist);
	gtk_clist_thaw(clist);

	/* Mark contents clist's row as unselected */
	recbin->contents_clist_selected_row = -1;

	edv_status_bar_message(
		recbin->status_bar, "All objects unselected", FALSE
	);

	edv_recycle_bin_set_busy(recbin, FALSE);
}

/*
 *	Invert Selection.
 */
static void edv_recycle_bin_op_invert_selection(edv_recbin_struct *recbin)
{
	gint		row,
			total_rows;
	GList		*glist,
			*selection;
	GtkCList *clist;
	EDVCore *core;

	if(recbin == NULL)
		return;

	clist = GTK_CLIST(recbin->contents_clist);
	core = recbin->core;

	edv_recycle_bin_set_busy(recbin, TRUE);
	gtk_clist_freeze(clist);

	/* Get copy of selected rows list from clist */
	selection = (clist->selection != NULL) ?
		g_list_copy(clist->selection) : NULL;

	for(row = 0, total_rows = clist->rows;
		row < total_rows;
		row++
	)
	{
		for(glist = selection;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			if(row == (gint)glist->data)
			{
				gtk_clist_unselect_row(clist, row, 0);
				break;
			}
		}
		/* Row not selected? */
		if(glist == NULL)
			gtk_clist_select_row(clist, row, 0);
	}

	g_list_free(selection);

	gtk_clist_thaw(clist);
	edv_status_bar_message(
		recbin->status_bar, "Selection inverted", FALSE
	);
	edv_recycle_bin_set_busy(recbin, FALSE);
}


/*
 *	Refresh.
 */
void edv_recycle_bin_op_refresh(edv_recbin_struct *recbin)
{
	GtkWidget	*toplevel,
			*sb;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVCore *core;

	if(recbin == NULL)
		return;

	toplevel = recbin->toplevel;
	core = recbin->core;
	cfg_list = core->cfg_list;
	sb = recbin->status_bar;

	edv_recycle_bin_set_busy(recbin, TRUE);
	GUIBlockInput(toplevel, TRUE);

	/* Update the Contents GtkCList */
	clist = GTK_CLIST(recbin->contents_clist);
	if(clist != NULL)
	{
		/* Record the last scroll position */
		const gfloat	last_x = GTK_ADJUSTMENT_GET_VALUE(clist->hadjustment),
							last_y = GTK_ADJUSTMENT_GET_VALUE(clist->vadjustment);

		gtk_clist_freeze(clist);

		/* Reget the listing */
		edv_recycle_bin_list_get_listing(
			recbin,
			EDV_GET_B(EDV_CFG_PARM_LISTS_SHOW_PROGRESS)
		);

		gtk_clist_thaw(clist);

		/* Scroll back to the original position */
		edv_clist_scroll_to_position(clist, last_x, last_y);
	}

	edv_recycle_bin_update_display(recbin);
	edv_status_bar_message(sb, "Refreshed contents listing", FALSE);

	GUIBlockInput(toplevel, FALSE);
	edv_recycle_bin_set_busy(recbin, FALSE);
}

/*
 *	Refresh All.
 */
static void edv_recycle_bin_op_refresh_all(edv_recbin_struct *recbin)
{
	const gchar *index_path;
	CfgList *cfg_list;
	EDVCore *core;

	if(recbin == NULL)
		return;

	core = recbin->core;
	cfg_list = core->cfg_list;

	/* Get the path to the recycled objects index file */
	index_path = CFGItemListGetValueS(
		cfg_list, EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX
	);

	/* Update the number of recycled objects on the core */
	core->last_nrecycle_bin_items = edv_recycle_bin_index_get_total(
		index_path
	);

	/* Refresh the Recycle Bin Desktop Icon */
	edv_recycle_bin_desktop_icon_update_display(core->recbin_deskicon);

	/* Refresh the Recycle Bin */
	edv_recycle_bin_op_refresh(recbin);
}


/*
 *	Sets the contents list filter.
 */
static void edv_recycle_bin_op_contents_filter(edv_recbin_struct *recbin)
{
	if(recbin == NULL)
		return;

	edv_recycle_bin_set_busy(recbin, TRUE);

	if(edv_query_name_filter(
		recbin->core,
		&recbin->contents_list_filter,
		recbin->toplevel
	))
	{                                 
		edv_recycle_bin_set_title(recbin);
		edv_recycle_bin_op_refresh(recbin);
	}

	edv_recycle_bin_set_busy(recbin, FALSE);
}

/*
 *      MIME Types.
 */
static void edv_recycle_bin_op_mime_types(edv_recbin_struct *recbin)
{
	gint i;
	gchar *type_string = NULL;
	GtkWidget *toplevel;
	GtkCList *clist;
	EDVCore *core;
	if(recbin == NULL)
		return;

	toplevel = recbin->toplevel;
	clist = GTK_CLIST(recbin->contents_clist);
	core = recbin->core;

	edv_recycle_bin_set_busy(recbin, TRUE);

	i = edv_clist_get_selected_last(clist, NULL);
	if(i > -1)
	{
		EDVRecycledObject *obj = EDV_RECYCLED_OBJECT(
			gtk_clist_get_row_data(clist, i)
		);
		if(obj != NULL)
		{
			(void)edv_match_object_type_string(
				core->mime_types_list,
				obj->type,
				obj->name,
				obj->permissions,
				&type_string
			);
		}
	}

	edv_map_mime_types(
		core,
		type_string,
		core->geometry_flags,
		(core->geometry_flags != 0) ? &core->geometry : NULL,
		toplevel
	);

	g_free(type_string);

	edv_recycle_bin_set_busy(recbin, FALSE);
}
