#include <string.h>
#if defined(HAVE_REGEX)
# include <regex.h>
#else
# include <fnmatch.h>
#endif
#include <gtk/gtk.h>

#include "cfg.h"

#include "guiutils.h"
#include "fprompt.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_recycled_obj.h"
#include "libendeavour2-base/edv_recycle_bin_index.h"
#include "libendeavour2-base/edv_recycled_obj_stat.h"
#include "libendeavour2-base/edv_id.h"
#include "edv_ids_list.h"
#include "edv_date_format.h"
#include "edv_pixmap.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "edv_obj_info_match.h"
#include "edv_utils_gtk.h"
#include "edv_status_bar.h"
#include "recycle_bin.h"
#include "recycle_bin_cb.h"
#include "recycle_bin_list.h"
#include "edv_recycled_obj_op.h"
#include "edv_emit.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


typedef struct _EDVRecBinContentsListFPromptData	EDVRecBinContentsListFPromptData;
#define EDV_RECYCLE_BIN_CONTENTS_LIST_FPROMPT_DATA(p)	((EDVRecBinContentsListFPromptData *)(p))


static EDVRecycledObject *edv_recycle_bin_list_new_error_object(
	const guint index,
	const gchar *name
);

/* Columns */
gint edv_recycle_bin_list_get_column_index_by_type(
	edv_recbin_struct *recbin,
	const EDVRecycleBinColumnType column_type
);
gint edv_recycle_bin_list_get_column_width_by_type(
	edv_recbin_struct *recbin,
	const EDVRecycleBinColumnType column_type
);
EDVRecycleBinColumnType edv_recycle_bin_list_get_column_type_by_index(
	edv_recbin_struct *recbin,
	const gint column_num
);
void edv_recycle_bin_list_set_column_width_by_type(
	edv_recbin_struct *recbin,
	const EDVRecycleBinColumnType column_type,
	const gint width
);
static void edv_recycle_bin_list_reset_columns(edv_recbin_struct *recbin);
void edv_recycle_bin_list_resize_column_optimul(
	edv_recbin_struct *recbin,
	const gint column_num
);
void edv_recycle_bin_list_resize_columns_optimul(edv_recbin_struct *recbin);

/* Cell Setting */
static void edv_recycle_bin_list_set_cell_index(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column
);
static void edv_recycle_bin_list_set_cell_name(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column
);
static void edv_recycle_bin_list_set_cell_size(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column,
	const gboolean hide_dir_size,
	const gboolean hide_link_size,
	const EDVSizeFormat size_format,
	const gulong block_size
);
static void edv_recycle_bin_list_set_cell_storage_size(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column,
	const gboolean hide_dir_size,
	const gboolean hide_link_size,
	const EDVSizeFormat size_format,
	const gulong block_size
);
static void edv_recycle_bin_list_set_cell_type(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column
);
static void edv_recycle_bin_list_set_cell_permissions(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column,
	gboolean hide_link_permissions
);
static void edv_recycle_bin_list_set_cell_owner(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column
);
static void edv_recycle_bin_list_set_cell_group(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column
);
static void edv_recycle_bin_list_set_cell_date_accessed(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column,
	EDVDateRelativity date_relativity, const gchar *date_format
);
static void edv_recycle_bin_list_set_cell_date_modified(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column,
	EDVDateRelativity date_relativity, const gchar *date_format
);
static void edv_recycle_bin_list_set_cell_date_changed(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column,
	EDVDateRelativity date_relativity, const gchar *date_format
);
static void edv_recycle_bin_list_set_cell_date_deleted(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column,
	const EDVDateRelativity date_relativity, const gchar *date_format
);
static void edv_recycle_bin_list_set_cell_linked_to(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column
);
static void edv_recycle_bin_list_set_cell_original_location(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column
);
static void edv_recycle_bin_list_set_cell_capacity_used(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column,
	const gulong recbin_size_max
);
static void EDVRecBinContentsSetRow(
	edv_recbin_struct *recbin,
	const gint row,
	EDVRecycledObject *obj
);

static gint edv_recycle_bin_list_append_object(
	edv_recbin_struct *recbin,
	EDVRecycledObject *obj
);
static void edv_recycle_bin_list_append_listing(
	edv_recbin_struct *recbin,
	const gboolean show_progress
);

gint edv_recycle_bin_list_find_row_by_index(
	edv_recbin_struct *recbin,
	const guint index
);

/* Realize Listing */
void edv_recycle_bin_list_realize_listing(edv_recbin_struct *recbin);

/* Get Listing */
void edv_recycle_bin_list_get_listing(
	edv_recbin_struct *recbin,
	const gboolean show_progress
);
void edv_recycle_bin_list_clear(edv_recbin_struct *recbin);

/* Renaming */
static void edv_recycle_bin_list_fprompt_rename_apply_cb(
	gpointer data, const gchar *value
);
static void edv_recycle_bin_list_fprompt_cancel_cb(gpointer data);
void edv_recycle_bin_list_rename_query(
	edv_recbin_struct *recbin,
	const gint row,
	const gint column
);

/* Callbacks */
void edv_recycle_bin_list_recycled_object_added_cb(
	edv_recbin_struct *recbin, const guint index
);
void edv_recycle_bin_list_recycled_object_modified_cb(
	edv_recbin_struct *recbin, const guint index
);
void edv_recycle_bin_list_recycled_object_removed_cb(
	edv_recbin_struct *recbin, const guint index
);


struct _EDVRecBinContentsListFPromptData {
	edv_recbin_struct	*recbin;
	gint		row,
			column;
};


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Creates a new error object.
 *
 *	The name specifies the name of the error object.
 */
static EDVRecycledObject *edv_recycle_bin_list_new_error_object(
	const guint index,
	const gchar *name
)
{
	EDVRecycledObject *obj = edv_recycled_object_new();
	if(obj == NULL)
		return(NULL);

	obj->name = STRDUP(name);
	obj->index = index;
	obj->type = EDV_OBJECT_TYPE_ERROR;

	return(obj);
}


/*
 *	Gets the column index displaying the column type.
 *
 *	The column_type specifies the EDVRecycleBinColumnType.
 *
 *	Returns the column index that is set to column_type or
 *	negative on error.
 */
gint edv_recycle_bin_list_get_column_index_by_type(
	edv_recbin_struct *recbin,
	const EDVRecycleBinColumnType column_type
)
{
	gint column_num;
	GtkCList *clist;
	CfgList *cfg_list;
	CfgIntList *column_types_intlist;
	EDVCore *core;

	if(recbin == NULL)
		return(-2);

	clist = GTK_CLIST(recbin->contents_clist);
	core = recbin->core;
	cfg_list = core->cfg_list;

	column_types_intlist = EDV_GET_INT_LIST(
		EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN
	);
	if(column_types_intlist == NULL)
		return(-2);

	column_num = (gint)g_list_index(
		column_types_intlist->list,
		(gpointer)column_type
	);
	if(column_num < 0)
		return(-2);

	if(column_num >= clist->columns)
		return(clist->columns - 1);

	return(column_num);
}

/*
 *	Gets the width of the column displaying the column type.
 *
 *	The column_type specifies the column type.
 *
 *	Returns the column width.
 */
gint edv_recycle_bin_list_get_column_width_by_type(
	edv_recbin_struct *recbin,
	const EDVRecycleBinColumnType column_type
)
{
	GtkCList *clist;
	const gint column_num = edv_recycle_bin_list_get_column_index_by_type(
		recbin,
		column_type
	);
	if(column_num < 0)
		return(0);

	clist = GTK_CLIST(recbin->contents_clist);

	return(clist->column[column_num].width);
}

/*
 *	Gets the column type that the column index is displaying.
 *
 *	The column_num specifies the column index.
 *
 *	Returns the EDVRecycleBinColumnType that column_num is
 *	displaying.
 */
EDVRecycleBinColumnType edv_recycle_bin_list_get_column_type_by_index(
	edv_recbin_struct *recbin,
	const gint column_num
)
{
	CfgList *cfg_list;
	CfgIntList *column_types_intlist;
	EDVCore *core;

	if(recbin == NULL)
		return(EDV_RECYCLE_BIN_COLUMN_TYPE_NAME);

	core = recbin->core;
	cfg_list = core->cfg_list;

	column_types_intlist = EDV_GET_INT_LIST(
		EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN
	);
	if(column_types_intlist == NULL)
		return(EDV_RECYCLE_BIN_COLUMN_TYPE_NAME);

	return((EDVRecycleBinColumnType)g_list_nth_data(
		column_types_intlist->list,
		column_num
	));
}

/*
 *	Sets the width of the column that is displaying the column
 *	type.
 *
 *	The column_type specifies the column type.
 * 
 *	The width specifies the width of the column.
 */
void edv_recycle_bin_list_set_column_width_by_type(
	edv_recbin_struct *recbin,
	const EDVRecycleBinColumnType column_type,
	const gint width
)
{
	GtkCList *clist;
	const gint column_num = edv_recycle_bin_list_get_column_index_by_type(
		recbin,
		column_type
	);
	if(column_num < 0)
		return;

	clist = GTK_CLIST(recbin->contents_clist);

	gtk_clist_set_column_width(
		clist,
		column_num,
		width
	);
}

/*
 *	Resets the column headings to the names and ordering specified
 *	by the configuration.
 *
 *	All inputs assumed valid.
 */
static void edv_recycle_bin_list_reset_columns(edv_recbin_struct *recbin)
{
	gint		i,
			width;
	const gchar *title = NULL;
	GList *glist;
	GtkJustification justify = GTK_JUSTIFY_LEFT;
	EDVCore *core = recbin->core;
	GtkRcStyle *lists_rcstyle = core->lists_rcstyle;
	GtkCList *clist = GTK_CLIST(recbin->contents_clist);
	const gint ncolumns = clist->columns;
	CfgIntList	*column_types_intlist,
			*column_width_intlist;
	CfgList *cfg_list = core->cfg_list;
	EDVRecycleBinColumnType column_type;

	/* Get the column types mapping */
	column_types_intlist = EDV_GET_INT_LIST(
		EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN
	);
	if(column_types_intlist == NULL)
		return;

	/* Get the column widths */
	column_width_intlist = EDV_GET_INT_LIST(
		EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN_WIDTH
	);
	if(column_width_intlist == NULL)
		return;

	gtk_clist_freeze(clist);

	/* Update the clist's column settings */
	gtk_clist_column_titles_active(clist);
	gtk_clist_column_titles_show(clist);
	gtk_clist_set_auto_sort(
		clist,
		FALSE
	);
	gtk_clist_set_sort_type(
		clist,
		GTK_SORT_DESCENDING
	);

	/* Iterate through each column */
	for(glist = column_types_intlist->list, i = 0;
	    glist != NULL;
	    glist = g_list_next(glist), i++
	)
	{
		column_type = (EDVRecycleBinColumnType)glist->data;

		/* Get the width for this column type */
		width = (gint)g_list_nth_data(
			column_width_intlist->list,
			(guint)column_type
		);

		/* Get column title and justification  based on the
		 * column type
		 */
		switch(column_type)
		{
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_INDEX:
			title = "Index";
			justify = GTK_JUSTIFY_RIGHT;
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_NAME:
			title = "Name";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_SIZE:
			title = "Size";
			justify = GTK_JUSTIFY_RIGHT;
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_STORAGE_SIZE:
			title = "Storage Size";
			justify = GTK_JUSTIFY_RIGHT;
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_TYPE:
			title = "Type";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_PERMISSIONS:
			title = "Permissions";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_OWNER:
			title = "Owner";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_GROUP:
			title = "Group";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_DATE_ACCESS:
			title = "Date Access";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_DATE_MODIFIED:
			title = "Date Modified";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_DATE_CHANGED:
			title = "Date Changed";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_DATE_DELETED:
			title = "Date Deleted";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_LINKED_TO:
			title = "Linked To";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_ORIGINAL_LOCATION:
			title = "Original Location";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_CAPACITY_USED:
			title = "Capacity Used";
			justify = GTK_JUSTIFY_LEFT;
			break;
		}

		gtk_clist_set_column_visibility(
			clist,
			i,
			TRUE
		);
		gtk_clist_set_column_auto_resize(
			clist,
			i,
			FALSE
		);
		gtk_clist_set_column_title(
			clist,
			i,
			title
		);
		gtk_clist_set_column_justification(
			clist,
			i,
			justify
		);
		gtk_clist_set_column_width(
			clist,
			i,
			width
		);
	}
	/* Reset the rest of the columns */
	while(i < ncolumns)
	{
		gtk_clist_set_column_visibility(
			clist,
			i,
			FALSE
		);
		i++;
	}

	/* Set RC style after column headings have been mapped */
	if(lists_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			GTK_WIDGET(clist), lists_rcstyle
		);

	gtk_clist_thaw(clist);
}

/*
 *	Resizes the column to optimul size.
 */
void edv_recycle_bin_list_resize_column_optimul(
	edv_recbin_struct *recbin,
	const gint column_num
)
{
	gint		ncolumns,
			width;
	GtkCList *clist;

	if(recbin == NULL)
		return;

	clist = GTK_CLIST(recbin->contents_clist);
	ncolumns = clist->columns;
	if((column_num < 0) || (column_num >= ncolumns))
		return;

	gtk_clist_freeze(clist);

	width = gtk_clist_optimal_column_width(
		clist,
		column_num
	);
	if(width > 0)
		gtk_clist_set_column_width(
			clist,
			column_num,
			width
		);

	gtk_clist_thaw(clist);
}

/*
 *	Resizes all the columns to optimul widths.
 */
void edv_recycle_bin_list_resize_columns_optimul(edv_recbin_struct *recbin)
{
	gint		i,
			ncolumns;
	GtkCList *clist;

	if(recbin == NULL)
		return;

	clist = GTK_CLIST(recbin->contents_clist);
	ncolumns = clist->columns;

	gtk_clist_freeze(clist);

	for(i = 0; i < ncolumns; i++)
		edv_recycle_bin_list_resize_column_optimul(recbin, i);

	gtk_clist_thaw(clist);
}


/*
 *	Sets the cell's value.
 */
static void edv_recycle_bin_list_set_cell_index(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column
)
{
	const gint border_minor = 2;
	const guint index = obj->index;
 	gchar num_str[40];

	/* Format the index string */
	if(index != 0)
		(void)g_snprintf(
			num_str, sizeof(num_str),
			"#%ld",
			obj->index
		);
	else
		*num_str = '\0';

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		num_str
	);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, -border_minor
	);
	gtk_clist_thaw(clist);
}

static void edv_recycle_bin_list_set_cell_name(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column
)
{
	const gchar *name = obj->name;
	EDVPixmap	*icon,
			*icon_opened,
			*icon_inaccessable,
			*icon_hidden;

	/* Get the pixmap and mask for the object's icon */
	(void)edv_match_object_icon(
		NULL,				/* No devices */
		core->mime_types_list,
		obj->type,
		name,
		TRUE,				/* Assume link valid */
		obj->permissions,
		EDV_ICON_SIZE_20,
		&icon,
		&icon_opened,
		&icon_inaccessable,
		&icon_hidden
	);
	/* Check if an alternate state icon should be used
	 *
	 * Hidden
	 */
	if(edv_path_is_hidden(name))
	{
		if(edv_pixmap_is_loaded(icon_hidden))
		{
			(void)edv_pixmap_unref(icon);
			icon = edv_pixmap_ref(icon_hidden);
		}
	}

	/* Set the cell */
	gtk_clist_freeze(clist);
	if(edv_pixmap_is_loaded(icon))
		gtk_clist_set_pixtext(
			clist,
			row, column,
			(name != NULL) ? name : "(null)",
			EDV_LIST_PIXMAP_TEXT_SPACING,
			icon->pixmap, icon->mask
		);
	else
		gtk_clist_set_text(
			clist,
			row, column,
			(name != NULL) ? name : "(null)"
		);
	gtk_clist_set_cell_style(
		clist,
		row, column,
		NULL
	);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);

	(void)edv_pixmap_unref(icon);
	(void)edv_pixmap_unref(icon_opened);
	(void)edv_pixmap_unref(icon_inaccessable);
	(void)edv_pixmap_unref(icon_hidden);
}

static void edv_recycle_bin_list_set_cell_size(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column,
	const gboolean hide_dir_size,
	const gboolean hide_link_size,
	const EDVSizeFormat size_format,
	const gulong block_size
)
{
	const gint border_minor = 2;
	const EDVObjectType type = obj->type;
	const gchar *s;

	/* Format the size string based on the object's type */
	if(hide_dir_size && (type == EDV_OBJECT_TYPE_DIRECTORY))
		s = "";
	else if(hide_link_size && (type == EDV_OBJECT_TYPE_LINK))
		s = "";
	else if((type == EDV_OBJECT_TYPE_DEVICE_BLOCK) ||
			(type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
	)
		s = "";
	else
		s = edv_str_size_format(
			obj->size,
			size_format,
			block_size,
			',',
			TRUE
		);

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(s != NULL) ? s : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, -border_minor
	);
	gtk_clist_thaw(clist);
}

static void edv_recycle_bin_list_set_cell_storage_size(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column,
	const gboolean hide_dir_size,
	const gboolean hide_link_size,
	const EDVSizeFormat size_format,
	const gulong block_size
)
{
	const gint border_minor = 2;
	const EDVObjectType type = obj->type;
	const gchar *s;

	/* Format the size string based on the object's type */
	if(hide_dir_size && (type == EDV_OBJECT_TYPE_DIRECTORY))
		s = "";
	else if(hide_link_size && (type == EDV_OBJECT_TYPE_LINK))
		s = "";
	else if((type == EDV_OBJECT_TYPE_DEVICE_BLOCK) ||
			(type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
	)
		s = "";
	else
		s = edv_str_size_format(
			obj->storage_size,
			size_format,
			block_size,
			',',
			TRUE
		);

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(s != NULL) ? s : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, -border_minor
	);
	gtk_clist_thaw(clist);
}

static void edv_recycle_bin_list_set_cell_type(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column
)
{
	gchar *type_string;

	/* Get the type string for this object */
	(void)edv_match_object_type_string(
		core->mime_types_list,
		obj->type,
		obj->name,
		obj->permissions,
		&type_string
	);

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(type_string != NULL) ? type_string : ""
	);
	gtk_clist_set_cell_style(
		clist,
		row, column,
		NULL
	);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);

	g_free(type_string);
}

static void edv_recycle_bin_list_set_cell_permissions(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column,
	gboolean hide_link_permissions
)
{
	const EDVPermissionFlags permissions = obj->permissions;
	gchar *s;

	/* Get permissions string */
	if(hide_link_permissions && (obj->type == EDV_OBJECT_TYPE_LINK))
		s = NULL;
	else
		s = edv_str_permissions(permissions);

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(s != NULL) ? s : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);

	g_free(s);
}

static void edv_recycle_bin_list_set_cell_owner(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column
)
{
	/* Get owner name from object's user id */
	gchar *owner_name = edv_uid_uid_to_name(
		core->uids_list,
		obj->owner_id, NULL
	);

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(owner_name != NULL) ? owner_name : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);

	g_free(owner_name);
}

static void edv_recycle_bin_list_set_cell_group(
	EDVCore *core, edv_recbin_struct *recbin,
	GtkCList *clist, EDVRecycledObject *obj,
	const gint row, const gint column
)
{
	/* Get group name from object's group id */
	gchar *group_name = edv_gid_gid_to_name(
		core->gids_list,
		obj->group_id, NULL
	);

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(group_name != NULL) ? group_name : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);

	g_free(group_name);
}

static void edv_recycle_bin_list_set_cell_date_accessed(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column,
	const EDVDateRelativity date_relativity, const gchar *date_format
)
{
	/* Get/format the date string */
	const gulong t = obj->access_time;
	gchar *s = (t > 0l) ?
		edv_date_string_format(t, date_format, date_relativity) :
		NULL;

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(s != NULL) ? s : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);

	g_free(s);
}

static void edv_recycle_bin_list_set_cell_date_modified(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column,
	const EDVDateRelativity date_relativity, const gchar *date_format
)
{
	/* Get/format the date string */
	const gulong t = obj->modify_time;
	gchar *s = (t > 0l) ?
		edv_date_string_format(t, date_format, date_relativity) :
		NULL;

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(s != NULL) ? s : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);

	g_free(s);
}

static void edv_recycle_bin_list_set_cell_date_changed(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column,
	const EDVDateRelativity date_relativity, const gchar *date_format
)
{
	/* Get/format the date string */
	const gulong t = obj->change_time;
	gchar *s = (t > 0l) ?
		edv_date_string_format(t, date_format, date_relativity) :
		NULL;

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(s != NULL) ? s : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);

	g_free(s);
}

static void edv_recycle_bin_list_set_cell_date_deleted(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column,
	const EDVDateRelativity date_relativity, const gchar *date_format
)
{
	/* Get/format the date string */
	const gulong t = obj->deleted_time;
	gchar *s = (t > 0l) ?
		edv_date_string_format(t, date_format, date_relativity) :
		NULL;

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(s != NULL) ? s : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);

	g_free(s);
}

static void edv_recycle_bin_list_set_cell_linked_to(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column
)
{
	/* Get the link's target value */
	const gchar *target = ((obj->link_target != NULL) &&
		(obj->type == EDV_OBJECT_TYPE_LINK)) ?
		obj->link_target : "";

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist, row, column, target
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist, row, column, 0, 0
	);
	gtk_clist_thaw(clist);
}

static void edv_recycle_bin_list_set_cell_original_location(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column
)
{
	const gchar *location = (obj->original_path != NULL) ?
		obj->original_path : "";
	EDVPixmap       *icon = NULL,
					*icon_inaccessable = NULL,
					*icon_hidden = NULL;
	EDVVFSObject *loc_obj = edv_vfs_object_lstat(location);
	if(loc_obj != NULL)
	{
		/* Get the pixmap and mask for the location icon */
		(void)edv_match_object_icon(
			core->devices_list,
			core->mime_types_list,
			loc_obj->type,
			loc_obj->path,
			EDV_VFS_OBJECT_LINK_TARGET_EXISTS(loc_obj),
			loc_obj->permissions,
			EDV_ICON_SIZE_20,
			&icon,
			NULL,
			&icon_inaccessable,
			&icon_hidden
		);
		/* Hidden */
		if(edv_is_object_hidden(loc_obj))
		{
			if(edv_pixmap_is_loaded(icon_hidden))
			{
				(void)edv_pixmap_unref(icon);
				icon = edv_pixmap_ref(icon_hidden);
			}
		}
		edv_vfs_object_delete(loc_obj);
	}
	else
	{
		EDVMIMEType *m = edv_mime_types_list_match_type(
			core->mime_types_list,
			NULL,
			EDV_MIME_TYPE_TYPE_INODE_DIRECTORY,
			FALSE
		);
		if(m != NULL)
		{
			edv_mime_type_realize(m, FALSE);
			(void)edv_pixmap_unref(icon);
			icon = edv_pixmap_ref(m->small_icon[
				EDV_MIME_TYPE_ICON_STATE_STANDARD
			]);
		}
	}

	/* Set the cell */
	gtk_clist_freeze(clist);
	if(edv_pixmap_is_loaded(icon))
		gtk_clist_set_pixtext(
			clist,
			row, column,
			location,
			EDV_LIST_PIXMAP_TEXT_SPACING,
			icon->pixmap, icon->mask
		);
	else
		gtk_clist_set_text(
			clist,
			row, column,
			location
		);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);

	(void)edv_pixmap_unref(icon);
	(void)edv_pixmap_unref(icon_inaccessable);
	(void)edv_pixmap_unref(icon_hidden);
}

static void edv_recycle_bin_list_set_cell_capacity_used(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GtkCList *clist,
	EDVRecycledObject *obj,
	const gint row, const gint column,
	const gulong recbin_size_max
)
{
	gfloat usage_coeff;
	GtkWidget *w = GTK_WIDGET(clist);
	GtkCListColumn *column_ptr = ((column >= 0) && (column < clist->columns)) ?
		&clist->column[column] : NULL;
	const gint	width = (column_ptr != NULL) ?
		((column_ptr->width_set) ? column_ptr->width : -1) : -1,
			height = GTK_CLIST_ROW_HEIGHT_SET(clist) ?
		(clist->row_height - 4) : -1;
	EDVPixmap *p;

	if(recbin_size_max == 0l)
	{
		gtk_clist_freeze(clist);
		gtk_clist_set_text(
			clist,
			row, column,
			""
		);
		gtk_clist_thaw(clist);
		return;
	}

	/* Calculate this recycled object's usage coefficient */
	usage_coeff = CLIP(
		((gfloat)obj->size / (gfloat)recbin_size_max), 0.0f, 1.0f
	);

	/* Create the usage pixmap */
	p = edv_new_progress_pixmap(
		w->window,
		gtk_widget_get_style(w),
		width, height,
		usage_coeff,
		TRUE,				/* Draw value */
		GTK_ORIENTATION_HORIZONTAL,
		FALSE				/* Not reverse */
	);

	/* Set the cell */
	gtk_clist_freeze(clist);
	if(edv_pixmap_is_loaded(p))
		gtk_clist_set_pixmap(
			clist,
			row, column,
			p->pixmap, p->mask
		);
	else 
		gtk_clist_set_text(
			clist,
			row, column,
			""
		);
	gtk_clist_set_cell_style(
		clist,
		row, column,
		NULL
	);
	gtk_clist_thaw(clist);

	(void)edv_pixmap_unref(p);
}

/*
 *	Sets the Contents List's row to the values of the recycled
 *	object.
 *
 *	The clist specifies the Contents List.
 *
 *	The obj specifies the recycled object.
 *
 *	The row specifies the row.
 *
 *	All inputs assumed valid.
 */
static void EDVRecBinContentsSetRow(
	edv_recbin_struct *recbin,
	const gint row,
	EDVRecycledObject *obj
)
{
	gboolean	hide_dir_size,
					hide_link_size,
					hide_link_permissions;
	gint i;
	gulong		recbin_size_max,
					block_size;
	const gchar *date_format;
	GList *glist;
	CfgIntList *column_types_intlist;
	EDVSizeFormat size_format;
	EDVDateRelativity date_relativity;
	EDVRecycleBinColumnType column_type;
	EDVCore *core = recbin->core;
	GtkCList *clist = GTK_CLIST(recbin->contents_clist);
	CfgList *cfg_list = core->cfg_list;

	/* Get column types mapping */
	column_types_intlist = EDV_GET_INT_LIST(
		EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN
	);
	if(column_types_intlist == NULL)
		return;

	/* Get additional display options */
	size_format = (EDVSizeFormat)EDV_GET_I(
		EDV_CFG_PARM_SIZE_FORMAT
	);
	block_size = EDV_GET_UL(EDV_CFG_PARM_BLOCK_SIZE);
	date_relativity = (EDVDateRelativity)EDV_GET_I(
		EDV_CFG_PARM_DATE_RELATIVITY
	);
	date_format = EDV_GET_S(EDV_CFG_PARM_DATE_FORMAT);
	hide_dir_size = EDV_GET_B(EDV_CFG_PARM_RECBIN_CONTENTS_HIDE_DIR_SIZE);
	hide_link_size = EDV_GET_B(EDV_CFG_PARM_RECBIN_CONTENTS_HIDE_LINK_SIZE);
	hide_link_permissions = EDV_GET_B(EDV_CFG_PARM_RECBIN_CONTENTS_HIDE_LINK_PERMISSIONS);

	recbin_size_max = (gulong)EDV_GET_L(
		EDV_CFG_PARM_RECBIN_SIZE_WARN
	);

	gtk_clist_freeze(clist);

	/* Iterate through each column */
	for(i = 0, glist = column_types_intlist->list;
		glist != NULL;
		i++, glist = g_list_next(glist)
	)
	{
		column_type = (EDVRecycleBinColumnType)glist->data;
		switch(column_type)
		{
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_INDEX:
			edv_recycle_bin_list_set_cell_index(
				core, recbin, clist, obj,
				row, i
			);
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_NAME:
			edv_recycle_bin_list_set_cell_name(
				core, recbin, clist, obj,
				row, i
			);
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_SIZE:
			edv_recycle_bin_list_set_cell_size(
				core, recbin, clist, obj,
				row, i, hide_dir_size, hide_link_size,
				size_format, block_size
			);
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_STORAGE_SIZE:
			edv_recycle_bin_list_set_cell_storage_size(
				core, recbin, clist, obj,
				row, i, hide_dir_size, hide_link_size,
				size_format, block_size
			);
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_TYPE:
			edv_recycle_bin_list_set_cell_type(
				core, recbin, clist, obj,
				row, i
			);
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_PERMISSIONS:
			edv_recycle_bin_list_set_cell_permissions(
				core, recbin, clist, obj,
				row, i, hide_link_permissions
			);
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_OWNER:
			edv_recycle_bin_list_set_cell_owner(
				core, recbin, clist, obj,
				row, i
			);
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_GROUP:
			edv_recycle_bin_list_set_cell_group(
				core, recbin, clist, obj,
				row, i
			);
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_DATE_ACCESS:
			edv_recycle_bin_list_set_cell_date_accessed(
				core, recbin, clist, obj,
				row, i, date_relativity, date_format
			);
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_DATE_MODIFIED:
			edv_recycle_bin_list_set_cell_date_modified(
				core, recbin, clist, obj,
				row, i, date_relativity, date_format
			);
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_DATE_CHANGED:
			edv_recycle_bin_list_set_cell_date_changed(
				core, recbin, clist, obj,
				row, i, date_relativity, date_format
			);
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_DATE_DELETED:
			edv_recycle_bin_list_set_cell_date_deleted(
				core, recbin, clist, obj,
				row, i, date_relativity, date_format

			);
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_LINKED_TO:
			edv_recycle_bin_list_set_cell_linked_to(
				core, recbin, clist, obj,
				row, i
			);
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_ORIGINAL_LOCATION:
			edv_recycle_bin_list_set_cell_original_location(
				core, recbin, clist, obj,
				row, i
			);
			break;
		  case EDV_RECYCLE_BIN_COLUMN_TYPE_CAPACITY_USED:
			edv_recycle_bin_list_set_cell_capacity_used(
				core, recbin, clist, obj,
				row, i,
				recbin_size_max
			);
			break;
		}
	}

	gtk_clist_thaw(clist);
}

/*
 *	Appends the recycled object to the Contents List.
 *
 *	The clist specifies the Contents List.
 *
 *	The obj specifies the recycled object who's values will be
 *	used to append a new row on the Contents List and will be
 *	transfered to the Content List's row data. The obj should
 *	not be referenced again after this call.
 *
 *	Returns the new row index or -1 on error.
 *
 *	All inputs assumed valid.
 */
static gint edv_recycle_bin_list_append_object(
	edv_recbin_struct *recbin,
	EDVRecycledObject *obj
)
{
	gint		i,
			new_row;
	gchar **strv;
	GtkCList *clist = GTK_CLIST(recbin->contents_clist);
	const gint ncolumns = MAX(clist->columns, 1);

	gtk_clist_freeze(clist);

	/* Allocate the row cell values */
	strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
		strv[i] = "";

	/* Append the new row */
	new_row = gtk_clist_append(clist, strv);

	/* Delete the row cell values */
	g_free(strv);

	/* Failed to append row? */
	if(new_row < 0)
	{
		gtk_clist_thaw(clist);
		edv_recycled_object_delete(obj);
		return(-1);
	}

	/* Set the new row's values */
	EDVRecBinContentsSetRow(
		recbin,
		new_row,
		obj
	);

	/* Set this recycled object as the new row's data */
	gtk_clist_set_row_data_full(
		clist,
		new_row,
		obj, (GtkDestroyNotify)edv_recycled_object_delete
	);

	gtk_clist_thaw(clist);

	return(new_row);
}

/*
 *	Appends the recycled objects in the recycle bin directory to
 *	the Contents List.
 *
 *	The clist specifies the Contents List.  
 *
 *	If show_progress is TRUE then the progress and messages will
 *	be displayed on the status bar during this operation.
 *
 *	All inputs assumed valid.
 */
static void edv_recycle_bin_list_append_listing(
	edv_recbin_struct *recbin,
	const gboolean show_progress
)
{
#ifdef HAVE_REGEX
	regex_t *regex_filter;
#endif
	gint nobjs, nobjs_loaded, last_progress_percent;
	const gchar	*recbin_index_file,
					*filter = recbin->contents_list_filter;
	GtkWidget *sb = recbin->status_bar;
	GtkCList *clist = GTK_CLIST(recbin->contents_clist);
	EDVCore *core = recbin->core;
	CfgList *cfg_list = core->cfg_list;
	EDVRecycledObject *obj;
	EDVRecycleBinIndex *rp;

#if defined(HAVE_REGEX)
	/* Compile the regex search criteria */
	if(STRISEMPTY(filter) ?
		FALSE : strcmp(filter, "*")
	)
	{
		regex_filter = (regex_t *)g_malloc(sizeof(regex_t));
		if(regcomp(
			regex_filter,
			filter,
#ifdef REG_EXTENDED
			REG_EXTENDED |		/* Use POSIX extended regex */
#endif
			REG_NOSUB		/* Do not report subpattern matches */
		))
		{
			g_free(regex_filter);
			regex_filter = NULL;
		}
	}
	else
	{
		regex_filter = NULL;
	}
#else
	if(STRISEMPTY(filter) ?
		TRUE : !strcmp(filter, "*")
	)
		filter = NULL;
#endif

	/* Get the full path to the recycle bin index file */
	recbin_index_file = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);
	if(STRISEMPTY(recbin_index_file))
		return;

	/* Get the total number of recycled objects */
	nobjs = edv_recycle_bin_index_get_total(recbin_index_file);

	/* Report the number of recycled objects being loaded? */
	if(show_progress && (nobjs > 0))
	{
		gchar *msg = g_strdup_printf(
"Loading Recycle Bin contents (%i %s)...",
			nobjs,
			(nobjs == 1) ? "recycled object" : "recycled objects"
		);
		edv_status_bar_message(sb, msg, TRUE);
		g_free(msg);
	}

	/* Open the recycle bin index file */
	rp = edv_recycle_bin_index_open(recbin_index_file);

	last_progress_percent = -1;
	nobjs_loaded = 0;

#define UPDATE_PROGRESS	{				\
 if(show_progress && (nobjs > 0)) {			\
  const gint progress_percent = nobjs_loaded * 100 / nobjs; \
  if(progress_percent > last_progress_percent) {	\
   edv_status_bar_progress(				\
    sb,							\
    (gfloat)progress_percent / 100.0f,			\
    TRUE						\
   );							\
   last_progress_percent = progress_percent;		\
  }							\
 }							\
}

	/* Begin reading the recycle bin index file */
	gtk_clist_freeze(clist);
	for(obj = edv_recycle_bin_index_next(rp);
		obj != NULL;
		obj = edv_recycle_bin_index_next(rp)
	)
	{
		if(obj->name != NULL)
		{
			const gchar *name = obj->name;
			EDVRecycledObject *tar_obj;

			/* Filter check */
#if defined(HAVE_REGEX)
			if(regex_filter != NULL)
			{
				if(regexec(
					regex_filter,
					name,
					0, NULL,
					0
				) == REG_NOMATCH)
				{
					nobjs_loaded++;
					UPDATE_PROGRESS
					continue;
				}
			}
#else
			if(filter != NULL)
			{
				if(fnmatch(filter, name, 0) == FNM_NOMATCH)
				{
					nobjs_loaded++;
					UPDATE_PROGRESS
					continue;
				}
			}
#endif

			/* Make a copy of the recycled object obtained from
			 * the index file
			 */
			tar_obj = edv_recycled_object_copy(obj);
			if(tar_obj != NULL)
			{
				/* Append/transfer the object to the listing */
				edv_recycle_bin_list_append_object(recbin, tar_obj);
				nobjs_loaded++;
				UPDATE_PROGRESS
			}
		}
		else
		{
			EDVRecycledObject *tar_obj = edv_recycle_bin_list_new_error_object(
				obj->index,
				NULL
			);
			if(tar_obj != NULL)
			{
				/* Append/transfer the object to the listing */
				edv_recycle_bin_list_append_object(recbin, tar_obj);
				nobjs_loaded++;
				UPDATE_PROGRESS
			}
		}
	}
	gtk_clist_thaw(clist);

#undef UPDATE_PROGRESS

	/* Close the recycle bin index file */
	edv_recycle_bin_index_close(rp);

#ifdef HAVE_REGEX
	if(regex_filter != NULL)
	{
		regfree(regex_filter);
		g_free(regex_filter);
	}
#endif
}

/*
 *	Finds the row by index.
 *
 *	The index specifies the recycled object index to match
 *	with a row's recycled object index.
 *
 *	Returns the matched row or negative on error.
 */
gint edv_recycle_bin_list_find_row_by_index(
	edv_recbin_struct *recbin,
	const guint index
)
{
	gint i, n;
	GtkCList *clist;
	EDVRecycledObject *obj;

	if(recbin == NULL)
		return(-2);

	clist = GTK_CLIST(recbin->contents_clist);

	n = clist->rows;
	for(i = 0; i < n; i++)
	{
		obj = EDV_RECYCLED_OBJECT(gtk_clist_get_row_data(clist, i));
		if(obj == NULL)
			continue;

		if(obj->index == index)
			return(i);
	}

	return(-1);
}

/*
 *	Updates all the existing rows on the Contents GtkCList by
 *	getting each row's recycled object and updating the row's
 *	cell values with it.
 */
void edv_recycle_bin_list_realize_listing(edv_recbin_struct *recbin)
{
	gint i, n;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVCore *core;
	EDVRecycledObject *obj;

	if(recbin == NULL)
		return;

	recbin->stop_count = 0;

	clist = GTK_CLIST(recbin->contents_clist);
	core = recbin->core;
	cfg_list = core->cfg_list;

	gtk_clist_freeze(clist);

	/* Reset the columns */
	edv_recycle_bin_list_reset_columns(recbin);

	/* Automatically resize the columns to optimul widths based on
	 * their headings?
	 */
	if(EDV_GET_B(EDV_CFG_PARM_RECBIN_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
		edv_recycle_bin_list_resize_columns_optimul(recbin);

	/* Update the rows */
	n = clist->rows;
	for(i = 0; i < n; i++)
	{
		obj = EDV_RECYCLED_OBJECT(gtk_clist_get_row_data(clist, i));
		if(obj == NULL)
			continue;

		EDVRecBinContentsSetRow(
			recbin,
			i,				/* Row number */
			obj
		);
	}

	/* Automatically resize the columns to optimul widths based on
	 * their row values?
	 */
	if(EDV_GET_B(EDV_CFG_PARM_RECBIN_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
		edv_recycle_bin_list_resize_columns_optimul(recbin);

	gtk_clist_thaw(clist);
}


/*
 *	Clears the Contents List, resets the columns, and gets a new
 *	listing of recycled objects.
 *
 *	The path specifies the full path to the location.
 *
 *	If show_progress is TRUE then the progress and messages will
 *	be displayed on the status bar during this operation.
 */
void edv_recycle_bin_list_get_listing(
	edv_recbin_struct *recbin,
	const gboolean show_progress
)
{
	GtkWidget *sb;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVCore *core;

	if(recbin == NULL)
		return;

	recbin->stop_count = 0;

	clist = GTK_CLIST(recbin->contents_clist);
	sb = recbin->status_bar;
	core = recbin->core;
	cfg_list = core->cfg_list;

	/* Report the initial progress? */
	if(show_progress)
	{
		edv_status_bar_message(
			sb,
"Loading Recycle Bin contents...",
			FALSE
		);
		edv_status_bar_progress(sb, 0.0f, TRUE);
	}

	gtk_clist_freeze(clist);

	/* Clear the listing */
	edv_recycle_bin_list_clear(recbin);

	/* Reset the columns */
	edv_recycle_bin_list_reset_columns(recbin);

	/* Automatically resize the columns to optimul widths based on
	 * their headings?
	 */
	if(EDV_GET_B(EDV_CFG_PARM_RECBIN_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
		edv_recycle_bin_list_resize_columns_optimul(recbin);

	/* Append the list of recycled objects to the listing */
	edv_recycle_bin_list_append_listing(
		recbin,
		show_progress
	);

	/* Automatically resize the columns to optimul widths based on
	 * their row values?
	 */
	if(EDV_GET_B(EDV_CFG_PARM_RECBIN_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
		edv_recycle_bin_list_resize_columns_optimul(recbin);

	gtk_clist_thaw(clist);

	/* Report the final progress? */
	if(show_progress)
	{
		/* Reset progress */
		edv_status_bar_message(
			sb,
"Recycle Bin contents loaded",
			FALSE
		);
		edv_status_bar_progress(sb, 0.0f, TRUE);
	}
}

/*
 *	Deletes all items in the given clist.
 */
void edv_recycle_bin_list_clear(edv_recbin_struct *recbin)
{
	GtkCList *clist;

	if(recbin == NULL)
		return;

	clist = GTK_CLIST(recbin->contents_clist);
	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	gtk_clist_thaw(clist);
}


/*
 *	FPrompt rename apply callback.
 */
static void edv_recycle_bin_list_fprompt_rename_apply_cb(
	gpointer data, const gchar *value
)
{
	gint row;
	edv_recbin_struct *recbin;
	EDVRecBinContentsListFPromptData *d = EDV_RECYCLE_BIN_CONTENTS_LIST_FPROMPT_DATA(data);
	if(d == NULL)
		return;

	recbin = d->recbin;
	row = d->row;

	/* Inputs valid? */
	if((recbin != NULL) && (row > -1) && (value != NULL))
	{
		GtkWidget	*toplevel = recbin->toplevel,
					*sb = recbin->status_bar;
		GtkCList *clist = GTK_CLIST(recbin->contents_clist);
		EDVCore *core = recbin->core;

		/* Get the selected object */
		EDVRecycledObject *obj = EDV_RECYCLED_OBJECT(gtk_clist_get_row_data(
			clist,
			row
		));
		if(obj != NULL)
		{
			gboolean yes_to_all = FALSE;
			const guint index = obj->index;
			gchar *old_name = STRDUP(obj->name);
			const gchar *new_name = value;
			const gchar *error_msg;
			GList *modified_indicies_list;

			/* Rename */
			(void)edv_recycled_object_op_rename(
				core,
				index,
				new_name,
				&modified_indicies_list,
				toplevel,
				FALSE,			/* Do not show progress */
				TRUE,			/* Interactive */
				&yes_to_all
			);

			/* Unmap the progress dialog */
			ProgressDialogBreakQuery(FALSE);
			ProgressDialogSetTransientFor(NULL);

			/* Check for errors */
			error_msg = edv_recycled_object_op_get_error(core);
			if(!STRISEMPTY(error_msg))
			{
				/* Report the error */
				edv_play_sound_error(core);
				edv_message_error(
					"Rename Error",
					error_msg,
					NULL,
					toplevel
				);
			}

			if(modified_indicies_list != NULL)
			{
				guint index;
				gchar *msg;
				GList *glist;

				/* Notify about the modified recycled objects */
				for(glist = modified_indicies_list;
					glist != NULL;
					glist = g_list_next(glist)
				)
				{
					index = (guint)glist->data;
					if(index == 0)
						continue;

					edv_emit_recycled_object_modified(core, index);
				}

				/* Update the status bar message */
				msg = g_strdup_printf(
					"Object \"%s\" renamed to \"%s\"",
					old_name,
					new_name
				);
				edv_status_bar_message(sb, msg, FALSE);
				g_free(msg);

				/* Delete the modified indicies list */
				g_list_free(modified_indicies_list);
			}
			else
			{
				/* Did not get the modified indicies list so this
				 * implies failure
				 */
				edv_status_bar_message(
					sb,
					"Rename object failed",
					FALSE
				);
 		}

			g_free(old_name);
		}
	}

	g_free(d);
}

/*
 *	FPrompt rename cancel callback.
 */
static void edv_recycle_bin_list_fprompt_cancel_cb(gpointer data)
{
	EDVRecBinContentsListFPromptData *d = EDV_RECYCLE_BIN_CONTENTS_LIST_FPROMPT_DATA(data);
	if(d == NULL)
		return;

	g_free(d);
}

/*
 *	Prompts to rename the object.
 *
 *	The row and column specifies the object on the list to map
 *	the rename prompt at. If column is -1 then the column that
 *	is set to EDV_RECYCLE_BIN_COLUMN_TYPE_NAME will be used.
 */
void edv_recycle_bin_list_rename_query(
	edv_recbin_struct *recbin,
	const gint row,
	const gint column
)
{
	gint		cx, cy,
			px, py,
			pwidth, pheight,
			name_column;
	GtkWidget *toplevel;
	GtkCList *clist;
	CfgList *cfg_list;
	CfgIntList *column_type_intlist;
	EDVRecycledObject *obj;
	EDVCore *core;

	if((recbin == NULL) || FPromptIsQuery())
		return;

	toplevel = recbin->toplevel;
	clist = GTK_CLIST(recbin->contents_clist);
	core = recbin->core;
	cfg_list = core->cfg_list;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
		return;

	edv_recycle_bin_sync_data(recbin);

	/* Row does not exist? */
	if((row < 0) || (row >= clist->rows))
		return;

	/* Find which column is displaying the name */
	name_column = -1;
	column_type_intlist = EDV_GET_INT_LIST(
		EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN
	);
	if(column_type_intlist != NULL)
	{
		gint i = 0;
		GList *glist;

		for(glist = column_type_intlist->list;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			if((EDVRecycleBinColumnType)glist->data ==
				EDV_RECYCLE_BIN_COLUMN_TYPE_NAME
			)
			{
				name_column = i;
				break;
			}
			i++;
		}
	}
	/* No column displaying the name? */
	if(name_column < 0)
		return;

	/* Get the cell's geometry */
	if(!gtk_clist_get_cell_geometry(
		clist, name_column, row,
		&cx, &cy, &pwidth, &pheight
	))
		return;

	/* Get the root window relative coordinates */
	px = 0;
	py = 0;
	gdk_window_get_deskrelative_origin(
		clist->clist_window, &px, &py
	);
	px += cx + 0;
	py += cy - 2;           /* Move up a little */

	/* Get the recycled object */
	obj = EDV_RECYCLED_OBJECT(gtk_clist_get_row_data(clist, row));
	if(obj == NULL)
		return;

	/* Check if the object's name is a special notation that
	 * may not be renamed
	 */
	if(!STRISEMPTY(obj->name))
	{
		const gchar *name = obj->name;
		if(!strcmp((const char *)name, ".") ||
		   !strcmp((const char *)name, "..") ||
		   !strcmp((const char *)name, "/")
		)
			return;
	}

	/* Is the specified column the name column or -1? */
	if((column == name_column) || (column < 0))
	{
		gchar *value = STRDUP(obj->name);
		EDVRecBinContentsListFPromptData *d = EDV_RECYCLE_BIN_CONTENTS_LIST_FPROMPT_DATA(
			g_malloc(sizeof(EDVRecBinContentsListFPromptData))
		);
		if(d != NULL)
		{
			d->recbin = recbin;
			d->row = row;
			d->column = column;
		}

		/* Map floating prompt to change values */
		FPromptSetTransientFor(toplevel);
		FPromptSetPosition(px, py);
		FPromptMapQuery(
			NULL,			/* No label */
			value,
			NULL,			/* No tooltip message */
			FPROMPT_MAP_TO_POSITION,	/* Map code */
			pwidth, -1,		/* Width and height */
			0,			/* No buttons */
			d,			/* Callback data */
			NULL,			/* No browse callback */
			edv_recycle_bin_list_fprompt_rename_apply_cb,
			edv_recycle_bin_list_fprompt_cancel_cb
		);

		g_free(value);
	}
}


/*
 *	Recycled object added callback.
 */
void edv_recycle_bin_list_recycled_object_added_cb(
	edv_recbin_struct *recbin, const guint index
)
{
	gint row;
	const gchar *recbin_index_file;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVCore *core;

	if(recbin == NULL)
		return;

	clist = GTK_CLIST(recbin->contents_clist);
	core = recbin->core;
	cfg_list = core->cfg_list;

	/* Get the full path to recycle bin index file */
	recbin_index_file = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);
	if(STRISEMPTY(recbin_index_file))
		return;

	/* Check if the recycled object specified by the index already
	 * exists in the list
	 */
	row = edv_recycle_bin_list_find_row_by_index(recbin, index);
	if(row > -1)
	{
		/* Recycled object already in list, do nothing (no update) */
	}
	else
	{
		/* Recycled object not in list, so add it */
		EDVRecycledObject *obj = edv_recycled_object_stat(
			recbin_index_file,
			index
		);
		if(obj != NULL)
		{
			/* Add the recycled object to the list */
			edv_recycle_bin_list_append_object(
				recbin,
				obj
			);
/* 			obj = NULL; */

			/* Resize the columns */
			if(EDV_GET_B(EDV_CFG_PARM_RECBIN_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
				edv_recycle_bin_list_resize_columns_optimul(recbin);
		}
	}
}

/*
 *	Recycled object modified callback.
 */
void edv_recycle_bin_list_recycled_object_modified_cb(
	edv_recbin_struct *recbin, const guint index
)
{
	gint row;
	const gchar *recbin_index_file;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVCore *core;

	if(recbin == NULL)
		return;

	clist = GTK_CLIST(recbin->contents_clist);
	core = recbin->core;
	cfg_list = core->cfg_list;

	/* Get the full path to recycle bin index file */
	recbin_index_file = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);
	if(STRISEMPTY(recbin_index_file))
		return;

	/* Update the modified recycled object in the list */
	row = edv_recycle_bin_list_find_row_by_index(recbin, index);
	if(row > -1)
	{
		/* Get the modified recycled object's statistics */
		EDVRecycledObject	*obj_values = edv_recycled_object_stat(
			recbin_index_file,
			index
		),
									*obj = EDV_RECYCLED_OBJECT(gtk_clist_get_row_data(
			clist,
			row
		));
		if((obj_values != NULL) && (obj != NULL))
		{
			/* Update the recycled object */
			edv_recycled_object_set_object(
				obj,			/* Target */
				obj_values			/* Source */
			);

			gtk_clist_freeze(clist);

			EDVRecBinContentsSetRow(
				recbin,
				row,
				obj
			);

			/* Resize the columns */
			if(EDV_GET_B(EDV_CFG_PARM_RECBIN_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
				edv_recycle_bin_list_resize_columns_optimul(recbin);

			gtk_clist_thaw(clist);
		}
		edv_recycled_object_delete(obj_values);
	}
	else
	{
		/* The modified recycled object was not in the list, so
		 * add the modified recycled object to the list
		 */
		EDVRecycledObject *obj = edv_recycled_object_stat(
			recbin_index_file,
			index
		);
		if(obj != NULL)
		{
			edv_recycle_bin_list_append_object(
				recbin,
				obj
			);
/*			obj = NULL; */

			/* Resize the columns */
			if(EDV_GET_B(EDV_CFG_PARM_RECBIN_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
				edv_recycle_bin_list_resize_columns_optimul(recbin);
		}
	}
}

/*
 *	Recycled object removed callback.
 */
void edv_recycle_bin_list_recycled_object_removed_cb(
	edv_recbin_struct *recbin, const guint index
)
{
	gint row;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVCore *core;

	if(recbin == NULL)
		return;

	clist = GTK_CLIST(recbin->contents_clist);
	core = recbin->core;
	cfg_list = core->cfg_list;

	/* Remove all the rows who's object's index matches the
	 * removed index
	 */
	row = edv_recycle_bin_list_find_row_by_index(recbin, index);
	if(row > -1)
	{
		gtk_clist_freeze(clist);
		do {
			gtk_clist_remove(
				clist,
				row
			);
			row = edv_recycle_bin_list_find_row_by_index(
				recbin,
				index
			);
		} while(row > -1);
		gtk_clist_thaw(clist);

		/* Resize the columns */
		if(EDV_GET_B(EDV_CFG_PARM_RECBIN_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
			edv_recycle_bin_list_resize_columns_optimul(recbin);
	}
}
