#include <errno.h>
#include <gtk/gtk.h>

#include "url.h"

#include "cdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_recycled_obj.h"
#include "libendeavour2-base/edv_recycled_obj_stat.h"
#include "edv_utils_gtk.h"
#include "edv_recycle_obj.h"
#include "recycle_bin.h"
#include "recycle_bin_desktop_icon.h"
#include "recycle_bin_dnd.h"
#include "edv_dnd.h"

#include "config.h"


static gint edv_recycle_bin_drag_data_received_nexus(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GdkDragContext *dc,
	const guint info,
	GtkSelectionData *selection_data,
	GtkWidget *toplevel
);

/* Contents GtkCList */
void edv_recycle_bin_list_drag_data_get_cb(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info,
	guint t,
	gpointer data
);
void edv_recycle_bin_list_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info,
	guint t,
	gpointer data
);
void edv_recycle_bin_list_drag_data_delete_cb(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
);

/* Desktop Icon */
void edv_recycle_bin_desktop_icon_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info,
	guint t,
	gpointer data
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Drag data received nextus.
 *
 *	If recbin is not NULL then recbin specifies the source
 *	Recycle Bin.
 *
 *	The dc specifies the GdkDragContext.
 *
 *	The info specifies the source object's location type,
 *
 *	The selection_data specifies the DND data describing the
 *	source objects in the form of a null-separated list of
 *	URL strings which will be parsed into a list of URL strings.
 *
 *	Returns 0 on success or non-zero on error.
 */
static gint edv_recycle_bin_drag_data_received_nexus(
	EDVCore *core,
	edv_recbin_struct *recbin,
	GdkDragContext *dc,
	const guint info,
	GtkSelectionData *selection_data,
	GtkWidget *toplevel
)
{
	gint status;

	if(recbin != NULL)
		edv_recycle_bin_set_busy(recbin, TRUE);

	/* Process the DND operation to the Reccyle Bin */
	status = edv_dnd_vfs_to_recycle_bin(
		core,
		dc,
		info,				/* Source location type */
		selection_data,			/* Source objects */
		TRUE,				/* Verbose */
		toplevel,
		(recbin != NULL) ? recbin->status_bar : NULL
	);

	if(recbin != NULL)
		edv_recycle_bin_set_busy(recbin, FALSE);

	return(status);
}


/*
 *	DND "drag_data_get" signal callback.
 */
void edv_recycle_bin_list_drag_data_get_cb(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gboolean data_sent = FALSE;
	GList		*glist,
			*urls_list = NULL;
	GtkWidget *w;
	GtkCList *clist;
	URLStruct *url;
	EDVRecycledObject *obj;
	EDVCore *core;
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((dc == NULL) || (recbin == NULL))
		return;

	w = recbin->contents_clist;
	clist = GTK_CLIST(w);
	core = recbin->core;

	edv_recycle_bin_sync_data(recbin);

	/* Generate a list of URLs from the selected rows */
	for(glist = clist->selection;                        
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		obj = EDV_RECYCLED_OBJECT(gtk_clist_get_row_data(
			clist,
			(gint)glist->data
		));
		if(obj == NULL)
			continue;

		url = url_new();
		if(url == NULL)
			break;

		/* For recycled objects, the path must be set as the
		 * recycled object's index value as a string
		 */
		url->path = g_strdup_printf(
			"%ld",
			obj->index
		);

		/* For recycled objects, the path argument must be set
		 * as the recycled object's original name
		 */
		url->path_arg = STRDUP(obj->name);

		urls_list = g_list_append(
			urls_list,
			url
		);
	}

	/* Encode the DND data from the URL list */
	if(urls_list != NULL) 
	{
		gint buf_len;
		guint8 *buf = url_encode(
			urls_list,
			&buf_len
		);
		if(buf != NULL)
		{
			/* Send out buffer */
			gtk_selection_data_set(
				selection_data,
				GDK_SELECTION_TYPE_STRING,
				8,		/* Bits Per Character */
				buf,		/* Data */
				buf_len		/* Length */
			);
			data_sent = TRUE;
			g_free(buf);
		}

		/* Delete the URL list */
		g_list_foreach(
			urls_list,
			(GFunc)url_delete,
			NULL
		);
		g_list_free(urls_list);
	}

	/* If failed to send out data then respond with error */
	if(!data_sent)
	{
		const char *s = "Error";
		gtk_selection_data_set(
			selection_data,
			GDK_SELECTION_TYPE_STRING,
			8,			/* Bits Per Character */
			s,			/* Data */
			STRLEN(s)		/* Length */
		);
		data_sent = TRUE;
	}
}

/*
 *	Recycle Bin Contents List "drag_data_received" signal callback.
 */
void edv_recycle_bin_list_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info,
	guint t,
	gpointer data
)
{
	GtkWidget *toplevel;
	EDVCore *core;
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((dc == NULL) || (recbin == NULL))
		return;

	if(EDV_RECYCLE_BIN_IS_PROCESSING(recbin) || (recbin->freeze_count > 0))
		return;

	/* No source data? */
	if(selection_data == NULL)
		return;
	if(selection_data->length <= 0)
		return;

	toplevel = recbin->toplevel;
	core = recbin->core;

	edv_recycle_bin_sync_data(recbin);

	/* Handle the received data */
	edv_recycle_bin_drag_data_received_nexus(
		core,
		recbin,
		dc,
		info,
		selection_data,
		toplevel
	);
}

/*
 *	DND "drag_data_delete" signal callback.
 */
void edv_recycle_bin_list_drag_data_delete_cb(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{

}


/*
 *      Recycle Bin Desktop Icon "drag_data_received" signal callback.
 */
void edv_recycle_bin_desktop_icon_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info,
	guint t,
	gpointer data
)
{
	GtkWidget *toplevel;
	EDVCore *core;
	EDVRecycleBinDesktopIcon *rbdi = EDV_RECYCLE_BIN_DESKTOP_ICON(data);
	if((dc == NULL) || (rbdi == NULL))
		return;

	/* No source data? */
	if(selection_data == NULL)
		return;
	if(selection_data->length <= 0)
		return;

	toplevel = (widget != NULL) ? gtk_widget_get_toplevel(widget) : NULL;
	core = rbdi->core;

	/* Handle the received data */
	edv_recycle_bin_drag_data_received_nexus(
		core,
		NULL,				/* No Recycle Bin */
		dc,
		info,
		selection_data,
		toplevel
	);
}
