#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "cfg.h"

#include "guiutils.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_property.h"
#include "libendeavour2-base/edv_recycled_obj.h"
#include "edv_list_seek.h"
#include "edv_utils_gtk.h"
#include "edv_status_bar.h"
#include "recycle_bin.h"
#include "recycle_bin_cb.h"
#include "recycle_bin_list.h"
#include "recycle_bin_dnd.h"
#include "recycle_bin_op.h"
#include "edv_cb.h"
#include "edv_op.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


/* General */
void edv_recycle_bin_realize_cb(GtkWidget *widget, gpointer data);
gint edv_recycle_bin_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
gint edv_recycle_bin_key_event_cb(
	 GtkWidget *widget, GdkEventKey *key, gpointer data
);
gint edv_recycle_bin_button_event_cb(
	GtkWidget *widget, GdkEventButton *button, gpointer data
);

/* GtkHandleBox */
void edv_recycle_bin_handle_child_attached_cb(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
);
void edv_recycle_bin_handle_child_detached_cb(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
);

/* Contents GtkCList */
static gint edv_recycle_bin_column_sort_index_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint edv_recycle_bin_column_sort_size_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint edv_recycle_bin_column_sort_storage_size_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint edv_recycle_bin_column_sort_date_nexus(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2,
	const gint sort_code
);
static gint edv_recycle_bin_column_sort_date_accessed_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint edv_recycle_bin_column_sort_date_modified_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint edv_recycle_bin_column_sort_date_changed_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint edv_recycle_bin_column_sort_date_deleted_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint edv_recycle_bin_column_sort_capacity_used_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
void edv_recycle_bin_resize_column_cb(
	GtkCList *clist, gint column, gint width, gpointer data
);
void edv_recycle_bin_click_column_cb(
	GtkCList *clist, gint column, gpointer data
);
void edv_recycle_bin_select_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
void edv_recycle_bin_unselect_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);

/* GtkMenuItem */
void edv_recycle_bin_menu_item_cb(GtkWidget *widget, gpointer data);
gint edv_recycle_bin_menu_item_enter_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
gint edv_recycle_bin_menu_item_leavr_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);

/* EDVFindBar */
void edv_recycle_bin_find_bar_start_cb(
	GtkWidget *bar,
	gpointer data
);
void edv_recycle_bin_find_bar_end_cb(
	GtkWidget *bar,
	const gint nmatches,
	gpointer data
);
void edv_recycle_bin_find_bar_match_cb(
	GtkWidget *bar,
	const gchar *path,
	GList *properties_list,
	const gint line_index,
	const gchar *excerpt,
	gpointer data
);

/* EDVStatusBar */
void edv_recycle_bin_find_bar_status_message_cb(
	GtkWidget *widget,
	const gchar *message, 
	gpointer data
);
void edv_recycle_bin_find_bar_status_progress_cb(
	GtkWidget *widget,
	const gfloat progress, 
	gpointer data
);

/* Window Created/Deleted */
void edv_recycle_bin_window_created_cb(
	edv_recbin_struct *recbin,
	const EDVWindowType win_type,
	gpointer win
);
void edv_recycle_bin_window_deleted_cb(
	edv_recbin_struct *recbin,
	const EDVWindowType win_type,
	gpointer win
);

/* Reconfigured Callback */
void edv_recycle_bin_reconfigured_cb(edv_recbin_struct *recbin);

/* Master Write Protect Changed */
void edv_recycle_bin_master_write_protect_changed_cb(
	edv_recbin_struct *recbin,
	const gboolean state
);

/* Delete Method Changed */
void edv_recycle_bin_delete_method_changed_cb(
	edv_recbin_struct *recbin,
	const EDVDeleteMethod delete_method
);

/* EDVRecycledObjects */
void edv_recycle_bin_recycled_object_added_cb(
	edv_recbin_struct *recbin,
	const guint index
);
void edv_recycle_bin_recycled_object_modified_cb(
	edv_recbin_struct *recbin,
	const guint index
);
void edv_recycle_bin_recycled_object_removed_cb(
	edv_recbin_struct *recbin,
	const guint index
);

/* EDVMIMETypes */
void edv_recycle_bin_mime_type_added_cb(
	edv_recbin_struct *recbin,
	const gint mt_num, EDVMIMEType *mt
);
void edv_recycle_bin_mime_type_modified_cb(
	edv_recbin_struct *recbin,
	const gint mt_num, EDVMIMEType *mt
);
void edv_recycle_bin_mime_type_removed_cb(
	edv_recbin_struct *recbin,
	const gint mt_num
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Toplevel GtkWindow "realize" signal callback.
 */
void edv_recycle_bin_realize_cb(GtkWidget *widget, gpointer data)
{
	GdkWindow *window;
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((widget == NULL) || (recbin == NULL))
		return;

	window = widget->window;
	if(window != NULL)
	{
		GdkGeometry geo;
		geo.min_width = 320;
		geo.min_height = 240;
		geo.base_width = 0;
		geo.base_height = 0;
		geo.width_inc = 1;
		geo.height_inc = 1;
		gdk_window_set_geometry_hints(
			window,
			&geo,
			GDK_HINT_MIN_SIZE | GDK_HINT_BASE_SIZE |
				GDK_HINT_RESIZE_INC
		);
		gdk_window_set_hints(
			window,
			0, 0,				/* Position */
			geo.min_width, geo.min_height,	/* Min size */
			geo.max_width, geo.max_height,	/* Max size */
			GDK_HINT_MIN_SIZE
		);
	}

	recbin->flags |= EDV_RECYCLE_BIN_REALIZED;
}

/*
 *	Toplevel GtkWindow "delete_event" signal callback.
 */
gint edv_recycle_bin_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if(recbin == NULL)
		return(FALSE);

	if(EDV_RECYCLE_BIN_IS_PROCESSING(recbin) || (recbin->freeze_count > 0))
	{
		recbin->stop_count++;
		return(TRUE);
	}

	edv_recycle_bin_op_close(recbin);

	return(TRUE);
}

/*
 *	Any GtkWidget "key_press_event" or "key_release_event" signal
 *	callback.
 */
gint edv_recycle_bin_key_event_cb(
	 GtkWidget *widget, GdkEventKey *key, gpointer data
)
{
	gboolean is_press;
	gint		status = FALSE,
			etype;
	guint		keyval,
			state;
	GtkWidget	*w,
			*focus_widget;
	CfgList *cfg_list;
	EDVCore *core;
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((widget == NULL) || (key == NULL) || (recbin == NULL))
		return(status);

	if(EDV_RECYCLE_BIN_IS_PROCESSING(recbin) || (recbin->freeze_count > 0))
		return(status);

	w = recbin->toplevel;
	focus_widget = GTK_WINDOW(w)->focus_widget;
	core = recbin->core;
	cfg_list = core->cfg_list;

	etype = key->type;
	is_press = (etype == GDK_KEY_PRESS) ? TRUE : FALSE;
	keyval = key->keyval;
	state = key->state;

/* Stop emit of signal */
#define STOP_SIGNAL_EMISSION(_w_) {		\
 gtk_signal_emit_stop_by_name(			\
  GTK_OBJECT(_w_),				\
  is_press ?					\
   "key_press_event" : "key_release_event"	\
 );						\
}

	/* If the focus_widget is not a GtkEditable then check if the
	 * keyval is an accelerator key before all subsequence checks
	 */
	if((focus_widget != NULL) ?
		!GTK_IS_EDITABLE(focus_widget) : TRUE
	)
	{
		EDVRecycleBinOpID op = (EDVRecycleBinOpID)edv_match_accel_key_op_id(
			cfg_list, EDV_CFG_PARM_RECBIN_ACCELERATOR_KEYS,
			keyval, state
		);
		if(op > 0)
		{
			if(is_press)
			{
				EDVRecycleBinOp *opid = edv_recycle_bin_op_match_by_id(
					recbin, op
				);
				if(opid != NULL)
					edv_recycle_bin_op_cb(NULL, -1, opid);
			}
			STOP_SIGNAL_EMISSION(widget);
			status = TRUE;
			return(status);
		}
	}

	/* Check which widget this signal is for
	 *
	 * Contents GtkCList
	 */
	if(widget == recbin->contents_clist)
	{
		GtkCList *clist = GTK_CLIST(widget);
		gint row = edv_clist_get_selected_last(clist, NULL);

		switch(keyval)
		{
		  case GDK_space:
		  case GDK_KP_Space:
			row = clist->focus_row;
			if((row >= 0) && (row < clist->rows) && is_press)
			{
				gboolean already_selected = FALSE;

				/* Check if this row is already selected */
				GList *glist = clist->selection;
				while(glist != NULL)
				{
					if(row == (gint)glist->data)
					{
						already_selected = TRUE;
						break;
					}
					glist = g_list_next(glist);
				}

				gtk_clist_freeze(clist);
				if(already_selected)
					gtk_clist_unselect_row(clist, row, 0);
				else
					gtk_clist_select_row(clist, row, 0);
				gtk_clist_thaw(clist);
			}
			STOP_SIGNAL_EMISSION(widget);
			status = TRUE;
			break;

		  default:
			/* For all other alphanumeric character keys and while
			 * no modifier keys are held, attempt to seek to the
			 * item who's name starts with the letter of the key
			 * that was pressed
			 */
			if(EDV_GET_B(EDV_CFG_PARM_LISTS_KEYBOARD_SCROLL_TO_KEY_NAME) &&
				isalnum((int)keyval) && is_press
			)
			{
				const gboolean backwards = (state & GDK_MOD1_MASK) ? TRUE : FALSE;
				const gint start_row = (backwards) ?
					(clist->rows - 1) : 0;

				/* Find the column that is set to display the name */
				gint column_num_name = edv_recycle_bin_list_get_column_index_by_type(
					recbin,
					EDV_ARCHIVER_COLUMN_TYPE_NAME
				);
				if(column_num_name < 0)
					column_num_name = 0;

				/* Seek to the row who's text matches this
				 * key event's character
				 */
				gtk_clist_freeze(clist);
				edv_clist_seek_character(
					clist,
					column_num_name,	/* Column */
					start_row,
					backwards,
					0,			/* Text index */
					keyval			/* Character */
				);
				gtk_clist_thaw(clist);

				STOP_SIGNAL_EMISSION(widget);
				status = TRUE;
			}
			break;
		}
	}

#undef STOP_SIGNAL_EMISSION

	return(status);
}

/*
 *	Any GtkWidget "button_press_event" signal callback.
 */
gint edv_recycle_bin_button_event_cb(
	GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	gint		status = FALSE,
			etype;
	guint state;
	CfgList *cfg_list;
	EDVCore *core;
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((widget == NULL) || (button == NULL) || (recbin == NULL))
		return(status);

	if(EDV_RECYCLE_BIN_IS_PROCESSING(recbin) || (recbin->freeze_count > 0))
		return(status);

	core = recbin->core;
	cfg_list = core->cfg_list;
	etype = (gint)button->type;
	state = button->state;

	/* Check which widget this signal is for
	 *
	 * Contents GtkCList
	 */
	if(widget == recbin->contents_clist)
	{
		gint	row, column,
					rows_selected = 0,
					selected_row = -1;
		GList *glist;
		GtkCList *clist = GTK_CLIST(widget);

		/* Find the row and column */
		if(!gtk_clist_get_selection_info(
			clist, button->x, button->y, &row, &column
		))
		{
			row = -1;
			column = 0;
		}

		/* Get number of selected rows and highest selected row */
		glist = clist->selection;
		while(glist != NULL)
		{
			rows_selected++;
			selected_row = (gint)glist->data;
			glist = g_list_next(glist);
		}

		/* Handle by button number */
		switch(button->button)
		{
		  case GDK_BUTTON2:
			switch((EDVListsPointerOpButton2)EDV_GET_I(EDV_CFG_PARM_LISTS_POINTER_OP_BUTTON2))
			{
			  case EDV_LISTS_POINTER_OP_BUTTON2_NONE:
			  case EDV_LISTS_POINTER_OP_BUTTON2_SCROLL_XY:
				/* Generic list callback handles this */
				break;
			  case EDV_LISTS_POINTER_OP_BUTTON2_RENAME:
				if(etype == GDK_BUTTON_PRESS)
				{
					/* Rename */
					edv_recycle_bin_list_rename_query(
						recbin,
						row, column
					);
				}
				status = TRUE;
				break;
			  case EDV_LISTS_POINTER_OP_BUTTON2_PASTE:
				if(etype == GDK_BUTTON_PRESS)
				{
					/* Paste objects */

				}
				status = TRUE;
				break;
			}
			break;

		  case GDK_BUTTON3:
			if(etype == GDK_BUTTON_PRESS)
			{
				/* Select item before mapping menu? */
				if(EDV_GET_B(EDV_CFG_PARM_RIGHT_CLICK_MENU_SELECTS) &&
				   (row >= 0) && (row < clist->rows)
				)
				{
					/* Select the row that the button was pressed over.
					 * if no key modifiers are held then this will also
					 * unselect all previously selected rows.
					 */
					gtk_clist_freeze(clist);
					if(!(state & GDK_CONTROL_MASK) &&
					   !(state & GDK_SHIFT_MASK)
					)
						gtk_clist_unselect_all(clist);
					clist->focus_row = row;
					gtk_clist_select_row(clist, row, 0);
					gtk_clist_thaw(clist);
				}

				/* Update all the menus and then map the
				 * right-click menu
				 */
				edv_recycle_bin_update_display(recbin);
				gtk_menu_popup(
					GTK_MENU(recbin->contents_clist_menu),
					NULL, NULL,
					NULL, NULL,
					button->button, button->time
				);
			}
			status = TRUE;
			break;
		}
	}

	return(status);
}


/*
 *	GtkHandleBox "child_attached" signal callback.
 */
void edv_recycle_bin_handle_child_attached_cb(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
)
{
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((handle_box == NULL) || (recbin == NULL))
		return;

	gtk_widget_queue_resize(
		gtk_widget_get_toplevel(GTK_WIDGET(handle_box))
	);
}

/*
 *	GtkHandleBox "child_detached" signal callback.
 */
void edv_recycle_bin_handle_child_detached_cb(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
)
{
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((handle_box == NULL) || (recbin == NULL))
		return;

	gtk_widget_queue_resize(
		gtk_widget_get_toplevel(GTK_WIDGET(handle_box))
	);
}


/*
 *	Contents GtkCList column sort indicies callback.
 */
static gint edv_recycle_bin_column_sort_index_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	EDVRecycledObject	*obj1 = EDV_RECYCLED_OBJECT(
		((const GtkCListRow *)ptr1)->data
	),
				*obj2 = EDV_RECYCLED_OBJECT(
		((const GtkCListRow *)ptr2)->data
	);
	if((obj1 == NULL) || (obj2 == NULL))
		return(0);

	if(obj1->index <= obj2->index)
		return(obj1->index < obj2->index);
	else
		return(-1);
}

/*
 *	Contents GtkCList column sort size callback.
 */
static gint edv_recycle_bin_column_sort_size_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	EDVRecycledObject	*obj1 = EDV_RECYCLED_OBJECT(
		((const GtkCListRow *)ptr1)->data
	),
				*obj2 = EDV_RECYCLED_OBJECT(
		((const GtkCListRow *)ptr2)->data
	);
	if((obj1 == NULL) || (obj2 == NULL))
		return(0);

	if(obj1->size <= obj2->size)
		return(obj1->size < obj2->size);
	else
		return(-1);
}

/*
 *	Contents GtkCList column sort storage size callback.
 */
static gint edv_recycle_bin_column_sort_storage_size_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	EDVRecycledObject	*obj1 = EDV_RECYCLED_OBJECT(
		((const GtkCListRow *)ptr1)->data
	),
				*obj2 = EDV_RECYCLED_OBJECT(
		((const GtkCListRow *)ptr2)->data
	);
	if((obj1 == NULL) || (obj2 == NULL))
		return(0);

	if(obj1->storage_size <= obj2->storage_size)
		return(obj1->storage_size < obj2->storage_size);
	else
		return(-1);
}

/*
 *	Contents GtkCList column sort date nexus.
 *
 *	The ptr1 and ptr2 specifies the two GtkCListRow * rows
 *	that contain the object date data to compare.
 *
 *	The sort_code specifies which date value to compare.
 */
static gint edv_recycle_bin_column_sort_date_nexus(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2,
	const gint sort_code
)
{
	EDVRecycledObject	*obj1 = EDV_RECYCLED_OBJECT(
		((const GtkCListRow *)ptr1)->data
	),
				*obj2 = EDV_RECYCLED_OBJECT(
		((const GtkCListRow *)ptr2)->data
	);
	if((clist == NULL) || (obj1 == NULL) || (obj2 == NULL))
		return(0);

	/* Handle by the sort code */
	switch(sort_code)
	{
	  case 0:       /* Access time */
		if(obj1->access_time <= obj2->access_time)
			return(obj1->access_time < obj2->access_time);
		else
			return(-1);
		break;

	  case 1:       /* Modify time */
		if(obj1->modify_time <= obj2->modify_time)
			return(obj1->modify_time < obj2->modify_time);
		else
			return(-1);
		break;

	  case 2:       /* Change time */
		if(obj1->change_time <= obj2->change_time)
			return(obj1->change_time < obj2->change_time);
		else
			return(-1);
		break;

	  case 3:	/* Delete time */
		if(obj1->deleted_time <= obj2->deleted_time)
			return(obj1->deleted_time < obj2->deleted_time);
		else
			return(-1);
		break;
	}

	return(0);
}

static gint edv_recycle_bin_column_sort_date_accessed_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	return(edv_recycle_bin_column_sort_date_nexus(
		clist, ptr1, ptr2, 0
	));
}

static gint edv_recycle_bin_column_sort_date_modified_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	return(edv_recycle_bin_column_sort_date_nexus(
		clist, ptr1, ptr2, 1
	));
}

static gint edv_recycle_bin_column_sort_date_changed_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	return(edv_recycle_bin_column_sort_date_nexus(
		clist, ptr1, ptr2, 2
	));
}

static gint edv_recycle_bin_column_sort_date_deleted_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	return(edv_recycle_bin_column_sort_date_nexus(
		clist, ptr1, ptr2, 3
	));
}

/*
 *	Contents GtkCList column sort capacity used callback.
 */
static gint edv_recycle_bin_column_sort_capacity_used_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	EDVRecycledObject	*obj1 = EDV_RECYCLED_OBJECT(
		((const GtkCListRow *)ptr1)->data
	),
				*obj2 = EDV_RECYCLED_OBJECT(
		((const GtkCListRow *)ptr2)->data
	);
	if((obj1 == NULL) || (obj2 == NULL))
		return(0);

	if(obj1->size <= obj2->size)
		return(obj1->size < obj2->size);
	else
		return(-1);
}

/*
 *	Contents GtkCList "resize_column" signal callback.
 */
void edv_recycle_bin_resize_column_cb(
	GtkCList *clist, gint column, gint width, gpointer data
)
{
	CfgList *cfg_list;
	EDVCore *core;
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((clist == NULL) || (recbin == NULL))
		return;

	if((EDV_RECYCLE_BIN_IS_PROCESSING(recbin)) || (recbin->freeze_count > 0))
		return;

	core = recbin->core;
	cfg_list = core->cfg_list;

	/* Check which clist this signal is for */
	if(GTK_WIDGET(clist) == recbin->contents_clist)
	{
		/* Get the column type that the column is displaying */
		const EDVRecycleBinColumnType column_type = edv_recycle_bin_list_get_column_type_by_index(
			recbin,
			column
		);

		/* Get column widths intlist */
		CfgIntList *column_width_intlist = EDV_GET_INT_LIST(
			EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN_WIDTH
		);
		if(column_width_intlist != NULL)
		{
			GList *glist = g_list_nth(
				column_width_intlist->list,
				(guint)column_type
			);
			if(glist != NULL)
			{
				glist->data = (gpointer)width;
			}
			else
			{
				/* The column type is not in the column widths
				 * list so create a new column widths list from
				 * the current column widths list and include
				 * the missing column type
				 */
				const gint m = EDV_RECYCLE_BIN_CONTENTS_CLIST_COLUMNS_MAX;
				gint i;
				GList *glist_new = NULL;

				glist = column_width_intlist->list;

				for(i = 0; i < m; i++)
				{
					if(glist != NULL)
					{
						glist_new = g_list_append(
							glist_new,
							glist->data
						);
						glist = g_list_next(glist);
					}
					else
						glist_new = g_list_append(
							glist_new,
							NULL
						);
				}

				g_list_free(column_width_intlist->list);
				column_width_intlist->list = glist_new;

				glist = g_list_nth(
					glist_new,
					(guint)column_type
				);
				if(glist != NULL)
					glist->data = (gpointer)width;
			}
		}
	}
}

/*
 *	Contents GtkCList "click_column" signal callback.
 */
void edv_recycle_bin_click_column_cb(
	GtkCList *clist, gint column, gpointer data
)
{
	GtkWidget	*widget,
			*toplevel;
	EDVCore *core;
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((clist == NULL) || (recbin == NULL))
		return;

	if((EDV_RECYCLE_BIN_IS_PROCESSING(recbin)) || (recbin->freeze_count > 0))
		return;

	widget = GTK_WIDGET(clist);
	toplevel = recbin->toplevel;
	core = recbin->core;

	/* Check which widget this signal is for
	 *
	 * Contents GtkCList
	 */
	if(GTK_WIDGET(clist) == recbin->contents_clist)
	{
		CfgIntList *column_types_intlist;
		GtkCListCompareFunc cmp_func = NULL;
		GtkCListCompareFunc cmp_func_str =
			 (GtkCListCompareFunc)EDVCListColumnSortStringCB;
#if 0
		GtkCListCompareFunc cmp_func_num =
			(GtkCListCompareFunc)edv_clist_column_sort_number_cb;
#endif

		edv_recycle_bin_set_busy(recbin, TRUE);
		GUIBlockInput(toplevel, TRUE);
		recbin->flags |= EDV_RECYCLE_BIN_PROCESSING;

		/* Get the column types mapping list */
		column_types_intlist = CFGItemListGetValueIntList(
			core->cfg_list, EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN
		);
		if(column_types_intlist != NULL)
		{
			EDVRecycleBinColumnType column_type = (EDVRecycleBinColumnType)g_list_nth_data(
				column_types_intlist->list,
				column
			);
			switch(column_type)
			{
			  case EDV_RECYCLE_BIN_COLUMN_TYPE_INDEX:
				cmp_func = edv_recycle_bin_column_sort_index_cb;
				break;
			  case EDV_RECYCLE_BIN_COLUMN_TYPE_NAME:
				cmp_func = cmp_func_str;
				break;
			  case EDV_RECYCLE_BIN_COLUMN_TYPE_SIZE:
				cmp_func = edv_recycle_bin_column_sort_size_cb;
				break;
			  case EDV_RECYCLE_BIN_COLUMN_TYPE_STORAGE_SIZE:
				cmp_func = edv_recycle_bin_column_sort_storage_size_cb;
				break;
			  case EDV_RECYCLE_BIN_COLUMN_TYPE_TYPE:
				cmp_func = cmp_func_str;
				break;
			  case EDV_RECYCLE_BIN_COLUMN_TYPE_PERMISSIONS:
				cmp_func = cmp_func_str;
				break;
			  case EDV_RECYCLE_BIN_COLUMN_TYPE_OWNER:
				cmp_func = cmp_func_str;
				break;
			  case EDV_RECYCLE_BIN_COLUMN_TYPE_GROUP:
				cmp_func = cmp_func_str;
				break;
			  case EDV_RECYCLE_BIN_COLUMN_TYPE_DATE_ACCESS:
				cmp_func = edv_recycle_bin_column_sort_date_accessed_cb;
				break;
			  case EDV_RECYCLE_BIN_COLUMN_TYPE_DATE_MODIFIED:
				cmp_func = edv_recycle_bin_column_sort_date_modified_cb;
				break;
			  case EDV_RECYCLE_BIN_COLUMN_TYPE_DATE_CHANGED:
				cmp_func = edv_recycle_bin_column_sort_date_changed_cb;
				break;
			  case EDV_RECYCLE_BIN_COLUMN_TYPE_DATE_DELETED:
				cmp_func = edv_recycle_bin_column_sort_date_deleted_cb;
				break;
			  case EDV_RECYCLE_BIN_COLUMN_TYPE_LINKED_TO:
				cmp_func = cmp_func_str;
				break;
			  case EDV_RECYCLE_BIN_COLUMN_TYPE_ORIGINAL_LOCATION:
				cmp_func = cmp_func_str;
				break;
			  case EDV_RECYCLE_BIN_COLUMN_TYPE_CAPACITY_USED:
				cmp_func = edv_recycle_bin_column_sort_capacity_used_cb;
				break;
			}
		}


		gtk_clist_freeze(clist);

		/* Set or invert the sort column on the GtkCList based
		 * on if the clicked on column is already the current
		 * sort column
		 */
		if(column != clist->sort_column)
			gtk_clist_set_sort_column(
				clist,
				column
			);
		else
			gtk_clist_set_sort_type(
				clist,
				(clist->sort_type == GTK_SORT_ASCENDING) ?
					GTK_SORT_DESCENDING : GTK_SORT_ASCENDING
			);
		if(cmp_func != NULL)
			gtk_clist_set_compare_func(clist, cmp_func);

		/* Sort the rows, this will call the GtkCList column
		 * sort callbacks
		 */
		gtk_clist_sort(clist);

		gtk_clist_thaw(clist);

		recbin->flags &= ~EDV_RECYCLE_BIN_PROCESSING;
		GUIBlockInput(toplevel, FALSE);
		edv_recycle_bin_set_busy(recbin, FALSE);

		edv_recycle_bin_update_display(recbin);
	}
}

/*
 *	Recycle Bin GtkCList "select_row" signal callback.
 */
void edv_recycle_bin_select_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((clist == NULL) || (recbin == NULL))
		return;

	if(EDV_RECYCLE_BIN_IS_PROCESSING(recbin))
		return;

	/* Check which clist this signal is for */
	if(GTK_WIDGET(clist) == recbin->contents_clist)
	{
		/* Get the total number of objects selected */
		const gint nselected = g_list_length(clist->selection);

		/* Get the object */
		EDVRecycledObject *obj = EDV_RECYCLED_OBJECT(
			gtk_clist_get_row_data(clist, row)
		);

		/* Update the selected row */
		recbin->contents_clist_selected_row = row;

		/* Set the DND drag icon */
		GUIDNDSetDragIconFromCListSelection(clist);

		if(obj != NULL)
		{
			/* Update status bar message */
			if(!STRISEMPTY(obj->name))
			{
				gchar *s, *size_str = NULL;
				const gchar *type_str = NULL;

				/* Get object type string and size string */
				switch(obj->type)
				{
				  case EDV_OBJECT_TYPE_UNKNOWN:
					type_str = "Object";
					size_str = NULL;
					break;
					break;
				  case EDV_OBJECT_TYPE_FILE:
					type_str = "File";
					size_str = g_strdup_printf(
						" (%s %s)",
						edv_str_size_delim(obj->size),
						(obj->size == 1l) ? "byte" : "bytes"
					);
					break;
				  case EDV_OBJECT_TYPE_DIRECTORY:
					type_str = "Directory";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_LINK:
					type_str = "Link";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_DEVICE_BLOCK:
					type_str = "Block device";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
					type_str = "Character device";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_FIFO:
					type_str = "FIFO pipe";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_SOCKET:
					type_str = "Socket";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_ERROR:
					type_str = "Error";
					size_str = NULL;
					break;
				}

				/* Set the status bar message */
				if(nselected > 1)
				{
					gulong total_size = 0l;
					EDVRecycledObject *obj;
					GList *glist = clist->selection;
					while(glist != NULL)
					{
						obj = EDV_RECYCLED_OBJECT(gtk_clist_get_row_data(
							clist, (gint)glist->data
						));
						if(obj != NULL)
							total_size += obj->size;
						glist = g_list_next(glist);
					}
					g_free(size_str);
					size_str = g_strdup_printf(
						"%s %s",
						edv_str_size_delim(total_size),
						(total_size == 1l) ? "byte" : "bytes"
					);
					s = g_strdup_printf(
						"%s objects selected (totaling %s)",
						edv_str_size_delim(nselected),
						size_str
					);
				}
				else if(!strcmp((const char *)obj->name, ".."))
					s = g_strdup_printf(
						"Parent directory selected"
					);
				else
					s = g_strdup_printf(
						"%s \"%s\" selected%s",
						type_str, obj->name,
						(size_str != NULL) ? size_str : ""
					);
				edv_status_bar_message(
					recbin->status_bar, s, FALSE
				);
				g_free(s);
				g_free(size_str);
			}
			else
			{
				edv_status_bar_message(
					recbin->status_bar,
					"Object with no name selected",
					FALSE
				);
			}
		}

		/* Check if selected row is fully visible, if not then adjust
		 * scroll position to try and make it visible
		 */
		if(gtk_clist_row_is_visible(clist, row) !=
			GTK_VISIBILITY_FULL
		)
			gtk_clist_moveto(
				clist,
				row, -1,	/* Row, column */
				0.5f, 0.0f	/* Row, column */
			);

/*		edv_recycle_bin_set_title(recbin); */
		edv_recycle_bin_update_display(recbin);

		/* Double click? */
		if(event != NULL)
		{
			if(event->type == GDK_2BUTTON_PRESS)
			{
				/* Recover */
				edv_recycle_bin_op_recover(recbin);
			}
		}
	}
}

/*
 *	Recycle Bin GtkCList "unselect_row" signal callback.
 */
void edv_recycle_bin_unselect_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((clist == NULL) || (recbin == NULL))
		return;

	if(EDV_RECYCLE_BIN_IS_PROCESSING(recbin))
		return;

	/* Check which clist this signal is for */
	if(GTK_WIDGET(clist) == recbin->contents_clist)
	{
		gchar *s;

		/* Get total number of objects selected */
		const gint nselected = g_list_length(clist->selection);

		/* Update the status bar message */
		if(nselected > 0)
		{
			gchar *size_str;
			gulong total_size = 0l;
			EDVRecycledObject *obj;
			GList *glist = clist->selection;
			while(glist != NULL)
			{
				obj = EDV_RECYCLED_OBJECT(gtk_clist_get_row_data(
					clist, (gint)glist->data
				));
				if(obj != NULL)
					total_size += obj->size;
				glist = g_list_next(glist);
			}
			size_str = g_strdup_printf(
				"%s %s",
				edv_str_size_delim(total_size),
				(total_size == 1l) ? "byte" : "bytes"
			);
			s = g_strdup_printf(
				"%s %s selected (totaling %s)",
				edv_str_size_delim(nselected),
				(nselected == 1) ? "object" : "objects",
				size_str
			);
			g_free(size_str);
		}
		else
			s = STRDUP("No objects selected");
		edv_status_bar_message(
			recbin->status_bar,
			s,
			FALSE
		);
		g_free(s);

		edv_recycle_bin_update_display(recbin);
	}
}


/*
 *	Menu item activate callback.
 *
 *	The given client data must be a EDVRecycleBinOp *.
 */
void edv_recycle_bin_menu_item_cb(GtkWidget *widget, gpointer data)
{
	edv_recycle_bin_op_cb(NULL, -1, data);
}

/*
 *	Menu item "enter_notify_event" signal callback.
 *
 *	The given client data must be a EDVRecycleBinOp *.
 */
gint edv_recycle_bin_menu_item_enter_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	edv_recycle_bin_op_enter_cb(NULL, -1, data);
	return(TRUE);
}

/*
 *	Menu item "leave_notify_event" signal callback.
 *
 *	The given client data must be a EDVRecycleBinOp *.
 */
gint edv_recycle_bin_menu_item_leavr_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	edv_recycle_bin_op_leave_cb(NULL, -1, data);
	return(TRUE);
}


/*
 *	Find bar start find callback.
 */
void edv_recycle_bin_find_bar_start_cb(
	GtkWidget *bar,
	gpointer data
)
{
	GtkCList *clist;
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if(recbin == NULL)
		return;

	clist = GTK_CLIST(recbin->contents_clist);

	edv_recycle_bin_set_busy(recbin, TRUE);

	gtk_clist_freeze(clist);
	gtk_clist_unselect_all(clist);
	gtk_clist_thaw(clist);

	edv_recycle_bin_update_display(recbin);

	recbin->freeze_count++;
}

/*
 *	Find bar end find callback.
 */
void edv_recycle_bin_find_bar_end_cb(
	GtkWidget *bar,
	const gint nmatches,
	gpointer data
)
{
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if(recbin == NULL)
		return;

	recbin->freeze_count--;
	edv_recycle_bin_set_busy(recbin, FALSE);
}


/*
 *	Find bar match callback.
 */
void edv_recycle_bin_find_bar_match_cb(
	GtkWidget *bar,
	const gchar *path,
	GList *properties_list,
	const gint line_index,
	const gchar *excerpt,
	gpointer data
)
{
	gint row;
	guint index;
	GtkCList *clist;
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((path == NULL) || (recbin == NULL))
		return;

	index = (guint)edv_properties_list_get_ul(
		properties_list,
		EDV_PROP_NAME_INDEX
	);

	clist = GTK_CLIST(recbin->contents_clist);

	row = edv_recycle_bin_list_find_row_by_index(recbin, index);
	if((row >= 0) && (row < clist->rows))
		gtk_clist_select_row(
			clist,
			row,
			0
		);
}


/*
 *	Status message callback.
 */
void edv_recycle_bin_find_bar_status_message_cb(
	GtkWidget *widget,
	const gchar *message, 
	gpointer data
)
{
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if(recbin == NULL)
		return;

	edv_status_bar_message(recbin->status_bar, message, FALSE);
}

/*
 *	Status progress callback.
 */
void edv_recycle_bin_find_bar_status_progress_cb(
	GtkWidget *widget,
	const gfloat progress,
	gpointer data
)
{
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if(recbin == NULL)
		return;

	edv_status_bar_progress(recbin->status_bar, progress, FALSE);
}


/*
 *	Window created callback.
 */
void edv_recycle_bin_window_created_cb(
	edv_recbin_struct *recbin,
	const EDVWindowType win_type,
	gpointer win
)
{
	if(recbin == NULL)
		return;

	if(recbin == EDV_RECBIN(win))
		return;

	edv_recycle_bin_update_display(recbin);
}

/*
 *	Window deleted callback.
 */
void edv_recycle_bin_window_deleted_cb(
	edv_recbin_struct *recbin,
	const EDVWindowType win_type,
	gpointer win
)
{
	if(recbin == NULL)
		return;

	if(recbin == EDV_RECBIN(win))
		return;

	edv_recycle_bin_update_display(recbin);
}


/*
 *	Reconfigured callback.
 */
void edv_recycle_bin_reconfigured_cb(edv_recbin_struct *recbin)
{
	GtkRcStyle	*standard_rcstyle,
			*lists_rcstyle;
	GtkWidget *w;
	CfgItem *cfg_list;
	EDVCore *core;

	if(recbin == NULL)
		return;

	if(EDV_RECYCLE_BIN_IS_PROCESSING(recbin))
		return;

	core = recbin->core;
	cfg_list = core->cfg_list;
	standard_rcstyle = core->standard_rcstyle;
	lists_rcstyle = core->lists_rcstyle;

	/* Reset last state markers so that resources get explicitly
	 * checked due to reconfiguring
	 */
	recbin->last_nrecycle_bin_items = -1;
	recbin->last_write_protect_state = -1;

	/* Update the title */
	edv_recycle_bin_set_title(recbin);

	/* Update the Accelkey Labels */
	edv_recycle_bin_accelkeys_regenerate(recbin);

	/* Regenerate the Tool Bar */
	edv_recycle_bin_tool_bar_regenerate(recbin);

	/* Show/hide the Tool Bar */
	w = recbin->tool_bar_handle;
	if(w != NULL)
	{
		if(EDV_GET_B(EDV_CFG_PARM_RECBIN_SHOW_TOOL_BAR))
		{
			gtk_widget_show(w);
			recbin->flags |= EDV_RECYCLE_BIN_TOOL_BAR_MAPPED;
		}
		else
		{
			gtk_widget_hide(w);
			recbin->flags &= ~EDV_RECYCLE_BIN_TOOL_BAR_MAPPED;
		}
	}

	/* Show/hide the Find Bar */
	w = recbin->find_bar_handle;
	if(w != NULL)
	{
		if(EDV_GET_B(EDV_CFG_PARM_RECBIN_SHOW_FIND_BAR))
		{
			gtk_widget_show(w);
			recbin->flags |= EDV_RECYCLE_BIN_FIND_BAR_MAPPED;
		}
		else
		{
			gtk_widget_hide(w);
			recbin->flags &= ~EDV_RECYCLE_BIN_FIND_BAR_MAPPED;
		}
	}

	/* Show/hide the Status Bar */
	w = recbin->status_bar;
	if(w != NULL)
	{
		if(EDV_GET_B(EDV_CFG_PARM_RECBIN_SHOW_STATUS_BAR))
		{
			gtk_widget_show(w);
			recbin->flags |= EDV_RECYCLE_BIN_STATUS_BAR_MAPPED;
		}
		else
		{
			gtk_widget_hide(w);
			recbin->flags &= ~EDV_RECYCLE_BIN_STATUS_BAR_MAPPED;
		}
	}


	/* Update the RC styles */
	w = recbin->toplevel;
	if((w != NULL) && (standard_rcstyle != NULL))
		gtk_widget_modify_style_recursive(w, standard_rcstyle);
	w = recbin->contents_clist;
	if((w != NULL) && (lists_rcstyle != NULL))
		gtk_widget_modify_style_recursive(w, lists_rcstyle);
	w = recbin->contents_clist_menu;
	if((w != NULL) && (standard_rcstyle != NULL))
		gtk_widget_modify_style_recursive(w, standard_rcstyle);

	/* Realize the listings */
	edv_recycle_bin_list_realize_listing(recbin);

	edv_recycle_bin_update_display(recbin);

	/* Notify GTK about possible size changes (such as the tool
	 * bar resizing)
	 */
	gtk_widget_queue_resize(recbin->toplevel);
}


/*
 *	Master Write Protect changed callback.
 */
void edv_recycle_bin_master_write_protect_changed_cb(
	edv_recbin_struct *recbin,
	const gboolean state
)
{
	if(recbin == NULL)
		return;

	if(EDV_RECYCLE_BIN_IS_PROCESSING(recbin))
		return;

	edv_recycle_bin_update_display(recbin);
}

/*
 *	Delete Method changed callback.
 */
void edv_recycle_bin_delete_method_changed_cb(
	edv_recbin_struct *recbin,
	const EDVDeleteMethod delete_method
)
{
	if(recbin == NULL)
		return;

	if(EDV_RECYCLE_BIN_IS_PROCESSING(recbin))
		return;

	edv_recycle_bin_update_display(recbin);
}

/*
 *	Recycled object added callback.
 */
void edv_recycle_bin_recycled_object_added_cb(
	edv_recbin_struct *recbin, const guint index
)
{
	EDVCore *core;

	if(recbin == NULL)
		return;

	if(EDV_RECYCLE_BIN_IS_PROCESSING(recbin))
		return;

	core = recbin->core;

	/* Contents List */
	edv_recycle_bin_list_recycled_object_added_cb(recbin, index);

	/* Update menus for the purpose of updating the Recycle Bin
	 * toplevel window's WM icon. Do this only if there is a change
	 * in the number of recycled objects from 0.
	 */
	if(core->last_nrecycle_bin_items != recbin->last_nrecycle_bin_items)
		edv_recycle_bin_update_display(recbin);
}

/*
 *	Recycled object modified callback.
 */
void edv_recycle_bin_recycled_object_modified_cb(
	edv_recbin_struct *recbin, const guint index
)
{
	EDVCore *core;

	if(recbin == NULL)
		return;

	if(EDV_RECYCLE_BIN_IS_PROCESSING(recbin))
		return;

	core = recbin->core;

	/* Contents List */
	edv_recycle_bin_list_recycled_object_modified_cb(recbin, index);

	/* Update menus for the purpose of updating the Recycle Bin
	 * toplevel window's WM icon. Do this only if there is a
	 * change in the number of recycled objects from 0
	 */
	if(core->last_nrecycle_bin_items != recbin->last_nrecycle_bin_items)
		edv_recycle_bin_update_display(recbin);
}

/*
 *	Recycled object removed callback.
 */
void edv_recycle_bin_recycled_object_removed_cb(
	edv_recbin_struct *recbin, const guint index
)
{
	EDVCore *core;

	if(recbin == NULL)
		return;

	if(EDV_RECYCLE_BIN_IS_PROCESSING(recbin))
		return;

	core = recbin->core;

	/* Contents List */
	edv_recycle_bin_list_recycled_object_removed_cb(recbin, index);

	/* Update menus for the purpose of updating the Recycle Bin
	 * toplevel window's WM icon. Do this only if there is a change
	 * in the number of recycled objects from 0
	 */
	if(core->last_nrecycle_bin_items != recbin->last_nrecycle_bin_items)
		edv_recycle_bin_update_display(recbin);
}


/*
 *	MIME Type added callback.
 */
void edv_recycle_bin_mime_type_added_cb(
	edv_recbin_struct *recbin,
	const gint mt_num, EDVMIMEType *mt
)
{
	/* Treat a MIME Type added the same as it would be for a MIME
	 * Type modified, forward signal to the MIME Type modified
	 * callback
	 */
	edv_recycle_bin_mime_type_modified_cb(recbin, mt_num, mt);
}

/*
 *	MIME Type modified callback.
 */
void edv_recycle_bin_mime_type_modified_cb(
	edv_recbin_struct *recbin,
	const gint mt_num, EDVMIMEType *mt
)
{
	if(recbin == NULL)
		return;

	if(EDV_RECYCLE_BIN_IS_PROCESSING(recbin))
		return;

	/* Realize listings */
	edv_recycle_bin_list_realize_listing(recbin);

/*	edv_recycle_bin_update_display(recbin); */
}

/*
 *	MIME Type removed callback.
 */
void edv_recycle_bin_mime_type_removed_cb(
	edv_recbin_struct *recbin,
	const gint mt_num
)
{
	if(recbin == NULL)
		return;

	if(EDV_RECYCLE_BIN_IS_PROCESSING(recbin))
		return;

	/* Realize listings */
	edv_recycle_bin_list_realize_listing(recbin);

/*	edv_recycle_bin_update_display(recbin); */
}
