#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "cfg.h"

#include "guiutils.h"
#include "cfg_gtk.h"
#include "tool_bar.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_recycled_obj.h"
#include "libendeavour2-base/edv_recycle_bin_index.h"
#include "edv_utils_gtk.h"
#include "edv_list_cb.h"
#include "edv_find_bar.h"
#include "edv_status_bar.h"
#include "recycle_bin.h"
#include "recycle_bin_cb.h"
#include "recycle_bin_op.h"
#include "recycle_bin_list.h"
#include "recycle_bin_dnd.h"
#include "edv_cb.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_hsep_20x20.xpm"
#include "images/icon_save_20x20.xpm"
#include "images/icon_trace_20x20.xpm"
#include "images/icon_run_20x20.xpm"
#include "images/icon_terminal2_20x20.xpm"
#include "images/icon_close_20x20.xpm"
#include "images/icon_exit_20x20.xpm"
#include "images/icon_recover_20x20.xpm"
#include "images/icon_purge_20x20.xpm"
#include "images/icon_purge_all_20x20.xpm"
#include "images/icon_rename_20x20.xpm"
#include "images/icon_chmod_20x20.xpm"
#include "images/icon_owned_20x20.xpm"
#include "images/icon_time_stamp_20x20.xpm"
#include "images/icon_select_all_20x20.xpm"
#include "images/icon_unselect_all_20x20.xpm"
#include "images/icon_invert_selection_20x20.xpm"
#include "images/icon_properties_20x20.xpm"
#include "images/icon_reload_20x20.xpm"
#include "images/icon_mimetypes_20x20.xpm"
#include "images/icon_search_20x20.xpm"
#include "images/icon_tool_bar_20x20.xpm"
#include "images/icon_status_bar_20x20.xpm"
#include "images/icon_wildcards_20x20.xpm"
#include "images/icon_resize_column_20x20.xpm"
#include "images/icon_options_20x20.xpm"
#include "images/icon_customize_20x20.xpm"
#include "images/icon_write_protect_off_20x20.xpm"
#include "images/icon_write_protect_on_20x20.xpm"
#include "images/icon_help_20x20.xpm"
#include "images/icon_about_20x20.xpm"

#include "images/icon_endeavour_file_browser_20x20.xpm"
#include "images/icon_endeavour_image_browser_20x20.xpm"
#include "images/icon_endeavour_archiver_20x20.xpm"


/* Utilities */
EDVRecycleBinOp *edv_recycle_bin_op_match_by_id(
	edv_recbin_struct *recbin,
	const EDVRecycleBinOpID id
);

/* Title */
void edv_recycle_bin_set_title(edv_recbin_struct *recbin);

/* Selected Objects */
GList *edv_recycle_bin_get_selected_objects(
	edv_recbin_struct *recbin,
	const gboolean copy
);

/* Regenerate Style, Menus, and Tool Bar */
void edv_recycle_bin_accelkeys_regenerate(edv_recbin_struct *recbin);
void edv_recycle_bin_tool_bar_regenerate(edv_recbin_struct *recbin);
void edv_recycle_bin_list_menu_regenerate(edv_recbin_struct *recbin);

/* Sync Configuration */
void edv_recycle_bin_sync_configuration(edv_recbin_struct *recbin);

/* EDVRecycleBinOp */
static EDVRecycleBinOp *edv_recycle_bin_op_new(void);
static void edv_recycle_bin_op_delete(EDVRecycleBinOp *opid);
GList *edv_recycle_bin_ops_list_new(edv_recbin_struct *recbin);
void edv_recycle_bin_ops_list_delete(GList *op_ids_list);

/* Accel Keys */
CfgAccelkeyList *edv_recycle_bin_accelkeys_new(void);

/* EDVRecycleBin */
static void EDVRecBinCreateMenuBar(
	edv_recbin_struct *recbin,
	GtkWidget *parent
);
edv_recbin_struct *edv_recycle_bin_new(
	EDVCore *core,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry
);
void edv_recycle_bin_sync_data(edv_recbin_struct *recbin);
void edv_recycle_bin_update_display(edv_recbin_struct *recbin);
void edv_recycle_bin_set_busy(edv_recbin_struct *recbin, const gboolean busy);
gboolean edv_recycle_bin_is_mapped(edv_recbin_struct *recbin);
void edv_recycle_bin_map(edv_recbin_struct *recbin);
void edv_recycle_bin_unmap(edv_recbin_struct *recbin);
void edv_recycle_bin_delete(edv_recbin_struct *recbin);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Returns the Recycle Bin Operation ID that matches the given
 *	operation.
 */
EDVRecycleBinOp *edv_recycle_bin_op_match_by_id(
	edv_recbin_struct *recbin,
	const EDVRecycleBinOpID id
)
{
	GList *glist;
	EDVRecycleBinOp *op;

	if(recbin == NULL)
		return(NULL);

	for(glist = recbin->op_ids_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		op = EDV_RECYCLE_BIN_OP(glist->data);
		if(op == NULL)
			continue;

		if(op->id == id)
			return(op);
	}

	return(NULL);
}


/*
 *	Sets the Recycle Bin title.
 */
void edv_recycle_bin_set_title(edv_recbin_struct *recbin)
{
	gboolean	show_program_name,
			show_user_name,
			show_location;
#if defined(PROG_LANGUAGE_FRENCH)
	const gchar *prog_name = "Corbeille";
#else
	const gchar *prog_name = "Recycle Bin";
#endif
	const gchar	*user_name,
					*filter;
	gchar		*title,
					*path;
	GtkWidget *w;
	CfgList *cfg_list;
	EDVCore *core;

	if(recbin == NULL)
		return;

	w = recbin->toplevel;
	core = recbin->core;
	cfg_list = core->cfg_list;

	/* Get the values to show in the title */
	if(!STRISEMPTY(core->title))
		prog_name = core->title;

	user_name = core->effective_user_id_host_str;
	filter = recbin->contents_list_filter;

	path = edv_recycle_bin_index_get_recbin_directory_path(
		EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX)
	);

	show_program_name = EDV_GET_B(EDV_CFG_PARM_RECBIN_TITLE_WITH_PROGRAM);
	if(show_program_name && STRISEMPTY(prog_name))
		show_program_name = FALSE;

	show_user_name = EDV_GET_B(EDV_CFG_PARM_RECBIN_TITLE_WITH_USER);
	if(show_user_name && STRISEMPTY(user_name))
		show_user_name = FALSE;

	show_location = EDV_GET_B(EDV_CFG_PARM_RECBIN_TITLE_WITH_LOCATION);
	if(show_location && STRISEMPTY(path))
		show_location = FALSE;

	/* Format the new title string */
	title = g_strdup("");
	if(EDV_GET_B(EDV_CFG_PARM_RECBIN_TITLE_REVERSE))
	{
		if(show_location)
		{
			gchar *s;
			if(STRISEMPTY(filter))
				s = g_strconcat(
					title,
					path,
					NULL
				);
			else
				s = g_strconcat(
					title,
					path,
					" - ",
					filter,
					NULL
				);
			g_free(title);
			title = s;
		}
		if(show_user_name)
		{
			gchar *s = g_strconcat(
				title,
				(show_location) ? " - " : "",
				user_name,
				NULL
			);
			g_free(title);
			title = s;
		}
		if(show_program_name)
		{
			gchar *s = g_strconcat(
				title,
				(show_user_name || show_location) ? " - " : "",
				prog_name,
				NULL
			);
			g_free(title);
			title = s;
		}
	}
	else
	{
		if(show_program_name)
		{
			gchar *s = g_strconcat(
				title,
				prog_name,
				(show_user_name || show_location) ? ": " : "",
				NULL
			);
			g_free(title);
			title = s;
		}
		if(show_user_name)
		{
			gchar *s = g_strconcat(
				title,
				user_name,
				(show_location) ? "" : "",
				NULL
			);
			g_free(title);
			title = s;
		}
		if(show_location)
		{
			gchar *s;
			if(STRISEMPTY(filter))
				s = g_strconcat(
					title,
					path,
					NULL
				);
			else
				s = g_strconcat(
					title,
					path,
					": ",
					filter,
					NULL
				);
			g_free(title);
			title = s;
		}
	}

	g_free(path);

	/* Set the new title */
	gtk_window_set_title(GTK_WINDOW(w), title);

	g_free(title);
}


/*
 *	Gets a list of the selected recycled objects.
 *
 *	If copy is TRUE then each object in the returned list will
 *	be a copy of the original object.
 *
 *	Returns a GList of EDVRecycledObject * objects. If
 *	copy was TRUE then the calling function must delete each
 *	object in the list and the list itself, otherwise the
 *	calling function need only delete the returned list.
 */
GList *edv_recycle_bin_get_selected_objects(
	edv_recbin_struct *recbin, const gboolean copy
)
{
	gint row;
	GList *glist, *objs_list;
	GtkCList *clist;
	EDVRecycledObject *obj;

	if(recbin == NULL)
		return(NULL);

	clist = GTK_CLIST(recbin->contents_clist);

	objs_list = NULL;
	for(glist = clist->selection;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		row = (gint)glist->data;
		obj = EDV_RECYCLED_OBJECT(gtk_clist_get_row_data(clist, row));
		if(obj == NULL)
			continue;

		objs_list = g_list_append(
			objs_list,
			(copy) ? edv_recycled_object_copy(obj) : obj
		);
	}

	return(objs_list);
}


/*
 *	Resets the Recycle Bin's accelkeys displayed on the widgets.
 */
void edv_recycle_bin_accelkeys_regenerate(edv_recbin_struct *recbin)
{
	GtkAccelGroup *accelgrp = NULL; 
	CfgAccelkeyList *ak_list;
	CfgList *cfg_list;
	EDVCore *core;

	if(recbin == NULL)
		return;
			   
	core = recbin->core;
	cfg_list = core->cfg_list;

	ak_list = EDV_GET_ACCELKEY_LIST(EDV_CFG_PARM_RECBIN_ACCELERATOR_KEYS);
	if(ak_list == NULL)                                                    
		return;

#define UPDATE_ACCELKEY(_w_,_opid_)	{	\
 if(((_w_) != NULL) && ((_opid_) > 0)) {	\
  GList *glist;					\
  CfgAccelkey *ak;				\
						\
  for(glist = ak_list->list;			\
      glist != NULL;				\
      glist = g_list_next(glist)		\
  ) {						\
   ak = CFG_ACCELKEY(glist->data);		\
   if(ak == NULL)				\
    continue;					\
						\
   if(ak->op_id == (gint)(_opid_)) {		\
    GUIMenuItemSetAccelKey(			\
     (_w_), accelgrp, ak->key, ak->modifiers	\
    );						\
    break;					\
   }						\
  }						\
 }						\
}

	UPDATE_ACCELKEY(
		recbin->recover_mi,
		EDV_RECYCLE_BIN_OP_RECOVER
	);
	UPDATE_ACCELKEY(
		recbin->purge_mi,
		EDV_RECYCLE_BIN_OP_PURGE
	);
	UPDATE_ACCELKEY(
		recbin->purge_all_mi,
		EDV_RECYCLE_BIN_OP_PURGE_ALL
	);
	UPDATE_ACCELKEY(
		recbin->sync_recycle_bin_mi,
		EDV_RECYCLE_BIN_OP_SYNC_RECYCLE_BIN
	);
	UPDATE_ACCELKEY(
		recbin->sync_disks_mi,
		EDV_RECYCLE_BIN_OP_SYNC_DISKS
	);
	UPDATE_ACCELKEY(
		recbin->history_mi,
		EDV_RECYCLE_BIN_OP_HISTORY
	);
	UPDATE_ACCELKEY(
		recbin->run_mi,
		EDV_RECYCLE_BIN_OP_RUN
	);
	UPDATE_ACCELKEY(
		recbin->close_mi,
		EDV_RECYCLE_BIN_OP_CLOSE
	);

	UPDATE_ACCELKEY(
		recbin->edit_recover_mi,
		EDV_RECYCLE_BIN_OP_RECOVER
	);
	UPDATE_ACCELKEY(
		recbin->edit_purge_mi,
		EDV_RECYCLE_BIN_OP_PURGE
	);
	UPDATE_ACCELKEY(
		recbin->edit_purge_all_mi,
		EDV_RECYCLE_BIN_OP_PURGE_ALL
	);
	UPDATE_ACCELKEY(
		recbin->rename_mi,
		EDV_RECYCLE_BIN_OP_RENAME
	);
	UPDATE_ACCELKEY(
		recbin->chmod_mi,
		EDV_RECYCLE_BIN_OP_CHMOD
	);
	UPDATE_ACCELKEY(
		recbin->chown_mi,
		EDV_RECYCLE_BIN_OP_CHOWN
	);
	UPDATE_ACCELKEY(
		recbin->chtime_mi,
		EDV_RECYCLE_BIN_OP_CHTIME
	);
	UPDATE_ACCELKEY(
		recbin->select_all_mi,
		EDV_RECYCLE_BIN_OP_SELECT_ALL
	);
	UPDATE_ACCELKEY(
		recbin->unselect_all_mi,
		EDV_RECYCLE_BIN_OP_UNSELECT_ALL
	);
	UPDATE_ACCELKEY(
		recbin->invert_selection_mi,
		EDV_RECYCLE_BIN_OP_INVERT_SELECTION
	);
	UPDATE_ACCELKEY(
		recbin->find_mi,
		EDV_RECYCLE_BIN_OP_FIND
	);
	UPDATE_ACCELKEY(
		recbin->properties_mi,
		EDV_RECYCLE_BIN_OP_PROPERTIES
	);

	UPDATE_ACCELKEY(
		recbin->refresh_mi,
		EDV_RECYCLE_BIN_OP_REFRESH
	);
	UPDATE_ACCELKEY(
		recbin->refresh_all_mi,
		EDV_RECYCLE_BIN_OP_REFRESH_ALL
	);
	UPDATE_ACCELKEY(
		recbin->show_tool_bar_micheck,
		EDV_RECYCLE_BIN_OP_SHOW_TOOL_BAR
	);
	UPDATE_ACCELKEY(
		recbin->show_find_bar_micheck,
		EDV_RECYCLE_BIN_OP_SHOW_FIND_BAR
	);
	UPDATE_ACCELKEY(
		recbin->show_status_bar_micheck,
		EDV_RECYCLE_BIN_OP_SHOW_STATUS_BAR
	);
	UPDATE_ACCELKEY(
		recbin->contents_list_filter_mi,
		EDV_RECYCLE_BIN_OP_CONTENTS_LIST_FILTER
	);
	UPDATE_ACCELKEY(
		recbin->contents_list_auto_resize_columns_micheck,
		EDV_RECYCLE_BIN_OP_CONTENTS_LIST_AUTO_RESIZE_COLUMNS
	);
	UPDATE_ACCELKEY(
		recbin->mime_types_mi,
		EDV_RECYCLE_BIN_OP_MIME_TYPES
	);

	UPDATE_ACCELKEY(
		recbin->write_protect_mi,
		EDV_RECYCLE_BIN_OP_WRITE_PROTECT
	);
	UPDATE_ACCELKEY(  
		recbin->delete_method_recycle_micheck,
		EDV_RECYCLE_BIN_OP_DELETE_METHOD_RECYCLE
	);
	UPDATE_ACCELKEY(
		recbin->delete_method_purge_micheck,
		EDV_RECYCLE_BIN_OP_DELETE_METHOD_PURGE
	);
	UPDATE_ACCELKEY(
		recbin->options_mi,
		EDV_RECYCLE_BIN_OP_OPTIONS
	);
	UPDATE_ACCELKEY(
		recbin->customize_mi,
		EDV_RECYCLE_BIN_OP_CUSTOMIZE
	);

	UPDATE_ACCELKEY(
		recbin->window_new_browser_mi,
		EDV_RECYCLE_BIN_OP_NEW_BROWSER
	);
	UPDATE_ACCELKEY(
		recbin->window_new_imbr_mi,
		EDV_RECYCLE_BIN_OP_NEW_IMBR
	);
	UPDATE_ACCELKEY(
		recbin->window_new_archiver_mi,
		EDV_RECYCLE_BIN_OP_NEW_ARCHIVER
	);

	UPDATE_ACCELKEY(
		recbin->help_contents_mi,
		EDV_RECYCLE_BIN_OP_HELP_CONTENTS
	);
	UPDATE_ACCELKEY(
		recbin->help_recycle_bin_mi,
		EDV_RECYCLE_BIN_OP_HELP_RECYCLE_BIN
	);
	UPDATE_ACCELKEY(
		recbin->help_keys_list_mi,
		EDV_RECYCLE_BIN_OP_HELP_KEYS_LIST
	);
	UPDATE_ACCELKEY(
		recbin->help_common_operations_mi,
		EDV_RECYCLE_BIN_OP_HELP_COMMON_OPERATIONS
	);
	UPDATE_ACCELKEY(
		recbin->help_about_mi,
		EDV_RECYCLE_BIN_OP_HELP_ABOUT
	);


	UPDATE_ACCELKEY(
		recbin->contents_clist_recover_mi,
		EDV_RECYCLE_BIN_OP_RECOVER
	);
	UPDATE_ACCELKEY(
		recbin->contents_clist_purge_mi,
		EDV_RECYCLE_BIN_OP_PURGE
	);
	UPDATE_ACCELKEY(
		recbin->contents_clist_purge_all_mi,
		EDV_RECYCLE_BIN_OP_PURGE_ALL
	);
	UPDATE_ACCELKEY(
		recbin->contents_clist_rename_mi,
		EDV_RECYCLE_BIN_OP_RENAME
	);
	UPDATE_ACCELKEY(
		recbin->contents_clist_chmod_mi,
		EDV_RECYCLE_BIN_OP_CHMOD
	);
	UPDATE_ACCELKEY(
		recbin->contents_clist_chown_mi,
		EDV_RECYCLE_BIN_OP_CHOWN
	);
	UPDATE_ACCELKEY(
		recbin->contents_clist_chtime_mi,
		EDV_RECYCLE_BIN_OP_CHTIME
	);
	UPDATE_ACCELKEY(
		recbin->contents_clist_properties_mi,
		EDV_RECYCLE_BIN_OP_PROPERTIES
	);

#undef UPDATE_ACCELKEY
}

/*
 *	Recreates the Recycle Bin's Tool Bar.
 */
void edv_recycle_bin_tool_bar_regenerate(edv_recbin_struct *recbin)
{
	const gint border_minor = 2;
	GList		*glist,
			*items_list;
	GtkWidget	*w,
			*parent;
	CfgIntList *intlist;
	CfgList *cfg_list;
	ToolBarButtonDecals tb_button_decals;
	GtkReliefStyle tb_relief;
	ToolBarItem *item;
	EDVRecycleBinOp *opid;
	EDVCore *core;

	if(recbin == NULL)
		return;

	parent = recbin->tool_bar_handle;
	core = recbin->core;
	cfg_list = core->cfg_list;

	/* Get the Tool Bar configuration */
	tb_button_decals = (ToolBarButtonDecals)EDV_GET_I(
		EDV_CFG_PARM_RECBIN_TOOL_BAR_BUTTON_DECALS
	);
	tb_relief = (GtkReliefStyle)EDV_GET_I(
		EDV_CFG_PARM_RECBIN_TOOL_BAR_RELIEF
	);
	intlist = EDV_GET_INT_LIST(EDV_CFG_PARM_RECBIN_TOOL_BAR);
	if(intlist == NULL)
		return;

	/* Delete the existing Tool Bar */
	GTK_WIDGET_DESTROY(recbin->tool_bar);
	recbin->tool_bar = NULL;

	/* Create the ToolBarItems list */
	items_list = NULL;
	for(glist = intlist->list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		/* Get the OPID */
		opid = edv_recycle_bin_op_match_by_id(
			recbin,
			(EDVRecycleBinOpID)glist->data
		);
		if(opid == NULL)
			continue;

		/* Exclude this OPID from Tool Bars? */
		if(opid->flags & EDV_OPID_EX_TOOL_BAR)
			continue;

		/* Create a new ToolBarItem from this OPID */
		item = ToolBarItemNew(
			opid->tb_item_type,
			opid->button_name,
			opid->button_icon_data,
			opid->tooltip,
			opid->id,
			edv_recycle_bin_op_cb, opid,
			edv_recycle_bin_op_enter_cb, opid,
			edv_recycle_bin_op_leave_cb, opid
		);
		if(item == NULL)
			continue;

		/* Append the new ToolBarItem to the list */
		items_list = g_list_append(
			items_list,
			item
		);
	}

	/* Create the Tool Bar */
	recbin->tool_bar = w = ToolBarNew(
		GTK_ORIENTATION_HORIZONTAL,
		tb_button_decals,
		tb_relief,
		GTK_POLICY_AUTOMATIC,
		items_list
	);
	gtk_widget_set_name(w, EDV_WIDGET_NAME_TOOL_BAR);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_minor);
	gtk_widget_show(w);

	/* Delete the Tool Bar items list */
	if(items_list != NULL)
	{
		g_list_foreach(
			items_list,
			(GFunc)ToolBarItemDelete,
			NULL
		);
		g_list_free(items_list);
	}
}

/*
 *	Recreates the Contents GtkCList's Right-Click GtkMenu.
 */
void edv_recycle_bin_list_menu_regenerate(edv_recbin_struct *recbin)
{
	edv_pixmap_data *icon_data = NULL;
	EDVRecycleBinOpID op;
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget *w, *menu;

	if(recbin == NULL)
		return;

	/* Recreate the menu */
	GTK_WIDGET_DESTROY(recbin->contents_clist_menu);
	recbin->contents_clist_menu = menu = GUIMenuCreate();

#define ADD_MENU_ITEM_LABEL	{		\
 EDVRecycleBinOp *opid =			\
  edv_recycle_bin_op_match_by_id(recbin, op);		\
 if(opid != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp,	\
   (icon_data != NULL) ?			\
    icon_data : opid->menu_icon_data,		\
   opid->menu_name,				\
   opid->accel_key, opid->accel_mods,		\
   edv_recycle_bin_menu_item_cb, opid			\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)edv_recycle_bin_menu_item_enter_cb, opid,	\
   (gpointer)edv_recycle_bin_menu_item_leavr_cb, opid	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}
#define ADD_MENU_SEP		{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL,						\
  NULL,						\
  0, 0,						\
  NULL, NULL					\
 );						\
}
	op = EDV_RECYCLE_BIN_OP_RECOVER;
	ADD_MENU_ITEM_LABEL
	GUISetMenuItemSetDefault(w, TRUE);
	recbin->contents_clist_recover_mi = w;

	ADD_MENU_SEP

	op = EDV_RECYCLE_BIN_OP_PURGE;
	ADD_MENU_ITEM_LABEL
	recbin->contents_clist_purge_mi = w;

	op = EDV_RECYCLE_BIN_OP_PURGE_ALL;
	ADD_MENU_ITEM_LABEL
	recbin->contents_clist_purge_all_mi = w;

	ADD_MENU_SEP

	op = EDV_RECYCLE_BIN_OP_RENAME;
	ADD_MENU_ITEM_LABEL
	recbin->contents_clist_rename_mi = w;

	op = EDV_RECYCLE_BIN_OP_CHMOD;
	ADD_MENU_ITEM_LABEL
	recbin->contents_clist_chmod_mi = w;

	op = EDV_RECYCLE_BIN_OP_CHOWN;
	ADD_MENU_ITEM_LABEL
	recbin->contents_clist_chown_mi = w;

	op = EDV_RECYCLE_BIN_OP_CHTIME;
	ADD_MENU_ITEM_LABEL
	recbin->contents_clist_chtime_mi = w;

	ADD_MENU_SEP

	op = EDV_RECYCLE_BIN_OP_PROPERTIES;
	ADD_MENU_ITEM_LABEL
	recbin->contents_clist_properties_mi = w;

#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_SEP
}


/*
 *	Sets the Recycle Bin values to the configuration.
 */
void edv_recycle_bin_sync_configuration(edv_recbin_struct *recbin)
{
	CfgList *cfg_list;
	EDVCore *core;
	
	if(recbin == NULL)
		return;

	core = recbin->core;
	cfg_list = core->cfg_list;

	/* Toplevel GtkWindow position and size */
	CfgGtkGetGtkWindow(
		cfg_list,
		EDV_CFG_PARM_RECBIN_X,
		EDV_CFG_PARM_RECBIN_Y,
		EDV_CFG_PARM_RECBIN_WIDTH,
		EDV_CFG_PARM_RECBIN_HEIGHT,
		recbin->toplevel
	);

	/* EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN_WIDTH parameter is updated
	 * as they change in edv_recycle_bin_resize_column_cb() or when the
	 * configuration list is changed directly
	 */

	/* EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN parameter is updated
	 * when the configuration list is changed directly
	 */

	/* Contents list filter */
	if(STRISEMPTY(recbin->contents_list_filter))
		EDV_SET_S(
			EDV_CFG_PARM_RECBIN_CONTENTS_LIST_FILTER,
			"*"
		);
	else
		EDV_SET_S(
			EDV_CFG_PARM_RECBIN_CONTENTS_LIST_FILTER,
			recbin->contents_list_filter
		);
}


/*
 *	Creates a new Operation ID.
 */
static EDVRecycleBinOp *edv_recycle_bin_op_new(void)
{
	return(EDV_RECYCLE_BIN_OP(g_malloc0(sizeof(EDVRecycleBinOp))));
}

/*
 *	Deletes the Operation ID.
 */
static void edv_recycle_bin_op_delete(EDVRecycleBinOp *op)
{
	if(op == NULL)
		return;

	g_free(op->button_name);
	g_free(op->menu_name);
	g_free(op->tooltip);
	g_free(op);
}

/*
 *	Creates a new Operation IDs list.
 */
GList *edv_recycle_bin_ops_list_new(edv_recbin_struct *recbin)
{
	guint		accel_key,
			accel_mods;
	const gchar	*button_name,
			*menu_name,
			*tooltip;
	guint8		**button_icon_data,
			**menu_icon_data;
	GList *op_ids_list = NULL;
	ToolBarItemType tb_item_type;
	EDVOpIDFlags flags;
	EDVRecycleBinOpID id;
	EDVRecycleBinOp *op;
	void (*func_cb)(ToolBarItem *, gint, gpointer);
	void (*enter_func_cb)(ToolBarItem *, gint, gpointer);

#define DO_APPEND_OPID	{			\
 op = edv_recycle_bin_op_new();			\
 if(op != NULL) {				\
  op->id = id;					\
  op->flags = flags;				\
  op->tb_item_type = tb_item_type;		\
  op->func_cb = func_cb;			\
  op->enter_func_cb = enter_func_cb;		\
  op->button_name = STRDUP(button_name);	\
  op->menu_name = STRDUP(menu_name);		\
  op->tooltip = STRDUP(tooltip);		\
  op->accel_key = accel_key;			\
  op->accel_mods = accel_mods;			\
  op->button_icon_data = button_icon_data;	\
  op->menu_icon_data = menu_icon_data;		\
  op->recbin = recbin;				\
  op_ids_list = g_list_append(op_ids_list, op);	\
 }						\
}

	func_cb = edv_recycle_bin_op_cb;
	enter_func_cb = edv_recycle_bin_op_enter_cb;


	id = EDV_RECYCLE_BIN_OP_SEPARATOR;
	flags = EDV_OPID_ALLOW_MULTIPLE | EDV_OPID_NO_OP;
	tb_item_type = TOOL_BAR_ITEM_SEPARATOR;
	button_icon_data = (edv_pixmap_data *)icon_hsep_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Separador";
	menu_name = "Separador";
	tooltip = "Separador";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Sparateur";
	menu_name = "Sparateur";
	tooltip = "Sparateur";
#elif defined(PROG_LANGUAGE_GERMAN)
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#elif defined(PROG_LANGUAGE_ITALIAN)
	button_name = "Separatore";
	menu_name = "Separatore";
	tooltip = "Separatore";
#elif defined(PROG_LANGUAGE_DUTCH)
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	button_name = "Separador";
	menu_name = "Separador";
	tooltip = "Separador";
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#else
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID


	id = EDV_RECYCLE_BIN_OP_CLOSE;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_close_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Close";
	menu_name = "Close";
	tooltip = "Close this window";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Cierre";
	menu_name = "Cierre";
	tooltip = "Cierre esta ventana";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Fermer";
	menu_name = "Fermer";
	tooltip = "Fermer cette fentre";
#endif
	accel_key = GDK_w;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_EXIT;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_exit_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Exit";
	menu_name = "Exit";
	tooltip = "Close all windows in the application and exit";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "La Salida";
	menu_name = "La Salida";
	tooltip = "Cierra todas ventanas en la aplicacin y la";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Quitter";
	menu_name = "Quitter";
	tooltip = "Ferme toutes fentres dans l'application et sortie";
#endif
	accel_key = GDK_q;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID


	id = EDV_RECYCLE_BIN_OP_SYNC_DISKS;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_save_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Sync Disks";
	menu_name = "Sync Disks";
	tooltip = "Flush any pending data transfers to disk";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Sincronizar Discos";
	menu_name = "Sincronizar Discos";
	tooltip = "Limpan cualquiera transferencias pendientes de datos al disco";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Synchro";
	menu_name = "Synchroniser Les Disques";
	tooltip = "Mettre  jour les transferts de donnes en attente au disque";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_HISTORY;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_trace_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "History";
	menu_name = "History...";
	tooltip = "History of commands processed by this application";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Historia";
	menu_name = "La Historia...";
	tooltip = "La Historia de rdenes procesadas por esta aplicacin";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Historique";
	menu_name = "Historique...";
	tooltip = "Historique des commandes traites par cette application";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_RUN;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_run_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Run";
	menu_name = "Run...";
	tooltip = "Run a program";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Corra";
	menu_name = "Corra...";
	tooltip = "Corra un programa";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Excuter";
	menu_name = "Excuter...";
	tooltip = "Excute un programme";
#endif
	accel_key = GDK_r;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_RUN_TERMINAL;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_terminal2_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Terminal";
	menu_name = "Terminal";
	tooltip = "Run a terminal";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "La Terminal";
	menu_name = "La Terminal";
	tooltip = "Corre una terminal";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Terminal";
	menu_name = "Terminal";
	tooltip = "Ouvrir un terminal";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_WRITE_PROTECT;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_write_protect_on_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Protect";
	menu_name = "Master Write Protect";
	tooltip = "Turn the master write protect on/off";
	accel_key = GDK_y;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_DELETE_METHOD_RECYCLE;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = edv_get_recycle_bin_icon_data(
		EDV_ICON_SIZE_20,
		1,
		NULL
	);
	menu_icon_data = button_icon_data;
	button_name = "Recycle";
	menu_name = "Delete Method: Recycle";
	tooltip = "Set the delete method to recycle";
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID
				  
	id = EDV_RECYCLE_BIN_OP_DELETE_METHOD_PURGE;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_purge_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Purge";
	menu_name = "Delete Method: Purge";
	tooltip = "Set the delete method to purge";
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID


	id = EDV_RECYCLE_BIN_OP_SYNC_RECYCLE_BIN;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = edv_get_recycle_bin_icon_data(
		EDV_ICON_SIZE_20,
		1,
		NULL
	);
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "SynRecBin";
	menu_name = "Sync Recycle Bin";
	tooltip = "Compact recycled objects and fix any errors";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "SynCorbeille";
	menu_name = "Compacte la Corbeille";
	tooltip = "Compacte la Corbeille et corrige toutes les erreurs";
#elif defined(PROG_LANGUAGE_GERMAN)
	button_name = "SynRecBin";
	menu_name = "Sync Recycle Bin";
	tooltip = "Compact recycled objects and fix any errors";
#elif defined(PROG_LANGUAGE_ITALIAN)
	button_name = "SynRecBin";
	menu_name = "Sync Recycle Bin";
	tooltip = "Compact recycled objects and fix any errors";
#elif defined(PROG_LANGUAGE_DUTCH)
	button_name = "SynRecBin";
	menu_name = "Sync Recycle Bin";
	tooltip = "Compact recycled objects and fix any errors";
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	button_name = "SynRecBin";
	menu_name = "Sync Recycle Bin";
	tooltip = "Compact recycled objects and fix any errors";
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	button_name = "SynRecBin";
	menu_name = "Sync Recycle Bin";
	tooltip = "Compact recycled objects and fix any errors";
#else
	button_name = "SynRecBin";
	menu_name = "Sync Recycle Bin";
	tooltip = "Compact recycled objects and fix any errors";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID


	id = EDV_RECYCLE_BIN_OP_RECOVER;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_recover_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Recover";
	menu_name = "Recover";
	tooltip = "Recover the selected object(s)";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Recupere";
	menu_name = "Recupere";
	tooltip = "Recupere objetos escogido(s)";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Retrouver";
	menu_name = "Retrouver";
	tooltip = "Retrouve l'objet choisi(s)";
#endif
	accel_key = GDK_Return;
	accel_mods = 0;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_PURGE;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_purge_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Purge";
	menu_name = "Purge";
	tooltip = "Permanently discard the selected object(s)";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Purga";
	menu_name = "Purga";
	tooltip = "Tira Permanentemente objetos escogido(s)";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Purge";
	menu_name = "Purge";
	tooltip = "D'Une faon permanente rejeter l'objet choisi(s)";
#endif
	accel_key = GDK_Delete;
	accel_mods = 0;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_PURGE_ALL;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_purge_all_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Purge All";
	menu_name = "Purge All";
	tooltip = "Permanently discard all of the recycled objects";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "PurgTodo";
	menu_name = "Purgue Todo";
	tooltip = "Tira Permanentemente todo el contenido";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "PurgTout";
	menu_name = "Purger Tout";
	tooltip = "d'une faon permanente rejeter tous objets recycls";
#endif
	accel_key = GDK_Delete;
	accel_mods = GDK_SHIFT_MASK;
	DO_APPEND_OPID


	id = EDV_RECYCLE_BIN_OP_RENAME;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_rename_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Renombrar";
	menu_name = "Renombrar";
	tooltip = "Renombrar archivo escogido";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Renommer";
	menu_name = "Renommer";
	tooltip = "Renomme l'objet choisi";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Zmie nazw";
	menu_name = "Zmie nazw";
	tooltip = "Zmie nazw zaznaczonego (zaznaczonych) obiektu (obiektw)";
#else
	button_name = "Rename";
	menu_name = "Rename";
	tooltip = "Rename the selected object";
#endif
	accel_key = GDK_F2;
	accel_mods = 0;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_CHMOD;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_chmod_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "ChMod";
	menu_name = "Cambie permisos...";
	tooltip = "El Cambio escogi los permisos de objetos";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "ChMod";
	menu_name = "Changer Le Mode...";
	tooltip = "Le Changement a choisi des permissions des objets";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "ChMod";
	menu_name = "Zmiana uprawnie...";
	tooltip = "Zmie uprawnienia dostpu do zaznaczonego (zaznaczonych) obiektu (obiektw)";
#else
	button_name = "ChMod";
	menu_name = "Change Permissions...";
	tooltip = "Change the permissions of the selected object(s)";
#endif
	accel_key = GDK_F9;
	accel_mods = 0;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_CHOWN;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_owned_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "ChOwn";
	menu_name = "Change Ownership...";
	tooltip = "Chown escogido(s)";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "ChOwn";
	menu_name = "Change Ownership...";
	tooltip = "Chown a choisi l'objet(s)";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "ChOwn";
	menu_name = "Zmiana waciciela...";
	tooltip = "Zmie wacicela zaznaczonego (zaznaczonych) obiektu (obiektw)";
#else
	button_name = "ChOwn";
	menu_name = "Change Ownership...";
	tooltip = "Change the ownership of the selected object(s)";
#endif
	accel_key = GDK_F10;
	accel_mods = 0;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_CHTIME;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_time_stamp_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "ChTime";
	menu_name = "Change Time Stamps...";
	tooltip = "Change the time stamps of the selected object(s)";
	accel_key = GDK_F11;
	accel_mods = 0;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_PROPERTIES;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_properties_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Props";
	menu_name = "Las Propiedades";
	tooltip = "Modifican escogi las propiedades de objeto";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Props";
	menu_name = "Les Proprits";
	tooltip = "Modifier a choisi les proprits de l'objet";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Waciwoci";
	menu_name = "Waciwoci...";
	tooltip = "Modyfikuj waciwoci zaznaczonego obiektu";
#else
	button_name = "Properties";
	menu_name = "Properties...";
	tooltip = "View the selected object's properties";
#endif
	accel_key = GDK_Return;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID


	id = EDV_RECYCLE_BIN_OP_SELECT_ALL;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_select_all_20x20_xpm;
	menu_icon_data = NULL;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "SelAll";
	menu_name = "Select All";
	tooltip = "Select all objects";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "SelAll";
	menu_name = "Escoja Todo";
	tooltip = "Escoge todo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "SelAll";
	menu_name = "Choisir Tout";
	tooltip = "Choisir tous objets";
#endif
	accel_key = GDK_a;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_UNSELECT_ALL;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_unselect_all_20x20_xpm;
	menu_icon_data = NULL;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "UnSelAll";
	menu_name = "Unselect All";
	tooltip = "Unselect all objects";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "UnSelAll";
	menu_name = "Unescoja Todo";
	tooltip = "Unescoge todo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "UnSelAll";
	menu_name = "Aucun";
	tooltip = "Aucun objets slects";
#endif
	accel_key = GDK_u;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_INVERT_SELECTION;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_invert_selection_20x20_xpm;
	menu_icon_data = NULL;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "InvSel";
	menu_name = "Invert Selection";
	tooltip =
 "Select all unselected objects and unselect all selected objects";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "InvSel";
	menu_name = "Invierta Seleccin";
	tooltip =
 "Selectos todos objetos y unselect no seleccionados todos objetos escogidos";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "InvSel";
	menu_name = "Inverser Slection";
	tooltip = "Rejete tous les objets slections et slectionne les autres";

#endif
	accel_key = GDK_i;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_FIND;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_search_20x20_xpm;;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Find";
	menu_name = "Find...";
	tooltip = "Find object(s) by specific attributes";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Buscar";
	menu_name = "Buscar";
	tooltip = "Buscar objeto(s) por atributos especificados";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Find";
	menu_name = "Recherche";
	tooltip = "Recherche le objet(s) suivant les attributs spcifis ";
#endif
	accel_key = GDK_f;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID


	id = EDV_RECYCLE_BIN_OP_REFRESH;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_reload_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Refresh";
	menu_name = "Refresh";
	tooltip = "Refresh the recycled objects listing";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Refresque";
	menu_name = "Refresque";
	tooltip = "Refresque objetos de recycled que listan";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Rafrachir";
	menu_name = "Rafrachir";
	tooltip = "Rafrachit des objets recycls pour numrer";
#endif
	accel_key = GDK_F5;
	accel_mods = 0;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_REFRESH_ALL;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_reload_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Refresh All";
	menu_name = "Refresh All";
	tooltip = "Refresh all listings";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Refresque Todo";
	menu_name = "Refresque Todo";
	tooltip = "Refresca todas listas";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Rafrachir Tout";
	menu_name = "Rafrachir Tout";
	tooltip = "Rafrachir toutes listes";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_SHOW_TOOL_BAR;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	flags = EDV_OPID_EX_TOOL_BAR;
	button_icon_data = (edv_pixmap_data *)icon_tool_bar_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "ToolBar";
	menu_name = "Tool Bar";
	tooltip = "Show/hide tool bar";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "ToolBar";
	menu_name = "Tool Bar";
	tooltip = "La Exposicin/la barra de herramienta de cuero";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Outils";
	menu_name = "Barre d'outils";
	tooltip = "Monter/cacher la barre d'outil";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_SHOW_FIND_BAR;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_tool_bar_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "FindBar";
	menu_name = "Find Bar";
	tooltip = "Show/hide find bar";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "FindBar";
	menu_name = "Find Bar";
	tooltip = "La exposicin/la barra del hallazgo del cuero";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Recherche";
	menu_name = "Barre de Recherche";
	tooltip = "Montrer/cacher la barre de Recherche";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_SHOW_STATUS_BAR;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_status_bar_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "StatusBar";
	menu_name = "Status Bar";
	tooltip = "Show/hide status bar";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "StatusBar";
	menu_name = "Status Bar";
	tooltip = "La exposicin/la barra de la posicin del cuero";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Etat";
	menu_name = "Barre d'tat";
	tooltip = "monter/cacher la barre d'tat";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID


	id = EDV_RECYCLE_BIN_OP_CONTENTS_LIST_FILTER;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_wildcards_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Filter";
	menu_name = "Set Name Filter...";
	tooltip = "Set the filter to list only the objects that have\
 a specific name pattern";
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_CONTENTS_LIST_AUTO_RESIZE_COLUMNS;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_resize_column_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "ReSzCol";
	menu_name = "Auto Resize Columns";
	tooltip = "Automatically resize the columns each time the list is updated";
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID


	id = EDV_RECYCLE_BIN_OP_MIME_TYPES;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_mimetypes_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "MIMETypes";
	menu_name = "MIME Types...";
	tooltip = "View/modify MIME Types";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "MIMETypes";
	menu_name = "MIME Types...";
	tooltip = "El panorama/modifica MIME Types";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "MIMETypes";
	menu_name = "MIME Types...";
	tooltip = "Voir/modifie MIME Types";
#endif
	accel_key = GDK_m;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID


	id = EDV_RECYCLE_BIN_OP_NEW_BROWSER;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_endeavour_file_browser_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "FBrowser";
	menu_name = "New File Browser";
	tooltip = "Create a new file browser";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "FBrowser";
	menu_name = "Nuevo Examinador De Archivos";
	tooltip = "Muestra un nuevo examinador de archivos";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Dossier";
	menu_name = "Nouveau navigateur de Fichier";
	tooltip = "Crer un nouveau navigateur de fichier";
#endif
	accel_key = GDK_1;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_NEW_IMBR;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_endeavour_image_browser_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "IBrowser";
	menu_name = "New Image Browser";
	tooltip = "Create a new image browser";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "IBrowser";
	menu_name = "Nuevo Examinador De Imagenes";
	tooltip = "Muestra un nuevo examinador de imagenes";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Galerie";
	menu_name = "Nouveau Navigateur d'Image";
	tooltip = "Crer un nouveau navigateur d'image";
#endif
	accel_key = GDK_2;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_NEW_ARCHIVER;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_endeavour_archiver_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Archiver";
	menu_name = "New Archiver";
	tooltip = "Create a new archiver";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Archiver";
	menu_name = "Nuevo Compresor De Archivos";
	tooltip = "Muestra un archivo comprimido nuevo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Archive";
	menu_name = "Nouvelle Archive";
	tooltip = "Crer un nouvelle archive";
#endif
	accel_key = GDK_3;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID


	id = EDV_RECYCLE_BIN_OP_OPTIONS;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_options_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Options";
	menu_name = "Options...";
	tooltip = "Modify this application's options";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Las Opciones";
	menu_name = "Las Opciones...";
	tooltip = "Modifican estas opciones de la aplicacin";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Options";
	menu_name = "Options";
	tooltip = "Modifier les options de l'application";
#endif
	accel_key = GDK_o;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_CUSTOMIZE;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_customize_20x20_xpm;
	menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Custom";
	menu_name = "Customize...";
	tooltip = "Customize the look and feel of this application";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Personal";
	menu_name = "Personaliza...";
	tooltip = "Personaliza la mirada y se siente de esta aplicacin";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Prfrences";
	menu_name = "Prfrences...";
	tooltip = "Rgle les prfrences pour cette application";
#endif
	accel_key = GDK_t;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID


	id = EDV_RECYCLE_BIN_OP_HELP_CONTENTS;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "Help";
	menu_name = "Contents";
	tooltip = "Index of all help pages";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "La Ayuda";
	menu_name = "El Contenido";
	tooltip = "El Indice de toda ayuda pagina";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Aide";
	menu_name = "Contenu d'aide";
	tooltip = "Index de toutes les pages d'aide";
#endif
	accel_key = GDK_F1;
	accel_mods = 0;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_HELP_FILE_BROWSER;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "Help";
	menu_name = "File Browser";
	tooltip = "Help on using the file browser";
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_HELP_IMAGE_BROWSER;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "Help";
	menu_name = "Image Browser";
	tooltip = "Help on using the image browser";
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_HELP_ARCHIVER;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "Help";
	menu_name = "Archiver";
	tooltip = "Help on using the archiver";
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_HELP_RECYCLE_BIN;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Aide";
	menu_name = "Corbeille";
	tooltip = "aide sur l'utilisation de la Corbeille";
#else
	button_name = "Help";
	menu_name = "Recycle Bin";
	tooltip = "Help on using the recycle bin";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_HELP_KEYS_LIST;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "HelpKeys";
	menu_name = "Keys List";
	tooltip = "List of all keyboard key operations";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "HelpKeys";
	menu_name = "La Lista De Llaves";
	tooltip = "La Lista de todo teclado las operaciones claves";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Raccourcis";
	menu_name = "Raccourcis clavier";
	tooltip = "Liste des raccourcis clavier pour les oprations principales";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_HELP_COMMON_OPERATIONS;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "HelpOp";
	menu_name = "Common Operations";
	tooltip = "Help on the most common operations";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "HelpOp";
	menu_name = "Las Operaciones Comunes";
	tooltip = "La Ayuda en las operaciones ms comunes";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "Oprations";
	menu_name = "Les Oprations Communes";
	tooltip = "Aide sur montage des composants";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	id = EDV_RECYCLE_BIN_OP_HELP_ABOUT;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_about_20x20_xpm;
	menu_icon_data = (edv_pixmap_data *)icon_about_20x20_xpm;
#ifdef PROG_LANGUAGE_ENGLISH
	button_name = "About";
	menu_name = "About";
	tooltip = "About this application";
#endif
#ifdef PROG_LANGUAGE_SPANISH
	button_name = "Acerca";
	menu_name = "Acerca De";
	tooltip = "Acerca de esta aplicacin";
#endif
#ifdef PROG_LANGUAGE_FRENCH
	button_name = "A propos";
	menu_name = "A propos";
	tooltip = "A propos de cette application";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID


	return(op_ids_list);
#undef DO_APPEND_OPID
}

/*
 *	Deletes the Operation IDs list.
 */
void edv_recycle_bin_ops_list_delete(GList *op_ids_list)
{
	if(op_ids_list == NULL)
		return;

	g_list_foreach(
		op_ids_list,
		(GFunc)edv_recycle_bin_op_delete,
		NULL
	);
	g_list_free(op_ids_list);
}


/*
 *	Creates a new Accel Key List with default values.
 */
CfgAccelkeyList *edv_recycle_bin_accelkeys_new(void)
{
	GList		*glist,
			*op_ids_list,
			*ak_glist = NULL;
	CfgAccelkeyList *ak_list;
	EDVRecycleBinOp *opid;

	op_ids_list = edv_recycle_bin_ops_list_new(NULL);
	for(glist = op_ids_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		opid = EDV_RECYCLE_BIN_OP(glist->data);
		if(opid == NULL)
			continue;

		if(opid->flags & EDV_OPID_NO_OP)
			continue;

		ak_glist = g_list_append(
			ak_glist,
			CFGAccelkeyNew(
				(gint)opid->id,
				opid->accel_key,
				opid->accel_mods
			)
		);
	}
	edv_recycle_bin_ops_list_delete(op_ids_list);

	ak_list = CFGAccelkeyListNew(ak_glist);

	g_list_foreach(ak_glist, (GFunc)CFGAccelkeyDelete, NULL);
	g_list_free(ak_glist);

	return(ak_list);
}


/*
 *	Creates the Recycle Bin Menu Bar.
 *
 *	This function should be called by edv_recycle_bin_new().
 */
static void EDVRecBinCreateMenuBar(
	edv_recbin_struct *recbin,
	GtkWidget *parent
)
{
	edv_pixmap_data *icon_data = NULL;
	EDVRecycleBinOpID id;
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget	*w,
			*menu,
			*menu_bar;

	/* Create Menu Bar */
	recbin->menu_bar = menu_bar = w = GUIMenuBarCreate(NULL);
	gtk_widget_set_name(w, EDV_WIDGET_NAME_MENU_BAR);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);

#define ADD_MENU_ITEM_LABEL		{	\
 EDVRecycleBinOp *op =				\
  edv_recycle_bin_op_match_by_id(recbin, id);	\
 if(op != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp,	\
   (icon_data != NULL) ?			\
    icon_data : op->menu_icon_data,		\
   op->menu_name,				\
   op->accel_key, op->accel_mods,		\
   edv_recycle_bin_menu_item_cb, op			\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)edv_recycle_bin_menu_item_enter_cb, op,	\
   (gpointer)edv_recycle_bin_menu_item_leavr_cb, op	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_ITEM_CHECK	{		\
 EDVRecycleBinOp *op =				\
  edv_recycle_bin_op_match_by_id(recbin, id);	\
 if(op != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu,					\
   GUI_MENU_ITEM_TYPE_CHECK,			\
   accelgrp,					\
   NULL,					\
   op->menu_name,				\
   op->accel_key, op->accel_mods,		\
   edv_recycle_bin_menu_item_cb, op			\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)edv_recycle_bin_menu_item_enter_cb, op,	\
   (gpointer)edv_recycle_bin_menu_item_leavr_cb, op	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_SEP	{			\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL,						\
  NULL,						\
  0, 0,						\
  NULL, NULL					\
 );						\
}

	/* Create Menus */
	/* File */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		id = EDV_RECYCLE_BIN_OP_RECOVER;
		ADD_MENU_ITEM_LABEL
		recbin->recover_mi = w;

		ADD_MENU_SEP

		id = EDV_RECYCLE_BIN_OP_PURGE;
		ADD_MENU_ITEM_LABEL
		recbin->purge_mi = w;

		id = EDV_RECYCLE_BIN_OP_PURGE_ALL;
		ADD_MENU_ITEM_LABEL
		recbin->purge_all_mi = w;

		ADD_MENU_SEP

		id = EDV_RECYCLE_BIN_OP_SYNC_RECYCLE_BIN;
		ADD_MENU_ITEM_LABEL
		recbin->sync_recycle_bin_mi = w;

		id = EDV_RECYCLE_BIN_OP_SYNC_DISKS;
		ADD_MENU_ITEM_LABEL
		recbin->sync_disks_mi = w;

		ADD_MENU_SEP

		id = EDV_RECYCLE_BIN_OP_HISTORY;
		ADD_MENU_ITEM_LABEL
		recbin->history_mi = w;

		ADD_MENU_SEP

		id = EDV_RECYCLE_BIN_OP_RUN;
		ADD_MENU_ITEM_LABEL
		recbin->run_mi = w;

		ADD_MENU_SEP

		id = EDV_RECYCLE_BIN_OP_CLOSE;
		ADD_MENU_ITEM_LABEL
		recbin->close_mi = w;
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Archivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fichier"
#elif defined(PROG_LANGUAGE_GERMAN)
"Akte"
#elif defined(PROG_LANGUAGE_ITALIAN)
"File"
#elif defined(PROG_LANGUAGE_DUTCH)
"Dossier"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Arquivo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Arkiv"
#else
"File"
#endif
		, GTK_JUSTIFY_LEFT
	);

	/* Edit */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		id = EDV_RECYCLE_BIN_OP_RECOVER;
		ADD_MENU_ITEM_LABEL
		recbin->edit_recover_mi = w;

		ADD_MENU_SEP

		id = EDV_RECYCLE_BIN_OP_PURGE;
		ADD_MENU_ITEM_LABEL
		recbin->edit_purge_mi = w;

		id = EDV_RECYCLE_BIN_OP_PURGE_ALL;
		ADD_MENU_ITEM_LABEL
		recbin->edit_purge_all_mi = w;

		ADD_MENU_SEP

		id = EDV_RECYCLE_BIN_OP_RENAME;
		ADD_MENU_ITEM_LABEL
		recbin->rename_mi = w;

		id = EDV_RECYCLE_BIN_OP_CHMOD;
		ADD_MENU_ITEM_LABEL
		recbin->chmod_mi = w;

		id = EDV_RECYCLE_BIN_OP_CHOWN;
		ADD_MENU_ITEM_LABEL
		recbin->chown_mi = w;

		id = EDV_RECYCLE_BIN_OP_CHTIME;
		ADD_MENU_ITEM_LABEL
		recbin->chtime_mi = w;

		ADD_MENU_SEP

		id = EDV_RECYCLE_BIN_OP_SELECT_ALL;
		ADD_MENU_ITEM_LABEL
		recbin->select_all_mi = w;

		id = EDV_RECYCLE_BIN_OP_UNSELECT_ALL;
		ADD_MENU_ITEM_LABEL
		recbin->unselect_all_mi = w;

		id = EDV_RECYCLE_BIN_OP_INVERT_SELECTION;
		ADD_MENU_ITEM_LABEL
		recbin->invert_selection_mi = w;

		ADD_MENU_SEP

		id = EDV_RECYCLE_BIN_OP_FIND;
		ADD_MENU_ITEM_LABEL
		recbin->find_mi = w;

		ADD_MENU_SEP

		id = EDV_RECYCLE_BIN_OP_PROPERTIES;
		ADD_MENU_ITEM_LABEL
		recbin->properties_mi = w;
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Editar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Editer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Redigieren"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Redigere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bewerking"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Edite"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Rediger"
#else
"Edit"
#endif
		, GTK_JUSTIFY_LEFT
	);


	/* View Menu */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		id = EDV_RECYCLE_BIN_OP_REFRESH;
		ADD_MENU_ITEM_LABEL
		recbin->refresh_mi = w;

		id = EDV_RECYCLE_BIN_OP_REFRESH_ALL;
		ADD_MENU_ITEM_LABEL
		recbin->refresh_all_mi = w;

		ADD_MENU_SEP

		id = EDV_RECYCLE_BIN_OP_SHOW_TOOL_BAR;
		ADD_MENU_ITEM_CHECK
		recbin->show_tool_bar_micheck = w;

		id = EDV_RECYCLE_BIN_OP_SHOW_FIND_BAR;
		ADD_MENU_ITEM_CHECK
		recbin->show_find_bar_micheck = w;

		id = EDV_RECYCLE_BIN_OP_SHOW_STATUS_BAR;
		ADD_MENU_ITEM_CHECK
		recbin->show_status_bar_micheck = w;

		ADD_MENU_SEP

		id = EDV_RECYCLE_BIN_OP_CONTENTS_LIST_AUTO_RESIZE_COLUMNS;
		ADD_MENU_ITEM_CHECK
		recbin->contents_list_auto_resize_columns_micheck = w;

		id = EDV_RECYCLE_BIN_OP_CONTENTS_LIST_FILTER;
		ADD_MENU_ITEM_LABEL
		recbin->contents_list_filter_mi = w;

		ADD_MENU_SEP

		id = EDV_RECYCLE_BIN_OP_MIME_TYPES;
		ADD_MENU_ITEM_LABEL
		recbin->mime_types_mi = w;
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Vista"
#elif defined(PROG_LANGUAGE_FRENCH)
"Voir"
#elif defined(PROG_LANGUAGE_GERMAN)
"Blick"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Veduta"
#elif defined(PROG_LANGUAGE_DUTCH)
"Overzicht"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Vista"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Sikt"
#else
"View"
#endif
		, GTK_JUSTIFY_LEFT
	);


	/* Settings Menu */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		id = EDV_RECYCLE_BIN_OP_WRITE_PROTECT;
		ADD_MENU_ITEM_LABEL
		recbin->write_protect_mi = w;

		id = EDV_RECYCLE_BIN_OP_DELETE_METHOD_RECYCLE;
		ADD_MENU_ITEM_CHECK
		recbin->delete_method_recycle_micheck = w;

		id = EDV_RECYCLE_BIN_OP_DELETE_METHOD_PURGE;
		ADD_MENU_ITEM_CHECK
		recbin->delete_method_purge_micheck = w;


		ADD_MENU_SEP

		id = EDV_RECYCLE_BIN_OP_OPTIONS;
		ADD_MENU_ITEM_LABEL
		recbin->options_mi = w;

		id = EDV_RECYCLE_BIN_OP_CUSTOMIZE;
		ADD_MENU_ITEM_LABEL
		recbin->customize_mi = w;
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Configuracion"
#elif defined(PROG_LANGUAGE_FRENCH)
"Rglages"
#elif defined(PROG_LANGUAGE_GERMAN)
"Setzen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Montaggi"
#elif defined(PROG_LANGUAGE_DUTCH)
"Zetten"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Por"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Innstillinger"
#else
"Settings"
#endif
		, GTK_JUSTIFY_LEFT
	);

	/* Windows */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		id = EDV_RECYCLE_BIN_OP_NEW_BROWSER;
		ADD_MENU_ITEM_LABEL
		recbin->window_new_browser_mi = w;

		id = EDV_RECYCLE_BIN_OP_NEW_IMBR;
		ADD_MENU_ITEM_LABEL
		recbin->window_new_imbr_mi = w;

		id = EDV_RECYCLE_BIN_OP_NEW_ARCHIVER;
		ADD_MENU_ITEM_LABEL
		recbin->window_new_archiver_mi = w;
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Ventanas"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fentres"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fenster"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Finestre"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ramen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Janelas"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Vinduer"
#else
"Windows"
#endif
		, GTK_JUSTIFY_LEFT
	);

	/* Help */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		id = EDV_RECYCLE_BIN_OP_HELP_CONTENTS;
		ADD_MENU_ITEM_LABEL
		recbin->help_contents_mi = w;

		ADD_MENU_SEP

		id = EDV_RECYCLE_BIN_OP_HELP_RECYCLE_BIN;
		ADD_MENU_ITEM_LABEL
		recbin->help_recycle_bin_mi = w;

		id = EDV_RECYCLE_BIN_OP_HELP_KEYS_LIST;
		ADD_MENU_ITEM_LABEL
		recbin->help_keys_list_mi = w;

		id = EDV_RECYCLE_BIN_OP_HELP_COMMON_OPERATIONS;
		ADD_MENU_ITEM_LABEL
		recbin->help_common_operations_mi = w;

		ADD_MENU_SEP

		id = EDV_RECYCLE_BIN_OP_HELP_ABOUT;
		ADD_MENU_ITEM_LABEL
		recbin->help_about_mi = w;
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Ayuda"
#elif defined(PROG_LANGUAGE_FRENCH)
"Aide"
#elif defined(PROG_LANGUAGE_GERMAN)
"Hilfe"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Aiuto"
#elif defined(PROG_LANGUAGE_DUTCH)
"Hulp"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Ajuda"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Hjelp"
#else
"Help"
#endif
		, GTK_JUSTIFY_RIGHT
	);

#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_CHECK
#undef ADD_MENU_SEP
}

/*
 *	Creates a new Recycle Bin.
 */
edv_recbin_struct *edv_recycle_bin_new(
	EDVCore *core,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry
)
{
	const gchar	*wm_name,
					*wm_class;
	GtkRcStyle	*standard_rcstyle,
					*lists_rcstyle;
	GtkWidget	*w,
					*parent, *parent2,
					*toplevel;
	GtkCList *clist;
	const GtkTargetEntry dnd_tar_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,	0,	EDV_DND_INFO_TEXT_PLAIN},
{GUI_TARGET_NAME_TEXT_URI_LIST,	0,	EDV_DND_INFO_TEXT_URI_LIST},
{GUI_TARGET_NAME_STRING,	0,	EDV_DND_INFO_STRING}
	};
	const GtkTargetEntry dnd_src_types[] = {
{EDV_DND_TARGET_RECYCLED_OBJECT, 0,	EDV_DND_INFO_RECYCLED_OBJECT}
	};
	CfgList *cfg_list;
	edv_recbin_struct *recbin;

	if(core == NULL)
		return(NULL);

	cfg_list = core->cfg_list;
	standard_rcstyle = core->standard_rcstyle;
	lists_rcstyle = core->lists_rcstyle;
	wm_name = core->wm_name;
	wm_class = core->wm_class;

	/* Create the Recycle Bin */
	recbin = EDV_RECBIN(g_malloc0(sizeof(edv_recbin_struct)));
	if(recbin == NULL)
		return(NULL);

	recbin->toplevel = toplevel = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	recbin->core = core;
#if 0
	recbin->flags = 0;
	recbin->freeze_count = 0;
	recbin->busy_count = 0;
	recbin->stop_count = 0;
#endif
	recbin->op_ids_list = edv_recycle_bin_ops_list_new(recbin);

	if(EDV_GET_B(EDV_CFG_PARM_RECBIN_SHOW_TOOL_BAR))
		recbin->flags |= EDV_RECYCLE_BIN_TOOL_BAR_MAPPED;
	if(EDV_GET_B(EDV_CFG_PARM_RECBIN_SHOW_FIND_BAR))
		recbin->flags |= EDV_RECYCLE_BIN_FIND_BAR_MAPPED;
	if(EDV_GET_B(EDV_CFG_PARM_RECBIN_SHOW_STATUS_BAR))
		recbin->flags |= EDV_RECYCLE_BIN_STATUS_BAR_MAPPED;

	recbin->contents_list_filter = STRDUP(EDV_GET_S(
		EDV_CFG_PARM_RECBIN_CONTENTS_LIST_FILTER
	));
	recbin->contents_clist_selected_row = -1;
	recbin->last_nrecycle_bin_items = -1;
	recbin->last_write_protect_state = -1;

	recbin->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	if(!STRISEMPTY(wm_name) && !STRISEMPTY(wm_class))
		gtk_window_set_wmclass(GTK_WINDOW(w), wm_name, wm_class);
	else
		gtk_window_set_wmclass(
			GTK_WINDOW(w), EDV_RECYCLE_BIN_WM_CLASS_WINDOW_NAME, PROG_NAME
		);
	gtk_widget_set_name(w, EDV_WIDGET_NAME_RECYCLE_BIN_TOPLEVEL);
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, FALSE);
	if((geometry_flags != 0) && (geometry != NULL))
	{
		if((geometry_flags & GDK_GEOMETRY_X) || (geometry_flags & GDK_GEOMETRY_Y))
			gtk_widget_set_uposition(w, geometry->x, geometry->y);

		if((geometry_flags & GDK_GEOMETRY_WIDTH) || (geometry_flags & GDK_GEOMETRY_HEIGHT))
			gtk_widget_set_usize(w, geometry->width, geometry->height);
		else
			CfgGtkSetGtkWindow(
				cfg_list,
				NULL, NULL,
				EDV_CFG_PARM_RECBIN_WIDTH,
				EDV_CFG_PARM_RECBIN_HEIGHT,
				w
			);
	}
	else
	{
		CfgGtkSetGtkWindow(
			cfg_list,
			EDV_CFG_PARM_RECBIN_X,
			EDV_CFG_PARM_RECBIN_Y,
			EDV_CFG_PARM_RECBIN_WIDTH,
			EDV_CFG_PARM_RECBIN_HEIGHT,
			w
		);
	}
	gtk_widget_add_events(
		w,
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "delete_event",
		GTK_SIGNAL_FUNC(edv_recycle_bin_delete_event_cb), recbin
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(edv_recycle_bin_key_event_cb), recbin
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(edv_recycle_bin_key_event_cb), recbin
	);
	parent = w;

	/* Main GtkVBox */
	recbin->main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;

	/* Bars GtkVBox */
	w = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* GtkHandleBox for the menu bar */
	recbin->menu_bar_handle = w = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_attached",
		GTK_SIGNAL_FUNC(edv_recycle_bin_handle_child_attached_cb), recbin
	);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_detached",
		GTK_SIGNAL_FUNC(edv_recycle_bin_handle_child_detached_cb), recbin
	);
	gtk_widget_show(w);
	/* Create the menu bar */
	EDVRecBinCreateMenuBar(recbin, w);


	/* Tool Bar GtkHandleBox */
	recbin->tool_bar_handle = w = gtk_handle_box_new();
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_attached",
		GTK_SIGNAL_FUNC(edv_recycle_bin_handle_child_attached_cb), recbin
	);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_detached",
		GTK_SIGNAL_FUNC(edv_recycle_bin_handle_child_detached_cb), recbin
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	if(recbin->flags & EDV_RECYCLE_BIN_TOOL_BAR_MAPPED)
		gtk_widget_show(w);
	/* Build the tool bar later */
/*	recbin->tool_bar = NULL; */


	/* GtkHandleBox for the Find Bar */
	recbin->find_bar_handle = w = gtk_handle_box_new();
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_attached",
		GTK_SIGNAL_FUNC(edv_recycle_bin_handle_child_attached_cb), recbin
	);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_detached",
		GTK_SIGNAL_FUNC(edv_recycle_bin_handle_child_detached_cb), recbin
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	if(recbin->flags & EDV_RECYCLE_BIN_FIND_BAR_MAPPED)
		gtk_widget_show(w);
	parent2 = w;

	/* Create the Find Bar */
	recbin->find_bar = w = edv_find_bar_new(
		core,
		EDV_LOCATION_TYPE_RECYCLE_BIN,
		NULL, NULL,				/* No get location cb */
		edv_recycle_bin_find_bar_start_cb,
		recbin,
		edv_recycle_bin_find_bar_end_cb,
		recbin,
		edv_recycle_bin_find_bar_match_cb,
		recbin,
		edv_recycle_bin_find_bar_status_message_cb,
		recbin,
		edv_recycle_bin_find_bar_status_progress_cb,
		recbin
	);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);


	/* GtkScrolledWindow for the Contents CList */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
		GTK_SCROLLED_WINDOW(w),
		GTK_POLICY_AUTOMATIC,
		GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Contents GtkCList */
	recbin->contents_clist = w = gtk_clist_new(
		EDV_RECYCLE_BIN_CONTENTS_CLIST_COLUMNS_MAX
	);
	clist = GTK_CLIST(w);
	gtk_widget_set_name(w, EDV_WIDGET_NAME_RECYCLE_BIN_CONTENTS_LIST);
	gtk_widget_add_events(
		w,
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "motion_notify_event",
		GTK_SIGNAL_FUNC(edv_clist_motion_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(edv_recycle_bin_key_event_cb), recbin
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(edv_recycle_bin_key_event_cb), recbin
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_recycle_bin_button_event_cb), recbin
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "resize_column",
		GTK_SIGNAL_FUNC(edv_recycle_bin_resize_column_cb), recbin
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "click_column",
		GTK_SIGNAL_FUNC(edv_recycle_bin_click_column_cb), recbin
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "select_row",
		GTK_SIGNAL_FUNC(edv_recycle_bin_select_row_cb), recbin
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "unselect_row",
		GTK_SIGNAL_FUNC(edv_recycle_bin_unselect_row_cb), recbin
	);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED);
	if(clist->columns > 0)
	{
		CfgIntList *column_widths_intlist = EDV_GET_INT_LIST(
			EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN_WIDTH
		);
		if(column_widths_intlist != NULL)
		{
			EDVRecycleBinColumnType column_type = 0;
			GList *glist = column_widths_intlist->list;
			while(glist != NULL)
			{
				edv_recycle_bin_list_set_column_width_by_type(
					recbin,
					column_type,
					MAX((gint)glist->data, 10)	/* Width */
				);
				column_type++;
				glist = g_list_next(glist);
			}
		}
	}
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
	GUIDNDSetSrc(
		w,
		dnd_src_types,
		sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_MOVE,			/* Actions */
		GDK_BUTTON1_MASK,			/* Buttons */
		NULL,
		edv_recycle_bin_list_drag_data_get_cb,
		edv_recycle_bin_list_drag_data_delete_cb,
		NULL,
		recbin
	);
	GUIDNDSetTar(
		w,
		dnd_tar_types,
		sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_MOVE,			/* Actions */
		GDK_ACTION_MOVE,			/* Default action if same */
		GDK_ACTION_MOVE,			/* Default action */
		edv_recycle_bin_list_drag_data_received_cb,
		recbin,
		TRUE				/* Highlight */
	);
	gtk_widget_show(w);



	/* Create the Tool Bar */
	edv_recycle_bin_tool_bar_regenerate(recbin);

	/* Create the Contents List Menu */
	edv_recycle_bin_list_menu_regenerate(recbin);


	/* Status Bar */
	recbin->status_bar = w = edv_status_bar_new(core);
	gtk_box_pack_end(GTK_BOX(parent), w, FALSE, FALSE, 0);
	if(recbin->flags & EDV_RECYCLE_BIN_STATUS_BAR_MAPPED)
		gtk_widget_show(w);


	/* Set the initial RC styles */
	if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			recbin->toplevel,
			standard_rcstyle
		);
	if(lists_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			recbin->contents_clist,
			lists_rcstyle
		);
	if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			recbin->contents_clist_menu,
			standard_rcstyle
		);

	/* Set the initial Accel Keys */
	edv_recycle_bin_accelkeys_regenerate(recbin);

	recbin->freeze_count--;

	return(recbin);
}

/*
 *	Syncronizes all data in the recycle bin's widgets to internal data
 *	on the recycle bin.
 */
void edv_recycle_bin_sync_data(edv_recbin_struct *recbin)
{
	if(recbin == NULL)
		return;

/* This function may be removed in the future, since there really is
 * no floating data from widgets to be synced to disk on the recycle bin
 */
}

/*
 *	Updates all widgets on the given recycle bin to reflect the current
 *	data.
 */
void edv_recycle_bin_update_display(edv_recbin_struct *recbin)
{
	gboolean	write_protect,
					write_protect_changed = FALSE,
					sensitive,
					state;
	gint		i,
					selected_obj_row = -1;
	guint8		**recbin_icon_data,
					**recbin_icon_large_data;
	GList *glist;
	GtkWidget	*w,
					*tb;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVDeleteMethod delete_method;
	EDVCore *core;

	if(recbin == NULL)
		return;

	recbin->freeze_count++;

	tb = recbin->tool_bar;
	core = recbin->core;
	cfg_list = core->cfg_list;

#define MENU_ITEM_SET_TOGGLE(_w_,_state_)		\
 GUIMenuItemSetCheck(					\
  (_w_),						\
  (_state_),						\
  FALSE							\
 )
#define TOOL_BAR_ITEM_SET_SENSITIVE(_op_,_sensitive_)	\
 ToolBarSetItemSensitive(				\
  tb,							\
  (_op_),						\
  (_sensitive_)						\
 )
#define TOOL_BAR_ITEM_SET_TOGGLE(_op_,_state_)		\
 ToolBarSetItemToggle(					\
  tb,							\
  (_op_),						\
  (_state_)						\
 )
#define TOOL_BAR_ITEM_UPDATE(_op_,_text_,_icon_data_,_tip_) \
 ToolBarUpdateItem(					\
  tb,							\
  (_op_),						\
  (_text_),						\
  (_icon_data_),					\
  (_tip_)						\
 )

	/* Get the master write protect state */
	write_protect = EDV_GET_B(EDV_CFG_PARM_WRITE_PROTECT);
	if(recbin->last_write_protect_state < 0)
	{
		write_protect_changed = TRUE;
	}
	else
	{
		if((write_protect && !recbin->last_write_protect_state) ||
		   (!write_protect && recbin->last_write_protect_state)
		)
			write_protect_changed = TRUE;
	}
	recbin->last_write_protect_state = write_protect ? 1 : 0;

	/* Delete method */
	delete_method = (EDVDeleteMethod)EDV_GET_I(
		EDV_CFG_PARM_DELETE_METHOD
	);

	/* Get the last selected row on the contents clist */
	clist = GTK_CLIST(recbin->contents_clist);
	glist = clist->selection_end;
	selected_obj_row = (glist != NULL) ? (gint)glist->data : -1;

	/* Get the Recycle Bin icon data */
	recbin_icon_data = edv_get_recycle_bin_icon_data(
		EDV_ICON_SIZE_20,
		core->last_nrecycle_bin_items,
		NULL
	);
	recbin_icon_large_data = edv_get_recycle_bin_icon_data(
		EDV_ICON_SIZE_48,
		core->last_nrecycle_bin_items,
		NULL
	);
	recbin->last_nrecycle_bin_items = core->last_nrecycle_bin_items;


	/* Update the Find Bar */
	edv_find_bar_update_display(recbin->find_bar);


	/* Update the Menu Bar and Tool Bar */

	/* File */
	sensitive = (selected_obj_row > -1) ? TRUE : FALSE;
	/* Recover */
	GTK_WIDGET_SET_SENSITIVE(recbin->recover_mi, sensitive);
	GTK_WIDGET_SET_SENSITIVE(recbin->edit_recover_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_RECYCLE_BIN_OP_RECOVER, sensitive);
	/* Purge */
	GTK_WIDGET_SET_SENSITIVE(recbin->purge_mi, sensitive);
	GTK_WIDGET_SET_SENSITIVE(recbin->edit_purge_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_RECYCLE_BIN_OP_PURGE, sensitive);
	/* Sync Recycle bin */
	if(recbin_icon_data != NULL)
	{
		GUIMenuItemSetPixmap(
			recbin->sync_recycle_bin_mi,
			recbin_icon_data
		);
		TOOL_BAR_ITEM_UPDATE(
			EDV_RECYCLE_BIN_OP_SYNC_RECYCLE_BIN,
			NULL,
			recbin_icon_data,
			NULL
		);
	}

	/* Edit */
	sensitive = (selected_obj_row > -1) ? TRUE : FALSE;
	/* Rename */
	GTK_WIDGET_SET_SENSITIVE(recbin->rename_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_RECYCLE_BIN_OP_RENAME, sensitive);
	/* Change Mode */
	GTK_WIDGET_SET_SENSITIVE(recbin->chmod_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_RECYCLE_BIN_OP_CHMOD, sensitive);
	/* Change Ownership */
	GTK_WIDGET_SET_SENSITIVE(recbin->chown_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_RECYCLE_BIN_OP_CHOWN, sensitive);
	/* Change Timestamps */
	GTK_WIDGET_SET_SENSITIVE(recbin->chtime_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_RECYCLE_BIN_OP_CHTIME, sensitive);
	/* Properties */
	GTK_WIDGET_SET_SENSITIVE(recbin->properties_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_RECYCLE_BIN_OP_PROPERTIES, sensitive);

	/* View */
	/* Show Tool Bar */
	state = EDV_GET_B(EDV_CFG_PARM_RECBIN_SHOW_TOOL_BAR);
	MENU_ITEM_SET_TOGGLE(recbin->show_tool_bar_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_RECYCLE_BIN_OP_SHOW_TOOL_BAR, state);
	/* Show Find Bar */
	state = EDV_GET_B(EDV_CFG_PARM_RECBIN_SHOW_FIND_BAR);
	MENU_ITEM_SET_TOGGLE(recbin->show_find_bar_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_RECYCLE_BIN_OP_SHOW_FIND_BAR, state);
	/* Show Status Bar */
	state = EDV_GET_B(EDV_CFG_PARM_RECBIN_SHOW_STATUS_BAR);
	MENU_ITEM_SET_TOGGLE(recbin->show_status_bar_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_RECYCLE_BIN_OP_SHOW_STATUS_BAR, state);

	/* Auto Resize Columns */
	state = EDV_GET_B(EDV_CFG_PARM_RECBIN_CONTENTS_LIST_AUTO_RESIZE_COLUMNS);
	MENU_ITEM_SET_TOGGLE(recbin->contents_list_auto_resize_columns_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_RECYCLE_BIN_OP_CONTENTS_LIST_AUTO_RESIZE_COLUMNS, state);


	/* Settings */
	/* Master Write Protect */
	if(write_protect_changed)
	{
		w = recbin->write_protect_mi;
		GUIMenuItemSetPixmap(
			w,
			(guint8 **)(write_protect ?
				icon_write_protect_on_20x20_xpm : icon_write_protect_off_20x20_xpm)
		);
		GUIMenuItemSetLabel(
			w,
			write_protect ?
				"Master Write Protect: On" :
				"Master Write Protect: Off"
		);
		TOOL_BAR_ITEM_UPDATE(
			EDV_RECYCLE_BIN_OP_WRITE_PROTECT,
			NULL,
			(guint8 **)(write_protect ?
				icon_write_protect_on_20x20_xpm : icon_write_protect_off_20x20_xpm),
			NULL
		);
	}

	/* Delete Method: Recycle */
	state = (delete_method == EDV_DELETE_METHOD_RECYCLE) ?
		TRUE : FALSE;
	MENU_ITEM_SET_TOGGLE(recbin->delete_method_recycle_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_RECYCLE_BIN_OP_DELETE_METHOD_RECYCLE, state);

	/* Delete Method: Purge */
	state = (delete_method == EDV_DELETE_METHOD_PURGE) ? TRUE : FALSE;
	MENU_ITEM_SET_TOGGLE(recbin->delete_method_purge_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_RECYCLE_BIN_OP_DELETE_METHOD_PURGE, state);


	/* Windows
	 *
	 * New Browser
	 */
	w = recbin->window_new_browser_mi;
	i = EDV_GET_I(EDV_CFG_PARM_MAX_BROWSERS);
	sensitive = ((core->total_browsers < i) || (i < 0)) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(w, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_RECYCLE_BIN_OP_NEW_BROWSER, sensitive);

	/* New Image Browser */
	w = recbin->window_new_imbr_mi;
	i = EDV_GET_I(EDV_CFG_PARM_MAX_IMBRS);
	sensitive = ((core->total_imbrs < i) || (i < 0)) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(w, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_RECYCLE_BIN_OP_NEW_IMBR, sensitive);

	/* New Archiver */
	w = recbin->window_new_archiver_mi;
	i = EDV_GET_I(EDV_CFG_PARM_MAX_ARCHIVERS);
	sensitive = ((core->total_archivers < i) || (i < 0)) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(w, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_RECYCLE_BIN_OP_NEW_ARCHIVER, sensitive);


	/* Contents GtkCList right-click menu */
	sensitive = (selected_obj_row > -1) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(
		recbin->contents_clist_recover_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		recbin->contents_clist_purge_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		recbin->contents_clist_rename_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		recbin->contents_clist_chmod_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		recbin->contents_clist_chown_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		recbin->contents_clist_chtime_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		recbin->contents_clist_properties_mi, sensitive
	);

	/* Status Bar */
	edv_status_bar_update_display(recbin->status_bar);


	/* Toplevel GdkWindow's WM icon */
	w = recbin->toplevel;
	if((w->window != NULL) && (recbin_icon_large_data != NULL))
		GUISetWMIcon(
			w->window,
			(guint8 **)recbin_icon_large_data
		);

#undef MENU_ITEM_SET_TOGGLE
#undef TOOL_BAR_ITEM_SET_SENSITIVE
#undef TOOL_BAR_ITEM_SET_TOGGLE
#undef TOOL_BAR_ITEM_UPDATE

	recbin->freeze_count--;
}

/*
 *	Sets the Recycle Bin as busy or ready.
 */
void edv_recycle_bin_set_busy(edv_recbin_struct *recbin, const gboolean busy)
{
	GdkCursor *cur;
	GtkWidget *w;
	EDVCore *core;

	if(recbin == NULL)
		return;

	w = recbin->toplevel;
	core = recbin->core;

	if(busy)
	{
		recbin->busy_count++;
		if(recbin->busy_count > 1)
			return;
		cur = edv_get_cursor(core, EDV_CURSOR_CODE_BUSY);
	}
	else
	{
		recbin->busy_count--;
		if(recbin->busy_count < 0)
			recbin->busy_count = 0;
		if(recbin->busy_count > 0)
			return;
		cur = NULL;
	}

	if(w->window != NULL)
	{
		gdk_window_set_cursor(w->window, cur);
		gdk_flush();
	}
}

/*
 *	Checks if the Recycle Bin is mapped.
 */
gboolean edv_recycle_bin_is_mapped(edv_recbin_struct *recbin)
{
	if(recbin == NULL)
		return(FALSE);

	return(GTK_WIDGET_MAPPED(recbin->toplevel));
}

/*
 *	Maps the Recycle Bin.
 */
void edv_recycle_bin_map(edv_recbin_struct *recbin)
{
	if(recbin == NULL)
		return;

	gtk_widget_show_raise(recbin->toplevel);
	recbin->flags |= EDV_RECYCLE_BIN_MAPPED;
}

/*
 *	Unmaps the Recycle Bin.
 */
void edv_recycle_bin_unmap(edv_recbin_struct *recbin)
{
	if(recbin == NULL)
		return;

	gtk_widget_hide(recbin->toplevel);
	recbin->flags &= ~EDV_RECYCLE_BIN_MAPPED;
}

/*
 *	Deletes the Recycle Bin.
 */
void edv_recycle_bin_delete(edv_recbin_struct *recbin)
{
	if(recbin == NULL)
		return;

	/* Reset the last selected item references */
	recbin->contents_clist_selected_row = -1;


	edv_recycle_bin_unmap(recbin);

	recbin->freeze_count++;

	GTK_WIDGET_DESTROY(recbin->contents_clist_menu);

	GTK_WIDGET_DESTROY(recbin->toplevel);

	/* Delete all Operation IDs */
	edv_recycle_bin_ops_list_delete(recbin->op_ids_list);

	g_free(recbin->contents_list_filter);

	recbin->freeze_count--;

	g_free(recbin);
}
