#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <gtk/gtk.h>

#include "cfg.h"

#include "guiutils.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_property.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_recycle_bin_index.h"
#include "edv_mpeg_audio.h"
#include "edv_id3.h"				/* edv_id3_length_format() */
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "edv_open.h"
#include "prop_page.h"
#include "prop_page_mpeg_audio.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_open_20x20.xpm"


typedef struct _EDVMPEGAudioPropPage	EDVMPEGAudioPropPage;
#define EDV_MPEG_AUDIO_PROP_PAGE(p)	((EDVMPEGAudioPropPage *)(p))


/*
 *	Flags:
 */
typedef enum {
	EDV_MPEG_AUDIO_PROP_PAGE_CURRENTLY_DISPLAYED	\
					= (1 << 0),
	EDV_MPEG_AUDIO_PROP_PAGE_GOT_VALUES	\
					= (1 << 1)
} EDVMPEGAudioPropPageFlags;


/* Check Support */
gboolean edv_mpeg_audio_prop_page_query_create_cb(
	EDVPropPageContext *ctx,
	gint *version_major_rtn,
	gint *version_minor_rtn,
	gint *version_release_rtn,
	gchar **page_name_rtn,
	edv_pixmap_data **pixmap_data_20x20_rtn,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list
);

/* Create */
gpointer edv_mpeg_audio_prop_page_create_cb(
	EDVPropPageContext *ctx,
	GtkWidget *parent
);

/* Update */
void edv_mpeg_audio_prop_page_update_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list,
	const int error_code,
	gpointer data
);

/* Page switched callback */
void edv_mpeg_audio_prop_page_page_switched_cb(
	EDVPropPageContext *ctx,
	const gboolean to,
	gpointer data
);

/* Destroy */
void edv_mpeg_audio_prop_page_destroy_cb(
	EDVPropPageContext *ctx,
	gpointer data
);

/* Callbacks */
static gint edv_mpeg_audio_prop_page_get_values_idle_cb(gpointer data);
static void edv_mpeg_audio_prop_page_open_cb(GtkWidget *widget, gpointer data);

/* Operations */
static void edv_mpeg_audio_prop_page_open(EDVMPEGAudioPropPage *p);

/* Get Values */
static void edv_mpeg_audio_prop_page_get_values(EDVMPEGAudioPropPage *p);

/* Update Widgets */
static void edv_mpeg_audio_prop_page_update_widgets(EDVMPEGAudioPropPage *p);


#define EDV_MPEG_AUDIO_PROP_PAGE_NAME	"MPEG Audio"

#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	MPEG Audio Properties Page:
 */
struct _EDVMPEGAudioPropPage {
	GtkWidget	*toplevel;
	gint		freeze_count;
	EDVPropPageContext	*ctx;
	EDVMPEGAudioPropPageFlags	flags;

	GtkWidget	*version_label,
			*layer_label,

			*time_label,
			*nframes_label,
			*bytes_label,
			*file_size_label,
			*open_btn,

			*bitrate_label,
			*sample_rate_label,
			*channel_mode_label,
			*crc_16_protection_label,
			*padding_label,
			*private_label;

	/* Statistics */
	guint		get_info_toid;
};


/*
 *	Check support callback.
 */
gboolean edv_mpeg_audio_prop_page_query_create_cb(
	EDVPropPageContext *ctx,
	gint *version_major_rtn,
	gint *version_minor_rtn,
	gint *version_release_rtn,
	gchar **page_name_rtn,
	edv_pixmap_data **pixmap_data_20x20_rtn,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list
)
{
	*version_major_rtn = PROG_VERSION_MAJOR;
	*version_minor_rtn = PROG_VERSION_MINOR;
	*version_release_rtn = PROG_VERSION_RELEASE;
	*page_name_rtn = g_strdup(EDV_MPEG_AUDIO_PROP_PAGE_NAME);

	switch(location_type)
	{
	    case EDV_LOCATION_TYPE_VFS:
	    case EDV_LOCATION_TYPE_RECYCLE_BIN:
		if((type == EDV_OBJECT_TYPE_FILE) ||
		   (type == EDV_OBJECT_TYPE_LINK)
		)
		{
			const gchar *name = edv_properties_list_get_s(
				properties_list,
				EDV_PROP_NAME_NAME
			);
			/* Report TRUE if this object name's extension
			 * is an MPEG
			 */
			if(edv_name_has_extension(
				name,
				".mpeg .mpg .mp2 .mp3"
			))
				return(TRUE);
			else
				return(FALSE);
		}
		break;

	    case EDV_LOCATION_TYPE_ARCHIVE:
		break;
	}

	return(FALSE);
}

/*
 *	Create callback.
 */
gpointer edv_mpeg_audio_prop_page_create_cb(
	EDVPropPageContext *ctx,
	GtkWidget *parent
)
{
	const gint	border_major = 5,
			border_minor = 2;
	GtkWidget	*w,
			*parent2, *parent3, *parent4, *parent5;
	EDVMPEGAudioPropPage *p = EDV_MPEG_AUDIO_PROP_PAGE(g_malloc0(
		sizeof(EDVMPEGAudioPropPage)
	));
	if(p == NULL)
		return(NULL);

	p->ctx = ctx;
	p->toplevel = parent;

	p->freeze_count++;

	/* Compatability GtkFrame */
	w = gtk_frame_new("Compatability");
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_hbox_new(FALSE, 2 * border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Version GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_label_new("Version:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	p->version_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Layer GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_label_new("Layer:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	p->layer_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Length GtkFrame */
	w = gtk_frame_new("Length");
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_hbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Time GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_label_new("Time:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	p->time_label = w = gtk_label_new("Scanning...");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Frames GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_label_new("Frames:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	p->nframes_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Bytes GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_label_new("Bytes:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	p->bytes_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* File Size GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_label_new("File Size:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	p->file_size_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* GtkVBox for the Open GtkButton */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_end(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Open GtkButton */
	p->open_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_open_20x20_xpm,
		"Open",
		NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_mpeg_audio_prop_page_open_cb), p
	);
	gtk_widget_show(w);


	/* Stream GtkFrame */
	w = gtk_frame_new("Stream");
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent2 = w;


	w = gtk_hbox_new(FALSE, 2 * border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Bitrate GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_label_new("Bitrate:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	p->bitrate_label = w = gtk_label_new("Scanning...");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	w = gtk_hbox_new(FALSE, 2 * border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Sample Rate GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_label_new("Sample Rate:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	p->sample_rate_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Channel Mode GtkLabel */
	p->channel_mode_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	w = gtk_hbox_new(FALSE, 2 * border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* CRC 16 Protection GtkLabel */
	p->crc_16_protection_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Padding GtkLabel */
	p->padding_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Private GtkLabel */
	p->private_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);




	p->freeze_count--;

	return(p);
}


/*
 *	Update callback.
 */
void edv_mpeg_audio_prop_page_update_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list,
	const int error_code,
	gpointer data
)
{
	EDVMPEGAudioPropPage *p = EDV_MPEG_AUDIO_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	/* Update the displayed information only if this page is
	 * currently displayed or we have previously obtained
	 * values that are now out of date
	 */
	if((p->flags & EDV_MPEG_AUDIO_PROP_PAGE_CURRENTLY_DISPLAYED) ||
	   (p->flags & EDV_MPEG_AUDIO_PROP_PAGE_GOT_VALUES)
	)
	{
		edv_mpeg_audio_prop_page_get_values(p);
	}

	edv_mpeg_audio_prop_page_update_widgets(p);

	p->freeze_count--;
}


/*
 *	Page switched callback.
 */
void edv_mpeg_audio_prop_page_page_switched_cb(
	EDVPropPageContext *ctx,
	const gboolean to,
	gpointer data
)
{
	EDVMPEGAudioPropPage *p = EDV_MPEG_AUDIO_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	if(to)
		p->flags |= EDV_MPEG_AUDIO_PROP_PAGE_CURRENTLY_DISPLAYED;
	else
		p->flags &= ~EDV_MPEG_AUDIO_PROP_PAGE_CURRENTLY_DISPLAYED;

	if(to)
	{
		/* Did not get the image information yet? */
		if(!(p->flags & EDV_MPEG_AUDIO_PROP_PAGE_GOT_VALUES))
		{
			if(p->get_info_toid == 0)
				p->get_info_toid = gtk_idle_add_priority(
					G_PRIORITY_LOW,
					edv_mpeg_audio_prop_page_get_values_idle_cb, p
				);
		}
	}

	p->freeze_count--;
}


/*
 *	Destroy callback.
 */
void edv_mpeg_audio_prop_page_destroy_cb(
	EDVPropPageContext *ctx,
	gpointer data
)
{
	EDVMPEGAudioPropPage *p = EDV_MPEG_AUDIO_PROP_PAGE(data);

	p->freeze_count++;

	p->get_info_toid = GTK_TIMEOUT_REMOVE(p->get_info_toid);

	p->freeze_count--;

	g_free(p);
}


/*
 *	Get info timeout callback.
 */
static gint edv_mpeg_audio_prop_page_get_values_idle_cb(gpointer data)
{
	EDVMPEGAudioPropPage *p = EDV_MPEG_AUDIO_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return(TRUE);

	p->freeze_count++;

	edv_mpeg_audio_prop_page_get_values(p);

	p->get_info_toid = 0;

	p->freeze_count--;

	return(FALSE);
}

/*
 *	Open callback.
 */
static void edv_mpeg_audio_prop_page_open_cb(GtkWidget *widget, gpointer data)
{
	EDVMPEGAudioPropPage *p = EDV_MPEG_AUDIO_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	edv_mpeg_audio_prop_page_open(p);

	p->freeze_count--;
}


/*
 *	Open.
 */
static void edv_mpeg_audio_prop_page_open(EDVMPEGAudioPropPage *p)
{
	const gchar *path;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	const EDVLocationType location_type = edv_prop_page_get_location_type(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);

	p->freeze_count++;

	switch(location_type)
	{
	    case EDV_LOCATION_TYPE_VFS:
		edv_prop_page_set_busy(ctx, TRUE);
		path = edv_properties_list_get_s(
			properties_list,
			EDV_PROP_NAME_PATH
		);
		if(path != NULL)
		{
			GList *paths_list = NULL;
			paths_list = g_list_append(
				paths_list,
				g_strdup(path)
			);
			(void)edv_open(
				core,
				paths_list,
				NULL,		/* Default command */
				toplevel,
				TRUE		/* Verbose */
			);
			g_list_foreach(paths_list, (GFunc)g_free, NULL);
			g_list_free(paths_list);
		}
		edv_prop_page_set_busy(ctx, FALSE);
		break;

	    case EDV_LOCATION_TYPE_RECYCLE_BIN:
	    case EDV_LOCATION_TYPE_ARCHIVE:
		break;
	}

	p->freeze_count--;
}


/*
 *	Opens the MPEG stream from the file, updates the values
 *	displayed on the GtkWidgets, and sets the
 *	EDV_MPEG_AUDIO_PROP_PAGE_GOT_VALUES flag.
 */
static void edv_mpeg_audio_prop_page_get_values(EDVMPEGAudioPropPage *p)
{
	FILE *fp;
	gulong index;
	gchar *path;
	EDVPropPageContext *ctx = p->ctx;
	const EDVLocationType location_type = edv_prop_page_get_location_type(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);
	CfgList *cfg_list = edv_prop_page_get_cfg_list(ctx);

	edv_prop_page_set_busy(ctx, TRUE);
	p->freeze_count++;


	/* Get the path to this object based on the location type */
	path = NULL;
	switch(location_type)
	{
	    case EDV_LOCATION_TYPE_VFS:
		path = STRDUP(edv_properties_list_get_s(
			properties_list,
			EDV_PROP_NAME_PATH
		));
		break;

	    case EDV_LOCATION_TYPE_RECYCLE_BIN:
		index = edv_properties_list_get_ul(
			properties_list,
			EDV_PROP_NAME_INDEX
		);
		path = edv_recycle_bin_index_get_recycled_object_path(
			EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX),
			index
		);
		break;

	    case EDV_LOCATION_TYPE_ARCHIVE:
		break;
	}

	/* Open the MPEG stream from the file */
	fp = fopen(
		(const char *)path,
		"rb"
	);
	if(fp != NULL)
	{
		EDVVFSObject *obj = edv_vfs_object_fstat((gint)fileno(fp));
		EDVMPEGAudioFrameHeader *header = edv_mpeg_audio_stream_next_frame(fp);
		if(header != NULL)
		{
			gint	bitrate_kbps_min = 0,
				bitrate_kbps_max = 0,
				bitrate_kbps_avg = 0;
			gulong	nframes,
				ms,
				mpeg_data_length_bytes;
			gchar *s;
			EDVMPEGAudioVBRHeader *vbr = header->vbr_header;

			/* If the VBR header was not found then measure
			 * the length of MPEG data in the stream
			 */
			if(vbr != NULL)
			{
				gulong sec;

				nframes = vbr->nframes;
				mpeg_data_length_bytes = vbr->nbytes;
				if(header->sample_rate_hz > 0)
					ms = nframes * edv_mpeg_audio_frame_samples(header) / header->sample_rate_hz * 1000l;
				else
					ms = 0l;

				sec = ms / 1000l;
				if(sec != 0l)
					bitrate_kbps_avg = mpeg_data_length_bytes /
						sec * 8l / 1000l;
			}
			else
			{
				nframes = 0l;
				mpeg_data_length_bytes = 0l;
				ms = 0l;
			}
			if(ms == 0l)
			{
				rewind(fp);
				mpeg_data_length_bytes = edv_mpeg_audio_scan_stream_length(
					fp,
					&nframes,
					&ms,
					&bitrate_kbps_min,
					&bitrate_kbps_max
				);
			}

			/* Version */
			s = g_strdup_printf(
				"%i.%i",
				header->version_major,
				header->version_minor
			);
			gtk_label_set_text(
				GTK_LABEL(p->version_label),
				s
			);
			g_free(s);

			/* Layer */
			s = g_strdup_printf(
				"%i",
				header->layer
			);
			gtk_label_set_text(
				GTK_LABEL(p->layer_label),
				s
			);
			g_free(s);

			/* Time Length */
			s = edv_id3_length_format(ms);
			gtk_label_set_text(
				GTK_LABEL(p->time_label),
				s
			);
			g_free(s);

			/* Frames */
			s = STRDUP(edv_str_size_delim(
				nframes
			));
			gtk_label_set_text(
				GTK_LABEL(p->nframes_label),
				s
			);
			g_free(s);

			/* Bytes Length */
			s = STRDUP(edv_str_size_delim(
				mpeg_data_length_bytes
			));
			gtk_label_set_text(
				GTK_LABEL(p->bytes_label),
				s
			);
			g_free(s);

			/* File Size */
			s = g_strdup_printf(
				"%s bytes",
				edv_str_size_delim(
					(obj != NULL) ? obj->size : 0l
				)
			);
			gtk_label_set_text(
				GTK_LABEL(p->file_size_label),
				s
			);
			g_free(s);

			/* Bitrate */
			if(vbr != NULL)
				s = g_strdup_printf(
					"%i kbps Variable",
					bitrate_kbps_avg
				);
			else
				s = g_strdup_printf(
					"%i kbps",
					header->bitrate_kbps
				);
			gtk_label_set_text(
				GTK_LABEL(p->bitrate_label),
				s
			);
			g_free(s);

			/* Sample Rate */
			s = g_strdup_printf(
				"%i Hz",
				header->sample_rate_hz
			);
			gtk_label_set_text(
				GTK_LABEL(p->sample_rate_label),
				s
			);
			g_free(s);

			/* Channel Mode */
			s = NULL;
			switch(header->channel_mode)
			{
			    case EDV_MPEG_AUDIO_CHANNEL_MODE_MONO:
				s = "Mono";
				break;
			    case EDV_MPEG_AUDIO_CHANNEL_MODE_DUAL_MONO:
				s = "Dual Mono";
				break;
			    case EDV_MPEG_AUDIO_CHANNEL_MODE_JOINT_STEREO:
				s = "Joint Stereo";
				break;
			    case EDV_MPEG_AUDIO_CHANNEL_MODE_STEREO:
				s = "Stereo";
				break;
			}
			gtk_label_set_text(
				GTK_LABEL(p->channel_mode_label),
				(s != NULL) ? s : ""
			);

			/* Protection */
			gtk_label_set_text(
				GTK_LABEL(p->crc_16_protection_label),
				(header->crc_16_protection) ? "CRC 16 BITS" : ""
			);

			/* Padding */
			gtk_label_set_text(
				GTK_LABEL(p->padding_label),
				(header->padding) ? "PADDED" : ""
			);


			/* Private */
			gtk_label_set_text(
				GTK_LABEL(p->private_label),
				(header->private) ? "PRIVATE" : ""
			);





			edv_mpeg_audio_frame_header_delete(header);
		}

		edv_vfs_object_delete(obj);

		(void)fclose(fp);
	}



	/* Mark that we have obtained the image information */
	p->flags |= EDV_MPEG_AUDIO_PROP_PAGE_GOT_VALUES;

	g_free(path);

	edv_mpeg_audio_prop_page_update_widgets(p);

	p->freeze_count--;
	edv_prop_page_set_busy(ctx, FALSE);
}


/*
 *	Updates the values displayed on the GtkWidgets.
 */
static void edv_mpeg_audio_prop_page_update_widgets(EDVMPEGAudioPropPage *p)
{
	GtkWidget *w;
	EDVPropPageContext *ctx = p->ctx;
	const EDVLocationType location_type = edv_prop_page_get_location_type(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);
	const gchar *path = edv_properties_list_get_s(
		properties_list,
		EDV_PROP_NAME_PATH
	);
	EDVCore *core = edv_prop_page_get_core(ctx);
	EDVMIMEType *m = edv_mime_types_list_match_path(
		core->mime_types_list,
		path
	);

	p->freeze_count++;

	/* Update the Open GtkButton's label and sensitivity */
	w = p->open_btn;
	if(m != NULL)
	{
		EDVMIMETypeCommand *command = EDV_MIME_TYPE_COMMAND(g_list_nth_data(
			m->commands_list,
			0
		));
		if(command != NULL)
		{
			GUIButtonPixmapUpdate(
				w,
				NULL,
				command->name
			);
		}
		switch(location_type)
		{
		    case EDV_LOCATION_TYPE_VFS:
			gtk_widget_set_sensitive(w, TRUE);
			break;
		    case EDV_LOCATION_TYPE_RECYCLE_BIN:
		    case EDV_LOCATION_TYPE_ARCHIVE:
			gtk_widget_set_sensitive(w, FALSE);
			break;
		}
	}
	else
	{
		GUIButtonPixmapUpdate(
		       w,
		       NULL,
		       "Open"
		);
		switch(location_type)
		{
		    case EDV_LOCATION_TYPE_VFS:
		    case EDV_LOCATION_TYPE_RECYCLE_BIN:
		    case EDV_LOCATION_TYPE_ARCHIVE:
			gtk_widget_set_sensitive(w, FALSE);
			break;
		}
	}

	p->freeze_count--;
}
