#include <errno.h>
#include <gtk/gtk.h>

#include "url.h"
#include "cfg.h"

#include "guiutils.h"
#include "cdialog.h"
#include "fb.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_link.h"
#include "libendeavour2-base/edv_property.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_recycled_obj.h"
#include "libendeavour2-base/edv_recycled_obj_stat.h"
#include "edv_pixmap.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "edv_obj_info_match.h"
#include "edv_utils_gtk.h"
#include "prop_page.h"
#include "prop_page_link.h"
#include "edv_vfs_obj_op.h"
#include "edv_recycled_obj_op.h"
#include "edv_op.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_folder_opened_20x20.xpm"


typedef struct _EDVLinkPropPage		EDVLinkPropPage;
#define EDV_LINK_PROP_PAGE(p)		((EDVLinkPropPage *)(p))


/*
 *	Flags:
 */
typedef enum {
	EDV_LINK_PROP_PAGE_HAS_CHANGES	= (1 << 7)
} EDVLinkPropPageFlags;


/* Check Supported */
gboolean edv_link_prop_page_query_create_cb(
	EDVPropPageContext *ctx,
	gint *version_major_rtn,
	gint *version_minor_rtn,
	gint *version_release_rtn,
	gchar **page_name_rtn,
	edv_pixmap_data **pixmap_data_20x20_rtn,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list
);

/* Create */
gpointer edv_link_prop_page_create_cb(
	EDVPropPageContext *ctx,
	GtkWidget *parent
);

/* Update */
void edv_link_prop_page_update_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list,
	const int error_code,
	gpointer data
);

/* Apply */
gboolean edv_link_prop_page_apply_vfs_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	GList *properties_list,
	gpointer data
);
gboolean edv_link_prop_page_apply_recycle_bin_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	GList *properties_list,
	gpointer data
);

/* Destroy */
void edv_link_prop_page_apply_destroy_cb(
	EDVPropPageContext *ctx,
	gpointer data
);

/* Callbacks */
static gint edv_link_prop_page_auto_link_target_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void edv_link_prop_page_auto_link_target_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info,
	guint t,
	gpointer data
);
static void edv_link_prop_page_changed_cb(GtkWidget *widget, gpointer data);
static void edv_link_prop_page_browse_cb(GtkWidget *widget, gpointer data);
static void edv_link_prop_page_make_relative_path_cb(GtkWidget *widget, gpointer data);
static void edv_link_prop_page_target_properties_cb(GtkWidget *widget, gpointer data);

/* Auto Link Target Draw */
static void edv_link_prop_page_auto_link_target_draw(EDVLinkPropPage *p);

/* Get Link Statistics */
static gboolean edv_link_prop_page_get_statistics(
	const gchar *path,
	const gchar *target,
	gchar **relativity_rtn,
	gint *nhops_rtn,
	gulong *size_rtn
);

/* Update Widgets */
static void edv_link_prop_page_update_widgets(EDVLinkPropPage *p);


#define EDV_LINK_PROP_PAGE_NAME		"Link"

#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Link Page:
 */
struct _EDVLinkPropPage {
	GtkWidget	*toplevel;
	GdkGC		*gc;
	gint		freeze_count;
	EDVPropPageContext	*ctx;
	EDVLinkPropPageFlags	flags;

	GtkWidget	*source_icon_pm,
			*source_label,
			*target_icon_pm,
			*target_entry,
			*target_make_relative_path_btn,
			*target_properties_btn,
			*auto_link_target_da,
			*statistics_frame,
			*statistics_label;
	EDVPixmap	*auto_link_icon;
};


/*
 *	Check supported callback.
 */
gboolean edv_link_prop_page_query_create_cb(
	EDVPropPageContext *ctx,
	gint *version_major_rtn,
	gint *version_minor_rtn,
	gint *version_release_rtn,
	gchar **page_name_rtn,
	edv_pixmap_data **pixmap_data_20x20_rtn,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list
)
{
	*version_major_rtn = PROG_VERSION_MAJOR;
	*version_minor_rtn = PROG_VERSION_MINOR;
	*version_release_rtn = PROG_VERSION_RELEASE;
	*page_name_rtn = g_strdup(EDV_LINK_PROP_PAGE_NAME);

	switch(location_type)
	{
	    case EDV_LOCATION_TYPE_VFS:
	    case EDV_LOCATION_TYPE_RECYCLE_BIN:
	    case EDV_LOCATION_TYPE_ARCHIVE:
		if(type == EDV_OBJECT_TYPE_LINK)
		{
			return(TRUE);
		}
		break;
	}

	return(FALSE);
}

/*
 *	Create callback.
 */
gpointer edv_link_prop_page_create_cb(
	EDVPropPageContext *ctx,
	GtkWidget *parent
)
{
	const gint	border_major = 5,
			border_minor = 2;
	gchar *icons_path;
	GtkWidget	*w,
			*parent2, *parent3, *parent4;
	EDVPixmap *default_icon;
	EDVMIMEType *m;
	const EDVLocationType location_type = edv_prop_page_get_location_type(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);
	CfgList *cfg_list = edv_prop_page_get_cfg_list(ctx);
	EDVLinkPropPage *p = EDV_LINK_PROP_PAGE(g_malloc0(
		sizeof(EDVLinkPropPage)
	));
	if(p == NULL)
		return(NULL);

	/* Get the icons directory path */
	icons_path = g_strconcat(
	    EDV_GET_S(EDV_CFG_PARM_DIR_GLOBAL),
	    G_DIR_SEPARATOR_S,
	    EDV_NAME_ICONS_SUBDIR,
	    NULL
	);

	/* Get the default icon */
	m = edv_mime_types_list_match_type(
		core->mime_types_list,
		NULL,				/* No index return */
		EDV_MIME_TYPE_TYPE_INODE_UNKNOWN,
		FALSE				/* Not case sensitive */
	);
	if(m != NULL)
	{
		edv_mime_type_realize(m, FALSE);
		default_icon = edv_pixmap_ref(m->small_icon[
			EDV_MIME_TYPE_ICON_STATE_STANDARD
		]);
	}
	else
		default_icon = NULL;

	p->ctx = ctx;
	p->toplevel = parent;

	p->freeze_count++;


	/* Source GtkFrame */
	w = gtk_frame_new("Source");
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Source GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Source Icon GtkPixmap */
	p->source_icon_pm = w = edv_pixmap_new_gtk_pixmap(default_icon);
	if(w != NULL)
	{
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
	}

	/* Source GtkLabel */
	p->source_label = w = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Target GtkFrame */
	w = gtk_frame_new("Target");
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Target GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Target Icon GtkPixmap */
	p->target_icon_pm = w = edv_pixmap_new_gtk_pixmap(default_icon);
	if(w != NULL)
	{
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
	}

	/* Target GtkEntry */
	p->target_entry = w = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	edv_entry_set_dnd(core, w);
	edv_entry_set_complete_path(core, w);
	gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(edv_link_prop_page_changed_cb), p
	);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Entre el valor del destino de la conexin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Entrer la valeur de destination du lien"
#elif defined(PROG_LANGUAGE_GERMAN)
"Tragen Sie das Reiseziel der valueEnter von dem Kettenglied\
 das Reiseziel der Wert von dem Kettenglied ein"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Entrare il valueEnter di destinazione della maglia il valore di\
 destinazione della maglia"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga de bestemming van de schakel valueenter de bestemming van de\
 schakel waarde binnen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Entre o valueEnter de destino do elo o valor de destino do elo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"G inn i leddets destinasjon valueenter leddets\
 destinasjonsverdi"
#else
"Enter the link's target value"
#endif
	);
	gtk_widget_show(w);

	if(location_type == EDV_LOCATION_TYPE_VFS)
	{
		/* Browse GtkButton */
		w = GUIButtonPixmap(
			(guint8 **)icon_folder_opened_20x20_xpm
		);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_signal_connect(
			GTK_OBJECT(w), "clicked",
			GTK_SIGNAL_FUNC(edv_link_prop_page_browse_cb), p
		);
		GUISetWidgetTip(w, "Browse");
		gtk_widget_show(w);
	}

	/* Make Relative Path, Autolink, and Link Target Properties */
	if(location_type == EDV_LOCATION_TYPE_VFS)
	{
		gchar *path;

		w = gtk_hbox_new(FALSE, border_minor);
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent3 = w;

		w = gtk_vbox_new(FALSE, border_minor);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent4 = w;

		/* Make Relative Path button */
		p->target_make_relative_path_btn = w = gtk_button_new_with_label(
			"Make Relative Path"
		);
		gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
		gtk_signal_connect(
			GTK_OBJECT(w), "clicked",
			GTK_SIGNAL_FUNC(edv_link_prop_page_make_relative_path_cb), p
		);
		GUISetWidgetTip(
			w,
"Click this to convert the target value from an absolute path\
 into a relative path"
		);
		gtk_widget_show(w);


		w = gtk_vbox_new(FALSE, border_minor);
		gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
		gtk_widget_show(w);
		parent4 = w;



		w = gtk_frame_new(NULL);
		gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
		gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
		gtk_widget_show(w);
		parent4 = w;

		/* Auto Link Target GtkDrawingArea */
	        p->auto_link_target_da = w = gtk_drawing_area_new();
	        gtk_widget_set_usize(w, 48, 48);
	        gtk_widget_add_events(
	                w,
	                GDK_STRUCTURE_MASK | GDK_EXPOSURE_MASK |
	                GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK
	        );
	        gtk_signal_connect(
	                GTK_OBJECT(w), "configure_event",
	                GTK_SIGNAL_FUNC(edv_link_prop_page_auto_link_target_event_cb), p
	        );
	        gtk_signal_connect(
	                GTK_OBJECT(w), "expose_event",
	                GTK_SIGNAL_FUNC(edv_link_prop_page_auto_link_target_event_cb), p
	        );
	        gtk_container_add(GTK_CONTAINER(parent4), w);
		if(w != NULL)
		{
			const GtkTargetEntry dnd_tar_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,	0,	EDV_DND_INFO_TEXT_PLAIN},
{GUI_TARGET_NAME_TEXT_URI_LIST,	0,	EDV_DND_INFO_TEXT_URI_LIST},
{GUI_TARGET_NAME_STRING,	0,	EDV_DND_INFO_STRING}
			};
			GUIDNDSetTar(
		                w,
		                dnd_tar_types,
		                sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
		                GDK_ACTION_COPY,	/* Actions */
		                GDK_ACTION_COPY,	/* Default action if same */
				GDK_ACTION_COPY,	/* Default action */
		                edv_link_prop_page_auto_link_target_drag_data_received_cb,
				p,
		                TRUE			/* Highlight */
		        );
		}
		GUISetWidgetTip(
			w,
"Drag an object here to automatically set this link's target value\
 to refer to it"
		);
	        gtk_widget_show(w);

		/* Autolink icon */
		path = g_strconcat(
			icons_path,
			G_DIR_SEPARATOR_S,
			"icon_wand_48x48.xpm",
			NULL
		);
		p->auto_link_icon = edv_pixmap_new_from_file(path);
		g_free(path);


		w = gtk_vbox_new(FALSE, border_minor);
		gtk_box_pack_end(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent4 = w;

		/* Link Target Properties button */
		p->target_properties_btn = w = gtk_button_new_with_label(
			"Target Properties"
		);
		gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
		gtk_signal_connect(
			GTK_OBJECT(w), "clicked",
			GTK_SIGNAL_FUNC(edv_link_prop_page_target_properties_cb), p
		);
		GUISetWidgetTip(
			w,
"Click this to view the properties of the target object"
		);
		gtk_widget_show(w);
	}

	/* Statistics */
	if(location_type == EDV_LOCATION_TYPE_VFS)
	{
		p->statistics_frame = w = gtk_frame_new("Statistics");
		gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
		gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
		gtk_widget_show(w);
		parent2 = w;

		w = gtk_vbox_new(FALSE, border_major);
		gtk_container_add(GTK_CONTAINER(parent2), w);
		gtk_container_border_width(GTK_CONTAINER(w), border_major);
		gtk_widget_show(w);
		parent2 = w;

		w = gtk_hbox_new(FALSE, border_major);
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent3 = w;

		w = gtk_label_new(
"Relativity:\n\
Total Hops:\n\
Size:"
		);
	        gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);

		p->statistics_label = w = gtk_label_new(
"Unknown\n\
0"
		);
	        gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
	}

	p->freeze_count--;

	(void)edv_pixmap_unref(default_icon);
	g_free(icons_path);

	return(p);
}


/*
 *	Update callback.
 */
void edv_link_prop_page_update_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list,
	const int error_code,
	gpointer data
)
{
	EDVLinkPropPage *p = EDV_LINK_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	edv_link_prop_page_update_widgets(p);

	p->freeze_count--;
}

/*
 *	Apply VFS object callback.
 */
gboolean edv_link_prop_page_apply_vfs_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	GList *properties_list,
	gpointer data
)
{
	gboolean	status = FALSE,
			yes_to_all = FALSE;
	const gchar *path;
	GtkWidget       *w,
			*toplevel = edv_prop_page_get_toplevel(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);
	EDVLinkPropPage *p = EDV_LINK_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return(status);

	p->freeze_count++;

	/* Do not apply if we did not make any changes */
	if(!(p->flags & EDV_LINK_PROP_PAGE_HAS_CHANGES))
	{
		p->freeze_count--;
		return(status);
	}

	/* Get the values that are needed to refer to the object that
	 * we want to apply this page's values to
	 */
	path = edv_properties_list_get_s(
		properties_list,
		EDV_PROP_NAME_PATH
	);
	if(path == NULL)
	{
		p->freeze_count--;
		return(status);
	}

	/* Link Target */
	w = p->target_entry;
	if(w != NULL)
	{
		gchar	*cur_target = STRDUP(edv_properties_list_get_s(
				properties_list,
				EDV_PROP_NAME_LINK_TARGET
			)),
			*new_target = STRDUP(gtk_entry_get_text(GTK_ENTRY(w)));

		/* Links may not have empty target values, if the target
		 * value is empty then set it to "undefined"
		 */
		if(STRISEMPTY(cur_target))
		{
			g_free(cur_target);
			cur_target = g_strdup("undefined");
		}
		if(STRISEMPTY(new_target))
		{
			g_free(new_target);
			new_target = g_strdup("undefined");
		}

		/* Need to change the value? */
		if(((cur_target != NULL) && (new_target != NULL)) ?
			strcmp((const char *)cur_target, (const char *)new_target) : TRUE
		)
		{
			/* Set the new target value */
			const gchar *error_msg;
			GList *modified_paths_list;

			/* Set the new target value */
			if(edv_vfs_object_op_relink(
				core,
				path,		/* Link */
				new_target,	/* Target */
				&modified_paths_list,
				toplevel,
				TRUE,
				TRUE,
				&yes_to_all
			) == 0)
				status = TRUE;

			/* Unmap the progress dialog */
			ProgressDialogBreakQuery(FALSE);
			ProgressDialogSetTransientFor(NULL);

			/* Check for errors */
			error_msg = edv_vfs_object_op_get_error(core);
			if(!STRISEMPTY(error_msg))
			{
				/* Report the error */
				edv_play_sound_error(core);
				edv_message_error(
					"Change Link Target Error",
					error_msg,
					NULL,
					toplevel
				);
			}

			/* Delete the modified paths list */
			if(modified_paths_list != NULL)
			{
				g_list_foreach(
					modified_paths_list,
					(GFunc)g_free,
					NULL
				);
				g_list_free(modified_paths_list);
			}
		}

		g_free(cur_target);
		g_free(new_target);
	}

	/* Remove our has changes marker */
	p->flags &= ~EDV_LINK_PROP_PAGE_HAS_CHANGES;

	p->freeze_count--;

	return(status);
}

/*
 *      Apply recycled object callback.
 */
gboolean edv_link_prop_page_apply_recycle_bin_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	GList *properties_list,
	gpointer data
)
{
	gboolean	status = FALSE,
			yes_to_all = FALSE;
	gulong index;
	GtkWidget       *w,
			*toplevel = edv_prop_page_get_toplevel(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);
	EDVLinkPropPage *p = EDV_LINK_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return(status);

	p->freeze_count++;

	/* Do not apply if we did not make any changes */
	if(!(p->flags & EDV_LINK_PROP_PAGE_HAS_CHANGES))
	{
		p->freeze_count--;
		return(status);
	}

	/* Get the values that are needed to refer to the object that
	 * we want to apply this page's values to
	 */
	index = edv_properties_list_get_ul(
		properties_list,
		EDV_PROP_NAME_INDEX
	);
	if(index == 0l)
	{
		p->freeze_count--;
		return(status);
	}

	/* Link Target */
	w = p->target_entry;
	if(w != NULL)
	{
		gchar	*cur_target = STRDUP(edv_properties_list_get_s(
			properties_list,
			EDV_PROP_NAME_LINK_TARGET
		)),
			*new_target = STRDUP(gtk_entry_get_text(GTK_ENTRY(w)));

		/* Links may not have empty target values, if the target
		 * value is empty then set it to "undefined"
		 */
		if(STRISEMPTY(cur_target))
		{
			g_free(cur_target);
			cur_target = g_strdup("undefined");
		}
		if(STRISEMPTY(new_target))
		{
			g_free(new_target);
			new_target = g_strdup("undefined");
		}

		/* Need to change the value? */
		if(((cur_target != NULL) && (new_target != NULL)) ?
			strcmp((const char *)cur_target, (const char *)new_target) : TRUE
		)
		{
			const gchar *error_msg;
			GList *modified_indicies_list;

			/* Set the new target value */
			if(edv_recycled_object_op_relink(
				core,
				index,
				new_target,
				&modified_indicies_list,
				toplevel,
				FALSE,			/* Do not show progress */
				TRUE,			/* Interactive */
				&yes_to_all
			) == 0)
				status = TRUE;

			/* Unmap the progress dialog */
			ProgressDialogBreakQuery(FALSE);
			ProgressDialogSetTransientFor(NULL);

			/* Check for errors */
			error_msg = edv_recycled_object_op_get_error(core);
			if(!STRISEMPTY(error_msg))
			{
				/* Report the error */
				edv_play_sound_error(core);
				edv_message_error(
					"Change Link Target Error",
					error_msg,
					NULL,
					toplevel
				);
			}

			g_list_free(modified_indicies_list);
		}

		g_free(cur_target);
		g_free(new_target);
	}

	/* Remove our has changes marker */
	p->flags &= ~EDV_LINK_PROP_PAGE_HAS_CHANGES;

	p->freeze_count--;

	return(status);
}

/*
 *      Destroy callback.
 */
void edv_link_prop_page_apply_destroy_cb(
	EDVPropPageContext *ctx,
	gpointer data
)
{
	EDVLinkPropPage *p = EDV_LINK_PROP_PAGE(data);
	(void)GDK_GC_UNREF(p->gc);
	(void)edv_pixmap_unref(p->auto_link_icon);
	g_free(p);
}


/*
 *	Auto link target GtkDrawingArea event signal callback.
 */
static gint edv_link_prop_page_auto_link_target_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	EDVLinkPropPage *p = EDV_LINK_PROP_PAGE(data);

	switch((gint)event->type)
	{
	    case GDK_CONFIGURE:
		break;
	    case GDK_EXPOSE:
		edv_link_prop_page_auto_link_target_draw(p);
		status = TRUE;
		break;
	}

	return(status);
}

/*
 *	Auto link target GtkDrawingArea "drag_data_received" signal
 *	callback.
 */
static void edv_link_prop_page_auto_link_target_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info,
	guint t,
	gpointer data
)
{
	EDVLinkPropPage *p = EDV_LINK_PROP_PAGE(data);
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);
	if((widget == NULL) || (dc == NULL) || (p == NULL))
		return;

	/* No source data? */
	if(selection_data == NULL)
		return;
	if(selection_data->length <= 0)
		return;

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	if((info == EDV_DND_INFO_TEXT_PLAIN) ||
	   (info == EDV_DND_INFO_TEXT_URI_LIST) ||
	   (info == EDV_DND_INFO_STRING)
	)
	{
		GList *urls_list = url_decode(
			(const guint8 *)selection_data->data,
			selection_data->length
		);
		if(urls_list != NULL)
		{
			const gchar	*path = edv_properties_list_get_s(
				properties_list,
				EDV_PROP_NAME_PATH
			),
					*target;
			GtkEntry *entry = GTK_ENTRY(p->target_entry);
			GList *glist;
			URLStruct *url;
			for(glist = urls_list;
			    glist != NULL;
			    glist = g_list_next(glist)
			)
			{
				url = URL(glist->data);
				if(url == NULL)
					continue;

#if 0
/* TODO handle different protocols */
				if(url->protocol != NULL)
				{

				}
#endif

				target = url->path;
				if(STRISEMPTY(target))
					continue;

				if(g_path_is_absolute(target))
				{
					/* Query the user to make the
					 * path a relative path
					 */
					gint response;
					gchar *msg = g_strdup_printf(
"Would you like to use a relative path from:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s",
						path,
						target
					);
					edv_play_sound_question(core);
					CDialogSetTransientFor(toplevel);
					response = CDialogGetResponse(
						"Set Relative Path",
						msg,
						NULL,
						CDIALOG_ICON_QUESTION,
						CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
						CDIALOG_BTNFLAG_YES
					);
					g_free(msg);
					CDialogSetTransientFor(NULL);
					if(response == CDIALOG_RESPONSE_YES)
					{
						/* Make the selected path a relative path */
						gchar *relative_target = edv_path_plot_relative(
							path,
							target
						);
						gtk_entry_set_text(
							entry,
							(relative_target != NULL) ? relative_target : target
						);
						g_free(relative_target);
						gtk_entry_set_position(entry, -1);
					}
					else
					{
						/* Set the link's target */
						gtk_entry_set_text(entry, target);
						gtk_entry_set_position(entry, -1);
					}
				}
				else
				{
					/* Set the link's target */
					gtk_entry_set_text(entry, target);
					gtk_entry_set_position(entry, -1);
				}
			}

			/* Mark that we have made changes */
			p->flags |= EDV_LINK_PROP_PAGE_HAS_CHANGES;
			edv_prop_page_set_has_changes(
				ctx,
				TRUE
			);

			/* Since this call freezes our page, we need
			 * explicitly update the values displayed on
			 * our page
			 */
			edv_link_prop_page_update_widgets(p);

			g_list_foreach(
				urls_list,
				(GFunc)url_delete,
				NULL
			);
			g_list_free(urls_list);
		}
	}

	p->freeze_count--;
}

/*
 *	Any GtkWidget "changed" signal callback.
 */
static void edv_link_prop_page_changed_cb(GtkWidget *widget, gpointer data)
{
	EDVLinkPropPage *p = EDV_LINK_PROP_PAGE(data);
	EDVPropPageContext *ctx = p->ctx;

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;                      /* Ignore our own update */

	/* Ignore if we have already made changes */
	if(p->flags & EDV_LINK_PROP_PAGE_HAS_CHANGES)
	{
		/* Since we are frozen we need to explicitly update
		 * the values displayed on our page
		 *
		 * This is needed in the changed callback because
		 * we want to update the displayed values each time
		 * the user changes something
		 */
		edv_link_prop_page_update_widgets(p);

		p->freeze_count--;
		return;
	}

	/* Mark that we have made changes */
	p->flags |= EDV_LINK_PROP_PAGE_HAS_CHANGES;

	/* Notify the EDVPropDlg that changes have been made and
	 * emit an update (which we will ignore)
	 */
	edv_prop_page_set_has_changes(
		ctx,
		TRUE
	);

	/* Since we are frozen we need to explicitly update the
	 * values displayed on our page
	 *
	 * This is needed in the changed callback because we want
	 * to update the displayed values each time the user
	 * changes something
	 */
	edv_link_prop_page_update_widgets(p);

	p->freeze_count--;
}

/*
 *      Link value browse callback.
 */
static void edv_link_prop_page_browse_cb(GtkWidget *widget, gpointer data)
{
	gboolean response;
	gint            npaths = 0,
			nftypes = 0;
	gchar           *cur_path,
			**paths_list = NULL;
	const gchar     *s,
			*path;
	GtkEntry *entry;
	fb_type_struct  **ftypes_list = NULL,
			*ftype_rtn = NULL;
	EDVLinkPropPage *p = EDV_LINK_PROP_PAGE(data);
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);

	if(p->freeze_count > 0)
		return;

	/* Get the starting location
	 *
	 * If the current link's target value is an absolute
	 * path then use its parent as the starting location,
	 * otherwise use the parent of the link
	 */
	entry = GTK_ENTRY(p->target_entry);

	edv_prop_page_set_busy(ctx, TRUE);
	p->freeze_count++;

	path = edv_properties_list_get_s(
		properties_list,
		EDV_PROP_NAME_PATH
	);
	s = gtk_entry_get_text(entry);
	if(s != NULL)
	{
		if(g_path_is_absolute(s))
			cur_path = g_dirname(s);
		else if(path != NULL)
			cur_path = g_dirname(path);
		else
			cur_path = NULL;
	}
	else if(path != NULL)
	{
		cur_path = g_dirname(path);
	}
	else
	{
		cur_path = NULL;
	}

	/* Create the file types list */
	FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		"*.*", "All Files"
	);

	/* Query the user for the link's value */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
		"Set Link Value",
		"Set", "Cancel",
		cur_path,
		ftypes_list, nftypes,
		&paths_list, &npaths,
		&ftype_rtn
	);
	g_free(cur_path);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(response && (npaths > 0))
	{
		gchar *target = STRDUP(paths_list[npaths - 1]);
		if(target != NULL)
		{
			/* Query the user to make the selected path a
			 * relative path
			 */
			gint response;
			gchar *msg = g_strdup_printf(
"Would you like to use a relative path from:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s",
				path,
				target
			);
			edv_play_sound_question(core);
			CDialogSetTransientFor(toplevel);
			response = CDialogGetResponse(
				"Set Relative Path",
				msg,
				NULL,
				CDIALOG_ICON_QUESTION,
				CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
				CDIALOG_BTNFLAG_YES
			);
			g_free(msg);
			CDialogSetTransientFor(NULL);
			if(response == CDIALOG_RESPONSE_YES)
			{
				/* Make the selected path a relative path */
				gchar *relative_target = edv_path_plot_relative(
					path,
					target
				);
				gtk_entry_set_text(
					entry,
					(relative_target != NULL) ?
						relative_target : target
				);
				g_free(relative_target);
				gtk_entry_set_position(entry, -1);
			}
			else
			{
				/* Set the link's target */
				gtk_entry_set_text(
					entry,
					target
				);
				gtk_entry_set_position(
					entry,
					-1
				);
			}

			/* Mark that we have made changes */
			p->flags |= EDV_LINK_PROP_PAGE_HAS_CHANGES;
			edv_prop_page_set_has_changes(
				ctx,
				TRUE
			);

			/* Since this call freezes our page, we need
			 * explicitly update the values displayed on
			 * our page
			 */
			edv_link_prop_page_update_widgets(p);

			g_free(target);
		}
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	p->freeze_count--;
	edv_prop_page_set_busy(ctx, FALSE);
}

/*
 *      Link Make Relative Path callback.
 */
static void edv_link_prop_page_make_relative_path_cb(GtkWidget *widget, gpointer data)
{
	const gchar *path;
	gchar		*target,
			*relative_target;
	GtkEntry *entry;
	EDVLinkPropPage *p = EDV_LINK_PROP_PAGE(data);
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);

	if(p->freeze_count > 0)
		return;

	edv_prop_page_set_busy(ctx, TRUE);
	p->freeze_count++;

	path = edv_properties_list_get_s(
		properties_list,
		EDV_PROP_NAME_PATH
	);
	if(path == NULL)
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	entry = GTK_ENTRY(p->target_entry);

	/* Get the link's current target value */
	target = STRDUP(gtk_entry_get_text(entry));
	if(target == NULL)
	{
		edv_play_sound_warning(core);
		edv_message_warning(
			"Make Relative Path Failed",
"Unable to obtain the current target value.",
			NULL,
			toplevel
		);
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	/* Check if the target value is already a relative path */
	if(!g_path_is_absolute(target))
	{
		edv_play_sound_warning(core);
		edv_message_warning(
			"Make Relative Path Failed",
"The target value is already a relative path.\n\
\n\
If you want to remake the relative path then set\n\
the target value to an absolute path and try again.",
			NULL,
			toplevel
		);
		g_free(target);
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	/* Set the link's target as a relative path */
	relative_target = edv_path_plot_relative(
		path,
		target
	);
	if(relative_target != NULL)
	{
		gtk_entry_set_text(
			entry,
			relative_target
		);
		gtk_entry_set_position(
			entry,
			-1
		);
		g_free(relative_target);
	}
	else
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to plot a relative path from:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",
			path,
			target,
			g_strerror(error_code)
		);
		edv_play_sound_warning(core);
		edv_message_warning(
			"Make Relative Path Failed",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
		g_free(target);
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	g_free(target);

	/* Mark that we have made changes */
	p->flags |= EDV_LINK_PROP_PAGE_HAS_CHANGES;
	edv_prop_page_set_has_changes(
		ctx,
		TRUE
	);

	/* Since this call freezes our page, we need explicitly
	 * update the values displayed on our page
	 */
	edv_link_prop_page_update_widgets(p);

	p->freeze_count--;
	edv_prop_page_set_busy(ctx, FALSE);
}

/*
 *      Link target properties callback.
 */
static void edv_link_prop_page_target_properties_cb(GtkWidget *widget, gpointer data)
{
	const gchar *value;
	GtkEntry *entry;
	EDVLinkPropPage *p = EDV_LINK_PROP_PAGE(data);
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	const EDVLocationType location_type = edv_prop_page_get_location_type(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);

	if(p->freeze_count > 0)
		return;

	entry = GTK_ENTRY(p->target_entry);

	edv_prop_page_set_busy(ctx, TRUE);
	p->freeze_count++;

	/* Get the link's target value */
	value = gtk_entry_get_text(entry);
	if(STRISEMPTY(value))
	{
		edv_play_sound_warning(core);
		edv_message_warning(
			"Target Properties Failed",
"Unable to obtain the current target value.",
			NULL,
			toplevel
		);
	}
	else if(location_type == EDV_LOCATION_TYPE_VFS)
	{
		gchar *target_full_path;

		/* Is the target value an absolute path? */
		if(g_path_is_absolute(value))
		{
			target_full_path = g_strdup(value);
		}
		else
		{
			/* Not an absolute path, prefix the parent path of
			 * the link to the target path
			 */
			const gchar *path = edv_properties_list_get_s(
				properties_list,
				EDV_PROP_NAME_PATH
			);
			gchar *parent_path = (path != NULL) ?
				g_dirname(path) : NULL;
			target_full_path = g_strconcat(
				parent_path,
				G_DIR_SEPARATOR_S,
				value,
				NULL
			);
			g_free(parent_path);
		}

		/* Got the full target path? */
		if(target_full_path != NULL)
		{
			/* Remove tailing deliminators and reduce any
			 * occurances of ../ and ./
			 */
			edv_path_simplify(target_full_path);

			/* Create a new Properties Dialog to display
			 * the target object
			 */
			edv_new_properties_dialog_vfs(
				core,
				target_full_path,
				NULL,		/* Default page */
				NULL,		/* No extended properties list */
				core->geometry_flags,
				(core->geometry_flags != 0) ? &core->geometry : NULL,
				toplevel
			);

			g_free(target_full_path);
		}
		else
		{
			const gchar *path = edv_properties_list_get_s(
				properties_list,
				EDV_PROP_NAME_PATH
			);
			gchar *parent_path = (path != NULL) ?
				g_dirname(path) : NULL;
			gchar *msg = g_strdup_printf(
"Unable to form an absolute path to the target:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s",
				value,
				parent_path
			);
			g_free(parent_path);
			edv_play_sound_warning(core);
			edv_message_warning(
				"Target Properties Failed",
				msg,
				NULL,
				toplevel
			);
			g_free(msg);
		}
	}

	p->freeze_count--;
	edv_prop_page_set_busy(ctx, FALSE);
}


/*
 *	Redraws the auto link target GtkDrawingArea.
 */
static void edv_link_prop_page_auto_link_target_draw(EDVLinkPropPage *p)
{
	gint width, height;
	GtkWidget *w = p->auto_link_target_da;
	GdkWindow *window = w->window;
	GdkDrawable *drawable = (GdkDrawable *)window;
	const GtkStateType state = GTK_WIDGET_STATE(w);
	GdkGC *gc;
	GtkStyle *style = gtk_widget_get_style(w);

	gdk_window_get_size((GdkWindow *)drawable, &width, &height);
	if((width <= 0) || (height <= 0))
		return;

	gc = p->gc;
	if(gc == NULL)
		gc = gdk_gc_new((GdkWindow *)drawable);
	if(gc == NULL)
		return;

	/* Draw the background */
	gdk_gc_set_foreground(gc, &style->base[state]);
	gdk_draw_rectangle(
		drawable,
		gc,
		TRUE,				/* Fill */
		0, 0,
		width, height
	);

	/* Draw the icon */
	if(p->auto_link_icon != NULL)
	{
		EDVPixmap *icon = p->auto_link_icon;
		const gint	x = (width - icon->width) / 2,
				y = (height - icon->height) / 2;
		gdk_gc_set_clip_mask(gc, icon->mask);
		gdk_gc_set_clip_origin(gc, x, y);
		gdk_draw_pixmap(
			drawable,
			gc,
			(GdkDrawable *)icon->pixmap,
			0, 0,
			x, y,
			icon->width, icon->height
		);
		gdk_gc_set_clip_mask(gc, NULL);
	}

#if 0
	if(style->font != NULL)
	{
		const gchar *s;
		GdkFont *font = style->font;
		const gint font_height = font->ascent + font->descent;
		GdkTextBounds b;

		gdk_gc_set_foreground(gc, &style->text[state]);
		gdk_gc_set_font(gc, font);

		s = "Autolink";
		gdk_string_bounds(font, s, &b);
		gdk_draw_string(
			drawable,
			font,
			gc,
			((width - b.width) / 2) - b.lbearing,
			(gint)(height * 0.25f) - (font_height / 2) + font->ascent,
			s
		);

		s = "Drag an object here";
		gdk_string_bounds(font, s, &b);
		gdk_draw_string(
			drawable,
			font,
			gc,
			((width - b.width) / 2) - b.lbearing,
			(gint)(height * 0.75f) - (font_height / 2) + font->ascent,
			s
		);
	}
#endif

	/* Send drawable to window if drawable is not the window */
	if(drawable != (GdkDrawable *)window)
		gdk_draw_pixmap(
			(GdkDrawable *)window,
			style->black_gc,
			drawable,
			0, 0,
			0, 0,
			width, height
		);
}

/*
 *	Get link statistics.
 */
static gboolean edv_link_prop_page_get_statistics(
	const gchar *path,
	const gchar *target,
	gchar **relativity_rtn,
	gint *nhops_rtn,
	gulong *size_rtn
)
{
	gchar		*parent_path,
			*cur_path,
			*relative_target;
	EDVVFSObject *obj;

	*relativity_rtn = NULL;
	*nhops_rtn = 0;
	*size_rtn = 0l;

	/* Get the target of the specified link */
	if(target == NULL)
		return(FALSE);

	*size_rtn = (gulong)strlen((const char *)target);

	/* Get the full path to the first link destination, use
	 * edv_path_evaluatel() since target may be either a full
	 * or relative path
	 */
	parent_path = g_dirname(path);
	cur_path = edv_path_evaluate(
		parent_path,			/* Parent */
		target				/* Full or relative child path */
	);
	g_free(parent_path);

	/* Hop through each link until the final destination is
	 * reached, count the number of hops and record the final
	 * destination as cur_path or set cur_path to NULL if the
	 * final destination could not be reached because there was
	 * a non-existant location (broken link)
	 */
	while(cur_path != NULL)
	{
		/* If this location does not exist locally then stop
		 * and do not count this hop
		 */
		if(!edv_path_lexists(cur_path))
		{
			/* Break indicating a broken link */
			g_free(cur_path);
			cur_path = NULL;
			continue;
		}

		/* Count this hop */
		*nhops_rtn = *nhops_rtn + 1;

		/* Get this object's local statistics */
		obj = edv_vfs_object_lstat(cur_path);
		if(obj == NULL)
		{
			/* Unable to get this object's local
			 * statistics but it was reported to exist
			 * so report this as the last hop
			 */
			break;
		}

		/* If this location is a link then get its destination
		 * and continue on to the next location
		 */
		if(EDV_VFS_OBJECT_IS_LINK(obj))
		{
			gchar *next_target_full = edv_link_get_target_full(cur_path);
			g_free(cur_path);
			cur_path = next_target_full;
			edv_vfs_object_delete(obj);
			continue;
		}

		/* Final destination reached and it exists */
		edv_vfs_object_delete(obj);
		break;
	}

	/* Missing location (broken link)? */
	if(cur_path == NULL)
		return(FALSE);

	/* Plot a relative path from the specified location to the
	 * final target location
	 */
	relative_target = edv_path_plot_relative(
		path,
		cur_path
	);
	if(relative_target != NULL)
	{
		/* Break down the relative path and determine the
		 * relativity of the the final target location to
		 * the specified location
		 */
		gchar **strv = g_strsplit(
			relative_target,
			G_DIR_SEPARATOR_S,
			-1
		);
		if(strv != NULL)
		{
			gint i;
			const gchar *s;
			gint	nascend = 0,
				ndescend = 0,
				generation_gap;
			for(i = 0; strv[i] != NULL; i++)
			{
				s = strv[i];
				if(STRISEMPTY(s))
					continue;

				if(!strcmp(s, ".."))
				{
					nascend++;
				}
				else if(!strcmp(s, "."))
				{
					/* Goes nowhere, do not count */
				}
				else
				{
					ndescend++;
				}
			}
			generation_gap = ndescend - nascend;

#if 0
g_print("%s d=%i a=%i\n", relative_target, ndescend, nascend);
#endif

			g_free(*relativity_rtn);
			if((ndescend == 1) && (nascend == 0))
			{
				*relativity_rtn = g_strdup("Sibling");
			}
			else if((ndescend > 1) && (nascend == 0))
			{
				*relativity_rtn = g_strdup("Descendent");
			}
			else if(generation_gap < 0)
			{
				*relativity_rtn = g_strdup("Ancestor");
			}
			else if((generation_gap >= 0) && (nascend > 0))
			{
				*relativity_rtn = g_strdup("Cousin");
			}
			else
			{
				relativity_rtn = NULL;
			}

			g_strfreev(strv);
		}
		g_free(relative_target);
	}

	g_free(cur_path);

	return(TRUE);
}


/*
 *	Updates the values displayed on the GtkWidgets.
 */
static void edv_link_prop_page_update_widgets(EDVLinkPropPage *p)
{
	gboolean	target_is_full_path,
			target_exists;
	const gchar	*path,
			*link_target;
	const gboolean this_page_has_changes = (p->flags & EDV_LINK_PROP_PAGE_HAS_CHANGES) ? TRUE : FALSE;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	const EDVLocationType location_type = edv_prop_page_get_location_type(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);

	p->freeze_count++;

	/* If there are changes then get the current values from this
	 * page's GtkWidgets, otherwise get the current values from
	 * the properties list and set those values to this page's
	 * GtkWidgets, this is to ensure that this page's GtkWidgets
	 * get their values initially set for the first time this
	 * update callback is called and to prevent subsequent
	 * modification of the values displayed on the GtkWidgets
	 * after the user has changed the values on the GtkWidgets
	 */
	if(this_page_has_changes)
	{
		link_target = gtk_entry_get_text(GTK_ENTRY(p->target_entry));
	}
	else
	{
		link_target = edv_properties_list_get_s(
			properties_list,
			EDV_PROP_NAME_LINK_TARGET
		);
		gtk_entry_set_text(
			GTK_ENTRY(p->target_entry),
			(link_target != NULL) ? link_target : ""
		);
	}

	/* Set the source icon and label only if there are no changes */
	if(!this_page_has_changes)
	{
		gchar *s;
		EDVPixmap	*icon = NULL,
				*icon_inaccessable = NULL,
				*icon_hidden = NULL;
		const gchar *path = edv_properties_list_get_s(
			properties_list,
			EDV_PROP_NAME_PATH
		);
		EDVVFSObject *obj = edv_vfs_object_lstat(path);
		if(obj != NULL)
		{
			/* Get the appropriate icon for the target object */
			(void)edv_match_object_icon(
				core->devices_list,
				core->mime_types_list,
				obj->type,
				obj->path,
				TRUE,		/* Assume link valid */
				obj->permissions,
				EDV_ICON_SIZE_20,
				&icon,
				NULL,
				&icon_inaccessable,
				&icon_hidden
			);
			/* Hidden? */
			if(edv_is_object_hidden(obj))
			{
				if(edv_pixmap_is_loaded(icon_hidden))
				{
					(void)edv_pixmap_unref(icon);
					icon = edv_pixmap_ref(icon_hidden);
				}
			}
			/* No access? */
			if(!edv_is_object_accessable(core, obj))
			{
				if(edv_pixmap_is_loaded(icon_inaccessable))
				{
					(void)edv_pixmap_unref(icon);
					icon = edv_pixmap_ref(icon_inaccessable);
				}
			}

			edv_vfs_object_delete(obj);
		}

		/* Update the icon */
		if(edv_pixmap_is_loaded(icon))
			edv_pixmap_set_gtk_pixmap(
				icon,
				p->source_icon_pm
			);

		/* Set the label */
		s = edv_path_shorten(
			path,
			50
		);
		if(s != NULL)
		{
			gtk_label_set_text(
				GTK_LABEL(p->source_label),
				s
			);
			g_free(s);
		}

		(void)edv_pixmap_unref(icon);
		(void)edv_pixmap_unref(icon_inaccessable);
		(void)edv_pixmap_unref(icon_hidden);
	}

	/* Set the target */
	switch(location_type)
	{
	    case EDV_LOCATION_TYPE_VFS:
		target_is_full_path = FALSE;
		target_exists = FALSE;
		path = edv_properties_list_get_s(
			properties_list,
			EDV_PROP_NAME_PATH
		);
		if(!STRISEMPTY(link_target))
		{
			/* Get the full path to the target and check if the
			 * specified target value is a full path
			 */
			gchar *target_full_path;
			if(g_path_is_absolute(link_target))
			{
				target_full_path = g_strdup(link_target);
				edv_path_simplify(target_full_path);
				target_is_full_path = TRUE;
			}
			else if(!STRISEMPTY(path))
			{
				/* Not an absolute path, prefix the parent path
				 * of the link to the target path
				 */
				gchar *parent_path = g_dirname(path);
				target_full_path = g_strconcat(
					parent_path,
					G_DIR_SEPARATOR_S,
					link_target,
					NULL
				);
				g_free(parent_path);
			}
			else
			{
				target_full_path = NULL;
			}

			/* Got the full target path? */
			if(target_full_path != NULL)
			{
				GtkWidget *w;
				EDVPixmap	*icon = NULL,
						*icon_inaccessable = NULL,
						*icon_hidden = NULL;

				/* Check if the target object exists locally
				 * by attempting to get its statistics
				 */
				EDVVFSObject *tar_obj = edv_vfs_object_lstat(target_full_path);
				if(tar_obj != NULL)
				{
					/* Target object exists */
					target_exists = TRUE;

					/* Get the appropriate icon for the target object */
					(void)edv_match_object_icon(
						core->devices_list,
						core->mime_types_list,
						tar_obj->type,
						tar_obj->path,
						TRUE,		/* Assume link valid */
						tar_obj->permissions,
						EDV_ICON_SIZE_20,
						&icon,
						NULL,
						&icon_inaccessable,
						&icon_hidden
					);
					/* Hidden? */
					if(edv_is_object_hidden(tar_obj))
					{
						if(edv_pixmap_is_loaded(icon_hidden))
						{
							(void)edv_pixmap_unref(icon);
							icon = edv_pixmap_ref(icon_hidden);
						}
					}
					/* No access? */
					if(!edv_is_object_accessable(core, tar_obj))
					{
						if(edv_pixmap_is_loaded(icon_inaccessable))
						{
							(void)edv_pixmap_unref(icon);
							icon = edv_pixmap_ref(icon_inaccessable);
						}
					}

					edv_vfs_object_delete(tar_obj);
				}
				else
				{
					/* Target object does not exist, get the
					 * broken link icon from the MIME Type
					 */
					EDVMIMEType *m = edv_mime_types_list_match_type(
						core->mime_types_list,
						NULL,	/* No index return */
						EDV_MIME_TYPE_TYPE_INODE_LINK,
						FALSE	/* Not case sensitive */
					);
					if(m != NULL)
					{
						edv_mime_type_realize(m, FALSE);
						(void)edv_pixmap_unref(icon);
						icon = edv_pixmap_ref(m->small_icon[
							EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE
						]);
					}
				}

				/* Update the link icon */
				w = p->target_icon_pm;
				if(edv_pixmap_is_loaded(icon))
				{
					edv_pixmap_set_gtk_pixmap(icon, w);
					gtk_widget_show(w);
				}
				else
				{
					gtk_widget_hide(w);
				}

				(void)edv_pixmap_unref(icon);
				(void)edv_pixmap_unref(icon_inaccessable);
				(void)edv_pixmap_unref(icon_hidden);

				g_free(target_full_path);
			}
		}
		/* Update the make relative path GtkButton */
		GTK_WIDGET_SET_SENSITIVE(
			p->target_make_relative_path_btn,
			target_is_full_path
		);
		/* Update the target properties GtkButton */
		GTK_WIDGET_SET_SENSITIVE(
			p->target_properties_btn,
			target_exists
		);
#if 0
		/* Update the auto link from icon */
		if(p->auto_link_from_icon_pm != NULL)
		{
			EDVMIMEType *m = edv_mime_types_list_match_type(
				core->mime_types_list,
				NULL,		/* No index return */
				EDV_MIME_TYPE_TYPE_INODE_LINK,
				FALSE		/* Not case sensitive */
			);
			if(m != NULL)
			{
				EDVPixmap *icon;
				edv_mime_type_realize(m, FALSE);
				icon = edv_pixmap_ref(m->small_icon[
					EDV_MIME_TYPE_ICON_STATE_STANDARD
				]);
				if(edv_pixmap_is_loaded(icon))
					edv_pixmap_set_gtk_pixmap(
						icon,
						p->auto_link_from_icon_pm
					);
				(void)edv_pixmap_unref(icon);
			}
		}
		/* Update the auto link from label */
		if(p->auto_link_from_path_label != NULL)
		{
			gchar *s = edv_path_shorten(
				path,
				50
			);
			if(s != NULL)
			{
				gtk_label_set_text(
					GTK_LABEL(p->auto_link_from_path_label),
					s
				);
				g_free(s);
			}
		}
#endif
		/* Update the statistics label */
		if(p->statistics_frame != NULL)
		{
			gchar	*s,
				*size_str,
				*relativity;
			gint nhops;
			gulong size;
			(void)edv_link_prop_page_get_statistics(
				path,
				link_target,
				&relativity,
				&nhops,
				&size
			);
			size_str = STRDUP(edv_str_size_delim(size));
			s = g_strdup_printf(
"%s\n\
%i\n\
%s %s",
				(relativity != NULL) ? relativity : "unknown",
				nhops,
				size_str,
				(size == 1l) ? "byte" : "bytes"
			);
			g_free(size_str);
			g_free(relativity);
			gtk_label_set_text(
				GTK_LABEL(p->statistics_label),
				s
			);
			g_free(s);
		}
		break;

	    case EDV_LOCATION_TYPE_RECYCLE_BIN:
	    case EDV_LOCATION_TYPE_ARCHIVE:
		if(core->mime_types_list != NULL)
		{
			GtkWidget *w;
			EDVPixmap *icon = NULL;

			/* Get the link icon from the MIME Type */
			EDVMIMEType *m = edv_mime_types_list_match_type(
				core->mime_types_list,
				NULL,		/* No index return */
				EDV_MIME_TYPE_TYPE_INODE_LINK,
				FALSE		/* Not case sensitive */
			);
			if(m != NULL)
			{
				edv_mime_type_realize(m, FALSE);
				(void)edv_pixmap_unref(icon);
				icon = edv_pixmap_ref(m->small_icon[
					EDV_MIME_TYPE_ICON_STATE_STANDARD
				]);
			}

			/* Update the link icon */
			w = p->target_icon_pm;
			if(edv_pixmap_is_loaded(icon))
			{
				edv_pixmap_set_gtk_pixmap(icon, w);
				gtk_widget_show(w);
			}
			else
			{
				gtk_widget_hide(w);
			}

			(void)edv_pixmap_unref(icon);
		}
		break;
	}

	gtk_widget_queue_resize(toplevel);

	p->freeze_count--;
}
