#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <gtk/gtk.h>

#include "cfg.h"

#include "guirgbimg.h"
#include "guiutils.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_property.h"
#include "libendeavour2-base/edv_property_directory.h"
#include "libendeavour2-base/edv_recycle_bin_index.h"
#include "edv_date_format.h"
#include "edv_pixmap.h"
#include "edv_image_io.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "edv_open.h"
#include "prop_page.h"
#include "prop_page_image.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_open_20x20.xpm"


typedef struct _EDVImagePropPage	EDVImagePropPage;
#define EDV_IMAGE_PROP_PAGE(p)		((EDVImagePropPage *)(p))


/*
 *	Flags:
 */
typedef enum {
	EDV_IMAGE_PROP_PAGE_CURRENTLY_DISPLAYED	\
					= (1 << 0),
	EDV_IMAGE_PROP_PAGE_GOT_VALUES	= (1 << 1)
} EDVImagePropPageFlags;


/* Check Support */
gboolean edv_image_prop_page_query_create_cb(
	EDVPropPageContext *ctx,
	gint *version_major_rtn,
	gint *version_minor_rtn,
	gint *version_release_rtn,
	gchar **page_name_rtn,
	edv_pixmap_data **pixmap_data_20x20_rtn,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list
);

/* Create */
gpointer edv_image_prop_page_create_cb(
	EDVPropPageContext *ctx,
	GtkWidget *parent
);

/* Update */
void edv_image_prop_page_update_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list,
	const int error_code,
	gpointer data
);

/* Page switched callback */
void edv_image_prop_page_page_switched_cb(
	EDVPropPageContext *ctx,
	const gboolean to,
	gpointer data
);

/* Destroy */
void edv_image_prop_page_destroy_cb(
	EDVPropPageContext *ctx,
	gpointer data
);

/* Callbacks */
static gint edv_image_prop_page_thumb_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void edv_image_prop_page_thumb_realize_cb(
	GtkWidget *widget, gpointer data
);
static gint edv_prop_dlg_image_text_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static gint edv_image_prop_page_get_values_idle_cb(gpointer data);
static void edv_image_prop_page_open_cb(GtkWidget *widget, gpointer data);

/* Utilities */
static gchar *edv_image_prop_page_single_line_string(gchar *s);

/* Operations */
static void edv_image_prop_page_open(EDVImagePropPage *p);

/* Thumb Draw/Set */
static void edv_image_prop_page_thumb_draw(EDVImagePropPage *p);
static EDVPixmap *edv_image_prop_page_thumb_set_rgba_iterate(
	EDVImagePropPage *p,
	const guint8 *rgba,
	const gint width, const gint height,
	const gint bpl,
	const guint8 *bg_color
);
static void edv_image_prop_page_thumb_set_rgba(
	EDVImagePropPage *p,
	const guint8 *rgba,
	const gint width, const gint height,
	const gint bpl,
	const guint8 *bg_color,
	const gint orig_width, const gint orig_height
);
/*
static void edv_image_prop_page_thumb_clear(EDVPropID3Page *p);
 */

/* Get Values */
static void edv_image_prop_page_get_values(EDVImagePropPage *p);

/* Update Widgets */
static void edv_image_prop_dlg_update_widgets(EDVImagePropPage *p);


#define EDV_IMAGE_PROP_PAGE_NAME	"Image"

#define EDV_IMAGE_PROP_PAGE_THUMB_WIDTH	100
#define EDV_IMAGE_PROP_PAGE_THUMB_HEIGHT	\
					100
#define EDV_IMAGE_PROP_PAGE_THUMB_PADDING_X	\
					5
#define EDV_IMAGE_PROP_PAGE_THUMB_PADDING_Y	\
					5
#define EDV_IMAGE_PROP_PAGE_THUMB_SHADOW_WIDTH	\
					5
#define EDV_IMAGE_PROP_PAGE_THUMB_SHADOW_HEIGHT	\
					5

#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define FCLOSE(_fp_)	(((_fp_) != NULL) ? (gint)fclose(_fp_) : -1)


/*
 *	Image Properties Page:
 */
struct _EDVImagePropPage {
	GtkWidget	*toplevel;
	gint		freeze_count;
	EDVPropPageContext	*ctx;
	EDVImagePropPageFlags	flags;

	GtkWidget	*info_label,
			*thumb_da,
			*open_btn,
			*title_entry,
			*author_entry,
			*creator_entry,
			*comments_toplevel,
			*comments_text;

	EDVPixmap	*thumb;

	/* Statistics */
	guint		get_info_toid;
};


/*
 *	Check support callback.
 */
gboolean edv_image_prop_page_query_create_cb(
	EDVPropPageContext *ctx,
	gint *version_major_rtn,
	gint *version_minor_rtn,
	gint *version_release_rtn,
	gchar **page_name_rtn,
	edv_pixmap_data **pixmap_data_20x20_rtn,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list
)
{
	*version_major_rtn = PROG_VERSION_MAJOR;
	*version_minor_rtn = PROG_VERSION_MINOR;
	*version_release_rtn = PROG_VERSION_RELEASE;
	*page_name_rtn = g_strdup(EDV_IMAGE_PROP_PAGE_NAME);

	switch(location_type)
	{
	    case EDV_LOCATION_TYPE_VFS:
	    case EDV_LOCATION_TYPE_RECYCLE_BIN:
		if((type == EDV_OBJECT_TYPE_FILE) ||
		   (type == EDV_OBJECT_TYPE_LINK)
		)
		{
			const gchar *name = edv_properties_list_get_s(
				properties_list,
				EDV_PROP_NAME_NAME
			);

			/* Report TRUE if this object name's extension
			 * match an image format that we recognize and
			 * will be able to display information for
			 */
			return(edv_image_is_supported_extension(
				edv_name_get_extension(name)
			));
		}
		break;

	    case EDV_LOCATION_TYPE_ARCHIVE:
		break;
	}

	return(FALSE);
}

/*
 *	Create callback.
 */
gpointer edv_image_prop_page_create_cb(
	EDVPropPageContext *ctx,
	GtkWidget *parent
)
{
	const gint	border_major = 5,
			border_minor = 2;
	GtkRcStyle *rcstyle;
	GtkWidget	*w,
			*parent2, *parent3, *parent4, *parent5,
			*sb;
	GtkEditable *editable;
	GtkText *text;
	EDVImagePropPage *p = EDV_IMAGE_PROP_PAGE(g_malloc0(
		sizeof(EDVImagePropPage)
	));
	if(p == NULL)
		return(NULL);

	p->ctx = ctx;
	p->toplevel = parent;

	p->freeze_count++;

	/* Information GtkFrame */
	w = gtk_frame_new("Information");
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Information GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent2 = w;

	/* Title GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_label_new("Title:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	p->title_entry = w = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_entry_set_editable(GTK_ENTRY(w), FALSE);
	gtk_widget_show(w);

	/* Author GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_label_new("Author:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	p->author_entry = w = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_entry_set_editable(GTK_ENTRY(w), FALSE);
	gtk_widget_show(w);

	/* GtkHBox to make two columns for the general image
	 * information and the thumb preview
	 */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Left Column GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	w = gtk_label_new(
"Width:\n\
Height:\n\
Aspect:\n\
Frames:\n\
Play Time:\n\
Frame Size:\n\
Total Size:\n\
File Size:\n\
Last Modified:"
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Info GtkLabel */
	p->info_label = w = gtk_label_new(
"Opening...\n\
-\n\
-\n\
-\n\
-\n\
-\n\
-\n\
-\n\
-"
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Right Side GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_end(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* GtkFrame */
	w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	/* Thumb GtkDrawingArea */
	p->thumb_da = w = gtk_drawing_area_new();
	gtk_widget_set_usize(
		w,
		EDV_IMAGE_PROP_PAGE_THUMB_WIDTH +
			(2 * EDV_IMAGE_PROP_PAGE_THUMB_PADDING_X) +
			EDV_IMAGE_PROP_PAGE_THUMB_SHADOW_WIDTH,
		EDV_IMAGE_PROP_PAGE_THUMB_HEIGHT +
			(2 * EDV_IMAGE_PROP_PAGE_THUMB_PADDING_Y) +
			EDV_IMAGE_PROP_PAGE_THUMB_SHADOW_HEIGHT
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "expose_event",
		GTK_SIGNAL_FUNC(edv_image_prop_page_thumb_event_cb), p
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "realize",
		GTK_SIGNAL_FUNC(edv_image_prop_page_thumb_realize_cb), p
	);
	gtk_container_add(GTK_CONTAINER(parent5), w);
	gtk_widget_show(w);

	/* GtkHBox for the Open GtkButton */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_end(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	/* Open GtkButton */
	p->open_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_open_20x20_xpm,
		"Open",
		NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_image_prop_page_open_cb), p
	);
	gtk_widget_show(w);


	/* Creator GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_label_new("Creator:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	p->creator_entry = w = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_entry_set_editable(GTK_ENTRY(w), FALSE);
	gtk_widget_show(w);


	/* Comments GtkFrame */
	p->comments_toplevel = w = gtk_frame_new("Comments");
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	parent2 = w;

	/* Comments GtkTable */
	w = gtk_table_new(1, 2, FALSE);
	gtk_table_set_row_spacing(GTK_TABLE(w), 0, border_minor);
	gtk_table_set_col_spacing(GTK_TABLE(w), 0, border_minor);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent3 = w;

	/* Comments Text GtkRcStyle */
	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = g_strdup(
#if defined(PROG_LANGUAGE_POLISH)
"-adobe-courier-medium-r-normal-*-12-*-*-*-*-*-iso8859-2"
#else
"-adobe-courier-medium-r-normal-*-12-*-*-*-*-*-iso8859-1"
#endif
	);
	/* GtkText */
	p->comments_text = w = gtk_text_new(NULL, NULL);
	editable = GTK_EDITABLE(w);
	text = GTK_TEXT(w);
/*	gtk_widget_set_usize(w, -1, 80); */
	text->default_tab_width = 8;
	gtk_text_set_editable(text, FALSE);
	gtk_text_set_word_wrap(text, TRUE);
	gtk_table_attach(
		GTK_TABLE(parent3),
		w,
		0, 1,
		0, 1,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		0, 0
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_prop_dlg_image_text_event_cb), p
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(edv_prop_dlg_image_text_event_cb), p
	);
	gtk_widget_realize(w);
	gtk_widget_modify_style(w, rcstyle);
	GUIEditableEndowPopupMenu(w, GUI_EDITABLE_POPUP_MENU_UNDO);
	gtk_widget_show(w);
	GTK_RC_STYLE_UNREF(rcstyle);
	/* Vertical GtkScrollBar */
	sb = gtk_vscrollbar_new(GTK_TEXT(w)->vadj);
	gtk_table_attach(
		GTK_TABLE(parent3),
		sb,
		1, 2,
		0, 1,
		GTK_FILL,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		0, 0
	);
	gtk_widget_show(sb);


	p->freeze_count--;

	return(p);
}


/*
 *	Update callback.
 */
void edv_image_prop_page_update_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list,
	const int error_code,
	gpointer data
)
{
	EDVImagePropPage *p = EDV_IMAGE_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	/* Update the displayed information only if this page is
	 * currently displayed or we have previously obtained
	 * values that are now out of date
	 */
	if((p->flags & EDV_IMAGE_PROP_PAGE_CURRENTLY_DISPLAYED) ||
	   (p->flags & EDV_IMAGE_PROP_PAGE_GOT_VALUES)
	)
	{
		edv_image_prop_page_get_values(p);
	}

	edv_image_prop_dlg_update_widgets(p);

	p->freeze_count--;
}


/*
 *	Page switched callback.
 */
void edv_image_prop_page_page_switched_cb(
	EDVPropPageContext *ctx,
	const gboolean to,
	gpointer data
)
{
	EDVImagePropPage *p = EDV_IMAGE_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	if(to)
		p->flags |= EDV_IMAGE_PROP_PAGE_CURRENTLY_DISPLAYED;
	else
		p->flags &= ~EDV_IMAGE_PROP_PAGE_CURRENTLY_DISPLAYED;

	if(to)
	{
		/* Did not get the image information yet? */
		if(!(p->flags & EDV_IMAGE_PROP_PAGE_GOT_VALUES))
		{
			if(p->get_info_toid == 0)
				p->get_info_toid = gtk_idle_add_priority(
					G_PRIORITY_LOW,
					edv_image_prop_page_get_values_idle_cb, p
				);
		}
	}

	p->freeze_count--;
}


/*
 *	Destroy callback.
 */
void edv_image_prop_page_destroy_cb(
	EDVPropPageContext *ctx,
	gpointer data
)
{
	EDVImagePropPage *p = EDV_IMAGE_PROP_PAGE(data);

	p->freeze_count++;

	p->get_info_toid = GTK_TIMEOUT_REMOVE(p->get_info_toid);
	p->thumb = edv_pixmap_unref(p->thumb);

	p->freeze_count--;

	g_free(p);
}


/*
 *	Thumb GtkDrawingArea event signal callback.
 */
static gint edv_image_prop_page_thumb_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	EDVImagePropPage *p = EDV_IMAGE_PROP_PAGE(data);
	if((widget == NULL) || (event == NULL) || (p == NULL))
		return(status);

	if(p->freeze_count > 0)
		return(status);

	p->freeze_count++;

	switch((gint)event->type)
	{
	    case GDK_EXPOSE:
		edv_image_prop_page_thumb_draw(p);
		status = TRUE;
		break;
	}

	p->freeze_count--;

	return(status);
}

/*
 *	Thumb GtkDrawingArea "realize" signal callback.
 */
static void edv_image_prop_page_thumb_realize_cb(
	GtkWidget *widget, gpointer data
)
{
	gchar *icons_path;
	GdkWindow *window;
	GtkStyle *style;
	EDVPropPageContext *ctx;
	EDVCore *core;
	CfgList *cfg_list;
	EDVImagePropPage *p = EDV_IMAGE_PROP_PAGE(data);
	if((widget == NULL) || (p == NULL))
		return;

	p->freeze_count++;

	window = widget->window;
	style = gtk_widget_get_style(widget);
	ctx = p->ctx;
	core = edv_prop_page_get_core(ctx);
	cfg_list = edv_prop_page_get_cfg_list(ctx);

	/* Get the icons directory */
	icons_path = g_strconcat(
		EDV_GET_S(EDV_CFG_PARM_DIR_GLOBAL),
		G_DIR_SEPARATOR_S,
		EDV_NAME_ICONS_SUBDIR,
		NULL
	);

	gdk_window_set_background(
		window,
		&style->bg[GTK_STATE_ACTIVE]
	);

	/* Create the "loading" GdkPixmap to be displayed on the thumb */
	if(p->thumb == NULL)
	{
		gchar *path = g_strconcat(
			icons_path,
			G_DIR_SEPARATOR_S,
			"icon_file_image_pending_48x48.xpm",
			NULL
		);
		p->thumb = edv_pixmap_new_from_file(path);
		g_free(path);
	}

	g_free(icons_path);

	p->freeze_count--;
}

/*
 *	Comments GtkText any event signal callback.
 */
static gint edv_prop_dlg_image_text_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	GdkEventButton *button;
	GtkText *text;
	EDVImagePropPage *p = EDV_IMAGE_PROP_PAGE(data);
	if((widget == NULL) || (event == NULL) || (p == NULL))
		return(status);

	if(p->freeze_count > 0)
		return(status);

	p->freeze_count++;

	text = GTK_TEXT(widget);

	switch((gint)event->type)
	{
	    case GDK_BUTTON_PRESS:
	       button = (GdkEventButton *)event;
		switch(button->button)
		{
		  case GDK_BUTTON4:
			/* Scroll up */
			if(text->vadj != NULL)
			{
				GtkAdjustment *adj = text->vadj;
				const gfloat inc = MAX(
					(0.25f * adj->page_size),
					adj->step_increment
				);
				gfloat v = adj->value - inc;
				if(v > (adj->upper - adj->page_size))
					v = adj->upper - adj->page_size;
				if(v < adj->lower)
					v = adj->lower;
				gtk_adjustment_set_value(adj, v);
			}
			/* Need to mark the GtkText button as 0 or else it will
			 * keep marking
			 */
			text->button = 0;
			gtk_grab_remove(widget);
			gtk_signal_emit_stop_by_name(
				GTK_OBJECT(widget), "button_press_event"
			);
			status = TRUE;
			break;

		  case GDK_BUTTON5:
			/* Scroll down */
			if(text->vadj != NULL)
			{
				GtkAdjustment *adj = text->vadj;
				const gfloat inc = MAX(
					(0.25f * adj->page_size),
					adj->step_increment
				);
				gfloat v = adj->value + inc;
				if(v > (adj->upper - adj->page_size))
					v = adj->upper - adj->page_size;
				if(v < adj->lower)
					v = adj->lower;
				gtk_adjustment_set_value(adj, v);
			}
			/* Need to mark the GtkText button as 0 or else it will
			 * keep marking
			 */
			text->button = 0;
			gtk_grab_remove(widget);
			gtk_signal_emit_stop_by_name(
				GTK_OBJECT(widget), "button_press_event"
			);
			status = TRUE;
			break;
		}
		break;

	  case GDK_BUTTON_RELEASE:
		button = (GdkEventButton *)event;
		switch(button->button)
		{
		  case GDK_BUTTON4:
			/* Need to mark the GtkText button as 0 or else it will
			 * keep marking
			 */
			text->button = 0;
			gtk_grab_remove(widget);
			gtk_signal_emit_stop_by_name(
				GTK_OBJECT(widget), "button_release_event"
			);
			status = TRUE;
			break;
		  case GDK_BUTTON5:
			/* Need to mark the GtkText button as 0 or else it will
			 * keep marking
			 */
			text->button = 0;
			gtk_grab_remove(widget);
			gtk_signal_emit_stop_by_name(
				GTK_OBJECT(widget), "button_release_event"
			);
			status = TRUE;
		       break;
		}
		break;
	}

	p->freeze_count--;

	return(status);
}

/*
 *	Get info timeout callback.
 */
static gint edv_image_prop_page_get_values_idle_cb(gpointer data)
{
	EDVImagePropPage *p = EDV_IMAGE_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return(TRUE);

	p->freeze_count++;

	edv_image_prop_page_get_values(p);

	p->get_info_toid = 0;

	p->freeze_count--;

	return(FALSE);
}

/*
 *	Open callback.
 */
static void edv_image_prop_page_open_cb(GtkWidget *widget, gpointer data)
{
	EDVImagePropPage *p = EDV_IMAGE_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	edv_image_prop_page_open(p);

	p->freeze_count--;
}


/*
 *	Removes all control characters from the string.
 */
static gchar *edv_image_prop_page_single_line_string(gchar *s)
{
	gchar *s2;

	if(s == NULL)
		return(g_strdup(""));

	s2 = edv_strsub(
		s,
		"\n",
		" "
	);
	g_free(s);
	s = s2;

	s2 = edv_strsub(
		s,
		"\r",
		" "
	);
	g_free(s);
	s = s2;

	s2 = edv_strsub(
		s,
		"\f",
		" "
	);
	g_free(s);
	s = s2;

	s2 = edv_strsub(
		s,
		"\b",
		" "
	);
	g_free(s);
	s = s2;

	return(s);
}


/*
 *	Open.
 */
static void edv_image_prop_page_open(EDVImagePropPage *p)
{
	const gchar *path;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	const EDVLocationType location_type = edv_prop_page_get_location_type(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);

	p->freeze_count++;

	switch(location_type)
	{
	    case EDV_LOCATION_TYPE_VFS:
		edv_prop_page_set_busy(ctx, TRUE);
		path = edv_properties_list_get_s(
			properties_list,
			EDV_PROP_NAME_PATH
		);
		if(path != NULL)
		{
			GList *paths_list = NULL;
			paths_list = g_list_append(
				paths_list,
				g_strdup(path)
			);
			(void)edv_open(
				core,
				paths_list,
				NULL,		/* Default command */
				toplevel,
				TRUE		/* Verbose */
			);
			g_list_foreach(paths_list, (GFunc)g_free, NULL);
			g_list_free(paths_list);
		}
		edv_prop_page_set_busy(ctx, FALSE);
		break;

	    case EDV_LOCATION_TYPE_RECYCLE_BIN:
	    case EDV_LOCATION_TYPE_ARCHIVE:
		break;
	}

	p->freeze_count--;
}


/*
 *	Redraws the thumb GtkDrawingArea.
 */
static void edv_image_prop_page_thumb_draw(EDVImagePropPage *p)
{
	gint width, height;
	GtkWidget *w = p->thumb_da;
	GdkWindow *window = w->window;
	GdkDrawable *drawable = (GdkDrawable *)window;
	GtkStyle *style = gtk_widget_get_style(w);
	if((drawable == NULL) || (style == NULL))
		return;

	gdk_window_get_size(
		(GdkWindow *)drawable,
		&width, &height
	);
	if((width <= 0) || (height <= 0))
		return;

	gdk_draw_rectangle(
		drawable,
		style->bg_gc[GTK_STATE_ACTIVE],
		TRUE,				/* Fill */
		0, 0,
		width, height
	);

	/* Thumb EDVPixmap available to be drawn? */
	if(p->thumb != NULL)
	{
		gint	x, y,
			shadow_x, shadow_y;
		GdkGC *gc;
		EDVPixmap *thumb = p->thumb;
		GdkBitmap *mask = thumb->mask;
		GdkPixmap *pixmap = thumb->pixmap;

		x = (width - thumb->width - EDV_IMAGE_PROP_PAGE_THUMB_SHADOW_WIDTH) / 2;
		y = (height - thumb->height - EDV_IMAGE_PROP_PAGE_THUMB_SHADOW_HEIGHT) / 2;

		shadow_x = x + EDV_IMAGE_PROP_PAGE_THUMB_SHADOW_WIDTH;
		shadow_y = y + EDV_IMAGE_PROP_PAGE_THUMB_SHADOW_HEIGHT;

		/* Draw the shadow */
		gc = style->black_gc;
		gdk_gc_set_clip_origin(gc, shadow_x, shadow_y);
		gdk_gc_set_clip_mask(gc, mask);
		gdk_draw_rectangle(
			drawable,
			gc,
			TRUE,			/* Fill */
			shadow_x, shadow_y,
			thumb->width, thumb->height
		);
		gdk_gc_set_clip_mask(gc, NULL);

		/* Draw the thumb */
		gc = style->white_gc;
		gdk_gc_set_clip_origin(gc, x, y);
		gdk_gc_set_clip_mask(gc, mask);
		gdk_draw_pixmap(
			drawable,
			gc,
			(GdkDrawable *)pixmap,
			0, 0,
			x, y,
			thumb->width, thumb->height
		);
		gdk_gc_set_clip_mask(gc, NULL);
	}

	/* Draw the focus */
	if(GTK_WIDGET_HAS_FOCUS(w) && GTK_WIDGET_SENSITIVE(w))
	{
		gtk_paint_focus(
			style,
			(GdkWindow *)drawable,
			NULL,			/* Entire area */
			w,
			NULL,			/* No detail */
			0, 0,
			width - 1, height - 1
		);
	}

	/* Send the drawable to window if drawable is not the window */
	if(drawable != (GdkDrawable *)window)
		gdk_draw_pixmap(
			window,
			style->white_gc,
			drawable,
			0, 0,
			0, 0,
			width, height
		);
}


/*
 *	Called by edv_image_prop_page_thumb_set_rgba() to create
 *	a new EDVPixmap from the RGBA image data, the specified RGBA
 *	image data will be combined with the specified background
 *	color and sent to the new EDVPixmap.
 *
 *	The specified RGBA image data should already be resized to
 *	the sized prefered by the thumb.
 */
static EDVPixmap *edv_image_prop_page_thumb_set_rgba_iterate(
	EDVImagePropPage *p,
	const guint8 *rgba,
	const gint width, const gint height,
	const gint bpl,
	const guint8 *bg_color
)
{
	const gint	bpp = 4;		/* RGBA */
	const guint32 bg_pixel32 = (guint32)(
			((guint32)bg_color[0] << 0) &
			((guint32)bg_color[1] << 8) &
			((guint32)bg_color[2] << 16) &
			0xFF000000
	);
	gint y;
	guint8		*line_ptr, *line_end,
			*combined_rgba;
	GtkWidget *w = p->thumb_da;
	GtkStyle *style = gtk_widget_get_style(w);
	GdkWindow *window = (w != NULL) ? w->window : NULL;

	/* Create the new GdkPixmap */
	EDVPixmap *thumb = edv_pixmap_new_ref();
	if(thumb == NULL)
		return(NULL);

	thumb->width = width;
	thumb->height = height;
	thumb->pixmap = gdk_pixmap_new(
		window,
		thumb->width, thumb->height,
		-1
	);
	if(thumb->pixmap == NULL)
		return(thumb);

	/* Combine the opened image data to the background and send
	 * it to the new GdkPixmap
	 *
	 * Allocate the combined RGBA image data
	 */
	combined_rgba = (guint8 *)g_malloc(
		height * bpl * sizeof(guint8)
	);
	if(combined_rgba == NULL)
		return(thumb);

	/* Clear the combined RGBA image data with the background
	 * color
	 */
	for(y = 0; y < height; y++)
	{
		for(line_ptr = combined_rgba + (y * bpl),
		    line_end = line_ptr + (width * bpp);
		    line_ptr < line_end;
		    line_ptr += bpp
		)
			*(guint32 *)line_ptr = bg_pixel32;
	}

	/* Combine/flatten the specified image data to the combined
	 * image data
	 */
	GUIImageBufferCopyArea(
		bpp,
		rgba,
		width, height, bpl,
		combined_rgba,
		width, height, bpl,
		0, 0,
		TRUE,				/* Blend */
		NULL, NULL
	);
	
	/* Send the combined image data to the new GdkPixmap */
	gdk_draw_rgb_32_image(
		(GdkDrawable *)thumb->pixmap,
		style->white_gc,
		0, 0,
		width, height,
		GDK_RGB_DITHER_NONE,
		combined_rgba,
		bpl
	);

	/* Delete the combined RGBA image data */
	g_free(combined_rgba);

	return(thumb);
}

/*
 *	Sets the RGBA image data for the thumb GtkDrawingArea,
 *	recreates the thumb GtkDrawingArea's GdkPixmap, and
 *	queues the thumb GtkDrawingArea to redraw.
 */
static void edv_image_prop_page_thumb_set_rgba(
	EDVImagePropPage *p,
	const guint8 *rgba,
	const gint width, const gint height,
	const gint bpl,
	const guint8 *bg_color,
	const gint orig_width, const gint orig_height
)
{
	const gint	bpp = 4,		/* RGBA */
			req_width = EDV_IMAGE_PROP_PAGE_THUMB_WIDTH,
			req_height = EDV_IMAGE_PROP_PAGE_THUMB_HEIGHT;
	EDVPixmap *thumb = NULL;
	GtkWidget *w = p->thumb_da;

	/* Unref the thumb's old GdkPixmap */
	p->thumb = edv_pixmap_unref(p->thumb);

	/* No RGBA image data specified? */
	if((rgba == NULL) || (width <= 0) || (height <= 0) ||
	   (bpl <= 0) || (orig_width <= 0) || (orig_height <= 0)
	)
	{
		/* Queue the thumb to redraw with no GdkPixmap */
		gtk_widget_queue_draw(w);
		return;
	}

	/* Resize the image to the prefered thumb size if the
	 * specified image is larger
	 */
	if((width > req_width) || (height > req_height))
	{
		guint8 *rs_rgba;
		gfloat aspect = (orig_width > 0) ?
			(gfloat)orig_height / (gfloat)orig_width : 1.0f;
		gint	rs_width = req_width,
			rs_height = (gint)(rs_width * aspect),
			rs_bpl;
		if(rs_height > req_height)
		{
			rs_height = req_height;
			rs_width = (aspect > 0.0f) ?
				(gint)(rs_height / aspect) : req_width;
		}
		if(rs_width < 1)
			rs_width = 1;
		if(rs_height < 1)
			rs_height = 1;
		rs_bpl = rs_width * bpp;

		/* Allocate the resized RGBA image data */
		rs_rgba = (guint8 *)g_malloc(
			rs_height * rs_bpl * sizeof(guint8)
		);
		if(rs_rgba != NULL)
		{
			/* Copy/resize the specified RGBA image data
			 * to our resized RGBA image data
			 */
			GUIImageBufferResize(
				bpp,
				rgba,
				width, height, bpl,
				rs_rgba,
				rs_width, rs_height, rs_bpl,
				NULL, NULL
			);

			/* Create a new EDVPixmap from the resized
			 * RGBA image data, the specified RGBA image
			 * data will be combined with the specified
			 * background color and sent to the new
			 * EDVPixmap
			 */
			thumb = edv_image_prop_page_thumb_set_rgba_iterate(
				p,
				rs_rgba,
				rs_width, rs_height,
				rs_bpl,
				bg_color
			);

			/* Delete the resized RGBA image data */
			g_free(rs_rgba);
		}
	}
	else
	{
		/* Create a new EDVPixmap from the RGBA image data,
		 * the specified RGBA image data will be combined with
		 * the specified background color and sent to the new
		 * EDVPixmap
		 */
		thumb = edv_image_prop_page_thumb_set_rgba_iterate(
			p,
			rgba,
			width, height,
			bpl,
			bg_color
		);
	}

	/* Was a new EDVPixmap created? */
	if(thumb != NULL)
	{
		/* Set the thumb's new EDVPixmap */
		p->thumb = thumb;

		/* Queue the thumb to redraw */
		gtk_widget_queue_draw(w);
	}
}

#if 0
/*
 *	Unrefs the thumb's GdkPixmap and queues the thumb to redraw.
 */
static void edv_image_prop_page_thumb_clear(EDVPropID3Page *p)
{
	/* Unref the thumb's old GdkPixmap */
	p->thumb = edv_pixmap_unref(p->thumb);

	/* Queue the thumb to redraw */
	gtk_widget_queue_draw(p->thumb_da);
}
#endif


/*
 *	Gets the information by opening the image from file and
 *	updating the displayed information, thumb, and sets the
 *	EDV_IMAGE_PROP_PAGE_GOT_VALUES flag.
 */
static void edv_image_prop_page_get_values(EDVImagePropPage *p)
{
	const gint	req_width = EDV_IMAGE_PROP_PAGE_THUMB_WIDTH,
			req_height = EDV_IMAGE_PROP_PAGE_THUMB_HEIGHT;
	guint8		*rgba,
			bg_color[4];
	gint		width, height,
			bpl,
			orig_width, orig_height,
			orig_nframes;
	gulong		index,
			play_time_ms,
			modified_time_sec;
	gchar		*path,
			*creator,
			*title,
			*author,
			*comments;
	GtkStyle *style;
	EDVPropPageContext *ctx = p->ctx;
	const EDVLocationType location_type = edv_prop_page_get_location_type(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);
	CfgList *cfg_list = edv_prop_page_get_cfg_list(ctx);

	edv_prop_page_set_busy(ctx, TRUE);
	p->freeze_count++;

	/* Set the initial default background color from the
	 * configuration
	 */
	style = gtk_widget_get_style(p->thumb_da);
	if(style != NULL)
	{
		GdkColor *c = &style->white;
		bg_color[0] = (guint8)(c->red >> 8);
		bg_color[1] = (guint8)(c->green >> 8);
		bg_color[2] = (guint8)(c->blue >> 8);
		bg_color[3] = 0xFF;
	}

	/* Open the image by the location type */
	rgba = NULL;
	switch(location_type)
	{
	    case EDV_LOCATION_TYPE_VFS:
		path = STRDUP(edv_properties_list_get_s(
			properties_list,
			EDV_PROP_NAME_PATH
		));
		rgba = edv_image_open_rgba_thumb(
			path,
			req_width, req_height,
			&width, &height,
			&bpl,
			bg_color,
			&orig_width, &orig_height,
			&orig_nframes,
			&play_time_ms,
			&creator,
			&title,
			&author,
			&comments,
			&modified_time_sec,
			NULL, NULL
		);
		g_free(path);
		break;

	    case EDV_LOCATION_TYPE_RECYCLE_BIN:
		/* Opening a recycled object does not always work,
		 * because the extension is not known and its format
		 * is usually not determineable
		 */
		index = edv_properties_list_get_ul(
			properties_list,
			EDV_PROP_NAME_INDEX
		);
		path = edv_recycle_bin_index_get_recycled_object_path(
			EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX),
			index
		);
		rgba = edv_image_open_rgba_thumb(
			path,
			req_width, req_height,
			&width, &height,
			&bpl,
			bg_color,
			&orig_width, &orig_height,
			&orig_nframes,
			&play_time_ms,
			&creator,
			&title,
			&author,
			&comments,
			&modified_time_sec,
			NULL, NULL
		);
		g_free(path);
		break;

	    case EDV_LOCATION_TYPE_ARCHIVE:
		break;
	}

	/* Opened the image successfully? */
	if(rgba != NULL)
	{
		const gulong file_size = edv_properties_list_get_ul(
			properties_list,
			EDV_PROP_NAME_SIZE
		);
		gchar	*s,
			*play_time_sec_s,
			*frame_size_s,
			*total_size_s,
			*file_size_s,
			*modified_time_s;
		GtkEditable *editable;
		GtkText *text;

		/* Set the thumb with the opened RGBA image data */
		edv_image_prop_page_thumb_set_rgba(
			p,
			rgba,
			width, height,
			bpl,
			bg_color,
			orig_width, orig_height
		);

		/* Update the Information GtkLabel */
		play_time_sec_s = (play_time_ms != 0l) ?
			g_strdup_printf(
				"%.2f sec",
				(gfloat)play_time_ms / 1000l
			) : g_strdup("-");
		frame_size_s = g_strdup_printf(
			"%s bytes",
			edv_str_size_delim(orig_width * orig_height * 4)
		);
		total_size_s = g_strdup_printf(
			"%s bytes",
			edv_str_size_delim(orig_width * orig_height * 4 * orig_nframes)
		);
		file_size_s = g_strdup_printf(
			"%s %s",
			edv_str_size_delim(file_size),
			(file_size == 1l) ? "byte" : "bytes"
		);
		modified_time_s = (modified_time_sec != 0l) ?
			edv_date_string_format(
				modified_time_sec,
				EDV_GET_S(EDV_CFG_PARM_DATE_FORMAT),
				(EDVDateRelativity)EDV_GET_I(EDV_CFG_PARM_DATE_RELATIVITY)
			) : g_strdup("-");
		s = g_strdup_printf(
"%i\n\
%i\n\
%.2f\n\
%i\n\
%s\n\
%s\n\
%s\n\
%s\n\
%s",
			orig_width,
			orig_height,
			(orig_height > 0) ? (gfloat)orig_width / (gfloat)orig_height : 0.0f,
			orig_nframes,
			play_time_sec_s,
			frame_size_s,
			total_size_s,
			file_size_s,
			modified_time_s
		);
		g_free(play_time_sec_s);
		g_free(frame_size_s);
		g_free(total_size_s);
		g_free(file_size_s);
		g_free(modified_time_s);
		gtk_label_set_text(
			GTK_LABEL(p->info_label),
			s
		);
		g_free(s);

		/* Update the Title GtkEntry */
		title = edv_image_prop_page_single_line_string(title);
		gtk_entry_set_text(
			GTK_ENTRY(p->title_entry),
			title
		);

		/* Update the Author GtkEntry */
		author = edv_image_prop_page_single_line_string(author);
		gtk_entry_set_text(
			GTK_ENTRY(p->author_entry),
			author
		);

		/* Update the Creator GtkEntry */
		creator = edv_image_prop_page_single_line_string(creator);
		gtk_entry_set_text(
			GTK_ENTRY(p->creator_entry),
			creator
		);

		/* Update the Comments GtkText */
		text = GTK_TEXT(p->comments_text);
		editable = GTK_EDITABLE(text);
	        gtk_events_process();
	        gtk_text_freeze(text);
	        gtk_text_set_point(text, 0);
	        gtk_editable_delete_text(editable, 0, -1);
	        gtk_text_thaw(text);
		if(STRISEMPTY(comments))
		{
			gtk_widget_hide(p->comments_toplevel);
		}
		else
		{
			gtk_events_process();
	                gtk_text_freeze(text);
	                gtk_text_insert(
	                        text,
	                        NULL,
	                        NULL, NULL,
	                        comments, -1
	                );
	                gtk_text_thaw(text);
			gtk_widget_show(p->comments_toplevel);
		}

		/* Queue the thumb GtkDrawingArea to redraw after
		 * all information has been updated and events
		 * processed
		 */
		gtk_widget_queue_draw(p->thumb_da);
	}

	g_free(rgba);
	g_free(creator);
	g_free(title);
	g_free(author);
	g_free(comments);

	/* Mark that we have obtained the image information */
	p->flags |= EDV_IMAGE_PROP_PAGE_GOT_VALUES;

	p->freeze_count--;
	edv_prop_page_set_busy(ctx, FALSE);
}


/*
 *	Updates the values displayed on the GtkWidgets.
 */
static void edv_image_prop_dlg_update_widgets(EDVImagePropPage *p)
{
	GtkWidget *w;
	EDVPropPageContext *ctx = p->ctx;
	const EDVLocationType location_type = edv_prop_page_get_location_type(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);
	const gchar *path = edv_properties_list_get_s(
		properties_list,
		EDV_PROP_NAME_PATH
	);
	EDVCore *core = edv_prop_page_get_core(ctx);
	EDVMIMEType *m = edv_mime_types_list_match_path(
		core->mime_types_list,
		path
	);

	p->freeze_count++;

	/* Update the Open GtkButton's label and sensitivity */
	w = p->open_btn;
	if(m != NULL)
	{
		EDVMIMETypeCommand *command = EDV_MIME_TYPE_COMMAND(g_list_nth_data(
			m->commands_list,
			0
		));
		if(command != NULL)
		{
			GUIButtonPixmapUpdate(
				w,
				NULL,
				command->name
			);
		}
		switch(location_type)
		{
		    case EDV_LOCATION_TYPE_VFS:
			gtk_widget_set_sensitive(w, TRUE);
			break;
		    case EDV_LOCATION_TYPE_RECYCLE_BIN:
		    case EDV_LOCATION_TYPE_ARCHIVE:
			gtk_widget_set_sensitive(w, FALSE);
			break;
		}
	}
	else
	{
		GUIButtonPixmapUpdate(
		       w,
		       NULL,
		       "Open"
		);
		switch(location_type)
		{
		    case EDV_LOCATION_TYPE_VFS:
		    case EDV_LOCATION_TYPE_RECYCLE_BIN:
		    case EDV_LOCATION_TYPE_ARCHIVE:
			gtk_widget_set_sensitive(w, FALSE);
			break;
		}
	}

	p->freeze_count--;
}
