#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "url.h"
#include "cfg.h"

#include "guirgbimg.h"
#include "guiutils.h"
#include "pulist.h"
#include "cdialog.h"
#include "pdialog.h"
#include "progressdialog.h"
#include "fb.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_property.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_recycle_bin_index.h"
#include "edv_date_format.h"
#include "edv_id3.h"
#include "edv_id3_ids_list.h"
#include "edv_pixmap.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "edv_obj_info_match.h"
#include "edv_list_cb.h"
#include "edv_open.h"
#include "edv_emit.h"
#include "edv_utils_gtk.h"
#include "prop_page.h"
#include "prop_page_id3.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_add_20x20.xpm"
#include "images/icon_edit_20x20.xpm"
#include "images/icon_remove_20x20.xpm"
#include "images/icon_folder_opened_20x20.xpm"
#include "images/icon_open_20x20.xpm"
#include "images/icon_open_32x32.xpm"
#include "images/icon_save_20x20.xpm"
#include "images/icon_save_32x32.xpm"


typedef struct _EDVID3PropPage		EDVID3PropPage;
#define EDV_ID3_PROP_PAGE(p)		((EDVID3PropPage *)(p))


/*
 *	Flags:
 */
typedef enum {
	EDV_ID3_PROP_PAGE_CURRENTLY_DISPLAYED	\
					= (1 << 0),
	EDV_ID3_PROP_PAGE_GOT_VALUES	= (1 << 1),
	EDV_ID3_PROP_PAGE_HAVE_IMAGE	= (1 << 2),
	EDV_ID3_PROP_PAGE_HAS_CHANGES	= (1 << 7)
} EDVID3PropPageFlags;


/* Check Support */
gboolean edv_id3_prop_page_query_create_cb(
	EDVPropPageContext *ctx,
	gint *version_major_rtn,
	gint *version_minor_rtn,
	gint *version_release_rtn,
	gchar **page_name_rtn,
	edv_pixmap_data **pixmap_data_20x20_rtn,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list
);

/* Create */
static void edv_id3_prop_page_create_cover_page(
	EDVID3PropPage *p,
	EDVPropPageContext *ctx,
	GtkWidget *parent
);
static void edv_id3_prop_page_create_tag_page(
	EDVID3PropPage *p,
	EDVPropPageContext *ctx,
	GtkWidget *parent
);
static void edv_id3_prop_page_create_frames_page(
	EDVID3PropPage *p,
	EDVPropPageContext *ctx,
	GtkWidget *parent
);
static void edv_id3_prop_page_create_output_page(
	EDVID3PropPage *p,
	EDVPropPageContext *ctx,
	GtkWidget *parent
);
gpointer edv_id3_prop_page_create_cb(
	EDVPropPageContext *ctx,
	GtkWidget *parent
);

/* Update */
void edv_id3_prop_page_update_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list,
	const int error_code,
	gpointer data
);

/* Page switched callback */
void edv_id3_prop_page_page_switched_cb(
	EDVPropPageContext *ctx,
	const gboolean to,
	gpointer data
);

/* Apply callback */
gboolean edv_id3_prop_page_apply_vfs_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	GList *properties_list,
	gpointer data
);
gboolean edv_prop_dlg_i3d_page_apply_recycle_bin_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	GList *properties_list,
	gpointer data
);

/* Destroy */
void edv_id3_prop_page_destroy_cb(
	EDVPropPageContext *ctx,
	gpointer data
);

/* Callbacks */
static gint edv_id3_prop_page_thumb_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void edv_id3_prop_page_thumb_realize_cb(
	GtkWidget *widget, gpointer data
);
static void edv_id3_prop_page_thumb_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info,
	guint t,
	gpointer data
);
static gint edv_id3_prop_page_text_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void edv_id3_prop_page_changed_cb(GtkWidget *widget, gpointer data);
static void edv_id3_prop_page_editable_changed_cb(GtkWidget *widget, gpointer data);
static void edv_id3_prop_page_genre_changed_cb(
	GtkWidget *widget, const gint i, gpointer data
);
static void edv_id3_prop_page_tag_toggled_cb(GtkWidget *widget, gpointer data);
static gint edv_id3_prop_page_frames_list_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static gint edv_id3_prop_page_fields_list_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void edv_id3_prop_page_select_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
static void edv_id3_prop_page_unselect_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
static gint edv_id3_prop_page_get_values_idle_cb(gpointer data);
static void edv_id3_prop_page_open_cb(GtkWidget *widget, gpointer data);
static void edv_id3_prop_page_set_image_cb(GtkWidget *widget, gpointer data);
static void edv_id3_prop_page_remove_image_cb(GtkWidget *widget, gpointer data);
static void edv_id3_prop_page_extract_image_cb(GtkWidget *widget, gpointer data);
static void edv_id3_prop_page_add_frame_cb(GtkWidget *widget, gpointer data);
static void edv_id3_prop_page_edit_frame_cb(GtkWidget *widget, gpointer data);
static void edv_id3_prop_page_remove_frame_cb(GtkWidget *widget, gpointer data);
static void edv_id3_prop_page_add_field_cb(GtkWidget *widget, gpointer data);
static void edv_id3_prop_page_edit_field_cb(GtkWidget *widget, gpointer data);
static void edv_id3_prop_page_remove_field_cb(GtkWidget *widget, gpointer data);
static void edv_id3_prop_page_insert_file_cb(GtkWidget *widget, gpointer data);

/* Cleanup String */
static gchar *edv_id3_prop_page_single_line_string(gchar *s);

/* Language */
static const gchar *edv_id3_prop_page_get_language(EDVID3PropPage *p);

/* New EDVID3Frame */
static EDVID3Frame *edv_id3_prop_page_new_frame(
	EDVID3PropPage *p,
	const gint i,
	const gchar *id,
	const EDVID3TextEncoding text_encoding,
	GList *fields_list,
	const gchar *description
);

/* Get EDVID3Tag */
static EDVID3Tag *edv_id3_prop_page_get_tag(
	EDVID3PropPage *p,
	const gboolean allow_create
);

/* Thumb Draw/Set */
static void edv_id3_prop_page_thumb_draw(EDVID3PropPage *p);
static EDVPixmap *edv_id3_prop_page_thumb_set_rgba_iterate(
	EDVID3PropPage *p,
	const guint8 *rgba,
	const gint width, const gint height,
	const gint bpl,
	const guint8 *bg_color
);
static void edv_id3_prop_page_thumb_set_rgba(
	EDVID3PropPage *p,
	GList *rgba_list,
	const gint width, const gint height,
	const gint bpl,
	const gint nframes,
	const guint8 *bg_color,
	const gint orig_width, const gint orig_height
);
static void edv_id3_prop_page_thumb_clear(EDVID3PropPage *p);

/* Operations */
static void edv_id3_prop_page_open(EDVID3PropPage *p);
static gint edv_id3_prop_page_set_image_path(
	EDVID3PropPage *p,
	const gchar *path
);
static void edv_id3_prop_page_set_image_query(EDVID3PropPage *p);
static void edv_id3_prop_page_remove_image(EDVID3PropPage *p);
static gint edv_id3_prop_page_extract_image_path(
	EDVID3PropPage *p,
	const gchar *path
);
static void edv_id3_prop_page_extract_image_query(EDVID3PropPage *p);
static EDVID3Field *edv_id3_prop_page_new_field_from_file_path(
        EDVID3PropPage *p,
        const gchar *path
);
static void edv_id3_prop_page_insert_file_query(EDVID3PropPage *p);
static void edv_id3_prop_page_add_frame_query(EDVID3PropPage *p);
static void edv_id3_prop_page_edit_frame_query(EDVID3PropPage *p);
static void edv_id3_prop_page_remove_frame_query(EDVID3PropPage *p);
static void edv_id3_prop_page_add_field_query(EDVID3PropPage *p);
static void edv_id3_prop_page_edit_field_query(EDVID3PropPage *p);
static void edv_id3_prop_page_remove_field_query(EDVID3PropPage *p);

/* Get Values */
static void edv_id3_prop_page_get_values(EDVID3PropPage *p);

/* Set Values */
static gint edv_id3_prop_page_set_frame_string(
	EDVID3PropPage *p,
	EDVID3Frame *frame,
	const gchar *s
);
static gint edv_id3_prop_page_set_frame_string_by_id(
	EDVID3PropPage *p,
	const gchar *id,
	const gchar *s,
	const gboolean allow_create_frame
);

/* Save Tags To Path */
static gint edv_id3_prop_page_save_tag_to_path(
	EDVID3PropPage *p,
	const gchar *path
);

/* Update Widgets */
static void edv_id3_prop_page_update_widgets(EDVID3PropPage *p);
static void edv_id3_prop_page_update_widgets_frame(
	EDVID3PropPage *p,
	const gchar *id,
	EDVID3Frame *frame
);
static void edv_id3_prop_page_update_widgets_frame_image(
	EDVID3PropPage *p,
	EDVID3Frame *frame			/* EDV_ID3_FRAME_ID_IMAGE */
);
static void edv_id3_prop_page_frames_list_update(
	EDVID3PropPage *p,
	EDVID3Frame *frame
);
/*
static void edv_id3_prop_page_frames_list_update_all(EDVID3PropPage *p);
 */
static void edv_id3_prop_page_fields_list_update(
	EDVID3PropPage *p,
	const gint row,
	EDVID3Field *field
);
static void edv_id3_prop_page_fields_list_update_all(
	EDVID3PropPage *p,
	EDVID3Frame *frame
);


#define EDV_ID3_PROP_PAGE_NAME		"ID3"

#define EDV_ID3_PROP_PAGE_THUMB_WIDTH	100
#define EDV_ID3_PROP_PAGE_THUMB_HEIGHT	100
#define EDV_ID3_PROP_PAGE_THUMB_PADDING_X	\
					5
#define EDV_ID3_PROP_PAGE_THUMB_PADDING_Y	\
					5
#define EDV_ID3_PROP_PAGE_THUMB_SHADOW_WIDTH	\
					5
#define EDV_ID3_PROP_PAGE_THUMB_SHADOW_HEIGHT	\
					5

#define EDV_ID3_PROP_PAGE_EDITABLE_FRAME_ID_KEY	\
					"EDV/ID3PropPage/Editable/Frame/ID"

#if defined(PROG_LANGUAGE_SPANISH)
#define EDV_ID3_PROP_PAGE_DEF_LANGUAGE	\
					"ESL"
#elif defined(PROG_LANGUAGE_FRENCH)
#define EDV_ID3_PROP_PAGE_DEF_LANGUAGE	\
					"FRA"
#elif defined(PROG_LANGUAGE_GERMAN)
#define EDV_ID3_PROP_PAGE_DEF_LANGUAGE	\
					"GER"
#elif defined(PROG_LANGUAGE_ITALIAN)
#define EDV_ID3_PROP_PAGE_DEF_LANGUAGE	\
					"ITA"
#elif defined(PROG_LANGUAGE_DUTCH)
#define EDV_ID3_PROP_PAGE_DEF_LANGUAGE	\
					"DUT"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
#define EDV_ID3_PROP_PAGE_DEF_LANGUAGE	\
					"POR"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
#define EDV_ID3_PROP_PAGE_DEF_LANGUAGE	\
					"NOR"
#elif defined(PROG_LANGUAGE_POLISH)
#define EDV_ID3_PROP_PAGE_DEF_LANGUAGE	\
					"POL"
#else	/* PROG_LANGUAGE_ENGLISH */
#define EDV_ID3_PROP_PAGE_DEF_LANGUAGE	\
					"ENG"
#endif

#define EDV_ID3_PROP_PAGE_THUMB_TOOLTIP	"Click here to change the image"
#define EDV_ID3_PROP_PAGE_LENGTH_TOOLTIP	\
					"[hours:][minutes:]<seconds.decimals>"


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define FCLOSE(_fp_)	(((_fp_) != NULL) ? (gint)fclose(_fp_) : -1)


/*
 *	ID3 Properties Page:
 */
struct _EDVID3PropPage {
	GtkWidget	*toplevel;
	gint		freeze_count;
	EDVPropPageContext	*ctx;
	EDVID3PropPageFlags	flags;

	GtkWidget	*notebook,
			*thumb_toplevel,
			*thumb_da,
			*thumb_menu,
			*set_image_mi,
			*remove_image_mi,
			*extract_image_mi,
			*year_entry,
			*track_entry,
			*length_entry,
			*open_btn,
			*title_entry,
			*author_entry,
			*album_entry,
			*genre_pulistbox,
/*			*codec_entry, */
			*comments_toplevel,
			*comments_text,

			*tag_version_label,

			*flags_toplevel,
			*unsyncronization_check,
			*extended_header_check,
			*experimental_check,
			*footer_present_check,

			*extended_flags_toplevel,
			*crc_16_check,
			*restrictions_check,

			*restriction_text_toplevel,
			*restriction_text_encoding_none_radio,
			*restriction_text_encoding_latin1_utf8_radio,
			*restriction_text_size_none_radio,
			*restriction_text_size_1024_radio,
			*restriction_text_size_128_radio,
			*restriction_text_size_30_radio,

			*restriction_image_toplevel,
			*restriction_image_encoding_none_radio,
			*restriction_image_encoding_jpeg_png_radio,
			*restriction_image_size_none_radio,
			*restriction_image_size_256x256_radio,
			*restriction_image_size_64x64_radio,
			*restriction_image_size_64x64_exact_radio,

			*frames_list,
			*frames_menu,
			*frame_add_mi,
			*frame_edit_mi,
			*frame_remove_mi,
			*frame_add_btn,
			*frame_edit_btn,
			*frame_remove_btn,
			*fields_list,
			*fields_menu,
			*field_add_mi,
			*field_edit_mi,
			*field_remove_mi,
			*field_insert_file_mi,
			*field_add_btn,
			*field_edit_btn,
			*field_remove_btn,
			*field_insert_file_btn,

			*optimize_check,
			*compress_check,
			*append_check;

	EDVPixmap	*thumb;

	EDVID3Tag	*tag;			/* Currently opened
						 * EDVID3Tag */

	gchar		*last_user_selected_path;

	/* Statistics */
	guint		get_values_toid;
};


/*
 *	Check support callback.
 */
gboolean edv_id3_prop_page_query_create_cb(
	EDVPropPageContext *ctx,
	gint *version_major_rtn,
	gint *version_minor_rtn,
	gint *version_release_rtn,
	gchar **page_name_rtn,
	edv_pixmap_data **pixmap_data_20x20_rtn,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list
)
{
	*version_major_rtn = PROG_VERSION_MAJOR;
	*version_minor_rtn = PROG_VERSION_MINOR;
	*version_release_rtn = PROG_VERSION_RELEASE;
	*page_name_rtn = g_strdup(EDV_ID3_PROP_PAGE_NAME);

	switch(location_type)
	{
	    case EDV_LOCATION_TYPE_VFS:
	    case EDV_LOCATION_TYPE_RECYCLE_BIN:
		if((type == EDV_OBJECT_TYPE_FILE) ||
		   (type == EDV_OBJECT_TYPE_LINK)
		)
		{
			const gchar *name = edv_properties_list_get_s(
				properties_list,
				EDV_PROP_NAME_NAME
			);
			/* Report TRUE if this object name's extension
			 * match a format that we recognize to have
			 * ID3 tags
			 */
			if(edv_name_has_extension(
				name,
				".aac .mp2 .mp3 .mp4 .mpeg .mpg .ogg"
			))
				return(TRUE);
			else
				return(FALSE);
		}
		break;

	    case EDV_LOCATION_TYPE_ARCHIVE:
		break;
	}

	return(FALSE);
}


/*
 *	Creates the Cover Page.
 */
static void edv_id3_prop_page_create_cover_page(
	EDVID3PropPage *p,
	EDVPropPageContext *ctx,
	GtkWidget *parent
)
{
	const gint	border_major = 5,
			border_minor = 2;
	GtkRcStyle *rcstyle;
	GtkWidget	*w,
			*parent2, *parent3, *parent4, *parent5,
			*sb, *menu;
	GtkEditable *editable;
	GtkText *text;
	EDVCore *core = edv_prop_page_get_core(ctx);

	p->freeze_count++;

	/* Cover GtkFrame */
	w = gtk_frame_new("Cover");
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Cover GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent2 = w;

	/* GtkHBox for two columns */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Thumb GtkHBox */
	p->thumb_toplevel = w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Thumb GtkFrame */
	w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Thumb GtkDrawingArea */
	p->thumb_da = w = gtk_drawing_area_new();
	GTK_WIDGET_SET_FLAGS(
		w,
		GTK_CAN_FOCUS
	);
	gtk_widget_set_usize(
		w,
		EDV_ID3_PROP_PAGE_THUMB_WIDTH +
			(2 * EDV_ID3_PROP_PAGE_THUMB_PADDING_X) +
			EDV_ID3_PROP_PAGE_THUMB_SHADOW_WIDTH,
		EDV_ID3_PROP_PAGE_THUMB_HEIGHT +
			(2 * EDV_ID3_PROP_PAGE_THUMB_PADDING_Y) +
			EDV_ID3_PROP_PAGE_THUMB_SHADOW_HEIGHT
	);
	gtk_widget_add_events(
		w,
		GDK_STRUCTURE_MASK | GDK_EXPOSURE_MASK |
		GDK_FOCUS_CHANGE_MASK |
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
		GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "configure_event",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_thumb_event_cb), p
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "expose_event",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_thumb_event_cb), p
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "focus_in_event",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_thumb_event_cb), p
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "focus_out_event",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_thumb_event_cb), p
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_thumb_event_cb), p
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_thumb_event_cb), p
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_thumb_event_cb), p
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_thumb_event_cb), p
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "realize",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_thumb_realize_cb), p
	);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	if(w != NULL)
	{
		const GtkTargetEntry dnd_tar_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,	0,	EDV_DND_INFO_TEXT_PLAIN},
{GUI_TARGET_NAME_TEXT_URI_LIST,	0,	EDV_DND_INFO_TEXT_URI_LIST},
{GUI_TARGET_NAME_STRING,	0,	EDV_DND_INFO_STRING}
		};
		GUIDNDSetTar(
			w,
			dnd_tar_types,
			sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
			GDK_ACTION_COPY,	/* Actions */
			GDK_ACTION_COPY,	/* Default action if same */
			GDK_ACTION_COPY,	/* Default action */
			edv_id3_prop_page_thumb_drag_data_received_cb,
			p,
			TRUE			/* Highlight */
		);
	}
	GUISetWidgetTip(
		w,
		EDV_ID3_PROP_PAGE_THUMB_TOOLTIP
	);
	gtk_widget_show(w);

	/* Thumb GtkMenu */
	p->thumb_menu = menu = GUIMenuCreate();
	if(menu != NULL)
	{
		guint	accel_key,
			accel_mods;
		gpointer data = p;
		guint8 **icon;
		const gchar *label;
		GtkAccelGroup *accelgrp = NULL;
		void (*func_cb)(GtkWidget *w, gpointer) = NULL;

#define ADD_MENU_ITEM_LABEL	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_LABEL,			\
  accelgrp,					\
  icon, label,					\
  accel_key, accel_mods,			\
  func_cb, data					\
 );						\
}
#define ADD_MENU_ITEM_CHECK	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_CHECK,			\
  accelgrp,					\
  NULL,						\
  label,					\
  accel_key, accel_mods,			\
  func_cb, data					\
 );						\
}
#define ADD_MENU_ITEM_SUBMENU_LABEL	{	\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SUBMENU,			\
  accelgrp,					\
  icon, label,					\
  accel_key, accel_mods,			\
  func_cb, data					\
 );						\
 GUIMenuItemSetSubMenu(w, submenu);		\
}
#define ADD_MENU_ITEM_SEPARATOR	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL, NULL,					\
  0, 0,						\
  NULL, NULL					\
 );						\
}
		/* Set */
		icon = (guint8 **)icon_add_20x20_xpm;
		label = "Set Image...";
		accel_key = 0;
		accel_mods = 0;
		func_cb = edv_id3_prop_page_set_image_cb;
		ADD_MENU_ITEM_LABEL
		p->set_image_mi = w;

		/* Remove */
		icon = (guint8 **)icon_remove_20x20_xpm;
		label = "Remove Image";
		accel_key = 0;
		accel_mods = 0;
		func_cb = edv_id3_prop_page_remove_image_cb;
		ADD_MENU_ITEM_LABEL
		p->remove_image_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		/* Extract */
		icon = (guint8 **)icon_save_20x20_xpm;
		label = "Extract Image...";
		accel_key = 0;
		accel_mods = 0;
		func_cb = edv_id3_prop_page_extract_image_cb;
		ADD_MENU_ITEM_LABEL
		p->extract_image_mi = w;

#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_CHECK
#undef ADD_MENU_ITEM_SUBMENU_LABEL
#undef ADD_MENU_ITEM_SEPARATOR
	}


	/* GtkVBox for the GtkEntries */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

#define ADD_ENTRY(_parent_,_id_,_name_,_entry_rtn_)	{	\
 gchar *name = ((_name_) != NULL) ? g_strdup(_name_) :		\
  edv_id3_id_to_verbose_name(_id_);				\
 gchar *label_text = g_strconcat(name, ":", NULL);		\
 GtkWidget *w, *_parent2, *_parent3;				\
								\
 w = gtk_hbox_new(FALSE, border_minor);				\
 gtk_box_pack_start(GTK_BOX(_parent_), w, FALSE, FALSE, 0);	\
 gtk_widget_show(w);						\
 _parent2 = w;							\
								\
 w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);			\
 gtk_widget_set_usize(w, 50, -1);                               \
 gtk_box_pack_start(GTK_BOX(_parent2), w, FALSE, FALSE, 0);	\
 gtk_widget_show(w);						\
 _parent3 = w;							\
								\
 w = gtk_label_new(label_text);					\
 gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);	\
 gtk_container_add(GTK_CONTAINER(_parent3), w);			\
 gtk_widget_show(w);						\
								\
 (_entry_rtn_) = w = gtk_entry_new();				\
 gtk_object_set_data_full(					\
  GTK_OBJECT(w), EDV_ID3_PROP_PAGE_EDITABLE_FRAME_ID_KEY,	\
  g_strdup(_id_), g_free					\
 );								\
 gtk_widget_set_usize(w, 90, -1);				\
 gtk_box_pack_start(GTK_BOX(_parent2), w, FALSE, FALSE, 0);	\
 gtk_entry_set_editable(GTK_ENTRY(w), TRUE);			\
 gtk_signal_connect(						\
  GTK_OBJECT(w), "changed",					\
  GTK_SIGNAL_FUNC(edv_id3_prop_page_changed_cb), p		\
 );								\
 gtk_signal_connect(						\
  GTK_OBJECT(w), "changed",					\
  GTK_SIGNAL_FUNC(edv_id3_prop_page_editable_changed_cb), p	\
 );								\
 GUIEditableEndowPopupMenu(w, 0);				\
 gtk_widget_show(w);						\
								\
 g_free(label_text);						\
 g_free(name);							\
}
	ADD_ENTRY(parent4, EDV_ID3_FRAME_ID_YEAR, NULL, p->year_entry);
	ADD_ENTRY(parent4, EDV_ID3_FRAME_ID_TRACK, NULL, p->track_entry);
	ADD_ENTRY(parent4, EDV_ID3_FRAME_ID_LENGTH, NULL, p->length_entry);
	GUISetWidgetTip(
		p->length_entry,
		EDV_ID3_PROP_PAGE_LENGTH_TOOLTIP
	);
#undef ADD_ENTRY

	/* GtkVBox for the Open GtkButton */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	/* Open GtkButton */
	p->open_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_open_20x20_xpm,
		"Open",
		NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_open_cb), p
	);
	gtk_widget_show(w);

#define ADD_ENTRY(_parent_,_id_,_name_,_entry_rtn_)	{	\
 gchar *name = ((_name_) != NULL) ? g_strdup(_name_) :		\
  edv_id3_id_to_verbose_name(_id_);				\
 gchar *label_text = g_strconcat(name, ":", NULL);		\
 GtkWidget *w, *_parent2;					\
								\
 w = gtk_hbox_new(FALSE, border_minor);				\
 if(GTK_IS_HBOX(_parent_))					\
  gtk_box_pack_start(GTK_BOX(_parent_), w, TRUE, TRUE, 0);	\
 else								\
  gtk_box_pack_start(GTK_BOX(_parent_), w, FALSE, FALSE, 0);	\
 gtk_widget_show(w);						\
 _parent2 = w;							\
								\
 w = gtk_label_new(label_text);					\
 gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);	\
 gtk_box_pack_start(GTK_BOX(_parent2), w, FALSE, FALSE, 0);	\
 gtk_widget_show(w);						\
								\
 (_entry_rtn_) = w = gtk_entry_new();				\
 gtk_object_set_data_full(					\
  GTK_OBJECT(w), EDV_ID3_PROP_PAGE_EDITABLE_FRAME_ID_KEY,	\
  g_strdup(_id_), g_free					\
 );								\
 gtk_box_pack_start(GTK_BOX(_parent2), w, TRUE, TRUE, 0);	\
 gtk_entry_set_editable(GTK_ENTRY(w), TRUE);			\
 gtk_signal_connect(						\
  GTK_OBJECT(w), "changed",					\
  GTK_SIGNAL_FUNC(edv_id3_prop_page_changed_cb), p		\
 );								\
 gtk_signal_connect(						\
  GTK_OBJECT(w), "changed",					\
  GTK_SIGNAL_FUNC(edv_id3_prop_page_editable_changed_cb), p	\
 );								\
 GUIEditableEndowPopupMenu(w, 0);				\
 gtk_widget_show(w);						\
								\
 g_free(label_text);						\
 g_free(name);							\
}
	ADD_ENTRY(parent2, EDV_ID3_FRAME_ID_TITLE, NULL, p->title_entry);
	ADD_ENTRY(parent2, EDV_ID3_FRAME_ID_ARTIST, NULL, p->author_entry);
	ADD_ENTRY(parent2, EDV_ID3_FRAME_ID_ALBUM, NULL, p->album_entry);

	/* Genre */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	if(TRUE)
	{
		gchar *name = edv_id3_id_to_verbose_name(EDV_ID3_FRAME_ID_GENRE);
		gchar *label_text = g_strconcat(name, ":", NULL);
		w = gtk_label_new(label_text);
		gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		g_free(label_text);
		g_free(name);
	}
	p->genre_pulistbox = w = PUListBoxNew(-1, -1);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	PUListBoxSetChangedCB(
		w,
		edv_id3_prop_page_genre_changed_cb,
		p
	);
	gtk_widget_show(w);
	w = PUListBoxGetPUList(w);
	if(core->run_flags & EDV_RUN_SAFE_MODE)
		PUListSetShadowStyle(w, PULIST_SHADOW_NONE);
	if(w != NULL)
	{
		gint i;
		gchar *s;
		GList	*glist,
			*genre_list = edv_id3_genre_list_new();
		EDVID3Genre *genre;
		PUListClear(w);
		for(glist = genre_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			genre = EDV_ID3_GENRE(glist->data);
			if(genre == NULL)
				continue;

			s = g_strdup_printf(
				"%s (%i)",
				genre->name,
				(gint)genre->index
			);
			i = PUListAddItem(
				w,
				s
			);
			g_free(s);
			PUListSetItemDataFull(
				w,
				i,
				edv_id3_genre_copy(genre),
				(GtkDestroyNotify)edv_id3_genre_delete
			);
		}
		PUListBoxSetLinesVisible(
			p->genre_pulistbox,
			MIN(i + 1, 10)
		);
		genre_list = edv_id3_genre_list_delete(genre_list);
	}

/*	ADD_ENTRY(parent2, EDV_ID3_FRAME_ID_CODEC, NULL, p->codec_entry); */

#undef ADD_ENTRY

#if 0
	/* Comments GtkFrame */
	p->comments_toplevel = w = gtk_frame_new("Comments");
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;
#endif
	/* Comments GtkTable */
	p->comments_toplevel = w = gtk_table_new(1, 2, FALSE);
	gtk_table_set_row_spacing(GTK_TABLE(w), 0, border_minor);
	gtk_table_set_col_spacing(GTK_TABLE(w), 0, border_minor);
#if 0
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
#endif
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Comments Text GtkRcStyle */
	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = g_strdup(
#if defined(PROG_LANGUAGE_POLISH)
"-adobe-courier-medium-r-normal-*-12-*-*-*-*-*-iso8859-2"
#else
"-adobe-courier-medium-r-normal-*-12-*-*-*-*-*-iso8859-1"
#endif
	);
	/* GtkText */
	p->comments_text = w = gtk_text_new(NULL, NULL);
	editable = GTK_EDITABLE(w);
	text = GTK_TEXT(w);
/*	gtk_widget_set_usize(w, -1, 80); */
	text->default_tab_width = 8;
	gtk_object_set_data_full(
		GTK_OBJECT(w), EDV_ID3_PROP_PAGE_EDITABLE_FRAME_ID_KEY,
		g_strdup(EDV_ID3_FRAME_ID_COMMENTS), g_free
	);
	gtk_text_set_editable(text, TRUE);
	gtk_text_set_word_wrap(text, TRUE);
	gtk_table_attach(
		GTK_TABLE(parent3),
		w,
		0, 1,
		0, 1,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		0, 0
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_text_event_cb), p
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_text_event_cb), p
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_changed_cb), p
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_editable_changed_cb), p
	);
	gtk_widget_realize(w);
	gtk_widget_modify_style(w, rcstyle);
	GUIEditableEndowPopupMenu(w, GUI_EDITABLE_POPUP_MENU_UNDO);
	gtk_widget_show(w);
	GTK_RC_STYLE_UNREF(rcstyle);
	/* Vertical GtkScrollBar */
	sb = gtk_vscrollbar_new(GTK_TEXT(w)->vadj);
	gtk_table_attach(
		GTK_TABLE(parent3),
		sb,
		1, 2,
		0, 1,
		GTK_FILL,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		0, 0
	);
	gtk_widget_show(sb);


	p->freeze_count--;
}

/*
 *	Creates the Tag Page.
 */
static void edv_id3_prop_page_create_tag_page(
	EDVID3PropPage *p,
	EDVPropPageContext *ctx,
	GtkWidget *parent
)
{
	const gint border_major = 5;
	GSList *gslist;
	GtkWidget	*w,
			*parent2, *parent3;

	p->freeze_count++;

	/* Compatability GtkFrame */
	w = gtk_frame_new("Compatability");
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Version GtkLabel */
	w = gtk_label_new("Version:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	p->tag_version_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Flags GtkFrame */
	p->flags_toplevel = w = gtk_frame_new("Flags");
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Unsyncronization GtkCheckButton */
	p->unsyncronization_check = w = gtk_check_button_new_with_label("Unsync");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_changed_cb), p
	);
	GUISetWidgetTip(
		w,
"Check this to indicate that all frames are unsynchronized. The\
 purpose of unsyncronization is to make the ID3v2 tag more\
 compatable when inserted into MPEG and AAC files so that\
 decoders do not confuse ID3v2 data with the native format data."
	);
	gtk_widget_show(w);

	/* Extended Header GtkCheckButton */
	p->extended_header_check = w = gtk_check_button_new_with_label("Extended");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_changed_cb), p
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_tag_toggled_cb), p
	);
	GUISetWidgetTip(
		w,
"Check this to include an extended header. The purpose of the\
 extended header is to provide further information about the\
 data contained in the ID3v2 tag (such as text and image size\
 limits)."
	);
	gtk_widget_show(w);

	/* Experimental GtkCheckButton */
	p->experimental_check = w = gtk_check_button_new_with_label("Experimental");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_changed_cb), p
	);
	GUISetWidgetTip(
		w,
"Check this to indicate that the data in the ID3v2 tag is in an\
 \"experimental\" state."
	);
	gtk_widget_show(w);

	/* Footer Present GtkCheckButton */
	p->footer_present_check = w = gtk_check_button_new_with_label("Footer");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_changed_cb), p
	);
	GUISetWidgetTip(
		w,
"Check this to add a footer to the ID3v2 tag. A footer can help\
 certain programs speed up the search process for ID3v2 tags.\
 Appended ID3v2 tags (placed at the end of the file) are required\
 to have a footer."
	);
	gtk_widget_show(w);


	/* Extended Flags GtkContainer */
	p->extended_flags_toplevel = w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* CRC 16 GtkCheckButton */
	p->crc_16_check = w = gtk_check_button_new_with_label("CRC 16");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_changed_cb), p
	);
	GUISetWidgetTip(
		w,
"Check this to add CRC 16-bit checksum data to the ID3v2 tag data\
 when writing it to the file."
	);
	gtk_widget_show(w);

	/* Restrictions GtkCheckButton */
	p->restrictions_check = w = gtk_check_button_new_with_label("Restrictions");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_changed_cb), p
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_tag_toggled_cb), p
	);
	GUISetWidgetTip(
		w,
"Check this to enable additional restrictions that apply to writing\
 the ID3v2 tag to the file."
	);
	gtk_widget_show(w);


	/* Text Restrictions GtkFrame */
	p->restriction_text_toplevel = w = gtk_frame_new("Text Restrictions");
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_label_new("Encoding:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* None GtkRadioButton */
	gslist = NULL;
	p->restriction_text_encoding_none_radio = w = gtk_radio_button_new_with_label(
		gslist,
		"None"
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_changed_cb), p
	);
	gtk_widget_show(w);

	/* Latin-1/UTF-8 GtkRadioButton */
	gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));
	p->restriction_text_encoding_latin1_utf8_radio = w = gtk_radio_button_new_with_label(
		gslist,
		"Latin-1/UTF-8"
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_changed_cb), p
	);
	gtk_widget_show(w);

	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_label_new("Size:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* None GtkRadioButton */
	gslist = NULL;
	p->restriction_text_size_none_radio = w = gtk_radio_button_new_with_label(
		gslist,
		"None"
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_changed_cb), p
	);
	gtk_widget_show(w);

	/* 1024 GtkRadioButton */
	gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));
	p->restriction_text_size_1024_radio = w = gtk_radio_button_new_with_label(
		gslist,
		"< 1024"
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_changed_cb), p
	);
	gtk_widget_show(w);

	/* 128 GtkRadioButton */
	gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));
	p->restriction_text_size_128_radio = w = gtk_radio_button_new_with_label(
		gslist,
		"< 128"
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_changed_cb), p
	);
	gtk_widget_show(w);

	/* 30 GtkRadioButton */
	gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));
	p->restriction_text_size_30_radio = w = gtk_radio_button_new_with_label(
		gslist,
		"< 30"
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_changed_cb), p
	);
	gtk_widget_show(w);


	/* Image GtkFrame */
	p->restriction_image_toplevel = w = gtk_frame_new("Image Restrictions");
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_label_new("Encoding:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* None GtkRadioButton */
	gslist = NULL;
	p->restriction_image_encoding_none_radio = w = gtk_radio_button_new_with_label(
		gslist,
		"None"
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_changed_cb), p
	);
	gtk_widget_show(w);

	/* JPEG/PNG GtkRadioButton */
	gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));
	p->restriction_image_encoding_jpeg_png_radio = w = gtk_radio_button_new_with_label(
		gslist,
		"JPEG/PNG"
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_changed_cb), p
	);
	gtk_widget_show(w);


	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_label_new("Size:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* None GtkRadioButton */
	gslist = NULL;
	p->restriction_image_size_none_radio = w = gtk_radio_button_new_with_label(
		gslist,
		"None"
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_changed_cb), p
	);
	gtk_widget_show(w);

	/* 256x256 GtkRadioButton */
	gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));
	p->restriction_image_size_256x256_radio = w = gtk_radio_button_new_with_label(
		gslist,
		"<= 256x256"
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_changed_cb), p
	);
	gtk_widget_show(w);

	/* 64x64 GtkRadioButton */
	gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));
	p->restriction_image_size_64x64_radio = w = gtk_radio_button_new_with_label(
		gslist,
		"<= 64x64"
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_changed_cb), p
	);
	gtk_widget_show(w);

	/* 64x64 Exact GtkRadioButton */
	gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));
	p->restriction_image_size_64x64_exact_radio = w = gtk_radio_button_new_with_label(
		gslist,
		"= 64x64"
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_changed_cb), p
	);
	gtk_widget_show(w);



	p->freeze_count--;
}

/*
 *	Creates the Frames Page.
 */
static void edv_id3_prop_page_create_frames_page(
	EDVID3PropPage *p,
	EDVPropPageContext *ctx,
	GtkWidget *parent
)
{
	const gint	border_major = 5,
			border_minor = 2;
	gchar *heading[3];
	GtkWidget	*w,
			*parent2, *parent3, *parent4,
			*menu;
	GtkCList *clist;
	EDVCore *core = edv_prop_page_get_core(ctx);

	p->freeze_count++;

	/* Frames GtkFrame */
	w = gtk_frame_new("Frames");
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Create the GtkScrolledWindow for the Frames GtkCList */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
		GTK_SCROLLED_WINDOW(w),
		GTK_POLICY_AUTOMATIC,
		GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Frames GtkCList */
	heading[0] = "ID";
	heading[1] = "Encoding";
	heading[2] = "Description";
	p->frames_list = w = gtk_clist_new_with_titles(
		3,
		heading
	);
	clist = GTK_CLIST(w);
	gtk_widget_add_events(
		w,
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "motion_notify_event",
		GTK_SIGNAL_FUNC(edv_clist_motion_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_frames_list_event_cb), p
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_frames_list_event_cb), p
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_frames_list_event_cb), p
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_frames_list_event_cb), p
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "select_row",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_select_row_cb), p
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "unselect_row",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_unselect_row_cb), p
	);
	gtk_container_add(GTK_CONTAINER(parent4), w);
/*	gtk_widget_realize(w); */
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_clist_column_titles_show(clist);
	gtk_clist_column_titles_passive(clist);
	gtk_clist_set_column_justification(
		clist, 0, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_column_auto_resize(clist, 0, TRUE);
	gtk_clist_set_column_resizeable(clist, 0, TRUE);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_SINGLE);
	gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
	gtk_widget_show(w);

	/* Frames GtkMenu */
	p->frames_menu = menu = GUIMenuCreate();
	if(menu != NULL)
	{
		guint	accel_key,
			accel_mods;
		gpointer data = p;
		guint8 **icon;
		const gchar *label;
		GtkAccelGroup *accelgrp = NULL;
		void (*func_cb)(GtkWidget *w, gpointer) = NULL;

#define ADD_MENU_ITEM_LABEL	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_LABEL,			\
  accelgrp,					\
  icon, label,					\
  accel_key, accel_mods,			\
  func_cb, data					\
 );						\
}
#define ADD_MENU_ITEM_CHECK	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_CHECK,			\
  accelgrp,					\
  NULL,						\
  label,					\
  accel_key, accel_mods,			\
  func_cb, data					\
 );						\
}
#define ADD_MENU_ITEM_SUBMENU_LABEL	{	\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SUBMENU,			\
  accelgrp,					\
  icon, label,					\
  accel_key, accel_mods,			\
  func_cb, data					\
 );						\
 GUIMenuItemSetSubMenu(w, submenu);		\
}
#define ADD_MENU_ITEM_SEPARATOR	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL, NULL,					\
  0, 0,						\
  NULL, NULL					\
 );						\
}
		/* Add */
		icon = (guint8 **)icon_add_20x20_xpm;
		label = "Add...";
		accel_key = 0;
		accel_mods = 0;
		func_cb = edv_id3_prop_page_add_frame_cb;
		ADD_MENU_ITEM_LABEL
		p->frame_add_mi = w;

		/* Edit */
		icon = (guint8 **)icon_edit_20x20_xpm;
		label = "Edit...";
		accel_key = 0;
		accel_mods = 0;
		func_cb = edv_id3_prop_page_edit_frame_cb;
		ADD_MENU_ITEM_LABEL
		p->frame_edit_mi = w;

		/* Remove */
		icon = (guint8 **)icon_remove_20x20_xpm;
		label = "Remove";
		accel_key = 0;
		accel_mods = 0;
		func_cb = edv_id3_prop_page_remove_frame_cb;
		ADD_MENU_ITEM_LABEL
		p->frame_remove_mi = w;

#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_CHECK
#undef ADD_MENU_ITEM_SUBMENU_LABEL
#undef ADD_MENU_ITEM_SEPARATOR
	}

	/* GtkHBox for the Add, Edit, Remove, GtkButtons */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Add GtkButton */
	p->frame_add_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_add_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Agregue"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ajouter"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fgen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aggiungere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Toevoeg"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Adicione"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Tilfy"
#else
"Add"
#endif
		"...", NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_add_frame_cb), p
	);
	gtk_widget_show(w);

	/* Edit GtkButton */
	p->frame_edit_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_edit_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Redacte"
#elif defined(PROG_LANGUAGE_FRENCH)
"Editer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Redigieren"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Redigere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bewerking"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Edite"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Rediger"
#else
"Edit"
#endif
		"...", NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_edit_frame_cb), p
	);
	gtk_widget_show(w);

	/* Remove GtkButton */
	p->frame_remove_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_remove_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Quite"
#elif defined(PROG_LANGUAGE_FRENCH)
"Enlever"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nehmen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Togliere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verwijdeer"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Retire"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Fjern"
#else
"Remove"
#endif
		, NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_remove_frame_cb), p
	);
	gtk_widget_show(w);


	/* Fields GtkFrame */
	w = gtk_frame_new("Fields");
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Create the GtkScrolledWindow for the Fields GtkCList */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
		GTK_SCROLLED_WINDOW(w),
		GTK_POLICY_AUTOMATIC,
		GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Fields GtkCList */
	heading[0] = "Type";
	heading[1] = "Value";
	p->fields_list = w = gtk_clist_new_with_titles(
		2,
		heading
	);
	clist = GTK_CLIST(w);
	gtk_widget_set_usize(w, -1, 80);
	gtk_widget_add_events(
		w,
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "motion_notify_event",
		GTK_SIGNAL_FUNC(edv_clist_motion_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_fields_list_event_cb), p
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_fields_list_event_cb), p
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_fields_list_event_cb), p
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_fields_list_event_cb), p
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "select_row",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_select_row_cb), p
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "unselect_row",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_unselect_row_cb), p
	);
	gtk_container_add(GTK_CONTAINER(parent4), w);
/*	gtk_widget_realize(w); */
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_clist_column_titles_show(clist);
	gtk_clist_column_titles_passive(clist);
	gtk_clist_set_column_justification(
		clist, 0, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_column_auto_resize(clist, 0, TRUE);
	gtk_clist_set_column_resizeable(clist, 0, TRUE);
	gtk_clist_set_column_justification(
		clist, 1, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_column_auto_resize(clist, 1, TRUE);
	gtk_clist_set_column_resizeable(clist, 1, TRUE);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_SINGLE);
	gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
	gtk_widget_show(w);

	/* Fields GtkMenu */
	p->fields_menu = menu = GUIMenuCreate();
	if(menu != NULL)
	{
		guint	accel_key,
			accel_mods;
		gpointer data = p;
		guint8 **icon;
		const gchar *label;
		GtkAccelGroup *accelgrp = NULL;
		void (*func_cb)(GtkWidget *w, gpointer) = NULL;

#define ADD_MENU_ITEM_LABEL	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_LABEL,			\
  accelgrp,					\
  icon, label,					\
  accel_key, accel_mods,			\
  func_cb, data					\
 );						\
}
#define ADD_MENU_ITEM_CHECK	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_CHECK,			\
  accelgrp,					\
  NULL,						\
  label,					\
  accel_key, accel_mods,			\
  func_cb, data					\
 );						\
}
#define ADD_MENU_ITEM_SUBMENU_LABEL	{	\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SUBMENU,			\
  accelgrp,					\
  icon, label,					\
  accel_key, accel_mods,			\
  func_cb, data					\
 );						\
 GUIMenuItemSetSubMenu(w, submenu);		\
}
#define ADD_MENU_ITEM_SEPARATOR	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL, NULL,					\
  0, 0,						\
  NULL, NULL					\
 );						\
}
		/* Add */
		icon = (guint8 **)icon_add_20x20_xpm;
		label = "Add...";
		accel_key = 0;
		accel_mods = 0;
		func_cb = edv_id3_prop_page_add_field_cb;
		ADD_MENU_ITEM_LABEL
		p->field_add_mi = w;

		/* Edit */
		icon = (guint8 **)icon_edit_20x20_xpm;
		label = "Edit...";
		accel_key = 0;
		accel_mods = 0;
		func_cb = edv_id3_prop_page_edit_field_cb;
		ADD_MENU_ITEM_LABEL
		p->field_edit_mi = w;

		/* Remove */
		icon = (guint8 **)icon_remove_20x20_xpm;
		label = "Remove";
		accel_key = 0;
		accel_mods = 0;
		func_cb = edv_id3_prop_page_remove_field_cb;
		ADD_MENU_ITEM_LABEL
		p->field_remove_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		/* Insert File */
		icon = (guint8 **)icon_folder_opened_20x20_xpm;
		label = "Insert From File...";
		accel_key = 0;
		accel_mods = 0;
		func_cb = edv_id3_prop_page_insert_file_cb;
		ADD_MENU_ITEM_LABEL
		p->field_insert_file_mi = w;

#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_CHECK
#undef ADD_MENU_ITEM_SUBMENU_LABEL
#undef ADD_MENU_ITEM_SEPARATOR
	}

	/* GtkHBox for the Add, Edit, Remove, GtkButtons */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Add GtkButton */
	p->field_add_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_add_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Agregue"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ajouter"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fgen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aggiungere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Toevoeg"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Adicione"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Tilfy"
#else
"Add"
#endif
		"...", NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_add_field_cb), p
	);
	gtk_widget_show(w);

	/* Edit GtkButton */
	p->field_edit_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_edit_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Redacte"
#elif defined(PROG_LANGUAGE_FRENCH)
"Editer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Redigieren"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Redigere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bewerking"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Edite"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Rediger"
#else
"Edit"
#endif
		"...", NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_edit_field_cb), p
	);
	gtk_widget_show(w);

	/* Remove GtkButton */
	p->field_remove_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_remove_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Quite"
#elif defined(PROG_LANGUAGE_FRENCH)
"Enlever"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nehmen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Togliere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verwijdeer"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Retire"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Fjern"
#else
"Remove"
#endif
		, NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_remove_field_cb), p
	);
	gtk_widget_show(w);

	/* Insert File GtkButton */
	p->field_insert_file_btn = w = GUIButtonPixmap(
		(guint8 **)icon_folder_opened_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_id3_prop_page_insert_file_cb), p
	);
	GUISetWidgetTip(
		w,
"Click this to insert a new field containing data from an existing file"
	);
	gtk_widget_show(w);


	p->freeze_count--;
}

/*
 *	Creates the Output Options Page.
 */
static void edv_id3_prop_page_create_output_page(
	EDVID3PropPage *p,
	EDVPropPageContext *ctx,
	GtkWidget *parent
)
{
	const gint	border_major = 5;
	GtkWidget	*w,
			*parent2, *parent3;

	p->freeze_count++;


	/* Options GtkFrame */
	w = gtk_frame_new("Options");
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Optimize GtkCheckButton */
	p->optimize_check = w = gtk_check_button_new_with_label("Optimize");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);
	GUISetWidgetTip(
		w,
"Check this to minimize the size of the ID3v2 tag data (without using\
 compression) when writing it to the file. Subsequent ID3v* tags in the\
 file will also be discarded. The default value is checked."
	);
	gtk_widget_show(w);

	/* Compress GtkCheckButton */
	p->compress_check = w = gtk_check_button_new_with_label("Compress");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), FALSE);
	GUISetWidgetTip(
		w,
"Check this to compress the ID3v2 tag data using GZip (zlib) when\
 writing it to the file. Some older players are unable to play\
 audio streams with compressed ID3v2 tags. The default value is\
 unchecked."
	);
	gtk_widget_show(w);

	/* Append GtkCheckButton */
	p->append_check = w = gtk_check_button_new_with_label("Append");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), FALSE);
	GUISetWidgetTip(
		w,
"Check this to append the ID3v2 tag to the end when writing it to the\
 file. The default value is unchecked."
	);
	gtk_widget_show(w);

	p->freeze_count--;
}

/*
 *	Create callback.
 */
gpointer edv_id3_prop_page_create_cb(
	EDVPropPageContext *ctx,
	GtkWidget *parent
)
{
	const gint border_major = 5;
	GtkWidget	*w,
			*notebook;
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(g_malloc0(
		sizeof(EDVID3PropPage)
	));
	if(p == NULL)
		return(NULL);

	p->ctx = ctx;
	p->toplevel = parent;

	p->freeze_count++;

	/* Main GtkNotebook */
	p->notebook = w = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(w), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(w), TRUE);
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(w), TRUE);
	gtk_notebook_set_show_border(GTK_NOTEBOOK(w), TRUE);
	gtk_widget_show(w);
	notebook = w;

	/* Cover Page */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_notebook_append_page(
		GTK_NOTEBOOK(notebook),
		w,
		gtk_label_new("Cover")
	);
	gtk_widget_show(w);
	edv_id3_prop_page_create_cover_page(
		p,
		ctx,
		w
	);

	/* Tag Page */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_notebook_append_page(
		GTK_NOTEBOOK(notebook),
		w,
		gtk_label_new("Tag")
	);
	gtk_widget_show(w);
	edv_id3_prop_page_create_tag_page(
		p,
		ctx,
		w
	);

	/* Frames Page */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_notebook_append_page(
		GTK_NOTEBOOK(notebook),
		w,
		gtk_label_new("Frames")
	);
	gtk_widget_show(w);
	edv_id3_prop_page_create_frames_page(
		p,
		ctx,
		w
	);

	/* Output Page */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_notebook_append_page(
		GTK_NOTEBOOK(notebook),
		w,
		gtk_label_new("Output")
	);
	gtk_widget_show(w);
	edv_id3_prop_page_create_output_page(
		p,
		ctx,
		w
	);


	edv_id3_prop_page_update_widgets(p);

	p->freeze_count--;

	return(p);
}


/*
 *	Update callback.
 */
void edv_id3_prop_page_update_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list,
	const int error_code,
	gpointer data
)
{
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	/* Update the displayed information only if this page is
	 * currently displayed or we have previously obtained
	 * values that are now out of date and we did not make any
	 * changes
	 */
	if(!(p->flags & EDV_ID3_PROP_PAGE_HAS_CHANGES))
	{
		if((p->flags & EDV_ID3_PROP_PAGE_CURRENTLY_DISPLAYED) ||
		   (p->flags & EDV_ID3_PROP_PAGE_GOT_VALUES)
		)
			edv_id3_prop_page_get_values(p);
	}

	edv_id3_prop_page_update_widgets(p);

	p->freeze_count--;
}


/*
 *	Page switched callback.
 */
void edv_id3_prop_page_page_switched_cb(
	EDVPropPageContext *ctx,
	const gboolean to,
	gpointer data
)
{
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	if(to)
		p->flags |= EDV_ID3_PROP_PAGE_CURRENTLY_DISPLAYED;
	else
		p->flags &= ~EDV_ID3_PROP_PAGE_CURRENTLY_DISPLAYED;

	if(to)
	{
		/* Did not get the image information yet? */
		if(!(p->flags & EDV_ID3_PROP_PAGE_GOT_VALUES))
		{
			/* Queue get information */
			if(p->get_values_toid == 0)
				p->get_values_toid = gtk_idle_add_priority(
					G_PRIORITY_LOW,
					edv_id3_prop_page_get_values_idle_cb, p
				);
		}
	}

	p->freeze_count--;
}


/*
 *	Apply EDVVFSObject callback.
 */
gboolean edv_id3_prop_page_apply_vfs_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	GList *properties_list,
	gpointer data
)
{
	gboolean status = FALSE;
	gint status2;
	gchar *path;
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);

	if(p->freeze_count > 0)
		return(status);

	p->freeze_count++;

	/* Do not apply if we did not make any changes */
	if(!(p->flags & EDV_ID3_PROP_PAGE_HAS_CHANGES))
	{
		p->freeze_count--;
		return(status);
	}

	/* Get the path to the file */
	path = STRDUP(edv_properties_list_get_s(
		properties_list,
		EDV_PROP_NAME_PATH
	));

	/* Save the tags to the file */
	status2 = edv_id3_prop_page_save_tag_to_path(
		p,
		path
	);
	if(status2 != 0)
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"An error occured while saving the ID3 tags to:\n\
\n\
    %s\n\
\n\
%s.",
			path,
			g_strerror(error_code)
		);
		edv_play_sound_warning(core);
		edv_message_warning(
			"ID3 Tags Save Error",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
	}

	g_free(path);

	/* Remove our has changes marker */
	p->flags &= ~EDV_ID3_PROP_PAGE_HAS_CHANGES;

	p->freeze_count--;

	return(TRUE);
}

/*
 *	Apply EDVRecycledObject callback.
 */
gboolean edv_prop_dlg_i3d_page_apply_recycle_bin_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	GList *properties_list,
	gpointer data
)
{
	gboolean status = FALSE;
	gint status2;
	gulong index;
	gchar *path;
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);
	CfgList *cfg_list = edv_prop_page_get_cfg_list(ctx);
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);

	if(p->freeze_count > 0)
		return(status);

	p->freeze_count++;

	/* Do not apply if we did not make any changes */
	if(!(p->flags & EDV_ID3_PROP_PAGE_HAS_CHANGES))
	{
		p->freeze_count--;
		return(status);
	}

	/* Get the path to the file */
	index = edv_properties_list_get_ul(
		properties_list,
		EDV_PROP_NAME_INDEX
	);
	path = edv_recycle_bin_index_get_recycled_object_path(
		EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX),
		index
	);

	/* Save the tags to the file */
	status2 = edv_id3_prop_page_save_tag_to_path(
		p,
		path
	);
	if(status2 != 0)
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"An error occured while saving the ID3 tags to:\n\
\n\
    %s\n\
\n\
%s.",
			path,
			g_strerror(error_code)
		);
		edv_play_sound_warning(core);
		edv_message_warning(
			"ID3 Tags Save Error",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
	}

	g_free(path);

	/* Remove our has changes marker */
	p->flags &= ~EDV_ID3_PROP_PAGE_HAS_CHANGES;

	p->freeze_count--;

	return(TRUE);
}


/*
 *	Destroy callback.
 */
void edv_id3_prop_page_destroy_cb(
	EDVPropPageContext *ctx,
	gpointer data
)
{
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);

	p->freeze_count++;

	p->get_values_toid = GTK_TIMEOUT_REMOVE(p->get_values_toid);

	GTK_WIDGET_DESTROY(p->thumb_menu);
	p->thumb_menu = NULL;
	GTK_WIDGET_DESTROY(p->frames_menu);
	p->frames_menu = NULL;
	GTK_WIDGET_DESTROY(p->fields_menu);
	p->fields_menu = NULL;

	edv_id3_tag_delete(p->tag);
	p->tag = NULL;

	p->thumb = edv_pixmap_unref(p->thumb);

	g_free(p->last_user_selected_path);

	p->freeze_count--;

	g_free(p);
}


/*
 *	Thumb GtkDrawingArea event signal callback.
 */
static gint edv_id3_prop_page_thumb_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	gboolean press = FALSE;
	GdkEventFocus *focus;
	GdkEventKey *key;
	GdkEventButton *button;
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);
	if((widget == NULL) || (event == NULL) || (p == NULL))
		return(status);

	if(p->freeze_count > 0)
		return(status);

	p->freeze_count++;

	switch((gint)event->type)
	{
	    case GDK_EXPOSE:
		edv_id3_prop_page_thumb_draw(p);
		status = TRUE;
		break;

	    case GDK_FOCUS_CHANGE:
		focus = (GdkEventFocus *)event;
		if(focus->in && !GTK_WIDGET_HAS_FOCUS(widget))
		{
			GTK_WIDGET_SET_FLAGS(widget, GTK_HAS_FOCUS);
			gtk_widget_queue_draw(widget);
		}
		else if(!focus->in && GTK_WIDGET_HAS_FOCUS(widget))
		{
			GTK_WIDGET_UNSET_FLAGS(widget, GTK_HAS_FOCUS);
			gtk_widget_queue_draw(widget);
		}
		status = TRUE;
		break;

	    case GDK_KEY_PRESS:
		press = TRUE;
	    case GDK_KEY_RELEASE:
		key = (GdkEventKey *)event;
		switch(key->keyval)
		{
		    case GDK_Return:
		    case GDK_3270_Enter:
		    case GDK_KP_Enter:
		    case GDK_ISO_Enter:
		    case GDK_space:
			if(press)
				edv_id3_prop_page_set_image_query(p);
			status = TRUE;
			break;
		}
		break;

	    case GDK_BUTTON_PRESS:
		button = (GdkEventButton *)event;
		switch(button->button)
		{
		    case GDK_BUTTON1:
			edv_id3_prop_page_set_image_query(p);
			status = TRUE;
			break;

		    case GDK_BUTTON3:
			gtk_menu_popup(
				GTK_MENU(p->thumb_menu),
				NULL, NULL,
				NULL, NULL,
				button->button, button->time
			);
			status = TRUE;
			break;
		}
		break;

	    case GDK_BUTTON_RELEASE:
		button = (GdkEventButton *)event;
		switch(button->button)
		{
		    case GDK_BUTTON1:
			status = TRUE;
			break;

		    case GDK_BUTTON3:
			status = TRUE;
			break;
		}
		break;
	}

	p->freeze_count--;

	return(status);
}

/*
 *	Thumb GtkDrawingArea "realize" signal callback.
 */
static void edv_id3_prop_page_thumb_realize_cb(
	GtkWidget *widget, gpointer data
)
{
	GdkWindow *window;
	GtkStyle *style;
	EDVPropPageContext *ctx;
	EDVCore *core;
	CfgList *cfg_list;
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);
	if((widget == NULL) || (p == NULL))
		return;

	p->freeze_count++;

	window = widget->window;
	style = gtk_widget_get_style(widget);
	ctx = p->ctx;
	core = edv_prop_page_get_core(ctx);
	cfg_list = edv_prop_page_get_cfg_list(ctx);

	gdk_window_set_background(
		window,
		&style->bg[GTK_STATE_ACTIVE]
	);

	p->freeze_count--;
}

/*
 *	Thumb GtkDrawingArea "drag_data_received" signal callback.
 */
static void edv_id3_prop_page_thumb_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info,
	guint t,
	gpointer data
)
{
	EDVPropPageContext *ctx;
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);
	if((widget == NULL) || (dc == NULL) || (p == NULL))
		return;

	/* No source data? */
	if(selection_data == NULL)
		return;
	if(selection_data->length <= 0)
		return;

	if(p->freeze_count > 0)
		return;

	ctx = p->ctx;

	edv_prop_page_set_busy(ctx, TRUE);
	p->freeze_count++;

	if((info == EDV_DND_INFO_TEXT_PLAIN) ||
	   (info == EDV_DND_INFO_TEXT_URI_LIST) ||
	   (info == EDV_DND_INFO_STRING)
	)
	{
		GList *urls_list = url_decode(
			(const guint8 *)selection_data->data,
			selection_data->length
		);
		if(urls_list != NULL)
		{
			GList *glist;
			URLStruct *url;
			for(glist = urls_list;
			    glist != NULL;
			    glist = g_list_next(glist)
			)
			{
				url = URL(glist->data);
				if(url == NULL)
					continue;

#if 0
/* TODO handle different protocols */
				if(url->protocol != NULL)
				{

				}
#endif

				/* Add/update the EDV_ID3_FRAME_ID_IMAGE
				 * frame's data with the data from the
				 * selected image, update the thumb, and
				 * set the EDV_ID3_PROP_PAGE_HAVE_IMAGE
				 * flag
				 */
				if(edv_id3_prop_page_set_image_path(
					p,
					url->path
				) == 0)
				{
					/* Record the last user selected
					 * path
					 */
					g_free(p->last_user_selected_path);
					p->last_user_selected_path = g_strdup(url->path);
				}

				/* Mark that we have made changes */
				if(!(p->flags & EDV_ID3_PROP_PAGE_HAS_CHANGES))
				{
					p->flags |= EDV_ID3_PROP_PAGE_HAS_CHANGES;
					edv_prop_page_set_has_changes(
						ctx,
						TRUE
					);
				}
			}

			/* Delete the URLs list */
			g_list_foreach(
				urls_list,
				(GFunc)url_delete,
				NULL
			);
			g_list_free(urls_list);
		}
	}

	p->freeze_count--;
	edv_prop_page_set_busy(ctx, FALSE);
}

/*
 *	Comments GtkText any event signal callback.
 */
static gint edv_id3_prop_page_text_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	GdkEventButton *button;
	GtkText *text;
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);
	if((widget == NULL) || (event == NULL) || (p == NULL))
		return(status);

	if(p->freeze_count > 0)
		return(status);

	p->freeze_count++;

	text = GTK_TEXT(widget);

	switch((gint)event->type)
	{
	    case GDK_BUTTON_PRESS:
	       button = (GdkEventButton *)event;
		switch(button->button)
		{
		  case GDK_BUTTON4:
			/* Scroll up */
			if(text->vadj != NULL)
			{
				GtkAdjustment *adj = text->vadj;
				const gfloat inc = MAX(
					(0.25f * adj->page_size),
					adj->step_increment
				);
				gfloat v = adj->value - inc;
				if(v > (adj->upper - adj->page_size))
					v = adj->upper - adj->page_size;
				if(v < adj->lower)
					v = adj->lower;
				gtk_adjustment_set_value(adj, v);
			}
			/* Need to mark the GtkText button as 0 or else it will
			 * keep marking
			 */
			text->button = 0;
			gtk_grab_remove(widget);
			gtk_signal_emit_stop_by_name(
				GTK_OBJECT(widget), "button_press_event"
			);
			status = TRUE;
			break;

		  case GDK_BUTTON5:
			/* Scroll down */
			if(text->vadj != NULL)
			{
				GtkAdjustment *adj = text->vadj;
				const gfloat inc = MAX(
					(0.25f * adj->page_size),
					adj->step_increment
				);
				gfloat v = adj->value + inc;
				if(v > (adj->upper - adj->page_size))
					v = adj->upper - adj->page_size;
				if(v < adj->lower)
					v = adj->lower;
				gtk_adjustment_set_value(adj, v);
			}
			/* Need to mark the GtkText button as 0 or else it will
			 * keep marking
			 */
			text->button = 0;
			gtk_grab_remove(widget);
			gtk_signal_emit_stop_by_name(
				GTK_OBJECT(widget), "button_press_event"
			);
			status = TRUE;
			break;
		}
		break;

	  case GDK_BUTTON_RELEASE:
		button = (GdkEventButton *)event;
		switch(button->button)
		{
		  case GDK_BUTTON4:
			/* Need to mark the GtkText button as 0 or else it will
			 * keep marking
			 */
			text->button = 0;
			gtk_grab_remove(widget);
			gtk_signal_emit_stop_by_name(
				GTK_OBJECT(widget), "button_release_event"
			);
			status = TRUE;
			break;
		  case GDK_BUTTON5:
			/* Need to mark the GtkText button as 0 or else it will
			 * keep marking
			 */
			text->button = 0;
			gtk_grab_remove(widget);
			gtk_signal_emit_stop_by_name(
				GTK_OBJECT(widget), "button_release_event"
			);
			status = TRUE;
		       break;
		}
		break;
	}

	p->freeze_count--;

	return(status);
}


/*
 *	Any GtkWidget changed signal callback.
 */
static void edv_id3_prop_page_changed_cb(GtkWidget *widget, gpointer data)
{
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);
	EDVPropPageContext *ctx = p->ctx;

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;			/* Ignore our own update */

	/* Ignore if we have already made changes */
	if(p->flags & EDV_ID3_PROP_PAGE_HAS_CHANGES)
	{
		p->freeze_count--;
		return;
	}

	/* Mark that one of our GtkWidgets has made changes */
	p->flags |= EDV_ID3_PROP_PAGE_HAS_CHANGES;

	/* Notify the EDVPropDlg that changes have been made and
	 * emit an update (which we will ignore)
	 */
	edv_prop_page_set_has_changes(
		ctx,
		TRUE
	);

	p->freeze_count--;
}

/*
 *	Any GtkEditable changed signal callback.
 *
 *	Gets the current value from the GtkEditable and updates the
 *	EDVID3Frame's EDVID3Field values associated with the
 *	GtkEditable.
 */
static void edv_id3_prop_page_editable_changed_cb(GtkWidget *widget, gpointer data)
{
	const gchar *id;
	GtkEditable *editable = (GtkEditable *)widget;
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	/* Get the EDVID3Frame ID associated with this GtkEditable */
	id = (const gchar *)gtk_object_get_data(
		GTK_OBJECT(widget),
		EDV_ID3_PROP_PAGE_EDITABLE_FRAME_ID_KEY
	);
	if(id != NULL)
	{
		EDVID3Frame *frame;
		GtkCList *clist = GTK_CLIST(p->frames_list);
		gchar *s = gtk_editable_get_chars(
			editable,
			0, -1
		);

		/* Convert the value if this is the length frame */
		if(!g_strcasecmp(id, EDV_ID3_FRAME_ID_LENGTH))
		{
			const gulong ms = edv_id3_length_parse(s);
			g_free(s);
			s = g_strdup_printf("%ld", ms);
		}

		/* Update the string value on this EDVID3Frame */
		(void)edv_id3_prop_page_set_frame_string_by_id(
			p,
			id,
			s,
			TRUE			/* Allow create frame */
		);

		g_free(s);

		frame = edv_id3_tag_get_frame_by_id(
			p->tag,
			id
		);

		/* Update the displayed value for this EDVID3Frame
		 * on the Frames GtkCList
		 */
		edv_id3_prop_page_frames_list_update(
			p,
			frame
		);

		/* Update the displayed values for this EDVID3Frame's
		 * EDVID3Fields on the Fields GtkCList if this
		 * EDVID3Frame is currently selected
		 */
		if(clist->selection_end != NULL)
		{
			if(frame == EDV_ID3_FRAME(gtk_clist_get_row_data(
				clist,
				(gint)clist->selection_end->data
			)))
				edv_id3_prop_page_fields_list_update_all(
					p,
					frame
				);
		}
	}

	p->freeze_count--;
}

/*
 *	Genre PopupList Box changed signal callback.
 */
static void edv_id3_prop_page_genre_changed_cb(
	GtkWidget *widget, const gint i, gpointer data
)
{
	EDVID3Genre *genre;
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);
	EDVPropPageContext *ctx = p->ctx;

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	genre = EDV_ID3_GENRE(PUListBoxGetSelectedData(widget));
	if(genre != NULL)
	{
		EDVID3Frame *frame;
		GtkCList *clist = GTK_CLIST(p->frames_list);
		gchar *s = g_strdup_printf(
			"%i",
			(gint)genre->index
		);
		const gchar *id = EDV_ID3_FRAME_ID_GENRE;

		/* Update the string value on this EDVID3Frame */
		(void)edv_id3_prop_page_set_frame_string_by_id(
			p,
			id,
			s,
			TRUE			/* Allow create frame */
		);

		g_free(s);

		frame = edv_id3_tag_get_frame_by_id(
			p->tag,
			id
		);

		/* Update the displayed value for this EDVID3Frame
		 * on the Frames GtkCList
		 */
		edv_id3_prop_page_frames_list_update(
			p,
			frame
		);

		/* Update the displayed values for this EDVID3Frame's
		 * EDVID3Fields on the Fields GtkCList if this
		 * EDVID3Frame is currently selected
		 */
		if(clist->selection_end != NULL)
		{
			if(frame == EDV_ID3_FRAME(gtk_clist_get_row_data(
				clist,
				(gint)clist->selection_end->data
			)))
				edv_id3_prop_page_fields_list_update_all(
					p,
					frame
				);
		}
	}

	/* Mark that we have made changes */
	if(!(p->flags & EDV_ID3_PROP_PAGE_HAS_CHANGES))
	{
		p->flags |= EDV_ID3_PROP_PAGE_HAS_CHANGES;
		edv_prop_page_set_has_changes(
			ctx,
			TRUE
		);
	}

	p->freeze_count--;
}

/*
 *	Tag Any GtkToggleButton "toggled" signal callback.
 */
static void edv_id3_prop_page_tag_toggled_cb(GtkWidget *widget, gpointer data)
{
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	edv_id3_prop_page_update_widgets(p);

	p->freeze_count--;
}

/*
 *	Frames GtkCList event signal callback.
 */
static gint edv_id3_prop_page_frames_list_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	gboolean press = FALSE;
	GdkEventKey *key;
	GdkEventButton *button;
	GtkCList *clist;
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);
	EDVPropPageContext *ctx;
	CfgList *cfg_list;
	if((widget == NULL) || (event == NULL) || (p == NULL))
		return(status);

	if(p->freeze_count > 0)
		return(status);

	p->freeze_count++;

	clist = GTK_CLIST(widget);
	ctx = p->ctx;
	cfg_list = edv_prop_page_get_cfg_list(ctx);

	switch((gint)event->type)
	{
	    case GDK_KEY_PRESS:
		press = TRUE;
	    case GDK_KEY_RELEASE:
		key = (GdkEventKey *)event;
		switch(key->keyval)
		{
		    case GDK_Return:
		    case GDK_3270_Enter:
		    case GDK_KP_Enter:
		    case GDK_ISO_Enter:
		    case GDK_space:
			if(press)
				edv_id3_prop_page_edit_frame_query(p);
			status = TRUE;
			break;
		}
		break;

	    case GDK_BUTTON_PRESS:
		button = (GdkEventButton *)event;
		switch(button->button)
		{
			gint row, column;
		    case GDK_BUTTON3:
			if(!gtk_clist_get_selection_info(
				clist,
				(gint)button->x, (gint)button->y,
				&row, &column
			))
			{
				row = -1;
				column = 0;
			}
			/* Select the item before mapping the menu? */
			if(EDV_GET_B(EDV_CFG_PARM_RIGHT_CLICK_MENU_SELECTS) &&
			   (row >= 0) && (row < clist->rows)
			)
			{
				gtk_clist_freeze(clist);
				if(!(button->state & GDK_CONTROL_MASK) &&
				   !(button->state & GDK_SHIFT_MASK)
				)
					gtk_clist_unselect_all(clist);
				clist->focus_row = row;
				gtk_clist_select_row(
					clist,
					row, column
				);
				gtk_clist_thaw(clist);
			}
			edv_id3_prop_page_update_widgets(p);
			gtk_menu_popup(
				GTK_MENU(p->frames_menu),
				NULL, NULL,
				NULL, NULL,
				button->button, button->time
			);
			status = TRUE;
			break;
		}
		break;

	    case GDK_BUTTON_RELEASE:
		button = (GdkEventButton *)event;
		switch(button->button)
		{
		    case GDK_BUTTON3:
			status = TRUE;
			break;
		}
		break;
	}

	p->freeze_count--;

	return(status);
}

/*
 *	Fields GtkCList event signal callback.
 */
static gint edv_id3_prop_page_fields_list_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	gboolean press = FALSE;
	GdkEventKey *key;
	GdkEventButton *button;
	GtkCList *clist;
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);
	EDVPropPageContext *ctx;
	CfgList *cfg_list;
	if((widget == NULL) || (event == NULL) || (p == NULL))
		return(status);

	if(p->freeze_count > 0)
		return(status);

	p->freeze_count++;

	clist = GTK_CLIST(widget);
	ctx = p->ctx;
	cfg_list = edv_prop_page_get_cfg_list(ctx);

	switch((gint)event->type)
	{
	    case GDK_KEY_PRESS:
		press = TRUE;
	    case GDK_KEY_RELEASE:
		key = (GdkEventKey *)event;
		switch(key->keyval)
		{
		    case GDK_Return:
		    case GDK_3270_Enter:
		    case GDK_KP_Enter:
		    case GDK_ISO_Enter:
		    case GDK_space:
			if(press)
				edv_id3_prop_page_edit_field_query(p);
			status = TRUE;
			break;
		}
		break;

	    case GDK_BUTTON_PRESS:
		button = (GdkEventButton *)event;
		switch(button->button)
		{
			gint row, column;
		    case GDK_BUTTON3:
			if(!gtk_clist_get_selection_info(
				clist,
				(gint)button->x, (gint)button->y,
				&row, &column
			))
			{
				row = -1;
				column = 0;
			}
			/* Select the item before mapping the menu? */
			if(EDV_GET_B(EDV_CFG_PARM_RIGHT_CLICK_MENU_SELECTS) &&
			   (row >= 0) && (row < clist->rows)
			)
			{
				gtk_clist_freeze(clist);
				if(!(button->state & GDK_CONTROL_MASK) &&
				   !(button->state & GDK_SHIFT_MASK)
				)
					gtk_clist_unselect_all(clist);
				clist->focus_row = row;
				gtk_clist_select_row(
					clist,
					row, column
				);
				gtk_clist_thaw(clist);
			}
			edv_id3_prop_page_update_widgets(p);
			gtk_menu_popup(
				GTK_MENU(p->fields_menu),
				NULL, NULL,
				NULL, NULL,
				button->button, button->time
			);
			status = TRUE;
			break;
		}
		break;

	    case GDK_BUTTON_RELEASE:
		button = (GdkEventButton *)event;
		switch(button->button)
		{
		    case GDK_BUTTON3:
			status = TRUE;
			break;
		}
		break;
	}

	p->freeze_count--;

	return(status);
}

/*
 *	GtkCList "select_row" signal callback.
 */
static void edv_id3_prop_page_select_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	if(gtk_clist_row_is_visible(clist, row) != GTK_VISIBILITY_FULL)
		gtk_clist_moveto(
			clist,
			row, -1,
			0.5f, 0.0f
		);

	if(clist == GTK_CLIST(p->frames_list))
	{
		EDVID3Frame *frame = EDV_ID3_FRAME(gtk_clist_get_row_data(
			clist,
			row
		));
		if(frame != NULL)
		{
			/* Search through the EDVID3Tag to make sure
			 * that the EDVID3Frame exists
			 */
			if(edv_id3_tag_frame_exists(p->tag, frame))
			{
				edv_id3_prop_page_fields_list_update_all(
					p,
					frame
				);
			}

			/* Double clicked? */
			if(event != NULL)
			{
				if(event->type == GDK_2BUTTON_PRESS)
					edv_id3_prop_page_edit_frame_query(p);
			}
		}
	}
	else if(clist == GTK_CLIST(p->fields_list))
	{
		/* Double clicked? */
		if(event != NULL)
		{
			if(event->type == GDK_2BUTTON_PRESS)
				edv_id3_prop_page_edit_field_query(p);
		}
	}

	edv_id3_prop_page_update_widgets(p);

	p->freeze_count--;
}

/*
 *	GtkCList "unselect_row" signal callback.
 */
static void edv_id3_prop_page_unselect_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	if(clist == GTK_CLIST(p->frames_list))
	{
		edv_id3_prop_page_fields_list_update_all(
			p,
			NULL
		);
	}

	edv_id3_prop_page_update_widgets(p);

	p->freeze_count--;
}

/*
 *	Get values timeout callback.
 */
static gint edv_id3_prop_page_get_values_idle_cb(gpointer data)
{
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return(TRUE);

	p->freeze_count++;

	edv_id3_prop_page_get_values(p);

	p->get_values_toid = 0;

	p->freeze_count--;

	return(FALSE);
}

/*
 *	Open callback.
 */
static void edv_id3_prop_page_open_cb(GtkWidget *widget, gpointer data)
{
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	edv_id3_prop_page_open(p);

	p->freeze_count--;
}

/*
 *	Set image callback.
 */
static void edv_id3_prop_page_set_image_cb(GtkWidget *widget, gpointer data)
{
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	edv_id3_prop_page_set_image_query(p);

	p->freeze_count--;
}

/*
 *	Clear image callback.
 */
static void edv_id3_prop_page_remove_image_cb(GtkWidget *widget, gpointer data)
{
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	edv_id3_prop_page_remove_image(p);

	p->freeze_count--;
}

/*
 *	Extract image callback.
 */
static void edv_id3_prop_page_extract_image_cb(GtkWidget *widget, gpointer data)
{
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	edv_id3_prop_page_extract_image_query(p);

	p->freeze_count--;
}

/*
 *	Add frame callback.
 */
static void edv_id3_prop_page_add_frame_cb(GtkWidget *widget, gpointer data)
{
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	edv_id3_prop_page_add_frame_query(p);

	p->freeze_count--;
}

/*
 *	Edit frame callback.
 */
static void edv_id3_prop_page_edit_frame_cb(GtkWidget *widget, gpointer data)
{
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	edv_id3_prop_page_edit_frame_query(p);

	p->freeze_count--;
}

/*
 *	Remove frame callback.
 */
static void edv_id3_prop_page_remove_frame_cb(GtkWidget *widget, gpointer data)
{
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	edv_id3_prop_page_remove_frame_query(p);

	p->freeze_count--;
}

/*
 *	Add field callback.
 */
static void edv_id3_prop_page_add_field_cb(GtkWidget *widget, gpointer data)
{
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	edv_id3_prop_page_add_field_query(p);

	p->freeze_count--;
}

/*
 *	Edit field callback.
 */
static void edv_id3_prop_page_edit_field_cb(GtkWidget *widget, gpointer data)
{
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	edv_id3_prop_page_edit_field_query(p);

	p->freeze_count--;
}

/*
 *	Remove field callback.
 */
static void edv_id3_prop_page_remove_field_cb(GtkWidget *widget, gpointer data)
{
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	edv_id3_prop_page_remove_field_query(p);

	p->freeze_count--;
}

/*
 *	Insert file callback.
 */
static void edv_id3_prop_page_insert_file_cb(GtkWidget *widget, gpointer data)
{
	EDVID3PropPage *p = EDV_ID3_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	edv_id3_prop_page_insert_file_query(p);

	p->freeze_count--;
}


/*
 *	Removes all control characters from the string.
 */
static gchar *edv_id3_prop_page_single_line_string(gchar *s)
{
	gchar *s2;

	if(s == NULL)
		return(g_strdup(""));

	s2 = edv_strsub(
		s,
		"\n",
		" "
	);
	g_free(s);
	s = s2;

	s2 = edv_strsub(
		s,
		"\r",
		" "
	);
	g_free(s);
	s = s2;

	s2 = edv_strsub(
		s,
		"\f",
		" "
	);
	g_free(s);
	s = s2;

	s2 = edv_strsub(
		s,
		"\b",
		" "
	);
	g_free(s);
	s = s2;

	return(s);
}


/*
 *	Gets the current language.
 */
static const gchar *edv_id3_prop_page_get_language(EDVID3PropPage *p)
{
/* TODO */
	return(EDV_ID3_PROP_PAGE_DEF_LANGUAGE);
}


/*
 *	Creates a new EDVID3Frame on the first EDVID3Tag in the
 *	EDVID3Tags list, if the EDVID3Tags list is empty then a new
 *	EDVID3Tag will be created and appended a new EDVID3Tags list
 *	and a new EDVID3Frame will be appended to the new EDVID3Tag.
 *
 *	The p specifies the EDVID3PropPage with the EDVID3Tags list.
 *
 *	The i specifies the index position on the first EDVID3Tag to
 *	insert the new EDVID3Frame at. If i is negative then the
 *	new EDVID3Frame will be appended.
 *
 *	Returns the a new dynamically allocated EDVID3Frame or NULL
 *	on error.
 */
static EDVID3Frame *edv_id3_prop_page_new_frame(
	EDVID3PropPage *p,
	const gint i,
	const gchar *id,
	const EDVID3TextEncoding text_encoding,
	GList *fields_list,
	const gchar *description

)
{
	gint		_i,
			row;
	gchar **cell_text;
	GtkCList *clist = GTK_CLIST(p->frames_list);
	const gint ncolumns = MAX(clist->columns, 1);
	EDVID3Frame *frame;

	/* Get the first EDVID3Tag in the list or create one as needed */
	EDVID3Tag *tag = edv_id3_prop_page_get_tag(
		p,
		TRUE				/* Allow create */
	);
	if(tag == NULL)
		return(NULL);

	/* Create a new EDVID3Frame */
	frame = edv_id3_frame_new_with_values(
		id,
		text_encoding,
		fields_list,
		description
	);
	if(frame == NULL)
		return(NULL);

	/* Allocate the cell values */
	cell_text = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(_i = 0; _i < ncolumns; _i++)
		cell_text[_i] = "";

	gtk_clist_freeze(clist);

	/* Insert a new EDVID3Frame? */
	if(i > -1)
	{
		/* Insert the new EDVID3Frame to the EDVID3Tag's
		 * EDVID3Frames list
		 */
		tag->frames_list = g_list_insert(
			tag->frames_list,
			frame,
			i
		);

		/* Insert a new row on the Frames GtkCList */
		row = gtk_clist_insert(
			clist,
			i,
			cell_text
		);
	}
	else
	{
		/* Append the new EDVID3Frame to the EDVID3Tag's
		 * EDVID3Frames list
		 */
		tag->frames_list = g_list_append(
			tag->frames_list,
			frame
		);

		/* Append a new row on the Frames GtkCList */
		row = gtk_clist_append(
			clist,
			cell_text
		);
	}
	if(row > -1)
		gtk_clist_set_row_data(
			clist,
			row,
			frame			/* Shared */
		);

	g_free(cell_text);

	/* Update the displayed values for the new EDVID3Frame on the
	 * Frames GtkCList
	 */
	edv_id3_prop_page_frames_list_update(
		p,
		frame
	);
	edv_id3_prop_page_update_widgets(p);

	gtk_clist_thaw(clist);

	return(frame);
}


/*
 *	Gets the EDVID3Tag on the EDVID3Tags list, if the EDVID3Tags
 *	list is empty and allow_create is TRUE then a new EDVID3Tag
 *	will be created and appended to the EDVID3Tags list.
 *
 *	The p specifies the EDVID3PropPage with the EDVID3Tags list.
 *
 *	The i specifies the EDVID3Tag to get.
 *
 *	If allow_create is TRUE and the EDVID3Tags list is empty then
 *	a new EDVID3Tag will be created and appended to the EDVID3Tags
 *	list. 
 *
 *	Returns the EDVID3Tag or NULL on error.
 */
static EDVID3Tag *edv_id3_prop_page_get_tag(
	EDVID3PropPage *p,
	const gboolean allow_create
)
{
	if(p->tag != NULL)
		return(p->tag);

	if(allow_create)
	{
		EDVID3Tag *tag = edv_id3_tag_new();
		if(tag != NULL)
		{

			/* Set the new EDVID3Tag's version to
			 * ID3v2.4.0 since the tags that we create
			 * conform to that version
			 */
			tag->version_major = 4;
			tag->version_minor = 0;

			p->tag = tag;
		}
		return(tag);
	}

	return(NULL);
}

/*
 *	Redraws the thumb GtkDrawingArea.
 */
static void edv_id3_prop_page_thumb_draw(EDVID3PropPage *p)
{
	GtkStateType state;
	gint width, height;
	GdkWindow *window;
	GdkDrawable *drawable;
	GtkStyle *style;
	GtkWidget *w = p->thumb_da;

	if(w == NULL)
		return;

	state = GTK_WIDGET_STATE(w);
	window = w->window;
	drawable = (GdkDrawable *)window;
	style = gtk_widget_get_style(w);
	if((drawable == NULL) || (style == NULL))
		return;

	gdk_window_get_size(
		(GdkWindow *)drawable,
		&width, &height
	);
	if((width <= 0) || (height <= 0))
		return;

	/* Draw the base */
	gdk_draw_rectangle(
		drawable,
		style->bg_gc[GTK_STATE_ACTIVE],
		TRUE,				/* Fill */
		0, 0,
		width, height
	);

	/* Thumb EDVPixmap available to be drawn? */
	if(p->thumb != NULL)
	{
		gint    x, y,
			shadow_x, shadow_y;
		GdkGC *gc;
		EDVPixmap *thumb = p->thumb;
		GdkBitmap *mask = thumb->mask;
		GdkPixmap *pixmap = thumb->pixmap;

		x = (width - thumb->width - EDV_ID3_PROP_PAGE_THUMB_SHADOW_WIDTH) / 2;
		y = (height - thumb->height - EDV_ID3_PROP_PAGE_THUMB_SHADOW_HEIGHT) / 2;

		shadow_x = x + EDV_ID3_PROP_PAGE_THUMB_SHADOW_WIDTH;
		shadow_y = y + EDV_ID3_PROP_PAGE_THUMB_SHADOW_HEIGHT;

		/* Draw the shadow */
		gc = style->black_gc;
		gdk_gc_set_clip_origin(gc, shadow_x, shadow_y);
		gdk_gc_set_clip_mask(gc, mask);
		gdk_draw_rectangle(
			drawable,
			gc,
			TRUE,                   /* Fill */
			shadow_x, shadow_y,
			thumb->width, thumb->height
		);
		gdk_gc_set_clip_mask(gc, NULL);

		/* Draw the thumb */
		gc = style->white_gc;
		gdk_gc_set_clip_origin(gc, x, y);
		gdk_gc_set_clip_mask(gc, mask);
		gdk_draw_pixmap(
			drawable,
			gc,
			(GdkDrawable *)pixmap,
			0, 0,
			x, y,
			thumb->width, thumb->height
		);
		gdk_gc_set_clip_mask(gc, NULL);
	}
	else
	{
		/* No image */
		GdkFont *font = style->font;
		if(font != NULL)
		{
			const gchar *s = "No Image";
			const gint font_height = font->ascent + font->descent;
			GdkTextBounds b;
			gdk_string_bounds(font, s, &b);
	                gdk_draw_string(
	                        drawable,
				font,
				style->text_gc[GTK_STATE_ACTIVE],
				((width - b.width) / 2) - b.lbearing,
				((height - font_height) / 2) + font->ascent,
				s
			);
		}
	}

	/* Draw the focus */
	if(GTK_WIDGET_HAS_FOCUS(w) && GTK_WIDGET_SENSITIVE(w))
	{
		gtk_paint_focus(
			style,
			(GdkWindow *)drawable,
			NULL,			/* Entire area */
			w,
			NULL,			/* No detail */
			0, 0,
			width - 1, height - 1
		);
	}

	/* Send drawable to window if drawable is not the window */
	if(drawable != (GdkDrawable *)window)
		gdk_draw_pixmap(
			window,
			style->white_gc,
			drawable,
			0, 0,
			0, 0,
			width, height
		);
}

/*
 *	Called by edv_id3_prop_page_thumb_set_rgba() to create
 *	a new GdkPixmap from the RGBA image data, the specified RGBA
 *	image data will be combined with the specified background
 *	color and sent to the new GdkPixmap.
 *
 *	The specified RGBA image data should already be resized to
 *	the sized prefered by the thumb.
 */
static EDVPixmap *edv_id3_prop_page_thumb_set_rgba_iterate(
	EDVID3PropPage *p,
	const guint8 *rgba,
	const gint width, const gint height,
	const gint bpl,
	const guint8 *bg_color
)
{
	const gint	bpp = 4;		/* RGBA */
	const guint32 bg_pixel32 = (guint32)(
			((guint32)bg_color[0] << 0) &
			((guint32)bg_color[1] << 8) &
			((guint32)bg_color[2] << 16) &
			0xFF000000
	);
	gint y;
	guint8		*line_ptr, *line_end,
			*combined_rgba;
	GtkWidget *w = p->thumb_da;
	GdkWindow *window = w->window;
	GtkStyle *style = gtk_widget_get_style(w);

	/* Create the new GdkPixmap */
	EDVPixmap *thumb = edv_pixmap_new_ref();
	if(thumb == NULL)
		return(NULL);

	thumb->width = width;
	thumb->height = height;
	thumb->pixmap = gdk_pixmap_new(
		window,
		thumb->width, thumb->height,
		-1
	);
	if(thumb->pixmap == NULL)
		return(thumb);

	/* Combine the opened image data to the background and send
	 * it to the new GdkPixmap
	 *
	 * Allocate the combined RGBA image data
	 */
	combined_rgba = (guint8 *)g_malloc(
		height * bpl * sizeof(guint8)
	);
	if(combined_rgba == NULL)
		return(thumb);

	/* Clear the combined RGBA image data with the background
	 * color
	 */
	for(y = 0; y < height; y++)
	{
		for(line_ptr = combined_rgba + (y * bpl),
		    line_end = line_ptr + (width * bpp);
		    line_ptr < line_end;
		    line_ptr += bpp
		)
			*(guint32 *)line_ptr = bg_pixel32;
	}

	/* Combine/flatten the specified image data to the combined
	 * image data
	 */
	GUIImageBufferCopyArea(
		bpp,
		rgba,
		width, height, bpl,
		combined_rgba,
		width, height, bpl,
		0, 0,
		TRUE,				/* Blend */
		NULL, NULL
	);

	/* Send the combined image data to the new GdkPixmap */
	gdk_draw_rgb_32_image(
		(GdkDrawable *)thumb->pixmap,
		style->white_gc,
		0, 0,
		width, height,
		GDK_RGB_DITHER_NONE,
		combined_rgba,
		bpl
	);

	/* Delete the combined RGBA image data */
	g_free(combined_rgba);

	return(thumb);
}

/*
 *	Sets the RGBA image data for the thumb GtkDrawingArea,
 *	recreates the thumb GtkDrawingArea's GdkPixmap, and
 *	queues the thumb GtkDrawingArea to redraw.
 *
 *	Does not set or unset the EDV_ID3_PROP_PAGE_HAVE_IMAGE flag.
 */
static void edv_id3_prop_page_thumb_set_rgba(
	EDVID3PropPage *p,
	GList *rgba_list,
	const gint width, const gint height,
	const gint bpl,
	const gint nframes,
	const guint8 *bg_color,
	const gint orig_width, const gint orig_height
)
{
	const gint	bpp = 4,		/* RGBA */
			req_width = EDV_ID3_PROP_PAGE_THUMB_WIDTH,
			req_height = EDV_ID3_PROP_PAGE_THUMB_HEIGHT;
	EDVPixmap *thumb = NULL;
	GtkWidget *w = p->thumb_da;
	guint8 *rgba = (guint8 *)g_list_nth_data(
		rgba_list,
		0				/* Get the first frame */
	);

	/* Unref the thumb's old GdkPixmap */
	p->thumb = edv_pixmap_unref(p->thumb);

	/* No RGBA image data specified? */
	if((rgba == NULL) || (width <= 0) || (height <= 0) ||
	   (bpl <= 0) || (orig_width <= 0) || (orig_height <= 0)
	)
	{
		/* Queue the thumb to redraw with no GdkPixmap */
		gtk_widget_queue_draw(w);
		return;
	}

	/* Resize the image to the prefered thumb size if the
	 * specified image is larger
	 */
	if((width > req_width) || (height > req_height))
	{
		guint8 *rs_rgba;
		gfloat aspect = (orig_width > 0) ?
			(gfloat)orig_height / (gfloat)orig_width : 1.0f;
		gint	rs_width = req_width,
			rs_height = (gint)(rs_width * aspect),
			rs_bpl;
		if(rs_height > req_height)
		{
			rs_height = req_height;
			rs_width = (aspect > 0.0f) ?
				(gint)(rs_height / aspect) : req_width;
		}
		if(rs_width < 1)
			rs_width = 1;
		if(rs_height < 1)
			rs_height = 1;
		rs_bpl = rs_width * bpp;

		/* Allocate the resized RGBA image data */
		rs_rgba = (guint8 *)g_malloc(
			rs_height * rs_bpl * sizeof(guint8)
		);
		if(rs_rgba != NULL)
		{
			/* Copy/resize the specified RGBA image data
			 * to our resized RGBA image data
			 */
			GUIImageBufferResize(
				bpp,
				rgba,
				width, height, bpl,
				rs_rgba,
				rs_width, rs_height, rs_bpl,
				NULL, NULL
			);

			/* Create a new GdkPixmap from the resized
			 * RGBA image data, the specified RGBA image
			 * data will be combined with the specified
			 * background color and sent to the new
			 * GdkPixmap
			 */
			thumb = edv_id3_prop_page_thumb_set_rgba_iterate(
				p,
				rs_rgba,
				rs_width, rs_height,
				rs_bpl,
				bg_color
			);

			/* Delete the resized RGBA image data */
			g_free(rs_rgba);
		}
	}
	else
	{
		/* Create a new GdkPixmap from the RGBA image data,
		 * the specified RGBA image data will be combined with
		 * the specified background color and sent to the new
		 * GdkPixmap
		 */
		thumb = edv_id3_prop_page_thumb_set_rgba_iterate(
			p,
			rgba,
			width, height,
			bpl,
			bg_color
		);
	}

	/* Was a new EDVPixmap created? */
	if(thumb != NULL)
	{
		/* Set the thumb's new GdkPixmap */
		p->thumb = thumb;

		/* Queue the thumb to redraw */
		gtk_widget_queue_draw(w);
	}
}

/*
 *	Unrefs the thumb's GdkPixmap and queues the thumb to redraw.
 *
 *	Does not set or unset the EDV_ID3_PROP_PAGE_HAVE_IMAGE flag.
 */
static void edv_id3_prop_page_thumb_clear(EDVID3PropPage *p)
{
	/* Unref the thumb's old EDVPixmap */
	p->thumb = edv_pixmap_unref(p->thumb);

	/* Queue the thumb to redraw */
	gtk_widget_queue_draw(p->thumb_da);
}


/*
 *	Open.
 */
static void edv_id3_prop_page_open(EDVID3PropPage *p)
{
	const gchar *path;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	const EDVLocationType location_type = edv_prop_page_get_location_type(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);

	p->freeze_count++;

	switch(location_type)
	{
	    case EDV_LOCATION_TYPE_VFS:
		edv_prop_page_set_busy(ctx, TRUE);
		path = edv_properties_list_get_s(
			properties_list,
			EDV_PROP_NAME_PATH
		);
		if(path != NULL)
		{
			GList *paths_list = NULL;
			paths_list = g_list_append(
				paths_list,
				g_strdup(path)
			);
			(void)edv_open(
				core,
				paths_list,
				NULL,		/* Default command */
				toplevel,
				TRUE		/* Verbose */
			);
			g_list_foreach(paths_list, (GFunc)g_free, NULL);
			g_list_free(paths_list);
		}
		edv_prop_page_set_busy(ctx, FALSE);
		break;

	    case EDV_LOCATION_TYPE_RECYCLE_BIN:
	    case EDV_LOCATION_TYPE_ARCHIVE:
		break;
	}

	p->freeze_count--;
}

/*
 *	Add/updates the EDV_ID3_FRAME_ID_IMAGE frame, updates the
 *	thumb, and sets the EDV_ID3_PROP_PAGE_HAVE_IMAGE flag.
 *
 *	Does not set or unset the EDV_ID3_PROP_PAGE_HAS_CHANGES
 *	flag.
 *
 *	Returns 0 on success or non-zero on error.
 */
static gint edv_id3_prop_page_set_image_path(
	EDVID3PropPage *p,
	const gchar *path
)
{
	FILE *fp;
	gchar *mime_type_string = NULL;
	gulong io_buf_len = 0l;
	GtkCList *clist;
	EDVVFSObject *obj;
	EDVID3Field *field;
	EDVID3Frame *frame;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);

	edv_prop_page_set_busy(ctx, TRUE);
	p->freeze_count++;

	if(STRISEMPTY(path))
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return(-2);
	}

	/* Open the image file for reading */
	fp = fopen(
		(const char *)path,
		"rb"
	);
	if(fp == NULL)
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
			g_strerror(error_code),
			path
		);
		edv_play_sound_warning(core);
		edv_message_warning(
			"Set Image Failed",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return(-1);
	}

	/* Get the statistics of the stream */
	obj = edv_vfs_object_fstat(fileno(fp));
	if(obj != NULL)
	{
		io_buf_len = obj->block_size;
		(void)edv_match_object_type_string(
			core->mime_types_list,
			obj->type,
			path,
			obj->permissions,
			&mime_type_string
		);
		edv_vfs_object_delete(obj);
	}
	if(io_buf_len == 0l)
		io_buf_len = 1l;

	/* Get or create the EDV_ID3_FRAME_ID_IMAGE frame */
	frame = edv_id3_tag_get_frame_by_id(
		p->tag,
		EDV_ID3_FRAME_ID_IMAGE
	);
	if(frame == NULL)
		frame = edv_id3_prop_page_new_frame(
			p,
			-1,			/* Append */
			EDV_ID3_FRAME_ID_IMAGE,
			EDV_ID3_TEXT_ENCODING_ISO_8859_1,
			NULL,
			NULL
		);
	if(frame == NULL)
	{
		g_free(mime_type_string);
		(void)fclose(fp);
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return(-3);
	}

	/* Delete all the existing fields on the frame */
	frame->fields_list = edv_id3_fields_list_delete(frame->fields_list);

	/* Set the new values and append new fields to the frame */
	frame->text_encoding = EDV_ID3_TEXT_ENCODING_ISO_8859_1;

	/* MIME Type */
	field = edv_id3_frame_append_field(frame);
	if(field != NULL)
	{
		gchar *s = (mime_type_string != NULL) ?
			g_strdup(mime_type_string) : g_strdup("");
		field->type = EDV_ID3_FIELD_TYPE_MIME_TYPE;
		field->data = s;
		field->data_length = STRLEN(s) + 1;
	}

	/* Image Code */
	field = edv_id3_frame_append_field(frame);
	if(field != NULL)
	{
		gint8 x = 3;			/* 0 = Other
						 * 3 = Cover */
		field->type = EDV_ID3_FIELD_TYPE_INT8;
		field->data_length = (gint)sizeof(x);
		field->data = g_memdup(&x, field->data_length);
	}

	/* Description (EDV_ID3_FIELD_TYPE_LANGUAGE is not needed) */
	field = edv_id3_frame_append_field(frame);
	if(field != NULL)
	{
		gchar *s = g_strdup("");
		field->type = EDV_ID3_FIELD_TYPE_STRING;
		field->data = s;
		field->data_length = STRLEN(s) + 1;
	}

	/* Data */
	field = edv_id3_frame_append_field(frame);
	if(field != NULL)
	{
	        /* Begin reading the data from the file to the frame's
		 * data
		 *
		 * Allocate the I/O buffer to read each block
		 */
		guint8 *io_buf = (guint8 *)g_malloc(io_buf_len * sizeof(guint8));

		field->type = EDV_ID3_FIELD_TYPE_DATA;

		if(io_buf != NULL)
		{
			size_t units_read;

	                while(!feof(fp))
		        {
	                        /* Read this block */
		                units_read = fread(
			                io_buf,
				        sizeof(guint8),
	                                (size_t)io_buf_len,
		                        fp
			        );
				if(units_read > 0l)
				{
					const gint i = field->data_length;
					field->data_length += (gint)(units_read * sizeof(guint8));
					field->data = g_realloc(
						field->data,
						field->data_length * sizeof(guint8)
					);
					if(field->data == NULL)
					{
						field->data_length = 0;
						break;
					}
					(void)memcpy(
						(guint8 *)field->data + i,
						io_buf,
						units_read * sizeof(guint8)
					);
				}
			}

			/* Delete the I/O buffer */
			g_free(io_buf);
		}
	}

	/* Close the image file */
	(void)fclose(fp);

	/* Open the image data from the EDVID3Frame, display it on the
	 * thumb, and set the EDV_ID3_PROP_PAGE_HAVE_IMAGE flag
	 */
	edv_id3_prop_page_update_widgets_frame_image(
		p,
		frame
	);

	/* Update the displayed value for this frame on the Frames
	 * GtkCList
	 */
	edv_id3_prop_page_frames_list_update(
		p,
		frame
	);

	/* Update the displayed values on the Fields GtkCList if this
	 * EDVID3Frame is selected
	 */
	clist = GTK_CLIST(p->frames_list);
	if(clist->selection_end != NULL)
	{
		if(frame == EDV_ID3_FRAME(gtk_clist_get_row_data(
			clist,
			(gint)clist->selection_end->data
		)))
			edv_id3_prop_page_fields_list_update_all(
				p,
				frame
			);
	}

	edv_id3_prop_page_update_widgets(p);

	g_free(mime_type_string);

	p->freeze_count--;
	edv_prop_page_set_busy(ctx, FALSE);

	return(0);
}

/*
 *	Queries the user to set a new image.
 */
static void edv_id3_prop_page_set_image_query(EDVID3PropPage *p)
{
	gboolean response;
	gint            npaths = 0,
			nftypes = 0;
	gchar           *cur_path,
			**paths_list = NULL;
	fb_type_struct  **ftypes_list = NULL,
			*ftype_rtn = NULL;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	const EDVLocationType location_type = edv_prop_page_get_location_type(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);

	edv_prop_page_set_busy(ctx, TRUE);
	p->freeze_count++;

	/* Get the path to the previously selected image or the
	 * the path to the directory of the object
	 */
	cur_path = NULL;
	if(STRISEMPTY(p->last_user_selected_path))
	{
		switch(location_type)
		{
		    case EDV_LOCATION_TYPE_VFS:
			cur_path = g_dirname(edv_properties_list_get_s(
				properties_list,
				EDV_PROP_NAME_PATH
			));
			break;
		    case EDV_LOCATION_TYPE_RECYCLE_BIN:
		    case EDV_LOCATION_TYPE_ARCHIVE:
			break;
		}
	}
	else
		cur_path = g_dirname(p->last_user_selected_path);

	/* Create the file types list */
	FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		"*.*", "All Images"
	);

	/* Query the user for a new image */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
		"Set Image",
		"Set", "Cancel",
		cur_path,
		ftypes_list, nftypes,
		&paths_list, &npaths,
		&ftype_rtn
	);
	g_free(cur_path);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(response && (npaths > 0))
	{
		const gchar *path = paths_list[npaths - 1];
		if(!STRISEMPTY(path))
		{
			/* Add/update the EDV_ID3_FRAME_ID_IMAGE
			 * frame's data with the data from the
			 * selected image, update the thumb, and
			 * set the EDV_ID3_PROP_PAGE_HAVE_IMAGE
			 * flag
			 */
			if(edv_id3_prop_page_set_image_path(
				p,
				path
			) == 0)
			{
				/* Record the last user selected path */
				g_free(p->last_user_selected_path);
				p->last_user_selected_path = g_strdup(path);
			}

			/* Mark that we have made changes */
			if(!(p->flags & EDV_ID3_PROP_PAGE_HAS_CHANGES))
			{
				p->flags |= EDV_ID3_PROP_PAGE_HAS_CHANGES;
				edv_prop_page_set_has_changes(
					ctx,
					TRUE
				);
			}
		}
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	p->freeze_count--;
	edv_prop_page_set_busy(ctx, FALSE);
}

/*
 *	Removes the image.
 */
static void edv_id3_prop_page_remove_image(EDVID3PropPage *p)
{
	EDVID3Frame *frame;
	EDVPropPageContext *ctx = p->ctx;
	EDVID3Tag *tag = p->tag;

	p->freeze_count++;

	/* Remove the image frame */
	frame = edv_id3_tag_get_frame_by_id(
		tag,
		EDV_ID3_FRAME_ID_IMAGE
	);
	if(frame != NULL)
	{
		gint row;
		GtkCList *clist = GTK_CLIST(p->frames_list);

		/* If this EDVID3Frame is selected on the
		 * Frames GtkCList then clear the Fields
		 * GtkCList
		 */
		if(clist->selection_end != NULL)
		{
			if(frame == EDV_ID3_FRAME(gtk_clist_get_row_data(
				clist,
				(gint)clist->selection_end->data
			)))
				edv_id3_prop_page_fields_list_update_all(
					p,
					NULL
				);
		}

		/* Remove the row displaying this frame on
		 * the Frames GtkCList
		 */
		row = gtk_clist_find_row_from_data(
			clist,
			frame
		);
		if(row > -1)
			gtk_clist_remove(
				clist,
				row
			);

		/* Remove this EDVID3Frame from the EDVID3Tag's
		 * list
		 */
		tag->frames_list = g_list_remove(
			tag->frames_list,
			frame
		);

		/* Delete this EDVID3Frame */
		edv_id3_frame_delete(frame);
	}

	/* Clear the thumb and remove the EDV_ID3_PROP_PAGE_HAVE_IMAGE
	 * flag
	 */
	edv_id3_prop_page_update_widgets_frame_image(
		p,
		NULL
	);

	/* Mark that we have made changes */
	if(!(p->flags & EDV_ID3_PROP_PAGE_HAS_CHANGES))
	{
		p->flags |= EDV_ID3_PROP_PAGE_HAS_CHANGES;
		edv_prop_page_set_has_changes(
			ctx,
			TRUE
		);
	}

	edv_id3_prop_page_update_widgets(p);

	p->freeze_count--;
}

/*
 *	Extracts the image.
 */
static gint edv_id3_prop_page_extract_image_path(
	EDVID3PropPage *p,
	const gchar *path
)
{
	FILE *fp;
	gboolean path_existed;
	gint status;
	guint8 *io_buf;
	gulong io_buf_len = 0l;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	EDVVFSObject *obj;
	EDVCore *core = edv_prop_page_get_core(ctx);
	EDVID3Field *field;
	EDVID3Frame *frame;

	edv_prop_page_set_busy(ctx, TRUE);
	p->freeze_count++;

	if(STRISEMPTY(path))
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return(-2);
	}

	/* Get the EDVID3Frame of the image to save */
	frame = edv_id3_tag_get_frame_by_id(
		p->tag,
		EDV_ID3_FRAME_ID_IMAGE
	);
	if(frame == NULL)
	{
		edv_play_sound_warning(core);
		edv_message_warning(
			"Extract Image",
"No image (APIC) frame found.",
			NULL,
			toplevel
		);
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return(-1);
	}
	/* Get the EDVID3Field containing the image data on the
	 * EDVID3Frame
	 */
	field = edv_id3_frame_get_field_by_type(
		frame,
		EDV_ID3_FIELD_TYPE_DATA
	);
	if(field == NULL)
	{
		edv_play_sound_warning(core);
		edv_message_warning(
			"Extract Image",
"No data field found on the image (APIC) frame.",
			NULL,
			toplevel
		);
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return(-1);
	}
	if((field->data == NULL) || (field->data_length <= 0))
	{
		edv_play_sound_warning(core);
		edv_message_warning(
			"Extract Image",
"No data set on the image (APIC) frame's data field.",
			NULL,
			toplevel
		);
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return(-1);
	}

	path_existed = edv_path_exists(path);

	/* Query overwrite */
	if(path_existed)
	{
		/* Confirm overwrite */
		gint response;
		gchar *msg = g_strdup_printf(
"Overwrite existing file:\n\
\n\
    %s",
			path
		);
		edv_play_sound_warning(core);
		CDialogSetTransientFor(toplevel);
		response = CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Confirme Escriba Para Reemplazar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Confirmer craser"
#elif defined(PROG_LANGUAGE_GERMAN)
"Besttigen Sie berschreibt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Confermare Sovrascrivere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bevestiig Beschrijft"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Confirme Overwrite"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Bekreft Overskriver"
#else
"Confirm Overwrite"
#endif
			,
			msg,
			NULL,
			CDIALOG_ICON_WARNING,
			CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
			CDIALOG_BTNFLAG_NO
		);
		g_free(msg);
		CDialogSetTransientFor(NULL);
		if(response != CDIALOG_RESPONSE_YES)
		{
			p->freeze_count--;
			edv_prop_page_set_busy(ctx, FALSE);
			return(-5);
		}
	}

	/* Create/truncate/open the file for writing */
	fp = fopen(
		(const char *)path,
		"wb"
	);
	if(fp == NULL)
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
			g_strerror(error_code),
			path
		);
		edv_message_warning(
			"Extract Image Failed",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return(-1);
	}

	/* Get the stream's statistics */
	obj = edv_vfs_object_fstat((gint)fileno(fp));
	if(obj != NULL)
	{
		io_buf_len = obj->block_size;
		edv_vfs_object_delete(obj);
	}
	if(io_buf_len == 0l)
		io_buf_len = 1l;

	/* Map the progress dialog? */
	if(!ProgressDialogIsQuery())
	{
		gchar *msg = g_strdup_printf(
"Extracting Image To:\n\
\n\
    %s",
			path
		);
		ProgressDialogSetTransientFor(toplevel);
		ProgressDialogMap(
			"Extracting Image",
			msg,
			(const guint8 **)icon_save_32x32_xpm,
			"Stop"
		);
		g_free(msg);
		ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			0.0f,
			EDV_PROGRESS_BAR_NTICKS,
			TRUE
		);
		gdk_flush();
	}

	/* Allocate the IO buffer */
	io_buf = (guint8 *)g_malloc(io_buf_len * sizeof(guint8));

	/* Write the field's data to the stream */
	status = 0;
	if(io_buf != NULL)
	{
		size_t	units_to_write,
			units_written;
		const gulong data_length = field->data_length;
		const guint8	*buf = (guint8 *)field->data,
				*buf_ptr = buf,
				*buf_end = buf_ptr + data_length;
		while(buf_ptr < buf_end)
		{
			/* Calculate the size of the block to write */
			units_to_write = (size_t)MIN(
				buf_end - buf_ptr,
				io_buf_len / sizeof(guint8)
			);

			/* Write this block to the stream */
			units_written = fwrite(
				buf_ptr,
				sizeof(guint8),
				units_to_write,
				fp
			);
/*			if((units_written != units_to_write) || ferror(fp)) */
			if(units_written != units_to_write)
			{
				const gint error_code = (gint)errno;
				gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
					g_strerror(error_code),
					path
				);
				edv_message_warning(
					"Extract Image Failed",
					msg,
					NULL,
					toplevel
				);
				g_free(msg);
				status = -1;
				break;
			}

			/* Increment the data pointer position */
			buf_ptr += units_written;

			/* Report the progress and check for user abort */
			ProgressDialogUpdate(
		                NULL, NULL, NULL, NULL,
				(gfloat)(buf_ptr - buf) /
					(gfloat)data_length,
				EDV_PROGRESS_BAR_NTICKS,
	                        TRUE
		        );
			if(ProgressDialogStopCount() > 0)
			{
				status = -4;
				break;
			}
		}

		g_free(io_buf);
	}

	/* Unmap the progress dialog */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Close the file */
	if(fclose(fp) != 0)
	{
		if(status == 0)
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
				g_strerror(error_code),
				path
			);
			edv_message_warning(
				"Extract Image Failed",
				msg,
				NULL,
				toplevel
			);
			g_free(msg);
			status = -1;
		}
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return(status);
	}

	edv_id3_prop_page_update_widgets(p);

	/* Notify about the object being added or modified */
	obj = edv_vfs_object_stat(path);
	if(obj != NULL)
	{
		if(path_existed)
			edv_emit_vfs_object_modified(
				core,
				path,
				path,
				obj
			);
		else
			edv_emit_vfs_object_added(
				core,
				path,
				obj
			);
		edv_vfs_object_delete(obj);
	}

	p->freeze_count--;
	edv_prop_page_set_busy(ctx, FALSE);

	return(status);
}

/*
 *	Queries the user to extract the image.
 */
static void edv_id3_prop_page_extract_image_query(EDVID3PropPage *p)
{
	gboolean response;
	gint            npaths = 0,
			nftypes = 0;
	const gchar *mime_type_type;
	gchar           *cur_path,
			**paths_list = NULL;
	fb_type_struct  **ftypes_list = NULL,
			*ftype_rtn = NULL;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	const EDVLocationType location_type = edv_prop_page_get_location_type(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);
	EDVMIMEType *m;
	EDVID3Frame *frame;

	edv_prop_page_set_busy(ctx, TRUE);
	p->freeze_count++;

	/* Get the path to the previously selected image or the
	 * the path to the directory of the object
	 */
	cur_path = NULL;
	if(STRISEMPTY(p->last_user_selected_path))
	{
		switch(location_type)
		{
		    case EDV_LOCATION_TYPE_VFS:
			cur_path = g_dirname(edv_properties_list_get_s(
				properties_list,
				EDV_PROP_NAME_PATH
			));
			break;
		    case EDV_LOCATION_TYPE_RECYCLE_BIN:
		    case EDV_LOCATION_TYPE_ARCHIVE:
			break;
		}
	}
	else
		cur_path = g_dirname(p->last_user_selected_path);


	/* Create the file types list and get the image's MIME Type */
	m = NULL;
	mime_type_type = NULL;
	frame = edv_id3_tag_get_frame_by_id(
		p->tag,
		EDV_ID3_FRAME_ID_IMAGE
	);
	if(frame != NULL)
	{
		EDVID3Field *field = edv_id3_frame_get_field_by_type(
			frame,
			EDV_ID3_FIELD_TYPE_MIME_TYPE
		);
		mime_type_type = edv_id3_field_get_string(field);
		m = edv_mime_types_list_match_type(
			core->mime_types_list,
			NULL,
			mime_type_type,
			FALSE
		);
		if(m != NULL)
		{
			FileBrowserTypeListNew(
				&ftypes_list, &nftypes,
				m->value, m->description
			);
		}
		else if(mime_type_type != NULL)
		{
			const gchar	*name = (const gchar *)strrchr(
				(const char *)mime_type_type,
				'/'
			),
					*ext;
			gchar *extensions;
			if(name != NULL)
				name++;
			else
				name = mime_type_type;
			ext = (const gchar *)strrchr(
				(const char *)name,
				'-'
			);
			if(ext != NULL)
				ext++;
			else
				ext = name;
			extensions = g_strconcat(
				".",
				ext,
				NULL
			);
			FileBrowserTypeListNew(
				&ftypes_list, &nftypes,
				extensions, mime_type_type
			);
			g_free(extensions);
		}
	}
	FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		"*.*", "All Images"
	);

	/* Warn if no MIME Type was found for this image */
	if(mime_type_type == NULL)
	{
		edv_play_sound_warning(core);
		edv_message_warning(
			"Extract Image Warning",
"No MIME Type was found on this ID3 tag's image (APIC)\n\
frame, therefore, the format of the image data is uncertain.\n\
\n\
Most images found in ID3 tags are either Joint Photographic\n\
Experts Group (JPEG) or Portable Network Graphics (PNG) formats\n\
you can attempt to save them in either .jpg or .png formats\n\
(respectively).",
			NULL,
			toplevel
		);
	}

	/* Query the user for the image to extract to */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
		"Save Image",
		"Save", "Cancel",
		cur_path,
		ftypes_list, nftypes,
		&paths_list, &npaths,
		&ftype_rtn
	);
	g_free(cur_path);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(response && (npaths > 0))
	{
		const gchar *path = paths_list[npaths - 1];
		if(!STRISEMPTY(path))
		{
			/* Extract the image data from the
			 * EDV_ID3_FRAME_ID_IMAGE frame's data and
			 * save it to a file
			 */
			if(edv_id3_prop_page_extract_image_path(
				p,
				path
			) == 0)
			{
				/* Record the last user selected path */
				g_free(p->last_user_selected_path);
				p->last_user_selected_path = g_strdup(path);
			}
		}
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	p->freeze_count--;
	edv_prop_page_set_busy(ctx, FALSE);
}

/*
 *	Creates a new EDVID3Field with data from a file.
 */
static EDVID3Field *edv_id3_prop_page_new_field_from_file_path(
        EDVID3PropPage *p,
        const gchar *path
)
{
	FILE *fp;
	gint status;
	guint8 *io_buf;
	gulong		file_size = 0l,
			io_buf_len = 0l;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	EDVVFSObject *obj;
	EDVID3Field *field;

	edv_prop_page_set_busy(ctx, TRUE);
	p->freeze_count++;

	if(STRISEMPTY(path))
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return(NULL);
	}

	/* Open the file for reading */
	fp = fopen(
		(const char *)path,
		"rb"
	);
	if(fp == NULL)
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
			g_strerror(error_code),
			path
		);
		edv_message_warning(
			"Insert File Failed",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return(NULL);
	}

	/* Get the stream's statistics */
	obj = edv_vfs_object_fstat((gint)fileno(fp));
	if(obj != NULL)
	{
		file_size = obj->size;
		io_buf_len = obj->block_size;
		edv_vfs_object_delete(obj);
	}
	if(io_buf_len == 0l)
		io_buf_len = 1l;

	/* Allocate the read buffer */
	io_buf = (guint8 *)g_malloc(io_buf_len * sizeof(guint8));

	/* Create a new EDVID3Field */
	field = edv_id3_field_new_with_values(
		EDV_ID3_FIELD_TYPE_DATA,
		NULL, 0
	);
	if(field == NULL)
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return(NULL);
	}

	/* Map the progress dialog? */
	if(!ProgressDialogIsQuery())
	{
		gchar *msg = g_strdup_printf(
"Inserting Field From File:\n\
\n\
    %s",
			path
		);
		ProgressDialogSetTransientFor(toplevel);
		ProgressDialogMap(
			"Inserting Field From File",
			msg,
			(const guint8 **)icon_open_32x32_xpm,
			"Stop"
		);
		g_free(msg);
		ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			0.0f,
			EDV_PROGRESS_BAR_NTICKS,
			TRUE
		);
		gdk_flush();
	}

	/* Read the field's data from the stream */
	status = 0;
	if(io_buf != NULL)
	{
		gint i;
		size_t	units_to_read,
			units_read;

		while(!feof(fp))
		{
			/* Calculate the size of the block to write */
			units_to_read = (size_t)io_buf_len / sizeof(guint8);

			/* Read this block from the stream */
			units_read = fread(
				io_buf,
				sizeof(guint8),
				units_to_read,
				fp
			);
			if(ferror(fp))
			{
				const gint error_code = (gint)errno;
				gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
					g_strerror(error_code),
					path
				);
				edv_message_warning(
					"Insert File Failed",
					msg,
					NULL,
					toplevel
				);
				g_free(msg);
				status = -1;
				break;
			}

			/* Append this block to the EDVID3Field */
			i = field->data_length;
			field->data_length += (gint)(units_read * sizeof(guint8));
			field->data = g_realloc(
				field->data,
				field->data_length * sizeof(guint8)
			);
			if(field->data == NULL)
			{
				field->data_length = 0;
				status = -3;
				break;
			}
			(void)memcpy(
				((guint8 *)field->data) + i,
				io_buf,
				units_read * sizeof(guint8)
			);

			/* Report the progress and check for user abort */
			if(file_size > 0l)
			{
				ProgressDialogUpdate(
			                NULL, NULL, NULL, NULL,
					(gfloat)field->data_length /
						(gfloat)file_size,
					EDV_PROGRESS_BAR_NTICKS,
		                        TRUE
			        );
				if(ProgressDialogStopCount() > 0)
				{
					status = -4;
					break;
				}
			}
		}

		/* Delete the read buffer */
		g_free(io_buf);
	}

	/* Unmap the progress dialog */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Close the file */
	(void)fclose(fp);

	p->freeze_count--;
	edv_prop_page_set_busy(ctx, FALSE);

	return(field);
}

/*
 *	Queries the user to insert a new EDVID3Field with data
 *	from a file.
 */
static void edv_id3_prop_page_insert_file_query(EDVID3PropPage *p)
{
	gboolean response;
	gint            row,
			npaths = 0,
			nftypes = 0;
	gchar           *cur_path,
			**paths_list = NULL;
	GList *glist;
	fb_type_struct  **ftypes_list = NULL,
			*ftype_rtn = NULL;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget	*toplevel = edv_prop_page_get_toplevel(ctx),
			*w = p->frames_list;
	GtkCList *clist = GTK_CLIST(w);
	const EDVLocationType location_type = edv_prop_page_get_location_type(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);
	EDVID3Frame *frame;

	edv_prop_page_set_busy(ctx, TRUE);
	p->freeze_count++;

	/* Get the selected EDVID3Frame */
	glist = clist->selection_end;
	row = (glist != NULL) ? (gint)glist->data : -1;
	frame = EDV_ID3_FRAME(gtk_clist_get_row_data(
		clist,
		row
	));
	if(frame == NULL)
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	/* Get the path to the previously selected image or the
	 * the path to the directory of the object
	 */
	cur_path = NULL;
	if(STRISEMPTY(p->last_user_selected_path))
	{
		switch(location_type)
		{
		    case EDV_LOCATION_TYPE_VFS:
			cur_path = g_dirname(edv_properties_list_get_s(
				properties_list,
				EDV_PROP_NAME_PATH
			));
			break;
		    case EDV_LOCATION_TYPE_RECYCLE_BIN:
		    case EDV_LOCATION_TYPE_ARCHIVE:
			break;
		}
	}
	else
		cur_path = g_dirname(p->last_user_selected_path);

	/* Create the file types list and get the image's MIME Type */
	FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		"*.*", "All Files"
	);

	/* Query the user for the file to insert */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
		"Insert File",
		"Insert", "Cancel",
		cur_path,
		ftypes_list, nftypes,
		&paths_list, &npaths,
		&ftype_rtn
	);
	g_free(cur_path);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(response && (npaths > 0))
	{
		/* Open the file, read the data, create a new
		 * EDVID3Field, and add the data to the new EDVID3Field
		 */
		const gchar *path = paths_list[npaths - 1];
		EDVID3Field *field = edv_id3_prop_page_new_field_from_file_path(
			p,
			path
		);
		if(field != NULL)
		{
			const gint ncolumns = MAX(clist->columns, 1);
			gint	i,
				row,
				new_row;
			GList *glist;

			/* Put the Fields GtkCList into context */
			GtkWidget *w = p->fields_list;
			GtkCList *clist = GTK_CLIST(w);

			/* Allocate the cell values */
			gchar **cell_text = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
			for(i = 0; i < ncolumns; i++)
				cell_text[i] = "";

			glist = clist->selection_end;
			row = (glist != NULL) ? (gint)glist->data : -1;

			gtk_clist_freeze(clist);

			/* Insert or append the new row to the Fields
			 * GtkCList and the new EDVID3Field to the
			 * EDVID3Frame
			 */
			if(row > -1)
			{
				frame->fields_list = g_list_insert(
					frame->fields_list,
					field,
					row
				);
				new_row = gtk_clist_insert(
					clist,
					row,
					cell_text
				);
			}
			else
			{
				frame->fields_list = g_list_append(
					frame->fields_list,
					field
				);
				new_row = gtk_clist_append(
					clist,
					cell_text
				);
			}

			g_free(cell_text);

			if(new_row > -1)
			{
				gtk_clist_unselect_all(clist);
				gtk_clist_select_row(
					clist,
					new_row, -1
				);
			}

			/* Update the displayed values for this
			 * EDVID3Field on the Fields GtkCList
			 */
			edv_id3_prop_page_fields_list_update(
				p,
				new_row,
				field
			);

			/* Record the last file path set by the user */
			g_free(p->last_user_selected_path);
			p->last_user_selected_path = g_strdup(path);

			gtk_clist_thaw(clist);

			edv_id3_prop_page_update_widgets(p);
		}
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	p->freeze_count--;
	edv_prop_page_set_busy(ctx, FALSE);
}

/*
 *	Queries the user to add a frame.
 */
static void edv_id3_prop_page_add_frame_query(EDVID3PropPage *p)
{
	gint i, row;
	const gchar *id = "NEW";
	gchar **cell_text;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *w = p->frames_list;
	GList *glist;
	GtkCList *clist = GTK_CLIST(w);
	const gint ncolumns = MAX(clist->columns, 1);
	EDVID3Frame *frame;

	edv_prop_page_set_busy(ctx, TRUE);
	p->freeze_count++;

	/* Allocate the cell values */
	cell_text = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
		cell_text[i] = "";

	/* Set the default cell values */
	if(ncolumns > 0)
		cell_text[0] = (gchar *)id;

	gtk_clist_freeze(clist);

	frame = NULL;

	/* If a row was selected on the Frames GtkCList then attempt
	 * to insert a new EDVID3Frame (frame will != NULL and row
	 * will be set if a new EDVID3Frame was successfully inserted)
	 */
	glist = clist->selection_end;
	row = (glist != NULL) ? (gint)glist->data : -1;
	if(row > -1)
	{
		gint i;

		/* Find the EDVID3Tag in the list that contains the
		 * current selected EDVID3Frame
		 */
		EDVID3Frame *selected_frame = EDV_ID3_FRAME(gtk_clist_get_row_data(
			clist,
			row
		));

		/* Get/create the EDVID3Tag */
		EDVID3Tag *tag = edv_id3_prop_page_get_tag(
			p,
			TRUE			/* Allow create */
		);
		if(tag != NULL)
		{
			/* Is the selected EDVID3Frame on this
			 * EDVID3Tag?
			 */
			i = g_list_index(
				tag->frames_list,
				selected_frame
			);
			if(i > -1)
			{
				/* Insert a new EDVID3Frame on to
				 * this EDVID3Tag and insert a new
				 * row on the Frames GtkCList
				 */
				frame = edv_id3_frame_new_with_values(
					id,
					EDV_ID3_TEXT_ENCODING_ISO_8859_1,
					NULL,
					NULL
				);
				if(frame != NULL)
				{
					tag->frames_list = g_list_insert(
						tag->frames_list,
						frame,
						i
					);
					row = gtk_clist_insert(
						clist,
						row,
						cell_text
					);
					gtk_clist_set_row_data(
						clist,
						row,
						frame/* Shared */
					);
				}
			}
		}
	}
	/* If no new EDVID3Frame was created above then it means we
	 * need to append a new EDVID3Frame
	 */
	if(frame == NULL)
	{
		/* Append a new EDVID3Frame
		 *
		 * Get the last EDVID3Tag or append a new one as needed
		 */
		EDVID3Tag *tag = edv_id3_prop_page_get_tag(
			p,
			TRUE
		);
		if(tag != NULL)
		{
			/* Append a new EDVID3Frame to this
			 * EDVID3Tag and append a new row to the
			 * Frames GtkCList
			 */
			frame = edv_id3_frame_new_with_values(
				id,
				EDV_ID3_TEXT_ENCODING_ISO_8859_1,
				NULL,
				NULL
			);
			if(frame != NULL)
			{
				tag->frames_list = g_list_append(
					tag->frames_list,
					frame
				);
				row = gtk_clist_append(
					clist,
					cell_text
				);
				gtk_clist_set_row_data(
					clist,
					row,
					frame	/* Shared */
				);
			}
		}
	}

	g_free(cell_text);

	/* Failed to create a new EDVID3Frame? */
	if(frame == NULL)
	{
		gtk_clist_thaw(clist);
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	/* Update the displayed values for the new EDVID3Frame on the
	 * Frames GtkCList
	 */
	edv_id3_prop_page_frames_list_update(
		p,
		frame
	);

	/* Select the new row (but callbacks will not be frozen) */
	gtk_clist_unselect_all(clist);
	gtk_clist_select_row(
		clist,
		row, -1
	);
	if(gtk_clist_row_is_visible(clist, row) != GTK_VISIBILITY_FULL)
		gtk_clist_moveto(
			clist,
			row, -1,
			0.5f, 0.0f
		);

	gtk_clist_thaw(clist);

	/* Mark that we have made changes */
	if(!(p->flags & EDV_ID3_PROP_PAGE_HAS_CHANGES))
	{
		p->flags |= EDV_ID3_PROP_PAGE_HAS_CHANGES;
		edv_prop_page_set_has_changes(
			ctx,
			TRUE
		);
	}

	edv_id3_prop_page_fields_list_update_all(
		p,
		frame
	);

	edv_id3_prop_page_update_widgets(p);

	edv_id3_prop_page_edit_frame_query(p);

	p->freeze_count--;
	edv_prop_page_set_busy(ctx, FALSE);
}

/*
 *	Queries the user to edit a frame.
 */
static void edv_id3_prop_page_edit_frame_query(EDVID3PropPage *p)
{
	gint		strc,
			row;
	gchar **strv;
	GList		*glist,
			*text_encodings_list;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);
	GtkWidget *w = p->frames_list;
	GtkCList *clist = GTK_CLIST(w);
	EDVID3Frame *frame;

	edv_prop_page_set_busy(ctx, TRUE);
	p->freeze_count++;

	/* Get the selected EDVID3Frame */
	glist = clist->selection_end;
	if(glist == NULL)
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	row = (gint)glist->data;

	frame = EDV_ID3_FRAME(gtk_clist_get_row_data(
		clist,
		row
	));
	if(frame == NULL)
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	/* Create the EDVID3TextEncodings list */
	text_encodings_list = NULL;
	text_encodings_list = g_list_append(
		text_encodings_list,
		g_strdup("UNKNOWN")
	);
	text_encodings_list = g_list_append(
		text_encodings_list,
		g_strdup("UTF-8")
	);
	text_encodings_list = g_list_append(
		text_encodings_list,
		g_strdup("UTF-16")
	);
	text_encodings_list = g_list_append(
		text_encodings_list,
		g_strdup("UTF-16 BE")
	);
	text_encodings_list = g_list_append(
		text_encodings_list,
		g_strdup("ISO-8859-1")
	);

	/* Set up the Prompt Dialog and query the user for new values */
	PDialogDeleteAllPrompts();
	PDialogSetSize(320, -1);
	PDialogAddPrompt(
		NULL,
		"ID:",
		frame->id
	);
	PDialogAddPromptPopupList(
		NULL,
		"Text Encoding:",
		text_encodings_list,
		(gint)frame->text_encoding,
		5
	);
	PDialogAddPrompt(
		NULL,
		"Description:",
		frame->description
	);
	PDialogSetTransientFor(toplevel);
	strv = PDialogGetResponse(
		"Edit Frame",
		NULL,
		NULL,
		PDIALOG_ICON_EDIT,
		"Set", "Cancel",
		PDIALOG_BTNFLAG_SUBMIT | PDIALOG_BTNFLAG_CANCEL,
		PDIALOG_BTNFLAG_SUBMIT,
		&strc
	);
	if((strv != NULL) && (strc >= 3))
	{
		const gchar *id = strv[0];
		gint response = CDIALOG_RESPONSE_YES;
		EDVID3Frame *frame2;

		/* Warn if another EDVID3Frame already has this ID */
		frame2 = edv_id3_tag_get_frame_by_id(
			p->tag,
			id
		);
		if((frame2 != NULL) && (frame2 != frame))
		{
			gchar *msg = g_strdup_printf(
"Another frame already exists with the ID \"%s\",\n\
are you sure you want to set this field's values?",
				id
			);
			edv_play_sound_warning(core);
			CDialogSetTransientFor(toplevel);
			response = CDialogGetResponse(
				"Another Frame ID Exists",
				msg,
				NULL,
				CDIALOG_ICON_WARNING,
				CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
				CDIALOG_BTNFLAG_NO
			);
			g_free(msg);
			CDialogSetTransientFor(NULL);
		}
		if(response == CDIALOG_RESPONSE_YES)
		{
			gint i;
			GList *glist;

			/* Set the ID */
			g_free(frame->id);
			frame->id = STRDUP(id);

			/* Determine and set the EDVID3FieldType */
			for(i = 0,
			    glist = text_encodings_list;
			    glist != NULL;
			    i++,
			    glist = g_list_next(glist)
			)
			{
				if(!g_strcasecmp(
					(const gchar *)glist->data,
					strv[1]
				))
				{
					frame->text_encoding = (EDVID3TextEncoding)i;
					break;
				}
			}

			/* Set description */
			g_free(frame->description);
			frame->description = STRDUP(strv[2]);

			/* Update the displayed values for this
			 * EDVID3Frame
			 */
			edv_id3_prop_page_update_widgets_frame(
				p,
				frame->id,
				frame
			);

	                /* Mark that we have made changes */
	                if(!(p->flags & EDV_ID3_PROP_PAGE_HAS_CHANGES))
	                {
	                        p->flags |= EDV_ID3_PROP_PAGE_HAS_CHANGES;
	       		        edv_prop_page_set_has_changes(
					ctx,
					TRUE
				);
			}

			edv_id3_prop_page_update_widgets(p);
		}
	}

	/* Delete the EDVID3TextEncodings list */
	g_list_foreach(text_encodings_list, (GFunc)g_free, NULL);
	g_list_free(text_encodings_list);

	p->freeze_count--;
	edv_prop_page_set_busy(ctx, FALSE);
}

/*
 *	Queries the user to remove a frame.
 */
static void edv_id3_prop_page_remove_frame_query(EDVID3PropPage *p)
{
	gint		response,
			row;
	gchar *msg;
	GList *glist;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	GtkWidget *w = p->frames_list;
	GtkCList *clist = GTK_CLIST(w);
	EDVCore *core = edv_prop_page_get_core(ctx);
	EDVID3Frame *frame;
	EDVID3Tag *tag;

	edv_prop_page_set_busy(ctx, TRUE);
	p->freeze_count++;

	/* Get the selected EDVID3Frame */
	glist = clist->selection_end;
	if(glist == NULL)
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	row = (gint)glist->data;

	frame = EDV_ID3_FRAME(gtk_clist_get_row_data(
		clist,
		row
	));
	if(frame == NULL)
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	/* Confirm remove */
	msg = g_strdup_printf(
"Remove frame \"%s\"?",
		frame->id
	);
	edv_play_sound_question(core);
	CDialogSetTransientFor(toplevel);
	response = CDialogGetResponse(
		"Confirm Remove",
		msg,
		NULL,
		CDIALOG_ICON_QUESTION,
		CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
		CDIALOG_BTNFLAG_NO
	);
	g_free(msg);
	CDialogSetTransientFor(NULL);

	if(response != CDIALOG_RESPONSE_YES)
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	/* Clear all values on GtkWidgets displaying this frame */
	edv_id3_prop_page_update_widgets_frame(
		p,
		frame->id,
		NULL				/* Clear values */
	);
	edv_id3_prop_page_fields_list_update_all(
		p,
		NULL
	);

	gtk_clist_freeze(clist);

	tag = p->tag;
	if(tag != NULL)
	{
		if(g_list_index(tag->frames_list, frame) > -1)
		{
			gtk_clist_remove(
				clist,
				row
			);
			tag->frames_list = g_list_remove(
				tag->frames_list,
				frame
			);
			edv_id3_frame_delete(frame);
		}
	}
	gtk_clist_thaw(clist);

	/* Mark that we have made changes */
	if(!(p->flags & EDV_ID3_PROP_PAGE_HAS_CHANGES))
	{
		p->flags |= EDV_ID3_PROP_PAGE_HAS_CHANGES;
		edv_prop_page_set_has_changes(
			ctx,
			TRUE
		);
	}

	edv_id3_prop_page_update_widgets(p);

	p->freeze_count--;
	edv_prop_page_set_busy(ctx, FALSE);
}

/*
 *	Queries the user to add a field.
 */
static void edv_id3_prop_page_add_field_query(EDVID3PropPage *p)
{
	gint		i,
			row,
			new_row;
	gchar **cell_text;
	GList *glist;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *w = p->frames_list;
	GtkCList *clist = GTK_CLIST(w);
	const gint ncolumns = MAX(clist->columns, 1);
	EDVID3Field *field;
	EDVID3Frame *frame;

	edv_prop_page_set_busy(ctx, TRUE);
	p->freeze_count++;

	/* Get the selected EDVID3Frame */
	glist = clist->selection_end;
	if(glist == NULL)
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	row = (gint)glist->data;

	frame = EDV_ID3_FRAME(gtk_clist_get_row_data(
		clist,
		row
	));
	if(frame == NULL)
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	/* Put the Fields GtkCList into context */
	w = p->fields_list;
	clist = GTK_CLIST(w);
	glist = clist->selection_end;
	row = (glist != NULL) ? (gint)glist->data : -1;

	/* Create a new EDVID3Field and insert it to the EDVID3Frame's
	 * list and Fields GtkCList
	 */
	field = edv_id3_field_new();
	if(field == NULL)
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
	}

	cell_text = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
		cell_text[i] = "";

	gtk_clist_freeze(clist);

	if(row > -1)
	{
		frame->fields_list = g_list_insert(
			frame->fields_list,
			field,
			row
		);
		new_row = gtk_clist_insert(
			clist,
			row,
			cell_text
		);
	}
	else
	{
		frame->fields_list = g_list_append(
			frame->fields_list,
			field
		);
		new_row = gtk_clist_append(
			clist,
			cell_text
		);
	}

	g_free(cell_text);

	if(new_row < 0)
	{
		gtk_clist_thaw(clist);
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
	}

	/* Select the new row (but callbacks will not be frozen) */
	gtk_clist_unselect_all(clist);
	gtk_clist_select_row(
		clist,
		new_row, -1
	);
	if(gtk_clist_row_is_visible(clist, new_row) != GTK_VISIBILITY_FULL)
		gtk_clist_moveto(
			clist,
			new_row, -1,
			0.5f, 0.0f
		);

	gtk_clist_thaw(clist);

	/* Mark that we have made changes */
	if(!(p->flags & EDV_ID3_PROP_PAGE_HAS_CHANGES))
	{
		p->flags |= EDV_ID3_PROP_PAGE_HAS_CHANGES;
		edv_prop_page_set_has_changes(
			ctx,
			TRUE
		);
	}

	edv_id3_prop_page_update_widgets(p);

	edv_id3_prop_page_edit_field_query(p);

	p->freeze_count--;
	edv_prop_page_set_busy(ctx, FALSE);
}

/*
 *	Queries the user to edit a field.
 */
static void edv_id3_prop_page_edit_field_query(EDVID3PropPage *p)
{
	gint		strc,
			row;
	gchar		*s,
			**strv;
	GList		*glist,
			*types_list;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);
	GtkWidget *w = p->frames_list;
	GtkCList *clist = GTK_CLIST(w);
	EDVID3Field *field;
	EDVID3Frame *frame;

	edv_prop_page_set_busy(ctx, TRUE);
	p->freeze_count++;

	/* Get the selected EDVID3Frame */
	glist = clist->selection_end;
	if(glist == NULL)
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	row = (gint)glist->data;

	frame = EDV_ID3_FRAME(gtk_clist_get_row_data(
		clist,
		row
	));
	if(frame == NULL)
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	/* Put the Fields GtkCList into context */
	w = p->fields_list;
	clist = GTK_CLIST(w);

	/* Get the selected EDVID3Field */
	glist = clist->selection_end;
	if(glist == NULL)
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	row = (gint)glist->data;

	field = EDV_ID3_FIELD(g_list_nth_data(
		frame->fields_list,
		(guint)row
	));
	if(field == NULL)
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	/* If the EDID3Field's data is too long then query the user to
	 * confirm editing of this EDID3Field
	 */
	if(field->type == EDV_ID3_FIELD_TYPE_DATA)
	{
		if(field->data_length > 1024)
		{
			gint response;
			gchar *msg = g_strdup_printf(
"This field's data is %i bytes long,\n\
are you sure you want to edit this field?\n\
\n\
Note that you can also insert a file's data\n\
into this field by clicking on the insert\n\
file button.",
				field->data_length
			);
			edv_play_sound_question(core);
			CDialogSetTransientFor(toplevel);
			response = CDialogGetResponse(
				"Confirm Edit",
				msg,
				NULL,
				CDIALOG_ICON_QUESTION,
				CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
				CDIALOG_BTNFLAG_YES
			);
			g_free(msg);
			CDialogSetTransientFor(NULL);
			if(response != CDIALOG_RESPONSE_YES)
			{
				p->freeze_count--;
				edv_prop_page_set_busy(ctx, FALSE);
				return;
			}
		}
	}

	/* Create the EDVID3FieldTypes list, each item's index
	 * corresponds to the EDVID3FieldTypes enumeration
	 */
	types_list = NULL;
	types_list = g_list_append(
		types_list,
		g_strdup("UNKNOWN")
	);
	types_list = g_list_append(
		types_list,
		g_strdup("INT8")
	);
	types_list = g_list_append(
		types_list,
		g_strdup("INT16")
	);
	types_list = g_list_append(
		types_list,
		g_strdup("INT32")
	);
	types_list = g_list_append(
		types_list,
		g_strdup("INT64/INT32PLUS")
	);
	types_list = g_list_append(
		types_list,
		g_strdup("STRING")
	);
	types_list = g_list_append(
		types_list,
		g_strdup("BINARY")
	);
	types_list = g_list_append(
		types_list,
		g_strdup("LANGUAGE")
	);
	types_list = g_list_append(
		types_list,
		g_strdup("MIME_TYPE")
	);

	/* Set up the Prompt Dialog and query the user for new values */
	PDialogDeleteAllPrompts();
	PDialogSetSize(320, -1);
	PDialogAddPromptPopupList(
		NULL,
		"Type:",
		types_list,
		(gint)field->type,
		MIN(g_list_length(types_list), 10)
	);
	s = NULL;
	switch(field->type)
	{
	    case EDV_ID3_FIELD_TYPE_UNKNOWN:
		break;
	    case EDV_ID3_FIELD_TYPE_INT8:
	    case EDV_ID3_FIELD_TYPE_INT16:
	    case EDV_ID3_FIELD_TYPE_INT32:
		s = g_strdup_printf(
			"%i",
			edv_id3_field_get_number(field)
		);
		break;
	    case EDV_ID3_FIELD_TYPE_INT64:
		s = g_strdup_printf(
			"%ld",
			edv_id3_field_get_long(field)
		);
		break;
	    case EDV_ID3_FIELD_TYPE_STRING:
	    case EDV_ID3_FIELD_TYPE_LANGUAGE:
	    case EDV_ID3_FIELD_TYPE_MIME_TYPE:
		s = STRDUP(edv_id3_field_get_string(field));
		break;
	    case EDV_ID3_FIELD_TYPE_DATA:
		if((field->data != NULL) && (field->data_length > 0))
		{
			const gint n = field->data_length;
			gint i;
			gchar *sn;
			const guint8 *data = (guint8 *)field->data;

			s = g_strdup("");
			for(i = 0; i < n; i++)
			{
				sn = g_strdup_printf(
					"%.2X",
					data[i]
				);
				if(sn != NULL)
				{
					gchar *s_tmp = g_strconcat(
						s,
						(i > 0) ? " " : "",
						sn,
						NULL
					);
					if(s_tmp != NULL)
					{
						g_free(s);
						s = s_tmp;
					}
					g_free(sn);
				}
			}
		}
		break;
	}
	PDialogAddPrompt(
		NULL,
		"Value:",
		(s != NULL) ? s : ""
	);
	g_free(s);
	PDialogSetTransientFor(toplevel);
	strv = PDialogGetResponse(
		"Edit Field",
		NULL,
		NULL,
		PDIALOG_ICON_EDIT,
		"Set", "Cancel",
		PDIALOG_BTNFLAG_SUBMIT | PDIALOG_BTNFLAG_CANCEL,
		PDIALOG_BTNFLAG_SUBMIT,
		&strc
	);
	if((strv != NULL) && (strc >= 2))
	{
		gint i;
		GList *glist;
		const gchar *s = strv[1];

		/* Determine and set the EDVID3FieldType */
		for(i = 0,
		    glist = types_list;
		    glist != NULL;
		    i++,
		    glist = g_list_next(glist)
		)
		{
			if(!g_strcasecmp(
				(const gchar *)glist->data,
				strv[0]
			))
			{
				field->type = (EDVID3FieldType)i;
				break;
			}
		}

		/* Set the new values on the EDVID3Frame */
		switch(field->type)
		{
		    case EDV_ID3_FIELD_TYPE_UNKNOWN:
			g_free(field->data);
			field->data = NULL;
			field->data_length = 0;
			break;
		    case EDV_ID3_FIELD_TYPE_INT8:
			{
				guint8 v8 = ATOI(s);
				g_free(field->data);
				field->data_length = sizeof(v8);
				field->data = g_memdup(
					&v8,
					field->data_length
				);
			}
			break;
		    case EDV_ID3_FIELD_TYPE_INT16:
			{
				guint16 v16 = ATOI(s);
				g_free(field->data);
				field->data_length = sizeof(v16);
				field->data = g_memdup(
					&v16,
					field->data_length
				);
			}
			break;
		    case EDV_ID3_FIELD_TYPE_INT32:
			{
				guint32 v32 = ATOI(s);
				g_free(field->data);
				field->data_length = sizeof(v32);
				field->data = g_memdup(
					&v32,
					field->data_length
				);
			}
			break;
		    case EDV_ID3_FIELD_TYPE_INT64:
			{
				guint64 v64 = (guint64)ATOL(s);
				g_free(field->data);
				field->data_length = sizeof(v64);
				field->data = g_memdup(
					&v64,
					field->data_length
				);
			}
			break;
		    case EDV_ID3_FIELD_TYPE_STRING:
			g_free(field->data);
			field->data = STRDUP(s);
			field->data_length = (field->data != NULL) ?
				(gint)strlen((char *)field->data) + 1 : 0;
			break;
		    case EDV_ID3_FIELD_TYPE_DATA:
			/* Parse the binary data */
			if(s != NULL)
			{
				guint8 v8 = 0;
				int	i,
					c,
					vp = 0;
				const gchar *s_ptr = s;

				g_free(field->data);
				field->data = NULL;
				field->data_length = 0;
				while(*s_ptr != '\0')
				{
					c = *s_ptr;
					if((c == ' ') || (c == '\t') ||
					   (c == '\n') || (c == '\r') 
					)
					{
						s_ptr++;
						continue;
					}

					if(vp == 0)
					{
						if(isdigit(c))
							v8 += 16 * (c - '0');
						else
							v8 += 16 * MIN(
								(toupper((int)c) - 'A' + 10),
								15
							);
						vp++;
					}
					else	/* vp == 1 */
					{
						if(isdigit(c))
							v8 += c - '0';
						else
							v8 += MIN(
								(toupper((int)c) - 'A' + 10),
								15
							);
						i = field->data_length;
						field->data_length++;
						field->data = g_realloc(
							field->data,
							field->data_length * sizeof(guint8)
						);
						if(field->data == NULL)
						{
							field->data_length = 0;
							break;
						}
						((guint8 *)field->data)[i] = v8;
						v8 = 0x00;
						vp = 0;
					}

					s_ptr++;
				}
			}
			break;
		    case EDV_ID3_FIELD_TYPE_LANGUAGE:
			g_free(field->data);
			field->data = STRDUP(s);
			field->data_length = (field->data != NULL) ?
				(gint)strlen((char *)field->data) + 1 : 0;
			break;
		    case EDV_ID3_FIELD_TYPE_MIME_TYPE:
			g_free(field->data);
			field->data = STRDUP(s);
			field->data_length = (field->data != NULL) ?
				(gint)strlen((char *)field->data) + 1 : 0;
			break;
		}

		/* Update the displayed values for this EDVID3Frame,
		 * this may also update/reformat the EDVID3Field's
		 * value we just set
		 */
		edv_id3_prop_page_update_widgets_frame(
			p,
			frame->id,
			frame
		);

		/* Update the displayed values for this EDVID3Field */
		edv_id3_prop_page_fields_list_update(
			p,
			row,
			field
		);

		/* Mark that we have made changes */
		if(!(p->flags & EDV_ID3_PROP_PAGE_HAS_CHANGES))
		{
			p->flags |= EDV_ID3_PROP_PAGE_HAS_CHANGES;
			edv_prop_page_set_has_changes(
				ctx,
				TRUE
			);
		}

		edv_id3_prop_page_update_widgets(p);
	}

	/* Delete the EDVID3FieldTypes list */
	g_list_foreach(types_list, (GFunc)g_free, NULL);
	g_list_free(types_list);

	p->freeze_count--;
	edv_prop_page_set_busy(ctx, FALSE);
}

/*
 *	Queries the user to remove a field.
 */
static void edv_id3_prop_page_remove_field_query(EDVID3PropPage *p)
{
	gint		response,
			row;
	gchar *msg;
	GList *glist;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	GtkWidget *w = p->frames_list;
	GtkCList *clist = GTK_CLIST(w);
	EDVCore *core = edv_prop_page_get_core(ctx);
	EDVID3Frame *frame;
	EDVID3Field *field;

	edv_prop_page_set_busy(ctx, TRUE);
	p->freeze_count++;

	/* Get the selected EDVID3Frame */
	glist = clist->selection_end;
	if(glist == NULL)
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	row = (gint)glist->data;

	frame = EDV_ID3_FRAME(gtk_clist_get_row_data(
		clist,
		row
	));
	if(frame == NULL)
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	/* Put the Fields GtkCList into context */
	w = p->fields_list;
	clist = GTK_CLIST(w);

	/* Get the selected EDVID3Field */
	glist = clist->selection_end;
	if(glist == NULL)
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	row = (gint)glist->data;

	field = EDV_ID3_FIELD(g_list_nth_data(
		frame->fields_list,
		(guint)row
	));
	if(field == NULL)
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	/* Confirm remove */
	msg = g_strdup_printf(
"Remove field %i?",
		row + 1
	);
	edv_play_sound_question(core);
	CDialogSetTransientFor(toplevel);
	response = CDialogGetResponse(
		"Confirm Remove",
		msg,
		NULL,
		CDIALOG_ICON_QUESTION,
		CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
		CDIALOG_BTNFLAG_NO
	);
	g_free(msg);
	CDialogSetTransientFor(NULL);

	if(response != CDIALOG_RESPONSE_YES)
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	/* Remove the EDVID3Field from the EDVID3Frame and the
	 * Fields GtkCList and then delete the EDVID3Field
	 */
	frame->fields_list = g_list_remove(
		frame->fields_list,
		field
	);

	gtk_clist_freeze(clist);
	gtk_clist_remove(
		clist,
		row
	);
	gtk_clist_thaw(clist);

	edv_id3_field_delete(field);

	/* Mark that we have made changes */
	if(!(p->flags & EDV_ID3_PROP_PAGE_HAS_CHANGES))
	{
		p->flags |= EDV_ID3_PROP_PAGE_HAS_CHANGES;
		edv_prop_page_set_has_changes(
			ctx,
			TRUE
		);
	}

	edv_id3_prop_page_update_widgets(p);

	p->freeze_count--;
	edv_prop_page_set_busy(ctx, FALSE);
}


/*
 *	Clears the currently displayed values, deletes the current
 *	EDVID3Tags list, opens the EDVID3Tags list from the file
 *	specified in the current properties list, updates the
 *	information displayed on the GtkWidgets and sets the
 *	EDV_ID3_PROP_PAGE_GOT_VALUES flag.
 */
static void edv_id3_prop_page_get_values(EDVID3PropPage *p)
{
	gulong index;
	gchar *path;
	GtkWidget *w;
	EDVPropPageContext *ctx = p->ctx;
	const EDVLocationType location_type = edv_prop_page_get_location_type(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);
	CfgList *cfg_list = edv_prop_page_get_cfg_list(ctx);
	EDVID3Tag *tag;

	edv_prop_page_set_busy(ctx, TRUE);
	p->freeze_count++;

	/* Reset/clear the values displayed on the GtkWidgets
	 *
	 * Comments GtkText
	 */
/*	gtk_widget_hide(p->comments_toplevel); */
	w = p->comments_text;
	if(w != NULL)
	{
		GtkEditable *editable = GTK_EDITABLE(w);
		GtkText *text = GTK_TEXT(w);
		gtk_events_process();
		gtk_text_freeze(text);
		gtk_text_set_point(
			text,
			0
		);
		gtk_editable_delete_text(
			editable,
			0, -1
		);
		gtk_text_thaw(text);
	}

	/* Frames GtkCList */
	w = p->frames_list;
	if(w != NULL)
	{
		GtkCList *clist = GTK_CLIST(w);
		gtk_clist_freeze(clist);
		gtk_clist_clear(clist);
		gtk_clist_thaw(clist);
	}

	/* Fields GtkCList */
	w = p->fields_list;
	if(w != NULL)
	{
		GtkCList *clist = GTK_CLIST(w);
		gtk_clist_freeze(clist);
		gtk_clist_clear(clist);
		gtk_clist_thaw(clist);
	}

	/* Mark that no image is loaded */
	p->flags &= ~EDV_ID3_PROP_PAGE_HAVE_IMAGE;

	/* Delete the current EDVID3Tag */
	edv_id3_tag_delete(p->tag);
	p->tag = NULL;

	/* Do not delete the last_user_selected_path set by the user */


	/* Get the path to this object based on the location type */
	path = NULL;
	switch(location_type)
	{
	    case EDV_LOCATION_TYPE_VFS:
		path = STRDUP(edv_properties_list_get_s(
			properties_list,
			EDV_PROP_NAME_PATH
		));
		break;

	    case EDV_LOCATION_TYPE_RECYCLE_BIN:
		index = edv_properties_list_get_ul(
			properties_list,
			EDV_PROP_NAME_INDEX
		);
		path = edv_recycle_bin_index_get_recycled_object_path(
			EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX),
			index
		);
		break;

	    case EDV_LOCATION_TYPE_ARCHIVE:
		break;
	}

	/* Get the ID3 tag from the file */
	p->tag = tag = edv_id3_tag_open_file(
		path,
		0
	);
	if(tag != NULL)
	{
		const EDVID3TagFlags flags = tag->flags;

		/* Set the values displayed on the GtkWidgets for the
		 * EDVID3Tag
		 *
		 * Unsyncronization
		 */
		gtk_toggle_button_set_active(
			GTK_TOGGLE_BUTTON(p->unsyncronization_check),
			(flags & EDV_ID3_TAG_FLAG_UNSYNCHRONISATION) ? TRUE : FALSE
		);

		/* Extended Header */
		gtk_toggle_button_set_active(
			GTK_TOGGLE_BUTTON(p->extended_header_check),
			(flags & EDV_ID3_TAG_FLAG_EXTENDED_HEADER) ? TRUE : FALSE
		);

		/* Experimental */
		gtk_toggle_button_set_active(
			GTK_TOGGLE_BUTTON(p->experimental_check),
			(flags & EDV_ID3_TAG_FLAG_EXPERIMENTAL) ? TRUE : FALSE
		);

		/* Footer Present */
		gtk_toggle_button_set_active(
			GTK_TOGGLE_BUTTON(p->footer_present_check),
			(flags & EDV_ID3_TAG_FLAG_FOOTER_PRESENT) ? TRUE : FALSE
		);

		/* Extended Flags */
		if(flags & EDV_ID3_TAG_FLAG_EXTENDED_HEADER)
		{
			const EDVID3TagExtendedFlags extended_flags = tag->extended_flags;

			/* CRC 16 */
			gtk_toggle_button_set_active(
				GTK_TOGGLE_BUTTON(p->crc_16_check),
				(extended_flags & EDV_ID3_TAG_EXTENDED_FLAG_CRC_DATA_PRESENT) ? TRUE : FALSE
			);

			/* Restrictions */
			gtk_toggle_button_set_active(
				GTK_TOGGLE_BUTTON(p->restrictions_check),
				(extended_flags & EDV_ID3_TAG_EXTENDED_FLAG_RESTRICTIONS) ? TRUE : FALSE
			);

			/* Restriction Flags */
			if(extended_flags & EDV_ID3_TAG_EXTENDED_FLAG_RESTRICTIONS)
			{
				const EDVID3TagRestrictionFlags restriction_flags = tag->restriction_flags;

				/* Tag Size */
/* TODO */

				/* Text Encoding */
				if(restriction_flags & EDV_ID3_TAG_RESTRICTION_TEXT_ENCODING_LATIN1_UTF8)
					w = p->restriction_text_encoding_latin1_utf8_radio;
				else
					w = p->restriction_text_encoding_none_radio;
				gtk_toggle_button_set_active(
					GTK_TOGGLE_BUTTON(w),
					TRUE
				);

				/* Text Size */
				if(restriction_flags & EDV_ID3_TAG_RESTRICTION_TEXT_SIZE_1024_CHARS)
					w = p->restriction_text_size_1024_radio;
				else if(restriction_flags & EDV_ID3_TAG_RESTRICTION_TEXT_SIZE_128_CHARS)
					w = p->restriction_text_size_128_radio;
				else if(restriction_flags & EDV_ID3_TAG_RESTRICTION_TEXT_SIZE_30_CHARS)
					w = p->restriction_text_size_30_radio;
				else
					w = p->restriction_text_size_none_radio;
				gtk_toggle_button_set_active(
					GTK_TOGGLE_BUTTON(w),
					TRUE
				);

				/* Image Encoding */
				if(restriction_flags & EDV_ID3_TAG_RESTRICTION_IMAGE_ENCODING_PNG_JPEG)
					w = p->restriction_image_encoding_jpeg_png_radio;
				else
					w = p->restriction_image_encoding_none_radio;
				gtk_toggle_button_set_active(
					GTK_TOGGLE_BUTTON(w),
					TRUE
				);

				/* Image Size */
				if(restriction_flags & EDV_ID3_TAG_RESTRICTION_IMAGE_SIZE_256_256)
					w = p->restriction_image_size_256x256_radio;
				else if(restriction_flags & EDV_ID3_TAG_RESTRICTION_IMAGE_SIZE_64_64)
					w = p->restriction_image_size_64x64_radio;
				else if(restriction_flags & EDV_ID3_TAG_RESTRICTION_IMAGE_SIZE_64_64_EXACT)
					w = p->restriction_image_size_64x64_exact_radio;
				else
					w = p->restriction_image_size_none_radio;
				gtk_toggle_button_set_active(
					GTK_TOGGLE_BUTTON(w),
					TRUE
				);
			}
		}

		/* Set the values displayed on the GtkWidgets for
		 * all the EDVID3Frames
		 */
		if(tag->frames_list != NULL)
		{
			GList *glist;
			EDVID3Frame *frame;
			for(glist = tag->frames_list;
			    glist != NULL;
			    glist = g_list_next(glist)
			)
			{
				frame = EDV_ID3_FRAME(glist->data);
				if(frame == NULL)
					continue;

				edv_id3_prop_page_update_widgets_frame(
					p,
					frame->id,
					frame
				);
			}
		}

		/* Queue the thumb GtkDrawingArea to redraw after
		 * all information has been updated and events
		 * processed
		 */
		gtk_widget_queue_draw(p->thumb_da);
	}

	/* Mark that we have obtained the image information */
	p->flags |= EDV_ID3_PROP_PAGE_GOT_VALUES;

	g_free(path);

	edv_id3_prop_page_update_widgets(p);

	p->freeze_count--;
	edv_prop_page_set_busy(ctx, FALSE);
}


/*
 *	Sets the string value to the EDVID3Frame and .
 */
static gint edv_id3_prop_page_set_frame_string(
	EDVID3PropPage *p,
	EDVID3Frame *frame,
	const gchar *s
)
{
	gint status;
	const gchar *language = edv_id3_prop_page_get_language(p);
	EDVID3Field *field;

	if(frame == NULL)
		return(-2);

	/* Add/update the language field */
	field = edv_id3_frame_get_field_by_type(
		frame,
		EDV_ID3_FIELD_TYPE_LANGUAGE
	);
	if(field == NULL)
		field = edv_id3_frame_append_field(frame);
	if(field != NULL)
	{
		field->type = EDV_ID3_FIELD_TYPE_LANGUAGE;
		g_free(field->data);
		field->data = STRDUP(language);
		field->data_length = (field->data != NULL) ?
			((gint)strlen((char *)field->data) + 1) : 0;
	}

	/* Add/update the string field */
	field = edv_id3_frame_get_field_by_type(
		frame,
		EDV_ID3_FIELD_TYPE_STRING
	);
	if(field == NULL)
		field = edv_id3_frame_append_field(frame);
	if(field != NULL)
	{
		field->type = EDV_ID3_FIELD_TYPE_STRING;
		g_free(field->data);
		field->data = STRDUP(s);
		field->data_length = (field->data != NULL) ?
			((gint)strlen((char *)field->data) + 1) : 0;
		status = 0;
	}
	else
		status = -3;

	return(status);
}

/*
 *	Sets the string value to the EDVID3Frame by ID.
 */
static gint edv_id3_prop_page_set_frame_string_by_id(
	EDVID3PropPage *p,
	const gchar *id,
	const gchar *s,
	const gboolean allow_create_frame
)
{
	EDVID3Frame *frame = edv_id3_tag_get_frame_by_id(
		p->tag,
		id
	);
	if(frame == NULL)
	{
		if(!allow_create_frame)
			return(-1);

		/* No EDVID3Frame found in the list, create a new
		 * new EDVID3Frame and a new EDVID3Tag as needed
		 */
		frame = edv_id3_prop_page_new_frame(
			p,
			-1,			/* Append */
			id,
			EDV_ID3_TEXT_ENCODING_ISO_8859_1,
			NULL,			/* No fields list */
			NULL			/* No description */
		);
	}
	if(frame == NULL)
		return(-1);

	return(edv_id3_prop_page_set_frame_string(
		p,
		frame,
		s
	));
}

/*
 *	Saves the tags list to the file.
 */
static gint edv_id3_prop_page_save_tag_to_path(
	EDVID3PropPage *p,
	const gchar *path
)
{
	gint status = 0;
	EDVID3Tag *tag;

	p->freeze_count++;

	if(STRISEMPTY(path))
	{
		p->freeze_count--;
		return(-2);
	}

	/* Set/update the tag's values before saving */
	tag = p->tag;
	if(tag != NULL)
	{
		/* The tags that we edited must be saved as
		 * ID3v2.4.0 or higher because of the types of
		 * frame IDs that we introduce
		 */
		if(tag->version_major < 4)
			tag->version_major = 4;

		/* Unsyncronization */
		if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
			p->unsyncronization_check
		)))
			tag->flags |= EDV_ID3_TAG_FLAG_UNSYNCHRONISATION;
		else
			tag->flags &= ~EDV_ID3_TAG_FLAG_UNSYNCHRONISATION;

		/* Extended Header */
		if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
			p->extended_header_check
		)))
			tag->flags |= EDV_ID3_TAG_FLAG_EXTENDED_HEADER;
		else
			tag->flags &= ~EDV_ID3_TAG_FLAG_EXTENDED_HEADER;

		/* Experimental */
		if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
			p->experimental_check
		)))
			tag->flags |= EDV_ID3_TAG_FLAG_EXPERIMENTAL;
		else
			tag->flags &= ~EDV_ID3_TAG_FLAG_EXPERIMENTAL;

		/* Footer Present */
		if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
			p->footer_present_check
		)))
			tag->flags |= EDV_ID3_TAG_FLAG_FOOTER_PRESENT;
		else
			tag->flags &= ~EDV_ID3_TAG_FLAG_FOOTER_PRESENT;

		/* Set values on the EDVID3Tag dependent on the setting
		 * of the EDV_ID3_TAG_FLAG_EXTENDED_HEADER flag
		 */
		if(tag->flags & EDV_ID3_TAG_FLAG_EXTENDED_HEADER)
		{
			/* This is not an update */
			tag->extended_flags &= ~EDV_ID3_TAG_EXTENDED_FLAG_IS_UPDATE;

			/* CRC 16 */
			if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
				p->crc_16_check
			)))
				tag->extended_flags |= EDV_ID3_TAG_EXTENDED_FLAG_CRC_DATA_PRESENT;
			else
				tag->extended_flags &= ~EDV_ID3_TAG_EXTENDED_FLAG_CRC_DATA_PRESENT;

			/* Restrictions */
			if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
				p->restrictions_check
			)))
				tag->extended_flags |= EDV_ID3_TAG_EXTENDED_FLAG_RESTRICTIONS;
			else
				tag->extended_flags &= ~EDV_ID3_TAG_EXTENDED_FLAG_RESTRICTIONS;

			/* Set values on the EDVID3Tag dependent on the
			 * setting of the
			 * EDV_ID3_TAG_EXTENDED_FLAG_RESTICTIONS flag
			 */
			if(tag->extended_flags & EDV_ID3_TAG_EXTENDED_FLAG_RESTRICTIONS)
			{
				/* Tag Size Restrictions */
				tag->restriction_flags &= ~EDV_ID3_TAG_RESTRICTION_TAG_SIZE_MASK;
/* TODO */

				/* Text Encoding Restrictions */
				tag->restriction_flags &= ~EDV_ID3_TAG_RESTRICTION_TEXT_ENCODING_MASK;
				if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
					p->restriction_text_encoding_latin1_utf8_radio
				)))
				{
					tag->extended_flags |= EDV_ID3_TAG_EXTENDED_FLAG_RESTRICTIONS;
					tag->restriction_flags |= EDV_ID3_TAG_RESTRICTION_TEXT_ENCODING_LATIN1_UTF8;
				}

				/* Text Size Restrictions */
				tag->restriction_flags &= ~EDV_ID3_TAG_RESTRICTION_TEXT_SIZE_MASK;
				if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
					p->restriction_text_size_1024_radio
				)))
				{
					tag->extended_flags |= EDV_ID3_TAG_EXTENDED_FLAG_RESTRICTIONS;
					tag->restriction_flags |= EDV_ID3_TAG_RESTRICTION_TEXT_SIZE_1024_CHARS;
				}
				else if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
					p->restriction_text_size_128_radio
				)))
				{
					tag->extended_flags |= EDV_ID3_TAG_EXTENDED_FLAG_RESTRICTIONS;
					tag->restriction_flags |= EDV_ID3_TAG_RESTRICTION_TEXT_SIZE_128_CHARS;
				}
				else if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
					p->restriction_text_size_30_radio
				)))
				{
					tag->extended_flags |= EDV_ID3_TAG_EXTENDED_FLAG_RESTRICTIONS;
					tag->restriction_flags |= EDV_ID3_TAG_RESTRICTION_TEXT_SIZE_30_CHARS;
				}

				/* Image Encoding Restrictions */
				tag->restriction_flags &= ~EDV_ID3_TAG_RESTRICTION_IMAGE_ENCODING_MASK;
				if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
					p->restriction_image_encoding_jpeg_png_radio
				)))
				{
					tag->extended_flags |= EDV_ID3_TAG_EXTENDED_FLAG_RESTRICTIONS;
					tag->restriction_flags |= EDV_ID3_TAG_RESTRICTION_IMAGE_ENCODING_PNG_JPEG;
				}

				/* Image Size Restrictions */
				tag->restriction_flags &= ~EDV_ID3_TAG_RESTRICTION_IMAGE_SIZE_MASK;
				if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
					p->restriction_image_size_256x256_radio
				)))
				{
					tag->extended_flags |= EDV_ID3_TAG_EXTENDED_FLAG_RESTRICTIONS;
					tag->restriction_flags |= EDV_ID3_TAG_RESTRICTION_IMAGE_SIZE_256_256;
				}
				else if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
					p->restriction_image_size_64x64_radio
				)))
				{
					tag->extended_flags |= EDV_ID3_TAG_EXTENDED_FLAG_RESTRICTIONS;
					tag->restriction_flags |= EDV_ID3_TAG_RESTRICTION_IMAGE_SIZE_64_64;
				}
				else if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
					p->restriction_image_size_64x64_exact_radio
				)))
				{
					tag->extended_flags |= EDV_ID3_TAG_EXTENDED_FLAG_RESTRICTIONS;
					tag->restriction_flags |= EDV_ID3_TAG_RESTRICTION_IMAGE_SIZE_64_64_EXACT;
				}
			}
		}

		/* Save the tags to the file */
		status = edv_id3_tag_save_file(
			p->tag,
			path,
			(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(p->optimize_check)) ? EDV_ID3_TAG_IO_TRUNCATE_EXISTING_TAGS : 0) |
			(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(p->compress_check)) ? EDV_ID3_TAG_IO_COMPRESS_GZIP : 0) |
			(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(p->append_check)) ? EDV_ID3_TAG_IO_APPEND : 0)
		);
	}

	p->freeze_count--;

	return(status);
}


/*
 *	Updates the GtkWidget states.
 */
static void edv_id3_prop_page_update_widgets(EDVID3PropPage *p)
{
	gboolean b;
	gchar *s;
	GtkWidget *w;
	EDVID3Tag *tag;
	EDVPropPageContext *ctx = p->ctx;
	const EDVLocationType location_type = edv_prop_page_get_location_type(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);
	const gchar *path = edv_properties_list_get_s(
		properties_list,
		EDV_PROP_NAME_PATH
	);
	EDVMIMEType *m = edv_mime_types_list_match_path(
		core->mime_types_list,
		path
	);

	p->freeze_count++;

	/* Update the Open GtkButton's label and sensivitity */
	w = p->open_btn;
	if(m != NULL)
	{
		EDVMIMETypeCommand *command = EDV_MIME_TYPE_COMMAND(g_list_nth_data(
			m->commands_list,
			0
		));
		if(command != NULL)
		{
			GUIButtonPixmapUpdate(
				w,
				NULL,
				command->name
			);
		}
		switch(location_type)
		{
		    case EDV_LOCATION_TYPE_VFS:
			gtk_widget_set_sensitive(w, TRUE);
			break;
		    case EDV_LOCATION_TYPE_RECYCLE_BIN:
		    case EDV_LOCATION_TYPE_ARCHIVE:
			gtk_widget_set_sensitive(w, FALSE);
			break;
		}
	}
	else
	{
		GUIButtonPixmapUpdate(
			w,
			NULL,
			"Open"
		);
		switch(location_type)
		{
		    case EDV_LOCATION_TYPE_VFS:
		    case EDV_LOCATION_TYPE_RECYCLE_BIN:
		    case EDV_LOCATION_TYPE_ARCHIVE:
			gtk_widget_set_sensitive(w, FALSE);
			break;
		}
	}

	/* Tag Version */
	tag = p->tag;
	if(tag != NULL)
		s = g_strdup_printf(
			"%i.%i.%i",
			2,
			tag->version_major,
			tag->version_minor
		);
	else
		s = g_strdup("");
	gtk_label_set_text(
		GTK_LABEL(p->tag_version_label),
		s
	);
	g_free(s);

	/* Extended Flags */
	b = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(p->extended_header_check));
	gtk_widget_set_sensitive(
		p->extended_flags_toplevel,
		b
	);

	/* Restriction Flags */
	if(b)
		b = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(p->restrictions_check));
	gtk_widget_set_sensitive(
		p->restriction_text_toplevel,
		b
	);
	gtk_widget_set_sensitive(
		p->restriction_image_toplevel,
		b
	);


	/* Thumb GtkMenu */
	gtk_widget_set_sensitive(
		p->set_image_mi,
		TRUE
	);
	b = (p->flags & EDV_ID3_PROP_PAGE_HAVE_IMAGE) ? TRUE : FALSE;
	gtk_widget_set_sensitive(
		p->remove_image_mi,
		b
	);
	gtk_widget_set_sensitive(
		p->extract_image_mi,
		b
	);

	/* Frames GtkCList */
	w = p->frames_list;
	b = (GTK_CLIST(w)->selection != NULL) ? TRUE : FALSE;
	gtk_widget_set_sensitive(
		p->frame_add_btn,
		TRUE
	);
	gtk_widget_set_sensitive(
		p->frame_add_mi,
		TRUE
	);
	gtk_widget_set_sensitive(
		p->frame_edit_btn,
		b
	);
	gtk_widget_set_sensitive(
		p->frame_edit_mi,
		b
	);
	gtk_widget_set_sensitive(
		p->frame_remove_btn,
		b
	);
	gtk_widget_set_sensitive(
		p->frame_remove_mi,
		b
	);

	/* Fields GtkCList */
	gtk_widget_set_sensitive(
		p->field_add_btn,
		b
	);
	gtk_widget_set_sensitive(
		p->field_add_mi,
		b
	);
	gtk_widget_set_sensitive(
		p->field_insert_file_btn,
		b
	);
	gtk_widget_set_sensitive(
		p->field_insert_file_mi,
		b
	);

	w = p->fields_list;
	b = (GTK_CLIST(w)->selection != NULL) ? TRUE : FALSE;
	gtk_widget_set_sensitive(
		p->field_edit_btn,
		b
	);
	gtk_widget_set_sensitive(
		p->field_edit_mi,
		b
	);
	gtk_widget_set_sensitive(
		p->field_remove_btn,
		b
	);
	gtk_widget_set_sensitive(
		p->field_remove_mi,
		b
	);


	p->freeze_count--;
}

/*
 *	Updates the displayed values on the GtkWidgets and the Frames
 *	GtkCList that are displaying values for a EDVID3Frame.
 *
 *	The id specifies the EDVID3Frame's ID.
 *
 *	The frame specifies the EDVID3Frame. If frame is NULL then the
 *	GtkWidgets that are displaying the values for the EDVID3Frame
 *	will have their values cleared.
 */
static void edv_id3_prop_page_update_widgets_frame(
	EDVID3PropPage *p,
	const gchar *id,
	EDVID3Frame *frame
)
{
	p->freeze_count++;

	if(id == NULL)
	{
		p->freeze_count--;
		return;
	}

	/* Title */
	if(!g_strcasecmp(id, EDV_ID3_FRAME_ID_TITLE))
	{
		GtkWidget *w = p->title_entry;
		if(frame != NULL)
		{
			EDVID3Field *field = edv_id3_frame_get_field_by_type(
				frame,
				EDV_ID3_FIELD_TYPE_STRING
			);
			gchar *s = STRDUP(edv_id3_field_get_string(field));
			s = edv_id3_prop_page_single_line_string(s);
			gtk_entry_set_text(
				GTK_ENTRY(w),
				s
			);
			g_free(s);
			if(!STRISEMPTY(frame->description))
				GUISetWidgetTip(
					w,
					frame->description
				);
		}
		else
		{
			gtk_entry_set_text(
				GTK_ENTRY(w),
				""
			);
			GUISetWidgetTip(
				w,
				NULL
			);
		}
	}
	/* Author */
	else if(!g_strcasecmp(id, EDV_ID3_FRAME_ID_ARTIST))
	{
		GtkWidget *w = p->author_entry;
		if(frame != NULL)
		{
			EDVID3Field *field = edv_id3_frame_get_field_by_type(
				frame,
				EDV_ID3_FIELD_TYPE_STRING
			);
			gchar *s = STRDUP(edv_id3_field_get_string(field));
			s = edv_id3_prop_page_single_line_string(s);
			gtk_entry_set_text(
				GTK_ENTRY(w),
				s
			);
			g_free(s);
			if(!STRISEMPTY(frame->description))
				GUISetWidgetTip(
					w,
					frame->description
				);
		}
		else
		{
			gtk_entry_set_text(
				GTK_ENTRY(w),
				""
			);
			GUISetWidgetTip(
				w,
				NULL
			);
		}
	}
	/* Album */
	else if(!g_strcasecmp(id, EDV_ID3_FRAME_ID_ALBUM))
	{
		GtkWidget *w = p->album_entry;
		if(frame != NULL)
		{
			EDVID3Field *field = edv_id3_frame_get_field_by_type(
				frame,
				EDV_ID3_FIELD_TYPE_STRING
			);
			gchar *s = STRDUP(edv_id3_field_get_string(field));
			s = edv_id3_prop_page_single_line_string(s);
			gtk_entry_set_text(
				GTK_ENTRY(w),
				s
			);
			g_free(s);
			if(!STRISEMPTY(frame->description))
				GUISetWidgetTip(
					w,
					frame->description
				);
		}
		else
		{
			gtk_entry_set_text(
				GTK_ENTRY(w),
				""
			);
			GUISetWidgetTip(
				w,
				NULL
			);
		}
	}
	/* Year */
	else if(!g_strcasecmp(id, EDV_ID3_FRAME_ID_YEAR))
	{
		GtkWidget *w = p->year_entry;
		if(frame != NULL)
		{
			EDVID3Field *field = edv_id3_frame_get_field_by_type(
				frame,
				EDV_ID3_FIELD_TYPE_STRING
			);
			if(EDV_ID3_FIELD_IS_NUMBER(field))
			{
				EDVPropPageContext *ctx = p->ctx;
				CfgList *cfg_list = edv_prop_page_get_cfg_list(ctx);
				const gulong t = (gulong)edv_id3_field_get_long(field);
				gchar *s = edv_date_string_format(
					t,
					EDV_GET_S(EDV_CFG_PARM_DATE_FORMAT),
					(EDVDateRelativity)EDV_GET_I(EDV_CFG_PARM_DATE_RELATIVITY)
				);
				s = edv_id3_prop_page_single_line_string(s);
				gtk_entry_set_text(
					GTK_ENTRY(w),
					s
				);
				g_free(s);
			}
			else if(EDV_ID3_FIELD_IS_STRING(field))
			{
				gchar *s = STRDUP(edv_id3_field_get_string(field));
				s = edv_id3_prop_page_single_line_string(s);
				gtk_entry_set_text(
					GTK_ENTRY(w),
					s
				);
				g_free(s);
			}
			if(!STRISEMPTY(frame->description))
				GUISetWidgetTip(
					w,
					frame->description
				);
		}
		else
		{
			gtk_entry_set_text(
				GTK_ENTRY(w),
				""
			);
			GUISetWidgetTip(
				w,
				NULL
			);
		}
	}
	/* Track */
	else if(!g_strcasecmp(id, EDV_ID3_FRAME_ID_TRACK))
	{
		GtkWidget *w = p->track_entry;
		if(frame != NULL)
		{
			EDVID3Field *field = edv_id3_frame_get_field_by_type(
				frame,
				EDV_ID3_FIELD_TYPE_STRING
			);
			if(EDV_ID3_FIELD_IS_NUMBER(field))
			{
				const gint n = edv_id3_field_get_number(field);
				gchar *s = g_strdup_printf("%i", n);
				s = edv_id3_prop_page_single_line_string(s);
				gtk_entry_set_text(
					GTK_ENTRY(w),
					s
				);
				g_free(s);
			}
			else if(EDV_ID3_FIELD_IS_STRING(field))
			{
				gchar *s = STRDUP(edv_id3_field_get_string(field));
				s = edv_id3_prop_page_single_line_string(s);
				gtk_entry_set_text(
					GTK_ENTRY(w),
					s
				);
				g_free(s);
			}
			if(!STRISEMPTY(frame->description))
				GUISetWidgetTip(
					w,
					frame->description
				);
		}
		else
		{
			gtk_entry_set_text(
				GTK_ENTRY(w),
				""
			);
			GUISetWidgetTip(
				w,
				NULL
			);
		}
	}
	/* Length */
	else if(!g_strcasecmp(id, EDV_ID3_FRAME_ID_LENGTH))
	{
		GtkWidget *w = p->length_entry;
		if(frame != NULL)
		{
			EDVID3Field *field = edv_id3_frame_get_field_by_type(
				frame,
				EDV_ID3_FIELD_TYPE_STRING
			);
			if(field != NULL)
			{
				const gchar *sn = edv_id3_field_get_string(field);
				const gulong ms = (gulong)ATOL(sn);
				gchar *s = edv_id3_length_format(ms);
				s = edv_id3_prop_page_single_line_string(s);
				gtk_entry_set_text(
					GTK_ENTRY(w),
					s
				);
				g_free(s);
			}
			if(STRISEMPTY(frame->description))
			{
				GUISetWidgetTip(
					w,
					EDV_ID3_PROP_PAGE_LENGTH_TOOLTIP
				);
			}
			else
			{
				gchar *s = g_strconcat(
					frame->description,
					"\n",
					EDV_ID3_PROP_PAGE_LENGTH_TOOLTIP,
					NULL
				);
				GUISetWidgetTip(
					w,
					s
				);
				g_free(s);
			}
		}
		else
		{
			gtk_entry_set_text(
				GTK_ENTRY(w),
				""
			);
			GUISetWidgetTip(
				w,
				NULL
			);
		}
	}
	/* Genre */
	else if(!g_strcasecmp(id, EDV_ID3_FRAME_ID_GENRE))
	{
		GtkWidget	*w = p->genre_pulistbox,
				*list = PUListBoxGetPUList(w);
		if(frame != NULL)
		{
			EDVID3Field *field = edv_id3_frame_get_field_by_type(
				frame,
				EDV_ID3_FIELD_TYPE_STRING
			);
			if(EDV_ID3_FIELD_IS_NUMBER(field))
			{
				const gint n = PUListGetTotalItems(list);
				gint i;
				EDVID3Genre *genre;
				const EDVID3GenreIndex genre_index = (EDVID3GenreIndex)edv_id3_field_get_number(field);
				for(i = 0; i < n; i++)
				{
					genre = EDV_ID3_GENRE(PUListGetItemData(
						list,
						i
					));
					if(genre == NULL)
						continue;
					if(genre->index == genre_index)
					{
						PUListBoxSelect(w, i);
						break;
					}
				}
			}
			else if(EDV_ID3_FIELD_IS_STRING(field))
			{
				gchar *s = STRDUP(edv_id3_field_get_string(field));
				s = edv_id3_prop_page_single_line_string(s);
				if(s != NULL)
				{
					const gint n = PUListGetTotalItems(list);
					gint i;
					EDVID3GenreIndex genre_index;
					EDVID3Genre *genre;
					if(isdigit(*s))
						genre_index = (EDVID3GenreIndex)atoi((gchar *)s);
					else
						genre_index = edv_id3_genre_name_to_index(s);
					for(i = 0; i < n; i++)
					{
						genre = EDV_ID3_GENRE(PUListGetItemData(
							list,
							i
						));
						if(genre == NULL)
							continue;
						if(genre->index == genre_index)
						{
							PUListBoxSelect(w, i);
							break;
						}
					}
				}
				g_free(s);
			}
			if(!STRISEMPTY(frame->description))
				PUListBoxSetTip(
					w,
					frame->description
				);
		}
		else
		{
			PUListBoxSelect(w, 0);
			PUListBoxSetTip(
				w,
				NULL
			);
		}
	}
	/* Comments */
	else if(!g_strcasecmp(id, EDV_ID3_FRAME_ID_COMMENTS))
	{
		GtkWidget *w = p->comments_text;
		if(frame != NULL)
		{
			EDVID3Field *field = edv_id3_frame_get_field_by_type(
				frame,
				EDV_ID3_FIELD_TYPE_STRING
			);
			const gchar *value = edv_id3_field_get_string(field);
			if(!STRISEMPTY(value))
			{
				GtkText *text = GTK_TEXT(w);
				gtk_events_process();
		                gtk_text_freeze(text);
		                gtk_text_insert(
		                        text,
		                        NULL,
		                        NULL, NULL,
		                        value, -1
		                );
		                gtk_text_thaw(text);
	/*			gtk_widget_show(p->comments_toplevel); */
			}
		}
		else
		{
			GtkEditable *editable = GTK_EDITABLE(w);
			GtkText *text = GTK_TEXT(w);
			gtk_events_process();
	                gtk_text_freeze(text);
			gtk_text_set_point(
				text,
				0
			);
			gtk_editable_delete_text(
				editable,
				0, -1
			);
	                gtk_text_thaw(text);
		}
	}
	/* Image */
	else if(!g_strcasecmp(id, EDV_ID3_FRAME_ID_IMAGE))
	{
		edv_id3_prop_page_update_widgets_frame_image(
			p,
			frame
		);
	}

	/* Add/update this value on the Frames GtkCList */
	edv_id3_prop_page_frames_list_update(
		p,
		frame
	);


	p->freeze_count--;
}

/*
 *	Opens the image data from the EDVID3Frame, displays it on the
 *	thumb, and sets the EDV_ID3_PROP_PAGE_HAVE_IMAGE flag.
 *
 *	The frame specifies the EDVID3Frame which must have an ID of
 *	EDV_ID3_FRAME_ID_IMAGE. If frame is NULL then the thumb will
 *	be cleared and the EDV_ID3_PROP_PAGE_HAVE_IMAGE flag will
 *	be unset.
 */
static void edv_id3_prop_page_update_widgets_frame_image(
	EDVID3PropPage *p,
	EDVID3Frame *frame
)
{
	gint		image_code,
			width, height,
			bpl,
			nframes;
	const gchar *mime_type_type;
	guint8		bg_color[4];
	GList		*delay_list = NULL,
			*rgba_list = NULL;
	GtkWidget *w = p->thumb_da;
	GtkStyle *style = gtk_widget_get_style(w);
	EDVID3Field *field;

	p->freeze_count++;

	/* Clear? */
	if(frame == NULL)
	{
		/* Clear the thumb and remove the
		 * EDV_ID3_PROP_PAGE_HAVE_IMAGE flag
		 */
		edv_id3_prop_page_thumb_clear(p);
		p->flags &= ~EDV_ID3_PROP_PAGE_HAVE_IMAGE;
		GUISetWidgetTip(
			w,
			EDV_ID3_PROP_PAGE_THUMB_TOOLTIP
		);
		p->freeze_count--;
		return;
	}

	/* Set the initial default background color from the
	 * configuration
	 */
	if(style != NULL)
	{
		GdkColor *c = &style->white;
		bg_color[0] = (guint8)(c->red >> 8);
		bg_color[1] = (guint8)(c->green >> 8);
		bg_color[2] = (guint8)(c->blue >> 8);
		bg_color[3] = 0xFF;
	}

	/* Get get MIME Type from the EDVID3Frame */
	field = edv_id3_frame_get_field_by_type(
		frame,
		EDV_ID3_FIELD_TYPE_MIME_TYPE
	);
	mime_type_type = edv_id3_field_get_string(field);

	/* Get get image code from the EDVID3Frame */
	field = edv_id3_frame_get_field_by_type(
		frame,
		EDV_ID3_FIELD_TYPE_INT8
	);
	image_code = edv_id3_field_get_number(field);

	/* Get/read the RGBA image data from the EDVID3Frame */
	field = edv_id3_frame_get_field_by_type(
		frame,
		EDV_ID3_FIELD_TYPE_DATA
	);
	rgba_list = edv_id3_field_get_image_rgba(
		field,
		&width, &height,
		&bpl,
		&nframes,
		&delay_list,
		bg_color
	);
	if(rgba_list != NULL)
	{
		/* Set the thumb's GtkDrawingArea's RGBA image data,
		 * map the thumb, and queue it to redraw
		 */
		edv_id3_prop_page_thumb_set_rgba(
			p,
			rgba_list,
			width, height,
			bpl,
			nframes,
			bg_color,
			width, height		/* Original size */
		);

		/* Mark that an image has been loaded */
		p->flags |= EDV_ID3_PROP_PAGE_HAVE_IMAGE;
	}

	/* Update the thumb's GtkDrawingArea tooltip */
	if(STRISEMPTY(frame->description))
	{
		const gchar *image_code_s = "Other Image";
		gchar *s;
		switch(image_code)
		{
		    case 3:
			image_code_s = "Cover Image";
			break;
		}
		s = g_strdup_printf(
			"%s\n%ix%i\n%s\n%s",
			image_code_s,
			width, height,
			(mime_type_type != NULL) ? mime_type_type : "MIME Type Unknown",
			EDV_ID3_PROP_PAGE_THUMB_TOOLTIP
		);
		GUISetWidgetTip(
			w,
			s
		);
		g_free(s);
	}
	else
	{
		const gchar *image_code_s = "Other Image";
		gchar *s;
		switch(image_code)
		{
		    case 3:
			image_code_s = "Cover Image";
			break;
		}
		s = g_strdup_printf(
			"%s\n%s\n%ix%i\n%s\n%s",
			frame->description,
			image_code_s,
			width, height,
			(mime_type_type != NULL) ? mime_type_type : "MIME Type Unknown",
			EDV_ID3_PROP_PAGE_THUMB_TOOLTIP
		);
		GUISetWidgetTip(
			w,
			s
		);
		g_free(s);
	}

	/* Delete the loaded image data */
	if(rgba_list != NULL)
	{
		g_list_foreach(rgba_list, (GFunc)g_free, NULL);
		g_list_free(rgba_list);
	}
	g_list_free(delay_list);

	p->freeze_count--;
}

/*
 *	Updates the displayed values for the EDVID3Frame on the Frames
 *	GtkCList.
 *
 *	The frame specifies the EDVID3Frame on the Frames GtkCList to
 *	update. If the EDVID3Frame is not set or not found on the
 *	Frames GtkCList then a new row displaying the EDVID3Frame
 *	will be appended.
 */
static void edv_id3_prop_page_frames_list_update(
	EDVID3PropPage *p,
	EDVID3Frame *frame
)
{
	gint		column,
			row;
	GtkWidget *w = p->frames_list;
	GtkCList *clist = GTK_CLIST(w);
	const gint ncolumns = MAX(clist->columns, 1);

	p->freeze_count++;

	if(frame == NULL)
	{
		p->freeze_count--;
		return;
	}

	gtk_clist_freeze(clist);

	/* Check if a row is already displaying this EDVID3Frame */
	row = gtk_clist_find_row_from_data(
		clist,
		frame
	);
	if(row < 0)
	{
		gint i;
		gchar **cell_text = (gchar **)g_malloc(
			ncolumns * sizeof(gchar *)
		);
		for(i = 0; i < ncolumns; i++)
			cell_text[i] = "";
		row = gtk_clist_append(
			clist,
			cell_text
		);
		g_free(cell_text);
		if(row < 0)
		{
			gtk_clist_thaw(clist);
			p->freeze_count--;
			return;
		}
		gtk_clist_set_row_data(
			clist,
			row,
			frame			/* Shared */
		);
	}

	/* ID */
	column = 0;
	if(ncolumns > column)
	{
		const gchar *text = frame->id;
		gtk_clist_set_text(
			clist,
			row, column,
			(text != NULL) ? text : ""
		);
	}

	/* Text Encoding */
	column = 1;
	if(ncolumns > column)
	{
		const gchar *text = NULL;
		switch(frame->text_encoding)
		{
		    case EDV_ID3_TEXT_ENCODING_UNKNOWN:
			break;
		    case EDV_ID3_TEXT_ENCODING_UTF_8:
			text = "UTF-8";
			break;
		    case EDV_ID3_TEXT_ENCODING_UTF_16:
			text = "UTF-16";
			break;
		    case EDV_ID3_TEXT_ENCODING_UTF_16_BE:
			text = "UTF-16 BE";
			break;
		    case EDV_ID3_TEXT_ENCODING_ISO_8859_1:
			text = "ISO-8859-1";
			break;
		}
		gtk_clist_set_text(
			clist,
			row, column,
			(text != NULL) ? text : ""
		);
	}

	/* Description */
	column = 2;
	if(ncolumns > column)
	{
		const gchar *text = frame->description;
		gtk_clist_set_text(
			clist,
			row, column,
			(text != NULL) ? text : ""
		);
	}

	gtk_clist_columns_autosize(clist);

	gtk_clist_thaw(clist);

	p->freeze_count--;
}

#if 0
/*
 *	Updates all the displayed values of all the EDVID3Frames
 *	on the Frames GtkCList.
 */
static void edv_id3_prop_page_frames_list_update_all(EDVID3PropPage *p)
{
	GList *glist;
	EDVID3Tag *tag;

	p->freeze_count++;

	tag = p->tag;
	if(tag != NULL)
	{
		if(tag->frames_list != NULL)
		{
			GList *glist;
			EDVID3Frame *frame;
			for(glist = tag->frames_list;
			    glist != NULL;
			    glist = g_list_next(glist)
			)
			{
				frame = EDV_ID3_FRAME(glist->data);
				edv_id3_prop_page_frames_list_update(
					p,
					frame
				);
			}
		}
	}

	p->freeze_count--;
}
#endif

/*
 *	Updates the Fields GtkCList's row with values from a
 *	EDVID3Field.
 *
 *	The row specifies the GtkCList row to update.
 *
 *	The field specifies the EDVID3Field with the values. If field
 *	is NULL then empty values will be displayed on the row.
 */
static void edv_id3_prop_page_fields_list_update(
	EDVID3PropPage *p,
	const gint row,
	EDVID3Field *field
)
{
	gint column;
	gchar *s;
	GtkCList *clist = GTK_CLIST(p->fields_list);
	const gint ncolumns = clist->columns;

	p->freeze_count++;

	gtk_clist_freeze(clist);

	/* Type */
	s = NULL;
	if(field != NULL)
	{
		switch(field->type)
		{
		    case EDV_ID3_FIELD_TYPE_UNKNOWN:
			s = g_strdup("UNKNOWN");
			break;
		    case EDV_ID3_FIELD_TYPE_INT8:
			s = g_strdup("INT8");
			break;
		    case EDV_ID3_FIELD_TYPE_INT16:
			s = g_strdup("INT16");
			break;
		    case EDV_ID3_FIELD_TYPE_INT32:
			s = g_strdup("INT32");
			break;
		    case EDV_ID3_FIELD_TYPE_INT64:
			s = g_strdup("INT64/INT32PLUS");
			break;
		    case EDV_ID3_FIELD_TYPE_STRING:
			s = g_strdup("STRING");
			break;
		    case EDV_ID3_FIELD_TYPE_DATA:
			s = g_strdup("BINARY");
			break;
		    case EDV_ID3_FIELD_TYPE_LANGUAGE:
			s = g_strdup("LANGUAGE");
			break;
		    case EDV_ID3_FIELD_TYPE_MIME_TYPE:
			s = g_strdup("MIME_TYPE");
			break;
		}
	}
	column = 0;
	if(column < ncolumns)
		gtk_clist_set_text(
			clist,
			row, column,
			(s != NULL) ? s : ""
		);
	g_free(s);

	/* Value */
	s = NULL;
	if(field != NULL)
	{
		switch(field->type)
		{
		    case EDV_ID3_FIELD_TYPE_UNKNOWN:
			break;
		    case EDV_ID3_FIELD_TYPE_INT8:
			s = g_strdup_printf(
				"%i",
				edv_id3_field_get_number(field)
			);
			break;
		    case EDV_ID3_FIELD_TYPE_INT16:
			s = g_strdup_printf(
				"%i",
				edv_id3_field_get_number(field)
			);
			break;
		    case EDV_ID3_FIELD_TYPE_INT32:
			s = g_strdup_printf(
				"%i",
				edv_id3_field_get_number(field)
			);
			break;
		    case EDV_ID3_FIELD_TYPE_INT64:
			s = g_strdup_printf(
				"%ld",
				edv_id3_field_get_long(field)
			);
			break;
		    case EDV_ID3_FIELD_TYPE_STRING:
			s = STRDUP(edv_id3_field_get_string(field));
			break;
		    case EDV_ID3_FIELD_TYPE_DATA:
			if((field->data != NULL) && (field->data_length > 0))
			{
				const gint	limit = 10,
						n = field->data_length;
				gint i;
				gchar *sn;
				const guint8 *data = (guint8 *)field->data;

				s = g_strdup("");
				for(i = 0; i < n; i++)
				{
					if(i >= limit)
						break;

					sn = g_strdup_printf(
						"%.2X",
						data[i]
					);
					if(sn != NULL)
					{
						gchar *s_tmp = g_strconcat(
							s,
							(i > 0) ? " " : "",
							sn,
							NULL
						);
						if(s_tmp != NULL)
						{
							g_free(s);
							s = s_tmp;
						}
						g_free(sn);
					}
				}
				if(i >= limit)
				{
					gchar *s_tmp = g_strconcat(
						s,
						"...",
						NULL
					);
					if(s_tmp != NULL)
					{
						g_free(s);
						s = s_tmp;
					}
				}
				sn = g_strdup_printf(
					" (%i %s)",
					n,
					(n == 1) ? "byte" : "bytes"
				);
				if(sn != NULL)
				{
					gchar *s_tmp = g_strconcat(
						s,
						sn,
						NULL
					);
					if(s_tmp != NULL)
					{
						g_free(s);
						s = s_tmp;
					}
					g_free(sn);
				}
			}
			break;
		    case EDV_ID3_FIELD_TYPE_LANGUAGE:
			s = STRDUP(edv_id3_field_get_string(field));
			break;
		    case EDV_ID3_FIELD_TYPE_MIME_TYPE:
			s = STRDUP(edv_id3_field_get_string(field));
			break;
		}
	}
	column = 1;
	if(column < ncolumns)
		gtk_clist_set_text(
			clist,
			row, column,
			(s != NULL) ? s : ""
		);
	g_free(s);

	gtk_clist_columns_autosize(clist);

	gtk_clist_thaw(clist);

	p->freeze_count--;
}

/*
 *	Clears the Fields GtkCList and appends a new list of
 *	EDVID3Fields from the EDVID3Frame to the Fields GtkCList.
 *
 *	The frame specifies the EDVID3Frame with the EDVID3Fields to
 *	add to the Fields GtkCList. If frame is NULL then the Fields
 *	GtkCList will be cleared.
 */
static void edv_id3_prop_page_fields_list_update_all(
	EDVID3PropPage *p,
	EDVID3Frame *frame
)
{
	GtkWidget *w = p->fields_list;
	GtkCList *clist = GTK_CLIST(w);
	const gint ncolumns = MAX(clist->columns, 1);

	p->freeze_count++;

	gtk_clist_freeze(clist);

	gtk_clist_clear(clist);

	if(frame != NULL)
	{
		gint i, row;
		gchar **cell_text = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
		GList *glist;
		EDVID3Field *field;

		for(i = 0; i < ncolumns; i++)
			cell_text[i] = "";

		for(glist = frame->fields_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			field = EDV_ID3_FIELD(glist->data);
			if(field == NULL)
				continue;

			row = gtk_clist_append(
				clist,
				cell_text
			);
			if(row < 0)
				break;

			edv_id3_prop_page_fields_list_update(
				p,
				row,
				field
			);
		}

		g_free(cell_text);
	}

	gtk_clist_columns_autosize(clist);

	gtk_clist_thaw(clist);

	p->freeze_count--;
}
