#include <errno.h>
#include <gtk/gtk.h>

#include "cfg.h"

#include "guiutils.h"
#include "menu_button.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_property.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_recycled_obj.h"
#include "libendeavour2-base/edv_recycled_obj_stat.h"
#include "libendeavour2-base/edv_id.h"
#include "edv_ids_list.h"
#include "edv_date_format.h"
#include "edv_pixmap.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "edv_obj_info_match.h"
#include "edv_utils_gtk.h"
#include "mime_type_edit_dlg.h"
#include "obj_op_dlg.h"
#include "prop_page.h"
#include "prop_page_general.h"
#include "edv_vfs_obj_op.h"
#include "edv_recycled_obj_op.h"
#include "edv_emit.h"
#include "edv_op.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_time_stamp_20x20.xpm"


typedef struct _EDVGeneralPropPage	EDVGeneralPropPage;
#define EDV_GENERAL_PROP_PAGE(p)	((EDVGeneralPropPage *)(p))


/*
 *	Flags:
 */
typedef enum {
	EDV_GENERAL_PROP_PAGE_HAS_CHANGES	\
					= (1 << 7)
} EDVGeneralPropPageFlags;


/* Check Support */
gboolean edv_general_prop_page_query_create_cb(
	EDVPropPageContext *ctx,
	gint *version_major_rtn,
	gint *version_minor_rtn,
	gint *version_release_rtn,
	gchar **page_name_rtn,
	edv_pixmap_data **pixmap_data_20x20_rtn,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list
);

/* Create */
gpointer edv_general_prop_page_create_cb(
	EDVPropPageContext *ctx,
	GtkWidget *parent
);

/* Update */
void edv_general_prop_page_update_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list,
	const int error_code,
	gpointer data
);

/* Apply */
gboolean edv_general_prop_page_apply_vfs_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	GList *properties_list,
	gpointer data
);
gboolean edv_general_prop_page_apply_recycle_bin_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	GList *properties_list,
	gpointer data
);

/* Destroy */
void edv_general_prop_page_destroy_cb(
	EDVPropPageContext *ctx,
	gpointer data
);

/* Callbacks */
static void edv_general_prop_page_changed_cb(GtkWidget *widget, gpointer data);

static void edv_general_prop_page_owner_map_pulist_cb(GtkWidget *widget, gpointer data);
static void edv_general_prop_page_group_map_pulist_cb(GtkWidget *widget, gpointer data);

static void edv_general_prop_page_add_mime_type_cb(GtkWidget *widget, gpointer data);
static void edv_general_prop_page_edit_mime_type_cb(GtkWidget *widget, gpointer data);

static void edv_general_prop_page_touch_cb(GtkWidget *widget, gpointer data);
static void edv_general_prop_page_touch_specific_cb(GtkWidget *widget, gpointer data);

/* Operations */
static void edv_general_prop_page_add_mime_type(EDVGeneralPropPage *p);
static void edv_general_prop_page_edit_mime_type(EDVGeneralPropPage *p);

static void edv_general_prop_page_touch(EDVGeneralPropPage *p);
static void edv_general_prop_page_touch_specific(EDVGeneralPropPage *p);

/* Update Widgets */
static void edv_general_prop_page_update_widgets(EDVGeneralPropPage *p);


#define EDV_GENERAL_PROP_PAGE_NAME	"General"

#define EDV_PROG_DLG_USE_PERMISSION_CHECK_BUTTON	\
					FALSE

#define EDV_PROP_DLG_VALUE_NOT_AVAILABLE_STR	\
					"*Not Available*"
#define EDV_PROP_DLG_TIME_STAMP_NOT_SET_STR	\
					"*Not Set*"


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	General Page:
 */
struct _EDVGeneralPropPage {
	GtkWidget	*toplevel;
	gint		freeze_count;
	EDVPropPageContext	*ctx;
	EDVGeneralPropPageFlags	flags;

	GtkWidget	*icon_pm,
			*name_label,
			*location_label,
			*type_label,
			*size_label,
			*add_mime_type_btn,
			*edit_mime_type_btn,
			*timestamps_frame,
			*date_access_label,
			*date_modify_label,
			*date_change_label,
			*date_deleted_label,
			*date_touch_btn,
			*ownership_frame,
			*owner_entry,
			*owner_btn,
			*group_entry,
			*group_btn,
			*permissions_frame,
			*ur_check,
			*uw_check,
			*ux_check,
			*gr_check,
			*gw_check,
			*gx_check,
			*or_check,
			*ow_check,
			*ox_check,
			*suid_check,
			*sgid_check,
			*sticky_check;

	edv_mime_type_edit_dlg_struct *mime_type_edit_dlg;
};


/*
 *	Check support callback.
 *
 *	Called to test if we want to create a page for the object
 *	based on the specified object's properties.
 *
 *	Returns TRUE if we want to create a page or FALSE if we do
 *	not want to.
 */
gboolean edv_general_prop_page_query_create_cb(
	EDVPropPageContext *ctx,
	gint *version_major_rtn,
	gint *version_minor_rtn,
	gint *version_release_rtn,
	gchar **page_name_rtn,
	edv_pixmap_data **pixmap_data_20x20_rtn,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list
)
{
	/* Always create the General Page */
	*version_major_rtn = PROG_VERSION_MAJOR;
	*version_minor_rtn = PROG_VERSION_MINOR;
	*version_release_rtn = PROG_VERSION_RELEASE;
	*page_name_rtn = g_strdup(EDV_GENERAL_PROP_PAGE_NAME);
	return(TRUE);
}


/*
 *	Creates callback.
 *
 *	Called when we need to create a new page.
 *
 *	Returns our pointer to the the page data or NULL on error or
 *	if we do not want to create a page.
 */
gpointer edv_general_prop_page_create_cb(
	EDVPropPageContext *ctx,
	GtkWidget *parent
)
{
	const gint	border_major = 5,
			border_minor = 2;
	gint font_size;
	const gchar *font_encoding;
	gchar *font_name_h1_bold;
	GList *properties_list = edv_prop_page_get_properties_list(ctx);
	GdkFont *font;
	GtkRcStyle *rcstyle;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	GtkStyle *style = gtk_widget_get_style(toplevel);
	GtkAccelGroup *accelgrp = edv_prop_page_get_accelgrp(ctx);
	GtkWidget	*w,
			*menu,
			*parent2, *parent3, *parent4, *parent5;
	EDVPixmap *icon;
	const EDVLocationType location_type = edv_prop_page_get_location_type(ctx);
	const EDVObjectType type = (EDVObjectType)edv_properties_list_get_i(
		properties_list,
		EDV_PROP_NAME_TYPE
	);
	EDVMIMEType *m;
	EDVCore *core = edv_prop_page_get_core(ctx);
	EDVGeneralPropPage *p = EDV_GENERAL_PROP_PAGE(g_malloc0(
		sizeof(EDVGeneralPropPage)
	));
	if(p == NULL)
		return(NULL);

	p->ctx = ctx;
	p->toplevel = parent;

	p->freeze_count++;

	/* Get the base font size */
	font = style->font;
	font_size = GDK_FONT_GET_FONT_NAME_SIZE(font);
	if(font_size < 3)
		font_size = 3;

	/* Format the font names */
#if defined(PROG_LANGUAGE_POLISH)
	font_encoding = "iso8859-2";
#else
	font_encoding = "iso8859-1";
#endif

	font_name_h1_bold = g_strdup_printf(
"-adobe-helvetica-bold-r-normal-*-%i-*-*-*-*-*-%s",
		font_size + 2,
		font_encoding
	);

	m = edv_mime_types_list_match_type(
		core->mime_types_list,
		NULL,				/* No index return */
		EDV_MIME_TYPE_TYPE_INODE_UNKNOWN,
		FALSE				/* Not case sensitive */
	);
	if(m != NULL)
	{
		edv_mime_type_realize(m, FALSE);
		icon = edv_pixmap_ref(m->medium_icon[
			EDV_MIME_TYPE_ICON_STATE_STANDARD
		]);
	}
	else
	{
		icon = NULL;
	}

	/* Identification GtkFrame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Identificacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Identification"
#elif defined(PROG_LANGUAGE_GERMAN)
"Identifikation"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Identificazione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Identificatie"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Identificao"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Identification"
#else
"Identification"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_minor);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Name GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Icon */
	if(edv_pixmap_is_loaded(icon))
	{
		p->icon_pm = w = edv_pixmap_new_gtk_pixmap(icon);
		if(w != NULL)
		{
			gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
			gtk_widget_show(w);
		}
	}
	else
	{
		p->icon_pm = NULL;
	}
	/* Label */
	p->name_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = STRDUP(font_name_h1_bold);
	gtk_widget_modify_style(w, rcstyle);
	GTK_RC_STYLE_UNREF(rcstyle);
	gtk_widget_show(w);

	/* GtkVBox for the MIME Type buttons */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_end(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Add MIME Type GtkButton */
	p->add_mime_type_btn = w = gtk_button_new_with_label(
		"Define Type"
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_general_prop_page_add_mime_type_cb), p
	);
	GUISetWidgetTip(
		w,
"Add a new MIME Type for this type of object"
	);

	/* Edit MIME Type GtkButton */
	p->edit_mime_type_btn = w = gtk_button_new_with_label(
		"Edit Type"
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_general_prop_page_edit_mime_type_cb), p
	);
	GUISetWidgetTip(
		w,
"Edit the MIME Type for this type of object"
	);

	/* Location GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Label */
	switch(location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
		w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"La Ubicacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Emplacement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Ort"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Posizione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Plaats"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Localidade"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Plassering"
#else
"Location"
#endif
			":"
		);
		break;
	  case EDV_LOCATION_TYPE_RECYCLE_BIN:
		w = gtk_label_new("Index:");
		break;
	  case EDV_LOCATION_TYPE_ARCHIVE:
		w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"La Ubicacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Emplacement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Ort"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Posizione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Plaats"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Localidade"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Plassering"
#else
"Location In Archive"
#endif
			":"
		);
		break;
	}
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Label */
	p->location_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Type & Size GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Type GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Label */
	w = gtk_label_new("Type:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Type label */
	p->type_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Size */
	if(type == EDV_OBJECT_TYPE_FILE)
	{
		/* Size GtkHBox */
		w = gtk_hbox_new(FALSE, border_major);
		gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
		gtk_widget_show(w);
		parent4 = w;
		/* Label */
		w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Tamao"
#elif defined(PROG_LANGUAGE_FRENCH)
"Taille"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gre"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Misura"
#elif defined(PROG_LANGUAGE_DUTCH)
"Maat"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Tamanho"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Strrelse"
#else
"Size"
#endif
			":"
		);
		gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
		gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		/* Size Label */
		p->size_label = w = gtk_label_new("");
		gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
		gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		/* Units Label */
		w = gtk_label_new("bytes");
		gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
		gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
	}
	else
	{
		/* Size GtkHBox */
		w = gtk_hbox_new(FALSE, border_major);
		gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
		gtk_widget_show(w);
		parent4 = w;
		/* Label */
		w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Tamao"
#elif defined(PROG_LANGUAGE_FRENCH)
"Taille"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gre"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Misura"
#elif defined(PROG_LANGUAGE_DUTCH)
"Maat"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Tamanho"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Strrelse"
#else
"Size"
#endif
			":"
		);
		gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
		gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
		GTK_WIDGET_SET_SENSITIVE(w, FALSE);
		gtk_widget_show(w);
	}

	/* Time Stamps GtkFrame */
	p->timestamps_frame = w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"El tiempo Estampa"
#elif defined(PROG_LANGUAGE_FRENCH)
"Horodateur"
#elif defined(PROG_LANGUAGE_GERMAN)
"Zeit Stempelt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Tempo Timbra"
#elif defined(PROG_LANGUAGE_DUTCH)
"Tijd Stempelt"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Tempo Sela"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Tid Stamps"
#else
"Time Stamps"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	/* Date labels & touch button GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Date labels GtkVBox */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Last Accessed GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Prefix label */
	w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);
	gtk_widget_set_usize(w, 100, -1);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Dure Accesado"
#elif defined(PROG_LANGUAGE_FRENCH)
"Dernier Accs"
#elif defined(PROG_LANGUAGE_GERMAN)
"Leisten Zugegriffen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Durare Accesso"
#elif defined(PROG_LANGUAGE_DUTCH)
"Leest Had Toegang Tot"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Dure Acedido"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Siste Accessed"
#else
"Last Accessed"
#endif
		":"
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_container_add(GTK_CONTAINER(parent5), w);
	gtk_widget_show(w);
	/* Date label */
	p->date_access_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Last Modified GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Prefix label */
	w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);
	gtk_widget_set_usize(w, 100, -1);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Dure Modificado"
#elif defined(PROG_LANGUAGE_FRENCH)
"Dernire Modification"
#elif defined(PROG_LANGUAGE_GERMAN)
"Dauern Modifiziert"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Durare Modificato"
#elif defined(PROG_LANGUAGE_DUTCH)
"Duur Wijzigde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Dure Modificado"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Siste Modified"
#else
"Last Modified"
#endif
		":"
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_container_add(GTK_CONTAINER(parent5), w);
	gtk_widget_show(w);
	/* Date label */
	p->date_modify_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Last Changed GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Prefix label */
	w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);
	gtk_widget_set_usize(w, 100, -1);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Dure Cambiado"
#elif defined(PROG_LANGUAGE_FRENCH)
"Dernier changement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Dauern Gendert"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Durare Cambiato"
#elif defined(PROG_LANGUAGE_DUTCH)
"Duur Veranderde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Dure Mudado"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Siste Changed"
#else
"Last Changed"
#endif
		":"
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_container_add(GTK_CONTAINER(parent5), w);
	gtk_widget_show(w);
	/* Date label */
	p->date_change_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	if(location_type == EDV_LOCATION_TYPE_RECYCLE_BIN)
	{
		/* Deleted On GtkHBox */
		w = gtk_hbox_new(FALSE, border_major);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent4 = w;
		/* Prefix GtkLabel */
		w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);
		gtk_widget_set_usize(w, 100, -1);
		gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent5 = w;
		w = gtk_label_new("Deleted On:");
		gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
		gtk_container_add(GTK_CONTAINER(parent5), w);
		gtk_widget_show(w);
		/* Date GtkLabel */
		p->date_deleted_label = w = gtk_label_new("");
		gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
		gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
	}

	/* Touch button GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_end(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Touch button */
	p->date_touch_btn = w = menu_hbutton_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Toque"
#elif defined(PROG_LANGUAGE_FRENCH)
"Maintenant"
#elif defined(PROG_LANGUAGE_GERMAN)
"Berhrung"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Tocco"
#elif defined(PROG_LANGUAGE_DUTCH)
"Aanraking"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Toque"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Berring"
#else
"Touch"
#endif
		,
		(guint8 **)icon_time_stamp_20x20_xpm,
		&menu
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_general_prop_page_touch_cb), p
	);
	menu_button_set_map_trigger(w, MENU_BUTTON_MAP_TYPE_PRESSED_DRAG);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Modifique Sellos De Tiempo (el clic y el obstculo para el men)"
#elif defined(PROG_LANGUAGE_FRENCH)
"Modifier la datation (dclic et trane pour le menu)"
#elif defined(PROG_LANGUAGE_GERMAN)
"Modifizieren Sie Zeit Briefmarken (klicken und widerstand fr men)"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Modificare I Francobolli Di Tempo (lo scatto e trascinare per il menu)"
#elif defined(PROG_LANGUAGE_DUTCH)
"Wijziig Tijd Stempels (klik en ruk voor menu)"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Modifique Selos De Tempo (estalido e arrasta para cardpio)"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Modifiser Time Stamps (klikk og hindring for meny)"
#else
"Modify Time Stamps (click and drag for menu)"
#endif
	);
	gtk_widget_show(w);
	/* Touch menu */
	if(menu != NULL)
	{
		guint accel_key, accel_mods;
		const gchar *label;
		edv_pixmap_data *icon;
		gpointer data = p;
		void (*func_cb)(GtkWidget *w, gpointer);

#define ADD_MENU_ITEM_LABEL	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_LABEL,			\
  accelgrp,					\
  icon,						\
  label,					\
  accel_key, accel_mods,			\
  func_cb, data					\
 );						\
}
#define ADD_MENU_ITEM_SEPARATOR	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL,						\
  NULL,						\
  0, 0,						\
  NULL, NULL					\
 );						\
}
		icon = NULL;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"La Fecha Actual & Tiempo"
#elif defined(PROG_LANGUAGE_FRENCH)
"La Date & heure courante"
#elif defined(PROG_LANGUAGE_GERMAN)
"Jetziges Datum & Zeit"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Data Attuale & Il Tempo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Huidig Datum & Tijd"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Data Atual & Tempo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Nvrende Date & Time"
#else
"Current Date & Time"
#endif
		;
		accel_key = 0;
		accel_mods = 0;
		func_cb = edv_general_prop_page_touch_cb;
		ADD_MENU_ITEM_LABEL

		icon = NULL;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"La Fecha Fija & Tiempo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Rgler ~La date & l'heure"
#elif defined(PROG_LANGUAGE_GERMAN)
"Festes Datum & Zei"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Ha Regolato La Data & Il Tempo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Vast Datum & Tijd"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Data fixa & Tempo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Fast Date & Time"
#else
"Set Date & Time"
#endif
		"...";
		accel_key = 0;
		accel_mods = 0;
		func_cb = edv_general_prop_page_touch_specific_cb;
		ADD_MENU_ITEM_LABEL

		gtk_widget_modify_style_recursive(
			menu,
			core->standard_rcstyle
		);

#undef ADD_MENU_ITEM_SEPARATOR
#undef ADD_MENU_ITEM_LABEL
	}


	/* Ownership GtkFrame */
	p->ownership_frame = w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Propiedad"
#elif defined(PROG_LANGUAGE_FRENCH)
"Propritaire"
#elif defined(PROG_LANGUAGE_GERMAN)
"Eigentumsrecht"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Propriet"
#elif defined(PROG_LANGUAGE_DUTCH)
"Eigendomsrecht"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Posse"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Eiendomsrett"
#else
"Ownership"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_minor);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Owner & Group GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Owner GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Label */
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Dueo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Propritaire"
#elif defined(PROG_LANGUAGE_GERMAN)
"Eigentmer"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Proprietario"
#elif defined(PROG_LANGUAGE_DUTCH)
"Eigenaar"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Proprietrio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Eier"
#else
"Owner"
#endif
		":"
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Entry and map button GtkHBox */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Entry */
	p->owner_entry = w = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(parent5), w, TRUE, TRUE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(edv_general_prop_page_changed_cb), p
	);
	GUIEditableEndowPopupMenu(w, 0);
	gtk_widget_show(w);
	/* Popup list map button */
	p->owner_btn = w = PUListNewMapButtonArrow(
		GTK_ARROW_DOWN, GTK_SHADOW_OUT,
		edv_general_prop_page_owner_map_pulist_cb, p
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Group GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Label */
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Grupo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Groupe de propritaire"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gruppe"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Gruppo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Groep"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Grupo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Gruppe"
#else
"Group"
#endif
		":"
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Entry and map button GtkHBox */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Entry */
	p->group_entry = w = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(parent5), w, TRUE, TRUE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(edv_general_prop_page_changed_cb), p
	);
	GUIEditableEndowPopupMenu(w, 0);
	gtk_widget_show(w);
	/* Popup list map button */
	p->group_btn = w = PUListNewMapButtonArrow(
		GTK_ARROW_DOWN, GTK_SHADOW_OUT,
		edv_general_prop_page_group_map_pulist_cb, p
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Permissions GtkFrame */
	p->permissions_frame = w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
		"Los Permisos"
#elif defined(PROG_LANGUAGE_FRENCH)
		"Permissions"
#else
		"Permissions"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* GtkVBox for standard permission labels and toggle buttons */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Labels GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
		"El Dueo"
#elif defined(PROG_LANGUAGE_FRENCH)
		"Propritaire"
#else
		"Owner"
#endif
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_CENTER);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
		"El Grupo"
#elif defined(PROG_LANGUAGE_FRENCH)
		"Groupe"
#else
		"Group"
#endif
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_CENTER);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
		"Otro"
#elif defined(PROG_LANGUAGE_FRENCH)
		"Autre"
#else
		"Other"
#endif
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_CENTER);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_widget_show(w);

	/* Toggle buttons GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

#define NEW_PERMISSION_TOGGLE_BUTTON(_label_,_tip_,_parent_)	{\
 if(EDV_PROG_DLG_USE_PERMISSION_CHECK_BUTTON) {			\
  /* Create permission button as a GtkCheckButton */		\
  w = gtk_check_button_new_with_label(_label_);			\
 } else {							\
  /* Create permission button as a GtkToggleButton */		\
  GdkFont *font = style->font;					\
  const gint font_height = (font != NULL) ?			\
   (font->ascent + font->descent) : 0;				\
  w = gtk_toggle_button_new_with_label(_label_);		\
  if(font_height > 0)						\
   gtk_widget_set_usize(					\
    w, MIN(font_height + 5, 20), MIN(font_height + 5, 20)	\
   );								\
 }								\
 gtk_box_pack_start(GTK_BOX(_parent_), w, FALSE, FALSE, 0);	\
 gtk_signal_connect(						\
  GTK_OBJECT(w), "toggled",					\
  GTK_SIGNAL_FUNC(edv_general_prop_page_changed_cb), p	\
 );								\
 if((_tip_) != NULL)						\
  GUISetWidgetTip(w, (_tip_));					\
 gtk_widget_show(w);						\
}
	/* Owner GtkToggleButton set GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Read GtkToggleButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
		"R",
"Check this to allow the owner of the object to read from it",
		parent5
	);
	p->ur_check = w;
	/* Write GtkToggleButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
		"W",
"Check this to allow the owner of the object to write to it",
		parent5
	);
	p->uw_check = w;
	/* Execute GtkToggleButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
		"X",
"Check this to allow the owner of the object to execute it.\
 If the object is a directory then this allows the owner to list\
 and access its contents",
		parent5
	);
	p->ux_check = w;

	/* Group GtkToggleButton set GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Read GtkToggleButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
		"R",
"Check this to allow any group member of the object to read from it",
		parent5
	);
	p->gr_check = w;
	/* Write GtkToggleButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
		"W",
"Check this to allow any group member of the object to write to it",
		parent5
	);
	p->gw_check = w;
	/* Execute GtkToggleButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
		"X",
"Check this to allow any group member of the object to execute it.\
 If the object is a directory then this allows any group member to list\
 and access its contents",
		parent5
	);
	p->gx_check = w;

	/* Other/Anonymous GtkToggleButton set GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Read GtkToggleButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
		"R",
"Check this to allow anyone to read from the object",
		parent5
	);
	p->or_check = w;
	/* Write GtkToggleButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
		"W",
"Check this to allow anyone to write to the object",
		parent5
	);
	p->ow_check = w;
	/* Execute GtkToggleButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
		"X",
"Check this to allow anyone to execute the object.\
 If the object is a directory then this allows anyone to list\
 and access its contents",
		parent5
	);
	p->ox_check = w;

#undef NEW_PERMISSION_TOGGLE_BUTTON

	/* SetUID, SetGID, and Sticky GtkHBox */
	w = gtk_hbox_new(TRUE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* SetUID GtkCheckButton */
	p->suid_check = w = gtk_check_button_new_with_label("SetUID");
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(edv_general_prop_page_changed_cb), p
	);
	GUISetWidgetTip(
		w,
"Check this to set the Set User ID Bit on the object.\
 When this is set, the executed processes of the object will\
 inherit the object's owner permissions instead of the\
 executer's permissions"
	);
	gtk_widget_show(w);

	/* SetGID GtkCheckButton */
	p->sgid_check = w = gtk_check_button_new_with_label("SetGID");
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(edv_general_prop_page_changed_cb), p
	);
	GUISetWidgetTip(
		w,
"Check this to set the Set Group ID Bit on the object.\
 When this is set, the executed processes of the object will\
 inherit the object's group permissions instead of the\
 executer's permissions"
	);
	gtk_widget_show(w);

	/* Sticky GtkCheckButton */
	p->sticky_check = w = gtk_check_button_new_with_label("Sticky");
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(edv_general_prop_page_changed_cb), p
	);
	GUISetWidgetTip(
		w,
"Check this to set the Sticky Bit on the object.\
 When this is set on a directory, only owners of objects can\
 modify them. The affects of this bit on other types of objects\
 vary, see help"
	);
	gtk_widget_show(w);


	(void)edv_pixmap_unref(icon);
	g_free(font_name_h1_bold);


	p->freeze_count--;

	return(p);
}


/*
 *	Update callback.
 *
 *	Called when we need to update our displayed values for the
 *	object's properties.
 */
void edv_general_prop_page_update_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list,
	const int error_code,
	gpointer data
)
{
	EDVGeneralPropPage *p = EDV_GENERAL_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	edv_general_prop_page_update_widgets(p);

	p->freeze_count--;
}

/*
 *	Apply VFS object callback.
 */
gboolean edv_general_prop_page_apply_vfs_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	GList *properties_list,
	gpointer data
)
{
	gboolean	status = FALSE,
			yes_to_all = FALSE;
	const gchar *path;
	GtkWidget	*w,
			*toplevel = edv_prop_page_get_toplevel(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);
	EDVGeneralPropPage *p = EDV_GENERAL_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return(status);

	p->freeze_count++;

	/* Do not apply if we did not make any changes */
	if(!(p->flags & EDV_GENERAL_PROP_PAGE_HAS_CHANGES))
	{
		p->freeze_count--;
		return(status);
	}

	/* Get the values that are needed to refer to the object that
	 * we want to apply this page's values to
	 */
	path = edv_properties_list_get_s(
		properties_list,
		EDV_PROP_NAME_PATH
	);
	if(path == NULL)
	{
		p->freeze_count--;
		return(status);
	}

	/* Permissions */
	w = p->permissions_frame;
	if((w != NULL) ? GTK_WIDGET_SENSITIVE(w) : FALSE)
	{
		EDVPermissionFlags permissions = 0x00000000;

		/* Get the current permission values */
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(p->ur_check))
			permissions |= EDV_PERMISSION_UR;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(p->uw_check))
			permissions |= EDV_PERMISSION_UW;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(p->ux_check))
			permissions |= EDV_PERMISSION_UX;

		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(p->gr_check))
			permissions |= EDV_PERMISSION_GR;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(p->gw_check))
			permissions |= EDV_PERMISSION_GW;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(p->gx_check))
			permissions |= EDV_PERMISSION_GX;

		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(p->or_check))
			permissions |= EDV_PERMISSION_OR;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(p->ow_check))
			permissions |= EDV_PERMISSION_OW;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(p->ox_check))
			permissions |= EDV_PERMISSION_OX;

		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(p->suid_check))
			permissions |= EDV_PERMISSION_SETUID;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(p->sgid_check))
			permissions |= EDV_PERMISSION_SETGID;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(p->sticky_check))
			permissions |= EDV_PERMISSION_STICKY;

		/* Do we need to change the values? */
		if((EDVPermissionFlags)edv_properties_list_get_i(
			properties_list,
			EDV_PROP_NAME_PERMISSIONS
		) != permissions)
		{
			const gchar *error_msg;
			GList	*paths_list,
				*modified_paths_list;

			/* Create the paths list */
			paths_list = NULL;
			paths_list = g_list_append(
				paths_list,
				g_strdup(path)
			);

			/* Set the new permissions */
			if(edv_vfs_object_op_chmod(
				core,
				paths_list,
				permissions,
				&modified_paths_list,
				toplevel,
				FALSE,		/* Do not show progress */
				TRUE,		/* Interactive */
				&yes_to_all,
				FALSE,		/* Not recursive */
				TRUE		/* Archive */
			) == 0)
				status = TRUE;

			/* Unmap the progress dialog just in case */
			ProgressDialogBreakQuery(FALSE);
			ProgressDialogSetTransientFor(NULL);

			/* Check for errors */
			error_msg = edv_vfs_object_op_get_error(core);
			if(!STRISEMPTY(error_msg))
			{
				/* Report the error */
				edv_play_sound_error(core);
				edv_message_error(
					"Change Permissions Error",
					error_msg,
					NULL,
					toplevel
				);
			}

			/* Delete the modified paths list */
			if(modified_paths_list != NULL)
			{
				g_list_foreach(modified_paths_list, (GFunc)g_free, NULL);
				g_list_free(modified_paths_list);
			}

			if(paths_list != NULL)
			{
				g_list_foreach(paths_list, (GFunc)g_free, NULL);
				g_list_free(paths_list);
			}
		}
	}

	/* Owner & Group */
	w = p->ownership_frame;
	if((w != NULL) ? GTK_WIDGET_SENSITIVE(w) : FALSE)
	{
		gint	owner_id = 0,
			group_id = 0;
		const gchar *error_msg;

		/* Get the owner id to be changed to */
		w = p->owner_entry;
		if(w != NULL)
			owner_id = edv_uid_name_to_uid(
				core->uids_list,
				gtk_entry_get_text(GTK_ENTRY(w))
			);

		/* Get the group id to be changed to */
		w = p->group_entry;
		if(w != NULL)
			group_id = edv_gid_name_to_gid(
				core->gids_list,
				gtk_entry_get_text(GTK_ENTRY(w))
			);

		/* Do we need to change the values? */
		if((edv_properties_list_get_i(properties_list, EDV_PROP_NAME_OWNER_ID) != owner_id) ||
		   (edv_properties_list_get_i(properties_list, EDV_PROP_NAME_GROUP_ID) != group_id)
		)
		{
			GList	*paths_list,
				*modified_paths_list;

			/* Create the paths list */
			paths_list = NULL;
			paths_list = g_list_append(
				paths_list,
				g_strdup(path)
			);

			/* Set the new owner and group */
			if(edv_vfs_object_op_chown(
				core,
				paths_list,
				owner_id, group_id,
				&modified_paths_list,
				toplevel,
				FALSE,		/* Do not show progress */
				TRUE,		/* Interactive */
				&yes_to_all,
				FALSE,		/* Not recursive */
				TRUE		/* Archive */
			) == 0)
				status = TRUE;

			/* Unmap the progress dialog */
			ProgressDialogBreakQuery(FALSE);
			ProgressDialogSetTransientFor(NULL);

			/* Check for errors */
			error_msg = edv_vfs_object_op_get_error(core);
			if(!STRISEMPTY(error_msg))
			{
				/* Report the error */
				edv_play_sound_error(core);
				edv_message_error(
					"Change Ownership Error",
					error_msg,
					NULL,
					toplevel
				);
			}

			/* Delete the modified paths list */
			if(modified_paths_list != NULL)
			{
				g_list_foreach(modified_paths_list, (GFunc)g_free, NULL);
				g_list_free(modified_paths_list);
			}

			if(paths_list != NULL)
			{
				g_list_foreach(paths_list, (GFunc)g_free, NULL);
				g_list_free(paths_list);
			}
		}
	}

	/* Remove our has changes marker */
	p->flags &= ~EDV_GENERAL_PROP_PAGE_HAS_CHANGES;

	p->freeze_count--;

	return(status);
}

/*
 *	Apply recycled object callback.
 */
gboolean edv_general_prop_page_apply_recycle_bin_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	GList *properties_list,
	gpointer data
)
{
	gboolean	status = FALSE,
			yes_to_all = FALSE;
	gulong index;
	GtkWidget	*w,
			*toplevel = edv_prop_page_get_toplevel(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);
	EDVGeneralPropPage *p = EDV_GENERAL_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return(status);

	p->freeze_count++;

	/* Do not apply if we did not make any changes */
	if(!(p->flags & EDV_GENERAL_PROP_PAGE_HAS_CHANGES))
	{
		p->freeze_count--;
		return(status);
	}

	/* Get the values that are needed to refer to the object that
	 * we want to apply this page's values to
	 */
	index = edv_properties_list_get_ul(
		properties_list,
		EDV_PROP_NAME_INDEX
	);
	if(index == 0l)
	{
		p->freeze_count--;
		return(status);
	}

	/* Permissions */
	w = p->permissions_frame;
	if((w != NULL) ? GTK_WIDGET_SENSITIVE(w) : FALSE)
	{
		EDVPermissionFlags permissions = 0x00000000;

		/* Get the permissions to change to */
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(p->ur_check))
			permissions |= EDV_PERMISSION_UR;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(p->uw_check))
			permissions |= EDV_PERMISSION_UW;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(p->ux_check))
			permissions |= EDV_PERMISSION_UX;

		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(p->gr_check))
			permissions |= EDV_PERMISSION_GR;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(p->gw_check))
			permissions |= EDV_PERMISSION_GW;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(p->gx_check))
			permissions |= EDV_PERMISSION_GX;

		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(p->or_check))
			permissions |= EDV_PERMISSION_OR;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(p->ow_check))
			permissions |= EDV_PERMISSION_OW;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(p->ox_check))
			permissions |= EDV_PERMISSION_OX;

		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(p->suid_check))
			permissions |= EDV_PERMISSION_SETUID;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(p->sgid_check))
			permissions |= EDV_PERMISSION_SETGID;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(p->sticky_check))
			permissions |= EDV_PERMISSION_STICKY;

		/* Do we need to change the values? */
		if((EDVPermissionFlags)edv_properties_list_get_i(
			properties_list,
			EDV_PROP_NAME_PERMISSIONS
		) != permissions)
		{
			const gchar *error_msg;
			GList	*indicies_list = NULL,
				*modified_indicies_list;

			indicies_list = g_list_append(
				indicies_list,
				(gpointer)index
			);

			/* Set the new permissions */
			if(edv_recycled_object_op_chmod(
				core,
				indicies_list,
				permissions,
				&modified_indicies_list,
				toplevel,
				FALSE,		/* Do not show progress */
				TRUE,		/* Interactive */
				&yes_to_all
			) == 0)
				status = TRUE;

			/* Unmap the progress dialog */
			ProgressDialogBreakQuery(FALSE);
			ProgressDialogSetTransientFor(NULL);

			/* Check for errors */
			error_msg = edv_recycled_object_op_get_error(core);
			if(!STRISEMPTY(error_msg))
			{
				/* Report the error */
				edv_play_sound_error(core);
				edv_message_error(
					"Change Permissions Error",
					error_msg,
					NULL,
					toplevel
				);
			}

			g_list_free(modified_indicies_list);
			g_list_free(indicies_list);
		}
	}

	/* Owner & Group */
	w = p->ownership_frame;
	if((w != NULL) ? GTK_WIDGET_SENSITIVE(w) : FALSE)
	{
		gint	owner_id = 0,
			group_id = 0;

		/* Get the user id to be changed to */
		w = p->owner_entry;
		if(w != NULL)
			owner_id = edv_uid_name_to_uid(
				core->uids_list,
				gtk_entry_get_text(GTK_ENTRY(w))
			);

		/* Get the group id to be changed to */
		w = p->group_entry;
		if(w != NULL)
			group_id = edv_gid_name_to_gid(
				core->gids_list,
				gtk_entry_get_text(GTK_ENTRY(w))
			);

		/* Do we need to change the values? */
		if((edv_properties_list_get_i(properties_list, EDV_PROP_NAME_OWNER_ID) != owner_id) ||
		   (edv_properties_list_get_i(properties_list, EDV_PROP_NAME_GROUP_ID) != group_id)
		)
		{
			const gchar *error_msg;
			GList	*indicies_list = NULL,
					*modified_indicies_list;

			indicies_list = g_list_append(
				indicies_list,
				(gpointer)index
			);

			/* Set the new owner and group */
			if(edv_recycled_object_op_chown(
				core,
				indicies_list,
				owner_id, group_id,
				&modified_indicies_list,
				toplevel,
				FALSE,		/* Do not show progress */
				TRUE,		/* Interactive */
				&yes_to_all
			) == 0)
				status = TRUE;

			/* Unmap the progress dialog */
			ProgressDialogBreakQuery(FALSE);
			ProgressDialogSetTransientFor(NULL);

			/* Check for errors */
			error_msg = edv_recycled_object_op_get_error(core);
			if(!STRISEMPTY(error_msg))
			{
				/* Report the error */
				edv_play_sound_error(core);
				edv_message_error(
					"Change Ownership Error",
					error_msg,
					NULL,
					toplevel
				);
			}

			g_list_free(modified_indicies_list);
			g_list_free(indicies_list);
		}
	}

	/* Remove our has changes marker */
	p->flags &= ~EDV_GENERAL_PROP_PAGE_HAS_CHANGES;

	p->freeze_count--;

	return(status);
}


/*
 *	Destroy callback.
 */
void edv_general_prop_page_destroy_cb(
	EDVPropPageContext *ctx,
	gpointer data
)
{
	EDVGeneralPropPage *p = EDV_GENERAL_PROP_PAGE(data);

	p->freeze_count++;

	EDVMimeTypeEditDlgDelete(p->mime_type_edit_dlg);

	p->freeze_count--;

	g_free(p);
}


/*
 *	Any GtkWidget "changed" signal callback.
 */
static void edv_general_prop_page_changed_cb(GtkWidget *widget, gpointer data)
{
	EDVGeneralPropPage *p = EDV_GENERAL_PROP_PAGE(data);
	EDVPropPageContext *ctx = p->ctx;

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;			/* Ignore our own update */

	/* Ignore if we have already made changes */
	if(p->flags & EDV_GENERAL_PROP_PAGE_HAS_CHANGES)
	{
		p->freeze_count--;
		return;
	}

	/* Mark that one of our GtkWidgets has made changes */
	p->flags |= EDV_GENERAL_PROP_PAGE_HAS_CHANGES;

	/* Notify the EDVPropDlg that changes have been made and
	 * emit an update (which we will ignore)
	 */
	edv_prop_page_set_has_changes(
		ctx,
		TRUE
	);

	p->freeze_count--;
}

/*
 *	Maps the popup list for the owner entry.
 */
static void edv_general_prop_page_owner_map_pulist_cb(GtkWidget *widget, gpointer data)
{
	gint		nitems,
			nitems_visible;
	gchar *old_value;
	const gchar *value;
	GtkWidget *pulist;
	GtkEntry *entry;
	EDVGeneralPropPage *p = EDV_GENERAL_PROP_PAGE(data);
	EDVPropPageContext *ctx = p->ctx;
	EDVCore *core = edv_prop_page_get_core(ctx);

	if(p->freeze_count > 0)
		return;

	entry = GTK_ENTRY(p->owner_entry);
	pulist = core->users_pulist;
	if(PUListIsQuery(pulist))
		return;

	p->freeze_count++;

	old_value = STRDUP(gtk_entry_get_text(entry));

	nitems = PUListGetTotalItems(pulist);
	nitems_visible = MIN(15, nitems);

	/* Block input and get value */
	value = PUListMapQuery(
		pulist,				/* Popup List */
		old_value,			/* Initial Value */
		nitems_visible,
		PULIST_RELATIVE_BELOW,		/* Popup Relativity */
		GTK_WIDGET(entry),		/* Relative GtkWidget */
		widget				/* Map Trigger Widget */
	);
	if(value != NULL)
	{
		/* Set new value if it is different from the old value  */
		if((old_value != NULL) ?
			strcmp(
				(const char *)old_value,
				(const char *)value
			) : TRUE
		)
		{
			gtk_entry_set_text(entry, value);

			/* Mark that we have made changes */
		        p->flags |= EDV_GENERAL_PROP_PAGE_HAS_CHANGES;
		        edv_prop_page_set_has_changes(
		                ctx,
		                TRUE
		        );
		}
	}

	g_free(old_value);

	p->freeze_count--;
}

/*
 *	Maps the popup list for the group entry.
 */
static void edv_general_prop_page_group_map_pulist_cb(GtkWidget *widget, gpointer data)
{
	gint		nitems,
			nitems_visible;
	gchar *old_value;
	const gchar *value;
	GtkWidget *pulist;
	GtkEntry *entry;
	EDVGeneralPropPage *p = EDV_GENERAL_PROP_PAGE(data);
	EDVPropPageContext *ctx = p->ctx;
	EDVCore *core = edv_prop_page_get_core(ctx);

	if(p->freeze_count > 0)
		return;

	entry = GTK_ENTRY(p->group_entry);
	pulist = core->groups_pulist;
	if(PUListIsQuery(pulist))
		return;

	p->freeze_count++;

	old_value = STRDUP(gtk_entry_get_text(entry));

	nitems = PUListGetTotalItems(pulist);
	nitems_visible = MIN(15, nitems);

	/* Block input and get value */
	value = PUListMapQuery(
		pulist,				/* Popup List */
		old_value,			/* Initial Value */
		nitems_visible,
		PULIST_RELATIVE_BELOW,		/* Popup Relativity */
		GTK_WIDGET(entry),		/* Relative GtkWidget */
		widget				/* Map Trigger GtkWidget */
	);
	if(value != NULL)
	{
		/* Set new value if it is different from the old value  */
		if((old_value != NULL) ?
			strcmp(
				(const char *)old_value,
				(const char *)value
			) : TRUE
		)
		{
			gtk_entry_set_text(entry, value);

			/* Mark that we have made changes */
		        p->flags |= EDV_GENERAL_PROP_PAGE_HAS_CHANGES;
		        edv_prop_page_set_has_changes(
		                ctx,
		                TRUE
		        );
		}
	}

	g_free(old_value);

	p->freeze_count--;
}


/*
 *	Add MIME Type callback.
 */
static void edv_general_prop_page_add_mime_type_cb(GtkWidget *widget, gpointer data)
{
	EDVGeneralPropPage *p = EDV_GENERAL_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	edv_general_prop_page_add_mime_type(p);
}

/*
 *	Edit MIME Type callback.
 */
static void edv_general_prop_page_edit_mime_type_cb(GtkWidget *widget, gpointer data)
{
	EDVGeneralPropPage *p = EDV_GENERAL_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	edv_general_prop_page_edit_mime_type(p);
}


/*
 *	Date touch callback.
 */
static void edv_general_prop_page_touch_cb(GtkWidget *widget, gpointer data)
{
	EDVGeneralPropPage *p = EDV_GENERAL_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	edv_general_prop_page_touch(p);
}

/*
 *	Date touch specific.
 */
static void edv_general_prop_page_touch_specific_cb(GtkWidget *widget, gpointer data)
{
	EDVGeneralPropPage *p = EDV_GENERAL_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	edv_general_prop_page_touch_specific(p);
}


/*
 *	Add MIME Type.
 */
static void edv_general_prop_page_add_mime_type(EDVGeneralPropPage *p)
{
	gint mt_num;
	const gchar	*path,
			*ext;
	EDVMIMEType *m;
	edv_mime_type_edit_dlg_struct *mime_type_edit_dlg;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);

	edv_prop_page_set_busy(ctx, TRUE);
	p->freeze_count++;

	path = edv_properties_list_get_s(
		properties_list,
		EDV_PROP_NAME_PATH
	);
	if(path == NULL)
		path = edv_properties_list_get_s(
			properties_list,
			EDV_PROP_NAME_NAME
		);

	/* Create a new MIME Type */
	m = edv_mime_type_new_values();
	if(m == NULL)
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	/* Set the new MIME Type's value as the mask for the current
	 * object's extension
	 */
	ext = edv_path_get_extension(path);
	if(ext != NULL)
	{
		m->mt_class = EDV_MIME_TYPE_CLASS_FORMAT;
		m->value = g_strconcat(
			"*",
			ext,
			NULL
		);
		m->type = g_strconcat(
			"misc/x-",
			ext + 1,
			NULL
		);
		m->description = g_strconcat(
			ext + 1,
			" file format",
			NULL
		);
	}
	else
	{
		m->mt_class = EDV_MIME_TYPE_CLASS_UNIQUE;
		m->value = STRDUP(path);
		m->type = STRDUP("misc/untitled");
		m->description = STRDUP("New MIME Type");
	}


	/* Append the new MIME Type to the list */
	mt_num = g_list_length(core->mime_types_list);
	core->mime_types_list = g_list_append(
		core->mime_types_list,
		m
	);

	/* Notify about this MIME Type being added */
	edv_emit_mime_type_added(
		core,
		mt_num,
		m
	);

	/* Create the MIME Type Edit Dialog as needed */
	mime_type_edit_dlg = p->mime_type_edit_dlg;
	if(mime_type_edit_dlg == NULL)
		p->mime_type_edit_dlg = mime_type_edit_dlg = EDVMimeTypeEditDlgNew(core);
	if(mime_type_edit_dlg != NULL)
	{
		edv_center_window_to_window(
			toplevel,
			mime_type_edit_dlg->toplevel
		);
		EDVMimeTypeEditDlgMap(mime_type_edit_dlg);
		EDVMimeTypeEditDlgGetValues(mime_type_edit_dlg, mt_num);
		EDVMimeTypeEditDlgResetHasChanges(mime_type_edit_dlg, FALSE);
	}
	else
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	p->freeze_count--;
	edv_prop_page_set_busy(ctx, FALSE);
}

/*
 *	Edit MIME Type.
 */
static void edv_general_prop_page_edit_mime_type(EDVGeneralPropPage *p)
{
	gint mt_num;
	const gchar *type;
	EDVMIMEType *m;
	edv_mime_type_edit_dlg_struct *mime_type_edit_dlg;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);

	edv_prop_page_set_busy(ctx, TRUE);
	p->freeze_count++;

	/* Get the current MIME Type's type */
	type = edv_prop_page_get_mime_type_type(ctx);
	if(STRISEMPTY(type))
	{
		/* No MIME Type defined, so add a new one */
		edv_general_prop_page_add_mime_type(p);

		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	/* Get the current MIME Type */
	m = edv_mime_types_list_match_type(
		core->mime_types_list,
		&mt_num,
		type,
		FALSE				/* Not case sensitive */
	);
	if(m == NULL)
	{
		/* No MIME Type defined, so add a new one */
		edv_general_prop_page_add_mime_type(p);

		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	/* If the MIME Type is a system object then it suggests it
	 * does not have a MIME Type defined yet
	 */
	if(m->mt_class == EDV_MIME_TYPE_CLASS_SYSTEM)
	{
		/* No MIME Type defined, so add a new one */
		edv_general_prop_page_add_mime_type(p);

		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	/* Create the MIME Type Edit Dialog as needed */
	mime_type_edit_dlg = p->mime_type_edit_dlg;
	if(mime_type_edit_dlg == NULL)
		p->mime_type_edit_dlg = mime_type_edit_dlg = EDVMimeTypeEditDlgNew(core);
	if(mime_type_edit_dlg != NULL)
	{
		edv_center_window_to_window(
			toplevel,
			mime_type_edit_dlg->toplevel
		);
		EDVMimeTypeEditDlgMap(mime_type_edit_dlg);
		EDVMimeTypeEditDlgGetValues(mime_type_edit_dlg, mt_num);
		EDVMimeTypeEditDlgResetHasChanges(mime_type_edit_dlg, FALSE);
	}
	else
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	p->freeze_count--;
	edv_prop_page_set_busy(ctx, FALSE);
}


/*
 *	Sets the object's time stamps to the current time.
 */
static void edv_general_prop_page_touch(EDVGeneralPropPage *p)
{
	gboolean yes_to_all = FALSE;
	gulong index;
	const gchar *path;
	const gulong cur_time = edv_time();
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	const EDVLocationType location_type = edv_prop_page_get_location_type(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);
	const EDVObjectType type = (EDVObjectType)edv_properties_list_get_i(
		properties_list,
		EDV_PROP_NAME_TYPE
	);
	EDVCore *core = edv_prop_page_get_core(ctx);

	edv_prop_page_set_busy(ctx, TRUE);
	p->freeze_count++;

	/* Check if the master write protect is on */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	/* Handle by the location type */
	switch(location_type)
	{
	    case EDV_LOCATION_TYPE_VFS:
		path = edv_properties_list_get_s(
			properties_list,
			EDV_PROP_NAME_PATH
		);
		if(!STRISEMPTY(path))
		{
			gint status;
			const gchar	*error_msg;
			GList		*paths_list,
					*modified_paths_list;

			/* If this is a link then warn about changing
			 * the time stamps of links
			 */
			if(type == EDV_OBJECT_TYPE_LINK)
			{
				gint response;
				edv_play_sound_warning(core);
				CDialogSetTransientFor(toplevel);
				response = CDialogGetResponse(
"Changing Time Stamps Warning",
"Changing the time stamps on a link will effectively\n\
change the time stamps of its target object.\n\
\n\
Are you sure you want to change the time stamps\n\
on the target object?",
"Links do not have time stamps, instead, their time\n\
stamps are determined by the time stamps of the\n\
target object. So changing the time stamps on a link\n\
will effectively change the time stamps of the target\n\
object.",
					CDIALOG_ICON_WARNING,
					CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO |
						CDIALOG_BTNFLAG_HELP,
					CDIALOG_BTNFLAG_YES
				);
				CDialogSetTransientFor(NULL);
				if(response != CDIALOG_RESPONSE_YES)
				{
					p->freeze_count--;
					edv_prop_page_set_busy(ctx, FALSE);
					return;
				}
			}

			/* Create the paths list */
			paths_list = NULL;
			paths_list = g_list_append(paths_list, STRDUP(path));

			/* Set the time stamps to the current time */
			status = edv_vfs_object_op_chtime(
				core,
				paths_list,
				cur_time,	/* Access time */
				cur_time,	/* Modify time */
				&modified_paths_list,
				toplevel,
				FALSE,		/* Do not show progress */
				TRUE,		/* Interactive */
				&yes_to_all,
				FALSE,		/* Not recursive */
				TRUE		/* Archive */
			);

			/* Unmap the progress dialog */
			ProgressDialogBreakQuery(FALSE);
			ProgressDialogSetTransientFor(NULL);

			/* Get the error message (if any) */
			error_msg = edv_vfs_object_op_get_error(core);
			if(!STRISEMPTY(error_msg))
			{
				/* Print the error message */
				edv_play_sound_error(core);
				edv_message_error(
					"Change Time Stamps Error",
					error_msg,
					NULL,
					toplevel
				);
			}

			/* Notify about the object being modified */
			if(modified_paths_list != NULL)
			{
				const gchar *path;
				GList *glist;
				EDVVFSObject *obj;

				for(glist = modified_paths_list;
				    glist != NULL;
				    glist = g_list_next(glist)
				)
				{
					path = (const gchar *)glist->data;
					if(path == NULL)
						continue;

					obj = edv_vfs_object_lstat(path);
					if(obj != NULL)
					{
						edv_emit_vfs_object_modified(
							core,
							path,
							path,
							obj
						);
						edv_vfs_object_delete(obj);
					}
				}

				/* Since this call freezes our page,
				 * we need explicitly update the
				 * values displayed on our page
				 */
				edv_general_prop_page_update_widgets(p);

				/* Delete the modified paths list */
				g_list_foreach(modified_paths_list, (GFunc)g_free, NULL);
				g_list_free(modified_paths_list);
			}

			/* Play the "completed" sound on success */
			if(status == 0)
				edv_play_sound_completed(core);

			/* Delete the paths list */
			if(paths_list != NULL)
			{
				g_list_foreach(paths_list, (GFunc)g_free, NULL);
				g_list_free(paths_list);
			}
		}
		break;

	    case EDV_LOCATION_TYPE_RECYCLE_BIN:
		index = edv_properties_list_get_ul(
			properties_list,
			EDV_PROP_NAME_INDEX
		);
		if(index != 0l)
		{
			gint status;
			const gchar *error_msg;
			GList	*indicies_list = NULL,
				*modified_indicies_list;

			/* Create the indicies list */
			indicies_list = g_list_append(
				indicies_list,
				(gpointer)index
			);

			/* Set the time stamps to the current time */
			status = edv_recycled_object_op_chtime(
				core,
				indicies_list,
				cur_time,	/* Access time */
				cur_time,	/* Modify time */
				cur_time,	/* Delete time */
				&modified_indicies_list,
				toplevel,
				FALSE,		/* Do not show progress */
				TRUE,		/* Interactive */
				&yes_to_all
			);

			/* Unmap the progress dialog */
			ProgressDialogBreakQuery(FALSE);
			ProgressDialogSetTransientFor(NULL);

			/* Get the error message (if any) */
			error_msg = edv_recycled_object_op_get_error(core);
			if(!STRISEMPTY(error_msg))
			{
				/* Print the error message */
				edv_play_sound_error(core);
				edv_message_error(
					"Change Time Stamps Error",
					error_msg,
					NULL,
					toplevel
				);
			}

			/* Notify about the recycled object being modified */
			if(modified_indicies_list != NULL)
			{
				gulong index;
				GList *glist;

				for(glist = modified_indicies_list;
				    glist != NULL;
				    glist = g_list_next(glist)
				)
				{
					index = (gulong)glist->data;
					if(index == 0l)
						continue;

					edv_emit_recycled_object_modified(
						core,
						index
					);
				}

				/* Since this call freezes our page,
				 * we need explicitly update the
				 * values displayed on our page
				 */
				edv_general_prop_page_update_widgets(p);

				/* Delete the modified indicies list */
				g_list_free(modified_indicies_list);
			}

			/* Play the "completed" sound on success */
			if(status == 0)
				edv_play_sound_completed(core);

			/* Delete the indicies list */
			g_list_free(indicies_list);
		}
		break;

	    case EDV_LOCATION_TYPE_ARCHIVE:
		edv_play_sound_warning(core);
		edv_message_warning(
"Changing Time Stamps Warning",
"The time stamps of an object in an archive can not be modified.",
			NULL,
			toplevel
		);
		break;
	}

	p->freeze_count--;
	edv_prop_page_set_busy(ctx, FALSE);
}

/*
 *	Queries the user for the new time stamps to set on the object
 *	and sets the object's time stamps.
 */
static void edv_general_prop_page_touch_specific(EDVGeneralPropPage *p)
{
	gulong index;
	const gchar *path;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	const EDVLocationType location_type = edv_prop_page_get_location_type(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);
	CfgList *cfg_list = edv_prop_page_get_cfg_list(ctx);

	edv_prop_page_set_busy(ctx, TRUE);
	p->freeze_count++;

	/* Check if the master write protect is on */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		p->freeze_count--;
		edv_prop_page_set_busy(ctx, FALSE);
		return;
	}

	/* Handle by the location type */
	switch(location_type)
	{
	    case EDV_LOCATION_TYPE_VFS:
		path = edv_properties_list_get_s(
			properties_list,
			EDV_PROP_NAME_PATH
		);
		if(!STRISEMPTY(path))
		{
			GList *objs_list = NULL;

			/* Create the objects list */
			EDVVFSObject *obj = edv_vfs_object_lstat(path);
			if(obj != NULL)
				objs_list = g_list_append(
					objs_list,
					obj
				);

			/* Map the object operations dialog to change time stamps */
			EDVObjOpDlgMapValues(
				edv_get_object_operations_dialog(core),
				EDV_OBJ_OP_DLG_OP_CHTIME,
				EDV_LOCATION_TYPE_VFS,
				objs_list,
				NULL,
				toplevel
			);

			/* Delete the objects list */
			if(objs_list != NULL)
			{
				g_list_foreach(objs_list, (GFunc)edv_vfs_object_delete, NULL);
				g_list_free(objs_list);
			}
		}
		break;

	    case EDV_LOCATION_TYPE_RECYCLE_BIN:
		index = edv_properties_list_get_ul(
			properties_list,
			EDV_PROP_NAME_INDEX
		);
		if(index != 0l)
		{
			GList *objs_list = NULL;

			/* Create the recycled objects list */
			EDVRecycledObject *obj = edv_recycled_object_stat(
				EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX),
				index
			);
			if(obj != NULL)
				objs_list = g_list_append(
					objs_list,
					obj
				);

			/* Map the object operations dialog to change time stamps */
			EDVObjOpDlgMapValues(
				edv_get_object_operations_dialog(core),
				EDV_OBJ_OP_DLG_OP_CHTIME,
				EDV_LOCATION_TYPE_RECYCLE_BIN,
				objs_list,
				NULL,
				toplevel
			);

			/* Delete the recycled objects list */
			if(objs_list != NULL)
			{
				g_list_foreach(
					objs_list,
					(GFunc)edv_recycled_object_delete,
					NULL
				);
				g_list_free(objs_list);
			}
		}
		break;

	    case EDV_LOCATION_TYPE_ARCHIVE:
		edv_play_sound_warning(core);
		edv_message_warning(
"Changing Time Stamps Warning",
"The time stamps of an object in an archive can not be modified.",
			NULL,
			toplevel
		);
		break;
	}

	p->freeze_count--;
	edv_prop_page_set_busy(ctx, FALSE);
}


/*
 *	Updates the values displayed on the GtkWidgets.
 */
static void edv_general_prop_page_update_widgets(EDVGeneralPropPage *p)
{
	gboolean	write_protect,
			read_only,
			sensitive;
	gint mt_num;
	const gchar	*date_format,
			*name,
			*path,
			*mime_type;
	GtkWidget *w;
	EDVDateRelativity date_relativity;
	EDVPermissionFlags permissions;
	EDVMIMEType *m;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	const EDVLocationType location_type = edv_prop_page_get_location_type(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);
	const EDVObjectType type = (EDVObjectType)edv_properties_list_get_i(
		properties_list,
		EDV_PROP_NAME_TYPE
	);
	EDVCore *core = edv_prop_page_get_core(ctx);
	CfgList *cfg_list = edv_prop_page_get_cfg_list(ctx);
	const gboolean this_page_has_changes = (p->flags & EDV_GENERAL_PROP_PAGE_HAS_CHANGES) ? TRUE : FALSE;

	p->freeze_count++;

	date_relativity = (EDVDateRelativity)EDV_GET_I(
		EDV_CFG_PARM_DATE_RELATIVITY
	);
	date_format = EDV_GET_S(EDV_CFG_PARM_DATE_FORMAT);
	write_protect = EDV_GET_B(EDV_CFG_PARM_WRITE_PROTECT);

	/* Determine if the object is read only */
	read_only = TRUE;
	switch(location_type)
	{
	    case EDV_LOCATION_TYPE_VFS:
		read_only = write_protect;
		break;
	    case EDV_LOCATION_TYPE_RECYCLE_BIN:
		read_only = write_protect;
		break;
	    case EDV_LOCATION_TYPE_ARCHIVE:
		read_only = TRUE;
		break;
	}

	/* Get the name */
	name = edv_properties_list_get_s(
		properties_list,
		EDV_PROP_NAME_NAME
	);

	/* Get the path */
	path = edv_properties_list_get_s(
		properties_list,
		EDV_PROP_NAME_PATH
	);
	if(path == NULL)
		path = name;

	/* Get the permissions */
	permissions = (EDVPermissionFlags)edv_properties_list_get_i(
		properties_list,
		EDV_PROP_NAME_PERMISSIONS
	);

	/* Get the MIME Type */
	mime_type = edv_prop_page_get_mime_type_type(ctx);
	if(!STRISEMPTY(mime_type))
	{
		m = edv_mime_types_list_match_type(
			core->mime_types_list,
			&mt_num,
			mime_type,
			FALSE			/* Not case sensitive */
		);
	}
	else
	{
		m = NULL;
		mt_num = -1;
	}


	/* Icon */
	w = p->icon_pm;
	if(w != NULL)
	{
		EDVPixmap	*icon = NULL,
				*icon_hidden = NULL;
		(void)edv_match_object_icon(
			core->devices_list,
			core->mime_types_list,
			type,
			path,
			TRUE,			/* Assume link valid */
			permissions,
			EDV_ICON_SIZE_32,
			&icon,
			NULL,
			NULL,
			&icon_hidden
		);
		if(edv_path_is_hidden(path))
		{
			if(edv_pixmap_is_loaded(icon_hidden))
			{
				(void)edv_pixmap_unref(icon);
				icon = edv_pixmap_ref(icon_hidden);
			}
		}
		if(edv_pixmap_is_loaded(icon))
		{
			edv_pixmap_set_gtk_pixmap(icon, w);
			gtk_widget_show(w);
		}
		else
		{
			gtk_widget_hide(w);
		}
		(void)edv_pixmap_unref(icon);
		(void)edv_pixmap_unref(icon_hidden);
	}

	/* Name */
	w = p->name_label;
	if(w != NULL)
	{
		gchar *s;
		if(name != NULL)
			s = edv_path_shorten(name, 30);
		else
			s = g_strdup(EDV_PROP_DLG_VALUE_NOT_AVAILABLE_STR);
		gtk_label_set_text(GTK_LABEL(w), (s != NULL) ? s : "");
		g_free(s);
	}

	/* Add and Edit MIME Type Buttons */
	w = p->add_mime_type_btn;
	if(w != NULL)
	{
		if(m != NULL)
		{
			if(m->mt_class == EDV_MIME_TYPE_CLASS_SYSTEM)
				gtk_widget_show(w);
			else
				gtk_widget_hide(w);
		}
		else
			gtk_widget_show(w);
	}
	w = p->edit_mime_type_btn;
	if(w != NULL)
	{
		if(m != NULL)
		{
			if(m->mt_class == EDV_MIME_TYPE_CLASS_SYSTEM)
				gtk_widget_hide(w);
			else
				gtk_widget_show(w);
		}
		else
			gtk_widget_hide(w);
	}

	/* Location */
	w = p->location_label;
	if(w != NULL)
	{
		gchar *location = NULL;

		switch(location_type)
		{
		    case EDV_LOCATION_TYPE_VFS:
			if(!STRISEMPTY(path))
			{
				location = g_dirname(path);
				if(location != NULL)
				{
					gchar *s = edv_path_shorten(location, 45);
					g_free(location);
					location = s;
				}
				else
				    location = STRDUP(EDV_PROP_DLG_VALUE_NOT_AVAILABLE_STR);
			}
			else
			{
				location = STRDUP(EDV_PROP_DLG_VALUE_NOT_AVAILABLE_STR);
			}
			break;
		    case EDV_LOCATION_TYPE_RECYCLE_BIN:
			location = g_strdup_printf(
				"#%ld",
				edv_properties_list_get_ul(
					properties_list,
					EDV_PROP_NAME_INDEX
				)
			);
			break;
		    case EDV_LOCATION_TYPE_ARCHIVE:
			if(!STRISEMPTY(path))
			{
				const gchar *in_arch_path = path;
				gchar *s;

				if(g_path_is_absolute(in_arch_path))
					s = g_strdup(in_arch_path);
				else
					s = g_strconcat(
						".",
						G_DIR_SEPARATOR_S,
						in_arch_path,
						NULL
					);
				if(s != NULL)
				{
					location = g_dirname(s);
					g_free(s);
					if(location != NULL)
					{
						s = edv_path_shorten(location, 34);
						g_free(location);
						location = s;
					}
				}
				else
					location = STRDUP(EDV_PROP_DLG_VALUE_NOT_AVAILABLE_STR);
			}
			else
			{
				location = STRDUP(EDV_PROP_DLG_VALUE_NOT_AVAILABLE_STR);
			}
			break;
		}
		gtk_label_set_text(GTK_LABEL(w), (location != NULL) ? location : "");
		g_free(location);
	}

	/* Type */
	w = p->type_label;
	if(w != NULL)
	{
		gchar *s;
		const gchar *mime_type = edv_prop_page_get_mime_type_type(ctx);
		if(mime_type != NULL)
		{
			s = edv_path_shorten(mime_type, 20);
		}
		else
		{
			s = STRDUP(edv_object_type_to_object_name(type));
		}
		gtk_label_set_text(GTK_LABEL(w), (s != NULL) ? s : "");
		g_free(s);
	}

	/* Size */
	w = p->size_label;
	if(w != NULL)
	{
		gchar *s = STRDUP(edv_str_size_delim(edv_properties_list_get_ul(
			properties_list,
			EDV_PROP_NAME_SIZE
		)));
		gtk_label_set_text(GTK_LABEL(w), (s != NULL) ? s : "");
		g_free(s);
	}

	/* Touch GtkButton */
	sensitive = !read_only;
	GTK_WIDGET_SET_SENSITIVE(p->date_touch_btn, sensitive);

	/* Last Accessed */
	w = p->date_access_label;
	if(w != NULL)
	{
		gchar *s;
		const gulong t = edv_properties_list_get_ul(
			properties_list,
			EDV_PROP_NAME_ACCESS_TIME
		);
		if(t > 0l)
			s = edv_date_string_format(
				t,
				date_format, date_relativity
			);
		else
			s = g_strdup(EDV_PROP_DLG_TIME_STAMP_NOT_SET_STR);
		gtk_label_set_text(GTK_LABEL(w), (s != NULL) ? s : "");
		g_free(s);
	}

	/* Last Modified */
	w = p->date_modify_label;
	if(w != NULL)
	{
		gchar *s;
		const gulong t = edv_properties_list_get_ul(
			properties_list,
			EDV_PROP_NAME_MODIFY_TIME
		);
		if(t > 0l)
			s = edv_date_string_format(
				t,
				date_format, date_relativity
			);
		else
			s = g_strdup(EDV_PROP_DLG_TIME_STAMP_NOT_SET_STR);
		gtk_label_set_text(GTK_LABEL(w), (s != NULL) ? s : "");
		g_free(s);
	}

	/* Last Changed */
	w = p->date_change_label;
	if(w != NULL)
	{
		gchar *s;
		const gulong t = edv_properties_list_get_ul(
			properties_list,
			EDV_PROP_NAME_CHANGE_TIME
		);
		if(t > 0l)
			s = edv_date_string_format(
				t,
				date_format, date_relativity
			);
		else
			s = g_strdup(EDV_PROP_DLG_TIME_STAMP_NOT_SET_STR);
		gtk_label_set_text(GTK_LABEL(w), (s != NULL) ? s : "");
		g_free(s);
	}

	/* Deleted On */
	w = p->date_deleted_label;
	if(w != NULL)
	{
		if(location_type == EDV_LOCATION_TYPE_RECYCLE_BIN)
		{
			gchar *s;
			const gulong t = edv_properties_list_get_ul(
				properties_list,
				EDV_PROP_NAME_DELETED_TIME
			);
			if(t > 0l)
				s = edv_date_string_format(
					t,
					date_format, date_relativity
				);
			else
				s = g_strdup(EDV_PROP_DLG_TIME_STAMP_NOT_SET_STR);
			gtk_label_set_text(GTK_LABEL(w), (s != NULL) ? s : "");
			g_free(s);
		}
	}

	/* Owner */
	w = p->owner_entry;
	if((w != NULL) && !this_page_has_changes)
	{
		gchar *s = STRDUP(edv_properties_list_get_s(
			properties_list,
			EDV_PROP_NAME_OWNER_NAME
		));
		if(s == NULL)
			s = edv_uid_uid_to_name(
				core->uids_list,
				edv_properties_list_get_i(
					properties_list,
					EDV_PROP_NAME_OWNER_ID
				),
				NULL
			);
		gtk_entry_set_text(GTK_ENTRY(w), (s != NULL) ? s : "");
		g_free(s);
	}

	/* Group */
	w = p->group_entry;
	if((w != NULL) && !this_page_has_changes)
	{
		gchar *s = STRDUP(edv_properties_list_get_s(
			properties_list,
			EDV_PROP_NAME_GROUP_NAME
		));
		if(s == NULL)
			s = edv_gid_gid_to_name(
				core->gids_list,
				edv_properties_list_get_i(
					properties_list,
					EDV_PROP_NAME_GROUP_ID
				),
				NULL
			);
		gtk_entry_set_text(GTK_ENTRY(w), (s != NULL) ? s : "");
		g_free(s);
	}

	/* Set the Permissions GtkFrame insensitive for object
	 * types that do not have permissions
	 */
	sensitive = (type == EDV_OBJECT_TYPE_LINK) ? FALSE : TRUE;
	GTK_WIDGET_SET_SENSITIVE(p->permissions_frame, sensitive);

	/* Permissions */
	w = p->ur_check;
	if((w != NULL) && !this_page_has_changes)
	{
		gboolean b;
		if(type == EDV_OBJECT_TYPE_LINK)
			b = FALSE;
		else
			b = (permissions & EDV_PERMISSION_UR) ? TRUE : FALSE;
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);
	}
	w = p->uw_check;
	if((w != NULL) && !this_page_has_changes)
	{
		gboolean b;
		if(type == EDV_OBJECT_TYPE_LINK)
			b = FALSE;
		else
			b = (permissions & EDV_PERMISSION_UW) ? TRUE : FALSE;
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);
	}
	w = p->ux_check;
	if((w != NULL) && !this_page_has_changes)
	{
		gboolean b;
		if(type == EDV_OBJECT_TYPE_LINK)
			b = FALSE;
		else
			b = (permissions & EDV_PERMISSION_UX) ? TRUE : FALSE;
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);
	}

	w = p->gr_check;
	if((w != NULL) && !this_page_has_changes)
	{
		gboolean b;
		if(type == EDV_OBJECT_TYPE_LINK)
			b = FALSE;
		else
			b = (permissions & EDV_PERMISSION_GR) ? TRUE : FALSE;
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);
	}
	w = p->gw_check;
	if((w != NULL) && !this_page_has_changes)
	{
		gboolean b;
		if(type == EDV_OBJECT_TYPE_LINK)
			b = FALSE;
		else
			b = (permissions & EDV_PERMISSION_GW) ? TRUE : FALSE;
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);
	}
	w = p->gx_check;
	if((w != NULL) && !this_page_has_changes)
	{
		gboolean b;
		if(type == EDV_OBJECT_TYPE_LINK)
			b = FALSE;
		else
			b = (permissions & EDV_PERMISSION_GX) ? TRUE : FALSE;
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);
	}

	w = p->or_check;
	if((w != NULL) && !this_page_has_changes)
	{
		gboolean b;
		if(type == EDV_OBJECT_TYPE_LINK)
			b = FALSE;
		else
			b = (permissions & EDV_PERMISSION_OR) ? TRUE : FALSE;
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);
	}
	w = p->ow_check;
	if((w != NULL) && !this_page_has_changes)
	{
		gboolean b;
		if(type == EDV_OBJECT_TYPE_LINK)
			b = FALSE;
		else
			b = (permissions & EDV_PERMISSION_OW) ? TRUE : FALSE;
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);
	}
	w = p->ox_check;
	if((w != NULL) && !this_page_has_changes)
	{
		gboolean b;
		if(type == EDV_OBJECT_TYPE_LINK)
			b = FALSE;
		else
			b = (permissions & EDV_PERMISSION_OX) ? TRUE : FALSE;
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);
	}

	/* SetUID, SetGID, and Sticky */
	w = p->suid_check;
	if((w != NULL) && !this_page_has_changes)
	{
		gboolean b;
		if(type == EDV_OBJECT_TYPE_LINK)
			b = FALSE;
		else
			b = (permissions & EDV_PERMISSION_SETUID) ? TRUE : FALSE;
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);
	}
	w = p->sgid_check;
	if((w != NULL) && !this_page_has_changes)
	{
		gboolean b;
		if(type == EDV_OBJECT_TYPE_LINK)
			b = FALSE;
		else
			b = (permissions & EDV_PERMISSION_SETGID) ? TRUE : FALSE;
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);
	}
	w = p->sticky_check;
	if((w != NULL) && !this_page_has_changes)
	{
		gboolean b;
		if(type == EDV_OBJECT_TYPE_LINK)
			b = FALSE;
		else
			b = (permissions & EDV_PERMISSION_STICKY) ? TRUE : FALSE;
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);
	}

	gtk_widget_queue_resize(toplevel);

	p->freeze_count--;
}
