#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <gtk/gtk.h>

#include "cfg.h"

#include "guiutils.h"
#include "fb.h"
#include "icon_picker.h"
#include "icon_sel_dlg.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_stream.h"
#include "libendeavour2-base/edv_process.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_property.h"
#include "libendeavour2-base/edv_property_directory.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "edv_pixmap.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "edv_utils_gtk.h"
#include "prop_page.h"
#include "prop_page_directory.h"
#include "edv_folder.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


typedef struct _EDVDirectoryPropPage	EDVDirectoryPropPage;
#define EDV_DIRECTORY_PROP_PAGE(p)	((EDVDirectoryPropPage *)(p))


/*
 *	Flags:
 */
typedef enum {
	EDV_DIRECTORY_PROP_PAGE_CURRENTLY_DISPLAYED	\
					= (1 << 0),
	EDV_DIRECTORY_PROP_PAGE_GOT_STATISTICS	\
					= (1 << 1),
	EDV_DIRECTORY_PROP_PAGE_HAS_CHANGES	\
					= (1 << 7)
} EDVDirectoryPropPageFlags;


/* Check Support */
gboolean edv_directory_prop_page_query_create_cb(
	EDVPropPageContext *ctx,
	gint *version_major_rtn,
	gint *version_minor_rtn,
	gint *version_release_rtn,
	gchar **page_name_rtn,
	edv_pixmap_data **pixmap_data_20x20_rtn,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list
);

/* Create */
gpointer edv_directory_prop_page_create_cb(
	EDVPropPageContext *ctx,
	GtkWidget *parent
);

/* Update */
void edv_directory_prop_page_update_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list,
	const int error_code,
	gpointer data
);

/* Page switched callback */
void edv_directory_prop_page_page_switched_cb(
	EDVPropPageContext *ctx,
	const gboolean to,
	gpointer data
);

/* Apply */
gboolean edv_directory_prop_page_apply_vfs_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	GList *properties_list,
	gpointer data
);
gboolean edv_directory_prop_page_apply_recycle_bin_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	GList *properties_list,
	gpointer data
);

/* Destroy */
void edv_directory_prop_page_destroy_cb(
	EDVPropPageContext *ctx,
	gpointer data
);

/* Callbacks */
static gint edv_directory_prop_page_get_statistics_timeout_cb(gpointer data);
static void edv_directory_prop_page_set_icons_from_image_cb(GtkWidget *widget, gpointer data);
static void edv_directory_prop_page_remove_customizations_cb(GtkWidget *widget, gpointer data);
static void edv_directory_prop_page_icon_switched_cb(
	GtkWidget *w,
	const gint icon_num, IconPickerIcon *icon,
	gpointer data
);
static void edv_directory_prop_page_icon_changed_cb(GtkWidget *widget, gpointer data);

/* Operations */
static void edv_directory_prop_page_set_icons_from_image(EDVDirectoryPropPage *p);
static void edv_directory_prop_page_set_icons_from_image_vfs(EDVDirectoryPropPage *p);
static void edv_directory_prop_page_remove_customizations(EDVDirectoryPropPage *p);

/* Update Widgets */
static void edv_directory_prop_page_update_widgets(EDVDirectoryPropPage *p);


#define EDV_DIRECTORY_PROP_PAGE_NAME	"Directory"

#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define FCLOSE(_fp_)	(((_fp_) != NULL) ? (gint)fclose(_fp_) : -1)


/*
 *	Directory Page:
 */
struct _EDVDirectoryPropPage {
	GtkWidget	*toplevel;
	gint		freeze_count;
	EDVPropPageContext	*ctx;

	EDVDirectoryPropPageFlags	flags;

	gchar		*cur_large_icon_path[2],
			*cur_medium_icon_path[2],
			*cur_small_icon_path[2];

	GtkWidget	*statistics_label,
			*set_icons_from_image_btn,
			*remove_customizations_btn,
			*icon_large_ip,
			*icon_medium_ip,
			*icon_small_ip,
			*icon_sel_dlg;

	/* Statistics */
	guint		get_statistics_toid;
	gint		get_statistics_pid;
	FILE		*statistics_stdout;
	gchar		*statistics_linebuf;
};


/*
 *	Check support callback.
 */
gboolean edv_directory_prop_page_query_create_cb(
	EDVPropPageContext *ctx,
	gint *version_major_rtn,
	gint *version_minor_rtn,
	gint *version_release_rtn,
	gchar **page_name_rtn,
	edv_pixmap_data **pixmap_data_20x20_rtn,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list
)
{
	*version_major_rtn = PROG_VERSION_MAJOR;
	*version_minor_rtn = PROG_VERSION_MINOR;
	*version_release_rtn = PROG_VERSION_RELEASE;
	*page_name_rtn = g_strdup(EDV_DIRECTORY_PROP_PAGE_NAME);

	switch(location_type)
	{
	    case EDV_LOCATION_TYPE_VFS:
		if(type == EDV_OBJECT_TYPE_DIRECTORY)
		{
			return(TRUE);
		}
		break;

	    case EDV_LOCATION_TYPE_RECYCLE_BIN:
	    case EDV_LOCATION_TYPE_ARCHIVE:
		break;
	}

	return(FALSE);
}

/*
 *	Create callback.
 */
gpointer edv_directory_prop_page_create_cb(
	EDVPropPageContext *ctx,
	GtkWidget *parent
)
{
	const gint	border_major = 5;
	GtkWidget	*w,
			*parent2, *parent3, *parent4, *parent5;
	const EDVLocationType location_type = edv_prop_page_get_location_type(ctx);
	CfgList *cfg_list = edv_prop_page_get_cfg_list(ctx);
	EDVDirectoryPropPage *p = EDV_DIRECTORY_PROP_PAGE(g_malloc0(
		sizeof(EDVDirectoryPropPage)
	));
	if(p == NULL)
		return(NULL);

	p->ctx = ctx;
	p->toplevel = parent;

	p->freeze_count++;

	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Left Side GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Statistics GtkFrame */
	w = gtk_frame_new("Statistics");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_end(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	w = gtk_label_new(
"Total:\n\
Objects:\n\
Subdirs:\n\
Depth:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Usage GtkLabel */
	p->statistics_label = w = gtk_label_new(
"Calculating...\n\
-\n\
-\n\
-");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Customizations GtkFrame */
	w = gtk_frame_new("Customizations");
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;

	if(location_type == EDV_LOCATION_TYPE_VFS)
	{
		w = gtk_hbox_new(FALSE, border_major);
		gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent5 = w;

		/* Set icons from image GtkButton */
		p->set_icons_from_image_btn = w = gtk_button_new_with_label(
			"Select Image"
		);
		gtk_widget_set_usize(
			w,
			GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
		);
		gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
		gtk_signal_connect(
			GTK_OBJECT(w), "clicked",
			GTK_SIGNAL_FUNC(edv_directory_prop_page_set_icons_from_image_cb), p
		);
		GUISetWidgetTip(
			w,
"Click this to create a new set of icons for this directory from an\
 image of any format and size that you select. This directory's icons\
 will then be automatically set to the newly created set of icons,\
 which will be placed in this directory. Any existing set of icons\
 previously created will be overwritten"
		);
		gtk_widget_show(w);


		w = gtk_hbox_new(FALSE, border_major);
		gtk_box_pack_end(GTK_BOX(parent4), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent5 = w;

		/* Remove Customizations GtkButton */
		p->remove_customizations_btn = w = gtk_button_new_with_label(
			"Remove"
		);
		gtk_widget_set_usize(
			w,
			GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
		);
		gtk_box_pack_end(GTK_BOX(parent5), w, FALSE, FALSE, 0);
		gtk_signal_connect(
			GTK_OBJECT(w), "clicked",
			GTK_SIGNAL_FUNC(edv_directory_prop_page_remove_customizations_cb), p
		);
		GUISetWidgetTip(
			w,
"Click this to remove this directory's properties file and icon\
 files. Removing those files will effectively remove all customizations\
 for this directory excluding those specified by its associated MIME Type"
		);
		gtk_widget_show(w);
	}


	/* Right Side GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	if(location_type == EDV_LOCATION_TYPE_VFS)
	{
		IconPickerFlags ip_flags;

		/* Create the Icon Selector Dialog on the Properties
		 * Dialog if no other page has requested one, the Icon
		 * Selector Dialog will have a reference count added for
		 * us
		 *
		 * We need the Icon Selector Dialog to select icons with
		 * the icon pickers
		 */
		GtkWidget *icon_sel_dlg = edv_prop_page_create_icon_selector(ctx);
		p->icon_sel_dlg = icon_sel_dlg;

		/* Icons GtkFrame */
		w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Iconos"
#elif defined(PROG_LANGUAGE_FRENCH)
"Icne"
#elif defined(PROG_LANGUAGE_GERMAN)
"Abbilder"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Le Icone"
#elif defined(PROG_LANGUAGE_DUTCH)
"Beelden"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Os cones"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Ikoner"
#else
"Icons"
#endif
		);
		gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
		gtk_widget_show(w);
		parent4 = w;

		/* Icon Pickers GtkVBox */
		w = gtk_vbox_new(FALSE, 2 * border_major);
		gtk_container_border_width(GTK_CONTAINER(w), border_major);
		gtk_container_add(GTK_CONTAINER(parent4), w);
		gtk_widget_show(w);
		parent4 = w;

		/* Set the Icon Picker flags */
		ip_flags = ICON_PICKER_CAN_CLEAR_ICON |
			ICON_PICKER_ACCEPT_SMALLER_SIZE |
			ICON_PICKER_ACCEPT_LARGER_SIZE |
			ICON_PICKER_WARN_DIFFERENT_SIZE |
			ICON_PICKER_ACCEPT_DND_DROP |
			ICON_PICKER_PROVIDE_DND_DRAG;
		if(EDV_GET_B(EDV_CFG_PARM_LISTS_DOUBLE_BUFFER))
			ip_flags |= ICON_PICKER_DOUBLE_BUFFER;
		if(EDV_GET_B(EDV_CFG_PARM_SHOW_TEXTTIPS))
			ip_flags |= ICON_PICKER_SHOW_TEXTTIPS;

		/* Large Icon Picker */
		p->icon_large_ip = w = icon_picker_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Grande"
#elif defined(PROG_LANGUAGE_FRENCH)
"Grand"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gro"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Grande"
#elif defined(PROG_LANGUAGE_DUTCH)
"Groot"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Grande"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stor"
#else
"Large"
#endif
			" 48x48:",
			ip_flags,
			ICON_PICKER_ICON_POSITION_CENTER,
			48, 48,				/* Icon size */
			48, 48,
/*		54, 54, */			/* Display size */
			80,				/* Label width */
			icon_sel_dlg
		);
		gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
		icon_picker_set_switched_icon_cb(
			w,
			edv_directory_prop_page_icon_switched_cb, p
		);
		icon_picker_set_changed_cb(
			w,
			edv_directory_prop_page_icon_changed_cb, p
		);
		gtk_widget_show(w);

		/* Medium Icon Picker */
		p->icon_medium_ip = w = icon_picker_new(
#if defined(PROG_LANGUAGE_SPANISH)
"El Medio"
#elif defined(PROG_LANGUAGE_FRENCH)
"Moyen"
#elif defined(PROG_LANGUAGE_GERMAN)
"Medium"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Mezzo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Middel"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mdio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Middel"
#else
"Medium"
#endif
			" 32x32:",
			ip_flags,
			ICON_PICKER_ICON_POSITION_CENTER,
			32, 32,				/* Icon size */
			32, 32,
/*		38, 38,	*/			/* Display size */
			80,				/* Label width */
			icon_sel_dlg
		);
		gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
		icon_picker_set_switched_icon_cb(
			w,
			edv_directory_prop_page_icon_switched_cb, p
		);
		icon_picker_set_changed_cb(
			w,
			edv_directory_prop_page_icon_changed_cb, p
		);
		gtk_widget_show(w);

		/* Small Icon Picker */
		p->icon_small_ip = w = icon_picker_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Pequeo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Petit"
#elif defined(PROG_LANGUAGE_GERMAN)
"Klein"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Piccolo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Klein"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Pequeno"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Liten"
#else
"Small"
#endif
			" 20x20:",
			ip_flags,
			ICON_PICKER_ICON_POSITION_CENTER,
			20, 20,				/* Icon size */
			20, 20,
/*		26, 26, */			/* Display size */
			80,				/* Label width */
			icon_sel_dlg
		);
		gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
		icon_picker_set_switched_icon_cb(
			w,
			edv_directory_prop_page_icon_switched_cb, p
		);
		icon_picker_set_changed_cb(
			w,
			edv_directory_prop_page_icon_changed_cb, p
		);
		gtk_widget_show(w);
	}

	p->freeze_count--;

	return(p);
}


/*
 *	Update callback.
 */
void edv_directory_prop_page_update_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list,
	const int error_code,
	gpointer data
)
{
	EDVDirectoryPropPage *p = EDV_DIRECTORY_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	edv_directory_prop_page_update_widgets(p);

	p->freeze_count--;
}


/*
 *	Page switched callback.
 */
void edv_directory_prop_page_page_switched_cb(
	EDVPropPageContext *ctx,
	const gboolean to,
	gpointer data
)
{
	EDVDirectoryPropPage *p = EDV_DIRECTORY_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	if(to)
		p->flags |= EDV_DIRECTORY_PROP_PAGE_CURRENTLY_DISPLAYED;
	else
		p->flags &= ~EDV_DIRECTORY_PROP_PAGE_CURRENTLY_DISPLAYED;

	/* Queue the get statistics process if our page was just
	 * switched to and the statistics have not been obtained yet
	 */
	if(p->flags & EDV_DIRECTORY_PROP_PAGE_CURRENTLY_DISPLAYED)
	{
		if(!(p->flags & EDV_DIRECTORY_PROP_PAGE_GOT_STATISTICS))
		{
			if(p->get_statistics_toid == 0)
				p->get_statistics_toid = gtk_timeout_add(
					1000l,
					edv_directory_prop_page_get_statistics_timeout_cb, p
				);
		}
	}

	p->freeze_count--;
}


/*
 *	Apply VFS object callback.
 */
gboolean edv_directory_prop_page_apply_vfs_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	GList *properties_list,
	gpointer data
)
{
	gboolean status = FALSE;
	const gchar *path;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);
	EDVDirectoryPropPage *p = EDV_DIRECTORY_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return(status);

	p->freeze_count++;

	/* Do not apply if we did not make any changes */
	if(!(p->flags & EDV_DIRECTORY_PROP_PAGE_HAS_CHANGES))
	{
		p->freeze_count--;
		return(status);
	}

	path = edv_properties_list_get_s(
		properties_list,
		EDV_PROP_NAME_PATH
	);
	if(path == NULL)
	{
		p->freeze_count--;
		return(status);
	}

#define SET_ICON(_w_,_paths_list_,_icon_state_,_icon_size_,_set_icon_func_) {	\
 /* Get the icon from the icon picker */			\
 IconPickerIcon *icon = ICON_PICKER_ICON(g_list_nth_data(	\
  icon_picker_get_icons_list(_w_),				\
  (_icon_state_)						\
 ));								\
 if(icon != NULL) {						\
  /* Get the current and new icon paths */			\
  gboolean need_change = FALSE;					\
  const gchar *cur_icon_path = (_paths_list_)[(_icon_state_)],	\
	      *new_icon_path = STRISEMPTY(icon->path) ?		\
  NULL : icon->path;						\
								\
  /* Check if the value needs to be changed */			\
  if((new_icon_path != NULL) && (cur_icon_path != NULL)) {	\
   if(strcmp(							\
    (const char *)new_icon_path,				\
    (const char *)cur_icon_path					\
   ))								\
    need_change = TRUE;						\
  }								\
  else if((new_icon_path != NULL) && (cur_icon_path == NULL))	\
  {								\
   need_change = TRUE;						\
  }								\
  else if((new_icon_path == NULL) && (cur_icon_path != NULL))	\
  {								\
   need_change = TRUE;						\
  }								\
								\
  /* Change the value? */					\
  if(need_change) {						\
   /* Set the new icon path to the properties file */		\
   if((_set_icon_func_)(					\
    path,							\
    (_icon_size_),						\
    new_icon_path						\
   )) {								\
    const gint error_code = (gint)errno;			\
    gchar *msg = g_strdup_printf(				\
"Unable to %s the icon:\n\
\n\
    %s\n\
\n\
%s.",								\
     STRISEMPTY(new_icon_path) ? "clear" : "change",		\
     path,							\
     g_strerror(error_code)					\
    );								\
    edv_play_sound_error(core);					\
    edv_message_error(						\
     "Change Icon Error",					\
     msg,							\
     NULL,							\
     toplevel							\
     );								\
    g_free(msg);						\
   } else {							\
    if(!status)							\
     status = TRUE;						\
   }								\
  }								\
 }								\
}

	SET_ICON(
		p->icon_large_ip,
		p->cur_large_icon_path,
		0,					/* Standard */
		EDV_ICON_SIZE_48,			/* Large */
		edv_property_directory_set_icon_path
	);
	SET_ICON(
		p->icon_large_ip,
		p->cur_large_icon_path,
		1,					/* Opened */
		EDV_ICON_SIZE_48,			/* Large */
		edv_property_directory_set_icon_opened_path
	);

	SET_ICON(
		p->icon_medium_ip,
		p->cur_medium_icon_path,
		0,					/* Standard */
		EDV_ICON_SIZE_32,			/* Medium */
		edv_property_directory_set_icon_path
	);
	SET_ICON(
		p->icon_medium_ip,
		p->cur_medium_icon_path,
		1,					/* Opened */
		EDV_ICON_SIZE_32,			/* Medium */
		edv_property_directory_set_icon_opened_path
	);

	SET_ICON(
		p->icon_small_ip,
		p->cur_small_icon_path,
		0,					/* Standard */
		EDV_ICON_SIZE_20,			/* Small */
		edv_property_directory_set_icon_path
	);
	SET_ICON(
		p->icon_small_ip,
		p->cur_small_icon_path,
		1,					/* Opened */
		EDV_ICON_SIZE_20,			/* Small */
		edv_property_directory_set_icon_opened_path
	);
 
#undef SET_ICON

	/* Remove our has changes marker */
	p->flags &= ~EDV_DIRECTORY_PROP_PAGE_HAS_CHANGES;

	p->freeze_count--;

	return(status);
}

/*
 *	Apply recycled object callback.
 */
gboolean edv_directory_prop_page_apply_recycle_bin_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	GList *properties_list,
	gpointer data
)
{
	return(FALSE);
}

/*
 *	Destroy callback.
 */
void edv_directory_prop_page_destroy_cb(
	EDVPropPageContext *ctx,
	gpointer data
)
{
	EDVDirectoryPropPage *p = EDV_DIRECTORY_PROP_PAGE(data);

	p->freeze_count++;

	p->get_statistics_toid = GTK_TIMEOUT_REMOVE(p->get_statistics_toid);

	if(p->get_statistics_pid > 0)
	{
		(void)kill(
			(int)p->get_statistics_pid,
			SIGINT
		);
		p->get_statistics_pid = 0;
	}

	(void)FCLOSE(p->statistics_stdout);
	p->statistics_stdout = NULL;

	g_free(p->statistics_linebuf);
	p->statistics_linebuf = NULL;

	p->icon_sel_dlg = IconSelDlgUnref(p->icon_sel_dlg);

	g_free(p->cur_large_icon_path[0]);
	g_free(p->cur_medium_icon_path[0]);
	g_free(p->cur_small_icon_path[0]);
	g_free(p->cur_large_icon_path[1]);
	g_free(p->cur_medium_icon_path[1]);
	g_free(p->cur_small_icon_path[1]);

	p->freeze_count--;

	g_free(p);
}


/*
 *	Get statistics timeout callback.
 */
static gint edv_directory_prop_page_get_statistics_timeout_cb(gpointer data)
{
	EDVDirectoryPropPage *p = EDV_DIRECTORY_PROP_PAGE(data);
	EDVPropPageContext *ctx = p->ctx;
	CfgList *cfg_list = edv_prop_page_get_cfg_list(ctx);

	if(p->freeze_count > 0)
		return(TRUE);

	p->freeze_count++;

	/* Is the get statistics process marked as running? */
	if(p->get_statistics_pid > 0)
	{
		/* Check if the process is no longer running */
		if(!edv_pid_exists(p->get_statistics_pid))
		{
			/* Mark the process as no longer running */
			p->get_statistics_pid = 0;
		}

		/* Check for, read, parse, and display any data from
		 * the get statistics stdout stream then check if the
		 * process is no longer running and close the stream
		 * and stop calling this callback if it is no longer
		 * running
		 */
		if(p->statistics_stdout != NULL)
		{
			const gulong block_size = MAX(
				EDV_GET_UL(EDV_CFG_PARM_BLOCK_SIZE),
				1l
			);
			gint i;
			gulong	total_size_blocks,
				total_children,
				total_subdirectories,  
				ngenerations;
			gchar	**line_buf_ptr = &p->statistics_linebuf,
				*msg,
				*total_size_kb_s,
				*total_size_mb_s,
				*total_children_s,
				*total_subdirectories_s,
				*ngenerations_s;
			FILE *fp = p->statistics_stdout;

			/* Read all data currently in the stream */
			while(edv_stream_read_strptrbrk(
				fp,
				line_buf_ptr,
				"\n\r",		/* End characters */
				FALSE,		/* Exclude end character */
				FALSE		/* Nonblocking */
			))
			{
				total_size_blocks = 0l;
				total_children = 0l;
				total_subdirectories = 0l;
				ngenerations = 0l;
				i = (gint)sscanf(
					(const char *)*line_buf_ptr,
					"%ld %ld %ld %ld",
					&total_size_blocks,
					&total_children,
					&total_subdirectories,
					&ngenerations
				);

				/* Format the data for display on the statistics label */
				total_size_kb_s = STRDUP(edv_str_size_delim(
					total_size_blocks
				));
				total_size_mb_s = STRDUP(edv_str_size_delim(
					total_size_blocks / block_size
				));
				total_children_s = STRDUP(edv_str_size_delim(total_children));
				total_subdirectories_s = STRDUP(edv_str_size_delim(total_subdirectories));
				ngenerations_s = STRDUP(edv_str_size_delim(ngenerations));
				msg = g_strdup_printf(
"%s kb (%s mb)\n\
%s\n\
%s\n\
%s",
					total_size_kb_s, total_size_mb_s,
					total_children_s,
					total_subdirectories_s,
					ngenerations_s
				);
				g_free(total_size_kb_s);
				g_free(total_size_mb_s);
				g_free(total_children_s);
				g_free(total_subdirectories_s);
				g_free(ngenerations_s);
				if(msg != NULL)
				{
					/* Display the data on the statistics label */
					gtk_label_set_text(
						GTK_LABEL(p->statistics_label),
						msg
					);
					g_free(msg);
				}

				g_free(*line_buf_ptr);
				*line_buf_ptr = NULL;
			}
		}

		/* Was the process marked as no longer running? */
		if(p->get_statistics_pid == 0)
		{
			/* Close the stream */
			(void)FCLOSE(p->statistics_stdout);
			p->statistics_stdout = NULL;

			/* Clear the line buffer */
			g_free(p->statistics_linebuf);
			p->statistics_linebuf = NULL;

			/* Mark that we have obtained the statistics, clear
			 * the get statistcs process pid, and stop calling
			 * this callback
			 */
			p->flags |= EDV_DIRECTORY_PROP_PAGE_GOT_STATISTICS;
			p->get_statistics_toid = 0;
			p->freeze_count--;
			return(FALSE);
		}
	}
	else
	{
		/* The get statistics process is not running
		 *
		 * If we have already obtained the statistics then do
		 * not call this function again, otherwise start the
		 * get statistics process if we have not obtained the
		 * statistics yet
		 */
		if(p->flags & EDV_DIRECTORY_PROP_PAGE_GOT_STATISTICS)
		{
			/* Stop calling this function */
			p->get_statistics_toid = 0;
			p->freeze_count--;
			return(FALSE);
		}
		else
		{
			GList *properties_list = edv_prop_page_get_properties_list(ctx);

			/* Start the get statistics process based on the
			 * location type
			 */
			switch(edv_prop_page_get_location_type(ctx))
			{
			    case EDV_LOCATION_TYPE_VFS:
				if(properties_list != NULL)
				{
					FILE *cstdout;
					gchar	*path = STRDUP(edv_properties_list_get_s(
						properties_list,
						EDV_PROP_NAME_PATH
					)),
						*cmd = g_strdup_printf(
"\"%s%c%s%c%s\" \"%s\" --block-size %ld",
							EDV_PATH_DEF_GLOBAL_LIB_DIR,
							G_DIR_SEPARATOR,
							EDV_NAME_BIN_SUBDIR,
							G_DIR_SEPARATOR,
							EDV_NAME_DIRECTORY_STATISTICS,
							path,
							EDV_GET_UL(EDV_CFG_PARM_BLOCK_SIZE)
					),
							*shell_prog;
					const gchar	*shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
							*shell_args = edv_strarg(
							shell_cmd,
							&shell_prog,
							TRUE,		/* Parse escapes */
							TRUE		/* Parse quotes */
					);
					const gint pid = edv_system_shell_streams(
							cmd,
							shell_prog,
							shell_args,
							NULL,
							&cstdout,
							NULL
					);
					g_free(path);
					g_free(cmd);
					g_free(shell_prog);
					if(pid > -1)
					{
						(void)FCLOSE(p->statistics_stdout);
						p->statistics_stdout = cstdout;
						g_free(p->statistics_linebuf);
						p->statistics_linebuf = NULL;
						p->get_statistics_pid = pid;
					}
					else
					{
						/* Failed to execute the process */
						(void)FCLOSE(cstdout);
						p->get_statistics_toid = 0;
						p->freeze_count--;
						return(FALSE);
					}
				}
				break;

			    case EDV_LOCATION_TYPE_RECYCLE_BIN:
				p->get_statistics_toid = 0;
				p->freeze_count--;
				return(FALSE);
				break;

			    case EDV_LOCATION_TYPE_ARCHIVE:
				p->get_statistics_toid = 0;
				p->freeze_count--;
				return(FALSE);
				break;
			}
		}
	}

	p->freeze_count--;

	return(TRUE);
}


/*
 *	Set icons from image callback.
 */
static void edv_directory_prop_page_set_icons_from_image_cb(GtkWidget *widget, gpointer data)
{
	EDVDirectoryPropPage *p = EDV_DIRECTORY_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	edv_directory_prop_page_set_icons_from_image(p);

	p->freeze_count--;
}

/*
 *	Remove customizations callback.
 */
static void edv_directory_prop_page_remove_customizations_cb(GtkWidget *widget, gpointer data)
{
	EDVDirectoryPropPage *p = EDV_DIRECTORY_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	edv_directory_prop_page_remove_customizations(p);

	p->freeze_count--;
}

/*
 *	Icon Picker icon switched signal callback.
 */
static void edv_directory_prop_page_icon_switched_cb(
	GtkWidget *w,
	const gint icon_num, IconPickerIcon *icon,
	gpointer data
)
{
#if 0
/* Do nothing */
	EDVDirectoryPropPage *p = EDV_DIRECTORY_PROP_PAGE(data);
/*	EDVPropPageContext *ctx = p->ctx; */

	if(p->freeze_count > 0)
		return;

#endif
}

/*
 *	Icon Picker icon changed signal callback.
 */
static void edv_directory_prop_page_icon_changed_cb(GtkWidget *widget, gpointer data)
{
	IconPickerIcon *icon;
	EDVDirectoryPropPage *p = EDV_DIRECTORY_PROP_PAGE(data);
	EDVPropPageContext *ctx = p->ctx;

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	/* Get the icon picker's icon that has changed */
	icon = icon_picker_get_icon(
		widget,
		0				/* Standard state */
	);
	if(icon != NULL)
	{
		/* Set or clear the new icon to the directory's properties */
		const gchar *path = icon->path;
		if(STRISEMPTY(path))
		{

		}
		else
		{

		}

		/* Mark that we have made changes */
		p->flags |= EDV_DIRECTORY_PROP_PAGE_HAS_CHANGES;
		edv_prop_page_set_has_changes(
			ctx,
			TRUE
		);
	}

	p->freeze_count--;
}


/*
 *	Set icons from image.
 */
static void edv_directory_prop_page_set_icons_from_image(EDVDirectoryPropPage *p)
{
	EDVPropPageContext *ctx = p->ctx;

	edv_prop_page_set_busy(ctx, TRUE);
	p->freeze_count++;

	/* Set the icons from an image based on the location type */
	switch(edv_prop_page_get_location_type(ctx))
	{
	    case EDV_LOCATION_TYPE_VFS:
		edv_directory_prop_page_set_icons_from_image_vfs(p);
		break;

	    case EDV_LOCATION_TYPE_RECYCLE_BIN:
		/* Recycled objects have no icons to set */
		break;

	    case EDV_LOCATION_TYPE_ARCHIVE:
		/* Archive objects have no icons to set */
		break;
	}

	p->freeze_count--;
	edv_prop_page_set_busy(ctx, FALSE);
}

static void edv_directory_prop_page_set_icons_from_image_vfs(EDVDirectoryPropPage *p)
{
	gboolean response;
	gint		npaths = 0,
			nftypes = 0;
	gchar **paths_list = NULL;
	fb_type_struct  **ftypes_list = NULL,
			*ftype_rtn = NULL;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);

	/* Get the path to this directory */
	gchar *directory_path = STRDUP(edv_properties_list_get_s(
		properties_list,
		EDV_PROP_NAME_PATH
	));

	/* Create the file types list */
	FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		"*.*", "All Images"
	);

	/* Query the user for a new image */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
		"Select Image",
		"Select", "Cancel",
		directory_path,
		ftypes_list, nftypes,
		&paths_list, &npaths,
		&ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(response && (npaths > 0))
	{
		/* Get the path to the selected image file */
		gchar *image_path = STRDUP(paths_list[npaths - 1]);
		if(image_path != NULL)
		{
			const gboolean prev_has_changes = (p->flags & EDV_DIRECTORY_PROP_PAGE_HAS_CHANGES) ? TRUE : FALSE;

			/* Need to temporary remove our has changes
			 * marker so that the signals emitted by
			 * edv_folder_set_icons_from_image() will
			 * cause this properties dialog's values to
			 * be updated
			 */
			if(prev_has_changes)
				p->flags &= ~EDV_DIRECTORY_PROP_PAGE_HAS_CHANGES;

			/* Create the icon files from the image in
			 * this directory, update the directory's
			 * properties file, and update this properties
			 * dialog's values
			 */
			(void)edv_folder_set_icons_from_image(
				core,
				directory_path,
				image_path,
				TRUE,		/* Show progress */
				TRUE,		/* Interactive */
				toplevel
			);

			/* Since this call freezes our page, we need
			 * to explicitly update the values displayed
			 * on our page
			 */
			edv_directory_prop_page_update_widgets(p);

			/* Restore our has changes marker */
			if(prev_has_changes)
				p->flags |= EDV_DIRECTORY_PROP_PAGE_HAS_CHANGES;

			g_free(image_path);
		}
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	g_free(directory_path);
}

/*
 *	Remove customizations.
 */
static void edv_directory_prop_page_remove_customizations(EDVDirectoryPropPage *p)
{
	gchar *path;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	const EDVLocationType location_type = edv_prop_page_get_location_type(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);

	/* Remove customizations by the location type */
	switch(location_type)
	{
	    case EDV_LOCATION_TYPE_VFS:
		path = STRDUP(edv_properties_list_get_s(
			properties_list,
			EDV_PROP_NAME_PATH
		));
		if(path != NULL)
		{
			const gboolean prev_has_changes = (p->flags & EDV_DIRECTORY_PROP_PAGE_HAS_CHANGES) ? TRUE : FALSE;

			/* Need to temporary remove our has changes
			 * marker so that the signals emitted by
			 * edv_folder_set_icons_from_image() will
			 * cause this properties dialog's values to
			 * be updated
			 */
			if(prev_has_changes)
				p->flags &= ~EDV_DIRECTORY_PROP_PAGE_HAS_CHANGES;

			/* Remove the properties file and icon files,
			 * thus removing all the customizations from
			 * this directory, and update this properties
			 * dialog
			 */
			(void)edv_folder_remove_all_customizations(
				core,
				path,
				TRUE,		/* Show progress */
				TRUE,		/* Interactive */
				toplevel
			);

			/* Since this call freezes our page, we need
			 * to explicitly update the values displayed
			 * on our page
			 */
			edv_directory_prop_page_update_widgets(p);

			/* Restore our has changes marker */
			if(prev_has_changes)
				p->flags |= EDV_DIRECTORY_PROP_PAGE_HAS_CHANGES;

			g_free(path);
		}
		break;

	    case EDV_LOCATION_TYPE_RECYCLE_BIN:
		/* Recycled objects have no customizations to remove */
		break;

	    case EDV_LOCATION_TYPE_ARCHIVE:
		/* Archive objects have no customizations to remove */
		break;
	}
}


/*
 *	Updates the values displayed on the GtkWidgets.
 */
static void edv_directory_prop_page_update_widgets(EDVDirectoryPropPage *p)
{
	const gchar *path;
	const gboolean this_page_has_changes = (p->flags & EDV_DIRECTORY_PROP_PAGE_HAS_CHANGES) ? TRUE : FALSE;
	EDVPropPageContext *ctx = p->ctx;
	const EDVLocationType location_type = edv_prop_page_get_location_type(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);

	p->freeze_count++;

	switch(location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
		path = edv_properties_list_get_s(
			properties_list,
			EDV_PROP_NAME_PATH
		);
		if(path != NULL)
		{
			GtkWidget *w;

			/* Get the directory MIME Type who's
			 * information will be used for defaults
			 * here
			 */
			EDVMIMEType *m = edv_mime_types_list_match_path(
				core->mime_types_list,
				path
			);
			if(m == NULL)
				m = edv_mime_types_list_match_type(
					core->mime_types_list,
					NULL,
					EDV_MIME_TYPE_TYPE_INODE_DIRECTORY,
					FALSE
				);

			/* Update the Set Icons From Image GtkButton */
			GTK_WIDGET_SET_SENSITIVE(
				p->set_icons_from_image_btn,
				TRUE
			);

			/* Update the Remove Customizations Gtkbutton */
			w = p->remove_customizations_btn;
			if(w != NULL)
			{
				/* Check if a directory properties
				 * file exists in this directory and
				 * set the Remove Customizations
				 * GtkButton's sensitivity accordingly
				 */
				gchar *prop_path = g_strconcat(
					path,
					G_DIR_SEPARATOR_S,
					EDV_NAME_DIRECTORY_PROPERTIES_FILE,
					NULL
				);
				GTK_WIDGET_SET_SENSITIVE(
					w,
					edv_path_lexists(prop_path)
				);
				g_free(prop_path);
			}

#define GET_ICON(_paths_list_,_label_,_icon_state_,_icon_size_,_get_icon_func_) { \
 IconPickerIcon *icon = icon_picker_icon_new();			\
 if(icon != NULL) {						\
  gchar *icon_path;						\
  icon->label = g_strdup(_label_);				\
  /* Get the path to the icon file specified in			\
   * this directory's properties file, if this			\
   * directory's properties file does not exist			\
   * or no icon file was specified then icon_path		\
   * will be NULL						\
   */								\
  icon_path = (_get_icon_func_)(				\
   path,			/* This directory */		\
   (_icon_size_)		/* Large icon */		\
  );								\
  g_free((_paths_list_)[(_icon_state_)]);			\
  (_paths_list_)[(_icon_state_)] = STRDUP(icon_path);		\
  if(icon_path != NULL) {					\
   EDVPixmap *p = edv_pixmap_new_from_file(icon_path);		\
   if(edv_pixmap_is_loaded(p)) {				\
    icon->path = g_strdup(icon_path);				\
    icon->pixmap = GDK_PIXMAP_REF(p->pixmap);			\
    icon->mask = GDK_BITMAP_REF(p->mask);			\
    icon->width = p->width;					\
    icon->height = p->height;					\
   }								\
   (void)edv_pixmap_unref(p);					\
   g_free(icon_path);						\
  }								\
  icons_list = g_list_append(					\
   icons_list,							\
   icon								\
  );								\
 }								\
}
			/* Large icon */
			w = p->icon_large_ip;
			if((w != NULL) && !this_page_has_changes)
			{
				GList *icons_list = NULL;

				GET_ICON(
					p->cur_large_icon_path,
					"Standard",
					0,			/* Standard */
					EDV_ICON_SIZE_48,	/* Large */
					edv_property_directory_get_icon_path
				);
				GET_ICON(
					p->cur_large_icon_path,
					"Opened",
					1,			/* Opened */
					EDV_ICON_SIZE_48,	/* Large */
					edv_property_directory_get_icon_opened_path
				);

				icon_picker_set_icons_list(w, icons_list);
				icon_picker_set_has_changes(w, FALSE);
			}

			/* Medium icon */
			w = p->icon_medium_ip;
			if((w != NULL) && !this_page_has_changes)
			{
				GList *icons_list = NULL;

				GET_ICON(
					p->cur_medium_icon_path,
					"Standard",
					0,			/* Standard */
					EDV_ICON_SIZE_32,	/* Medium */
					edv_property_directory_get_icon_path
				);
				GET_ICON(
					p->cur_medium_icon_path,
					"Opened",
					1,			/* Opened */
					EDV_ICON_SIZE_32,	/* Medium */
					edv_property_directory_get_icon_opened_path
				);

				icon_picker_set_icons_list(w, icons_list);
				icon_picker_set_has_changes(w, FALSE);
			}

			/* Small icon */
			w = p->icon_small_ip;
			if((w != NULL) && !this_page_has_changes)
			{
				GList *icons_list = NULL;

				GET_ICON(
					p->cur_small_icon_path,
					"Standard",
					0,			/* Standard */
					EDV_ICON_SIZE_20,	/* Small */
					edv_property_directory_get_icon_path
				);
				GET_ICON(
					p->cur_small_icon_path,
					"Opened",
					1,			/* Opened */
					EDV_ICON_SIZE_20,	/* Small */
					edv_property_directory_get_icon_opened_path
				);

				icon_picker_set_icons_list(w, icons_list);
				icon_picker_set_has_changes(w, FALSE);
			}
#undef GET_ICON
		}
		break;

	  case EDV_LOCATION_TYPE_RECYCLE_BIN:
		GTK_WIDGET_SET_SENSITIVE(
			p->set_icons_from_image_btn,
			FALSE
		);
		/* Update the Remove Customizations Gtkbutton */
		GTK_WIDGET_SET_SENSITIVE(
			p->remove_customizations_btn,
			FALSE
		);
		break;

	  case EDV_LOCATION_TYPE_ARCHIVE:
		GTK_WIDGET_SET_SENSITIVE(
			p->set_icons_from_image_btn,
			FALSE
		);
		/* Update the Remove Customizations Gtkbutton */
		GTK_WIDGET_SET_SENSITIVE(
			p->remove_customizations_btn,
			FALSE
		);
		break;
	}

	p->freeze_count--;
}
