#include <stdlib.h>
#include <errno.h>
#include <fcntl.h>				/* mknod() */
#include <sys/types.h>				/* mknod() */
#include <sys/stat.h>				/* mknod() */
#include <gtk/gtk.h>
#include <unistd.h>				/* mknod() */

#include "cfg.h"

#include "guiutils.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_property.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_recycled_obj.h"
#include "libendeavour2-base/edv_archive_obj.h"
#include "edv_utils_gtk.h"
#include "prop_page.h"
#include "prop_page_device_node.h"

#include "config.h"


typedef struct _EDVDeviceNodePropPage	EDVDeviceNodePropPage;
#define EDV_DEVICE_NODE_PROP_PAGE(p)	((EDVDeviceNodePropPage *)(p))


/*
 *	Flags:
 */
typedef enum {
	EDV_DEVICE_NODE_PROP_PAGE_HAS_CHANGES	\
					= (1 << 7)
} EDVDeviceNodePropPageFlags;


/* Check Support */
gboolean edv_device_node_prop_page_query_create_cb(
	EDVPropPageContext *ctx,
	gint *version_major_rtn,
	gint *version_minor_rtn,
	gint *version_release_rtn,
	gchar **page_name_rtn,
	edv_pixmap_data **pixmap_data_20x20_rtn,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list
);

/* Create */
gpointer edv_device_node_prop_page_create_cb(
	EDVPropPageContext *ctx,
	GtkWidget *parent 
);

/* Update */
void edv_device_node_prop_page_update_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list,
	const int error_code,
	gpointer data
);

/* Apply */
gboolean edv_device_node_prop_page_apply_vfs_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	GList *properties_list,
	gpointer data
);
gboolean edv_device_node_prop_page_apply_recycle_bin_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	GList *properties_list,
	gpointer data
);

/* Destroy */
void edv_device_node_prop_page_destroy_cb(
	EDVPropPageContext *ctx,
	gpointer data
);


/* Callbacks */
static void edv_device_node_prop_page_changed_cb(GtkWidget *widget, gpointer data);

/* Update Widgets */
static void edv_device_node_prop_page_update_widgets(EDVDeviceNodePropPage *p);


#define EDV_DEVICE_NODE_PROP_PAGE_NAME	"Device Node"

#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Device Node Page:
 */
struct _EDVDeviceNodePropPage {
	GtkWidget	*toplevel;
	gint		freeze_count;
	EDVPropPageContext	*ctx;
	EDVDeviceNodePropPageFlags	flags;

	GtkWidget	*major_entry,
			*minor_entry;
};


/*
 *	Check support callback.
 */
gboolean edv_device_node_prop_page_query_create_cb(
	EDVPropPageContext *ctx,
	gint *version_major_rtn,
	gint *version_minor_rtn,
	gint *version_release_rtn,
	gchar **page_name_rtn,
	edv_pixmap_data **pixmap_data_20x20_rtn,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list
)
{
	*version_major_rtn = PROG_VERSION_MAJOR;
	*version_minor_rtn = PROG_VERSION_MINOR;
	*version_release_rtn = PROG_VERSION_RELEASE;
	*page_name_rtn = g_strdup(EDV_DEVICE_NODE_PROP_PAGE_NAME);

	if((type != EDV_OBJECT_TYPE_DEVICE_BLOCK) &&
	   (type != EDV_OBJECT_TYPE_DEVICE_CHARACTER)
	)
		return(FALSE);

	return(TRUE);
}

/*
 *	Create callback.
 */
gpointer edv_device_node_prop_page_create_cb(
	EDVPropPageContext *ctx,
	GtkWidget *parent
)
{
	const gint	border_major = 5,
			border_minor = 2;
	GtkWidget	*w,
			*parent2, *parent3;
	EDVDeviceNodePropPage *p = EDV_DEVICE_NODE_PROP_PAGE(g_malloc0(
		sizeof(EDVDeviceNodePropPage)
	));
	if(p == NULL)
		return(NULL);

	p->ctx = ctx;
	p->toplevel = parent;

	p->freeze_count++;

	/* Device numbers GtkFrame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"El Artefacto Numeros"
#elif defined(PROG_LANGUAGE_FRENCH)
"Numro de composant"
#elif defined(PROG_LANGUAGE_GERMAN)
"Vorrichtung Zhlt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"I Numeri Di Congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Apparaat Nummert"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Artifcio Numera"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Innretning Numbers"
#else
"Device Numbers"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_hbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Major GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Major GtkLabel */
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Mayor:"
#elif defined(PROG_LANGUAGE_FRENCH)
"Majeur:"
#elif defined(PROG_LANGUAGE_GERMAN)
"Major:"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Maggiore:"
#elif defined(PROG_LANGUAGE_DUTCH)
"Majoor:"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Importante:"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Major:"
#else
"Major:"
#endif
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Major GtkEntry */
	p->major_entry = w = gtk_entry_new();
	gtk_widget_set_usize(w, 80, -1);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(edv_device_node_prop_page_changed_cb), p
	);
	GUIEditableEndowPopupMenu(w, 0);
	gtk_widget_show(w);

	/* Minor GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Minor GtkLabel */
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"El Menor:"
#elif defined(PROG_LANGUAGE_FRENCH)
"Mineur:"
#elif defined(PROG_LANGUAGE_GERMAN)
"Minderjhriger:"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Minore:"
#elif defined(PROG_LANGUAGE_DUTCH)
"Minderjarige:"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Menor:"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Mindrerig:"
#else
"Minor:"
#endif
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Minor GtkEntry */
	p->minor_entry = w = gtk_entry_new();
	gtk_widget_set_usize(w, 80, -1);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(edv_device_node_prop_page_changed_cb), p
	);
	GUIEditableEndowPopupMenu(w, 0);
	gtk_widget_show(w);


	p->freeze_count--;

	return(p);
}


/*
 *	Update callback.
 */
void edv_device_node_prop_page_update_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list,
	const int error_code,
	gpointer data
)
{
	EDVDeviceNodePropPage *p = EDV_DEVICE_NODE_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	edv_device_node_prop_page_update_widgets(p);

	p->freeze_count--;
}

/*
 *	Apply VFS object callback.
 */
gboolean edv_device_node_prop_page_apply_vfs_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	GList *properties_list,
	gpointer data
)
{
	gboolean status = FALSE;
	gint		major,
			minor,
			cur_major,
			cur_minor;
	const gchar *path;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);
	EDVDeviceNodePropPage *p = EDV_DEVICE_NODE_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return(status);

	p->freeze_count++;

	/* Do not apply if we did not make any changes */
	if(!(p->flags & EDV_DEVICE_NODE_PROP_PAGE_HAS_CHANGES))
	{
		p->freeze_count--;
		return(status);
	}

	/* Get the values that are needed to refer to the object that
	 * we want to apply this page's values to
	 */
	path = edv_properties_list_get_s(
		properties_list,
		EDV_PROP_NAME_PATH
	);
	if(STRISEMPTY(path))
	{
		p->freeze_count--;
		return(status);
	}

	/* Get the new major & minor numbers to be set */
	major = ATOI(gtk_entry_get_text(GTK_ENTRY(p->major_entry))),
	minor = ATOI(gtk_entry_get_text(GTK_ENTRY(p->minor_entry)));

	/* Get the current major & minor numbers */
	edv_device_numbers_parse(
		edv_properties_list_get_i(
			properties_list,
			"device_type"
		),
		&cur_major, &cur_minor
	);

	/* Do we need to change the values? */
	if((major != cur_major) || (minor != cur_minor))
	{
		/* Set the device number by recreating the device node */
		const gint device_type = edv_device_numbers_format(
			major, minor
		);

		/* Remove the existing device */
		if(edv_unlink(path))
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"Unable to change the device numbers of:\n\
\n\
    %s\n\
\n\
%s.",
				path,
				g_strerror(error_code)
			);
			edv_play_sound_error(core);
			edv_message_error(
				"Change Device Numbers Error",
				msg,
				NULL,
				toplevel
			);
			g_free(msg);
		}
		/* Recreate the device */
		else if(mknod(
			(const char *)path,
			edv_edv_permissions_to_stat_mode((EDVPermissionFlags)edv_properties_list_get_i(
				properties_list,
				"permissions"
			)),
			(dev_t)device_type
		))
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"Unable to change the device numbers of:\n\
\n\
    %s\n\
\n\
%s.",
				path,
				g_strerror(error_code)
			);
			edv_play_sound_error(core);
			edv_message_error(
				"Change Device Numbers Error",
				msg,
				NULL,
				toplevel
			);
			g_free(msg);
		}
		else
		{
			/* Restore the ownership */
			(void)edv_lchown(
				path,
				edv_properties_list_get_i(
					properties_list,
					"owner_id"
				),
				edv_properties_list_get_i(
					properties_list,
					"group_id"
				)
			);

			/* Restore the time stamps */
			(void)edv_utime(
				path,
				edv_properties_list_get_ul(
					properties_list,
					"access_time"
				),
				edv_properties_list_get_ul(
					properties_list,
					"modify_time"
				)
			);

			status = TRUE;
		}
	}

	/* Remove our has changes marker */
	p->flags &= ~EDV_DEVICE_NODE_PROP_PAGE_HAS_CHANGES;

	p->freeze_count--;

	return(status);
}

/*
 *      Apply recycled object callback.
 */
gboolean edv_device_node_prop_page_apply_recycle_bin_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	GList *properties_list,
	gpointer data
)
{
	return(FALSE);
}


/*
 *      Destroy callback.
 */
void edv_device_node_prop_page_destroy_cb(
	EDVPropPageContext *ctx,
	gpointer data
)
{
	EDVDeviceNodePropPage *p = EDV_DEVICE_NODE_PROP_PAGE(data);
	g_free(p);
}


/*
 *	Any GtkWidget "changed" signal callback.
 */
static void edv_device_node_prop_page_changed_cb(GtkWidget *widget, gpointer data)
{
	EDVDeviceNodePropPage *p = EDV_DEVICE_NODE_PROP_PAGE(data);
	EDVPropPageContext *ctx = p->ctx;

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;			/* Ignore our own update */

	/* Ignore if we have already made changes */
	if(p->flags & EDV_DEVICE_NODE_PROP_PAGE_HAS_CHANGES)
	{
		p->freeze_count--;
		return;
	}

	/* Mark that we have made changes */
	p->flags |= EDV_DEVICE_NODE_PROP_PAGE_HAS_CHANGES;

	/* Notify the EDVPropDlg that changes have been made and
	 * emit an update (which we will ignore)
	 */
	edv_prop_page_set_has_changes(
		ctx,
		TRUE
	);

	p->freeze_count--;
}


/*
 *	Updates the values displayed on the GtkWidgets.
 */
static void edv_device_node_prop_page_update_widgets(EDVDeviceNodePropPage *p)
{
	gint		major,
			minor;
	const gboolean this_page_has_changes = (p->flags & EDV_DEVICE_NODE_PROP_PAGE_HAS_CHANGES) ? TRUE : FALSE;
	EDVPropPageContext *ctx = p->ctx;
	GList *properties_list = edv_prop_page_get_properties_list(ctx);

	p->freeze_count++;

	/* If there are changes then get the current values from this
	 * page's GtkWidgets, otherwise get the current values from
	 * the properties list and set those values to this page's
	 * GtkWidgets, this is to ensure that this page's GtkWidgets
	 * get their values initially set for the first time this
	 * update callback is called and to prevent subsequent
	 * modification of the values displayed on the GtkWidgets
	 * after the user has changed the values on the GtkWidgets
	 */
	if(this_page_has_changes)
	{
		major = ATOI(gtk_entry_get_text(GTK_ENTRY(p->major_entry)));
		minor = ATOI(gtk_entry_get_text(GTK_ENTRY(p->minor_entry)));
	}
	else
	{
		gchar *s;

		edv_device_numbers_parse(
			edv_properties_list_get_i(
				properties_list,
				"device_type"
			),
			&major, &minor
		);

		s = g_strdup_printf("%i", major);
		gtk_entry_set_text(GTK_ENTRY(p->major_entry), s);
		g_free(s);

		s = g_strdup_printf("%i", minor);
		gtk_entry_set_text(GTK_ENTRY(p->minor_entry), s);
		g_free(s);
	}

	p->freeze_count--;
}
