#include <errno.h>
#include <sys/stat.h>
#include <gtk/gtk.h>

#include "cfg.h"

#include "guiutils.h"
#include "pie_chart.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_property.h"
#include "edv_pixmap.h"
#include "edv_device.h"
#include "edv_devices_list.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "edv_utils_gtk.h"
#include "edv_device_mount.h"
#include "device_edit_dlg.h"
#include "prop_page.h"
#include "prop_page_device.h"
#include "edv_emit.h"
#include "edv_op.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_mount_20x20.xpm"
#include "images/icon_unmount_20x20.xpm"
#include "images/icon_eject_20x20.xpm"

#include "images/icon_fsck_32x32.xpm"
#include "images/icon_tools_32x32.xpm"
#include "images/icon_floppy_32x32.xpm"


typedef struct _EDVDevicePropPage	EDVDevicePropPage;
#define EDV_DEVICE_PROP_PAGE(p)		((EDVDevicePropPage *)(p))


/*
 *	Flags:
 */
typedef enum {
	EDV_DEVICE_PROP_PAGE_HAS_CHANGES	\
					= (1 << 7)
} EDVDevicePropPageFlags;


/* Check Support */
gboolean edv_device_prop_page_query_create_cb(
	EDVPropPageContext *ctx,
	gint *version_major_rtn,
	gint *version_minor_rtn,
	gint *version_release_rtn,
	gchar **page_name_rtn,
	edv_pixmap_data **pixmap_data_20x20_rtn,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list
);

/* Create */
gpointer edv_device_prop_page_create_cb(
	EDVPropPageContext *ctx,
	GtkWidget *parent
);

/* Update */
void edv_device_prop_page_update_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list,
	const int error_code,
	gpointer data
);

/* Destroy */
void edv_device_prop_page_destroy_cb(
	EDVPropPageContext *ctx,
	gpointer data
);

/* Callbacks */
static void edv_device_prop_page_edit_device_cb(GtkWidget *widget, gpointer data);
static void edv_device_prop_page_mount_cb(GtkWidget *widget, gpointer data);
static void edv_device_prop_page_eject_cb(GtkWidget *widget, gpointer data);
static void edv_device_prop_page_run_check_cb(GtkWidget *widget, gpointer data);
static void edv_device_prop_page_run_tools_cb(GtkWidget *widget, gpointer data);
static void edv_device_prop_page_run_format_cb(GtkWidget *widget, gpointer data);

/* Operations */
static void edv_device_prop_page_edit_device(EDVDevicePropPage *p);

/* Update Widgets */
static void edv_device_prop_page_update_widgets(EDVDevicePropPage *p);


#define EDV_DEVICE_PROP_PAGE_NAME	"Device"

#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Device Properties Page:
 */
struct _EDVDevicePropPage {
	GtkWidget	*toplevel;
	gint		freeze_count;
	EDVPropPageContext	*ctx;
	EDVDevicePropPageFlags	flags;

	GtkWidget	*icon_pm,
			*edit_device_btn,
			*device_path_label,
			*mount_path_label,
			*fs_type_label,
			*usage_frame,
			*usage_vbox,
			*usage_pie_chart,
			*mount_btn,
			*eject_btn,
			*check_btn,
			*tools_btn,
			*format_btn;

	gint		dev_num;

	edv_device_edit_dlg_struct	*device_edit_dlg;
};


/*
 *	Check support callback.
 */
gboolean edv_device_prop_page_query_create_cb(
	EDVPropPageContext *ctx,
	gint *version_major_rtn,
	gint *version_minor_rtn,
	gint *version_release_rtn,
	gchar **page_name_rtn,
	edv_pixmap_data **pixmap_data_20x20_rtn,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list
)
{
	gint dev_num;
	const gchar *path;
	EDVDevice *dev;
	EDVCore *core = edv_prop_page_get_core(ctx);

	*version_major_rtn = PROG_VERSION_MAJOR;
	*version_minor_rtn = PROG_VERSION_MINOR;
	*version_release_rtn = PROG_VERSION_RELEASE;
	*page_name_rtn = g_strdup(EDV_DEVICE_PROP_PAGE_NAME);

	if(location_type != EDV_LOCATION_TYPE_VFS)
		return(FALSE);

	path = edv_properties_list_get_s(
		properties_list,
		EDV_PROP_NAME_PATH
	);
	if(path == NULL)
		return(FALSE);

	/* Refresh all the device mount states and device stats and
	 * then get the device path or mount path that matches the
	 * object
	 */
	edv_devices_list_update_mount_states(core->devices_list);
	edv_devices_list_update_statistics(core->devices_list);
	dev = edv_devices_list_match_mount_path(
		core->devices_list,
		&dev_num,
		path
	);
	if(dev == NULL)
		dev = edv_devices_list_match_device_path(
			core->devices_list,
			&dev_num,
			path
	   );
	if((dev == NULL) || (dev_num < 0))
		return(FALSE);

	return(TRUE);
}


/*
 *	Create callback.
 */
gpointer edv_device_prop_page_create_cb(
	EDVPropPageContext *ctx,
	GtkWidget *parent
)
{
	const gint	border_major = 5,
			border_minor = 2;
	gint font_size;
	const gchar *font_encoding;
	gchar *font_name_h1_bold;
	GdkFont *font;
	GtkRcStyle *rcstyle;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	GtkStyle *style = gtk_widget_get_style(toplevel);
	GtkWidget	*w,
			*parent2, *parent3, *parent4;
	EDVPixmap *icon = NULL;
	EDVMIMEType *m;
	EDVCore *core = edv_prop_page_get_core(ctx);
	EDVDevicePropPage *p = EDV_DEVICE_PROP_PAGE(g_malloc0(
		sizeof(EDVDevicePropPage)
	));
	if(p == NULL)
		return(NULL);

	p->ctx = ctx;
	p->toplevel = parent;
	p->dev_num = -1;

	p->freeze_count++;

	/* Get the base font size */
	font = style->font;
	font_size = GDK_FONT_GET_FONT_NAME_SIZE(font);
	if(font_size < 3)
		font_size = 3;

	/* Format the font names */
#if defined(PROG_LANGUAGE_POLISH)
	font_encoding = "iso8859-2";
#else
	font_encoding = "iso8859-1";
#endif

	font_name_h1_bold = g_strdup_printf(
"-adobe-helvetica-bold-r-normal-*-%i-*-*-*-*-*-%s",
		font_size + 2,
		font_encoding
	);

	/* Get a generic icon for initial display */
	m = edv_mime_types_list_match_type(
		core->mime_types_list,
		NULL,				/* No index return */
		edv_prop_page_get_mime_type_type(ctx),
		FALSE				/* Not case sensitive */
	);
	if(m == NULL)
		m = edv_mime_types_list_match_type(
			core->mime_types_list,
			NULL,				/* No index return */
			EDV_MIME_TYPE_TYPE_INODE_UNKNOWN,
			FALSE				/* Not case sensitive */
		);
	if(m != NULL)
	{
		edv_mime_type_realize(m, FALSE);
		(void)edv_pixmap_unref(icon);
		icon = edv_pixmap_ref(m->medium_icon[
			EDV_MIME_TYPE_ICON_STATE_STANDARD
		]);
	}

	/* Identification GtkFrame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Identificacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Identification"
#elif defined(PROG_LANGUAGE_GERMAN)
"Identifikation"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Identificazione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Identificatie"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Identificao"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Identification"
#else
"Identification"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_minor);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Device GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Icon */
	if(edv_pixmap_is_loaded(icon))
	{
		p->icon_pm = w = edv_pixmap_new_gtk_pixmap(icon);
		if(w != NULL)
		{
			gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
			gtk_widget_show(w);
		}
	}
	/* Name */
	p->device_path_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = STRDUP(font_name_h1_bold);
	gtk_widget_modify_style(w, rcstyle);
	GTK_RC_STYLE_UNREF(rcstyle);
	gtk_widget_show(w);

	/* GtkVBox for the Device Edit button */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_end(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Edit Device GtkButton */
	p->edit_device_btn = w = gtk_button_new_with_label(
		"Edit Device"
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_device_prop_page_edit_device_cb), p
	);
	GUISetWidgetTip(
		w,
"Edit the Device for this object"
	);
	gtk_widget_show(w);


	/* Mount Path GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Prefix GtkLabel */
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Monte Sendero:"
#elif defined(PROG_LANGUAGE_FRENCH)
"Point de montage:"
#elif defined(PROG_LANGUAGE_GERMAN)
"Stellen Sie Pfad Auf:"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Montare Il Sentiero:"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bestijg Pad:"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Monte Caminho:"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Monter Path:"
#else
"Mount Path:"
#endif
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Mount Path GtkLabel */
	p->mount_path_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Filesystem GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Prefix GtkLabel */
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Archive El De Sistema:"
#elif defined(PROG_LANGUAGE_FRENCH)
"Systme de fichier:"
#elif defined(PROG_LANGUAGE_GERMAN)
"Dateisystem:"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Schedare Il Di Sistema:"
#elif defined(PROG_LANGUAGE_DUTCH)
"Archiveer Systeem:"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Arquive De Sistema:"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Arkiver System:"
#else
"Filesystem:"
#endif
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Filesystem Type GtkLabel */
	p->fs_type_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Usage GtkFrame */
	p->usage_frame = w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"El Uso"
#elif defined(PROG_LANGUAGE_FRENCH)
"Usage"
#elif defined(PROG_LANGUAGE_GERMAN)
"Brauch"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Uso"
#elif defined(PROG_LANGUAGE_DUTCH)
"Gebruik"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Uso"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Praksis"
#else
"Usage"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;


	/* Services GtkFrame */
	w = gtk_frame_new("Services");
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_hbox_new(TRUE, 0);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;


	/* Check GtkButton */
	p->check_btn = w = GUIButtonPixmap(
		(guint8 **)icon_fsck_32x32_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_device_prop_page_run_check_cb), p
	);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Archive Cheque De Sistema"
#elif defined(PROG_LANGUAGE_FRENCH)
"Controler le systme de fichier"
#elif defined(PROG_LANGUAGE_GERMAN)
"Dateisystem Kontrolle"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Schedare L'Assegno Di Sistema"
#elif defined(PROG_LANGUAGE_DUTCH)
"Archiveer Systeemcontrole"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Arquive Sistema Verificar"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Arkiver Systemkontroll"
#else
"File System Check"
#endif
	);
	gtk_widget_show(w);

	/* Tools GtkButton */
	p->tools_btn = w = GUIButtonPixmap(
		(guint8 **)icon_tools_32x32_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_device_prop_page_run_tools_cb), p
	);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Las Herramientas"
#elif defined(PROG_LANGUAGE_FRENCH)
"Outils"
#elif defined(PROG_LANGUAGE_GERMAN)
"Werkzeuge"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Gli Attrezzi"
#elif defined(PROG_LANGUAGE_DUTCH)
"Werktuigen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"As Ferramentas"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Redskaper"
#else
"Tools"
#endif
	);
	gtk_widget_show(w);

	/* Format GtkButton */
	p->format_btn = w = GUIButtonPixmap(
		(guint8 **)icon_floppy_32x32_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_device_prop_page_run_format_cb), p
	);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Formatear Medios"
#elif defined(PROG_LANGUAGE_FRENCH)
"Formatter"
#elif defined(PROG_LANGUAGE_GERMAN)
"Formatieren Sie Medien"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Stampa Di Formato"
#elif defined(PROG_LANGUAGE_DUTCH)
"Formatteer Media"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Imprensa De Formato"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Formater Media"
#else
"Format Media"
#endif
	);
	gtk_widget_show(w);


	(void)edv_pixmap_unref(icon);
	g_free(font_name_h1_bold);

	p->freeze_count--;

	return(p);
}


/*
 *	Update callback.
 */
void edv_device_prop_page_update_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list,
	const int error_code,
	gpointer data
)
{
	EDVDevicePropPage *p = EDV_DEVICE_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	edv_device_prop_page_update_widgets(p);

	p->freeze_count--;
}

/*
 *	Destroy callback.
 */
void edv_device_prop_page_destroy_cb(
	EDVPropPageContext *ctx,
	gpointer data
)
{
	EDVDevicePropPage *p = EDV_DEVICE_PROP_PAGE(data);

	p->freeze_count++;

	EDVDeviceEditDlgDelete(p->device_edit_dlg);

	p->freeze_count--;

	g_free(p);
}


/*
 *	Edit Device callback.
 */
static void edv_device_prop_page_edit_device_cb(GtkWidget *widget, gpointer data)
{
	EDVDevicePropPage *p = EDV_DEVICE_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	edv_device_prop_page_edit_device(p);

	p->freeze_count--;
}


/*
 *	Mount/Unmount Device callback.
 */
static void edv_device_prop_page_mount_cb(GtkWidget *widget, gpointer data)
{
	gboolean original_mount_state;
	gint status;
	EDVDevice *dev;
	EDVDevicePropPage *p = EDV_DEVICE_PROP_PAGE(data);
	const gint dev_num = p->dev_num;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));
	if(dev == NULL)
	{
		p->freeze_count--;
		return;
	}

	edv_prop_page_set_busy(ctx, TRUE);

	original_mount_state = EDV_DEVICE_IS_MOUNTED(dev);

	/* Unmount or mount? */
 	if(EDV_DEVICE_IS_MOUNTED(dev))
		status = edv_device_unmount(
			core,
			dev,
			TRUE,			/* Show progress */
			TRUE,			/* Verbose */
			toplevel
		);
	else
		status = edv_device_mount(
			core,
			dev,
			TRUE,			/* Show progress */
			TRUE,			/* Verbose */
			toplevel
		);

	/* Update all the devices' statistics */
	edv_devices_list_update_statistics(core->devices_list);

	/* Mount/unmount error? */
	if(status != 0)
	{
		/* Get the error message (if any) */
		const gchar *last_error = edv_device_mount_get_error(core);
		if(!STRISEMPTY(last_error))
		{
			edv_play_sound_error(core);
			edv_message_error(
				original_mount_state ?
					"Unmount Failed" : "Mount Failed",
				last_error,
				NULL,
				toplevel
			);
		}
	}
	else
	{
		/* Notify about this object being mounted/unmounted */
		edv_emit_device_mount(
			core,
			dev_num, dev,
			EDV_DEVICE_IS_MOUNTED(dev)
		);

		/* Since this call freezes our page, we need to
		 * explicitly update the values displayed on our
		 * page
		 */
		edv_device_prop_page_update_widgets(p);

/* Warning, some GtkWidgets may be invalid at this point if fetched
 * earlier in this function
 */
	}

	edv_prop_page_set_busy(ctx, FALSE);

	p->freeze_count--;
}

/*
 *	Eject Device callback.
 */
static void edv_device_prop_page_eject_cb(GtkWidget *widget, gpointer data)
{
	gboolean original_mount_state;
	gint status;
	EDVDevice *dev;
	EDVDevicePropPage *p = EDV_DEVICE_PROP_PAGE(data);
	const gint dev_num = p->dev_num;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));
	if(dev == NULL)
	{
		p->freeze_count--;
		return;
	}

	edv_prop_page_set_busy(ctx, TRUE);

	original_mount_state = EDV_DEVICE_IS_MOUNTED(dev);

	/* Need to unmount first? */
	if(EDV_DEVICE_IS_MOUNTED(dev))
		edv_device_unmount(
			core,
			dev,
			TRUE,
			TRUE,
			toplevel
		);

	/* Eject */
	status = edv_device_eject(
		core,
		dev,
		TRUE,
		TRUE,
		toplevel
	);

	/* Update all device mount states and stats */
	edv_devices_list_update_mount_states(core->devices_list);
	edv_devices_list_update_statistics(core->devices_list);

	/* Eject error? */
	if(status != 0)
	{
		/* Get the error message (if any) */
		const gchar *last_error = edv_device_mount_get_error(core);
		if(!STRISEMPTY(last_error))
		{
			edv_play_sound_error(core);
			edv_message_error(
				"Eject Failed",
				last_error,
				NULL,
				toplevel
			);
		}
	}
	else
	{   
		/* Notify about this object being ejected */
		edv_emit_device_mount(
			core,
			dev_num,
			dev,
			EDV_DEVICE_IS_MOUNTED(dev)
		);

		/* Since this call freezes our page, we need to
		 * explicitly update the values displayed on our
		 * page
		 */
		edv_device_prop_page_update_widgets(p);
	}

	edv_prop_page_set_busy(ctx, FALSE);

	p->freeze_count--;
}

/*
 *      Runs the file system check command.
 */
static void edv_device_prop_page_run_check_cb(GtkWidget *widget, gpointer data)
{
	EDVDevice *dev;
	EDVDevicePropPage *p = EDV_DEVICE_PROP_PAGE(data);
	const gint dev_num = p->dev_num;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));
	if(dev == NULL)
	{
		p->freeze_count--;
		return;
	}

	edv_prop_page_set_busy(ctx, TRUE);

	edv_run_device_check(
		core,
		dev,
		toplevel
	);

	edv_prop_page_set_busy(ctx, FALSE);

	p->freeze_count--;
}

/*
 *	Runs the device tools command.
 */
static void edv_device_prop_page_run_tools_cb(GtkWidget *widget, gpointer data)
{
	EDVDevice *dev;
	EDVDevicePropPage *p = EDV_DEVICE_PROP_PAGE(data);
	const gint dev_num = p->dev_num;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));
	if(dev == NULL)
	{
		p->freeze_count--;
		return;
	}

	edv_prop_page_set_busy(ctx, TRUE);

	edv_run_device_tools(
		core,
		dev,
		toplevel
	);

	edv_prop_page_set_busy(ctx, FALSE);

	p->freeze_count--;
}

/*
 *      Runs the device format command.
 */
static void edv_device_prop_page_run_format_cb(GtkWidget *widget, gpointer data)
{
	EDVDevice *dev;
	EDVDevicePropPage *p = EDV_DEVICE_PROP_PAGE(data);
	const gint dev_num = p->dev_num;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));
	if(dev == NULL) 
	{
		p->freeze_count--;
		return;
	}

	edv_prop_page_set_busy(ctx, TRUE);

	edv_run_device_format(
		core,
		dev,
		toplevel
	);

	edv_prop_page_set_busy(ctx, FALSE);

	p->freeze_count--;
}


/*
 *	Edit Device.
 */
static void edv_device_prop_page_edit_device(EDVDevicePropPage *p)
{
	edv_device_edit_dlg_struct *device_edit_dlg;
	const gint dev_num = p->dev_num;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);

	p->freeze_count++;

	edv_prop_page_set_busy(ctx, TRUE);

	/* Create the Device Edit Dialog as needed */
	device_edit_dlg = p->device_edit_dlg;
	if(device_edit_dlg == NULL)
		p->device_edit_dlg = device_edit_dlg = EDVDeviceEditDlgNew(core);
	if(device_edit_dlg != NULL)
	{
		edv_center_window_to_window(
			toplevel,
			device_edit_dlg->toplevel
		);
		EDVDeviceEditDlgMap(device_edit_dlg);
		EDVDeviceEditDlgGetValues(
			device_edit_dlg,
			dev_num
		);
		EDVDeviceEditDlgResetHasChanges(
			device_edit_dlg,
			FALSE
		);
	}

	edv_prop_page_set_busy(ctx, FALSE);

	p->freeze_count--;
}


/*
 *      Updates the values displayed on the GtkWidgets.
 */
static void edv_device_prop_page_update_widgets(EDVDevicePropPage *p)
{
	const gint	border_major = 5,
			border_minor = 2;
	gint		font_size,
			dev_num;
	const gchar	*path,
			*font_encoding;
	gchar *font_name_h1_bold;
	GdkFont *font;
	GtkRcStyle *standard_rcstyle;
	EDVPropPageContext *ctx = p->ctx;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	GtkStyle *style = gtk_widget_get_style(toplevel);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);
	EDVCore *core = edv_prop_page_get_core(ctx);
	CfgList *cfg_list = edv_prop_page_get_cfg_list(ctx);
	GtkWidget	*w,
			*parent, *parent2, *parent3;
	EDVDevice *dev;

	p->freeze_count++;

	/* Get the path */
	path = edv_properties_list_get_s(
		properties_list,
		EDV_PROP_NAME_PATH
	);

	/* Refresh all the device mount states and device stats and
	 * then get the device path or mount path that matches the
	 * object
	 */
	edv_devices_list_update_mount_states(core->devices_list);
	edv_devices_list_update_statistics(core->devices_list);
	dev = edv_devices_list_match_mount_path(
		core->devices_list,
		&dev_num,
		path
	);
	if(dev == NULL)
		dev = edv_devices_list_match_device_path(
			core->devices_list,
			&dev_num,
			path
	   );
	if((dev == NULL) || (dev_num < 0))
		return;

	/* Update the device number on the page */
	p->dev_num = dev_num;

	/* Update the icon */
	w = p->icon_pm;
	if(w != NULL)
	{
		const gint n = EDV_DEVICE_TOTAL_ICON_STATES;
		gint i;
		EDVPixmap *icon = NULL;

		edv_device_realize(dev, FALSE);

		for(i = 0; i < n; i++)
		{
			if(edv_pixmap_is_loaded(icon))
				break;

			icon = edv_pixmap_ref(dev->medium_icon[i]);
		}
		for(i = 0; i < n; i++)
		{
			if(edv_pixmap_is_loaded(icon))
				break;

			icon = edv_pixmap_ref(dev->large_icon[i]);
		}
		for(i = 0; i < n; i++)
		{
			if(edv_pixmap_is_loaded(icon))
				break;

			icon = edv_pixmap_ref(dev->small_icon[i]);
		}
		if(edv_pixmap_is_loaded(icon))
		{
			edv_pixmap_set_gtk_pixmap(icon, w);
			(void)edv_pixmap_unref(icon);
		}
	}

	/* Update the device path */
	w = p->device_path_label;
	if(w != NULL)
	{
		gchar *s;
		if(dev->name != NULL)
		{
			if(dev->device_path != NULL)
				s = g_strconcat(
					dev->name,
					" (",
					dev->device_path,
					")",
					NULL
				);
			else
				s = g_strdup(dev->name);
		}
		else if(dev->device_path != NULL)
		{
			s = g_strdup(dev->device_path);
		}
		else
		{
			s = g_strdup("");
		}
		gtk_label_set_text(GTK_LABEL(w), (s != NULL) ? s : "");
		g_free(s);
	}

	/* Update the mount path */
	w = p->mount_path_label;
	if(w != NULL)
	{
		const gchar *s = dev->mount_path;
		gtk_label_set_text(GTK_LABEL(w), (s != NULL) ? s : "");
	}

	/* Update the filesystem type */
	w = p->fs_type_label;
	if(w != NULL)
	{
		const gchar *s = dev->fs_type_name;
		gtk_label_set_text(GTK_LABEL(w), (s != NULL) ? s : "");
	}

	GTK_WIDGET_SET_SENSITIVE(
		p->check_btn,
		STRISEMPTY(dev->command_check) ? FALSE : TRUE
	);

	GTK_WIDGET_SET_SENSITIVE(
		p->tools_btn,
		STRISEMPTY(dev->command_tools) ? FALSE : TRUE
	);

	GTK_WIDGET_SET_SENSITIVE(
		p->format_btn,
		STRISEMPTY(dev->command_format) ? FALSE : TRUE
	);


	/* Begin recreating the device usage GtkWidgets
	 *
	 * Destroy any existing device usage GtkWidgets
	 */
	GTK_WIDGET_DESTROY(p->usage_pie_chart);
	p->usage_pie_chart = NULL;
	GTK_WIDGET_DESTROY(p->mount_btn);
	p->mount_btn = NULL;
	GTK_WIDGET_DESTROY(p->eject_btn);
	p->eject_btn = NULL;
	GTK_WIDGET_DESTROY(p->usage_vbox);
	p->usage_vbox = NULL;

	standard_rcstyle = core->standard_rcstyle;

	/* Get the GtkContainer that will be the parent for the
	 * GtkWidgets that we are about to recreate
	 */
	parent = p->usage_frame;

	/* Get the base font size */
	font = style->font;
	font_size = GDK_FONT_GET_FONT_NAME_SIZE(font);
	if(font_size < 3)
		font_size = 3;

	/* Format the font names */
#if defined(PROG_LANGUAGE_POLISH)
	font_encoding = "iso8859-2";
#else
	font_encoding = "iso8859-1";
#endif

	font_name_h1_bold = g_strdup_printf(
"-adobe-helvetica-bold-r-normal-*-%i-*-*-*-*-*-%s",
		font_size + 2,
		font_encoding
	);

	/* Usage GtkVBox */
	p->usage_vbox = w = gtk_vbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent = w;


	/* Create the Usage Pie Chart if mounted */
	if(EDV_DEVICE_IS_MOUNTED(dev))
	{
		const gulong	adj_division = 10l,
							block_size = EDV_GET_UL(EDV_CFG_PARM_BLOCK_SIZE),
							total_kb = dev->blocks_total,
							free_kb = dev->blocks_free,
							available_kb = dev->blocks_available,
							used_kb = MAX(total_kb - free_kb, 0l);
		gchar *ss_kb, *ss_mb, *text, *text2;
		GdkColor c;
		GtkAdjustment *adj;

		/* GtkHBox for multiple columns */
		w = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent2 = w;

		/* GtkHBox to center the Usage Pie Chart */
		w = gtk_hbox_new(TRUE, 0);
		gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
		gtk_widget_show(w);
		parent3 = w;


		adj = (GtkAdjustment *)gtk_adjustment_new(
			0.0f, 0.0f,
			(gfloat)(total_kb / adj_division),
			0.0f, 0.0f, 0.0f
		);
		GDK_COLOR_SET_COEFF(&c, 0.0f, 0.0f, 1.0f);
		ss_kb = STRDUP(edv_str_size_delim(total_kb));
		ss_mb = STRDUP(edv_str_size_delim(
			(block_size > 0l) ?
				(total_kb / block_size) : (total_kb / 1024l)
		));
		text = g_strdup_printf(
			"%s: %s kb (%s mb)",
#if defined(PROG_LANGUAGE_SPANISH)
"La Capacidad Total"
#elif defined(PROG_LANGUAGE_FRENCH)
"Capacit Totale"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gesamte Kapazitt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Capacit Totale"
#elif defined(PROG_LANGUAGE_DUTCH)
"Totale Capaciteit"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Capacidade Total"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Total Capacity"
#else
"Total Capacity"
#endif
			, ss_kb, ss_mb   
		);
		g_free(ss_kb);
		g_free(ss_mb);

		ss_kb = STRDUP(edv_str_size_delim(free_kb));
		ss_mb = STRDUP(edv_str_size_delim(
			(block_size > 0l) ?
				(free_kb / block_size) : (free_kb / 1024l)
		));
		text2 = g_strdup_printf(
			"%s kb (%s mb)",
			ss_kb, ss_mb
		);
		g_free(ss_kb);
		g_free(ss_mb);

		p->usage_pie_chart = w = PieChartNew(
			(EDV_GET_B(EDV_CFG_PARM_LISTS_DOUBLE_BUFFER) ?
				PIE_CHART_DOUBLE_BUFFER : 0) |
				PIE_CHART_SHOW_LABELS |
				PIE_CHART_SHOW_SHADOW |
				PIE_CHART_SHOW_OUTLINE,
			110, 70,
			adj,
			&c,
			NULL,
			text,
#if defined(PROG_LANGUAGE_SPANISH)
"Libre:"
#elif defined(PROG_LANGUAGE_FRENCH)
"Libre:"
#elif defined(PROG_LANGUAGE_GERMAN)
"Frei:"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Libero:"
#elif defined(PROG_LANGUAGE_DUTCH)
"Vrij:"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Livre:"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Fri:"
#else
"Free:"
#endif
			,
			text2
		);
		adj = (GtkAdjustment *)GTK_OBJECT_UNREF(GTK_OBJECT(adj));
		g_free(text);
		g_free(text2);
		gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
		gtk_widget_show(w);

		GDK_COLOR_SET_COEFF(&c, 0.0f, 1.0f, 1.0f);
		adj = (GtkAdjustment *)gtk_adjustment_new(
			0.0f, 0.0f,
			(gfloat)(available_kb / adj_division),
			0.0f, 0.0f, 0.0f
		);
		ss_kb = STRDUP(edv_str_size_delim(available_kb));
		ss_mb = STRDUP(edv_str_size_delim(
			(block_size > 0l) ?
				(available_kb / block_size) : (available_kb / 1024l)
		));
		text = g_strdup_printf(
			"%s kb (%s mb)",
			ss_kb, ss_mb
		);
		g_free(ss_kb);
		g_free(ss_mb);
		PieChartValueAdd(
			w,
			adj,
			&c,
#if defined(PROG_LANGUAGE_SPANISH)
"Libre & Disponible:"
#elif defined(PROG_LANGUAGE_FRENCH)
"Libre & Disponible:"
#elif defined(PROG_LANGUAGE_GERMAN)
"Frei & Verfgbar:"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Libero & Disponibile:"                           
#elif defined(PROG_LANGUAGE_DUTCH)
"Vrij & Verkrijgbare:"              
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Livre & Disponvel:"            
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Fri & Available:"             
#else           
"Free & Available:"         
#endif
			,
			text
		);
		adj = (GtkAdjustment *)GTK_OBJECT_UNREF(GTK_OBJECT(adj));
		g_free(text);

		GDK_COLOR_SET_COEFF(&c, 1.0f, 0.0f, 1.0f);
		adj = (GtkAdjustment *)gtk_adjustment_new(
			0.0f, 0.0f,
			(gfloat)(used_kb / adj_division),
			0.0f, 0.0f, 0.0f
		);
		ss_kb = STRDUP(edv_str_size_delim(used_kb));
		ss_mb = STRDUP(edv_str_size_delim(
			(block_size > 0l) ?
				(used_kb / block_size) : (used_kb / 1024l)
		));
		text = g_strdup_printf(
			"%s kb (%s mb)",
			ss_kb, ss_mb
		);
		g_free(ss_kb);
		g_free(ss_mb);
		PieChartValueAdd(
			w,
			adj,
			&c,
#if defined(PROG_LANGUAGE_SPANISH)
"Usado:"
#elif defined(PROG_LANGUAGE_FRENCH)
"Utilis:"
#elif defined(PROG_LANGUAGE_GERMAN)
"Benutzt:"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Usato:"
#elif defined(PROG_LANGUAGE_DUTCH)
"Gebruikt:"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Usado:"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Brukt:"
#else           
"Used:"
#endif
			,
			text 
		);
		adj = (GtkAdjustment *)GTK_OBJECT_UNREF(GTK_OBJECT(adj));
		g_free(text);


		/* Right column vbox for buttons */
		w = gtk_vbox_new(FALSE, border_minor);
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent3 = w;

		/* Unmount button */
		p->mount_btn = w = GUIButtonPixmap(
			(guint8 **)icon_unmount_20x20_xpm
		);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_signal_connect(
			GTK_OBJECT(w), "clicked",
			GTK_SIGNAL_FUNC(edv_device_prop_page_mount_cb), p
		);
		GUISetWidgetTip(w, "Unmount");
		GTK_WIDGET_SET_SENSITIVE(w, !EDV_DEVICE_IS_NO_UNMOUNT(dev));
		if(standard_rcstyle != NULL)
			gtk_widget_modify_style_recursive(w, standard_rcstyle);
		gtk_widget_show(w);

		/* Eject button */
		p->eject_btn = w = GUIButtonPixmap(
			(guint8 **)icon_eject_20x20_xpm
		);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_signal_connect(
			GTK_OBJECT(w), "clicked",
			GTK_SIGNAL_FUNC(edv_device_prop_page_eject_cb), p
		);
		GUISetWidgetTip(w, "Eject");
		GTK_WIDGET_SET_SENSITIVE(
			w,
			!STRISEMPTY(dev->command_eject) ? TRUE : FALSE
		);
		if(standard_rcstyle != NULL)
			gtk_widget_modify_style_recursive(w, standard_rcstyle);
		gtk_widget_show(w);
	}
	else
	{
		GtkRcStyle *rcstyle;

		w = gtk_hbox_new(TRUE, 0);
		gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent2 = w;

		/* Device not mounted label */
		w = gtk_label_new("(Device Not Mounted)");
		gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_CENTER);
		gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
		rcstyle = gtk_rc_style_new();
		rcstyle->font_name = STRDUP(font_name_h1_bold);
		gtk_widget_modify_style(w, rcstyle);
		GTK_RC_STYLE_UNREF(rcstyle);
		gtk_widget_show(w);

		w = gtk_hbox_new(TRUE, 0);
		gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent2 = w;

		/* Mount button */
		p->mount_btn = w = GUIButtonPixmapLabelH(
			(guint8 **)icon_mount_20x20_xpm, "Mount", NULL
		);
		gtk_widget_set_usize(
			w,
			GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
		);
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_signal_connect(
			GTK_OBJECT(w), "clicked",
			GTK_SIGNAL_FUNC(edv_device_prop_page_mount_cb), p
		);
		GUISetWidgetTip(w, "Click to mount the device");
		GTK_WIDGET_SET_SENSITIVE(w, !EDV_DEVICE_IS_NO_UNMOUNT(dev));
		if(standard_rcstyle != NULL)
			gtk_widget_modify_style_recursive(w, standard_rcstyle);
		gtk_widget_show(w);

		/* Eject button */
		p->eject_btn = w = GUIButtonPixmapLabelH(
			(guint8 **)icon_eject_20x20_xpm, "Eject", NULL
		);
		gtk_widget_set_usize(
			w,
			GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
		);
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_signal_connect(
			GTK_OBJECT(w), "clicked",
			GTK_SIGNAL_FUNC(edv_device_prop_page_eject_cb), p
		);
		GUISetWidgetTip(w, "Click to eject the media from the device");
		GTK_WIDGET_SET_SENSITIVE(
			w,
			!STRISEMPTY(dev->command_eject) ? TRUE : FALSE
		);
		if(standard_rcstyle != NULL)
			gtk_widget_modify_style_recursive(w, standard_rcstyle);
		gtk_widget_show(w);
	}


	g_free(font_name_h1_bold);


	p->freeze_count--;
}
