#include <stdlib.h>
#include <errno.h>
#include <gtk/gtk.h>

#include "cfg.h"

#include "guiutils.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_property.h"
#include "libendeavour2-base/edv_fs_type.h"
#include "libendeavour2-base/edv_recycle_bin_index.h"
#include "libendeavour2-base/edv_id.h"
#include "edv_ids_list.h"
#include "edv_date_format.h"
#include "edv_pixmap.h"
#include "edv_mime_type.h"
#include "edv_device.h"
#include "edv_devices_list.h"
#include "edv_utils_gtk.h"
#include "edv_list_cb.h"
#include "prop_page.h"
#include "prop_page_details.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


typedef struct _EDVDetailsPropPage	EDVDetailsPropPage;
#define EDV_DETAILS_PROP_PAGE(p)	((EDVDetailsPropPage *)(p))


/* Check Support */
gboolean edv_details_prop_page_query_create_cb(
	EDVPropPageContext *ctx,
	gint *version_major_rtn,
	gint *version_minor_rtn,
	gint *version_release_rtn,
	gchar **page_name_rtn,
	edv_pixmap_data **pixmap_data_20x20_rtn,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list
);

/* Create */
gpointer edv_details_prop_page_create_cb(
	EDVPropPageContext *ctx,
	GtkWidget *parent
);

/* Update */
void edv_details_prop_page_update_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list,
	const int error_code,
	gpointer data
);

/* Destroy */
void edv_details_prop_page_destroy_cb(
	EDVPropPageContext *ctx,
	gpointer data
);

/* Callbacks */
static void edv_details_prop_page_verbose_cb(GtkWidget *widget, gpointer data);
static void edv_details_prop_page_raw_cb(GtkWidget *widget, gpointer data);
static void edv_details_prop_page_select_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);


#define EDV_DETAILS_PROP_PAGE_NAME	"Details"

#define EDV_DETAILS_PROP_PAGE_VALUE_NOT_AVAILABLE_STRING	\
					"*Not Available*"
#define EDV_DETAILS_PROP_PAGE_TIME_STAMP_NOT_SET_STRING	\
					"*Not Set*"


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Details Properties Page:
 */
struct _EDVDetailsPropPage {
	GtkWidget	*toplevel;
	gint		freeze_count;
	EDVPropPageContext	*ctx;

	GtkWidget	*verbose_radio,
			*raw_radio,
			*clist;
};


/*
 *	Check support callback.
 */
gboolean edv_details_prop_page_query_create_cb(
	EDVPropPageContext *ctx,
	gint *version_major_rtn,
	gint *version_minor_rtn,
	gint *version_release_rtn,
	gchar **page_name_rtn,
	edv_pixmap_data **pixmap_data_20x20_rtn,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list
)
{
	/* Always create the Details Properties Page */
	*version_major_rtn = PROG_VERSION_MAJOR;
	*version_minor_rtn = PROG_VERSION_MINOR;
	*version_release_rtn = PROG_VERSION_RELEASE;
	*page_name_rtn = g_strdup(EDV_DETAILS_PROP_PAGE_NAME);
	return(TRUE);
}

/*
 *	Create callback.
 */
gpointer edv_details_prop_page_create_cb(
	EDVPropPageContext *ctx,
	GtkWidget *parent
)
{
	const gint	border_major = 5,
			border_minor = 2;
	gchar *heading[2];
	GSList *gslist;
	GtkWidget	*w,
			*parent2, *parent3;
	GtkCList *clist;
	EDVCore *core = edv_prop_page_get_core(ctx);
	EDVDetailsPropPage *p = EDV_DETAILS_PROP_PAGE(g_malloc0(
		sizeof(EDVDetailsPropPage)
	));
	if(p == NULL)
		return(NULL);

	p->ctx = ctx;
	p->toplevel = parent;

	p->freeze_count++;

	/* Details GtkFrame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Los Detalles"
#elif defined(PROG_LANGUAGE_FRENCH)
"Dtails"
#elif defined(PROG_LANGUAGE_GERMAN)
"Statistik"
#elif defined(PROG_LANGUAGE_ITALIAN)
"I Dettagli"
#elif defined(PROG_LANGUAGE_DUTCH)
"Statistieken"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Os Detalhes"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Detaljer"
#else
"Details"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Radios GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_end(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"El Despliegue"
#elif defined(PROG_LANGUAGE_FRENCH)
"Affichage"
#elif defined(PROG_LANGUAGE_GERMAN)
"Ausstellung"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Mostra"
#elif defined(PROG_LANGUAGE_DUTCH)
"Tentoonstelling"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Exposio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Utstilling"
#else
"Display"
#endif
		":"
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Verbose Radio */
	gslist = NULL;
	p->verbose_radio = w = gtk_radio_button_new_with_label(
		gslist,
#if defined(PROG_LANGUAGE_SPANISH)
"Detallado"
#elif defined(PROG_LANGUAGE_FRENCH)
"Dtaill"
#elif defined(PROG_LANGUAGE_GERMAN)
"Verbos"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Dettagliato"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verbose"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Verbose"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Verbose"
#else
"Verbose"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(edv_details_prop_page_verbose_cb), p
	);
	gtk_widget_show(w);
	gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));

	/* Raw Radio */
	p->raw_radio = w = gtk_radio_button_new_with_label(
		gslist,
#if defined(PROG_LANGUAGE_SPANISH)
"Crudo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Brute"
#elif defined(PROG_LANGUAGE_GERMAN)
"Roh"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Crudo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Rauw"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Cru"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"R"
#else
"Raw"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(edv_details_prop_page_raw_cb), p
	);
	gtk_widget_show(w);


	/* Create the GtkScrolledWindow for the GtkCList */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
		GTK_SCROLLED_WINDOW(w),
		GTK_POLICY_AUTOMATIC,
		GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Details GtkCList */
	heading[0] = "Property";
	heading[1] = "Value";
	p->clist = w = gtk_clist_new_with_titles(
		sizeof(heading) / sizeof(gchar *),
		heading
	);
	clist = GTK_CLIST(w);
	gtk_widget_add_events(
		w,
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "motion_notify_event",
		GTK_SIGNAL_FUNC(edv_clist_motion_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "select_row",
		GTK_SIGNAL_FUNC(edv_details_prop_page_select_row_cb), p
	);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_realize(w);
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_clist_column_titles_show(clist);
	gtk_clist_column_titles_passive(clist);
	gtk_clist_set_column_justification(
		clist, 0, GTK_JUSTIFY_RIGHT
	);
	gtk_clist_set_column_auto_resize(clist, 0, TRUE);
	gtk_clist_set_column_resizeable(clist, 0, TRUE);
	gtk_clist_set_column_justification(
		clist, 1, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_column_auto_resize(clist, 1, TRUE);
	gtk_clist_set_column_resizeable(clist, 1, TRUE);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_SINGLE);
	gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
	gtk_widget_show(w);


	/* Set the initial value to verbose */
	gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(p->verbose_radio),
		TRUE
	);

	p->freeze_count--;

	return(p);
}


/*
 *	Update callback.
 */
void edv_details_prop_page_update_cb(
	EDVPropPageContext *ctx,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GList *properties_list,
	const int error_code,
	gpointer data
)
{
	const gint border_minor = 2;
	gboolean verbose;
	gint		i,
			row,
			value_pixmap_width, value_pixmap_height;
	gulong block_size;
	gfloat last_x, last_y;
	gchar		*s,
			**strv;
	const gchar	*path,
			*date_format;
	GtkWidget *toplevel = edv_prop_page_get_toplevel(ctx);
	EDVDateRelativity date_relativity;
	EDVCore *core = edv_prop_page_get_core(ctx);
	CfgList *cfg_list = edv_prop_page_get_cfg_list(ctx);
	GList *meta_data_list = edv_prop_page_get_meta_data_list(ctx);
	EDVDetailsPropPage *p = EDV_DETAILS_PROP_PAGE(data);
	GtkWidget *w = p->clist;
	GtkCList *clist = GTK_CLIST(w);
	GtkStyle *style = gtk_widget_get_style(w);
	const gint ncolumns = MAX(clist->columns, 1);

	p->freeze_count++;

	/* Record the last scroll positions */
	last_x = GTK_ADJUSTMENT_GET_VALUE(clist->hadjustment);
	last_y = GTK_ADJUSTMENT_GET_VALUE(clist->vadjustment);

	/* Calculate the prefered size of the value pixmap */
	value_pixmap_width = 120;
	value_pixmap_height = GTK_CLIST_ROW_HEIGHT_SET(clist) ?
		(clist->row_height - 4) : -1;

	/* Begin updating */
	gtk_clist_freeze(clist);

	/* Clear any existing details */
	gtk_clist_clear(clist);

	/* Get the display options */
	date_relativity = (EDVDateRelativity)EDV_GET_I(
		EDV_CFG_PARM_DATE_RELATIVITY
	);
	date_format = EDV_GET_S(EDV_CFG_PARM_DATE_FORMAT);
	block_size = EDV_GET_L(EDV_CFG_PARM_BLOCK_SIZE);
	if(block_size == 0l)
		block_size = 1024l;

	/* Get the verbose state */
	verbose = GTK_TOGGLE_BUTTON_GET_ACTIVE(p->verbose_radio);

	/* Allocate the row cell values */
	strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
		strv[i] = "";

#define APPEND_LINE(_prop_,_val_)	{		\
 if(ncolumns >= 2) {					\
  /* Copy the value and shorten it as needed */		\
  gchar *dval = STRDUP(_val_);				\
  if(dval != NULL) {					\
   const gint max_value_length = 80;			\
   if(strlen(dval) > max_value_length)			\
    dval[max_value_length] = '\0';			\
  }							\
  strv[0] = ((_prop_) != NULL) ? (_prop_) : "";		\
  strv[1] = (dval != NULL) ? dval : "";			\
  row = gtk_clist_append(clist, strv);			\
  g_free(dval);						\
  gtk_clist_set_selectable(clist, row, FALSE);		\
  gtk_clist_set_shift(					\
   clist, row, 0, 0, -border_minor			\
  );							\
  gtk_clist_set_shift(					\
   clist, row, 1, 0, 0					\
  );							\
 } else {						\
  row = -1;						\
 }							\
}
#define APPEND_LINE_PIXMAP(_prop_,_pixmap_,_mask_) {	\
 if(ncolumns >= 2) {					\
  strv[0] = ((_prop_) != NULL) ? (_prop_) : "";		\
  strv[1] = "";						\
  row = gtk_clist_append(clist, strv);			\
  if((_pixmap_) != NULL)				\
   gtk_clist_set_pixmap(				\
    clist, row, 1,					\
    (_pixmap_), (_mask_)				\
   );							\
  gtk_clist_set_selectable(clist, row, FALSE);		\
  gtk_clist_set_shift(					\
   clist, row, 0, 0, -border_minor			\
  );							\
  gtk_clist_set_shift(					\
   clist, row, 1, 0, 0					\
  );							\
 } else {						\
  row = -1;						\
 }							\
}

	/* Add the lines based on the location type */
	switch(location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
		path = edv_properties_list_get_s(
			properties_list,
			EDV_PROP_NAME_PATH
		);
		if(path != NULL)
		{
			/* Get the object's type and permissions */
			const EDVPermissionFlags permissions = (EDVPermissionFlags)edv_properties_list_get_i(
				properties_list,
				EDV_PROP_NAME_PERMISSIONS
			);

			/* Get the device that this object is on */
			gint dev_num;
			EDVDevice *dev = edv_devices_list_match_object(
				core->devices_list,
				&dev_num,
				path
			);

			/* Name */
			s = STRDUP(edv_properties_list_get_s(
				properties_list,
				EDV_PROP_NAME_NAME
			));
			APPEND_LINE("Name", s);
			g_free(s);

			/* Location */
			s = g_dirname(path);
			if(s == NULL)
				s = g_strdup(EDV_DETAILS_PROP_PAGE_VALUE_NOT_AVAILABLE_STRING);
			APPEND_LINE("Location", s);
			g_free(s);

			/* Device */
			if(verbose)
			{
				if(dev != NULL)
				{
					if(!STRISEMPTY(dev->name))
						s = g_strdup(dev->name);
					else if(!STRISEMPTY(dev->device_path))
						s = g_strdup(dev->device_path);
					else
						s = g_strdup_printf(
							"#%ld",
							edv_properties_list_get_ul(
								properties_list,
								EDV_PROP_NAME_DEVICE_INDEX
							)
						);
				}
				else
				{
					s = g_strdup_printf(
						"#%ld",
						edv_properties_list_get_ul(
							properties_list,
							EDV_PROP_NAME_DEVICE_INDEX
						)
					);
				}
			}
			else
				s = g_strdup_printf(
					"%ld",
					edv_properties_list_get_ul(
						properties_list,
						EDV_PROP_NAME_DEVICE_INDEX
					)
				);
			APPEND_LINE("Device", s);
			g_free(s);

			/* Filesystem */
			if(dev != NULL)
			{
				if(verbose)
				{
					if(dev->fs_type_code != EDV_FS_TYPE_EMPTY)
						s = g_strdup_printf(
							"%s (0x%.8X)",
							dev->fs_type_name,
							dev->fs_type_code
						);
					else
						s = STRDUP(dev->fs_type_name);
				}
				else
				{
					s = STRDUP(dev->fs_type_name);
				}
				APPEND_LINE("Filesystem", s);
				g_free(s);
			}

			/* Mount point information (only if this object is
			 * a directory serving as a mount point)
			 */
			if(core->devices_list != NULL)
			{
				/* Get the device who's mount point is this directory */
				gint dev_num;
				EDVDevice *dev = edv_devices_list_match_mount_path(
					core->devices_list,
					&dev_num,
					path
				);
				if(dev != NULL)
				{
					const gulong	capacity_used = (dev->blocks_total > dev->blocks_free) ? 
						(dev->blocks_total - dev->blocks_free) : 0l,
							indicies_used = (dev->indicies_total > dev->indicies_free) ?
						(dev->indicies_total - dev->indicies_free) : 0l;

					/* Capacity Usage */
					if(verbose)
					{
						const gfloat coeff = (dev->blocks_total > 0l) ?
							((gfloat)capacity_used / (gfloat)dev->blocks_total) : 0.0f;
						EDVPixmap *p = edv_new_progress_pixmap(
							w->window,
							style,
							value_pixmap_width,
							value_pixmap_height,
							coeff,
							TRUE,		/* Draw value */
							GTK_ORIENTATION_HORIZONTAL,
							FALSE		/* Not reverse */
						);
						if(p != NULL)
						{
							APPEND_LINE_PIXMAP("Capacity Usage", p->pixmap, p->mask);
							(void)edv_pixmap_unref(p);
						}
					}

					/* Capacity Total */
					if(verbose)
					{
						const gulong size = dev->blocks_total;
						s = g_strdup_printf(
							"%s %s",
							edv_str_size_delim(size),
							(size == 1l) ? "byte" : "bytes"
						);
						APPEND_LINE("Capacity Total", s);
					}
					else
					{
						s = g_strdup_printf(
							"%ld",
							dev->blocks_total
						);
						APPEND_LINE("Total Capacity", s);
					}
					g_free(s);

					/* Capacity Used */
					if(verbose)
					{
						const gulong size = capacity_used;
						s = g_strdup_printf(
							"%s %s",
							edv_str_size_delim(size),
							(size == 1l) ? "byte" : "bytes"
						);
						APPEND_LINE("Capacity Used", s);
					}
					else
					{
						s = g_strdup_printf(
							"%ld",
							capacity_used
						);
						APPEND_LINE("Used Capacity", s);
					}
					g_free(s);

					/* Capacity Free & Available */
					if(verbose)
					{
						const gulong size = dev->blocks_available;
						s = g_strdup_printf(
							"%s %s",
							edv_str_size_delim(size),
							(size == 1l) ? "byte" : "bytes"
						);
						APPEND_LINE("Capacity Free & Available", s);
					}
					else
					{
						s = g_strdup_printf(
							"%ld",
							dev->blocks_available
						);
						APPEND_LINE("Free & Available Capacity", s);
					}
					g_free(s);

					/* Capacity Free */
					if(verbose)
					{
						const gulong size = dev->blocks_free;
						s = g_strdup_printf(
							"%s %s",
							edv_str_size_delim(size),
							(size == 1l) ? "byte" : "bytes"
						);
						APPEND_LINE("Capacity Free", s);
					}
					else
					{
						s = g_strdup_printf(
							"%ld",
							dev->blocks_free
						);
						APPEND_LINE("Free Capacity", s);
					}
					g_free(s);

					/* Device IO Block Size */
					if(verbose)
					{
						const gulong io_block_size = dev->block_size;
						s = g_strdup_printf(
							"%s %s",
							edv_str_size_delim(io_block_size),
							(io_block_size == 1l) ? "byte" : "bytes"
						);
					}
					else
						s = g_strdup_printf(
							"%ld",
							dev->block_size
						);
					APPEND_LINE("Device IO Block Size", s);
					g_free(s);

					if(dev->indicies_total > 0l)
					{
						/* Index/INodes Usage */
						if(verbose)
						{
							const gfloat coeff = (dev->indicies_total > 0l) ?
								((gfloat)indicies_used / (gfloat)dev->indicies_total) : 0.0f;
							EDVPixmap *p = edv_new_progress_pixmap(
								w->window,
								style,
								value_pixmap_width,
								value_pixmap_height,
								coeff,
								TRUE,	/* Draw value */
								GTK_ORIENTATION_HORIZONTAL,
								FALSE	/* Not reverse */
							);
							if(p != NULL)
							{
								APPEND_LINE_PIXMAP("Indicies Usage", p->pixmap, p->mask);
								(void)edv_pixmap_unref(p);
							}
						}

						/* Indicies/INodes Total */
						if(verbose)
						{
							s = STRDUP(edv_str_size_delim(dev->indicies_total));
							APPEND_LINE("Indicies Total", s);
						}
						else
						{
							s = g_strdup_printf(
								"%ld",
								dev->indicies_total
							);
							APPEND_LINE("Total INodes", s);
						}
						g_free(s);

						/* Indicies/INodes Used */
						if(verbose)
						{
							s = STRDUP(edv_str_size_delim(indicies_used));
							APPEND_LINE("Indicies Used", s);
						}
						else
						{
							s = g_strdup_printf(
								"%ld",
								indicies_used
							);
							APPEND_LINE("Used INodes", s);
						}
						g_free(s);

						/* Indicies/INodes Available */
						if(verbose)
						{
							s = STRDUP(edv_str_size_delim(dev->indicies_available));
							APPEND_LINE("Indicies Free & Available", s);
						}
						else
						{
							s = g_strdup_printf(
								"%ld",
								dev->indicies_available
							);
							APPEND_LINE("Free & Available INodes", s);
						}
						g_free(s);

						/* Indicies/INodes Free */
						if(verbose)
						{
							s = STRDUP(edv_str_size_delim(dev->indicies_free));
							APPEND_LINE("Indicies Free", s);
						}
						else
						{
							s = g_strdup_printf(
								"%ld",
								dev->indicies_free
							);
							APPEND_LINE("Free INodes", s);
						}
						g_free(s);
					}

					/* Name Length Max */
					if(verbose)
					{
						const gulong size = dev->name_length_max;
						s = g_strdup_printf(
							"%s %s",
							edv_str_size_delim(size),
							(size == 1l) ? "bytes" : "bytes"
						);
					}
					else
						s = g_strdup_printf(
							"%ld",
							dev->name_length_max
						);
					APPEND_LINE("Name Length Max", s);
					g_free(s);
				}
			}

			/* Index/INode */
			if(verbose)
			{
				s = g_strdup_printf(
					"#%ld",
					edv_properties_list_get_ul(
						properties_list,
						EDV_PROP_NAME_INDEX
					)
				);
				APPEND_LINE("Index", s);
			}
			else
			{
				s = g_strdup_printf(
					"%ld",
					edv_properties_list_get_ul(
						properties_list,
						EDV_PROP_NAME_INDEX
					)
				);
				APPEND_LINE("INode", s);
			}
			g_free(s);

			/* Type */
			if(verbose)
				s = STRDUP(edv_prop_page_get_mime_type_type(ctx));
			else
				s = STRDUP(edv_object_type_to_object_name(type));
			APPEND_LINE("Type", s);
			g_free(s);

			/* Size */
			if(verbose)
			{
				const gulong size = edv_properties_list_get_ul(
					properties_list,
					EDV_PROP_NAME_SIZE
				);
				s = g_strdup_printf(
					"%s %s",
					edv_str_size_delim(size),
					(size == 1l) ? "byte" : "bytes"
				);
			}
			else
				s = g_strdup_printf(
					"%ld",
					edv_properties_list_get_ul(
						properties_list,
						EDV_PROP_NAME_SIZE
					)
				);
			APPEND_LINE("Size", s);
			g_free(s);

			/* Permissions/Mode */
			if(verbose)
			{
				s = edv_str_permissions(permissions);
				APPEND_LINE("Permissions", s);
			}
			else
			{
				s = g_strdup_printf(
					"0x%.8X",
					(guint32)edv_edv_permissions_to_stat_mode(permissions)
				);
				APPEND_LINE("Mode", s);
			}
			g_free(s);

			/* Hard Links */
			s = g_strdup_printf(
				"%ld",
				edv_properties_list_get_ul(
					properties_list,
					EDV_PROP_NAME_HARD_LINKS
				)
			);
			APPEND_LINE("Hard Links", s);
			g_free(s);

			/* Owner */
			if(verbose)
			{
				s =  STRDUP(edv_properties_list_get_s(
					properties_list,
					EDV_PROP_NAME_OWNER_NAME
				));
		        if(s == NULL)
		            s = edv_uid_uid_to_name(
		                core->uids_list,
		                edv_properties_list_get_i(
		                    properties_list,
		                    EDV_PROP_NAME_OWNER_ID
		                ),
		                NULL
		            );
				APPEND_LINE("Owner", s);
			}
			else
			{
				s = g_strdup_printf(
					"%i",
					edv_properties_list_get_i(
						properties_list,
						EDV_PROP_NAME_OWNER_ID
					)
				);
				APPEND_LINE("UID", s);
			}
			g_free(s);

			/* Group */
			if(verbose)
			{
				s =  STRDUP(edv_properties_list_get_s(
					properties_list,
					EDV_PROP_NAME_GROUP_NAME
				));
		        if(s == NULL)
		            s = edv_gid_gid_to_name(
		                core->gids_list,
		                edv_properties_list_get_i(
		                    properties_list,
		                    EDV_PROP_NAME_GROUP_ID
		                ),
		                NULL
		            );
				APPEND_LINE("Group", s);
			}
			else
			{
				s = g_strdup_printf(
					"%i",
					edv_properties_list_get_i(
						properties_list,
						EDV_PROP_NAME_GROUP_ID
					)
				);
				APPEND_LINE("GID", s);
			}
			g_free(s);

			/* Link Target */
			if(type == EDV_OBJECT_TYPE_LINK)
			{
				s = STRDUP(edv_properties_list_get_s(
					properties_list,
					EDV_PROP_NAME_LINK_TARGET
				));
				if(s == NULL)
					s = g_strdup(EDV_DETAILS_PROP_PAGE_VALUE_NOT_AVAILABLE_STRING);
				APPEND_LINE("Link Target", s);
				g_free(s);
			}

			/* Device Numbers */
			if((type == EDV_OBJECT_TYPE_DEVICE_BLOCK) ||
			   (type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
			)
			{
				if(verbose)
				{
					gint major = 0, minor = 0;
					edv_device_numbers_parse(
						edv_properties_list_get_i(
							properties_list,
							EDV_PROP_NAME_DEVICE_TYPE
						),
						&major, &minor
					);
					s = g_strdup_printf(
						"%i, %i",
						major, minor
					);
					APPEND_LINE("Device Numbers", s);
				}
				else
				{
					s = g_strdup_printf(
						"0x%.4X",
						(guint16)edv_properties_list_get_i(
							properties_list,
							EDV_PROP_NAME_DEVICE_TYPE
						)
					);
					APPEND_LINE("Device Type", s);
				}
				g_free(s);
			}

			/* IO Block Size */
			if(verbose)
			{
				const gulong io_block_size = edv_properties_list_get_ul(
					properties_list,
					EDV_PROP_NAME_BLOCK_SIZE
				);
				s = g_strdup_printf(
					"%s %s",
					edv_str_size_delim(io_block_size),
					(io_block_size == 1l) ? "byte" : "bytes"
				);
			}
			else
			{
				s = g_strdup_printf(
					"%ld",
					edv_properties_list_get_ul(
						properties_list,
						EDV_PROP_NAME_BLOCK_SIZE
					)
				);
			}
			APPEND_LINE("Optimul IO Block Size", s);
			g_free(s);

			/* Blocks Allocated */
			if(verbose)
			{
				const gulong blocks_allocated = edv_properties_list_get_ul(
					properties_list,
					EDV_PROP_NAME_BLOCKS
				);
				s = g_strdup_printf(
					"%s %s",
					edv_str_size_delim(blocks_allocated),
					(blocks_allocated == 1l) ? "block" : "blocks"
				);
				APPEND_LINE("Blocks Allocated", s);
			}
			else
			{
				s = g_strdup_printf(
					"%ld",
					edv_properties_list_get_ul(
						properties_list,
						EDV_PROP_NAME_BLOCKS
					)
				);
				APPEND_LINE("Blocks Allocated", s);
			}
			g_free(s);

			/* Last Accessed */
			if(verbose)
			{
				const gulong t = edv_properties_list_get_ul(
					properties_list,
					EDV_PROP_NAME_ACCESS_TIME
				);
				if(t > 0l)
					s = edv_date_string_format(
						t,
						date_format, date_relativity
					);
				else
					s = g_strdup(EDV_DETAILS_PROP_PAGE_TIME_STAMP_NOT_SET_STRING);
				APPEND_LINE("Last Accessed", s);
			}
			else
			{
				s = g_strdup_printf(
					"%ld",
					edv_properties_list_get_ul(
						properties_list,
						EDV_PROP_NAME_ACCESS_TIME
					)
				);
				APPEND_LINE("Accessed Time", s);
			}
			g_free(s);

			/* Last Modified */
			if(verbose)
			{
				const gulong t = edv_properties_list_get_ul(
					properties_list,
					EDV_PROP_NAME_MODIFY_TIME
				);
				if(t > 0l)
					s = edv_date_string_format(
						t,
						date_format, date_relativity
					);
				else
					s = g_strdup(EDV_DETAILS_PROP_PAGE_TIME_STAMP_NOT_SET_STRING);
				APPEND_LINE("Last Modified", s);
			}
			else
			{
				s = g_strdup_printf(
					"%ld",
					edv_properties_list_get_ul(
						properties_list,
						EDV_PROP_NAME_MODIFY_TIME
					)
				);
				APPEND_LINE("Modified Time", s);
			}
			g_free(s);

			/* Last Changed */
			if(verbose)
			{
				const gulong t = edv_properties_list_get_ul(
					properties_list,
					EDV_PROP_NAME_CHANGE_TIME
				);
				if(t > 0l)
					s = edv_date_string_format(
						t,
						date_format, date_relativity
					);
				else
					s = g_strdup(EDV_DETAILS_PROP_PAGE_TIME_STAMP_NOT_SET_STRING);
				APPEND_LINE("Last Changed", s);
			}
			else
			{
				s = g_strdup_printf(
					"%ld",
					edv_properties_list_get_ul(
						properties_list,
						EDV_PROP_NAME_CHANGE_TIME
					)
				);
				APPEND_LINE("Changed Time", s);
			}
			g_free(s);

			/* Mount point information (only if this object is
			 * a directory serving as a mount point)
			 */
			if(core->devices_list != NULL)
			{
				/* Get the device who's mount point is this directory */
				gint dev_num;
				EDVDevice *dev = edv_devices_list_match_mount_path(
					core->devices_list,
					&dev_num,
					path
				);
				if(dev != NULL)
				{
					/* Last Mounted */
					if(verbose)
					{
						const gulong t = dev->last_mount_time;
						if(t > 0l)
							s = edv_date_string_format(
								t,
								date_format, date_relativity
							);
						else
							s = g_strdup(EDV_DETAILS_PROP_PAGE_TIME_STAMP_NOT_SET_STRING);
						APPEND_LINE("Last Mounted", s);
					}
					else
					{
						s = g_strdup_printf(
							"%ld",
							dev->last_mount_time
						);
						APPEND_LINE("Mounted Time", s);
					}
					g_free(s);

					/* Last Checked */
					if(verbose)
					{
						const gulong t = dev->last_check_time;
						if(t > 0l)
							s = edv_date_string_format(
								t,
								date_format, date_relativity
							);
						else
							s = g_strdup(EDV_DETAILS_PROP_PAGE_TIME_STAMP_NOT_SET_STRING);
						APPEND_LINE("Last Checked", s);
					}
					else
					{
						s = g_strdup_printf(
							"%ld",
							dev->last_check_time
						);
						APPEND_LINE("Checked Time", s);
					}
					g_free(s);
				}
			}
		}
		break;

	  case EDV_LOCATION_TYPE_RECYCLE_BIN:
		if(properties_list != NULL)
		{
			const gulong	recbin_size_max = (gulong)EDV_GET_L(
				EDV_CFG_PARM_RECBIN_SIZE_WARN
			),
							size = edv_properties_list_get_ul(
				properties_list,
				EDV_PROP_NAME_SIZE
			),
							storage_size = edv_properties_list_get_ul(
				properties_list,
				EDV_PROP_NAME_STORAGE_SIZE
			);
			const EDVPermissionFlags permissions = (EDVPermissionFlags)edv_properties_list_get_i(
				properties_list,
				EDV_PROP_NAME_PERMISSIONS
			); 

			/* Name */
			s = STRDUP(edv_properties_list_get_s(
				properties_list,
				EDV_PROP_NAME_NAME
			));
			if(s == NULL)
				s = g_strdup(EDV_DETAILS_PROP_PAGE_VALUE_NOT_AVAILABLE_STRING);
			APPEND_LINE("Name", s);
			g_free(s);

			/* Location */
			if(verbose)
			{
				APPEND_LINE("Location", "Recycle Bin");
			}
			else
			{
				gchar *location = edv_recycle_bin_index_get_recycled_object_path(
					EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX),
					(guint)edv_properties_list_get_ul(
						properties_list,
						EDV_PROP_NAME_INDEX
					)
				);
				APPEND_LINE("Location", location);
				g_free(location);
			}

			/* Index */
			if(verbose)
				s = g_strdup_printf(
					"#%ld",
					edv_properties_list_get_ul(
						properties_list,
						EDV_PROP_NAME_INDEX
					)
				);
			else
				s = g_strdup_printf(
					"%ld",
					edv_properties_list_get_ul(
						properties_list,
						EDV_PROP_NAME_INDEX
					)
				);
			APPEND_LINE("Index", s);
			g_free(s);

			/* Original Location */
			s = STRDUP(edv_properties_list_get_s(
				properties_list,
				EDV_PROP_NAME_ORIGINAL_PATH
			));
			if(s == NULL)
				s = g_strdup(EDV_DETAILS_PROP_PAGE_VALUE_NOT_AVAILABLE_STRING);
			APPEND_LINE("Original Location", s);
			g_free(s);

			/* Type */
			if(verbose)
				s = STRDUP(edv_prop_page_get_mime_type_type(ctx));
			else
				s = STRDUP(edv_object_type_to_object_name(type));
			APPEND_LINE("Type", s);
			g_free(s);

			/* Size */
			if(verbose)
				s = g_strdup_printf(
					"%s %s",
					edv_str_size_delim(size),
					(size == 1l) ? "byte" : "bytes"
				);
			else
				s = g_strdup_printf(
					"%ld",
					size
				);
			APPEND_LINE("Size", s);
			g_free(s);

			/* Storage Size */
			if(verbose)
				s = g_strdup_printf(
					"%s %s",
					edv_str_size_delim(storage_size),
					(storage_size == 1l) ? "byte" : "bytes"
				);
			else
				s = g_strdup_printf(
					"%ld",
					storage_size
				);
			APPEND_LINE("Storage Size", s);
			g_free(s);

			/* Capacity Used */
			if(recbin_size_max > 0l)
			{
				const gfloat coeff = size / (gfloat)recbin_size_max;
				if(verbose)
				{
					EDVPixmap *p = edv_new_progress_pixmap(
						w->window,
						style,
						value_pixmap_width,
						value_pixmap_height,
						coeff,
						TRUE,		/* Draw value */
						GTK_ORIENTATION_HORIZONTAL,
						FALSE		/* Not reverse */
					);
					if(p != NULL)
					{
						APPEND_LINE_PIXMAP("Capacity Usage", p->pixmap, p->mask);
						(void)edv_pixmap_unref(p);
					}
				}
				else
				{
					s = g_strdup_printf(
						"%.2f",
						coeff
					);
					APPEND_LINE("Capacity Usage", s);
					g_free(s);
				}
			}

			/* Permissions/Mode */
			if(verbose)
			{
				s = edv_str_permissions(permissions);
				APPEND_LINE("Permissions", s);
			}
			else
			{
				s = g_strdup_printf(
					"0x%.8X",
					(guint32)edv_edv_permissions_to_stat_mode(permissions)
				);
				APPEND_LINE("Mode", s);
			}
			g_free(s);

			/* Owner */
			if(verbose)
			{
				s = edv_uid_uid_to_name(
					core->uids_list,
					edv_properties_list_get_i(
						properties_list,
						EDV_PROP_NAME_OWNER_ID
					),
					NULL
				);
				APPEND_LINE("Owner", s);
			}
			else
			{
				s = g_strdup_printf(
					"%i",
					edv_properties_list_get_i(
						properties_list,
						EDV_PROP_NAME_OWNER_ID
					)
				);
				APPEND_LINE("UID", s);
			}
			g_free(s);

			/* Group */
			if(verbose)
			{
				s = edv_gid_gid_to_name(
					core->gids_list,
					edv_properties_list_get_i(
						properties_list,
						EDV_PROP_NAME_GROUP_ID
					),
					NULL
				);
				APPEND_LINE("Group", s);
			}
			else
			{
				s = g_strdup_printf(
					"%i",
					edv_properties_list_get_i(
						properties_list,
						EDV_PROP_NAME_GROUP_ID
					)
				);
				APPEND_LINE("GID", s);
			}
			g_free(s);

			/* Link Target */
			if(type == EDV_OBJECT_TYPE_LINK)
			{
				s = STRDUP(edv_properties_list_get_s(
					properties_list,
					EDV_PROP_NAME_LINK_TARGET
				));
				if(s == NULL)
					s = g_strdup(EDV_DETAILS_PROP_PAGE_VALUE_NOT_AVAILABLE_STRING);
				APPEND_LINE("Link Target", s);
				g_free(s);
			}

			/* Last Accessed */
			if(verbose)
			{
				const gulong t = edv_properties_list_get_ul(
					properties_list,
					EDV_PROP_NAME_ACCESS_TIME
				);
				if(t > 0l)
					s = edv_date_string_format(
						t,
						date_format, date_relativity
					);
				else
					s = g_strdup(EDV_DETAILS_PROP_PAGE_TIME_STAMP_NOT_SET_STRING);
				APPEND_LINE("Last Accessed", s);
			}
			else
			{
				s = g_strdup_printf(
					"%ld",
					edv_properties_list_get_ul(
						properties_list,
						EDV_PROP_NAME_ACCESS_TIME
					)
				);
				APPEND_LINE("Accessed Time", s);
			}
			g_free(s);

			/* Last Modified */
			if(verbose)
			{
				const gulong t = edv_properties_list_get_ul(
					properties_list,
					EDV_PROP_NAME_MODIFY_TIME
				);
				if(t > 0l)
					s = edv_date_string_format(
						t,
						date_format, date_relativity
					);
				else
					s = g_strdup(EDV_DETAILS_PROP_PAGE_TIME_STAMP_NOT_SET_STRING);
				APPEND_LINE("Last Modified", s);
			}
			else
			{
				s = g_strdup_printf(
					"%ld",
					edv_properties_list_get_ul(
						properties_list,
						EDV_PROP_NAME_MODIFY_TIME
					)
				);
				APPEND_LINE("Modified Time", s);
			}
			g_free(s);

			/* Last Changed */
			if(verbose)
			{
				const gulong t = edv_properties_list_get_ul(
					properties_list,
					EDV_PROP_NAME_CHANGE_TIME
				);
				if(t > 0l)
					s = edv_date_string_format(
						t,
						date_format, date_relativity
					);
				else
					s = g_strdup(EDV_DETAILS_PROP_PAGE_TIME_STAMP_NOT_SET_STRING);
				APPEND_LINE("Last Changed", s);
			}
			else
			{
				s = g_strdup_printf(
					"%ld",
					edv_properties_list_get_ul(
						properties_list,
						EDV_PROP_NAME_CHANGE_TIME
					)
				);
				APPEND_LINE("Changed Time", s);
			}
			g_free(s);

			/* Deleted On/Deleted Time */
			if(verbose)
			{
				const gulong t = edv_properties_list_get_ul(
					properties_list,
					EDV_PROP_NAME_DELETED_TIME
				);
				if(t > 0l)
					s = edv_date_string_format(
						t,
						date_format, date_relativity
					);
				else
					s = g_strdup(EDV_DETAILS_PROP_PAGE_TIME_STAMP_NOT_SET_STRING);
				APPEND_LINE("Deleted On", s);
			}
			else
			{
				s = g_strdup_printf(
					"%ld",
					edv_properties_list_get_ul(
						properties_list,
						EDV_PROP_NAME_DELETED_TIME
					)
				);
				APPEND_LINE("Deleted Time", s);
			}
			g_free(s);
		}
		break;

	  case EDV_LOCATION_TYPE_ARCHIVE:
		if(properties_list != NULL)
		{
			const gulong	size = edv_properties_list_get_ul(
				properties_list,
				EDV_PROP_NAME_SIZE
			),
							storage_size = edv_properties_list_get_ul(
				properties_list,
				EDV_PROP_NAME_STORAGE_SIZE
			);
			const gfloat compression_ratio = edv_properties_list_get_f(
				properties_list,
				EDV_PROP_NAME_COMPRESSION_RATIO
			);
			const gchar *in_arch_path = edv_properties_list_get_s(
				properties_list,
				EDV_PROP_NAME_PATH
			);
			GdkBitmap *mask;
			GdkPixmap *pixmap;
			const EDVPermissionFlags permissions = (EDVPermissionFlags)edv_properties_list_get_i(
				properties_list,
				EDV_PROP_NAME_PERMISSIONS
			);

			/* Name */
			s = STRDUP(edv_properties_list_get_s(
				properties_list,
				EDV_PROP_NAME_NAME
			));
			if(s == NULL)
				s = g_strdup(EDV_DETAILS_PROP_PAGE_VALUE_NOT_AVAILABLE_STRING);
			APPEND_LINE("Name", s);
			g_free(s);

			/* Location */
			s = STRDUP(edv_prop_page_get_archive_path(ctx));
			if(s == NULL)
				s = g_strdup(EDV_DETAILS_PROP_PAGE_VALUE_NOT_AVAILABLE_STRING);
			APPEND_LINE("Location", s);
			g_free(s);

			/* Index */
			if(verbose)
				s = g_strdup_printf(
					"#%ld",
					edv_properties_list_get_ul(
						properties_list,
						EDV_PROP_NAME_INDEX
					)
				);
			else
				s = g_strdup_printf(
					"%ld",
					edv_properties_list_get_ul(
						properties_list,
						EDV_PROP_NAME_INDEX
					)
				);
			APPEND_LINE("Index", s);
			g_free(s);

			/* Location In Archive */
			if(!STRISEMPTY(in_arch_path))
			{
				gchar *location;

				if(g_path_is_absolute(in_arch_path))
					location = g_strdup(in_arch_path);
				else
					location = g_strconcat(
						".",
						G_DIR_SEPARATOR_S,
						in_arch_path,
						NULL
					);
				if(location != NULL)
				{
					s = g_dirname(location);
					g_free(location);
				}
				else
					s = g_strdup(EDV_DETAILS_PROP_PAGE_VALUE_NOT_AVAILABLE_STRING);
			}
			else
			{
				s = g_strdup(EDV_DETAILS_PROP_PAGE_VALUE_NOT_AVAILABLE_STRING);
			}
			APPEND_LINE("Location In Archive", s);
			g_free(s);

			/* Type */
			if(verbose)
				s = STRDUP(edv_prop_page_get_mime_type_type(ctx));
			else
				s = STRDUP(edv_object_type_to_object_name(type));
			APPEND_LINE("Type", s);
			g_free(s);

			/* Size */
			if(verbose)
				s = g_strdup_printf(
					"%s %s",
					edv_str_size_delim(size),
					(size == 1l) ? "byte" : "bytes"
				);
			else
				s = g_strdup_printf(
					"%ld",
					size
				);
			APPEND_LINE("Size", s);
			g_free(s);

			/* Storage Size */
			if(((storage_size > 0l) && (size > 0l)) ||
			   (size == 0l)
			)
			{
				if(verbose)
					s = g_strdup_printf(
						"%s %s",
						edv_str_size_delim(storage_size),
						(storage_size == 1l) ? "byte" : "bytes"
					);
				else
					s = g_strdup_printf(
						"%ld",
						storage_size
					);
			}
			else
			{
				if(verbose)
					s = g_strdup(EDV_DETAILS_PROP_PAGE_VALUE_NOT_AVAILABLE_STRING);
				else
					s = g_strdup_printf(
						"%ld",
						storage_size
					);
			}
			APPEND_LINE("Storage Size", s);
			g_free(s);

			/* Compression Ratio */
			s = NULL;
			pixmap = NULL;
			mask = NULL;
			if((type == EDV_OBJECT_TYPE_FILE) &&
			   (compression_ratio >= 0.0f)
			)
			{
				const gfloat ratio = compression_ratio;
				if(((storage_size > 0l) && (size > 0l)) ||
				   (size == 0l)
				)
				{
					if(verbose)
					{
						EDVPixmap *p = edv_new_progress_pixmap(
							w->window,
							style,
							value_pixmap_width,
							value_pixmap_height,
							ratio,
							TRUE,		/* Draw value */
							GTK_ORIENTATION_HORIZONTAL,
							FALSE		/* Not reverse */
						);
						if(p != NULL)
						{
							pixmap = GDK_PIXMAP_REF(p->pixmap);
							mask = GDK_PIXMAP_REF(p->mask);
							(void)edv_pixmap_unref(p);
						}
						else
							s = g_strdup(EDV_DETAILS_PROP_PAGE_VALUE_NOT_AVAILABLE_STRING);
					}
					else
						s = g_strdup_printf(
							"%.2f",
							ratio
						);
				}
				else
				{
					if(verbose)
						s = g_strdup(EDV_DETAILS_PROP_PAGE_VALUE_NOT_AVAILABLE_STRING);
					else
						s = g_strdup_printf(
							"%.2f",
							ratio
						);
				}
			}
			else
			{
				if(verbose)
					s = g_strdup(EDV_DETAILS_PROP_PAGE_VALUE_NOT_AVAILABLE_STRING);
				else
					s = g_strdup_printf(
						"%.2f",
						compression_ratio
					);
			}
			if(pixmap != NULL)
			{
				APPEND_LINE_PIXMAP("Compression Ratio", pixmap, mask);
				(void)GDK_PIXMAP_UNREF(pixmap);
				(void)GDK_BITMAP_UNREF(mask);
			}
			else
			{
				APPEND_LINE("Compression Ratio", s);
			}
			g_free(s);

			/* Encryption */
			s = STRDUP(edv_properties_list_get_s(
				properties_list,
				EDV_PROP_NAME_ENCRYPTION_NAME
			));
			if(s == NULL)
				s = g_strdup("None");
			APPEND_LINE("Encryption", s);
			g_free(s);

			/* Permissions/Mode */
			if(verbose)
			{
				s = edv_str_permissions(permissions);
				APPEND_LINE("Permissions", s);
			}
			else
			{
				s = g_strdup_printf(
					"0x%.8X",
					(guint32)edv_edv_permissions_to_stat_mode(permissions)
				);
				APPEND_LINE("Mode", s);
			}
			g_free(s);

			/* Owner */
			s = STRDUP(edv_properties_list_get_s(
				properties_list,
				EDV_PROP_NAME_OWNER_NAME
			));
			if(s == NULL)
				s = g_strdup(EDV_DETAILS_PROP_PAGE_VALUE_NOT_AVAILABLE_STRING);
			APPEND_LINE("Owner", s);
			g_free(s);

			/* Group */
			s = STRDUP(edv_properties_list_get_s(
				properties_list,
				EDV_PROP_NAME_GROUP_NAME
			));
			if(s == NULL)
				s = g_strdup(EDV_DETAILS_PROP_PAGE_VALUE_NOT_AVAILABLE_STRING);
			APPEND_LINE("Group", s);
			g_free(s);

			/* Link Target */
			if(type == EDV_OBJECT_TYPE_LINK)
			{
				s = STRDUP(edv_properties_list_get_s(
					properties_list,
					EDV_PROP_NAME_LINK_TARGET
				));
				if(s == NULL)
					s = g_strdup(EDV_DETAILS_PROP_PAGE_VALUE_NOT_AVAILABLE_STRING);
				APPEND_LINE("Link Target", s);
				g_free(s);
			}

			/* Device Numbers */
			if((type == EDV_OBJECT_TYPE_DEVICE_BLOCK) ||
			   (type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
			)
			{
				if(verbose)
				{
					gint major = 0, minor = 0;
					edv_device_numbers_parse(
						edv_properties_list_get_i(
							properties_list,
							EDV_PROP_NAME_DEVICE_TYPE
						),
						&major, &minor
					);
					s = g_strdup_printf(
						"%i, %i",
						major, minor
					);
					APPEND_LINE("Device Numbers", s);
				}
				else
				{
					s = g_strdup_printf(
						"0x%.4X",
						(guint16)edv_properties_list_get_i(
							properties_list,
							EDV_PROP_NAME_DEVICE_TYPE
						)
					);
					APPEND_LINE("Device Type", s);
				}
				g_free(s);
			}

			/* Storage Method */
			s = STRDUP(edv_properties_list_get_s(
				properties_list,
				EDV_PROP_NAME_STORAGE_METHOD
			));
			if(s == NULL)
				s = g_strdup(EDV_DETAILS_PROP_PAGE_VALUE_NOT_AVAILABLE_STRING);
			APPEND_LINE("Storage Method", s);
			g_free(s);

			/* CRC */
			s = STRDUP(edv_properties_list_get_s(
				properties_list,
				EDV_PROP_NAME_CRC
			));
			if(s == NULL)
				s = STRDUP(EDV_DETAILS_PROP_PAGE_VALUE_NOT_AVAILABLE_STRING);
			APPEND_LINE("CRC", s);
			g_free(s);

			/* Last Accessed */
			if(verbose)
			{
				const gulong t = edv_properties_list_get_ul(
					properties_list,
					EDV_PROP_NAME_ACCESS_TIME
				);
				if(t > 0l)
					s = edv_date_string_format(
						t,
						date_format, date_relativity
					);
				else
					s = g_strdup(EDV_DETAILS_PROP_PAGE_TIME_STAMP_NOT_SET_STRING);
				APPEND_LINE("Last Accessed", s);
			}
			else
			{
				s = g_strdup_printf(
					"%ld",
					edv_properties_list_get_ul(
						properties_list,
						EDV_PROP_NAME_ACCESS_TIME
					)
				);
				APPEND_LINE("Accessed Time", s);
			}
			g_free(s);

			/* Last Modified */
			if(verbose)
			{
				const gulong t = edv_properties_list_get_ul(
					properties_list,
					EDV_PROP_NAME_MODIFY_TIME
				);
				if(t > 0l)
					s = edv_date_string_format(
						t,
						date_format, date_relativity
					);
				else
					s = g_strdup(EDV_DETAILS_PROP_PAGE_TIME_STAMP_NOT_SET_STRING);
				APPEND_LINE("Last Modified", s);
			}
			else
			{
				s = g_strdup_printf(
					"%ld",
					edv_properties_list_get_ul(
						properties_list,
						EDV_PROP_NAME_MODIFY_TIME
					)
				);
				APPEND_LINE("Modified Time", s);
			}
			g_free(s);

			/* Last Changed */
			if(verbose)
			{
				const gulong t = edv_properties_list_get_ul(
					properties_list,
					EDV_PROP_NAME_CHANGE_TIME
				);
				if(t > 0l)
					s = edv_date_string_format(
						t,
						date_format, date_relativity
					);
				else
					s = g_strdup(EDV_DETAILS_PROP_PAGE_TIME_STAMP_NOT_SET_STRING);
				APPEND_LINE("Last Changed", s);
			}
			else
			{
				s = g_strdup_printf(
					"%ld",
					edv_properties_list_get_ul(
						properties_list,
						EDV_PROP_NAME_CHANGE_TIME
					)
				);
				APPEND_LINE("Changed Time", s);
			}
			g_free(s);
		}
		break;
	}


	/* Additional meta data list available?
	 *
	 * Note that since the Details Properties Page (should) be
	 * created last, all other pages should have called
	 * edv_prop_page_add_meta_data() to add their meta data to
	 * the list and thus this list should be up to date
	 */
	if(meta_data_list != NULL)
	{
		/* Append the meta data list */
		GList *glist;
		EDVProperty *p;
		for(glist = meta_data_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			p = EDV_PROPERTY(glist->data);
			if(p == NULL)
				continue;

			APPEND_LINE(p->name, p->value);
		}
	}

#undef APPEND_LINE_PIXMAP
#undef APPEND_LINE

	gtk_clist_columns_autosize(clist);

	/* Scroll back to the original position */
	edv_clist_scroll_to_position(clist, last_x, last_y);

	gtk_clist_thaw(clist);

	g_free(strv);

	gtk_widget_queue_resize(toplevel);

	p->freeze_count--;
}

/*
 *	Destroy callback.
 */
void edv_details_prop_page_destroy_cb(
	EDVPropPageContext *ctx,
	gpointer data
)
{
	EDVDetailsPropPage *p = EDV_DETAILS_PROP_PAGE(data);
	g_free(p);
}


/*
 *	Details verbose GtkRadioButton "toggled" signal callback.
 */
static void edv_details_prop_page_verbose_cb(
	GtkWidget *widget, gpointer data
)
{
	EDVDetailsPropPage *p = EDV_DETAILS_PROP_PAGE(data);
	EDVPropPageContext *ctx = p->ctx;
	const EDVLocationType location_type = edv_prop_page_get_location_type(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	/* Update the Details GtkCList */
	edv_details_prop_page_update_cb(
		ctx,
		(EDVObjectType)edv_properties_list_get_i(
			properties_list,
			EDV_PROP_NAME_TYPE
		),
		location_type,
		properties_list,
		0,				/* Success */
		p
	);

	p->freeze_count--;
}

/*
 *	Details raw GtkRadioButton "toggled" signal callback.
 */
static void edv_details_prop_page_raw_cb(
	GtkWidget *widget, gpointer data
)
{
	EDVDetailsPropPage *p = EDV_DETAILS_PROP_PAGE(data);
	EDVPropPageContext *ctx = p->ctx;
	const EDVLocationType location_type = edv_prop_page_get_location_type(ctx);
	GList *properties_list = edv_prop_page_get_properties_list(ctx);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	/* Update the Details GtkCList */
	edv_details_prop_page_update_cb(
		ctx,
		(EDVObjectType)edv_properties_list_get_i(
			properties_list,
			EDV_PROP_NAME_TYPE
		),
		location_type,
		properties_list,
		0,				/* Success */
		p
	);

	p->freeze_count--;
}

/*
 *	Details GtkCList "select_row" signal callback.
 */
static void edv_details_prop_page_select_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	EDVDetailsPropPage *p = EDV_DETAILS_PROP_PAGE(data);

	if(p->freeze_count > 0)
		return;

	p->freeze_count++;

	/* Scroll to the selected row if it is not visible */
	if(gtk_clist_row_is_visible(clist, row) !=
	   GTK_VISIBILITY_FULL
	)
		gtk_clist_moveto(
			clist,
			row, -1,		/* Row, column */
			0.5f, 0.0f		/* Row, column */
		);

	p->freeze_count--;
}
