#include <gtk/gtk.h>

#include "cfg.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_property.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_recycled_obj.h"
#include "libendeavour2-base/edv_archive_obj.h"
#include "edv_device.h"
#include "edv_mime_type.h"
#include "prop_dlg.h"
#include "prop_page.h"
#include "prop_page_private.h"

#include "config.h"


/* prop_page_private.h */
EDVPropPageContext *edv_prop_page_context_new(void);
void edv_prop_page_context_delete(EDVPropPageContext *ctx);

/* prop_page.h */
GtkWidget *edv_prop_page_get_toplevel(EDVPropPageContext *ctx);
GtkAccelGroup *edv_prop_page_get_accelgrp(EDVPropPageContext *ctx);
gint edv_prop_page_get_freeze_count(EDVPropPageContext *ctx);

EDVCore *edv_prop_page_get_core(EDVPropPageContext *ctx);
CfgList *edv_prop_page_get_cfg_list(EDVPropPageContext *ctx);

EDVLocationType edv_prop_page_get_location_type(EDVPropPageContext *ctx);

GList *edv_prop_page_get_properties_list(EDVPropPageContext *ctx);

GList *edv_prop_page_get_meta_data_list(EDVPropPageContext *ctx);
void edv_prop_page_add_meta_data(
	EDVPropPageContext *ctx,
	const gchar *name,
	const gchar *value
);
void edv_prop_page_remove_meta_data(
	EDVPropPageContext *ctx,
	const gchar *name
);

const gchar *edv_prop_page_get_mime_type_type(EDVPropPageContext *ctx);

const gchar *edv_prop_page_get_archive_path(EDVPropPageContext *ctx);

gboolean edv_prop_page_get_has_changes(EDVPropPageContext *ctx);
void edv_prop_page_set_has_changes(
	EDVPropPageContext *ctx,
	const gboolean has_changes
);

void edv_prop_page_set_busy(
	EDVPropPageContext *ctx,
	const gboolean busy
);

GtkWidget *edv_prop_page_create_icon_selector(EDVPropPageContext *ctx);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Creates a new EDVPropPageContext.
 *
 *	Returns a new dynamically allocated EDVPropPageContext. with
 *	all of its values zero'ed or NULL on error.
 */
EDVPropPageContext *edv_prop_page_context_new(void)
{
	return(EDV_PROP_PAGE_CONTEXT(g_malloc0(sizeof(EDVPropPageContext))));
}

/*
 *	Deletes the EDVPropPageContext.
 *
 *	The references to other windows (such as the EDVPropDlg) and
 *	other user set resources will not be deleted.
 */
void edv_prop_page_context_delete(EDVPropPageContext *ctx)
{
	if(ctx == NULL)
		return;

	/* Do not delete user set resources */

	g_free(ctx);
}


/*
 *	Gets the toplevel GtkWidget.
 */
GtkWidget *edv_prop_page_get_toplevel(EDVPropPageContext *ctx)
{
	if(ctx == NULL)
		return(NULL);

	return(edv_prop_dlg_get_toplevel(ctx->prop_dlg));
}

/*
 *	Gets the GtkAccelGroup.
 */
GtkAccelGroup *edv_prop_page_get_accelgrp(EDVPropPageContext *ctx)
{
	if(ctx == NULL)
		return(NULL);

	return(edv_prop_dlg_get_accelgrp(ctx->prop_dlg));
}

/*
 *	Gets the freeze count.
 */
gint edv_prop_page_get_freeze_count(EDVPropPageContext *ctx)
{
	if(ctx == NULL)
		return(0);

	return(edv_prop_dlg_get_freeze_count(ctx->prop_dlg));
}

/*
 *	Freezes the parent window.
 */
void edv_prop_page_get_freeze(EDVPropPageContext *ctx)
{
	if(ctx == NULL)
		return;

	return(edv_prop_dlg_freeze(ctx->prop_dlg));
}

/*
 *	Thaws the parent window.
 */
void edv_prop_page_get_thaw(EDVPropPageContext *ctx)
{
	if(ctx == NULL)
		return;

	return(edv_prop_dlg_thaw(ctx->prop_dlg));
}


/*
 *	Gets the EDVCore.
 */
EDVCore *edv_prop_page_get_core(EDVPropPageContext *ctx)
{
	if(ctx == NULL)
		return(NULL);

	return(edv_prop_dlg_get_core(ctx->prop_dlg));
}

/*
 *	Gets the CfgList.
 */
CfgList *edv_prop_page_get_cfg_list(EDVPropPageContext *ctx)
{
	if(ctx == NULL)
		return(NULL);

	return(edv_prop_dlg_get_cfg_list(ctx->prop_dlg));
}


/*
 *	Gets the EDV*Object's EDVLocationType.
 */
EDVLocationType edv_prop_page_get_location_type(EDVPropPageContext *ctx)
{
	if(ctx == NULL)
		return(EDV_LOCATION_TYPE_VFS);

	return(edv_prop_dlg_get_location_type(ctx->prop_dlg));
}


/*
 *	Gets the properties list describing the EDV*Object.
 *
 *	Returns the pointer to the GList of EDVProperty * properties
 *	or NULL on error. The returned pointer must not be modified
 *	or deleted.
 */
GList *edv_prop_page_get_properties_list(EDVPropPageContext *ctx)
{
	if(ctx == NULL)
		return(NULL);

	return(edv_prop_dlg_get_properties_list(ctx->prop_dlg));
}


/*
 *	Gets the meta data list describing the EDV*Object.
 *
 *	Returns the pointer to the GList of EDVProperty * meta datas
 *	or NULL on error. The returned pointer must not be modified
 *	or deleted.
 */
GList *edv_prop_page_get_meta_data_list(EDVPropPageContext *ctx)
{
	if(ctx == NULL)
		return(NULL);

	return(edv_prop_dlg_get_meta_data_list(ctx->prop_dlg));
}

/*
 *	Add/updates the meta data value.
 *
 *	The name specifies the name of the meta data to add/update.
 *
 *	The value specifies the value to set. If value is NULL or an
 *	empty string then the meta data will be removed.
 */
void edv_prop_page_add_meta_data(
	EDVPropPageContext *ctx,
	const gchar *name,
	const gchar *value
)
{
	if(ctx == NULL)
		return;

	return(edv_prop_dlg_add_meta_data(
		ctx->prop_dlg,
		name,
		value
	));
}

/*
 *	Removes the meta data.
 *
 *	The name specifies the name of the meta data to remove.
 */
void edv_prop_page_remove_meta_data(
	EDVPropPageContext *ctx,
	const gchar *name
)
{
	if(ctx == NULL)
		return;

	return(edv_prop_dlg_remove_meta_data(
		ctx->prop_dlg,
		name
	));
}


/*
 *	Gets the EDVMIMEType type.
 *
 *	Returns the pointer to the EDVMIMEType's type string or NULL
 *	on error.
 */
const gchar *edv_prop_page_get_mime_type_type(EDVPropPageContext *ctx)
{
	if(ctx == NULL)
		return(NULL);

	return(edv_prop_dlg_get_mime_type_type(ctx->prop_dlg));
}


/*
 *	Gets the archive path.
 */
const gchar *edv_prop_page_get_archive_path(EDVPropPageContext *ctx)
{
	if(ctx == NULL)
		return(NULL);

	return(edv_prop_dlg_get_archive_path(ctx->prop_dlg));
}


/*
 *	Switches to the page by name.
 *
 *	The page_name specifies the page to switch to.
 *
 *	Returns TRUE if successfully switched to the page or FALSE on
 *	error.
 */
gboolean edv_prop_page_set_page(
	EDVPropPageContext *ctx,
	const gchar *page_name
)
{
	if(ctx == NULL)
		return(FALSE);

	return(edv_prop_dlg_set_page(
		ctx->prop_dlg,
		page_name
	));
}


/*
 *	Gets the has changes value.
 */
gboolean edv_prop_page_get_has_changes(EDVPropPageContext *ctx)
{
	if(ctx == NULL)
		return(FALSE);

	return(edv_prop_dlg_get_has_changes(ctx->prop_dlg));
}

/*
 *	Sets the has changes flag.
 */
void edv_prop_page_set_has_changes(
	EDVPropPageContext *ctx,
	const gboolean has_changes
)
{
	if(ctx == NULL)
		return;

	edv_prop_dlg_set_has_changes(
		ctx->prop_dlg,
		has_changes
	);
}


/*
 *	Sets busy or ready.
 */
void edv_prop_page_set_busy(
	EDVPropPageContext *ctx,
	const gboolean busy
)
{
	if(ctx == NULL)
		return;

	edv_prop_dlg_set_busy(
		ctx->prop_dlg,
		busy
	);
}


/*
 *	Creates the Icon Selector Dialog on the Properties Dialog as
 *	needed.
 *
 *	Returns the Icon Selector Dialog with a reference count added
 *	or NULL on error.
 */
GtkWidget *edv_prop_page_create_icon_selector(EDVPropPageContext *ctx)
{
	if(ctx == NULL)
		return(NULL);

	return(edv_prop_dlg_create_icon_selector(ctx->prop_dlg));
}
