#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "cfg.h"

#include "guiutils.h"
#include "icon_picker.h"
#include "cdialog.h"
#include "fb.h"
#include "progressdialog.h"
#include "icon_sel_dlg.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_property.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_recycled_obj.h"
#include "libendeavour2-base/edv_recycled_obj_stat.h"
#include "libendeavour2-base/edv_archive_obj.h"
#include "libendeavour2-base/edv_convert_obj.h"
#include "libendeavour2-base/edv_id.h"
#include "edv_pixmap.h"
#include "edv_device.h"
#include "edv_devices_list.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "edv_archive_obj_stat.h"
#include "edv_obj_info_match.h"
#include "edv_utils_gtk.h"
#include "prop_dlg.h"
#include "prop_page.h"
#include "prop_page_private.h"
#include "prop_page_details.h"
#include "prop_page_device.h"
#include "prop_page_device_node.h"
#include "prop_page_directory.h"
#include "prop_page_general.h"
#include "prop_page_id3.h"
#include "prop_page_image.h"
#include "prop_page_link.h"
#include "prop_page_mpeg_audio.h"
#include "edv_emit.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_ok_20x20.xpm"
#include "images/icon_select_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_close_20x20.xpm"
#include "images/icon_properties_48x48.xpm"


typedef struct _EDVPropDlgPage		EDVPropDlgPage;
#define EDV_PROP_DLG_PAGE(p)		((EDVPropDlgPage *)(p))
#define EDV_PROP_DLG_PAGE_KEY		"EDV/PropDlg/Page"


/*
 *	Page Flags:
 */
typedef enum {
	EDV_PROP_DLG_PAGE_END		= (1 << 0) /* Append page to the
						 * end instead
						 * of insert from beginning */
} EDVPropDlgPageFlags;


/* Callbacks */
static void edv_prop_dlg_page_destroy_cb(gpointer data);
static gint edv_prop_dlg_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void edv_prop_dlg_switch_page_cb(
	GtkNotebook *notebook, GtkNotebookPage *page, guint page_num,
	gpointer data
);

static void edv_prop_dlg_ok_cb(GtkWidget *widget, gpointer data);
static void edv_prop_dlg_apply_cb(GtkWidget *widget, gpointer data);
static void edv_prop_dlg_cancel_cb(GtkWidget *widget, gpointer data);

void edv_prop_dlg_reconfigured_cb(EDVPropDlg *d);
void edv_prop_dlg_master_write_protect_changed_cb(
	EDVPropDlg *d,
	const gboolean state
);
void edv_prop_dlg_delete_method_changed_cb(
	EDVPropDlg *d,
	const EDVDeleteMethod delete_method
);

static void edv_prop_dlg_mime_types_list_changed_cb(EDVPropDlg *d);
void edv_prop_dlg_mime_type_added_cb(
	EDVPropDlg *d,
	const gint mt_num, EDVMIMEType *m
);
void edv_prop_dlg_mime_type_modified_cb(
	EDVPropDlg *d,
	const gint mt_num, EDVMIMEType *m
);
void edv_prop_dlg_mime_type_removed_cb(
	EDVPropDlg *d,
	const gint mt_num
);

static void edv_prop_dlg_devices_list_changed_cb(EDVPropDlg *d);
void edv_prop_dlg_device_added_cb(
	EDVPropDlg *d,
	const gint dev_num, EDVDevice *dev
);
void edv_prop_dlg_device_modified_cb(
	EDVPropDlg *d,
	const gint dev_num, EDVDevice *dev
);
void edv_prop_dlg_device_removed_cb(
	EDVPropDlg *d,
	const gint dev_num
);

void edv_prop_dlg_vfs_object_added_cb(
	EDVPropDlg *d,
	const gchar *path,
	EDVVFSObject *obj
);
void edv_prop_dlg_vfs_object_modified_cb(
	EDVPropDlg *d,
	const gchar *path,
	const gchar *new_path,
	EDVVFSObject *obj
);
void edv_prop_dlg_vfs_object_removed_cb(
	EDVPropDlg *d, const gchar *path
);

void edv_prop_dlg_device_mount_cb(
	EDVPropDlg *d,
	const gint dev_num, EDVDevice *dev,
	const gboolean mounted
);

void edv_prop_dlg_recycled_object_added_cb(
	EDVPropDlg *d,
	const gulong index
);
void edv_prop_dlg_recycled_object_modified_cb(
	EDVPropDlg *d,
	const gulong index
);
void edv_prop_dlg_recycled_object_removed_cb(
	EDVPropDlg *d,
	const gulong index
);

/* Utilities */
GList *edv_prop_dlg_update_properties_list(
	EDVCore *core,
	GList *properties_list,
	const EDVLocationType location_type,
	const gchar *path,
	const gulong index,
	const gchar *arch_path,
	GList **meta_data_list_rtn,
	gint *error_code_rtn
);
GtkWidget *edv_prop_dlg_create_icon_selector(EDVPropDlg *d);

/* Apply */
static gboolean edv_prop_dlg_apply(EDVPropDlg *d);
static gboolean edv_prop_dlg_apply_vfs(EDVPropDlg *d);
static gboolean edv_prop_dlg_apply_recycle_bin(EDVPropDlg *d);

/* EDVPropDlgPage */
static EDVPropDlgPage *edv_prop_dlg_page_new(void);
static void edv_prop_dlg_page_delete(EDVPropDlgPage *page);

/* EDVPropDlgPages */
void edv_prop_dlg_emit_update_pages(EDVPropDlg *d);
static void edv_prop_dlg_create_pages(EDVPropDlg *d);
static void edv_prop_dlg_clear_pages(EDVPropDlg *d);

/* EDVPropDlg */
EDVPropDlg *edv_prop_dlg_new(
	EDVCore *core,
	const EDVLocationType location_type,
	const gchar *path,
	const gulong index,
	const gchar *arch_path,
	GList *meta_data_list,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry
);
GtkWidget *edv_prop_dlg_get_toplevel(EDVPropDlg *d);
GtkAccelGroup *edv_prop_dlg_get_accelgrp(EDVPropDlg *d);
gint edv_prop_dlg_get_freeze_count(EDVPropDlg *d);
void edv_prop_dlg_freeze(EDVPropDlg *d);
void edv_prop_dlg_thaw(EDVPropDlg *d);
EDVCore *edv_prop_dlg_get_core(EDVPropDlg *d);
CfgList *edv_prop_dlg_get_cfg_list(EDVPropDlg *d);
EDVLocationType edv_prop_dlg_get_location_type(EDVPropDlg *d);
GList *edv_prop_dlg_get_properties_list(EDVPropDlg *d);
GList *edv_prop_dlg_get_meta_data_list(EDVPropDlg *d);
void edv_prop_dlg_add_meta_data(
	EDVPropDlg *d,
	const gchar *name,
	const gchar *value
);
void edv_prop_dlg_remove_meta_data(
	EDVPropDlg *d,
	const gchar *name
);
const gchar *edv_prop_dlg_get_mime_type_type(EDVPropDlg *d);
const gchar *edv_prop_dlg_get_archive_path(EDVPropDlg *d);
gboolean edv_prop_dlg_set_page(
	EDVPropDlg *d,
	const gchar *page_name
);
gboolean edv_prop_dlg_get_has_changes(EDVPropDlg *d);
void edv_prop_dlg_set_has_changes(
	EDVPropDlg *d,
	const gboolean has_changes
);
void edv_prop_dlg_update_display(EDVPropDlg *d);
void edv_prop_dlg_set_busy(EDVPropDlg *d, const gboolean busy);
gboolean edv_prop_dlg_is_mapped(EDVPropDlg *d);
void edv_prop_dlg_map(EDVPropDlg *d);
void edv_prop_dlg_unmap(EDVPropDlg *d);
void edv_prop_dlg_delete(EDVPropDlg *d);


#define EDV_PROP_DLG_TITLE		"Properties"

#define EDV_PROP_DLG_WIDTH		380
#define EDV_PROP_DLG_HEIGHT		450


/*
 *	Built-In Pages List:
 *
 *	Must fit the EDVPropDlgPage structure.
 *
 *	The last page must be all NULL's.
 */
#define EDV_PROP_DLG_PAGES_LIST		{			\
{	"General",						\
	0,					/* EDVPropDlgPageFlags */ \
	NULL,					/* Pointer to EDVPropDlg \
						 * when initialized, leave \
						 * this NULL */ \
	NULL,					/* Pointer to plugin handle \
						 * when intiailized, leave \
						 * this NULL */	\
	edv_general_prop_page_query_create_cb,			\
	edv_general_prop_page_create_cb,			\
	edv_general_prop_page_update_cb,			\
	NULL,							\
	edv_general_prop_page_apply_vfs_cb,			\
	edv_general_prop_page_apply_recycle_bin_cb,		\
	edv_general_prop_page_destroy_cb,			\
	NULL },							\
{	"Directory",						\
	0, NULL, NULL,						\
	edv_directory_prop_page_query_create_cb,		\
	edv_directory_prop_page_create_cb,			\
	edv_directory_prop_page_update_cb,			\
	edv_directory_prop_page_page_switched_cb,		\
	edv_directory_prop_page_apply_vfs_cb,			\
	edv_directory_prop_page_apply_recycle_bin_cb,		\
	edv_directory_prop_page_destroy_cb,			\
	NULL },							\
{	"Link",							\
	0, NULL, NULL,						\
	edv_link_prop_page_query_create_cb,			\
	edv_link_prop_page_create_cb,				\
	edv_link_prop_page_update_cb,				\
	NULL,							\
	edv_link_prop_page_apply_vfs_cb,			\
	edv_link_prop_page_apply_recycle_bin_cb,		\
	edv_link_prop_page_apply_destroy_cb,			\
	NULL },							\
{	"Device Node",						\
	0, NULL, NULL,						\
	edv_device_node_prop_page_query_create_cb,		\
	edv_device_node_prop_page_create_cb,			\
	edv_device_node_prop_page_update_cb,			\
	NULL,							\
	edv_device_node_prop_page_apply_vfs_cb,			\
	edv_device_node_prop_page_apply_recycle_bin_cb,		\
	edv_device_node_prop_page_destroy_cb,			\
	NULL },							\
{	"Device",						\
	0, NULL, NULL,						\
	edv_device_prop_page_query_create_cb,			\
	edv_device_prop_page_create_cb,				\
	edv_device_prop_page_update_cb,				\
	NULL,							\
	NULL,							\
	NULL,							\
	edv_device_prop_page_destroy_cb,			\
	NULL },							\
{	"Image",						\
	0, NULL, NULL,						\
	edv_image_prop_page_query_create_cb,			\
	edv_image_prop_page_create_cb,				\
	edv_image_prop_page_update_cb,				\
	edv_image_prop_page_page_switched_cb,			\
	NULL,							\
	NULL,							\
	edv_image_prop_page_destroy_cb,				\
	NULL },							\
{	"ID3",							\
	0, NULL, NULL,						\
	edv_id3_prop_page_query_create_cb,			\
	edv_id3_prop_page_create_cb,				\
	edv_id3_prop_page_update_cb,				\
	edv_id3_prop_page_page_switched_cb,			\
	edv_id3_prop_page_apply_vfs_cb,				\
	edv_prop_dlg_i3d_page_apply_recycle_bin_cb,		\
	edv_id3_prop_page_destroy_cb,				\
	NULL },							\
{	"MPEG Audio",						\
	0, NULL, NULL,						\
	edv_mpeg_audio_prop_page_query_create_cb,		\
	edv_mpeg_audio_prop_page_create_cb,			\
	edv_mpeg_audio_prop_page_update_cb,			\
	edv_mpeg_audio_prop_page_page_switched_cb,		\
	NULL,							\
	NULL,							\
	edv_mpeg_audio_prop_page_destroy_cb,			\
	NULL },							\
{	"Details",						\
	EDV_PROP_DLG_PAGE_END, NULL, NULL,			\
	edv_details_prop_page_query_create_cb,			\
	edv_details_prop_page_create_cb,			\
	edv_details_prop_page_update_cb,			\
	NULL,							\
	NULL,							\
	NULL,							\
	edv_details_prop_page_destroy_cb,			\
	NULL },							\
{	NULL,							\
	0, NULL, NULL,						\
	NULL,							\
	NULL,							\
	NULL,							\
	NULL,							\
	NULL,							\
	NULL,							\
	NULL,							\
	NULL }							\
}


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Properties Dialog:
 */
struct _EDVPropDlg {

	GtkWidget       *toplevel;
	GtkAccelGroup   *accelgrp;
	gint            busy_count,
			freeze_count;
	gboolean	has_changes;
	gchar		*title;
	EDVCore		*core;

	GtkWidget	*main_vbox,
			*notebook,
			*ok_btn,
			*apply_btn,
			*cancel_btn,
			*close_btn;

	/* Object's Properties */
	EDVLocationType	location_type;
	GList		*properties_list,	/* Basic object properties, a
						 * GList of EDVProperty *
						 * properties */
			*meta_data_list;	/* Format and type-specific
						 * properties, a GList of
						 * EDVProperty * properties */
	gchar		*mime_type,
			*arch_path;		/* Path to the archive that
						 * the object is in (if
						 * location_type is
						 * EDV_LOCATION_TYPE_ARCHIVE) */

	EDVPropPageContext	*page_context;

	/* Pages List, GList of EDVPropDlgPage * pages */
	GList		*pages_list;
	gint		cur_page_num;		/* Current displayed page or
						 * -1 for none */

	/* Icon Selection Dialog */
	GtkWidget	*icon_sel_dlg;
};


/*
 *	Page:
 */
struct _EDVPropDlgPage	{

	gchar		*name;
	EDVPropDlgPageFlags	flags;
	EDVPropDlg	*prop_dlg;

	gpointer	plugin_handle;		/* For future implmentation
						 * of loading plugins for
						 * EDVPropDlg pages */

	EDVPropPageQueryCreateFunc query_create_cb;
	EDVPropPageCreateFunc	create_cb;
	EDVPropPageUpdateFunc	update_cb;
	EDVPropPagePageSwitchedFunc	page_switched_cb;
	EDVPropPageApplyVFSFunc	apply_vfs_cb;
	EDVPropPageApplyRecycleBinFunc	apply_recbin_cb;
	EDVPropPageDestroyFunc	destroy_cb;

	/* User data obtained from create_cb() */
	gpointer	data;
};


/*
 *	EDVPropDlgPage "destroy" signal callback.
 */
static void edv_prop_dlg_page_destroy_cb(gpointer data)
{
	EDVPropDlg *d;
	EDVPropDlgPage *page = EDV_PROP_DLG_PAGE(data);
	if(page == NULL)
		return;

	d = page->prop_dlg;

	/* Notify this page that is destroyed */
	if(page->destroy_cb != NULL)
		page->destroy_cb(
			d->page_context,
			page->data
		);

	/* Remove this page from the Property Dialog's list of pages */
	if(page->prop_dlg != NULL)
		d->pages_list = g_list_remove(
			d->pages_list,
			page
		);

	/* Delete this page */
	edv_prop_dlg_page_delete(page);
}

/*
 *	GtkWindow "delete_event" signal callback.
 */
static gint edv_prop_dlg_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	EDVPropDlg *d = EDV_PROP_DLG(data);
	if(d == NULL)
		return(FALSE);

	if(d->freeze_count > 0)
		return(TRUE);

	edv_prop_dlg_unmap(d);

	return(TRUE);
}

/*
 *	GtkNotebook "switch_page" signal callback.
 */
static void edv_prop_dlg_switch_page_cb(
	GtkNotebook *notebook, GtkNotebookPage *page, guint page_num,
	gpointer data
)
{
	gint prev_page_num;
	EDVPropDlgPage *dpage;
	EDVPropDlg *d = EDV_PROP_DLG(data);
	if(d == NULL)
		return;

	if(d->freeze_count > 0)
		return;

	d->freeze_count++;

	/* Get the previous page number and set the current page number */
	prev_page_num = d->cur_page_num;
	d->cur_page_num = (gint)page_num;

	/* Was there a previous page displayed? */
	if(prev_page_num > -1)
	{
		/* Get the page that we are switching away from */
		dpage = EDV_PROP_DLG_PAGE(g_list_nth_data(
			d->pages_list,
			(guint)prev_page_num
		));
		if(dpage != NULL)
		{
			/* Notify the page that we are switching away from it */
			if(dpage->page_switched_cb != NULL)
				dpage->page_switched_cb(
					d->page_context,
					FALSE,	/* Away from this page */
					dpage->data
				);
		}
	}

	/* Get the page that we are switching to */
	dpage = EDV_PROP_DLG_PAGE(g_list_nth_data(
		d->pages_list,
		page_num
	));
	if(dpage != NULL)
	{
		/* Notify the page that we are switching to it */
		if(dpage->page_switched_cb != NULL)
			dpage->page_switched_cb(
				d->page_context,
				TRUE,		/* To this page */
				dpage->data
			);
	}

	d->freeze_count--;
}


/*
 *	OK callback.
 */
static void edv_prop_dlg_ok_cb(GtkWidget *widget, gpointer data)
{
	EDVPropDlg *d = EDV_PROP_DLG(data);
	if(d == NULL)
		return;

	if(d->freeze_count > 0)
		return;

	edv_prop_dlg_set_busy(d, TRUE);

	/* Call each page's apply callback to apply any changes */
	edv_prop_dlg_apply(d);

	/* Unmap the EDVPropDlg */
	edv_prop_dlg_unmap(d);

	/* Clear the has changes flag */
	edv_prop_dlg_set_has_changes(
		d,
		FALSE
	);

	edv_prop_dlg_set_busy(d, FALSE);
}

/*
 *	Apply callback.
 */
static void edv_prop_dlg_apply_cb(GtkWidget *widget, gpointer data)
{
	EDVPropDlg *d = EDV_PROP_DLG(data);
	if(d == NULL)
		return;

	if(d->freeze_count > 0)
		return;

	edv_prop_dlg_set_busy(d, TRUE);

	/* Call each page's apply callback to apply any changes */
	edv_prop_dlg_apply(d);

	/* Clear the has changes flag */
	edv_prop_dlg_set_has_changes(
		d,
		FALSE
	);

	edv_prop_dlg_set_busy(d, FALSE);
}

/*
 *	Cancel callback.
 */
static void edv_prop_dlg_cancel_cb(GtkWidget *widget, gpointer data)
{
	EDVPropDlg *d = EDV_PROP_DLG(data);
	if(d == NULL)
		return;

	if(d->freeze_count > 0)
		return;

	/* Unmap the EDVPropDlg */
	edv_prop_dlg_unmap(d);
}


/*
 *	Reconfigured callback.
 */
void edv_prop_dlg_reconfigured_cb(EDVPropDlg *d)
{
	GtkRcStyle	*standard_rcstyle,
			*lists_rcstyle;
	EDVCore *core;

	if(d == NULL)
		return;

	core = d->core;
	standard_rcstyle = core->standard_rcstyle;
	lists_rcstyle = core->lists_rcstyle;

	/* Notify all the pages to update their displayed values */
	edv_prop_dlg_emit_update_pages(d);

	/* Update the RC styles */
	if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			d->toplevel,
			standard_rcstyle
		);

	edv_prop_dlg_update_display(d);
}

/*
 *	Master Write Protect changed callback.
 */
void edv_prop_dlg_master_write_protect_changed_cb(
	EDVPropDlg *d,
	const gboolean state
)
{
	if(d == NULL)
		return;

	if(d->freeze_count > 0)
		return;

	edv_prop_dlg_emit_update_pages(d);

	edv_prop_dlg_update_display(d);
}

/*
 *	Delete Method changed callback.
 */
void edv_prop_dlg_delete_method_changed_cb(
	EDVPropDlg *d,
	const EDVDeleteMethod delete_method
)
{
	if(d == NULL)
		return;

	if(d->freeze_count > 0)
		return;

	edv_prop_dlg_update_display(d);
}


/*
 *	MIME Types list changed callback.
 */
static void edv_prop_dlg_mime_types_list_changed_cb(EDVPropDlg *d)
{
	const gchar *path;
	gchar *type_string;
	CfgList *cfg_list;
	EDVCore *core;

	if(d == NULL)
		return;

	core = d->core;
	cfg_list = core->cfg_list;

	path = edv_properties_list_get_s(
		d->properties_list,
		EDV_PROP_NAME_PATH
	);
	if(path == NULL)
		path = edv_properties_list_get_s(
			d->properties_list,
			EDV_PROP_NAME_NAME
		);

	/* Get the new MIME Type type string */
	(void)edv_match_object_type_string(
		core->mime_types_list,
		(EDVObjectType)edv_properties_list_get_i(
			d->properties_list,
			EDV_PROP_NAME_TYPE
		),
		path,
		(EDVPermissionFlags)edv_properties_list_get_i(
			d->properties_list,
			EDV_PROP_NAME_PERMISSIONS
		),
		&type_string
	);
	if(type_string != NULL)
	{
		/* Update the current MIME Type's type string */
		g_free(d->mime_type);
		d->mime_type = g_strdup(type_string);
		g_free(type_string);
	}

	/* Notify all the pages to update their displayed values */
	edv_prop_dlg_emit_update_pages(d);

	edv_prop_dlg_update_display(d);
}

/*
 *	MIME Type added callback.
 */
void edv_prop_dlg_mime_type_added_cb(
	EDVPropDlg *d,
	const gint mt_num, EDVMIMEType *m
)
{
	edv_prop_dlg_mime_types_list_changed_cb(d);
}

/*
 *	MIME Type modified callback.
 */
void edv_prop_dlg_mime_type_modified_cb(
	EDVPropDlg *d,
	const gint mt_num, EDVMIMEType *m
)
{
	edv_prop_dlg_mime_types_list_changed_cb(d);
}

/*
 *	MIME Type removed callback.
 */
void edv_prop_dlg_mime_type_removed_cb(
	EDVPropDlg *d,
	const gint mt_num
)
{
	edv_prop_dlg_mime_types_list_changed_cb(d);
}


/*
 *	Devices list changed callback.
 */
static void edv_prop_dlg_devices_list_changed_cb(EDVPropDlg *d)
{
	if(d == NULL)
		return;

	/* Notify all the pages to update their displayed values */
	edv_prop_dlg_emit_update_pages(d);

	edv_prop_dlg_update_display(d);
}

/*
 *	Device added callback.
 */
void edv_prop_dlg_device_added_cb(
	EDVPropDlg *d,
	const gint dev_num, EDVDevice *dev
)
{
	edv_prop_dlg_devices_list_changed_cb(d);
}

/*
 *	Device modified callback.
 */
void edv_prop_dlg_device_modified_cb(
	EDVPropDlg *d,
	const gint dev_num, EDVDevice *dev
)
{
	edv_prop_dlg_devices_list_changed_cb(d);
}

/*
 *	Device removed callback.
 */
void edv_prop_dlg_device_removed_cb(
	EDVPropDlg *d,
	const gint dev_num
)
{
	edv_prop_dlg_devices_list_changed_cb(d);
}


/*
 *	EDVVFSObject added callback.
 */
void edv_prop_dlg_vfs_object_added_cb(
	EDVPropDlg *d,
	const gchar *path,
	EDVVFSObject *obj
)
{
	gchar *cur_path;
	EDVLocationType location_type;
	EDVCore *core;

	if((d == NULL) || STRISEMPTY(path))
		return;

	location_type = d->location_type;
	core = d->core;

	switch(location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
		cur_path = STRDUP(edv_properties_list_get_s(
			d->properties_list,
			EDV_PROP_NAME_PATH
		));
		if(cur_path != NULL)
		{
			gint error_code;

			/* Update the properties list with the object's
			 * current values
			 *
			 * We do not check if the added object is our object
			 * because any change to any object may affect the
			 * properties of our object
			 */
			d->properties_list = edv_prop_dlg_update_properties_list(
				core,
				d->properties_list,
				location_type,
				cur_path,
				0,		/* Not used */
				NULL,		/* Not used */
				&d->meta_data_list,
				&error_code
			);

			g_free(cur_path);
		}
		break;

	  case EDV_LOCATION_TYPE_RECYCLE_BIN:
		break;

	  case EDV_LOCATION_TYPE_ARCHIVE:
		break;
	}

	/* Notify all the pages to update their displayed values */
	edv_prop_dlg_emit_update_pages(d);

	edv_prop_dlg_update_display(d);
}

/*
 *	EDVVFSObject modified callback.
 */
void edv_prop_dlg_vfs_object_modified_cb(
	EDVPropDlg *d,
	const gchar *path,
	const gchar *new_path,
	EDVVFSObject *obj
)
{
	gchar *cur_path;
	EDVLocationType location_type;
	EDVCore *core;

	if((d == NULL) || STRISEMPTY(path))
		return;

	if(new_path == NULL)
		new_path = path;

	location_type = d->location_type;
	core = d->core;

	switch(location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
		cur_path = STRDUP(edv_properties_list_get_s(
			d->properties_list,
			EDV_PROP_NAME_PATH
		));
		if(cur_path != NULL)
		{
			gboolean name_changed = FALSE;
			gint error_code;

			/* Is our object the one that was modified? */
			if(!strcmp((const char *)cur_path, (const char *)path))
			{
				/* Change in names or paths? */
				if(strcmp((const char *)cur_path, (const char *)new_path))
				{
					/* Update the name and path in the properties list */
					d->properties_list = edv_properties_list_set_s(
						d->properties_list,
						EDV_PROP_NAME_PATH,
						new_path,
						TRUE
					);
					d->properties_list = edv_properties_list_set_s(
						d->properties_list,
						EDV_PROP_NAME_NAME,
						g_basename(new_path),
						TRUE
					);

					g_free(cur_path);
					cur_path = STRDUP(edv_properties_list_get_s(
						d->properties_list,
						EDV_PROP_NAME_PATH
					));

					name_changed = TRUE;
				}
			}

			/* Update the properties list with the object's
			 * current values
			 *
			 * We do not check if the modified object is our object
			 * because any change to any object may affect the
			 * properties of our object
			 */
			d->properties_list = edv_prop_dlg_update_properties_list(
				core,
				d->properties_list,
				location_type,
				cur_path,
				0,		/* Not used */
				NULL,		/* Not used */
				&d->meta_data_list,
				&error_code
			);

			/* Did the name change? */
			if(name_changed)
			{
				/* We need to update the MIME Type when the
				 * name has changed because changing the name can
				 * also change the object's MIME Type
				 */
				gchar *type_string;
				(void)edv_match_object_type_string(
					core->mime_types_list,
					(EDVObjectType)edv_properties_list_get_i(
						d->properties_list,
						EDV_PROP_NAME_TYPE
					),
					cur_path,
					(EDVPermissionFlags)edv_properties_list_get_i(
						d->properties_list,
						EDV_PROP_NAME_PERMISSIONS
					),
					&type_string
				);
				if(type_string != NULL)
				{
					g_free(d->mime_type);
					d->mime_type = g_strdup(type_string);
					g_free(type_string);
				}
			}

			g_free(cur_path);
		}
		break;

	  case EDV_LOCATION_TYPE_RECYCLE_BIN:
		break;

	  case EDV_LOCATION_TYPE_ARCHIVE:
		break;
	}

	/* Notify all the pages to update their displayed values */
	edv_prop_dlg_emit_update_pages(d);

	edv_prop_dlg_update_display(d);
}

/*
 *	EDVVFSObject removed callback.
 */
void edv_prop_dlg_vfs_object_removed_cb(
	EDVPropDlg *d,
	const gchar *path
)
{
	gchar *cur_path;
	EDVLocationType location_type;
	EDVCore *core;

	if((d == NULL) || STRISEMPTY(path))
		return;

	location_type = d->location_type;
	core = d->core;

	switch(location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
		cur_path = STRDUP(edv_properties_list_get_s(
			d->properties_list,
			EDV_PROP_NAME_PATH
		));
		if(cur_path != NULL)
		{
			gint error_code;

			/* Is our object the one that was removed? */
			if(!strcmp((const char *)cur_path, (const char *)path))
			{
				/* Unmap this Properties Dialog since
				 * it no longer needs to display the
				 * now non-existant object
				 */
				edv_prop_dlg_unmap(d);
				g_free(cur_path);
				return;
			}

			/* Update the properties list with the object's
			 * current values
			 *
			 * This is needed because removing any object can
			 * cause the properties of our object to change
			 */
			d->properties_list = edv_prop_dlg_update_properties_list(
				core,
				d->properties_list,
				location_type,
				cur_path,
				0,		/* Not used */
				NULL,		/* Not used */
				&d->meta_data_list,
				&error_code
			);

			g_free(cur_path);
		}
		break;

	  case EDV_LOCATION_TYPE_RECYCLE_BIN:
		break;

	  case EDV_LOCATION_TYPE_ARCHIVE:
		break;
	}

	/* Notify all the pages to update their displayed values */
	edv_prop_dlg_emit_update_pages(d);

	edv_prop_dlg_update_display(d);
}


/*
 *	EDVDevice mount/unmount callback.
 */
void edv_prop_dlg_device_mount_cb(
	EDVPropDlg *d, 
	const gint dev_num, EDVDevice *dev,
	const gboolean mounted
)
{
	if(d == NULL)
		return;

	/* Notify all the pages to update their displayed values */
	edv_prop_dlg_emit_update_pages(d);

	edv_prop_dlg_update_display(d);
}


/*
 *	EDVRecycledObject added callback.
 */
void edv_prop_dlg_recycled_object_added_cb(
	EDVPropDlg *d,
	const gulong index
)
{
	gulong cur_index;
	EDVLocationType location_type;
	EDVCore *core;

	if(d == NULL)
		return;

	location_type = d->location_type;
	core = d->core;

	switch(location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
		break;

	  case EDV_LOCATION_TYPE_RECYCLE_BIN:
		cur_index = edv_properties_list_get_ul(
			d->properties_list,
			EDV_PROP_NAME_INDEX
		);
		if(cur_index != 0)
		{
			gint error_code;

			/* Update the properties list with the object's
			 * current values
			 *
			 * We do not check if the added object is our object
			 * because any change to any object may affect the
			 * properties of our object
			 */
			d->properties_list = edv_prop_dlg_update_properties_list(
				core,
				d->properties_list,
				location_type,
				NULL,		/* Not used */
				cur_index,
				NULL,		/* Not used */
				&d->meta_data_list,
				&error_code
			);
		}
		break;

	  case EDV_LOCATION_TYPE_ARCHIVE:
		break;
	}

	/* Notify all the pages to update their displayed values */
	edv_prop_dlg_emit_update_pages(d);

	edv_prop_dlg_update_display(d);
}

/*
 *	EDVRecycledObject modified callback.
 */
void edv_prop_dlg_recycled_object_modified_cb(
	EDVPropDlg *d,
	const gulong index
)
{
	gulong cur_index;
	EDVLocationType location_type;
	EDVCore *core;

	if(d == NULL)
		return;

	location_type = d->location_type;
	core = d->core;

	switch(location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
		break;

	  case EDV_LOCATION_TYPE_RECYCLE_BIN:
		cur_index = edv_properties_list_get_ul(
			d->properties_list,
			EDV_PROP_NAME_INDEX
		);
		if(cur_index != 0)
		{
			gboolean name_changed = FALSE;
			gint error_code;
			const gchar *cur_name;
			gchar *prev_name = STRDUP(edv_properties_list_get_s(
				d->properties_list,
				EDV_PROP_NAME_NAME
			));

			/* Update the properties list with the object's
			 * current values
			 *
			 * We do not check if the modified object is our object
			 * because any change to any object may affect the
			 * properties of our object
			 */
			d->properties_list = edv_prop_dlg_update_properties_list(
				core,
				d->properties_list,
				location_type,
				NULL,		/* Not used */
				cur_index,
				NULL,		/* Not used */
				&d->meta_data_list,
				&error_code
			);

			/* Check if the name has changed */
			cur_name = edv_properties_list_get_s(
				d->properties_list,
				EDV_PROP_NAME_NAME
			);
			if((prev_name != NULL) && (cur_name != NULL))
			{
				if(strcmp((const char *)prev_name, (const char *)cur_name))
					name_changed = TRUE;
			}

			/* Did the name change? */
			if(name_changed)
			{
				/* We need to update the MIME Type when the
				 * name has changed because changing the name can
				 * also change the object's MIME Type
				 */
				gchar *type_string;
				(void)edv_match_object_type_string(
					core->mime_types_list,
					(EDVObjectType)edv_properties_list_get_i(
						d->properties_list,
						EDV_PROP_NAME_TYPE
					),
					cur_name,
					(EDVPermissionFlags)edv_properties_list_get_i(
						d->properties_list,
						EDV_PROP_NAME_PERMISSIONS
					),
					&type_string
				);
				if(type_string != NULL)
				{
					g_free(d->mime_type);
					d->mime_type = g_strdup(type_string);
					g_free(type_string);
				}
			}

			g_free(prev_name);
		}
		break;

	  case EDV_LOCATION_TYPE_ARCHIVE:
		break;
	}

	/* Notify all the pages to update their displayed values */
	edv_prop_dlg_emit_update_pages(d);

	edv_prop_dlg_update_display(d);
}

/*
 *	EDVRecycledObject removed callback.
 */
void edv_prop_dlg_recycled_object_removed_cb(
	EDVPropDlg *d,
	const gulong index
)
{
	gulong cur_index;
	EDVLocationType location_type;
	EDVCore *core;

	if(d == NULL)
		return;

	location_type = d->location_type;
	core = d->core;

	switch(location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
		break;

	  case EDV_LOCATION_TYPE_RECYCLE_BIN:
		cur_index = edv_properties_list_get_ul(
			d->properties_list,
			EDV_PROP_NAME_INDEX
		);
		if(cur_index != 0)
		{
			gint error_code;

			/* Is the removed object our object? */
			if(cur_index == index)
			{
				edv_prop_dlg_unmap(d);
				return;
			}

			/* Update the properties list with the object's
			 * current values
			 *
			 * This is needed because removing any object can
			 * cause the properties of our object to change
			 */
			d->properties_list = edv_prop_dlg_update_properties_list(
				core,
				d->properties_list,
				location_type,
				NULL,		/* Not used */
				cur_index,
				NULL,		/* Not used */
				&d->meta_data_list,
				&error_code
			);
		}
		break;

	  case EDV_LOCATION_TYPE_ARCHIVE:
		break;
	}

	/* Notify all the pages to update their displayed values */
	edv_prop_dlg_emit_update_pages(d);

	edv_prop_dlg_update_display(d);
}


/*
 *	Creates/updates the properties list with the object's statistics.
 *
 *	The properties_list specifies the properties list, a GList of
 *	EDVProperty * properties.
 *
 *	The location_type specifies the object's location type.
 *
 *	The path specifies the object if location_type is
 *	EDV_LOCATION_TYPE_VFS or EDV_LOCATION_TYPE_ARCHIVE.
 *
 *	The index specifies the object if location_type is
 *	EDV_LOCATION_TYPE_RECYCLE_BIN.
 *
 *	The arch_path specifies the archive that the object is in
 *	if location_type is EDV_LOCATION_TYPE_ARCHIVE.
 *
 *	If meta_data_list_rtn is not NULL then *meta_data_list_rtn
 *	will be updated with the meta datas list obtained from the
 *	current object's meta datas list.
 *
 *	Returns the GList of EDVProperty * properties.
 */
GList *edv_prop_dlg_update_properties_list(
	EDVCore *core,
	GList *properties_list,
	const EDVLocationType location_type,
	const gchar *path,
	const gulong index,
	const gchar *arch_path,
	GList **meta_data_list_rtn,
	gint *error_code_rtn
)
{
	/* Clear the error property from the meta datas list */
	if(meta_data_list_rtn != NULL)
		*meta_data_list_rtn = edv_properties_list_remove(
			*meta_data_list_rtn,
			EDV_PROP_NAME_ERROR
		);

	/* Update the properties list based on the location type */
	switch(location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
		if(path != NULL)
		{
			EDVVFSObject *obj = edv_vfs_object_lstat(path);
			if(obj != NULL)
			{
				properties_list = edv_properties_list_delete(properties_list);
				properties_list = edv_convert_vfs_object_to_properties_list(
					properties_list,
					obj
				);
				if(meta_data_list_rtn != NULL)
					*meta_data_list_rtn = edv_properties_list_update(
						*meta_data_list_rtn,
						obj->meta_data_list,
						TRUE,	/* Create as needed */
						FALSE	/* Do not remove nonexistant */
					);
				edv_vfs_object_delete(obj);
				if(error_code_rtn != NULL)
					*error_code_rtn = 0;
				return(properties_list);
			}
			else
			{
				if(error_code_rtn != NULL)
					*error_code_rtn = errno;
				return(properties_list);
			}

		}
		break;

	  case EDV_LOCATION_TYPE_RECYCLE_BIN:
		if((core != NULL) && (index != 0l))
		{
			CfgList *cfg_list = core->cfg_list;
			EDVRecycledObject *obj = edv_recycled_object_stat(
				EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX),
				index
			);
			if(obj != NULL)
			{
				properties_list = edv_properties_list_delete(properties_list);
				properties_list = edv_convert_recycled_object_to_properties_list(
					properties_list,
					obj
				);
				if(meta_data_list_rtn != NULL)
					*meta_data_list_rtn = edv_properties_list_update(
						*meta_data_list_rtn,
						obj->meta_data_list,
						TRUE,	/* Create as needed */
						FALSE	/* Do not remove nonexistant */
					);
				edv_recycled_object_delete(obj);
				if(error_code_rtn != NULL)
					*error_code_rtn = 0;
				return(properties_list);
			}
			else
			{
				if(error_code_rtn != NULL)
					*error_code_rtn = errno;
				return(properties_list);
			}
		}
		break;

	  case EDV_LOCATION_TYPE_ARCHIVE:
		if((core != NULL) && (arch_path != NULL))
		{
			EDVArchiveObject *obj = edv_archive_object_stat(
				core->cfg_list,
				arch_path,
				path,
				NULL
			);
			if(obj != NULL)
			{
				properties_list = edv_properties_list_delete(properties_list);
				properties_list = edv_convert_archive_object_to_properties_list(
					properties_list,
					obj
				);
				if(meta_data_list_rtn != NULL)
					*meta_data_list_rtn = edv_properties_list_update(
						*meta_data_list_rtn,
						obj->meta_data_list,
						TRUE,	/* Create as needed */
						FALSE	/* Do not remove nonexistant */
					);
				edv_archive_object_delete(obj);
				if(error_code_rtn != NULL)
					*error_code_rtn = 0;
				return(properties_list);
			}
			else
			{
				if(error_code_rtn != NULL)
					*error_code_rtn = errno;
				return(properties_list);
			}
		}
		break;
	}

	if(error_code_rtn != NULL)
		*error_code_rtn = EINVAL;

	return(properties_list);
}

/*
 *	Creates the Icon Selector Dialog on the Properties Dialog as
 *	needed.
 *
 *	Returns the Icon Selector Dialog with a reference count added
 *	or NULL on error.
 */
GtkWidget *edv_prop_dlg_create_icon_selector(EDVPropDlg *d)
{
	gulong load_int;
	gchar *default_icons_path;
	GtkWidget *icon_sel_dlg;
	IconPickerFlags ip_flags;
	CfgList *cfg_list;
	EDVListsPointerOpButton2 lists_pointer_op_button2;
	EDVCore *core;

	if(d == NULL)
		return(NULL);

	/* If the Icon Selector Dialog has already been created then
	 * just add a reference count to the existing one and return it
	 */
	if(d->icon_sel_dlg != NULL)
		return(IconSelDlgRef(d->icon_sel_dlg));

	core = d->core;
	cfg_list = core->cfg_list;

	/* Set the Icon Picker flags */
	ip_flags = ICON_PICKER_CAN_CLEAR_ICON |
		ICON_PICKER_ACCEPT_SMALLER_SIZE |
		ICON_PICKER_ACCEPT_LARGER_SIZE |
		ICON_PICKER_WARN_DIFFERENT_SIZE |
		ICON_PICKER_ACCEPT_DND_DROP |
		ICON_PICKER_PROVIDE_DND_DRAG;
	if(EDV_GET_B(EDV_CFG_PARM_LISTS_DOUBLE_BUFFER))
		ip_flags |= ICON_PICKER_DOUBLE_BUFFER;
	if(EDV_GET_B(EDV_CFG_PARM_SHOW_TEXTTIPS))
		ip_flags |= ICON_PICKER_SHOW_TEXTTIPS;

	lists_pointer_op_button2 = (EDVListsPointerOpButton2)EDV_GET_I(
		EDV_CFG_PARM_LISTS_POINTER_OP_BUTTON2
	);

	/* Load images interval (in milliseconds) */
	load_int = edv_get_interval_from_load_images_priority(
		(EDVListsLoadImagesPriority)EDV_GET_I(
			EDV_CFG_PARM_LISTS_LOAD_IMAGES_PRIORITY
		)
	);

	/* Default icons location */
	default_icons_path = g_strconcat(
		EDV_GET_S(EDV_CFG_PARM_DIR_GLOBAL),
		G_DIR_SEPARATOR_S,
			EDV_NAME_ICONS_SUBDIR,
			NULL
		);

	/* Create the Icon Selector Dialog for this Properties Dialog */
	d->icon_sel_dlg = icon_sel_dlg = IconSelDlgNew(
		((ip_flags & ICON_PICKER_DOUBLE_BUFFER) ?
			ICON_SEL_DLG_DOUBLE_BUFFER : 0) |
		((ip_flags & ICON_PICKER_SHOW_TEXTTIPS) ?
			ICON_SEL_DLG_SHOW_TEXTTIPS : 0) |
		((lists_pointer_op_button2 == EDV_LISTS_POINTER_OP_BUTTON2_SCROLL_XY) ?
			ICON_SEL_DLG_ENABLE_DRAG_SCROLL : 0),
		GTK_ORIENTATION_HORIZONTAL,
		-1,				/* Default list height */
		50, 50,				/* Thumb size */
		0,				/* No thumb border */
		default_icons_path,
		load_int
	);

	g_free(default_icons_path);

	return(IconSelDlgRef(d->icon_sel_dlg));
}


/*
 *	Apply.
 */
static gboolean edv_prop_dlg_apply(EDVPropDlg *d)
{
	gboolean status;
	GtkWidget *toplevel;
	EDVCore *core;

	if(d == NULL)
		return(FALSE);

	d->freeze_count++;

	toplevel = d->toplevel;
	core = d->core;

	/* Check if the global write protect is on */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		d->freeze_count--;
		return(FALSE);
	}

	status = FALSE;
	switch(d->location_type)
	{
	    case EDV_LOCATION_TYPE_VFS:
		status = edv_prop_dlg_apply_vfs(d);
		break;
	    case EDV_LOCATION_TYPE_RECYCLE_BIN:
		status = edv_prop_dlg_apply_recycle_bin(d);
		break;
	    case EDV_LOCATION_TYPE_ARCHIVE:
		edv_play_sound_warning(core);
		edv_message_warning(
"Apply Failed",
"Unable to modify the properties of an object in an archive.",
			NULL,
			toplevel
		);
		break;
	}

	d->freeze_count--;

	return(status);
}

/*
 *	Apply VFS object.
 *
 *	Calls all the pages that have VFS object apply callbacks set
 *	and checks if any of the pages have reported that they have
 *	modified the object. If the object has been modified then a
 *	VFS object modified signal will be emitted.
 *
 *	Returns TRUE if any of the pages have reported that they have
 *	modified the VFS object or FALSE if no page reported modifying
 *	the VFS object.
 */
static gboolean edv_prop_dlg_apply_vfs(EDVPropDlg *d)
{
	gboolean status = FALSE;
	GList *glist;
	const EDVObjectType type = (EDVObjectType)edv_properties_list_get_i(
		d->properties_list,
		EDV_PROP_NAME_TYPE
	);
	EDVCore *core = d->core;
	EDVPropDlgPage *page;

	for(glist = d->pages_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		page = EDV_PROP_DLG_PAGE(glist->data);
		if(page == NULL)
			continue;

		if(page->apply_vfs_cb != NULL)
		{
			if(page->apply_vfs_cb(
				d->page_context,
				type,
				d->properties_list,
				page->data
			))
			{
				if(!status)
					status = TRUE;
			}
		}
	}

	/* Did any of the callbacks report that they made changes to
	 * the object?
	 */
	if(status)
	{
		gchar *path = STRDUP(edv_properties_list_get_s(
			d->properties_list,
			EDV_PROP_NAME_PATH
		));
		const gulong index = edv_properties_list_get_ul(
			d->properties_list,
			EDV_PROP_NAME_INDEX
		);
		if(path != NULL)
		{
			gint error_code;
			EDVVFSObject *obj;

			/* Update the properties list with the object's
			 * current values
			 */
			d->properties_list = edv_prop_dlg_update_properties_list(
				core,
				d->properties_list,
				d->location_type,
				path,
				index,		/* Not used */
				d->arch_path,	/* Not used */
				&d->meta_data_list,
				&error_code
			);

			/* Notify about this object being modified
			 *
			 * This will call the property dialog's VFS object
			 * modifyed callback which will instruct all the pages
			 * to update their displayed values
			 */
			obj = edv_vfs_object_lstat(path);
			if(obj != NULL)
			{
				edv_emit_vfs_object_modified(
					core,
					path,
					path,
					obj
				);
				edv_vfs_object_delete(obj);
			}
			else
			{
				const gint error_code = (gint)errno;
				if(error_code == ENOENT)
					edv_emit_vfs_object_removed(
						core,
						path
					);
			}

			g_free(path);
		}
	}

	return(status);
}

/*
 *	Apply recycled object.
 *
 *	Calls all the pages that have recycled object apply callbacks
 *	set and checks if any of the pages have reported that they
 *	have modified the recycled object. If the recycled object has
 *	been modified then a recycled object modified signal will be
 *	emitted.
 *
 *	Returns TRUE if any of the pages have reported that they have
 *	modified the recycled object or FALSE if no page reported
 *	modifying the recycled object.
 */
static gboolean edv_prop_dlg_apply_recycle_bin(EDVPropDlg *d)
{
	gboolean status = FALSE;
	GList *glist;
	const EDVObjectType type = (EDVObjectType)edv_properties_list_get_i(
		d->properties_list,
		EDV_PROP_NAME_TYPE
	);
	EDVCore *core = d->core;
	EDVPropDlgPage *page;

	for(glist = d->pages_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		page = EDV_PROP_DLG_PAGE(glist->data);
		if(page == NULL)
			continue;

		if(page->apply_recbin_cb != NULL)
		{
			if(page->apply_recbin_cb(
				d->page_context,
				type,
				d->properties_list,
				page->data
			))
			{
				if(!status)
					status = TRUE;
			}
		}
	}

	/* Did any of the callbacks report that they made changes to
	 * the object?
	 */
	if(status)
	{
		const gulong index = edv_properties_list_get_ul(
			d->properties_list,
			EDV_PROP_NAME_INDEX
		);
		if(index != 0l)
		{
			gint error_code;

			/* Update the properties list with the object's
			 * current values
			 */
			d->properties_list = edv_prop_dlg_update_properties_list(
				core,
				d->properties_list,
				d->location_type,
				NULL,		/* Not used */
				index,
				d->arch_path,	/* Not used */
				&d->meta_data_list,
				&error_code
			);

			/* Notify about this recycled object being modified
			 *
			 * This will call the property dialog's recycled
			 * object modifyed callback which will instruct all
			 * the pages to update their displayed values
			 */
			edv_emit_recycled_object_modified(core, index);
		}
	}

	return(status);
}


/*
 *	Creates a new page.
 */
static EDVPropDlgPage *edv_prop_dlg_page_new(void)
{
	return(EDV_PROP_DLG_PAGE(g_malloc0(sizeof(EDVPropDlgPage))));
}

/*
 *	Deletes the page.
 */
static void edv_prop_dlg_page_delete(EDVPropDlgPage *page)
{
	if(page == NULL)
		return;

	g_free(page->name);
	g_free(page);
}


/*
 *	Updates all the pages.
 */
void edv_prop_dlg_emit_update_pages(EDVPropDlg *d)
{
	GList *glist;
	const EDVObjectType type = (EDVObjectType)edv_properties_list_get_i(
		d->properties_list,
		EDV_PROP_NAME_TYPE
	);
	EDVPropDlgPage *page;

	if(d == NULL)
		return;

	d->freeze_count++;

	for(glist = d->pages_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		page = EDV_PROP_DLG_PAGE(glist->data);
		if(page == NULL)
			continue;

		if(page->update_cb != NULL)
			page->update_cb(
				d->page_context,
				type,
				d->location_type,
				d->properties_list,
				0,
				page->data
			);
	}

	d->freeze_count--;
}


/*
 *	Checks if a new page should be created by calling the source
 *	page's query_create_cb() callback, if it returns TRUE then
 *	a GtkVBox will be created and parented to the Property Dialog's
 *	GtkNotebook and the page's create_cb will be called to create
 *	the subsequent GtkWidgets. The new page will be appended to
 *	the Property Dialog's list of pages.
 *
 *	Returns the new page if one was created or NULL if not or an
 *	error occured.
 */
static EDVPropDlgPage *EDVPropDlgCreatePageIterate(
	EDVPropDlg *d,
	EDVPropDlgPage *src_page,
	const EDVObjectType type,
	const EDVLocationType location_type,
	GtkNotebook *notebook
)
{
	gboolean status;
	const gint border_major = 5;
	gint		version_major = 0,
			version_minor = 0,
			version_release = 0;
	gchar *page_name = NULL;
	edv_pixmap_data *pixmap_data_20x20 = NULL;
	GtkWidget	*w,
			*parent;
	EDVPropDlgPage *page;

	/* If this page does not have both the query_create_cb()
	 * and create_cb() callbacks set then completely ignore
	 * this page and do not create anything for it
	 */
	if((src_page->query_create_cb == NULL) ||
	   (src_page->create_cb == NULL)
	)
		return(NULL);

	/* Query the page's query_create_cb() to check if they want
	 * to create a page
	 */
	status = src_page->query_create_cb(
		d->page_context,
		&version_major,
		&version_minor,
		&version_release,
		&page_name,
		&pixmap_data_20x20,
		type,
		location_type,
		d->properties_list
	);

	g_free(page_name);

	/* Do not create a page? */
	if(!status)
		return(NULL);

	if((version_major != PROG_VERSION_MAJOR) ||
	   (version_minor != PROG_VERSION_MINOR) ||
	   (version_release != PROG_VERSION_RELEASE)
	)
		return(NULL);

	/* Allocate a new page */
	page = edv_prop_dlg_page_new();
	if(page == NULL)
		return(NULL);

	/* Append this page to the Properties Dialog's list of pages */
	d->pages_list = g_list_append(
		d->pages_list,
		page
	);

	/* Copy the values from the source page to the new page */
	page->prop_dlg = d;
	page->flags = src_page->flags;
	page->name = g_strdup(src_page->name);
/*	page->plugin_handle = NULL; */
	page->query_create_cb = src_page->query_create_cb;
	page->create_cb = src_page->create_cb;
	page->update_cb = src_page->update_cb;
	page->page_switched_cb = src_page->page_switched_cb;
	page->apply_vfs_cb = src_page->apply_vfs_cb;
	page->apply_recbin_cb = src_page->apply_recbin_cb;
	page->destroy_cb = src_page->destroy_cb;

	/* Create the GtkVBox on the main GtkNotebook that will
	 * server as the toplevel parent for this page
	 */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_object_set_data_full(
		GTK_OBJECT(w), EDV_PROP_DLG_PAGE_KEY,
		page, edv_prop_dlg_page_destroy_cb
	);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_notebook_append_page(
		notebook,
		w,
		gtk_label_new(page->name)
	);
	gtk_widget_show(w);
	parent = w;

	/* Call the page's create callback to create the new page
	 * and get its page handle as the user data for this new
	 * page
	 */
	page->data = page->create_cb(
		d->page_context,
		parent
	);
	if(page->data == NULL)
	{
		/* When create_cb() returns NULL it means it was unable
		 * to create the page data and we should remove the new
		 * page
		 *
		 * Remove the new page from the GtkNotebook which will
		 * destroy the toplevel GtkWidget for the new page and
		 * call edv_prop_dlg_page_destroy_cb() to delete
		 * the rest of the page and remove it from the Property
		 * Dialog's list of pages
		 */
		gtk_notebook_remove_page(
			notebook,
			g_list_length(d->pages_list) - 1
		);
		return(NULL);
	}

	return(page);
}

/*
 *	Creates all the pages specified in the source pages list.
 */
static void edv_prop_dlg_create_pages(EDVPropDlg *d)
{
	gint i;
	GtkNotebook *notebook = GTK_NOTEBOOK(d->notebook);
	const EDVObjectType type = (EDVObjectType)edv_properties_list_get_i(
		d->properties_list,
		EDV_PROP_NAME_TYPE
	);
	EDVPropDlgPage	*page,
			*src_page,
			src_pages_list[] = EDV_PROP_DLG_PAGES_LIST;

	d->freeze_count++;

	/* Iterate through the source pages list and check to see if
	 * if each page should/can be created and create each page
	 * accordingly
	 */
	for(i = 0; TRUE; i++)
	{
		src_page = &src_pages_list[i];

		/* End of source pages list? */
		if(src_page->name == NULL)
			break;

		/* Skip pages that should be appended, they will be
		 * created on the next iteration below
		 */
		if(src_page->flags & EDV_PROP_DLG_PAGE_END)
			continue;

		/* Append this page to the Properties Dialog's list of pages */
		page = EDVPropDlgCreatePageIterate(
			d,
			src_page,
			type,
			d->location_type,
			notebook
		);
	}

/* TODO create pages from plugins */

	/* Iterate through the source pages list and append pages that
	 * are set to be appended
	 */
	for(i = 0; TRUE; i++)
	{
		src_page = &src_pages_list[i];

		/* End of source pages list? */
		if(src_page->name == NULL)
			break;

		/* Skip pages that should not be appended */
		if(!(src_page->flags & EDV_PROP_DLG_PAGE_END))
			continue;

		/* Append this page to the Properties Dialog's list of pages */
		page = EDVPropDlgCreatePageIterate(
			d,
			src_page,
			type,
			d->location_type,
			notebook
		);
	}

	d->freeze_count--;
}

/*
 *	Deletes all the pages.
 */
static void edv_prop_dlg_clear_pages(EDVPropDlg *d)
{
	GtkNotebook *notebook = GTK_NOTEBOOK(d->notebook);

	/* Freeze to prevent page switch callbacks */
	d->freeze_count++;

	/* Check if the currently displayed page is the one that is
	 * being destroyed
	 */
	if(d->cur_page_num > -1)
	{
		EDVPropDlgPage *page = EDV_PROP_DLG_PAGE(g_list_nth_data(
			d->pages_list,
			(guint)d->cur_page_num
		));

		/* Mark that no page is being displayed because we are
		 * about to destroy all the pages
		 */
		d->cur_page_num = -1;

		/* Notify the page that we are switching away from it */
		if(page != NULL)
		{
			if(page->page_switched_cb != NULL)
				page->page_switched_cb(
					d->page_context,
					FALSE,	/* Away from this page */
					page->data
				);
		}
	}

	/* Destroy all the pages */
	while(d->pages_list != NULL)
	{
		/* Remove this page from the GtkNotebook which will
		 * destroy the toplevel GtkWidget for the page and call
		 * edv_prop_dlg_page_destroy_cb() to delete the rest
		 * of the page and remove it from the Property Dialog's
		 * pages list
		 */
		gtk_notebook_remove_page(
			notebook,
			0
		);
	}

	d->freeze_count--;
}


/*
 *	Creates a new Properties Dialog.
 */
EDVPropDlg *edv_prop_dlg_new(
	EDVCore *core,
	const EDVLocationType location_type,
	const gchar *path,
	const gulong index,
	const gchar *arch_path,
	GList *meta_data_list,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry
)
{
	const gint border_major = 5;
	gint error_code;
	GdkWindow *window;
	GtkStyle *style;
	GtkAccelGroup *accelgrp;
	GtkRcStyle	*standard_rcstyle,
			*lists_rcstyle;
	GtkWidget	*w,
			*parent, *parent2,
			*toplevel;
	CfgList *cfg_list;
	EDVPropDlg *d;
	EDVPropPageContext *page_context;

	if(core == NULL)
		return(NULL);

	cfg_list = core->cfg_list;
	standard_rcstyle = core->standard_rcstyle;
	lists_rcstyle = core->lists_rcstyle;

	d = EDV_PROP_DLG(g_malloc0(sizeof(EDVPropDlg)));
	if(d == NULL)
		return(NULL);

	d->toplevel = toplevel = gtk_window_new(GTK_WINDOW_DIALOG);
	d->accelgrp = accelgrp = gtk_accel_group_new();
#if 0
	d->busy_count = 0;
	d->freeze_count = 0;
	d->has_changes = FALSE;
#endif
	d->core = core;
	d->location_type = location_type;
#if 0
	d->properties_list = NULL;
	d->meta_data_list = NULL;
	d->mime_type = NULL;
	d->arch_path = NULL;
	d->pages_list = NULL;
#endif
	d->cur_page_num = -1;
#if 0
	d->icon_sel_dlg = NULL;
	d->mime_type_edit_dlg = NULL;
#endif

	d->freeze_count++;


	/* Create the EDVPropPageContext */
	d->page_context = page_context = edv_prop_page_context_new();
	if(page_context == NULL)
	{
		g_free(d);
		return(NULL);
	}
	page_context->prop_dlg = d;


	/* Copy the specified meta data to the Property Dialog's meta
	 * data list first, let subsequent operations that obtain more
	 * up to date data overwrite any value set on the Property
	 * Dialog's meta data list
	 */
	d->meta_data_list = edv_properties_list_update(
		d->meta_data_list,
		meta_data_list,
		TRUE,				/* Create as needed */
		FALSE				/* Do not remove nonexistant */
	);

	/* Get the object's properties which will fill this Property
	 * Dialog's properties list and meta datas list with the
	 * updated property values of the object
	 */
	d->properties_list = edv_prop_dlg_update_properties_list(
		core,
		d->properties_list,
		location_type,
		path,
		index,
		arch_path,
		&d->meta_data_list,
		&error_code
	);
	if(d->properties_list != NULL)
	{
		gulong index;
		gchar *type_string;
		const gchar *name;
		EDVObjectType type = EDV_OBJECT_TYPE_UNKNOWN;
		EDVPermissionFlags permissions;
		switch(location_type)
		{
		  case EDV_LOCATION_TYPE_VFS:
			/* Get the type */
			type = (EDVObjectType)edv_properties_list_get_i(
				d->properties_list,
				EDV_PROP_NAME_TYPE
			);
			/* Get the name */
			name = edv_properties_list_get_s(
				d->properties_list,
				EDV_PROP_NAME_NAME
			);
			/* Reget the path */
			path = edv_properties_list_get_s(
				d->properties_list,
				EDV_PROP_NAME_PATH
			);
			/* Get the permissions */
			permissions = (EDVPermissionFlags)edv_properties_list_get_i(
				d->properties_list,
				EDV_PROP_NAME_PERMISSIONS
			);
			/* Get the MIME Type */
			(void)edv_match_object_type_string(
				core->mime_types_list,
				type,
				path,
				permissions,
				&type_string
			);
			if(type_string != NULL)
			{
				g_free(d->mime_type);
				d->mime_type = g_strdup(type_string);
				g_free(type_string);
			}
			break;

		  case EDV_LOCATION_TYPE_RECYCLE_BIN:
			/* Get the type */
			type = (EDVObjectType)edv_properties_list_get_i(
				d->properties_list,
				EDV_PROP_NAME_TYPE
			);
			/* Get the index */
			index = edv_properties_list_get_ul(
				d->properties_list,
				EDV_PROP_NAME_INDEX
			);
			/* Get the name */
			name = edv_properties_list_get_s(
				d->properties_list,
				EDV_PROP_NAME_NAME
			);
			/* Get the permissions */
			permissions = (EDVPermissionFlags)edv_properties_list_get_i(
				d->properties_list,
				EDV_PROP_NAME_PERMISSIONS
			);
			/* Get the MIME Type */
			(void)edv_match_object_type_string(
				core->mime_types_list,
				type,
				name,
				permissions,
				&type_string
			);
			if(type_string != NULL)
			{
				g_free(d->mime_type);
				d->mime_type = g_strdup(type_string);
				g_free(type_string);
			}
			break;

		  case EDV_LOCATION_TYPE_ARCHIVE:
			/* Get the type */
			type = (EDVObjectType)edv_properties_list_get_i(
				d->properties_list,
				EDV_PROP_NAME_TYPE
			);
			/* Get the name */
			name = edv_properties_list_get_s(
				d->properties_list,
				EDV_PROP_NAME_NAME
			);
			/* Reget the path */
			path = edv_properties_list_get_s(
				d->properties_list,
				EDV_PROP_NAME_PATH
			);
			/* Get the permissions */
			permissions = (EDVPermissionFlags)edv_properties_list_get_i(
				d->properties_list,
				EDV_PROP_NAME_PERMISSIONS
			);
			/* Get the archive path */
			g_free(d->arch_path);
			d->arch_path = STRDUP(arch_path);
			/* Get the MIME Type */
			(void)edv_match_object_type_string(
				core->mime_types_list,
				type,
				name,		/* Use the name and not the
						 * full path for archive
						 * objects */
				permissions,
				&type_string
			);
			if(type_string != NULL)
			{
				g_free(d->mime_type);
				d->mime_type = g_strdup(type_string);
				g_free(type_string);
			}
			break;
		}
	}
	else
	{
		/* Unable to get the properties, set up the values to
		 * display the error
		 */
		d->properties_list = edv_properties_list_set_s(
			d->properties_list,
			EDV_PROP_NAME_PATH,
			path,
			TRUE
		);
		d->properties_list = edv_properties_list_set_s(
			d->properties_list,
			EDV_PROP_NAME_NAME,
			(path != NULL) ? g_basename(path) : NULL,
			TRUE
		);
		d->properties_list = edv_properties_list_set_ul(
			d->properties_list,
			EDV_PROP_NAME_INDEX,
			index,
			TRUE
		);

		/* Set the error property to display the error message */
		d->meta_data_list = edv_properties_list_set_s(
			d->meta_data_list,
			EDV_PROP_NAME_ERROR,
			g_strerror(error_code),
			TRUE
		);
	}


	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_window_set_wmclass(
		GTK_WINDOW(w),
		EDV_PROP_DLG_WM_CLASS_WINDOW_NAME,
		PROG_NAME
	);
	gtk_window_set_policy(GTK_WINDOW(w), FALSE, FALSE, FALSE);
	if((geometry_flags != 0) && (geometry != NULL))
	{
		if((geometry_flags & GDK_GEOMETRY_X) || (geometry_flags & GDK_GEOMETRY_Y))
			gtk_widget_set_uposition(w, geometry->x, geometry->y);
	}
	gtk_widget_set_usize(w, EDV_PROP_DLG_WIDTH, EDV_PROP_DLG_HEIGHT);
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
		gdk_window_set_decorations(
			window,
			GDK_DECOR_BORDER | GDK_DECOR_TITLE | GDK_DECOR_MENU |
			GDK_DECOR_MINIMIZE
		);
		gdk_window_set_functions(
			window,
			GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
		);
		GUISetWMIcon(
			window,
			(guint8 **)icon_properties_48x48_xpm
		);
	}
	gtk_widget_add_events( 
		w,
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "delete_event",
		GTK_SIGNAL_FUNC(edv_prop_dlg_delete_event_cb), d
	);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	style = gtk_widget_get_style(w);
	parent = w;


	/* Main GtkVBox */
	w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Main notebook GtkVBox */
	w = gtk_vbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Main GtkNotebook */
	d->notebook = w = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(w), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(w), TRUE);
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(w), TRUE);
	gtk_notebook_set_show_border(GTK_NOTEBOOK(w), TRUE);
/*	gtk_notebook_set_page(GTK_NOTEBOOK(w), 0); */
	gtk_signal_connect(
		GTK_OBJECT(w), "switch_page",
		GTK_SIGNAL_FUNC(edv_prop_dlg_switch_page_cb), d
	);
	gtk_widget_show(w);
	parent2 = w;


	/* GtkSeparator */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Buttons GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* OK GtkButton */
	d->ok_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_ok_20x20_xpm,
		"OK",
		NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_prop_dlg_ok_cb), d
	);
	gtk_accel_group_add(
		accelgrp, GDK_o, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_o);
	gtk_widget_show(w);

	/* Apply GtkButton */
	d->apply_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_select_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Aplique"
#elif defined(PROG_LANGUAGE_FRENCH)
"Appliquer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Verwenden"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Applicare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Toepas"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Aplique"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Gjeld"
#else
"Apply"
#endif
		,
		NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_prop_dlg_apply_cb), d
	);
	gtk_accel_group_add(
		accelgrp, GDK_a, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_a);
	gtk_widget_show(w);

	/* Cancel GtkButton */
	d->cancel_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_cancel_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cancele"
#elif defined(PROG_LANGUAGE_FRENCH)
"Annuler"
#elif defined(PROG_LANGUAGE_GERMAN)
"Heben"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Annullare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Annuleer"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Cancelamento"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kanseller"
#else
"Cancel"
#endif
		,
		NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_prop_dlg_cancel_cb), d
	);
	gtk_accel_group_add(
		accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
		accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);

	/* Close GtkButton */
	d->close_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_close_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cierre"
#elif defined(PROG_LANGUAGE_FRENCH)
"Quitter"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nah"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Vicino"
#elif defined(PROG_LANGUAGE_DUTCH)
"Einde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Prximo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Nr"
#else
"Close"
#endif
		,
		NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_prop_dlg_cancel_cb), d
	);
	gtk_accel_group_add(
		accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
		accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);


	/* Create the pages */
	edv_prop_dlg_create_pages(d);


	/* Set initial the RC styles */
	if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			d->toplevel,
			standard_rcstyle
		);

#if 0
	if(d->gen_toplevel != NULL)
	{
		w = menu_button_get_menu(d->gen_date_touch_btn);
		if((w != NULL) && (standard_rcstyle != NULL))
			gtk_widget_modify_style_recursive(
				w, standard_rcstyle
			);
	}
#endif

	/* Notify the pages to display their initial values */
	edv_prop_dlg_emit_update_pages(d);

	edv_prop_dlg_update_display(d);

	d->freeze_count--;

	return(d);
}

/*
 *	Gets the toplevel GtkWidget.
 */
GtkWidget *edv_prop_dlg_get_toplevel(EDVPropDlg *d)
{
	if(d == NULL)
		return(NULL);

	return(d->toplevel);
}

/*
 *	Gets the GtkAccelGroup.
 */
GtkAccelGroup *edv_prop_dlg_get_accelgrp(EDVPropDlg *d)
{
	if(d == NULL)
		return(NULL);

	return(d->accelgrp);
}

/*
 *	Gets the freeze count.
 */
gint edv_prop_dlg_get_freeze_count(EDVPropDlg *d)
{
	if(d == NULL)
		return(0);

	return(d->freeze_count);
}

/*
 *	Freezes the EDVPropDlg.
 */
void edv_prop_dlg_freeze(EDVPropDlg *d)
{
	if(d == NULL)
		return;

	d->freeze_count++;
}

/*
 *	Thaws the EDVPropDlg.
 */
void edv_prop_dlg_thaw(EDVPropDlg *d)
{
	if(d == NULL)
		return;

	d->freeze_count--;
}

/*
 *	Gets the EDVCore.
 */
EDVCore *edv_prop_dlg_get_core(EDVPropDlg *d)
{
	if(d == NULL)
		return(NULL);

	return(d->core);
}

/*
 *	Gets the CfgList from the EDVCore.
 */
CfgList *edv_prop_dlg_get_cfg_list(EDVPropDlg *d)
{
	EDVCore *core = edv_prop_dlg_get_core(d);
	if(core == NULL)
		return(NULL);

	return(core->cfg_list);
}

/*
 *	Gets the EDVLocationType.
 */
EDVLocationType edv_prop_dlg_get_location_type(EDVPropDlg *d)
{
	if(d == NULL)
		return(EDV_LOCATION_TYPE_VFS);

	return(d->location_type);
}

/*
 *	Gets the properties list describing the EDV*Object.
 *
 *	Returns the pointer to the GList of EDVProperty * properties
 *	or NULL on error. The returned pointer must not be modified
 *	or deleted.
 */
GList *edv_prop_dlg_get_properties_list(EDVPropDlg *d)
{
	if(d == NULL)
		return(NULL);

	return(d->properties_list);
}

/*
 *	Gets the meta data list describing the EDV*Object.
 *
 *	Returns the pointer to the GList of EDVProperty * meta datas
 *	or NULL on error. The returned pointer must not be modified
 *	or deleted.
 */
GList *edv_prop_dlg_get_meta_data_list(EDVPropDlg *d)
{
	if(d == NULL)
		return(NULL);

	return(d->meta_data_list);
}

/*
 *	Add/updates the meta data value.
 *
 *	The name specifies the name of the meta data to add/update.
 *
 *	The value specifies the value to set. If value is NULL or an
 *	empty string then the meta data will be removed.
 */
void edv_prop_dlg_add_meta_data(
	EDVPropDlg *d,
	const gchar *name,
	const gchar *value
)
{
	if(d == NULL)
		return;

	d->meta_data_list = edv_properties_list_set_s(
		d->meta_data_list,
		name,
		value,
		TRUE				/* Create as needed */
	);
}

/*
 *	Removes the meta data.
 *
 *	The name specifies the name of the meta data to remove.
 */
void edv_prop_dlg_remove_meta_data(
	EDVPropDlg *d,
	const gchar *name
)
{
	if(d == NULL)
		return;

	d->meta_data_list = edv_properties_list_remove(
		d->meta_data_list,
		name
	);
}

/*
 *	Gets the EDVMIMEType type.
 *
 *	Returns the pointer to the EDVMIMEType's type string or NULL
 *	on error.
 */
const gchar *edv_prop_dlg_get_mime_type_type(EDVPropDlg *d)
{
	if(d == NULL)
		return(NULL);

	return(d->mime_type);
}

/*
 *	Gets the archive path.
 *
 *	Returns the pointer to the archive path string or NULL if the
 *	EDVPropDlg's EDVLocationType is not EDV_LOCATION_TYPE_ARCHIVE.
 */
const gchar *edv_prop_dlg_get_archive_path(EDVPropDlg *d)
{
	if(d == NULL)
		return(NULL);

	return(d->arch_path);
}

/*
 *	Switch page.
 */
gboolean edv_prop_dlg_set_page(
	EDVPropDlg *d,
	const gchar *page_name
)
{
	GtkNotebook *notebook;

	if(d == NULL)
		return(FALSE);

	notebook = GTK_NOTEBOOK(d->notebook);
	if(page_name != NULL)
	{
		gint page_num;
		GList *glist;
		EDVPropDlgPage *page;

		for(page_num = 0,
		    glist = d->pages_list;
		    glist != NULL;
		    page_num++,
		    glist = g_list_next(glist)
		)
		{
			page = EDV_PROP_DLG_PAGE(glist->data);
			if(page == NULL)
				continue;

			if(page->name == NULL)
				continue;

			if(!g_strcasecmp(page->name, page_name))
				break;
		}
		if(glist != NULL)
		{
			gtk_notebook_set_page(
				notebook,
				page_num
			);
			return(TRUE);
		}
		else
		{
			return(FALSE);
		}
	}
	else
	{
		gtk_notebook_set_page(
			notebook,
			0
		);
		return(TRUE);
	}
}

/*
 *	Checks if the has changes flag is set.
 */
gboolean edv_prop_dlg_get_has_changes(EDVPropDlg *d)
{
	if(d == NULL)
		return(FALSE);

	return(d->has_changes);
}

/*
 *	Sets the has changes flag.
 */
void edv_prop_dlg_set_has_changes(
	EDVPropDlg *d,
	const gboolean has_changes
)
{
	if(d == NULL)
		return;

	if(d->has_changes == has_changes)
		return;

	d->has_changes = has_changes;
	edv_prop_dlg_update_display(d);
}

/*
 *	Updates the Properties Dialog's widgets to reflect current
 *	values.
 */
void edv_prop_dlg_update_display(EDVPropDlg *d)
{
	gboolean	sensitive,
			write_protect,
			has_changes,
			read_only;
	gchar *s;
	const gchar	*name,
			*path;
	GList *properties_list;
	CfgList *cfg_list;
	EDVCore *core;

	if(d == NULL)
		return;

	d->freeze_count++;

	properties_list = d->properties_list;
	core = d->core;
	cfg_list = core->cfg_list;
	write_protect = EDV_GET_B(EDV_CFG_PARM_WRITE_PROTECT);
	has_changes = d->has_changes;

	/* Determine if the object is read only */
	read_only = TRUE;
	switch(d->location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
		read_only = write_protect;
		break;
	  case EDV_LOCATION_TYPE_RECYCLE_BIN:
		read_only = write_protect;
		break;
	  case EDV_LOCATION_TYPE_ARCHIVE:
		read_only = TRUE;
		break;
	}

	/* Get the name and path */
	name = edv_properties_list_get_s(
		properties_list,
		EDV_PROP_NAME_NAME
	);
	path = edv_properties_list_get_s(
		properties_list,
		EDV_PROP_NAME_PATH
	);
	if(path == NULL)
		path = name;

#define MAP(_w_)	{		\
 if((_w_) != NULL)			\
  gtk_widget_show(_w_);			\
}
#define UNMAP(_w_)	{		\
 if((_w_) != NULL)			\
  gtk_widget_hide(_w_);			\
}

	/* Title */
	if(name != NULL)
		s = g_strconcat(
			EDV_PROP_DLG_TITLE,
			": ",
			name,
			(has_changes) ? " (*)" : "",
			NULL
		);
	else if(path != NULL)
		s = g_strconcat(
			EDV_PROP_DLG_TITLE,
			": ",
			path,
			(has_changes) ? " (*)" : "",
			NULL
		);
	else
		s = g_strdup(EDV_PROP_DLG_TITLE);
	gtk_window_set_title(GTK_WINDOW(d->toplevel), s);
	g_free(s);

	/* OK, Apply, Cancel, Close Buttons */
	sensitive = (read_only) ? FALSE : has_changes;
	GTK_WIDGET_SET_SENSITIVE(d->ok_btn, sensitive);
	GTK_WIDGET_SET_SENSITIVE(d->apply_btn, sensitive);
	sensitive = TRUE;
	GTK_WIDGET_SET_SENSITIVE(d->cancel_btn, sensitive);
	GTK_WIDGET_SET_SENSITIVE(d->close_btn, sensitive);
	if(has_changes)
	{
		MAP(d->ok_btn);
		MAP(d->apply_btn);
		MAP(d->cancel_btn);
		UNMAP(d->close_btn);
	}
	else
	{
		MAP(d->ok_btn);
		MAP(d->apply_btn);
		UNMAP(d->cancel_btn);
		MAP(d->close_btn);
	}

#undef UNMAP
#undef MAP

	d->freeze_count--;
}

/*
 *	Sets the Properties Dialog as busy or ready.
 */
void edv_prop_dlg_set_busy(EDVPropDlg *d, const gboolean busy)
{
	GdkCursor *cursor;
	GtkWidget *w;
	EDVCore *core;

	if(d == NULL)
		return;

	core = d->core;

	w = d->toplevel;
	if(w != NULL)
	{
		if(busy)
		{
			/* Increase the busy count */
			d->busy_count++;

			/* If already busy then don't change anything */
			if(d->busy_count > 1)
				return;

			cursor = edv_get_cursor(core, EDV_CURSOR_CODE_BUSY);
		}
		else
		{
			/* Reduce the busy count */
			d->busy_count--;
			if(d->busy_count < 0)
				d->busy_count = 0;

			/* If still busy then do not change anything */
			if(d->busy_count > 0)
				return;

			cursor = NULL;		/* Use default cursor */
		}

		/* Update toplevel window's cursor */
		if(w->window != NULL)
		{
			gdk_window_set_cursor(w->window, cursor);
			gdk_flush();
		}
	}
}

/*
 *	Checks if the Properties Dialog is mapped.
 */
gboolean edv_prop_dlg_is_mapped(EDVPropDlg *d)
{
	if(d == NULL)
		return(FALSE);

	return(GTK_WIDGET_MAPPED(d->toplevel));
}

/*
 *	Maps the Properties Dialog.
 */
void edv_prop_dlg_map(EDVPropDlg *d)
{
	GtkWidget *w;

	if(d == NULL)
		return;

	gtk_widget_show_raise(d->toplevel);

	w = d->close_btn;
	gtk_widget_grab_focus(w);
	gtk_widget_grab_default(w);
}

/*
 *	Unmaps the Properties Dialog.
 */
void edv_prop_dlg_unmap(EDVPropDlg *d)
{
	if(d == NULL)
		return;

	gtk_widget_hide(d->toplevel);
}

/*
 *	Deletes the Properties Dialog.
 */
void edv_prop_dlg_delete(EDVPropDlg *d)
{
	if(d == NULL)
		return;

	d->freeze_count++;

	/* Delete all the pages */
	edv_prop_dlg_clear_pages(d);

	/* Unref the Icon Selector Dialog */
	d->icon_sel_dlg = IconSelDlgUnref(d->icon_sel_dlg);

	edv_prop_dlg_unmap(d);

	/* Delete the EDVPropPageContext */
	edv_prop_page_context_delete(d->page_context);
	d->page_context = NULL;

	/* Destroy all the GtkWidgets */
	gtk_widget_destroy(d->toplevel);

	gtk_accel_group_unref(d->accelgrp);

	d->properties_list = edv_properties_list_delete(d->properties_list);
	d->meta_data_list = edv_properties_list_delete(d->meta_data_list);

	g_free(d->mime_type);
	g_free(d->arch_path);

	d->freeze_count--;

	g_free(d);
}
