#include <stdlib.h>
#include <string.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "guiutils.h"
#include "progressdialog.h"


typedef struct _ProgressDialogIcon	ProgressDialogIcon;
#define PROGRESS_DIALOG_ICON(p)		((ProgressDialogIcon *)(p))

typedef struct _ProgressDialog		ProgressDialog;
#define PROGRESS_DIALOG(p)		((ProgressDialog *)(p))
#define PROGRESS_DIALOG_KEY		"/ProgressDialog"


/*
 *	Flags:
 */
typedef enum {
	PROGRESS_DIALOG_MAPPED		= (1 << 0),
	PROGRESS_DIALOG_REALIZED	= (1 << 1)
} ProgressDialogFlags;


/* Callbacks */
static void ProgressDialogDestroyCB(gpointer data);
static gint ProgressDialogDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static gint ProgressDialogAnimationConfigureEventCB(
	GtkWidget *widget, GdkEventConfigure *configure, gpointer data
);
static gint ProgressDialogAnimationExposeEventCB(
	GtkWidget *widget, GdkEventExpose *expose, gpointer data
);
static void ProgressDialogRealizeCB(GtkWidget *widget, gpointer data);
static void ProgressDialogStopCB(GtkWidget *widget, gpointer data);

/* Utilities */
static void ProgressDialogSetIcon(
	ProgressDialog *pd,
	const guint8 **icon_data
);
static ProgressDialogIcon *ProgressDialogIconNewFromData(
	ProgressDialog *pd,
	const guint8 **icon_data
);
static ProgressDialogIcon *ProgressDialogIconNewFromFile(
	ProgressDialog *pd,
	const gchar *path
);
static void ProgressDialogIconDelete(ProgressDialogIcon *icon);
static void ProgressDialogClearAnimationIcons(ProgressDialog *pd);

/* Draw */
static void ProgressDialogDrawAnimation(ProgressDialog *pd);

/* Front Ends */
gint ProgressDialogInit(void);
void ProgressDialogSetStyle(GtkRcStyle *rc_style);
void ProgressDialogSetTransientFor(GtkWidget *w);
void ProgressDialogSetWMIconData(const guint8 **icon_data);
gboolean ProgressDialogIsQuery(void);
void ProgressDialogBreakQuery(const gboolean allow_gtk_iteration);
gint ProgressDialogStopCount(void);
GtkWidget *ProgressDialogGetToplevel(void);
void ProgressDialogMap(
	const gchar *title,
	const gchar *label,
	const guint8 **icon_data,
	const gchar *stop_btn_label
);
void ProgressDialogMapFile(
	const gchar *title,
	const gchar *label,
	const gchar *icon_path,
	const gchar *stop_btn_label
);
void ProgressDialogMapAnimation(
	const gchar *title,
	const gchar *label,
	const gchar *stop_btn_label,
	GList *start_icon_datas_list,
	GList *icon_datas_list,
	GList *end_icon_datas_list,
	const gulong animation_interval_ms,
	const guint16 animation_increment
);
void ProgressDialogMapAnimationFile(
	const gchar *title,
	const gchar *label,
	const gchar *stop_btn_label,
	GList *start_icon_paths_list,
	GList *icon_paths_list,
	GList *end_icon_paths_list,
	const gulong animation_interval_ms,
	const guint16 animation_increment
);
void ProgressDialogUpdate(
	const gchar *title,
	const gchar *label,
	const guint8 **icon_data,
	const gchar *stop_btn_label,
	const gfloat position,			/* 0.0 to 1.0 */
	const guint nblocks,			/* 0 for continuous */
	const gboolean allow_gtk_iteration
);
void ProgressDialogUpdateUnknown(
	const gchar *title,
	const gchar *label,
	const guint8 **icon_data,
	const gchar *stop_btn_label,
	const gboolean allow_gtk_iteration
);
void ProgressDialogUnmap(void);
void ProgressDialogShutdown(void);


#define PROGRESS_DIALOG_WIDTH		380
#define PROGRESS_DIALOG_HEIGHT		-1

#define PROGRESS_DIALOG_ANIMATION_AREA_WIDTH	270
#define PROGRESS_DIALOG_ANIMATION_AREA_HEIGHT	45


/* #define PROGRESS_DIALOG_PROGRESS_BAR_HEIGHT	(20 + (2 * 2)) */
#define PROGRESS_DIALOG_PROGRESS_BAR_HEIGHT	20


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Icon:
 *
 *	Used in animation.
 */
struct _ProgressDialogIcon {

	GdkPixmap	*pixmap;
	GdkBitmap	*mask;
	gint		width, height;

};

/*
 *	Progress Dialog:
 */
struct _ProgressDialog {

	GtkWidget	*toplevel;
	GtkAccelGroup	*accelgrp;
	gint		freeze_count,
			stop_count;
	ProgressDialogFlags	flags;

	GtkWidget	*main_vbox,
			*animation_box,		/* Animation GtkBox */
			*animation_da,		/* Animation GtkDrawingArea */
			*icon_box,		/* Icon GtkBox */
			*icon_pm,		/* Icon GtkPixmap */
			*label,			/* GtkLabel */
			*progress_bar,
			*stop_btn,
			*stop_btn_label;

	/* Animation */
	GdkGC		*animation_gc;
	GdkPixmap	*animation_buffer;	/* Animation GtkDrawingArea
						 * back buffer */
	GTimer		*animation_timer;
	guint16		animation_position,	/* In animation units from
						 * 0 to (guint16)-1 */
			animation_increment;	/* In animation units from
						 * 0 to (guint16)-1 */
	gulong		animation_interval_ms;
	GList		*start_icons_list,	/* GList of ProgressDialogIcon
						 * * icons */
			*icons_list,
			*end_icons_list;

	GtkWidget	*transient_for;		/* Reference */
};

static ProgressDialog	*progress_dialog = NULL;


/*
 *	Toplevel GtkWindow "destroy" signal callback.
 */
static void ProgressDialogDestroyCB(gpointer data)
{
	ProgressDialog *pd = PROGRESS_DIALOG(data);
	if(pd == NULL)
	    return;

	if(pd == progress_dialog)
	    progress_dialog = NULL;

	pd->freeze_count++;

	/* Delete all of the animation icons */
	ProgressDialogClearAnimationIcons(pd);

/*	pd->toplevel = NULL; */
	gtk_accel_group_unref(pd->accelgrp);

	GDK_PIXMAP_UNREF(pd->animation_buffer);
	GDK_GC_UNREF(pd->animation_gc);

	G_TIMER_DESTROY(pd->animation_timer);

	pd->freeze_count--;

	g_free(pd);
}

/*
 *	Toplevel GtkWindow "delete_event" signal callback.
 */
static gint ProgressDialogDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	ProgressDialogStopCB(widget, data);
	return(TRUE);
}

/*
 *	Animation GtkDrawingArea "configure_event" signal callback.
 */
static gint ProgressDialogAnimationConfigureEventCB(
	GtkWidget *widget, GdkEventConfigure *configure, gpointer data
)
{
	gint width, height;
	GdkWindow *window;
	ProgressDialog *pd = PROGRESS_DIALOG(data);
	if((widget == NULL) || (configure == NULL) || (pd == NULL))
	    return(FALSE);

	window = configure->window;
	width = configure->width;
	height = configure->height;

	/* Recreate the Animated GtkDrawingArea's back buffer
	 * GdkPixmap
	 */
	GDK_PIXMAP_UNREF(pd->animation_buffer);
	if((width > 0) && (height > 0))
	    pd->animation_buffer = gdk_pixmap_new(
		window,
		width, height,
		-1
	    );
	else
	    pd->animation_buffer = NULL;

	return(TRUE);
}

/*
 *	Animation GtkDrawingArea "expose_event" signal callback.
 */
static gint ProgressDialogAnimationExposeEventCB(
	GtkWidget *widget, GdkEventExpose *expose, gpointer data
)
{
	ProgressDialog *pd = PROGRESS_DIALOG(data);
	if((widget == NULL) || (pd == NULL))
	    return(FALSE);

	ProgressDialogDrawAnimation(pd);
	return(TRUE);
}

/*
 *	Toplevel GtkWindow "realize" signal callback.
 */
static void ProgressDialogRealizeCB(GtkWidget *widget, gpointer data)
{
	GdkWindow *window;
	ProgressDialog *pd = PROGRESS_DIALOG(data);
	if((widget == NULL) || (pd == NULL))
	    return;

	window = widget->window;
	if(window != NULL)
	{
	    GdkGeometry geo;
	    geo.min_width = 100;
	    geo.min_height = 70;
	    geo.max_width = gdk_screen_width() - 10;
	    geo.max_height = gdk_screen_height() - 10;
	    geo.base_width = 0;
	    geo.base_height = 0;
	    geo.width_inc = 1;
	    geo.height_inc = 1;
	    geo.min_aspect = 1.3f;
	    geo.max_aspect = 1.3f;
	    gdk_window_set_geometry_hints(
		window,
		&geo,
		GDK_HINT_MIN_SIZE | GDK_HINT_MAX_SIZE |
		GDK_HINT_BASE_SIZE | GDK_HINT_RESIZE_INC
	    );
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE | GDK_DECOR_MENU |
		GDK_DECOR_MINIMIZE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
	    );
	}

	/* Create the GC */
	if(pd->animation_gc == NULL)
	    pd->animation_gc = gdk_gc_new(window);

	/* Mark the Progress Dialog as realized */
	pd->flags |= PROGRESS_DIALOG_REALIZED;
}

/*
 *	Stop callback.
 */
static void ProgressDialogStopCB(GtkWidget *widget, gpointer data)
{
	ProgressDialog *pd = PROGRESS_DIALOG(data);
	if(pd == NULL)
	    return;

	if(pd->stop_count < 0)
	    pd->stop_count = 0;

	pd->stop_count++;
}


/*
 *	Sets the icon from XPM data.
 */
static void ProgressDialogSetIcon(
	ProgressDialog *pd,
	const guint8 **icon_data
)
{
	GdkBitmap *mask;
	GdkPixmap *pixmap;
	GtkWidget	*w,
			*parent = pd->icon_box;

	if(icon_data == NULL)
	    return;

	w = pd->toplevel;
	if(!GTK_WIDGET_REALIZED(w))
	    gtk_widget_realize(w);

	/* Load the icon */
	pixmap = GUIPixmapNewFromXPMData(
	    w->window,
	    &mask,
	    NULL,
	    (guint8 **)icon_data
	);
	if(pixmap == NULL)
	    return;

	/* Set the icon */
	w = pd->icon_pm;
	if(w != NULL)
	{
	    gtk_pixmap_set(GTK_PIXMAP(w), pixmap, mask);
	}
	else
	{
	    /* Icon GtkPixmap has not been created yet, so create it
	     * here and set the icon
	     */
	    pd->icon_pm = w = gtk_pixmap_new(pixmap, mask);
	    gtk_box_pack_start(GTK_BOX(parent), w, TRUE, FALSE, 0);
	    gtk_widget_show(w);
	}

	(void)GDK_PIXMAP_UNREF(pixmap);
	(void)GDK_BITMAP_UNREF(mask);

#if 0
	/* Set WM icon for toplevel */
/*
 Do not set toplevel icon, let it be parent relative to the transient
 window
 */
	GUISetWMIcon(toplevel->window, (guint8 **)icon_data);
#endif
}

/*
 *	Sets the icon from a XPM file.
 */
static void ProgressDialogSetIconFile(
	ProgressDialog *pd,
	const gchar *path
)
{
	GdkBitmap *mask;
	GdkPixmap *pixmap;
	GtkWidget	*w,
			*parent = pd->icon_box;

	if(path == NULL)
	    return;

	w = pd->toplevel;
	if(!GTK_WIDGET_REALIZED(w))
	    gtk_widget_realize(w);

	/* Load the icon */
	pixmap = GUIPixmapNewFromXPMFile(
	    w->window,
	    &mask,
	    NULL,
	    path
	);
	if(pixmap == NULL)
	    return;

	/* Set the icon */
	w = pd->icon_pm;
	if(w != NULL)
	{
	    gtk_pixmap_set(GTK_PIXMAP(w), pixmap, mask);
	}
	else
	{
	    /* Icon GtkPixmap has not been created yet, so create it
	     * here and set the icon
	     */
	    pd->icon_pm = w = gtk_pixmap_new(pixmap, mask);
	    gtk_box_pack_start(GTK_BOX(parent), w, TRUE, FALSE, 0);
	    gtk_widget_show(w);
	}

	(void)GDK_PIXMAP_UNREF(pixmap);
	(void)GDK_BITMAP_UNREF(mask);
}

/*
 *	Creates a new icon from XPM data.
 */
static ProgressDialogIcon *ProgressDialogIconNewFromData(
	ProgressDialog *pd,
	const guint8 **icon_data
)
{
	gint width, height;
	GdkBitmap *mask;
	GdkPixmap *pixmap;
	GtkWidget *w;
	ProgressDialogIcon *icon;

	if((pd == NULL) || (icon_data == NULL))
	    return(NULL);

	w = pd->toplevel;
	if(!GTK_WIDGET_REALIZED(w))
	    gtk_widget_realize(w);

	pixmap = GUIPixmapNewFromXPMData(
	    w->window,
	    &mask,
	    NULL,
	    (guint8 **)icon_data
	);
	if(pixmap == NULL)
	    return(NULL);

	/* Create a new animated icon */
	icon = PROGRESS_DIALOG_ICON(g_malloc0(
	    sizeof(ProgressDialogIcon)
	));
	if(icon == NULL)
	{
	    (void)GDK_PIXMAP_UNREF(pixmap);
	    (void)GDK_BITMAP_UNREF(mask);
	    return(NULL);
	}

	icon->pixmap = pixmap;
	icon->mask = mask;

	/* Get the size */
	gdk_window_get_size(pixmap, &width, &height);
	icon->width = width;
	icon->height = height;

	return(icon);
}

/*
 *	Creates a new icon from a XPM file.
 */
static ProgressDialogIcon *ProgressDialogIconNewFromFile(
	ProgressDialog *pd,
	const gchar *path
)
{
	gint width, height;
	GdkBitmap *mask;
	GdkPixmap *pixmap;
	GtkWidget *w;
	ProgressDialogIcon *icon;

	if((pd == NULL) || (path == NULL))
	    return(NULL);

	w = pd->toplevel;
	if(!GTK_WIDGET_REALIZED(w))
	    gtk_widget_realize(w);

	pixmap = GUIPixmapNewFromXPMFile(
	    w->window,
	    &mask,
	    NULL,
	    path
	);
	if(pixmap == NULL)
	    return(NULL);

	/* Create a new animated icon */
	icon = PROGRESS_DIALOG_ICON(g_malloc0(
	    sizeof(ProgressDialogIcon)
	));
	if(icon == NULL)
	{
	    (void)GDK_PIXMAP_UNREF(pixmap);
	    (void)GDK_BITMAP_UNREF(mask);
	    return(NULL);
	}

	icon->pixmap = pixmap;
	icon->mask = mask;

	/* Get the size */
	gdk_window_get_size(pixmap, &width, &height);
	icon->width = width;
	icon->height = height;

	return(icon);
}

/*
 *	Deletes the icon.
 */
static void ProgressDialogIconDelete(ProgressDialogIcon *icon)
{
	if(icon == NULL)
	    return;

	(void)GDK_PIXMAP_UNREF(icon->pixmap);
	(void)GDK_BITMAP_UNREF(icon->mask);
	g_free(icon);
}

/*
 *	Deletes all of the animation icons.
 */
static void ProgressDialogClearAnimationIcons(ProgressDialog *pd)
{
	if(pd == NULL)
	    return;

	if(pd->start_icons_list != NULL)
	{
	    g_list_foreach(
		pd->start_icons_list,
		(GFunc)ProgressDialogIconDelete,
		NULL
	    );
	    g_list_free(pd->start_icons_list);
	    pd->start_icons_list = NULL;
	}

	if(pd->end_icons_list != NULL)
	{
	    g_list_foreach(
		pd->end_icons_list,
		(GFunc)ProgressDialogIconDelete,
		NULL
	    );
	    g_list_free(pd->end_icons_list);
	    pd->end_icons_list = NULL;
	}

	if(pd->icons_list != NULL)
	{
	    g_list_foreach(
		pd->icons_list,
		(GFunc)ProgressDialogIconDelete,
		NULL
	    );
	    g_list_free(pd->icons_list);
	    pd->icons_list = NULL;
	}
}


/*
 *	Draws the Animation GtkDrawingArea.
 */
static void ProgressDialogDrawAnimation(ProgressDialog *pd)
{
	gint	width, height,
		start_icon_width = 0,
		end_icon_width = 0,
		nicons,
		frame,
		nframes,
		anim_per_frame;
	GdkGC *gc;
	GdkWindow *window;
	GdkPixmap *pixmap;
	GdkDrawable *drawable;
	GtkStateType state;
	GtkStyle *style;
	GtkWidget *w = pd->animation_da;

	if(!GTK_WIDGET_VISIBLE(w))
	    return;

	window = w->window;
	gc = pd->animation_gc;
	state = GTK_WIDGET_STATE(w);
	style = gtk_widget_get_style(w);
	if((window == NULL) || (gc == NULL) || (style == NULL))
	    return;

	pixmap = pd->animation_buffer;
	drawable = (GdkDrawable *)((pixmap != NULL) ? pixmap : window);

	gdk_window_get_size(
	    (GdkWindow *)drawable,
	    &width, &height
	);
	if((width <= 0) || (height <= 0))
	    return;


	/* Draw the background */
	gtk_style_apply_default_background(
	    style,
	    drawable,
	    FALSE,
	    state,
	    NULL,
	    0, 0,
	    width, height
	);


	/* Begin drawing the icons */

	/* Calculate total frames, this is the number of icons in the
	 * inside animatino frame plus two end point icons (the start and
	 * end icons)
	 */
	nicons = g_list_length(pd->icons_list);
	nframes = MAX((nicons + 2), 2);

	/* Calculate animation frame index */
	frame = (gint)(
	    (gfloat)pd->animation_position / (gfloat)((guint16)-1) *
	    nframes
	);

	/* Calculate number of animation units in one frame */
	if(nframes > 0)
	    anim_per_frame = (gint)((guint16)-1) / nframes;
	else
	    anim_per_frame = 0;

	/* Draw the start icons */
	if(pd->start_icons_list != NULL)
	{
	    ProgressDialogIcon *icon;

	    /* Get the icon based on the frame index */
	    if(frame == 0)
		icon = PROGRESS_DIALOG_ICON(g_list_nth_data(
		    pd->start_icons_list, 0
		));
	    else if(frame == (nframes - 1))
		icon = PROGRESS_DIALOG_ICON(g_list_nth_data(
		    pd->start_icons_list, 2
		));
	    else
		icon = PROGRESS_DIALOG_ICON(g_list_nth_data(
		    pd->start_icons_list, 1
		));

	    /* Got the icon? */
	    if(icon != NULL)
	    {
		gint    icon_width = icon->width,
			icon_height = icon->height,
			x = 0,
			y = height - icon_height - 1;
		GdkBitmap *icon_mask = icon->mask;
		GdkPixmap *icon_pixmap = icon->pixmap;

		/* Record start icon width */
		start_icon_width = icon_width;

		/* Icon pixmap available for drawing? */
		if(icon_pixmap != NULL)
		{
		    /* Set icon mask? */
		    if(icon_mask != NULL)
		    {
			gdk_gc_set_clip_mask(gc, icon_mask);
			gdk_gc_set_clip_origin(gc, x, y);
		    }
		    else
		    {
			gdk_gc_set_clip_mask(gc, NULL);
		    }
		    /* Draw icon */
		    gdk_draw_pixmap(
			drawable,
			gc,
			(GdkDrawable *)icon_pixmap,
			0, 0,
			x, y,
			icon_width, icon_height
		    );
		}
	    }
	}

	/* Draw the end icons */
	if(pd->end_icons_list != NULL)
	{
	    ProgressDialogIcon *icon;

	    /* Get the icon based on the frame index */
	    if(frame == 0)
		icon = PROGRESS_DIALOG_ICON(g_list_nth_data(
		    pd->end_icons_list, 0
		));
	    else if(frame == (nframes - 1))
		icon = PROGRESS_DIALOG_ICON(g_list_nth_data(
		    pd->end_icons_list, 2
		));
	    else
		icon = PROGRESS_DIALOG_ICON(g_list_nth_data(
		    pd->end_icons_list, 1
		));

	    /* Got the icon? */
	    if(icon != NULL)
	    {
		gint	icon_width = icon->width,
			icon_height = icon->height,
			x = width - icon->width - 1,
			y = height - icon->height - 1;
		GdkBitmap *icon_mask = icon->mask;
		GdkPixmap *icon_pixmap = icon->pixmap;

		/* Record end icon width */
		end_icon_width = icon->width;

		/* Pixmap available for drawing? */
		if(icon_pixmap != NULL)
		{
		    /* Mask available? */
		    if(icon_mask != NULL)
		    {
			gdk_gc_set_clip_mask(gc, icon_mask);
			gdk_gc_set_clip_origin(gc, x, y);
		    }
		    else
		    {
			gdk_gc_set_clip_mask(gc, NULL);
		    }

		    gdk_draw_pixmap(
			drawable,
			gc,
			(GdkDrawable *)icon_pixmap,
			0, 0,
			x, y,
			icon_width, icon_height
		    );
		}
	    }
	}


	/* Draw the intermediate icons */
	if((pd->icons_list != NULL) &&
	   (frame > 0) && (frame < (nframes - 1))
	)
	{
	    gint i, iframe_max;
	    gfloat iframe_coeff;
	    ProgressDialogIcon *icon = NULL;

	    /* Calculate the inside frame coeff */
	    iframe_max = (gint)((guint16)-1) - (2 * anim_per_frame);
	    if(iframe_max > 0)
		iframe_coeff = CLIP(
		    ((gfloat)pd->animation_position - (gfloat)anim_per_frame) /
		    (gfloat)iframe_max, 0.0f, 1.0f
		);
	    else
		iframe_coeff = 0.0f;

	    /* Calculate the icon index based on the inside frame coeff */
	    i = (gint)(iframe_coeff * nicons);

	    /* Get the icon based on the frame index */
	    icon = PROGRESS_DIALOG_ICON(g_list_nth_data(
		pd->icons_list, i
	    ));
	    if(icon != NULL)
	    {
		gint	y_border = 0,
			icon_width = icon->width,
			icon_height = icon->height,
			x, y, range;
		GdkBitmap *icon_mask = icon->mask;
		GdkPixmap *icon_pixmap = icon->pixmap;

		/* Calculate x position based on iframe_coeff and range */
		range = MAX(
		    width - start_icon_width - end_icon_width - icon_width,
		    0
		);
		x = (gint)(iframe_coeff * range) + start_icon_width;

		/* Calculate y position based on iframe_coeff and range */
		range = MAX(
		    (height / 2) - (icon_height / 2) - y_border,
		    0
		);
		if(iframe_coeff > 0.5f)
		{
		    const gfloat tc = (gfloat)((iframe_coeff - 0.5) / 0.5);
		    y = (gint)((tc * tc) * range);
		}
		else
		{
		    const gfloat tc = (gfloat)(1.0 - (iframe_coeff / 0.5));
		    y = (gint)((tc * tc) * range);
		}
		y += y_border;	/* Add 5 pixels margin */


		/* Pixmap available for drawing? */
		if(icon_pixmap != NULL)
		{
		    /* Mask available? */
		    if(icon_mask != NULL)
		    {
			gdk_gc_set_clip_mask(gc, icon_mask);
			gdk_gc_set_clip_origin(gc, x, y);
		    }
		    else
		    {
			gdk_gc_set_clip_mask(gc, NULL);
		    }

		    gdk_draw_pixmap(
			drawable,
			gc,
			(GdkDrawable *)icon_pixmap,
			0, 0,
			x, y,
			icon_width, icon_height
		    );
		}
	    }
	}
	gdk_gc_set_clip_mask(gc, NULL);


	/* Send the buffer to the window if it is not the window */
	if(drawable != (GdkDrawable *)window)
	    gdk_window_copy_area(
		window,
		gc,
		0, 0,
		(GdkWindow *)drawable,
		0, 0,
		width, height
	    );
}


/*
 *	Initializes the Progress Dialog.
 */
gint ProgressDialogInit(void)
{
	const gint border_major = 5;
	GtkAdjustment *adj;
	GtkWidget	*w,
			*parent, *parent2, *parent3, *parent4,
			*toplevel;
	GtkAccelGroup *accelgrp;
	GtkProgress *progress;
	GtkProgressBar *pb;
	ProgressDialog *pd;

	/* Already initialized? */
	if(progress_dialog != NULL)
	    return(0);

	/* Create the new Progress Dialog */
	progress_dialog = pd = PROGRESS_DIALOG(g_malloc0(
	    sizeof(ProgressDialog)
	));
	if(pd == NULL)
	    return(-3);

	pd->toplevel = toplevel = gtk_window_new(GTK_WINDOW_DIALOG);
	pd->accelgrp = accelgrp = gtk_accel_group_new();
/*
	pd->freeze_count = 0;
	pd->stop_count = 0;;
	pd->flags = 0;
	pd->transient_for = NULL;
 */
	pd->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_object_set_data_full(
	    GTK_OBJECT(w), PROGRESS_DIALOG_KEY,
	    pd, ProgressDialogDestroyCB
	);
#ifdef PROG_NAME
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "dialog", PROG_NAME
	);
#endif
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, TRUE);
	gtk_widget_set_usize(w, PROGRESS_DIALOG_WIDTH, PROGRESS_DIALOG_HEIGHT);
	gtk_window_set_title(GTK_WINDOW(w), "Progress");
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(ProgressDialogDeleteEventCB), pd
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "realize",
	    GTK_SIGNAL_FUNC(ProgressDialogRealizeCB), pd
	);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	parent = w;


	/* Main GtkVBox */
	pd->main_vbox = w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Animation GtkHBox */
	pd->animation_box = w = gtk_hbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	parent2 = w;

	/* Animation GtkDrawingArea */
	pd->animation_da = w = gtk_drawing_area_new();
	gtk_widget_set_events(
	    w,
	    GDK_STRUCTURE_MASK | GDK_EXPOSURE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "configure_event",
	    GTK_SIGNAL_FUNC(ProgressDialogAnimationConfigureEventCB), pd
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "expose_event",
	    GTK_SIGNAL_FUNC(ProgressDialogAnimationExposeEventCB), pd
	);
	gtk_drawing_area_size(
	    GTK_DRAWING_AREA(w),
	    PROGRESS_DIALOG_ANIMATION_AREA_WIDTH,
	    PROGRESS_DIALOG_ANIMATION_AREA_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Icon and Label GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* GtkVBox to align the icon */
	pd->icon_box = w = gtk_vbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	parent3 = w;

	/* No icon GtkPixmap at startup */
/*	pd->icon_pm = NULL; */

	/* GtkVBox to align the label */
	w = gtk_vbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* GtkHBox to align the label to the left */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* GtkLabel */
	pd->label = w = gtk_label_new("Processing...");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Progress Bar and Stop Button GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* GtkVBox to align the Progress Bar */
	w = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* GtkProgressBar */
	adj = (GtkAdjustment *)gtk_adjustment_new(
	    0.0f, 1.0f, 100.0f,
	    0.0f, 5.0f, 5.0f
	);
	pd->progress_bar = w = gtk_progress_bar_new_with_adjustment(adj);
	progress = GTK_PROGRESS(w);
	pb = GTK_PROGRESS_BAR(w);
	gtk_widget_set_usize(
	    w,
	    -1, PROGRESS_DIALOG_PROGRESS_BAR_HEIGHT
	);
	gtk_progress_bar_set_orientation(
	    pb,
	    GTK_PROGRESS_LEFT_TO_RIGHT
	);
	gtk_progress_bar_set_bar_style(
	    pb,
	    GTK_PROGRESS_CONTINUOUS
	);
	gtk_progress_set_activity_mode(
	    progress,
	    FALSE
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* GtkVBox to align the Stop Button */
	w = gtk_vbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Stop GtkButton */
	pd->stop_btn = w = gtk_button_new();
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(ProgressDialogStopCB), pd
	);
	gtk_accel_group_add(
	    accelgrp,
	    GDK_Escape, 0,
	    GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w),
	    "clicked"
	);
	gtk_widget_show(w);
	parent4 = w;
	/* Stop button GtkLabel */
	pd->stop_btn_label = w = gtk_label_new("Stop");
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);

	pd->freeze_count--;

	return(0);
}

/*
 *	Sets the Dialog's style.
 */
void ProgressDialogSetStyle(GtkRcStyle *rc_style)
{
	GtkWidget *w;
	ProgressDialog *pd = progress_dialog;
	if(pd == NULL)
	    return;

	w = pd->toplevel;
	if(w != NULL)
	{
	    if(rc_style != NULL)
	    {
		gtk_widget_modify_style(w, rc_style);
	    }
	    else
	    {
		rc_style = gtk_rc_style_new();
		gtk_widget_modify_style_recursive(w, rc_style);
		GTK_RC_STYLE_UNREF(rc_style)
	    }
	}
}

/*
 *	Sets the Dialog to be a transient for the given GtkWindow w.
 *
 *	If w is NULL then transient for will be unset.
 */
void ProgressDialogSetTransientFor(GtkWidget *w)
{
	ProgressDialog *pd = progress_dialog;
	if(pd == NULL)
	    return;

	if(pd->toplevel != NULL)
	{
	    if(w != NULL)
	    {
		if(!GTK_IS_WINDOW(GTK_OBJECT(w)))
		    return;

		if(GTK_WINDOW(w)->modal)
		    gtk_window_set_modal(GTK_WINDOW(w), FALSE);

		gtk_window_set_modal(
		    GTK_WINDOW(pd->toplevel), TRUE
		);
		gtk_window_set_transient_for(
		    GTK_WINDOW(pd->toplevel), GTK_WINDOW(w)
		);
		pd->transient_for = w;
	    }
	    else
	    {
		gtk_window_set_modal(
		    GTK_WINDOW(pd->toplevel), FALSE
		);
		gtk_window_set_transient_for(
		    GTK_WINDOW(pd->toplevel), NULL
		);
		pd->transient_for = NULL;
	    }
	}
}

/*
 *	Explicitly sets the progress dialog's toplevel WM icon.
 */
void ProgressDialogSetWMIconData(const guint8 **icon_data)
{
	GtkWidget *w;
	ProgressDialog *pd = progress_dialog;
	if(pd == NULL)
	    return;

	w = pd->toplevel;
	if(!GTK_WIDGET_REALIZED(w))
	    gtk_widget_realize(w);

	/* Set the WM icon */
	GUISetWMIcon(
	    w->window,
	    (guint8 **)icon_data
	);
}

/*
 *	Returns TRUE if dialog is mapped.
 */
gboolean ProgressDialogIsQuery(void)
{
	ProgressDialog *pd = progress_dialog;
	if(pd == NULL)
	    return(FALSE);

	return((pd->flags & PROGRESS_DIALOG_MAPPED) ? TRUE : FALSE);
}

/*
 *	Unmaps the progress dialog, breaking any queries.
 *
 *	If allow_gtk_iteration is TRUE then gtk_events_process()
 *	may be called one or more times during this call in order to
 *	ensure that changes get displayed during this call but may
 *	cause recursive function calls if not checked.
 */
void ProgressDialogBreakQuery(const gboolean allow_gtk_iteration)
{
	ProgressDialog *pd = progress_dialog;
	if(pd == NULL)
	    return;

	if(allow_gtk_iteration)
	    gtk_events_process();

	gdk_flush();

	ProgressDialogUnmap();

	gdk_flush();

	if(allow_gtk_iteration)
	    gtk_events_process();
}

/*
 *	Returns the number of stop counts.
 */
gint ProgressDialogStopCount(void)
{
	ProgressDialog *pd = progress_dialog;
	if(pd == NULL)
	    return(0);

	return(pd->stop_count);
}

/*
 *	Gets the progress dialog's toplevel widget.
 */
GtkWidget *ProgressDialogGetToplevel(void)
{
	ProgressDialog *pd = progress_dialog;
	if(pd == NULL)
	    return(NULL);

	return(pd->toplevel);
}


/*
 *	Maps the progress dialog for standard progress display.
 *
 *	If title is not NULL then title specifies the new title.
 *
 *	If label is not NULL then label specifies the new progress message.
 *
 *	The icon_data specifies the icon xpm data to load for the progress
 *	icon. The size of the icon should be 32 by 32.
 *
 *	If stop_btn_label is not NULL then stop_btn_label specifies the
 *	new stop button label.
 */
void ProgressDialogMap(
	const gchar *title,
	const gchar *label,
	const guint8 **icon_data,
	const gchar *stop_btn_label
)
{
	GtkWidget *w;
	ProgressDialog *pd = progress_dialog;
	if(pd == NULL)
	    return;

	/* Reset the stop count */
	pd->stop_count = 0;

	/* Set the title */
	if(title != NULL)
	    gtk_window_set_title(
		GTK_WINDOW(pd->toplevel),
		title
	    );

	/* Set the GtkLabel */
	if(label != NULL)
	    gtk_label_set_text(
		GTK_LABEL(pd->label),
		label
	    );

	/* Map the icon GtkBox */
	gtk_widget_show(pd->icon_box);

	/* Unmap the Animation GtkBox */
	gtk_widget_hide(pd->animation_box);

	/* Set the icon */
	if(icon_data != NULL)
  	    ProgressDialogSetIcon(
		pd,
		icon_data
	    );
	
	/* Set the stop GtkButton label */
	if(stop_btn_label != NULL)
	    gtk_label_set_text(
		GTK_LABEL(pd->stop_btn_label),
		stop_btn_label
	    );

	/* Set the Stop GtkButton as the default button */
	w = pd->stop_btn;
	gtk_widget_grab_focus(w);
	gtk_widget_grab_default(w);

	/* Map the progress dialog */
	if(!(pd->flags & PROGRESS_DIALOG_MAPPED))
	{
	    gtk_widget_show_raise(pd->toplevel);
	    pd->flags |= PROGRESS_DIALOG_MAPPED;
	}
}

/*
 *	Maps the progress dialog for standard progress display.
 *
 *	If title is not NULL then title specifies the new title.
 *
 *	If label is not NULL then label specifies the new progress message.
 *
 *	The icon_path specifies the icon xpm file path. The size of the
 *	icon should be 32 by 32.
 *
 *	If stop_btn_label is not NULL then stop_btn_label specifies the
 *	new stop button label.
 */
void ProgressDialogMapFile(
	const gchar *title,
	const gchar *label,
	const gchar *icon_path,
	const gchar *stop_btn_label
)
{
	GtkWidget *w;
	ProgressDialog *pd = progress_dialog;
	if(pd == NULL)
	    return;

	/* Reset the stop count */
	pd->stop_count = 0;

	/* Set the title */
	if(title != NULL)
	    gtk_window_set_title(
		GTK_WINDOW(pd->toplevel),
		title
	    );

	/* Set the GtkLabel */
	if(label != NULL)
	    gtk_label_set_text(
		GTK_LABEL(pd->label),
		label
	    );

	/* Map the icon GtkBox */
	gtk_widget_show(pd->icon_box);

	/* Unmap the Animation GtkBox */
	gtk_widget_hide(pd->animation_box);

	/* Set the icon */
	if(icon_path != NULL)
  	    ProgressDialogSetIconFile(
		pd,
		icon_path
	    );
	
	/* Set the stop GtkButton label */
	if(stop_btn_label != NULL)
	    gtk_label_set_text(
		GTK_LABEL(pd->stop_btn_label),
		stop_btn_label
	    );

	/* Set the Stop GtkButton as the default button */
	w = pd->stop_btn;
	gtk_widget_grab_focus(w);
	gtk_widget_grab_default(w);

	/* Map the progress dialog */
	if(!(pd->flags & PROGRESS_DIALOG_MAPPED))
	{
	    gtk_widget_show_raise(pd->toplevel);
	    pd->flags |= PROGRESS_DIALOG_MAPPED;
	}
}

/*
 *	Maps the progress dialog in animation mode and loads the
 *	animation icons from XPM data.
 *
 *	The start_icon_datas_list, icon_datas_list, and
 *	end_icon_datas_list specifies the animation icon datas list,
 *	each a GList of guint8 ** XPM data.
 *
 *	The animation_interval_ms specifies the animation interval in
 *	milliseconds (default value is 500 ms).
 *
 *	The animation_increment specifies the value to advance the
 *	animation position by in animation units per animation interval
 *	from (guint16)1 to (guint16)-1. Smaller values produce slower
 *	and finer animation while larger values produce faster and
 *	rougher animation (default value is (guint16)10000).
 */
void ProgressDialogMapAnimation(
	const gchar *title,
	const gchar *label,
	const gchar *stop_btn_label,
	GList *start_icon_datas_list,
	GList *icon_datas_list,
	GList *end_icon_datas_list,
	const gulong animation_interval_ms,
	const guint16 animation_increment
)
{
	GtkWidget *w;
	ProgressDialog *pd = progress_dialog;
	if(pd == NULL)
	    return;

	/* Reset the stop count */
	pd->stop_count = 0;

	/* Set the title */
	if(title != NULL)
	    gtk_window_set_title(
		GTK_WINDOW(pd->toplevel),
		title
	    );

	/* Set the message */
	if(label != NULL)
	    gtk_label_set_text(
		GTK_LABEL(pd->label),
		label
	    );

	/* Unmap the Icon GtkBox */
	gtk_widget_hide(pd->icon_box);

	/* Map the Animation GtkBox */
	gtk_widget_show(pd->animation_box);

	/* Set the stop GtkButton label */
	if(stop_btn_label != NULL)
	    gtk_label_set_text(
		GTK_LABEL(pd->stop_btn_label),
		stop_btn_label
	    );

	/* Delete the current animation icons */
	ProgressDialogClearAnimationIcons(pd);

	/* Set the new animation icons
	 *
	 * Start icons
	 */
	if(start_icon_datas_list != NULL)
	{
	    GList *glist;
	    for(glist = start_icon_datas_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
		pd->start_icons_list = g_list_append(
		    pd->start_icons_list,
		    ProgressDialogIconNewFromData(
			pd,
			(const guint8 **)glist->data
		    )
		);
	}

	/* Icons */
	if(icon_datas_list != NULL)
	{
	    GList *glist;
	    for(glist = icon_datas_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
		pd->icons_list = g_list_append(
		    pd->icons_list,
		    ProgressDialogIconNewFromData(
			pd,
			(const guint8 **)glist->data
		    )
		);
	}

	/* End icons */
	if(end_icon_datas_list != NULL)
	{
	    GList *glist;
	    for(glist = end_icon_datas_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
		pd->end_icons_list = g_list_append(
		    pd->end_icons_list,
		    ProgressDialogIconNewFromData(
			pd,
			(const guint8 **)glist->data
		    )
		);
	}

	/* Reset the animation values */
	if(pd->animation_timer == NULL)
	    pd->animation_timer = g_timer_new();
	g_timer_start(pd->animation_timer);	/* Restart from 0 */
	pd->animation_interval_ms = animation_interval_ms;
	pd->animation_position = 0;
	pd->animation_increment = animation_increment;

	/* Set the Stop GtkButton as the default GtkButton */
	w = pd->stop_btn;
	gtk_widget_grab_focus(w);
	gtk_widget_grab_default(w);

	/* Map the progress dialog */
	if(!(pd->flags & PROGRESS_DIALOG_MAPPED))
	{
	    gtk_widget_show_raise(pd->toplevel);
	    pd->flags |= PROGRESS_DIALOG_MAPPED;
	}
}

/*
 *	Maps the progress dialog in animation mode and loads the
 *	animation icons from XPM data.
 *
 *	The start_icon_paths_list, icon_paths_list, and
 *	end_icon_paths_list specifies the animation icon files list,
 *	each a GList of gchar * path to a XPM file.
 *
 *	The animation_interval_ms specifies the animation interval in
 *	milliseconds (default value is 500 ms).
 *
 *	The animation_increment specifies the value to advance the
 *	animation position by in animation units per animation interval
 *	from (guint16)1 to (guint16)-1. Smaller values produce slower
 *	and finer animation while larger values produce faster and
 *	rougher animation (default value is (guint16)10000).
 */
void ProgressDialogMapAnimationFile(
	const gchar *title,
	const gchar *label,
	const gchar *stop_btn_label,
	GList *start_icon_paths_list,
	GList *icon_paths_list,
	GList *end_icon_paths_list,
	const gulong animation_interval_ms,
	const guint16 animation_increment
)
{
	GtkWidget *w;
	ProgressDialog *pd = progress_dialog;
	if(pd == NULL)
	    return;

	/* Reset the stop count */
	pd->stop_count = 0;

	/* Set the title */
	if(title != NULL)
	    gtk_window_set_title(
		GTK_WINDOW(pd->toplevel),
		title
	    );

	/* Set the message */
	if(label != NULL)
	    gtk_label_set_text(
		GTK_LABEL(pd->label),
		label
	    );

	/* Unmap the Icon GtkBox */
	gtk_widget_hide(pd->icon_box);

	/* Map the Animation GtkBox */
	gtk_widget_show(pd->animation_box);

	/* Set the stop GtkButton label */
	if(stop_btn_label != NULL)
	    gtk_label_set_text(
		GTK_LABEL(pd->stop_btn_label),
		stop_btn_label
	    );

	/* Delete the current animation icons */
	ProgressDialogClearAnimationIcons(pd);

	/* Set the new animation icons
	 *
	 * Start icons
	 */
	if(start_icon_paths_list != NULL)
	{
	    GList *glist;
	    for(glist = start_icon_paths_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
		pd->start_icons_list = g_list_append(
		    pd->start_icons_list,
		    ProgressDialogIconNewFromFile(
			pd,
			(const gchar *)glist->data
		    )
		);
	}

	/* Icons */
	if(icon_paths_list != NULL)
	{
	    GList *glist;
	    for(glist = icon_paths_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
		pd->icons_list = g_list_append(
		    pd->icons_list,
		    ProgressDialogIconNewFromFile(
			pd,
			(const gchar *)glist->data
		    )
		);
	}

	/* End icons */
	if(end_icon_paths_list != NULL)
	{
	    GList *glist;
	    for(glist = end_icon_paths_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
		pd->end_icons_list = g_list_append(
		    pd->end_icons_list,
		    ProgressDialogIconNewFromFile(
			pd,
			(const gchar *)glist->data
		    )
		);
	}

	/* Reset the animation values */
	if(pd->animation_timer == NULL)
	    pd->animation_timer = g_timer_new();
	g_timer_start(pd->animation_timer);	/* Restart from 0 */
	pd->animation_interval_ms = animation_interval_ms;
	pd->animation_position = 0;
	pd->animation_increment = animation_increment;

	/* Set the Stop GtkButton as the default GtkButton */
	w = pd->stop_btn;
	gtk_widget_grab_focus(w);
	gtk_widget_grab_default(w);

	/* Map the progress dialog */
	if(!(pd->flags & PROGRESS_DIALOG_MAPPED))
	{
	    gtk_widget_show_raise(pd->toplevel);
	    pd->flags |= PROGRESS_DIALOG_MAPPED;
	}
}

/*
 *	Updates the progress dialog with the given values.
 *
 *	This function has no affect if ProgressDialogMap() was not called
 *	before to map the progress dialog.
 *
 *	If allow_gtk_iteration is TRUE then gtk_events_process()
 *	may be called one or more times during this call in order to
 *	ensure that changes get displayed during this call but may
 *	cause recursive function calls if not checked.
 */
void ProgressDialogUpdate(
	const gchar *title,
	const gchar *label,
	const guint8 **icon_data,
	const gchar *stop_btn_label,
	const gfloat position,			/* 0.0 to 1.0 */
	const guint nblocks,			/* 0 for continuous */
	const gboolean allow_gtk_iteration
)
{
	gboolean	updated_message,
			updated_progress;
	GTimer *animation_timer;
	GtkWidget *w;
	ProgressDialog *pd = progress_dialog;
	if(pd == NULL)
	    return;

	/* Do not update if not mapped */
	if(!(pd->flags & PROGRESS_DIALOG_MAPPED))
	    return;

	updated_message = FALSE;
	updated_progress = FALSE;

	/* Set the title? */
	if(title != NULL)
	{
	    gtk_window_set_title(
		GTK_WINDOW(pd->toplevel),
		title
	    );
	    updated_message = TRUE;
	}

	/* Set the label? */
	if(label != NULL)
	{
	    gtk_label_set_text(
		GTK_LABEL(pd->label),
		label
	    );
	    updated_message = TRUE;
	}

	/* Set the icon? */
	if(icon_data != NULL)
	{
	    ProgressDialogSetIcon(
		pd,
		icon_data
	    );
	    updated_message = TRUE;
	}

	/* Set the stop button label? */
	if(stop_btn_label != NULL)
	{
	    gtk_label_set_text(
		GTK_LABEL(pd->stop_btn_label),
		stop_btn_label
	    );
	    updated_message = TRUE;
	}

	/* Begin updating the GtkProgressBar's value */
	w = pd->progress_bar;
	if(w != NULL)
	{
	    GtkProgress *progress = GTK_PROGRESS(w);
	    GtkAdjustment *adj = progress->adjustment;
	    gfloat value = (position * (adj->upper - adj->lower)) + adj->lower;
	    GtkProgressBar *pb = GTK_PROGRESS_BAR(w);

	    /* Clip the value */
	    if(value > adj->upper)
		value = adj->upper;
	    else if(value < adj->lower)
		value = adj->lower;

	    /* Set the GtkProgressBar to display a known value */
	    if(progress->activity_mode)
	    {
		gtk_progress_set_activity_mode(
		    progress,
		    FALSE
		);
		updated_message = TRUE;
	    }
	    if(progress->show_text)
	    {
		gtk_progress_set_show_text(
		    progress,
		    FALSE
		);
		updated_message = TRUE;
	    }

	    /* Number of blocks to display specified? */
	    if(nblocks > 0)
	    {
		/* Set the GtkProgressBar to display blocks */
		if(pb->bar_style != GTK_PROGRESS_DISCRETE)
		{
		    gtk_progress_bar_set_bar_style(
			pb,
			GTK_PROGRESS_DISCRETE
		    );
		    updated_message = TRUE;
		}
		if(pb->blocks != nblocks)
		{
		    gtk_progress_bar_set_discrete_blocks(
			pb,
			nblocks
		    );
		    updated_message = TRUE;
		}
	    }
	    else
	    {
		/* Set the GtkProgressBar to display a continuous value */
		if(pb->bar_style != GTK_PROGRESS_CONTINUOUS)
		{
		    gtk_progress_bar_set_bar_style(
			pb,
			GTK_PROGRESS_CONTINUOUS
		    );
		    updated_message = TRUE;
		}
	    }

	    /* Set the GtkProgressBar's new value */
	    if(value != adj->value)
	    {
		gtk_progress_set_value(
		    progress,
		    value
		);
		updated_progress = TRUE;
	    }
	}

	/* Check if the Animation is mapped */
	w = pd->animation_box;
	animation_timer = pd->animation_timer;
	if(GTK_WIDGET_MAPPED(w) && (animation_timer != NULL))
	{
#if 1
	    const gulong elapsed_ms = G_TIMER_ELAPSED_MS(animation_timer);
	    gfloat tc;

	    g_timer_start(animation_timer);	/* Restart from 0 */

	    /* Calculate the time coefficient */
	    if(pd->animation_interval_ms > 0l)
		tc = (gfloat)elapsed_ms / (gfloat)pd->animation_interval_ms;
	    else
		tc = 0.0f;

	    /* Advance the animation position by the elapsed time */
	    pd->animation_position += (guint16)(
		pd->animation_increment * tc
	    );

	    /* Redraw the animation */
	    ProgressDialogDrawAnimation(pd);

	    /* With the animation mapped, there is always an update */
	    updated_progress = TRUE;
#else
/* For syncing animation position with progress (not used anymore) */
	    /* Set animation position to match progress position */
	    pd->animation_position = (guint16)(
		_position * (guint16)-1
	    );

	    /* Redraw animation */
	    ProgressDialogDrawAnimation(pd);
#endif
	}

	/* Process any GdkEvents so that the changes made above get
	 * displayed before the end of this call
	 */
	if(allow_gtk_iteration)
	{
	    if(updated_message)
		gtk_events_process();
	    else if(updated_progress)
		gtk_events_process_current();
	}
}

/*
 *	Updates the progress dialog with the given values.
 *
 *	This function has no affect if ProgressDialogMap() was not
 *	called before to map the progress dialog.
 *
 *	If allow_gtk_iteration is TRUE then gtk_events_process()
 *	may be called one or more times during this call in order to
 *	ensure that changes get displayed during this call but may
 *	cause recursive function calls if not checked.
 */
void ProgressDialogUpdateUnknown(
	const gchar *title,
	const gchar *label,
	const guint8 **icon_data,
	const gchar *stop_btn_label,
	const gboolean allow_gtk_iteration
)
{
	gboolean	updated_message,
			updated_progress;
	GTimer *animation_timer;
	GtkWidget *w;
	ProgressDialog *pd = progress_dialog;
	if(pd == NULL)
	    return;

	/* Do not update if not mapped */
	if(!(pd->flags & PROGRESS_DIALOG_MAPPED))
	    return;

	updated_message = FALSE;
	updated_progress = FALSE;

	/* Set the title? */
	if(title != NULL)
	{
	    gtk_window_set_title(
		GTK_WINDOW(pd->toplevel),
		title
	    );
	    updated_message = TRUE;
	}

	/* Set the label? */
	if(label != NULL)
	{
	    gtk_label_set_text(
		GTK_LABEL(pd->label),
		label
	    );
	    updated_message = TRUE;
	}

	/* Set the icon? */
	if(icon_data != NULL)
	{
	    ProgressDialogSetIcon(
		pd,
		icon_data
	    );
	    updated_message = TRUE;
	}

	/* Set the stop button label? */
	if(stop_btn_label != NULL)
	{
	    gtk_label_set_text(
		GTK_LABEL(pd->stop_btn_label),
		stop_btn_label
	    );
 	    updated_message = TRUE;
	}

	/* Begin updating progress bar position in activity mode */
	w = pd->progress_bar;
	if(w != NULL)
	{
	    GtkProgress *progress = GTK_PROGRESS(w);
	    GtkAdjustment *adj = progress->adjustment;
	    GtkProgressBar *pb = GTK_PROGRESS_BAR(w);
	    gfloat value = gtk_progress_get_value(progress) + 1.0f;
	    if(value > adj->upper)
		value = adj->lower;

	    if(!progress->activity_mode)
	    {
		gtk_progress_set_activity_mode(
		    progress,
		    TRUE
		);
		updated_message = TRUE;
	    }
	    if(progress->show_text)
	    {
		gtk_progress_set_show_text(
		    progress,
		    FALSE
		);
		updated_message = TRUE;
	    }
	    if(pb->bar_style != GTK_PROGRESS_CONTINUOUS)
	    {
		gtk_progress_bar_set_bar_style(
		    pb,
		    GTK_PROGRESS_CONTINUOUS
		);
		updated_message = TRUE;
	    }

	    gtk_progress_set_value(
		progress,
		value
	    );

	    /* There is always an update when the value is unknown */
 	    updated_progress = TRUE;
	}

	/* Check if the Animation is mapped */
	w = pd->animation_box;
	animation_timer = pd->animation_timer;
	if(GTK_WIDGET_MAPPED(w) && (animation_timer != NULL))
	{
	    const gulong elapsed_ms = G_TIMER_ELAPSED_MS(animation_timer);
	    gfloat tc;

	    g_timer_start(animation_timer);	/* Restart from 0 */

	    /* Calculate the time coefficient */
	    if(pd->animation_interval_ms > 0l)
		tc = (gfloat)elapsed_ms / (gfloat)pd->animation_interval_ms;
	    else
		tc = 0.0f;

	    /* Advance the animation position by the elapsed time */
	    pd->animation_position += (guint16)(
		pd->animation_increment * tc
	    );

	    /* Redraw the animation */
	    ProgressDialogDrawAnimation(pd);

	    /* With the animation mapped, the progress is always
	     * considered updated
	     */
	    updated_progress = TRUE;
	}

	/* Process any GdkEvents so that the changes made above get
	 * displayed before the end of this call
	 */
	if(allow_gtk_iteration)
	{
	    if(updated_message)
		gtk_events_process();
	    else if(updated_progress)
		gtk_events_process_current();
	}
}

/*
 *	Unmaps the progress dialog.
 */
void ProgressDialogUnmap(void)
{
	GtkWidget *w;
	ProgressDialog *pd = progress_dialog;
	if(pd == NULL)
	    return;

	/* Delete all of the animation icons */
	ProgressDialogClearAnimationIcons(pd);

	/* Do not check the current map state */

	w = pd->toplevel;
	gtk_widget_hide(w);
	gtk_widget_unmap(w);
	pd->flags &= ~PROGRESS_DIALOG_MAPPED;
}

/*
 *	Shuts down the Progress Dialog.
 */
void ProgressDialogShutdown(void)
{
	ProgressDialog *pd = progress_dialog;
	if(pd == NULL)
	    return;

	/* Unmap the Progress Dialog manually */
	gtk_widget_hide(pd->toplevel);

	/* Unset the transient for widget as needed */
	if(pd->transient_for != NULL)
	    ProgressDialogSetTransientFor(NULL);

	/* Delete the rest of the Progress Dialog by destroying its
	 * toplevel GtkWidget which will call
	 * ProgressDialogDestroyCB() to delete the rest of the
	 * Progress Dialog
	 */
	gtk_widget_destroy(pd->toplevel);
}
